% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/distances.R
\name{distTips}
\alias{distTips}
\title{Compute some phylogenetic distance between tips}
\usage{
distTips(
  x,
  tips = "all",
  method = c("patristic", "nNodes", "Abouheif", "sumDD"),
  useC = TRUE
)
}
\arguments{
\item{x}{a tree of class \code{\link[ape:read.tree]{phylo}},
\linkS4class{phylo4} or \linkS4class{phylo4d}.}

\item{tips}{A vector of integers identifying tips by their numbers, or a
vector of characters identifying tips by their names. Distances will be
computed between all possible pairs of tips.}

\item{method}{a character string (full or abbreviated without ambiguity)
specifying the method used to compute distances ; possible values are:\cr -
\code{patristic}: patristic distance, i.e. sum of branch lengths \cr -
\code{nNodes}: number of nodes on the path between the nodes \cr -
\code{Abouheif}: Abouheif's distance (see details) \cr - \code{sumDD}: sum
of direct descendants of all nodes on the path (see details) \cr}

\item{useC}{a logical indicating whether computations should be performed
using compiled C code (TRUE, default), or using a pure R version (FALSE). C
version is several orders of magnitude faster, and R version is kept for
backward compatibility.}
}
\value{
An object of class \code{dist}, containing phylogenetic distances.
}
\description{
The function \code{distTips} computes a given distance between a set of tips
of a phylogeny. A vector of tips is supplied: distances between all possible
pairs of these tips are computed.  The distances are computed from the
shortest path between the tips. Several distances can be used, defaulting to
the sum of branch lengths (see argument \code{method}).
}
\details{
An option (enabled by default) allows computations to be run using compiled
C code, which is much faster than pure R code. In this case, a matrix of all
pairwise distances is returned (i.e., \code{tips} argument is ignored).

\code{Abouheif} distance refers to the phylogenetic distance underlying the
test of Abouheif (see references). Let P be the set of all the nodes in the
path going from \code{node1} to \code{node2}. Let DDP be the number of
direct descendants from each node in P. Then, the so-called 'Abouheif'
distance is the product of all terms in DDP.\cr

\code{sumDD} refers to a phylogenetic distance quite similar to that of
Abouheif. We consider the same sets P and DDP. But instead of computing the
product of all terms in DDP, this distance computes the sum of all terms in
DDP.
}
\examples{

if(require(ape) & require(phylobase)){
## make a tree
x <- as(rtree(10),"phylo4")
plot(x, show.node=TRUE)
axisPhylo()
## compute different distances
distTips(x, 1:3)
distTips(x, 1:3, "nNodes")
distTips(x, 1:3, "Abouheif")
distTips(x, 1:3, "sumDD")

## compare C and pure R code outputs
x <- rtree(10)
all.equal(as.matrix(distTips(x)), as.matrix(distTips(x, useC=FALSE)))
all.equal(as.matrix(distTips(x, meth="nNode")),
   as.matrix(distTips(x, meth="nNode", useC=FALSE)))
all.equal(as.matrix(distTips(x, meth="Abou")),
   as.matrix(distTips(x, meth="Abou", useC=FALSE)))
all.equal(as.matrix(distTips(x, meth="sumDD")),
   as.matrix(distTips(x, meth="sumDD", useC=FALSE)))

## compare speed
x <- rtree(50)
tim1 <- system.time(distTips(x, useC=FALSE)) # old pure R version
tim2 <- system.time(distTips(x)) # new version using C
tim1[c(1,3)]/tim2[c(1,3)] # C is about a thousand time faster in this case
}

}
\references{
Pavoine, S.; Ollier, S.; Pontier, D. & Chessel, D. (2008)
Testing for phylogenetic signal in life history variable: Abouheif's test
revisited. \emph{Theoretical Population Biology}: \bold{73}, 79-91.
}
\seealso{
\code{\link{distTips}} which computes several phylogenetic
distances between tips.
}
\author{
Thibaut Jombart \email{tjombart@imperial.ac.uk}
}
\keyword{manip}
