% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/ergm_tidiers.R
\name{ergm_tidiers}
\alias{ergm_tidiers}
\alias{tidy.ergm}
\alias{glance.ergm}
\title{Tidying methods for an exponential random graph model}
\usage{
\method{tidy}{ergm}(x, conf.int = FALSE, conf.level = 0.95,
  exponentiate = FALSE, quick = FALSE, ...)

\method{glance}{ergm}(x, deviance = FALSE, mcmc = FALSE, ...)
}
\arguments{
\item{x}{an \pkg{ergm} object}

\item{conf.int}{whether to include a confidence interval}

\item{conf.level}{confidence level of the interval, used only if
\code{conf.int=TRUE}}

\item{exponentiate}{whether to exponentiate the coefficient estimates
and confidence intervals}

\item{quick}{whether to compute a smaller and faster version, containing
only the \code{term} and \code{estimate} columns.}

\item{...}{extra arguments passed to \code{\link[ergm]{summary.ergm}}}

\item{deviance}{whether to report null and residual deviance for the model,
along with degrees of freedom; defaults to \code{FALSE}}

\item{mcmc}{whether to report MCMC interval, burn-in and sample size used to
estimate the model; defaults to \code{FALSE}}
}
\value{
All tidying methods return a \code{data.frame} without rownames.
The structure depends on the method chosen.

\code{tidy.ergm} returns one row for each coefficient, with five columns:
  \item{term}{The term in the model being estimated and tested}
  \item{estimate}{The estimated coefficient}
  \item{std.error}{The standard error}
  \item{mcmc.error}{The MCMC error}
  \item{p.value}{The two-sided p-value}

If \code{conf.int=TRUE}, it also includes columns for \code{conf.low} and
\code{conf.high}.

\code{glance.ergm} returns a one-row data.frame with the columns
  \item{independence}{Whether the model assumed dyadic independence}
  \item{iterations}{The number of iterations performed before convergence}
  \item{logLik}{If applicable, the log-likelihood associated with the model}
  \item{AIC}{The Akaike Information Criterion}
  \item{BIC}{The Bayesian Information Criterion}

If \code{deviance=TRUE}, and if the model supports it, the
data frame will also contain the columns
  \item{null.deviance}{The null deviance of the model}
  \item{df.null}{The degrees of freedom of the null deviance}
  \item{residual.deviance}{The residual deviance of the model}
  \item{df.residual}{The degrees of freedom of the residual deviance}

Last, if \code{mcmc=TRUE}, the data frame will also contain
the columns
  \item{MCMC.interval}{The interval used during MCMC estimation}
  \item{MCMC.burnin}{The burn-in period of the MCMC estimation}
  \item{MCMC.samplesize}{The sample size used during MCMC estimation}
}
\description{
These methods tidy the coefficients of an exponential random graph model
estimated with the \pkg{ergm} package into a summary, and construct
a one-row glance of the model's statistics. The methods should work with
any model that conforms to the \pkg{ergm} class, such as those
produced from weighted networks by the \pkg{ergm.count} package.
}
\details{
There is no \code{augment} method for \pkg{ergm} objects.
}
\examples{

\dontrun{
if (require("ergm")) {
    # Using the same example as the ergm package
    # Load the Florentine marriage network data
    data(florentine)

    # Fit a model where the propensity to form ties between
    # families depends on the absolute difference in wealth
    gest <- ergm(flomarriage ~ edges + absdiff("wealth"))

    # Show terms, coefficient estimates and errors
    tidy(gest)

    # Show coefficients as odds ratios with a 99\% CI
    tidy(gest, exponentiate = TRUE, conf.int = TRUE, conf.level = 0.99)

    # Take a look at likelihood measures and other
    # control parameters used during MCMC estimation
    glance(gest)
    glance(gest, deviance = TRUE)
    glance(gest, mcmc = TRUE)
}
}
}
\references{
Hunter DR, Handcock MS, Butts CT, Goodreau SM, Morris M (2008b).
\pkg{ergm}: A Package to Fit, Simulate and Diagnose Exponential-Family
Models for Networks. \emph{Journal of Statistical Software}, 24(3).
\url{http://www.jstatsoft.org/v24/i03/}.
}
\seealso{
\code{\link[ergm]{ergm}},
\code{\link[ergm]{control.ergm}},
\code{\link[ergm]{summary.ergm}}
}
