% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/slidingwin.R
\name{slidingwin}
\alias{slidingwin}
\title{Test for a climate windows in data.}
\usage{
slidingwin(exclude = NA, xvar, cdate, bdate, baseline, type, refday,
  stat = "mean", func = "lin", range, cmissing = FALSE,
  cinterval = "day", k = 0, upper = NA, lower = NA, binary = FALSE,
  centre = list(NULL, "both"), spatial = NULL, cohort = NULL,
  cutoff.day = NULL, cutoff.month = NULL, furthest = NULL,
  closest = NULL, thresh = NULL, cvk = NULL)
}
\arguments{
\item{exclude}{Two values (distance and duration) which allow users
to exclude short-duration long-lag climate windows from analysis (e.g., 
windows with a duration of 10 days which occur over a month ago).
These windows are often considered to be biologically implausible.}

\item{xvar}{A list object containing all climate variables of interest. 
Please specify the parent environment and variable name (e.g. climate$Temp).}

\item{cdate}{The climate date variable (dd/mm/yyyy). Please specify the 
parent environment and variable name (e.g. climate$Date).}

\item{bdate}{The biological date variable (dd/mm/yyyy). Please specify the 
parent environment and variable name (e.g. Biol$Date).}

\item{baseline}{The baseline model structure used for model testing. 
Currently known to support lm, glm, lmer, glmer and coxph objects.}

\item{type}{"absolute" or "relative", whether you wish the climate window to be relative
(e.g. the number of days before each biological record is measured) or absolute
(e.g. number of days before a set point in time).}

\item{refday}{If type is "absolute", the day and month respectively of the 
year from which the absolute window analysis will start.}

\item{stat}{The aggregate statistics used to analyse the climate data. Can 
currently use basic R statistics (e.g. mean, min), as well as slope. 
Additional aggregate statistics can be created using the format 
function(x) (...). See FUN in \code{\link{apply}} for more detail.}

\item{func}{The functions used to fit the climate variable. Can be linear 
("lin"), quadratic ("quad"), cubic ("cub"), inverse ("inv") or log ("log").}

\item{range}{Two values signifying respectively the furthest and closest number 
of time intervals (set by cinterval) back from the refday or biological record to include 
in the climate window search.}

\item{cmissing}{Determines what should be done if there are 
missing climate data. Three approaches are possible: 
 - FALSE; the function will not run if missing climate data is encountered.
 An object 'missing' will be returned containing the dates of missing climate.
 - "method1"; missing climate data will be replaced with the mean climate
 of the preceding and following 2 days.
 - "method2"; missing climate data will be replaced with the mean climate
 of all records on the same date.
 
 Note: Other methods are possible. Users should consider those methods most
 appropriate for their data.}

\item{cinterval}{The resolution at which climate window analysis will be 
conducted. May be days ("day"), weeks ("week"), or months ("month"). Note the units
of parameter 'range' will differ depending on the choice
of cinterval.}

\item{k}{The number of folds used for k-fold cross validation. By default
this value is set to 0, so no cross validation occurs. Value should be a
minimum of 2 for cross validation to occur.}

\item{upper}{Cut-off values used to determine growing degree days or positive 
climate thresholds (depending on parameter thresh). Note that when values
of lower and upper are both provided, slidingwin will instead calculate an 
optimal climate zone.}

\item{lower}{Cut-off values used to determine chill days or negative 
climate thresholds (depending on parameter thresh). Note that when values
of lower and upper are both provided, slidingwin will instead calculate an 
optimal climate zone.}

\item{binary}{TRUE or FALSE. Determines whether to use values of upper and
lower to calculate binary climate data (binary = TRUE), or to use for
growing degree days (binary = FALSE).}

\item{centre}{A list item containing:
1. The variable used for mean centring (e.g. Year, Site, Individual). 
Please specify the parent environment and variable name (e.g. Biol$Year).
2. Whether the model should include both within-group means and variance ("both"),
only within-group means ("mean"), or only within-group variance ("var").}

\item{spatial}{A list item containing:
1. A factor that defines which spatial group (i.e. population) each biological
record is taken from. The length of this factor should correspond to the length 
of the biological dataset.
2. A factor that defines which spatial group (i.e. population) climate data
corresponds to. This length of this factor should correspond to the length of
the climate dataset.}

\item{cohort}{A variable used to group biological records that occur in the same biological
season but cover multiple years (e.g. southern hemisphere breeding season). Only required
when type is "absolute". The cohort variable should be in the same dataset as the variable bdate.}

\item{cutoff.day, cutoff.month}{Redundant parameters. Now replaced by refday.}

\item{furthest, closest}{Redundant parameters. Now replaced by range.}

\item{thresh}{Redundant parameter. Now replaced by binary.}

\item{cvk}{Redundant parameter. Now replaced by k.}
}
\value{
Will return a list with an output for each tested set of climate
 window parameters. Each list item contains three objects:
 
 \itemize{
 \item BestModel, a model object. The strongest climate window model based on AICc. 
 \item BestModelData, a dataframe used to fit the strongest climate window model.
 \item Dataset, a dataframe with information on all fitted climate windows. 
 Ordered using deltaAICc, with most negative deltaAICc values first. 
 See \code{\link{MassOutput}} as an example.}
 
 In addition, the returned list includes an object 'combos', a summary of all
 tested sets of climate window parameters.
}
\description{
Finds the time period when a biological variable is most strongly affected 
by climate. Note that climate data and biological data should be loaded as 
two seperate objects. Both objects should contain a date column to designate
when the data were recorded (dd/mm/yyyy).
}
\details{
Note that slidingwin allows you to test multiple possible parameters with the
same code (e.g. func, stat, xvar). See examples for more detail.
}
\examples{
\dontrun{
##EXAMPLE 1## 
 
# Test both a linear and quadratic variable climate window using datasets "Offspring"
# and "OffspringClimate".

# Load data.

data(Offspring) 
data(OffspringClimate)

# Test both linear and quadratic functions with climate variable temperature

OffspringWin <- slidingwin(xvar = list(Temp = OffspringClimate$Temperature), 
                          cdate = OffspringClimate$Date, 
                          bdate = Offspring$Date, 
                          baseline = glm(Offspring ~ 1, data = Offspring, family = poisson),
                          range = c(150, 0), 
                          type = "relative", stat = "mean", 
                          func = c("lin", "quad"), cmissing = FALSE, cinterval = "day")

# Examine tested combinations
 
OffspringWin$combos
     
# View output for func = "lin"
 
head(OffspringWin[[1]]$Dataset) 
summary(OffspringWin[[1]]$BestModel)
 
# View output for func = "quad"
 
head(OffspringWin[[2]]$Dataset)
summary(OffspringWin[[2]]$BestModel)
 
##EXAMPLE 2##
 
# Test for an absolute climate window with both 'mean' and 'max' aggregate statistics
# using datasets 'Mass' and 'MassClimate'.
 
# Load data.
 
data(Mass)
data(MassClimate)
 
# Test an absolute window, starting 20 May (refday = c(20, 5))
# Test for climate windows between 100 and 0 days ago (range = c(100, 0))
# Test both mean and max aggregate statistics (stat = c("mean", "max"))
# Fit a linear term (func = "lin")
# Test at the resolution of days (cinterval = "day")
 
MassWin <- slidingwin(xvar = list(Temp = MassClimate$Temp), cdate = MassClimate$Date, 
                     bdate = Mass$Date, baseline = lm(Mass ~ 1, data = Mass),
                     range = c(100, 0),
                     stat = c("mean", "max"), func = "lin",
                     type = "absolute", refday = c(20, 5),
                     cmissing = FALSE, cinterval = "day")
                       
# Examine tested combinations
 
MassWin$combos                      
 
# View output for mean temperature
 
head(MassWin[[1]]$Dataset)
summary(MassWin[[1]]$BestModel)
 
# View output for max temperature
 
head(MassWin[[2]]$Dataset)
summary(MassWin[[2]]$BestModel)
 
}
 
}
\author{
Liam D. Bailey and Martijn van de Pol
}

