#' @title Balance Statistics for `MatchThem` Objects
#' 
#' @description
#' Generates balance statistics for `mimids` and `wimids` objects from \pkg{MatchThem}.
#' 
#' @inheritParams bal.tab
#' @param x a `mimids` or `wimids` object; the output of a call to \pkgfun{MatchThem}{matchthem} or \pkgfun{MatchThem}{weightthem}.
#' @param distance an optional formula or data frame containing distance values (e.g., propensity scores) or a character vector containing their names. If a formula or variable names are specified, `bal.tab()` will look in the argument to `data`, if specified. The distance measure generated by `matchthem()` or `weightthem()` is automatically included and named "distance" or "prop.score", respectively.
#' @param s.d.denom `character`; how the denominator for standardized mean differences should be calculated, if requested. See [col_w_smd()] for allowable options. Abbreviations allowed. If not specified, the defaults depend on the options specified in the original function calls; see [bal.tab.matchit()] and [bal.tab.weightit()] for details on the defaults.
#' 
#' @returns
#' If clusters are not specified, an object of class `"bal.tab.imp"` containing balance summaries for each imputation and a summary of balance across imputations. See [`class-bal.tab.imp`] for details.
#' 
#' If clusters are specified, an object of class `"bal.tab.imp.cluster"` containing summaries between and across all clusters and imputations.
#' 
#' @details
#' `bal.tab.mimids()` and `bal.tab.wimids()` generate a list of balance summaries for the `mimids` or `wimids` object given.
#' 
#' @seealso
#' * [bal.tab()] for details of calculations
#' * [bal.tab.matchit()] and [bal.tab.weightit()]
#' 
#' @examplesIf all(sapply(c("mice", "MatchThem", "MatchIt", "WeightIt"), requireNamespace, quietly = TRUE))
#' library(mice)
#' library(MatchThem)
#' 
#' data("lalonde_mis", package = "cobalt")
#' 
#' #Imputing the missing data
#' imp <- mice(lalonde_mis, m = 5)
#' 
#' #Matching using within-imputation propensity scores
#' mt.out1 <- matchthem(treat ~ age + educ + race + 
#'                          married + nodegree + re74 + re75, 
#'                      data = imp, approach = "within")
#' bal.tab(mt.out1)
#' 
#' #Matching using across-imputation average propensity scores
#' mt.out2 <- matchthem(treat ~ age + educ + race + 
#'                          married + nodegree + re74 + re75, 
#'                      data = imp, approach = "across")
#' 
#' bal.tab(mt.out2)
#' 
#' #Weighting using within-imputation propensity scores
#' wt.out <- weightthem(treat ~ age + educ + race + 
#'                          married + nodegree + re74 + re75, 
#'                      data = imp, approach = "within",
#'                      estimand = "ATT")
#' 
#' bal.tab(wt.out)

#' @exportS3Method bal.tab mimids
bal.tab.mimids <-     function(x,
                               stats, int = FALSE, poly = 1, distance = NULL, addl = NULL, data = NULL, continuous, binary, s.d.denom, thresholds = NULL, weights = NULL, cluster = NULL, pairwise = TRUE, s.weights = NULL, abs = FALSE, subset = NULL, quick = TRUE,
                               ...) {
    
    tryCatch(args <- c(as.list(environment()), list(...))[-1], error = function(e) .err(conditionMessage(e)))
    
    #Adjustments to arguments
    
    args[vapply(args, rlang::is_missing, logical(1L))] <- NULL
    
    #Initializing variables
    X <- do.call("x2base", c(list(x), args), quote = TRUE)
    
    args[names(args) %in% names(X)] <- NULL
    
    X <- .assign_X_class(X)
    
    do.call("base.bal.tab", c(list(X), args),
            quote = TRUE)
}

#' @exportS3Method bal.tab wimids
bal.tab.wimids <- bal.tab.mimids
