\name{NPMLE.Plackett}
\alias{NPMLE.Plackett}
%- Also NEED an '\alias' for EACH other topic documented here.

\title{Semiparametric Inference for Copula Models with Dependent Truncation Data}

\description{Nonparametric maximum likelihood estimation for dependent truncation data under 
the Plackett copula models (Emura & Wang, 2012). The forms of the marginal distributions for X and Y are completely unspecified, but
the parametric form of copula is specified as the Plackett copula. 
}

\usage{
NPMLE.Plackett(x.trunc, y.trunc,
x.fix = median(x.trunc), y.fix = median(y.trunc), plotX = TRUE)
}
%- maybe also 'usage' for other objects documented here.

\arguments{
  \item{x.trunc}{vector of variables satisfying x.trunc<=y.trunc}
  \item{y.trunc}{vector of variables satisfying x.trunc<=y.trunc}
  \item{x.fix}{vector of fixed points at which marginal distribution function of X is calculated}
  \item{y.fix}{vector of fixed points at which marginal survival function of Y are calculated}
  \item{plotX}{if TRUE, the culumative distribution function for X is plotted}
}


\details{
The function produces the nonparametric maximum likelihood estimate (NPMLE) for the marginal distributions
 and the estimate of the association parameter under the Plackett copula model. The method follows Emura & Wang (2012).
The maximization of the likelihood (L) is conducted by minimizing -logL by "nlm". Standard errors are calculated
from the diagonal elements in the observed Fisher information matrix, which is obtained from the output of the "nlm".  
The method can handle ties but cannot handle right-censoring.
}

\value{
\item{alpha }{estimate of association parameter}
\item{alpha_se }{standard error of the estimate of association parameter}
\item{Hx }{estimate of the marginal cumulative reverse-hazard function for X}
\item{Hx_se }{standard error of the estimate of the marginal cumulative reverse-hazard function for X}
\item{Ay }{estimate of the marginal cumulative hazard function for Y}
\item{Ay_se }{standard error of the estimate of the marginal cumulative hazard function for Y}
\item{Fx }{estimate of the marginal distribution function for X}
\item{Fx_se }{standard error of the estimate of the marginal distribution function for X}
\item{Sy }{estimate of the marginal survival function for Y}
\item{Sy_se }{standard error of the estimate of the marginal survival function for Y}
\item{conv }{an integer indicating why the optimization process terminated in "nlm": If conv=1,
the maximization of the likelihood function is properly done. Please refer R function "nlm"
 for more details.}
\item{iteration}{the number of iterations until convergence}
\item{Grad}{L_2 norm for the gradient vector at the solution (close to zero if the solution is proper)}
\item{MinEigen}{Minimum eigenvalue of the Hessian matrix at the solution (positive if the solution is proper)}
}


\references{
Emura T, Wang W (2012)  Nonparametric maximum likelihood estimation for dependent truncation data based on copulas, Journal of Multivariate Analysis 110, 171-88
}
\author{Takeshi EMURA}


\examples{
##### Generate truncated data from Plackett copula with unit exponential marginals #####
set.seed(1)
m=25 #### sample size ####
alpha_true=1/5.11 #### Kendall's tau on (X,Y) = 0.5 ####
x.trunc=y.trunc=numeric(m)
l=1
while(l<=m){
  u=runif(1,min=0,max=1)
  v=runif(1,min=0,max=1)
  K=function(w){
    A=sqrt( (1+(alpha_true-1)*(u+w))^2-4*alpha_true*(alpha_true-1)*u*w )  
    2*v+(  1+(alpha_true-1)*u-(alpha_true+1)*w  )/A-1
  }
  low=0.00001*u
  up=1-0.00001*v
  for(j in 1:100){ #### Invert K(.) by the bisection method to ####
    mid=(low+up)/2;M<-K(mid);U<-K(up)
    if(M*U>0){up<-mid}else{low<-mid}
  }
  w=mid
  x=-log(1-u);y=-log(w)
  if((x<y)){
  x.trunc[l]=x;y.trunc[l]=y
  l=l+1
  }
}

NPMLE.Plackett(x.trunc, y.trunc, x.fix =1, y.fix = 1, plotX = TRUE)

#### compare the above estimate with the true value #####
c(alpha_true=alpha_true,Fx_true=1-exp(-1),Sy_true=exp(-1))
}

\keyword{ Copula }
\keyword{ Semiparametric method }% __ONLY ONE__ keyword per line
