% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/ergm.allstats.R
\name{ergm.allstats}
\alias{ergm.allstats}
\title{Calculate all possible vectors of statistics on a network for an ERGM}
\usage{
ergm.allstats(
  formula,
  zeroobs = TRUE,
  force = FALSE,
  maxNumChangeStatVectors = 2^16,
  ...
)
}
\arguments{
\item{formula}{an \code{\link{formula}} object of the form \code{y ~ <model
terms>}, where \code{y} is a network object or a matrix that can be coerced
to a \code{\link[network]{network}} object.  For the details on the possible
\code{<model terms>}, see \code{\link{ergmTerm}}.  To create a
\code{\link[network]{network}} object in , use the \code{network()}
function, then add nodal attributes to it using the \code{\%v\%} operator if
necessary.}

\item{zeroobs}{Logical: Should the vectors be centered so that the network
passed in the \code{formula} has the zero vector as its statistics?}

\item{force}{Logical: Should the algorithm be run even if it is determined
that the problem may be very large, thus bypassing the warning message that
normally terminates the function in such cases?}

\item{maxNumChangeStatVectors}{Maximum possible number of distinct values of
the vector of statistics.  It's good to use a power of 2 for this.}

\item{\dots}{further arguments; not currently used.}
}
\value{
Returns a list object with these two elements:
\item{weights}{integer of counts, one for each row of \code{statmat} telling
how many networks share the corresponding vector of statistics.}
\item{statmat}{matrix in which each row is a unique vector of statistics.}
}
\description{
\code{ergm.allstats} produces a matrix of network statistics for an
arbitrary \code{statnet} exponential-family random graph model.  One
possible use for this function is to calculate the exact loglikelihood
function for a small network via the \code{\link{ergm.exact}} function.
}
\details{
The mechanism for doing this is a recursive algorithm, where the number of
levels of recursion is equal to the number of possible dyads that can be
changed from 0 to 1 and back again.  The algorithm starts with the network
passed in \code{formula}, then recursively toggles each edge twice so that
every possible network is visited.

\code{ergm.allstats} should only be used for small networks, since the
number of possible networks grows extremely fast with the number of nodes.
An error results if it is used on a directed network of more than 6 nodes or
an undirected network of more than 8 nodes; use \code{force=TRUE} to
override this error.
}
\examples{

# Count by brute force all the edge statistics possible for a 7-node 
# undirected network
mynw <- network(matrix(0,7,7),dir=FALSE)
system.time(a <- ergm.allstats(mynw~edges))

# Summarize results
rbind(t(a$statmat),a$weights)

# Each value of a$weights is equal to 21-choose-k, 
# where k is the corresponding statistic (and 21 is 
# the number of dyads in an 7-node undirected network).  
# Here's a check of that fact:
as.vector(a$weights - choose(21, t(a$statmat)))

# Simple ergm.exact outpuf for this network.  
# We know that the loglikelihood for my empty 7-node network
# should simply be -21*log(1+exp(eta)), so we may check that
# the following two values agree:
-21*log(1+exp(.1234)) 
ergm.exact(.1234, mynw~edges, statmat=a$statmat, weights=a$weights)

}
\seealso{
\code{\link{ergm.exact}}
}
\keyword{models}
