#' Create an ezsim object from 4 important arguments(dgp,estimator,true,parameter_def). 
#' @name ezsim
#' @aliases ezsim dgp
#' @title Create an ezsim Object.
#' @param m The number of times you want to simulate.
#' @param estimator A function takes the return value of dgp as argument and return estimates of the dgp
#' @param dgp A function defines the data generating process(dgp). It returns an object (usually it is a \code{data.frame}) which can be used as argument of \code{estimator}. It will be evaluated under an environment generated by a set of parameter generated by parameter_def.
#' @param parameter_def A parameter_def object will be used in this simulation.
#' @param true_value A function defines the true value of estimators(TV). Similar to dgp, but it returns the true value of estimates. It is necessary for computing the bias and rmse of the estimator. The length of its return value must be the same as the lenght of \code{estimator}. If it is missing, True Value, Bias and rmse will be NA in the summary of ezsim.
#' @param run Whether the simulation will be ran right after the creation.
#' @param core numeric. How many core will be used in the simulation. The parallel computing is implmented by foreach and doSNOW.
#' @param display_name Display name for the name of parameter and estimator. see \code{\link{plotmath}} for details.
#' @param packages character vector of packages that the estiamtor depend on. Ignored when core=1. See \code{\link{foreach}} for details.
#' @return An ezsim object.
#' @author TszKin Julian Chan \email{ctszkin@@gmail.com}
#' @seealso \code{\link{createParDef}} \code{\link{setOthers}},\code{\link{setScalars}} \code{\link{summary.ezsim}}
#' @export
#' @examples         
#' \dontrun{
#' ## Example 1
#' ezsim_basic<-ezsim(
#'     m             = 100,
#'     run           = TRUE,
#'     core          = 1,
#'     display_name  = c(mean_hat="hat(mu)",sd_mean_hat="hat(sigma[hat(mu)])"),
#'     parameter_def = createParDef(list(n=seq(20,80,20),mu=c(0,2),sigma=c(1,3,5))),
#'     dgp           = function() rnorm(n,mu,sigma),
#'     estimator     = function(x) c(mean_hat = mean(x), 
#'                                  sd_mean_hat=sd(x)/sqrt(length(x)-1)),
#'     true_value    = function() c(mu, sigma / sqrt(n-1))
#' )
#' 
#' ## Test whether an ezsim object is valid. Print the result of the test and dont return the name of estimator.
#' test(ezsim_basic,print_result=TRUE,return_name=FALSE)
#' 
#' ## Subset of an ezsim object.
#' subset(ezsim_basic,subset=list(estimator='mean_hat',mu=0,n=c(20,40)))$sim
#' 
#' ## Summary of an ezsim object
#' summary(ezsim_basic)
#' 
#' ## Summary of a subset of ezsim object
#' summary(ezsim_basic,subset=list(estimator='mean_hat',n=c(20,40),sigma=c(1,3)))
#' 
#' ## More Summary Statistics
#' summary(ezsim_basic,simple=FALSE,subset=list(estimator='mean_hat',n=c(20,40),sigma=c(1,3)))
#' 
#' ## Customize the Summary Statistics
#' summary(ezsim_basic,stat=c("q25","median","q75"),Q025=quantile(value_of_estimator,0.025),Q975=quantile(value_of_estimator,0.975),subset=list(estimator='mean_hat',n=c(20,40),sigma=c(1,3)))
#' 
#' ## Plot an ezsim object
#' plot(ezsim_basic)
#' ## Subet of the Plot
#' plot(ezsim_basic,subset=list(estimator="sd_mean_hat",mu=0))
#' plot(ezsim_basic,subset=list(estimator="mean_hat",sigma=3))
#' ## Parameters Priority of the Plot
#' plot(ezsim_basic,subset=list(estimator="sd_mean_hat",mu=0),parameters_priority=c("sigma","n"))
#' plot(ezsim_basic,subset=list(estimator="mean_hat",sigma=c(1,3)),parameters_priority="mu")
#' 
#' ## Density Plot
#' plot(ezsim_basic,'density')
#' plot(ezsim_basic,"density",subset=list(estimator="mean_hat",sigma=3),parameters_priority="n",benchmark=dnorm)
#' plot(ezsim_basic,"density",subset=list(estimator="mean_hat",mu=0),parameters_priority="n" ,benchmark=dnorm)
#' 
#' ## Plot the summary ezsim
#' plot(summary(ezsim_basic,c("q25","q75")))
#' plot(summary(ezsim_basic,c("q25","q75"),subset=list(estimator='mean_hat')))
#' plot(summary(ezsim_basic,c("median"),subset=list(estimator='sd_mean_hat')))
#' 
#' ## Example 2
#' ezsim_ols<-ezsim(
#'     m             = 100,    
#'     run           = TRUE,
#'     core          = 1,
#'     display_name  = c(beta_hat='hat(beta)',es='sigma[e]^2',xs='sigma[x]^2',sd_beta_hat='hat(sigma)[hat(beta)]'),
#'     parameter_def = createParDef(scalars=list(xs=c(1,3),beta=c(0,2),n=seq(20,80,20),es=c(1,3))),
#'     dgp           = function(){
#'                         x<-rnorm(n,0,xs)
#'                         e<-rnorm(n,0,es)
#'                         y<-beta * x + e
#'                         data.frame(y,x)
#'                     },
#'     estimator     = function(d){
#'                         r<-summary(lm(y~x-1,data=d))
#'                         out<-r$coef[1,1:2]
#'                         names(out)<-c('beta_hat','sd_beta_hat')
#'                         out
#'                     },
#'     true_value    = function() c(beta, es/sqrt(n)/xs) 
#' )
#' summary(ezsim_ols)
#' plot(ezsim_ols)
#' plot(ezsim_ols,subset=list(beta=0))
#' 
#' plot(ezsim_ols,'density')
#' plot(ezsim_ols,'density',subset=list(es=1,xs=1))
#' 
#' 
#' ## example 3
#' ezsim_powerfun<-ezsim(
#'     run           = TRUE,   
#'     m             = 100,
#'     parameter_def = createParDef(scalars=list(xs=1,n=50,es=c(1,5),b=seq(-1,1,0.1))),
#'     display_name  = c(b='beta',es='sigma[e]^2',xs='sigma[x]^2'),
#'     dgp           = function(){
#'                         x<-rnorm(n,0,xs)
#'                         e<-rnorm(n,0,es)
#'                         y<-b * x + e
#'                         data.frame(y,x)
#'                     },
#'     estimator     = function(d){
#'                         r<-summary(lm(y~x-1,data=d))
#'                         stat<-r$coef[,1]/r$coef[,2]
#' 
#'                         # test whether b > 0
#'                         # level of significance : 5%
#'                         out <- stat > c(qnorm(.95), qt(0.95,df=r$df[2]))
#'                         names(out)<-c("z-test","t-test")
#'                         out
#'                     }
#' )
#' plot(ezsim_powerfun,'powerfun')
#' }

ezsim <-
function(m,estimator,dgp,parameter_def,true_value=NA,run=TRUE,core=1,display_name=NULL,packages=NULL){
    out<-list(m=m,estimator=estimator,true_value=true_value,dgp=dgp,parameter_def=parameter_def,core=core,display_name=display_name,packages=packages)
    class(out)<-"ezsim"
    i<-NULL
    
    ## Generate parameter
    out$scalar_parameters<- names(parameter_def$scalars)
    out$parameter_list<-generate(parameter_def)
    
    ## Test
    name_of_estimator<-test(out)
    
    ## Generate true value table
    out$TV_table<-
    foreach (i = out$parameter_list,.combine=rbind,.final=data.frame) %do% {
        true_value<-
            if(class(out$true_value)!='function')
                rep(NA,length(name_of_estimator))
            else
                run(out$true_value,i)
        scalars_par<-unlist(i[out$scalar_parameters])
        names(true_value)<-name_of_estimator
        t(c(scalars_par,true_value))
    }
    rownames(out$TV_table)<-NULL
    out$TV_table<-melt(out$TV_table,id.vars=unlist(out$scalar_parameters),variable_name="estimator")

    ## Run simulation
    if (run)
        run(out,core=core)
        
    return(out)
}

#' Run the Simulation of an ezsim object. The simulation result is store into the ezsim object in the argument directly, reassignment is not needed.
#' @name run.ezsim
#' @aliases run.ezsim
#' @title Run the Simulation
#' @method run ezsim
#' @param x An ezsim object
#' @param core Number of core to be used in parallel computing. if missing, the value in ezsim object will be used.
#' @param \dots unused
#' @author TszKin Julian Chan \email{ctszkin@@gmail.com}
#' @S3method run ezsim
#' @examples              
#' \dontrun{
#' ezsim_basic<-ezsim(
#'     m             = 100,
#'     run           = FALSE,
#'     display_name  = c(mean_hat="hat(mu)",sd_mean_hat="hat(sigma[hat(mu)])"),
#'     parameter_def = createParDef(list(n=seq(20,80,20),mu=c(0,2),sigma=c(1,3,5))),
#'     dgp           = function() rnorm(n,mu,sigma),
#'     estimator     = function(x) c(mean_hat = mean(x), 
#'                                  sd_mean_hat=sd(x)/sqrt(length(x)-1)),
#'     true_value    = function() c(mu, sigma / sqrt(n-1))
#' )
#' run(ezsim_basic)
#' }

run.ezsim <-
function(x,core,...){
    xx<-x
    if (missing(core))
        core <- xx$core
    counter<-1
    i<-NULL

    time_used<-system.time({
    if (core==1){
        pb<-txtProgressBar(min = 0, max = length(xx$parameter_list)*xx$m, style = 3)
        xx$sim<-
        foreach (i = x$parameter_list,.combine=rbind) %do% {
            temp<-as.data.frame(
            foreach (j = 1:x$m,.combine=rbind) %do% {
                setTxtProgressBar(pb,counter<-counter+1 )    
                x$estimator(run(x$dgp,i))
            })
            add2DataFrame(temp,unlist(i[x$scalar_parameters]))
        }
        close(pb)
    } else {
        numb_loop<-length(x$parameter_list)
        pb<-txtProgressBar(min = 0, max = numb_loop, style = 3)
        cl<-makeCluster(core) 
        registerDoSNOW(cl)
        if (is.null(x$packages))
            packages<-'ezsim'
        else    
            packages<-c('ezsim',x$packages)
        xx$sim<-
        foreach (i = x$parameter_list,.combine=rbind) %do% {
            setTxtProgressBar(pb,counter<-counter+1 )
            temp<-as.data.frame(
            foreach (j = 1:x$m,.combine=rbind,.packages=packages) %dopar% {
                x$estimator(ezsim:::run.function(x$dgp,i))
            })
            add2DataFrame(temp,unlist(i[x$scalar_parameters]))
        }    
        close(pb)
        stopCluster(cl)
    }
    })
    
    ## melt down the estimator
    id_vars<-xx$scalar_parameters
    xx$sim<-melt(xx$sim,id.vars=id_vars,variable_name='estimator')

    ## merge the true value
    xx$sim<-merge(xx$sim,xx$TV_table,by=c(id_vars,'estimator'),suffixes = c("_of_estimator","_of_TV"),all.x=TRUE)

    print(time_used)
    cat("DONE!\n")
    rownames(xx$sim)<-NULL
    eval.parent(substitute(x<-xx))
}

#' For each set of parameters, the simulation is ran once to obtain the value of estimator and true value to make sure everything in ezsim is properly defined. The test results will be shown in the console. The test will be ran automatically when you create an ezsim object.
#' @name test.ezsim
#' @aliases test.ezsim
#' @title Perform a Test for an ezsim Object
#' @method test ezsim
#' @param x An ezsim Object
#' @param return_name Whehter to return the name of estimator
#' @param print_result Whehter to print the return
#' @param \dots unused
#' @return Optional: names of estimator.
#' @author TszKin Julian Chan \email{ctszkin@@gmail.com}
#' @seealso \code{\link{ezsim}}
#' @S3method test ezsim
#' @examples          
#' ezsim_basic<-ezsim(
#'     m             = 100,
#'     run           = FALSE,
#'     display_name  = c(mean_hat="hat(mu)",sd_mean_hat="hat(sigma[hat(mu)])"),
#'     parameter_def = createParDef(list(n=seq(20,80,20),mu=c(0,2),sigma=c(1,3,5))),
#'     dgp           = function() rnorm(n,mu,sigma),
#'     estimator     = function(x) c(mean_hat = mean(x), 
#'                                  sd_mean_hat=sd(x)/sqrt(length(x)-1)),
#'     true_value    = function() c(mu, sigma / sqrt(n-1))
#' )
#' 
#' test(ezsim_basic,print_result=TRUE)

test.ezsim <-
function(x,return_name=TRUE,print_result=FALSE,...){
    cat("Running Test...\n")
    name_of_estimator<-''
    i<-NULL
    temp<-
    foreach (i = x$parameter_list,.combine=rbind) %do% {
        my_estimator<-x$estimator(run(x$dgp,i))
        name_of_estimator<-names(my_estimator)
        my_TV<-
            if(class(x$true_value)!='function')
                rep(NA,length(name_of_estimator))
            else
                run(x$true_value,i)
        if (length(my_estimator)!=length(my_TV))
            stop('length of estimator and true value are not the same!')
        names(my_TV)<-paste('TV_of_',names(my_estimator),sep='')
        c(unlist(i[x$scalar_parameters]),my_estimator,my_TV)
    }
    cat("Done!\n")

    rownames(temp)<-NULL
    if (print_result) {
        cat("Sample estimators and it's true value:\n")
        print(temp)
        cat('\n')
    }
    if (return_name)
        return(names(my_estimator))
}


#' Print an ezsim Object. See \code{\link{ezsim}} for details.
#' @name print.ezsim
#' @aliases print.ezsim
#' @title Print an ezsim Object.
#' @method print ezsim
#' @param x An ezsim Object
#' @param \dots unused
#' @author TszKin Julian Chan \email{ctszkin@@gmail.com}
#' @seealso \code{\link{ezsim}}
#' @S3method print ezsim 
print.ezsim <-
function(x,...){
    cat("Number of Simulation :",x$m ,"\n")
    cat("\n")
    cat("Estimator : \n")
    print(x$estimator)
    cat("\n")
    cat("dgp : \n")
    print(x$dgp)
    cat("\n")
    if (is.function(x$true_value)){
        cat("True value of estimator : \n")
        print(x$true_value)
        cat("\n")
    }
    print(x$parameter_def)
}

#' Return a subset of the simulation result of an ezsim object.
#' @name subset.ezsim
#' @aliases subset.ezsim
#' @title Return of the Simulation 
#' @method subset ezsim
#' @param x An ezsim Object
#' @param subset A list contains the subset of estimators and parameters. To select a subset of estimators: \code{list(estimator=c('name of estimator1','name of estimator2'))}. To select a subset of parameters: \code{list(mean=1:3,sd=4:5)}. Or both.
#' @param \dots unused
#' @return sim of ezsim
#' @author TszKin Julian Chan \email{ctszkin@@gmail.com}
#' @seealso \code{\link{ezsim}}
#' @note For internal use of ezsim.
#' @S3method subset ezsim
#' @examples        
#' \dontrun{
#' ezsim_basic<-ezsim(
#'     m             = 100,
#'     run           = TRUE,
#'     display_name  = c(mean_hat="hat(mu)",sd_mean_hat="hat(sigma[hat(mu)])"),
#'     parameter_def = createParDef(list(n=seq(20,80,20),mu=c(0,2),sigma=c(1,3,5))),
#'     dgp           = function() rnorm(n,mu,sigma),
#'     estimator     = function(x) c(mean_hat = mean(x), 
#'                                  sd_mean_hat=sd(x)/sqrt(length(x)-1)),
#'     true_value    = function() c(mu, sigma / sqrt(n-1))
#' )
#' subset(ezsim_basic,subset=list(estimator='mean_hat',mu=0,n=c(20,40)))$sim
#' }
  
subset.ezsim<-
function(x,subset,...){
    if (is.null(x$sim))
        stop('Please run the simulation first.')

    if (!missing(subset)){
        x$sim<-
        tryCatch({
            if (length(subset)==0)
                stop('length of subset is zero \n')
            if (any(names(subset) %in% c('value_of_TV','value_of_estimator')))
                stop('subset cant contain  \'value_of_TV\' and \'value_of_estimator\' \n')
            if (!all(names(subset) %in% names(x$sim) ))
                stop('Unknown name: ', names(subset)[!names(subset) %in% names(x$sim)] ,'\n' )
                
            index<-apply(mapply(function(name,value) as.matrix(x$sim[[name]] %in% value)  , name=names(subset) , value=subset),1, all)
            
            subset(x$sim,subset=index)
        },
        error = function(e) {
            print(e)
            cat('Error in subseting. Nothing is dropped out. \n')
            x$sim
        })
    }
    x
}

#' There are 3 different modes to plot an ezsim object. \code{'summary'}, \code{'density'} and \code{'powerfun'} plot the summary statistics,density function and power function of an ezsim object respectively.\cr\cr
#' \code{'summary'}: The y-variable of the plot are summary statistics of the estimator. Two confidence bounds will be shaded in the plot. 25\% and 75\% percentile will form a 50\% confidence bound. Similarly, 2.5\% and 97.5\% percentile will form a 95\% confidence bound.  Each plot have only one estimator. The scalars parameter has the longest length will be the x-variable of the plot. The rest of the scalars parameters will be become the facets of the plot (see \pkg{ggplot2}). \cr\cr \code{density} : Density plot of the estimator. Each plot have only one estimator. scalars parameter will appear as different colour and in different facets.\cr\cr \code{powerfun} : Plot the power function of test(s). Estimators have to be a test (value = 1 if rejecting the null hypothesis, value = 0 if fail to reject the null hypothesis) others parameters will not be shown in the graph.
#' @name plot.ezsim
#' @aliases plot.ezsim
#' @title Plot an ezsim Object
#' @method plot ezsim
#' @param x An ezsim object
#' @param type Type of plot
#' @param subset subset of estimators or parameters. See \code{\link{subset.ezsim}} for details.
#' @param parameters_priority Display priority of parameter. If any parameter is missing here, they will be sorted by length.
#' @param return_print If TRUE, return a list of ggplot2 object. If FALSE(default), all of the plot will be printed out.
#' @param ylab Label of y-axis
#' @param title Title of the plot
#' @param pdf_option A list of option pass to \code{\link{pdf}}. If it is not missing, the plot will export to a pdf file
#' @param null_hypothesis Null hypothesis of the test. For \code{type=='powerfun'} only.
#' @param benchmark Benchmark distribution. For \code{type=='density'} only.
#' @param \dots unused
#' @return Optional: a list of ggplot2 object
#' @author TszKin Julian Chan \email{ctszkin@@gmail.com}
#' @S3method plot ezsim
#' @seealso \code{\link{ezsim}},\code{\link{summary.ezsim}}, \code{\link{plot.summary.ezsim}},
#' @keywords post-simulation
#' @examples       
#' \dontrun{
#' ## example 1
#' ezsim_basic<-ezsim(
#'     m             = 100,
#'     run           = TRUE,
#'     core          = 1,
#'     display_name  = c(mean_hat="hat(mu)",sd_mean_hat="hat(sigma[hat(mu)])"),
#'     parameter_def = createParDef(list(n=seq(20,80,20),mu=c(0,2),sigma=c(1,3,5))),
#'     dgp           = function() rnorm(n,mu,sigma),
#'     estimator     = function(x) c(mean_hat = mean(x), 
#'                                  sd_mean_hat=sd(x)/sqrt(length(x)-1)),
#'     true_value    = function() c(mu, sigma / sqrt(n-1))
#' )
#' ## Plot an ezsim object
#' plot(ezsim_basic)
#' ## Subet of the Plot
#' plot(ezsim_basic,subset=list(estimator="sd_mean_hat",mu=0))
#' plot(ezsim_basic,subset=list(estimator="mean_hat",sigma=3))
#' ## Parameters Priority of the Plot
#' plot(ezsim_basic,subset=list(estimator="sd_mean_hat",mu=0),parameters_priority=c("sigma","n"))
#' plot(ezsim_basic,subset=list(estimator="mean_hat",sigma=c(1,3)),parameters_priority="mu")
#' 
#' ## Density Plot
#' plot(ezsim_basic,'density')
#' plot(ezsim_basic,"density",subset=list(estimator="mean_hat",sigma=3),parameters_priority="n",benchmark=dnorm)
#' plot(ezsim_basic,"density",subset=list(estimator="mean_hat",mu=0),parameters_priority="n" ,benchmark=dnorm)
#'  
#' ## example 2
#' ezsim_ols<-ezsim(
#'     m             = 100,    
#'     run           = TRUE,
#'     core          = 1,
#'     display_name  = c(beta_hat='hat(beta)',es='sigma[e]^2',xs='sigma[x]^2',sd_beta_hat='hat(sigma)[hat(beta)]'),
#'     parameter_def = createParDef(scalars=list(xs=c(1,3),beta=c(0,2),n=seq(20,80,20),es=c(1,3))),
#'     dgp           = function(){
#'                         x<-rnorm(n,0,xs)
#'                         e<-rnorm(n,0,es)
#'                         y<-beta * x + e
#'                         data.frame(y,x)
#'                     },
#'     estimator     = function(d){
#'                         r<-summary(lm(y~x-1,data=d))
#'                         out<-r$coef[1,1:2]
#'                         names(out)<-c('beta_hat','sd_beta_hat')
#'                         out
#'                     },
#'     true_value    = function() c(beta, es/sqrt(n)/xs) 
#' )
#' plot(ezsim_ols)
#' plot(ezsim_ols,subset=list(beta=0))
#' 
#' plot(ezsim_ols,'density')
#' plot(ezsim_ols,'density',subset=list(es=1,xs=1))
#' 
#'  
#' ## example 3
#' ezsim_powerfun<-ezsim(
#'     run           = TRUE,   
#'     m             = 100,
#'     parameter_def = createParDef(scalars=list(xs=1,n=50,es=c(1,5),b=seq(-1,1,0.1))),
#'     display_name  = c(b='beta',es='sigma[e]^2',xs='sigma[x]^2'),
#'     dgp           = function(){
#'                         x<-rnorm(n,0,xs)
#'                         e<-rnorm(n,0,es)
#'                         y<-b * x + e
#'                         data.frame(y,x)
#'                     },
#'     estimator     = function(d){
#'                         r<-summary(lm(y~x-1,data=d))
#'                         stat<-r$coef[,1]/r$coef[,2]
#' 
#'                         # test whether b > 0
#'                         # level of significance : 5%
#'                         out <- stat > c(qnorm(.95), qt(0.95,df=r$df[2]))
#'                         names(out)<-c("z-test","t-test")
#'                         out
#'                     }
#' )
#' plot(ezsim_powerfun,'powerfun') plot(ezsim_basic,'powerfun')
#' }
plot.ezsim <-
function(x,type=c('summary','density','powerfun' ),subset,parameters_priority,return_print=FALSE,ylab,title,pdf_option,null_hypothesis,benchmark,...){

    if (is.null(x$sim))
        stop('Please run the simulation first.')

    type=match.arg(type)
    out<-NULL
    i<-j<-Q025<-Q975<-Q25<-Q75<-TV<-Mean<-Median<-value_of_estimator<-NULL
    
    # scalars_name<-getScalarsName(x)

    ##Summmary 
    if (type=='summary'){
        if (missing(ylab))
            ylab<-'Summary Statistics'
        
        ## Compute summary statistics
        summ<-summary(x,c('mean','q25','q75','tv','median'), Q025=quantile(value_of_estimator,0.025),simple=FALSE, Q975=quantile(value_of_estimator,0.975),subset=subset )
        
        temp<-getScalarsName(summ,parameters_priority=parameters_priority)
                
        if (missing(title))
            title=''
        subtitle<-temp$subtitle
            
        x_var=head(temp$scalar_length_greater_one,1)
        other=tail(temp$scalar_length_greater_one,-1)    
        
        #########
              
        my_facet<-facet_grid(createFormula(other), labeller = label_both_parsed_recode(x$display_name))

        out<-dlply(summ,'estimator', 
            function(mydata) {
                mytitle<-''
                if (title=='')
                    mytitle<-paste('Summary','of',as.character(mydata$estimator),sep='~~')
                else
                    mytitle<-title
                if (subtitle!='')
                    mytitle<- paste(mytitle,subtitle,sep='~~')

                ggplot(aes_string(x=x_var),data=mydata)+

                geom_ribbon(aes(ymin=Q025,ymax=Q975,alpha=.1))+
                geom_ribbon(aes(ymin=Q25,ymax=Q75,alpha=.3),)+
                scale_alpha_continuous(name='Confidence Bound',to=c(.1,.3),breaks=c(0.1,.3),label=c('95%','50%'),limit=c(.1,.3))+
                
                geom_line(aes(y=TV,color='True Value'))+
                geom_line(aes(y=Mean,color='Mean'))+
                geom_line(aes(y=Median,color='Median'))+

                geom_point(aes(y=TV,color='True Value'))+
                geom_point(aes(y=Mean,color='Mean'))+
                geom_point(aes(y=Median,color='Median'))+
                
                scale_colour_manual(name='Summary Statistics',value=c('red','black','blue'))+
                my_facet + ylab(ylab)+xlab(parse(text=recode(x_var,x$display_name)))+
                opts(legend.position='bottom', legend.direction='horizontal',title=parse(text=mytitle))      
            }
        )
    }
    
    ## powerfun
    if (type=='powerfun'){
        if (missing(ylab))
            ylab<-'Probability of Rejecting Null Hypothesis'
            summary_x<- summary(x,stat='mean',subset=subset,simple=FALSE)


        mytitle<-
        if (missing(title))
            'Power~~Function'
        else
            title          

        out<-plot(summary_x,parameters_priority=parameters_priority,return_print=TRUE,ylab=ylab ,title=mytitle)
        
       
        if (!missing(null_hypothesis)){
            out<-out+
            eval(substitute(geom_vline(xintercept=null),list(null=null_hypothesis)))
        }
    }
    
    ## density
    if (type=='density'){
        if (missing(ylab))
            ylab<-'Density'
        
        x$sim<-subset.ezsim(x,subset=subset)$sim
        
        scalars_name <- getScalarsName(x,parameters_priority=parameters_priority)
 
        x_var=head(scalars_name$scalar_length_greater_one,1)
        other=tail(scalars_name$scalar_length_greater_one,-1)    

        # change name of estimator to display name
        x$sim$estimator<-factor(recode(as.character(x$sim$estimator),x$display_name))
        
        my_facet<-facet_grid(createFormula(other,right=FALSE), labeller = label_both_parsed_recode(x$display_name))
        
        
        
        if (!missing(benchmark) ){
            benchmark<-
            if (length(benchmark)!= length(unique(x$sim$estimator)))
                replicate(length(unique(x$sim$estimator)),benchmark)
            else
                list(benchmark)
        }    
            
        out<-
        foreach ( i = unique(x$sim$estimator), j=1:length(unique(x$sim$estimator)) ) %do% {
            temp<-subset.data.frame(x$sim,subset=x$sim$estimator==i)
            temp[[x_var]]<-factor(temp[[x_var]])
                                
            mytitle<-as.character(i)
            
            if (missing(title))
                mytitle<- paste('Density~~of',mytitle,sep='~~')
            else
                mytitle<- paste(title,'Density~~of',mytitle,sep='~~')
                
            if (scalars_name$subtitle!='')
                mytitle<-paste(mytitle,scalars_name$subtitle,sep='~~')

            temp_out<-
            ggplot(data=temp)+
            geom_density(aes_string(x='value_of_estimator',color=x_var,fill=x_var),alpha=0.1) + 
            scale_color_discrete(legend=FALSE) + 
            scale_fill_discrete(name=parse(text=x_var)) +
            my_facet + 
            ylab(ylab) + 
            xlab(parse(text=as.character(i))) +
            opts(legend.position='bottom', legend.direction='horizontal',title=parse(text=mytitle))
          
            
            
            if (!missing(benchmark)){
                temp_out<-temp_out+
                stat_function(fun = dnorm, colour="black",aes_string(x='value_of_estimator')) 
            }
            temp_out
        }
    }
        
    if (!missing(pdf_option)){
        do.call(pdf,pdf_option)
        if (class(out)=='ggplot')
            print(out)
        else
            lapply(out,print )
        dev.off()
    } 
    if (return_print){
        return(out)
    } else{
        if (class(out)=='ggplot')
            print(out)
        else
            lapply(out,function(x) {x11(); print(x)} )
        
    }
}
#' A quick summary to the simulation. Summary statistics included mean, true value (tv), bias, bias percentage (mean/tv-1), sd, rmse (root mean square error), min, q25 (first quarter), median, q75 (third quarter), max, p value of jb-test. See \code{\link{ezsim}} and \code{\link{plot.summary.ezsim}} for details and examples.
#' @name summary.ezsim
#' @aliases summary.ezsim
#' @title Summarize an ezsim Object
#' @usage  
#' \method{summary}{ezsim}(object,stat=c('mean','tv','bias',
#' 'biaspercentage','sd','rmse','min','q25','median',
#' 'q75','max','jb_test'),simple=TRUE,subset,...)
#' @method summary ezsim
#' @param object An ezsim object
#' @param stat Some preset summary statistics. Included,  \code{c('mean','tv','bias','biaspercentage','sd','rmse','min','q25','median','q75','max','jb_test')}
#' @param simple If True, shows only mean, true value, bias, sd and rmse of the estimator. If False, shows all statistics in stat.
#' @param subset subset of estimators or parameters. See \code{\link{subset.ezsim}} for details.
#' @param \dots Furhter summary statistics. Given in the form stat_name=stat. For example, Mean=mean
#' @return A summary.ezsim object
#' @author TszKin Julian Chan \email{ctszkin@@gmail.com}
#' @S3method summary ezsim
#' @seealso \code{\link{ezsim}}, \code{\link{plot.summary.ezsim}}, \code{\link{getScalarsName.summary.ezsim}}
#' @keywords post-simulation
#' @examples        
#' \dontrun{
#' ezsim_basic<-ezsim(
#'     m             = 100,
#'     run           = TRUE,
#'     core          = 1,
#'     display_name  = c(mean_hat="hat(mu)",sd_mean_hat="hat(sigma[hat(mu)])"),
#'     parameter_def = createParDef(list(n=seq(20,80,20),mu=c(0,2),sigma=c(1,3,5))),
#'     dgp           = function() rnorm(n,mu,sigma),
#'     estimator     = function(x) c(mean_hat = mean(x), 
#'                                  sd_mean_hat=sd(x)/sqrt(length(x)-1)),
#'     true_value    = function() c(mu, sigma / sqrt(n-1))
#' )
#' 
#' 
#' ## Subset of an ezsim object.
#' subset(ezsim_basic,subset=list(estimator='mean_hat',mu=0,n=c(20,40)))
#' 
#' ## Summary of an ezsim object
#' summary(ezsim_basic)
#' 
#' ## Summary of a subset of ezsim object
#' summary(ezsim_basic,subset=list(estimator='mean_hat',n=c(20,40),sigma=c(1,3)))
#' 
#' ## More Summary Statistics
#' summary(ezsim_basic,simple=FALSE,subset=list(estimator='mean_hat',n=c(20,40),sigma=c(1,3)))
#' 
#' ## Customize the Summary Statistics
#' summary(ezsim_basic,stat=c("q25","median","q75"),Q025=quantile(value_of_estimator,0.025),Q975=quantile(value_of_estimator,0.975),subset=list(estimator='mean_hat',n=c(20,40),sigma=c(1,3)))
#' }

summary.ezsim <-
function(object,stat=c('mean','tv','bias','biaspercentage',
    'sd','rmse','min','q25','median','q75','max','jb_test'),simple=TRUE,subset,...){
    
    if (is.null(object$sim))
        stop('Please run the simulation first.')
    if (length(stat)==0)
        stop('stat cant be empty')

    sim<-subset.ezsim(object,subset=subset)$sim

    ## preset summary statistics
    
    variable_name <- c('estimator',getScalarsName(object,TRUE)  )

    out<-NULL
    i<-NULL
    
    stat<-match.arg(stat, several.ok = TRUE)
    
    if (simple & length(stat)==12)
        stat=c('mean','tv','bias','sd','rmse','jb_test')

    if (length(stat)>0){
        stat_list<-
        foreach (i=stat,.combine=(function(...) paste(...,sep=',')) ) %do% {
            switch(i,
                mean='Mean=mean(value_of_estimator)',
                tv='TV=mean(value_of_TV)',
                bias='Bias=mean(value_of_estimator)-mean(value_of_TV)',
                biaspercentage='BiasPercentage=mean(value_of_estimator)/mean(value_of_TV)-1',
                sd='SD=sd(value_of_estimator)',
                rmse='rmse=sqrt(sd(value_of_estimator)^2+(mean(value_of_estimator)-mean(value_of_TV))^2)',
                min='Min=min(value_of_estimator)',
                q25='Q25=quantile(value_of_estimator,0.25)',
                median='Median=median(value_of_estimator)',
                q75='Q75=quantile(value_of_estimator,0.75)',
                max='Max=max(value_of_estimator)',
                jb_test='JB_test=jb.test(value_of_estimator)'
            )
        }

        out<-eval(parse(text=
            paste("ddply(sim,variable_name,summarize,",stat_list,')',sep='')
        ))
    }
    
    ## Custom summary statistics
    arg_list<-names(lapply(match.call()[-1],deparse))

    if (! all(arg_list %in% c('object','stat','subset','simple')  )){
        temp<-ddply(sim,variable_name,summarize,...)
        if (!is.null(out))
            out<-merge(out,temp,by=variable_name, suffixes = c("_1","_2"))
        else
            out<-temp
    }

    ## rename the estimator
    out$estimator<-factor(recode(as.character(out$estimator),object$display_name))

    ## Setup the summary.ezsim object
    # attr(out,'other_parameters')<-object$other_parameters
    attr(out,'scalar_parameters')<-getScalarsName(object,TRUE)
    attr(out,'display_name')<-object$display_name

    class(out)<-c('summary.ezsim',class(out))

    out
}    

#' Plot the summary statistics for several estimators in the same plot. Summary statistics abd estimators are separated by colour and linetype.
#' The longest scalars parameter will be the x-variable of the plot. The rest of the scalars parameters will be become the facets of the plot (see \pkg{ggplot2}).
#' others parameters will not be shown in the graph.
#' @name plot.summary.ezsim
#' @aliases plot.summary.ezsim
#' @usage  
#' \method{plot}{summary.ezsim}(x,parameters_priority,ylab='Summary Statistics',title,pdf_option,...)
#' @title Plot an summary.ezsim Object
#' @param x An summary.ezsim Object
#' @param parameters_priority Display priority of parameter. Any missed parameters will be sorted by length.
#' @param ylab Label of y-axis
#' @param title Title of the plot
#' @param pdf_option A list of option pass to \code{\link{pdf}}. If it is not missing, the plot will export to a pdf file
#' @param \dots unused
#' @return Optional: a ggplot2 object
#' @author TszKin Julian Chan \email{ctszkin@@gmail.com}
#' @S3method plot summary.ezsim
#' @seealso \code{\link{ezsim}},\code{\link{summary.ezsim}}, \code{\link{plot.summary.ezsim}},
#' @keywords post-simulation
#' @examples       
#' ezsim_basic<-ezsim(
#'     m             = 100,
#'     run           = TRUE,
#'     core          = 1,
#'     display_name  = c(mean_hat="hat(mu)",sd_mean_hat="hat(sigma[hat(mu)])"),
#'     parameter_def = createParDef(list(n=seq(20,80,20),mu=c(0,2),sigma=c(1,3,5))),
#'     dgp           = function() rnorm(n,mu,sigma),
#'     estimator     = function(x) c(mean_hat = mean(x), 
#'                                  sd_mean_hat=sd(x)/sqrt(length(x)-1)),
#'     true_value    = function() c(mu, sigma / sqrt(n-1))
#' )
#' ## Plot the summary ezsim
#' plot(summary(ezsim_basic,c("q25","q75")))
#' plot(summary(ezsim_basic,c("q25","q75"),subset=list(estimator='mean_hat')))
#' plot(summary(ezsim_basic,c("median"),subset=list(estimator='sd_mean_hat')))
plot.summary.ezsim<-
function(x,parameters_priority,ylab='Summary Statistics',title,pdf_option,...){

    display_name<-attr(x,'display_name')
         
    temp<-getScalarsName(x,parameters_priority=parameters_priority)
    ########### title
    title<-
    if (missing(title)){
        temp$subtitle
    }
    else {  
        if (temp$subtitle!='')
            paste(title,temp$subtitle,sep='~~')
        else 
            title
    }   

    x_var=head(temp$scalar_length_greater_one,1)
    other=tail(temp$scalar_length_greater_one,-1)    
            
    summ<-x

    summ<-melt(summ,id.vars=c('estimator',getScalarsName(x,TRUE)),variable_name='stat')
    
    my_facet<-facet_grid(createFormula(other), labeller = label_both_parsed_recode(display_name))

    out<-
        if (length(unique(summ$stat))==1 & length(unique(summ$estimator))==1){
            ggplot(data=summ, aes_string(x=x_var,y='value'),)+geom_line()
        } else if (length(unique(summ$stat))==1 & length(unique(summ$estimator))>1){
            ggplot(data=summ, aes_string(color='estimator', x=x_var,y='value'))+scale_colour_discrete(name='Estimators',breaks=unique(summ$estimator),labels=parse(text=paste(unique(summ$stat),'of', unique(summ$estimator),sep='~~')))
            
            #+scale_linetype(name='Summary Statistics',breaks=unique(summ$stat),labels=parse(text=unique(summ$stat)))
        } else if (length(unique(summ$stat))>1 & length(unique(summ$estimator))==1){
            ggplot(data=summ, aes_string(color='stat', x=x_var,y='value'))+scale_colour_discrete(name='Summary Statistics',breaks=unique(summ$stat),labels=parse(text=paste(unique(summ$stat),'of', unique(summ$estimator),sep='~~')))
            
            # +scale_linetype(name='Estimators',breaks=unique(summ$estimator),labels=parse(text=unique(summ$estimator)))
        } else {
            ggplot(data=summ, aes_string(linetype='stat',color='estimator', x=x_var,y='value'))+scale_colour_discrete(name='Estimators',breaks=unique(summ$estimator),labels=parse(text=unique(summ$estimator)))+scale_linetype(name='Summary Statistics',breaks=unique(summ$stat),labels=parse(text=unique(summ$stat)))
        }
        
    out <- out+geom_line()+geom_point() + my_facet+ylab(ylab)+xlab(parse(text=recode(x_var,display_name)))+opts(legend.position='bottom', legend.direction='horizontal')
    
    out<-out+opts(title=parse(text=title))
    
    if (!missing(pdf_option)){
        do.call(pdf,pdf_option)
        print(out)
        dev.off()
    } 
    else {
        return(out)
    }
}

#' Get names of scalars parameters from an summary.ezsim object.
#' @name getScalarsName.summary.ezsim
#' @aliases getScalarsName.summary.ezsim
#' @title Get Names of scalars Parameters.
#' @usage \method{getScalarsName}{summary.ezsim}(x,simple=FALSE,parameters_priority,...)
#' @param x an summary.ezsim object
#' @param simple If true, return only the name of scalar parameters. If False, split the scalar into two groups, one with fixed value, one with varied value. Also, subtitle is returned.
#' @param parameters_priority Priority in sorting parameters.
#' @param \dots unused
#' @return Names of scalars parameters. 
#' @author TszKin Julian Chan \email{ctszkin@@gmail.com}
#' @S3method getScalarsName summary.ezsim
#' @note For internal use of ezsim.
#' @seealso \code{\link{getScalarsName.ezsim}}
#' @examples       
#' \dontrun{
#' ezsim_basic<-ezsim(
#'     m             = 100,
#'     run           = TRUE,
#'     core          = 1,
#'     display_name  = c(mean_hat="hat(mu)",sd_mean_hat="hat(sigma[hat(mu)])"),
#'     parameter_def = createParDef(list(n=seq(20,80,20),mu=c(0,2),sigma=c(1,3,5))),
#'     dgp           = function() rnorm(n,mu,sigma),
#'     estimator     = function(x) c(mean_hat = mean(x), 
#'                                  sd_mean_hat=sd(x)/sqrt(length(x)-1)),
#'     true_value    = function() c(mu, sigma / sqrt(n-1))
#' )
#' 
#' getScalarsName(ezsim_basic)
#' getScalarsName(summary(ezsim_basic))
#' }

getScalarsName.summary.ezsim<-
function(x,simple=FALSE,parameters_priority,...){
    scalars_name <- attr(x,'scalar_parameters')
    display_name <- attr(x,'display_name')
    
    scalar_unique_length<-sapply(x[scalars_name], function(x) length(unique(x)))
    
    scalar_length_one<-names(scalar_unique_length)[scalar_unique_length==1]
    scalar_length_greater_one<-scalar_unique_length[scalar_unique_length>1]
    
    scalar_length_greater_one<-names(scalar_length_greater_one)[order(scalar_length_greater_one,decreasing=TRUE) ]
    
    subtitle<-''
    ## For scalars_name_length_one, we update the subtitle.
    if (length(scalar_length_one)>0 ){
        scalars_value_length_one<-unique(x[scalar_length_one])
        
        subtitle<-paste(paste(recode(scalar_length_one,display_name),scalars_value_length_one,sep='=='),collapse=',')
        subtitle<-paste('list(',subtitle,')',sep='')
        
       subtitle<-paste('group(\'(\',',subtitle,',\')\' )')
    }
    
    ## Update scalars_name_length_greater_one with parameters_priority
    if (!missing(parameters_priority)){
        parameters_priority<-
        tryCatch( match.arg(parameters_priority,scalar_length_greater_one,TRUE) ,
            error = function(e) scalar_length_greater_one )
        
        scalar_length_greater_one<-c(parameters_priority,setdiff(scalar_length_greater_one,parameters_priority))
    }
    
    if (!simple)
        return(list(scalar_length_greater_one=scalar_length_greater_one,scalar_length_one=scalar_length_one,subtitle=subtitle))
    else
        return(c(scalar_length_greater_one,scalar_length_one))
}

#' Get names of scalars parameters from an ezsim object.
#' @name getScalarsName.ezsim
#' @aliases getScalarsName.ezsim
#' @title Get Names of scalars Parameters.
#' @usage \method{getScalarsName}{ezsim}(x,simple=FALSE,parameters_priority,...)
#' @param x an ezsim object
#' @param simple If true, return only the name of scalar parameters. If False, split the scalar into two groups, one with fixed value, one with varied value. Also, subtitle is returned.
#' @param parameters_priority Priority in sorting parameters.
#' @param \dots unused
#' @return \item{scalar_length_greater_one}{ Name of scalar parameters with more than one elements} \item{scalar_length_one}{Name of scalar parameters with only one element} \item{subtitle}{subtitle for fixed scalar parameters}
#' @author TszKin Julian Chan \email{ctszkin@@gmail.com}
#' @S3method getScalarsName ezsim
#' @note For internal use of ezsim.
#' @seealso \code{\link{getScalarsName.summary.ezsim}}
#' @examples       
#' \dontrun{
#' ezsim_basic<-ezsim(
#'     m             = 100,
#'     run           = TRUE,
#'     core          = 1,
#'     display_name  = c(mean_hat="hat(mu)",sd_mean_hat="hat(sigma[hat(mu)])"),
#'     parameter_def = createParDef(list(n=seq(20,80,20),mu=c(0,2),sigma=c(1,3,5))),
#'     dgp           = function() rnorm(n,mu,sigma),
#'     estimator     = function(x) c(mean_hat = mean(x), 
#'                                  sd_mean_hat=sd(x)/sqrt(length(x)-1)),
#'     true_value    = function() c(mu, sigma / sqrt(n-1))
#' )
#' 
#' getScalarsName(ezsim_basic)
#' getScalarsName(summary(ezsim_basic))
#' }

getScalarsName.ezsim <-
function(x,simple=FALSE,parameters_priority,...){
    scalar_value<-unique(x$sim[!names(x$sim) %in% c('estimator','value_of_estimator','value_of_TV')])
    
    scalar_unique_length<-sapply(scalar_value, function(x) length(unique(x)))
    
    scalar_length_one<-names(scalar_value)[scalar_unique_length==1]
    
    # sort rest of them
    scalar_length_greater_one<-scalar_unique_length[scalar_unique_length>1]
    scalar_length_greater_one<-names(scalar_length_greater_one)[order(scalar_length_greater_one,decreasing=TRUE) ]

    subtitle<-''
    ## For scalars_name_length_one, we update the subtitle.
    if (length(scalar_length_one)>0 ){
        scalars_value_length_one<-unique(x$sim[scalar_length_one])
        
        subtitle<-paste(paste(recode(scalar_length_one,x$display_name),scalars_value_length_one,sep='=='),collapse=',')
        subtitle<-paste('list(',subtitle,')',sep='')
        
       subtitle<-paste('group(\'(\',',subtitle,',\')\' )')
    }
    
    ## Update scalars_name_length_greater_one with parameters_priority
    if (!missing(parameters_priority)){
        parameters_priority<-
        tryCatch( match.arg(parameters_priority,scalar_length_greater_one,TRUE) ,
            error = function(e) scalar_length_greater_one )
        
        scalar_length_greater_one<-c(parameters_priority,setdiff(scalar_length_greater_one,parameters_priority))
    }
    
    if (!simple)
        return(list(scalar_length_greater_one=scalar_length_greater_one,scalar_length_one=scalar_length_one,subtitle=subtitle))
    else
        return(c(scalar_length_greater_one,scalar_length_one))
}

#' Print a summary.ezsim Object in the console. See \code{\link{summary.ezsim}} for details
#' @name print.summary.ezsim
#' @aliases print.summary.ezsim
#' @title Print a summary.ezsim Object.
#' @usage \method{print}{summary.ezsim}(x,digits=4,...)
#' @param x A summary.ezsim Object
#' @param digits Number of digits the data will be rounded to.
#' @param \dots unused
#' @author TszKin Julian Chan \email{ctszkin@@gmail.com}
#' @S3method print summary.ezsim
#' @seealso \code{\link{summary.ezsim}}

print.summary.ezsim <-
function(x,digits=4,...){

    for( i in 1:length(x) ) 
        if (is.numeric(x[[i]]))
            x[[i]]<-round(x[[i]],digits=4)
            

    print.data.frame(x)
}

#' createParDef creates a new parameterDef object from a list of scalar parameters and a list of other parameters.
#' parameterDef is a short hand of "parameter definition". It defines parameters used by the \code{\link{dgp}} which is the most important part of a simulation. For each simulation,There is a particular set of parameter. parameterDef allow us to define several parameters for different simulation at once. There are two types of parameter in parameterDef, scalar parameters and other parameters.
#' Scalar parameters must be a scalar. Any vectors or matrix is regarded as a sequence of scalar parameters. For example, n=seq(10,50,10), first simulation  takes n=10, second simulation takes n=20 and so on.
#' Other parameters can be anything and it is others over the scalar parameters.
#' For example, we would like to know how would the sample size affect the variance of the sample mean of normally distributed variable. We can set n=seq(10,50,10), mean=1 and sd=2.  (see example)
#' @name createParDef
#' @aliases parameterDef createParDef
#' @title Create a parameterDef Object.
#' @param scalars A list of scalar parameters
#' @param others A list of other parameters
#' @return A parameterDef object
#' @author TszKin Julian Chan \email{ctszkin@@gmail.com}
#' @export
#' @seealso \code{\link{setOthers.parameterDef}},\code{\link{setScalars.parameterDef}},\code{\link{test.parameterDef}},\code{\link{generate.parameterDef}}
#' @keywords parameterDef
#' @examples       
#' par_def1<-createParDef(scalar=list(mean=1,sd=2,n=seq(10,50,10)))
#' 
#' par_def2<-createParDef()
#' setScalars(par_def2,mean=1,sd=2,n=seq(10,50,10))
#' 
#' identical(par_def1,par_def2)
#' 
#' test(par_def1, function() rnorm(n,mean,sd) )  # 10 random number
#' test(par_def1, function() rnorm(n,mean,sd), index=3)  # 30 random number
#' 
#' generate(par_def1)
#' 
#' # More than 1 scalars parameters 
#' par_def3<-createParDef(scalars=list(sd=2,mean=1:3,n=seq(10,50,10)))
#' 
#' generate(par_def3)

createParDef <-
function(scalars=list(),others=list()){
    x<-list(scalars=list(),others=list())
    class(x)<-"parameterDef"

    if (length(others)>0)
        x<-do.call(function(...) setOthers.parameterDef(x,...),others)
    if (length(scalars)>0)
        x<-do.call(function(...) setScalars.parameterDef(x,...),scalars)
    x
}

#' setOthers sets the scalar parameters of a parameterDef object. setOthers are "call by reference", so assignment is not needed to update the parameterDef object. In other words, they will overwrite the value of its argument(parameterDef object).
#' parameterDef is a short hand of "parameter definition". It defines parameters used by the \code{\link{dgp}} which is the most important part of a simulation. For each simulation,There is a particular set of parameter. parameterDef allow us to define several parameters for different simulation at once. There are two types of parameter in parameterDef, scalar parameters and other parameters.
#' Scalar parameters must be a scalar. Any vectors or matrix is regarded as a sequence of scalar parameters. For example, n=seq(10,50,10), first simulation  takes n=10, second simulation takes n=20 and so on.
#' Other parameters can be anything and it is others over the scalar parameters.
#' For example, we would like to know how would the sample size affect the variance of the sample mean of normally distributed variable. We can set n=seq(10,50,10), mean=1 and sd=2.  (see example)
#' @name setOthers.parameterDef
#' @aliases setOthers.parameterDef
#' @title Set a parameterDef Object.
#' @usage \method{setOthers}{parameterDef}(x,...)
#' @param x A parameterDef object
#' @param \dots Variables to be added to a parameterDef object
#' @return A parameterDef object
#' @author TszKin Julian Chan \email{ctszkin@@gmail.com}
#' @S3method setOthers parameterDef
#' @seealso \code{\link{setScalars.parameterDef}},\code{\link{createParDef}},\code{\link{test.parameterDef}},\code{\link{generate.parameterDef}}
#' @keywords parameterDef
#' @examples       
#' par_def1<-createParDef(scalar=list(mean=1,sd=2,n=seq(10,50,10)))
#' 
#' par_def2<-createParDef()
#' setScalars(par_def2,mean=1,sd=2,n=seq(10,50,10))
#' 
#' identical(par_def1,par_def2)
#' 
#' test(par_def1, function() rnorm(n,mean,sd) )  # 10 random number
#' test(par_def1, function() rnorm(n,mean,sd), index=3)  # 30 random number
#' 
#' generate(par_def1)
#' 
#' # More than 1 scalars parameters 
#' par_def3<-createParDef(scalars=list(sd=2,mean=1:3,n=seq(10,50,10)))
#' 
#' generate(par_def3)
#' 
#' # 
#' par_def4<-createParDef(scalar=list(mean=1,sd=2,n=seq(10,50,10)))
#' setOthers(par_def4,some_matrix=matrix(1:4,nrow=2),some_vector=1:6)
#' par_def4
#' generate(par_def4)

setOthers.parameterDef <-
function(x,...){
    xx<-x
    temp<-list(...)
    i<-NULL
    if (length(temp)==0)
        stop("Nothing to set")
    
    # remove parameter in x$scalars if parameter appear in ...
    if ( length(xx$scalars)!=0   && length(intersect(names(temp),names(xx$scalars)))>0){
        warning("Same variables name as scalars parameter. scalars paramter will be removed")
        xx$scalars <- xx$scalars[[names(xx$scalars) %in% names(temp)]] 
    }

    # concat those parameter which is not repeated
    repeated<-names(temp) %in% names(xx$others)
    xx$others<-c(xx$others,temp[!repeated])
    
    # for those who are repeated, replace the original value
    if (sum(repeated)>0){
        warning(paste("\nOriginal value of ",names(temp)[repeated] ," is replaced"))
        foreach (i = names(temp)[repeated]) %do% {
            xx$others[[i]]<-temp[[i]]
        }        
    }
    eval.parent(substitute(x<-xx))
}

#' setScalars sets the others parameters of a parameterDef object. setScalars are "call by reference", so assignment is not needed to update the parameterDef object. In other words, they will overwrite the value of its argument(parameterDef object).
#' parameterDef is a short hand of "parameter definition". It defines parameters used by the \code{\link{dgp}} which is the most important part of a simulation. For each simulation,There is a particular set of parameter. parameterDef allow us to define several parameters for different simulation at once. There are two types of parameter in parameterDef, scalar parameters and other parameters.
#' Scalar parameters must be a scalar. Any vectors or matrix is regarded as a sequence of scalar parameters. For example, n=seq(10,50,10), first simulation  takes n=10, second simulation takes n=20 and so on.
#' Other parameters can be anything and it is others over the scalar parameters.
#' For example, we would like to know how would the sample size affect the variance of the sample mean of normally distributed variable. We can set n=seq(10,50,10), mean=1 and sd=2.  (see example)#' @name setScalars.parameterDef
#' @aliases setScalars.parameterDef
#' @title Set a parameterDef Object.
#' @usage \method{setScalars}{parameterDef}(x,...)
#' @param x A parameterDef object
#' @param \dots Variables to be added to a parameterDef object
#' @return A parameterDef object
#' @author TszKin Julian Chan \email{ctszkin@@gmail.com}
#' @S3method setScalars parameterDef
#' @seealso \code{\link{setScalars.parameterDef}},\code{\link{createParDef}},\code{\link{test.parameterDef}},\code{\link{generate.parameterDef}}
#' @keywords parameterDef
#' @examples       
#' par_def1<-createParDef(scalar=list(mean=1,sd=2,n=seq(10,50,10)))
#' 
#' par_def2<-createParDef()
#' setScalars(par_def2,mean=1,sd=2,n=seq(10,50,10))
#' 
#' identical(par_def1,par_def2)
#' 
#' test(par_def1, function() rnorm(n,mean,sd) )  # 10 random number
#' test(par_def1, function() rnorm(n,mean,sd), index=3)  # 30 random number
#' 
#' generate(par_def1)
#' 
#' # More than 1 scalars parameters 
#' par_def3<-createParDef(scalars=list(sd=2,mean=1:3,n=seq(10,50,10)))
#' 
#' generate(par_def3)
setScalars.parameterDef <-
function(x,...){
    temp<-list(...)
    xx<-x
    i<-NULL
    j<-NULL
    flag<-
    foreach( i = temp, .combine=c ) %:% foreach(j=i, .combine=c )  %do% !(is.numeric(j) & length(j)==1)

    if ( sum(flag)!=0)
        stop("scalars Parameter must be numeric scaler")
    
    
    if (length(temp)==0)
        stop("Nothing to set")

    if (length(xx$others)!=0 && length(intersect(names(temp),names(xx$others)))>0 ){
        warning("Same variables name as others parameter. others paramter will be removed")
        xx$others <- xx$others[[names(xx$others) %in% names(temp)]] 
    }
    
    repeated<-names(temp) %in% names(xx$scalars)
    xx$scalars<-c(xx$scalars,temp[!repeated])
    
    if (sum(repeated)>0){
        warning(paste("\nOriginal value of ",names(temp)[repeated] ," is replaced"))
        foreach (i = names(temp)[repeated]) %do% {
            xx$scalars[[i]]<-temp[[i]]
        }        
    }
    eval.parent(substitute(x<-xx))
}


#' Generate Parameters from a parameterDef Object.
#' The scalars parameters in parameterDef is expanded and concatenated with the others parameters.
#' @name generate.parameterDef
#' @aliases generate.parameterDef
#' @title Generate Parameter
#' @usage \method{generate}{parameterDef}(x,...)
#' @param x A parameterDef Object
#' @param \dots unused
#' @return \item{other_parameters}{A list of other_parameters} \item{scalar_parameters}{A data.frame of scalar_parameters} \item{parameter_list}{A list of all parameters}
#' @author TszKin Julian Chan \email{ctszkin@@gmail.com}
#' @S3method generate parameterDef
#' @seealso \code{\link{setOthers.parameterDef}}, \code{\link{setScalars.parameterDef}}, \code{\link{test.parameterDef}}, \code{\link{generate.parameterDef}}
#' @examples         
#' par_def1<-createParDef(scalars=list(mean=1,sd=2,n=seq(10,50,10)))
#' generate(par_def1)
#' par_def2<-createParDef(scalars=list(sd=2,mean=1:3,n=seq(10,50,10)))
#' generate(par_def2)
generate.parameterDef <-
function(x,...){
    if (length(x$scalars)>0){
        scalar_parameters<-expand.grid(x$scalars)
        apply(scalar_parameters,1,function(xx) {
            temp<-c(x$others,xx)
            class(temp)<-c("parameters","list")
            temp
        })
    } else {
        temp<-x$others
        class(temp)<-c("parameters","list")
        list(temp)
    }
}

#' several set of parameters is generated from parameterDef. Function \code{fun} is evaulated under the \code{index-th} set of parameters and returns its value.
#' @name test.parameterDef
#' @aliases test.parameterDef
#' @title Test Whether a parameterDef Ojbect Work Properly for a dgp.
#' @method test parameterDef
#' @param x A parameterDef object
#' @param fun A function to be evaluated. 
#' @param index Which set of parameters to use.
#' @param \dots unused
#' @author TszKin Julian Chan \email{ctszkin@@gmail.com}
#' @S3method test parameterDef
#' @seealso \code{\link{parameterDef}}
#' @examples        
#' par_def<-createParDef()
#' setScalars(par_def,mean=1,sd=2,n=seq(10,50,10))
#' 
#' test(par_def, function() rnorm(n,mean,sd) )  # 10 random number
#' test(par_def, function() rnorm(n,mean,sd), index=3)  # 30 random number
#' 
#' ## Example 2
#' par_def<-createParDef()
#' setOthers(par_def,xs=1,b=1)
#' setScalars(par_def,n=seq(20,100,20),es=c(1,10))
#' 
#' dgp<-function(){
#' x<-rnorm(n,0,xs)
#' e<-rnorm(n,0,es)
#' y<-b * x + e
#' data.frame(y,x)
#' }
#' estimator<-function(d){
#' r<-summary(lm(y~x-1,data=d))
#' c(b=r$coef[,1],t=(r$coef[,1]-1)/r$coef[,2] )
#' }
#' 
#' true<-function(){
#' c(b,(b-1)/(es/sqrt(n)/xs)) 
#' }
#' test(par_def,dgp)
#' estimator(test(par_def,dgp))
#' test(par_def,true)
test.parameterDef <-
function(x,fun,index=1,...){
    out<-
    lapply(index, function(i)
        run(fun,generate(x)[[i]])
    )
    names(out) <- lapply(index, function(i)
        paste(paste(names(generate(x)[[i]]),
            generate(x)[[i]],sep='='),collapse=',')
    )
    if (length(out)==1)
        out[[1]]
    else   
        out
}

#' Print a parameters Object in the console
#' @name print.parameters
#' @aliases print.parameters
#' @title Print a parameters Object.
#' @method print parameters
#' @param x A parameters Object
#' @param \dots unused
#' @author TszKin Julian Chan \email{ctszkin@@gmail.com}
#' @S3method print parameters
#' @seealso \code{\link{parameterDef}}

print.parameters <-
function(x,...){
    y<-x[sapply(x,length)!=1]
    if (length(y)!=0){
        for (i in 1:length(y)){
            cat(names(y)[i]," : \n")
            print(y[[i]])
            cat("\n")
        }
    }
    
    y<-x[sapply(x,length)==1]
    cat(paste(names(y),y,sep='=',collapse=', '))
    cat("\n")
}

#' Print a parameterDef Object in the console
#' @name print.parameterDef
#' @aliases print.parameterDef
#' @title Print a parameterDef Object.
#' @method print parameterDef
#' @param x A parameterDef Object
#' @param \dots unused
#' @author TszKin Julian Chan \email{ctszkin@@gmail.com}
#' @S3method print parameterDef
#' @seealso \code{\link{createParDef}}

print.parameterDef <-
function(x,...){
    cat("Scalar Parameters:\n")
    print(x$scalars)
    cat("\n")
    cat("Others Parameters:\n")
    print(x$others)
}

#' Merge a vector to data.frame. Each element of the vector will be a new column. Each rows will be filled up with the same value.
#' @name add2DataFrame
#' @aliases add2DataFrame
#' @title Merge a Vector to \code{data.frame}
#' @param x \code{data.frame} 
#' @param y \code{vector} to be merged to the \code{data.frame} 
#' @param name Name of new columns in the \code{data.frame}. (Default = names(y))
#' @return a \code{data.frame} that contains all columns in x and all element of y as different columns.
#' @author TszKin Julian Chan \email{ctszkin@@gmail.com}
#' @export
#' @note For internal use of ezsim.

#' @examples      
#' d=data.frame(x=1:5,y=11:15)
#' v=c(a=100,b=200,c=300)
#' add2DataFrame(d,v)
add2DataFrame <-
function(x,y,name=names(y)){
    if (!is.vector(y) || is.list(y))
        stop("invalid y")
    if (length(y)!=length(name))
        stop("length of y is not the same as length of name")
    
    cbind(x,matrix(rep(y,each=dim(x)[1]),ncol=length(y),dimnames=list(NULL,name)))
}

#' Recode the value of a vector.
#' @name recode
#' @aliases recode
#' @title Recode the value of a vector
#' @param x a vector
#' @param value recode name and value
#' @return a vector
#' @author TszKin Julian Chan \email{ctszkin@@gmail.com}
#' @export
#' @note For internal use of ezsim.
#' @examples      
#' x=rep(1:5,each=2)
#' value=5:1
#' names(value)=1:5
#' recode(x,value)
recode<-function(x,value){
	xx<-x
	for (i in 1:length(value))
		x[xx==names(value)[i]] <- value[i]
	x
}

#' This function evaluates a function \code{x} under an environment which is created by a list. All elements of the list is local to the function; other words all elements of the list can be accessed directly by the function. 
#' In this function, a new environment is created and each element of \code{variables} is assigned to the new environment. Then the environment associated with the \code{x} is updated with the new environment. Finally \code{x(...)} is evaluated and return the result.
#' The main usage of \code{run.function} is to evaluate \code{dgp} or \code{true_value} under a list of parameter. See \code{\link{ezsim}} for details.
#' @name run.function
#' @aliases run.function
#' @title Evaluate Function Under Local Variables
#' @method run function
#' @param x A function to be called
#' @param variables A list to be converted to an environment
#' @param \dots Further arguments to \code{x}
#' @return Return value of the \code{x}.
#' @author TszKin Julian Chan \email{ctszkin@@gmail.com}
#' @S3method run "function"
#' @note For internal use of ezsim.
#' @seealso \code{\link{environment}}, \code{\link{ezsim}}
#' @examples      
#' run(function() rnorm(n,mean,sd),list(n=5,mean=5,sd=1))
run.function <-
function(x,variables=list,...){
    e <- new.env()
    mapply(assign, MoreArgs = list(envir=e),x=names(variables),value=variables)
    environment(x)<-e
    x(...)
}

#' Elements of the vector is evenly distributed to both of the formula. Each element in the formula is seperated by \code{+}.
#' @name createFormula
#' @aliases createFormula
#' @title Create Formula From a Vector of Character.
#' @param x A vector of character
#' @param right If there is only one element in \code{x}, should it appear in the left or right hand side of the formula.
#' @return Formula
#' @author TszKin Julian Chan \email{ctszkin@@gmail.com}
#' @export
#' @note For internal use of ezsim. It aims at creating formula for facets in \pkg{ggplot2}.
#' @seealso \code{\link{formula}}
 
#' @examples      
#' createFormula(letters[1])  ## . ~ a
#' createFormula(letters[1],right=FALSE)  ## a ~ .
#' createFormula(letters[1:3])  ## c ~ a + b
#' createFormula(letters[1:4])  ## c + d ~ a + b
#' createFormula(letters[1:4],right=FALSE) ## a + b ~ c + d
createFormula <-
function(x,right=TRUE){
    f<-'.~.'
    if (length(x)==1){
        if (right)
            f<-paste('.',x,sep='~')
        else
            f<-paste(x,'.',sep='~')
    } else if (length(x)>1){
        half_length<-length(x)%/%2
        
        if (right) 
            f<-paste(
                    paste(tail(x,half_length),collapse='+'),
                    paste(head(x,length(x)-half_length),
                    collapse='+')
                    ,sep='~'
                )
        else
            f<-paste(
                    paste(head(x,length(x)-half_length),collapse='+'),
                    paste(tail(x,half_length),
                    collapse='+')
                    ,sep='~'
                )
    }
    as.formula(f)
}

#' Return the p Value of Jarque Bera test. The Jarque Bera test  test the null hypothesis that the data are from a normal distribution. 
#' @name jb.test
#' @aliases jb.test
#' @title p Value of Jarque Bera test
#' @param x data
#' @return p Value of Jarque Bera test
#' @author TszKin Julian Chan \email{ctszkin@@gmail.com}
#' @export
#' @examples        
#' jb.test(rnorm(50))
#' jb.test(rt(50,3))
#' 
#' n=100
#' # size
#' mean(replicate(n,jb.test(rnorm(100)))<0.05)
#' 
#' # power
#' mean(replicate(n,jb.test(rt(100,3)))<0.05)
jb.test <-
function(x){
    m1<-mean(x)
    m2<-mean((x-m1)^2)
    m3<-mean((x-m1)^3)
    m4<-mean((x-m1)^4)
    s<-m3/m2^(3/2)
    k<-m4/m2^2-3
    jb_test=length(x)/6*(s^2+k^2/4)
    1 - pchisq(jb_test, df = 2)
}


#' Generic function
#' @name generate
#' @aliases generate
#' @title Generic function
#' @param x Object
#' @param \dots Further arguments
#' @author TszKin Julian Chan \email{ctszkin@@gmail.com}
#' @export
#' @seealso \code{\link{generate.parameterDef}}

generate <-
function(x,...){
    UseMethod("generate")
}

#' Generic function
#' @name getScalarsName
#' @aliases getScalarsName
#' @title Generic function
#' @param x Object
#' @param \dots Further arguments
#' @author TszKin Julian Chan \email{ctszkin@@gmail.com}
#' @export
#' @seealso \code{\link{getScalarsName.ezsim}}, \code{\link{getScalarsName.summary.ezsim}}

getScalarsName <-
function(x,...){
    UseMethod("getScalarsName")
}

#' Generic function
#' @name run
#' @aliases run
#' @title Generic function
#' @param x Object
#' @param \dots Further arguments
#' @author TszKin Julian Chan \email{ctszkin@@gmail.com}
#' @export
#' @seealso \code{\link{run.ezsim}}

run <-
function(x,...){
    UseMethod("run")
}

#' Generic function
#' @name test
#' @aliases test
#' @title Generic function
#' @param x Object
#' @param \dots Further arguments
#' @author TszKin Julian Chan \email{ctszkin@@gmail.com}
#' @export
#' @seealso \code{\link{test.parameterDef}}, \code{\link{test.ezsim}}

test <-
function(x,...){
    UseMethod("test")
}

#' Generic function
#' @name setOthers
#' @aliases setOthers
#' @title Generic function
#' @param x Object
#' @param \dots Further arguments
#' @author TszKin Julian Chan \email{ctszkin@@gmail.com}
#' @export
#' @seealso \code{\link{setOthers.parameterDef}}

setOthers <-
function(x,...){
    UseMethod("setOthers")
}

#' Generic function
#' @name setScalars
#' @aliases setScalars
#' @title Generic function
#' @param x Object
#' @param \dots Further arguments
#' @author TszKin Julian Chan \email{ctszkin@@gmail.com}
#' @export
#' @seealso \code{\link{setScalars.parameterDef}}

setScalars <-
function(x,...){
    UseMethod("setScalars")
}

#' Combine label_both and label_parsed in \pkg{ggplot2}. Also added a rename function to it
#' see label_both and label_parsed in \pkg{ggplot2} for details.
#' @name label_both_parsed_recode
#' @aliases label_both_parsed_recode
#' @title Combine label_both and label_parsed in \pkg{ggplot2}.
#' @param display_name A vector contains the display name. Names of the vector are the original name.
#' @return A function similar to label_both and label_parsed in \pkg{ggplot2} for details.
#' @author TszKin Julian Chan \email{ctszkin@@gmail.com}
#' @export
#' @references \url{http://cran.r-project.org/web/packages/ggplot2/index.html}
#' @seealso \code{\link{label_both}}, \code{\link{label_parsed}}
#' @keywords internal
label_both_parsed_recode <-
function(display_name){
    function(variable, value){
        variable=recode(variable,display_name)
        v<-paste(variable, value, sep = "==")
        lapply(as.character(v), function(x) parse(text = x))
    }
}