
# This library is free software; you can redistribute it and/or
# modify it under the terms of the GNU Library General Public
# License as published by the Free Software Foundation; either
# version 2 of the License, or (at your option) any later version.
#
# This library is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
# GNU Library General Public License for more details.
#
# You should have received A copy of the GNU Library General
# Public License along with this library; if not, write to the
# Free Foundation, Inc., 59 Temple Place, Suite 330, Boston,
# MA  02111-1307  USA

# Copyrights (C)
# for this R-port:
#   1999 - 2007, Diethelm Wuertz, GPL
#   Diethelm Wuertz <wuertz@itp.phys.ethz.ch>
#   info@rmetrics.org
#   www.rmetrics.org
# for the code accessed (or partly included) from other R-ports:
#   see R's copyright and license files
# for the code accessed (or partly included) from contributed R-ports
# and other sources
#   see Rmetrics's copyright file


# fEcofin::4B-JarqueBeraPValues
################################################################################
# FUNCTION:                 JARQUE-BERA DATA TABLES:
# .jbTable                   Finite sample p values for the Jarque Bera test
# .jbPlot                    Plots probabilities
# .pjb                       Returns probabilities for JB given quantiles
# .qjb                       Returns quantiles for JB given probabilities
# DATA:                     DESCRIPTION:
# .jbLM                      Jarque-Bera Lagrange Multiplier Data
# .jbALM                     Jarque Bera Augmented Lagrange Multiplier Data
################################################################################


################################################################################
# FUNCTION:           JARQUE-BERA DATA TABLES:
# .jbTable             Finite sample p values for the Jarque Bera test
# .jbPlot              Plots probabilt
# .pjb                 Returns probabilities for the JB Test given quantiles
# .qjb                 Returns quantiles for the ADF Test given probabilities


.jbTable =
function(type = c("LM", "ALM"), size = c("mini", "small", "all"))
{   # A function implemented by Diethelm Wuertz

    # Description:
    #   Finite sample p values for the Jarque Bera test

    # Details:
    #   The function jbTable() returns a data.frame with rows denoting
    #   size and columns denoting probabilities 0 < p < 1.

    # Note:
    #   x=-3:0; y=0:3; z=outer(x,y,"*"); rownames(z)=x; colnames(z)=y; z

    # Example:
    #   .jbTable()

    # FUNCTION:

    # Check Arguments:
    type = match.arg(type)
    size = match.arg(size)

    # Create Table:
    if (type == "LM") {
        Table = t(.jbLM())
    } else if (type == "ALM") {
        Table = t(.jbALM())
    }

    # Select Size:
    if (size == "small") {
        Table = t(Table)
        n = dim(Table)[1]
        Table = Table[c(matrix(1:(n-2), byrow = TRUE, ncol = 22)[, 1], n), ]
        Table = Table[-(1:17),]
        Table = t(Table)
    } else if (size == "mini") {
        p = c(
            "0.01", "0.0251189", "0.05", "0.1",
            "0.9", "0.95", "0.9751687", "0.99")
        Table = Table[, p]
        Table = rbind(Table, qchisq(as.numeric(p), df = 2))
        Table = round(Table, 3)
        colnames(Table) = substr(colnames(Table), 1, 5)
        nRows = dim(Table)[1]
        rownames(Table)[nRows] = "Inf"
    }
    ans = list(
        x = as.numeric(rownames(Table)),
        y = as.numeric(colnames(Table)),
        z = Table)

    # Add Control:
    attr(ans, "control") <-
        c(table = "jb", type = type, size = size)

    # Return Value:
    class(ans) = "gridData"
    ans
}


# ------------------------------------------------------------------------------


.jbPlot =
function(type = c("LM", "ALM"))
{   # A function implemented by Diethelm Wuertz

    # Match Arguments:
    type = match.arg(type)

    # Load Table:
    Y = .jbTable(size = "small")
    X = cbind(expand.grid(x = Y$x, y = Y$y), z = as.vector(Y$z))
    x = X[, 1] # N
    y = X[, 3] # q-Stat
    z = X[, 2] # p-Value

    # Interpolate:
    ans = akimaInterp(log(x), log(y), z, gridPoints = 101, extrap = FALSE)
    persp(ans, theta = 40, xlab = "log(N)", ylab = "log(q)", zlab = "p",
        main = paste(type, "Jarque Bera"))

    # Return Value:
    invisible(NULL)
}



# ------------------------------------------------------------------------------


.pjb =
function(q, N = Inf, type = c("LM", "ALM"))
{   # A function implemented by Diethelm Wuertz

    # Description:
    #   Returns probabilities for the JB Test given quantiles

    # Arguments:
    #   q = quantiles (0.020 < q < 12)
    #   N = sample sizes

    # Details:
    #   Uses "small" table for interpolation.

    # Example:
    #   q=c(0.03,1,5,9); for (N in c(4,1000,50000,Inf)) print(.pjb(q, N))

    # FUNCTION:

    # Match Arguments:
    type = match.arg(type)

    # Check N:
    stopifnot(length(N) == 1)

    # Grid Points:
    Y = .jbTable(type = type, size = "small")
    if (N < 10000) {
        # Interpolate on log-Scale:
        X = cbind(expand.grid(x = Y$x, y = Y$y), z = as.vector(Y$z))
        x = log(X[, 1]) # N
        y = log(X[, 3]) # q-Stat
        z = X[, 2] # p-Value
        # Out Points:
        xo = rep(log(N), times = length(q))
        yo = log(q)
        # Interpolate:
        p = linearInterpp(x, y, z, xo = xo, yo = yo)[, 3]
    } else if (N > 10000) {
        p = pchisq(as.numeric(q), df = 2)
    }

    # Result:
    if (N < 5) controlN = "< 5"
    if (N >= 5 & N <= 10000) controlN = as.character(N)
    if (N > 10000) controlN = "> 10000"
    if (!is.finite(N)) controlN = "Inf"
    names(p) = as.character(q)
    attr(p, "control") <- c(N = controlN)

    # Return Value:
    p
}


# ------------------------------------------------------------------------------


.qjb =
function(p, N = Inf, type = c("LM", "ALM"))
{   # A function implemented by Diethelm Wuertz

    # Description:
    #   Returns quantiles for the ADF Test given probabilities

    # Arguments:
    #   p = probabilities (0.01 < p < 0.99)
    #   N = sample sizes

    # Details:
    #   Uses "small" table for interpolation.

    # Example:
    #   p=(1:9)/10; for (N in c(4,1000,50000,Inf)) print(.qjb(p, N))

    # FUNCTION:

    # Match Arguments:
    type = match.arg(type)
    statistic = match.arg(statistic)

    # Check N:
    stopifnot(length(N) == 1)

    # Grid Points:
    Y = .jbTable(type = type, size = "small")
    if (N < 10000) {
        # Interpolate on log-Scale:
        X = cbind(expand.grid(x = Y$x, y = Y$y), z = as.vector(Y$z))
        x = log(X[, 1]) # N
        y = X[, 2] # p-Value
        z = X[, 3] # q-Stat
        # Out Points:
        xo = rep(log(N), times = length(p))
        yo = p
        # Interpolate - Scaling Required:
        xStd = sqrt(var(x))
        q = linearInterpp(x/xStd, y, z, xo = xo/xStd, yo = yo)[, 3]
    } else {
        # Asymptotic Limit:
        q = qchisq(as.numeric(p), df = 2)
    }

    # Result:
    if (N < 5) controlN = "< 5"
    if (N >= 5 & N <= 10000) controlN = as.character(N)
    if (N > 10000) controlN = "> 10000"
    if (!is.finite(N)) controlN = "Inf"
    names(q) = as.character(p)
    attr(q, "control") <- c(N = controlN)

    # Return Value:
    q
}


################################################################################
# DATA:               Description:
# .jbLM                Jarque-Bera Lagrange Multiplier Test Data
# .jbALM               Jarque Bera Augmented Lagrange Multiplier Test Data


.jbALM =
function()
{
structure(list("5" = c(3.6559e-05, 3.6676e-05, 3.8408e-05, 3.9805e-05,
4.0283e-05, 4.0549e-05, 4.062e-05, 4.1218e-05, 4.2351e-05, 4.2355e-05,
4.2797e-05, 4.3593e-05, 4.3764e-05, 4.3945e-05, 4.4328e-05, 4.4677e-05,
4.4694e-05, 4.5125e-05, 4.5133e-05, 4.5314e-05, 4.5404e-05, 4.6125e-05,
4.7878e-05, 4.791e-05, 4.8059e-05, 4.8158e-05, 4.8537e-05, 4.8671e-05,
4.9346e-05, 5.2005e-05, 5.2045e-05, 5.2874e-05, 5.4154e-05, 5.4401e-05,
5.6475e-05, 5.7065e-05, 5.7268e-05, 5.7759e-05, 5.8145e-05, 5.8421e-05,
5.8544e-05, 5.9025e-05, 6.1431e-05, 6.2246e-05, 6.2726e-05, 6.3127e-05,
6.4263e-05, 6.5175e-05, 6.558e-05, 6.5867e-05, 6.7439e-05, 6.7628e-05,
6.8728e-05, 6.9229e-05, 6.9781e-05, 7.047e-05, 7.4419e-05, 7.52e-05,
7.5643e-05, 7.6625e-05, 7.7835e-05, 7.8056e-05, 8.0169e-05, 8.0525e-05,
8.1766e-05, 8.2566e-05, 8.3138e-05, 8.445e-05, 8.5821e-05, 8.653e-05,
8.688e-05, 8.7287e-05, 8.8008e-05, 8.9306e-05, 8.9998e-05, 9.0808e-05,
9.0994e-05, 9.2429e-05, 9.4441e-05, 9.7687e-05, 9.8294e-05, 9.9436e-05,
9.9811e-05, 0.000100384, 0.000101058, 0.000102205, 0.000103164,
0.000103798, 0.000104172, 0.000106155, 0.000106266, 0.000108068,
0.000109299, 0.000110262, 0.000110657, 0.000113396, 0.000114168,
0.000114691, 0.000115894, 0.000118261, 0.000118738, 0.00012071,
0.00012185, 0.000122411, 0.000122816, 0.000123872, 0.000124869,
0.000125712, 0.000127507, 0.000129113, 0.000129633, 0.000130914,
0.000132618, 0.000134974, 0.000137935, 0.000138358, 0.000140343,
0.000142789, 0.000143747, 0.000145185, 0.00014727, 0.000149816,
0.000151305, 0.00015328, 0.000156449, 0.000157562, 0.000158683,
0.000160049, 0.000161523, 0.000164733, 0.000166998, 0.000169417,
0.000171718, 0.000174539, 0.000176047, 0.000178368, 0.000179273,
0.000180695, 0.000181991, 0.000182821, 0.000184236, 0.000186803,
0.000188006, 0.000191383, 0.000192252, 0.000195411, 0.000197557,
0.000199396, 0.000203267, 0.000206478, 0.000208246, 0.00021081,
0.00021277, 0.000214432, 0.000216338, 0.000219331, 0.000221487,
0.000223769, 0.000225692, 0.000229045, 0.000232534, 0.000236029,
0.000237355, 0.000238839, 0.000241658, 0.000243651, 0.000247937,
0.000251439, 0.000254616, 0.00025698, 0.000259024, 0.000263569,
0.000266388, 0.000268836, 0.000270824, 0.000275006, 0.000277729,
0.000280224, 0.000283939, 0.000286056, 0.00028816, 0.000291164,
0.000295574, 0.000297855, 0.000301406, 0.000303802, 0.000306588,
0.000309147, 0.000312405, 0.000316706, 0.000320339, 0.00032635,
0.000328857, 0.000332809, 0.00033598, 0.00033906, 0.000343894,
0.000347044, 0.000352935, 0.000356722, 0.000360606, 0.000363624,
0.000366725, 0.000371637, 0.000377895, 0.000380807, 0.000387119,
0.000391195, 0.00039438, 0.000398039, 0.000402309, 0.00040665,
0.000412242, 0.00041689, 0.000421805, 0.000426133, 0.000429776,
0.000435604, 0.000442123, 0.000445948, 0.000450778, 0.000456309,
0.000460125, 0.000465173, 0.00047096, 0.000478304, 0.000482793,
0.0004886, 0.000492719, 0.000500311, 0.000505597, 0.000512136,
0.000517525, 0.000521164, 0.000526838, 0.000530932, 0.000538273,
0.000543949, 0.000547949, 0.000557152, 0.000561684, 0.000566815,
0.000572831, 0.000578346, 0.000583322, 0.000588881, 0.000594856,
0.000604063, 0.000611878, 0.000616038, 0.000625048, 0.000632283,
0.000638615, 0.000646415, 0.000656107, 0.000663472, 0.000669256,
0.0006775, 0.000686414, 0.000697483, 0.000704603, 0.000709929,
0.000719191, 0.000728828, 0.000736136, 0.000742434, 0.000753303,
0.000761699, 0.000771737, 0.000781114, 0.000790353, 0.000800866,
0.00081076, 0.000818651, 0.000830237, 0.000839178, 0.000849784,
0.000858741, 0.000867707, 0.000878672, 0.000889377, 0.000898416,
0.000907651, 0.000919457, 0.000930054, 0.000945049, 0.000956511,
0.000965677, 0.000979332, 0.000988167, 0.00100013, 0.001014508,
0.001027825, 0.001035209, 0.001047369, 0.001058546, 0.0010707,
0.001079348, 0.001088831, 0.001103408, 0.001116266, 0.00113147,
0.001140678, 0.001154093, 0.001165199, 0.001181722, 0.001194202,
0.001213794, 0.001231295, 0.001246247, 0.001258361, 0.001274188,
0.001290861, 0.001306783, 0.001320243, 0.001333606, 0.001354469,
0.001375078, 0.001389923, 0.001405028, 0.001420399, 0.001437746,
0.001453279, 0.001471657, 0.001486939, 0.00150243, 0.001520999,
0.001541187, 0.001560932, 0.001582043, 0.001596843, 0.001612566,
0.001631022, 0.001653752, 0.001674666, 0.001695299, 0.001711676,
0.001725655, 0.00175016, 0.001764389, 0.001779125, 0.001783268,
0.001804581, 0.001823921, 0.001844709, 0.001865129, 0.001886846,
0.001909372, 0.001927753, 0.001952632, 0.001976229, 0.002013512,
0.002039507, 0.002062143, 0.002082904, 0.002100875, 0.002127102,
0.002153903, 0.002180398, 0.002205838, 0.002227415, 0.002251702,
0.002274242, 0.002296068, 0.002322629, 0.002351514, 0.002378494,
0.002404313, 0.002435294, 0.002466558, 0.002497381, 0.002520757,
0.002555872, 0.002586149, 0.002620634, 0.002648223, 0.002684808,
0.002717046, 0.002743266, 0.002777663, 0.0028076, 0.002838148,
0.002865398, 0.002894208, 0.002924336, 0.002964645, 0.003004024,
0.00303525, 0.003075351, 0.003113435, 0.003150581, 0.003188933,
0.003223459, 0.003254621, 0.003287804, 0.003325633, 0.003360841,
0.003400504, 0.003428985, 0.003463445, 0.003500397, 0.003542617,
0.003584709, 0.003628508, 0.003670112, 0.003716181, 0.003760418,
0.003807062, 0.003845951, 0.003889573, 0.003937353, 0.003977666,
0.00401694, 0.004072232, 0.004118754, 0.00416505, 0.004214084,
0.004260224, 0.004310062, 0.004355524, 0.00440579, 0.004455118,
0.004511028, 0.004561812, 0.004611664, 0.00466489, 0.004720059,
0.004775932, 0.004833971, 0.004890741, 0.004944213, 0.005005854,
0.00505933, 0.005120248, 0.005168977, 0.005232266, 0.005295132,
0.005348358, 0.005422287, 0.005483878, 0.005544922, 0.005605788,
0.005665343, 0.005734263, 0.005802923, 0.0058766, 0.005941258,
0.006009913, 0.006073892, 0.006144848, 0.006219229, 0.0063024,
0.006371055, 0.006450011, 0.006522412, 0.006599935, 0.006681315,
0.006760397, 0.006839474, 0.006915885, 0.006993508, 0.007068701,
0.0071574, 0.007231024, 0.007312241, 0.007391797, 0.007481554,
0.00756224, 0.007648281, 0.007734631, 0.007817361, 0.007917345,
0.008001533, 0.008083505, 0.008174944, 0.008280894, 0.008384592,
0.008487807, 0.008569222, 0.008657994, 0.008750164, 0.008858198,
0.008958549, 0.009072711, 0.009183631, 0.009288004, 0.009387484,
0.009487533, 0.009592086, 0.009694136, 0.009805331, 0.009916848,
0.010020197, 0.010142106, 0.010262439, 0.010384808, 0.010507209,
0.01062309, 0.010738243, 0.010865318, 0.010995926, 0.011133057,
0.011254801, 0.011389623, 0.011528398, 0.011666389, 0.011808989,
0.011937685, 0.012078173, 0.012219025, 0.012374169, 0.012527662,
0.012664523, 0.012818783, 0.012971324, 0.013128624, 0.013279506,
0.013432013, 0.01359048, 0.013747251, 0.013916016, 0.014075929,
0.014237417, 0.014409722, 0.01457517, 0.014738734, 0.014903102,
0.01507266, 0.015242032, 0.015432491, 0.015608658, 0.015778218,
0.015956313, 0.016135642, 0.016320512, 0.016504368, 0.01669306,
0.016887745, 0.017067288, 0.017257689, 0.017458599, 0.017618047,
0.017654392, 0.017850886, 0.018055074, 0.018256924, 0.018471204,
0.018691827, 0.018911858, 0.019132161, 0.019370125, 0.019601175,
0.0198019, 0.020023205, 0.020247257, 0.020483355, 0.020718216,
0.02096615, 0.021213161, 0.021444918, 0.02168673, 0.021943998,
0.022189844, 0.022438014, 0.022692875, 0.022953768, 0.023194174,
0.023466894, 0.023741721, 0.024003462, 0.024273669, 0.024548145,
0.024832661, 0.025123599, 0.025413644, 0.025713194, 0.02601006,
0.026311145, 0.026601403, 0.026907793, 0.027223156, 0.027520955,
0.02782744, 0.028161537, 0.028479236, 0.028782152, 0.029112315,
0.029444583, 0.029766705, 0.030123068, 0.030485933, 0.030857724,
0.031210782, 0.031567092, 0.031915004, 0.032263462, 0.032645941,
0.033012303, 0.033392447, 0.033756059, 0.034129974, 0.034521928,
0.03490533, 0.035304913, 0.035729048, 0.036145387, 0.036570441,
0.036989106, 0.037418021, 0.037851025, 0.038265821, 0.038690539,
0.039133699, 0.03958711, 0.040020153, 0.040497763, 0.040962912,
0.041423189, 0.0419003, 0.042392843, 0.042871775, 0.043339449,
0.04384165, 0.044348869, 0.044833052, 0.045360822, 0.045888484,
0.046420857, 0.046937915, 0.047488362, 0.048039346, 0.048603466,
0.049158463, 0.049693169, 0.050272649, 0.050840273, 0.051430164,
0.05198366, 0.052557767, 0.053164899, 0.053777287, 0.054405409,
0.055013734, 0.055632139, 0.056265096, 0.056922861, 0.057542263,
0.058216067, 0.058849992, 0.059524279, 0.060200388, 0.060877299,
0.06154046, 0.062244876, 0.062939094, 0.063664605, 0.064385143,
0.065100718, 0.065819771, 0.066573088, 0.067351128, 0.068140771,
0.0688953, 0.0696671, 0.070465232, 0.071261676, 0.072078251,
0.072883268, 0.073711564, 0.074567365, 0.07542192, 0.076259147,
0.077137372, 0.077985882, 0.078867069, 0.079771519, 0.080677168,
0.081583181, 0.082529855, 0.083441874, 0.084396566, 0.085339836,
0.086325387, 0.087318889, 0.088289864, 0.089263867, 0.090260613,
0.091245464, 0.092283721, 0.093309256, 0.094372035, 0.095455169,
0.096523546, 0.097616397, 0.098683139, 0.099755491, 0.100884681,
0.102045793, 0.103192646, 0.104378571, 0.105545575, 0.106778666,
0.107947284, 0.109160163, 0.110364112, 0.111557532, 0.11281333,
0.114094046, 0.115375528, 0.116664101, 0.117980128, 0.119260457,
0.120615258, 0.121921045, 0.123273521, 0.124665084, 0.126109963,
0.127548186, 0.128973472, 0.130398775, 0.131875454, 0.133349175,
0.134797574, 0.136330065, 0.137839837, 0.139393438, 0.140900964,
0.142464936, 0.144034355, 0.145642255, 0.147296223, 0.14890239,
0.15054162, 0.152200565, 0.153891336, 0.155563442, 0.157199276,
0.158911532, 0.160702021, 0.162481332, 0.164252507, 0.16604854,
0.167455813, 0.167835743, 0.169646917, 0.17148661, 0.173385494,
0.175241662, 0.1771372, 0.17905781, 0.180999398, 0.182956279,
0.184955185, 0.186974091, 0.189022823, 0.191037098, 0.193148419,
0.195206647, 0.197277391, 0.199355381, 0.201460009, 0.203645814,
0.205829662, 0.208047197, 0.210236284, 0.21247216, 0.214723947,
0.217015038, 0.219382442, 0.221686073, 0.224076367, 0.226476662,
0.228877142, 0.23129064, 0.233715918, 0.236188279, 0.238663389,
0.241217406, 0.24377106, 0.24636134, 0.248979605, 0.251597868,
0.254271087, 0.25693589, 0.259607447, 0.26232757, 0.265049731,
0.267817998, 0.270607806, 0.273456083, 0.276296098, 0.279147813,
0.282039394, 0.284960107, 0.287879685, 0.290796251, 0.293758423,
0.29677992, 0.299799993, 0.30287189, 0.305944694, 0.309053736,
0.312240621, 0.315392445, 0.318586034, 0.321785946, 0.325000891,
0.328269661, 0.331561527, 0.334816618, 0.338108058, 0.341347357,
0.344717983, 0.348110507, 0.351506647, 0.35489934, 0.358322916,
0.36175357, 0.365181081, 0.368676051, 0.372181411, 0.375682156,
0.379189192, 0.38267535, 0.386233732, 0.389732671, 0.393255781,
0.396773897, 0.400191642, 0.403621087, 0.407080955, 0.410587809,
0.414089443, 0.4176846, 0.421263562, 0.42500382, 0.428701138,
0.432484759, 0.436314009, 0.437129936, 0.440170684, 0.444088594,
0.448041803, 0.452085706, 0.456203167, 0.460330988, 0.464584055,
0.468785407, 0.473099838, 0.477436708, 0.481915165, 0.486393483,
0.490917553, 0.495466923, 0.50018802, 0.504887294, 0.509687129,
0.514541049, 0.519476109, 0.524544804, 0.52961961, 0.534683885,
0.539891003, 0.545214329, 0.550583655, 0.55167832, 0.555960002,
0.561414672, 0.566957344, 0.572538034, 0.5782339, 0.584027025,
0.589969188, 0.595893181, 0.601880301, 0.607975492, 0.614155629,
0.620443132, 0.626830721, 0.633250968, 0.639830353, 0.646450941,
0.653147649, 0.659971221, 0.666876303, 0.673860173, 0.680951186,
0.688225806, 0.695558795, 0.702920456, 0.710350724, 0.717983159,
0.725646298, 0.733459857, 0.741297232, 0.74929371, 0.757447067,
0.765714116, 0.773982596, 0.782442721, 0.791086879, 0.794735839,
0.799767722, 0.808591422, 0.817531539, 0.826498955, 0.835666556,
0.845033809, 0.854484607, 0.864122465, 0.873895621, 0.883795433,
0.893826403, 0.904018922, 0.914338122, 0.924807826, 0.935348082,
0.946080121, 0.957004636, 0.968120733, 0.979293621, 0.990710669,
1.002152242, 1.013938448, 1.025836324, 1.037968805, 1.050379702,
1.055510632, 1.062940458, 1.075761536, 1.088713978, 1.102037733,
1.11551473, 1.129153826, 1.143193984, 1.157441488, 1.171817283,
1.186501413, 1.201283837, 1.216374545, 1.231720581, 1.247168716,
1.2630186, 1.279144542, 1.29549473, 1.311920698, 1.328714083,
1.342254692, 1.37276528, 1.389734605, 1.406376996, 1.422858586,
1.439261387, 1.455559995, 1.471701219, 1.487624216, 1.504099111,
1.520962322, 1.537927972, 1.554957242, 1.572170121, 1.589748625,
1.60733102, 1.62501999, 1.642645517, 1.660150481, 1.670594148,
1.677755435, 1.695494183, 1.713110716, 1.730813853, 1.74844742,
1.766158693, 1.783694196, 1.801198584, 1.81882292, 1.836349178,
1.853861575, 1.871252549, 1.888808531, 1.906178158, 1.923555864,
1.940772901, 1.958078526, 1.975126557, 1.992199957, 2.009170137,
2.026089156, 2.042907185, 2.059687801, 2.076413749, 2.093014696,
2.102564189, 2.10952883, 2.125734251, 2.142035196, 2.158254424,
2.174484512, 2.190435419, 2.206415883, 2.222261112, 2.238041843,
2.253790013, 2.270125665, 2.2869378, 2.304000168, 2.321251373,
2.338715982, 2.356567111, 2.374565281, 2.392621909, 2.410829225,
2.429360718, 2.447821355, 2.466577153, 2.485541683, 2.504538568,
2.52388001, 2.543186692, 2.562646018, 2.582183464, 2.602012007,
2.622035545, 2.642375485, 2.662798792, 2.683120273, 2.70389943,
2.724580066, 2.741279991, 2.745695705, 2.766849489, 2.788026793,
2.809441135, 2.831283085, 2.853173826, 2.87531901, 2.897677823,
2.92051093, 2.943731731, 2.967106264, 2.991164615, 3.01561854,
3.040090934, 3.065436019, 3.090857625, 3.11792974, 3.145702342,
3.174686803, 3.206109241, 3.242212091, 3.279958988, 3.318669125,
3.35675389, 3.395459013, 3.425303655, 3.433755192, 3.471795255,
3.510981667, 3.548931951, 3.587523562, 3.626248745, 3.664812295,
3.703558356, 3.741761853, 3.780723526, 3.819654249, 3.857261047,
3.895989087, 3.934687348, 3.973056781, 4.011760826, 4.050490209,
4.088920944, 4.12721078, 4.165607436, 4.204421231, 4.243107593,
4.281935771, 4.321020017, 4.359378514, 4.398161397, 4.436673459,
4.475346417, 4.514077787, 4.552779139, 4.590818137, 4.628936666,
4.667406302, 4.705245663, 4.743356005, 4.781670633, 4.819955615,
4.858219507, 4.89560808, 4.93398262, 4.971425302, 5.009823753,
5.047578472, 5.085366075, 5.122947747, 5.159754386, 5.197300653,
5.234401093, 5.271903615, 5.30932371, 5.346110494, 5.383855278,
5.420643283, 5.45691495, 5.49388409, 5.530876045, 5.568373272,
5.605020768, 5.641716765, 5.678631587, 5.714879842, 5.751399892,
5.787478902, 5.823324743, 5.859400263, 5.89573311, 5.931866309,
5.967315962, 6.002580034, 6.037343653, 6.073066232, 6.108606441,
6.143113798, 6.178860221, 6.213273289, 6.247685564, 6.282528508,
6.316634316, 6.350582099, 6.384855859, 6.419133886, 6.453196623,
6.486968657, 6.520390854, 6.553850781, 6.58762841, 6.62072357,
6.654454683, 6.687042747, 6.720140895, 6.753075689, 6.786007897,
6.818330672, 6.850696084, 6.882542584, 6.914514778, 6.920744908,
6.946062613, 6.977247848, 7.008693225, 7.040543945, 7.072631739,
7.104919678, 7.136144364, 7.167093211, 7.197721754, 7.228185563,
7.259591738, 7.290630867, 7.320765425, 7.350420099, 7.380729583,
7.410480922, 7.440375107, 7.470130295, 7.499675875, 7.528809622,
7.558227233, 7.587029086, 7.616504966, 7.646071742, 7.675216172,
7.704100494, 7.732842997, 7.761607755, 7.790020541, 7.818283308,
7.845561353, 7.873713836, 7.900969888, 7.928157082, 7.955739041,
7.982874476, 8.009472991, 8.036645365, 8.063401152, 8.089832337,
8.116336355, 8.142869001, 8.17025125, 8.197095154, 8.222808316,
8.24874018, 8.273764856, 8.300440508, 8.326219988, 8.35176005,
8.377491624, 8.402435997, 8.427633164, 8.452325953, 8.477645638,
8.501716175, 8.525995915, 8.550201563, 8.574298136, 8.59843595,
8.621858836, 8.646143919, 8.67026502, 8.693772789, 8.717514433,
8.740145234, 8.763226511, 8.786932128, 8.810492184, 8.834823494,
8.857786718, 8.880190171, 8.902835238, 8.925810872, 8.947944605,
8.970204753, 8.991645846, 9.013430866, 9.034996353, 9.057113423,
9.07825175, 9.100504593, 9.121696445, 9.141974576, 9.162970039,
9.183334749, 9.204625212, 9.225603258, 9.246304542, 9.266736556,
9.287121497, 9.307729444, 9.328254872, 9.348347609, 9.367479725,
9.38786811, 9.407259712, 9.42673883, 9.446466403, 9.46557218,
9.485179078, 9.504749436, 9.524648668, 9.54293282, 9.561606294,
9.580496174, 9.59886285, 9.616491842, 9.635072574, 9.652969972,
9.671613141, 9.689034598, 9.70668887, 9.724031018, 9.74136574,
9.759116461, 9.777243599, 9.794866027, 9.811702549, 9.828521612,
9.844996506, 9.8616524, 9.878359138, 9.895180109, 9.912174651,
9.928179671, 9.94417254, 9.960673555, 9.976505496, 9.993406667,
10.009538752, 10.025753523, 10.041218527, 10.057201878, 10.072975631,
10.088614439, 10.104376473, 10.119590128, 10.135119874, 10.150882156,
10.165525367, 10.180417733, 10.194456372, 10.208753399, 10.22415649,
10.238400653, 10.253334916, 10.267356154, 10.28210723, 10.296336821,
10.310601345, 10.324935848, 10.338546803, 10.352776139, 10.366328338,
10.380284198, 10.393722178, 10.407099142, 10.42045738, 10.433365467,
10.447003241, 10.460593603, 10.47386568, 10.486927719, 10.498756024,
10.511531486, 10.524387582, 10.536882423, 10.548527626, 10.559648315,
10.570929166, 10.583391389, 10.595693271, 10.608067665, 10.62078206,
10.633188921, 10.644967279, 10.656031985, 10.667287819, 10.679087849,
10.690858076, 10.701695364, 10.712756851, 10.723045825, 10.734107949,
10.744611979, 10.755455037, 10.765965181, 10.777344592, 10.787781083,
10.798300089, 10.809227923, 10.819885359, 10.830846769, 10.840959665,
10.850784263, 10.859543045, 10.870194004, 10.880517957, 10.890061988,
10.89168529, 10.899771969, 10.909850907, 10.919490066, 10.929393884,
10.939043845, 10.948618931, 10.958295406, 10.968500347, 10.977998034,
10.987921108, 10.997532215, 11.006245954, 11.015476259, 11.024436626,
11.033370529, 11.041720773, 11.050045983, 11.0588774, 11.067330795,
11.076222292, 11.084880651, 11.093373352, 11.101687884, 11.109735978,
11.118289799, 11.126544854, 11.135637301, 11.144071032, 11.151828526,
11.159715303, 11.167161328, 11.174210972, 11.182480422, 11.190805113,
11.19875723, 11.206380996, 11.214922734, 11.221970046, 11.229348718,
11.236955829, 11.244222973, 11.251366929, 11.258770994, 11.266545261,
11.273657295, 11.280772842, 11.28811669, 11.294854833, 11.302478746,
11.309885333, 11.316523137, 11.323968751, 11.330713822, 11.337383463,
11.343992931, 11.350677552, 11.357642953, 11.364173773, 11.371159633,
11.378051491, 11.383806287, 11.389921657, 11.396705783, 11.402669756,
11.408633341, 11.415186452, 11.421165969, 11.426853434, 11.432273971,
11.438219106, 11.444554247, 11.451025914, 11.45696375, 11.463144051,
11.46904288, 11.474695084, 11.48071666, 11.48649488, 11.492379023,
11.497818848, 11.503649658, 11.508979531, 11.515022769, 11.520822264,
11.526789423, 11.533095325, 11.537919384, 11.543665853, 11.548865279,
11.553872003, 11.55864614, 11.563567127, 11.568388872, 11.573499955,
11.578913886, 11.583699882, 11.588243417, 11.592840986, 11.597966989,
11.602540464, 11.607639707, 11.612704392, 11.617439798, 11.622121866,
11.627121148, 11.631635279, 11.635050587, 11.639619272, 11.644076607,
11.647965112, 11.652341674, 11.656825259, 11.661720021, 11.666116593,
11.670753873, 11.675386303, 11.679152753, 11.683427101, 11.68761985,
11.691626984, 11.69602507, 11.700012695, 11.703674537, 11.707790994,
11.711953543, 11.716118663, 11.720709495, 11.724531684, 11.729097126,
11.732718966, 11.736733319, 11.740221506, 11.744020131, 11.747476446,
11.751837503, 11.755516125, 11.759256374, 11.763848731, 11.767239744,
11.771003887, 11.774936939, 11.778854116, 11.782648785, 11.785813552,
11.788990487, 11.792570748, 11.795924586, 11.79921413, 11.802408092,
11.805632093, 11.809186896, 11.81237092, 11.816021202, 11.819252806,
11.822448544, 11.826063892, 11.829670577, 11.832480832, 11.835469398,
11.838410273, 11.841293234, 11.844588347, 11.847104181, 11.849819607,
11.852890845, 11.856338286, 11.858991833, 11.861917869, 11.864778526,
11.867177281, 11.870228201, 11.873394798, 11.87638001, 11.879076471,
11.882044134, 11.885158075, 11.888082014, 11.890913514, 11.893170553,
11.895773593, 11.898124843, 11.900852511, 11.903289533, 11.905326539,
11.907902742, 11.910533308, 11.913553032, 11.915425503, 11.917671543,
11.920271324, 11.922935745, 11.925046526, 11.92747901, 11.929552435,
11.931732513, 11.933890218, 11.936161678, 11.939125542, 11.941633946,
11.944317733, 11.944839283, 11.946642755, 11.948507574, 11.950721877,
11.95232785, 11.954469619, 11.956887419, 11.959088347, 11.961262874,
11.963483908, 11.965558582, 11.967777326, 11.969750876, 11.971889937,
11.973826349, 11.975993186, 11.977993291, 11.980559967, 11.982998958,
11.984676614, 11.986882544, 11.988906996, 11.990627954, 11.992339515,
11.994140723, 11.99611366, 11.997936976, 11.999313313, 12.00078635,
12.002758376, 12.0045195, 12.006181924, 12.008206282, 12.010124401,
12.012084316, 12.013776784, 12.015409671, 12.017028808, 12.018417237,
12.020709917, 12.022546222, 12.02469638, 12.02633059, 12.028035333,
12.029396351, 12.030815478, 12.032528913, 12.034182196, 12.03545008,
12.037521823, 12.039183931, 12.041100911, 12.043122247, 12.044126667,
12.045457052, 12.047272456, 12.048988151, 12.050479656, 12.052016453,
12.053189401, 12.05464071, 12.055909128, 12.057186844, 12.057981798,
12.059420399, 12.060698348, 12.061876957, 12.063203118, 12.064891897,
12.066407561, 12.067420731, 12.068841843, 12.069685773, 12.071027635,
12.072317558, 12.073661757, 12.074845525, 12.076196753, 12.077598994,
12.078759512, 12.079983522, 12.081136295, 12.082789495, 12.083706977,
12.085259971, 12.086598381, 12.087548508, 12.088395038, 12.089482956,
12.090848552, 12.092192865, 12.093646172, 12.094470638, 12.095606161,
12.096772426, 12.09800063, 12.09942223, 12.100432654, 12.101428001,
12.102580101, 12.103609614, 12.104508214, 12.105899357, 12.106785153,
12.10775748, 12.108510362, 12.109476402, 12.110026756, 12.111082444,
12.112167034, 12.112991932, 12.114333712, 12.114928301, 12.116055919,
12.116721871, 12.117732397, 12.11884389, 12.11959541, 12.120530849,
12.121327681, 12.122068343, 12.122904473, 12.123409116, 12.12437726,
12.12542487, 12.126716395, 12.12777665, 12.128533811, 12.129230101,
12.130401991, 12.131541121, 12.132351733, 12.133045223, 12.133855134,
12.134785778, 12.135422587, 12.136074632, 12.13699641, 12.137888432,
12.138430212, 12.139422156, 12.139754324, 12.140701545, 12.141457708,
12.142606578, 12.143388033, 12.144133098, 12.144713033, 12.145435051,
12.146154356, 12.147036904, 12.148170127, 12.148453467, 12.14957943,
12.150441591, 12.151307603, 12.152437939, 12.152898164, 12.153353196,
12.15384459, 12.154590847, 12.155571826, 12.156155956, 12.156980789,
12.157615521, 12.158254376, 12.159022041, 12.159593026, 12.160416237,
12.161072634, 12.16200385, 12.162330447, 12.162980004, 12.163542662,
12.164175299, 12.164594985, 12.165190881, 12.16583361, 12.166210132,
12.166843129, 12.167625381, 12.168217752, 12.168658755, 12.16937553,
12.169799289, 12.17066088, 12.171702806, 12.171893359, 12.172603872,
12.173156918, 12.173938245, 12.174375685, 12.174689971, 12.175546494,
12.176120808, 12.176617496, 12.1770984, 12.177406497, 12.177977614,
12.17881195, 12.179805638, 12.180206143, 12.180688186, 12.180956316,
12.181325746, 12.181798968, 12.182965421, 12.183929354, 12.184479075,
12.184643527, 12.185140212, 12.185515954, 12.186039097, 12.186313845,
12.186978215, 12.187548552, 12.188467088, 12.1890111, 12.189568472,
12.18972195, 12.190223168, 12.190326266, 12.190778274, 12.190958881,
12.191588233, 12.191886642, 12.192104822, 12.192583484, 12.192931218,
12.19327084, 12.193527812, 12.194158978, 12.194648394, 12.194840662,
12.195569546, 12.196055701, 12.196239325, 12.196482815, 12.196835701,
12.197062238, 12.197269344, 12.197636324, 12.197902937, 12.198609671,
12.198898657, 12.199142544, 12.199346771, 12.199454728, 12.199843578,
12.200170625, 12.200548667, 12.201544611, 12.202700425, 12.202890685,
12.203221392, 12.203392576, 12.203803964, 12.204178636, 12.204495409,
12.204963434, 12.205253187, 12.205367933, 12.20551096, 12.205614496,
12.205977088, 12.206192557, 12.207018414, 12.20722794, 12.207628233,
12.207958388, 12.208287022, 12.208518169, 12.20860066, 12.209043961,
12.20926696, 12.209560283, 12.209843851, 12.210170631, 12.210404496,
12.210748197, 12.211071123, 12.211443952, 12.211841725, 12.2118983,
12.212017228, 12.212255364, 12.212579705, 12.212956036, 12.212961761,
12.213025412, 12.213711974, 12.213935289, 12.214304625, 12.214827171,
12.215180751, 12.215255984, 12.215736858, 12.215969605, 12.216021249,
12.216374261, 12.216910076, 12.216945196, 12.217198752, 12.217282665,
12.217497456, 12.217508068, 12.218441066, 12.218616399, 12.218885129,
12.219043, 12.219227705, 12.219339207, 12.219832854, 12.220223861,
12.22031109, 12.220495629, 12.220579302, 12.220664422, 12.220711628,
12.220853979, 12.220945108, 12.221080935, 12.221500277, 12.221973279,
12.222450244, 12.222635942, 12.222766767, 12.22285714, 12.222883223,
12.223183593, 12.223373956, 12.223523855, 12.223634285, 12.223913065,
12.22395152, 12.223995485, 12.224191992, 12.224262452, 12.224509264,
12.224589308, 12.224610402), "6" = c(3.1358e-05, 3.199e-05, 3.2215e-05,
3.2604e-05, 3.2964e-05, 3.43e-05, 3.4741e-05, 3.4961e-05, 3.5372e-05,
3.5422e-05, 3.5568e-05, 3.5881e-05, 3.6385e-05, 3.6488e-05, 3.6659e-05,
3.7058e-05, 3.7107e-05, 3.7398e-05, 3.7499e-05, 3.8212e-05, 3.8752e-05,
3.8811e-05, 3.8866e-05, 4.0702e-05, 4.0805e-05, 4.0869e-05, 4.115e-05,
4.1226e-05, 4.2528e-05, 4.2867e-05, 4.2897e-05, 4.5052e-05, 4.5384e-05,
4.5909e-05, 4.6076e-05, 4.6386e-05, 4.6524e-05, 4.6802e-05, 4.7191e-05,
4.7556e-05, 4.8027e-05, 4.8352e-05, 4.8502e-05, 4.9601e-05, 5.0181e-05,
5.0304e-05, 5.1075e-05, 5.1409e-05, 5.1589e-05, 5.249e-05, 5.3015e-05,
5.3929e-05, 5.421e-05, 5.4614e-05, 5.5116e-05, 5.5548e-05, 5.6369e-05,
5.7154e-05, 5.7758e-05, 5.8629e-05, 5.925e-05, 6.0019e-05, 6.1106e-05,
6.1425e-05, 6.1855e-05, 6.224e-05, 6.3563e-05, 6.407e-05, 6.5906e-05,
6.6758e-05, 6.7046e-05, 6.7223e-05, 6.8348e-05, 6.8514e-05, 6.9641e-05,
7.0231e-05, 7.1479e-05, 7.2577e-05, 7.3317e-05, 7.3954e-05, 7.4817e-05,
7.5618e-05, 7.5813e-05, 7.7122e-05, 7.8087e-05, 7.9237e-05, 8.012e-05,
8.0243e-05, 8.1779e-05, 8.231e-05, 8.4462e-05, 8.5127e-05, 8.5905e-05,
8.7885e-05, 8.9497e-05, 9.1079e-05, 9.2526e-05, 9.3303e-05, 9.4744e-05,
9.5542e-05, 9.6375e-05, 9.8544e-05, 9.902e-05, 9.9332e-05, 0.000100327,
0.000101334, 0.000101843, 0.000105041, 0.00010553, 0.000106943,
0.000108932, 0.000111273, 0.000112326, 0.000113296, 0.000114312,
0.00011713, 0.000120026, 0.000122239, 0.000124341, 0.000125049,
0.000125785, 0.00012669, 0.000127089, 0.00012754, 0.00012958,
0.000130862, 0.000131803, 0.000133833, 0.000134431, 0.000136624,
0.000137298, 0.000138366, 0.000139129, 0.000140502, 0.000142434,
0.000145286, 0.000146587, 0.000147148, 0.000149636, 0.000150534,
0.000153821, 0.000155387, 0.000157036, 0.000158884, 0.000160799,
0.000162778, 0.000164749, 0.000166989, 0.000169952, 0.000170893,
0.00017248, 0.000174911, 0.000176752, 0.000178576, 0.000181883,
0.000185364, 0.000187306, 0.000188943, 0.000190169, 0.000191289,
0.000194582, 0.000196643, 0.000199103, 0.000200883, 0.000202463,
0.000205172, 0.000209269, 0.000211035, 0.000214002, 0.000217027,
0.000219418, 0.000221686, 0.000224033, 0.000226431, 0.000228844,
0.000232905, 0.000234994, 0.000236859, 0.000239332, 0.000242202,
0.000244334, 0.000248079, 0.000251063, 0.000253493, 0.000257094,
0.000258939, 0.000263413, 0.00026642, 0.00026928, 0.000272847,
0.000275031, 0.000279454, 0.000282502, 0.000287279, 0.000288813,
0.000292289, 0.000295726, 0.000298568, 0.000300983, 0.000305033,
0.000309459, 0.000311539, 0.000313804, 0.000317825, 0.000324706,
0.000327784, 0.000330988, 0.000333835, 0.000337482, 0.000342636,
0.000346308, 0.000350424, 0.000354095, 0.000356466, 0.000361552,
0.000367086, 0.000370716, 0.000376316, 0.000382317, 0.00038593,
0.000391478, 0.000397068, 0.000399942, 0.000405609, 0.000409808,
0.000414337, 0.000419119, 0.000422939, 0.00042844, 0.000433489,
0.000438026, 0.000442673, 0.000447499, 0.00045592, 0.000460823,
0.000468677, 0.000473994, 0.000477837, 0.000481731, 0.000486912,
0.000494621, 0.000502888, 0.000510092, 0.000516883, 0.000524098,
0.000530554, 0.000536137, 0.000540917, 0.000546026, 0.000551482,
0.000558451, 0.000563719, 0.000571731, 0.000577546, 0.000584718,
0.000594017, 0.000600639, 0.000611098, 0.000618041, 0.000625433,
0.000633574, 0.000639748, 0.000648174, 0.000654127, 0.000663736,
0.000670907, 0.000678348, 0.000687715, 0.000694103, 0.00070408,
0.000713057, 0.000721742, 0.000731484, 0.000738256, 0.000747753,
0.000756947, 0.000767129, 0.000775268, 0.00078463, 0.000797949,
0.000806928, 0.000815876, 0.000826115, 0.000836937, 0.000847243,
0.000855766, 0.000863722, 0.000875365, 0.000882285, 0.000895156,
0.000901138, 0.000911201, 0.000922752, 0.00093168, 0.000942428,
0.000953781, 0.000966221, 0.000980283, 0.000988012, 0.001004412,
0.001015536, 0.001026814, 0.001042387, 0.001052762, 0.001062743,
0.001073587, 0.001086986, 0.001100474, 0.001114898, 0.001127425,
0.00114134, 0.00115469, 0.001168244, 0.00118286, 0.001197807,
0.001211545, 0.001226369, 0.001242923, 0.001253856, 0.00127046,
0.001284516, 0.001302275, 0.001321528, 0.001334926, 0.001352586,
0.001369274, 0.001386095, 0.001401396, 0.001415442, 0.001434787,
0.001448325, 0.001465383, 0.001478626, 0.001493416, 0.001512424,
0.001531111, 0.001550665, 0.001566872, 0.001585362, 0.001603616,
0.001618772, 0.001622189, 0.0016379, 0.001660578, 0.00167862,
0.00169572, 0.001717002, 0.001737744, 0.001755432, 0.001775606,
0.001798234, 0.001819496, 0.001839298, 0.001857857, 0.001882414,
0.001907021, 0.001926812, 0.001949346, 0.001974211, 0.001996867,
0.002021221, 0.002046505, 0.002070506, 0.002092403, 0.002118607,
0.002139654, 0.002165837, 0.002185864, 0.002207371, 0.002233436,
0.002260244, 0.002285607, 0.002313696, 0.002337864, 0.00236019,
0.002388151, 0.002415846, 0.00244324, 0.002473226, 0.002494795,
0.002521855, 0.00255456, 0.002584934, 0.002614476, 0.002642928,
0.002674447, 0.002708406, 0.002744966, 0.002780954, 0.002810208,
0.002838564, 0.002868369, 0.002909759, 0.002939206, 0.002973684,
0.003010544, 0.003043807, 0.003084327, 0.003116059, 0.003160138,
0.003195558, 0.003230986, 0.003267562, 0.003312263, 0.003358702,
0.003397273, 0.003433537, 0.003470614, 0.003518371, 0.003556198,
0.003592701, 0.003635947, 0.00368405, 0.00372651, 0.003762686,
0.003807754, 0.00385522, 0.003899725, 0.003946888, 0.003983832,
0.00402626, 0.004068371, 0.004114358, 0.004164022, 0.004213087,
0.004256585, 0.00430583, 0.004358012, 0.004410824, 0.004463931,
0.00452143, 0.004573049, 0.004629808, 0.004682287, 0.004731446,
0.004784838, 0.004838168, 0.004897421, 0.004956533, 0.005006963,
0.005066982, 0.00512283, 0.005181988, 0.005250067, 0.00530931,
0.005377239, 0.005434665, 0.005499989, 0.005563998, 0.00562873,
0.005690858, 0.005760167, 0.005828701, 0.005891394, 0.005962076,
0.006033243, 0.006097639, 0.006164996, 0.006237171, 0.006303471,
0.006379314, 0.006461709, 0.006534404, 0.006597755, 0.006671983,
0.006752468, 0.006827253, 0.006913804, 0.006998163, 0.007077921,
0.007164246, 0.007239587, 0.007326362, 0.007414885, 0.007508986,
0.007594542, 0.007685408, 0.007777279, 0.007864146, 0.007953763,
0.008046213, 0.008134679, 0.008219248, 0.008311884, 0.008408814,
0.008507289, 0.008602258, 0.008702739, 0.008812805, 0.008912228,
0.009017911, 0.009125268, 0.009238558, 0.009341648, 0.009444282,
0.009557411, 0.009674552, 0.009783983, 0.009893443, 0.010013104,
0.010133875, 0.010255026, 0.010370431, 0.010490746, 0.010624207,
0.010752412, 0.010881084, 0.010997724, 0.01111531, 0.011249304,
0.011386856, 0.011513152, 0.011649222, 0.011777298, 0.011912488,
0.012056016, 0.012192791, 0.012333772, 0.012475704, 0.01262172,
0.012785768, 0.012942093, 0.013097827, 0.013242494, 0.013393291,
0.013542815, 0.013687881, 0.013859042, 0.014000969, 0.014161072,
0.014307929, 0.014469906, 0.014643222, 0.014816175, 0.014998539,
0.015170773, 0.015337204, 0.015501237, 0.015672485, 0.015858415,
0.016030623, 0.016178362, 0.016215504, 0.016396593, 0.016591613,
0.016768061, 0.016980079, 0.017166765, 0.017355176, 0.017542901,
0.017740399, 0.017948503, 0.018152207, 0.018355998, 0.01856894,
0.018775423, 0.018999371, 0.019219394, 0.019425251, 0.01965511,
0.019888081, 0.020111296, 0.020329606, 0.020559568, 0.020789629,
0.021030006, 0.021284409, 0.021526297, 0.021771246, 0.022025405,
0.022272097, 0.022527079, 0.02280131, 0.023049242, 0.023310727,
0.023599445, 0.023864264, 0.024126937, 0.024396768, 0.024656958,
0.02493037, 0.025231085, 0.025517073, 0.025813018, 0.026105055,
0.026402359, 0.026706853, 0.02701713, 0.02732196, 0.027647032,
0.027948432, 0.028259155, 0.028579215, 0.028887198, 0.02922414,
0.029545817, 0.029877508, 0.030211525, 0.030568475, 0.030921872,
0.031268224, 0.031629401, 0.03199219, 0.032354484, 0.032711846,
0.033067724, 0.033446489, 0.033825924, 0.034210033, 0.034595255,
0.034972531, 0.03536243, 0.035759053, 0.036177683, 0.036579119,
0.036993551, 0.037424256, 0.037839047, 0.038276011, 0.038707558,
0.03914052, 0.039583098, 0.040033056, 0.040477639, 0.0409489,
0.041424137, 0.041866004, 0.042322992, 0.042795923, 0.043280602,
0.043757681, 0.044252212, 0.044763138, 0.045290131, 0.045784271,
0.046297535, 0.046823264, 0.047321812, 0.047851715, 0.048406534,
0.048953948, 0.049513915, 0.050070172, 0.050620642, 0.051158871,
0.051754218, 0.052312307, 0.052895072, 0.053477214, 0.054065234,
0.054675623, 0.055272491, 0.055880665, 0.056525101, 0.057148377,
0.057772583, 0.058426521, 0.059047357, 0.059700687, 0.060364097,
0.061015297, 0.061684283, 0.062374565, 0.063071958, 0.06377796,
0.064486051, 0.065189431, 0.065900462, 0.066620365, 0.067353583,
0.068095393, 0.06885572, 0.069633721, 0.070398052, 0.07119047,
0.071974699, 0.072745245, 0.07353188, 0.074317254, 0.075147921,
0.075951652, 0.076799269, 0.077633025, 0.07847584, 0.079325042,
0.08016444, 0.081028111, 0.081912885, 0.082801627, 0.083662253,
0.084552999, 0.085465385, 0.086382252, 0.08731652, 0.08821957,
0.089188922, 0.090138972, 0.091087401, 0.092071924, 0.093010932,
0.093986333, 0.094977364, 0.095965325, 0.096946211, 0.097962154,
0.099016624, 0.100044234, 0.101071376, 0.102108951, 0.103151843,
0.104169369, 0.105247715, 0.106337643, 0.107425966, 0.108549681,
0.109627863, 0.110764897, 0.111881418, 0.113048363, 0.114192813,
0.115365123, 0.116568379, 0.117785141, 0.119002019, 0.120214328,
0.121452633, 0.122674801, 0.123963044, 0.125253194, 0.1265536,
0.127851731, 0.129150895, 0.130452499, 0.131809665, 0.133148279,
0.134563422, 0.135935853, 0.137323133, 0.138743161, 0.140178673,
0.141643681, 0.143102921, 0.14428474, 0.144585384, 0.146064633,
0.147561724, 0.149135387, 0.150705516, 0.152289702, 0.153864596,
0.155436416, 0.15703033, 0.158684074, 0.160321311, 0.16199507,
0.163653702, 0.165327975, 0.167032974, 0.168739884, 0.170508516,
0.172268812, 0.174082475, 0.17585731, 0.177653481, 0.179501595,
0.181365929, 0.183233655, 0.18514769, 0.187101056, 0.189053639,
0.191023557, 0.192996216, 0.195030979, 0.197037404, 0.199012936,
0.201055581, 0.20315131, 0.205301021, 0.207410864, 0.209560116,
0.211799714, 0.214001306, 0.216263103, 0.218517794, 0.220788348,
0.223149574, 0.225526093, 0.227864271, 0.230260082, 0.232756763,
0.235188929, 0.237615219, 0.240098053, 0.242593152, 0.245147146,
0.247704284, 0.250344004, 0.252983703, 0.255704915, 0.258415153,
0.261139904, 0.26384025, 0.266656355, 0.269454278, 0.272283524,
0.275145342, 0.278025202, 0.280889228, 0.283907731, 0.286889664,
0.289941041, 0.293025842, 0.296117632, 0.299274712, 0.302508502,
0.305717984, 0.308926548, 0.312183529, 0.315486499, 0.318839891,
0.322245719, 0.325677067, 0.329172868, 0.332679828, 0.336207989,
0.339830293, 0.343411755, 0.34710179, 0.350806107, 0.354604019,
0.358376402, 0.362228052, 0.366069705, 0.369972243, 0.373867225,
0.377838246, 0.381831872, 0.385894652, 0.390046521, 0.390953072,
0.394198849, 0.398395836, 0.402627118, 0.40693311, 0.41125517,
0.415677971, 0.420141956, 0.424685307, 0.429267491, 0.433878042,
0.43847912, 0.44317343, 0.447887243, 0.452750497, 0.457633927,
0.462528327, 0.467533505, 0.47254767, 0.477597885, 0.482681923,
0.487823623, 0.493109727, 0.498454262, 0.503816645, 0.509230077,
0.510331876, 0.514627251, 0.520179694, 0.525797963, 0.53147953,
0.537149791, 0.54296638, 0.548862868, 0.554770919, 0.560744408,
0.566745705, 0.572901181, 0.579082664, 0.585332978, 0.591630761,
0.598046446, 0.604514322, 0.611098711, 0.617663425, 0.624332178,
0.631150707, 0.637879773, 0.644802997, 0.651688685, 0.658762193,
0.665889952, 0.673092478, 0.680292824, 0.687653482, 0.69508272,
0.702522201, 0.710090337, 0.717724366, 0.725483682, 0.733276872,
0.741123631, 0.744523772, 0.749175397, 0.757277375, 0.765389464,
0.773655983, 0.782006515, 0.790446221, 0.799037926, 0.807683152,
0.816419795, 0.825256251, 0.834246655, 0.843184506, 0.852261068,
0.861466906, 0.870751333, 0.880148681, 0.889716656, 0.899284005,
0.908896767, 0.918673285, 0.92854399, 0.938557144, 0.948585361,
0.958797206, 0.969190839, 0.973484459, 0.979672605, 0.990340093,
1.001161981, 1.012232096, 1.023425326, 1.034719518, 1.04632504,
1.058102299, 1.070011444, 1.082016031, 1.09435498, 1.106828199,
1.119507516, 1.132378671, 1.145303223, 1.1586069, 1.172015508,
1.185814255, 1.199950723, 1.211416412, 1.237306978, 1.251620409,
1.266094463, 1.280472648, 1.294774315, 1.309095562, 1.323301903,
1.337403375, 1.351625445, 1.365843858, 1.380071923, 1.394102256,
1.408186104, 1.422139982, 1.436258509, 1.45009706, 1.464005663,
1.477886583, 1.485998056, 1.491718409, 1.505608823, 1.519537746,
1.533418291, 1.547250357, 1.56096935, 1.574742824, 1.588467967,
1.602065274, 1.615773508, 1.629438976, 1.643074893, 1.656703826,
1.67048235, 1.684233943, 1.69778449, 1.711562168, 1.72519738,
1.738668346, 1.752408779, 1.766076353, 1.779790481, 1.793478187,
1.807169555, 1.820927716, 1.828851938, 1.834686993, 1.848371903,
1.862132179, 1.875943096, 1.889803931, 1.903730641, 1.91773716,
1.931715216, 1.945909864, 1.96012055, 1.974432486, 1.988819792,
2.003513476, 2.018211964, 2.033064261, 2.048389851, 2.064290158,
2.080517128, 2.09698635, 2.113989373, 2.131773028, 2.149857908,
2.168221376, 2.187099741, 2.206256945, 2.225851721, 2.246143178,
2.266609177, 2.287541484, 2.308922044, 2.331149519, 2.353651111,
2.377131243, 2.400908312, 2.425503795, 2.445511057, 2.450716679,
2.476638497, 2.50318257, 2.530272291, 2.557373435, 2.58493795,
2.612547438, 2.640463784, 2.669055946, 2.697861363, 2.72683001,
2.756300212, 2.785901668, 2.815934923, 2.846487264, 2.87737797,
2.90873349, 2.940263634, 2.972317062, 3.004688958, 3.037740406,
3.071353862, 3.105044443, 3.139214207, 3.174109478, 3.201823877,
3.209645293, 3.245716528, 3.282321457, 3.319329941, 3.356381173,
3.394201368, 3.433189924, 3.472424453, 3.511674264, 3.552420325,
3.593752095, 3.635291202, 3.67816185, 3.721014471, 3.763902505,
3.806975119, 3.850802463, 3.894095564, 3.937724678, 3.981477856,
4.026035245, 4.070352726, 4.114677704, 4.158596113, 4.203519252,
4.248606135, 4.293796242, 4.339020451, 4.384867763, 4.429742788,
4.475340175, 4.52148588, 4.567099339, 4.61317506, 4.659170489,
4.705563915, 4.751717005, 4.797306822, 4.843493425, 4.889480644,
4.936329448, 4.982966613, 5.029465536, 5.076526251, 5.122685421,
5.169865319, 5.216677718, 5.263712172, 5.311416797, 5.358789973,
5.406044187, 5.454000023, 5.500891196, 5.548564306, 5.596105395,
5.643472319, 5.692336717, 5.739869868, 5.787590522, 5.835452346,
5.883585915, 5.931261383, 5.979539234, 6.027380898, 6.075539533,
6.122902428, 6.171343447, 6.219081584, 6.267387505, 6.31543678,
6.363637609, 6.412609318, 6.459467785, 6.507645468, 6.556056258,
6.60355698, 6.651697034, 6.699711034, 6.747802459, 6.795248344,
6.84348736, 6.890704669, 6.938877503, 6.987358116, 7.036737005,
7.086255639, 7.134403227, 7.181532981, 7.230241034, 7.278177513,
7.326222116, 7.374200679, 7.422932132, 7.470005553, 7.51792501,
7.565439937, 7.574942038, 7.612558751, 7.660462506, 7.708477531,
7.754314487, 7.800808873, 7.849413885, 7.896443451, 7.944262058,
7.992587911, 8.039199322, 8.08591291, 8.13287917, 8.180237704,
8.227605582, 8.273691108, 8.321146435, 8.368347903, 8.413756924,
8.460510392, 8.507186518, 8.553309921, 8.600598262, 8.64744457,
8.694813423, 8.742570063, 8.789992211, 8.835152611, 8.88169509,
8.927965199, 8.974749292, 9.020500773, 9.066379533, 9.111963751,
9.159341348, 9.205783305, 9.250785576, 9.295622262, 9.342452185,
9.386786681, 9.431732322, 9.476260519, 9.521087114, 9.567043405,
9.612224461, 9.657189095, 9.702061099, 9.747191368, 9.791898059,
9.837009982, 9.88089322, 9.924135071, 9.967877366, 10.011109361,
10.056155434, 10.099599942, 10.143672807, 10.185499409, 10.228488011,
10.272368989, 10.315804865, 10.35983312, 10.403601461, 10.445774596,
10.488111719, 10.530833139, 10.573090363, 10.616459487, 10.658021896,
10.700152911, 10.743135262, 10.784125758, 10.825967949, 10.868603742,
10.909381005, 10.951435069, 10.992962297, 11.035257103, 11.076445774,
11.118304837, 11.158790196, 11.198999805, 11.240475958, 11.282442942,
11.323467823, 11.365433206, 11.405102891, 11.445151842, 11.485050939,
11.524204281, 11.561770464, 11.601193241, 11.641225482, 11.680781889,
11.719315097, 11.758645823, 11.797076224, 11.835749634, 11.875272649,
11.91278869, 11.952315678, 11.991285442, 12.028392465, 12.065397742,
12.103283918, 12.13986447, 12.178211411, 12.215436532, 12.251972379,
12.289632573, 12.327255127, 12.364407016, 12.40245052, 12.439478085,
12.476892999, 12.515170273, 12.551108151, 12.587855993, 12.62568361,
12.661234996, 12.69846867, 12.733031371, 12.770519767, 12.805812585,
12.841980406, 12.876973641, 12.911692418, 12.946238474, 12.980166022,
13.013471219, 13.048670018, 13.083841984, 13.11711976, 13.150714025,
13.184355568, 13.219024484, 13.252616183, 13.284961911, 13.319329111,
13.353231794, 13.386229098, 13.420905995, 13.452826108, 13.485346055,
13.519188365, 13.551998247, 13.585617934, 13.618141954, 13.650848499,
13.683106042, 13.715673292, 13.750399783, 13.782960783, 13.814283336,
13.845678265, 13.875794035, 13.905837791, 13.936702963, 13.967240936,
13.999207995, 14.031790859, 14.061778301, 14.092673621, 14.121659084,
14.1527083, 14.182485426, 14.213159169, 14.243000971, 14.273048605,
14.302563757, 14.331968411, 14.361632026, 14.39326471, 14.422626988,
14.451310207, 14.479986595, 14.509250852, 14.537722527, 14.567453181,
14.597467987, 14.626244329, 14.654681756, 14.683477498, 14.710888154,
14.738242368, 14.76692133, 14.795145266, 14.822718596, 14.850056694,
14.876576984, 14.90625144, 14.93414322, 14.959979689, 14.987092731,
15.012830461, 15.037120239, 15.062627933, 15.088482875, 15.116580408,
15.142875307, 15.168014967, 15.173175383, 15.194140158, 15.220682107,
15.248913729, 15.274293804, 15.301620624, 15.326225018, 15.349587202,
15.374260652, 15.39901838, 15.424383051, 15.450016588, 15.475083315,
15.501929227, 15.528635719, 15.551885627, 15.577342621, 15.601116995,
15.626797741, 15.650473911, 15.67380756, 15.697750772, 15.721586895,
15.746769288, 15.770218185, 15.795552211, 15.817566257, 15.842167215,
15.867048587, 15.888656193, 15.910736274, 15.933287741, 15.95600103,
15.979191671, 16.000590741, 16.022690176, 16.045633872, 16.068144385,
16.089615365, 16.111788391, 16.132191982, 16.153911828, 16.174505806,
16.197898553, 16.218112875, 16.240067172, 16.261699782, 16.283676834,
16.304367881, 16.326151781, 16.347341553, 16.368338311, 16.389025864,
16.40898225, 16.42744778, 16.445121285, 16.463814913, 16.482823531,
16.502912573, 16.522139051, 16.54345999, 16.563114059, 16.582710025,
16.602521548, 16.622217503, 16.640595781, 16.659082694, 16.679101469,
16.698794772, 16.717263719, 16.738028218, 16.755786152, 16.772081148,
16.789092209, 16.807565951, 16.82628045, 16.844605617, 16.863315635,
16.882110912, 16.899998036, 16.919035008, 16.936451778, 16.95386839,
16.974376646, 16.991082742, 17.008976222, 17.027245826, 17.044832081,
17.060835654, 17.080133535, 17.095761574, 17.115368094, 17.132343087,
17.149452636, 17.167334456, 17.185074055, 17.200586724, 17.218287343,
17.236346858, 17.253543795, 17.269716518, 17.285332207, 17.300761435,
17.31524487, 17.332690356, 17.346599086, 17.360626661, 17.3755371,
17.390862409, 17.404333159, 17.4190408, 17.434268738, 17.448927874,
17.465744779, 17.480802774, 17.499558918, 17.514786495, 17.531388122,
17.54909607, 17.56246693, 17.578254932, 17.594646452, 17.60988222,
17.625054914, 17.639335725, 17.653051928, 17.666224688, 17.679884514,
17.694084818, 17.709567309, 17.723563382, 17.738162464, 17.749922381,
17.76434886, 17.778498012, 17.791996104, 17.806420418, 17.82051673,
17.833523458, 17.846410401, 17.860503107, 17.87277778, 17.884837162,
17.898057968, 17.909021697, 17.92109648, 17.933589843, 17.946094622,
17.958736104, 17.96924387, 17.981464751, 17.992149466, 18.001987361,
18.014865349, 18.028235553, 18.040001197, 18.051186397, 18.062810946,
18.077065269, 18.086610612, 18.099308386, 18.110762952, 18.123326584,
18.133501947, 18.146537603, 18.157609657, 18.16895237, 18.181858243,
18.193637066, 18.204363544, 18.215889602, 18.22608638, 18.237327209,
18.247365324, 18.259188041, 18.271868913, 18.282116515, 18.292461586,
18.302295659, 18.312705869, 18.32509411, 18.337313209, 18.347481497,
18.356990282, 18.36684485, 18.378741097, 18.389415564, 18.40176925,
18.413421702, 18.422770719, 18.432889755, 18.441935978, 18.45338354,
18.466252281, 18.476196349, 18.488254874, 18.497697056, 18.509148473,
18.518985249, 18.527443759, 18.537977035, 18.539255886, 18.54595779,
18.554225986, 18.56287057, 18.573434601, 18.580269074, 18.590529938,
18.601775633, 18.611533741, 18.621507406, 18.629178951, 18.636110465,
18.644253967, 18.651786682, 18.662290895, 18.672452103, 18.680970838,
18.690778396, 18.699598866, 18.70819452, 18.714922057, 18.722849945,
18.731540757, 18.741295652, 18.749460393, 18.75856011, 18.767572894,
18.774551068, 18.783624088, 18.790850001, 18.798093074, 18.805584057,
18.816138662, 18.822468611, 18.830264074, 18.837481755, 18.846194734,
18.854074943, 18.859546229, 18.866051476, 18.871934708, 18.879769437,
18.887815052, 18.895745955, 18.903857637, 18.912650662, 18.92109785,
18.930456501, 18.93856076, 18.94747826, 18.954991725, 18.962747609,
18.97161652, 18.981063187, 18.987968724, 18.995455809, 19.000547029,
19.01047806, 19.016565238, 19.02465525, 19.030917333, 19.038013472,
19.04360459, 19.053117023, 19.058621601, 19.067517934, 19.076029748,
19.084042003, 19.090689661, 19.097079234, 19.104895179, 19.111042085,
19.118765764, 19.125627678, 19.133113557, 19.14031122, 19.145591381,
19.151579804, 19.157342784, 19.162798891, 19.171185234, 19.178233253,
19.184394533, 19.190429796, 19.195330823, 19.200271143, 19.205294138,
19.211547063, 19.216687486, 19.223146455, 19.229741886, 19.235181702,
19.243026474, 19.246856429, 19.252820461, 19.257340864, 19.260110082,
19.26500096, 19.272385536, 19.277298107, 19.283493591, 19.288401485,
19.292190536, 19.298108736, 19.305366553, 19.312991988, 19.31799428,
19.322395086, 19.328938498, 19.335072376, 19.340673446, 19.346078737,
19.351111145, 19.355398334, 19.360416096, 19.365708284, 19.369963149,
19.374551567, 19.378388782, 19.381543874, 19.387131349, 19.391142472,
19.398205213, 19.403951196, 19.411908639, 19.416011971, 19.420416503,
19.427128261, 19.430960108, 19.436183075, 19.440020194, 19.443441682,
19.44736406, 19.452223123, 19.455131591, 19.459474399, 19.46439275,
19.468459022, 19.471899205, 19.475355788, 19.479658404, 19.482249201,
19.485787749, 19.488041613, 19.492108141, 19.496696468, 19.501411647,
19.505846332, 19.508593368, 19.511266183, 19.514412194, 19.520130413,
19.523648143, 19.527435941, 19.529259501, 19.533427353, 19.536353512,
19.54224703, 19.546546825, 19.551361256, 19.555176304, 19.557115172,
19.562452156, 19.565730142, 19.568428247, 19.572462325, 19.575795225,
19.579071097, 19.583196925, 19.586494152, 19.58970746, 19.593056804,
19.596392369, 19.598789025, 19.603090899, 19.605910909, 19.608510973,
19.612225167, 19.615806019, 19.619662278, 19.62471347, 19.629282793,
19.633059436, 19.636846656, 19.639239743, 19.641020978, 19.647307512,
19.649423254, 19.650670073, 19.653680881, 19.656097037, 19.661159574,
19.664227598, 19.66734563, 19.669786912, 19.672452073, 19.676458095,
19.67819949, 19.679317961, 19.681669146, 19.684379576, 19.687946555,
19.691300451, 19.697548615, 19.700021226, 19.703673975, 19.705679137,
19.709832946, 19.712101766, 19.715656714, 19.718154876, 19.722325284,
19.72728388, 19.729672549, 19.731440481, 19.733560773, 19.737496326,
19.742843246, 19.744603838, 19.748654559, 19.75215159, 19.754520733,
19.757883357, 19.762410442, 19.765649411, 19.769292818, 19.770329963,
19.774106234, 19.778137536, 19.779885715, 19.78353691, 19.788532389,
19.791349877, 19.794897663, 19.797144684, 19.800203091, 19.801406131,
19.803085318, 19.806866775, 19.809642835, 19.812811466, 19.813253626,
19.818408266, 19.82218051, 19.825508862, 19.82811557, 19.833280911,
19.835106408, 19.838272913, 19.838838997, 19.844974336, 19.846656933,
19.848470314, 19.850671524, 19.852145568, 19.854877295, 19.858323009,
19.859226693, 19.8628968, 19.867283932, 19.870901899, 19.872972253,
19.876780473, 19.87886444, 19.879503669, 19.882044698, 19.884276432,
19.887518847, 19.888947756, 19.890883933, 19.891270276, 19.892829635,
19.897592751, 19.900369866, 19.900686529, 19.903289775, 19.904145471,
19.905207785, 19.905491685, 19.905921869, 19.906523517, 19.909574827,
19.911872845, 19.912724239, 19.91406781, 19.917057504, 19.918558906,
19.923833697, 19.925668188, 19.926230825, 19.92661756, 19.927152421,
19.929193425, 19.933772942, 19.935571024, 19.937695, 19.937985544,
19.942888699, 19.943748712, 19.94573214, 19.947220359, 19.947262021,
19.948872397, 19.950069197, 19.954000708, 19.954939627, 19.957210624,
19.958495312, 19.962567108, 19.963864689, 19.965915548, 19.966076808,
19.968001694, 19.9699005, 19.970797977, 19.971578762, 19.974282802,
19.975223833, 19.975260904, 19.97788226, 19.977949761, 19.979601496,
19.979641128, 19.981626192, 19.98346117, 19.984539273, 19.988127415,
19.988484237, 19.989670448, 19.98975718, 19.991561826, 19.993875471,
19.995170837, 19.997050402, 19.999600556, 20.001787308, 20.003113118,
20.004403039, 20.006901764, 20.007012622, 20.007909499), "8" = c(2.28e-05,
2.2812e-05, 2.316e-05, 2.3589e-05, 2.3888e-05, 2.4379e-05, 2.4737e-05,
2.4778e-05, 2.5034e-05, 2.5156e-05, 2.5391e-05, 2.5695e-05, 2.6095e-05,
2.6302e-05, 2.6341e-05, 2.6589e-05, 2.6728e-05, 2.69e-05, 2.6984e-05,
2.6996e-05, 2.7315e-05, 2.7389e-05, 2.7931e-05, 2.8255e-05, 2.851e-05,
2.8572e-05, 2.9144e-05, 2.919e-05, 2.957e-05, 2.9795e-05, 3.0015e-05,
3.0514e-05, 3.0957e-05, 3.0997e-05, 3.1479e-05, 3.1998e-05, 3.2246e-05,
3.2506e-05, 3.2689e-05, 3.3107e-05, 3.3609e-05, 3.3755e-05, 3.4042e-05,
3.4218e-05, 3.4358e-05, 3.4636e-05, 3.5106e-05, 3.5403e-05, 3.6139e-05,
3.6653e-05, 3.7016e-05, 3.7077e-05, 3.7884e-05, 3.8473e-05, 3.8942e-05,
3.9446e-05, 3.9851e-05, 4.0185e-05, 4.0489e-05, 4.1271e-05, 4.2231e-05,
4.2292e-05, 4.2846e-05, 4.3953e-05, 4.4846e-05, 4.5216e-05, 4.669e-05,
4.6805e-05, 4.8851e-05, 4.9228e-05, 4.9977e-05, 5.03e-05, 5.0763e-05,
5.2006e-05, 5.2528e-05, 5.4103e-05, 5.4594e-05, 5.5708e-05, 5.5897e-05,
5.64e-05, 5.6779e-05, 5.6942e-05, 5.7181e-05, 5.7937e-05, 5.8445e-05,
5.8784e-05, 5.9804e-05, 5.9958e-05, 6.0329e-05, 6.0695e-05, 6.1292e-05,
6.1952e-05, 6.307e-05, 6.3676e-05, 6.3951e-05, 6.5394e-05, 6.622e-05,
6.7122e-05, 6.7834e-05, 6.8521e-05, 7.0807e-05, 7.197e-05, 7.2642e-05,
7.3535e-05, 7.4161e-05, 7.6889e-05, 7.7603e-05, 7.8046e-05, 7.9375e-05,
8.0694e-05, 8.1491e-05, 8.2901e-05, 8.4528e-05, 8.4747e-05, 8.6876e-05,
8.8619e-05, 8.9506e-05, 9.1724e-05, 9.2358e-05, 9.3294e-05, 9.3665e-05,
9.4659e-05, 9.5959e-05, 9.6425e-05, 9.8345e-05, 9.9889e-05, 0.000100734,
0.000101445, 0.000102842, 0.000103659, 0.000105404, 0.000106238,
0.000107681, 0.000108286, 0.000110072, 0.000113073, 0.000114914,
0.000115967, 0.00011756, 0.000119455, 0.000120604, 0.00012227,
0.000123697, 0.000125357, 0.00012738, 0.000128337, 0.000129308,
0.000130703, 0.000132948, 0.000134258, 0.000135436, 0.000136896,
0.000138019, 0.00013941, 0.000140199, 0.000142829, 0.000144537,
0.000146235, 0.000148098, 0.000148963, 0.000149817, 0.000151312,
0.000153105, 0.000154607, 0.000156262, 0.000158941, 0.000161954,
0.000163595, 0.000165669, 0.000167183, 0.000169026, 0.000172253,
0.000173808, 0.000175941, 0.000177368, 0.000179305, 0.000181655,
0.000184148, 0.000186981, 0.000189283, 0.00019147, 0.000194023,
0.000197172, 0.000200484, 0.000203957, 0.000206211, 0.00020853,
0.000211897, 0.000216437, 0.000220409, 0.000222992, 0.000227507,
0.000231039, 0.000238991, 0.0002441, 0.000245779, 0.000247753,
0.000251293, 0.000254987, 0.000258395, 0.000260914, 0.000263864,
0.000265529, 0.000268109, 0.000270642, 0.000274252, 0.000276678,
0.000279451, 0.000282856, 0.000286136, 0.000288264, 0.000291791,
0.000296206, 0.00030079, 0.000304837, 0.000308009, 0.000314644,
0.000318984, 0.000323568, 0.000327327, 0.000332652, 0.00033579,
0.000338794, 0.000342813, 0.000345529, 0.000349641, 0.000353068,
0.000356047, 0.000358884, 0.000364317, 0.000368672, 0.000376388,
0.000379608, 0.000383847, 0.000387733, 0.000392518, 0.000397242,
0.000402386, 0.000408323, 0.000413784, 0.000418106, 0.000423652,
0.000428782, 0.000433451, 0.000437227, 0.000441753, 0.000446415,
0.000453371, 0.000457261, 0.000461481, 0.000468733, 0.000474327,
0.000479882, 0.000486549, 0.000492131, 0.000497539, 0.000504322,
0.000509915, 0.000515301, 0.000520851, 0.00052529, 0.000530835,
0.000537487, 0.0005431, 0.000548461, 0.000553975, 0.000560102,
0.000566405, 0.000572429, 0.000576488, 0.000582965, 0.00058882,
0.000595591, 0.000603294, 0.000611435, 0.000619107, 0.000626033,
0.000631185, 0.000636856, 0.00064634, 0.000654157, 0.000659826,
0.000664616, 0.000671269, 0.000677559, 0.000687098, 0.000694363,
0.000704079, 0.000713013, 0.000722547, 0.000732275, 0.000741766,
0.000750262, 0.000757802, 0.000764858, 0.000774529, 0.000784203,
0.000794508, 0.000802347, 0.000809793, 0.000818032, 0.000825956,
0.000834302, 0.00084253, 0.000849624, 0.000858686, 0.000868091,
0.000875481, 0.000887881, 0.00089623, 0.000904161, 0.00091556,
0.000924772, 0.00093569, 0.000947029, 0.000958888, 0.000968894,
0.000979329, 0.000992942, 0.001002606, 0.00101525, 0.001028405,
0.00103714, 0.001047028, 0.001060727, 0.001072586, 0.001090096,
0.001105159, 0.001120077, 0.001130928, 0.001142876, 0.001153758,
0.001166803, 0.001178538, 0.00119154, 0.001205293, 0.001218295,
0.001235437, 0.001249549, 0.001266396, 0.001275109, 0.001279743,
0.001294749, 0.001309518, 0.001321721, 0.001335911, 0.00135704,
0.001374593, 0.001389789, 0.001403727, 0.001418646, 0.0014373,
0.001455954, 0.001468745, 0.001484978, 0.001505553, 0.001522318,
0.001543129, 0.001559723, 0.001577728, 0.001596804, 0.001613726,
0.001631539, 0.001651259, 0.00167007, 0.001690471, 0.001711182,
0.001732451, 0.001754733, 0.001774663, 0.001796811, 0.001816174,
0.001838106, 0.001856437, 0.001876273, 0.001894254, 0.001912697,
0.001936009, 0.001957601, 0.001983866, 0.002008425, 0.002030555,
0.00205563, 0.002080617, 0.002102259, 0.002128004, 0.002152091,
0.002179772, 0.002205539, 0.002230916, 0.002257038, 0.00228153,
0.002309921, 0.002341933, 0.002367517, 0.002396025, 0.002419084,
0.002443645, 0.002473349, 0.002500988, 0.002530566, 0.002558867,
0.002592188, 0.002622773, 0.002653688, 0.002685852, 0.002710527,
0.002739292, 0.002771013, 0.002805532, 0.002841148, 0.002876093,
0.002908327, 0.002938022, 0.0029698, 0.003007237, 0.003046334,
0.003084801, 0.003117196, 0.003158254, 0.003191937, 0.00323091,
0.003273591, 0.003307939, 0.003346203, 0.003385846, 0.003422344,
0.003462461, 0.003499154, 0.003542888, 0.003580298, 0.003623424,
0.003660788, 0.003700015, 0.003746697, 0.003787726, 0.003829095,
0.003874217, 0.003919546, 0.003967354, 0.004012355, 0.004054909,
0.004104023, 0.00415392, 0.004201728, 0.00425125, 0.00430143,
0.004359122, 0.004407716, 0.004465701, 0.004516568, 0.004572003,
0.0046245, 0.004681527, 0.004736385, 0.004789013, 0.004845169,
0.004905531, 0.004960046, 0.00501498, 0.005066878, 0.005129343,
0.005185114, 0.005246013, 0.00529813, 0.005368111, 0.005427317,
0.005494596, 0.005563752, 0.005624566, 0.005687943, 0.005758409,
0.005822882, 0.005885038, 0.005953048, 0.006011803, 0.006085703,
0.006157038, 0.006222489, 0.006292354, 0.006362848, 0.006446602,
0.006523488, 0.006599065, 0.006668189, 0.006748726, 0.00683206,
0.006911043, 0.006984767, 0.007071469, 0.007148739, 0.007237793,
0.007318618, 0.007406666, 0.007488542, 0.007573191, 0.007658232,
0.007743015, 0.007830551, 0.007926813, 0.008014895, 0.008106883,
0.008193565, 0.008292405, 0.008394328, 0.008490269, 0.008590922,
0.008684052, 0.008786946, 0.008888797, 0.008982092, 0.009085557,
0.009201545, 0.009317563, 0.009422259, 0.009528887, 0.009625833,
0.009737365, 0.009852084, 0.009969622, 0.010088275, 0.010214621,
0.010326395, 0.010442125, 0.010553572, 0.010675566, 0.010788799,
0.010912427, 0.011040804, 0.011161991, 0.011294114, 0.011429435,
0.011562774, 0.01169382, 0.011817902, 0.01195551, 0.012082375,
0.012212543, 0.012353415, 0.01247507, 0.012621953, 0.012740155,
0.012770497, 0.012924713, 0.013067629, 0.013217593, 0.013367214,
0.013524067, 0.01368398, 0.013839744, 0.014004594, 0.014164991,
0.014337402, 0.014498624, 0.014660024, 0.014831457, 0.0149965,
0.015171485, 0.01534538, 0.015519828, 0.015692661, 0.015872933,
0.016053339, 0.016247976, 0.016431603, 0.016610825, 0.016806979,
0.016995114, 0.017192221, 0.017383511, 0.017583131, 0.017779577,
0.017983183, 0.018193519, 0.018387818, 0.018603722, 0.018826885,
0.01902499, 0.019246945, 0.019475961, 0.019695832, 0.019935096,
0.020165852, 0.020398797, 0.020630761, 0.020876896, 0.021112362,
0.021350738, 0.021593713, 0.021842834, 0.022090315, 0.022348177,
0.022597325, 0.022837385, 0.023115431, 0.023375345, 0.023648672,
0.023919933, 0.024196619, 0.024471426, 0.02475858, 0.02503864,
0.02532454, 0.025615194, 0.025919009, 0.026214867, 0.026510541,
0.026820288, 0.027123303, 0.027433286, 0.027733259, 0.028048981,
0.028369867, 0.028692202, 0.029016199, 0.029359406, 0.029694839,
0.03004354, 0.03039204, 0.030750588, 0.03109654, 0.031468257,
0.031825775, 0.032192117, 0.032561958, 0.032928156, 0.033309912,
0.033688603, 0.034070003, 0.034478652, 0.034867347, 0.035266087,
0.035661144, 0.036082008, 0.036504751, 0.036915964, 0.037336354,
0.037766036, 0.038201777, 0.038655974, 0.03908616, 0.039522686,
0.039970521, 0.040432871, 0.040897518, 0.04136469, 0.041838371,
0.042352158, 0.04284733, 0.04333027, 0.043826524, 0.044325448,
0.044836305, 0.045366177, 0.045882503, 0.046419001, 0.046956281,
0.047506174, 0.048052495, 0.048597493, 0.049141686, 0.049676446,
0.050264859, 0.050829934, 0.05140503, 0.051992186, 0.052588573,
0.053180573, 0.053781028, 0.054411726, 0.055009341, 0.055641297,
0.056281961, 0.056915064, 0.057567706, 0.058206095, 0.058829746,
0.059509998, 0.060193007, 0.06087266, 0.061569807, 0.062273044,
0.062997686, 0.063704762, 0.064443165, 0.065189993, 0.065934644,
0.066714981, 0.067509749, 0.068287634, 0.069081928, 0.069854008,
0.070630272, 0.071445561, 0.072280944, 0.07311026, 0.073957586,
0.074810048, 0.075642144, 0.076487533, 0.077373621, 0.078261148,
0.079147319, 0.080051452, 0.080943571, 0.081871586, 0.082803329,
0.083754969, 0.084705812, 0.085674887, 0.086642654, 0.087639976,
0.088627287, 0.089630189, 0.090646962, 0.091692892, 0.09274723,
0.093785038, 0.094849889, 0.095940257, 0.09699651, 0.098098452,
0.099200314, 0.100313497, 0.101444759, 0.102590136, 0.103762256,
0.104931346, 0.106096838, 0.107269783, 0.108468145, 0.109662826,
0.110910582, 0.112150573, 0.113411571, 0.114709742, 0.115973007,
0.117297362, 0.118646767, 0.11997447, 0.121334815, 0.122702891,
0.123811264, 0.124091098, 0.125460014, 0.126836251, 0.128271774,
0.129732614, 0.13120427, 0.132700762, 0.134209249, 0.135697013,
0.137217508, 0.13876452, 0.140332145, 0.141915971, 0.143528749,
0.145152896, 0.146784733, 0.148426499, 0.15008857, 0.151745556,
0.153435501, 0.155166311, 0.15690568, 0.158673514, 0.160443631,
0.162239034, 0.164076456, 0.165898749, 0.167775507, 0.169631601,
0.171525509, 0.173423735, 0.175382242, 0.177356872, 0.179356041,
0.181334604, 0.183339916, 0.18538738, 0.187431289, 0.189534436,
0.191652065, 0.193778518, 0.195897525, 0.198069218, 0.200272874,
0.202500957, 0.204731841, 0.206968297, 0.209256512, 0.211551608,
0.213879199, 0.21625388, 0.218646052, 0.221081823, 0.223491829,
0.225904585, 0.228393825, 0.230855618, 0.233419678, 0.235922216,
0.238515877, 0.24112284, 0.243787269, 0.246436373, 0.249114734,
0.251855348, 0.254579343, 0.257339068, 0.260101942, 0.262887423,
0.265705025, 0.268596005, 0.271514714, 0.274435041, 0.277391723,
0.280400208, 0.283434247, 0.286513739, 0.289590669, 0.292745725,
0.295891591, 0.29910332, 0.302339495, 0.30558642, 0.308917798,
0.312250308, 0.315640385, 0.319024184, 0.322481136, 0.325991074,
0.329465732, 0.333014956, 0.336589853, 0.340251984, 0.343921276,
0.347652578, 0.351395388, 0.352223076, 0.355135629, 0.35889926,
0.362728103, 0.366602715, 0.370521613, 0.374510156, 0.3784952,
0.382515305, 0.386630048, 0.390720381, 0.39487372, 0.399072776,
0.403324165, 0.407680189, 0.412044191, 0.416424216, 0.42087254,
0.425349544, 0.4298866, 0.434461227, 0.439133077, 0.443858952,
0.448612554, 0.453464061, 0.45833582, 0.459340398, 0.463240006,
0.468239589, 0.473249248, 0.478353892, 0.483447263, 0.488605375,
0.493848314, 0.49914226, 0.50445842, 0.509797407, 0.515257646,
0.520758271, 0.526345922, 0.531995482, 0.537674239, 0.543466823,
0.549325584, 0.555217013, 0.561197693, 0.567218139, 0.573297922,
0.579429007, 0.58573467, 0.592088199, 0.598437085, 0.604883223,
0.611417736, 0.618031619, 0.624736154, 0.631531172, 0.638340611,
0.64524873, 0.652244124, 0.659396918, 0.66659139, 0.669693571,
0.673860961, 0.681161295, 0.688531458, 0.696060367, 0.703660043,
0.711292447, 0.71903071, 0.726895704, 0.734903451, 0.742980308,
0.751208295, 0.759530752, 0.76787455, 0.776325622, 0.784826863,
0.793472292, 0.80233292, 0.811201683, 0.820197899, 0.829370213,
0.83862362, 0.848075532, 0.857569334, 0.867244348, 0.877052678,
0.881125049, 0.886997791, 0.897086448, 0.907314384, 0.91764927,
0.928094389, 0.938789897, 0.9494957, 0.960471449, 0.9717454,
0.983043016, 0.994536694, 1.00623505, 1.01805901, 1.030071146,
1.042364196, 1.054833285, 1.067607186, 1.080494646, 1.093636132,
1.104208989, 1.128514748, 1.141849832, 1.155273898, 1.168635055,
1.181934262, 1.195128327, 1.208390682, 1.221525066, 1.234648393,
1.247778324, 1.260835405, 1.273916462, 1.286924958, 1.29993269,
1.312930855, 1.32587909, 1.338847585, 1.351744329, 1.35928479,
1.364602862, 1.377499658, 1.390501709, 1.403427833, 1.416456057,
1.429317455, 1.442223848, 1.455145222, 1.468055254, 1.481015905,
1.493925624, 1.506857933, 1.519891146, 1.532870507, 1.545807336,
1.558924153, 1.572000845, 1.585037352, 1.598378866, 1.611668453,
1.62500304, 1.638344802, 1.651667122, 1.665097756, 1.6784059,
1.686115499, 1.691950252, 1.70562696, 1.719208488, 1.732876058,
1.746723304, 1.760610133, 1.77457915, 1.788646797, 1.802967043,
1.81714276, 1.831518608, 1.846147803, 1.860718234, 1.875500359,
1.890478124, 1.905523315, 1.9206989, 1.936118981, 1.951599892,
1.967286231, 1.98322145, 1.999286893, 2.01546844, 2.031802281,
2.048507608, 2.065456366, 2.082614107, 2.099976765, 2.117457182,
2.135290297, 2.153707001, 2.172289301, 2.191162162, 2.210281363,
2.2297733, 2.245233808, 2.24937258, 2.269666369, 2.290525076,
2.311572538, 2.332950881, 2.354495335, 2.376657697, 2.399216036,
2.422201856, 2.446010365, 2.470131093, 2.494904357, 2.519944342,
2.545749213, 2.572222917, 2.599346458, 2.626872735, 2.655344595,
2.683835387, 2.71367832, 2.743919656, 2.774904028, 2.80645133,
2.83850512, 2.871095262, 2.897347095, 2.904805399, 2.938388435,
2.972606078, 3.007603777, 3.042966878, 3.078456584, 3.115381366,
3.1518634, 3.1885622, 3.226642441, 3.26497256, 3.303695076, 3.342723113,
3.38243581, 3.421985981, 3.462304637, 3.503262773, 3.544286157,
3.585931455, 3.62785329, 3.670227411, 3.71248119, 3.755155794,
3.797933435, 3.841259643, 3.885894408, 3.929759129, 3.974307798,
4.019265401, 4.064949543, 4.110839474, 4.156890455, 4.203011213,
4.249113168, 4.29582669, 4.34202474, 4.389252724, 4.437448524,
4.485236909, 4.533934036, 4.582040199, 4.630614132, 4.678946381,
4.727792117, 4.777624257, 4.827000112, 4.877098933, 4.927114639,
4.977449769, 5.029108475, 5.079627796, 5.131269581, 5.182610188,
5.234167394, 5.285883798, 5.339287246, 5.390871571, 5.444862974,
5.497240035, 5.551432762, 5.604435786, 5.657284109, 5.711124623,
5.765909445, 5.820480502, 5.87518186, 5.929168871, 5.984423069,
6.040735393, 6.096393157, 6.152094095, 6.208344398, 6.264740426,
6.321072694, 6.377352402, 6.433127284, 6.490200214, 6.548706793,
6.606396788, 6.664173006, 6.722126424, 6.780992697, 6.838719769,
6.896715776, 6.956449206, 7.016184606, 7.075981253, 7.135544407,
7.195796704, 7.254557196, 7.312283326, 7.372379292, 7.433183134,
7.493902012, 7.554415423, 7.614870247, 7.627272752, 7.675837577,
7.736288252, 7.796884518, 7.858257025, 7.919406985, 7.981752905,
8.045213429, 8.108003412, 8.170588748, 8.232430468, 8.297021984,
8.359506065, 8.421736347, 8.484464073, 8.547927283, 8.611065047,
8.675095554, 8.738886031, 8.801602707, 8.865358206, 8.929622916,
8.994218326, 9.059126885, 9.123386265, 9.188268908, 9.253014946,
9.318797396, 9.382738531, 9.448285827, 9.513255395, 9.578791771,
9.644017333, 9.709347562, 9.77525105, 9.841709815, 9.907989365,
9.973245417, 10.038507623, 10.105614193, 10.172042382, 10.238949119,
10.302905127, 10.367532086, 10.43600332, 10.502185766, 10.56872062,
10.635909227, 10.70235079, 10.768636725, 10.836581067, 10.903746581,
10.970209039, 11.037421775, 11.104492363, 11.172054393, 11.241668452,
11.310623275, 11.378398264, 11.447298295, 11.51502099, 11.583459333,
11.65151267, 11.721558044, 11.788357028, 11.858195058, 11.924759195,
11.991748578, 12.060952797, 12.128016782, 12.194058001, 12.262664411,
12.331017188, 12.397395914, 12.465093856, 12.531072755, 12.600539333,
12.670092596, 12.736760097, 12.806629644, 12.875612877, 12.945231048,
13.01354459, 13.083829804, 13.150895046, 13.220510865, 13.288610695,
13.356329568, 13.423226214, 13.490285124, 13.557006975, 13.625140748,
13.694474333, 13.761689643, 13.828259605, 13.898311101, 13.96730172,
14.036310145, 14.104531385, 14.17328702, 14.24107332, 14.308522482,
14.379287396, 14.448930503, 14.517920324, 14.589418245, 14.656637284,
14.726516174, 14.794494003, 14.867794386, 14.937949316, 15.003259227,
15.070367795, 15.140610681, 15.210197895, 15.280703395, 15.352264383,
15.42048158, 15.48557498, 15.553982272, 15.624040115, 15.692995695,
15.761569883, 15.828928457, 15.895412908, 15.963421284, 16.030127014,
16.09794871, 16.166761478, 16.23574672, 16.302489728, 16.371388848,
16.43920874, 16.506885795, 16.571802757, 16.641142252, 16.710976928,
16.773499442, 16.844490026, 16.912585993, 16.981824909, 17.047460797,
17.112821282, 17.183118188, 17.252431046, 17.321239867, 17.38882739,
17.454342306, 17.525471777, 17.594155485, 17.660962706, 17.727756707,
17.791791923, 17.859368109, 17.924085092, 17.993737077, 18.059661551,
18.132479624, 18.201876384, 18.267116918, 18.333460235, 18.399926778,
18.465507401, 18.530466193, 18.597133064, 18.664659499, 18.73179639,
18.799320295, 18.868071384, 18.932517619, 18.998371711, 19.065325207,
19.135129722, 19.206402988, 19.273691046, 19.338139461, 19.406885167,
19.473083959, 19.537232301, 19.600145071, 19.66527158, 19.730233936,
19.79649186, 19.86045111, 19.926468894, 19.991627385, 20.055538038,
20.122037249, 20.185238368, 20.254847929, 20.31891861, 20.383928954,
20.44605776, 20.509409947, 20.572326647, 20.636874581, 20.701076558,
20.76658926, 20.825436459, 20.885143814, 20.949206335, 20.962417927,
21.014726219, 21.083604803, 21.147895218, 21.209535019, 21.267175645,
21.327704991, 21.385146147, 21.449076884, 21.50912981, 21.569890435,
21.633202692, 21.695265393, 21.755257195, 21.81610816, 21.878624764,
21.936959835, 22.000728697, 22.063649412, 22.125047488, 22.18977412,
22.24906804, 22.31225956, 22.367835787, 22.431846367, 22.489339194,
22.553204195, 22.618200938, 22.676226876, 22.734591116, 22.789937721,
22.847062773, 22.902150618, 22.965413707, 23.025198118, 23.084596008,
23.144268223, 23.20253327, 23.257846604, 23.319188133, 23.381403038,
23.438407848, 23.496097368, 23.559236142, 23.620251799, 23.682980362,
23.744993458, 23.804088598, 23.866095512, 23.922195545, 23.974418771,
24.032865524, 24.092052, 24.151864718, 24.208914274, 24.267641369,
24.327264961, 24.385181251, 24.442156575, 24.500127886, 24.557686834,
24.617290747, 24.671487695, 24.720860807, 24.775017544, 24.8324025,
24.886709398, 24.948032991, 25.009915907, 25.067002347, 25.118275757,
25.180902675, 25.239077029, 25.291556739, 25.349628348, 25.411259274,
25.469475073, 25.520758624, 25.574220653, 25.630552011, 25.689092621,
25.740724463, 25.795843692, 25.851973018, 25.905060695, 25.958654424,
26.008683845, 26.060838256, 26.114622002, 26.170929575, 26.226689164,
26.289224591, 26.34624303, 26.410656822, 26.465429206, 26.507433885,
26.559557384, 26.613107253, 26.669341682, 26.720453179, 26.77141133,
26.820491444, 26.870662671, 26.921196347, 26.97265077, 27.025075312,
27.076331415, 27.121942032, 27.171932724, 27.224284236, 27.275128795,
27.327362184, 27.378765522, 27.433210341, 27.486106715, 27.543201087,
27.590367409, 27.634553204, 27.684303908, 27.736025971, 27.788413301,
27.839616051, 27.890957015, 27.932478563, 27.987948787, 28.042338752,
28.087505677, 28.129032734, 28.177206892, 28.226775461, 28.274067764,
28.314901184, 28.363808336, 28.412769043, 28.4545, 28.504096197,
28.555906619, 28.596464665, 28.639751441, 28.693877851, 28.743565978,
28.785181062, 28.831409017, 28.873414696, 28.91752309, 28.963197293,
29.002324815, 29.04401841, 29.084036446, 29.129048255, 29.181361993,
29.228023606, 29.278092756, 29.328469966, 29.372420692, 29.412157326,
29.457102565, 29.499377902, 29.541727767, 29.578436879, 29.626563114,
29.678999513, 29.72022254, 29.760800146, 29.807763449, 29.85405887,
29.898907902, 29.948357346, 29.991517819, 30.032594708, 30.07463922,
30.124267378, 30.169081817, 30.209401093, 30.263639074, 30.308129909,
30.349351446, 30.403123431, 30.44888233, 30.495653538, 30.533500176,
30.566351881, 30.603677184, 30.651663398, 30.687620049, 30.732858335,
30.773318824, 30.811477931, 30.855031664, 30.903544112, 30.941117103,
30.975834588, 31.025712688, 31.067618283, 31.105898375, 31.145599777,
31.191265386, 31.227959395, 31.269915935, 31.310954887, 31.346646846,
31.352889927, 31.384324365, 31.428011124, 31.462954165, 31.504016524,
31.544844485, 31.58228507, 31.616305622, 31.64511888, 31.686394358,
31.726061907, 31.762005653, 31.806589278, 31.84998479, 31.887536925,
31.920902854, 31.955484534, 31.991722316, 32.030320873, 32.0745303,
32.106707037, 32.14247583, 32.191623521, 32.230764362, 32.265823792,
32.305011645, 32.338277477, 32.383669424, 32.430272816, 32.464246531,
32.508333962, 32.540497326, 32.585564407, 32.6183372, 32.648289415,
32.682194896, 32.72600897, 32.761464835, 32.79707999, 32.82678069,
32.867237252, 32.894701497, 32.935087907, 32.972836279, 33.00989734,
33.040617123, 33.075018113, 33.1130372, 33.153237725, 33.191355711,
33.234950756, 33.272237264, 33.299359961, 33.332173059, 33.368440899,
33.403786236, 33.43071725, 33.465359922, 33.504884616, 33.536527922,
33.572121561, 33.604600746, 33.644338797, 33.677148676, 33.723968369,
33.756777797, 33.786668005, 33.819663178, 33.845586285, 33.879792585,
33.917578751, 33.944733922, 33.977593552, 34.002491484, 34.032089128,
34.069687894, 34.100362078, 34.140252698, 34.169052531, 34.205086532,
34.242681427, 34.269904339, 34.286632252, 34.310873656, 34.336359937,
34.379330206, 34.417838005, 34.457667528, 34.48409544, 34.511662962,
34.556383326, 34.574373909, 34.59384773, 34.622776, 34.652918626,
34.671570908, 34.708763946, 34.742606917, 34.769327799, 34.796802843,
34.826760045, 34.857112897, 34.895166557, 34.932199949, 34.964148201,
34.993475794, 35.021601387, 35.060246574, 35.087442295, 35.116270704,
35.152080084, 35.18088866, 35.20392769, 35.239068458, 35.271617551,
35.303735924, 35.332733065, 35.365539575, 35.393998892, 35.413928106,
35.443482437, 35.47522191, 35.510203403, 35.555475209, 35.587438915,
35.616194642, 35.658813147, 35.685381991, 35.712500303, 35.732735208,
35.758990832, 35.794229028, 35.810323208, 35.840381538, 35.867725386,
35.897505779, 35.933119976, 35.955180105, 35.999102802, 36.037615012,
36.063284567, 36.092122847, 36.122018451, 36.133950333, 36.157990686,
36.191808184, 36.216811063, 36.244362433, 36.277293322, 36.292689995,
36.321439371, 36.351071211, 36.387053331, 36.420344893, 36.449569316,
36.475470372, 36.514657208, 36.549298926, 36.571840132, 36.596118682,
36.644681511, 36.672752095, 36.689927475, 36.703018157, 36.717730805,
36.731528636, 36.745371885, 36.767665426, 36.792505734, 36.812777102,
36.839060205, 36.866293985, 36.879110689, 36.919389097, 36.946112477,
36.975157608, 37.002638522, 37.048216862, 37.080182372, 37.12279247,
37.153786128, 37.164048184, 37.187479627, 37.203939046, 37.218494913,
37.237432421, 37.252627558, 37.271083611, 37.304928312, 37.320654104,
37.342707804, 37.367600859, 37.383665516, 37.41472861, 37.446731742,
37.458603008, 37.480648562, 37.494227721, 37.512388941, 37.525307312,
37.535783883, 37.572828989, 37.583118525, 37.593607992, 37.616633117,
37.642772183, 37.65226606, 37.66691065, 37.691888256, 37.700971356,
37.728365689, 37.76209639, 37.772605378, 37.791146591, 37.805691763,
37.818136785, 37.830278349, 37.843017707, 37.863756211, 37.878294213,
37.898404007, 37.912241994, 37.931888, 37.941723422, 37.951200749,
37.966186422, 37.982501157, 37.998681477, 38.01165574, 38.036296398,
38.068904148, 38.102479573, 38.116920667, 38.133608833, 38.154024262,
38.167964004, 38.192100592, 38.20938018, 38.217830319, 38.224011912,
38.234635981, 38.237552573, 38.25874712, 38.307815364, 38.331216763,
38.34196245, 38.354161121, 38.366450641, 38.37848219, 38.392107386,
38.404094261, 38.428170144, 38.453483275, 38.462261535, 38.481224212,
38.50442632, 38.515853613, 38.523028509, 38.539638532, 38.557009872,
38.587524189, 38.592305647, 38.610881872, 38.614367215, 38.6412593,
38.65745459, 38.68111097, 38.68795225, 38.691723163, 38.710037827,
38.719339733, 38.734874149, 38.742343104, 38.755714403, 38.778926544,
38.79971555, 38.808756531, 38.818340744, 38.827939833, 38.833521371,
38.840039093, 38.866540678, 38.880866062, 38.89557291, 38.901913442,
38.932555063, 38.94987501, 38.956322946, 39.002291834, 39.017333788,
39.028902337, 39.047773407, 39.057115663, 39.074730601, 39.082181529,
39.123272644, 39.132706021, 39.151654415, 39.16797747, 39.17753063,
39.186276642, 39.194824504, 39.228426236, 39.253941796, 39.259275338,
39.259417918, 39.279249545, 39.292181105, 39.297594537, 39.313091314,
39.320881108, 39.326795865, 39.338921447, 39.363385661, 39.365346295,
39.39426879, 39.398860142, 39.408102377, 39.421508791, 39.441496828,
39.458994494, 39.477998562, 39.485653631, 39.497097285, 39.509535633,
39.512546714, 39.530129569, 39.584164293, 39.586750201, 39.595527935,
39.634997157, 39.642019657, 39.644831225, 39.674707589, 39.689556372,
39.690841669, 39.701910067, 39.704323197, 39.739142813, 39.791900368,
39.806888728), "10" = c(2.4225e-05, 2.5001e-05, 2.5028e-05, 2.5168e-05,
2.5448e-05, 2.5592e-05, 2.5627e-05, 2.6466e-05, 2.6858e-05, 2.713e-05,
2.7503e-05, 2.791e-05, 2.7932e-05, 2.8384e-05, 2.8694e-05, 3.0007e-05,
3.0263e-05, 3.057e-05, 3.0724e-05, 3.1177e-05, 3.1587e-05, 3.1657e-05,
3.2154e-05, 3.2912e-05, 3.2968e-05, 3.3135e-05, 3.3257e-05, 3.3287e-05,
3.3494e-05, 3.3742e-05, 3.4116e-05, 3.5154e-05, 3.5939e-05, 3.6206e-05,
3.7256e-05, 3.7527e-05, 3.7763e-05, 3.781e-05, 3.8096e-05, 3.8664e-05,
3.8922e-05, 3.9349e-05, 3.9633e-05, 3.9985e-05, 4.0043e-05, 4.0148e-05,
4.0584e-05, 4.1557e-05, 4.1666e-05, 4.2075e-05, 4.2743e-05, 4.329e-05,
4.3696e-05, 4.3975e-05, 4.4737e-05, 4.5208e-05, 4.6093e-05, 4.6409e-05,
4.6688e-05, 4.7879e-05, 4.883e-05, 4.8879e-05, 4.9253e-05, 4.9833e-05,
5.1118e-05, 5.1219e-05, 5.1646e-05, 5.1814e-05, 5.2684e-05, 5.2701e-05,
5.3173e-05, 5.3436e-05, 5.3747e-05, 5.4653e-05, 5.6917e-05, 5.7128e-05,
5.8211e-05, 6.03e-05, 6.0963e-05, 6.1596e-05, 6.228e-05, 6.2504e-05,
6.3548e-05, 6.434e-05, 6.5712e-05, 6.6272e-05, 6.7376e-05, 6.813e-05,
6.8498e-05, 6.9095e-05, 6.9365e-05, 6.9811e-05, 7.0443e-05, 7.1881e-05,
7.2499e-05, 7.4594e-05, 7.5771e-05, 7.6547e-05, 7.6809e-05, 7.7907e-05,
7.8027e-05, 7.8581e-05, 7.9825e-05, 8.026e-05, 8.0927e-05, 8.1532e-05,
8.3105e-05, 8.4205e-05, 8.5499e-05, 8.7041e-05, 8.85e-05, 8.9813e-05,
9.1247e-05, 9.2216e-05, 9.3093e-05, 9.3974e-05, 9.5219e-05, 9.6165e-05,
9.7563e-05, 9.9645e-05, 0.000100551, 0.000101753, 0.000102921,
0.000104656, 0.000105939, 0.00010837, 0.000110445, 0.000111965,
0.000113912, 0.000115373, 0.000117454, 0.00011861, 0.000119411,
0.00012151, 0.000123028, 0.000123645, 0.000125413, 0.00012645,
0.000128183, 0.000129511, 0.00013143, 0.000132959, 0.000134459,
0.000136049, 0.000137792, 0.000139172, 0.000139788, 0.000141666,
0.000146505, 0.000148158, 0.000149205, 0.000150666, 0.000152529,
0.000154093, 0.000156246, 0.000157431, 0.000160167, 0.00016141,
0.000163207, 0.000164867, 0.000167001, 0.000167755, 0.000169191,
0.000172171, 0.00017379, 0.000175926, 0.000178467, 0.000180997,
0.000182945, 0.000185385, 0.000187643, 0.000189891, 0.000191571,
0.00019314, 0.000194793, 0.000197328, 0.000200341, 0.000201933,
0.000205221, 0.000206551, 0.00020783, 0.000210918, 0.000213375,
0.00021479, 0.000218125, 0.000220707, 0.00022311, 0.000225637,
0.000228361, 0.000231289, 0.000233744, 0.000236812, 0.000238756,
0.000240193, 0.000242877, 0.000246115, 0.000250456, 0.000253099,
0.000255775, 0.000261203, 0.000263945, 0.000267291, 0.000269194,
0.000270673, 0.000275675, 0.000278654, 0.000282103, 0.000284338,
0.000287037, 0.000289324, 0.000291168, 0.000293758, 0.000296576,
0.000300255, 0.000302861, 0.000306341, 0.000309799, 0.000314755,
0.00031778, 0.000321978, 0.000324472, 0.000328014, 0.000331378,
0.000335875, 0.000340354, 0.00034511, 0.000348074, 0.00035293,
0.00035831, 0.000363352, 0.000367647, 0.000370564, 0.000375274,
0.00037745, 0.00038115, 0.000385837, 0.000390884, 0.000395712,
0.000401569, 0.000407084, 0.000411823, 0.000417536, 0.000423198,
0.000427264, 0.000430404, 0.00043562, 0.000440752, 0.000445552,
0.000450792, 0.000456385, 0.000461838, 0.000467224, 0.000472282,
0.00047698, 0.000483492, 0.000488339, 0.000494931, 0.000502316,
0.00050852, 0.000516247, 0.000523579, 0.000527881, 0.000532479,
0.000539272, 0.000545489, 0.000550228, 0.000555487, 0.00056118,
0.000564777, 0.000571611, 0.000576607, 0.000583164, 0.000589741,
0.000598126, 0.000605435, 0.00061425, 0.000619276, 0.000626028,
0.000632102, 0.000640251, 0.000646643, 0.000654033, 0.000660557,
0.000667417, 0.000677758, 0.000685735, 0.000691887, 0.000699168,
0.000706745, 0.000714015, 0.000723926, 0.000729667, 0.000738865,
0.000746752, 0.000755425, 0.00076178, 0.000772133, 0.000779358,
0.00078779, 0.000795878, 0.000804403, 0.000812571, 0.000820073,
0.000829976, 0.000840124, 0.00085037, 0.000860979, 0.000869934,
0.000879113, 0.000891395, 0.000902612, 0.000912396, 0.000924129,
0.000932283, 0.000942624, 0.000951714, 0.000963946, 0.000972047,
0.000983117, 0.000993911, 0.00100722, 0.001016032, 0.001025059,
0.001037614, 0.00104898, 0.001061672, 0.001073096, 0.001084438,
0.001095447, 0.00110601, 0.001121114, 0.001132071, 0.001148787,
0.001158801, 0.001170939, 0.001185031, 0.00119737, 0.00121165,
0.00122594, 0.001243202, 0.001255405, 0.001258353, 0.001270821,
0.001286463, 0.001297719, 0.001315369, 0.001329078, 0.001341271,
0.001355607, 0.001371064, 0.001389898, 0.001401572, 0.00141618,
0.00143417, 0.001447981, 0.00146886, 0.00148523, 0.001502641,
0.001513301, 0.001531849, 0.001550612, 0.001569798, 0.001586407,
0.001607299, 0.001626509, 0.001646151, 0.001672829, 0.001693661,
0.001711699, 0.00173128, 0.001747788, 0.001765872, 0.001786737,
0.001802837, 0.001825232, 0.001842997, 0.001861864, 0.001884546,
0.001904873, 0.001926697, 0.001950839, 0.00197212, 0.001997991,
0.002019689, 0.002041605, 0.002063452, 0.002088863, 0.00211438,
0.00213958, 0.002162528, 0.00218851, 0.00221211, 0.002236291,
0.002268575, 0.002296608, 0.002322994, 0.002347513, 0.002374302,
0.002399944, 0.002430773, 0.002461524, 0.002490656, 0.002520554,
0.002548478, 0.00257372, 0.002598957, 0.002627531, 0.002656409,
0.002688046, 0.002715356, 0.002741661, 0.002773379, 0.002807606,
0.00283777, 0.002871217, 0.002903921, 0.00293988, 0.002967046,
0.00299868, 0.003033499, 0.003067124, 0.003108249, 0.003140737,
0.00317534, 0.003211613, 0.00325228, 0.003291963, 0.003327411,
0.003366988, 0.00340476, 0.003442823, 0.003489329, 0.003535798,
0.003573566, 0.003611861, 0.003656422, 0.003701289, 0.003739984,
0.003786672, 0.003827065, 0.00386885, 0.00391425, 0.003963245,
0.00401057, 0.004060403, 0.004106682, 0.004154989, 0.004194183,
0.00424595, 0.004291038, 0.004344861, 0.004394767, 0.004442752,
0.00449446, 0.004547394, 0.004600669, 0.004654978, 0.004706169,
0.004763747, 0.004818222, 0.004877675, 0.004932547, 0.00498662,
0.005045479, 0.005097891, 0.005151508, 0.005207862, 0.005266877,
0.005326688, 0.005384631, 0.005450511, 0.005507664, 0.005566323,
0.005631864, 0.005697411, 0.005765158, 0.005829613, 0.005899966,
0.005964903, 0.006031942, 0.006104284, 0.006174204, 0.00624336,
0.006314383, 0.006390861, 0.00646342, 0.006538994, 0.006615526,
0.006687375, 0.006770475, 0.006843772, 0.006919184, 0.006994462,
0.007070465, 0.007150353, 0.007226725, 0.00731064, 0.007391891,
0.007473752, 0.007556024, 0.00764626, 0.007734012, 0.007825967,
0.007919076, 0.00802116, 0.008111853, 0.008209204, 0.008303233,
0.008386756, 0.008483248, 0.008583645, 0.008683822, 0.008780641,
0.008883886, 0.008985742, 0.009092455, 0.009202802, 0.009310849,
0.009413854, 0.009527583, 0.009639835, 0.009747664, 0.009863143,
0.009975387, 0.01009423, 0.010204864, 0.010314531, 0.010430487,
0.010551237, 0.010673856, 0.010801442, 0.010921029, 0.011050686,
0.011181306, 0.01131156, 0.011439291, 0.011566681, 0.011705759,
0.011840097, 0.011977962, 0.012111255, 0.012211676, 0.012240932,
0.012365781, 0.012509429, 0.012642603, 0.012796205, 0.01294554,
0.013103294, 0.013264738, 0.013418084, 0.013579123, 0.013731395,
0.013893012, 0.01406172, 0.014211308, 0.014362421, 0.014510325,
0.014674199, 0.014851031, 0.015007389, 0.015176658, 0.015339182,
0.015510773, 0.015684383, 0.015864992, 0.016047122, 0.016227529,
0.016409899, 0.01659475, 0.016792587, 0.016981102, 0.017175539,
0.017371671, 0.017589851, 0.017803843, 0.018000676, 0.01819554,
0.018401578, 0.018612415, 0.01883125, 0.019043891, 0.019268324,
0.019503517, 0.019734829, 0.019961821, 0.020203782, 0.020433521,
0.020681929, 0.020922725, 0.021165153, 0.021407154, 0.021656389,
0.021905725, 0.022148888, 0.022405178, 0.022652755, 0.022908908,
0.023165239, 0.023433598, 0.023713281, 0.02397992, 0.024246172,
0.024532125, 0.024828038, 0.025108874, 0.025398733, 0.025683416,
0.025985819, 0.02626826, 0.026575902, 0.026875354, 0.027172917,
0.027474139, 0.027777756, 0.028087127, 0.028412294, 0.028729832,
0.029059364, 0.029401145, 0.029751104, 0.030084594, 0.030428127,
0.030769593, 0.031127862, 0.031469682, 0.0318204, 0.03217575,
0.032555648, 0.0329261, 0.033284671, 0.033656153, 0.034030418,
0.034431946, 0.034826781, 0.035225375, 0.03562973, 0.036052803,
0.036455516, 0.036864118, 0.037299802, 0.037708244, 0.038127955,
0.03854331, 0.038993573, 0.039438694, 0.039890855, 0.04036718,
0.040841554, 0.041320985, 0.041792569, 0.042277243, 0.042747995,
0.043240666, 0.04372332, 0.044226294, 0.044738225, 0.045245368,
0.045751954, 0.046264771, 0.046802853, 0.047330321, 0.047897432,
0.048472645, 0.049004561, 0.049559829, 0.050119604, 0.050688731,
0.051275263, 0.05186103, 0.052469743, 0.053056195, 0.05367326,
0.054296909, 0.054903195, 0.055519053, 0.056138938, 0.056785834,
0.057441479, 0.058102759, 0.058767484, 0.059428062, 0.060104002,
0.060784574, 0.061483474, 0.062178368, 0.062867735, 0.063606118,
0.064320759, 0.065059898, 0.065783975, 0.066526607, 0.067281231,
0.068043841, 0.068835335, 0.06961338, 0.070443216, 0.071238919,
0.072021939, 0.072847498, 0.073674398, 0.07449862, 0.075337892,
0.076195744, 0.077041267, 0.077929923, 0.078799218, 0.079692372,
0.080599076, 0.081524143, 0.082424014, 0.083358824, 0.084286916,
0.085250258, 0.086211018, 0.087183763, 0.088195821, 0.089201002,
0.090220677, 0.091239645, 0.092276235, 0.093285009, 0.094322618,
0.095407386, 0.096510119, 0.097585801, 0.098674986, 0.09979195,
0.100909263, 0.102064621, 0.103202874, 0.104376911, 0.105544313,
0.106711377, 0.107912459, 0.109102112, 0.11033684, 0.111582164,
0.112809591, 0.114108812, 0.115382327, 0.116627213, 0.117661543,
0.117941316, 0.119258751, 0.120593614, 0.121943931, 0.123280664,
0.12463858, 0.126025071, 0.12742727, 0.128841154, 0.130289047,
0.131754693, 0.133238281, 0.134710507, 0.136247629, 0.137743134,
0.139286699, 0.140821824, 0.142389314, 0.143980915, 0.145598921,
0.147191358, 0.148845291, 0.150505023, 0.152151739, 0.153819486,
0.15553998, 0.157271245, 0.158994918, 0.160738099, 0.162509672,
0.164356976, 0.166200634, 0.168027245, 0.169922355, 0.171794779,
0.173692627, 0.175612145, 0.177569154, 0.179512621, 0.181488549,
0.183495105, 0.185527007, 0.187543467, 0.189611425, 0.191691371,
0.193834519, 0.195992813, 0.198129879, 0.200301744, 0.202534656,
0.204746773, 0.207016903, 0.20932865, 0.211615588, 0.213945678,
0.216297269, 0.218672856, 0.221067501, 0.223462934, 0.225917706,
0.228395745, 0.230869142, 0.23343882, 0.236003036, 0.238597798,
0.241215807, 0.243876504, 0.246492833, 0.24920965, 0.251917199,
0.254627452, 0.257406511, 0.260227522, 0.263075256, 0.265938266,
0.268878288, 0.271780925, 0.274765767, 0.277713294, 0.2807101,
0.283740404, 0.286846509, 0.289945262, 0.293094907, 0.296274194,
0.299512585, 0.302770719, 0.306025966, 0.309321141, 0.312710867,
0.316102311, 0.319497247, 0.322942551, 0.326454484, 0.330055414,
0.333625284, 0.33441212, 0.337291852, 0.340963874, 0.344613546,
0.348359439, 0.352118614, 0.355917169, 0.359798028, 0.363652416,
0.367598431, 0.371530428, 0.375544639, 0.379615498, 0.383731286,
0.387914581, 0.39212411, 0.396389047, 0.400692473, 0.405011417,
0.409387284, 0.413818027, 0.418334421, 0.422848463, 0.427402861,
0.432005008, 0.436673604, 0.437614073, 0.441379857, 0.446179507,
0.451018204, 0.455892085, 0.460799699, 0.465786174, 0.470806677,
0.47596626, 0.48111333, 0.486329052, 0.491591803, 0.496882341,
0.5022939, 0.507737092, 0.513256105, 0.51881399, 0.524445256,
0.530151221, 0.535989549, 0.541810364, 0.547730745, 0.55362518,
0.559607153, 0.565703949, 0.571898344, 0.578107016, 0.584383065,
0.590722802, 0.597208127, 0.603694907, 0.610310738, 0.617009347,
0.623830077, 0.630641047, 0.637629264, 0.640601183, 0.644618556,
0.651688965, 0.658845271, 0.666173557, 0.673572342, 0.681025569,
0.688607362, 0.696211957, 0.703928007, 0.711784083, 0.719714277,
0.727739015, 0.73588594, 0.744122154, 0.752417292, 0.760874124,
0.769399108, 0.778178356, 0.786996826, 0.795991102, 0.80504834,
0.814172897, 0.823426799, 0.832830868, 0.842377757, 0.846363531,
0.85215248, 0.86198027, 0.871876115, 0.882002275, 0.892334688,
0.902815419, 0.913514317, 0.924325045, 0.935251615, 0.94631074,
0.95765088, 0.969150702, 0.980839232, 0.992651603, 1.004721483,
1.017029927, 1.029595927, 1.042383609, 1.055364582, 1.065822831,
1.089588749, 1.102887554, 1.115986841, 1.129013687, 1.142088081,
1.155122014, 1.168054628, 1.181013934, 1.193877236, 1.206866758,
1.219719384, 1.232613427, 1.245473904, 1.258265334, 1.271020749,
1.283808707, 1.296448498, 1.309156196, 1.316561623, 1.32184399,
1.334520662, 1.347277053, 1.359978324, 1.372564267, 1.385278944,
1.397938229, 1.410680688, 1.423389322, 1.435983966, 1.448805886,
1.461529731, 1.474352483, 1.487222226, 1.49996462, 1.512727784,
1.525730902, 1.538592609, 1.551536368, 1.56452481, 1.577426507,
1.590558497, 1.603630681, 1.616648313, 1.630059443, 1.63762561,
1.643219284, 1.656585498, 1.669975882, 1.683516733, 1.697056188,
1.710677915, 1.72442837, 1.738094876, 1.751870337, 1.765810092,
1.779771781, 1.794100116, 1.808594942, 1.822962682, 1.837538442,
1.852253172, 1.867078269, 1.882192989, 1.8972627, 1.912673916,
1.928163454, 1.943743882, 1.959581958, 1.97559315, 1.991824462,
2.008213669, 2.024915481, 2.042074733, 2.059229514, 2.076704614,
2.094395294, 2.112314412, 2.130608955, 2.149022346, 2.167853861,
2.183035475, 2.187058742, 2.20694757, 2.226933355, 2.247341366,
2.268099014, 2.289155829, 2.310692129, 2.33283739, 2.35532754,
2.378378329, 2.402117893, 2.42590095, 2.450302445, 2.474988016,
2.500529939, 2.526537017, 2.55311487, 2.580165332, 2.607978334,
2.636139097, 2.665115585, 2.694506568, 2.724417178, 2.754895226,
2.785916947, 2.811025773, 2.81794638, 2.850065236, 2.882475117,
2.915930248, 2.950171571, 2.98427354, 3.01874264, 3.053849141,
3.089619855, 3.126150857, 3.162287838, 3.19857915, 3.23567223,
3.273021333, 3.310720697, 3.348990311, 3.387073692, 3.425804784,
3.465283425, 3.504396867, 3.54411344, 3.584366072, 3.625220431,
3.66632396, 3.707150494, 3.749009833, 3.790599445, 3.832533905,
3.874461323, 3.916141677, 3.958686522, 4.002039039, 4.045415005,
4.089267101, 4.133307119, 4.176915717, 4.221469346, 4.266776978,
4.312767451, 4.358835903, 4.404149778, 4.451132412, 4.497563508,
4.544109084, 4.590352067, 4.637949165, 4.686424056, 4.734352493,
4.783296213, 4.832393764, 4.88083519, 4.929887698, 4.979114905,
5.02769901, 5.079203274, 5.129565274, 5.179824512, 5.230974345,
5.282660006, 5.335120354, 5.38797424, 5.439726374, 5.492744368,
5.545662683, 5.598356701, 5.652399163, 5.705643885, 5.759993548,
5.814601616, 5.869126237, 5.924807709, 5.981040947, 6.037170741,
6.0925258, 6.148510761, 6.205538697, 6.262134187, 6.319122418,
6.377794959, 6.434954897, 6.491948609, 6.55065277, 6.609803727,
6.669719311, 6.730233674, 6.790059987, 6.850920024, 6.91159371,
6.97252503, 7.03269138, 7.093380511, 7.155343324, 7.218723021,
7.279546791, 7.340690408, 7.402439405, 7.416140464, 7.466322026,
7.528769847, 7.593571651, 7.658158366, 7.722466665, 7.788005272,
7.852762642, 7.919092889, 7.982961848, 8.048833383, 8.116539781,
8.183029681, 8.248632132, 8.314296064, 8.382035997, 8.449978963,
8.519449784, 8.58766761, 8.65568746, 8.724060282, 8.792044427,
8.859634513, 8.927911423, 8.996863783, 9.066125008, 9.135577064,
9.205572621, 9.274980703, 9.343339664, 9.413705917, 9.482928965,
9.552422342, 9.623659339, 9.695814087, 9.768783059, 9.841234752,
9.914416605, 9.98603228, 10.058713536, 10.132304508, 10.207202424,
10.279852175, 10.35339341, 10.425831216, 10.499989981, 10.574172038,
10.649454969, 10.725496837, 10.800189152, 10.876976604, 10.954301966,
11.02927477, 11.106128166, 11.182912384, 11.257644432, 11.335120748,
11.413649293, 11.490729804, 11.567351513, 11.6452045, 11.723702495,
11.801481253, 11.880060744, 11.962064231, 12.043005338, 12.121033514,
12.200140107, 12.279820416, 12.356806427, 12.433675544, 12.51372372,
12.594044637, 12.674762824, 12.753141544, 12.831627844, 12.913671056,
12.996475156, 13.078140277, 13.160497221, 13.242276658, 13.325624906,
13.407825816, 13.487160289, 13.568621774, 13.65004158, 13.730160433,
13.812298419, 13.895810898, 13.977013606, 14.060177248, 14.143415488,
14.229095, 14.313360071, 14.394766308, 14.477348132, 14.563107851,
14.645385083, 14.72770328, 14.81250938, 14.89479052, 14.981834834,
15.065301683, 15.150269881, 15.234286843, 15.316810409, 15.39912423,
15.484360159, 15.570739393, 15.65555925, 15.744572468, 15.832204763,
15.91483977, 16.006199891, 16.093526097, 16.177641972, 16.265066797,
16.3550137, 16.442000775, 16.528872659, 16.617325738, 16.705218095,
16.789364649, 16.877776953, 16.965008888, 17.047281211, 17.134189714,
17.227722461, 17.314699292, 17.398636082, 17.482850467, 17.571783862,
17.662838904, 17.747654044, 17.836963457, 17.923092004, 18.013895744,
18.106396985, 18.190017405, 18.279145087, 18.373596427, 18.465420023,
18.552518839, 18.644502974, 18.734212739, 18.826806068, 18.916325367,
19.008805871, 19.091387793, 19.181713594, 19.272957001, 19.363485941,
19.453801499, 19.538872831, 19.63301025, 19.732070148, 19.824719245,
19.910131738, 20.000346004, 20.090735134, 20.17931822, 20.271099604,
20.359345003, 20.449545386, 20.537318639, 20.630440942, 20.72471213,
20.8136158, 20.905402486, 21.006535023, 21.099003867, 21.190641204,
21.277209857, 21.365261401, 21.459067117, 21.54865688, 21.64268309,
21.735833261, 21.827008999, 21.921778544, 22.01167303, 22.096777103,
22.182372289, 22.273804355, 22.364793273, 22.456331699, 22.544747174,
22.63886179, 22.724791592, 22.81491718, 22.899415271, 22.99496967,
23.091840971, 23.18273801, 23.273321936, 23.361577772, 23.451518322,
23.543485669, 23.631391289, 23.721284846, 23.813494388, 23.830958786,
23.904237714, 23.99740631, 24.087594336, 24.172943911, 24.274161415,
24.362139995, 24.453532753, 24.540695793, 24.632500279, 24.723593001,
24.817718992, 24.907582427, 24.996702273, 25.085523164, 25.170573307,
25.261339745, 25.353369823, 25.448395648, 25.543219674, 25.63779333,
25.727406626, 25.819785142, 25.911685111, 25.999908648, 26.090735734,
26.185427567, 26.283638713, 26.376351374, 26.468050184, 26.560393839,
26.653506704, 26.744986408, 26.84148007, 26.935424712, 27.031613859,
27.12990126, 27.213413474, 27.297949511, 27.391606073, 27.481023046,
27.577244196, 27.660078273, 27.752969371, 27.847640785, 27.935775729,
28.03001622, 28.112326411, 28.202678142, 28.305711886, 28.41408588,
28.507798726, 28.601222202, 28.691577655, 28.778426427, 28.865139442,
28.952047073, 29.04189022, 29.13290208, 29.222468629, 29.310252568,
29.403966566, 29.496545685, 29.588955658, 29.670282655, 29.765237946,
29.853860713, 29.939157623, 30.031831597, 30.132635284, 30.223626553,
30.314716077, 30.404698773, 30.49653321, 30.598354005, 30.706591079,
30.794758086, 30.88567056, 30.983904059, 31.079917086, 31.170311984,
31.254085628, 31.349366271, 31.434231574, 31.522216021, 31.62488187,
31.714113276, 31.807101124, 31.898765083, 31.989075454, 32.077621503,
32.163954464, 32.246301303, 32.342922885, 32.42317931, 32.514293109,
32.599385166, 32.696940056, 32.797336076, 32.881616052, 32.963128604,
33.055534389, 33.144613213, 33.240354973, 33.333530041, 33.419772089,
33.511015226, 33.596997478, 33.690154576, 33.770456693, 33.854346738,
33.93568232, 34.026461214, 34.133964357, 34.224368934, 34.31696213,
34.395991053, 34.488531354, 34.573282935, 34.657860394, 34.748500015,
34.83593968, 34.921200877, 35.016495421, 35.103500147, 35.189098757,
35.278248657, 35.380007822, 35.453440189, 35.536598475, 35.628900353,
35.70795709, 35.805519717, 35.884035947, 35.974236628, 36.073947199,
36.166191944, 36.262461722, 36.348830108, 36.447486376, 36.537945741,
36.621765276, 36.72019365, 36.806406163, 36.884595414, 36.961244033,
37.034270501, 37.114565364, 37.209007141, 37.286227619, 37.371118267,
37.439248702, 37.523874232, 37.626546182, 37.714768819, 37.785740396,
37.872095337, 37.959705484, 38.031587597, 38.11013961, 38.193842456,
38.279066005, 38.356655473, 38.426962569, 38.497766592, 38.580200096,
38.665833922, 38.732337607, 38.823390354, 38.912840108, 39.005549346,
39.079872157, 39.167914141, 39.262178359, 39.344000365, 39.42523678,
39.49998407, 39.579711118, 39.658374457, 39.744554639, 39.824637137,
39.909778531, 39.976503627, 40.07086873, 40.146192319, 40.232737117,
40.33047521, 40.411234975, 40.511735513, 40.584150733, 40.678903543,
40.748665601, 40.821384285, 40.896221213, 40.979874976, 41.083133078,
41.172353606, 41.246389695, 41.313545639, 41.396839905, 41.479670109,
41.501513806, 41.585966498, 41.668554998, 41.752348114, 41.831414071,
41.912412532, 42.004031538, 42.084409904, 42.177761254, 42.253139879,
42.328232597, 42.406243606, 42.473229902, 42.548504366, 42.630339647,
42.692319198, 42.74748047, 42.804862445, 42.878076503, 42.963458529,
43.039196054, 43.10863784, 43.194069051, 43.277205491, 43.361188831,
43.436277396, 43.487676246, 43.544669136, 43.634713019, 43.728844753,
43.791675735, 43.859926068, 43.933711086, 44.005127785, 44.086927783,
44.157265008, 44.21662749, 44.292348231, 44.370159707, 44.45510652,
44.534029169, 44.626322887, 44.701067778, 44.803583406, 44.864293009,
44.95201579, 45.043863222, 45.123050925, 45.182140354, 45.25662336,
45.319886322, 45.391665942, 45.459489666, 45.545710844, 45.618135789,
45.66728339, 45.72744204, 45.799838467, 45.898369185, 45.994963194,
46.090238567, 46.146203488, 46.26281139, 46.346893944, 46.45014075,
46.529511793, 46.597497684, 46.658877955, 46.729144287, 46.800422843,
46.861403455, 46.92327215, 46.969716677, 47.023688664, 47.099679441,
47.184733527, 47.248407438, 47.32401346, 47.396937895, 47.452863472,
47.514186447, 47.576990958, 47.648578084, 47.705497443, 47.762521688,
47.831638098, 47.903169177, 47.951731454, 48.020834076, 48.095559152,
48.167585912, 48.220456741, 48.326118955, 48.418029654, 48.484154252,
48.555548343, 48.609736278, 48.677514557, 48.75564225, 48.835717134,
48.912387575, 48.960681064, 49.054169503, 49.128789242, 49.194258607,
49.253431104, 49.37758376, 49.448551612, 49.502782365, 49.539253929,
49.582457332, 49.66630225, 49.713330225, 49.757793563, 49.812926089,
49.886663198, 49.950486551, 50.026712999, 50.088839497, 50.127248006,
50.174739004, 50.240564471, 50.332476336, 50.400841704, 50.480282298,
50.532097998, 50.583808449, 50.662323506, 50.742654396, 50.780844233,
50.891342609, 50.957442039, 51.042725986, 51.093942464, 51.156948895,
51.195729006, 51.291043093, 51.381803663, 51.458790073, 51.531678716,
51.599976131, 51.65626566, 51.709147107, 51.76203002, 51.824975017,
51.878645675, 51.934547869, 52.001951575, 52.067250431, 52.1194897,
52.162675836, 52.235741806, 52.32566646, 52.389523072, 52.443109134,
52.49138602, 52.54272621, 52.595193784, 52.663805809, 52.788684196,
52.833450301, 52.875193543, 52.9159869, 52.975875823, 53.031362979,
53.068981885, 53.116212445, 53.14547418, 53.2727605, 53.338198482,
53.393187604, 53.417470109, 53.505323214, 53.546378814, 53.578433904,
53.615060914, 53.669437374, 53.780209647, 53.808896345, 53.854121481,
53.906261433, 53.961012623, 54.009645471, 54.049179711, 54.104410343,
54.169819551, 54.203007186, 54.288961166, 54.351679756, 54.396473216,
54.439717446, 54.469957439, 54.546661941, 54.642504156, 54.676294861,
54.762218371, 54.808256939, 54.882289295, 54.928723673, 55.048099579,
55.128384739, 55.17783498, 55.235278344, 55.304976843, 55.374097326,
55.467879464, 55.513557679, 55.543457793, 55.570730734, 55.686372953,
55.770528997, 55.87696501, 55.905337912, 55.947162423, 56.0312052,
56.176790596, 56.219229001, 56.245140746, 56.27486856, 56.330070588,
56.399630729, 56.450215642, 56.488855048, 56.546142414, 56.587661981,
56.648618888, 56.741137626, 56.802316218, 56.933502363, 56.990857005,
57.040105216, 57.097273315, 57.119688441, 57.235078939, 57.327216296,
57.406750052, 57.417746121, 57.474042919, 57.533842229, 57.599588486,
57.620970505, 57.6515351, 57.679454109, 57.740176231, 57.746021517,
57.753052499, 57.795981006, 57.822460492, 57.867960392, 57.879056695,
57.909533392, 58.01337581, 58.077555295, 58.117737276, 58.156706159,
58.205200391, 58.248586396, 58.268805623, 58.321966227, 58.405923894,
58.443694331, 58.520466117, 58.586773008, 58.59859899, 58.722980332,
58.768044116, 58.785127157, 58.807314622, 58.852421729, 58.886936484,
58.89957316, 58.91820026, 58.946018727, 59.008331298, 59.04888349,
59.148655135, 59.183619318, 59.18393932, 59.234868721, 59.286425847,
59.364700758, 59.537521765, 59.548197367, 59.598490284, 59.672819646,
59.727335913, 59.78205831, 59.818075066, 59.861644715, 59.901500831,
59.964907418, 59.97134798, 60.049307518, 60.102000229, 60.150915601,
60.191144727, 60.236815256, 60.244739861, 60.326130152, 60.368676594,
60.458258591, 60.461048872, 60.485896663, 60.519909409, 60.563425111,
60.595992007, 60.620363457, 60.67239429, 60.682697059, 60.770795465,
60.816598289, 60.84351731, 60.919060583, 60.931892622, 60.962964442,
60.965851849, 60.981282393, 61.008663477, 61.034751528, 61.062704721,
61.156373572, 61.198838093, 61.243416455, 61.262388603, 61.358863648,
61.379551606, 61.57937367, 61.579655353, 61.585538279, 61.587656687,
61.658794017, 61.667865819, 61.722849948, 61.829180575, 61.88010347,
61.881584817, 61.946075566, 61.982596065, 62.043244688, 62.120505964,
62.209069428), "20" = c(2.1738e-05, 2.2164e-05, 2.2221e-05, 2.2704e-05,
2.2864e-05, 2.3015e-05, 2.3582e-05, 2.3592e-05, 2.372e-05, 2.3884e-05,
2.3969e-05, 2.4259e-05, 2.4576e-05, 2.481e-05, 2.4906e-05, 2.5291e-05,
2.5399e-05, 2.5596e-05, 2.5674e-05, 2.574e-05, 2.6771e-05, 2.7183e-05,
2.7636e-05, 2.783e-05, 2.8432e-05, 2.8451e-05, 2.8745e-05, 2.8937e-05,
2.9495e-05, 2.9803e-05, 2.9873e-05, 3.0239e-05, 3.0634e-05, 3.1161e-05,
3.2279e-05, 3.2841e-05, 3.3132e-05, 3.408e-05, 3.4187e-05, 3.4496e-05,
3.4732e-05, 3.4892e-05, 3.513e-05, 3.6065e-05, 3.6173e-05, 3.6425e-05,
3.6823e-05, 3.7422e-05, 3.8441e-05, 3.8984e-05, 3.939e-05, 3.9863e-05,
4.0046e-05, 4.0391e-05, 4.0753e-05, 4.1457e-05, 4.1758e-05, 4.2048e-05,
4.2295e-05, 4.2706e-05, 4.3357e-05, 4.3567e-05, 4.4124e-05, 4.4534e-05,
4.4838e-05, 4.4991e-05, 4.5728e-05, 4.6237e-05, 4.7076e-05, 4.7239e-05,
4.7709e-05, 4.7879e-05, 4.8164e-05, 4.8893e-05, 4.9274e-05, 5.0724e-05,
5.1959e-05, 5.2958e-05, 5.3087e-05, 5.3999e-05, 5.4986e-05, 5.5417e-05,
5.592e-05, 5.6276e-05, 5.7091e-05, 5.7659e-05, 5.8588e-05, 5.8911e-05,
5.9347e-05, 6.0369e-05, 6.0784e-05, 6.1513e-05, 6.1974e-05, 6.3907e-05,
6.4217e-05, 6.53e-05, 6.6645e-05, 6.7606e-05, 6.7791e-05, 6.8862e-05,
6.9435e-05, 6.9947e-05, 7.1e-05, 7.1214e-05, 7.2755e-05, 7.3795e-05,
7.4935e-05, 7.6325e-05, 7.6983e-05, 7.8369e-05, 7.9595e-05, 8.008e-05,
8.0405e-05, 8.1462e-05, 8.4584e-05, 8.5489e-05, 8.811e-05, 8.8859e-05,
8.9357e-05, 9.077e-05, 9.2593e-05, 9.5605e-05, 9.6161e-05, 9.7325e-05,
9.7938e-05, 9.9273e-05, 0.000100154, 0.000100714, 0.000103258,
0.000104261, 0.000104848, 0.000105045, 0.000106708, 0.00010781,
0.000108898, 0.000109399, 0.000111046, 0.000112114, 0.000113095,
0.000114353, 0.000115575, 0.000116267, 0.000118189, 0.000118745,
0.000119753, 0.000120482, 0.000121507, 0.000122095, 0.00012346,
0.000125392, 0.000126507, 0.000129093, 0.000130551, 0.000132507,
0.000133447, 0.000134879, 0.000137351, 0.000139148, 0.00014054,
0.000142284, 0.0001439, 0.00014488, 0.000146531, 0.000147152,
0.000148488, 0.000149579, 0.000150744, 0.000152047, 0.000153371,
0.000154106, 0.00015606, 0.00015708, 0.000158669, 0.000160688,
0.000163011, 0.00016419, 0.000166789, 0.000168079, 0.000170232,
0.000172431, 0.000174337, 0.000175728, 0.000177814, 0.000179076,
0.000181937, 0.000184129, 0.000186687, 0.000189155, 0.000191258,
0.000192473, 0.000194548, 0.000196885, 0.000198096, 0.0002003,
0.000203664, 0.000206278, 0.00020807, 0.000209829, 0.000211699,
0.000213095, 0.000215329, 0.000217316, 0.000219443, 0.000221027,
0.000222918, 0.000225052, 0.000227004, 0.000228799, 0.000230623,
0.000233094, 0.00023592, 0.000237484, 0.000240574, 0.000243835,
0.000246973, 0.000251189, 0.00025431, 0.000257105, 0.000259253,
0.000263075, 0.000265521, 0.0002684, 0.000271877, 0.000274366,
0.000276811, 0.000281744, 0.000284244, 0.000286038, 0.00028939,
0.000295676, 0.000299303, 0.000303533, 0.000307541, 0.000312855,
0.000315645, 0.000319724, 0.000324628, 0.000329116, 0.000333104,
0.000336953, 0.000340689, 0.000343192, 0.00034782, 0.000351803,
0.000356268, 0.000360921, 0.00036406, 0.000368748, 0.000372736,
0.00037745, 0.000382522, 0.000387406, 0.000391622, 0.000394855,
0.000401068, 0.000405696, 0.000411255, 0.000417248, 0.000421599,
0.000425071, 0.000429984, 0.000435364, 0.000441678, 0.000447928,
0.000452185, 0.000456249, 0.000461861, 0.00046784, 0.00047247,
0.000478592, 0.00048337, 0.000488966, 0.00049625, 0.000501319,
0.000507632, 0.000513116, 0.000518072, 0.000523313, 0.000530276,
0.000535429, 0.000541265, 0.000547466, 0.000553451, 0.000562153,
0.000569329, 0.000575598, 0.000582245, 0.000589352, 0.000595505,
0.000602347, 0.000609259, 0.000615674, 0.000622014, 0.000628936,
0.000637991, 0.000643839, 0.000651651, 0.000659401, 0.000670168,
0.000681384, 0.000689048, 0.000697732, 0.000705547, 0.000712816,
0.000720061, 0.000727273, 0.000735817, 0.000742163, 0.000749501,
0.000758206, 0.000767112, 0.000775057, 0.000782356, 0.000788241,
0.000794981, 0.000804576, 0.000814258, 0.000822131, 0.000832088,
0.000843104, 0.000850698, 0.000857672, 0.000867094, 0.000876794,
0.000886068, 0.000897049, 0.000906755, 0.000917242, 0.000928563,
0.000938324, 0.00094862, 0.000959582, 0.000969364, 0.000979292,
0.000991039, 0.001002089, 0.001013865, 0.001027939, 0.00103962,
0.001050135, 0.001059193, 0.001062461, 0.00107824, 0.001090818,
0.001101962, 0.001115178, 0.001127784, 0.001137825, 0.001149981,
0.001160579, 0.001173463, 0.001187747, 0.00120098, 0.001213668,
0.001232128, 0.001246793, 0.001262567, 0.001276638, 0.001291745,
0.001305286, 0.00132156, 0.001335842, 0.001354054, 0.001370372,
0.001387485, 0.001404099, 0.001422324, 0.001438359, 0.001452987,
0.001471639, 0.001491647, 0.001507602, 0.001526961, 0.001545932,
0.001564062, 0.001582893, 0.001599386, 0.001619272, 0.001637045,
0.001655626, 0.001672941, 0.001692533, 0.00171318, 0.001733724,
0.001754772, 0.001775471, 0.001797629, 0.001817998, 0.001840528,
0.001860266, 0.001879788, 0.001900636, 0.001921016, 0.001941105,
0.00196539, 0.001992672, 0.002011813, 0.002032412, 0.002055079,
0.002075867, 0.002103918, 0.002124837, 0.002151084, 0.00218006,
0.002204168, 0.002233454, 0.002256756, 0.002281981, 0.002311798,
0.002336818, 0.002363182, 0.002388778, 0.002420165, 0.002445716,
0.002476578, 0.002505192, 0.002535139, 0.002559801, 0.002594495,
0.002619292, 0.002655543, 0.002685624, 0.002712511, 0.002744206,
0.002778383, 0.002816116, 0.002850697, 0.002882313, 0.002913297,
0.002949434, 0.002979888, 0.003014668, 0.003049562, 0.003088279,
0.003121772, 0.003158025, 0.003196779, 0.00323089, 0.003271365,
0.003307848, 0.003349713, 0.00338827, 0.003425124, 0.003471762,
0.003509171, 0.003553734, 0.003595781, 0.003643731, 0.003684672,
0.003730534, 0.00377162, 0.003815931, 0.003861329, 0.003906293,
0.00394849, 0.003994191, 0.004039181, 0.004084892, 0.00413336,
0.004175296, 0.004222356, 0.004268406, 0.004316003, 0.004368866,
0.004424117, 0.004472958, 0.004523195, 0.004571199, 0.004627289,
0.004683818, 0.004743053, 0.004794654, 0.004853115, 0.004907556,
0.004962485, 0.005012138, 0.005063829, 0.005122648, 0.005177126,
0.005235906, 0.005300626, 0.005361236, 0.005422632, 0.00549376,
0.005554307, 0.005617404, 0.005683021, 0.005752425, 0.005821914,
0.005888533, 0.005961669, 0.006024391, 0.006089921, 0.006158429,
0.006227853, 0.006295536, 0.006372634, 0.006448759, 0.006518778,
0.006598477, 0.006676279, 0.006750067, 0.006830177, 0.006906,
0.006980973, 0.007058179, 0.007145763, 0.007224822, 0.007307814,
0.007387226, 0.007466012, 0.007551021, 0.00763989, 0.007730255,
0.007813552, 0.007904621, 0.00799612, 0.008094422, 0.008190973,
0.008287425, 0.008383547, 0.008470813, 0.008567856, 0.008663977,
0.008760746, 0.00886911, 0.008969825, 0.009078002, 0.009171032,
0.009277574, 0.009386748, 0.009490449, 0.009603885, 0.009711037,
0.009822521, 0.009948059, 0.010058413, 0.010169728, 0.010288038,
0.010413405, 0.010524267, 0.010619268, 0.010644417, 0.010765856,
0.010898346, 0.011027812, 0.011155879, 0.011283296, 0.011416768,
0.011548391, 0.011686219, 0.01181406, 0.011950967, 0.012088283,
0.012231189, 0.01237423, 0.012527451, 0.01267378, 0.012819938,
0.012962279, 0.013116772, 0.013267997, 0.013422675, 0.013576091,
0.013731761, 0.013897283, 0.01407281, 0.014227037, 0.014387631,
0.014558172, 0.014718529, 0.014880458, 0.015053452, 0.015228204,
0.015401633, 0.015584076, 0.01577357, 0.015960091, 0.016145884,
0.016338155, 0.016526725, 0.016718581, 0.016907605, 0.017102862,
0.017292104, 0.017498125, 0.017693058, 0.017894404, 0.01810058,
0.018313131, 0.018525688, 0.018748961, 0.018952532, 0.019180808,
0.019408553, 0.019635838, 0.01986561, 0.020090225, 0.020334446,
0.020565416, 0.020802533, 0.021039759, 0.021277174, 0.02152185,
0.021780839, 0.022036802, 0.022288444, 0.02252659, 0.022776259,
0.023031832, 0.023295002, 0.023556568, 0.02381309, 0.024085019,
0.024356025, 0.02462488, 0.024920869, 0.025214909, 0.025510377,
0.025801832, 0.026098792, 0.026400144, 0.026708872, 0.027022399,
0.027337502, 0.027654443, 0.027970077, 0.028293166, 0.028619996,
0.028952127, 0.029285014, 0.029614978, 0.029961808, 0.03030048,
0.030640518, 0.030986692, 0.031341669, 0.031693112, 0.032071062,
0.032443039, 0.032819107, 0.033205587, 0.03359287, 0.033985402,
0.034368088, 0.034775676, 0.03518514, 0.0355713, 0.035977969,
0.036381636, 0.036807348, 0.03722512, 0.037646165, 0.038088373,
0.038519738, 0.038958947, 0.039412355, 0.039872338, 0.040335748,
0.040806541, 0.04126531, 0.041740714, 0.042201966, 0.042667851,
0.043148961, 0.043640955, 0.044146186, 0.044648078, 0.045140655,
0.045656294, 0.046192771, 0.046725155, 0.047279468, 0.047841521,
0.048378238, 0.048931074, 0.049485475, 0.050036681, 0.050613349,
0.051210901, 0.051783706, 0.052371462, 0.052971063, 0.053591109,
0.054194754, 0.054824948, 0.055433284, 0.056068564, 0.056709205,
0.057327402, 0.057993993, 0.058660753, 0.059335311, 0.060016228,
0.060701055, 0.06140114, 0.062111953, 0.062804692, 0.063521759,
0.064266362, 0.065019126, 0.065758322, 0.066487141, 0.067245564,
0.068024153, 0.068806415, 0.069613844, 0.070428008, 0.071224237,
0.072042407, 0.072844003, 0.073660222, 0.074508963, 0.075362091,
0.076211346, 0.077052226, 0.07791137, 0.078800091, 0.079698564,
0.080590185, 0.08150072, 0.082441213, 0.083388798, 0.084311804,
0.085255847, 0.086238772, 0.087237018, 0.088222273, 0.089220544,
0.090222164, 0.09126791, 0.092322419, 0.093381202, 0.094465764,
0.095557228, 0.096663489, 0.097775587, 0.098893855, 0.100043326,
0.101164997, 0.102328219, 0.103464346, 0.104395852, 0.104648447,
0.105851244, 0.107051945, 0.108265081, 0.109486873, 0.110705413,
0.111969023, 0.11326186, 0.11452615, 0.115818733, 0.117138356,
0.118460365, 0.119798257, 0.12118723, 0.122570138, 0.123976191,
0.125404718, 0.126843682, 0.128306592, 0.129767224, 0.131247854,
0.132734943, 0.13424515, 0.13575281, 0.137284489, 0.138834957,
0.140410238, 0.142034214, 0.143671018, 0.14529447, 0.14692176,
0.148593437, 0.150298132, 0.152012395, 0.15372701, 0.155462416,
0.157195034, 0.158984705, 0.160794716, 0.16261291, 0.16445612,
0.166310233, 0.168183763, 0.170097821, 0.17201673, 0.173996873,
0.175982709, 0.17798229, 0.179985419, 0.182059109, 0.184121459,
0.186188341, 0.188297482, 0.190406754, 0.192579061, 0.194730342,
0.196926631, 0.199154601, 0.201432716, 0.203682554, 0.20599302,
0.20831785, 0.210701685, 0.213060161, 0.215443731, 0.217885245,
0.220313972, 0.222806449, 0.225339306, 0.227899585, 0.230485911,
0.23310306, 0.235741064, 0.238372325, 0.241071267, 0.243783335,
0.246572057, 0.249342078, 0.252144108, 0.254930084, 0.257816901,
0.260763213, 0.263730565, 0.266779329, 0.269765166, 0.272834018,
0.27591923, 0.279026211, 0.2821618, 0.285381285, 0.288598803,
0.291796123, 0.295104229, 0.298464186, 0.301851634, 0.305304813,
0.306052404, 0.308783657, 0.312280535, 0.315809044, 0.319387442,
0.322970255, 0.326634551, 0.330314176, 0.334081149, 0.337856923,
0.341654674, 0.345495742, 0.349383073, 0.353314049, 0.357344901,
0.361391437, 0.365471267, 0.369596363, 0.373792564, 0.378036632,
0.382364809, 0.386706614, 0.39111015, 0.39554839, 0.400076763,
0.404629074, 0.4055502, 0.40922258, 0.413872244, 0.418650814,
0.423442293, 0.428253505, 0.433130218, 0.438011388, 0.443027854,
0.448041277, 0.453134624, 0.458405887, 0.463619489, 0.468840604,
0.474205853, 0.479642096, 0.485136876, 0.490676124, 0.496355682,
0.502053204, 0.507860021, 0.513722913, 0.519644769, 0.525658872,
0.531665227, 0.53774407, 0.544012912, 0.550270408, 0.556611739,
0.562993785, 0.569525648, 0.576165666, 0.582916898, 0.589688691,
0.596582753, 0.603608225, 0.606546515, 0.610657662, 0.617785675,
0.625014259, 0.632309719, 0.639748927, 0.647279766, 0.654883982,
0.662592452, 0.67046725, 0.678431805, 0.686533283, 0.694695606,
0.702986634, 0.711407735, 0.719992418, 0.728653949, 0.737455154,
0.746401519, 0.755453363, 0.764564917, 0.77379886, 0.783210999,
0.792738527, 0.802492791, 0.812368384, 0.816501157, 0.822412298,
0.83262617, 0.842993439, 0.853471297, 0.864248162, 0.87509903,
0.886146568, 0.897442748, 0.908838744, 0.920451765, 0.932286407,
0.944321314, 0.956539342, 0.969052686, 0.981751396, 0.994664249,
1.007762529, 1.021150752, 1.034903764, 1.045971156, 1.071147008,
1.08504851, 1.09894747, 1.112934178, 1.12678466, 1.14063567,
1.15442433, 1.168255229, 1.181962236, 1.195704255, 1.209387112,
1.223109377, 1.236793461, 1.250488417, 1.264155409, 1.277786606,
1.291357271, 1.304973665, 1.312958486, 1.318556824, 1.33214701,
1.345776225, 1.359515843, 1.373133829, 1.386759332, 1.400464537,
1.414080075, 1.427662151, 1.441278436, 1.454784708, 1.468475646,
1.482188694, 1.495796401, 1.509549761, 1.523354229, 1.537147022,
1.550975022, 1.564848446, 1.578661837, 1.592557944, 1.60650114,
1.62058513, 1.634622868, 1.648669883, 1.656926778, 1.662904237,
1.67715719, 1.691467323, 1.705861735, 1.720313521, 1.734779826,
1.749320517, 1.763857006, 1.778560208, 1.793317132, 1.808128119,
1.82313953, 1.838260911, 1.853316344, 1.868607652, 1.883931063,
1.899400637, 1.914908088, 1.930463792, 1.946190188, 1.962209563,
1.978289188, 1.994603034, 2.011028416, 2.027640967, 2.044069188,
2.0609797, 2.077829215, 2.094827272, 2.1121563, 2.129561953,
2.147269268, 2.165055853, 2.183271538, 2.201652034, 2.216383304,
2.220296146, 2.239072547, 2.258094496, 2.277097833, 2.296695778,
2.316442028, 2.336703773, 2.357021251, 2.377422945, 2.398320606,
2.419532696, 2.441185314, 2.463177549, 2.485412981, 2.507973035,
2.530856348, 2.554232711, 2.577635212, 2.601566362, 2.625679517,
2.650228605, 2.675290556, 2.700785776, 2.726713563, 2.753080571,
2.773604781, 2.779474412, 2.806890452, 2.834516935, 2.862565446,
2.890434042, 2.919462559, 2.948578755, 2.978466804, 3.008281381,
3.039189148, 3.070741253, 3.101981835, 3.133375521, 3.1656192,
3.19817405, 3.231395108, 3.264653605, 3.298528191, 3.332020173,
3.366773328, 3.401610069, 3.437096945, 3.472554345, 3.508468108,
3.544868085, 3.582038271, 3.619584691, 3.656439276, 3.693699426,
3.731434023, 3.76951363, 3.807693778, 3.846552162, 3.88609297,
3.925807265, 3.965728379, 4.005852726, 4.046162153, 4.086919106,
4.128644042, 4.170203881, 4.211527332, 4.25390359, 4.295896816,
4.338314449, 4.380751581, 4.424580169, 4.467944547, 4.512146529,
4.556031186, 4.600118197, 4.643789146, 4.688494866, 4.734390072,
4.779867065, 4.825232272, 4.871172697, 4.917329812, 4.964267632,
5.011349415, 5.058884445, 5.106295502, 5.154171531, 5.202431355,
5.250347925, 5.299443052, 5.348028736, 5.398153912, 5.448685118,
5.499437573, 5.549394869, 5.600478971, 5.651350172, 5.703244347,
5.754749218, 5.80714069, 5.859322785, 5.912968738, 5.9663709,
6.01985651, 6.075727556, 6.129659458, 6.184167214, 6.240156729,
6.295798237, 6.352257791, 6.407955272, 6.464258346, 6.520293649,
6.5766751, 6.632766125, 6.68950642, 6.74557258, 6.803399922,
6.861731797, 6.919786869, 6.931742664, 6.978862994, 7.038565284,
7.098122627, 7.157883189, 7.217716068, 7.27881002, 7.340433075,
7.401899123, 7.463081661, 7.52372135, 7.584670174, 7.647523403,
7.710991158, 7.776059666, 7.838460073, 7.903275306, 7.966897788,
8.030443345, 8.095294939, 8.160895804, 8.225326736, 8.290595616,
8.355701281, 8.421979912, 8.490540418, 8.557967813, 8.627737032,
8.69796975, 8.76464612, 8.834787584, 8.902892239, 8.974864462,
9.04455907, 9.114996327, 9.185543471, 9.256143616, 9.326276535,
9.3989204, 9.471711212, 9.545436676, 9.615871352, 9.690642189,
9.766056572, 9.840187402, 9.914450653, 9.989702416, 10.065507183,
10.140756498, 10.216127682, 10.291476283, 10.366652221, 10.445692542,
10.523656777, 10.598963145, 10.67793459, 10.754626484, 10.829988691,
10.909180264, 10.986958286, 11.069281166, 11.146798424, 11.224868599,
11.306601372, 11.38568575, 11.467293832, 11.547918622, 11.630175013,
11.710875847, 11.795179386, 11.877810374, 11.96222071, 12.043774485,
12.126337636, 12.210454876, 12.298304804, 12.381032687, 12.466099066,
12.552640516, 12.639025457, 12.725792235, 12.815096219, 12.902151487,
12.98779967, 13.076105984, 13.166304779, 13.258415076, 13.346521898,
13.432944986, 13.521615304, 13.608930936, 13.705297782, 13.795582379,
13.886958852, 13.978430244, 14.07076749, 14.164611537, 14.259113718,
14.352678787, 14.448788719, 14.540331847, 14.63807769, 14.734102132,
14.828524837, 14.924013658, 15.021167262, 15.120611159, 15.218521165,
15.313756291, 15.410973957, 15.508367981, 15.605626876, 15.707885728,
15.807728083, 15.906838076, 16.007656926, 16.100352435, 16.205647267,
16.302054699, 16.401711819, 16.505327933, 16.613051764, 16.71366502,
16.81635126, 16.916363929, 17.018989708, 17.119397414, 17.229002831,
17.335612537, 17.44635848, 17.549655892, 17.660274829, 17.767366268,
17.87427101, 17.987675728, 18.096612567, 18.204907084, 18.311662714,
18.421255605, 18.531193361, 18.643237008, 18.754681152, 18.863520003,
18.972502378, 19.087232682, 19.195876097, 19.309703701, 19.420056906,
19.538538214, 19.656585412, 19.77473586, 19.893604109, 20.013732996,
20.127669764, 20.236686879, 20.352602784, 20.479815258, 20.598888453,
20.712349413, 20.828058452, 20.947896707, 21.065203894, 21.189808576,
21.31192235, 21.427569818, 21.542641122, 21.661085589, 21.779559872,
21.89767617, 22.022703769, 22.144771994, 22.267941913, 22.393259725,
22.514636978, 22.640689832, 22.768612123, 22.89202267, 23.009280672,
23.132089256, 23.257321551, 23.375603093, 23.505029116, 23.625992109,
23.758502333, 23.87764876, 24.003120792, 24.131434875, 24.257996217,
24.393276955, 24.520260152, 24.649282554, 24.785768185, 24.917504049,
25.046816256, 25.168129888, 25.297801936, 25.425661178, 25.539681339,
25.675334564, 25.797834392, 25.934010143, 25.963193763, 26.072087573,
26.200788783, 26.33663465, 26.473642388, 26.606596223, 26.741222345,
26.875779379, 27.014474957, 27.145581151, 27.280969075, 27.417822508,
27.567778134, 27.700520273, 27.830713001, 27.967409715, 28.107566306,
28.247241366, 28.374605205, 28.506864319, 28.658122721, 28.797672392,
28.945087212, 29.0963734, 29.244544481, 29.386500841, 29.535451316,
29.664603134, 29.815664425, 29.961424418, 30.119823331, 30.262308468,
30.414748161, 30.55925268, 30.714182421, 30.875735476, 31.029672324,
31.175896196, 31.336718269, 31.486421033, 31.641463077, 31.779540592,
31.940514013, 32.085490789, 32.248118381, 32.392568625, 32.532276738,
32.689258095, 32.839487408, 33.000902334, 33.150571111, 33.308315508,
33.439345615, 33.585716656, 33.751230005, 33.901257413, 34.065056949,
34.228034292, 34.375555843, 34.542010007, 34.704826135, 34.862088038,
35.01425944, 35.166678289, 35.330063835, 35.481703564, 35.642863069,
35.798178597, 35.96058598, 36.111960441, 36.272718875, 36.427278001,
36.588014112, 36.770423835, 36.929400181, 37.096456206, 37.265701951,
37.423116433, 37.58269047, 37.763650849, 37.943815824, 38.117667292,
38.285359836, 38.427050374, 38.595170735, 38.738333047, 38.907994611,
39.063459541, 39.222057064, 39.378576673, 39.533715364, 39.716737424,
39.903944908, 40.086115317, 40.250411055, 40.425622557, 40.597422525,
40.776667221, 40.943420697, 41.114709879, 41.287887407, 41.468926079,
41.627576992, 41.798171396, 41.995100364, 42.169895945, 42.347677803,
42.512319717, 42.703643273, 42.879700228, 43.063387538, 43.237098813,
43.403134376, 43.59578143, 43.776332257, 43.959857809, 44.136891746,
44.304729692, 44.460772298, 44.633950166, 44.795123191, 44.962310099,
45.134201681, 45.328150206, 45.503915376, 45.71401254, 45.89800616,
46.095529998, 46.264885497, 46.436543957, 46.62608965, 46.830275263,
46.978328375, 47.138486294, 47.318158002, 47.498291573, 47.693259633,
47.878818937, 48.042940633, 48.191315382, 48.39859832, 48.595405679,
48.806128206, 48.959659331, 49.139823241, 49.339930039, 49.531630484,
49.703912614, 49.912108449, 50.085530654, 50.285660792, 50.488027259,
50.663953309, 50.889504002, 51.092554149, 51.298347426, 51.522537319,
51.737282679, 51.880856993, 52.116775455, 52.309843379, 52.514434064,
52.71728739, 52.930969139, 53.096184296, 53.308425189, 53.521320004,
53.679089813, 53.880569228, 54.048479958, 54.228672477, 54.418935054,
54.586751305, 54.786446117, 54.995215828, 55.198728011, 55.386684118,
55.581963841, 55.784685536, 55.986042817, 56.166438825, 56.339692303,
56.563076058, 56.758486576, 56.988142544, 57.18912896, 57.412195993,
57.611511132, 57.833726537, 58.050529422, 58.301941015, 58.510327181,
58.743947349, 58.932528945, 59.137855565, 59.358861778, 59.583992102,
59.881304219, 60.066231782, 60.263685594, 60.483380108, 60.507761042,
60.676082048, 60.873485363, 61.11840179, 61.333260855, 61.533419952,
61.751462725, 61.974838907, 62.165950149, 62.363935329, 62.557703775,
62.751444906, 62.909723147, 63.06619517, 63.288758637, 63.468265451,
63.77130707, 63.971473607, 64.159788759, 64.345321276, 64.558086425,
64.78873441, 64.948761622, 65.153255361, 65.412461461, 65.599534021,
65.825130455, 66.015147527, 66.221104826, 66.416652523, 66.627797515,
66.793122639, 67.026231073, 67.20800774, 67.419823391, 67.732410001,
67.957899541, 68.208295959, 68.392500607, 68.581187935, 68.840555801,
69.095078074, 69.332950467, 69.557699415, 69.712236524, 69.904037101,
70.141175621, 70.330683347, 70.607401872, 70.873608654, 71.073030149,
71.315751464, 71.505718916, 71.728399981, 71.985437846, 72.247074408,
72.422839636, 72.650518382, 72.891087341, 73.079714751, 73.282104991,
73.52734859, 73.773397567, 73.992250657, 74.234750453, 74.413445794,
74.660350445, 74.885235054, 75.071426379, 75.258767098, 75.46388081,
75.656448592, 75.833900543, 76.075094452, 76.355793546, 76.623073833,
76.858508105, 77.063715437, 77.219277916, 77.459726857, 77.63564141,
77.947559656, 78.14167472, 78.396445706, 78.590257005, 78.72264449,
79.016511421, 79.252393848, 79.433239825, 79.628187206, 79.859067895,
80.023439022, 80.187434007, 80.403162674, 80.605471905, 80.913954595,
81.124182368, 81.248044432, 81.472293182, 81.681197292, 81.890631713,
82.120379431, 82.257732935, 82.470034678, 82.670571112, 82.893470083,
83.08308427, 83.344835477, 83.698114933, 83.938086208, 84.128091741,
84.343981087, 84.541232082, 84.804394298, 85.021720136, 85.206610623,
85.550338201, 85.771973071, 85.979450507, 86.184537636, 86.430283294,
86.740601741, 87.129004258, 87.339532774, 87.550032691, 87.812460277,
88.005290895, 88.336000046, 88.498674087, 88.707318031, 89.030638588,
89.278743205, 89.521253438, 89.769914309, 90.036975461, 90.226632836,
90.41902623, 90.604884124, 90.813244166, 90.983387726, 91.216755744,
91.431628889, 91.666896964, 91.832806752, 92.011340533, 92.133715011,
92.277519741, 92.52732875, 92.799224004, 92.904251721, 93.22229394,
93.410129829, 93.650758786, 93.948652661, 94.351479254, 94.671381961,
94.847490597, 95.122994779, 95.335072102, 95.614094358, 96.074507714,
96.294110008, 96.648127084, 96.882460741, 97.106404772, 97.261329399,
97.644870762, 97.850639225, 97.945543327, 98.177419216, 98.480910386,
98.575623014, 98.774477041, 98.956630791, 99.144696059, 99.340220679,
99.536674742, 99.99320855, 100.177893879, 100.329155381, 100.44993558,
100.597918378, 100.763109498, 100.94371307, 101.153678727, 101.311839367,
101.430438257, 101.578860762, 101.823881729, 102.17179555, 102.461684737,
102.664374686, 102.964709876, 103.141442528, 103.291312518, 103.601921191,
103.871847861, 104.360261695, 104.728748251, 104.980757231, 105.253278959,
105.483450195, 105.56157741, 105.820836947, 106.238943297, 106.460567071,
106.571389735, 106.856502809, 107.021957387, 107.177718715, 107.652789409,
107.763471628, 107.980247986, 108.233260173, 108.450163059, 108.605297355,
108.846439465, 109.120053441, 109.324711923, 109.53211782, 109.851434801,
110.169604442, 110.313106726, 110.614933851, 110.955088274, 111.081738318,
111.408807437, 111.670948009, 111.853081469, 111.989439218, 112.101574801,
112.394644261, 112.483526604, 112.687516624, 112.766089398, 113.377482442,
113.484959196, 113.606811802, 113.716551507, 114.006853693, 114.192406057,
114.293383234, 114.644706396, 115.163970778, 115.698509077, 116.320815797,
116.821105423, 116.89808776, 117.607328149, 117.795788977, 118.081088046,
118.265115771, 118.678073383, 118.85750606, 118.958238405, 119.035764296,
119.121625508, 119.867184876, 120.139387451, 120.241000398, 120.491235983,
120.574248738, 120.881499423, 121.004672116, 121.342956851, 121.529488733,
121.660584521, 121.722714089, 122.060594137, 122.237059595, 122.350677802,
122.752070097, 122.897544387, 123.151597004, 123.221073949, 123.599122174,
123.774131605, 123.890683992, 124.358382896, 124.423141801, 124.466363107,
124.617381418, 124.776292159, 125.125186208, 125.367623599, 125.6085467,
125.647068425, 125.730697292, 125.916103177, 126.156308619, 126.479797574,
126.795154463, 127.005448825, 127.01680762, 127.306572899, 127.600265345,
127.695888804, 127.951327568, 128.641858837, 128.830184667, 129.253056492,
129.585925047, 129.894576922, 130.149186998, 130.599068079, 130.807754452,
130.939593639, 131.097325005, 131.131648253, 131.500492701, 131.689181219,
131.734235591, 131.944280865, 132.469694218, 132.730447788, 133.200888014,
133.424867868, 133.505314196, 133.624568779, 133.784010538, 133.824489228,
133.956346669, 134.079522074, 134.143642238, 134.485427343, 134.485827464,
134.654472668, 134.712571757, 134.720142325, 135.061623134, 135.537580798,
135.615527016, 135.857949492, 135.917030005, 136.149270873, 136.192068238,
136.298114, 136.489901117, 136.869211454, 136.948927687, 137.089489716
), "35" = c(1.5838e-05, 1.5993e-05, 1.6104e-05, 1.6612e-05, 1.6615e-05,
1.6621e-05, 1.6731e-05, 1.6991e-05, 1.7132e-05, 1.7212e-05, 1.7214e-05,
1.7315e-05, 1.7511e-05, 1.7607e-05, 1.7814e-05, 1.7877e-05, 1.8108e-05,
1.8454e-05, 1.9067e-05, 1.9129e-05, 1.9339e-05, 1.9512e-05, 1.9847e-05,
1.9883e-05, 2.011e-05, 2.0195e-05, 2.0252e-05, 2.0303e-05, 2.0576e-05,
2.1588e-05, 2.2032e-05, 2.2124e-05, 2.2326e-05, 2.2402e-05, 2.3037e-05,
2.3401e-05, 2.3432e-05, 2.365e-05, 2.3798e-05, 2.4518e-05, 2.4632e-05,
2.4817e-05, 2.5628e-05, 2.5791e-05, 2.6087e-05, 2.6285e-05, 2.695e-05,
2.7386e-05, 2.7995e-05, 2.8023e-05, 2.8454e-05, 2.8796e-05, 2.8909e-05,
2.9285e-05, 2.9779e-05, 3.0787e-05, 3.1447e-05, 3.1813e-05, 3.2358e-05,
3.2988e-05, 3.3341e-05, 3.3541e-05, 3.3681e-05, 3.5001e-05, 3.5246e-05,
3.5968e-05, 3.6426e-05, 3.6574e-05, 3.6929e-05, 3.7553e-05, 3.8091e-05,
3.8572e-05, 3.881e-05, 3.9102e-05, 3.9217e-05, 3.956e-05, 4.033e-05,
4.1878e-05, 4.2489e-05, 4.3064e-05, 4.3512e-05, 4.3855e-05, 4.4424e-05,
4.4929e-05, 4.5719e-05, 4.598e-05, 4.6461e-05, 4.6737e-05, 4.7406e-05,
4.8231e-05, 4.8628e-05, 4.8926e-05, 4.9624e-05, 5.0693e-05, 5.1576e-05,
5.1922e-05, 5.2416e-05, 5.3038e-05, 5.4095e-05, 5.4351e-05, 5.5548e-05,
5.6717e-05, 5.7708e-05, 5.8348e-05, 5.8917e-05, 6.0067e-05, 6.1404e-05,
6.2087e-05, 6.2662e-05, 6.3314e-05, 6.4057e-05, 6.4759e-05, 6.6117e-05,
6.7003e-05, 6.753e-05, 6.7898e-05, 6.9269e-05, 6.9954e-05, 7.1506e-05,
7.2117e-05, 7.2902e-05, 7.4234e-05, 7.4675e-05, 7.6382e-05, 7.7076e-05,
7.7595e-05, 7.8754e-05, 7.9869e-05, 8.1066e-05, 8.2687e-05, 8.4621e-05,
8.5339e-05, 8.5784e-05, 8.763e-05, 8.8125e-05, 8.8593e-05, 8.9589e-05,
9.0587e-05, 9.1324e-05, 9.2401e-05, 9.306e-05, 9.4567e-05, 9.5483e-05,
9.6193e-05, 9.7225e-05, 9.8182e-05, 9.9619e-05, 0.000100356,
0.000101525, 0.000103625, 0.000104584, 0.000105976, 0.000106918,
0.00010798, 0.000109622, 0.000111681, 0.000113557, 0.000115071,
0.000116623, 0.000118584, 0.000120122, 0.00012176, 0.000123429,
0.000124168, 0.000126009, 0.00012748, 0.00012874, 0.000129845,
0.00013276, 0.000135902, 0.000137063, 0.000139689, 0.000140897,
0.000142107, 0.000143982, 0.00014646, 0.000147254, 0.000149847,
0.000151374, 0.000152506, 0.000153432, 0.00015562, 0.000156779,
0.000158122, 0.000159756, 0.000160951, 0.000161816, 0.000164077,
0.000165887, 0.000167622, 0.000168438, 0.000170624, 0.000172999,
0.000175341, 0.000178304, 0.00017987, 0.000181772, 0.000183899,
0.000186577, 0.000189493, 0.000191105, 0.000193132, 0.000195439,
0.000198325, 0.000202338, 0.000206887, 0.000210397, 0.000212948,
0.000215017, 0.000216461, 0.000218075, 0.000220853, 0.000225595,
0.000227706, 0.000231228, 0.000234218, 0.000237073, 0.000239564,
0.000241176, 0.00024358, 0.000247207, 0.000250286, 0.000252956,
0.000256189, 0.000259091, 0.000263383, 0.000267118, 0.000269729,
0.00027394, 0.000278443, 0.000280649, 0.000282786, 0.000286287,
0.000288249, 0.000290536, 0.000292653, 0.000295826, 0.000301563,
0.000304813, 0.000308416, 0.00031279, 0.00031527, 0.000318879,
0.000323046, 0.00032662, 0.000329854, 0.000334565, 0.00033941,
0.000343289, 0.000346941, 0.000351556, 0.000356877, 0.000360327,
0.000364661, 0.00036991, 0.000374878, 0.000379473, 0.000383823,
0.000388846, 0.000392232, 0.000396403, 0.000400083, 0.000403981,
0.000409081, 0.000413603, 0.000418264, 0.000421555, 0.000425575,
0.00042981, 0.00043551, 0.000439811, 0.000444945, 0.000450706,
0.000453261, 0.000457886, 0.000463454, 0.00046705, 0.000475035,
0.000479659, 0.00048459, 0.000489614, 0.000496136, 0.000500632,
0.00050692, 0.000510856, 0.000517255, 0.00052203, 0.000527236,
0.000533631, 0.000538229, 0.000545991, 0.000551747, 0.00055631,
0.000565621, 0.000572764, 0.000577865, 0.000581712, 0.000587792,
0.00059303, 0.000601038, 0.000608666, 0.000614069, 0.000620664,
0.000627864, 0.000638384, 0.000644394, 0.000653675, 0.000664289,
0.000668917, 0.000676001, 0.00068269, 0.000691245, 0.000700188,
0.000708146, 0.000716491, 0.000725612, 0.000734373, 0.000741224,
0.000749843, 0.000759047, 0.000765883, 0.000776118, 0.000786667,
0.000795656, 0.000806572, 0.000817679, 0.000828352, 0.000839936,
0.000850519, 0.000861373, 0.000871637, 0.000882553, 0.000892962,
0.000903389, 0.000912493, 0.000921067, 0.000930819, 0.000940572,
0.000950951, 0.000960871, 0.000969695, 0.000971939, 0.000982773,
0.000996131, 0.001005247, 0.001015629, 0.001029532, 0.001042934,
0.001054315, 0.001066518, 0.001081945, 0.001094998, 0.00110756,
0.001118304, 0.001133894, 0.001148263, 0.001163945, 0.001179029,
0.001192059, 0.001205464, 0.001219202, 0.001234111, 0.001247365,
0.00126044, 0.00127697, 0.001293318, 0.001308622, 0.001323705,
0.001340462, 0.001354962, 0.001372486, 0.001384869, 0.00140073,
0.001415475, 0.001432907, 0.001449223, 0.001467179, 0.001487703,
0.001501946, 0.001516135, 0.001534297, 0.001552229, 0.001568826,
0.001588694, 0.001610315, 0.001631819, 0.001651673, 0.001671373,
0.001688917, 0.001702917, 0.001723256, 0.001743486, 0.001765982,
0.001789164, 0.00180835, 0.001830026, 0.001847676, 0.001868095,
0.00188701, 0.001909544, 0.001930472, 0.001952028, 0.001976809,
0.002000305, 0.002026502, 0.002050552, 0.002076755, 0.002100624,
0.002125069, 0.002149496, 0.002173566, 0.002197898, 0.002227051,
0.002252861, 0.002276812, 0.002306146, 0.002332473, 0.002362706,
0.0023871, 0.002416739, 0.00244292, 0.002472085, 0.002500707,
0.002530453, 0.002559565, 0.002585932, 0.002618624, 0.002648693,
0.00267914, 0.002715982, 0.00274909, 0.002777172, 0.002810995,
0.002845879, 0.00287797, 0.002913569, 0.002947819, 0.002980834,
0.003015768, 0.003051896, 0.003087067, 0.003128145, 0.003164832,
0.003198976, 0.003235145, 0.003267768, 0.003310994, 0.003347518,
0.003385268, 0.003426784, 0.0034725, 0.00351621, 0.00355555,
0.003597687, 0.003639912, 0.00368469, 0.003729842, 0.003773617,
0.003818895, 0.003859491, 0.003904334, 0.003944737, 0.00399032,
0.004039528, 0.004083856, 0.004131587, 0.004180103, 0.004231245,
0.004285071, 0.004333214, 0.004383039, 0.004433153, 0.004490229,
0.004541426, 0.004595148, 0.004645446, 0.004702012, 0.004756862,
0.004813816, 0.004874489, 0.004932172, 0.004984575, 0.005038498,
0.005095237, 0.005150921, 0.005214476, 0.005276814, 0.005338038,
0.005397096, 0.0054628, 0.005528096, 0.005592859, 0.005656579,
0.005728494, 0.005797678, 0.005865291, 0.005928406, 0.005997769,
0.006066799, 0.00613641, 0.006202617, 0.006279935, 0.006342081,
0.00641674, 0.006490055, 0.006563769, 0.00663595, 0.006714596,
0.006793708, 0.006884802, 0.006966275, 0.007049833, 0.007136003,
0.007214468, 0.007296821, 0.007375637, 0.007457329, 0.00754643,
0.007630677, 0.007727789, 0.007826501, 0.007915764, 0.008009827,
0.008108028, 0.008201704, 0.008301936, 0.008394186, 0.008498097,
0.008594066, 0.008687508, 0.008781382, 0.008886101, 0.008996425,
0.009106262, 0.009208665, 0.009312218, 0.009421165, 0.009546241,
0.009654455, 0.009765016, 0.009876264, 0.009973045, 0.009995391,
0.010111036, 0.010229762, 0.010346381, 0.010465167, 0.010582833,
0.010699074, 0.010833237, 0.010954832, 0.011084718, 0.011219761,
0.011350383, 0.01147113, 0.011608352, 0.011743531, 0.011883754,
0.012028317, 0.012167037, 0.012303834, 0.012446116, 0.012602307,
0.012741323, 0.012881523, 0.013026005, 0.013176168, 0.013333037,
0.013496065, 0.013652963, 0.0138178, 0.013971475, 0.014132179,
0.014296884, 0.014456723, 0.014617244, 0.014782907, 0.014954774,
0.015121069, 0.015292675, 0.01546569, 0.015642292, 0.015810248,
0.015996285, 0.016180602, 0.01637586, 0.016565833, 0.016757518,
0.016945396, 0.017141766, 0.017340004, 0.017537652, 0.017745562,
0.017949955, 0.018154414, 0.018359387, 0.018569604, 0.018783775,
0.018993221, 0.019217639, 0.019439444, 0.019667108, 0.019892088,
0.020123683, 0.020350773, 0.020575746, 0.020821431, 0.021073084,
0.021308832, 0.021547237, 0.021807076, 0.022066695, 0.022337239,
0.022597693, 0.022859676, 0.023135482, 0.023412527, 0.023702195,
0.023966669, 0.024256458, 0.024516734, 0.024794881, 0.025088435,
0.025395626, 0.025699252, 0.025993374, 0.026280394, 0.026587314,
0.026876081, 0.027190134, 0.027497275, 0.02780642, 0.028130569,
0.028466136, 0.028792656, 0.029135966, 0.029463954, 0.029803439,
0.030161794, 0.030502054, 0.030863475, 0.031225622, 0.03160424,
0.031975081, 0.032346037, 0.0327156, 0.033097032, 0.033477069,
0.033849079, 0.034246577, 0.03464701, 0.03505482, 0.035466933,
0.03588037, 0.036294581, 0.036701528, 0.037137808, 0.037575655,
0.03803493, 0.03848246, 0.038921593, 0.039352995, 0.039806329,
0.040263854, 0.040729023, 0.041199384, 0.041674853, 0.042155139,
0.042639445, 0.043139299, 0.043638372, 0.044160686, 0.04465973,
0.045176478, 0.045698965, 0.046255845, 0.046788018, 0.047348346,
0.047893499, 0.048456487, 0.049007845, 0.049561763, 0.050147681,
0.050723664, 0.051287229, 0.051879981, 0.052477479, 0.05305997,
0.053663422, 0.054288488, 0.054915319, 0.05554912, 0.056188568,
0.056843716, 0.057496682, 0.058161503, 0.05883591, 0.059497208,
0.060196317, 0.060898196, 0.061615731, 0.062333393, 0.063046273,
0.063772043, 0.064510662, 0.065263105, 0.06600264, 0.066770578,
0.067537014, 0.06832739, 0.069110714, 0.069916738, 0.070732124,
0.071546706, 0.072388907, 0.073224227, 0.074066768, 0.074901993,
0.07576499, 0.076655611, 0.077541992, 0.078455859, 0.079362795,
0.080282584, 0.081222109, 0.082136937, 0.083065114, 0.084014125,
0.084956492, 0.085939487, 0.086953917, 0.087955593, 0.088983699,
0.090048418, 0.091090137, 0.092129155, 0.093174026, 0.094219542,
0.095310199, 0.096418313, 0.097530333, 0.098646529, 0.099561669,
0.09979984, 0.100949651, 0.102123799, 0.10330525, 0.104506439,
0.105721404, 0.106977921, 0.108233695, 0.10946597, 0.110728912,
0.111993519, 0.113298479, 0.114640525, 0.11596575, 0.117282786,
0.118631128, 0.120026842, 0.121405224, 0.122831099, 0.124253392,
0.125685091, 0.127171228, 0.128662822, 0.130137362, 0.131661735,
0.133173379, 0.134759056, 0.136292765, 0.137887268, 0.139494372,
0.141109251, 0.14272606, 0.144387145, 0.146044188, 0.147725324,
0.149436942, 0.151160846, 0.152892434, 0.154637879, 0.156465248,
0.158297081, 0.160120828, 0.161956769, 0.163838249, 0.165718186,
0.167656211, 0.169605733, 0.171559338, 0.17354117, 0.175586077,
0.177603296, 0.179617572, 0.181724489, 0.183818171, 0.185924391,
0.188079157, 0.190242679, 0.192441604, 0.194641923, 0.196911684,
0.199159417, 0.201476044, 0.203800988, 0.206156464, 0.20854777,
0.210973084, 0.213428671, 0.215919498, 0.218426615, 0.220936999,
0.223510209, 0.226099777, 0.228776657, 0.231456224, 0.234138056,
0.236863087, 0.239607702, 0.242397409, 0.245190447, 0.24806333,
0.250957678, 0.25390331, 0.256855424, 0.25986237, 0.262917132,
0.265974269, 0.269052096, 0.272217653, 0.275413681, 0.278624276,
0.281874216, 0.285141307, 0.288486016, 0.291793022, 0.2951733,
0.298594224, 0.299378474, 0.302095267, 0.305623741, 0.309190504,
0.312798601, 0.316449755, 0.320163157, 0.323901091, 0.327688411,
0.331450561, 0.335345548, 0.33925286, 0.343215385, 0.347258545,
0.351317632, 0.355439833, 0.359614627, 0.36381065, 0.368106651,
0.372431291, 0.37679976, 0.381248876, 0.385761905, 0.390302784,
0.394856706, 0.399554803, 0.400529669, 0.404271275, 0.40906564,
0.413898937, 0.418846551, 0.423773953, 0.428794001, 0.433798472,
0.43893682, 0.444199867, 0.449480036, 0.454936108, 0.460353788,
0.465850012, 0.471448835, 0.477090068, 0.482793386, 0.488596682,
0.494410155, 0.500311342, 0.506341468, 0.512438161, 0.518588613,
0.524899265, 0.531276961, 0.537635087, 0.544159955, 0.550676521,
0.55739939, 0.564124492, 0.571003389, 0.577974208, 0.585004209,
0.592177608, 0.599437472, 0.606813524, 0.60998688, 0.614305501,
0.621966379, 0.62962423, 0.63738482, 0.645268463, 0.653210424,
0.661365196, 0.669569121, 0.677915474, 0.686397767, 0.695019308,
0.70366082, 0.712488022, 0.721493179, 0.730644838, 0.73998176,
0.749388438, 0.758945007, 0.768569438, 0.778395795, 0.788328174,
0.798469123, 0.808786066, 0.819275339, 0.829963452, 0.83436653,
0.840656699, 0.851777992, 0.862823102, 0.874187436, 0.88576081,
0.897667213, 0.909616193, 0.921813993, 0.934229221, 0.946892807,
0.95981637, 0.97292846, 0.986241445, 0.999882492, 1.013825632,
1.027954128, 1.042381226, 1.057089332, 1.072166194, 1.08435368,
1.112023707, 1.12740465, 1.142682019, 1.157984999, 1.173229156,
1.18844436, 1.203620769, 1.218823243, 1.233991629, 1.249218146,
1.264266682, 1.279348304, 1.294439814, 1.309474741, 1.324549802,
1.339463906, 1.354565194, 1.369529713, 1.378538685, 1.384754025,
1.399783859, 1.414845888, 1.429909492, 1.445131703, 1.460355225,
1.475388729, 1.490551648, 1.505601611, 1.520781451, 1.53583874,
1.550979098, 1.566182127, 1.581292676, 1.596435025, 1.611661666,
1.626907987, 1.642109555, 1.657390231, 1.672726101, 1.688039217,
1.703356304, 1.718781367, 1.734274855, 1.749635192, 1.75850715,
1.765161575, 1.780800186, 1.796552552, 1.81228478, 1.827976144,
1.843777098, 1.859725833, 1.875522506, 1.891504176, 1.907635738,
1.923646464, 1.939917857, 1.956276865, 1.972634626, 1.989100042,
2.005754381, 2.022408616, 2.039100163, 2.05589931, 2.072676231,
2.089646706, 2.106953554, 2.124083581, 2.141357346, 2.158594864,
2.176183056, 2.193753039, 2.211486601, 2.229489009, 2.247552575,
2.265840693, 2.28417654, 2.302642702, 2.320979666, 2.339792615,
2.354729264, 2.35853965, 2.377483097, 2.396792926, 2.416238803,
2.435644968, 2.45538105, 2.475273569, 2.495141048, 2.515400723,
2.535795037, 2.556518107, 2.577348767, 2.598479038, 2.619691357,
2.641461757, 2.663137368, 2.685225159, 2.707641229, 2.730129154,
2.75284215, 2.775804087, 2.798971505, 2.822380972, 2.846175351,
2.870395686, 2.889510135, 2.894845221, 2.919293296, 2.94448953,
2.969685762, 2.995473099, 3.021549062, 3.047692419, 3.074403477,
3.101699623, 3.129195253, 3.156615785, 3.184207799, 3.212705928,
3.241445392, 3.271059169, 3.300389128, 3.330277823, 3.360471858,
3.390996353, 3.421345723, 3.452670792, 3.484069033, 3.516211966,
3.548416342, 3.580471301, 3.613645842, 3.646969718, 3.680571607,
3.714724234, 3.748920418, 3.783291433, 3.817913573, 3.853088661,
3.888719296, 3.925072502, 3.961180358, 3.997600786, 4.034054362,
4.071468759, 4.108531794, 4.146260071, 4.18454227, 4.223535409,
4.26226376, 4.300595402, 4.340251187, 4.380218584, 4.419269321,
4.459291682, 4.500094692, 4.541120668, 4.582490729, 4.623873092,
4.665837469, 4.707791924, 4.749687923, 4.791834333, 4.83516217,
4.878101199, 4.921849009, 4.965457028, 5.0091109, 5.052462396,
5.096175685, 5.140556524, 5.184494286, 5.230344281, 5.276502524,
5.322360378, 5.368520862, 5.414613808, 5.460792713, 5.507248069,
5.553868839, 5.60202146, 5.650187795, 5.698260781, 5.747322635,
5.794906298, 5.843385116, 5.892413265, 5.941922685, 5.991102994,
6.042071054, 6.092520992, 6.143674549, 6.194755758, 6.245617252,
6.297311825, 6.348475101, 6.400920595, 6.453360111, 6.506283452,
6.560073434, 6.613785181, 6.667298269, 6.678777942, 6.722690663,
6.777692627, 6.832183551, 6.886478452, 6.941498993, 6.996606339,
7.052261316, 7.1085803, 7.164219432, 7.221463113, 7.277563678,
7.334695663, 7.39337572, 7.45125532, 7.509518914, 7.569355011,
7.626759506, 7.68503655, 7.744358157, 7.803460783, 7.86370427,
7.922737936, 7.985028939, 8.046502359, 8.106381184, 8.166275579,
8.226264372, 8.287874356, 8.35189338, 8.416082585, 8.481562371,
8.54421396, 8.608067523, 8.673613961, 8.737891572, 8.803511467,
8.869739133, 8.93729254, 9.004805306, 9.071543529, 9.137721189,
9.204160274, 9.272079183, 9.339530472, 9.407823579, 9.475968919,
9.544115017, 9.613499267, 9.681319684, 9.752622704, 9.822679086,
9.890778986, 9.963857479, 10.037350878, 10.107996217, 10.181040892,
10.25244534, 10.325537406, 10.398665784, 10.470260273, 10.542801165,
10.61684543, 10.691749849, 10.766845592, 10.841857426, 10.916229761,
10.991119406, 11.066293612, 11.142887686, 11.216226674, 11.292785416,
11.369860473, 11.446567251, 11.525729061, 11.603587584, 11.681288743,
11.760966713, 11.840379078, 11.919415783, 12.001328095, 12.083951378,
12.165095654, 12.248082849, 12.330684112, 12.412700066, 12.498345169,
12.580566496, 12.663942643, 12.748783656, 12.831586449, 12.91746963,
13.000097933, 13.087712716, 13.17443694, 13.26448119, 13.350396938,
13.437850859, 13.525234879, 13.610702002, 13.695615069, 13.784924776,
13.876389001, 13.966005982, 14.052650246, 14.139922823, 14.230512111,
14.317314069, 14.409773527, 14.503222841, 14.596060431, 14.687754719,
14.783808551, 14.874856058, 14.967336895, 15.059619825, 15.152345242,
15.246605223, 15.341199192, 15.438985803, 15.536176764, 15.631009404,
15.725277632, 15.81786761, 15.916997936, 16.014643322, 16.113313501,
16.211896456, 16.311966528, 16.407028309, 16.506039841, 16.604528025,
16.705394303, 16.803420332, 16.914975468, 17.014462716, 17.115460095,
17.217349842, 17.327123526, 17.430126988, 17.539572057, 17.644639158,
17.752439553, 17.853756444, 17.962947719, 18.065623365, 18.173488729,
18.277540062, 18.381227556, 18.490780511, 18.598536767, 18.712919864,
18.824786377, 18.936692715, 19.048688072, 19.160051409, 19.267718952,
19.383010495, 19.492393059, 19.602069881, 19.713872141, 19.827009743,
19.939404629, 20.055595157, 20.164909143, 20.278838284, 20.387661527,
20.506314018, 20.617256255, 20.735886258, 20.849596977, 20.967068468,
21.083157749, 21.201771303, 21.319861165, 21.433437512, 21.552389587,
21.669828196, 21.784998104, 21.907313216, 22.040682528, 22.158572666,
22.277741687, 22.40071919, 22.532338219, 22.657238411, 22.771417728,
22.890192732, 23.004341897, 23.12978721, 23.257794698, 23.384918228,
23.526752517, 23.654048159, 23.783959281, 23.89964595, 24.03384206,
24.155280571, 24.274852724, 24.401007015, 24.539515305, 24.568722045,
24.679865656, 24.806764428, 24.931938862, 25.066777847, 25.200914222,
25.340775173, 25.469630758, 25.601685871, 25.733822665, 25.86818825,
25.99732227, 26.134097034, 26.264598198, 26.409330788, 26.535822646,
26.670748749, 26.810898086, 26.952081671, 27.08861727, 27.233434629,
27.372947841, 27.524834136, 27.666763846, 27.80789388, 27.956681424,
28.099272809, 28.244556299, 28.387500862, 28.536928532, 28.681687778,
28.816108413, 28.954518996, 29.101342317, 29.253091315, 29.403627579,
29.547428635, 29.702648086, 29.848956361, 29.998165506, 30.150376405,
30.306134198, 30.456305278, 30.614718095, 30.75127775, 30.903908537,
31.047487376, 31.195520756, 31.354874686, 31.501976786, 31.644042532,
31.786019673, 31.948021173, 32.108306067, 32.268099436, 32.434228397,
32.580996241, 32.757302377, 32.922347932, 33.079652231, 33.239028059,
33.412426684, 33.574141262, 33.738584503, 33.889037928, 34.051848734,
34.231857035, 34.394311959, 34.581545838, 34.745610613, 34.905256852,
35.056858567, 35.219084883, 35.375877881, 35.541147403, 35.718704434,
35.89192333, 36.057879813, 36.230734075, 36.408231393, 36.583569456,
36.748469998, 36.915701895, 37.109143471, 37.253925994, 37.433222953,
37.627563675, 37.808669953, 37.973648167, 38.131947768, 38.319830737,
38.47177366, 38.646681082, 38.834831535, 39.008224259, 39.17462922,
39.364969968, 39.555618698, 39.716545413, 39.892491401, 40.070119748,
40.26232604, 40.444344336, 40.612473131, 40.779646754, 40.946133674,
41.135695427, 41.313783432, 41.52814243, 41.718436998, 41.912596271,
42.116072577, 42.309037975, 42.494516972, 42.68893015, 42.896715554,
43.071921396, 43.283009468, 43.50379794, 43.698157397, 43.897209707,
44.07791851, 44.294801576, 44.47819044, 44.669443078, 44.849038899,
45.041089975, 45.220775135, 45.423108661, 45.614548807, 45.793480878,
45.997146201, 46.209456977, 46.450568191, 46.632466165, 46.818945263,
47.007977263, 47.207082389, 47.392855276, 47.601083315, 47.779690884,
48.004955408, 48.199285998, 48.391672982, 48.602776913, 48.809876175,
49.004841654, 49.207657138, 49.447914654, 49.659904797, 49.870728279,
50.053017336, 50.268475305, 50.475861011, 50.698422973, 50.871062239,
51.076574119, 51.309283633, 51.508062313, 51.699919756, 51.904426338,
52.111923624, 52.362103174, 52.578417905, 52.779816919, 53.015904857,
53.228490166, 53.432060987, 53.634954186, 53.895803467, 54.160794402,
54.353551948, 54.540858606, 54.754143937, 54.987450942, 55.230772147,
55.442366844, 55.643630256, 55.859420741, 56.12489402, 56.375967898,
56.572110461, 56.759611369, 56.947011755, 57.159984529, 57.394343981,
57.646973117, 57.88114977, 58.094140997, 58.367073354, 58.582079745,
58.81990894, 59.048727047, 59.291640153, 59.512334001, 59.77520663,
59.986948454, 60.206481976, 60.444911386, 60.671173212, 60.875001568,
60.927112341, 61.081514695, 61.319250383, 61.571121928, 61.834253742,
62.080050068, 62.327148058, 62.539233344, 62.800192515, 63.064143965,
63.356242274, 63.588539175, 63.845301848, 64.030986938, 64.261239485,
64.535939951, 64.838442091, 65.056549729, 65.282718551, 65.543010344,
65.769599854, 66.02017046, 66.294417819, 66.549481803, 66.762635857,
67.153062311, 67.421544663, 67.636057351, 67.93459442, 68.277325697,
68.602135349, 68.790781477, 69.036326556, 69.343612254, 69.632916097,
69.915178879, 70.179567555, 70.477583157, 70.774053219, 71.050802421,
71.338423035, 71.649795779, 71.941766516, 72.126564485, 72.427574041,
72.694171387, 72.88436945, 73.114706342, 73.309914714, 73.521134768,
73.79956, 74.09813125, 74.32259345, 74.490376295, 74.730090786,
74.984631294, 75.22277406, 75.45426605, 75.767494445, 76.066941118,
76.331780505, 76.654049819, 76.955417797, 77.168800479, 77.467260607,
77.799639682, 78.059187753, 78.28290166, 78.467967999, 78.738299369,
78.934485836, 79.270323939, 79.528054768, 79.795558684, 80.115735813,
80.425413327, 80.679470831, 80.936585555, 81.27065012, 81.564398624,
81.778298193, 82.005878933, 82.279863921, 82.495373285, 82.8645289,
83.081219005, 83.403850885, 83.745760575, 84.074856356, 84.306339388,
84.540797173, 84.94941201, 85.195813147, 85.463832328, 85.857998684,
86.112027646, 86.423545776, 86.662214289, 86.94137535, 87.141268334,
87.409977131, 87.718416808, 88.05817132, 88.244517593, 88.573995381,
88.954991297, 89.186646405, 89.475760276, 89.781208791, 90.007003639,
90.265456954, 90.530469544, 90.913877617, 91.230719865, 91.557317451,
91.790999753, 92.049149527, 92.25340017, 92.467484859, 92.984058093,
93.206895261, 93.520893856, 93.819364766, 94.122124781, 94.659638941,
94.846973038, 95.128667955, 95.557566043, 95.715451337, 95.894378361,
96.24669127, 96.673559607, 96.902153342, 97.284312815, 97.521462192,
97.99382407, 98.322270422, 98.781956103, 99.089357134, 99.314349111,
99.882687003, 100.241854079, 100.514913428, 100.9176625, 101.134542486,
101.617530376, 101.802566475, 102.025938021, 102.27991377, 102.650488258,
102.943192268, 103.325619902, 103.805507101, 104.157014871, 104.531824428,
104.810052251, 105.109733346, 105.356049617, 105.652355979, 105.842524499,
106.292069678, 106.648033186, 106.905305573, 107.223635627, 107.618579572,
107.861943448, 108.135011176, 108.656808836, 108.831594814, 109.123835698,
109.545925708, 110.078682586, 110.594090879, 110.70289684, 111.106721969,
111.384652871, 111.631411606, 112.034810306, 112.394695675, 112.602290002,
112.907894543, 113.321193556, 113.527685474, 113.897368593, 114.502589489,
114.697635467, 115.247709702, 115.557911125, 115.852670754, 116.15782419,
116.340205967, 116.736946863, 117.064764662, 117.296290215, 117.731589281,
117.99572534, 118.590837052, 119.121000714, 119.674882126, 119.94283508,
120.206253868, 120.568903625, 121.152046212, 121.669182987, 122.041524631,
122.282689094, 122.930683165, 123.305373781, 123.633290883, 123.75835489,
124.020775424, 124.395167247, 124.660960054, 124.954198327, 125.394235156,
125.772752987, 126.557434518, 126.984130884, 127.691528655, 128.026087561,
128.215818381, 128.54131364, 128.77258964, 129.001192447, 129.341240553,
129.622599533, 129.813971222, 130.293562725, 130.384702495, 130.574321962,
130.843467083, 131.033702075, 131.258052909, 131.68693313, 131.883285782,
132.292775992, 133.058643708, 133.249261848, 133.481971694, 134.193959495,
134.713817623, 135.037614873, 135.539057531, 135.994479965, 136.489024963,
136.950885039, 137.359361168, 137.484532801, 137.934388228, 138.496425601,
139.175542225, 139.63351102, 139.741619977, 140.29157909, 140.900792247,
141.245560429, 141.336477477, 142.182582636, 142.743799713, 143.058631215,
143.336977981, 144.311671215, 144.523720381, 144.90157109, 145.104665052,
145.50385202, 146.162794373, 146.178586891, 146.378669443, 146.709680321,
147.05274606, 147.258097714, 147.89210362, 148.102902697, 148.789898309,
149.232782915, 149.881623253, 150.341254166, 150.573994454, 151.019686154,
151.822099327, 152.121611228, 152.260143944, 152.471919011, 152.966004801,
154.264182449, 154.632740822, 155.092632628, 155.522999128, 156.004061191,
156.638640877, 156.870600578, 157.633998329, 157.690945989, 157.800795006,
158.064984162, 158.160621873, 158.912623113, 159.343952879, 159.399939226,
160.244531766, 161.184128851, 161.942322924, 162.074446975, 162.452858883,
162.553329021, 163.052809407, 163.409906359, 163.532849309, 164.246944821,
165.035772344, 165.180417762, 165.653246239, 165.830398044, 165.891048339,
167.250532549, 167.38016672, 167.451886351, 167.510308486, 167.690623347,
168.036271544, 168.214724565, 168.820509427, 168.848688997, 169.824385267,
169.932498326, 170.369996402, 170.41262208, 170.953906206, 170.955586427,
171.55448653, 172.484466894, 173.338413642, 174.974229003, 175.110590676,
175.785117943, 175.874917918, 175.949020007, 176.234487621, 176.591346789,
176.785312131), "50" = c(2.019e-05, 2.0205e-05, 2.0416e-05, 2.0967e-05,
2.1027e-05, 2.1275e-05, 2.1334e-05, 2.1351e-05, 2.2074e-05, 2.2167e-05,
2.2597e-05, 2.278e-05, 2.3539e-05, 2.379e-05, 2.3848e-05, 2.4345e-05,
2.4669e-05, 2.5067e-05, 2.5254e-05, 2.5264e-05, 2.5556e-05, 2.5792e-05,
2.6088e-05, 2.6366e-05, 2.6483e-05, 2.6563e-05, 2.6965e-05, 2.7711e-05,
2.795e-05, 2.8933e-05, 2.9083e-05, 2.9389e-05, 2.9848e-05, 2.9859e-05,
3.0384e-05, 3.049e-05, 3.1071e-05, 3.1636e-05, 3.184e-05, 3.2211e-05,
3.2741e-05, 3.3051e-05, 3.33e-05, 3.3631e-05, 3.4768e-05, 3.5032e-05,
3.5603e-05, 3.5629e-05, 3.5952e-05, 3.6325e-05, 3.6705e-05, 3.7131e-05,
3.7446e-05, 3.7661e-05, 3.8345e-05, 3.8518e-05, 3.9274e-05, 3.9418e-05,
3.9694e-05, 3.9998e-05, 4.0665e-05, 4.0963e-05, 4.0995e-05, 4.1343e-05,
4.1639e-05, 4.198e-05, 4.2451e-05, 4.2566e-05, 4.2852e-05, 4.3271e-05,
4.4095e-05, 4.4522e-05, 4.545e-05, 4.5824e-05, 4.5858e-05, 4.6929e-05,
4.7618e-05, 4.8022e-05, 4.8134e-05, 4.8258e-05, 4.9184e-05, 5.0293e-05,
5.1012e-05, 5.1428e-05, 5.2679e-05, 5.3656e-05, 5.3943e-05, 5.48e-05,
5.5891e-05, 5.6133e-05, 5.6547e-05, 5.6787e-05, 5.7184e-05, 5.8349e-05,
5.9087e-05, 6.0303e-05, 6.143e-05, 6.1937e-05, 6.3259e-05, 6.38e-05,
6.4019e-05, 6.4622e-05, 6.596e-05, 6.6071e-05, 6.6573e-05, 6.7978e-05,
6.8583e-05, 6.9176e-05, 7.0741e-05, 7.2067e-05, 7.2774e-05, 7.3173e-05,
7.3848e-05, 7.4923e-05, 7.583e-05, 7.6872e-05, 7.694e-05, 7.7282e-05,
7.8358e-05, 8.0051e-05, 8.0958e-05, 8.2354e-05, 8.3093e-05, 8.3639e-05,
8.4391e-05, 8.5927e-05, 8.6997e-05, 8.8251e-05, 8.8738e-05, 8.9565e-05,
9.0689e-05, 9.1601e-05, 9.2106e-05, 9.3566e-05, 9.4525e-05, 9.5215e-05,
9.7199e-05, 9.84e-05, 9.9213e-05, 0.000100121, 0.000100739, 0.000102205,
0.000103723, 0.000104641, 0.000105743, 0.000107377, 0.000108245,
0.000109208, 0.000110706, 0.000111803, 0.000112875, 0.000113909,
0.000114563, 0.000115546, 0.000116294, 0.000118463, 0.000119355,
0.000121276, 0.000122472, 0.000124048, 0.000125229, 0.000125731,
0.000127716, 0.000130099, 0.000131506, 0.000132309, 0.000134032,
0.000135349, 0.00013712, 0.000139149, 0.000140299, 0.000142244,
0.000143997, 0.000145688, 0.000147833, 0.000149779, 0.000150872,
0.000153099, 0.000154144, 0.00015662, 0.000159186, 0.000160758,
0.000162182, 0.000164046, 0.000166308, 0.000168828, 0.000170254,
0.000172291, 0.00017385, 0.000175542, 0.000178345, 0.000180074,
0.000182621, 0.000185223, 0.000187768, 0.000189628, 0.000191391,
0.000192906, 0.000194547, 0.000196016, 0.00019728, 0.000199487,
0.000200744, 0.000201849, 0.00020362, 0.000206168, 0.000209615,
0.000212143, 0.000213832, 0.000215601, 0.000218118, 0.000221152,
0.000224171, 0.000229082, 0.00023196, 0.000235318, 0.000237483,
0.000240301, 0.000242489, 0.000246302, 0.000252736, 0.00025492,
0.000257647, 0.000260336, 0.000263194, 0.000265527, 0.000268476,
0.000273399, 0.000276454, 0.000278715, 0.00028274, 0.000285945,
0.000289186, 0.000291918, 0.000294555, 0.000297525, 0.00030007,
0.000303989, 0.00030838, 0.000312336, 0.000315495, 0.000317184,
0.00032051, 0.000324187, 0.000328083, 0.00033244, 0.00033489,
0.00033813, 0.000342433, 0.000345902, 0.000349911, 0.000354133,
0.000357791, 0.000362759, 0.00036559, 0.000370389, 0.000377629,
0.000381676, 0.000386494, 0.000390176, 0.000393593, 0.00039731,
0.000402234, 0.000406789, 0.000410469, 0.000414679, 0.000417982,
0.000421552, 0.000426087, 0.00043005, 0.000436443, 0.000440932,
0.000446794, 0.000450159, 0.000454889, 0.000459879, 0.000464413,
0.000469047, 0.000474044, 0.000478912, 0.000483837, 0.000491975,
0.000499812, 0.000505271, 0.000509994, 0.000514565, 0.000521893,
0.000527558, 0.000533706, 0.000539215, 0.000547162, 0.000553043,
0.000560445, 0.000567965, 0.000577584, 0.000583139, 0.000591322,
0.000599636, 0.000606256, 0.000611444, 0.000617304, 0.000625326,
0.00063197, 0.000640625, 0.00064919, 0.000654672, 0.000661542,
0.000668733, 0.000675739, 0.000683625, 0.000692334, 0.000701286,
0.000711183, 0.00071944, 0.000729635, 0.000736791, 0.00074419,
0.000750555, 0.000761188, 0.00076953, 0.000777571, 0.00078845,
0.000797485, 0.000804767, 0.000811944, 0.00082077, 0.000830016,
0.000840515, 0.000850539, 0.000861607, 0.0008704, 0.000880649,
0.00089128, 0.000904131, 0.000916312, 0.000928041, 0.000936455,
0.000945878, 0.00095837, 0.000973362, 0.000983831, 0.000986457,
0.000997865, 0.001010027, 0.001021199, 0.001033114, 0.001043783,
0.001054089, 0.001065257, 0.001076954, 0.001088244, 0.001102624,
0.001114654, 0.001127719, 0.001142337, 0.001154759, 0.00117152,
0.001186376, 0.001200129, 0.001215598, 0.001229731, 0.001242811,
0.001257455, 0.001270463, 0.001284917, 0.001303007, 0.001318175,
0.001333988, 0.001351157, 0.001365691, 0.001383647, 0.001398886,
0.001413365, 0.001428922, 0.001447756, 0.001468414, 0.001485601,
0.001502279, 0.001521855, 0.00153792, 0.001556678, 0.001575165,
0.001593123, 0.001612234, 0.001631421, 0.001648469, 0.001667297,
0.001686437, 0.001704749, 0.001727203, 0.001747751, 0.001764788,
0.001786623, 0.001804504, 0.001824461, 0.001847478, 0.001867076,
0.001888228, 0.001906468, 0.001925635, 0.001948575, 0.001970081,
0.001994098, 0.002017878, 0.002041074, 0.002065082, 0.002088412,
0.002114554, 0.002135873, 0.00216032, 0.002188545, 0.002216817,
0.002240926, 0.002267331, 0.002292851, 0.002316881, 0.002342883,
0.002371198, 0.002395838, 0.002427362, 0.002457652, 0.002486512,
0.002515326, 0.002543462, 0.002575498, 0.002601825, 0.002630253,
0.002660622, 0.002691821, 0.002721928, 0.002753853, 0.002785027,
0.002817186, 0.002847036, 0.002876844, 0.00290863, 0.00294403,
0.002981901, 0.003018732, 0.003051169, 0.003083025, 0.003126739,
0.003166002, 0.003204325, 0.003240019, 0.003275286, 0.003310562,
0.00334948, 0.003389144, 0.003431512, 0.003471749, 0.003513312,
0.003553074, 0.003595635, 0.003637422, 0.003678817, 0.003718264,
0.003761746, 0.003802768, 0.00384361, 0.003884008, 0.003927128,
0.003973577, 0.004027514, 0.004073117, 0.004123935, 0.004173584,
0.004222091, 0.004273434, 0.004323656, 0.004372338, 0.004422993,
0.004472049, 0.004522206, 0.004577358, 0.004626582, 0.00467467,
0.004734686, 0.004787722, 0.004839646, 0.00489532, 0.004952117,
0.005008494, 0.005067644, 0.005122521, 0.005177938, 0.005234363,
0.005298586, 0.005360208, 0.005424168, 0.005480237, 0.005541287,
0.005601648, 0.005663107, 0.005729266, 0.005795892, 0.005858724,
0.005921718, 0.005989611, 0.006062575, 0.006133056, 0.006209556,
0.006276743, 0.006345358, 0.00642221, 0.006501667, 0.006580791,
0.006647121, 0.006718227, 0.006793842, 0.006876317, 0.006958449,
0.007040752, 0.007119127, 0.007200652, 0.007287408, 0.007377221,
0.007462955, 0.007554616, 0.007645002, 0.007727608, 0.007812044,
0.007904918, 0.007990284, 0.008083001, 0.008182769, 0.008278197,
0.008374989, 0.008475591, 0.008569257, 0.0086686, 0.00877213,
0.008880439, 0.008981845, 0.009081389, 0.009190709, 0.009298055,
0.009407666, 0.009520156, 0.009630344, 0.009735829, 0.009829074,
0.009854459, 0.009971109, 0.010082016, 0.010192336, 0.010311958,
0.010436564, 0.010559461, 0.01068318, 0.010804546, 0.010923772,
0.011049966, 0.011174198, 0.011305745, 0.011427668, 0.011560674,
0.011698991, 0.011845247, 0.011979833, 0.01211508, 0.012262094,
0.012403306, 0.012546504, 0.012687185, 0.012833786, 0.012976501,
0.013121119, 0.013279543, 0.013446275, 0.013610222, 0.013774658,
0.013940335, 0.014106974, 0.014279082, 0.01445019, 0.014613915,
0.014789451, 0.014954726, 0.015112349, 0.015299877, 0.015491741,
0.015676348, 0.015850882, 0.016035657, 0.01621702, 0.01640175,
0.016593356, 0.016788842, 0.016975379, 0.017179957, 0.017371198,
0.017575981, 0.017777139, 0.017975951, 0.018177396, 0.018385431,
0.018611967, 0.018828409, 0.019044306, 0.019268147, 0.01948952,
0.019718585, 0.019940762, 0.020182237, 0.02040204, 0.020625111,
0.020854586, 0.021096072, 0.02133607, 0.021587116, 0.021842935,
0.022101658, 0.022346041, 0.022592346, 0.02285937, 0.023127305,
0.023396694, 0.023670296, 0.023943095, 0.024226516, 0.024502629,
0.024786481, 0.025080325, 0.025372198, 0.025662521, 0.025955196,
0.026260947, 0.02656299, 0.026870854, 0.027196179, 0.027504632,
0.027824485, 0.028154031, 0.028465083, 0.028787331, 0.029114063,
0.02946049, 0.029805179, 0.030147747, 0.030506497, 0.03085849,
0.031224698, 0.031580482, 0.031946636, 0.032312024, 0.032688964,
0.033080968, 0.033474751, 0.033858837, 0.034256236, 0.03465799,
0.035056252, 0.035460665, 0.035879336, 0.036291442, 0.036713613,
0.037143392, 0.037568506, 0.037997038, 0.038452816, 0.038889142,
0.039336507, 0.039797575, 0.040270269, 0.040738851, 0.041201527,
0.041671824, 0.042168853, 0.042642444, 0.043130634, 0.043633223,
0.044142788, 0.04466566, 0.045188869, 0.045713938, 0.046243019,
0.046779246, 0.0473012, 0.047843657, 0.048393645, 0.048939967,
0.049518326, 0.05008889, 0.050657872, 0.051242017, 0.051833437,
0.05243651, 0.053038194, 0.053636275, 0.054250441, 0.05485575,
0.055505535, 0.056125001, 0.056782215, 0.057434922, 0.058109072,
0.058781715, 0.059462804, 0.060154942, 0.060856807, 0.061558963,
0.06226504, 0.062980929, 0.063692469, 0.064436331, 0.065184959,
0.065949201, 0.066723026, 0.067478962, 0.068250581, 0.069057533,
0.069856517, 0.070651543, 0.071464038, 0.072307868, 0.073163538,
0.074019444, 0.074886019, 0.075753024, 0.076602328, 0.077475498,
0.0783897, 0.079281085, 0.080200506, 0.081151792, 0.082099971,
0.083035468, 0.084008478, 0.085004586, 0.0859882, 0.086964651,
0.087980512, 0.089020068, 0.090048608, 0.091125148, 0.092172902,
0.093257448, 0.094325228, 0.095426289, 0.096550719, 0.097694725,
0.098588335, 0.098825885, 0.099976394, 0.101151923, 0.102320207,
0.103494321, 0.104718159, 0.105953669, 0.107185903, 0.108436853,
0.109715266, 0.111028126, 0.112334616, 0.113642614, 0.114965663,
0.116282802, 0.117625703, 0.119017119, 0.120380169, 0.121773836,
0.123181211, 0.124613975, 0.126057113, 0.127503255, 0.128979355,
0.13048784, 0.132041852, 0.133574549, 0.135118359, 0.136649158,
0.138213466, 0.139834582, 0.141472027, 0.143118906, 0.144781297,
0.146485308, 0.148161617, 0.149901371, 0.151629511, 0.153397236,
0.155201614, 0.156999459, 0.1588623, 0.16073841, 0.162630655,
0.164511766, 0.166464898, 0.168401847, 0.170386021, 0.172381462,
0.174415983, 0.176432392, 0.178478142, 0.180567198, 0.182676788,
0.184813853, 0.186982449, 0.189135122, 0.191341016, 0.19362012,
0.195912056, 0.198191857, 0.200491915, 0.202835625, 0.20523448,
0.207653147, 0.210035343, 0.212510834, 0.214980491, 0.217519911,
0.220094051, 0.22262155, 0.225278365, 0.227923194, 0.230612742,
0.233306794, 0.236061468, 0.238822682, 0.241609939, 0.244439204,
0.247310784, 0.250240186, 0.253224473, 0.256232882, 0.259226277,
0.262313694, 0.265369846, 0.268491052, 0.27161151, 0.274777471,
0.278005432, 0.281229269, 0.284548546, 0.287954334, 0.291368837,
0.294836603, 0.298366343, 0.299137766, 0.3018811, 0.305441595,
0.309037614, 0.312674294, 0.316419235, 0.320228187, 0.324070106,
0.327953675, 0.331801264, 0.335769089, 0.339718216, 0.343733789,
0.347865557, 0.352021201, 0.356248815, 0.360474324, 0.364790642,
0.369096523, 0.373532645, 0.378022593, 0.382550627, 0.387094336,
0.391750874, 0.396495955, 0.40124321, 0.402220565, 0.406020387,
0.410945347, 0.415892796, 0.420977009, 0.426048472, 0.431204123,
0.436455595, 0.44177867, 0.447145143, 0.452541706, 0.458067543,
0.463648593, 0.469306223, 0.47505397, 0.480828718, 0.486655393,
0.492566206, 0.498676605, 0.50480208, 0.510982564, 0.517300879,
0.523653378, 0.530090956, 0.536649173, 0.543320049, 0.550056611,
0.556870628, 0.563769291, 0.57082874, 0.57800615, 0.585238626,
0.592581448, 0.599961126, 0.607418736, 0.615033014, 0.618261558,
0.622806081, 0.630665596, 0.638593621, 0.646691172, 0.654835212,
0.663181553, 0.671703182, 0.680239255, 0.688960083, 0.697723261,
0.706688472, 0.715905465, 0.725236579, 0.73463827, 0.744197905,
0.75378801, 0.7636659, 0.773750397, 0.783896406, 0.794162292,
0.804699414, 0.815454584, 0.82633625, 0.837416987, 0.84859961,
0.853247157, 0.860099833, 0.871628583, 0.883427729, 0.895495442,
0.90776317, 0.920179879, 0.932941919, 0.945949612, 0.959147493,
0.97259477, 0.986210538, 1.00004985, 1.014185334, 1.028615517,
1.043322803, 1.058332147, 1.073696355, 1.089373553, 1.105356484,
1.118276445, 1.147661032, 1.164020459, 1.180450911, 1.196750836,
1.212878991, 1.229225917, 1.245395746, 1.261723465, 1.277908044,
1.294037943, 1.310161329, 1.326241806, 1.342205676, 1.358231206,
1.374397167, 1.390360146, 1.40641944, 1.422523054, 1.432032614,
1.438685042, 1.454787126, 1.470909388, 1.487076006, 1.503287203,
1.519514218, 1.535792104, 1.551922922, 1.568079577, 1.584173962,
1.600396208, 1.616577693, 1.632857498, 1.649201794, 1.665351141,
1.681636891, 1.698012009, 1.714318771, 1.730680527, 1.747022947,
1.763442521, 1.779778216, 1.796316106, 1.812868601, 1.829309044,
1.838785356, 1.845827026, 1.86230921, 1.878802198, 1.895452371,
1.912101853, 1.928733429, 1.945538507, 1.962420539, 1.979421551,
1.996462631, 2.013498816, 2.030613433, 2.047837191, 2.065122056,
2.082434233, 2.099795685, 2.117267175, 2.13456954, 2.152320156,
2.170074342, 2.187894106, 2.205518802, 2.223308709, 2.241534133,
2.25968779, 2.277858724, 2.296261799, 2.314498788, 2.333132557,
2.351575097, 2.370333891, 2.389286255, 2.408172581, 2.427147705,
2.446253369, 2.461560209, 2.465594036, 2.484901583, 2.50443451,
2.524351281, 2.544281476, 2.564251508, 2.584523308, 2.604606747,
2.624904791, 2.645436775, 2.666381482, 2.687229888, 2.708409421,
2.729584016, 2.750969357, 2.772609136, 2.794635949, 2.816627952,
2.838299731, 2.860782748, 2.883215686, 2.905730454, 2.92846844,
2.951630185, 2.974981098, 2.993455636, 2.998475671, 3.022387748,
3.046310016, 3.070665678, 3.095533475, 3.120807054, 3.14592327,
3.171031788, 3.196903039, 3.222361075, 3.248646058, 3.274782573,
3.30174249, 3.328530719, 3.355661884, 3.38346905, 3.411378792,
3.439314228, 3.467542328, 3.496344735, 3.52500378, 3.554479529,
3.584258485, 3.613726717, 3.643616307, 3.674388692, 3.704998475,
3.735709567, 3.76778341, 3.799745033, 3.832309362, 3.864371779,
3.896775593, 3.930001549, 3.963691344, 3.997732562, 4.031512831,
4.066013035, 4.100700047, 4.136295332, 4.171473658, 4.207118296,
4.242645009, 4.278634318, 4.31462059, 4.351472655, 4.388591004,
4.42651482, 4.464009973, 4.501888051, 4.540169681, 4.578993857,
4.617579826, 4.656698414, 4.696797048, 4.736354943, 4.776514658,
4.816890641, 4.857257165, 4.898134709, 4.938700998, 4.979694834,
5.02117416, 5.062909223, 5.105412232, 5.147127262, 5.18915082,
5.231975932, 5.275387341, 5.318538881, 5.361979784, 5.406105308,
5.450499386, 5.494745512, 5.540110393, 5.585506041, 5.631138216,
5.677105657, 5.722760031, 5.768741417, 5.815144868, 5.861634109,
5.908812855, 5.955524657, 6.002580338, 6.050448109, 6.098968838,
6.147338268, 6.195223898, 6.243734375, 6.292424748, 6.342672299,
6.39283463, 6.442564422, 6.492266927, 6.542678589, 6.553304998,
6.594018526, 6.644116408, 6.69636615, 6.74820814, 6.800969876,
6.853544109, 6.905761693, 6.958642131, 7.011736455, 7.063905479,
7.117918696, 7.172376666, 7.226015415, 7.281270474, 7.33736543,
7.391992489, 7.447535096, 7.50304389, 7.55831746, 7.614919777,
7.671352282, 7.727474495, 7.782610126, 7.839447888, 7.895729449,
7.952197578, 8.010463926, 8.068968663, 8.127615098, 8.186073522,
8.245363699, 8.305348147, 8.365383973, 8.424933473, 8.485564831,
8.54635179, 8.607510295, 8.668127431, 8.729654648, 8.790350442,
8.852899381, 8.917082577, 8.979831834, 9.043268675, 9.10715488,
9.172802047, 9.237049387, 9.300585061, 9.366996556, 9.433758709,
9.499058873, 9.562536943, 9.627843964, 9.69394471, 9.762830448,
9.827178709, 9.89571097, 9.963212904, 10.030188592, 10.099325481,
10.167546435, 10.235508082, 10.303641906, 10.375561796, 10.446002456,
10.514730578, 10.586345163, 10.656783188, 10.727836073, 10.797960329,
10.869306185, 10.941246923, 11.012923088, 11.085177434, 11.160327386,
11.230405514, 11.3060048, 11.381839478, 11.456490363, 11.528921819,
11.602004267, 11.677406485, 11.754288798, 11.834019485, 11.910024292,
11.987309649, 12.064980787, 12.140495923, 12.21959513, 12.298104837,
12.377282373, 12.458576379, 12.539113037, 12.617735047, 12.701535036,
12.781975083, 12.865860406, 12.946361785, 13.025266016, 13.107410095,
13.187905517, 13.265938039, 13.34784859, 13.427544901, 13.509089895,
13.59253879, 13.675030565, 13.758292714, 13.841978144, 13.926452526,
14.011530072, 14.096842369, 14.183310064, 14.271493005, 14.356937586,
14.44307979, 14.531169948, 14.618772395, 14.7076259, 14.798789238,
14.887280907, 14.9756141, 15.062954441, 15.150753042, 15.240733694,
15.332175828, 15.425775414, 15.520496355, 15.616938465, 15.710324273,
15.802542921, 15.896022834, 15.982838266, 16.07650801, 16.172319582,
16.267809031, 16.364248492, 16.460968264, 16.562365533, 16.659085246,
16.752179578, 16.850410344, 16.947500887, 17.049795686, 17.146470837,
17.249576864, 17.34751568, 17.447677838, 17.545330976, 17.651865113,
17.758939625, 17.863534392, 17.967390846, 18.073659062, 18.173249608,
18.272244134, 18.376164353, 18.480662749, 18.586501152, 18.69270695,
18.797222908, 18.909760403, 19.020031076, 19.123905294, 19.224268228,
19.331424337, 19.436870004, 19.541426705, 19.652450475, 19.761889913,
19.867168517, 19.972835494, 20.086160797, 20.196255773, 20.307269148,
20.416191989, 20.529828013, 20.642827783, 20.753954204, 20.871722199,
20.995331078, 21.104263827, 21.21489194, 21.335694554, 21.450918373,
21.55686499, 21.666760447, 21.779970668, 21.902382162, 22.00810243,
22.124779975, 22.245804573, 22.359751981, 22.483956045, 22.605617556,
22.730061178, 22.853890368, 22.962607397, 23.081863519, 23.200390161,
23.229185027, 23.328817491, 23.454272604, 23.586029129, 23.707075311,
23.82574447, 23.941869964, 24.066879821, 24.197226639, 24.321630377,
24.451009061, 24.570565379, 24.69266521, 24.819263413, 24.948734671,
25.077438309, 25.198194297, 25.338747439, 25.46825159, 25.595711176,
25.717365041, 25.847123961, 25.979722219, 26.107325375, 26.23858713,
26.378887971, 26.510138391, 26.639420733, 26.77631126, 26.907696558,
27.030038178, 27.160051988, 27.299079452, 27.449337052, 27.579224008,
27.722399625, 27.863710572, 27.995880172, 28.133811239, 28.271983745,
28.410577147, 28.571545575, 28.703561428, 28.844019399, 28.999889402,
29.142198269, 29.277164318, 29.42532668, 29.560029115, 29.701749769,
29.845412722, 29.990949578, 30.138327286, 30.275913704, 30.432826493,
30.582521188, 30.732761896, 30.888842588, 31.05472399, 31.206805322,
31.346518717, 31.494540266, 31.629411846, 31.79190212, 31.928149452,
32.086781699, 32.230850768, 32.38588119, 32.542979379, 32.687314462,
32.850935859, 33.004353791, 33.160483977, 33.323579807, 33.475604099,
33.620996621, 33.757968703, 33.915771195, 34.079734014, 34.242759592,
34.386900748, 34.539219174, 34.703639332, 34.85366148, 35.019146,
35.197395684, 35.383732377, 35.553833786, 35.736180127, 35.898790285,
36.092078468, 36.250287438, 36.412574055, 36.584176835, 36.772568106,
36.95145678, 37.134599853, 37.287700903, 37.464970491, 37.626723877,
37.801322621, 37.987248894, 38.159280184, 38.32892814, 38.501615971,
38.686873975, 38.878723867, 39.079412905, 39.257991858, 39.44861108,
39.640068121, 39.814088007, 40.012238217, 40.203860139, 40.38900056,
40.553432292, 40.775508864, 40.949962235, 41.141954456, 41.348395416,
41.546161159, 41.728761042, 41.898837561, 42.099791834, 42.289896716,
42.466387406, 42.659980268, 42.841283443, 43.016813558, 43.197873545,
43.414196983, 43.58423683, 43.757160407, 43.963492429, 44.114584298,
44.333136265, 44.530474856, 44.706820228, 44.906459152, 45.107554433,
45.318101853, 45.481907782, 45.686493016, 45.874588191, 46.064131703,
46.250397328, 46.439016275, 46.633904658, 46.833072302, 47.038167215,
47.203696919, 47.386224783, 47.590636278, 47.780030431, 47.973333609,
48.196272239, 48.383078288, 48.616068148, 48.810610859, 48.99600004,
49.206858779, 49.407206006, 49.611263094, 49.784958123, 49.990156573,
50.185537115, 50.404981107, 50.595744363, 50.859935184, 51.055635924,
51.247828906, 51.423927373, 51.625806476, 51.8375713, 52.040927093,
52.258940981, 52.457244294, 52.686069887, 52.937674746, 53.140272777,
53.393206688, 53.632899563, 53.850245706, 54.089652035, 54.266376605,
54.488288344, 54.71245731, 54.983553496, 55.222554648, 55.450285422,
55.671789559, 55.89384767, 56.079198477, 56.306620894, 56.549688893,
56.799028376, 57.002619111, 57.217730462, 57.446654763, 57.661903139,
57.955889695, 58.013263956, 58.195091998, 58.411827253, 58.675586124,
58.882383256, 59.139542133, 59.362357623, 59.609137833, 59.841876322,
60.031931999, 60.31225422, 60.521416794, 60.747505979, 60.930806652,
61.166597325, 61.395747592, 61.654372888, 61.842287464, 62.108628071,
62.322630855, 62.533100003, 62.791021019, 63.069856924, 63.332797237,
63.612707849, 63.796829863, 64.039997027, 64.324044978, 64.593574802,
64.815242108, 65.086088464, 65.345814855, 65.601629547, 65.823073156,
66.027994955, 66.225033197, 66.540324238, 66.764013702, 67.088103348,
67.382161242, 67.64717566, 67.899594655, 68.113741756, 68.39869876,
68.643600351, 68.93447417, 69.126502545, 69.382181323, 69.63964634,
69.884535833, 70.171274294, 70.419199646, 70.775601506, 70.996800336,
71.173534234, 71.375764377, 71.727002274, 71.997415142, 72.283115015,
72.613503028, 72.911513096, 73.154816727, 73.398500871, 73.639752857,
73.880191077, 74.169666333, 74.43986113, 74.681434294, 74.961173364,
75.274468172, 75.546948457, 75.816196087, 76.012611693, 76.216579817,
76.458326874, 76.626314544, 76.883480262, 77.140592211, 77.426883939,
77.689840637, 77.96771686, 78.233420334, 78.574019966, 78.872274749,
79.126786287, 79.246641878, 79.607247888, 79.847961934, 80.212418598,
80.45224006, 80.731541045, 81.030353276, 81.222252945, 81.508484105,
81.955770046, 82.328901321, 82.652340989, 83.107576065, 83.432473123,
83.617854292, 83.866061774, 84.122204508, 84.278244423, 84.659538859,
84.950686193, 85.251073327, 85.52690369, 85.731738836, 86.106358457,
86.326792226, 86.660267866, 87.078308367, 87.242853687, 87.617322011,
87.849641858, 88.217792146, 88.522702039, 88.788377199, 89.076601867,
89.299948105, 89.722055847, 90.029259956, 90.384101246, 90.635831804,
90.925042732, 91.240603358, 91.471691018, 91.802060329, 92.195688522,
92.508420522, 92.834117224, 93.102390816, 93.359753425, 93.590067441,
93.862030077, 94.393626341, 94.775853636, 94.964898062, 95.340244136,
95.740770569, 96.158286342, 96.293194222, 96.838133001, 97.058510034,
97.382584382, 97.698626546, 97.963019908, 98.158797115, 98.496854009,
98.695505819, 98.938196698, 99.448068893, 99.775985099, 100.137129989,
100.506655626, 100.76165646, 101.009123937, 101.337222539, 101.71602578,
101.966468987, 102.342175987, 102.738282759, 103.278812345, 103.71625785,
104.016329515, 104.328515389, 104.74418731, 105.558513651, 106.032649865,
106.169687827, 106.585181231, 107.07669284, 107.516654738, 107.719881969,
107.995710718, 108.374618611, 108.686222925, 109.011611195, 109.246349146,
109.557211276, 109.955095259, 110.436555104, 111.140469686, 111.251522072,
111.557258393, 111.821977607, 112.377533242, 112.563918941, 113.078999246,
113.266967639, 113.493153052, 113.79451872, 114.016494654, 114.516271962,
114.849115425, 115.228905996, 115.478174546, 115.715688907, 115.861175954,
116.165624709, 116.606972913, 116.928202812, 117.100782759, 117.668705637,
118.165544465, 118.672325476, 118.908869496, 119.409560174, 119.998277567,
120.40854695, 120.663832815, 120.913449786, 121.152295821, 121.539890948,
121.82716221, 122.674972882, 122.995344101, 123.138182321, 123.303640388,
123.508839627, 124.063932583, 124.239657114, 124.584833095, 125.074496234,
125.352673047, 125.528865859, 125.85949649, 126.083210263, 126.214190065,
126.466184688, 126.710877595, 127.314327173, 128.063691959, 128.669197263,
129.030548426, 129.355996268, 129.896928566, 130.321890716, 130.498929634,
131.379276914, 131.70962744, 131.899100531, 132.116289728, 132.204170119,
132.337403009, 132.650638441, 133.581317785, 133.84849024, 134.306377149,
134.630435996, 134.90528433, 135.269906734, 135.75058761, 135.891452841,
136.065011972, 136.809221, 137.366318542, 137.825687805, 138.019182248,
138.303150816, 138.904173617, 139.210909351, 139.431835499, 139.811327395,
140.406700273, 140.846375091, 141.557698364, 141.943003869, 142.634193374,
142.897554909, 143.161553852, 143.497879344, 143.992127334, 144.434422288,
144.719460382, 145.158311782, 145.537956758, 145.715784909, 146.137565229,
147.177112331, 147.620457417, 148.170679611, 148.273362105, 149.453550032,
150.786979716, 151.761124986, 151.975193258, 152.111637244, 152.338572269,
152.52566726, 152.9126127, 153.279203738, 153.882093625, 154.018566471,
154.1564456, 154.237081026, 155.28415731, 155.560639406, 155.640571972,
156.060846946, 156.360826452, 156.538788176, 157.066712895, 157.806475258,
158.407153976, 159.003007, 159.785329768, 160.039781621, 161.039888476,
161.142944264, 161.468406361, 161.66252347, 162.14094407, 162.364007521,
162.490318729, 162.619644464, 163.08779663, 163.410916439, 163.623779739,
164.098744512, 164.196449462, 165.906565417, 165.918403527, 166.126431718,
166.359293923, 166.698406416, 167.133046834, 167.529927369, 168.321726778,
168.805410097, 169.419541702, 169.528177179, 169.603115977, 170.034649233,
170.041649743, 170.135530717, 171.166731181, 171.576821338, 172.736813929,
173.114361229, 173.367993743), "75" = c(1.8209e-05, 1.8358e-05,
1.8595e-05, 1.8845e-05, 1.8975e-05, 1.8984e-05, 1.9016e-05, 1.9259e-05,
1.9486e-05, 1.9643e-05, 1.9892e-05, 2.0345e-05, 2.0628e-05, 2.0633e-05,
2.1077e-05, 2.1952e-05, 2.2248e-05, 2.2447e-05, 2.2561e-05, 2.2641e-05,
2.282e-05, 2.2961e-05, 2.299e-05, 2.34e-05, 2.3692e-05, 2.3727e-05,
2.3851e-05, 2.4299e-05, 2.4405e-05, 2.4899e-05, 2.507e-05, 2.6398e-05,
2.6825e-05, 2.7007e-05, 2.7588e-05, 2.7889e-05, 2.825e-05, 2.8534e-05,
2.8895e-05, 2.9347e-05, 2.9366e-05, 2.9713e-05, 3.0176e-05, 3.0329e-05,
3.1059e-05, 3.154e-05, 3.1955e-05, 3.2221e-05, 3.2644e-05, 3.3042e-05,
3.3167e-05, 3.3424e-05, 3.4031e-05, 3.4534e-05, 3.6139e-05, 3.707e-05,
3.779e-05, 3.8827e-05, 3.9541e-05, 4.0614e-05, 4.119e-05, 4.1238e-05,
4.1263e-05, 4.1697e-05, 4.2633e-05, 4.2807e-05, 4.3634e-05, 4.4122e-05,
4.5485e-05, 4.5822e-05, 4.6302e-05, 4.6463e-05, 4.7353e-05, 4.8224e-05,
4.8441e-05, 4.9122e-05, 4.9863e-05, 5.0651e-05, 5.0882e-05, 5.2099e-05,
5.2871e-05, 5.3077e-05, 5.3194e-05, 5.3979e-05, 5.44e-05, 5.4774e-05,
5.5434e-05, 5.6488e-05, 5.6894e-05, 5.7206e-05, 5.8385e-05, 5.8711e-05,
5.9576e-05, 6.0386e-05, 6.1128e-05, 6.1679e-05, 6.2635e-05, 6.3313e-05,
6.4207e-05, 6.4953e-05, 6.5589e-05, 6.6554e-05, 6.788e-05, 6.8734e-05,
6.9736e-05, 7.0604e-05, 7.1055e-05, 7.2407e-05, 7.3112e-05, 7.345e-05,
7.4273e-05, 7.4897e-05, 7.5803e-05, 7.655e-05, 7.7304e-05, 7.8027e-05,
7.8739e-05, 7.9328e-05, 8.0842e-05, 8.1369e-05, 8.2282e-05, 8.2969e-05,
8.3413e-05, 8.4045e-05, 8.4987e-05, 8.5315e-05, 8.5631e-05, 8.642e-05,
8.7994e-05, 8.9127e-05, 8.9782e-05, 9.2127e-05, 9.2905e-05, 9.3674e-05,
9.4246e-05, 9.4891e-05, 9.6312e-05, 9.7111e-05, 9.8182e-05, 9.8757e-05,
0.000100241, 0.000101603, 0.000102887, 0.000103939, 0.00010481,
0.0001062, 0.000107163, 0.0001074, 0.0001083, 0.000108953, 0.000110116,
0.000111488, 0.000112145, 0.000112867, 0.000114772, 0.000115471,
0.00011636, 0.000117553, 0.000120367, 0.000121003, 0.000122141,
0.000122922, 0.000124501, 0.000125819, 0.000128184, 0.000129041,
0.000130143, 0.000131921, 0.00013404, 0.000135609, 0.000137434,
0.000138335, 0.000140331, 0.000141991, 0.000144433, 0.000146497,
0.000147756, 0.000149652, 0.000151815, 0.000153255, 0.000154405,
0.000155756, 0.000157732, 0.000159797, 0.00016085, 0.00016345,
0.00016678, 0.000168849, 0.000171128, 0.000173241, 0.000174616,
0.000176496, 0.000177661, 0.0001794, 0.000180594, 0.00018324,
0.000185568, 0.000187418, 0.00018905, 0.000190426, 0.000192994,
0.000194505, 0.000196091, 0.000199302, 0.000200963, 0.000203452,
0.000206776, 0.000208346, 0.00021183, 0.000214626, 0.000216146,
0.000218594, 0.000223144, 0.000225564, 0.000228351, 0.000231216,
0.000232935, 0.00023609, 0.000238753, 0.000242563, 0.00024576,
0.000249661, 0.000252239, 0.000254357, 0.000257446, 0.000260258,
0.000263194, 0.000266796, 0.000271683, 0.00027558, 0.000280933,
0.0002836, 0.000286864, 0.00029088, 0.000293586, 0.000297092,
0.000302018, 0.000304347, 0.000306476, 0.000309727, 0.000312121,
0.000314905, 0.000319326, 0.000322143, 0.00032537, 0.000329516,
0.000332285, 0.000335477, 0.000338738, 0.000343615, 0.000347261,
0.000350993, 0.00035489, 0.00035796, 0.000361577, 0.000364868,
0.000368124, 0.000372072, 0.000376801, 0.000381117, 0.000385449,
0.000389568, 0.000395624, 0.000400479, 0.000406477, 0.000409348,
0.000414134, 0.000417686, 0.000422739, 0.000428939, 0.00043483,
0.000438978, 0.00044368, 0.000447884, 0.000452944, 0.00045863,
0.000464167, 0.000469564, 0.000475943, 0.000481879, 0.000488077,
0.000493646, 0.000499456, 0.000505542, 0.000510833, 0.000516672,
0.000524122, 0.000530081, 0.000537184, 0.000543616, 0.00054903,
0.000555146, 0.000562827, 0.000568957, 0.000575447, 0.000581738,
0.000590355, 0.000597067, 0.000602745, 0.000609943, 0.000616543,
0.000622768, 0.000629051, 0.000636296, 0.000644788, 0.000651431,
0.00065975, 0.000667505, 0.000676204, 0.000684367, 0.000691226,
0.000700118, 0.000707285, 0.000718799, 0.000730087, 0.000736719,
0.000743694, 0.000754704, 0.000765657, 0.00077262, 0.000781369,
0.000788502, 0.000797208, 0.000806223, 0.000817279, 0.000827593,
0.000837212, 0.000847842, 0.000859441, 0.000870424, 0.000881582,
0.000891865, 0.000900259, 0.000909138, 0.000917939, 0.000928429,
0.000940948, 0.000951801, 0.000962542, 0.00097281, 0.000980746,
0.000983438, 0.000995566, 0.00100667, 0.001018491, 0.001029957,
0.00104087, 0.001051583, 0.001067963, 0.001079373, 0.001092534,
0.001106844, 0.001119831, 0.001131053, 0.001142726, 0.00115765,
0.001173307, 0.00118404, 0.001197976, 0.001213841, 0.001226223,
0.001240792, 0.001258141, 0.001270767, 0.00128509, 0.001298669,
0.001310963, 0.001324475, 0.001338006, 0.001351257, 0.001366754,
0.001382734, 0.001401566, 0.001417831, 0.001436156, 0.001453497,
0.001469923, 0.001487274, 0.00150751, 0.00152247, 0.00153929,
0.001555941, 0.001571556, 0.001589423, 0.001605776, 0.001624971,
0.001643278, 0.001662343, 0.001684805, 0.001702351, 0.001723984,
0.001746558, 0.001766554, 0.00178634, 0.001806953, 0.001825658,
0.001847151, 0.001869745, 0.001893413, 0.001914612, 0.001934095,
0.001953136, 0.001976794, 0.001995929, 0.002019236, 0.002040252,
0.002061997, 0.00208561, 0.002108347, 0.002131309, 0.002158775,
0.002185866, 0.002211248, 0.002237469, 0.002263212, 0.002290097,
0.002314808, 0.002341747, 0.00236629, 0.002393074, 0.002419411,
0.002448152, 0.002477594, 0.002504078, 0.002535949, 0.002567649,
0.002593114, 0.0026226, 0.0026539, 0.002681041, 0.002712663,
0.002744175, 0.002772973, 0.002804142, 0.002834663, 0.002868875,
0.002905541, 0.002941829, 0.002972421, 0.003004403, 0.003040666,
0.003073111, 0.003109522, 0.003147188, 0.00318628, 0.003223647,
0.003258683, 0.003296016, 0.003329591, 0.003368827, 0.003407368,
0.003442325, 0.003483133, 0.003521489, 0.003561483, 0.003600918,
0.003642232, 0.003685098, 0.003727781, 0.003772552, 0.003813257,
0.003852405, 0.003895134, 0.003944163, 0.003993108, 0.004042176,
0.004087417, 0.004132161, 0.004182091, 0.004229945, 0.004282899,
0.004327777, 0.004381896, 0.004435419, 0.004486858, 0.00453835,
0.004587396, 0.004643133, 0.004697991, 0.004752347, 0.004807964,
0.004860343, 0.004912878, 0.004971717, 0.005027785, 0.005086806,
0.005147312, 0.005205217, 0.005270105, 0.005332397, 0.005396119,
0.005453283, 0.005516801, 0.005583042, 0.005646586, 0.005714503,
0.005778433, 0.005842576, 0.005912496, 0.005976558, 0.006046083,
0.006110896, 0.006180662, 0.006256144, 0.006320255, 0.006394294,
0.006469572, 0.006556447, 0.006631952, 0.006706346, 0.006785633,
0.006867494, 0.006948544, 0.007028668, 0.007111222, 0.007194139,
0.007273466, 0.007365966, 0.007456298, 0.007537812, 0.007625499,
0.007721537, 0.007815314, 0.007902847, 0.007992783, 0.00808695,
0.008182568, 0.008274713, 0.00836983, 0.008466153, 0.008555319,
0.008656258, 0.008759551, 0.008857032, 0.008954869, 0.009069621,
0.009171844, 0.009279351, 0.009388548, 0.009503001, 0.009611676,
0.009705323, 0.009725512, 0.009831866, 0.009943281, 0.010050742,
0.010167536, 0.010287974, 0.010420793, 0.010537934, 0.010654138,
0.010775616, 0.010892855, 0.011015208, 0.011136739, 0.011262669,
0.011398287, 0.011522628, 0.011659548, 0.011793156, 0.011927567,
0.012058741, 0.012193749, 0.012336499, 0.012480777, 0.012624995,
0.012774382, 0.012923971, 0.013080718, 0.013227255, 0.013373483,
0.01352752, 0.013686408, 0.013843372, 0.014005478, 0.014165016,
0.01433344, 0.014501415, 0.014665907, 0.014843232, 0.015022472,
0.015198322, 0.015368289, 0.015547129, 0.015734833, 0.015913916,
0.016096434, 0.016288053, 0.01648043, 0.016657719, 0.016861903,
0.017060612, 0.017257382, 0.017458803, 0.017663088, 0.017875744,
0.018082364, 0.018297807, 0.018516931, 0.018734564, 0.018950813,
0.019165184, 0.019381779, 0.019596919, 0.0198221, 0.020051345,
0.020283417, 0.020526649, 0.020767427, 0.021011533, 0.021259273,
0.021499816, 0.021750506, 0.021997156, 0.022263522, 0.02251955,
0.022771712, 0.023027473, 0.02329574, 0.023572599, 0.0238611,
0.024132474, 0.024423853, 0.024713213, 0.025007192, 0.025285099,
0.025570107, 0.025854399, 0.026139919, 0.026450642, 0.026750809,
0.027056901, 0.027369447, 0.027686566, 0.0280112, 0.028334473,
0.028670597, 0.029020824, 0.029366381, 0.029717523, 0.030053876,
0.030418594, 0.030765126, 0.031129199, 0.0314916, 0.031862319,
0.032228778, 0.032594581, 0.032971976, 0.033364999, 0.03375905,
0.034158006, 0.034560072, 0.034952103, 0.035356497, 0.035772058,
0.036189038, 0.036608262, 0.03703296, 0.037470208, 0.037900245,
0.038338367, 0.038793272, 0.039245729, 0.039690489, 0.04014047,
0.04060804, 0.041069827, 0.041540644, 0.042046353, 0.042537979,
0.043037598, 0.043526439, 0.044021293, 0.044549004, 0.045048404,
0.045587785, 0.046121267, 0.046671876, 0.047208008, 0.04775404,
0.048289526, 0.04886904, 0.049450655, 0.050011345, 0.050593458,
0.051212101, 0.051815751, 0.052420298, 0.053040891, 0.053652957,
0.054296226, 0.054930164, 0.055578928, 0.056233111, 0.056882915,
0.057542318, 0.058221232, 0.05889295, 0.059584029, 0.06029178,
0.060992831, 0.061719016, 0.062446732, 0.063162001, 0.063922656,
0.064676632, 0.065426584, 0.066197884, 0.066974644, 0.067762881,
0.068551557, 0.069350287, 0.070135562, 0.070954051, 0.071776225,
0.072620064, 0.073473578, 0.074342041, 0.075210557, 0.07608673,
0.076973962, 0.077870805, 0.078769338, 0.079697353, 0.08061394,
0.081562918, 0.082509665, 0.083477014, 0.084435753, 0.085400907,
0.086393829, 0.087390137, 0.088409441, 0.089458632, 0.090508247,
0.091579829, 0.092640506, 0.093731129, 0.094852477, 0.095952482,
0.097072379, 0.097964457, 0.098201534, 0.099341076, 0.100509767,
0.101713823, 0.102904584, 0.104105727, 0.105327133, 0.106575237,
0.107832932, 0.109092855, 0.110345798, 0.111637774, 0.112933263,
0.11425739, 0.115595088, 0.116938802, 0.118307762, 0.119709125,
0.121112734, 0.122520632, 0.123929965, 0.125377851, 0.126827128,
0.128300375, 0.129811467, 0.131342677, 0.132869569, 0.134419975,
0.136015991, 0.137602408, 0.139193988, 0.140817598, 0.142491197,
0.144150489, 0.145862943, 0.14755778, 0.149315189, 0.151102712,
0.152856271, 0.154659404, 0.156474048, 0.158285836, 0.160143439,
0.162001354, 0.163929702, 0.165843246, 0.167822765, 0.169804134,
0.171807259, 0.173823377, 0.175905066, 0.177967484, 0.180106201,
0.182215309, 0.184372179, 0.186569749, 0.188806861, 0.190995496,
0.19324511, 0.195577572, 0.197924431, 0.200268507, 0.202641804,
0.205021261, 0.20744692, 0.209928246, 0.212391455, 0.214875364,
0.217461202, 0.220049915, 0.222676123, 0.225290871, 0.227967772,
0.230678246, 0.233415559, 0.236224426, 0.239043453, 0.241870895,
0.244738529, 0.247687795, 0.250689288, 0.253680523, 0.25670441,
0.259744662, 0.262867688, 0.266028492, 0.269220539, 0.27244601,
0.275716591, 0.279031205, 0.282381191, 0.285772514, 0.289207871,
0.292642262, 0.296188788, 0.299757843, 0.300548528, 0.303404921,
0.30704805, 0.310743352, 0.314515441, 0.318301876, 0.322166027,
0.326094918, 0.329992448, 0.333956971, 0.338016755, 0.34216987,
0.346306501, 0.35050006, 0.354729422, 0.359046304, 0.363384846,
0.36774712, 0.3722477, 0.376776538, 0.381332539, 0.386049716,
0.390777495, 0.395548472, 0.400447644, 0.405369419, 0.406402025,
0.410354489, 0.415450448, 0.42060558, 0.425849448, 0.431071201,
0.43640345, 0.441821473, 0.447362758, 0.452925482, 0.45853981,
0.464233822, 0.469984177, 0.475898695, 0.481820761, 0.487806738,
0.493916992, 0.500083429, 0.506348127, 0.512781878, 0.519175607,
0.525691558, 0.532232556, 0.538921631, 0.545727666, 0.552648067,
0.55962564, 0.566769163, 0.574040408, 0.581344226, 0.588778745,
0.596302746, 0.603964616, 0.611760845, 0.619510527, 0.627542045,
0.630937479, 0.635592839, 0.643811069, 0.652090033, 0.660536191,
0.669237722, 0.678002852, 0.686921521, 0.6958268, 0.704905439,
0.714223795, 0.72370052, 0.733276446, 0.742994173, 0.752871948,
0.762897065, 0.773071609, 0.783463282, 0.794061287, 0.804811005,
0.815757311, 0.826785213, 0.838040552, 0.849555766, 0.861259704,
0.87315773, 0.878065251, 0.885186324, 0.897512756, 0.910102704,
0.922768182, 0.935775566, 0.949002143, 0.962515411, 0.976170001,
0.990154511, 1.004441002, 1.018976185, 1.033810491, 1.048958559,
1.064283846, 1.079987859, 1.096060668, 1.112487295, 1.129326882,
1.146574464, 1.160445004, 1.191919324, 1.209372977, 1.226845118,
1.244281568, 1.261659876, 1.279229691, 1.296695401, 1.314097158,
1.331465028, 1.348875436, 1.366139801, 1.383412344, 1.400702726,
1.417937779, 1.435356445, 1.452643622, 1.470012259, 1.487264251,
1.497450564, 1.504509134, 1.521702347, 1.539071316, 1.556339335,
1.573577304, 1.590810444, 1.608094458, 1.625567124, 1.642782158,
1.660030598, 1.67723626, 1.694557683, 1.711923057, 1.729349994,
1.746885701, 1.764407816, 1.781890256, 1.799371332, 1.816880647,
1.834391532, 1.851973254, 1.869567437, 1.887044013, 1.904711718,
1.922385523, 1.932547799, 1.940102804, 1.957846271, 1.975620608,
1.993539463, 2.011264185, 2.029033551, 2.047026763, 2.064880134,
2.082726477, 2.100669735, 2.118726991, 2.136689988, 2.154806695,
2.17315354, 2.191428602, 2.209933002, 2.228328905, 2.246841141,
2.26552183, 2.284213929, 2.302888518, 2.321733987, 2.340537653,
2.359510467, 2.378234426, 2.397211544, 2.416392532, 2.435595798,
2.454965227, 2.474064991, 2.493537387, 2.513288958, 2.532838582,
2.552457238, 2.572379341, 2.588089316, 2.592202859, 2.61243664,
2.63261092, 2.652630575, 2.672868122, 2.693301353, 2.713612861,
2.734274068, 2.755008069, 2.775752077, 2.796729773, 2.817743725,
2.83877383, 2.860096482, 2.881452323, 2.902964544, 2.924669414,
2.946813128, 2.968694573, 2.990861741, 3.012897854, 3.03528377,
3.057663163, 3.080302146, 3.103319593, 3.121543415, 3.126508971,
3.149678316, 3.173329761, 3.196779226, 3.220669257, 3.244684839,
3.269037798, 3.293622245, 3.318190542, 3.343092455, 3.368193024,
3.393675425, 3.418789213, 3.444443201, 3.470080507, 3.496359765,
3.522821579, 3.548891213, 3.575207422, 3.602176054, 3.629367867,
3.65683497, 3.68457476, 3.712845929, 3.740917839, 3.769203326,
3.797510577, 3.826200645, 3.855194991, 3.884484747, 3.914073063,
3.943879219, 3.974408307, 4.004282869, 4.034875954, 4.066354418,
4.097371377, 4.128695993, 4.1600797, 4.19253721, 4.22470417,
4.257484573, 4.290775072, 4.324579452, 4.358018252, 4.391247593,
4.425188169, 4.459351299, 4.493511793, 4.528132167, 4.563168989,
4.598868451, 4.634499869, 4.670373586, 4.705908242, 4.742462814,
4.778792822, 4.815770356, 4.85275991, 4.890100906, 4.927356557,
4.965162751, 5.003401443, 5.041401564, 5.080080519, 5.119401048,
5.158942436, 5.198289711, 5.237685871, 5.277948549, 5.317805083,
5.358410016, 5.398834599, 5.439629729, 5.480951975, 5.522130396,
5.563668472, 5.605920014, 5.648803103, 5.692456682, 5.735564607,
5.779189807, 5.8220187, 5.865351768, 5.908886283, 5.952360706,
5.996611388, 6.041258903, 6.08544067, 6.131433118, 6.176874267,
6.221492379, 6.265415028, 6.311295886, 6.357968626, 6.40465048,
6.414448645, 6.451014598, 6.498563339, 6.546141057, 6.593393883,
6.640314733, 6.687799851, 6.735116619, 6.784281433, 6.832543656,
6.881884679, 6.931261623, 6.980186951, 7.030937217, 7.079258239,
7.129949421, 7.181707482, 7.232551193, 7.283565483, 7.335043097,
7.386912463, 7.43746725, 7.488450509, 7.540758909, 7.593169162,
7.645501103, 7.69703803, 7.750038173, 7.804118575, 7.858346217,
7.911086545, 7.96517272, 8.02042631, 8.076060684, 8.131035978,
8.186655349, 8.240487751, 8.296698718, 8.352038488, 8.410008291,
8.465849742, 8.522965471, 8.578244388, 8.63457062, 8.690840262,
8.749025447, 8.805081189, 8.86252695, 8.921324861, 8.981771027,
9.038123372, 9.095093989, 9.152921685, 9.213846321, 9.273190647,
9.332211436, 9.394243357, 9.454306924, 9.514819665, 9.576033798,
9.639634888, 9.701459228, 9.762013374, 9.824436304, 9.888852969,
9.953750125, 10.016785766, 10.080221859, 10.145598187, 10.212625501,
10.275339238, 10.339237172, 10.405209369, 10.468222157, 10.533534823,
10.598107802, 10.662540413, 10.728495724, 10.794875871, 10.859869366,
10.926429558, 10.994766673, 11.062150743, 11.130684002, 11.20039583,
11.270132011, 11.338996132, 11.407757522, 11.479850258, 11.548726068,
11.616260212, 11.687582845, 11.756759988, 11.827487689, 11.899654102,
11.970475962, 12.045045956, 12.117944479, 12.190759113, 12.265784447,
12.337665908, 12.412453749, 12.487755745, 12.558513236, 12.630304005,
12.703465837, 12.7802537, 12.853333775, 12.926211091, 13.003419358,
13.083700373, 13.159079242, 13.236353333, 13.31268431, 13.391201099,
13.46784636, 13.546357213, 13.623796586, 13.70008304, 13.779906833,
13.857349337, 13.93588803, 14.014943562, 14.096881632, 14.173679669,
14.254448161, 14.33038347, 14.411931829, 14.495159986, 14.578858381,
14.662825102, 14.744792836, 14.82670875, 14.91429547, 14.996403528,
15.07667691, 15.161991714, 15.252356336, 15.338831795, 15.424946902,
15.50632982, 15.59292541, 15.678697721, 15.766528144, 15.85640411,
15.944389489, 16.032153897, 16.121342753, 16.207011689, 16.297181648,
16.38776141, 16.472206898, 16.56185599, 16.651248428, 16.745149389,
16.83843384, 16.932784863, 17.022796321, 17.119759701, 17.219046539,
17.307433287, 17.395548794, 17.487700308, 17.580898771, 17.674084945,
17.772625616, 17.867943499, 17.956172338, 18.057518464, 18.160555814,
18.256335841, 18.347090214, 18.445782853, 18.544262725, 18.63873842,
18.741796974, 18.845503282, 18.938574795, 19.040432312, 19.138586266,
19.233860211, 19.330831603, 19.428621442, 19.534302634, 19.635999178,
19.734399095, 19.839814369, 19.943141348, 20.049125943, 20.148494615,
20.245182057, 20.351349058, 20.456861228, 20.567819726, 20.672598151,
20.779623416, 20.888068427, 20.993195211, 21.096552866, 21.204740573,
21.312256003, 21.333924869, 21.423039034, 21.528863991, 21.631734233,
21.728797768, 21.838600661, 21.946920278, 22.061648426, 22.167613938,
22.273926199, 22.395074726, 22.501123518, 22.609542174, 22.717862372,
22.836833068, 22.948838218, 23.067905723, 23.179842124, 23.293301006,
23.403957458, 23.518336114, 23.631660115, 23.743037947, 23.857283032,
23.979005147, 24.091383601, 24.203636288, 24.320197499, 24.444638741,
24.571321971, 24.684898462, 24.797546338, 24.907696977, 25.034094425,
25.152548681, 25.273103678, 25.395236072, 25.528804541, 25.638819484,
25.769004677, 25.884008235, 25.997167191, 26.115346842, 26.227453484,
26.338522948, 26.468604894, 26.595613021, 26.717997476, 26.848805838,
26.986031708, 27.108755125, 27.244104805, 27.369431021, 27.513268205,
27.653894213, 27.781524731, 27.909038448, 28.040556524, 28.166113302,
28.290329297, 28.419710183, 28.549340373, 28.681572947, 28.818640296,
28.941932245, 29.07204936, 29.200178493, 29.337812515, 29.469524345,
29.601581123, 29.726207553, 29.849300516, 29.979348722, 30.11008238,
30.234714973, 30.376084008, 30.497922793, 30.633158902, 30.778205959,
30.910981776, 31.04842856, 31.179571136, 31.331294238, 31.472103385,
31.599134862, 31.740162775, 31.875439537, 32.013388812, 32.140550281,
32.28425395, 32.434005091, 32.571660875, 32.717762521, 32.867454242,
32.999253951, 33.145492345, 33.311219581, 33.444267939, 33.613546135,
33.778490176, 33.931689709, 34.081554534, 34.230612542, 34.383303307,
34.537522421, 34.705861764, 34.849273263, 35.012521122, 35.170883037,
35.337746999, 35.504600548, 35.661705591, 35.829051995, 35.991783124,
36.148844671, 36.303007401, 36.469257458, 36.622006651, 36.786963241,
36.958462097, 37.139263655, 37.286589283, 37.44795492, 37.602378946,
37.746851777, 37.909340708, 38.056488316, 38.228535231, 38.370671446,
38.526201997, 38.67362326, 38.833034461, 38.982522461, 39.133022676,
39.291371776, 39.433407386, 39.625010528, 39.780401775, 39.943582568,
40.106462974, 40.265451154, 40.441465986, 40.621906039, 40.765672289,
40.928891851, 41.065404182, 41.237857735, 41.402345626, 41.57042902,
41.748819579, 41.954029178, 42.141957786, 42.311822701, 42.488029712,
42.679327567, 42.845754667, 42.996298863, 43.19942955, 43.355474143,
43.542617843, 43.711343125, 43.877996255, 44.013338068, 44.164955521,
44.352878754, 44.570268569, 44.726686011, 44.931862213, 45.137161974,
45.355106755, 45.533617863, 45.743479608, 45.943534577, 46.137783808,
46.327690356, 46.498164823, 46.729438398, 46.94005661, 47.128413935,
47.299936715, 47.491864208, 47.695297751, 47.860920329, 48.05450139,
48.222039178, 48.413566081, 48.632790614, 48.866861504, 49.031384245,
49.248830156, 49.427340509, 49.632737843, 49.866839942, 50.034958677,
50.227452412, 50.420133641, 50.606405855, 50.851737147, 51.033967013,
51.220022225, 51.383812545, 51.412590707, 51.584768363, 51.793253809,
51.981775983, 52.18059352, 52.419303766, 52.680054338, 52.881236143,
53.083332567, 53.287725074, 53.484021235, 53.707952213, 53.902501871,
54.171602671, 54.367330308, 54.577247166, 54.791125607, 55.034589736,
55.199706481, 55.504680329, 55.739737409, 55.881983065, 56.061306789,
56.257981376, 56.469989991, 56.667093756, 56.902999371, 57.120706526,
57.388879106, 57.588072291, 57.774327907, 58.034546492, 58.257361347,
58.499627191, 58.665231733, 58.903875938, 59.068591999, 59.331640232,
59.59891297, 59.852743397, 60.095377277, 60.336085388, 60.616277046,
60.87670842, 61.130402551, 61.344660424, 61.592131808, 61.828741334,
62.014295261, 62.25849466, 62.486908431, 62.666434124, 62.88308536,
63.160095611, 63.30195569, 63.521508568, 63.794651377, 64.046969358,
64.242529395, 64.424866796, 64.680201719, 64.841233258, 65.101885797,
65.430784567, 65.699693818, 65.914932732, 66.10013208, 66.337089003,
66.554217329, 66.737388189, 66.924812082, 67.191845469, 67.373396074,
67.654250349, 67.822270291, 67.988302537, 68.316997837, 68.532218883,
68.731274497, 68.974300134, 69.170073195, 69.427940594, 69.724558577,
70.013121415, 70.221567567, 70.401102032, 70.603774652, 70.8029222,
71.032380609, 71.237046756, 71.556353115, 71.957864437, 72.158432553,
72.390350177, 72.673183779, 72.908713318, 73.118441779, 73.449990639,
73.745602563, 73.962503297, 74.230535031, 74.414504787, 74.675638418,
74.852151796, 75.210934335, 75.527064654, 75.728301186, 76.003644239,
76.199978853, 76.462854301, 76.652718435, 76.860784292, 77.19629523,
77.566595624, 77.866918526, 78.261408882, 78.54316052, 78.841946779,
79.18179811, 79.493847943, 79.791090943, 79.953072464, 80.461991677,
80.75749843, 81.004045254, 81.249533685, 81.411426191, 81.771852894,
82.12384154, 82.323479345, 82.636618117, 82.941961363, 83.363029317,
83.596675405, 83.871213496, 84.157037636, 84.413573179, 84.685586188,
85.115006391, 85.443223563, 85.696174931, 85.825915339, 86.027608243,
86.378842569, 86.68734833, 86.878211831, 87.179844612, 87.481498786,
87.730569121, 88.10327244, 88.346254718, 88.725454093, 88.991925784,
89.373420139, 89.640224596, 90.000401389, 90.428242912, 90.654743899,
90.932248147, 91.143774286, 91.462434799, 91.803206817, 91.998159496,
92.326679894, 92.733522583, 93.046938964, 93.350857294, 93.572884047,
93.836150594, 94.200595473, 94.434116252, 94.895158325, 95.097075978,
95.585855113, 95.952748123, 96.298747757, 96.734245729, 96.959600476,
97.326333153, 97.74323943, 98.059616368, 98.211484432, 98.34865051,
98.677349857, 98.972340143, 99.393307159, 99.859871659, 100.120892154,
100.341315649, 100.542263718, 100.884989985, 101.209652637, 101.494645106,
102.01792074, 102.531202599, 102.852517962, 103.145958904, 103.593121992,
103.739777823, 104.081820536, 104.304192262, 104.679931707, 104.939346016,
105.157818389, 105.574176705, 106.220991213, 106.605145597, 106.868605872,
107.083617821, 107.471452485, 107.877856029, 108.045994456, 108.349528324,
108.56177967, 109.054125635, 109.327054305, 109.502323685, 109.897114907,
110.664880229, 111.000497994, 111.253280957, 111.506870445, 112.220381101,
112.516873859, 112.946147213, 113.163092471, 113.715504408, 114.019605,
114.605068717, 114.775355308, 114.861644549, 115.053893952, 115.24237858,
115.387284944, 115.956221007, 116.046435335, 116.566491796, 116.973471041,
117.280988567, 117.631935704, 117.958298681, 118.182900985, 118.638072007,
118.911506259, 119.04184583, 119.73271414, 120.566878209, 120.968571472,
121.598199692, 121.920499304, 122.497426345, 123.006710012, 123.228803912,
123.729878752, 124.061621816, 124.343555335, 124.456428003, 124.690555015,
125.503325047, 125.740359182, 125.884879381, 126.721358459, 127.102857763,
127.523346739, 127.796053194, 128.099269351, 128.270418922, 128.5038192,
129.105458953, 129.459765672, 129.678612501, 130.394600773, 131.16848018,
131.525039246, 131.6963765, 132.232038619, 132.511170688, 133.771679732,
134.716874656, 135.134554139, 135.530753077, 135.953787819, 136.297396611,
136.987373535, 137.381437802, 138.244585679, 138.382736579, 139.214249305,
140.001275516, 140.941763139, 141.136300375, 141.589193846, 141.870084565,
142.120332494, 142.332948588, 142.742126237, 143.186477971, 144.332316837,
144.518526615, 145.66070562, 147.13773539, 147.322039686, 147.642162434,
147.935372929, 148.694068888, 148.771569971, 148.925469098, 149.034911642,
149.129365377, 149.24072046, 149.560597049, 149.708982929, 150.334315906,
151.090898581, 151.121089253, 152.091815628, 152.104337605, 152.857317674,
153.070094619, 154.039484412, 154.171415905, 154.66542309, 155.669673599,
155.780557184, 156.179207475, 156.740731425, 156.965143864, 157.248451124,
157.278619943, 158.125477862, 158.437160821, 158.476637147, 159.015198749,
159.221400347, 159.780332745, 159.826486461, 159.90542881, 160.844455667,
161.510348581, 161.537187694, 161.716769583), "100" = c(1.9123e-05,
1.9736e-05, 1.9952e-05, 2.0636e-05, 2.0833e-05, 2.1049e-05, 2.1131e-05,
2.1634e-05, 2.2694e-05, 2.2743e-05, 2.3126e-05, 2.362e-05, 2.3687e-05,
2.3859e-05, 2.4541e-05, 2.4736e-05, 2.5204e-05, 2.554e-05, 2.5815e-05,
2.6004e-05, 2.6578e-05, 2.6893e-05, 2.7228e-05, 2.7514e-05, 2.7789e-05,
2.7903e-05, 2.8467e-05, 2.8575e-05, 2.8726e-05, 2.9109e-05, 2.9279e-05,
2.9874e-05, 3.0426e-05, 3.049e-05, 3.1097e-05, 3.1635e-05, 3.234e-05,
3.3326e-05, 3.3606e-05, 3.4237e-05, 3.4386e-05, 3.4627e-05, 3.4792e-05,
3.4833e-05, 3.4958e-05, 3.5527e-05, 3.6286e-05, 3.6837e-05, 3.7093e-05,
3.7321e-05, 3.7355e-05, 3.7483e-05, 3.7763e-05, 3.8108e-05, 3.8205e-05,
3.8293e-05, 3.8877e-05, 3.9397e-05, 3.9723e-05, 3.9914e-05, 4.0052e-05,
4.0103e-05, 4.0897e-05, 4.1775e-05, 4.2192e-05, 4.258e-05, 4.3222e-05,
4.3455e-05, 4.3676e-05, 4.3895e-05, 4.4154e-05, 4.4725e-05, 4.5157e-05,
4.553e-05, 4.5852e-05, 4.6142e-05, 4.6896e-05, 4.7973e-05, 4.8088e-05,
4.9365e-05, 4.9703e-05, 5.0113e-05, 5.0681e-05, 5.224e-05, 5.248e-05,
5.4033e-05, 5.4703e-05, 5.5175e-05, 5.6291e-05, 5.7562e-05, 5.8735e-05,
5.9062e-05, 5.9322e-05, 5.9656e-05, 5.9818e-05, 6.0815e-05, 6.1705e-05,
6.2042e-05, 6.2381e-05, 6.2936e-05, 6.3332e-05, 6.3835e-05, 6.4751e-05,
6.5128e-05, 6.5633e-05, 6.624e-05, 6.6896e-05, 6.7665e-05, 6.8232e-05,
6.9476e-05, 7.0098e-05, 7.0734e-05, 7.1626e-05, 7.2225e-05, 7.3412e-05,
7.4465e-05, 7.6432e-05, 7.7738e-05, 7.8472e-05, 8.0004e-05, 8.0352e-05,
8.1719e-05, 8.2535e-05, 8.3804e-05, 8.4997e-05, 8.6929e-05, 8.7507e-05,
8.8777e-05, 9.0586e-05, 9.1215e-05, 9.1547e-05, 9.2228e-05, 9.2981e-05,
9.4217e-05, 9.4445e-05, 9.5717e-05, 9.6896e-05, 9.7878e-05, 9.9746e-05,
0.000100579, 0.000101858, 0.000102555, 0.000103623, 0.000105202,
0.000106517, 0.000107988, 0.000109884, 0.00011022, 0.000111651,
0.000113622, 0.000114348, 0.000116007, 0.00011761, 0.00011986,
0.000121414, 0.00012394, 0.000125299, 0.00012678, 0.000129606,
0.000130116, 0.000131909, 0.000133311, 0.000134832, 0.000135912,
0.000138205, 0.000138976, 0.000139919, 0.000141929, 0.000142882,
0.000143994, 0.000145668, 0.000146902, 0.000149005, 0.000151125,
0.000152102, 0.000153419, 0.000155524, 0.000157266, 0.000158737,
0.000159652, 0.000161895, 0.000163563, 0.000164985, 0.000166591,
0.000167898, 0.00016985, 0.00017229, 0.00017411, 0.000175558,
0.000177782, 0.000179258, 0.000181096, 0.00018321, 0.000184826,
0.000186249, 0.00018785, 0.000190313, 0.000192498, 0.000194745,
0.000197116, 0.000199871, 0.00020106, 0.000203148, 0.000204865,
0.000208034, 0.000208694, 0.000210388, 0.000212994, 0.00021677,
0.000218258, 0.000219863, 0.000222198, 0.000225352, 0.000227863,
0.00023057, 0.00023452, 0.000237159, 0.000239998, 0.000244228,
0.000246669, 0.000248932, 0.000252597, 0.000255055, 0.000258993,
0.000261771, 0.000264088, 0.000266675, 0.000270702, 0.000275055,
0.0002782, 0.000280691, 0.000283919, 0.00028632, 0.000289595,
0.000293073, 0.000297494, 0.000302661, 0.000307915, 0.000311205,
0.000314912, 0.000318602, 0.000322669, 0.000327312, 0.000331092,
0.000335333, 0.000338742, 0.000341412, 0.000344957, 0.000349583,
0.000352737, 0.000356837, 0.000360053, 0.000363805, 0.000368092,
0.000370919, 0.000374723, 0.000377892, 0.000382669, 0.000387794,
0.00039156, 0.000395127, 0.000399463, 0.000402481, 0.000406971,
0.000412853, 0.000418201, 0.000423111, 0.00042941, 0.000436019,
0.00043888, 0.000442802, 0.000449236, 0.000456701, 0.000462892,
0.000468006, 0.00047141, 0.000476438, 0.000482089, 0.000485858,
0.000490543, 0.000496345, 0.000501564, 0.000505997, 0.000512174,
0.000517906, 0.000523832, 0.000528643, 0.000533687, 0.000540468,
0.000545894, 0.000552255, 0.000556628, 0.000563068, 0.000571389,
0.000577492, 0.000584669, 0.000591133, 0.000598408, 0.000606794,
0.000612552, 0.000619958, 0.000626562, 0.000633478, 0.00063813,
0.000645626, 0.000653479, 0.00065949, 0.000667466, 0.000674238,
0.000684431, 0.000692875, 0.000703054, 0.000709123, 0.00071684,
0.000726268, 0.000734288, 0.000744044, 0.000751393, 0.000759239,
0.000767587, 0.000776897, 0.000785953, 0.000795834, 0.000804611,
0.000815249, 0.000827512, 0.000837703, 0.000847796, 0.000859153,
0.00086853, 0.000879598, 0.000889529, 0.000900081, 0.00090954,
0.000919101, 0.000929832, 0.00094066, 0.000948699, 0.000959802,
0.000973483, 0.000983215, 0.000985892, 0.000993671, 0.001005588,
0.001013978, 0.001022825, 0.00103482, 0.001046951, 0.001061273,
0.001074148, 0.001087183, 0.001098411, 0.001109094, 0.001120692,
0.00113239, 0.00114252, 0.00115819, 0.00117008, 0.001182838,
0.001196416, 0.001209245, 0.001221912, 0.001237803, 0.00125397,
0.001266098, 0.001281255, 0.001295335, 0.001310962, 0.001326587,
0.001348241, 0.001365574, 0.001382849, 0.001398391, 0.001413821,
0.001429027, 0.001445955, 0.001462817, 0.001482272, 0.00149878,
0.00151357, 0.001529473, 0.001546028, 0.001565174, 0.001583239,
0.001601323, 0.001620996, 0.001639951, 0.00165844, 0.001675148,
0.001696394, 0.001715796, 0.001734586, 0.001751732, 0.001769274,
0.00178858, 0.001812623, 0.001833139, 0.001852008, 0.001871982,
0.001894286, 0.001915239, 0.001934137, 0.001955934, 0.001979119,
0.002003646, 0.002024837, 0.002046225, 0.002066941, 0.002093272,
0.002116134, 0.002142677, 0.002171402, 0.002197694, 0.002224777,
0.002249493, 0.002273528, 0.002299737, 0.002319065, 0.002344646,
0.002375195, 0.00240534, 0.002430435, 0.002461458, 0.002490433,
0.002522199, 0.002550691, 0.002580889, 0.002611167, 0.002639702,
0.002669305, 0.002696471, 0.002727623, 0.002761267, 0.002789493,
0.002823371, 0.002851605, 0.002885755, 0.002919118, 0.002952602,
0.002989136, 0.003023418, 0.003059531, 0.003096386, 0.003130381,
0.00316893, 0.003205376, 0.00323832, 0.0032757, 0.003313049,
0.003349296, 0.003392303, 0.003434958, 0.003476086, 0.003520079,
0.003562206, 0.0036053, 0.003642699, 0.003683055, 0.003723767,
0.003765996, 0.003807741, 0.003852537, 0.003894684, 0.003940451,
0.00398654, 0.004035055, 0.004089241, 0.004140856, 0.004186253,
0.004232987, 0.004279517, 0.004330918, 0.00438271, 0.004431728,
0.00448848, 0.004541072, 0.004596412, 0.004649194, 0.00470279,
0.004752375, 0.004809188, 0.004865869, 0.004923529, 0.00497731,
0.005031294, 0.005087978, 0.005146283, 0.005208921, 0.005272617,
0.005333221, 0.005397769, 0.005464588, 0.005529424, 0.005595526,
0.005654626, 0.00572722, 0.005792532, 0.0058563, 0.005919167,
0.005987743, 0.006056266, 0.006131331, 0.006197538, 0.006271319,
0.006346223, 0.006418705, 0.006490712, 0.006565145, 0.006645397,
0.006719306, 0.006795755, 0.006872029, 0.006953585, 0.007034997,
0.00711846, 0.007196521, 0.00728679, 0.007368389, 0.007449817,
0.007538234, 0.007619667, 0.007704934, 0.007793076, 0.007884724,
0.007978876, 0.008072783, 0.0081736, 0.008281882, 0.008380715,
0.00847982, 0.008577965, 0.008675058, 0.008774886, 0.008868612,
0.008969667, 0.009067206, 0.009173741, 0.009284438, 0.009394798,
0.009507099, 0.009619416, 0.009699258, 0.009722345, 0.009831479,
0.009948376, 0.010060003, 0.010179634, 0.010301919, 0.010421608,
0.010546812, 0.010674739, 0.010796063, 0.010919421, 0.011050537,
0.011175416, 0.011298151, 0.011430511, 0.011566219, 0.011698505,
0.011835163, 0.011972124, 0.01210739, 0.012249038, 0.012390303,
0.012526863, 0.01267491, 0.01281721, 0.012961447, 0.013115288,
0.013266601, 0.013424832, 0.013580005, 0.0137387, 0.013902144,
0.014065324, 0.014233493, 0.014401464, 0.014564179, 0.014725362,
0.014890533, 0.015062726, 0.01523923, 0.015417249, 0.015591146,
0.015772141, 0.015963732, 0.016148264, 0.016331476, 0.016521939,
0.016707817, 0.016899835, 0.017094265, 0.017282997, 0.017479894,
0.017693381, 0.017893474, 0.01810576, 0.018310885, 0.018531894,
0.01873805, 0.018953945, 0.01916331, 0.019379277, 0.019602773,
0.019828262, 0.020073568, 0.020304731, 0.020530559, 0.020765301,
0.021000495, 0.021242669, 0.02149153, 0.021739526, 0.021992409,
0.022246602, 0.022514247, 0.02277531, 0.02304282, 0.023325388,
0.023590887, 0.02385785, 0.024130806, 0.024407336, 0.024695083,
0.024990257, 0.025279681, 0.025568798, 0.025868637, 0.02616843,
0.026482825, 0.026783048, 0.027099044, 0.027410685, 0.027730788,
0.028058081, 0.028372469, 0.028714981, 0.029054309, 0.02939348,
0.029738073, 0.030086838, 0.030447463, 0.030803786, 0.031164702,
0.031526441, 0.031888593, 0.032257269, 0.032627869, 0.033014322,
0.033403902, 0.03378436, 0.034167788, 0.03456021, 0.034965271,
0.03538025, 0.035794578, 0.036188385, 0.036616109, 0.037051644,
0.03748871, 0.03791356, 0.038355481, 0.038818998, 0.039282844,
0.039734698, 0.0401957, 0.040657765, 0.041143854, 0.041611208,
0.042076669, 0.042573072, 0.043062815, 0.043571414, 0.044085495,
0.044600245, 0.045140964, 0.045663528, 0.046187515, 0.046723169,
0.047267174, 0.047829929, 0.048386001, 0.048958803, 0.049550977,
0.050135109, 0.050729451, 0.051335523, 0.051930803, 0.052525926,
0.053125691, 0.053728763, 0.054356884, 0.054986896, 0.055627428,
0.056261619, 0.056917912, 0.057576154, 0.058241762, 0.05890784,
0.05958429, 0.060295983, 0.060989141, 0.061703659, 0.062424345,
0.063167199, 0.063882837, 0.064612425, 0.065387315, 0.066159205,
0.066937137, 0.067705687, 0.068478873, 0.06928452, 0.070107003,
0.070938806, 0.071762584, 0.072589234, 0.07344909, 0.074322353,
0.075202338, 0.076080115, 0.07696713, 0.077843215, 0.078749015,
0.079662908, 0.080605526, 0.081532992, 0.082490116, 0.083468841,
0.084461531, 0.085471766, 0.086490668, 0.087505908, 0.088538408,
0.089598091, 0.090637846, 0.091680295, 0.092742893, 0.093867425,
0.094981593, 0.096108692, 0.097238576, 0.098145782, 0.098387071,
0.099546163, 0.100707451, 0.101887982, 0.103083634, 0.104282477,
0.105497597, 0.106717768, 0.107980756, 0.109242205, 0.110521689,
0.111825166, 0.113135694, 0.114478149, 0.115822795, 0.11717763,
0.11856233, 0.119934555, 0.121355302, 0.12279279, 0.124217213,
0.125703239, 0.127147152, 0.128635854, 0.130143632, 0.131712147,
0.133260961, 0.134860132, 0.136458456, 0.138051489, 0.139679571,
0.14135716, 0.14299787, 0.144684446, 0.146388746, 0.148126265,
0.149860309, 0.151634206, 0.153435632, 0.155232123, 0.157038443,
0.158910643, 0.160828937, 0.162760547, 0.16471665, 0.166653622,
0.168633603, 0.17064477, 0.172673229, 0.174738041, 0.176801118,
0.17890264, 0.181041185, 0.183198041, 0.185385549, 0.187588075,
0.18979868, 0.192062379, 0.194358695, 0.196694416, 0.199068703,
0.20143532, 0.203815191, 0.206250554, 0.208642799, 0.211140495,
0.213691436, 0.216237989, 0.218849038, 0.221463411, 0.224056087,
0.22674043, 0.229474628, 0.232263108, 0.235051529, 0.237847573,
0.240676717, 0.243585808, 0.246503828, 0.249423073, 0.252426458,
0.25545919, 0.258560529, 0.261644852, 0.264780169, 0.267952551,
0.271108013, 0.274387195, 0.277680098, 0.281032299, 0.284436096,
0.287930551, 0.291392844, 0.294913221, 0.298464447, 0.302067865,
0.302866179, 0.305765385, 0.309499973, 0.313281163, 0.317024931,
0.320863042, 0.324767988, 0.328742382, 0.332784942, 0.336884245,
0.341012398, 0.345207072, 0.349495822, 0.353769971, 0.358136125,
0.362555614, 0.367029314, 0.371581937, 0.376201105, 0.380871492,
0.385594562, 0.390363844, 0.395213589, 0.400130828, 0.405083568,
0.410094579, 0.411142967, 0.415208794, 0.420385988, 0.425634291,
0.430947258, 0.436339581, 0.441758566, 0.447284784, 0.45283934,
0.458493395, 0.464250719, 0.470078457, 0.475989084, 0.481978556,
0.488186721, 0.494390803, 0.500659614, 0.5069182, 0.513335866,
0.519853628, 0.526433153, 0.533194312, 0.539972074, 0.546831044,
0.553858873, 0.560998548, 0.568223941, 0.575568732, 0.582930802,
0.590441819, 0.59811447, 0.605885845, 0.613730439, 0.621816271,
0.629924828, 0.638125071, 0.641625772, 0.64639394, 0.654882897,
0.663412644, 0.672162886, 0.680982311, 0.689984745, 0.699080533,
0.708362398, 0.717792309, 0.727375216, 0.737110552, 0.747140151,
0.757165129, 0.767396654, 0.777730774, 0.788352708, 0.799205904,
0.810188502, 0.821165443, 0.832495589, 0.843932535, 0.855583208,
0.867495303, 0.879591821, 0.891885809, 0.897069671, 0.904445966,
0.917334769, 0.930273848, 0.943564544, 0.957026111, 0.970740588,
0.984635751, 0.998886729, 1.013504371, 1.028289947, 1.043408279,
1.058838128, 1.074533627, 1.090653623, 1.10703649, 1.12377564,
1.140833833, 1.158139026, 1.175934777, 1.190389381, 1.223344276,
1.241695033, 1.259929639, 1.278140051, 1.296418503, 1.314758992,
1.332843576, 1.351072081, 1.369200395, 1.387473607, 1.405717115,
1.423841322, 1.442050346, 1.460176126, 1.478383854, 1.496489438,
1.514732699, 1.532836691, 1.543532804, 1.550909578, 1.568987845,
1.586977166, 1.605126541, 1.623231686, 1.641306711, 1.659490364,
1.677771581, 1.6959189, 1.71399534, 1.732282071, 1.750381662,
1.76851203, 1.786650804, 1.804829891, 1.823163932, 1.841365502,
1.859781362, 1.878052886, 1.896369509, 1.914555083, 1.932804026,
1.951207791, 1.969544488, 1.988001858, 1.998648816, 2.006568487,
2.025029377, 2.043483311, 2.061984762, 2.080526447, 2.099087856,
2.117964273, 2.136814162, 2.155561761, 2.174395219, 2.193129546,
2.211880715, 2.230759441, 2.249592048, 2.268505882, 2.287497344,
2.306659656, 2.325845501, 2.34505346, 2.36438555, 2.383692682,
2.403107458, 2.42246014, 2.441847727, 2.46146558, 2.481106634,
2.500851827, 2.52067845, 2.540459441, 2.560366445, 2.580267646,
2.600016837, 2.619908502, 2.640134651, 2.660528144, 2.676722365,
2.680911648, 2.701496917, 2.721845732, 2.742086461, 2.76263693,
2.783425016, 2.804113072, 2.824825713, 2.845854982, 2.867115101,
2.888483661, 2.909721328, 2.931324628, 2.952880628, 2.974426523,
2.996359045, 3.018151836, 3.040338996, 3.062564121, 3.084618751,
3.106767478, 3.129087548, 3.151949795, 3.174268846, 3.197325023,
3.214957342, 3.220055584, 3.243227524, 3.266543575, 3.289845114,
3.313525443, 3.336948938, 3.360804975, 3.384875949, 3.409251078,
3.433551089, 3.457674977, 3.482315638, 3.507305149, 3.532233625,
3.557411439, 3.582311305, 3.60768399, 3.633047387, 3.658693866,
3.684473274, 3.710545985, 3.736630047, 3.763068599, 3.789600896,
3.815879919, 3.842818705, 3.870443563, 3.897682001, 3.925288005,
3.953430454, 3.981463666, 4.009757473, 4.038593523, 4.067408415,
4.096485059, 4.125592162, 4.154997487, 4.184644105, 4.214237697,
4.244015142, 4.274031686, 4.304614323, 4.335417316, 4.366245081,
4.397552377, 4.429529384, 4.461454754, 4.493614847, 4.526068669,
4.558149457, 4.591443575, 4.624637156, 4.657445148, 4.690782066,
4.724888033, 4.758689777, 4.792872641, 4.826934207, 4.861205036,
4.895823693, 4.930694747, 4.965461324, 5.00099421, 5.036996045,
5.073537375, 5.109977083, 5.146154744, 5.182741297, 5.220301543,
5.257672098, 5.295594878, 5.33341371, 5.371444662, 5.409448231,
5.447766379, 5.486103557, 5.525331583, 5.564952086, 5.605136964,
5.645385357, 5.684714459, 5.724318925, 5.765124006, 5.805250946,
5.846419405, 5.887498485, 5.928816969, 5.970434187, 6.013029542,
6.055542665, 6.097530565, 6.139759494, 6.181938464, 6.224672682,
6.266919032, 6.310602644, 6.319228461, 6.354216069, 6.397330919,
6.441223077, 6.485744816, 6.530380118, 6.575784456, 6.619825517,
6.665108547, 6.710602577, 6.75663204, 6.803141006, 6.850104264,
6.896897732, 6.942793349, 6.989697344, 7.036717364, 7.085155543,
7.132727981, 7.18099408, 7.228371465, 7.27669633, 7.325209385,
7.374040645, 7.423356573, 7.472546127, 7.521695246, 7.569786515,
7.619802702, 7.670012191, 7.71910529, 7.769988525, 7.820946916,
7.872064392, 7.922901616, 7.975215454, 8.026144514, 8.07688214,
8.129637279, 8.182374528, 8.234662312, 8.288021092, 8.34057135,
8.394283163, 8.446082016, 8.499806804, 8.553675131, 8.609369488,
8.6620522, 8.715763926, 8.770086341, 8.824775254, 8.881693003,
8.937049413, 8.992898489, 9.049055245, 9.104894386, 9.160646475,
9.216956721, 9.275116328, 9.333340232, 9.390784923, 9.449016372,
9.50646488, 9.563686604, 9.622521719, 9.681531798, 9.738670099,
9.800107077, 9.85848011, 9.91956498, 9.976990243, 10.03463491,
10.093436221, 10.151692049, 10.212892633, 10.273951013, 10.335522331,
10.398225737, 10.458400533, 10.519311953, 10.580252786, 10.642322503,
10.70510135, 10.768778204, 10.830827661, 10.894366295, 10.960396927,
11.024767696, 11.08951034, 11.155880259, 11.222258719, 11.285870855,
11.352252173, 11.419608505, 11.485644473, 11.551440886, 11.615856704,
11.681192503, 11.750287242, 11.816001012, 11.883887722, 11.949926136,
12.020577384, 12.090693776, 12.159407341, 12.229277842, 12.299259644,
12.370316728, 12.439274951, 12.511092358, 12.581548866, 12.65227825,
12.721501997, 12.790438396, 12.861543615, 12.930779049, 13.000770429,
13.072080135, 13.143180366, 13.212809564, 13.285428981, 13.359380395,
13.431341531, 13.504044305, 13.575839527, 13.647261996, 13.721559067,
13.795718043, 13.869060812, 13.943432271, 14.019501646, 14.099102945,
14.174350835, 14.250755981, 14.328294253, 14.405651662, 14.482217891,
14.559847896, 14.637764536, 14.718470198, 14.79793134, 14.87649053,
14.955967338, 15.036405503, 15.117511011, 15.198831476, 15.283218208,
15.363329305, 15.441672615, 15.522314067, 15.60582989, 15.68537137,
15.771151705, 15.850688148, 15.933005664, 16.01262109, 16.101482506,
16.183814548, 16.268989192, 16.355948798, 16.439327088, 16.523613982,
16.610893142, 16.692920499, 16.779214643, 16.860652144, 16.947213437,
17.032206789, 17.11972712, 17.209762326, 17.295772149, 17.382041588,
17.471531336, 17.557492254, 17.648792976, 17.736008294, 17.823434188,
17.914142547, 18.005470864, 18.095088954, 18.182741695, 18.275305707,
18.369577609, 18.456380658, 18.555212297, 18.646882625, 18.735820822,
18.824593939, 18.909328632, 19.001431787, 19.095629735, 19.185234472,
19.2806776, 19.37414659, 19.473415555, 19.566581831, 19.666503903,
19.760605062, 19.860898976, 19.965429219, 19.9859773, 20.064950074,
20.159082046, 20.256768823, 20.354042192, 20.451186062, 20.545396182,
20.636420582, 20.742463188, 20.833976859, 20.941631322, 21.043264162,
21.159357681, 21.255764897, 21.365730503, 21.461982849, 21.561033598,
21.656395251, 21.75678034, 21.861898052, 21.96947395, 22.065206678,
22.173594889, 22.288187707, 22.38779479, 22.48580285, 22.600403678,
22.700338422, 22.797301166, 22.908715626, 23.019917859, 23.126405986,
23.238798651, 23.340786779, 23.442777929, 23.554125886, 23.670807147,
23.782232206, 23.898181451, 24.005250518, 24.110350567, 24.220675672,
24.330566636, 24.43895827, 24.571508017, 24.682207966, 24.795030034,
24.907368246, 25.022623764, 25.135683245, 25.253792982, 25.379412593,
25.492529282, 25.610061099, 25.749490176, 25.872807329, 25.997476102,
26.121050747, 26.23002038, 26.343981703, 26.462579927, 26.576578429,
26.697093245, 26.807030677, 26.937120612, 27.044940275, 27.168230057,
27.300077762, 27.423007439, 27.555139848, 27.675055011, 27.800557365,
27.915164363, 28.044188637, 28.162326573, 28.280562148, 28.405919259,
28.523030872, 28.669408278, 28.817504696, 28.952997537, 29.073190992,
29.191103657, 29.300394021, 29.426521415, 29.54448823, 29.659531813,
29.79081771, 29.918332517, 30.047590085, 30.167594994, 30.30426537,
30.435824742, 30.5640073, 30.695852072, 30.83477991, 30.964500437,
31.095271023, 31.227633587, 31.358786825, 31.49856077, 31.633751353,
31.762977956, 31.900222275, 32.016917968, 32.142966743, 32.272639404,
32.417842411, 32.57845748, 32.71610002, 32.872513345, 33.014897696,
33.140530225, 33.290345502, 33.424577229, 33.579212745, 33.751569261,
33.904713422, 34.043849783, 34.186554922, 34.35264227, 34.495360038,
34.647204411, 34.790056528, 34.954931964, 35.105109415, 35.260160194,
35.390649079, 35.570461617, 35.720331706, 35.877022805, 36.029290089,
36.179920903, 36.36586039, 36.524726106, 36.660428878, 36.79770728,
36.931554531, 37.100845858, 37.256673565, 37.414217752, 37.570193665,
37.721500583, 37.877191468, 38.029309219, 38.178942646, 38.318994734,
38.486111457, 38.646558175, 38.81730785, 38.995998302, 39.131339817,
39.295081229, 39.449178106, 39.590162102, 39.744463481, 39.894258439,
40.04279831, 40.197616194, 40.35186313, 40.517373771, 40.681355964,
40.847780981, 41.030728848, 41.209571401, 41.39298602, 41.548504038,
41.716651628, 41.902003817, 42.033179844, 42.223000977, 42.378928959,
42.525423084, 42.682316932, 42.833236976, 42.986745523, 43.139716574,
43.305227978, 43.463214002, 43.637174519, 43.821761434, 43.969838548,
44.147367919, 44.33349284, 44.513131051, 44.672468313, 44.860831049,
45.069472779, 45.262703108, 45.408208815, 45.612001166, 45.790935886,
45.974160855, 46.147554876, 46.310164374, 46.479135288, 46.631766944,
46.854931997, 47.026711487, 47.211373874, 47.392520311, 47.444267804,
47.599620005, 47.803528973, 47.962988212, 48.16351657, 48.317043883,
48.527408832, 48.712331506, 48.916998574, 49.106040004, 49.242068507,
49.43801286, 49.623064385, 49.8246239, 50.039845231, 50.217892299,
50.424061507, 50.605502238, 50.785192125, 50.946861814, 51.120437498,
51.313812223, 51.575699155, 51.76371781, 52.083012544, 52.297261725,
52.515104821, 52.709056962, 52.935190034, 53.132179257, 53.291722185,
53.498534609, 53.725532485, 53.911785619, 54.132286096, 54.327380597,
54.560402889, 54.737926107, 54.953883363, 55.152438322, 55.373668992,
55.59756117, 55.776336093, 55.979199967, 56.229384847, 56.46125835,
56.640243605, 56.816436874, 57.007429129, 57.213674807, 57.465554746,
57.620279566, 57.86934145, 58.067368041, 58.226435358, 58.470631273,
58.711195362, 58.985311286, 59.214948899, 59.47896391, 59.64032395,
59.817483287, 60.110994577, 60.377248787, 60.64026148, 60.909753918,
61.132109229, 61.353885499, 61.581454092, 61.708588944, 61.979374248,
62.259163443, 62.465466998, 62.843924637, 63.042576119, 63.257487572,
63.533838701, 63.741937703, 63.913545966, 64.027392163, 64.342879658,
64.569647866, 64.836478249, 64.984210443, 65.265788699, 65.454095958,
65.732401898, 65.954793394, 66.092898262, 66.33489902, 66.664698177,
66.975662561, 67.207247075, 67.46700461, 67.749702592, 67.950110013,
68.208857935, 68.437779424, 68.759661705, 69.032601786, 69.248924671,
69.407632145, 69.770840018, 70.085811938, 70.308574142, 70.451819405,
70.643657157, 70.865828746, 71.110948792, 71.263652962, 71.508761168,
71.727512947, 71.990814746, 72.154652118, 72.437434805, 72.717810938,
72.97155564, 73.132604357, 73.444351561, 73.686892162, 74.123365889,
74.339574791, 74.639576626, 74.889601653, 75.087648514, 75.347038165,
75.634658776, 75.844795138, 76.188090561, 76.412034558, 76.754802253,
77.022798967, 77.352898311, 77.556325173, 77.890942669, 78.130954524,
78.407111573, 78.691491787, 78.992449735, 79.22695994, 79.522656558,
79.807288044, 80.09361574, 80.326329179, 80.503875417, 80.714686652,
80.957172643, 81.329180156, 81.597886246, 81.845945644, 82.283833744,
82.537224429, 82.856471995, 83.03108103, 83.230005709, 83.487324539,
83.741797991, 84.024876701, 84.212157139, 84.397795214, 84.612835841,
84.897015014, 85.119241758, 85.452297538, 85.537105644, 85.789824138,
85.988925851, 86.285147657, 86.559642573, 86.766213874, 87.056631415,
87.355454834, 87.686476779, 87.927524426, 88.284321369, 88.782422055,
89.000104756, 89.172147359, 89.558718073, 89.792711548, 90.101448628,
90.395377065, 90.567468107, 90.989023411, 91.340142003, 91.492357433,
91.979132755, 92.360561213, 92.509027692, 92.727077468, 93.147930493,
93.369201642, 93.865180726, 94.442804987, 94.738619676, 95.035378562,
95.262206312, 95.652085786, 96.034797018, 96.320700157, 96.803298707,
97.236201607, 97.428213743, 97.655475304, 97.901223714, 98.109797852,
98.271089247, 98.687897937, 98.895617908, 99.156067897, 99.353253776,
99.526105691, 99.697430828, 100.208679731, 100.40560192, 100.836053715,
101.075007006, 101.682773305, 101.916597011, 102.285744305, 103.025965727,
103.493002676, 103.630410442, 104.001680848, 104.364029796, 104.761578674,
104.955528705, 105.313571983, 105.589806324, 105.921709585, 106.100173972,
106.222786823, 106.638331694, 106.94084932, 107.325246019, 107.625274975,
107.893441491, 108.367787813, 108.507041526, 108.719415221, 108.900811147,
109.077495715, 109.284140155, 110.358751024, 110.490369916, 110.631810481,
111.048746449, 111.538270853, 111.786151932, 112.470819804, 113.132351715,
113.427692227, 113.8394687, 114.303716552, 114.615806958, 115.17932686,
115.475137011, 115.769925784, 116.006903477, 116.265410554, 116.639226837,
117.249933157, 117.550767636, 117.892786729, 118.379305118, 118.506628227,
118.888893289, 119.018371205, 120.303393162, 120.511243772, 120.772383152,
121.06779382, 121.43919971, 122.017141891, 122.126027702, 122.400559633,
122.84113843, 122.994405824, 123.486689262, 123.597118998, 124.122056497,
124.537805158, 124.793372484, 125.380030839, 125.43549198, 125.728966358,
126.161799141, 126.372566715, 126.552562284, 126.633200907, 126.764228333,
126.771328563, 127.195916924, 127.687547295, 128.041555058, 128.314543611,
128.425594574, 128.807364195, 128.982890873, 129.171080161, 129.580369458,
129.778617759, 130.549455562, 130.807768178, 131.569525327, 131.884708787,
132.633064884, 133.184368793, 133.243752989, 133.400407284, 133.730267437,
134.00506061, 135.53264541, 135.970348108, 135.990834176, 136.286830846,
136.388315409, 136.683834232, 137.604891782, 138.423097682, 138.438229158,
139.177871354, 139.218644489, 139.40778058, 139.590905156, 139.751564313,
140.210275536, 140.70280496, 140.890712927, 141.956570663, 142.152200153,
142.264160895, 142.357345549, 142.534606479, 142.71300523, 142.829034193,
143.049391658, 143.097203521, 144.112741345, 144.210741426, 145.100980871
), "150" = c(1.9209e-05, 1.9266e-05, 1.9301e-05, 1.9535e-05,
2.0039e-05, 2.0117e-05, 2.0198e-05, 2.0204e-05, 2.0249e-05, 2.0324e-05,
2.035e-05, 2.1512e-05, 2.1673e-05, 2.1829e-05, 2.1993e-05, 2.2179e-05,
2.2366e-05, 2.3111e-05, 2.3364e-05, 2.352e-05, 2.3613e-05, 2.366e-05,
2.4037e-05, 2.4195e-05, 2.4553e-05, 2.4653e-05, 2.5106e-05, 2.5199e-05,
2.5548e-05, 2.6034e-05, 2.6154e-05, 2.6735e-05, 2.6916e-05, 2.7002e-05,
2.7159e-05, 2.7403e-05, 2.7573e-05, 2.7828e-05, 2.7956e-05, 2.8197e-05,
2.8288e-05, 2.8699e-05, 2.9637e-05, 3.0071e-05, 3.0371e-05, 3.0501e-05,
3.0727e-05, 3.1221e-05, 3.1367e-05, 3.1558e-05, 3.1793e-05, 3.2056e-05,
3.2677e-05, 3.3674e-05, 3.4e-05, 3.4605e-05, 3.5032e-05, 3.5197e-05,
3.548e-05, 3.6065e-05, 3.6371e-05, 3.6883e-05, 3.7237e-05, 3.8027e-05,
3.8502e-05, 3.8891e-05, 3.9612e-05, 3.9756e-05, 4.0315e-05, 4.0638e-05,
4.0901e-05, 4.0978e-05, 4.154e-05, 4.2053e-05, 4.2476e-05, 4.2776e-05,
4.3082e-05, 4.4092e-05, 4.4418e-05, 4.5558e-05, 4.5951e-05, 4.7088e-05,
4.8377e-05, 4.9394e-05, 4.9651e-05, 5.0047e-05, 5.0922e-05, 5.1487e-05,
5.1885e-05, 5.3283e-05, 5.3711e-05, 5.4488e-05, 5.4753e-05, 5.6288e-05,
5.7102e-05, 5.7933e-05, 5.8333e-05, 5.8518e-05, 5.9063e-05, 6.0049e-05,
6.0527e-05, 6.126e-05, 6.1862e-05, 6.2151e-05, 6.268e-05, 6.364e-05,
6.5382e-05, 6.5684e-05, 6.7271e-05, 6.762e-05, 6.8295e-05, 6.8456e-05,
6.9219e-05, 6.9599e-05, 7.0224e-05, 7.0703e-05, 7.1836e-05, 7.2595e-05,
7.3442e-05, 7.4138e-05, 7.5643e-05, 7.6607e-05, 7.7083e-05, 7.7903e-05,
7.8991e-05, 8.0343e-05, 8.0803e-05, 8.1694e-05, 8.2245e-05, 8.2529e-05,
8.3792e-05, 8.5219e-05, 8.5946e-05, 8.6949e-05, 8.8239e-05, 8.9189e-05,
8.9904e-05, 9.0878e-05, 9.1554e-05, 9.2447e-05, 9.3341e-05, 9.5047e-05,
9.5904e-05, 9.6332e-05, 9.8204e-05, 9.9086e-05, 0.000100829,
0.000102437, 0.000103887, 0.000104378, 0.000106452, 0.000107481,
0.000108854, 0.000111219, 0.000112348, 0.000113042, 0.000114549,
0.00011565, 0.000116729, 0.000118046, 0.000119042, 0.000121094,
0.000122709, 0.000123284, 0.000126069, 0.000127537, 0.000129352,
0.000130951, 0.000132867, 0.00013494, 0.000137414, 0.000139637,
0.000140917, 0.000142684, 0.000145365, 0.000146804, 0.000149112,
0.000149886, 0.000150809, 0.000152012, 0.000153721, 0.000155046,
0.000157607, 0.000159211, 0.000162224, 0.000163138, 0.000164967,
0.00016671, 0.000168807, 0.000171142, 0.000172717, 0.000174608,
0.000176936, 0.000178293, 0.000180019, 0.000181391, 0.000182968,
0.000185453, 0.000188242, 0.000190804, 0.000193598, 0.000195793,
0.000198141, 0.000199567, 0.000202616, 0.000205451, 0.000207751,
0.000209888, 0.000212397, 0.000216493, 0.000220311, 0.000223579,
0.00022702, 0.000229481, 0.00023341, 0.000236008, 0.00023828,
0.000241997, 0.000245358, 0.000248036, 0.000250723, 0.000253303,
0.000256883, 0.00026097, 0.000264923, 0.000266807, 0.000269362,
0.000274145, 0.000276577, 0.000279066, 0.000281686, 0.000286045,
0.000288957, 0.000293109, 0.000296052, 0.000300781, 0.000303818,
0.000307882, 0.000311584, 0.00031394, 0.000317333, 0.000320439,
0.000323793, 0.00032794, 0.000331983, 0.000336753, 0.000341343,
0.000344421, 0.000348137, 0.000351596, 0.000355304, 0.000358702,
0.000363238, 0.000366897, 0.000369307, 0.000373568, 0.0003777,
0.000380811, 0.0003841, 0.000389384, 0.000393733, 0.000396633,
0.000402125, 0.000407118, 0.000412417, 0.000419166, 0.000424132,
0.000428141, 0.000431434, 0.000435846, 0.000441744, 0.000447026,
0.000452197, 0.000457972, 0.000463947, 0.000469749, 0.000474203,
0.000480184, 0.000485393, 0.000490314, 0.000497012, 0.000501512,
0.000505962, 0.000512703, 0.000517436, 0.00052214, 0.00052685,
0.000533264, 0.00053958, 0.000546622, 0.000553668, 0.000558398,
0.00056583, 0.000571899, 0.000576716, 0.000585049, 0.000592836,
0.000599855, 0.000605246, 0.000609586, 0.00061859, 0.000623986,
0.000632282, 0.000640022, 0.000648492, 0.000658548, 0.000665672,
0.000675037, 0.000682543, 0.000690109, 0.000697321, 0.000703117,
0.000711553, 0.000721094, 0.000731306, 0.000740175, 0.000749969,
0.000757918, 0.000766087, 0.000774596, 0.000785399, 0.000792099,
0.000801819, 0.000809301, 0.000819394, 0.000826904, 0.000835573,
0.00084507, 0.000855751, 0.000867107, 0.000876879, 0.000889382,
0.00089814, 0.000909527, 0.000921999, 0.00093197, 0.000941361,
0.000951956, 0.000961844, 0.000971769, 0.000981907, 0.000984219,
0.000992704, 0.001004394, 0.001014136, 0.001024868, 0.001035487,
0.001047662, 0.001059543, 0.001069911, 0.001082643, 0.001095399,
0.001106777, 0.001118147, 0.001129171, 0.001140519, 0.001152551,
0.001166198, 0.00118048, 0.001194656, 0.001208546, 0.001222889,
0.001239045, 0.001250223, 0.001262297, 0.001277119, 0.001293477,
0.001311051, 0.00132215, 0.001335341, 0.001350174, 0.001367013,
0.001382509, 0.001397501, 0.001411737, 0.00142731, 0.001444936,
0.001464886, 0.001482234, 0.001499864, 0.001517679, 0.001535527,
0.001551517, 0.001569959, 0.001589284, 0.001610149, 0.001626794,
0.001645596, 0.001664102, 0.001685776, 0.001708923, 0.001730672,
0.001750067, 0.001770937, 0.001788731, 0.001809514, 0.001830973,
0.001851454, 0.001874298, 0.001894531, 0.001916731, 0.001938906,
0.001957087, 0.001979217, 0.002004787, 0.002031799, 0.00205643,
0.002081182, 0.002103499, 0.002126747, 0.002152169, 0.00217749,
0.002203974, 0.002228975, 0.002259612, 0.002286029, 0.0023153,
0.00234346, 0.00237027, 0.002394372, 0.002427467, 0.002456457,
0.002484014, 0.002512054, 0.002541989, 0.002572352, 0.002604637,
0.002639559, 0.002668955, 0.002696096, 0.002721039, 0.002752422,
0.002780927, 0.002816409, 0.002844758, 0.002873905, 0.002906011,
0.002937581, 0.002970006, 0.003002503, 0.003039264, 0.003070429,
0.00310381, 0.003143803, 0.003181708, 0.003216285, 0.003250617,
0.003290988, 0.003331647, 0.003373168, 0.003409567, 0.003450917,
0.003485742, 0.003524639, 0.003566031, 0.003600709, 0.00363598,
0.003676708, 0.003716079, 0.003758229, 0.003804388, 0.003850031,
0.003894573, 0.003939693, 0.003983266, 0.004027376, 0.004073081,
0.004119863, 0.004166484, 0.004218594, 0.004267054, 0.004318628,
0.004366887, 0.004418196, 0.0044704, 0.004521671, 0.004569359,
0.004620276, 0.004671006, 0.004721935, 0.004782458, 0.004844255,
0.004906974, 0.004967281, 0.005023869, 0.005086915, 0.005147863,
0.005208199, 0.005269726, 0.005326915, 0.005394873, 0.005456659,
0.005518488, 0.005589432, 0.005652312, 0.005715877, 0.005785148,
0.005853957, 0.005925648, 0.005996879, 0.006064518, 0.00613666,
0.006205688, 0.00627108, 0.006346729, 0.00641574, 0.006487584,
0.0065577, 0.006627695, 0.006704586, 0.006778104, 0.006862298,
0.006940008, 0.007019684, 0.007094273, 0.007181114, 0.007269223,
0.007346352, 0.00743766, 0.007520812, 0.007604526, 0.007694145,
0.007787871, 0.007876896, 0.007968051, 0.008064467, 0.00815917,
0.00825965, 0.00835467, 0.008453838, 0.008553793, 0.008652778,
0.00874863, 0.00885062, 0.008953958, 0.009054532, 0.009153646,
0.009255317, 0.009362676, 0.009464359, 0.00956855, 0.00965899,
0.009683599, 0.009794953, 0.009908462, 0.010019258, 0.010132014,
0.010247506, 0.010374947, 0.010491109, 0.010617553, 0.010742057,
0.010864534, 0.010993838, 0.011119865, 0.011244577, 0.01137169,
0.011510356, 0.011637166, 0.011763629, 0.011906354, 0.012041406,
0.012176883, 0.012316055, 0.012448896, 0.01260042, 0.012757456,
0.012907494, 0.013056594, 0.013201688, 0.013349392, 0.013512159,
0.013676678, 0.013843369, 0.014005153, 0.01416332, 0.01432679,
0.014489734, 0.014657898, 0.014824118, 0.014988932, 0.015165028,
0.015346204, 0.015520916, 0.015703037, 0.015882276, 0.016065362,
0.016251736, 0.016449595, 0.016639782, 0.016838109, 0.017031241,
0.017232101, 0.017438966, 0.017650015, 0.017850685, 0.018058165,
0.01827138, 0.018484927, 0.018688502, 0.018901011, 0.019115238,
0.019334286, 0.019557331, 0.019793446, 0.020018629, 0.020251789,
0.020469302, 0.02071475, 0.020948897, 0.02118787, 0.021433971,
0.02168887, 0.02194115, 0.022207504, 0.022482188, 0.022737656,
0.023006649, 0.023257728, 0.023524258, 0.023788663, 0.024057881,
0.024338012, 0.024625854, 0.024912549, 0.025194394, 0.025498007,
0.025782923, 0.026077775, 0.026381096, 0.026687164, 0.02700652,
0.027329318, 0.02764156, 0.027983087, 0.028294943, 0.028628184,
0.028952782, 0.029289095, 0.029613275, 0.02996641, 0.030318435,
0.030671567, 0.031033978, 0.031400628, 0.031770589, 0.032138668,
0.032500274, 0.032860853, 0.033237082, 0.033633372, 0.03401924,
0.034424996, 0.034824139, 0.035231585, 0.035631552, 0.036048411,
0.03647871, 0.036909612, 0.03733694, 0.037763118, 0.03822293,
0.038674598, 0.039118793, 0.039576998, 0.040031581, 0.040513355,
0.04099384, 0.041476738, 0.041955281, 0.042460166, 0.042962506,
0.043462464, 0.043976968, 0.044490141, 0.045007743, 0.045537144,
0.046079805, 0.046610763, 0.047164328, 0.047710712, 0.048262004,
0.048803022, 0.049374493, 0.049951988, 0.050532784, 0.051123868,
0.05171429, 0.052331877, 0.052934776, 0.053555453, 0.05420203,
0.054847554, 0.055480076, 0.056143167, 0.056804618, 0.057474366,
0.058164362, 0.058846488, 0.059531407, 0.060207672, 0.060922599,
0.061634598, 0.062362757, 0.063084271, 0.063834237, 0.064593961,
0.065353001, 0.066115954, 0.066911169, 0.067719084, 0.068534602,
0.069341583, 0.070170131, 0.071005563, 0.071839742, 0.07266433,
0.073511696, 0.074341789, 0.075222515, 0.076107836, 0.077009206,
0.077903871, 0.078815314, 0.079745747, 0.080671582, 0.081626961,
0.082572326, 0.083533787, 0.08453035, 0.085521444, 0.086526662,
0.087536672, 0.088554301, 0.089593384, 0.090653149, 0.091725828,
0.092808557, 0.093908738, 0.094998315, 0.096136735, 0.097278905,
0.098176157, 0.098412309, 0.099543202, 0.100734052, 0.101897018,
0.103094858, 0.104334122, 0.10557443, 0.106825384, 0.108104276,
0.109386581, 0.110677161, 0.111994266, 0.113322766, 0.114679571,
0.116039815, 0.117414018, 0.118777364, 0.120151828, 0.121542323,
0.122979912, 0.124445556, 0.125926394, 0.127401592, 0.128878936,
0.130418263, 0.131984142, 0.133537039, 0.135118768, 0.136722454,
0.138370796, 0.140024473, 0.141697778, 0.143384769, 0.145117843,
0.146842324, 0.148617637, 0.150399009, 0.152180567, 0.153991164,
0.155847046, 0.157693722, 0.159548181, 0.161437609, 0.163376298,
0.165345115, 0.167297701, 0.169303745, 0.171363683, 0.173388819,
0.175455913, 0.177579386, 0.179700162, 0.181825652, 0.183993413,
0.186190792, 0.18842856, 0.190687565, 0.192979258, 0.195337401,
0.197654163, 0.200027412, 0.202428448, 0.204871914, 0.20731886,
0.209798911, 0.212316168, 0.214850911, 0.217415847, 0.220044854,
0.222687158, 0.225381043, 0.228125375, 0.230892041, 0.233692454,
0.23652038, 0.239346873, 0.242247909, 0.2451886, 0.248178074,
0.251181702, 0.254221443, 0.257295703, 0.260425371, 0.263593842,
0.266817695, 0.270049137, 0.273396651, 0.276688871, 0.280046411,
0.283400174, 0.28689764, 0.290446511, 0.294036905, 0.297603518,
0.301264558, 0.304953539, 0.305768251, 0.30872109, 0.312545747,
0.31643173, 0.320269832, 0.324229703, 0.328239636, 0.332297222,
0.336387533, 0.34054535, 0.344791243, 0.34907268, 0.353397449,
0.357815892, 0.362214495, 0.366720616, 0.371282002, 0.375925639,
0.380599146, 0.385282603, 0.390104285, 0.395025723, 0.399939513,
0.405037514, 0.410085008, 0.415242229, 0.416318529, 0.420446161,
0.425741929, 0.431128059, 0.436545067, 0.442131587, 0.447668589,
0.453368629, 0.459158945, 0.46495933, 0.470908015, 0.476919741,
0.48295277, 0.489105404, 0.495388019, 0.501714505, 0.5080855,
0.514565686, 0.521179752, 0.527873493, 0.534722434, 0.541630909,
0.548608661, 0.555744433, 0.563038393, 0.570363212, 0.577818951,
0.585339038, 0.593085013, 0.600890839, 0.608792205, 0.616832066,
0.625002554, 0.63328792, 0.641680087, 0.650151915, 0.653791163,
0.658785949, 0.667541697, 0.676479017, 0.68550293, 0.694685314,
0.704076276, 0.713545799, 0.723292472, 0.733126825, 0.743197246,
0.753354382, 0.763704123, 0.774228709, 0.784903205, 0.795723431,
0.806705282, 0.817884797, 0.829242268, 0.840772728, 0.852578403,
0.864544684, 0.876773225, 0.88915324, 0.901751453, 0.914667526,
0.919992957, 0.927853313, 0.941219337, 0.954915962, 0.968787294,
0.983010706, 0.997460685, 1.012195279, 1.027148186, 1.042409492,
1.057951064, 1.073872073, 1.090089567, 1.106738999, 1.123660718,
1.140914803, 1.158483603, 1.176456015, 1.194786311, 1.21364817,
1.228988638, 1.263899681, 1.283092052, 1.302245419, 1.321478917,
1.340804783, 1.359878908, 1.379025305, 1.398404019, 1.417720878,
1.437015955, 1.456331263, 1.475583287, 1.494763583, 1.51403115,
1.533209632, 1.552469461, 1.571579208, 1.590869144, 1.602059938,
1.609921574, 1.628952389, 1.648040394, 1.667275613, 1.686256888,
1.70544642, 1.724510948, 1.743807567, 1.762879892, 1.782099228,
1.801168486, 1.820364982, 1.839696734, 1.858777096, 1.877969407,
1.897300212, 1.916660327, 1.936083281, 1.955532039, 1.974773445,
1.994096791, 2.01343133, 2.032804082, 2.052201449, 2.071476677,
2.082625216, 2.090883697, 2.110365732, 2.129818754, 2.149201968,
2.168664682, 2.188301879, 2.207850257, 2.227503553, 2.247233134,
2.267056488, 2.286557549, 2.306379768, 2.326196514, 2.346144146,
2.36584226, 2.385659055, 2.405513167, 2.425612482, 2.44577107,
2.465872158, 2.485872603, 2.506154441, 2.526308077, 2.546537254,
2.566698965, 2.587199825, 2.607754844, 2.62809371, 2.648871064,
2.669342264, 2.689797998, 2.710415792, 2.731220205, 2.75185095,
2.772815392, 2.789468646, 2.793779953, 2.814889031, 2.835991018,
2.857014957, 2.878180484, 2.899222618, 2.920521504, 2.941971562,
2.963347643, 2.985024711, 3.006855294, 3.028514056, 3.050428723,
3.072443648, 3.094589812, 3.116342126, 3.138640297, 3.160844844,
3.183085567, 3.205420505, 3.227820387, 3.250354435, 3.27287626,
3.295374459, 3.317700779, 3.335572458, 3.34066878, 3.363654278,
3.38664237, 3.409930827, 3.433326123, 3.456470972, 3.480066384,
3.503739987, 3.5276592, 3.551634902, 3.575248894, 3.599220043,
3.623321351, 3.647576016, 3.672384114, 3.696916172, 3.721446905,
3.74599116, 3.770898942, 3.796142142, 3.821488138, 3.847201958,
3.873183511, 3.898832894, 3.924644533, 3.95067504, 3.976764808,
4.002996545, 4.028910522, 4.05554462, 4.0819449, 4.108665059,
4.135594854, 4.162982416, 4.190314987, 4.217973411, 4.245509425,
4.273407304, 4.301634164, 4.329574761, 4.357764825, 4.386404699,
4.414728159, 4.443847358, 4.472758331, 4.501775439, 4.530980609,
4.560204882, 4.589632054, 4.619621588, 4.650036963, 4.680558962,
4.7111407, 4.741947196, 4.772771885, 4.803435526, 4.834288059,
4.865738199, 4.897332763, 4.928716135, 4.960858053, 4.992977516,
5.025250368, 5.058042516, 5.09062992, 5.123874724, 5.15726287,
5.190639149, 5.22412927, 5.257881804, 5.291950184, 5.326403924,
5.360841857, 5.395402145, 5.430794777, 5.46591735, 5.501136022,
5.536728739, 5.572849897, 5.608270132, 5.643414926, 5.679543094,
5.716579126, 5.753491009, 5.790155866, 5.826989719, 5.863877875,
5.900788105, 5.938836104, 5.976292972, 6.014345824, 6.05369238,
6.092503849, 6.131287735, 6.170954773, 6.209940382, 6.218244651,
6.249274787, 6.288673135, 6.328793735, 6.368711815, 6.408809428,
6.449309352, 6.490471537, 6.531628593, 6.5724677, 6.614271642,
6.656041702, 6.698733614, 6.741479165, 6.78408798, 6.826176864,
6.868457831, 6.910717357, 6.95419047, 6.997197301, 7.040490342,
7.0838244, 7.126722972, 7.170762369, 7.215889032, 7.259807009,
7.304854488, 7.34906056, 7.394813012, 7.442716949, 7.487876978,
7.534068448, 7.579655872, 7.625361408, 7.670278126, 7.716939368,
7.762691803, 7.809283307, 7.855620691, 7.902363062, 7.948221336,
7.994626752, 8.042806842, 8.090490943, 8.139535437, 8.187107,
8.236001777, 8.285207488, 8.332845675, 8.382319939, 8.431507822,
8.481324011, 8.530332087, 8.580520558, 8.629512642, 8.679398664,
8.728809916, 8.780474745, 8.830941551, 8.883177964, 8.933966989,
8.985254392, 9.035658292, 9.085363442, 9.135780872, 9.186670979,
9.237491328, 9.289983018, 9.341241777, 9.395001643, 9.447298173,
9.499060001, 9.552540339, 9.607500954, 9.660551317, 9.714706409,
9.769169735, 9.823341978, 9.877919465, 9.933945738, 9.988586028,
10.042290144, 10.09773533, 10.152542944, 10.2086893, 10.263669234,
10.319335404, 10.378578374, 10.434531599, 10.492996975, 10.548724776,
10.605893969, 10.662732168, 10.720024988, 10.778782761, 10.83803235,
10.896441145, 10.953396269, 11.011170223, 11.070391406, 11.129889423,
11.191591451, 11.25150065, 11.312335717, 11.37368522, 11.434101089,
11.495814735, 11.558197208, 11.618822013, 11.679866823, 11.742269202,
11.803053721, 11.865590042, 11.927425992, 11.988182339, 12.050378393,
12.113386106, 12.178229991, 12.238566272, 12.301996494, 12.366712977,
12.429187201, 12.4970749, 12.565285395, 12.63242785, 12.696345647,
12.758253174, 12.823228141, 12.890890156, 12.954059553, 13.023429223,
13.090104649, 13.160154633, 13.226974018, 13.294955456, 13.361624127,
13.429643537, 13.501107279, 13.570070648, 13.637213338, 13.707380645,
13.775601833, 13.844464829, 13.914392577, 13.983901644, 14.053795569,
14.127320703, 14.201164818, 14.269442399, 14.340328559, 14.407761051,
14.481858246, 14.555890656, 14.627843818, 14.698391439, 14.770029173,
14.841740956, 14.916590636, 14.987521051, 15.060422634, 15.132387934,
15.204080589, 15.277752128, 15.346554112, 15.4247981, 15.494487129,
15.571482928, 15.647368696, 15.725762317, 15.807202948, 15.881872162,
15.95762509, 16.032645734, 16.10757998, 16.186654963, 16.265327794,
16.340827671, 16.418073051, 16.493097045, 16.574672446, 16.655598977,
16.733136216, 16.814195744, 16.894296904, 16.971773311, 17.048584831,
17.127850551, 17.211572745, 17.289662521, 17.371926383, 17.453964902,
17.530388359, 17.61089009, 17.692679213, 17.771410325, 17.853956979,
17.939639445, 18.025395515, 18.104442145, 18.183549711, 18.267441923,
18.284662954, 18.350070797, 18.438150349, 18.523182917, 18.610147845,
18.69312926, 18.775366505, 18.858783562, 18.94853262, 19.034474505,
19.118915993, 19.202202217, 19.288157352, 19.37588032, 19.459207745,
19.550700046, 19.639404368, 19.730392143, 19.823324872, 19.909235586,
20.00124172, 20.094775065, 20.179529225, 20.266007322, 20.355041299,
20.443519508, 20.535639187, 20.622304749, 20.712032534, 20.807270878,
20.899019132, 21.005937513, 21.104916375, 21.19975323, 21.292758952,
21.385154091, 21.486564237, 21.578409514, 21.665648918, 21.754104141,
21.856912721, 21.948925903, 22.051888743, 22.143247496, 22.240964955,
22.337985216, 22.439677371, 22.533825211, 22.642516024, 22.731364487,
22.826609879, 22.926710012, 23.021424739, 23.120865967, 23.221988811,
23.316972581, 23.419320654, 23.517520205, 23.620515839, 23.728205212,
23.834633548, 23.930271151, 24.038851613, 24.146088478, 24.243135459,
24.339301119, 24.441338671, 24.546136231, 24.653160544, 24.762509631,
24.860530871, 24.967707536, 25.064465975, 25.183034126, 25.301744892,
25.409394687, 25.508607258, 25.611405817, 25.713602733, 25.821592959,
25.925758148, 26.029308767, 26.138375087, 26.240983979, 26.366841709,
26.48189632, 26.588243355, 26.700230326, 26.806412814, 26.916228884,
27.02612163, 27.139969821, 27.251518165, 27.364519358, 27.478455053,
27.574224885, 27.676738541, 27.80493614, 27.92766427, 28.04124914,
28.158975752, 28.276331003, 28.386924827, 28.50682812, 28.617692003,
28.733397598, 28.844657743, 28.966054599, 29.113993841, 29.235966265,
29.348217021, 29.474532857, 29.586448881, 29.698707991, 29.817276477,
29.935632125, 30.054812943, 30.168777068, 30.289124287, 30.407279396,
30.539791091, 30.664465683, 30.797933351, 30.920090535, 31.056674474,
31.174176123, 31.284817637, 31.407654055, 31.556960526, 31.681500827,
31.815707235, 31.946883149, 32.072851986, 32.2007432, 32.336588616,
32.466500293, 32.599487101, 32.738094267, 32.863332148, 32.972170844,
33.106176995, 33.272477821, 33.389633483, 33.533499454, 33.672718191,
33.835902037, 33.963880068, 34.092696995, 34.212697187, 34.335509782,
34.472248111, 34.618256087, 34.736731127, 34.856149868, 35.012330776,
35.136305059, 35.269171878, 35.416146466, 35.538960216, 35.693082589,
35.810482294, 35.943883838, 36.110198849, 36.253154794, 36.387187411,
36.562147206, 36.693376701, 36.83608153, 36.992485171, 37.139566476,
37.267063237, 37.405605243, 37.539385418, 37.705852678, 37.855654329,
38.004734076, 38.131697518, 38.260821175, 38.398728269, 38.522675238,
38.644662118, 38.811868655, 38.942624928, 39.107181931, 39.248291205,
39.37763067, 39.535448819, 39.688538863, 39.83272592, 39.974445565,
40.156933524, 40.314548943, 40.449244148, 40.620607475, 40.753267246,
40.905510714, 41.059373094, 41.216669239, 41.396509419, 41.531204715,
41.689675314, 41.712915785, 41.822429505, 41.958077118, 42.120697661,
42.325762547, 42.524704096, 42.665937193, 42.831410314, 42.992697816,
43.161381676, 43.301190999, 43.454568672, 43.619811065, 43.805449747,
43.963447813, 44.113215257, 44.291819552, 44.431779851, 44.600845312,
44.778555936, 44.94508012, 45.103218213, 45.289346245, 45.507210325,
45.68091046, 45.84033842, 46.006359718, 46.147118244, 46.280130207,
46.453748866, 46.620696606, 46.809104588, 47.004695163, 47.099198122,
47.280243034, 47.434295688, 47.612747639, 47.745331148, 47.892350165,
48.097294021, 48.263678236, 48.484588977, 48.657536917, 48.819812233,
49.00850858, 49.193071808, 49.375617172, 49.553510616, 49.776053167,
49.928045804, 50.130686354, 50.317121978, 50.491366299, 50.685694626,
50.884383876, 51.108329326, 51.255784518, 51.390244664, 51.595071457,
51.731658974, 51.903382006, 52.123556514, 52.328826702, 52.487319222,
52.696353867, 52.911456714, 53.185203421, 53.386873343, 53.545522515,
53.712945549, 53.91457674, 54.106682703, 54.25693709, 54.38890716,
54.590516004, 54.809708722, 54.937697825, 55.156648744, 55.405741033,
55.691990692, 55.856788013, 56.033952988, 56.179614644, 56.423014921,
56.649110296, 56.796119707, 57.02603986, 57.248791508, 57.436175759,
57.721114677, 57.996637493, 58.233471666, 58.374964422, 58.523273635,
58.702811404, 58.860729466, 59.182038038, 59.382771074, 59.580384417,
59.771838792, 59.948069013, 60.098335768, 60.294237534, 60.458415791,
60.647062942, 60.811086094, 60.974473075, 61.19175168, 61.394393375,
61.602029385, 61.77196373, 61.974715774, 62.189746587, 62.405119423,
62.648591693, 62.846388486, 62.995747513, 63.229361704, 63.394877887,
63.551556128, 63.786425153, 64.015790052, 64.286496448, 64.538849881,
64.683318973, 64.786162871, 64.971834399, 65.157983104, 65.367461552,
65.514241554, 65.708323939, 65.823609966, 66.092103134, 66.19795172,
66.440758602, 66.601566747, 66.93669226, 67.099946805, 67.273351165,
67.501869005, 67.68456076, 67.845817523, 68.127284918, 68.296983179,
68.444393797, 68.721363514, 68.935150755, 69.091932876, 69.391845032,
69.625883646, 69.862222902, 70.061045559, 70.258922589, 70.682206693,
70.972035962, 71.23497067, 71.468214184, 71.593079103, 71.760056246,
71.870314024, 71.976940497, 72.12897241, 72.418103886, 72.58011066,
72.812942854, 73.242885397, 73.694689057, 74.020807682, 74.243411704,
74.431780659, 74.742091381, 74.87380285, 75.264784775, 75.329518769,
75.603299298, 75.796405759, 76.028932357, 76.496469318, 76.669027056,
77.046998272, 77.289822068, 77.521829352, 77.79532109, 77.93191839,
78.293177117, 78.482247118, 78.931621122, 79.085183366, 79.30430046,
79.564928235, 79.970190875, 80.292773044, 80.70935374, 80.894917376,
80.992732855, 81.142954556, 81.341075829, 81.665616295, 81.739644264,
82.014387968, 82.406429306, 82.606499108, 82.69401089, 83.194140565,
83.363863663, 83.763035651, 84.036021322, 84.573069426, 84.804418825,
85.206461797, 85.347888477, 85.519732285, 85.695328579, 86.322893153,
86.754714386, 87.221275818, 87.37577449, 87.604192887, 87.969769716,
88.066768059, 88.241059266, 88.419519505, 88.599986167, 88.881373443,
89.203839874, 89.370277352, 89.550356681, 89.635580163, 89.893652918,
90.202337805, 90.670814166, 90.804583082, 90.951729226, 91.277238472,
91.569690866, 91.677524709, 91.847404312, 92.039090724, 92.180680152,
92.307414317, 92.583843435, 93.005981837, 93.700550728, 94.456859936,
94.581693803, 95.102880208, 95.562090237, 95.609385944, 95.813106521,
96.155005253, 96.70417523, 97.065745103, 97.293777524, 97.564486136,
97.788876371, 98.029374153, 98.210728844, 98.36566515, 98.513544133,
98.814789963, 99.318378249, 99.502890023, 100.157649313, 100.208979339,
100.544209853, 101.314563052, 101.846579642, 101.962494859, 102.252103634,
102.635716389, 102.794067674, 103.224928888, 103.472835681, 103.966524855,
104.366526708, 104.725762659, 104.842643487, 104.867126103, 105.190874976,
105.244888414, 105.387687089, 105.49526082, 105.58070767, 106.060291913,
106.447970581, 106.966834137, 107.210577625, 107.265886291, 108.234541543,
108.81521351, 109.03096719, 109.18432753, 109.521676321, 109.608622847,
109.785218701, 109.95608561, 110.086046898, 110.70640086, 110.95032831,
111.124088474, 111.549262037, 111.57264881, 112.328399817, 112.907398037,
113.776148019, 113.952913642, 114.640236529, 115.302287876, 115.477509226,
116.202323228, 117.366893575, 117.425215005, 117.602484375, 117.767606988,
118.054945041, 118.285974691, 118.363912982, 118.625795695, 118.741753297,
119.29791533, 119.533339446, 120.026177301, 120.256734263, 120.490747476,
120.640560254, 120.868682469, 122.193519437, 122.452736993, 122.516255501,
122.759462284, 123.56389322, 123.880086358, 123.929853593, 124.859012097,
125.264156223, 125.757184848, 126.04863531, 126.368628376, 126.60945319,
127.028283541, 127.028687015), "200" = c(1.8636e-05, 1.9213e-05,
1.9232e-05, 1.9704e-05, 1.9718e-05, 1.9897e-05, 2.0203e-05, 2.0472e-05,
2.0805e-05, 2.098e-05, 2.1046e-05, 2.1483e-05, 2.1744e-05, 2.1922e-05,
2.1938e-05, 2.2356e-05, 2.2914e-05, 2.3012e-05, 2.3626e-05, 2.3852e-05,
2.4254e-05, 2.4464e-05, 2.5054e-05, 2.521e-05, 2.5568e-05, 2.5723e-05,
2.6082e-05, 2.6093e-05, 2.6527e-05, 2.7124e-05, 2.7411e-05, 2.7874e-05,
2.7991e-05, 2.8156e-05, 2.8418e-05, 2.8572e-05, 2.8666e-05, 2.9334e-05,
2.9429e-05, 2.9583e-05, 2.97e-05, 3.013e-05, 3.095e-05, 3.1271e-05,
3.1302e-05, 3.1597e-05, 3.1915e-05, 3.2588e-05, 3.314e-05, 3.3365e-05,
3.3942e-05, 3.467e-05, 3.4986e-05, 3.5262e-05, 3.5726e-05, 3.6029e-05,
3.6043e-05, 3.6148e-05, 3.6611e-05, 3.6937e-05, 3.7137e-05, 3.7571e-05,
3.7733e-05, 3.8403e-05, 3.911e-05, 3.9502e-05, 4.0587e-05, 4.0879e-05,
4.176e-05, 4.2466e-05, 4.2742e-05, 4.3117e-05, 4.3664e-05, 4.3904e-05,
4.4467e-05, 4.4697e-05, 4.5742e-05, 4.636e-05, 4.6932e-05, 4.8466e-05,
4.9113e-05, 4.9927e-05, 5.0533e-05, 5.1645e-05, 5.2242e-05, 5.2442e-05,
5.3204e-05, 5.4129e-05, 5.4849e-05, 5.5246e-05, 5.5854e-05, 5.6111e-05,
5.6312e-05, 5.6722e-05, 5.737e-05, 5.8837e-05, 5.9687e-05, 6.0363e-05,
6.1074e-05, 6.1271e-05, 6.1636e-05, 6.2422e-05, 6.3414e-05, 6.3665e-05,
6.4261e-05, 6.4851e-05, 6.65e-05, 6.6869e-05, 6.7877e-05, 6.8643e-05,
6.9486e-05, 7.0551e-05, 7.1543e-05, 7.2023e-05, 7.325e-05, 7.4106e-05,
7.4423e-05, 7.6256e-05, 7.6695e-05, 7.7144e-05, 7.7513e-05, 7.8922e-05,
7.9314e-05, 8.1084e-05, 8.1933e-05, 8.2978e-05, 8.424e-05, 8.4965e-05,
8.5519e-05, 8.6465e-05, 8.7058e-05, 8.8135e-05, 8.8729e-05, 9.0135e-05,
9.1051e-05, 9.2841e-05, 9.3547e-05, 9.3939e-05, 9.51e-05, 9.6759e-05,
9.8934e-05, 0.000100323, 0.000101066, 0.000101908, 0.000103578,
0.000104666, 0.000107027, 0.000108159, 0.000109576, 0.000110451,
0.000111113, 0.000112635, 0.000113699, 0.000114645, 0.000115044,
0.000116234, 0.00011674, 0.000117748, 0.000119689, 0.000121057,
0.000122354, 0.000123621, 0.000124916, 0.000126427, 0.000128402,
0.000130348, 0.000131843, 0.000133781, 0.000135121, 0.000135967,
0.000137393, 0.000138675, 0.00014017, 0.000143162, 0.00014471,
0.000147298, 0.000149209, 0.000150672, 0.000151543, 0.000153535,
0.000154805, 0.000156719, 0.000158916, 0.000161186, 0.000162251,
0.000163802, 0.000165148, 0.000166639, 0.000168844, 0.000170789,
0.000174667, 0.000176802, 0.000178001, 0.000180665, 0.000182909,
0.000184865, 0.000186243, 0.000189682, 0.000191329, 0.000193209,
0.000195273, 0.000200568, 0.000202424, 0.00020403, 0.00020641,
0.000209406, 0.000210857, 0.000213229, 0.000215127, 0.000216838,
0.000219794, 0.000222775, 0.000224718, 0.000226465, 0.000227711,
0.00022918, 0.000231863, 0.000234754, 0.000236938, 0.000240967,
0.000243657, 0.000245168, 0.00024794, 0.000250607, 0.000251986,
0.000255835, 0.00025843, 0.000260384, 0.000262374, 0.00026526,
0.000268084, 0.000270915, 0.000273269, 0.000276266, 0.000278931,
0.000281352, 0.000285889, 0.000288106, 0.000291717, 0.000295053,
0.000298494, 0.000302464, 0.000308077, 0.000311148, 0.000314344,
0.000319998, 0.000324207, 0.000329727, 0.000332571, 0.000335357,
0.00033907, 0.000343501, 0.000347506, 0.000350806, 0.000355224,
0.000358465, 0.0003634, 0.000368383, 0.000373065, 0.000376463,
0.000381795, 0.000386381, 0.000390592, 0.000394659, 0.000399685,
0.000405705, 0.000410026, 0.0004156, 0.0004202, 0.000423147,
0.000428265, 0.000432606, 0.000438762, 0.000442666, 0.000447524,
0.000451722, 0.000459106, 0.000464626, 0.000468742, 0.000474915,
0.000479997, 0.000488534, 0.000494311, 0.000499717, 0.00050414,
0.000511822, 0.000518393, 0.000524082, 0.000529974, 0.000534372,
0.000541378, 0.000548251, 0.00055505, 0.000560229, 0.000566375,
0.000572599, 0.000579543, 0.000587117, 0.000593716, 0.000602215,
0.000608265, 0.000615558, 0.000623587, 0.000631318, 0.000637739,
0.000646247, 0.000654436, 0.000661276, 0.000668727, 0.000675739,
0.000684243, 0.000691679, 0.000700963, 0.000710871, 0.000717996,
0.000726604, 0.000736848, 0.000744711, 0.000754958, 0.000764012,
0.000772257, 0.000779154, 0.000787516, 0.000798813, 0.000808192,
0.000816631, 0.000824918, 0.000836883, 0.00084813, 0.000855657,
0.000868043, 0.000877798, 0.000886432, 0.000895475, 0.000908221,
0.000918255, 0.000929525, 0.000939035, 0.000948888, 0.000958652,
0.000968359, 0.000969876, 0.000980905, 0.000991868, 0.001002077,
0.001013777, 0.001023903, 0.00103429, 0.001046722, 0.00105786,
0.001069415, 0.001082703, 0.001096922, 0.001107378, 0.001122616,
0.001137419, 0.001149493, 0.001160372, 0.00117501, 0.001188356,
0.00120182, 0.001218298, 0.001233003, 0.001244216, 0.001259633,
0.001274956, 0.001293711, 0.001307482, 0.001321872, 0.001336854,
0.001355619, 0.001369931, 0.001385321, 0.001403443, 0.001419927,
0.001436244, 0.001454611, 0.001470302, 0.001487046, 0.001504962,
0.001522199, 0.001538988, 0.001559332, 0.001575863, 0.001591542,
0.00161372, 0.001629682, 0.001645857, 0.00166541, 0.001687192,
0.001706494, 0.001726837, 0.001747109, 0.001767527, 0.00179032,
0.001811993, 0.001830563, 0.001850797, 0.001873823, 0.001894346,
0.00191731, 0.001940305, 0.001963323, 0.0019814, 0.002002824,
0.002024741, 0.002048417, 0.00206877, 0.002093077, 0.002117632,
0.00214743, 0.002169793, 0.002195027, 0.002221812, 0.002246458,
0.002270474, 0.0022945, 0.00232506, 0.002354154, 0.002383784,
0.002413561, 0.00244441, 0.002470785, 0.002500162, 0.002526505,
0.002556812, 0.002590508, 0.002618943, 0.002650213, 0.00267705,
0.002706798, 0.002741068, 0.00277123, 0.002806126, 0.002837426,
0.002874071, 0.002904996, 0.002940871, 0.002972902, 0.003008194,
0.003041873, 0.003076286, 0.003106361, 0.003144506, 0.003179303,
0.003214812, 0.003250668, 0.003285169, 0.00332233, 0.003362555,
0.003397766, 0.003440486, 0.003484029, 0.003528346, 0.003567445,
0.00361041, 0.003656506, 0.003696836, 0.003741734, 0.003790018,
0.003835233, 0.003880564, 0.003929148, 0.00397557, 0.004014799,
0.004064482, 0.004107896, 0.004160721, 0.004211434, 0.004255754,
0.004300305, 0.004348065, 0.004397154, 0.004446507, 0.00449987,
0.00455317, 0.004605118, 0.004656998, 0.004708154, 0.004763113,
0.004818406, 0.00487695, 0.004936955, 0.004990665, 0.005050818,
0.005111181, 0.005165068, 0.005225665, 0.005286184, 0.005351968,
0.005412143, 0.005477135, 0.005535261, 0.005602001, 0.005662953,
0.005734093, 0.005798341, 0.005862579, 0.005928264, 0.005999368,
0.006066714, 0.006140033, 0.006211306, 0.006292629, 0.006372148,
0.006450939, 0.006533447, 0.006609052, 0.006684714, 0.006756851,
0.006838988, 0.006923507, 0.007007105, 0.007089463, 0.007165184,
0.007247284, 0.007328469, 0.007407872, 0.007492555, 0.007580202,
0.007665227, 0.007754288, 0.007845082, 0.007937065, 0.008032675,
0.008130521, 0.00823234, 0.008330306, 0.008422014, 0.008524285,
0.008624331, 0.008715513, 0.008819424, 0.008924958, 0.009021932,
0.009117123, 0.009214659, 0.009316627, 0.009431332, 0.009544219,
0.009651135, 0.009735376, 0.00975775, 0.009876774, 0.009995504,
0.010105024, 0.010220441, 0.010336726, 0.010457245, 0.010574215,
0.010699849, 0.010820631, 0.01094908, 0.011076971, 0.011199548,
0.011319421, 0.011451062, 0.011590155, 0.011724306, 0.01185725,
0.011996628, 0.012136317, 0.012280874, 0.01242082, 0.012570277,
0.012712658, 0.012861217, 0.013018819, 0.013164963, 0.013310636,
0.013467251, 0.013618424, 0.013771455, 0.01393094, 0.014090198,
0.014252515, 0.014423584, 0.014600756, 0.014769463, 0.014942907,
0.015112242, 0.01529081, 0.015468011, 0.015648196, 0.015828806,
0.016014868, 0.01620707, 0.016396163, 0.016583302, 0.016775035,
0.016978805, 0.017166717, 0.017373863, 0.017576629, 0.017778667,
0.017974725, 0.018182786, 0.018389635, 0.018597012, 0.018811853,
0.019035356, 0.019253256, 0.019462046, 0.019692114, 0.019901372,
0.02012977, 0.020362329, 0.020597038, 0.020828249, 0.021070732,
0.021325507, 0.021571398, 0.021827307, 0.022079104, 0.022333397,
0.022596667, 0.022860254, 0.023129639, 0.023389831, 0.023658388,
0.023928536, 0.024206127, 0.024483943, 0.024768648, 0.025062739,
0.025353605, 0.025653183, 0.025945174, 0.026241078, 0.026540924,
0.026847516, 0.027147472, 0.0274531, 0.027779204, 0.028115318,
0.028455781, 0.028794059, 0.029136912, 0.029491239, 0.029825249,
0.030175704, 0.03052925, 0.030878543, 0.031231839, 0.031595759,
0.031958276, 0.032332646, 0.032719573, 0.033098374, 0.033488525,
0.033869294, 0.034267266, 0.034682705, 0.035066825, 0.035473487,
0.035866309, 0.036286152, 0.036714756, 0.037147876, 0.037575004,
0.038015222, 0.038469257, 0.038917346, 0.039364709, 0.039813758,
0.04027474, 0.040737438, 0.041233876, 0.041707807, 0.042195368,
0.042670456, 0.043177195, 0.043685177, 0.044207404, 0.044717508,
0.045238255, 0.045746829, 0.046264458, 0.046790664, 0.047341751,
0.047912274, 0.048465871, 0.049029156, 0.049604469, 0.050181274,
0.050765026, 0.051340778, 0.051938478, 0.052542991, 0.053157135,
0.053795025, 0.054433683, 0.055067955, 0.055718836, 0.056358214,
0.057016808, 0.057684392, 0.05836487, 0.059042351, 0.059748154,
0.060449234, 0.06117771, 0.061879246, 0.062605492, 0.063337822,
0.064070053, 0.064823467, 0.06559567, 0.066365071, 0.067137861,
0.067923133, 0.068704759, 0.069506434, 0.070305462, 0.071118011,
0.071941437, 0.07279858, 0.073659977, 0.074508644, 0.075400693,
0.076288275, 0.07718141, 0.078094654, 0.079014458, 0.079949233,
0.080866774, 0.081852202, 0.082816911, 0.083773172, 0.084734998,
0.085727658, 0.08673367, 0.087730857, 0.088775486, 0.089852987,
0.090936911, 0.092016902, 0.093103408, 0.094203391, 0.095330008,
0.096475502, 0.097620148, 0.098544536, 0.09877277, 0.099974701,
0.101146114, 0.102334592, 0.103546736, 0.104747265, 0.105962606,
0.10722774, 0.10848837, 0.109768033, 0.111072075, 0.112399257,
0.113731514, 0.115093055, 0.116487472, 0.117877632, 0.119273948,
0.120679424, 0.122090238, 0.123559805, 0.125021594, 0.126530132,
0.128037995, 0.129553655, 0.131105084, 0.13265533, 0.134232255,
0.135822368, 0.13746299, 0.13906442, 0.140731961, 0.142369707,
0.144060554, 0.145782782, 0.147490782, 0.1492541, 0.151064506,
0.152886983, 0.15468909, 0.156524161, 0.158435529, 0.160345667,
0.162295393, 0.16423719, 0.16619505, 0.168188289, 0.170187792,
0.172193173, 0.174262726, 0.176356117, 0.178486941, 0.180659794,
0.182819488, 0.185007467, 0.187248042, 0.189477289, 0.191767672,
0.194053318, 0.196376774, 0.198736661, 0.201167338, 0.20354455,
0.206009213, 0.208522702, 0.211062441, 0.213631798, 0.216234354,
0.218862433, 0.221539933, 0.224232656, 0.226971498, 0.229753949,
0.232492793, 0.235289289, 0.238156997, 0.24105281, 0.243986055,
0.246974154, 0.250049192, 0.253085231, 0.256221563, 0.259334778,
0.262490539, 0.265632696, 0.268873678, 0.272160531, 0.275503682,
0.278843598, 0.282259332, 0.285693675, 0.289187029, 0.292714487,
0.296305652, 0.299952914, 0.303658532, 0.307391111, 0.308206834,
0.31122571, 0.315095416, 0.318973134, 0.322901015, 0.32690405,
0.330938616, 0.335083367, 0.339194457, 0.343408424, 0.347659013,
0.352022118, 0.356386807, 0.360861146, 0.365417873, 0.370046464,
0.37463788, 0.379265913, 0.384019504, 0.388766201, 0.393668419,
0.398636474, 0.403678432, 0.40878006, 0.414001851, 0.41924891,
0.420326516, 0.424533958, 0.42993483, 0.435350499, 0.440918189,
0.446522727, 0.452240483, 0.458080134, 0.46398987, 0.469922433,
0.47588112, 0.481963183, 0.488131808, 0.494392097, 0.500788603,
0.507188454, 0.513769963, 0.520421191, 0.527122386, 0.534023597,
0.541034914, 0.548035689, 0.555201431, 0.562414822, 0.569836122,
0.577264237, 0.584880001, 0.592642548, 0.600475956, 0.608397077,
0.616534013, 0.624790313, 0.633088559, 0.641596996, 0.650144714,
0.658783353, 0.66251263, 0.667637858, 0.676625138, 0.685816804,
0.695123929, 0.704523699, 0.714096283, 0.723783632, 0.733746002,
0.743802063, 0.753966755, 0.764431645, 0.775098038, 0.785866544,
0.796745555, 0.807939091, 0.819226055, 0.830656034, 0.842385865,
0.854257456, 0.866476839, 0.878855747, 0.891403751, 0.904209932,
0.917256039, 0.930440928, 0.935945673, 0.943891485, 0.957657637,
0.971642146, 0.98605472, 1.000503495, 1.015344212, 1.030429598,
1.0460056, 1.06184131, 1.077933795, 1.094154527, 1.110759015,
1.127751306, 1.145098222, 1.162828483, 1.18099477, 1.199547256,
1.218569738, 1.237851369, 1.253591767, 1.289393896, 1.309311412,
1.329189177, 1.349032796, 1.368884357, 1.388887566, 1.408731625,
1.428480398, 1.448384258, 1.46842933, 1.488017014, 1.507793418,
1.527512168, 1.547282218, 1.567099212, 1.587009541, 1.606912381,
1.626638258, 1.63820754, 1.646476437, 1.666432619, 1.686228766,
1.705852287, 1.72542747, 1.745229761, 1.765113468, 1.785032723,
1.805036447, 1.824934335, 1.844897511, 1.864651179, 1.884632423,
1.904578033, 1.924423167, 1.944334499, 1.96410882, 1.983988622,
2.003809497, 2.023771154, 2.043520871, 2.063537681, 2.083348738,
2.103329714, 2.123563522, 2.135021231, 2.143375966, 2.163583266,
2.183598231, 2.203998194, 2.224098434, 2.244329128, 2.264307784,
2.284578176, 2.304631548, 2.324843856, 2.345145621, 2.365563349,
2.385825244, 2.406239904, 2.42639859, 2.446845911, 2.467406762,
2.487657829, 2.508205331, 2.528769177, 2.549501872, 2.569995224,
2.590866391, 2.611622361, 2.632220492, 2.652951498, 2.673672193,
2.694450439, 2.715474932, 2.736292066, 2.757209347, 2.778203477,
2.799128209, 2.820131414, 2.841420122, 2.858191232, 2.862517581,
2.883439903, 2.904775995, 2.926259373, 2.947720524, 2.969187645,
2.990918215, 3.012366444, 3.03407187, 3.055734632, 3.07777233,
3.099787099, 3.121661799, 3.143808771, 3.166015535, 3.187770933,
3.210207448, 3.232684508, 3.255130518, 3.277754009, 3.300437264,
3.323042095, 3.345643735, 3.368215495, 3.390912706, 3.408644918,
3.413487735, 3.436364238, 3.459330576, 3.4825545, 3.505682068,
3.528763536, 3.552240843, 3.57556027, 3.599196263, 3.622687853,
3.646368218, 3.670081175, 3.694197306, 3.718612262, 3.742765816,
3.766807402, 3.79128352, 3.815795227, 3.840305005, 3.864943409,
3.889446712, 3.914472269, 3.939219914, 3.964161489, 3.989298116,
4.014521592, 4.039470887, 4.064723828, 4.089959972, 4.115418618,
4.141434841, 4.167151009, 4.193207622, 4.219410286, 4.245265081,
4.271795277, 4.298598364, 4.32530948, 4.351700717, 4.379030163,
4.405958249, 4.433045934, 4.460143199, 4.487371224, 4.51528676,
4.542988218, 4.570785451, 4.599190428, 4.627139115, 4.655396757,
4.684234218, 4.713116076, 4.742119985, 4.770804327, 4.79998944,
4.829877202, 4.859607441, 4.888888226, 4.917583175, 4.947322475,
4.977811394, 5.008210082, 5.039025664, 5.069635972, 5.100385473,
5.13117422, 5.162507246, 5.19333023, 5.225527106, 5.257648592,
5.289813999, 5.321543708, 5.354046525, 5.386416659, 5.418201481,
5.451132407, 5.48445374, 5.517873023, 5.550812938, 5.583797745,
5.616979633, 5.650291054, 5.684400347, 5.719065312, 5.753362559,
5.78741594, 5.822128081, 5.856606595, 5.891779031, 5.927418415,
5.96305232, 5.998319556, 6.033867752, 6.070084572, 6.105773662,
6.14177495, 6.149254512, 6.178291672, 6.215415329, 6.251982207,
6.288593285, 6.325864839, 6.362690101, 6.401043976, 6.437863347,
6.475981541, 6.514407383, 6.552343737, 6.59017869, 6.629102988,
6.667254556, 6.706632538, 6.746340655, 6.786191206, 6.825867751,
6.865708795, 6.904803794, 6.944873512, 6.985518923, 7.027281028,
7.069131984, 7.109400003, 7.152286573, 7.194059564, 7.236742202,
7.277475301, 7.319381877, 7.360405657, 7.40310346, 7.445209538,
7.487772918, 7.530912999, 7.572945386, 7.616622185, 7.660166598,
7.703368781, 7.74647237, 7.790774999, 7.835731806, 7.879938237,
7.92470079, 7.969013666, 8.013756675, 8.059360897, 8.104917816,
8.150846108, 8.196343666, 8.241621845, 8.288283643, 8.333996286,
8.380311477, 8.426730123, 8.472901068, 8.518539603, 8.565958492,
8.614206635, 8.660192526, 8.706033165, 8.754252772, 8.803187267,
8.851401208, 8.898870184, 8.947897225, 8.998337365, 9.04519606,
9.095058268, 9.143144768, 9.192069011, 9.241580184, 9.289824885,
9.340317658, 9.390496657, 9.440605284, 9.490009067, 9.540083503,
9.590499956, 9.642502527, 9.692597631, 9.743444614, 9.794333812,
9.846529464, 9.898368195, 9.950277083, 10.003603213, 10.056727367,
10.109774873, 10.162274513, 10.214743321, 10.267143205, 10.321199455,
10.372026018, 10.423638314, 10.475611195, 10.529270362, 10.582930612,
10.636574662, 10.69157518, 10.748335265, 10.802025038, 10.856691752,
10.910330597, 10.964616495, 11.018809816, 11.075318005, 11.132653705,
11.189787161, 11.245719723, 11.303242359, 11.360874884, 11.416695867,
11.475322314, 11.530537083, 11.589879303, 11.6486315, 11.70799926,
11.765705109, 11.823186067, 11.881532265, 11.9391904, 12.001853875,
12.061135007, 12.121226697, 12.183225423, 12.243826415, 12.304047582,
12.36276449, 12.423480804, 12.484094388, 12.54520049, 12.607006629,
12.666487374, 12.725708208, 12.790119381, 12.850215386, 12.912369859,
12.975951644, 13.041596183, 13.103659273, 13.167344342, 13.230070095,
13.295634035, 13.356845142, 13.420721062, 13.485399103, 13.54741414,
13.612038789, 13.674163608, 13.736280043, 13.801357479, 13.868016571,
13.93100509, 13.996015781, 14.061335817, 14.125711938, 14.189719488,
14.25885632, 14.323394663, 14.39047329, 14.456146254, 14.522720821,
14.585799932, 14.654761129, 14.723475261, 14.789377934, 14.858826742,
14.923157221, 14.992101384, 15.063935046, 15.130997476, 15.200739437,
15.261799144, 15.330967571, 15.402903799, 15.475465667, 15.548509979,
15.62084701, 15.692771549, 15.758371428, 15.830227948, 15.899759074,
15.971222279, 16.041004395, 16.110288062, 16.180594851, 16.25100964,
16.324014171, 16.39829739, 16.470660567, 16.546425371, 16.626053565,
16.695859795, 16.772377753, 16.847603243, 16.9210289, 16.994400404,
17.06963003, 17.141656842, 17.155589687, 17.21225867, 17.282142193,
17.359755179, 17.437289269, 17.512600173, 17.587649703, 17.662218623,
17.737055743, 17.812720208, 17.889422943, 17.961420513, 18.032983416,
18.111247756, 18.185473714, 18.264609389, 18.337447102, 18.418432626,
18.499648524, 18.578471892, 18.65259713, 18.741028321, 18.826756339,
18.905333585, 18.987153685, 19.065917473, 19.149289041, 19.225736716,
19.312945598, 19.393589446, 19.469206134, 19.5496455, 19.637021214,
19.719066148, 19.806847583, 19.887532241, 19.973476221, 20.05288714,
20.129488566, 20.209628824, 20.286657163, 20.370212855, 20.453879832,
20.541101632, 20.632929945, 20.717598409, 20.798256856, 20.891831678,
20.971717902, 21.053063618, 21.144092852, 21.229180997, 21.313130788,
21.414109502, 21.504835221, 21.588120065, 21.673305646, 21.771155496,
21.862987837, 21.940502544, 22.033669563, 22.122487537, 22.222803675,
22.31104445, 22.408895629, 22.502984508, 22.591089408, 22.679079574,
22.771966713, 22.85936858, 22.952630765, 23.046614028, 23.146974768,
23.229052147, 23.324979338, 23.414929269, 23.510793344, 23.602235133,
23.697351516, 23.79332145, 23.896597706, 23.997165269, 24.097319692,
24.180200789, 24.273271144, 24.366115217, 24.457460229, 24.56592532,
24.665170195, 24.76275921, 24.854563523, 24.957252207, 25.057575737,
25.167911447, 25.271930335, 25.368404335, 25.473239876, 25.579528749,
25.672276661, 25.772623118, 25.868261876, 25.95990979, 26.059359404,
26.164759753, 26.275555902, 26.384427982, 26.49105829, 26.609052357,
26.735268823, 26.850479496, 26.949089394, 27.05931773, 27.157399328,
27.258962763, 27.363648688, 27.471450244, 27.572731335, 27.680224161,
27.783500682, 27.887351305, 27.987602589, 28.09331525, 28.205556821,
28.292382599, 28.402793929, 28.497462221, 28.598007395, 28.731706787,
28.839935938, 28.935079205, 29.043169903, 29.146429756, 29.25309353,
29.379088539, 29.487831399, 29.594728051, 29.70773233, 29.8158949,
29.935178851, 30.046920386, 30.157467568, 30.279122439, 30.386050134,
30.484624767, 30.589612035, 30.690337731, 30.797188595, 30.929234075,
31.032636876, 31.153942812, 31.287774567, 31.40486342, 31.509695169,
31.608282487, 31.730573083, 31.852519508, 31.999073219, 32.128745294,
32.243312463, 32.382975708, 32.532494641, 32.634858679, 32.767329496,
32.886119733, 32.986530541, 33.093921631, 33.229429122, 33.360370598,
33.493755624, 33.605288061, 33.718558838, 33.849187836, 33.957493423,
34.090959636, 34.224662894, 34.356917412, 34.462660061, 34.600421623,
34.7450766, 34.852543641, 34.975607386, 35.084964176, 35.196939452,
35.305394428, 35.417459648, 35.545870008, 35.68086729, 35.816526255,
35.974467826, 36.078503639, 36.209726303, 36.325125044, 36.457407402,
36.611542519, 36.743595245, 36.875386738, 37.005274097, 37.119312105,
37.218990846, 37.365904004, 37.50358233, 37.524144267, 37.632658614,
37.762809606, 37.869729817, 37.982158445, 38.141085194, 38.27200866,
38.412874682, 38.529457077, 38.654878478, 38.787395146, 38.942997952,
39.07469668, 39.230090378, 39.355640194, 39.463010075, 39.616785313,
39.739932487, 39.842464819, 39.959872887, 40.069801088, 40.169840708,
40.33944118, 40.474904955, 40.617587148, 40.760370885, 40.893935474,
41.040671638, 41.185758214, 41.298776906, 41.424411518, 41.540129743,
41.705433528, 41.831327967, 42.002211472, 42.168476371, 42.292618007,
42.424554546, 42.57667584, 42.725967968, 42.92266187, 43.095985035,
43.225155596, 43.346838732, 43.517270043, 43.733776982, 43.872754737,
44.071888565, 44.226277488, 44.360688154, 44.515174581, 44.673961747,
44.826340502, 44.966120158, 45.17537499, 45.307591814, 45.479782758,
45.655395003, 45.819505943, 45.981001163, 46.105982578, 46.219803927,
46.331451641, 46.480642867, 46.663820984, 46.808454727, 46.977474269,
47.110616719, 47.26647254, 47.395713148, 47.558866271, 47.765825909,
47.914882468, 48.092224758, 48.212204438, 48.360635744, 48.495899393,
48.614907839, 48.771585189, 48.976620513, 49.137120355, 49.275983232,
49.482698411, 49.624851745, 49.780760907, 49.91009775, 50.042681915,
50.278311881, 50.462763906, 50.623029135, 50.760414181, 50.977033813,
51.121993452, 51.300104057, 51.561956784, 51.685807535, 51.833533419,
51.973717108, 52.171596835, 52.333853059, 52.438546667, 52.633699299,
52.798214885, 52.989942078, 53.172671246, 53.321126702, 53.459467445,
53.651958598, 53.784009567, 53.990813364, 54.142473607, 54.334508736,
54.531833715, 54.673490445, 54.800390385, 54.951286312, 55.113367153,
55.238990005, 55.405970965, 55.600955548, 55.777946678, 55.914374748,
56.140852473, 56.349635266, 56.660403469, 56.81945408, 57.027815937,
57.249297824, 57.411856101, 57.506742524, 57.697167529, 58.026102709,
58.222700068, 58.314448725, 58.582571842, 58.729533363, 58.879258943,
59.06859432, 59.210758228, 59.386701089, 59.599951257, 59.796965484,
59.9550775, 60.120246791, 60.322031073, 60.482747917, 60.688406733,
60.842738391, 61.085487585, 61.231641686, 61.461300402, 61.697695823,
61.944979854, 62.087499379, 62.256141638, 62.607353093, 62.846803039,
63.09443414, 63.312983284, 63.536142351, 63.822343526, 64.029650434,
64.248192204, 64.458039626, 64.653171251, 64.816313447, 65.044893395,
65.305329132, 65.449282034, 65.671633671, 65.826384664, 65.954526357,
66.100148931, 66.306695366, 66.678261944, 66.976064103, 67.177442874,
67.312804058, 67.573976282, 67.891638189, 68.105114274, 68.317694984,
68.598623332, 68.774576331, 68.989480998, 69.159704762, 69.494810479,
69.786481431, 69.893570421, 70.157425708, 70.358771239, 70.489290863,
70.721003495, 71.018416385, 71.276621836, 71.585232626, 71.871170288,
72.260588262, 72.66393895, 72.983922891, 73.2650231, 73.535025807,
73.736707596, 73.815833406, 73.898991442, 74.11235073, 74.388104162,
74.626280882, 74.801524961, 75.01194028, 75.417777408, 75.543836306,
75.620534248, 75.843681015, 75.998204068, 76.296730263, 76.612598354,
76.858948164, 77.341389189, 77.526995161, 77.802480388, 78.104362542,
78.413656253, 78.668838755, 78.964684047, 79.03414987, 79.269147582,
79.663117037, 80.172334824, 80.450658603, 80.659122593, 80.957901193,
81.102643835, 81.264957316, 81.367486546, 81.562811528, 82.062840607,
82.288446942, 82.380158453, 82.625199318, 82.796351469, 82.870921411,
83.115666528, 83.479875449, 83.604987442, 83.810624111, 84.361931671,
84.509860984, 84.826035951, 85.017909509, 85.298648745, 85.545390294,
85.815839518, 86.233354514, 86.405859782, 86.597944978, 86.927384156,
87.166009655, 87.498677227, 87.610388951, 88.022562668, 88.425392429,
88.519432712, 88.642249618, 89.145419845, 89.798235781, 90.068771988,
90.20839128, 90.577801247, 90.874263146, 91.206690598, 91.32782688,
91.478583617, 91.614531256, 91.70393229, 92.541810545, 92.635386104,
92.872202025, 93.079860078, 93.166258289, 93.317619451, 93.542463605,
93.770281442, 94.007954449, 94.380468999, 94.809084445, 94.927246199,
95.074374462, 95.15900041, 95.261176296, 95.338496935, 95.598058832,
96.15664072, 96.34633546, 96.401267232, 96.93744457, 97.127068064,
97.441467943, 97.618983636, 97.827069307, 97.937286048, 97.977902264,
98.29424893, 98.443088233, 99.073355646, 99.420148146, 99.55166846,
100.213840398, 100.344890539, 100.590990234, 101.550268895, 101.553877109,
101.873257328, 102.226779312, 102.227853579, 102.481848587, 102.67932765,
102.969760257, 103.248680503, 103.540700979, 103.545883043, 103.760473408,
103.939613938, 104.026635264, 104.642273705, 105.050434126, 105.586357375,
105.804818035, 105.97494225, 106.138004121, 107.370832448, 107.767710322,
107.792296213, 108.774176058, 109.24672626, 109.309625236, 109.682944576,
109.744572439, 109.987799235, 110.04558376, 110.070904945), "300" = c(1.9087e-05,
1.9332e-05, 1.9338e-05, 1.9612e-05, 1.9839e-05, 1.9909e-05, 2.0128e-05,
2.0205e-05, 2.141e-05, 2.1527e-05, 2.1544e-05, 2.2177e-05, 2.2204e-05,
2.2232e-05, 2.2472e-05, 2.2657e-05, 2.3425e-05, 2.3781e-05, 2.4379e-05,
2.4411e-05, 2.4604e-05, 2.4656e-05, 2.4852e-05, 2.5047e-05, 2.5295e-05,
2.5338e-05, 2.5819e-05, 2.6025e-05, 2.6626e-05, 2.6746e-05, 2.6796e-05,
2.7301e-05, 2.7409e-05, 2.7529e-05, 2.7754e-05, 2.7804e-05, 2.8049e-05,
2.8508e-05, 2.8775e-05, 2.8877e-05, 2.9206e-05, 2.9407e-05, 2.9604e-05,
2.9971e-05, 3.0335e-05, 3.1044e-05, 3.1491e-05, 3.1781e-05, 3.2345e-05,
3.2569e-05, 3.3025e-05, 3.3139e-05, 3.3661e-05, 3.3842e-05, 3.3912e-05,
3.4007e-05, 3.4379e-05, 3.4787e-05, 3.5415e-05, 3.6148e-05, 3.6427e-05,
3.6686e-05, 3.7053e-05, 3.7652e-05, 3.7913e-05, 3.8297e-05, 3.8788e-05,
3.9553e-05, 4.0323e-05, 4.0378e-05, 4.1899e-05, 4.2262e-05, 4.2491e-05,
4.2792e-05, 4.3085e-05, 4.4161e-05, 4.5194e-05, 4.5605e-05, 4.5763e-05,
4.6086e-05, 4.6286e-05, 4.7013e-05, 4.8155e-05, 4.8281e-05, 4.8788e-05,
4.9551e-05, 4.9859e-05, 5.0515e-05, 5.1611e-05, 5.3549e-05, 5.4194e-05,
5.4634e-05, 5.51e-05, 5.5864e-05, 5.7323e-05, 5.7931e-05, 5.9042e-05,
5.9709e-05, 6.0254e-05, 6.1166e-05, 6.137e-05, 6.2421e-05, 6.289e-05,
6.3792e-05, 6.5545e-05, 6.6118e-05, 6.6648e-05, 6.7274e-05, 6.7471e-05,
6.8264e-05, 6.9129e-05, 7.0703e-05, 7.1732e-05, 7.2992e-05, 7.3802e-05,
7.4297e-05, 7.4739e-05, 7.5832e-05, 7.6687e-05, 7.751e-05, 7.7993e-05,
7.8227e-05, 7.9343e-05, 8.1618e-05, 8.2815e-05, 8.3317e-05, 8.391e-05,
8.465e-05, 8.51e-05, 8.5835e-05, 8.7312e-05, 8.8639e-05, 8.923e-05,
9.0093e-05, 9.1327e-05, 9.2068e-05, 9.3069e-05, 9.4449e-05, 9.5319e-05,
9.6644e-05, 9.8258e-05, 0.000100619, 0.000101625, 0.000102775,
0.000103746, 0.000104664, 0.000105545, 0.000106226, 0.000108413,
0.000109258, 0.000110308, 0.000112019, 0.000113194, 0.000114922,
0.00011623, 0.000116974, 0.00011816, 0.00011883, 0.000120299,
0.000122613, 0.000124469, 0.000125279, 0.000126634, 0.000127945,
0.000129274, 0.000130058, 0.000131684, 0.000132749, 0.000135233,
0.000136519, 0.00013774, 0.000139689, 0.000140874, 0.000142057,
0.00014411, 0.000145628, 0.000147836, 0.000148915, 0.000151674,
0.0001534, 0.000155096, 0.000157337, 0.000158458, 0.000159474,
0.000161177, 0.000162411, 0.000164044, 0.000165096, 0.000166314,
0.000167594, 0.000168872, 0.000169814, 0.00017237, 0.000175016,
0.000176535, 0.000179147, 0.000181441, 0.000183986, 0.000186282,
0.000188917, 0.00019079, 0.000193156, 0.000194979, 0.000197373,
0.000199454, 0.000202034, 0.000204425, 0.000206484, 0.000207537,
0.000209895, 0.000212987, 0.000217102, 0.000220819, 0.000222272,
0.000225474, 0.000227448, 0.000228964, 0.000231192, 0.000234124,
0.000237233, 0.000240298, 0.000241873, 0.000246176, 0.000250192,
0.000253457, 0.00025689, 0.000259518, 0.000264291, 0.00026849,
0.000272755, 0.000275855, 0.000278937, 0.000282382, 0.000288152,
0.000291452, 0.000294065, 0.000297999, 0.000300084, 0.00030214,
0.000304723, 0.000310045, 0.000314278, 0.000317407, 0.000321455,
0.000325995, 0.000329806, 0.00033364, 0.000336149, 0.000338037,
0.000342425, 0.000346378, 0.000350411, 0.000355369, 0.000358284,
0.000362876, 0.000366631, 0.00036967, 0.000374428, 0.000379274,
0.00038336, 0.000388201, 0.000393538, 0.000399278, 0.000404938,
0.000408497, 0.000412514, 0.000418056, 0.000422606, 0.00042823,
0.000432608, 0.000437874, 0.000443213, 0.000446679, 0.000451743,
0.000455162, 0.000461611, 0.00046836, 0.000473414, 0.000477972,
0.000483147, 0.000488357, 0.000494907, 0.000501081, 0.000508039,
0.000514447, 0.000518705, 0.00052556, 0.000531495, 0.00053695,
0.000541634, 0.000546463, 0.000552729, 0.000558698, 0.00056645,
0.000571538, 0.000576786, 0.000583082, 0.000590113, 0.000597351,
0.000605398, 0.000612807, 0.000620299, 0.000627636, 0.000632587,
0.000638867, 0.000649246, 0.000656053, 0.000664808, 0.000671967,
0.00068091, 0.000687882, 0.000696226, 0.000704155, 0.000712414,
0.000719326, 0.000730654, 0.000738913, 0.000747223, 0.0007565,
0.00076516, 0.000775272, 0.000784426, 0.000795125, 0.000803627,
0.000813208, 0.000825752, 0.000835159, 0.000844141, 0.000854039,
0.000868711, 0.000878066, 0.000889747, 0.000899167, 0.000910703,
0.000922475, 0.000934059, 0.000945268, 0.000955175, 0.000965957,
0.000975633, 0.00098424, 0.000987034, 0.000996822, 0.001008061,
0.00102186, 0.001034384, 0.001048513, 0.001060112, 0.001072452,
0.001083517, 0.001098441, 0.001109753, 0.001124114, 0.001136094,
0.001147298, 0.001157864, 0.001171216, 0.001185279, 0.001199691,
0.001211711, 0.001227647, 0.001239501, 0.001251695, 0.001266602,
0.001282222, 0.00129646, 0.00130932, 0.001321207, 0.001336787,
0.00135033, 0.001365703, 0.001380564, 0.001397868, 0.001413164,
0.001430932, 0.001447326, 0.001462744, 0.001482443, 0.001503365,
0.001518496, 0.001536632, 0.001554337, 0.001569718, 0.001589168,
0.001607901, 0.001624456, 0.001642999, 0.001658663, 0.001680993,
0.001702483, 0.001722469, 0.001742523, 0.001761454, 0.001779252,
0.001801673, 0.001821305, 0.001839324, 0.001860134, 0.001882392,
0.001903433, 0.001926032, 0.001946337, 0.001972518, 0.001995879,
0.00202458, 0.002048945, 0.002072349, 0.002095042, 0.002121843,
0.002149502, 0.002173636, 0.002196294, 0.002223306, 0.002250253,
0.002275175, 0.002300392, 0.002326707, 0.002353856, 0.002378692,
0.002402059, 0.002428653, 0.002457244, 0.002482654, 0.002507338,
0.002535387, 0.002562352, 0.002590664, 0.002621356, 0.002652535,
0.002681807, 0.00271205, 0.002744047, 0.002776923, 0.002809024,
0.002839992, 0.002873651, 0.002907045, 0.002939849, 0.00297317,
0.003004815, 0.003041535, 0.003077671, 0.003110854, 0.003149724,
0.003191214, 0.003223273, 0.003264026, 0.003306453, 0.003344419,
0.0033848, 0.003426641, 0.003465652, 0.003502313, 0.003543123,
0.003587834, 0.003623553, 0.003668131, 0.003711475, 0.003759089,
0.0038025, 0.003847424, 0.003891292, 0.003939333, 0.003984853,
0.00403006, 0.004079343, 0.004126803, 0.00417572, 0.004227694,
0.004269541, 0.004316952, 0.004364783, 0.004415727, 0.004467046,
0.004519398, 0.004574272, 0.004625755, 0.00467834, 0.004731261,
0.004790355, 0.004844773, 0.004899672, 0.004954102, 0.005008321,
0.005065865, 0.005125678, 0.005185946, 0.005242005, 0.00530067,
0.00535942, 0.005423442, 0.005483337, 0.00554625, 0.005611636,
0.005672218, 0.005739257, 0.005800053, 0.005867203, 0.005927657,
0.005996269, 0.006065751, 0.006141852, 0.006217915, 0.006287729,
0.006355394, 0.006425831, 0.006498608, 0.006566091, 0.006643253,
0.006720284, 0.006800723, 0.006878689, 0.006959649, 0.007045243,
0.007129076, 0.007208403, 0.007295135, 0.0073787, 0.007460221,
0.007548922, 0.00764502, 0.00773719, 0.007823177, 0.007920985,
0.008014114, 0.008107085, 0.008195817, 0.008290218, 0.008399718,
0.008500579, 0.008595087, 0.00869057, 0.008787535, 0.008893321,
0.008997173, 0.009101655, 0.009204832, 0.00930841, 0.009419618,
0.009529949, 0.009644836, 0.009732203, 0.009753369, 0.009872621,
0.010001239, 0.010105596, 0.010224849, 0.010348911, 0.010468678,
0.010590131, 0.010713871, 0.010831679, 0.010959708, 0.011084372,
0.011212773, 0.011349906, 0.011477108, 0.011612292, 0.011752188,
0.011876578, 0.012015782, 0.012164229, 0.012308984, 0.012448955,
0.012593118, 0.012736479, 0.012884089, 0.01302912, 0.013172017,
0.013321327, 0.013477562, 0.013633515, 0.013786587, 0.013947586,
0.014114776, 0.014279578, 0.014448141, 0.014613965, 0.014787641,
0.014954839, 0.01513954, 0.015313284, 0.015487373, 0.01566419,
0.015845022, 0.016029482, 0.01621414, 0.016407936, 0.016607842,
0.016807044, 0.016997585, 0.017196331, 0.017398729, 0.017611989,
0.017821983, 0.018040877, 0.018235352, 0.018444692, 0.018660641,
0.0188686, 0.019091658, 0.019315847, 0.019542469, 0.019766496,
0.019989686, 0.020224088, 0.020463392, 0.020693516, 0.020927082,
0.021177022, 0.021433357, 0.021679765, 0.021931464, 0.022182028,
0.022443883, 0.022698857, 0.022968803, 0.023235849, 0.023504034,
0.023774381, 0.02405783, 0.024340156, 0.024621945, 0.024908732,
0.025187977, 0.025491747, 0.025788609, 0.026096511, 0.026402192,
0.026699354, 0.026994529, 0.027311611, 0.027639651, 0.027956998,
0.028291412, 0.028634313, 0.028965138, 0.029311339, 0.029648234,
0.029998201, 0.03034332, 0.030702186, 0.031052829, 0.031417265,
0.031784407, 0.032169706, 0.032549544, 0.032938366, 0.033327378,
0.033704401, 0.034105844, 0.034497567, 0.034887289, 0.035301499,
0.035701939, 0.036113386, 0.036528249, 0.036957822, 0.037384029,
0.037822902, 0.03826398, 0.038713053, 0.039176135, 0.039622776,
0.040083857, 0.040556416, 0.041041214, 0.041514883, 0.042008619,
0.042503806, 0.042991733, 0.043489954, 0.044004042, 0.044538253,
0.045034, 0.045554199, 0.046091627, 0.046646091, 0.04718862,
0.047715135, 0.048271222, 0.048844693, 0.049410007, 0.049980481,
0.050550677, 0.051138066, 0.051735984, 0.052327678, 0.052924346,
0.053529954, 0.054170371, 0.054805875, 0.055442472, 0.056084441,
0.05672317, 0.057392575, 0.058068681, 0.058735359, 0.059414269,
0.060124658, 0.060842672, 0.061580211, 0.062298397, 0.063051539,
0.063787213, 0.064537366, 0.065283842, 0.066047875, 0.066822007,
0.067607635, 0.068394006, 0.069204212, 0.070021073, 0.070862005,
0.071699313, 0.072527276, 0.073378054, 0.074230993, 0.075111618,
0.076012259, 0.076915538, 0.077806951, 0.078737246, 0.079697082,
0.080632591, 0.08162023, 0.082572133, 0.08353105, 0.084517055,
0.085501063, 0.086501146, 0.087521842, 0.088539982, 0.089576696,
0.090640562, 0.091686375, 0.092757534, 0.093844297, 0.094939967,
0.09605792, 0.097178582, 0.098366153, 0.09930958, 0.09955393,
0.100739591, 0.10194473, 0.103132352, 0.104364977, 0.105566312,
0.106822762, 0.10805741, 0.109337794, 0.110628089, 0.111972316,
0.113322227, 0.114657834, 0.116019013, 0.117392826, 0.11876706,
0.120177467, 0.121587343, 0.123032361, 0.124523154, 0.126016455,
0.127504056, 0.129004872, 0.130540134, 0.132120996, 0.133677661,
0.135278693, 0.13691735, 0.13854684, 0.140232394, 0.141891028,
0.143606851, 0.14532046, 0.147029056, 0.14877237, 0.150556234,
0.15234505, 0.154140328, 0.155956008, 0.157817737, 0.159682892,
0.161573687, 0.163512694, 0.165491328, 0.167508428, 0.16955935,
0.171581228, 0.173626931, 0.175701721, 0.177812212, 0.17996001,
0.182098969, 0.184282445, 0.186512792, 0.188790872, 0.191052012,
0.193340662, 0.195666988, 0.198038625, 0.200448685, 0.202871575,
0.205343514, 0.207840608, 0.210379608, 0.212956005, 0.215515282,
0.218160845, 0.220802515, 0.223516542, 0.226226438, 0.229024412,
0.231829033, 0.234691177, 0.237593271, 0.240490256, 0.243438996,
0.246406163, 0.249422885, 0.252455544, 0.255570192, 0.258713575,
0.26187484, 0.265095762, 0.26833329, 0.271603272, 0.274917043,
0.278308496, 0.281758521, 0.285235408, 0.288788575, 0.292351928,
0.295932063, 0.299550269, 0.30325344, 0.30701843, 0.310838386,
0.311680668, 0.314673337, 0.318581629, 0.322523756, 0.326547269,
0.330595112, 0.334718944, 0.33887486, 0.343095843, 0.347379217,
0.351737477, 0.356156686, 0.360651878, 0.36518786, 0.369767332,
0.374453753, 0.379175528, 0.383937526, 0.388814883, 0.39376857,
0.398792481, 0.403858897, 0.408899773, 0.414086162, 0.419343747,
0.42465002, 0.42574664, 0.430098158, 0.435497752, 0.441071309,
0.446701504, 0.452391074, 0.458108375, 0.463941611, 0.469950407,
0.475986986, 0.482110283, 0.488339593, 0.494688845, 0.501103207,
0.507706893, 0.514272349, 0.521008376, 0.527765433, 0.534638599,
0.541597767, 0.548754525, 0.555972659, 0.56330119, 0.570711414,
0.578260546, 0.585912212, 0.593761274, 0.601576555, 0.609635624,
0.617823499, 0.626048753, 0.63450087, 0.643074274, 0.651724749,
0.660623167, 0.669647466, 0.673451769, 0.67874401, 0.687926369,
0.697332342, 0.706911912, 0.716668348, 0.726507986, 0.736562899,
0.746839561, 0.757253651, 0.767763594, 0.778433226, 0.789242022,
0.800339231, 0.811580185, 0.823005571, 0.834630958, 0.846464603,
0.858596075, 0.87101139, 0.883512883, 0.896233922, 0.909236815,
0.922419484, 0.935872624, 0.949670262, 0.955440208, 0.963692301,
0.977790598, 0.992319429, 1.007124351, 1.022177732, 1.037513126,
1.053179588, 1.069160185, 1.085420373, 1.102105374, 1.119119883,
1.136326101, 1.154039419, 1.172026647, 1.190625736, 1.209479766,
1.228713788, 1.248339713, 1.268419464, 1.284573287, 1.321639362,
1.342272788, 1.362982694, 1.383557776, 1.404172208, 1.424717429,
1.445346956, 1.466109536, 1.486743558, 1.50731015, 1.527901032,
1.548570269, 1.569160195, 1.589818618, 1.610373325, 1.630926231,
1.651296888, 1.6717993, 1.683985695, 1.692470813, 1.713058868,
1.733561761, 1.754068315, 1.774671533, 1.795251148, 1.815888073,
1.836465441, 1.857220372, 1.877883177, 1.898644687, 1.919343311,
1.940040504, 1.960783269, 1.981272282, 2.001958552, 2.022493137,
2.043221768, 2.064102652, 2.084688097, 2.105348588, 2.125878877,
2.146750614, 2.167517857, 2.188262054, 2.200339442, 2.209150257,
2.229922151, 2.250779745, 2.271510262, 2.292365533, 2.313091359,
2.333993595, 2.355093362, 2.37601713, 2.397002683, 2.417999798,
2.439139822, 2.460261012, 2.48130311, 2.502156437, 2.523542037,
2.544661658, 2.565841366, 2.586674902, 2.607832329, 2.629271917,
2.650655046, 2.67168481, 2.693133031, 2.714260852, 2.735529673,
2.757065259, 2.77859457, 2.799976656, 2.82139857, 2.842790766,
2.864028486, 2.885699258, 2.907258302, 2.929147849, 2.946180164,
2.950752243, 2.972720065, 2.994601921, 3.016672139, 3.038656344,
3.060818347, 3.082813812, 3.104965901, 3.127013767, 3.149260912,
3.171504942, 3.193858861, 3.2162127, 3.238628307, 3.260908359,
3.28336206, 3.305656622, 3.32802014, 3.350363301, 3.372636555,
3.395329315, 3.418168983, 3.440874391, 3.463512199, 3.48658826,
3.504469608, 3.509562085, 3.53241123, 3.555374175, 3.578673729,
3.601990387, 3.625083391, 3.648260501, 3.671710265, 3.695146217,
3.718529478, 3.742065682, 3.765631529, 3.789165666, 3.812906431,
3.836868473, 3.860997857, 3.884929784, 3.909061313, 3.933240899,
3.957446023, 3.981430549, 4.005536511, 4.029852826, 4.054606346,
4.079111314, 4.103872024, 4.128593247, 4.15341375, 4.178151025,
4.203117016, 4.22830198, 4.253041353, 4.278538093, 4.303947437,
4.329276012, 4.35471727, 4.380527794, 4.406411084, 4.431938335,
4.457783404, 4.483829115, 4.510179523, 4.536371088, 4.563108387,
4.589343527, 4.616078711, 4.641845068, 4.668366153, 4.694623603,
4.721344721, 4.748186031, 4.774970427, 4.801900909, 4.829124922,
4.856471319, 4.884234435, 4.912157482, 4.93986162, 4.967664893,
4.995252061, 5.023271175, 5.05136667, 5.079754205, 5.107768971,
5.136233292, 5.164382591, 5.193800929, 5.223264227, 5.252421698,
5.281439149, 5.310579708, 5.340026948, 5.369766026, 5.399875459,
5.430239327, 5.459964812, 5.490118474, 5.520173935, 5.550437788,
5.580372518, 5.610513218, 5.641549577, 5.672279615, 5.703207627,
5.733745963, 5.764776724, 5.796226147, 5.827932334, 5.859777139,
5.892006702, 5.923903596, 5.955564198, 5.988000023, 6.020788212,
6.053347654, 6.085829355, 6.092458146, 6.119485946, 6.152368204,
6.18497756, 6.219272103, 6.253143057, 6.287258207, 6.320964193,
6.355056044, 6.389817216, 6.423600284, 6.458056725, 6.493014459,
6.526972482, 6.561279225, 6.596212589, 6.631986246, 6.667439185,
6.702988837, 6.737537991, 6.774032075, 6.810035283, 6.845477207,
6.881774115, 6.918132893, 6.954506569, 6.990622635, 7.027203853,
7.064007537, 7.101599088, 7.137714009, 7.175196264, 7.212979989,
7.250432895, 7.288135691, 7.325505591, 7.364112319, 7.402237568,
7.441408991, 7.479538909, 7.519254155, 7.558502738, 7.596909745,
7.635898125, 7.675591236, 7.71522184, 7.754434236, 7.793762139,
7.833612765, 7.873658911, 7.914049403, 7.954950937, 7.995005268,
8.03475742, 8.07464853, 8.114760601, 8.155643333, 8.197054627,
8.238720786, 8.279884155, 8.321561683, 8.365412223, 8.408735198,
8.451301333, 8.493771736, 8.536356254, 8.577114604, 8.620044556,
8.660700068, 8.703769188, 8.747156516, 8.789871, 8.834265227,
8.878548959, 8.923838216, 8.967494432, 9.010749257, 9.05733793,
9.102325008, 9.146125483, 9.191856802, 9.240260057, 9.285359264,
9.329607027, 9.374516915, 9.419639637, 9.464845975, 9.509167384,
9.555804173, 9.600952891, 9.646896998, 9.694008777, 9.740328458,
9.78682622, 9.832579076, 9.87879154, 9.927539069, 9.97430561,
10.02186452, 10.071087239, 10.119248981, 10.170338348, 10.220149125,
10.268403035, 10.316606051, 10.365110346, 10.413777061, 10.461853065,
10.510402409, 10.560750472, 10.610620211, 10.660012871, 10.710360265,
10.760990028, 10.808423607, 10.856726494, 10.906510564, 10.957801798,
11.008168817, 11.057974289, 11.109290872, 11.161157479, 11.210586146,
11.264079576, 11.314258283, 11.366590084, 11.417608699, 11.468715011,
11.522585336, 11.574983983, 11.625973118, 11.678095491, 11.728874727,
11.781379003, 11.832272762, 11.88613428, 11.940300589, 11.991479917,
12.044033727, 12.097139726, 12.149169323, 12.201442842, 12.256681731,
12.311959977, 12.364127735, 12.418561934, 12.474139204, 12.525817559,
12.581844148, 12.636539851, 12.695379832, 12.751402927, 12.807027106,
12.865752819, 12.923426872, 12.978686201, 13.036804073, 13.091074293,
13.144753943, 13.20309434, 13.258738372, 13.313410092, 13.370522317,
13.42992948, 13.48777027, 13.546099945, 13.602302127, 13.659976091,
13.718611013, 13.778944643, 13.836824804, 13.892893069, 13.951240969,
14.008323966, 14.065578153, 14.122786617, 14.18338984, 14.247440683,
14.309038672, 14.367322104, 14.421583202, 14.481870146, 14.541091579,
14.599444127, 14.657918123, 14.720336414, 14.783480528, 14.844676767,
14.909272396, 14.974679383, 15.035148549, 15.101607485, 15.166862086,
15.233456369, 15.297882842, 15.364401176, 15.427923517, 15.491143719,
15.551126027, 15.611360062, 15.676272864, 15.688788981, 15.738177819,
15.805231793, 15.868125519, 15.936280487, 15.996160302, 16.062882312,
16.126102437, 16.189001821, 16.253005339, 16.316881228, 16.38219183,
16.452607623, 16.520149576, 16.586930477, 16.650767363, 16.716649427,
16.787081275, 16.851178118, 16.915573691, 16.986815594, 17.046213355,
17.112507303, 17.182867894, 17.258674628, 17.32847402, 17.397945902,
17.467722045, 17.531480952, 17.60154609, 17.675351224, 17.748197248,
17.811500605, 17.876363856, 17.95514218, 18.027542112, 18.098793132,
18.168009851, 18.233302712, 18.301314188, 18.380448017, 18.446667317,
18.515106497, 18.585673658, 18.658280829, 18.728630739, 18.79793989,
18.868606947, 18.937077452, 19.005572935, 19.068236117, 19.139257452,
19.213528516, 19.293014909, 19.361923595, 19.437547106, 19.51859661,
19.594194941, 19.671706302, 19.750447928, 19.815661015, 19.894181786,
19.964643477, 20.043222981, 20.11968579, 20.192526802, 20.26811786,
20.344141639, 20.424038024, 20.49609047, 20.575278341, 20.649487876,
20.725433715, 20.800636703, 20.87475852, 20.956674609, 21.039976212,
21.11946536, 21.194307988, 21.267459395, 21.345332688, 21.4201833,
21.498029434, 21.581756309, 21.653618184, 21.729477883, 21.815030515,
21.890799323, 21.975475034, 22.055556787, 22.137696914, 22.221375531,
22.302854579, 22.38647792, 22.469143506, 22.553970771, 22.630862482,
22.711567994, 22.794066681, 22.872643651, 22.946196787, 23.028839616,
23.116275591, 23.20718837, 23.30125571, 23.381125252, 23.470297797,
23.553918784, 23.639891105, 23.722692867, 23.802456031, 23.883410805,
23.976649417, 24.065167977, 24.16828007, 24.283107438, 24.37288418,
24.45705714, 24.553748468, 24.647664343, 24.727967718, 24.824199026,
24.911756746, 24.997690662, 25.095568321, 25.183765455, 25.273868154,
25.360616958, 25.437108589, 25.521000882, 25.616168891, 25.700834515,
25.767479136, 25.856350786, 25.951573089, 26.029939764, 26.109312557,
26.205400367, 26.290803909, 26.394082449, 26.498194169, 26.593068061,
26.68053386, 26.778906424, 26.87187343, 26.958317208, 27.05604682,
27.152114947, 27.250965719, 27.322918754, 27.397707899, 27.488660848,
27.574933427, 27.687431023, 27.826490326, 27.933988634, 28.026068067,
28.123089268, 28.211508468, 28.292108244, 28.40770038, 28.523512805,
28.610398375, 28.702910603, 28.809385108, 28.920854669, 29.020639831,
29.10249943, 29.190488635, 29.294813102, 29.391177183, 29.478205221,
29.584203651, 29.708322482, 29.796443266, 29.90685421, 30.024465906,
30.147896591, 30.244801941, 30.321304184, 30.432348724, 30.541558986,
30.633427483, 30.741421127, 30.859838638, 30.952433423, 31.045138958,
31.149805426, 31.250755119, 31.354462958, 31.485599219, 31.612786368,
31.713272641, 31.823166156, 31.959267657, 32.048982105, 32.150148017,
32.262405687, 32.375226177, 32.472261857, 32.556845977, 32.579439537,
32.652378611, 32.770832365, 32.926519411, 33.0698871, 33.174325964,
33.325934711, 33.452925667, 33.58132586, 33.72897591, 33.833433927,
33.940992988, 34.081508598, 34.202566363, 34.31729451, 34.442211032,
34.538608894, 34.656003856, 34.736329629, 34.851079948, 34.954199931,
35.08209198, 35.194321886, 35.295039325, 35.390833875, 35.482731847,
35.606794616, 35.719633662, 35.844736725, 35.966767873, 36.096817642,
36.235677461, 36.344762266, 36.450605921, 36.556068723, 36.661544707,
36.784236709, 36.924255998, 37.020517356, 37.136820491, 37.211736102,
37.337083101, 37.45197252, 37.583901629, 37.709131502, 37.846322928,
37.980336628, 38.10136568, 38.189401999, 38.307848573, 38.459911387,
38.577783057, 38.680748082, 38.800734057, 38.953377846, 39.087798509,
39.220065532, 39.33046205, 39.437779151, 39.587205046, 39.694632064,
39.850918062, 39.964820334, 40.050171844, 40.218076174, 40.391751673,
40.509948263, 40.625621617, 40.758439512, 40.921556903, 41.02538218,
41.15638064, 41.28440626, 41.421204114, 41.543248896, 41.677606339,
41.805604604, 41.99173644, 42.113656218, 42.229028819, 42.354421234,
42.470063385, 42.565830989, 42.674902611, 42.835707567, 42.963450727,
43.116212286, 43.220941839, 43.349999963, 43.467051436, 43.622608002,
43.739359743, 43.861585363, 43.981756383, 44.055050028, 44.172141648,
44.286492039, 44.41799461, 44.563047498, 44.690258344, 44.792321106,
44.948511483, 45.129020465, 45.236866212, 45.380872636, 45.481569175,
45.587261257, 45.723941197, 45.815994898, 45.994982967, 46.155507591,
46.300439285, 46.525311516, 46.745955337, 46.834397609, 46.9759107,
47.127268236, 47.256851593, 47.343882945, 47.469767794, 47.590042276,
47.704004602, 47.842224752, 47.965281196, 48.130880661, 48.249598292,
48.414075931, 48.508881882, 48.613717357, 48.696960346, 48.903444337,
49.015990933, 49.269755065, 49.463911846, 49.592440362, 49.725988258,
49.841233547, 50.021015081, 50.19713033, 50.374835085, 50.529856958,
50.783723907, 51.061421949, 51.206029168, 51.400599599, 51.489432924,
51.605883774, 51.761362833, 51.93454465, 52.165021094, 52.293927755,
52.374530163, 52.509759374, 52.647606807, 52.807475207, 52.972900934,
53.224596514, 53.402449521, 53.513920131, 53.61306836, 53.749342772,
53.890570831, 54.057048744, 54.146160941, 54.236195145, 54.482788839,
54.580983189, 54.863526418, 55.113865761, 55.262274549, 55.408646429,
55.613969193, 55.692846575, 55.857414341, 55.925698066, 56.073820325,
56.217187679, 56.305445012, 56.472038969, 56.565442188, 56.912445348,
57.178476351, 57.303349524, 57.580141994, 57.801895371, 57.891361588,
58.087076905, 58.397744804, 58.574840465, 58.772743036, 58.965996171,
59.093217119, 59.25062406, 59.493871632, 59.771810995, 60.047577313,
60.174698933, 60.429835294, 60.570693981, 60.83587879, 61.055046613,
61.411802903, 61.537326941, 61.737380689, 61.811396205, 62.05833106,
62.292763054, 62.408802675, 62.495176099, 62.657041983, 62.75375763,
62.979816993, 63.124716489, 63.343134876, 63.544214773, 63.690019088,
63.877477271, 64.079216847, 64.345686683, 64.517202053, 64.751400444,
64.938097741, 65.250508374, 65.386393998, 65.751598488, 65.9370796,
66.201001133, 66.416031694, 66.75376154, 66.878536523, 66.990974981,
67.130480494, 67.2040714, 67.326438866, 67.441642871, 67.636025211,
68.054447254, 68.176630802, 68.511974182, 68.644317239, 68.794866624,
68.995827568, 69.168872399, 69.519025161, 69.640821428, 69.740916418,
69.880351356, 70.164464237, 70.35480103, 70.394126052, 70.479649804,
70.614052871, 70.955358349, 71.199471438, 71.609215262, 71.65558816,
71.827032225, 71.932431278, 72.219496937, 72.354165379, 72.428538236,
72.525042598, 72.569267777, 72.687652999, 72.854233513, 73.273157871,
73.41170583, 73.7820163, 73.858262564, 74.198313466, 74.250389328,
74.380197959, 74.422749945, 74.664260877, 74.954266994, 75.189406521,
75.371494657, 75.561270458, 75.74400318, 76.038301527, 76.083965641,
76.585055223, 76.728008585, 76.823534319, 77.109007087, 77.526347613,
77.741686544, 77.932022, 78.109018685, 78.290622628, 78.40670286,
78.642369516, 78.772056863, 79.085083227, 79.41920218, 79.746599796,
79.876978945, 79.89027833, 79.986354038, 80.192640924, 80.220118074,
80.228772874, 80.342773432, 80.604354208, 80.805180304, 80.984728073,
81.307549957, 81.311944457, 81.465300195, 81.563148052, 81.639796217,
81.666937949, 81.850788785, 82.207255894, 82.325674477, 82.63091307,
82.856215301, 83.195624484, 83.202771087, 83.510743103, 83.57616216,
84.072782868, 84.252073469, 84.364236645, 84.629921624, 84.746423784,
84.886318259, 84.912137754, 85.014864268, 85.067113038, 85.320290885,
85.458397964, 85.599074607, 85.871143975, 85.925313707, 86.190437996,
86.250100145, 86.329171101, 86.693697706, 86.968610227, 86.971093193
), "500" = c(1.97e-05, 1.9944e-05, 1.9983e-05, 2.0238e-05, 2.0244e-05,
2.0285e-05, 2.0325e-05, 2.0736e-05, 2.1091e-05, 2.1204e-05, 2.1281e-05,
2.2057e-05, 2.2115e-05, 2.229e-05, 2.2512e-05, 2.2851e-05, 2.313e-05,
2.3532e-05, 2.3571e-05, 2.368e-05, 2.3739e-05, 2.3872e-05, 2.4161e-05,
2.4179e-05, 2.4441e-05, 2.449e-05, 2.5053e-05, 2.5068e-05, 2.5197e-05,
2.5337e-05, 2.5861e-05, 2.6467e-05, 2.6514e-05, 2.6871e-05, 2.7309e-05,
2.7427e-05, 2.7442e-05, 2.8577e-05, 2.8783e-05, 2.9131e-05, 2.9258e-05,
2.954e-05, 2.9907e-05, 3.0175e-05, 3.14e-05, 3.1577e-05, 3.2466e-05,
3.274e-05, 3.3532e-05, 3.3878e-05, 3.4036e-05, 3.4207e-05, 3.4509e-05,
3.4703e-05, 3.4968e-05, 3.5446e-05, 3.5665e-05, 3.6668e-05, 3.7155e-05,
3.799e-05, 3.8296e-05, 3.8625e-05, 3.8927e-05, 3.9559e-05, 3.964e-05,
3.9812e-05, 4.0118e-05, 4.0137e-05, 4.0367e-05, 4.0492e-05, 4.1057e-05,
4.169e-05, 4.2832e-05, 4.3505e-05, 4.3596e-05, 4.3814e-05, 4.4471e-05,
4.4701e-05, 4.5484e-05, 4.6175e-05, 4.7398e-05, 4.8349e-05, 4.851e-05,
4.9741e-05, 5.0642e-05, 5.1103e-05, 5.1802e-05, 5.2618e-05, 5.3055e-05,
5.3496e-05, 5.3662e-05, 5.4177e-05, 5.4649e-05, 5.5044e-05, 5.5234e-05,
5.5743e-05, 5.6493e-05, 5.7328e-05, 5.8191e-05, 5.894e-05, 5.9704e-05,
6.1206e-05, 6.1443e-05, 6.1626e-05, 6.2086e-05, 6.3076e-05, 6.4848e-05,
6.571e-05, 6.6692e-05, 6.8113e-05, 7.0122e-05, 7.0436e-05, 7.1023e-05,
7.2005e-05, 7.2972e-05, 7.3904e-05, 7.4264e-05, 7.5809e-05, 7.7062e-05,
7.7802e-05, 7.8477e-05, 7.9415e-05, 8.022e-05, 8.1138e-05, 8.2012e-05,
8.2705e-05, 8.4549e-05, 8.4785e-05, 8.6967e-05, 8.7618e-05, 8.8601e-05,
8.9938e-05, 9.0656e-05, 9.1397e-05, 9.2234e-05, 9.2691e-05, 9.3366e-05,
9.4076e-05, 9.4667e-05, 9.5603e-05, 9.6394e-05, 9.8617e-05, 9.9596e-05,
0.00010081, 0.000102696, 0.000103561, 0.000105044, 0.000105745,
0.000107135, 0.000109165, 0.000110916, 0.000111876, 0.000112684,
0.00011353, 0.000114921, 0.000116752, 0.000118428, 0.000120226,
0.000122814, 0.000124159, 0.00012693, 0.000128407, 0.000129883,
0.000131533, 0.000133238, 0.000134704, 0.000135865, 0.000138226,
0.000139409, 0.000140705, 0.000142384, 0.000146004, 0.000149602,
0.000150779, 0.000152058, 0.000154972, 0.000156466, 0.000157537,
0.000160011, 0.000162907, 0.000164615, 0.000167151, 0.000168396,
0.000170381, 0.000172046, 0.000174711, 0.00017683, 0.000178973,
0.000180837, 0.000182141, 0.000183649, 0.000184857, 0.000186404,
0.000189918, 0.000192693, 0.000194373, 0.000197199, 0.000198555,
0.000200209, 0.000202486, 0.000205435, 0.000206989, 0.000209848,
0.000213838, 0.000216693, 0.000218422, 0.000221093, 0.00022381,
0.000225557, 0.000228925, 0.000231284, 0.0002343, 0.000237468,
0.000240286, 0.000242273, 0.000245297, 0.00024898, 0.000252378,
0.000255657, 0.000258206, 0.000260337, 0.000262056, 0.000264133,
0.000267489, 0.000269818, 0.000272336, 0.000276012, 0.000278198,
0.000282051, 0.000284641, 0.00028829, 0.000291269, 0.000294198,
0.000297389, 0.000302201, 0.000305018, 0.000308845, 0.00031323,
0.000316407, 0.000319657, 0.000322816, 0.000327732, 0.000331208,
0.000336997, 0.000340226, 0.000343539, 0.000346992, 0.00034998,
0.000352988, 0.000355662, 0.00035913, 0.000363089, 0.000367133,
0.000370487, 0.000375114, 0.000380365, 0.00038405, 0.000388166,
0.000392428, 0.000396587, 0.000401087, 0.000404903, 0.000409736,
0.000415312, 0.000419328, 0.000424049, 0.000430218, 0.000433421,
0.000438705, 0.000442534, 0.00044626, 0.000452213, 0.000458034,
0.00046298, 0.000469792, 0.000475702, 0.000482393, 0.000489115,
0.000494765, 0.00050104, 0.000506087, 0.000513102, 0.00051937,
0.000523753, 0.000531199, 0.000538758, 0.000544299, 0.000549686,
0.000553736, 0.000559418, 0.000567113, 0.00057399, 0.000581225,
0.000587304, 0.000593946, 0.000600864, 0.000606527, 0.000613226,
0.000618437, 0.000625763, 0.000634865, 0.000645351, 0.000654454,
0.000661152, 0.000670038, 0.000677093, 0.000684377, 0.000691344,
0.000697905, 0.000704475, 0.000709912, 0.000718457, 0.000725613,
0.000732882, 0.000739818, 0.000746306, 0.000754762, 0.000763565,
0.000770902, 0.000778932, 0.000787266, 0.000798047, 0.000810289,
0.00081893, 0.000828753, 0.000838138, 0.000849021, 0.000860539,
0.000869692, 0.000881994, 0.000891243, 0.000901577, 0.000912615,
0.000923754, 0.000933915, 0.00094278, 0.000955433, 0.000964704,
0.000975864, 0.000987972, 0.000996228, 0.000998289, 0.001011446,
0.001025966, 0.001038797, 0.001050427, 0.001063202, 0.001077491,
0.00109058, 0.001103811, 0.001114912, 0.001128197, 0.001139917,
0.001154491, 0.001166947, 0.001179503, 0.001191707, 0.001204658,
0.001217521, 0.001232613, 0.00124883, 0.001264102, 0.001278377,
0.001296099, 0.001309706, 0.001323689, 0.001341117, 0.001355673,
0.001371666, 0.001387257, 0.001404922, 0.001418401, 0.001434993,
0.001453185, 0.001468345, 0.001483339, 0.001498794, 0.001512723,
0.001532127, 0.001549396, 0.00156466, 0.001583566, 0.001603135,
0.001620737, 0.001636756, 0.001653349, 0.001674848, 0.001692437,
0.001713233, 0.001731685, 0.001750003, 0.001767867, 0.001790564,
0.001812403, 0.001836621, 0.001856474, 0.001876067, 0.001899063,
0.001922769, 0.001943074, 0.001967506, 0.001986681, 0.002009203,
0.002030947, 0.002054146, 0.002079176, 0.002103867, 0.002129129,
0.002153058, 0.002176376, 0.002202615, 0.002228694, 0.00225659,
0.002279068, 0.002307734, 0.002337141, 0.002362277, 0.002388474,
0.002417281, 0.002444215, 0.002471354, 0.00249934, 0.00253213,
0.002560206, 0.002586961, 0.002622037, 0.002647132, 0.002674516,
0.002706007, 0.002736184, 0.002765154, 0.002793209, 0.002819409,
0.002858238, 0.002888903, 0.00291575, 0.002948836, 0.002987097,
0.003015453, 0.003053146, 0.003089763, 0.003121222, 0.003153287,
0.003183766, 0.00322596, 0.00326276, 0.003294227, 0.003331335,
0.003370408, 0.003412199, 0.00344956, 0.003489948, 0.003533415,
0.0035732, 0.003613696, 0.003658453, 0.003701226, 0.003746879,
0.003792429, 0.003834269, 0.003877497, 0.003924373, 0.003968051,
0.004013051, 0.004059906, 0.004110375, 0.004157271, 0.004211352,
0.004260205, 0.004308627, 0.004358076, 0.004411577, 0.004461445,
0.004514042, 0.004568073, 0.004621293, 0.004677201, 0.004732631,
0.00478636, 0.004839322, 0.004895318, 0.004948547, 0.005004741,
0.005061532, 0.005115878, 0.005172406, 0.005228346, 0.00528848,
0.005349877, 0.005407188, 0.005471568, 0.005534579, 0.005599721,
0.005666649, 0.005736224, 0.005800532, 0.005869781, 0.005942246,
0.006012544, 0.006085984, 0.00615612, 0.006220708, 0.006288287,
0.006360389, 0.006444181, 0.006518026, 0.006592994, 0.006663444,
0.006744742, 0.00682145, 0.006896436, 0.006969377, 0.007047639,
0.007129305, 0.007209968, 0.007289341, 0.007370093, 0.00745556,
0.007544798, 0.007634132, 0.007722977, 0.007810447, 0.007903546,
0.007990608, 0.008085429, 0.008180621, 0.008277731, 0.008377622,
0.008474104, 0.008576991, 0.008670032, 0.00877322, 0.008875303,
0.008971242, 0.009077409, 0.009177906, 0.009278141, 0.00938234,
0.00949319, 0.009597469, 0.009710022, 0.009804856, 0.009826319,
0.009951527, 0.010063638, 0.010184044, 0.010308565, 0.010425699,
0.010547371, 0.010664089, 0.010788515, 0.010900412, 0.011023268,
0.011150013, 0.011284519, 0.011408973, 0.011541029, 0.011665342,
0.011804744, 0.011943172, 0.012088401, 0.012228158, 0.012368974,
0.012501117, 0.012653009, 0.012807731, 0.012959454, 0.013110476,
0.013252675, 0.013415943, 0.013572353, 0.013733341, 0.013892527,
0.014056672, 0.014215885, 0.014379751, 0.01455405, 0.01472471,
0.014892915, 0.015066768, 0.01523584, 0.015413346, 0.015589366,
0.01577671, 0.015957867, 0.016142334, 0.016331621, 0.016531673,
0.016720406, 0.01691199, 0.017102795, 0.017294199, 0.017481852,
0.017687483, 0.017885158, 0.018082419, 0.018306292, 0.018504279,
0.018714142, 0.018951513, 0.019171128, 0.019394184, 0.019605482,
0.019830118, 0.020058196, 0.020285916, 0.020536247, 0.02077748,
0.021033338, 0.021292928, 0.021542596, 0.021793982, 0.022050311,
0.022310138, 0.022571149, 0.022825189, 0.023091918, 0.023356528,
0.023638442, 0.023911346, 0.024198393, 0.024467791, 0.024741915,
0.025017292, 0.025304712, 0.025603532, 0.025906872, 0.026202457,
0.026498064, 0.026809601, 0.027109765, 0.02742716, 0.027750473,
0.028068445, 0.028394141, 0.028729113, 0.029079292, 0.029418175,
0.029754094, 0.030101793, 0.030442487, 0.03079557, 0.031158446,
0.031520292, 0.031881374, 0.032267385, 0.032643884, 0.033028329,
0.033409475, 0.033791961, 0.034184057, 0.034585558, 0.035002073,
0.035420676, 0.035831154, 0.036242219, 0.036679966, 0.037106225,
0.037536914, 0.037985319, 0.038445855, 0.038904837, 0.039363937,
0.039835449, 0.04030913, 0.040779193, 0.041272767, 0.041752603,
0.042243748, 0.042736618, 0.043229083, 0.043733217, 0.044241122,
0.044758646, 0.045272801, 0.04578619, 0.04632146, 0.04685639,
0.047382241, 0.047923098, 0.048490071, 0.049049804, 0.049625813,
0.050202515, 0.050780716, 0.051378455, 0.051960502, 0.052575071,
0.053189203, 0.053826321, 0.054453624, 0.055075876, 0.05572356,
0.056375629, 0.057044552, 0.057725179, 0.058391762, 0.059058433,
0.059750035, 0.060453552, 0.061177912, 0.061902092, 0.06265109,
0.063386742, 0.06412957, 0.064891575, 0.065694349, 0.066458303,
0.067262268, 0.068054507, 0.068853437, 0.069644421, 0.070476291,
0.071310045, 0.072130398, 0.072964973, 0.073838409, 0.074693518,
0.075586243, 0.076479538, 0.07738226, 0.078283726, 0.079197481,
0.080137992, 0.081091649, 0.082075983, 0.083043655, 0.084036607,
0.085041779, 0.086022798, 0.08705722, 0.088083532, 0.089127876,
0.090195634, 0.091253413, 0.092318669, 0.093437004, 0.094570048,
0.095702388, 0.096837986, 0.097973434, 0.099131808, 0.100072291,
0.100320201, 0.101513417, 0.102698149, 0.10391127, 0.105144001,
0.106412041, 0.107675654, 0.108978731, 0.110242422, 0.111575239,
0.112888246, 0.114231775, 0.115601612, 0.116990482, 0.118374852,
0.119743853, 0.121155327, 0.122596344, 0.124090471, 0.125571515,
0.127090103, 0.128599865, 0.130153986, 0.131708626, 0.133279775,
0.134857118, 0.136460305, 0.138123575, 0.139776351, 0.141443196,
0.143132084, 0.144810165, 0.146553683, 0.148333836, 0.150122392,
0.15194274, 0.153762944, 0.155631505, 0.157479967, 0.159405529,
0.16136957, 0.163315123, 0.165267465, 0.167222146, 0.169219418,
0.171244337, 0.173322507, 0.175366517, 0.177485068, 0.179655372,
0.181779709, 0.184005067, 0.186248711, 0.188497792, 0.190755255,
0.193047542, 0.1953721, 0.19777337, 0.200181484, 0.202630071,
0.205106395, 0.207572188, 0.210093357, 0.212656639, 0.215229749,
0.217861582, 0.220516809, 0.223241078, 0.225992385, 0.228733537,
0.231563725, 0.234361444, 0.237193673, 0.240155489, 0.243129407,
0.246084799, 0.249113988, 0.252164324, 0.255295566, 0.258398858,
0.261579603, 0.26476068, 0.268021502, 0.271296517, 0.274618228,
0.278014058, 0.281484027, 0.284950085, 0.288484707, 0.292138151,
0.295742103, 0.299400894, 0.303123112, 0.306901369, 0.310718428,
0.314579857, 0.315436447, 0.318519723, 0.322543976, 0.326613684,
0.330686582, 0.33479719, 0.338985599, 0.343161639, 0.347431236,
0.351822197, 0.356317594, 0.360781573, 0.365365758, 0.369930643,
0.37455834, 0.379300568, 0.384124524, 0.388936969, 0.393837002,
0.398871141, 0.403922579, 0.409105152, 0.414267633, 0.41961366,
0.42496687, 0.430485784, 0.43161693, 0.435985642, 0.441588473,
0.447261385, 0.453016587, 0.458851875, 0.464816239, 0.47075364,
0.476846005, 0.483063724, 0.489303247, 0.495588391, 0.501987382,
0.508526286, 0.515118436, 0.521876733, 0.528727072, 0.535676838,
0.542747452, 0.549916789, 0.557148928, 0.56452152, 0.572000243,
0.579656106, 0.587387097, 0.595236125, 0.603236987, 0.611435693,
0.619651933, 0.628024504, 0.636428813, 0.64499214, 0.653739987,
0.662698497, 0.671668955, 0.68084657, 0.684779376, 0.69011536,
0.699581745, 0.709248264, 0.719032737, 0.728947774, 0.739078313,
0.749368438, 0.759833201, 0.770498616, 0.781371953, 0.792353585,
0.803520229, 0.814938289, 0.826545773, 0.838274306, 0.850262924,
0.862436965, 0.874897669, 0.887557002, 0.900431277, 0.913695201,
0.927023516, 0.940679916, 0.954620249, 0.96871756, 0.974574147,
0.983079611, 0.997679643, 1.01267496, 1.027832416, 1.043368252,
1.059190054, 1.075386324, 1.091947718, 1.108820162, 1.126071813,
1.143645297, 1.161546972, 1.179774126, 1.198469917, 1.217562804,
1.237080328, 1.256978788, 1.277391382, 1.298110271, 1.314961558,
1.353460623, 1.374886945, 1.396180855, 1.417510506, 1.438840846,
1.460115682, 1.481477664, 1.502863887, 1.524250281, 1.545562548,
1.566843687, 1.588247254, 1.609545595, 1.630819739, 1.652212026,
1.673682126, 1.694856465, 1.716248928, 1.728743609, 1.737581797,
1.758736928, 1.780027218, 1.801316787, 1.822756027, 1.844278817,
1.865711095, 1.886938482, 1.908424436, 1.930113413, 1.951484823,
1.972931702, 1.994484026, 2.015789811, 2.037349815, 2.058802054,
2.080236793, 2.101721081, 2.123306575, 2.144636665, 2.166227664,
2.187660963, 2.209191036, 2.230746412, 2.252247027, 2.264608035,
2.273756135, 2.295308987, 2.316960761, 2.338495179, 2.360032016,
2.381912198, 2.403518422, 2.424948846, 2.446526922, 2.468108272,
2.489726773, 2.511382009, 2.533013014, 2.554678881, 2.576414699,
2.598193546, 2.619787737, 2.641339168, 2.663088069, 2.684731333,
2.70643649, 2.728393448, 2.750123736, 2.771934097, 2.793887825,
2.815750151, 2.837626829, 2.859596351, 2.881707041, 2.903873094,
2.925836544, 2.94787248, 2.970031409, 2.99217461, 3.014345616,
3.03208964, 3.036671758, 3.059046279, 3.081248763, 3.1035679,
3.12588334, 3.148244279, 3.170573832, 3.192853822, 3.215451041,
3.237747304, 3.260133041, 3.28246644, 3.304824738, 3.327143631,
3.349999192, 3.372479787, 3.395179266, 3.417749455, 3.440487879,
3.462971172, 3.485836225, 3.508663951, 3.531701586, 3.554634357,
3.577584154, 3.595245373, 3.600315073, 3.623523079, 3.646603142,
3.669440499, 3.692714006, 3.716102105, 3.739199875, 3.762576909,
3.785862635, 3.809221213, 3.832675143, 3.856137632, 3.879687524,
3.903029091, 3.926871056, 3.950344083, 3.974185554, 3.997658432,
4.021526295, 4.045251112, 4.069206659, 4.093399884, 4.117153396,
4.141151445, 4.16496735, 4.189201653, 4.21347343, 4.23762429,
4.261841505, 4.286341501, 4.310675139, 4.335179354, 4.359468648,
4.383783194, 4.408380545, 4.433551277, 4.458696924, 4.483077203,
4.507982505, 4.532737691, 4.557976931, 4.582652438, 4.607772282,
4.632611069, 4.657710626, 4.682934654, 4.708236048, 4.733691499,
4.759325542, 4.784381534, 4.809745331, 4.835335399, 4.86032553,
4.886197688, 4.912344972, 4.93782563, 4.963898939, 4.989883119,
5.015829429, 5.042736522, 5.068829712, 5.095698305, 5.122685583,
5.149148733, 5.176081124, 5.202689278, 5.230032404, 5.25653991,
5.283066239, 5.31006036, 5.337150202, 5.364856513, 5.391422672,
5.418294959, 5.445206121, 5.472665132, 5.500604861, 5.528373943,
5.556043246, 5.583358897, 5.611603446, 5.63963887, 5.667275979,
5.695262699, 5.724193194, 5.75270062, 5.781740196, 5.810958605,
5.840624419, 5.869167608, 5.897674599, 5.926443407, 5.955914816,
5.985315264, 6.014549632, 6.043728889, 6.049655438, 6.07292758,
6.102373632, 6.132204144, 6.162200589, 6.191102907, 6.221486434,
6.251129691, 6.281926125, 6.312207541, 6.34289062, 6.373207948,
6.403852959, 6.434823335, 6.465140737, 6.496311001, 6.528421081,
6.559358266, 6.590335719, 6.621238519, 6.652888703, 6.684748067,
6.716725626, 6.747804056, 6.780452514, 6.812444437, 6.844583081,
6.877323886, 6.910149748, 6.943783461, 6.975607308, 7.007954351,
7.040786757, 7.073987297, 7.107190663, 7.14057606, 7.17325032,
7.207333647, 7.240147458, 7.275012567, 7.308373545, 7.342467284,
7.376980257, 7.411720302, 7.445263375, 7.479858674, 7.514062765,
7.548456853, 7.584222237, 7.620034162, 7.653837768, 7.688398246,
7.723817438, 7.758375198, 7.792585746, 7.829135064, 7.864587114,
7.900083837, 7.935229632, 7.971195855, 8.007451713, 8.043261435,
8.080101646, 8.117274938, 8.153552797, 8.190161448, 8.227582119,
8.264812378, 8.30237384, 8.339235792, 8.376005909, 8.413187647,
8.450019308, 8.486229476, 8.52355978, 8.561429162, 8.598826978,
8.635277298, 8.672974254, 8.711490584, 8.749096205, 8.787616235,
8.826184791, 8.864413402, 8.90244856, 8.941071731, 8.980620649,
9.020017893, 9.060321991, 9.099480893, 9.139955826, 9.178350701,
9.217803088, 9.25831286, 9.298756437, 9.338514827, 9.379766641,
9.419698892, 9.461694392, 9.501706771, 9.541884484, 9.584348095,
9.626121558, 9.666595918, 9.708971148, 9.751851482, 9.793636981,
9.834655135, 9.877370026, 9.917993769, 9.95939969, 10.000303379,
10.042298134, 10.087097681, 10.130225649, 10.17424344, 10.218876053,
10.261360957, 10.303065821, 10.345419129, 10.389759186, 10.43209536,
10.473429145, 10.517403399, 10.561713836, 10.605322388, 10.648439138,
10.692552509, 10.734681247, 10.778158291, 10.824499998, 10.870063643,
10.916230157, 10.962806112, 11.006263664, 11.050800143, 11.092996119,
11.135127693, 11.181092262, 11.227555561, 11.270805372, 11.315460124,
11.360598234, 11.405571661, 11.450047618, 11.498209635, 11.543342618,
11.590228471, 11.635567596, 11.682601089, 11.72947624, 11.77632491,
11.824852059, 11.869456739, 11.917888379, 11.963890266, 12.010741397,
12.057600952, 12.107167418, 12.152821429, 12.198587396, 12.247742336,
12.29096171, 12.338659361, 12.386763156, 12.432677794, 12.479559076,
12.528434359, 12.575431976, 12.62285412, 12.67184735, 12.721492224,
12.771738615, 12.822260367, 12.873978099, 12.923227915, 12.974567607,
13.025770448, 13.075069278, 13.128010097, 13.176756879, 13.225490267,
13.275210678, 13.327460415, 13.377245568, 13.427565442, 13.480206394,
13.531701093, 13.582942734, 13.634294081, 13.683198379, 13.734206582,
13.788302286, 13.843154494, 13.894021613, 13.943224657, 13.993044985,
14.042461523, 14.096946715, 14.14900589, 14.20063466, 14.211196745,
14.253395282, 14.303325663, 14.356659649, 14.413074136, 14.464604646,
14.517272678, 14.567619326, 14.620637942, 14.676135821, 14.728504125,
14.780923879, 14.834070033, 14.88805798, 14.938942139, 14.990469544,
15.044076032, 15.099953233, 15.156412884, 15.214235817, 15.271213138,
15.325798609, 15.378597177, 15.438135445, 15.491867021, 15.543077721,
15.599954339, 15.655099226, 15.708897669, 15.766123228, 15.824043884,
15.879576539, 15.939120348, 15.992220692, 16.044966872, 16.101306686,
16.161617882, 16.216714661, 16.272656164, 16.329363337, 16.387933775,
16.449113047, 16.505669995, 16.555903753, 16.610588781, 16.672664881,
16.729278013, 16.78697127, 16.838599093, 16.895893754, 16.956822843,
17.018039892, 17.084414287, 17.143379198, 17.200685619, 17.259229469,
17.321749126, 17.381149615, 17.443208858, 17.503905954, 17.560281903,
17.614933678, 17.681351405, 17.74221697, 17.802667037, 17.858522943,
17.917738437, 17.97863753, 18.037817944, 18.098155391, 18.159425683,
18.211840657, 18.27526074, 18.33692478, 18.396258526, 18.458238818,
18.514456863, 18.573363424, 18.636278843, 18.698026625, 18.758652828,
18.819790183, 18.885907793, 18.947757511, 19.015342568, 19.071780516,
19.130766847, 19.190779655, 19.250576806, 19.307703312, 19.37973949,
19.444307555, 19.513320431, 19.578422798, 19.641058076, 19.701525268,
19.765950912, 19.831485681, 19.898244367, 19.970120491, 20.040461079,
20.113085378, 20.182634957, 20.250021823, 20.316093874, 20.385526512,
20.452960709, 20.528490799, 20.590835456, 20.657576237, 20.734601163,
20.796432382, 20.857074999, 20.924296268, 20.981758081, 21.050520711,
21.101586835, 21.169267627, 21.23809613, 21.307697124, 21.375825204,
21.439853831, 21.505956285, 21.573004444, 21.643790428, 21.712069434,
21.778388207, 21.849227372, 21.922472106, 21.99394987, 22.067064639,
22.133753656, 22.206486166, 22.279696782, 22.353360427, 22.425205731,
22.503982638, 22.588001214, 22.660306287, 22.719754145, 22.784789866,
22.845650145, 22.917175247, 22.98864013, 23.058736717, 23.126449875,
23.206097285, 23.281201599, 23.353807142, 23.438471345, 23.513137164,
23.581148347, 23.651792828, 23.720141593, 23.789544528, 23.865779724,
23.936527792, 24.020555006, 24.107845845, 24.199307866, 24.263359017,
24.347087384, 24.4346385, 24.507818781, 24.571974714, 24.646841362,
24.710487497, 24.783059591, 24.856796319, 24.950652825, 25.042550922,
25.123896877, 25.193670372, 25.274957094, 25.349462172, 25.435576491,
25.51316589, 25.586882888, 25.657849681, 25.756303689, 25.835507009,
25.907729142, 25.994210371, 26.075654291, 26.156078657, 26.237219479,
26.323004184, 26.392211925, 26.467561447, 26.538090853, 26.617539618,
26.686705331, 26.769525404, 26.856198329, 26.928412953, 27.013049688,
27.094563009, 27.172581464, 27.248735817, 27.326394553, 27.416558689,
27.431447543, 27.48948758, 27.56575701, 27.637779629, 27.727198467,
27.819583175, 27.89072311, 27.967084764, 28.044717681, 28.132956826,
28.206619771, 28.28977456, 28.38057562, 28.457359878, 28.535054562,
28.609218937, 28.70687242, 28.798699172, 28.87704667, 28.94972847,
29.045949652, 29.116868084, 29.207316102, 29.281657173, 29.365005391,
29.45307, 29.538765423, 29.609066235, 29.666093065, 29.747033087,
29.823210163, 29.910621168, 29.985310853, 30.069042216, 30.193806117,
30.286142091, 30.349338339, 30.433071836, 30.533425606, 30.629249697,
30.718788208, 30.816179692, 30.931175554, 31.021387436, 31.110903951,
31.202371237, 31.271299302, 31.379835205, 31.486582745, 31.570969892,
31.68538211, 31.781328558, 31.863820266, 31.947949911, 32.028391635,
32.103321532, 32.171981007, 32.24530075, 32.333256798, 32.443546143,
32.53381135, 32.617076506, 32.725431383, 32.810984415, 32.910608235,
32.973158191, 33.088711761, 33.192357766, 33.283193512, 33.371393106,
33.475948005, 33.559347219, 33.658391025, 33.750009475, 33.818083105,
33.90824308, 33.9838755, 34.080877763, 34.146295692, 34.250507807,
34.371608214, 34.476316879, 34.533721491, 34.640746452, 34.749101717,
34.863275024, 34.964454776, 35.050573054, 35.123931035, 35.207383079,
35.295835004, 35.380436979, 35.47389825, 35.607238842, 35.703077944,
35.777426108, 35.880933773, 35.964963294, 36.074860653, 36.24639341,
36.308937188, 36.412852652, 36.520897541, 36.614058124, 36.678693471,
36.825285754, 36.911731444, 37.00304522, 37.103617561, 37.240839212,
37.324188178, 37.442605899, 37.594380244, 37.689401444, 37.783864425,
37.868888391, 37.965533896, 38.032190738, 38.121662374, 38.216231759,
38.32931494, 38.485550915, 38.587984541, 38.741413355, 38.834721886,
38.942599059, 39.049938067, 39.157737534, 39.275502717, 39.397776124,
39.481108636, 39.566361926, 39.655743772, 39.761800044, 39.846485686,
39.925713715, 40.038230526, 40.164410971, 40.261000263, 40.372527731,
40.472335121, 40.583788392, 40.670698911, 40.791787272, 40.912643027,
41.104750893, 41.228516347, 41.342489963, 41.511861814, 41.608119358,
41.780778758, 41.926089127, 42.081861777, 42.202222703, 42.264025815,
42.445242848, 42.541322329, 42.657827999, 42.77568877, 42.823277773,
42.935205122, 43.025586067, 43.129570971, 43.205017138, 43.271345137,
43.462115004, 43.60013578, 43.756092841, 43.938932433, 44.149401556,
44.264182962, 44.434910694, 44.581969545, 44.703570962, 44.796600856,
44.947405349, 45.089628008, 45.267919786, 45.36112533, 45.608989372,
45.715181997, 45.8500343, 45.944624804, 46.131300224, 46.258716502,
46.355977484, 46.438045156, 46.595332139, 46.667334279, 46.775681755,
46.91836877, 47.006755457, 47.123090488, 47.308801367, 47.535576793,
47.649553016, 47.781304591, 47.898501697, 47.999404392, 48.102567936,
48.25556793, 48.450372783, 48.504637579, 48.651699065, 48.69032531,
48.862987574, 48.997370434, 49.128709174, 49.222352914, 49.44965026,
49.58989109, 49.774413399, 49.893380294, 49.988531117, 50.050008469,
50.108822427, 50.204190817, 50.307292073, 50.522654511, 50.612358502,
50.677841791, 50.869632516, 51.050259049, 51.108077675, 51.204289716,
51.387720181, 51.542672449, 51.620595912, 51.782608466, 51.891135673,
52.021941434, 52.141056866, 52.208811497, 52.369019918, 52.491592533,
52.690732579, 52.913379394, 52.986028544, 53.027035955, 53.077566526,
53.185383664, 53.243235923, 53.299567343, 53.414008843, 53.564701117,
53.883016079, 53.950198336, 54.054204705, 54.206611743, 54.376925407,
54.43146155, 54.457764314, 54.621225025, 54.77791546, 54.846948549,
54.982962435, 55.076132239, 55.186184502, 55.406078433, 55.517433791,
55.542549545, 55.652733031, 55.857984019, 56.002805709, 56.084355346,
56.141074618, 56.657298498, 56.785265565, 56.939642344, 56.998013267,
57.040987899, 57.309535748, 57.337197562, 57.36815285, 57.555620079,
57.668623466, 57.69157889, 58.067514642, 58.287528205, 58.360098721,
58.585726772, 58.802189428, 59.020926565, 59.120140031, 59.229413804,
59.37564499, 59.509482713, 59.893036155, 59.935568408, 60.160397592,
60.602575441, 60.84460666, 60.981398125, 61.181194777, 61.308163952,
61.359593739, 61.684023446, 61.889594463, 61.944413133, 61.964160486,
62.303363846, 62.404444046, 62.506208328, 62.541168016, 62.826164005,
62.853688955, 63.045562184, 63.206955035, 63.243528429, 64.06303519,
64.171759143, 64.232523107, 64.337782626, 64.349614158, 64.368753581,
64.430457602, 64.500135056, 64.68965015, 64.82217617, 64.877767252,
65.057042406, 65.379512885, 65.474871588, 65.983938144, 66.257717242,
66.285394449, 66.383499839, 66.386942025, 66.607800951, 66.661988401,
66.753245688, 66.897145459, 66.953560625, 67.008028367, 67.115040522,
67.77741031, 67.787354346, 67.973519995, 68.195748441, 68.239942723,
68.248092333), "800" = c(2.1893e-05, 2.1902e-05, 2.2009e-05,
2.266e-05, 2.2676e-05, 2.2721e-05, 2.2914e-05, 2.3017e-05, 2.3423e-05,
2.345e-05, 2.3513e-05, 2.3763e-05, 2.3792e-05, 2.4215e-05, 2.4554e-05,
2.5176e-05, 2.5287e-05, 2.5443e-05, 2.5991e-05, 2.5997e-05, 2.6229e-05,
2.6301e-05, 2.6455e-05, 2.6556e-05, 2.7122e-05, 2.7165e-05, 2.7312e-05,
2.7466e-05, 2.8109e-05, 2.8329e-05, 2.858e-05, 2.9273e-05, 2.9941e-05,
3.0024e-05, 3.0363e-05, 3.0808e-05, 3.1331e-05, 3.1931e-05, 3.2726e-05,
3.3217e-05, 3.3476e-05, 3.3758e-05, 3.3946e-05, 3.4071e-05, 3.4556e-05,
3.522e-05, 3.5651e-05, 3.5973e-05, 3.6554e-05, 3.6954e-05, 3.7381e-05,
3.7978e-05, 3.8078e-05, 3.8399e-05, 3.8886e-05, 3.9406e-05, 4.0024e-05,
4.0122e-05, 4.0319e-05, 4.0933e-05, 4.277e-05, 4.323e-05, 4.3591e-05,
4.4218e-05, 4.431e-05, 4.4524e-05, 4.5251e-05, 4.5416e-05, 4.5876e-05,
4.6116e-05, 4.6365e-05, 4.6688e-05, 4.7616e-05, 4.7863e-05, 4.9092e-05,
4.9639e-05, 5.0232e-05, 5.0554e-05, 5.0733e-05, 5.1021e-05, 5.1514e-05,
5.1771e-05, 5.2004e-05, 5.2709e-05, 5.3655e-05, 5.4334e-05, 5.4747e-05,
5.4906e-05, 5.5313e-05, 5.665e-05, 5.7395e-05, 5.8157e-05, 5.8722e-05,
5.9079e-05, 5.9402e-05, 6.0363e-05, 6.0897e-05, 6.1575e-05, 6.334e-05,
6.3806e-05, 6.459e-05, 6.5698e-05, 6.6313e-05, 6.6872e-05, 6.7939e-05,
6.8746e-05, 6.9874e-05, 7.0824e-05, 7.169e-05, 7.2554e-05, 7.3318e-05,
7.5074e-05, 7.6365e-05, 7.7243e-05, 7.7418e-05, 7.7963e-05, 7.9888e-05,
8.0361e-05, 8.1935e-05, 8.4886e-05, 8.5553e-05, 8.653e-05, 8.6842e-05,
8.9222e-05, 9.0786e-05, 9.1307e-05, 9.2643e-05, 9.3598e-05, 9.5172e-05,
9.6337e-05, 9.7605e-05, 9.8755e-05, 9.9836e-05, 0.000100424,
0.000103086, 0.000104228, 0.000106217, 0.000107273, 0.000109191,
0.000110832, 0.000111734, 0.000112587, 0.000113577, 0.000115047,
0.000116802, 0.000117927, 0.000118735, 0.000119217, 0.000120207,
0.000121418, 0.000122531, 0.00012359, 0.000124389, 0.000125864,
0.000126465, 0.000127486, 0.000128304, 0.000130423, 0.000131529,
0.000132888, 0.00013391, 0.00013503, 0.000136359, 0.000137481,
0.000139368, 0.000139933, 0.000140946, 0.000143432, 0.000144724,
0.000146572, 0.000148128, 0.000149323, 0.000150918, 0.000153289,
0.000154081, 0.000155665, 0.000158201, 0.00015933, 0.000162022,
0.000164503, 0.000165915, 0.000167519, 0.000169083, 0.00017096,
0.00017251, 0.000173973, 0.000176431, 0.000179508, 0.000180385,
0.000184865, 0.000185983, 0.000187985, 0.000189375, 0.000190749,
0.000191786, 0.000194657, 0.000196482, 0.000197806, 0.000198736,
0.000201775, 0.00020399, 0.0002057, 0.000209008, 0.000211062,
0.000213917, 0.000215152, 0.000217809, 0.000218658, 0.000220943,
0.000223293, 0.000226876, 0.000229599, 0.000232357, 0.000234291,
0.000236596, 0.000240029, 0.000242794, 0.000246294, 0.000250072,
0.000252709, 0.000255721, 0.000259856, 0.00026259, 0.000266243,
0.000267987, 0.000270875, 0.00027313, 0.000277023, 0.000281084,
0.00028328, 0.000285663, 0.00028816, 0.000290253, 0.000293171,
0.000296414, 0.000299633, 0.00030456, 0.000308232, 0.000311586,
0.000315057, 0.000317871, 0.000321626, 0.00032407, 0.000327871,
0.000333027, 0.00033725, 0.00034093, 0.000344211, 0.000349868,
0.000354533, 0.000357877, 0.000362396, 0.000366756, 0.000369826,
0.000373055, 0.000376904, 0.000381521, 0.000385033, 0.000387353,
0.000390587, 0.000394571, 0.000398687, 0.000402215, 0.00040681,
0.000409704, 0.000414273, 0.000418482, 0.000425298, 0.000431251,
0.000434446, 0.000441245, 0.00044567, 0.000451293, 0.000456763,
0.000461034, 0.000466459, 0.000472551, 0.000476219, 0.000481997,
0.00048657, 0.000494044, 0.000498925, 0.00050474, 0.000510148,
0.000515132, 0.000520106, 0.000526909, 0.000533595, 0.000540787,
0.000546664, 0.00055316, 0.000559215, 0.000565504, 0.00056962,
0.000579005, 0.00058409, 0.000590503, 0.000597237, 0.000604523,
0.000612222, 0.000619853, 0.000625454, 0.000632735, 0.000641787,
0.000646966, 0.000655684, 0.000664097, 0.000670137, 0.000679031,
0.000688298, 0.000693453, 0.000701739, 0.000710478, 0.000718121,
0.000726332, 0.000733597, 0.000743798, 0.000754744, 0.00076243,
0.000772565, 0.000781084, 0.000789398, 0.000797051, 0.000805367,
0.000812958, 0.000820608, 0.000829814, 0.00083849, 0.000849068,
0.00085729, 0.000868924, 0.000882317, 0.000892775, 0.000905387,
0.0009142, 0.000926095, 0.000935957, 0.000946719, 0.000955582,
0.000964835, 0.000975535, 0.000978027, 0.000989677, 0.001002425,
0.001013647, 0.001025301, 0.001038343, 0.001049223, 0.001059754,
0.001074322, 0.001086267, 0.001096376, 0.001107606, 0.001121111,
0.00113392, 0.001148533, 0.001163598, 0.001177995, 0.001192658,
0.001205774, 0.001222305, 0.001232428, 0.001246027, 0.001259694,
0.001272485, 0.001286247, 0.001301557, 0.001318183, 0.001332928,
0.001346852, 0.00136382, 0.001380124, 0.001396616, 0.001412953,
0.001429477, 0.001448531, 0.00146338, 0.001479385, 0.001497569,
0.001513477, 0.001530031, 0.001545452, 0.001562343, 0.001582354,
0.001602998, 0.001624102, 0.001640631, 0.001660048, 0.001676688,
0.00169588, 0.00171578, 0.001735114, 0.001752758, 0.001772751,
0.001794321, 0.001814492, 0.001837852, 0.001859488, 0.001882692,
0.001907399, 0.001930547, 0.001953546, 0.001977064, 0.001997619,
0.002018371, 0.002042553, 0.002066635, 0.002091066, 0.00211045,
0.002132572, 0.002157048, 0.002181055, 0.002202417, 0.002228629,
0.002253783, 0.002284293, 0.002311309, 0.00233943, 0.002363998,
0.002393925, 0.002422803, 0.00245502, 0.002487043, 0.002515692,
0.002544325, 0.00257122, 0.002599607, 0.002629866, 0.002659069,
0.002689838, 0.002722049, 0.002752441, 0.002786692, 0.002818533,
0.002850075, 0.002883164, 0.00291698, 0.002949004, 0.002982972,
0.003021833, 0.003056928, 0.003093081, 0.003126405, 0.003161834,
0.003197792, 0.003236167, 0.003274769, 0.003314445, 0.003349982,
0.003391251, 0.003429288, 0.003463771, 0.003501516, 0.003543242,
0.003583591, 0.00362566, 0.003667097, 0.003712741, 0.00376017,
0.003801897, 0.003849918, 0.003890579, 0.003936804, 0.003978508,
0.004019743, 0.004069363, 0.004117139, 0.004163419, 0.004211549,
0.004257328, 0.004308134, 0.004360544, 0.004414358, 0.004465139,
0.004518793, 0.004570038, 0.00462457, 0.004682671, 0.004736812,
0.004789042, 0.004843169, 0.004899443, 0.004949941, 0.005003634,
0.005057592, 0.005110574, 0.005171471, 0.00523075, 0.00528807,
0.005346244, 0.005411608, 0.005476439, 0.005539967, 0.005601882,
0.005667017, 0.00573001, 0.005793323, 0.005860569, 0.005935708,
0.006008763, 0.00607753, 0.006149349, 0.006222253, 0.006298778,
0.006370354, 0.006445616, 0.006516818, 0.006588337, 0.006660636,
0.006747091, 0.006827054, 0.006901024, 0.006983115, 0.00706461,
0.007143028, 0.007232914, 0.007320005, 0.007401199, 0.0074872,
0.007576984, 0.007657817, 0.007749743, 0.007841012, 0.007933139,
0.008030785, 0.008129027, 0.008223166, 0.008327276, 0.008425086,
0.008524674, 0.008632296, 0.008733218, 0.008836098, 0.008937887,
0.009043222, 0.009152198, 0.009254235, 0.009367703, 0.009473852,
0.009575527, 0.009686481, 0.009769218, 0.009792217, 0.009902418,
0.010026022, 0.010146904, 0.010266542, 0.010384129, 0.01050974,
0.010635592, 0.010751319, 0.010873001, 0.011004728, 0.011135711,
0.011257777, 0.011385405, 0.011521332, 0.011666669, 0.011809837,
0.011943742, 0.012082127, 0.012222172, 0.012365701, 0.012509281,
0.012651573, 0.012797467, 0.012960157, 0.013112849, 0.013260049,
0.013415051, 0.013576135, 0.013738897, 0.013895495, 0.014039967,
0.014209334, 0.014379092, 0.014545903, 0.014700954, 0.014862792,
0.015034728, 0.015213007, 0.015393285, 0.01557109, 0.015752577,
0.015945395, 0.016136139, 0.016321526, 0.016514897, 0.016707772,
0.016908357, 0.017106486, 0.017304843, 0.017506921, 0.017712706,
0.017918313, 0.018118253, 0.018330266, 0.018549923, 0.018758818,
0.018982637, 0.019200861, 0.019416119, 0.019644887, 0.01986298,
0.020103442, 0.02034397, 0.020588293, 0.020827861, 0.021063946,
0.021307471, 0.021570898, 0.021816949, 0.022083201, 0.022343371,
0.022594253, 0.02287206, 0.023154642, 0.023434142, 0.023700869,
0.023976413, 0.024261397, 0.024538661, 0.024829309, 0.025125379,
0.025424826, 0.025711322, 0.026014637, 0.026318188, 0.026610296,
0.026927895, 0.027245928, 0.02755476, 0.027884533, 0.028208641,
0.0285382, 0.028864484, 0.029194034, 0.029542145, 0.029891379,
0.03023763, 0.030598527, 0.030963777, 0.031333738, 0.031691125,
0.032058322, 0.032443967, 0.032820665, 0.033193418, 0.033590285,
0.033989198, 0.034382465, 0.034774929, 0.035180574, 0.035595082,
0.036009271, 0.036459304, 0.03687094, 0.03729721, 0.037725616,
0.038170023, 0.038614859, 0.039070676, 0.039534443, 0.039990994,
0.04046947, 0.040967968, 0.041444551, 0.041927187, 0.042403978,
0.042900614, 0.043409676, 0.043905071, 0.044410698, 0.044944462,
0.045484966, 0.046019253, 0.046560907, 0.047083615, 0.047625864,
0.048188976, 0.048754097, 0.049328825, 0.049899915, 0.050467578,
0.051077114, 0.05166214, 0.052268136, 0.052884235, 0.053503218,
0.054135817, 0.054794348, 0.055418525, 0.056097142, 0.056767614,
0.057433286, 0.058122365, 0.058812838, 0.059538491, 0.060245854,
0.060946019, 0.061653976, 0.062390087, 0.063139449, 0.063905329,
0.064656737, 0.065419071, 0.066182582, 0.06696832, 0.067764235,
0.068576976, 0.069394604, 0.07019258, 0.071055138, 0.071879723,
0.072715215, 0.073574886, 0.074413495, 0.075307484, 0.076177311,
0.077078194, 0.077989004, 0.078898069, 0.079817065, 0.080752404,
0.08170525, 0.082680811, 0.083659657, 0.084660089, 0.085650587,
0.086653397, 0.087694544, 0.088719524, 0.089765393, 0.090832988,
0.091929296, 0.09301539, 0.094131263, 0.095250453, 0.09636557,
0.097497818, 0.098642968, 0.099815722, 0.100741097, 0.100986829,
0.102184206, 0.103414452, 0.104628164, 0.105879026, 0.107133302,
0.108424716, 0.109708981, 0.111001187, 0.112330242, 0.113673553,
0.11506138, 0.116423135, 0.117813754, 0.119233761, 0.120642659,
0.122056329, 0.123517768, 0.124988957, 0.126461191, 0.127934509,
0.129467213, 0.131000046, 0.132560615, 0.134156971, 0.135765881,
0.137395538, 0.138999871, 0.140643275, 0.142339564, 0.144033827,
0.145748547, 0.147508402, 0.149280574, 0.151063694, 0.152885029,
0.154729303, 0.156582149, 0.158462153, 0.16038291, 0.162297565,
0.164263056, 0.166258044, 0.168312664, 0.170353174, 0.172373984,
0.174467498, 0.176545447, 0.178661064, 0.18088925, 0.183078598,
0.185295442, 0.187507078, 0.189790353, 0.192093644, 0.194379918,
0.19669623, 0.199086957, 0.20150316, 0.20394855, 0.206473153,
0.208999474, 0.211566514, 0.214127965, 0.216744224, 0.219370674,
0.2220666, 0.224777677, 0.227538004, 0.230321505, 0.233185515,
0.236092347, 0.238981513, 0.241908837, 0.244921732, 0.247913854,
0.250960539, 0.254054172, 0.257205407, 0.260347177, 0.263557157,
0.266853651, 0.270136154, 0.273482276, 0.276883861, 0.280325081,
0.283840042, 0.287310663, 0.290894022, 0.294518394, 0.298186838,
0.301876385, 0.305627291, 0.309459209, 0.313306182, 0.317227568,
0.31810622, 0.321232361, 0.325252638, 0.329343622, 0.33343904,
0.33771673, 0.341939035, 0.346262294, 0.350638607, 0.355083949,
0.359585177, 0.364173845, 0.368796446, 0.373510116, 0.378203576,
0.382990266, 0.387798296, 0.392750128, 0.397764329, 0.402821634,
0.407967615, 0.413165566, 0.418395838, 0.423742201, 0.429214152,
0.434788661, 0.435947563, 0.440380609, 0.44605069, 0.45181447,
0.45765743, 0.463628461, 0.469656492, 0.475781671, 0.481964566,
0.488229742, 0.494605222, 0.501064905, 0.507597788, 0.514243547,
0.520972875, 0.527792493, 0.534740625, 0.541748744, 0.548977472,
0.55625362, 0.563658153, 0.571142141, 0.578844812, 0.586676595,
0.59446735, 0.602515793, 0.610628109, 0.618843446, 0.62719823,
0.635679778, 0.644317963, 0.653088605, 0.66206949, 0.67114144,
0.680290568, 0.689714788, 0.693715706, 0.699205616, 0.708831122,
0.718643229, 0.728607294, 0.738789671, 0.749037416, 0.759507997,
0.770107344, 0.780814381, 0.791757493, 0.802913037, 0.814388577,
0.826054603, 0.837820779, 0.84982377, 0.862095671, 0.874630872,
0.8873844, 0.900359476, 0.913451738, 0.926766185, 0.940384575,
0.95419243, 0.968408647, 0.98275778, 0.988795118, 0.997327417,
1.012415841, 1.027758592, 1.043240467, 1.059245802, 1.075472089,
1.092016025, 1.108829724, 1.12614191, 1.143620887, 1.161456298,
1.179829906, 1.198374782, 1.217455712, 1.236858219, 1.256840163,
1.277297237, 1.298269818, 1.319367771, 1.336728906, 1.376043392,
1.398217521, 1.420067403, 1.442151297, 1.464159984, 1.48617226,
1.508023269, 1.529902955, 1.551905604, 1.573671871, 1.595480024,
1.617231801, 1.639158823, 1.660872542, 1.682773073, 1.704677541,
1.726515544, 1.748398169, 1.761260373, 1.770217205, 1.792231389,
1.814008731, 1.83597056, 1.857833336, 1.879722705, 1.901778315,
1.923717778, 1.945535539, 1.967571615, 1.989602417, 2.011643036,
2.033444537, 2.055292805, 2.077319108, 2.099385655, 2.121316178,
2.14325926, 2.165251789, 2.187053153, 2.209248578, 2.231237312,
2.253088568, 2.275088607, 2.297119967, 2.309767155, 2.31921637,
2.341216257, 2.363280221, 2.385120609, 2.407158119, 2.429176811,
2.451144148, 2.473386666, 2.495501082, 2.517585983, 2.539896478,
2.562060141, 2.584042276, 2.606360476, 2.628522692, 2.650842902,
2.672808473, 2.694698735, 2.717006377, 2.739196552, 2.761113627,
2.78336799, 2.805639758, 2.827843603, 2.850219248, 2.872549379,
2.895181632, 2.917627156, 2.940019478, 2.962280424, 2.984832201,
3.007365256, 3.029681056, 3.052329001, 3.074827523, 3.092300771,
3.096874714, 3.119575495, 3.142136539, 3.164588553, 3.186810249,
3.209313801, 3.232024346, 3.254615378, 3.277063208, 3.299862458,
3.322566339, 3.345330293, 3.368014196, 3.39088405, 3.413524629,
3.436075156, 3.458880349, 3.481891559, 3.504861983, 3.528040224,
3.55109573, 3.57391785, 3.596861112, 3.620123357, 3.643165536,
3.661131955, 3.666145579, 3.688867259, 3.712041864, 3.735238996,
3.758247082, 3.781165398, 3.804362563, 3.827442807, 3.850660684,
3.874025317, 3.897369852, 3.921043839, 3.94458255, 3.968156226,
3.991506999, 4.015172432, 4.038670573, 4.062266609, 4.085533913,
4.109116041, 4.132775056, 4.156717224, 4.180188832, 4.204401977,
4.228159823, 4.251897218, 4.276002014, 4.29981368, 4.324016166,
4.347907391, 4.371936769, 4.395820556, 4.419589814, 4.443979471,
4.468068928, 4.492329479, 4.516321934, 4.540715617, 4.564998548,
4.589390031, 4.613919575, 4.637869153, 4.66226927, 4.686662101,
4.711917469, 4.736846053, 4.760957669, 4.785995667, 4.810252856,
4.835027217, 4.860195484, 4.885270147, 4.909938551, 4.9347809,
4.959488489, 4.984315386, 5.009385868, 5.034513536, 5.059719777,
5.084988269, 5.11017069, 5.13537018, 5.160499257, 5.185999059,
5.211067067, 5.236926146, 5.262136134, 5.287822919, 5.313337418,
5.338841438, 5.364197051, 5.389806563, 5.415768144, 5.442181049,
5.4679655, 5.493675047, 5.519392774, 5.545573632, 5.57169423,
5.598234637, 5.62427682, 5.650839368, 5.677492878, 5.704290346,
5.731379446, 5.757543187, 5.784113581, 5.810768616, 5.83734246,
5.863921935, 5.890431601, 5.917054645, 5.944186162, 5.970959458,
5.99881401, 6.025396998, 6.03089898, 6.052340459, 6.079771825,
6.106655769, 6.134683647, 6.162407944, 6.190608383, 6.218316948,
6.245662719, 6.273890775, 6.302184973, 6.329999975, 6.357144692,
6.384763923, 6.412369835, 6.440686528, 6.46952286, 6.498057177,
6.526283145, 6.5543846, 6.583585695, 6.612140952, 6.640366961,
6.668816972, 6.697844948, 6.726909172, 6.75675271, 6.785967402,
6.815588411, 6.844665528, 6.874757375, 6.904090732, 6.933740514,
6.963282194, 6.993320614, 7.02289244, 7.051698685, 7.080798593,
7.11009493, 7.139514759, 7.169046369, 7.198828947, 7.228328411,
7.258845106, 7.288239995, 7.318815319, 7.349604464, 7.379561882,
7.409312393, 7.44018222, 7.470760318, 7.502016472, 7.533623656,
7.56535796, 7.596207331, 7.628385955, 7.659384824, 7.691742835,
7.723026167, 7.754222882, 7.786070681, 7.817886813, 7.84997606,
7.88123126, 7.911936023, 7.94496744, 7.97645191, 8.008498446,
8.040909145, 8.073808722, 8.105421479, 8.138219601, 8.171721295,
8.205119666, 8.237791944, 8.270105585, 8.303522463, 8.337566954,
8.372361125, 8.406577199, 8.439260106, 8.473503932, 8.507778883,
8.541934327, 8.575753034, 8.608879293, 8.642501161, 8.676440446,
8.71189948, 8.743923165, 8.777167537, 8.812294972, 8.847180873,
8.881888556, 8.917488393, 8.952729018, 8.98716833, 9.022540719,
9.057842104, 9.092485723, 9.126744044, 9.160402521, 9.195788719,
9.23116562, 9.265794187, 9.301202475, 9.338162972, 9.373484655,
9.408353611, 9.444393176, 9.479036449, 9.517213353, 9.553721606,
9.589615136, 9.626004982, 9.660891424, 9.697208687, 9.736625667,
9.773432944, 9.810521077, 9.847454873, 9.885706029, 9.92339542,
9.961316879, 9.998292698, 10.035453688, 10.071997963, 10.109658308,
10.148422956, 10.188809961, 10.226848413, 10.265468258, 10.303608119,
10.341843931, 10.382266285, 10.419944273, 10.459886116, 10.499809752,
10.53804072, 10.577180535, 10.616327221, 10.655973896, 10.695804431,
10.735939387, 10.776561902, 10.815704153, 10.854146661, 10.893888875,
10.933147962, 10.970662027, 11.011336718, 11.049700464, 11.090699247,
11.130086151, 11.168892563, 11.207605519, 11.247034156, 11.288067063,
11.327803575, 11.367341485, 11.405707197, 11.449320379, 11.489666557,
11.532128202, 11.573106372, 11.6130937, 11.656442836, 11.698949247,
11.738080526, 11.776481304, 11.81639912, 11.857269107, 11.899526503,
11.943497852, 11.985577006, 12.031009116, 12.076177133, 12.120860373,
12.165167843, 12.206572758, 12.251727446, 12.294394861, 12.33567683,
12.380802852, 12.420385568, 12.463337965, 12.508546474, 12.552369172,
12.596903923, 12.64190704, 12.685898756, 12.728717067, 12.770503197,
12.812961281, 12.857954526, 12.900435339, 12.946698689, 12.990528925,
13.035474722, 13.076479134, 13.120001498, 13.129291003, 13.166804176,
13.214732289, 13.257259047, 13.300120899, 13.346581536, 13.389411081,
13.431753643, 13.479022657, 13.522919358, 13.567301091, 13.611633486,
13.654955578, 13.700371539, 13.748485101, 13.794761316, 13.837445714,
13.883674373, 13.931823517, 13.979775264, 14.026028006, 14.068044696,
14.114379656, 14.157125707, 14.202818274, 14.253493894, 14.299955341,
14.342601009, 14.387253866, 14.437065298, 14.48426294, 14.535417093,
14.584451141, 14.633012911, 14.683969647, 14.729901888, 14.778521038,
14.825301348, 14.872281866, 14.919650535, 14.964202056, 15.015388536,
15.061473183, 15.109978299, 15.162076589, 15.207977559, 15.254847024,
15.306420235, 15.354703521, 15.403805688, 15.45452657, 15.503807076,
15.55395911, 15.60077864, 15.65117561, 15.698094951, 15.747333657,
15.789388212, 15.840168304, 15.892783305, 15.941581708, 15.987731409,
16.032250885, 16.084870703, 16.128949973, 16.180954568, 16.225403552,
16.27918709, 16.333171583, 16.379103747, 16.424273663, 16.47200581,
16.524443695, 16.57294203, 16.61893648, 16.666279146, 16.718970442,
16.768022838, 16.821661281, 16.873037859, 16.920961917, 16.972359901,
17.022553878, 17.069130372, 17.121323801, 17.175086044, 17.224240886,
17.275576121, 17.325636757, 17.379311066, 17.425406462, 17.476268892,
17.524224389, 17.570928519, 17.623968294, 17.673100885, 17.735475369,
17.78648834, 17.836768743, 17.895679392, 17.951777598, 18.002663724,
18.050458173, 18.101985601, 18.155806605, 18.210564194, 18.26108134,
18.311715875, 18.367538411, 18.417966355, 18.473257745, 18.523795652,
18.580403548, 18.641774827, 18.692901115, 18.741177582, 18.796262462,
18.843708323, 18.900377316, 18.952654955, 19.008196936, 19.059718504,
19.122322895, 19.179454519, 19.234001031, 19.298921026, 19.350365439,
19.408375965, 19.466766158, 19.521940104, 19.584523307, 19.643518093,
19.699371327, 19.751746404, 19.807248852, 19.861385732, 19.919336425,
19.964625857, 20.021566625, 20.082580667, 20.147304452, 20.198719785,
20.25463039, 20.318721352, 20.387048411, 20.436274627, 20.502425483,
20.555513586, 20.611285269, 20.664885752, 20.720924399, 20.782465444,
20.836368847, 20.886097231, 20.946164748, 21.012435199, 21.061313386,
21.125194834, 21.184505601, 21.2526885, 21.308315856, 21.377800146,
21.441556162, 21.49773483, 21.560168803, 21.620036136, 21.676517118,
21.73768738, 21.799022031, 21.855883578, 21.907625016, 21.965223719,
22.03003566, 22.087747099, 22.144148411, 22.199816977, 22.258942125,
22.311589622, 22.37880356, 22.43149822, 22.488480501, 22.5560014,
22.626635855, 22.691870717, 22.751040183, 22.809056696, 22.877029657,
22.948884475, 22.996738283, 23.061693537, 23.121182578, 23.188859441,
23.265200517, 23.325813327, 23.390584041, 23.454188879, 23.520306235,
23.569458652, 23.629235055, 23.687301112, 23.735821938, 23.751209156,
23.805083681, 23.873991355, 23.946291476, 24.011013453, 24.071211737,
24.136030259, 24.193533708, 24.247043543, 24.317916094, 24.399933772,
24.475720584, 24.541753875, 24.617121843, 24.677084832, 24.746204466,
24.811074628, 24.882933062, 24.932799523, 24.9999446, 25.06692317,
25.132236244, 25.214356774, 25.274836467, 25.328806137, 25.388712242,
25.446382589, 25.543938289, 25.622247082, 25.696728923, 25.762632945,
25.814579855, 25.901158009, 25.986216869, 26.068179941, 26.13950064,
26.199231532, 26.25940674, 26.329825266, 26.40741818, 26.479498323,
26.567216891, 26.640856721, 26.707558594, 26.778486161, 26.840121213,
26.938788293, 27.008949806, 27.068263185, 27.148098514, 27.216085759,
27.275838226, 27.339299482, 27.40412306, 27.461528178, 27.535828248,
27.601984676, 27.688922761, 27.734290921, 27.800452411, 27.877709499,
27.984265425, 28.037263663, 28.113821418, 28.17659324, 28.265240422,
28.322455181, 28.421412092, 28.496773782, 28.573469022, 28.66708372,
28.730393547, 28.820229358, 28.880702905, 28.937323379, 29.0129735,
29.091727321, 29.150947972, 29.232407726, 29.317418694, 29.397422841,
29.475959645, 29.548809522, 29.602744795, 29.662559457, 29.738393099,
29.813686476, 29.890528322, 29.960826028, 30.036489766, 30.127508908,
30.221089042, 30.326222711, 30.4312754, 30.481502771, 30.553316856,
30.626831952, 30.679407996, 30.761744989, 30.873191181, 30.941610863,
31.043958544, 31.117211014, 31.194543852, 31.27264059, 31.326284112,
31.393054991, 31.474399505, 31.548640457, 31.62055429, 31.713766659,
31.790015963, 31.864970275, 31.943787716, 32.013027647, 32.082004603,
32.172656065, 32.236935787, 32.331670505, 32.38689445, 32.472647589,
32.527415867, 32.571175348, 32.623539798, 32.724366924, 32.808395994,
32.943335333, 33.036537919, 33.138722223, 33.234695947, 33.376142128,
33.475178065, 33.542500899, 33.617535227, 33.713382832, 33.805022864,
33.852138516, 33.949673954, 34.08727226, 34.15206706, 34.230577019,
34.324676089, 34.450086268, 34.542624543, 34.661693727, 34.704420309,
34.776526094, 34.867141622, 34.918434223, 35.036851472, 35.106893738,
35.205392236, 35.306328384, 35.39064962, 35.497487722, 35.581798905,
35.705251468, 35.768072277, 35.816987147, 35.882582127, 35.963577159,
36.086917595, 36.197663775, 36.287173786, 36.367969766, 36.49583745,
36.588924984, 36.708882785, 36.776223405, 36.856368345, 36.966322878,
37.013421774, 37.123437984, 37.200615883, 37.322401878, 37.405612585,
37.51548204, 37.680763231, 37.787490689, 37.874224758, 37.971573045,
38.027378982, 38.092702755, 38.171529377, 38.257456401, 38.293416601,
38.377716667, 38.454602588, 38.49786473, 38.594090799, 38.678686688,
38.744916819, 38.897789808, 39.042198317, 39.159105956, 39.203140667,
39.344081954, 39.451577432, 39.559252852, 39.615118976, 39.697713457,
39.865352439, 39.928894723, 39.961899338, 40.032429754, 40.124718314,
40.244781454, 40.357176818, 40.444321896, 40.519619418, 40.569466592,
40.651894292, 40.755092928, 41.002247576, 41.1237225, 41.209821951,
41.28446626, 41.331922788, 41.373546316, 41.464425185, 41.687163857,
41.81291491, 41.898025621, 41.921146752, 42.095364382, 42.18654113,
42.433314098, 42.603134744, 42.816127388, 42.940028346, 43.158734236,
43.188863885, 43.332898793, 43.447438733, 43.537000753, 43.568575732,
43.742660177, 43.857188127, 43.875025716, 43.946281275, 44.048399145,
44.152941363, 44.236576091, 44.303144586, 44.480316122, 44.559917156,
44.703988463, 44.79398793, 44.931778421, 45.015029062, 45.215586777,
45.260966668, 45.437088807, 45.497420054, 45.549476122, 45.591530916,
45.699767495, 45.771613363, 45.840251636, 45.96468841, 46.123086025,
46.286149339, 46.310877762, 46.355604313, 46.449315881, 46.587356908,
46.680996919, 46.738661737, 46.829714851, 46.940102732, 47.033280514,
47.10112526, 47.160480507, 47.245722598, 47.450569272, 47.487882139,
47.498798053, 47.69435473, 47.817997451, 47.863719194, 47.951326334,
48.151659967, 48.281375044, 48.401229452, 48.710408317, 48.766256927,
48.887501988, 48.96643506, 49.039796998, 49.28902341, 49.431882573,
49.474275807, 49.48343802, 49.49926844, 49.684161369, 49.718410024,
49.888151571, 50.004911515, 50.111569239, 50.349143357, 50.392764952,
50.527973502, 50.551497006, 50.612023374, 50.678599284, 50.680105259,
50.773608071, 50.998386953, 51.051960335, 51.075112178, 51.160680186,
51.297643482, 51.464863223, 51.597712248, 51.631321181, 51.82144497,
51.88745397, 52.063125792, 52.09622365, 52.293769355, 52.330576047,
52.473793829, 52.485810909, 52.536092517, 52.573066687, 52.618268756,
52.673426782, 52.704436463, 52.909786321, 53.082462484, 53.285493357,
53.307505053, 53.311451692, 53.380818928, 53.408503812, 53.484401825,
53.504780051, 53.584312313, 53.859344723, 53.935241806, 54.020528083
), "1000" = c(1.8831e-05, 1.9142e-05, 1.9501e-05, 1.9818e-05,
1.984e-05, 1.989e-05, 2.002e-05, 2.0128e-05, 2.0695e-05, 2.0749e-05,
2.0791e-05, 2.1032e-05, 2.1252e-05, 2.1517e-05, 2.1607e-05, 2.2275e-05,
2.255e-05, 2.2938e-05, 2.3004e-05, 2.3188e-05, 2.3553e-05, 2.3672e-05,
2.3967e-05, 2.4204e-05, 2.4366e-05, 2.4602e-05, 2.4758e-05, 2.4956e-05,
2.506e-05, 2.5428e-05, 2.5607e-05, 2.5775e-05, 2.6001e-05, 2.6275e-05,
2.6546e-05, 2.6823e-05, 2.6963e-05, 2.7428e-05, 2.7757e-05, 2.7868e-05,
2.7913e-05, 2.812e-05, 2.8587e-05, 2.861e-05, 2.9445e-05, 3.0006e-05,
3.0374e-05, 3.0438e-05, 3.0765e-05, 3.1069e-05, 3.1672e-05, 3.2053e-05,
3.3189e-05, 3.335e-05, 3.3738e-05, 3.3764e-05, 3.433e-05, 3.4492e-05,
3.4954e-05, 3.5349e-05, 3.5812e-05, 3.6456e-05, 3.6747e-05, 3.7421e-05,
3.7733e-05, 3.8301e-05, 3.8516e-05, 3.8866e-05, 3.9533e-05, 3.9715e-05,
4.1151e-05, 4.1309e-05, 4.1679e-05, 4.2395e-05, 4.3754e-05, 4.4305e-05,
4.4629e-05, 4.5262e-05, 4.5881e-05, 4.6338e-05, 4.7329e-05, 4.7559e-05,
4.8243e-05, 4.8524e-05, 4.9008e-05, 4.9198e-05, 4.9426e-05, 5.0106e-05,
5.0154e-05, 5.0922e-05, 5.1278e-05, 5.1687e-05, 5.2189e-05, 5.2752e-05,
5.3973e-05, 5.4566e-05, 5.5089e-05, 5.5328e-05, 5.6563e-05, 5.7362e-05,
5.8426e-05, 5.8924e-05, 5.963e-05, 5.9902e-05, 6.1248e-05, 6.2351e-05,
6.2799e-05, 6.4531e-05, 6.489e-05, 6.5372e-05, 6.606e-05, 6.6745e-05,
6.7731e-05, 6.826e-05, 6.9252e-05, 6.9701e-05, 7.0865e-05, 7.1508e-05,
7.2541e-05, 7.3932e-05, 7.4465e-05, 7.54e-05, 7.572e-05, 7.6541e-05,
7.7678e-05, 7.8778e-05, 7.9052e-05, 7.9984e-05, 8.088e-05, 8.1908e-05,
8.3424e-05, 8.5262e-05, 8.5938e-05, 8.6851e-05, 8.7787e-05, 8.8534e-05,
8.9194e-05, 9.0233e-05, 9.1437e-05, 9.2168e-05, 9.2632e-05, 9.4028e-05,
9.5333e-05, 9.6002e-05, 9.6639e-05, 9.9184e-05, 9.9853e-05, 0.000101483,
0.000102473, 0.000103331, 0.000105185, 0.000106685, 0.000109144,
0.000111409, 0.000111834, 0.000113906, 0.000114641, 0.000116536,
0.000117921, 0.000119041, 0.000119614, 0.000120629, 0.000122626,
0.000124721, 0.000125783, 0.00012769, 0.000128977, 0.000130377,
0.00013174, 0.000132896, 0.000134183, 0.000136036, 0.000137511,
0.000140106, 0.000142088, 0.000144223, 0.000146098, 0.000147427,
0.000148375, 0.000149752, 0.000151733, 0.000154652, 0.000156811,
0.000158378, 0.000159743, 0.000161676, 0.000164777, 0.000166027,
0.000167649, 0.000170659, 0.00017247, 0.000173772, 0.000176016,
0.000177024, 0.000179362, 0.000182079, 0.000183315, 0.000185916,
0.000187875, 0.000190079, 0.000193504, 0.000195946, 0.000198063,
0.000200456, 0.000202556, 0.000205594, 0.000207543, 0.000210429,
0.00021438, 0.000219016, 0.000221449, 0.000224224, 0.00022742,
0.000229111, 0.000231316, 0.000233457, 0.000235678, 0.000237937,
0.000241682, 0.000244393, 0.000247127, 0.000250042, 0.00025319,
0.000257905, 0.000261873, 0.000265739, 0.000268935, 0.000273918,
0.000275916, 0.000279777, 0.000283646, 0.000287528, 0.000291418,
0.00029479, 0.000299933, 0.000302859, 0.000305932, 0.000309156,
0.000312037, 0.000316222, 0.000319999, 0.000323641, 0.000326283,
0.000330831, 0.000335766, 0.000340369, 0.000344423, 0.000347875,
0.000351407, 0.000355255, 0.000360127, 0.000364714, 0.00036694,
0.000370827, 0.000375966, 0.000380953, 0.00038506, 0.000389836,
0.000394696, 0.00039932, 0.000404635, 0.000408629, 0.000412333,
0.000417637, 0.000423393, 0.000428976, 0.000434426, 0.000439901,
0.000444721, 0.000449745, 0.000455535, 0.000459099, 0.00046408,
0.000470056, 0.000473278, 0.000477708, 0.000483448, 0.0004886,
0.000493002, 0.000499561, 0.000505128, 0.000511285, 0.000516145,
0.000520792, 0.000528501, 0.000533625, 0.000539173, 0.000545329,
0.000551721, 0.000559162, 0.000564951, 0.000568665, 0.00057613,
0.000582756, 0.000588679, 0.000594392, 0.000599947, 0.000607402,
0.000614383, 0.000621601, 0.000631243, 0.00063795, 0.000646368,
0.000654538, 0.000661427, 0.000668156, 0.000672878, 0.0006776,
0.000685749, 0.000692521, 0.000699704, 0.000708502, 0.00071609,
0.000726536, 0.000738112, 0.000746894, 0.00075553, 0.000764042,
0.000772826, 0.000778907, 0.000788746, 0.000797974, 0.000807398,
0.000817177, 0.000826255, 0.000835449, 0.000845968, 0.000855347,
0.000866616, 0.000879401, 0.000889529, 0.00089911, 0.000909027,
0.000919485, 0.000926885, 0.000938671, 0.000949347, 0.000957969,
0.000967435, 0.000978015, 0.000988909, 0.000992033, 0.001003493,
0.001016336, 0.001026553, 0.001038261, 0.001051702, 0.00106524,
0.001076915, 0.001092405, 0.001107826, 0.00112223, 0.001137722,
0.001148658, 0.001162477, 0.001172969, 0.001188305, 0.001203701,
0.00121687, 0.001226662, 0.001241338, 0.001254894, 0.001268025,
0.001282884, 0.001297077, 0.001313568, 0.001330555, 0.001344908,
0.001357774, 0.001372637, 0.001387591, 0.001404824, 0.001420479,
0.001437448, 0.001453514, 0.001470401, 0.00148432, 0.001504585,
0.001521643, 0.001539319, 0.001556735, 0.001577152, 0.001598571,
0.001616988, 0.001634755, 0.001654204, 0.001673988, 0.001698089,
0.001719109, 0.001737325, 0.001757723, 0.001777745, 0.001801074,
0.001821901, 0.001842811, 0.001867895, 0.001887495, 0.00190771,
0.001931679, 0.001951296, 0.001975133, 0.001999261, 0.002022076,
0.002048604, 0.002071568, 0.002097553, 0.002120675, 0.002142932,
0.002170979, 0.002193384, 0.002217586, 0.002241877, 0.00226668,
0.002291301, 0.002318756, 0.002345017, 0.002373803, 0.002404089,
0.002429513, 0.0024558, 0.002484006, 0.002511594, 0.00253958,
0.002568832, 0.002602107, 0.002628354, 0.002661155, 0.002691263,
0.002725158, 0.002761141, 0.002792718, 0.002823887, 0.002858624,
0.002897944, 0.002932138, 0.002964582, 0.002995815, 0.003027581,
0.003058516, 0.003092977, 0.00312467, 0.003157857, 0.003191624,
0.003228286, 0.003263682, 0.003302119, 0.003336411, 0.003375771,
0.003418443, 0.003454369, 0.003492942, 0.003532497, 0.003572721,
0.003615988, 0.003658587, 0.003701982, 0.003741535, 0.003783262,
0.003825894, 0.003868944, 0.003913608, 0.003965671, 0.004013516,
0.004059136, 0.004110817, 0.004161111, 0.004206606, 0.004260255,
0.00430716, 0.004353003, 0.004408605, 0.00445696, 0.004505008,
0.004561379, 0.004616125, 0.004670491, 0.004722347, 0.00477513,
0.004827696, 0.00488731, 0.004941642, 0.004999087, 0.005057162,
0.005121535, 0.005178845, 0.005229103, 0.00528881, 0.005351526,
0.005414943, 0.005477281, 0.005536567, 0.005596472, 0.00566041,
0.005719861, 0.005792105, 0.005857393, 0.005922047, 0.005991064,
0.006060255, 0.006135529, 0.006203784, 0.006269676, 0.006341454,
0.006415918, 0.006488169, 0.006563139, 0.006636057, 0.006712959,
0.006787729, 0.006874638, 0.006954628, 0.007034086, 0.007122534,
0.007206399, 0.007280899, 0.007367742, 0.007445252, 0.007534555,
0.00762118, 0.007706134, 0.007792467, 0.007886938, 0.007980935,
0.008070705, 0.008155491, 0.008245538, 0.008343997, 0.008438387,
0.00853453, 0.008632708, 0.008737292, 0.008836311, 0.008932541,
0.009033401, 0.009142858, 0.009252902, 0.00935777, 0.009458801,
0.009572018, 0.009679469, 0.009787962, 0.009872206, 0.009893676,
0.010007352, 0.010122074, 0.010241251, 0.01036192, 0.010481401,
0.010603249, 0.010724845, 0.010841424, 0.010963673, 0.011096024,
0.011217766, 0.011352485, 0.011488823, 0.011625145, 0.011759818,
0.011898241, 0.01203948, 0.012178188, 0.012326735, 0.012479584,
0.01262155, 0.012763071, 0.012910439, 0.013067569, 0.013218536,
0.013377016, 0.01353656, 0.013693711, 0.013859333, 0.014016357,
0.014182133, 0.014351333, 0.01450924, 0.014672225, 0.014841737,
0.015013117, 0.015186573, 0.015358727, 0.015539616, 0.015717922,
0.015900128, 0.016091696, 0.016281045, 0.016480062, 0.016673505,
0.0168659, 0.017060086, 0.017254799, 0.017458814, 0.017646362,
0.017849777, 0.018059918, 0.018260021, 0.018470204, 0.018684829,
0.018896851, 0.019116701, 0.019343012, 0.019572058, 0.019789199,
0.020020568, 0.020253758, 0.020497761, 0.020735199, 0.020978864,
0.021205526, 0.021450934, 0.021706857, 0.021971348, 0.022221221,
0.022489461, 0.022749158, 0.023016282, 0.023283544, 0.023571905,
0.023839772, 0.024120188, 0.024406722, 0.024675413, 0.024966611,
0.025257132, 0.025551942, 0.025843156, 0.026155755, 0.026467078,
0.026775527, 0.027098007, 0.027421433, 0.027732527, 0.028074661,
0.028393065, 0.028728419, 0.029079931, 0.029424072, 0.029766285,
0.030114922, 0.030474926, 0.030833824, 0.031183639, 0.031562322,
0.031917815, 0.032271253, 0.032646235, 0.033028529, 0.033403773,
0.03380155, 0.034220253, 0.034615694, 0.03501507, 0.035404703,
0.035814454, 0.036228474, 0.036662526, 0.037082535, 0.037500522,
0.037925681, 0.038364523, 0.0388085, 0.039268765, 0.039715888,
0.040153152, 0.04062285, 0.041098779, 0.041559959, 0.042045771,
0.042546463, 0.043040514, 0.043532782, 0.044035527, 0.044551965,
0.045050116, 0.045587582, 0.046124441, 0.046673303, 0.047222311,
0.047778042, 0.048337713, 0.048917929, 0.049493394, 0.050073782,
0.05065817, 0.051244813, 0.051859501, 0.052456569, 0.053078514,
0.053706162, 0.054347497, 0.054969816, 0.055634717, 0.056309726,
0.056971701, 0.057639983, 0.058296249, 0.058968779, 0.059643094,
0.060341648, 0.061051603, 0.061766701, 0.062497118, 0.06323322,
0.063998163, 0.064768316, 0.065550574, 0.066321168, 0.067100079,
0.067910935, 0.068716914, 0.069530004, 0.070348212, 0.07117516,
0.072000484, 0.07285043, 0.073704265, 0.074551032, 0.075418101,
0.076307865, 0.077202923, 0.078095091, 0.079016835, 0.07998026,
0.080897274, 0.081833592, 0.082819843, 0.083794119, 0.084780507,
0.085779284, 0.086790461, 0.08780895, 0.088852945, 0.089873877,
0.090938873, 0.092020621, 0.093138692, 0.09425505, 0.095368387,
0.096487628, 0.097652789, 0.098785939, 0.099950163, 0.100901441,
0.101137481, 0.102349245, 0.103555948, 0.104781472, 0.106042275,
0.107329692, 0.108606793, 0.109886572, 0.111218699, 0.112512009,
0.113868176, 0.115221805, 0.116572641, 0.117954235, 0.119360882,
0.120805975, 0.122219298, 0.123707126, 0.125195441, 0.126695203,
0.128204094, 0.129742534, 0.131270745, 0.132840577, 0.134465507,
0.136088821, 0.137725215, 0.139351935, 0.141036667, 0.142711743,
0.144447093, 0.146164335, 0.147916855, 0.149686634, 0.151488183,
0.153332836, 0.155173166, 0.157020275, 0.158904769, 0.160820542,
0.162773242, 0.164713805, 0.166667584, 0.168700283, 0.170719035,
0.172774246, 0.174850698, 0.176982315, 0.179116289, 0.181252729,
0.183472267, 0.185675665, 0.187918043, 0.1902228, 0.192527271,
0.194859387, 0.197192606, 0.199620119, 0.202088932, 0.204540141,
0.2070694, 0.209569562, 0.212142651, 0.214745761, 0.217326347,
0.219998753, 0.222693178, 0.225410345, 0.228162505, 0.230982099,
0.233836495, 0.23669527, 0.239609791, 0.242555136, 0.24551827,
0.248564922, 0.251598282, 0.254718267, 0.257872649, 0.261051257,
0.264241438, 0.267461931, 0.270782302, 0.274145104, 0.277499684,
0.280941577, 0.284435378, 0.287956372, 0.291500312, 0.295126846,
0.298824974, 0.302519523, 0.306319398, 0.310100197, 0.313956537,
0.31789319, 0.318731663, 0.321826499, 0.325852808, 0.32995603,
0.334105016, 0.33834902, 0.342568574, 0.346889548, 0.351256979,
0.355650587, 0.360139307, 0.364717612, 0.369347188, 0.374038842,
0.378787035, 0.383602337, 0.388449325, 0.393447984, 0.398479601,
0.403559852, 0.408735618, 0.413952236, 0.419330541, 0.424699893,
0.43019176, 0.435751796, 0.436883562, 0.441345601, 0.447028115,
0.45282325, 0.458716948, 0.464722408, 0.470768486, 0.476865681,
0.48308393, 0.489411666, 0.495854893, 0.502407023, 0.509000476,
0.515702593, 0.522487383, 0.529422003, 0.536437217, 0.543536646,
0.55074706, 0.558053418, 0.565556492, 0.57308375, 0.580756646,
0.588575781, 0.596585307, 0.604650253, 0.612764635, 0.621006734,
0.629430251, 0.63795406, 0.646579683, 0.655417893, 0.664378938,
0.673551021, 0.682841359, 0.692254168, 0.696282608, 0.701794209,
0.711445434, 0.721210193, 0.731188987, 0.741376654, 0.751817116,
0.762421053, 0.773114687, 0.784061013, 0.795130969, 0.806488029,
0.817928589, 0.82958244, 0.841522499, 0.85352701, 0.865821494,
0.878428845, 0.891239808, 0.904248945, 0.917498537, 0.930919608,
0.944719735, 0.958724266, 0.973038783, 0.987622989, 0.993637177,
1.002297896, 1.017373358, 1.032772741, 1.048447606, 1.064422093,
1.080710692, 1.097502878, 1.114365096, 1.131641103, 1.149287043,
1.167325463, 1.185917776, 1.204847027, 1.223968012, 1.243615421,
1.263696718, 1.284154559, 1.305122204, 1.326626572, 1.344070649,
1.383839078, 1.405911746, 1.428047918, 1.45024018, 1.472109506,
1.494367297, 1.516382815, 1.538507228, 1.560497563, 1.582667983,
1.604648622, 1.626736448, 1.648938561, 1.671232193, 1.693445935,
1.715730761, 1.737893232, 1.760075171, 1.773090818, 1.782200007,
1.804348156, 1.826386877, 1.84853163, 1.870597324, 1.892718846,
1.914871648, 1.936779005, 1.958909151, 1.981004128, 2.003197837,
2.025464654, 2.04755, 2.069700182, 2.091718373, 2.113888201,
2.136060398, 2.158289624, 2.180264191, 2.202414254, 2.224715916,
2.247021345, 2.269176049, 2.291387516, 2.313476111, 2.326170573,
2.335429921, 2.357633495, 2.379940342, 2.402192813, 2.42439467,
2.446633232, 2.468877569, 2.491060592, 2.513314823, 2.535630828,
2.557749792, 2.580162936, 2.602435893, 2.624704815, 2.646972352,
2.669225722, 2.691639318, 2.714125337, 2.736523733, 2.758779515,
2.781210104, 2.803541992, 2.825991542, 2.848390192, 2.870893882,
2.89320733, 2.915455918, 2.937845091, 2.960346084, 2.982679852,
3.005146649, 3.02774239, 3.050376772, 3.072941576, 3.095418238,
3.11321039, 3.117867616, 3.140406779, 3.162938027, 3.18562352,
3.208289242, 3.230859931, 3.253624376, 3.276398362, 3.299011984,
3.321948803, 3.344713921, 3.3676211, 3.390499039, 3.413612734,
3.436630201, 3.459511265, 3.48223841, 3.504908259, 3.528112984,
3.550958409, 3.573780646, 3.596638328, 3.619695026, 3.64256239,
3.665633321, 3.683343545, 3.688536772, 3.711710091, 3.734617698,
3.757598595, 3.780931714, 3.804213312, 3.827652854, 3.850988818,
3.8741239, 3.89718276, 3.920286634, 3.94340705, 3.96671481, 3.989896523,
4.013379229, 4.036672778, 4.060051041, 4.08303672, 4.10640057,
4.129916476, 4.15328391, 4.176743757, 4.200091387, 4.223996806,
4.247754929, 4.271171647, 4.294870338, 4.31839101, 4.341864425,
4.365810642, 4.389813131, 4.413505721, 4.437363576, 4.461160072,
4.485244755, 4.509465226, 4.533544868, 4.557521075, 4.58143756,
4.605331702, 4.629649597, 4.654338849, 4.67856908, 4.70255081,
4.727261444, 4.751578732, 4.775787232, 4.800085036, 4.824469706,
4.848994082, 4.873263151, 4.8977181, 4.921976094, 4.946515242,
4.971465689, 4.996271438, 5.020856011, 5.0453246, 5.069943696,
5.094500117, 5.119505243, 5.144680646, 5.169359087, 5.194003475,
5.219107787, 5.243822841, 5.269076907, 5.294390922, 5.319732421,
5.344801075, 5.369903605, 5.395309396, 5.420751242, 5.446184675,
5.471371425, 5.496730322, 5.522403766, 5.54783593, 5.573171939,
5.598851893, 5.624625043, 5.650378491, 5.676160128, 5.70193199,
5.727979434, 5.753988551, 5.779702337, 5.806023759, 5.832503437,
5.858193893, 5.884700562, 5.911013967, 5.937288068, 5.962911845,
5.989544556, 6.016038968, 6.021765405, 6.042805886, 6.069234376,
6.096545452, 6.123852764, 6.150387925, 6.177645232, 6.2037812,
6.230156743, 6.256649381, 6.282750935, 6.310588855, 6.337615521,
6.364788367, 6.392144377, 6.420203983, 6.447116437, 6.474764521,
6.502364754, 6.529769238, 6.557052052, 6.584991279, 6.612458475,
6.641386247, 6.668930084, 6.697800322, 6.726509995, 6.754995941,
6.78236368, 6.81095288, 6.83950134, 6.868194685, 6.896105638,
6.924874595, 6.953415695, 6.981442754, 7.009990791, 7.039180124,
7.067832482, 7.095821835, 7.124633947, 7.15370117, 7.1821442,
7.210691259, 7.239253077, 7.268739477, 7.298970966, 7.328093601,
7.357994507, 7.387132966, 7.417568761, 7.447799146, 7.477969791,
7.506752168, 7.537082903, 7.566409369, 7.595804973, 7.626173505,
7.656846318, 7.68689076, 7.716452796, 7.746099372, 7.77776348,
7.807903097, 7.838637291, 7.869475446, 7.898007097, 7.92821631,
7.959625664, 7.990043053, 8.020759328, 8.05230695, 8.082776387,
8.113834765, 8.14555263, 8.176218372, 8.208642221, 8.238427411,
8.26994779, 8.301681934, 8.332287351, 8.363891556, 8.395981186,
8.428496048, 8.459894708, 8.49111943, 8.523945296, 8.55721011,
8.589619734, 8.622282831, 8.65560952, 8.688868298, 8.721887673,
8.754682965, 8.786957854, 8.819588891, 8.852277166, 8.884767417,
8.918327862, 8.951848485, 8.986729974, 9.020028922, 9.054197551,
9.087858035, 9.121601111, 9.156251012, 9.189536396, 9.221878206,
9.255244412, 9.288354318, 9.322815121, 9.357896545, 9.392101157,
9.427771911, 9.460937578, 9.495234357, 9.528826316, 9.563500448,
9.59595505, 9.630535381, 9.666271632, 9.700189242, 9.735576007,
9.770534437, 9.807182428, 9.841594744, 9.876326976, 9.91230452,
9.946693093, 9.982556069, 10.016478678, 10.049248392, 10.083488076,
10.120434286, 10.156617455, 10.191231776, 10.227669692, 10.261932306,
10.299243926, 10.336041444, 10.371497603, 10.407478623, 10.443792893,
10.481012545, 10.518198094, 10.555726248, 10.592950408, 10.630847645,
10.667084599, 10.704363788, 10.741465101, 10.780027854, 10.817483581,
10.856015358, 10.891600093, 10.92869582, 10.966812815, 11.004168609,
11.04336318, 11.080330401, 11.117357754, 11.155394926, 11.195138652,
11.231398745, 11.268142915, 11.305026688, 11.341543142, 11.380313808,
11.419718955, 11.459786995, 11.50033222, 11.536696798, 11.575432834,
11.614170071, 11.652981088, 11.693134444, 11.733089121, 11.77359691,
11.812959144, 11.850637831, 11.889562859, 11.929322324, 11.967571689,
12.008827485, 12.047957621, 12.088055734, 12.127124644, 12.166144201,
12.208062367, 12.24706399, 12.287613053, 12.325297741, 12.36359824,
12.403306211, 12.441049272, 12.479638651, 12.52369663, 12.562895259,
12.604811586, 12.646891343, 12.685372543, 12.694183872, 12.726526437,
12.765540985, 12.804047067, 12.845086988, 12.886512108, 12.92831334,
12.97413392, 13.016569002, 13.058837558, 13.101586108, 13.143724826,
13.181878851, 13.22570256, 13.267672894, 13.310237277, 13.354195143,
13.397693393, 13.440979897, 13.484366577, 13.530128598, 13.57341327,
13.611674474, 13.653824313, 13.698783583, 13.74047497, 13.785807849,
13.829057291, 13.870731713, 13.912606294, 13.954542175, 13.997548482,
14.040949724, 14.084558518, 14.129733131, 14.174162924, 14.219230172,
14.261752239, 14.308388386, 14.352873902, 14.395960984, 14.440837194,
14.485350136, 14.531649666, 14.576011129, 14.622350059, 14.669013838,
14.708765729, 14.751921189, 14.795611794, 14.838199706, 14.886728044,
14.93109873, 14.977750113, 15.023322421, 15.06886313, 15.115933439,
15.158866464, 15.203266694, 15.250714664, 15.298237878, 15.345906951,
15.395587413, 15.444056879, 15.486395821, 15.532539748, 15.576879824,
15.621783792, 15.672698406, 15.717593684, 15.762149729, 15.809596243,
15.859370598, 15.909364924, 15.961375554, 16.008070824, 16.05394639,
16.100260324, 16.144621574, 16.194003304, 16.241985535, 16.287939274,
16.334207179, 16.382593015, 16.432362851, 16.481542894, 16.528434225,
16.582259021, 16.627020912, 16.680404374, 16.725547532, 16.775800901,
16.822178737, 16.864473323, 16.909612064, 16.954185203, 17.002200096,
17.051927144, 17.10193172, 17.157367535, 17.206684221, 17.258661628,
17.304405373, 17.352511131, 17.397075433, 17.454371981, 17.496695656,
17.552031316, 17.600399453, 17.65432924, 17.703313804, 17.752919931,
17.798583922, 17.843800373, 17.898277992, 17.947852932, 17.998242634,
18.045790778, 18.092802917, 18.140853076, 18.199572732, 18.253502547,
18.309656398, 18.358886443, 18.415068449, 18.464751709, 18.518029333,
18.570531179, 18.62187785, 18.666691487, 18.715465416, 18.764028918,
18.82051235, 18.876644652, 18.925186741, 18.973339741, 19.026380699,
19.076011142, 19.125579551, 19.183127591, 19.244029212, 19.294022443,
19.347156877, 19.396727184, 19.445968916, 19.496414085, 19.541852539,
19.592602197, 19.644559356, 19.698066736, 19.752424665, 19.807691036,
19.857758586, 19.916726404, 19.965574953, 20.018483409, 20.066149161,
20.124708796, 20.183695448, 20.231089928, 20.280293444, 20.338143311,
20.398798893, 20.457772518, 20.507573396, 20.559734743, 20.615718072,
20.674441509, 20.725215281, 20.783642318, 20.839945679, 20.891877506,
20.94442334, 21.019720982, 21.07876198, 21.123918647, 21.170105093,
21.212029057, 21.260800123, 21.31763232, 21.380429751, 21.441556798,
21.502431956, 21.572200868, 21.62365481, 21.680448912, 21.739553323,
21.783557937, 21.832141772, 21.897005295, 21.949836253, 22.010772533,
22.068266279, 22.128408257, 22.190515511, 22.25255695, 22.300474346,
22.359662529, 22.421617503, 22.473638276, 22.536997916, 22.545415883,
22.58918761, 22.638187697, 22.698457257, 22.763820349, 22.817376495,
22.879515426, 22.947892536, 22.99799628, 23.054069494, 23.11505506,
23.167447037, 23.236528349, 23.291146449, 23.369116477, 23.421573008,
23.48878331, 23.558270001, 23.628399599, 23.683992459, 23.734091184,
23.805924283, 23.847554292, 23.908566924, 23.961814435, 24.016743473,
24.089719449, 24.16022511, 24.220052191, 24.270765975, 24.337532482,
24.389976359, 24.454080976, 24.512254581, 24.572558729, 24.654944574,
24.695677753, 24.749540759, 24.814204302, 24.858737767, 24.931233955,
24.992179039, 25.060447301, 25.092992469, 25.154777062, 25.218725824,
25.293797482, 25.34357924, 25.429152612, 25.494202565, 25.540830175,
25.59809365, 25.664819399, 25.733851887, 25.786902342, 25.866363677,
25.923075397, 25.978994686, 26.054298241, 26.123382853, 26.190039929,
26.267816716, 26.332069734, 26.388126427, 26.475857331, 26.558140374,
26.626492777, 26.690365055, 26.752955816, 26.831423161, 26.906452556,
26.997905555, 27.072086566, 27.146476931, 27.21099222, 27.272204686,
27.350488857, 27.408512933, 27.463030087, 27.518802224, 27.601366792,
27.649959267, 27.704055943, 27.791772698, 27.876490665, 27.949639781,
28.009888243, 28.093863429, 28.140187601, 28.203912993, 28.302100835,
28.3655676, 28.447006244, 28.514784632, 28.6049529, 28.670819875,
28.737618424, 28.803050868, 28.87911912, 28.939529575, 29.016809383,
29.103513291, 29.181198248, 29.266550497, 29.356275841, 29.429731814,
29.492536822, 29.565058072, 29.632386315, 29.704904732, 29.78034884,
29.850361071, 29.943931226, 29.994528565, 30.101369155, 30.176326077,
30.2787371, 30.33613008, 30.420846687, 30.507811518, 30.614494856,
30.671369558, 30.748988114, 30.822563604, 30.896796931, 30.97548012,
30.998035328, 31.045200578, 31.105519869, 31.186833119, 31.247985659,
31.326722259, 31.403664689, 31.471977457, 31.519508927, 31.588296573,
31.670240239, 31.794897452, 31.860682101, 31.929100395, 31.993388537,
32.082887201, 32.148651088, 32.270805055, 32.312923209, 32.378650647,
32.466763002, 32.555345388, 32.600763047, 32.695147983, 32.826931126,
32.908446833, 32.980622789, 33.018418806, 33.091115085, 33.207859333,
33.25051713, 33.318266474, 33.369126816, 33.460685698, 33.513588683,
33.576806132, 33.654352287, 33.738226828, 33.797625398, 33.850966556,
33.966606511, 34.046071698, 34.199562718, 34.231912672, 34.301489215,
34.39047182, 34.475633542, 34.533846546, 34.566296088, 34.700325648,
34.761165595, 34.887411478, 34.926356476, 35.017190867, 35.115052424,
35.183205026, 35.242440239, 35.322072345, 35.356361151, 35.433978736,
35.488493334, 35.553823113, 35.670885448, 35.717978937, 35.816382766,
35.855958609, 35.916447393, 36.012703652, 36.056839413, 36.180130673,
36.255417912, 36.285070573, 36.37349818, 36.459180469, 36.591145993,
36.676566967, 36.781880863, 36.868041904, 36.890602929, 36.962764205,
37.05491354, 37.220399543, 37.322887558, 37.37650548, 37.47200531,
37.565898358, 37.744283404, 37.841628463, 37.94794763, 38.026369713,
38.149533609, 38.279448449, 38.357674014, 38.470426126, 38.530587654,
38.596442164, 38.647046362, 38.70437958, 38.786333193, 38.956533933,
39.026417972, 39.242454732, 39.320147952, 39.378449597, 39.445039913,
39.472177553, 39.55301151, 39.646585409, 39.734841165, 39.889621083,
39.933069446, 40.050579339, 40.178527136, 40.240936243, 40.325372096,
40.484943889, 40.592794474, 40.687148401, 40.720749425, 40.79969198,
40.929677338, 40.950532522, 41.104924058, 41.146859656, 41.252021409,
41.280141669, 41.333841313, 41.481031674, 41.533859543, 41.591232294,
41.665856534, 41.717116687, 41.721367056, 41.90600096, 42.000494836,
42.104114953, 42.16057281, 42.264203912, 42.409380514, 42.600897125,
42.679597246, 42.705197902, 42.783365402, 42.816655515, 42.929218243,
43.009425395, 43.07093373, 43.259050541, 43.291113523, 43.416344158,
43.505744313, 43.570326205, 43.679930558, 43.898057405, 44.002086306,
44.089833968, 44.146438582, 44.193472054, 44.393909584, 44.570327604,
44.629770954, 44.666507049, 44.765437873, 44.804370014, 45.009913481,
45.186232314, 45.311265244, 45.508858924, 45.58845445, 45.624818966,
45.69282507, 45.745497609, 45.861473033, 45.966155619, 46.090869802,
46.101902651, 46.317067637, 46.463401265, 46.547772946, 46.548442085,
46.632923257, 46.692206239, 46.695471465, 46.761312925, 47.052056927,
47.205664456, 47.330914993, 47.776049076, 47.778531342, 47.851009041,
47.87258814, 48.024027734, 48.129251112, 48.204660235, 48.23214584,
48.687249608, 48.792678621, 48.937994876, 49.062211337, 49.118003431,
49.27004682, 49.312498646, 49.375249109, 49.416792448, 49.508937595,
49.523261652, 49.5899737, 49.814258192, 49.896687927, 49.934163258,
50.099350611, 50.102919909, 50.413135224, 50.43944627, 50.562703481
), "1400" = c(1.9557e-05, 1.9601e-05, 1.9625e-05, 1.9929e-05,
1.9955e-05, 2.0042e-05, 2.006e-05, 2.0257e-05, 2.1044e-05, 2.1085e-05,
2.1311e-05, 2.1606e-05, 2.1724e-05, 2.1838e-05, 2.2111e-05, 2.2419e-05,
2.2495e-05, 2.2612e-05, 2.2719e-05, 2.2833e-05, 2.3089e-05, 2.3244e-05,
2.3659e-05, 2.4371e-05, 2.4395e-05, 2.4433e-05, 2.4716e-05, 2.4745e-05,
2.5356e-05, 2.5966e-05, 2.6005e-05, 2.6134e-05, 2.6703e-05, 2.708e-05,
2.7145e-05, 2.7279e-05, 2.7283e-05, 2.8251e-05, 2.841e-05, 2.9157e-05,
2.9187e-05, 2.9535e-05, 3.047e-05, 3.0866e-05, 3.1362e-05, 3.1685e-05,
3.1987e-05, 3.2242e-05, 3.2465e-05, 3.2842e-05, 3.3493e-05, 3.3862e-05,
3.3923e-05, 3.4949e-05, 3.5281e-05, 3.576e-05, 3.5915e-05, 3.6158e-05,
3.6568e-05, 3.6839e-05, 3.7119e-05, 3.7647e-05, 3.8419e-05, 3.9212e-05,
3.9983e-05, 4.0495e-05, 4.1125e-05, 4.1378e-05, 4.1661e-05, 4.1737e-05,
4.1981e-05, 4.2461e-05, 4.2893e-05, 4.3133e-05, 4.4181e-05, 4.4583e-05,
4.5543e-05, 4.587e-05, 4.599e-05, 4.6231e-05, 4.6458e-05, 4.7177e-05,
4.7727e-05, 4.7886e-05, 4.8726e-05, 4.9618e-05, 5.041e-05, 5.077e-05,
5.2288e-05, 5.2772e-05, 5.3548e-05, 5.4146e-05, 5.4311e-05, 5.4732e-05,
5.4842e-05, 5.5679e-05, 5.5927e-05, 5.6439e-05, 5.687e-05, 5.7435e-05,
5.8812e-05, 5.9758e-05, 6.0292e-05, 6.1324e-05, 6.2087e-05, 6.2915e-05,
6.4371e-05, 6.5017e-05, 6.6313e-05, 6.7112e-05, 6.7938e-05, 6.8685e-05,
6.9392e-05, 6.9998e-05, 7.0587e-05, 7.2148e-05, 7.2743e-05, 7.3587e-05,
7.3818e-05, 7.4572e-05, 7.6178e-05, 7.7476e-05, 7.7757e-05, 7.8547e-05,
8.0024e-05, 8.0702e-05, 8.1934e-05, 8.2579e-05, 8.3466e-05, 8.4076e-05,
8.5469e-05, 8.6683e-05, 8.7617e-05, 8.8398e-05, 8.9228e-05, 8.9771e-05,
9.0812e-05, 9.1902e-05, 9.2882e-05, 9.4565e-05, 9.5605e-05, 9.6031e-05,
9.8194e-05, 9.9315e-05, 0.000100213, 0.000100975, 0.000102078,
0.000103376, 0.00010586, 0.0001062, 0.00010784, 0.00010892, 0.000109354,
0.000112153, 0.000114079, 0.000115041, 0.000116503, 0.000118004,
0.000119533, 0.000120685, 0.000122652, 0.000123493, 0.000125554,
0.000126894, 0.000127969, 0.000129365, 0.000131223, 0.00013321,
0.000134443, 0.000135532, 0.000137165, 0.000138554, 0.00014024,
0.000141623, 0.000143389, 0.000145038, 0.00014725, 0.000148859,
0.000150055, 0.000152129, 0.000153294, 0.000154962, 0.000156253,
0.000158049, 0.000159876, 0.000162387, 0.000165482, 0.000166971,
0.000168011, 0.000170428, 0.000172522, 0.000174213, 0.000176774,
0.000180178, 0.000181671, 0.000183707, 0.000185607, 0.000188375,
0.00019003, 0.000194522, 0.000195828, 0.000199953, 0.000202452,
0.000204246, 0.000206417, 0.000209427, 0.0002121, 0.000214602,
0.000216642, 0.00021891, 0.000221794, 0.000224474, 0.000226374,
0.000228446, 0.000230724, 0.000233799, 0.000235531, 0.000237626,
0.000240728, 0.000242749, 0.000244096, 0.0002467, 0.000248703,
0.000251638, 0.000254492, 0.000257057, 0.000258687, 0.000261876,
0.000264932, 0.000267357, 0.000270939, 0.000275707, 0.000279746,
0.00028242, 0.000284697, 0.000288236, 0.000292896, 0.000295844,
0.000298731, 0.000300956, 0.000305146, 0.000309508, 0.000312106,
0.00031683, 0.000321729, 0.000324717, 0.000328472, 0.000331666,
0.000334458, 0.000338883, 0.000341967, 0.000346886, 0.000352725,
0.000357213, 0.000361518, 0.000364486, 0.000370484, 0.000376278,
0.000380732, 0.000385111, 0.000389379, 0.000394179, 0.000399642,
0.000403693, 0.000407652, 0.000412439, 0.000419271, 0.000424349,
0.000428005, 0.000434295, 0.000438989, 0.000443843, 0.000449389,
0.000455214, 0.000460679, 0.000466644, 0.000472731, 0.000477975,
0.000482114, 0.00048518, 0.000490528, 0.000496096, 0.000500686,
0.000507698, 0.000514557, 0.00052048, 0.000526646, 0.000534217,
0.000538039, 0.000543445, 0.000549774, 0.000556369, 0.000564253,
0.000570288, 0.000576017, 0.000581693, 0.00058947, 0.00059829,
0.000604755, 0.000614711, 0.000620847, 0.000625014, 0.000631248,
0.000638484, 0.000647157, 0.000654985, 0.000662842, 0.000670113,
0.000675472, 0.000684512, 0.000692168, 0.000699811, 0.000707461,
0.000715373, 0.000722661, 0.000734186, 0.00074225, 0.000752589,
0.000758674, 0.000768136, 0.000776424, 0.00078547, 0.00079446,
0.000804174, 0.000812214, 0.00082138, 0.000830745, 0.000839456,
0.000849544, 0.000860428, 0.000870718, 0.00088152, 0.000893621,
0.000903423, 0.00091238, 0.000925255, 0.000937035, 0.000947013,
0.000957079, 0.000969447, 0.000978208, 0.000980085, 0.000991836,
0.001004085, 0.001018679, 0.001029151, 0.001043152, 0.001055715,
0.001068996, 0.001078719, 0.001092298, 0.001104814, 0.001119417,
0.001130732, 0.001141218, 0.001153837, 0.001167904, 0.001182688,
0.001200456, 0.001215957, 0.001228867, 0.00124244, 0.001259623,
0.001273253, 0.001288586, 0.001307358, 0.001321937, 0.001336386,
0.001351942, 0.001369839, 0.001385874, 0.001401235, 0.001416045,
0.001433032, 0.00144952, 0.001466444, 0.001486053, 0.001503982,
0.001522727, 0.001541739, 0.001559081, 0.001578047, 0.001596441,
0.001614798, 0.001630265, 0.001653219, 0.0016719, 0.001689752,
0.001712831, 0.00173261, 0.001749922, 0.001768849, 0.001788587,
0.001807618, 0.001828062, 0.001848814, 0.00186584, 0.001888492,
0.001918593, 0.001941275, 0.001965903, 0.00198733, 0.002007108,
0.002030074, 0.002053358, 0.002076604, 0.002103523, 0.002130251,
0.002155788, 0.002178897, 0.002205229, 0.002230042, 0.002257287,
0.002285922, 0.002313924, 0.002343839, 0.002368653, 0.002394848,
0.002419802, 0.002453267, 0.002483225, 0.002508459, 0.002536248,
0.002561742, 0.002592595, 0.002623472, 0.002654859, 0.002683004,
0.002716109, 0.002746949, 0.002774998, 0.002804684, 0.002838779,
0.002874351, 0.002910434, 0.002940372, 0.002974277, 0.003010553,
0.003050948, 0.003086573, 0.003125085, 0.003157816, 0.003194046,
0.003232077, 0.003263791, 0.003302824, 0.003337126, 0.003375039,
0.003410213, 0.003450267, 0.003492988, 0.003533796, 0.003571347,
0.00361361, 0.003655543, 0.003699853, 0.003739652, 0.003779751,
0.003822727, 0.003863705, 0.003913137, 0.003955758, 0.004001968,
0.004044548, 0.004091201, 0.004139522, 0.004187619, 0.004233574,
0.004280428, 0.004322545, 0.004378182, 0.004429462, 0.004480681,
0.004532145, 0.004584918, 0.004637844, 0.004690066, 0.004746317,
0.004802738, 0.004852106, 0.004908287, 0.004968449, 0.005024628,
0.005084318, 0.005146878, 0.005210677, 0.005273441, 0.005333148,
0.005394902, 0.005459906, 0.005520311, 0.005579774, 0.005641954,
0.005706832, 0.00577238, 0.005834312, 0.005902485, 0.005968862,
0.0060369, 0.0061138, 0.006186009, 0.006254117, 0.006333239,
0.00640549, 0.006482765, 0.006560922, 0.006644949, 0.006722578,
0.006797858, 0.006871935, 0.006951907, 0.007023561, 0.007106232,
0.007186456, 0.007264733, 0.00735579, 0.007439022, 0.007519836,
0.007607464, 0.007694593, 0.007782785, 0.00786916, 0.007959373,
0.00805087, 0.008143756, 0.00823737, 0.00833466, 0.008439538,
0.008538559, 0.00864399, 0.008750809, 0.008852931, 0.008963641,
0.009066083, 0.009162719, 0.009273034, 0.009377397, 0.009486681,
0.009599999, 0.009714458, 0.009816843, 0.009907784, 0.009927709,
0.010035715, 0.010150855, 0.010269228, 0.010391223, 0.010506832,
0.010630338, 0.010751478, 0.010873165, 0.010997625, 0.011128655,
0.011258405, 0.011386759, 0.011517397, 0.011654793, 0.011794429,
0.011931445, 0.012071038, 0.012207362, 0.012347995, 0.012494234,
0.012649439, 0.012788618, 0.012933547, 0.013078183, 0.013246245,
0.013404049, 0.013560675, 0.013724574, 0.013889011, 0.014055387,
0.014222291, 0.014379636, 0.01455, 0.014710012, 0.014881961,
0.015053958, 0.015226723, 0.015405334, 0.015590425, 0.015767063,
0.015951342, 0.016136543, 0.016314456, 0.016503396, 0.016689336,
0.016888426, 0.017099743, 0.017286571, 0.017486986, 0.017683444,
0.017889572, 0.018088607, 0.018305716, 0.018525876, 0.018740443,
0.018964886, 0.019189766, 0.019412312, 0.019643218, 0.019879451,
0.020114311, 0.0203453, 0.020589621, 0.02082256, 0.021063936,
0.021300216, 0.021547513, 0.02180482, 0.022050384, 0.022297139,
0.022567489, 0.022813711, 0.023078821, 0.023350439, 0.023617028,
0.023900061, 0.02417602, 0.024455067, 0.0247462, 0.025039303,
0.025348163, 0.025637247, 0.025936391, 0.026241111, 0.026546469,
0.02684471, 0.027162462, 0.027484937, 0.027808041, 0.028136588,
0.028455341, 0.028790073, 0.029123789, 0.029475088, 0.029813726,
0.030159971, 0.030527102, 0.030874864, 0.031227721, 0.031586913,
0.031957736, 0.032329096, 0.032720133, 0.033099701, 0.033476326,
0.03386464, 0.034248205, 0.034639483, 0.035057258, 0.03545921,
0.035863928, 0.036295067, 0.036721147, 0.037142062, 0.037576998,
0.038012293, 0.038448243, 0.038899462, 0.039359219, 0.039809974,
0.040262606, 0.040741054, 0.041212774, 0.041690477, 0.042178779,
0.042680297, 0.043195303, 0.043682423, 0.044199546, 0.044714167,
0.045231155, 0.045755741, 0.046295281, 0.046837374, 0.047381292,
0.047922926, 0.048467159, 0.04903878, 0.049627283, 0.0502102,
0.050798438, 0.051399406, 0.052001552, 0.052618398, 0.053239398,
0.053866395, 0.054483365, 0.055116062, 0.055771517, 0.056436735,
0.057105354, 0.057786069, 0.058468412, 0.059167178, 0.059851067,
0.060549971, 0.061274538, 0.061990045, 0.062730573, 0.063470839,
0.064219338, 0.06498654, 0.065749979, 0.066519619, 0.067305165,
0.068096954, 0.06891395, 0.069708536, 0.070505665, 0.071319241,
0.072162339, 0.073024865, 0.073885269, 0.074737337, 0.07563517,
0.076516944, 0.077418042, 0.078325576, 0.079234849, 0.080180006,
0.081124847, 0.082073589, 0.083048382, 0.084042716, 0.085067235,
0.086076347, 0.087109409, 0.088140916, 0.0891933, 0.090235756,
0.091292953, 0.092374706, 0.093465365, 0.094567083, 0.095681907,
0.096809007, 0.097942489, 0.099072308, 0.10029988, 0.101255485,
0.101505427, 0.102713332, 0.103922698, 0.105182569, 0.106439688,
0.10771328, 0.109005999, 0.11032077, 0.111639575, 0.11297243,
0.114310675, 0.11569249, 0.11705603, 0.118490002, 0.119895866,
0.12135149, 0.122825132, 0.124289847, 0.125762078, 0.127280547,
0.128822104, 0.130364789, 0.131907171, 0.133476832, 0.135093577,
0.136749031, 0.138374213, 0.140010093, 0.141665908, 0.143339537,
0.145041514, 0.146782593, 0.14852736, 0.150305668, 0.152093342,
0.153898626, 0.155751041, 0.157646134, 0.159560227, 0.161519814,
0.163489461, 0.165467719, 0.167472244, 0.169463415, 0.171541062,
0.173634356, 0.175744214, 0.17784306, 0.179978569, 0.182184542,
0.184379828, 0.186653988, 0.188929394, 0.19121866, 0.193562295,
0.195893499, 0.198272992, 0.200692338, 0.203126283, 0.205581658,
0.208089745, 0.210676288, 0.21328748, 0.215908496, 0.218547441,
0.221196538, 0.223872012, 0.226620084, 0.229390607, 0.232199419,
0.235052623, 0.237897171, 0.240853717, 0.243834408, 0.24680919,
0.249849058, 0.252920831, 0.256008741, 0.259154912, 0.262325247,
0.265568106, 0.268830401, 0.27216281, 0.275501727, 0.278905485,
0.282386265, 0.28581591, 0.289377276, 0.292923268, 0.296574941,
0.300300053, 0.304100342, 0.30790601, 0.311712065, 0.315603497,
0.319528103, 0.320377937, 0.323504665, 0.327554713, 0.331662893,
0.335846275, 0.340091743, 0.344405611, 0.348697837, 0.35306978,
0.357561025, 0.362095857, 0.366636915, 0.371251927, 0.37596912,
0.380716551, 0.385526774, 0.390478314, 0.39549135, 0.400586075,
0.405681307, 0.410863975, 0.416145536, 0.421437226, 0.426877126,
0.432265756, 0.437806164, 0.438978108, 0.443466667, 0.449238548,
0.455070023, 0.461002859, 0.467003844, 0.473026121, 0.479222357,
0.485566167, 0.491928437, 0.498408743, 0.504983153, 0.511685146,
0.518382375, 0.525181218, 0.53214871, 0.539209454, 0.546301132,
0.553529914, 0.560890753, 0.568354416, 0.575945912, 0.58365622,
0.591468391, 0.599477336, 0.607599377, 0.615820732, 0.624218049,
0.632663353, 0.641268509, 0.64992217, 0.658794824, 0.6678033,
0.676993841, 0.686331142, 0.695815806, 0.699923255, 0.705458428,
0.715222588, 0.725199804, 0.735359521, 0.745724047, 0.756176628,
0.766868512, 0.777693014, 0.788733898, 0.799919373, 0.811257965,
0.822908287, 0.834665519, 0.846662414, 0.858995446, 0.871505097,
0.884236407, 0.897058733, 0.91014572, 0.923488343, 0.937071926,
0.95093981, 0.965115281, 0.979460192, 0.994154618, 1.00026155,
1.009207995, 1.024366062, 1.039831033, 1.055517978, 1.071695064,
1.088352245, 1.105110137, 1.122224043, 1.139617399, 1.157374167,
1.175637434, 1.194226724, 1.213243712, 1.232604118, 1.252384045,
1.272724701, 1.293443883, 1.314748915, 1.336396107, 1.353798141,
1.393638995, 1.415929908, 1.438464448, 1.461034321, 1.483354823,
1.50551998, 1.527941618, 1.550020506, 1.572475171, 1.594934385,
1.617219714, 1.639548815, 1.661932716, 1.684015365, 1.706393622,
1.728680392, 1.751047476, 1.773249438, 1.786361301, 1.795504179,
1.817890506, 1.840253578, 1.862623338, 1.884909908, 1.907102564,
1.929428066, 1.951991186, 1.974465086, 1.996832042, 2.019287073,
2.041516779, 2.063863715, 2.086207553, 2.108657496, 2.130994022,
2.153315191, 2.175607016, 2.198041478, 2.220549746, 2.242788571,
2.26516972, 2.287299587, 2.309725988, 2.332157271, 2.345103122,
2.354551075, 2.37701133, 2.399401657, 2.421769415, 2.444178797,
2.466565659, 2.488894511, 2.511171545, 2.533719597, 2.556142569,
2.578551591, 2.600900907, 2.62327121, 2.645558439, 2.66831711,
2.690619328, 2.71308115, 2.735698359, 2.758171514, 2.780766253,
2.803369848, 2.825784526, 2.848376415, 2.871118814, 2.893709853,
2.916312831, 2.938985498, 2.961532574, 2.984063769, 3.007084765,
3.029819122, 3.05242974, 3.075065342, 3.097786239, 3.120290678,
3.138595326, 3.143138807, 3.165753352, 3.188303087, 3.211117964,
3.233946875, 3.256693017, 3.279476065, 3.302219065, 3.325076308,
3.348031573, 3.371023197, 3.393853706, 3.416629654, 3.43935658,
3.461963312, 3.48464817, 3.507600205, 3.530247976, 3.553243547,
3.576311715, 3.599366772, 3.622050514, 3.644821641, 3.667910881,
3.691090515, 3.709192935, 3.714203426, 3.737225722, 3.760089618,
3.783195889, 3.806151918, 3.829701503, 3.853052996, 3.875989394,
3.899086651, 3.922245962, 3.945285274, 3.968822263, 3.992161372,
4.015503732, 4.038958159, 4.062369924, 4.085294831, 4.108853405,
4.131933677, 4.155126151, 4.178815259, 4.202170367, 4.225806498,
4.249240514, 4.272659753, 4.295975589, 4.319120688, 4.34299789,
4.36636751, 4.390012879, 4.413420188, 4.437250575, 4.46100466,
4.484519299, 4.507947343, 4.531598058, 4.555535266, 4.579071294,
4.603076404, 4.626996338, 4.650868516, 4.674578279, 4.698443937,
4.722529387, 4.746595483, 4.770731486, 4.795174471, 4.819377821,
4.843151148, 4.867003296, 4.891154197, 4.915232147, 4.939400353,
4.963530931, 4.988021516, 5.012532819, 5.037044045, 5.061417823,
5.086344713, 5.110344002, 5.134820423, 5.159243266, 5.183732782,
5.208167514, 5.233020451, 5.257622551, 5.282321875, 5.307442883,
5.332176481, 5.356667318, 5.381561495, 5.406026596, 5.431092582,
5.455551068, 5.480755814, 5.505342677, 5.53033027, 5.555473671,
5.58020766, 5.605328796, 5.629677427, 5.654798694, 5.680152262,
5.70503731, 5.72998503, 5.7553273, 5.780395547, 5.805842275,
5.830729555, 5.855432595, 5.881516504, 5.90682925, 5.932589366,
5.957933664, 5.983568848, 6.008660684, 6.01369119, 6.034179043,
6.059871101, 6.085338134, 6.111235371, 6.13676503, 6.162626659,
6.188559255, 6.214220653, 6.239969623, 6.265655381, 6.291941068,
6.317879488, 6.344081815, 6.370280524, 6.396431332, 6.422679801,
6.448775838, 6.474601213, 6.500591937, 6.52708401, 6.553706743,
6.581246439, 6.607044297, 6.633551752, 6.659931823, 6.686663243,
6.713563833, 6.740568024, 6.766979809, 6.793661248, 6.820047929,
6.847084815, 6.873875951, 6.900916072, 6.9278946, 6.955439384,
6.982079631, 7.00893743, 7.036629894, 7.063436241, 7.090757243,
7.118159677, 7.145149518, 7.172546611, 7.199133025, 7.226137416,
7.253748899, 7.28066249, 7.307010725, 7.334955527, 7.362939006,
7.391100528, 7.41923321, 7.446797739, 7.474194501, 7.501337543,
7.529851429, 7.558395207, 7.586110388, 7.613972257, 7.642842644,
7.671579463, 7.7007358, 7.730473535, 7.759163455, 7.787658514,
7.816631701, 7.845221146, 7.874694048, 7.903333605, 7.931523082,
7.959998189, 7.988725627, 8.018115527, 8.047276419, 8.076474759,
8.105611668, 8.134499257, 8.163621593, 8.193563075, 8.222706938,
8.251210999, 8.280681378, 8.310227298, 8.339234082, 8.369460499,
8.398618924, 8.429747731, 8.459202266, 8.489648093, 8.520178161,
8.549956661, 8.580769044, 8.610630722, 8.641811188, 8.672023077,
8.701513402, 8.732364581, 8.764570164, 8.794801983, 8.825242383,
8.855579836, 8.88593896, 8.916458, 8.947141439, 8.97714099, 9.007531957,
9.038003424, 9.067751705, 9.098099079, 9.129150905, 9.159420953,
9.190520009, 9.22152027, 9.253642546, 9.28536682, 9.31678295,
9.347545514, 9.379531688, 9.411739382, 9.443229572, 9.473858987,
9.50708456, 9.539930882, 9.572888582, 9.604947669, 9.636768791,
9.669754048, 9.703692742, 9.735954215, 9.767598098, 9.799818398,
9.833339329, 9.865322259, 9.897244987, 9.929565793, 9.96263004,
9.99614191, 10.029423723, 10.061496404, 10.096368643, 10.129285302,
10.162407322, 10.196669142, 10.23015354, 10.263455248, 10.296873533,
10.331723228, 10.365737652, 10.398727153, 10.430592965, 10.465739684,
10.499886518, 10.5321491, 10.566349961, 10.600786636, 10.634535064,
10.667872464, 10.702385596, 10.736899479, 10.770501338, 10.804498042,
10.837408584, 10.869951332, 10.903533681, 10.936984931, 10.970744427,
11.003876846, 11.037739278, 11.070744779, 11.105535989, 11.141013069,
11.174567243, 11.207635996, 11.241241912, 11.275584966, 11.313721189,
11.348555874, 11.384298219, 11.418391208, 11.455344413, 11.490273534,
11.525401172, 11.560860492, 11.598069995, 11.635842157, 11.672348124,
11.709282926, 11.744394311, 11.779500305, 11.813783506, 11.847416003,
11.881482305, 11.91802442, 11.954398107, 11.990304612, 12.025341941,
12.062864302, 12.098314823, 12.137065368, 12.144080477, 12.171712535,
12.209689254, 12.247231269, 12.286496231, 12.320219878, 12.3549833,
12.391423273, 12.42813351, 12.46619794, 12.505127214, 12.542206183,
12.580501859, 12.621896682, 12.660755711, 12.698353826, 12.734886617,
12.774949035, 12.813491821, 12.85305132, 12.891474742, 12.930349454,
12.96765662, 13.004177392, 13.043219809, 13.080463743, 13.115670887,
13.152252192, 13.189413715, 13.233257507, 13.275458095, 13.314033418,
13.354720995, 13.394788099, 13.435832168, 13.478091542, 13.51721069,
13.555007043, 13.593909583, 13.633749595, 13.670609897, 13.708208597,
13.747615287, 13.788082682, 13.826151881, 13.863777676, 13.903548252,
13.938007739, 13.979227175, 14.019175241, 14.059401571, 14.101189731,
14.139567334, 14.183184046, 14.225310526, 14.266969155, 14.30768243,
14.348715649, 14.385657037, 14.423667384, 14.468381412, 14.504384227,
14.543107384, 14.582956718, 14.625876955, 14.663871271, 14.701288241,
14.744714608, 14.782306498, 14.823090854, 14.865584123, 14.90765267,
14.956032811, 14.995987686, 15.036447523, 15.077237903, 15.11872368,
15.15846607, 15.198468379, 15.239060853, 15.279264547, 15.323296986,
15.369676308, 15.409992044, 15.45101432, 15.49525383, 15.536879963,
15.58128018, 15.624105197, 15.664015155, 15.704989668, 15.750717728,
15.794980123, 15.843806669, 15.889077914, 15.930491587, 15.965719794,
16.006807194, 16.044612202, 16.085479522, 16.124280106, 16.163047626,
16.207151024, 16.251519402, 16.29573528, 16.337733325, 16.377904406,
16.420161829, 16.464119138, 16.509040972, 16.554276831, 16.588300722,
16.626627306, 16.668050989, 16.709773839, 16.748497735, 16.792322636,
16.830464613, 16.872421627, 16.91194236, 16.956655816, 17.000694098,
17.043050476, 17.082726985, 17.128539006, 17.174802834, 17.217439299,
17.261790481, 17.30679014, 17.349595891, 17.396218969, 17.437284917,
17.479042616, 17.51786286, 17.565512675, 17.612361664, 17.652306298,
17.696367194, 17.742567186, 17.796929383, 17.84050704, 17.879916745,
17.930596281, 17.974834859, 18.023546789, 18.062622227, 18.109707625,
18.158150143, 18.202170988, 18.255216235, 18.301217556, 18.344620355,
18.392987629, 18.442799352, 18.488035879, 18.539895773, 18.59228136,
18.641380028, 18.679036504, 18.722044537, 18.773120266, 18.818881312,
18.862535604, 18.906485207, 18.955018415, 18.998822255, 19.046437547,
19.092151946, 19.12964049, 19.177795994, 19.231511481, 19.273757432,
19.323876185, 19.375560923, 19.416849495, 19.446307171, 19.503348284,
19.549005217, 19.604543787, 19.647013786, 19.696427838, 19.734966535,
19.78097635, 19.824296597, 19.878802975, 19.928621596, 19.971260334,
20.022190161, 20.073094047, 20.124960571, 20.172635364, 20.222312984,
20.270958181, 20.312024481, 20.365015758, 20.427090044, 20.473703607,
20.515249168, 20.564696515, 20.614959446, 20.667044815, 20.681257654,
20.719347681, 20.776657764, 20.82620978, 20.885768934, 20.957814167,
21.018579301, 21.068308628, 21.11549699, 21.154744293, 21.207091806,
21.268854521, 21.324743588, 21.384390871, 21.44059625, 21.495914379,
21.538548784, 21.580694965, 21.622789161, 21.685927581, 21.732243053,
21.779867761, 21.838766132, 21.883938076, 21.929197269, 21.981940588,
22.03309007, 22.077548095, 22.130603323, 22.193208918, 22.24506311,
22.30418011, 22.365744186, 22.414007633, 22.470805793, 22.516511425,
22.570139557, 22.619967981, 22.671430524, 22.729383393, 22.791316107,
22.83306684, 22.889561274, 22.934715728, 22.978575088, 23.034771834,
23.088027372, 23.139198523, 23.186955089, 23.245435732, 23.296440737,
23.345197054, 23.394852522, 23.446039779, 23.501613925, 23.554599917,
23.61134638, 23.671664028, 23.743174295, 23.790787563, 23.849950095,
23.896432486, 23.939921082, 24.001680689, 24.030130502, 24.083803236,
24.123704073, 24.161995627, 24.213365715, 24.257040707, 24.30481365,
24.350418776, 24.392430971, 24.448896262, 24.502045212, 24.569909581,
24.608883276, 24.662411175, 24.704228261, 24.759181939, 24.802349416,
24.854108896, 24.892790956, 24.94282338, 25.027031351, 25.084693296,
25.130546146, 25.177756255, 25.253899956, 25.314612368, 25.387317465,
25.437248984, 25.485181644, 25.538943948, 25.584773691, 25.626985639,
25.692420443, 25.735012425, 25.780218571, 25.825892137, 25.896376421,
25.960326753, 26.004551508, 26.042243756, 26.100057753, 26.160301535,
26.215330788, 26.280559801, 26.308079421, 26.344841663, 26.390459988,
26.44947313, 26.520386866, 26.581978548, 26.620257114, 26.684970662,
26.712828385, 26.750231366, 26.819310444, 26.843492792, 26.876337118,
26.934328619, 26.999245942, 27.051379673, 27.114767464, 27.159227408,
27.2219473, 27.285635656, 27.323865602, 27.361623534, 27.430302937,
27.496971347, 27.567818744, 27.646721851, 27.734282655, 27.785641343,
27.86506394, 27.901761874, 27.97790915, 28.025542601, 28.082084824,
28.123988108, 28.189764201, 28.230136671, 28.275177373, 28.319527116,
28.384325614, 28.427420154, 28.495134184, 28.525313242, 28.607314746,
28.658748243, 28.699078971, 28.762344325, 28.822369002, 28.877885884,
28.896814614, 28.972039797, 29.023542786, 29.114303556, 29.175061912,
29.219473823, 29.272580375, 29.321260793, 29.378384734, 29.457956179,
29.537965447, 29.589877151, 29.686331105, 29.767352551, 29.833514032,
29.926747154, 29.993946064, 30.027962402, 30.090903986, 30.220045864,
30.270068908, 30.322087813, 30.373282942, 30.447759785, 30.499284655,
30.5222983, 30.580558184, 30.606605066, 30.629144683, 30.677999858,
30.762917982, 30.814702698, 30.905052401, 30.92580219, 31.00721972,
31.072389624, 31.203176724, 31.247013112, 31.285970415, 31.359586133,
31.416620031, 31.476244708, 31.548780655, 31.582602169, 31.62705909,
31.691524754, 31.746602792, 31.794163396, 31.827324028, 31.915208466,
31.969111809, 32.007857837, 32.047950771, 32.090151229, 32.130534268,
32.184262429, 32.229464681, 32.313650151, 32.34933069, 32.370706656,
32.419351548, 32.471737516, 32.514156137, 32.617641224, 32.69117151,
32.762280079, 32.850302649, 32.893861942, 33.017440115, 33.078530893,
33.113845267, 33.203980397, 33.230265606, 33.284780446, 33.352211395,
33.384701885, 33.466736377, 33.612706147, 33.696734426, 33.710193141,
33.803945565, 33.840470559, 33.909107039, 33.972815431, 34.122736792,
34.178420854, 34.283759864, 34.377837902, 34.430271562, 34.483913762,
34.545395999, 34.639271025, 34.691177686, 34.791209335, 34.866262859,
34.931677727, 35.020166197, 35.073025481, 35.179643203, 35.222288439,
35.31312208, 35.425258923, 35.449960631, 35.499491171, 35.516105026,
35.532973033, 35.569992184, 35.597828633, 35.790737615, 35.905389813,
35.958035108, 36.052691139, 36.111967815, 36.175239844, 36.206822674,
36.344621937, 36.397322601, 36.465226023, 36.591330558, 36.746498603,
36.789894411, 36.79410234, 36.915716594, 36.959222283, 37.040435602,
37.138384876, 37.153737189, 37.205095449, 37.23235986, 37.320857544,
37.380472389, 37.40338375, 37.431159451, 37.442680672, 37.451605439,
37.501508472, 37.551489034, 37.568619649, 37.613774606, 37.702834164,
37.782673476, 37.877107447, 38.011416295, 38.108680931, 38.182994424,
38.226432693, 38.285319348, 38.355185207, 38.460388418, 38.554045822,
38.749648961, 38.785236648, 38.916024443, 39.120586267, 39.187118426,
39.20587648, 39.389512046, 39.429515082, 39.544600096, 39.568374944,
39.593452873, 39.723959041, 39.735758904, 39.883765243, 40.04750588,
40.260945495, 40.27876909, 40.362889356, 40.419115362, 40.483215324,
40.622415666, 40.654353622, 40.677498121, 40.840584042, 40.891379468,
41.059325758, 41.16831433, 41.212693742, 41.216573687, 41.383086251,
41.3928515, 41.446492336, 41.604237595, 41.606580646, 41.661792192
), "1600" = c(1.5429e-05, 1.5429e-05, 1.5502e-05, 1.5676e-05,
1.5683e-05, 1.5711e-05, 1.5755e-05, 1.5941e-05, 1.6414e-05, 1.665e-05,
1.6774e-05, 1.6819e-05, 1.6889e-05, 1.7286e-05, 1.7391e-05, 1.7442e-05,
1.7472e-05, 1.77e-05, 1.7862e-05, 1.8308e-05, 1.8919e-05, 1.9114e-05,
1.9695e-05, 1.9749e-05, 2.0273e-05, 2.0302e-05, 2.0833e-05, 2.0961e-05,
2.1482e-05, 2.222e-05, 2.2246e-05, 2.272e-05, 2.2904e-05, 2.2957e-05,
2.3586e-05, 2.5046e-05, 2.5222e-05, 2.5395e-05, 2.5538e-05, 2.5665e-05,
2.6015e-05, 2.6388e-05, 2.7088e-05, 2.7603e-05, 2.802e-05, 2.8439e-05,
2.8468e-05, 2.8858e-05, 2.8881e-05, 2.9768e-05, 2.9863e-05, 3.0138e-05,
3.0389e-05, 3.0928e-05, 3.0953e-05, 3.0998e-05, 3.1756e-05, 3.2203e-05,
3.3321e-05, 3.3536e-05, 3.4646e-05, 3.5085e-05, 3.5139e-05, 3.5549e-05,
3.6116e-05, 3.6789e-05, 3.7246e-05, 3.7557e-05, 3.812e-05, 3.8288e-05,
3.8813e-05, 4.026e-05, 4.091e-05, 4.1086e-05, 4.1594e-05, 4.2132e-05,
4.2621e-05, 4.2878e-05, 4.35e-05, 4.3999e-05, 4.4135e-05, 4.4561e-05,
4.5127e-05, 4.5708e-05, 4.6021e-05, 4.65e-05, 4.7372e-05, 4.7701e-05,
4.8316e-05, 4.8954e-05, 4.9428e-05, 4.9802e-05, 5.0364e-05, 5.0848e-05,
5.1389e-05, 5.2806e-05, 5.3264e-05, 5.404e-05, 5.4984e-05, 5.5791e-05,
5.6093e-05, 5.6598e-05, 5.834e-05, 5.8741e-05, 5.9739e-05, 6.0615e-05,
6.0778e-05, 6.1008e-05, 6.1565e-05, 6.2533e-05, 6.2893e-05, 6.3532e-05,
6.401e-05, 6.4981e-05, 6.6291e-05, 6.7857e-05, 6.8277e-05, 6.8991e-05,
6.9518e-05, 7.1455e-05, 7.2671e-05, 7.3423e-05, 7.4514e-05, 7.5137e-05,
7.6229e-05, 7.7579e-05, 7.837e-05, 7.8967e-05, 8.0357e-05, 8.1973e-05,
8.2867e-05, 8.4336e-05, 8.5646e-05, 8.7194e-05, 8.8245e-05, 8.9166e-05,
9.0646e-05, 9.1774e-05, 9.2835e-05, 9.4343e-05, 9.5314e-05, 9.6482e-05,
9.7113e-05, 9.8441e-05, 9.9077e-05, 0.000100277, 0.000101397,
0.000103426, 0.00010544, 0.000106332, 0.000107546, 0.000109439,
0.000110119, 0.000112006, 0.000113459, 0.000114421, 0.000115262,
0.000116363, 0.000118349, 0.000120467, 0.000121316, 0.000122815,
0.00012392, 0.000126855, 0.000128914, 0.0001304, 0.000132091,
0.000133006, 0.00013478, 0.000135987, 0.000137777, 0.000139442,
0.000140694, 0.000142958, 0.00014442, 0.00014533, 0.000147008,
0.000148673, 0.00015085, 0.000153591, 0.000155925, 0.000157814,
0.00015873, 0.000160471, 0.000161474, 0.000163344, 0.000165458,
0.000167785, 0.000170045, 0.000172111, 0.000173982, 0.000175741,
0.000178115, 0.000180787, 0.000182875, 0.000184637, 0.000187237,
0.00018918, 0.000192298, 0.000195396, 0.00019726, 0.000200825,
0.000202917, 0.000204773, 0.000206044, 0.000207829, 0.000210567,
0.00021284, 0.000214591, 0.000216996, 0.000219819, 0.00022304,
0.000226093, 0.000228112, 0.000231362, 0.000235295, 0.000238009,
0.000240085, 0.000242728, 0.000245528, 0.000247448, 0.000250373,
0.000253172, 0.000255632, 0.000259746, 0.000262787, 0.000266097,
0.000268316, 0.000271554, 0.000274737, 0.000278844, 0.000281461,
0.00028401, 0.000286507, 0.000290678, 0.000294297, 0.000297514,
0.000300982, 0.00030483, 0.000310753, 0.000314905, 0.000317275,
0.000320542, 0.00032489, 0.000328711, 0.000332668, 0.000336881,
0.000340415, 0.00034426, 0.000347488, 0.00035232, 0.000356722,
0.000362091, 0.000366847, 0.000370819, 0.000375442, 0.000380031,
0.000383863, 0.000388287, 0.000392681, 0.000396285, 0.00040128,
0.000406373, 0.000411834, 0.00041859, 0.00042315, 0.000428721,
0.000432611, 0.000437113, 0.000442339, 0.000446627, 0.000451114,
0.000455989, 0.00046149, 0.000466454, 0.000471996, 0.000478517,
0.000483835, 0.000489808, 0.000494674, 0.000502333, 0.000509458,
0.000514011, 0.000521089, 0.000526352, 0.000532403, 0.000537597,
0.000543229, 0.000548652, 0.000554624, 0.000559728, 0.000566757,
0.000573237, 0.000579888, 0.00058489, 0.000593027, 0.000601964,
0.000608525, 0.000614732, 0.000622215, 0.000629067, 0.000636441,
0.000643431, 0.000650764, 0.000658451, 0.000668782, 0.000677106,
0.000684099, 0.000691977, 0.000700064, 0.00070754, 0.000717052,
0.000727087, 0.000733895, 0.000743611, 0.000751248, 0.000760583,
0.000767425, 0.000775764, 0.000783763, 0.000794093, 0.00080283,
0.000813, 0.000820202, 0.000831188, 0.000837282, 0.000847292,
0.000855275, 0.000865128, 0.000874738, 0.000885952, 0.000896486,
0.000908242, 0.000920581, 0.000931772, 0.000942032, 0.000953125,
0.000965111, 0.000977077, 0.000985375, 0.000994929, 0.000997963,
0.001008955, 0.001022849, 0.001033936, 0.001045234, 0.001057371,
0.001071207, 0.001085629, 0.001094893, 0.001106005, 0.001120353,
0.001132616, 0.001145793, 0.001156872, 0.001170388, 0.001182704,
0.001195934, 0.001208907, 0.001222069, 0.001237532, 0.001252475,
0.001267303, 0.001279582, 0.001293584, 0.001308188, 0.001320279,
0.001334946, 0.00135138, 0.001367365, 0.001382517, 0.001396851,
0.00141322, 0.001431087, 0.001446578, 0.001463723, 0.001480189,
0.001493637, 0.001513629, 0.001529922, 0.001551964, 0.001568307,
0.001586021, 0.00160327, 0.001624866, 0.001641832, 0.001659574,
0.001681092, 0.001701537, 0.001718758, 0.001738425, 0.001757453,
0.001777873, 0.00180048, 0.001824612, 0.001846783, 0.001863458,
0.001884942, 0.00190526, 0.001925835, 0.001946874, 0.001971254,
0.001996102, 0.00201698, 0.002037683, 0.002060103, 0.002082758,
0.002106409, 0.002129007, 0.002154989, 0.002181851, 0.002207108,
0.002231596, 0.002256836, 0.002282343, 0.002309303, 0.002334112,
0.002358345, 0.002387156, 0.002410807, 0.002445481, 0.002476899,
0.002507092, 0.002534101, 0.002562057, 0.002592023, 0.002626494,
0.002658379, 0.002688962, 0.002720578, 0.002750578, 0.002786117,
0.002817885, 0.002853243, 0.002891893, 0.00292367, 0.00295806,
0.002993221, 0.003031546, 0.003065154, 0.003097967, 0.003134769,
0.003171656, 0.003209061, 0.003249423, 0.003285864, 0.003322365,
0.003360666, 0.003400079, 0.003437071, 0.003473894, 0.003515581,
0.003559024, 0.00360032, 0.003649522, 0.003698003, 0.003741713,
0.003781833, 0.003827065, 0.003873217, 0.003916216, 0.003960069,
0.004001991, 0.004051017, 0.004094639, 0.004146773, 0.004192485,
0.004237869, 0.004288254, 0.004334161, 0.004386776, 0.00443606,
0.004491919, 0.004542123, 0.004596868, 0.004648514, 0.004704818,
0.004756557, 0.004812243, 0.004867444, 0.004926127, 0.004977904,
0.005042279, 0.005101005, 0.005168282, 0.005227751, 0.005288878,
0.005347916, 0.005411128, 0.005475387, 0.005537296, 0.00560163,
0.005665474, 0.005731182, 0.005799887, 0.005871427, 0.005941132,
0.006009724, 0.006076033, 0.006148237, 0.006218893, 0.006292189,
0.006359582, 0.006439367, 0.006510455, 0.006584907, 0.006662414,
0.006741152, 0.006818895, 0.00689376, 0.006971988, 0.007052107,
0.007136479, 0.007215388, 0.007296005, 0.007384794, 0.007464833,
0.007552816, 0.007643337, 0.007730512, 0.007823017, 0.007915192,
0.008008941, 0.008103251, 0.008198545, 0.00829169, 0.008387876,
0.008482239, 0.008579054, 0.008673823, 0.008770975, 0.008870707,
0.008970726, 0.009071687, 0.009173407, 0.009286794, 0.009395629,
0.009511466, 0.009617749, 0.009724064, 0.009840628, 0.009930635,
0.009953314, 0.010056113, 0.010174415, 0.010297218, 0.010414827,
0.010536746, 0.010658477, 0.010779294, 0.010904682, 0.011033844,
0.011165349, 0.011294265, 0.011428849, 0.011567471, 0.011703105,
0.011831625, 0.011961905, 0.012097813, 0.012246705, 0.012386111,
0.012535143, 0.012675752, 0.012823601, 0.012964074, 0.013109329,
0.013267414, 0.013417575, 0.013573353, 0.013723856, 0.013883887,
0.014044804, 0.014214641, 0.014373087, 0.014549717, 0.014719474,
0.014891004, 0.015065456, 0.015230503, 0.015410477, 0.01558535,
0.01576399, 0.015940998, 0.016131492, 0.016314916, 0.0164995,
0.01669181, 0.016886333, 0.017087569, 0.017289212, 0.017495236,
0.017695021, 0.017903128, 0.018123719, 0.018334283, 0.018543303,
0.018763286, 0.018974197, 0.01919034, 0.019419827, 0.019648392,
0.019868826, 0.0200986, 0.020326864, 0.020557739, 0.020800127,
0.021045922, 0.021294759, 0.021544889, 0.021797034, 0.022050806,
0.022318409, 0.022581664, 0.022838818, 0.023113601, 0.023382366,
0.023654476, 0.023943223, 0.024222676, 0.024504254, 0.024786238,
0.025072508, 0.025358893, 0.02564041, 0.025951102, 0.026253172,
0.026554094, 0.026870869, 0.027189375, 0.027500485, 0.027815363,
0.028147987, 0.028473854, 0.028811833, 0.029151045, 0.029484778,
0.029840035, 0.030182473, 0.030533674, 0.030903302, 0.031260466,
0.031623893, 0.031985371, 0.032351336, 0.032721524, 0.033089718,
0.033469262, 0.033866059, 0.034265423, 0.034659535, 0.035069018,
0.03548302, 0.035901449, 0.036299412, 0.036721638, 0.037147432,
0.037603547, 0.038036559, 0.038474564, 0.03893835, 0.039390503,
0.03985716, 0.040330201, 0.040808614, 0.041262245, 0.041744501,
0.042240743, 0.042737184, 0.043217702, 0.043720182, 0.044229595,
0.044751914, 0.045273508, 0.045808981, 0.04632464, 0.04684647,
0.047406408, 0.047951898, 0.048507716, 0.049067145, 0.049650962,
0.050251409, 0.050819259, 0.051411028, 0.052019155, 0.052621082,
0.053259754, 0.053892309, 0.054529077, 0.055152761, 0.055778699,
0.056427688, 0.05708661, 0.05775989, 0.05844234, 0.05914142,
0.059815449, 0.060542863, 0.061255231, 0.061978805, 0.062713398,
0.063427199, 0.064209904, 0.0649629, 0.065734681, 0.066501624,
0.06728921, 0.068061513, 0.068863178, 0.069677794, 0.070498,
0.071347865, 0.07218547, 0.073036561, 0.073887898, 0.074748795,
0.075646941, 0.076541067, 0.077463257, 0.078371762, 0.079294904,
0.080228659, 0.081174575, 0.082136122, 0.083112567, 0.084080915,
0.085056381, 0.086073347, 0.087087481, 0.088084922, 0.089127502,
0.090161475, 0.091218411, 0.092288735, 0.09340744, 0.094543748,
0.095669537, 0.096791278, 0.09795662, 0.099127195, 0.100324926,
0.101269506, 0.101512671, 0.102694862, 0.103907066, 0.105130157,
0.106343017, 0.107637831, 0.108917362, 0.110199767, 0.111497711,
0.112820924, 0.114166762, 0.115513864, 0.116889578, 0.118285102,
0.119715286, 0.121146989, 0.122557836, 0.123999963, 0.125490773,
0.127002164, 0.128559446, 0.130117799, 0.131666819, 0.133245883,
0.134843891, 0.136434495, 0.138071105, 0.139695146, 0.141377841,
0.143025099, 0.144712048, 0.146484536, 0.148265951, 0.150054487,
0.151863707, 0.153678822, 0.155546528, 0.157445894, 0.159378584,
0.16127648, 0.163233911, 0.165220328, 0.167205386, 0.169207173,
0.171241583, 0.173306295, 0.175410707, 0.177544072, 0.179671827,
0.181875594, 0.184126187, 0.186359403, 0.188623434, 0.190900709,
0.193216497, 0.195565062, 0.197960046, 0.200381666, 0.202850093,
0.205293192, 0.207810895, 0.210362019, 0.212956329, 0.215518686,
0.21816936, 0.2208806, 0.223603901, 0.226385057, 0.229143349,
0.231985751, 0.23482168, 0.237687093, 0.240638413, 0.243595793,
0.24661656, 0.249649324, 0.25271809, 0.255777426, 0.258916117,
0.262152814, 0.265407384, 0.268695049, 0.272022597, 0.275403282,
0.278797462, 0.282287125, 0.285795577, 0.289310268, 0.292934226,
0.296581763, 0.300287972, 0.304025139, 0.307837455, 0.311699543,
0.315601897, 0.31955116, 0.320436997, 0.323612939, 0.327697997,
0.331861897, 0.336009706, 0.340237773, 0.344577254, 0.348944171,
0.353366288, 0.357856636, 0.362355215, 0.366907623, 0.371539207,
0.376264112, 0.381048939, 0.385865788, 0.390800155, 0.395791822,
0.400878989, 0.406037887, 0.411262358, 0.416630108, 0.421979023,
0.427393993, 0.432926261, 0.438545755, 0.43974163, 0.444262503,
0.450077806, 0.45591108, 0.461838344, 0.467916429, 0.473967582,
0.480154108, 0.48641231, 0.492832985, 0.499267884, 0.505787923,
0.512445466, 0.519210777, 0.526033606, 0.533041678, 0.540132802,
0.547313913, 0.554534185, 0.56195817, 0.569539305, 0.577144126,
0.584818106, 0.592708947, 0.600656244, 0.608779217, 0.617033414,
0.625326132, 0.63386066, 0.642506026, 0.651271439, 0.660308695,
0.669343907, 0.678496284, 0.687811049, 0.697395995, 0.701433696,
0.707008968, 0.71688292, 0.726837425, 0.736949456, 0.747299233,
0.757839371, 0.768590847, 0.779304812, 0.790358829, 0.801545576,
0.813004891, 0.824652184, 0.83643861, 0.848561517, 0.860839569,
0.873237031, 0.885940878, 0.898807215, 0.912086718, 0.925448942,
0.93898047, 0.952988749, 0.967195111, 0.981772759, 0.996488979,
1.002659336, 1.011533585, 1.026872347, 1.042371669, 1.058205367,
1.074458722, 1.09104704, 1.107968177, 1.125166609, 1.142753445,
1.160645538, 1.178867673, 1.197612278, 1.216757517, 1.236216802,
1.25615546, 1.276430999, 1.297232504, 1.318443414, 1.34024465,
1.357931643, 1.398277137, 1.420627753, 1.442855232, 1.465339526,
1.487806492, 1.51027426, 1.532863565, 1.55518873, 1.57760317,
1.599794104, 1.622303248, 1.644646639, 1.667104133, 1.68940649,
1.711715774, 1.734081696, 1.756341451, 1.778712715, 1.791714751,
1.80093287, 1.823375273, 1.845778989, 1.86808116, 1.890502199,
1.913046114, 1.935507021, 1.9579107, 1.980191557, 2.002701846,
2.025318947, 2.047779854, 2.070306402, 2.092856202, 2.115372469,
2.137899054, 2.160152398, 2.182625092, 2.205053356, 2.227576069,
2.25009301, 2.272490638, 2.294855657, 2.31745729, 2.339687688,
2.352784426, 2.362338915, 2.384734743, 2.407454626, 2.429879675,
2.452345596, 2.474819495, 2.497560322, 2.520294619, 2.542873033,
2.565461007, 2.587916722, 2.610303005, 2.633030433, 2.655569141,
2.67815637, 2.700894697, 2.723508491, 2.746047139, 2.768836256,
2.791491986, 2.81414865, 2.836817195, 2.859535279, 2.882248664,
2.90494815, 2.927642139, 2.950500234, 2.973199561, 2.996005575,
3.01852218, 3.04143652, 3.064254766, 3.08696452, 3.109719517,
3.132284047, 3.150233208, 3.154904707, 3.177518944, 3.200070439,
3.223085186, 3.245889815, 3.268581446, 3.291363895, 3.314064341,
3.336920648, 3.359671254, 3.382824332, 3.405831244, 3.428880893,
3.451801983, 3.474295141, 3.497069478, 3.520121086, 3.543040798,
3.565812537, 3.588648035, 3.611885231, 3.635308733, 3.658352415,
3.681446271, 3.704592291, 3.72275618, 3.727782289, 3.751175736,
3.774372326, 3.79747392, 3.820661543, 3.84375931, 3.866915856,
3.890087249, 3.912815769, 3.93613948, 3.959403241, 3.982547884,
4.005537176, 4.029456016, 4.05290932, 4.076234494, 4.09950343,
4.122970868, 4.14622911, 4.169698177, 4.193558474, 4.217114459,
4.240544631, 4.264273449, 4.287878802, 4.311355283, 4.33495505,
4.358201356, 4.381357732, 4.404865589, 4.428253439, 4.451816733,
4.475485654, 4.498923768, 4.522709624, 4.546249185, 4.570363145,
4.593556634, 4.617323282, 4.641174163, 4.664844263, 4.68876457,
4.712317907, 4.73638299, 4.760072388, 4.784009097, 4.808013055,
4.832176243, 4.856159607, 4.879796281, 4.903847617, 4.927568606,
4.951370553, 4.975219739, 4.999425165, 5.02399793, 5.048271294,
5.072384001, 5.096851306, 5.121451596, 5.145068096, 5.169443984,
5.193592183, 5.217926387, 5.242406766, 5.267069849, 5.291370622,
5.3158874, 5.340285422, 5.364716514, 5.388935226, 5.413726713,
5.438723293, 5.463177482, 5.488151501, 5.513147549, 5.537689171,
5.562553809, 5.587369563, 5.611533421, 5.63634903, 5.660998501,
5.685690339, 5.710540349, 5.735240309, 5.760223278, 5.785379224,
5.81058182, 5.835242701, 5.860362919, 5.885938784, 5.911236382,
5.936892029, 5.962403731, 5.987832739, 6.012867366, 6.018207803,
6.038310414, 6.063555767, 6.088389549, 6.114251595, 6.139325991,
6.16505587, 6.19019936, 6.215035092, 6.240783547, 6.266751503,
6.292138139, 6.317962412, 6.343385909, 6.369749669, 6.395414942,
6.421238347, 6.447090618, 6.47375521, 6.499618794, 6.526007761,
6.552150251, 6.578060103, 6.60442248, 6.630879073, 6.656959637,
6.683034283, 6.709817901, 6.736579057, 6.762808405, 6.789650034,
6.816670682, 6.843444285, 6.870190848, 6.896653312, 6.922976714,
6.949410856, 6.975781186, 7.003134607, 7.029728354, 7.057342421,
7.084363633, 7.11062707, 7.137326882, 7.164559278, 7.191554273,
7.219005114, 7.245949975, 7.273036124, 7.299211891, 7.326472888,
7.353682459, 7.380198468, 7.407413287, 7.43503963, 7.462437045,
7.490824057, 7.518079384, 7.54522333, 7.573040329, 7.60051581,
7.627887504, 7.655744416, 7.683796803, 7.712069548, 7.740210449,
7.767995289, 7.79690927, 7.824213804, 7.851626407, 7.879881152,
7.908902077, 7.937769976, 7.966033022, 7.995271463, 8.0239938,
8.05159895, 8.081220385, 8.109303744, 8.138303267, 8.166646227,
8.195605657, 8.223856372, 8.253152374, 8.281561297, 8.310207454,
8.338788755, 8.367529671, 8.396683903, 8.425367224, 8.455255906,
8.484687715, 8.513571646, 8.542277191, 8.570569374, 8.599927875,
8.628936065, 8.657432311, 8.686885247, 8.716439257, 8.74657098,
8.77660767, 8.806689051, 8.836692891, 8.867154939, 8.894903293,
8.923066222, 8.953073646, 8.982715133, 9.01281534, 9.043260037,
9.074307116, 9.104462266, 9.134647217, 9.164655895, 9.194556846,
9.224431715, 9.255679884, 9.286348335, 9.316669342, 9.34801196,
9.377393309, 9.407564688, 9.438020907, 9.469406126, 9.501202486,
9.531037046, 9.562006923, 9.594707878, 9.624065648, 9.654667582,
9.6853032, 9.71519593, 9.745725499, 9.776807272, 9.80799562,
9.839917857, 9.871575762, 9.90374516, 9.934506229, 9.966715946,
9.999690644, 10.03175538, 10.064654045, 10.098590464, 10.129886474,
10.161656434, 10.193586504, 10.225453899, 10.257336906, 10.289431808,
10.32146145, 10.353853677, 10.385004416, 10.415340945, 10.447176691,
10.480413392, 10.513314592, 10.545860204, 10.578817059, 10.614763707,
10.645337414, 10.679973375, 10.712078659, 10.74468375, 10.776557157,
10.809167151, 10.843028001, 10.875888854, 10.907981212, 10.941427913,
10.974833425, 11.009198465, 11.042054854, 11.075696601, 11.108373206,
11.142076256, 11.177074479, 11.211640714, 11.245463319, 11.279385295,
11.31326608, 11.346739309, 11.380947317, 11.414235863, 11.447709762,
11.480948589, 11.515708904, 11.550479024, 11.584566456, 11.617981749,
11.65263368, 11.688025544, 11.721377891, 11.756258209, 11.790452322,
11.82394084, 11.859973549, 11.895459077, 11.930500015, 11.965009393,
11.971103554, 11.998278917, 12.035505633, 12.068894099, 12.106530595,
12.141621862, 12.174936423, 12.209782363, 12.244506078, 12.28142006,
12.317875797, 12.35078942, 12.386191379, 12.423800109, 12.461231787,
12.495682102, 12.534300463, 12.572178564, 12.610720193, 12.64481116,
12.681109192, 12.718231955, 12.753885483, 12.790180515, 12.827538488,
12.86407267, 12.901970935, 12.936523075, 12.971271903, 13.009024878,
13.04438126, 13.077419398, 13.114134972, 13.152412424, 13.189959603,
13.228409301, 13.268225079, 13.30440195, 13.338069146, 13.376742958,
13.414638707, 13.454468083, 13.494275836, 13.532079747, 13.569685351,
13.605382032, 13.643473958, 13.681424671, 13.719625017, 13.757164701,
13.79405773, 13.832919425, 13.870850099, 13.910856725, 13.948840593,
13.988370002, 14.026642593, 14.065658833, 14.099737238, 14.138177058,
14.176493667, 14.214922385, 14.251774564, 14.291584165, 14.330689447,
14.369386466, 14.412705725, 14.451034136, 14.489017769, 14.532634475,
14.570140818, 14.608799094, 14.645874725, 14.689380956, 14.730929552,
14.770403977, 14.807128265, 14.847827397, 14.88521805, 14.92364729,
14.961915961, 14.999697934, 15.036439638, 15.073776177, 15.110787482,
15.149795889, 15.194035627, 15.231205969, 15.273125529, 15.315532862,
15.356905307, 15.390934509, 15.43005507, 15.472395089, 15.514743319,
15.554932584, 15.594404232, 15.638060689, 15.672821312, 15.71311781,
15.750127997, 15.793500171, 15.841602912, 15.881364823, 15.924793596,
15.963267235, 16.000779479, 16.037777265, 16.077594446, 16.121969555,
16.165579533, 16.206087043, 16.244573797, 16.284087685, 16.327880821,
16.373187627, 16.409850952, 16.456786558, 16.498130944, 16.538940416,
16.580340108, 16.624824799, 16.670961077, 16.714110619, 16.755887196,
16.801665587, 16.848314295, 16.893805033, 16.939904473, 16.991652362,
17.039402926, 17.090611701, 17.133538273, 17.179692671, 17.216093297,
17.256501136, 17.29833338, 17.343507434, 17.385523175, 17.427808601,
17.473739931, 17.512458695, 17.55376207, 17.600292196, 17.640882113,
17.685526778, 17.730845234, 17.768556358, 17.810075665, 17.851405085,
17.892590298, 17.930533199, 17.974883739, 18.020051824, 18.066170861,
18.102111727, 18.147958696, 18.194983874, 18.240637892, 18.287260083,
18.323742784, 18.364275985, 18.401410317, 18.441368786, 18.491787677,
18.538702514, 18.590128474, 18.635309184, 18.680841185, 18.727965868,
18.772300879, 18.818803283, 18.864352329, 18.904939217, 18.95016447,
18.994401079, 19.046271059, 19.082408549, 19.131460414, 19.178004314,
19.218801982, 19.261740542, 19.301760184, 19.341236214, 19.388253209,
19.431356623, 19.472752027, 19.513076562, 19.551692396, 19.599232743,
19.638316153, 19.680382254, 19.729324554, 19.78311108, 19.844413854,
19.879252815, 19.919825462, 19.975068247, 20.025687891, 20.072518325,
20.110886998, 20.128081014, 20.169125058, 20.215976179, 20.263503132,
20.310612101, 20.345020355, 20.402118367, 20.460427196, 20.502555645,
20.551178063, 20.605504985, 20.652122001, 20.704488477, 20.754369038,
20.788472809, 20.827010383, 20.854642463, 20.903418966, 20.948287524,
21.006649087, 21.065962278, 21.110384924, 21.151994732, 21.213153618,
21.276295001, 21.328537984, 21.375643435, 21.429377049, 21.489267378,
21.538877529, 21.592730365, 21.641462657, 21.697726047, 21.741103077,
21.800450378, 21.855036304, 21.903819857, 21.955076884, 22.002407767,
22.045681067, 22.088619216, 22.133688958, 22.17279244, 22.22759466,
22.277606606, 22.349116865, 22.395332239, 22.454928026, 22.492114507,
22.5335903, 22.585986824, 22.643244159, 22.690518404, 22.765071433,
22.828359884, 22.867793703, 22.929819393, 22.978341318, 23.042315839,
23.104946734, 23.159874091, 23.205425075, 23.237584505, 23.29192309,
23.332039984, 23.38655006, 23.423923791, 23.480983059, 23.511270892,
23.564663701, 23.606431337, 23.665713367, 23.713914371, 23.750440349,
23.83692907, 23.875839203, 23.927170148, 23.979727538, 24.04744189,
24.11800154, 24.170320245, 24.212957101, 24.259692575, 24.331170263,
24.383081649, 24.454871211, 24.505407441, 24.564187953, 24.606364392,
24.667614039, 24.727986567, 24.785641141, 24.843561178, 24.893510814,
24.947860468, 25.003659619, 25.07199456, 25.136088418, 25.214366002,
25.247313836, 25.32245746, 25.364520735, 25.424114178, 25.479364044,
25.522802065, 25.575517863, 25.631853221, 25.689985402, 25.752332422,
25.802260334, 25.833019136, 25.866032057, 25.918868343, 25.958726527,
26.023962077, 26.055960592, 26.104636351, 26.142758848, 26.221864542,
26.267510941, 26.30606783, 26.338651575, 26.420018431, 26.467902508,
26.522116456, 26.55598812, 26.60118913, 26.674762784, 26.722190345,
26.761777447, 26.84437887, 26.892265327, 26.94118676, 26.98608192,
27.040424992, 27.0997155, 27.158187351, 27.238544127, 27.319151355,
27.380372826, 27.453694496, 27.527193999, 27.569440412, 27.636989289,
27.662847899, 27.712015067, 27.761327817, 27.836586881, 27.913129959,
27.981212181, 28.049944503, 28.137567088, 28.175747425, 28.270791609,
28.292786868, 28.346059669, 28.43816374, 28.509443421, 28.558549598,
28.621104419, 28.680815907, 28.773272175, 28.828602833, 28.900834703,
28.923200479, 28.994028217, 29.058703991, 29.119930943, 29.199810948,
29.243220861, 29.276462308, 29.314352725, 29.357288842, 29.397896218,
29.485177302, 29.531216534, 29.57464366, 29.660637701, 29.696946413,
29.75420616, 29.786526243, 29.855013669, 29.920288875, 29.989002565,
30.035939576, 30.105316534, 30.153648879, 30.234207363, 30.268747816,
30.384515317, 30.430541947, 30.506694486, 30.575730853, 30.630199164,
30.682712078, 30.72026582, 30.776704323, 30.814868589, 30.869474797,
30.944085274, 30.964573291, 31.009524066, 31.054147689, 31.081272822,
31.132684141, 31.212187568, 31.280157847, 31.369542877, 31.428073902,
31.461385688, 31.4946169, 31.543433571, 31.591121796, 31.6584126,
31.689342907, 31.735535054, 31.848386582, 31.897110002, 31.950285711,
31.992722355, 32.089818042, 32.151777874, 32.249571823, 32.297425024,
32.398991692, 32.469577166, 32.566622575, 32.630667244, 32.738736606,
32.780642994, 32.843150707, 32.897517289, 32.954468423, 33.006420947,
33.068993482, 33.139020076, 33.190405733, 33.235932381, 33.253707904,
33.324041461, 33.416805241, 33.540840672, 33.583914453, 33.666985197,
33.705743271, 33.80090168, 33.856271559, 33.915238873, 33.969648157,
34.000572567, 34.101838382, 34.207119684, 34.360237318, 34.434008667,
34.494776717, 34.513529009, 34.566050281, 34.658779631, 34.753837492,
34.801640701, 34.872282315, 34.964060761, 35.078031269, 35.120185776,
35.203808161, 35.291778556, 35.450368538, 35.52288054, 35.597986107,
35.690497, 35.730722665, 35.794333741, 35.838063034, 36.059208058,
36.129263774, 36.169818263, 36.240156283, 36.308225576, 36.337594598,
36.393600753, 36.45822071, 36.549820233, 36.562347847, 36.608831358,
36.705273047, 36.764182818, 36.842432099, 37.010863194, 37.101026885,
37.280058217, 37.30146787, 37.368160152, 37.442581065, 37.45807429,
37.494084094, 37.628309004, 37.831186753, 37.9813575, 38.039334241,
38.092772981, 38.302386917, 38.343143608, 38.478730355, 38.544825363,
38.570504001, 38.609939266, 38.731521011, 38.778094647, 38.834390457,
38.969807286, 39.217661473, 39.218565094, 39.290387909, 39.298300934,
39.308237003, 39.320686929, 39.354856367, 39.53870889, 39.571738646,
39.676400236, 39.75364742, 39.807568455, 39.827533501, 39.84619277,
39.98243036, 40.022169738, 40.137975383, 40.21487771, 40.219443896,
40.23214859, 40.315357471, 40.350356397, 40.432050237, 40.48518292,
40.748858137, 40.801278542, 40.833267647, 41.044011231, 41.109762981,
41.112099824, 41.146532736), "2400" = c(1.9153e-05, 1.9492e-05,
1.9722e-05, 2.0133e-05, 2.0616e-05, 2.1255e-05, 2.1271e-05, 2.1503e-05,
2.2242e-05, 2.2762e-05, 2.3285e-05, 2.4045e-05, 2.4393e-05, 2.4616e-05,
2.4878e-05, 2.5373e-05, 2.5784e-05, 2.5974e-05, 2.6069e-05, 2.6155e-05,
2.6289e-05, 2.6564e-05, 2.712e-05, 2.726e-05, 2.7466e-05, 2.759e-05,
2.7727e-05, 2.8275e-05, 2.8312e-05, 2.8625e-05, 2.8854e-05, 2.9597e-05,
3.0016e-05, 3.0173e-05, 3.0553e-05, 3.0956e-05, 3.1233e-05, 3.1432e-05,
3.2004e-05, 3.2614e-05, 3.2711e-05, 3.2871e-05, 3.4163e-05, 3.496e-05,
3.5163e-05, 3.5343e-05, 3.5898e-05, 3.6221e-05, 3.6644e-05, 3.7308e-05,
3.7632e-05, 3.8202e-05, 3.9174e-05, 3.9567e-05, 4.0097e-05, 4.0732e-05,
4.1156e-05, 4.1518e-05, 4.1676e-05, 4.182e-05, 4.218e-05, 4.2446e-05,
4.3502e-05, 4.3754e-05, 4.4179e-05, 4.4409e-05, 4.4773e-05, 4.4986e-05,
4.6416e-05, 4.6693e-05, 4.7664e-05, 4.8021e-05, 4.8412e-05, 4.9876e-05,
5.003e-05, 5.0835e-05, 5.1256e-05, 5.1492e-05, 5.2274e-05, 5.3425e-05,
5.4139e-05, 5.49e-05, 5.5465e-05, 5.598e-05, 5.7155e-05, 5.7506e-05,
5.8728e-05, 5.9033e-05, 5.9739e-05, 6.0075e-05, 6.041e-05, 6.114e-05,
6.2552e-05, 6.3085e-05, 6.3215e-05, 6.4046e-05, 6.4341e-05, 6.4674e-05,
6.5373e-05, 6.6745e-05, 6.728e-05, 6.7435e-05, 6.8311e-05, 6.8581e-05,
6.9132e-05, 6.9311e-05, 7.0258e-05, 7.1203e-05, 7.1958e-05, 7.2506e-05,
7.312e-05, 7.3499e-05, 7.4586e-05, 7.4897e-05, 7.5293e-05, 7.578e-05,
7.6852e-05, 7.71e-05, 7.7396e-05, 7.7832e-05, 7.8762e-05, 7.9235e-05,
7.9697e-05, 8.016e-05, 8.1243e-05, 8.2304e-05, 8.3192e-05, 8.4095e-05,
8.4602e-05, 8.5914e-05, 8.7528e-05, 8.8851e-05, 8.9921e-05, 9.0659e-05,
9.1322e-05, 9.1947e-05, 9.3708e-05, 9.4398e-05, 9.5692e-05, 9.6786e-05,
9.7833e-05, 9.9894e-05, 0.000100947, 0.000101344, 0.000102409,
0.000103426, 0.000104458, 0.000105045, 0.000107457, 0.000108787,
0.000109871, 0.000111403, 0.00011225, 0.000113849, 0.000115285,
0.000117037, 0.000118909, 0.000120592, 0.000122355, 0.000124865,
0.000125892, 0.000126447, 0.000128146, 0.000129765, 0.000130586,
0.000131929, 0.000133892, 0.000135507, 0.000137285, 0.00013882,
0.000140808, 0.000142258, 0.000143392, 0.000146107, 0.000147297,
0.00014845, 0.000149237, 0.00015119, 0.00015239, 0.000154087,
0.000156415, 0.000158396, 0.000159915, 0.000162057, 0.000165078,
0.000168089, 0.000170266, 0.000172299, 0.000174267, 0.000176141,
0.00017794, 0.000180112, 0.000182032, 0.000184665, 0.000186377,
0.000188278, 0.000190267, 0.000191473, 0.000193227, 0.000195082,
0.000196336, 0.000197967, 0.000200679, 0.000203023, 0.00020533,
0.000207138, 0.0002103, 0.000213078, 0.000214908, 0.000216858,
0.000218814, 0.00022039, 0.000222266, 0.000224067, 0.00022778,
0.000230371, 0.000232872, 0.000235019, 0.000237498, 0.000242325,
0.0002478, 0.000251056, 0.000253986, 0.000255367, 0.000257478,
0.00025986, 0.000262703, 0.000266565, 0.000271263, 0.00027502,
0.000277441, 0.000280389, 0.000283326, 0.000285581, 0.000290157,
0.000293012, 0.00029682, 0.000300095, 0.000302782, 0.000307121,
0.000311122, 0.000313893, 0.000318602, 0.000323209, 0.000325126,
0.00032861, 0.000331161, 0.000333796, 0.000336815, 0.000341932,
0.000345581, 0.000349392, 0.00035564, 0.000361531, 0.000366277,
0.000369529, 0.000373339, 0.000378272, 0.000382412, 0.000387271,
0.000390537, 0.000395817, 0.000400601, 0.000404521, 0.000409952,
0.000414135, 0.000419169, 0.00042465, 0.000428981, 0.000432551,
0.000437116, 0.000441972, 0.00044743, 0.000452246, 0.000457114,
0.000463828, 0.000468802, 0.000474012, 0.000479012, 0.000482649,
0.000487169, 0.000492929, 0.000498259, 0.000503745, 0.000509763,
0.000513576, 0.000518694, 0.000524898, 0.00053374, 0.000540728,
0.000547471, 0.000552903, 0.000558219, 0.00056694, 0.000573611,
0.000581725, 0.000588822, 0.000595552, 0.000601719, 0.000608019,
0.000613146, 0.000619941, 0.000629632, 0.000637084, 0.000643112,
0.000649759, 0.000656482, 0.000663163, 0.000671553, 0.00067943,
0.000686192, 0.000694356, 0.000703456, 0.000712528, 0.000721253,
0.000730114, 0.000738225, 0.00074729, 0.000757841, 0.000765947,
0.000773796, 0.000782988, 0.000794662, 0.000805332, 0.000814862,
0.000826721, 0.000835808, 0.000845238, 0.000854215, 0.000863486,
0.000872939, 0.000884606, 0.00089401, 0.000900543, 0.000913095,
0.000923649, 0.000936204, 0.0009463, 0.000957887, 0.000970848,
0.000979369, 0.000981168, 0.000991246, 0.001002185, 0.001011767,
0.001024025, 0.00103687, 0.001048088, 0.001063923, 0.001076834,
0.001088496, 0.001101524, 0.001112938, 0.001127152, 0.001142777,
0.001155587, 0.001171065, 0.001186109, 0.001199367, 0.001214884,
0.001227338, 0.001247847, 0.001262596, 0.00127828, 0.001293359,
0.001309843, 0.001322773, 0.00133929, 0.001354437, 0.001369085,
0.001384321, 0.001403076, 0.001417176, 0.0014344, 0.001451698,
0.001468393, 0.00148381, 0.001503589, 0.001519888, 0.001536722,
0.001553903, 0.001574655, 0.001594193, 0.001612789, 0.001629309,
0.001647077, 0.001666395, 0.001683459, 0.001705521, 0.001727294,
0.001748351, 0.001767089, 0.001786798, 0.001807303, 0.001826541,
0.001848261, 0.001870537, 0.001891286, 0.001910911, 0.001931361,
0.001955136, 0.001978998, 0.00199966, 0.00202569, 0.002047329,
0.00206937, 0.002094927, 0.002119931, 0.002144034, 0.002170549,
0.002193687, 0.002219459, 0.002241756, 0.002268696, 0.002297271,
0.002325325, 0.002348473, 0.002378283, 0.002405831, 0.002433776,
0.00246389, 0.002494876, 0.002521592, 0.002551777, 0.002580125,
0.002611893, 0.002644627, 0.002672017, 0.002704129, 0.002731378,
0.002765283, 0.002795026, 0.002824441, 0.002859184, 0.002890874,
0.002922257, 0.002955474, 0.002988574, 0.003027056, 0.003065257,
0.003105392, 0.003134643, 0.003169331, 0.003211305, 0.003245485,
0.003285037, 0.003321736, 0.003357099, 0.00339218, 0.003434201,
0.00347816, 0.003519709, 0.003562599, 0.00360059, 0.003641282,
0.003678794, 0.003724275, 0.003772538, 0.003810009, 0.003851523,
0.003895599, 0.003942072, 0.003986513, 0.00403523, 0.004083008,
0.004129935, 0.004176546, 0.004222643, 0.004275023, 0.004328916,
0.004382876, 0.004429201, 0.004485914, 0.00453693, 0.004589279,
0.0046401, 0.004693341, 0.004746714, 0.004803795, 0.004859776,
0.004923945, 0.004981061, 0.005039741, 0.005097638, 0.005160248,
0.00522229, 0.005277232, 0.005338483, 0.005400067, 0.0054634,
0.00552893, 0.005592599, 0.005659052, 0.005724859, 0.005788473,
0.005849801, 0.005918045, 0.005988118, 0.006055939, 0.006125635,
0.006193725, 0.006263315, 0.006339387, 0.00641222, 0.006479834,
0.006554937, 0.00663772, 0.006713929, 0.006788967, 0.006868985,
0.006952025, 0.007031115, 0.007112721, 0.007196273, 0.007282047,
0.007368182, 0.007456393, 0.007546719, 0.007633792, 0.007721986,
0.007807099, 0.007895626, 0.007985061, 0.008072466, 0.0081646,
0.008264197, 0.008360445, 0.008455306, 0.008543432, 0.008642079,
0.008737845, 0.0088393, 0.008940402, 0.009043367, 0.009144582,
0.009251194, 0.009357175, 0.009465266, 0.00957095, 0.009695366,
0.009806292, 0.009902308, 0.00992577, 0.010037538, 0.010158331,
0.010288325, 0.010406513, 0.010534424, 0.010661354, 0.010787162,
0.010910739, 0.011037053, 0.011167272, 0.011297004, 0.011434929,
0.011568578, 0.01170442, 0.011843383, 0.01198444, 0.012119753,
0.012261404, 0.012395913, 0.012551178, 0.012700056, 0.012848033,
0.012997318, 0.013146973, 0.013301658, 0.013461008, 0.013621841,
0.013777838, 0.01394194, 0.014113125, 0.014283692, 0.014445424,
0.014616024, 0.014789371, 0.014956366, 0.015134647, 0.015317981,
0.015491969, 0.015672617, 0.015857459, 0.016039319, 0.016225556,
0.01641352, 0.016596208, 0.016784688, 0.016972102, 0.017169735,
0.0173703, 0.017565036, 0.017771729, 0.01798368, 0.018182486,
0.018396953, 0.01861033, 0.018820867, 0.019029455, 0.019252175,
0.019475727, 0.019710381, 0.019937807, 0.02017058, 0.020414278,
0.020650266, 0.020889231, 0.021132142, 0.021371346, 0.021621174,
0.021869709, 0.022127233, 0.022386793, 0.022649859, 0.022905091,
0.023175456, 0.023438976, 0.023730174, 0.023996692, 0.024278546,
0.024561535, 0.024841194, 0.025134377, 0.025421826, 0.02571303,
0.026025286, 0.02632111, 0.026624597, 0.026934016, 0.027257675,
0.027576471, 0.027905865, 0.028227766, 0.02856755, 0.028894636,
0.029230826, 0.029573877, 0.029932435, 0.030291879, 0.0306357,
0.030993172, 0.031363867, 0.031730714, 0.032091554, 0.032472061,
0.032863473, 0.033241617, 0.03361345, 0.033997616, 0.034377245,
0.03477369, 0.035190776, 0.035605593, 0.036028216, 0.036450911,
0.036869428, 0.037282793, 0.037710328, 0.038133047, 0.038584131,
0.039033076, 0.039491126, 0.039949996, 0.040420422, 0.040878901,
0.041356349, 0.041845059, 0.042329189, 0.042816689, 0.043324664,
0.043819921, 0.0443279, 0.044851862, 0.045363607, 0.045895917,
0.04642139, 0.046962564, 0.047509965, 0.048098252, 0.048665128,
0.049232345, 0.049803657, 0.050390462, 0.050997381, 0.051585541,
0.052197707, 0.052808617, 0.053428572, 0.054060748, 0.054683726,
0.055324423, 0.055963631, 0.056642426, 0.057304287, 0.057997397,
0.058675559, 0.059388612, 0.060069427, 0.06076574, 0.061481034,
0.062192604, 0.062932705, 0.063676885, 0.064421131, 0.065190379,
0.065945866, 0.066732799, 0.067521079, 0.068305556, 0.069123327,
0.069923922, 0.070731114, 0.071581466, 0.07242485, 0.073295023,
0.074152419, 0.075055782, 0.075924499, 0.076834385, 0.077754906,
0.078662861, 0.079619888, 0.080596569, 0.081535211, 0.082508667,
0.083492704, 0.084462608, 0.085464914, 0.086473817, 0.087505933,
0.088547406, 0.089607528, 0.090669667, 0.091730786, 0.092805416,
0.093887623, 0.095015392, 0.096151222, 0.097274742, 0.098437138,
0.099603361, 0.100787651, 0.101749405, 0.101989427, 0.103179732,
0.104400774, 0.105665811, 0.106926862, 0.108210046, 0.10948751,
0.110793641, 0.112121204, 0.113457973, 0.114791346, 0.116132093,
0.117527714, 0.118941477, 0.120373764, 0.121799666, 0.123263359,
0.124732174, 0.126196211, 0.127696465, 0.129227298, 0.130777256,
0.132341779, 0.133944752, 0.135543525, 0.13716952, 0.138837114,
0.140534725, 0.142231041, 0.143945956, 0.145629695, 0.14736549,
0.149093119, 0.150892051, 0.152706299, 0.154535645, 0.156384737,
0.15823181, 0.160135302, 0.162104394, 0.164030433, 0.166014872,
0.167982987, 0.169950458, 0.171929672, 0.173992989, 0.176119038,
0.178246679, 0.180415451, 0.182611953, 0.184832862, 0.187069864,
0.189347965, 0.191643429, 0.194008765, 0.196367694, 0.198793521,
0.201206059, 0.203672428, 0.206181315, 0.208697848, 0.211258479,
0.213848654, 0.216445723, 0.219050253, 0.221743323, 0.224469637,
0.2272582, 0.230003321, 0.232844487, 0.235733013, 0.238611761,
0.241559776, 0.244521555, 0.247553311, 0.250638543, 0.253747546,
0.256869431, 0.26005347, 0.263238367, 0.266512181, 0.26980857,
0.273158513, 0.276567888, 0.279988631, 0.283503677, 0.287046086,
0.290600906, 0.294188883, 0.297839786, 0.301555879, 0.305335102,
0.309150856, 0.313007809, 0.316974041, 0.321002288, 0.321873529,
0.325040444, 0.329157292, 0.333307622, 0.337495445, 0.341771121,
0.346101248, 0.350460295, 0.354860951, 0.359400789, 0.363968212,
0.368608024, 0.373263973, 0.378002481, 0.382774555, 0.387693282,
0.392660537, 0.39767015, 0.402847624, 0.408018867, 0.413306734,
0.418651707, 0.424056519, 0.429539219, 0.435106702, 0.440750939,
0.441964974, 0.446486985, 0.452276991, 0.45812695, 0.464083164,
0.47014266, 0.476299878, 0.48254511, 0.488870073, 0.49527568,
0.501675951, 0.50821581, 0.514893522, 0.521681615, 0.528657754,
0.53565239, 0.542753602, 0.550104791, 0.557498608, 0.564944818,
0.572436326, 0.580040991, 0.587808355, 0.595729133, 0.603790937,
0.612046233, 0.62026104, 0.628657454, 0.637183631, 0.645900855,
0.654776078, 0.663842914, 0.672945603, 0.682125377, 0.691648177,
0.701226181, 0.705406565, 0.71101807, 0.720914188, 0.730891645,
0.741203406, 0.751520378, 0.762119895, 0.77280117, 0.783801911,
0.794992041, 0.806330198, 0.817756583, 0.829502947, 0.841352834,
0.853515953, 0.865866672, 0.878369375, 0.891148135, 0.904137931,
0.917417749, 0.930864078, 0.94476682, 0.958743063, 0.97308741,
0.987608917, 1.002365052, 1.008656581, 1.017446795, 1.032844594,
1.048552003, 1.064571925, 1.080895191, 1.097554972, 1.114406673,
1.131716608, 1.149521516, 1.167497854, 1.185986605, 1.204590036,
1.223804912, 1.243395506, 1.263401571, 1.284003332, 1.305072413,
1.326514523, 1.348509845, 1.366264043, 1.406722985, 1.429296459,
1.451858535, 1.474421144, 1.497129095, 1.519548638, 1.542244637,
1.564811693, 1.587390549, 1.610065995, 1.63275151, 1.655410587,
1.678109991, 1.700716645, 1.723288489, 1.745882557, 1.768364467,
1.790911864, 1.804231305, 1.813572108, 1.83619941, 1.858731141,
1.881366134, 1.904037308, 1.926763844, 1.949339424, 1.972217712,
1.994924062, 2.017531268, 2.040161468, 2.062846614, 2.085343567,
2.107812286, 2.130512936, 2.15305853, 2.175735157, 2.198326425,
2.220909954, 2.243521858, 2.266199902, 2.288763352, 2.311120579,
2.333632545, 2.356340251, 2.369358524, 2.379032954, 2.401654222,
2.424097374, 2.446931906, 2.46949432, 2.49229464, 2.515080228,
2.537707175, 2.560477374, 2.583114877, 2.605946346, 2.628709401,
2.651437851, 2.674244649, 2.696964037, 2.719564726, 2.742292786,
2.765194658, 2.787844427, 2.810622826, 2.833368533, 2.856151378,
2.878948576, 2.901603093, 2.924221071, 2.946973201, 2.969696127,
2.992224415, 3.014953747, 3.03797864, 3.060887882, 3.083923609,
3.106886005, 3.129886355, 3.152699307, 3.170578953, 3.175307637,
3.198216192, 3.221310086, 3.244169594, 3.266976765, 3.28978795,
3.312637209, 3.335654432, 3.358843421, 3.381597504, 3.404748246,
3.427844106, 3.4510632, 3.47399451, 3.496466441, 3.519258944,
3.542385786, 3.565513794, 3.588593931, 3.611638345, 3.634693735,
3.657871615, 3.681041625, 3.704042473, 3.727074556, 3.745038724,
3.75007812, 3.773187687, 3.795839563, 3.818796692, 3.841939517,
3.865048445, 3.88772525, 3.910859125, 3.933861602, 3.957093,
3.980541756, 4.00384177, 4.027145409, 4.050493364, 4.073872028,
4.097136203, 4.120336907, 4.143584459, 4.16677025, 4.190182492,
4.213735271, 4.236929005, 4.26024754, 4.28352139, 4.30689809,
4.330434826, 4.354422537, 4.377893612, 4.401179406, 4.424187745,
4.447341224, 4.471013337, 4.494451203, 4.517926901, 4.541565516,
4.564966006, 4.588316874, 4.611568942, 4.635635993, 4.659190705,
4.682654867, 4.706065242, 4.729277352, 4.753503578, 4.777203746,
4.800657149, 4.82398422, 4.847876923, 4.871316632, 4.895379257,
4.919073638, 4.943060744, 4.966620075, 4.99024534, 5.013760494,
5.037515828, 5.061139579, 5.084899562, 5.108385873, 5.13225388,
5.15611545, 5.180303785, 5.204475917, 5.228595871, 5.252287604,
5.276608875, 5.300831332, 5.324762525, 5.348252326, 5.372262503,
5.396610059, 5.420665917, 5.444813358, 5.468942514, 5.493406507,
5.516907707, 5.540955403, 5.565242131, 5.589643676, 5.613790351,
5.637752691, 5.662335726, 5.686745924, 5.71090422, 5.734933215,
5.759491365, 5.783628859, 5.808138592, 5.832487055, 5.856485434,
5.880370416, 5.904821664, 5.928953375, 5.953497626, 5.978251925,
6.002605664, 6.007657388, 6.027388084, 6.051922416, 6.077243594,
6.102628182, 6.127054703, 6.151896547, 6.17576339, 6.200853456,
6.225557253, 6.25021917, 6.274871443, 6.299729207, 6.323931267,
6.348962329, 6.373485379, 6.399010777, 6.424053851, 6.449416496,
6.474046053, 6.49914466, 6.524256107, 6.549602487, 6.574868617,
6.599894826, 6.62495081, 6.65083129, 6.675360399, 6.701256469,
6.726109352, 6.751545358, 6.777337937, 6.802819056, 6.828194474,
6.853345548, 6.877870158, 6.903467215, 6.928840474, 6.95413259,
6.980353972, 7.00581522, 7.030875906, 7.055656315, 7.081184886,
7.106438236, 7.13215598, 7.157305896, 7.182663102, 7.2083025,
7.234195328, 7.260670409, 7.287094747, 7.31309352, 7.338257132,
7.364269738, 7.390756197, 7.41682421, 7.443115068, 7.468566648,
7.495126082, 7.521338627, 7.54771955, 7.575065728, 7.601183593,
7.627634901, 7.653277688, 7.680908162, 7.706589399, 7.733746919,
7.759531395, 7.786135962, 7.811864251, 7.838383338, 7.865059194,
7.89083847, 7.918120264, 7.944856436, 7.971591303, 7.998737784,
8.025487378, 8.05228014, 8.078739599, 8.10614549, 8.132931985,
8.159622195, 8.185419678, 8.212126303, 8.238603872, 8.265175225,
8.291779316, 8.318075455, 8.345517013, 8.371463809, 8.399460221,
8.427345818, 8.453861992, 8.481493487, 8.507979386, 8.534935824,
8.562952007, 8.589803748, 8.617140311, 8.644119863, 8.672263833,
8.701595318, 8.729020982, 8.756069525, 8.783878402, 8.810489521,
8.837738658, 8.865887043, 8.894559761, 8.921430688, 8.949479449,
8.977692156, 9.006254859, 9.033933094, 9.061343366, 9.089244737,
9.117395375, 9.145215174, 9.173079308, 9.20140614, 9.229037861,
9.257139653, 9.285511487, 9.314554872, 9.343774831, 9.37108454,
9.397714828, 9.426414841, 9.454691162, 9.483656861, 9.513534937,
9.542283357, 9.571199997, 9.60002887, 9.628453641, 9.657220803,
9.686991921, 9.715782066, 9.74497249, 9.77321937, 9.803215998,
9.832271805, 9.860581283, 9.890113213, 9.917657911, 9.94673729,
9.977511966, 10.008074806, 10.037633152, 10.066604078, 10.095700628,
10.125403252, 10.154038903, 10.182790305, 10.212379868, 10.243061027,
10.273232265, 10.303182401, 10.332598323, 10.361749233, 10.391092594,
10.421823943, 10.450531089, 10.478921583, 10.509524886, 10.539126734,
10.569859478, 10.600164672, 10.6311694, 10.661360426, 10.693766339,
10.723881528, 10.753788664, 10.783010522, 10.812149633, 10.840952583,
10.871934822, 10.903924378, 10.934780107, 10.964355359, 10.994396615,
11.025717935, 11.056962736, 11.089530065, 11.119471748, 11.147708572,
11.177563181, 11.20759057, 11.239682694, 11.272678442, 11.303274729,
11.335242132, 11.366269883, 11.399480494, 11.429612219, 11.458482885,
11.488407237, 11.518325134, 11.52512206, 11.549899194, 11.57956235,
11.612249311, 11.644243795, 11.676724137, 11.710888662, 11.74191256,
11.774970802, 11.80541002, 11.837546668, 11.868604971, 11.899445624,
11.929589758, 11.96321781, 11.993032807, 12.024066916, 12.052984803,
12.085125778, 12.116481152, 12.146499526, 12.180899962, 12.213352099,
12.246426427, 12.281534613, 12.314202049, 12.347676978, 12.380985985,
12.416810229, 12.451102395, 12.483108241, 12.514396985, 12.547528039,
12.580161412, 12.61497225, 12.646978531, 12.677812983, 12.711021119,
12.74531074, 12.779762838, 12.814479864, 12.847368331, 12.881768069,
12.916285276, 12.947537908, 12.980487032, 13.011971425, 13.045148545,
13.080877891, 13.115617113, 13.150068266, 13.183345093, 13.213714147,
13.249106295, 13.279503076, 13.31506332, 13.346709795, 13.379462044,
13.415134486, 13.446360408, 13.47939584, 13.51384999, 13.549416402,
13.581638646, 13.616855601, 13.649134936, 13.681900493, 13.716356389,
13.74889091, 13.783028047, 13.819177624, 13.855491404, 13.892446257,
13.925508286, 13.959205497, 13.992171296, 14.025107766, 14.059934924,
14.093107134, 14.129321647, 14.162389382, 14.198062065, 14.234535691,
14.26719793, 14.302195359, 14.339332745, 14.373596653, 14.404218983,
14.43737897, 14.471398826, 14.507400124, 14.544312526, 14.576661101,
14.60905063, 14.640698961, 14.672863159, 14.709212187, 14.745470059,
14.781401676, 14.816912133, 14.84849429, 14.886581873, 14.924228153,
14.963740796, 14.995053621, 15.033135433, 15.067357853, 15.106256051,
15.141555235, 15.180113292, 15.210978686, 15.250337466, 15.28629051,
15.32178351, 15.357069813, 15.392616183, 15.425189542, 15.46117701,
15.490871853, 15.526865657, 15.561279944, 15.597760099, 15.632510979,
15.669414188, 15.706286588, 15.742585848, 15.782935574, 15.81671502,
15.854802125, 15.892984133, 15.931401662, 15.97202766, 16.003221466,
16.035391573, 16.076814201, 16.115426375, 16.154394011, 16.195768373,
16.232138708, 16.264639251, 16.313935604, 16.349794744, 16.389940581,
16.425132326, 16.462586271, 16.500528071, 16.535491967, 16.575394353,
16.609234748, 16.64962958, 16.693754846, 16.728961154, 16.764921143,
16.805416168, 16.84432766, 16.878758866, 16.913721319, 16.949134657,
16.988508505, 17.022197093, 17.061674505, 17.100265273, 17.139215007,
17.187168687, 17.22276253, 17.255249223, 17.289824822, 17.323769245,
17.364305797, 17.395482564, 17.428706097, 17.467536294, 17.503011444,
17.543017806, 17.568914538, 17.60688778, 17.645684903, 17.68140793,
17.727869603, 17.759431061, 17.792620945, 17.830920787, 17.869083124,
17.912768171, 17.964994109, 18.008171611, 18.043888596, 18.080896076,
18.123165615, 18.158301699, 18.192573376, 18.238444392, 18.279229834,
18.315597721, 18.35141685, 18.390166409, 18.438004791, 18.4823666,
18.525537521, 18.57256227, 18.613102742, 18.620828306, 18.654930676,
18.700423767, 18.742318311, 18.784476519, 18.821001284, 18.8681769,
18.910005592, 18.95745331, 18.989328181, 19.030615599, 19.064956848,
19.099134871, 19.132087527, 19.170240584, 19.200086434, 19.234533748,
19.267237744, 19.310652731, 19.350235088, 19.391656399, 19.435880374,
19.471660046, 19.508789936, 19.541360178, 19.592700814, 19.637995658,
19.683299181, 19.729549218, 19.761688166, 19.805248115, 19.85138005,
19.896704582, 19.92985764, 19.979067687, 20.024630269, 20.088255826,
20.130895533, 20.167553821, 20.20700895, 20.244735266, 20.275473973,
20.31015769, 20.353132648, 20.392663556, 20.440629313, 20.479473597,
20.51706807, 20.563908411, 20.590692162, 20.64965377, 20.704603435,
20.73733303, 20.780604039, 20.827684524, 20.868062176, 20.903415915,
20.936481457, 20.980736087, 21.010777197, 21.051857981, 21.093913066,
21.134793118, 21.18046786, 21.218070894, 21.265669413, 21.301981133,
21.344077241, 21.384869432, 21.429725494, 21.478047622, 21.519662534,
21.552490648, 21.582494777, 21.629294081, 21.673382025, 21.717180995,
21.775055105, 21.820476317, 21.878224849, 21.912958048, 21.96599438,
22.00234045, 22.037122397, 22.080628074, 22.124194935, 22.148372152,
22.196475029, 22.253673084, 22.301945583, 22.344504783, 22.400259581,
22.460879302, 22.503668475, 22.541288106, 22.57794609, 22.612970538,
22.661270264, 22.707505194, 22.758304251, 22.791100658, 22.852641031,
22.902908302, 22.939364323, 22.978060352, 23.018785377, 23.06248874,
23.111828385, 23.140127749, 23.171181253, 23.216241877, 23.288243536,
23.329025821, 23.354886756, 23.38627557, 23.424526296, 23.476009931,
23.520280985, 23.563230238, 23.586951711, 23.647798112, 23.686948492,
23.744669079, 23.781861524, 23.830822227, 23.866316751, 23.924133292,
23.975925233, 24.043604116, 24.095449349, 24.145992963, 24.180905177,
24.219765822, 24.249157577, 24.311605994, 24.34659419, 24.414415004,
24.452611599, 24.499535902, 24.536805401, 24.574944709, 24.624740812,
24.67768102, 24.722688075, 24.749055641, 24.802397755, 24.825749614,
24.856984186, 24.918764566, 24.980799726, 25.032143823, 25.054768976,
25.121250138, 25.166932852, 25.20244711, 25.279938861, 25.340021577,
25.420263553, 25.455630754, 25.492221479, 25.556389497, 25.608558624,
25.710206645, 25.751262298, 25.779238373, 25.828401798, 25.884944162,
25.924760029, 25.975359814, 26.009812559, 26.057963176, 26.120674194,
26.166581051, 26.206979856, 26.237896427, 26.270873956, 26.317384439,
26.366821362, 26.401364571, 26.446531678, 26.472995256, 26.498724433,
26.538291933, 26.581249779, 26.670932006, 26.700605711, 26.740688141,
26.793842907, 26.847833186, 26.905758644, 26.976578855, 27.035272635,
27.058859035, 27.111184703, 27.175663376, 27.232340591, 27.252715225,
27.314961788, 27.377518573, 27.454153218, 27.500959041, 27.519081871,
27.541555213, 27.567545512, 27.603237371, 27.658583199, 27.739732079,
27.811795326, 27.874854756, 27.930350646, 28.006557183, 28.105173525,
28.15034836, 28.193946595, 28.245966119, 28.292641094, 28.391663714,
28.440112376, 28.470293885, 28.499448199, 28.537668045, 28.595007472,
28.647764805, 28.6804083, 28.760091685, 28.80113382, 28.848771992,
28.886966288, 28.921503084, 28.968439079, 29.037542019, 29.059104622,
29.105256569, 29.170655724, 29.202450873, 29.249442786, 29.307647275,
29.408890453, 29.450444775, 29.496263345, 29.560889446, 29.564188309,
29.597443986, 29.60911376, 29.65686478, 29.705978868, 29.753793147,
29.788358272, 29.815073512, 29.836571601, 29.96737139, 30.0371142,
30.098446917, 30.117046309, 30.14094959, 30.174653995, 30.210417321,
30.241236502, 30.268993011, 30.306137509, 30.347789147, 30.368419126,
30.423165987, 30.534418476, 30.661301471, 30.716163701, 30.757285663,
30.763670566, 30.807206226, 30.850853407, 30.913580523, 30.980710517,
30.986785963, 31.043018816, 31.073526037, 31.259199138, 31.271292789,
31.330225052, 31.369661742, 31.42770653, 31.439397224, 31.541205829,
31.605715125, 31.682726075, 31.744113338, 31.839163575, 31.851321687,
31.907977003, 31.926927295, 31.977022226, 32.017193337, 32.099266726,
32.19644351, 32.253781196, 32.334006157, 32.384486348, 32.425345589,
32.505221661, 32.558528458, 32.603293048, 32.717461009, 32.71986227,
32.773478712, 32.801767866, 32.840196432, 32.842298686, 32.882915886,
32.909588819, 32.990004538, 33.102076421, 33.123914907, 33.134235011,
33.241262606, 33.351906295, 33.354758596, 33.413087189, 33.508692188,
33.539405125, 33.622069075, 33.687238449, 33.705881021, 33.814972433,
33.827191502, 33.910165747, 33.969176229, 33.971348443, 33.980933286,
34.297435343, 34.304800913, 34.312806565, 34.40857215, 34.451230164,
34.469225993, 34.628984252, 34.67606235, 34.729629297, 34.807616684,
34.822525063, 34.943056889, 34.950988975, 34.977555473), "10000" = c(1.8303e-05,
1.8337e-05, 1.894e-05, 2.006e-05, 2.0616e-05, 2.0635e-05, 2.0852e-05,
2.1219e-05, 2.2046e-05, 2.2137e-05, 2.2459e-05, 2.3079e-05, 2.3206e-05,
2.3207e-05, 2.3458e-05, 2.4331e-05, 2.4671e-05, 2.4737e-05, 2.4841e-05,
2.4851e-05, 2.5248e-05, 2.559e-05, 2.6783e-05, 2.6981e-05, 2.7196e-05,
2.7217e-05, 2.7853e-05, 2.7939e-05, 2.8256e-05, 2.8594e-05, 2.8677e-05,
2.8921e-05, 2.9374e-05, 2.9864e-05, 3.0041e-05, 3.0427e-05, 3.0438e-05,
3.069e-05, 3.1086e-05, 3.1241e-05, 3.1362e-05, 3.2071e-05, 3.2199e-05,
3.2411e-05, 3.2713e-05, 3.3599e-05, 3.4563e-05, 3.4651e-05, 3.507e-05,
3.5413e-05, 3.5815e-05, 3.6255e-05, 3.6643e-05, 3.6914e-05, 3.7204e-05,
3.7626e-05, 3.8373e-05, 3.8895e-05, 3.9556e-05, 3.9806e-05, 4.0083e-05,
4.022e-05, 4.1022e-05, 4.1427e-05, 4.1532e-05, 4.1639e-05, 4.1881e-05,
4.2106e-05, 4.2798e-05, 4.3362e-05, 4.4096e-05, 4.4397e-05, 4.5423e-05,
4.6303e-05, 4.6457e-05, 4.7375e-05, 4.7792e-05, 4.8351e-05, 4.8484e-05,
4.9128e-05, 4.9464e-05, 4.9885e-05, 5.0444e-05, 5.0921e-05, 5.1537e-05,
5.2147e-05, 5.2342e-05, 5.268e-05, 5.435e-05, 5.5125e-05, 5.5664e-05,
5.619e-05, 5.6855e-05, 5.7606e-05, 5.7824e-05, 5.9023e-05, 5.9372e-05,
5.9823e-05, 6.1617e-05, 6.2088e-05, 6.2324e-05, 6.2992e-05, 6.4238e-05,
6.5364e-05, 6.595e-05, 6.7344e-05, 6.871e-05, 6.9962e-05, 7.1203e-05,
7.1568e-05, 7.2722e-05, 7.3334e-05, 7.4072e-05, 7.4274e-05, 7.5619e-05,
7.73e-05, 7.8073e-05, 7.9197e-05, 8.0321e-05, 8.1256e-05, 8.2273e-05,
8.3252e-05, 8.4102e-05, 8.4628e-05, 8.5627e-05, 8.6755e-05, 8.7922e-05,
8.8881e-05, 8.9398e-05, 9.0467e-05, 9.1582e-05, 9.2734e-05, 9.4129e-05,
9.4725e-05, 9.5919e-05, 9.6808e-05, 9.7935e-05, 9.8878e-05, 0.000100629,
0.000101553, 0.0001027, 0.000103647, 0.000104208, 0.000105681,
0.000106755, 0.000107978, 0.000109177, 0.000109529, 0.000110914,
0.000111654, 0.000112635, 0.000114614, 0.000115485, 0.000117063,
0.000117673, 0.000119663, 0.000120846, 0.000122086, 0.000123385,
0.000125197, 0.000127083, 0.000128892, 0.000129552, 0.000131101,
0.000132734, 0.000133679, 0.000134561, 0.000135668, 0.000136856,
0.000137904, 0.000139449, 0.000140808, 0.000142062, 0.000143744,
0.000146475, 0.000149122, 0.000150445, 0.000151626, 0.00015258,
0.0001543, 0.000156797, 0.00015816, 0.000159377, 0.000161352,
0.000163214, 0.000165382, 0.000167032, 0.000168444, 0.000170228,
0.000170885, 0.00017249, 0.000175019, 0.0001768, 0.000179204,
0.000181401, 0.000184447, 0.000186961, 0.000188475, 0.000192126,
0.0001947, 0.000196625, 0.000197349, 0.000199277, 0.000200644,
0.00020297, 0.000207029, 0.000209945, 0.000212778, 0.000216641,
0.000218665, 0.000221947, 0.000224406, 0.000227463, 0.000230166,
0.000233049, 0.00023543, 0.000237867, 0.000240343, 0.000243235,
0.000247099, 0.000249734, 0.000251748, 0.000254475, 0.000257815,
0.000260877, 0.000262627, 0.000265926, 0.000268325, 0.000271849,
0.000275015, 0.00027803, 0.000281784, 0.000284617, 0.000288816,
0.000292692, 0.000295697, 0.00030057, 0.000302901, 0.000305838,
0.000311928, 0.000314299, 0.000317995, 0.000321843, 0.000324221,
0.000328063, 0.000332214, 0.000336191, 0.000340004, 0.00034376,
0.000348093, 0.000351061, 0.00035552, 0.000358922, 0.000362787,
0.000365847, 0.000369318, 0.000373959, 0.000378577, 0.00038177,
0.000386923, 0.000390738, 0.000395972, 0.000400395, 0.000406534,
0.000410696, 0.000418226, 0.000421179, 0.000424931, 0.000429044,
0.000434506, 0.000439545, 0.000444752, 0.000450245, 0.000454932,
0.000461596, 0.000467865, 0.000473512, 0.00047838, 0.00048301,
0.000488663, 0.000495128, 0.00049813, 0.000504403, 0.000508209,
0.00051488, 0.000522719, 0.000527574, 0.000533717, 0.000539028,
0.000544701, 0.00055261, 0.000558443, 0.000565986, 0.000571836,
0.000578784, 0.000586281, 0.000592469, 0.000599687, 0.000607181,
0.000615439, 0.000622763, 0.000630093, 0.000637304, 0.00064283,
0.000650441, 0.000660192, 0.000667997, 0.00067643, 0.000683077,
0.000693374, 0.000700816, 0.000709372, 0.000718097, 0.000724689,
0.000733432, 0.000742239, 0.00075197, 0.000759468, 0.000765666,
0.000774407, 0.000781968, 0.000791578, 0.00080313, 0.000811609,
0.000819368, 0.000829237, 0.000838324, 0.000849078, 0.000857443,
0.000865899, 0.000878023, 0.000886335, 0.000895565, 0.000906751,
0.000914945, 0.000928686, 0.00094037, 0.000951412, 0.000964488,
0.000975594, 0.00098316, 0.000985722, 0.000995229, 0.00100546,
0.001019708, 0.001035424, 0.001047896, 0.001059609, 0.00107532,
0.001087238, 0.001100763, 0.001111466, 0.001123084, 0.001133234,
0.001146442, 0.001159714, 0.0011733, 0.001184856, 0.001198339,
0.001211505, 0.001223125, 0.001236809, 0.001250145, 0.001265034,
0.001279095, 0.001294276, 0.001310794, 0.001324679, 0.001341616,
0.001357083, 0.001371886, 0.001385935, 0.001405009, 0.001419618,
0.001436754, 0.001455604, 0.001469692, 0.001489019, 0.001505829,
0.001523371, 0.001538548, 0.001555506, 0.00157146, 0.001592123,
0.001613119, 0.00163493, 0.001653, 0.001670373, 0.001689551,
0.001709102, 0.001732361, 0.001749755, 0.001766473, 0.001788218,
0.00180887, 0.001827117, 0.001853045, 0.001878118, 0.001900972,
0.001923752, 0.001947088, 0.001967865, 0.00199222, 0.00202023,
0.002045108, 0.002069757, 0.002093948, 0.002123402, 0.002146392,
0.002171351, 0.002197049, 0.002223444, 0.00225232, 0.002278578,
0.002302504, 0.002332455, 0.002361303, 0.002387365, 0.002414722,
0.002442247, 0.002467148, 0.002498257, 0.002526241, 0.002555715,
0.002585744, 0.002616531, 0.002643926, 0.002676576, 0.002708119,
0.00274177, 0.002775112, 0.002807657, 0.00284083, 0.002875997,
0.00290971, 0.002948077, 0.00297711, 0.003013059, 0.003050783,
0.003079547, 0.003115012, 0.003155865, 0.00319473, 0.003235155,
0.003272745, 0.00330964, 0.003345972, 0.003383841, 0.003423997,
0.003466717, 0.003509565, 0.003549721, 0.00358986, 0.003629772,
0.003674482, 0.003715063, 0.00375912, 0.003801217, 0.003845623,
0.003889473, 0.003931397, 0.003975703, 0.004014808, 0.004061808,
0.004106342, 0.004155496, 0.004202565, 0.004249485, 0.00430316,
0.00435166, 0.00440204, 0.004452725, 0.004504291, 0.004557279,
0.004608226, 0.004663661, 0.004724908, 0.004779036, 0.00483951,
0.004897672, 0.004958088, 0.005017107, 0.005072923, 0.005140537,
0.005195673, 0.005254143, 0.005317468, 0.005380794, 0.00544004,
0.005500631, 0.005560039, 0.005622974, 0.005686338, 0.00575437,
0.005825528, 0.005892081, 0.005958765, 0.00602396, 0.006089164,
0.006153105, 0.00622362, 0.006296984, 0.006370089, 0.006447801,
0.006529518, 0.006601043, 0.006684437, 0.00675836, 0.006833763,
0.006906313, 0.00698174, 0.007064847, 0.007144694, 0.00722458,
0.007305581, 0.00739159, 0.007479288, 0.007566585, 0.007649844,
0.007743732, 0.00783446, 0.007924979, 0.008019715, 0.008116429,
0.008206473, 0.008299385, 0.008400517, 0.00849683, 0.008591642,
0.008691608, 0.008790099, 0.008884169, 0.008992629, 0.009099502,
0.009212583, 0.009323444, 0.009432267, 0.00954452, 0.009647141,
0.009759267, 0.009872874, 0.009962008, 0.009988935, 0.010107149,
0.010227451, 0.010346999, 0.010460743, 0.010589978, 0.010721123,
0.010842559, 0.010973587, 0.011093196, 0.011220607, 0.011349833,
0.011482198, 0.011618728, 0.011760268, 0.011894125, 0.012035434,
0.01216777, 0.012311373, 0.01244833, 0.012596481, 0.012743697,
0.012893134, 0.013042469, 0.013200693, 0.013350211, 0.013513501,
0.013671186, 0.013823053, 0.013981148, 0.014135799, 0.014295219,
0.014470707, 0.014655186, 0.014826807, 0.015000175, 0.015169337,
0.015358402, 0.015542025, 0.015725115, 0.015903633, 0.016093072,
0.016281343, 0.016471517, 0.016654948, 0.016853411, 0.017045255,
0.017240435, 0.017449389, 0.017651692, 0.017853311, 0.018061509,
0.018276135, 0.018486391, 0.01869823, 0.018917399, 0.019145744,
0.019361558, 0.019588588, 0.019816836, 0.020041585, 0.020276514,
0.020505947, 0.020740168, 0.02098758, 0.021234446, 0.02147612,
0.021720615, 0.021977276, 0.022238081, 0.022476181, 0.022736306,
0.022994964, 0.023266106, 0.023533131, 0.023809474, 0.024081361,
0.024353867, 0.024637699, 0.024934712, 0.0252245, 0.025524914,
0.025836714, 0.026142091, 0.026447123, 0.026750176, 0.02706789,
0.027387659, 0.027718696, 0.028056738, 0.028403328, 0.028721211,
0.029064416, 0.02941367, 0.029757436, 0.030100206, 0.030438572,
0.030805344, 0.031153244, 0.031525959, 0.03189161, 0.032274692,
0.032656355, 0.033046355, 0.033440945, 0.033839004, 0.034222682,
0.034624061, 0.0350272, 0.035411024, 0.035810638, 0.036241157,
0.036660359, 0.037098227, 0.037553035, 0.037994774, 0.038428172,
0.038879033, 0.039345252, 0.039821756, 0.040291915, 0.040776216,
0.04125389, 0.04172787, 0.042217641, 0.042710923, 0.043208739,
0.043711809, 0.044205977, 0.04472478, 0.045245554, 0.045789234,
0.046321444, 0.046868041, 0.04741429, 0.04796682, 0.048536391,
0.049100994, 0.049679611, 0.050259973, 0.050830882, 0.051420732,
0.052041096, 0.052646271, 0.053251713, 0.053879845, 0.054497601,
0.055116373, 0.055765229, 0.056396684, 0.057036011, 0.057725294,
0.058388749, 0.059055309, 0.059748304, 0.060449164, 0.061167789,
0.06187649, 0.062598397, 0.063315042, 0.064061793, 0.064822259,
0.065594175, 0.066379303, 0.067169361, 0.067953824, 0.068754357,
0.069578965, 0.070388333, 0.071220865, 0.072055246, 0.072904181,
0.073742715, 0.074611715, 0.075500504, 0.076383249, 0.077293905,
0.078233887, 0.079154619, 0.080093874, 0.081043131, 0.081979083,
0.082939424, 0.083923849, 0.084939913, 0.085943175, 0.086962595,
0.088036229, 0.089067081, 0.090133954, 0.091208835, 0.092305686,
0.093390285, 0.094494277, 0.095630236, 0.096728254, 0.09785513,
0.099029353, 0.100206907, 0.101367521, 0.102328764, 0.102567519,
0.10378848, 0.105027238, 0.106290471, 0.107560951, 0.108831766,
0.110140564, 0.1114678, 0.112831253, 0.114167799, 0.115537274,
0.116894236, 0.118318666, 0.119690395, 0.121103962, 0.12255848,
0.124025249, 0.125508813, 0.127035163, 0.128572902, 0.130116712,
0.13172241, 0.133328622, 0.134923124, 0.13656252, 0.138198147,
0.139861722, 0.141554265, 0.143234433, 0.144922427, 0.146660419,
0.148401788, 0.150210941, 0.152021289, 0.153877672, 0.155745557,
0.157618774, 0.159507837, 0.161430965, 0.163402505, 0.165373663,
0.167399546, 0.169423541, 0.171465637, 0.173549199, 0.175632734,
0.177763532, 0.179926806, 0.182103437, 0.184302627, 0.186514311,
0.188784394, 0.191048309, 0.193357328, 0.195750567, 0.198161758,
0.200552749, 0.203012698, 0.205521324, 0.20800816, 0.210541592,
0.213100523, 0.215664225, 0.218275339, 0.220988424, 0.223669276,
0.226381463, 0.229172556, 0.231970987, 0.234828187, 0.237706274,
0.240582231, 0.243515722, 0.246514407, 0.249568276, 0.252651362,
0.255730806, 0.258877221, 0.262065591, 0.265269007, 0.268590454,
0.271908245, 0.275254382, 0.278687504, 0.282178812, 0.285747475,
0.28933672, 0.292921995, 0.29653879, 0.300286449, 0.30402311,
0.307802849, 0.311615832, 0.31552446, 0.319460304, 0.323456232,
0.324340318, 0.327507947, 0.331639584, 0.335798271, 0.340020591,
0.344326378, 0.348644686, 0.353042791, 0.357459906, 0.361988291,
0.366567543, 0.371294854, 0.376039538, 0.380822903, 0.385691754,
0.39065588, 0.395667367, 0.400760174, 0.405875199, 0.411046145,
0.416336947, 0.421710479, 0.427098513, 0.432595518, 0.43818315,
0.443880243, 0.445053144, 0.449642204, 0.45547946, 0.461390333,
0.467402279, 0.473490051, 0.479681682, 0.485898947, 0.492341087,
0.498806176, 0.505358117, 0.511993411, 0.518719565, 0.525601427,
0.532493648, 0.539574925, 0.546778092, 0.554011859, 0.561491672,
0.569037613, 0.576623889, 0.584343536, 0.592248384, 0.60024186,
0.608317201, 0.616556355, 0.624868626, 0.633422524, 0.64206286,
0.650936878, 0.659767589, 0.668872925, 0.678048343, 0.687433898,
0.696973015, 0.706628097, 0.710805793, 0.716455386, 0.726444395,
0.736603759, 0.746998242, 0.75758901, 0.768228239, 0.778995815,
0.790035958, 0.801332371, 0.812833166, 0.824610306, 0.836555835,
0.848582617, 0.860984076, 0.873476091, 0.886192609, 0.899281882,
0.912461412, 0.925910859, 0.939570655, 0.953460509, 0.967627299,
0.98208528, 0.996735354, 1.011766493, 1.017933861, 1.027064942,
1.042560556, 1.058533469, 1.074679299, 1.091284957, 1.108362585,
1.125521343, 1.143075597, 1.160996397, 1.179426602, 1.19806272,
1.217117641, 1.236643246, 1.256368155, 1.276915347, 1.29774021,
1.318950573, 1.340601299, 1.3628091, 1.380842179, 1.421959289,
1.444921251, 1.467922219, 1.490830777, 1.513717444, 1.536694072,
1.559677451, 1.582669639, 1.605599495, 1.62856648, 1.651452862,
1.67430227, 1.697029733, 1.720021935, 1.74281902, 1.76571911,
1.788652261, 1.811488163, 1.825065429, 1.834648835, 1.857659492,
1.880653428, 1.903585198, 1.926414674, 1.949449993, 1.972348345,
1.995288193, 2.01816068, 2.041002155, 2.063937811, 2.086752238,
2.10973726, 2.132746733, 2.155581877, 2.17864277, 2.201265812,
2.224105532, 2.247062412, 2.269866085, 2.292746725, 2.315711798,
2.33866809, 2.361534671, 2.384598276, 2.397952089, 2.407505662,
2.430397015, 2.453605935, 2.476394416, 2.499258656, 2.522112321,
2.544984341, 2.56766608, 2.590662875, 2.613502659, 2.636576062,
2.659441456, 2.68255794, 2.705550391, 2.728467233, 2.751402673,
2.77412918, 2.796991218, 2.819875409, 2.842947492, 2.865785223,
2.888847064, 2.911959149, 2.934914832, 2.95795179, 2.980728257,
3.003569196, 3.026581279, 3.049784386, 3.072714036, 3.095743092,
3.118644536, 3.141642229, 3.164680354, 3.187536185, 3.205801315,
3.21052212, 3.233370252, 3.256406645, 3.279566575, 3.302589653,
3.325605528, 3.348327778, 3.371440233, 3.39430153, 3.417148259,
3.439696631, 3.462784795, 3.485819841, 3.508696513, 3.531949894,
3.555086339, 3.578278224, 3.601345458, 3.624214828, 3.64728222,
3.670555218, 3.69370456, 3.716475712, 3.739562651, 3.762956623,
3.781172063, 3.786199254, 3.809223507, 3.832193338, 3.855468259,
3.878186266, 3.901264614, 3.924168067, 3.947411177, 3.97054078,
3.993403176, 4.016707928, 4.039632766, 4.062473727, 4.085463862,
4.108348511, 4.131166124, 4.154182229, 4.177136745, 4.200030547,
4.223271446, 4.246508018, 4.269508375, 4.292937589, 4.316259363,
4.339533648, 4.362860318, 4.386309115, 4.409186358, 4.432196883,
4.455435004, 4.478459514, 4.501474235, 4.524387819, 4.547719291,
4.570825365, 4.594071192, 4.617640061, 4.640690843, 4.663785893,
4.686778743, 4.709880046, 4.733074446, 4.756090145, 4.779269231,
4.802411961, 4.825567536, 4.848717571, 4.872021008, 4.89522175,
4.918537517, 4.941785641, 4.964887795, 4.987409063, 5.010650754,
5.034081209, 5.057140266, 5.080282881, 5.103494346, 5.127090049,
5.150197765, 5.173666255, 5.196313769, 5.219343697, 5.242706787,
5.266104606, 5.28843007, 5.31179762, 5.335063002, 5.358095657,
5.381773921, 5.404855256, 5.427903974, 5.451374372, 5.474543557,
5.498048198, 5.521556787, 5.544823874, 5.568715312, 5.592387767,
5.615401958, 5.638906152, 5.662272051, 5.685796065, 5.70985008,
5.733102006, 5.75655406, 5.780000138, 5.803556768, 5.826985829,
5.850296653, 5.873637922, 5.897440622, 5.920702517, 5.944161332,
5.967388922, 5.991193445, 5.996092176, 6.014539008, 6.037875026,
6.061418966, 6.084561038, 6.10861812, 6.131764887, 6.154988266,
6.178153078, 6.202126187, 6.225238346, 6.248542193, 6.272250154,
6.29618716, 6.319454628, 6.342947947, 6.366636523, 6.390371249,
6.414507299, 6.437881502, 6.461193967, 6.484746262, 6.508395649,
6.531686096, 6.555372715, 6.578693242, 6.60181271, 6.625367585,
6.64955699, 6.673381932, 6.696242505, 6.71937168, 6.743525818,
6.766840086, 6.790980752, 6.8145853, 6.838841491, 6.86264795,
6.886788576, 6.910197074, 6.933856574, 6.957815194, 6.980963675,
7.003870313, 7.027440211, 7.051120292, 7.075238583, 7.099032378,
7.122734951, 7.146020754, 7.170083005, 7.194248009, 7.217897071,
7.241302967, 7.265338728, 7.289632045, 7.313404019, 7.337609925,
7.361112204, 7.384561351, 7.40834751, 7.432623959, 7.457273304,
7.481536298, 7.504920422, 7.528416837, 7.552042179, 7.576039315,
7.599695076, 7.623850533, 7.647328672, 7.670432358, 7.694852912,
7.71869535, 7.742622393, 7.766676167, 7.790950779, 7.814596267,
7.837793647, 7.861046812, 7.88472287, 7.908964707, 7.933380992,
7.956880107, 7.980807791, 8.004623639, 8.029450176, 8.05219723,
8.07638889, 8.100281694, 8.123746689, 8.148681603, 8.172432663,
8.195774959, 8.218864135, 8.242572095, 8.266365529, 8.290318439,
8.31528443, 8.339013716, 8.362841603, 8.386989795, 8.410852119,
8.435657946, 8.460643822, 8.484101339, 8.507546763, 8.533000596,
8.557571051, 8.58140926, 8.605228778, 8.630156683, 8.654655691,
8.679603073, 8.703299017, 8.728051444, 8.75117255, 8.776254059,
8.800346029, 8.82487927, 8.84867469, 8.872517632, 8.8971912,
8.922075215, 8.94671743, 8.970307297, 8.995020376, 9.019104858,
9.044253649, 9.068231455, 9.093616892, 9.117245733, 9.141675234,
9.165152016, 9.189456509, 9.215074053, 9.239567511, 9.264067793,
9.289455618, 9.314248578, 9.338637213, 9.36211885, 9.38577755,
9.410195684, 9.434765484, 9.459400005, 9.485255611, 9.509378342,
9.53443768, 9.559501613, 9.58426713, 9.609966772, 9.635764354,
9.660132447, 9.685805777, 9.7116124, 9.736959861, 9.760736083,
9.78499257, 9.809710057, 9.834980423, 9.859177244, 9.884444286,
9.909244405, 9.9356693, 9.961102117, 9.986628475, 10.011220213,
10.03699538, 10.061479622, 10.085738824, 10.109073835, 10.133251649,
10.156565449, 10.18083581, 10.206387847, 10.23108992, 10.255367571,
10.279368187, 10.303645538, 10.328099319, 10.352190684, 10.375890258,
10.398739878, 10.423332855, 10.447428325, 10.471975475, 10.495961805,
10.521511716, 10.547146275, 10.570852906, 10.597959989, 10.623047625,
10.648012837, 10.671090486, 10.694900145, 10.718657347, 10.74482832,
10.771172507, 10.797016738, 10.822223059, 10.826834919, 10.84657588,
10.87443535, 10.901988974, 10.928255255, 10.951335758, 10.976401661,
11.001596466, 11.025809118, 11.050946321, 11.076334535, 11.100002781,
11.124407992, 11.147587527, 11.17324941, 11.196396373, 11.221365618,
11.247949298, 11.271157476, 11.296881227, 11.320730756, 11.34614757,
11.369895156, 11.394888601, 11.420855879, 11.446497465, 11.471077944,
11.498401379, 11.523141767, 11.549923732, 11.575094781, 11.600806584,
11.627496901, 11.652500516, 11.678562551, 11.703570271, 11.731505494,
11.756602444, 11.783951333, 11.809024653, 11.834166158, 11.858164782,
11.884455554, 11.911600392, 11.936472703, 11.963424567, 11.989160829,
12.015606151, 12.043371337, 12.06898879, 12.095706148, 12.119715339,
12.145102166, 12.170470282, 12.194635735, 12.22196279, 12.251404354,
12.2785704, 12.306052016, 12.331530247, 12.356977062, 12.379312854,
12.404761908, 12.42790716, 12.454078676, 12.482536139, 12.508558716,
12.534682775, 12.561452011, 12.586192688, 12.610647834, 12.635521885,
12.662931584, 12.688148057, 12.715401467, 12.744141513, 12.767711729,
12.790245941, 12.820444832, 12.845567498, 12.870919963, 12.895386872,
12.921744024, 12.946455973, 12.970770043, 12.996898456, 13.024094095,
13.047982984, 13.07458088, 13.104337458, 13.131820198, 13.157659638,
13.18318997, 13.210896882, 13.236695629, 13.265219358, 13.2920514,
13.31745077, 13.344154186, 13.368000987, 13.395629359, 13.421412393,
13.448931735, 13.475565069, 13.504810197, 13.528771447, 13.556771951,
13.582827368, 13.610612985, 13.633700736, 13.659210955, 13.682906844,
13.707047461, 13.73431614, 13.75495387, 13.784861981, 13.810314221,
13.837386963, 13.860216992, 13.887480257, 13.915356859, 13.940750233,
13.969161315, 13.995434436, 14.020971473, 14.047305079, 14.072122792,
14.099132024, 14.130305695, 14.157486477, 14.18876272, 14.216533472,
14.238134048, 14.266540252, 14.29023919, 14.316870963, 14.348337356,
14.37515995, 14.400726578, 14.427522299, 14.457021716, 14.480547374,
14.509327192, 14.538923442, 14.5602823, 14.587881298, 14.613752658,
14.640284915, 14.667473265, 14.693660697, 14.718976139, 14.745716052,
14.771410912, 14.798343641, 14.825483639, 14.853247248, 14.876544836,
14.903320871, 14.928510695, 14.957673493, 14.986341902, 15.01001126,
15.045309506, 15.079789256, 15.107206231, 15.130267367, 15.155529736,
15.187598213, 15.214912891, 15.241533538, 15.270850693, 15.295699013,
15.322364302, 15.349690192, 15.369327395, 15.396023269, 15.423292145,
15.448283931, 15.476333995, 15.507032093, 15.530132221, 15.559638982,
15.589004108, 15.62117424, 15.651743528, 15.681245663, 15.703992328,
15.737062612, 15.76230938, 15.790225484, 15.821170126, 15.855458619,
15.881562758, 15.920086362, 15.95399668, 15.975035196, 16.003524973,
16.027292877, 16.052799081, 16.082261235, 16.10837852, 16.114037506,
16.135273034, 16.16432965, 16.194825808, 16.222054388, 16.243600514,
16.276997663, 16.307089087, 16.342577692, 16.369077999, 16.395968827,
16.418086859, 16.441736973, 16.464305545, 16.491299199, 16.516317391,
16.550268283, 16.576956967, 16.59953087, 16.620198594, 16.65049636,
16.679202799, 16.704108673, 16.726856294, 16.755377445, 16.793410253,
16.821001859, 16.851811727, 16.876623427, 16.908413019, 16.928999988,
16.958861072, 16.980858536, 17.008992325, 17.043104091, 17.0813759,
17.115549316, 17.147611806, 17.173892456, 17.200365145, 17.23709191,
17.261980315, 17.29134072, 17.311983566, 17.342498968, 17.370250774,
17.395045231, 17.416688044, 17.446077078, 17.475751379, 17.495457326,
17.512998378, 17.537381739, 17.566458052, 17.595955772, 17.621865105,
17.643277383, 17.674329021, 17.702299171, 17.731950353, 17.763515276,
17.789640104, 17.82028669, 17.849478548, 17.871804513, 17.907584413,
17.931115083, 17.961941923, 17.990006818, 18.0219267, 18.062383203,
18.08215647, 18.11611255, 18.145112307, 18.18462439, 18.214480762,
18.234744338, 18.265269291, 18.292651253, 18.322646133, 18.351590819,
18.373921589, 18.401054655, 18.444926101, 18.46725276, 18.496203668,
18.522298594, 18.550829794, 18.584910721, 18.622713149, 18.65495772,
18.683201078, 18.706043366, 18.73615547, 18.763243157, 18.788988157,
18.812880816, 18.838356741, 18.866667035, 18.899868955, 18.935664815,
18.955230014, 18.977007112, 19.010133525, 19.030283534, 19.066891178,
19.098460612, 19.129509366, 19.159968189, 19.180497578, 19.212977012,
19.241216406, 19.264735574, 19.283699486, 19.310962859, 19.33714602,
19.362885891, 19.389140393, 19.409100122, 19.437236792, 19.464886296,
19.490010927, 19.514776335, 19.540721518, 19.563941389, 19.590612614,
19.621691378, 19.632779214, 19.668762877, 19.697025405, 19.721577666,
19.761964684, 19.787841933, 19.814217875, 19.840066339, 19.882463973,
19.916181931, 19.957381718, 19.980361627, 20.011576179, 20.03974939,
20.067670571, 20.103891372, 20.12565062, 20.176478426, 20.198941265,
20.247834401, 20.275746639, 20.30255529, 20.345959719, 20.374952586,
20.402047861, 20.433219707, 20.458077902, 20.488371106, 20.506971622,
20.525507433, 20.553707177, 20.587685409, 20.61549948, 20.668032671,
20.699486062, 20.750388516, 20.779656444, 20.806401857, 20.847203434,
20.864992258, 20.89066329, 20.920738217, 20.944893756, 20.969811768,
20.990771293, 21.038181814, 21.056173527, 21.087766104, 21.120846193,
21.140385183, 21.159979637, 21.176259146, 21.221702604, 21.293492973,
21.323149193, 21.353822596, 21.401432517, 21.45088487, 21.470719953,
21.490832577, 21.511275887, 21.535518395, 21.552485353, 21.571557117,
21.637564587, 21.679169207, 21.693083873, 21.736949056, 21.775312305,
21.796988207, 21.837014748, 21.865554775, 21.905760701, 21.921620073,
21.957927876, 21.976421716, 22.009296762, 22.03205728, 22.049296395,
22.0801241, 22.131541019, 22.181668184, 22.22018223, 22.23150351,
22.268193674, 22.296136396, 22.316232514, 22.340068785, 22.357068691,
22.395532214, 22.41738902, 22.464978369, 22.500287166, 22.553289024,
22.58083013, 22.608511748, 22.677472004, 22.718297735, 22.73043355,
22.765933757, 22.828699593, 22.832638374, 22.862640518, 22.913989586,
22.924485694, 22.950137874, 22.988286072, 23.016893213, 23.076528808,
23.102270273, 23.122198193, 23.138813756, 23.210604987, 23.234908705,
23.251078599, 23.282798201, 23.303527158, 23.359687384, 23.413695005,
23.463331432, 23.516923721, 23.547884034, 23.5578188, 23.578035392,
23.618222368, 23.659512966, 23.677507102, 23.716774125, 23.72526394,
23.765030403, 23.784719965, 23.801211711, 23.836378809, 23.865676896,
23.886617774, 23.917792245, 23.923424407, 23.946767631, 23.955371799,
23.992375172, 24.012117802, 24.081674162, 24.107289486, 24.121272775,
24.202178525, 24.278370988, 24.340387907, 24.401215567, 24.482842513,
24.521305793, 24.57541267, 24.650009567, 24.663654046, 24.685134372,
24.759934308, 24.809863597, 24.830675374, 24.875414591, 24.886750573,
24.931260653, 24.944690093, 24.961896968, 24.993601239, 25.010040085,
25.036293158, 25.051567619, 25.091424557, 25.128185758, 25.157452893,
25.170288074, 25.237982974, 25.275149098, 25.294860629, 25.3392105,
25.34391782, 25.36710952, 25.449619654, 25.482033357, 25.484894938,
25.578654784, 25.614380421, 25.638685836, 25.675688167, 25.701109126,
25.724240509, 25.76347697, 25.779297009, 25.819074003, 25.879944762,
25.890775034, 25.914374612, 25.957512147, 25.984316107, 25.992554517,
26.118772531, 26.137686038, 26.150101136, 26.198901379, 26.201253666,
26.213216461, 26.237276789, 26.240484045, 26.349777843, 26.36861032,
26.375267328, 26.401794736, 26.421851993, 26.487345848, 26.494794583,
26.50239174, 26.523854545, 26.589160054, 26.636574955, 26.67705635
)), .Names = c("5", "6", "8", "10", "20", "35", "50", "75", "100",
"150", "200", "300", "500", "800", "1000", "1400", "1600", "2400",
"10000"), row.names = c("1e-05", "1.01e-05", "1.02e-05", "1.04e-05",
"1.05e-05", "1.06e-05", "1.07e-05", "1.08e-05", "1.1e-05", "1.11e-05",
"1.12e-05", "1.14e-05", "1.15e-05", "1.16e-05", "1.17e-05", "1.19e-05",
"1.2e-05", "1.22e-05", "1.23e-05", "1.24e-05", "1.26e-05", "1.27e-05",
"1.29e-05", "1.3e-05", "1.32e-05", "1.33e-05", "1.35e-05", "1.36e-05",
"1.38e-05", "1.4e-05", "1.41e-05", "1.43e-05", "1.45e-05", "1.46e-05",
"1.48e-05", "1.5e-05", "1.51e-05", "1.53e-05", "1.55e-05", "1.57e-05",
"1.58e-05", "1.6e-05", "1.62e-05", "1.64e-05", "1.66e-05", "1.68e-05",
"1.7e-05", "1.72e-05", "1.74e-05", "1.76e-05", "1.78e-05", "1.8e-05",
"1.82e-05", "1.84e-05", "1.86e-05", "1.88e-05", "1.91e-05", "1.93e-05",
"1.95e-05", "1.97e-05", "2e-05", "2.02e-05", "2.04e-05", "2.07e-05",
"2.09e-05", "2.11e-05", "2.14e-05", "2.16e-05", "2.19e-05", "2.21e-05",
"2.24e-05", "2.26e-05", "2.29e-05", "2.32e-05", "2.34e-05", "2.37e-05",
"2.4e-05", "2.43e-05", "2.45e-05", "2.48e-05", "2.51e-05", "2.54e-05",
"2.57e-05", "2.6e-05", "2.63e-05", "2.66e-05", "2.69e-05", "2.72e-05",
"2.75e-05", "2.79e-05", "2.82e-05", "2.85e-05", "2.88e-05", "2.92e-05",
"2.95e-05", "2.99e-05", "3.02e-05", "3.05e-05", "3.09e-05", "3.13e-05",
"3.16e-05", "3.2e-05", "3.24e-05", "3.27e-05", "3.31e-05", "3.35e-05",
"3.39e-05", "3.43e-05", "3.47e-05", "3.51e-05", "3.55e-05", "3.59e-05",
"3.63e-05", "3.67e-05", "3.72e-05", "3.76e-05", "3.8e-05", "3.85e-05",
"3.89e-05", "3.94e-05", "3.98e-05", "4.03e-05", "4.07e-05", "4.12e-05",
"4.17e-05", "4.22e-05", "4.27e-05", "4.32e-05", "4.37e-05", "4.42e-05",
"4.47e-05", "4.52e-05", "4.57e-05", "4.62e-05", "4.68e-05", "4.73e-05",
"4.79e-05", "4.84e-05", "4.9e-05", "4.95e-05", "5.01e-05", "5.07e-05",
"5.13e-05", "5.19e-05", "5.25e-05", "5.31e-05", "5.37e-05", "5.43e-05",
"5.5e-05", "5.56e-05", "5.62e-05", "5.69e-05", "5.75e-05", "5.82e-05",
"5.89e-05", "5.96e-05", "6.03e-05", "6.1e-05", "6.17e-05", "6.24e-05",
"6.31e-05", "6.38e-05", "6.46e-05", "6.53e-05", "6.61e-05", "6.68e-05",
"6.76e-05", "6.84e-05", "6.92e-05", "7e-05", "7.08e-05", "7.16e-05",
"7.24e-05", "7.33e-05", "7.41e-05", "7.5e-05", "7.59e-05", "7.67e-05",
"7.76e-05", "7.85e-05", "7.94e-05", "8.04e-05", "8.13e-05", "8.22e-05",
"8.32e-05", "8.41e-05", "8.51e-05", "8.61e-05", "8.71e-05", "8.81e-05",
"8.91e-05", "9.02e-05", "9.12e-05", "9.23e-05", "9.33e-05", "9.44e-05",
"9.55e-05", "9.66e-05", "9.77e-05", "9.89e-05", "1e-04", "0.0001012",
"0.0001023", "0.0001035", "0.0001047", "0.0001059", "0.0001072",
"0.0001084", "0.0001096", "0.0001109", "0.0001122", "0.0001135",
"0.0001148", "0.0001161", "0.0001175", "0.0001189", "0.0001202",
"0.0001216", "0.000123", "0.0001245", "0.0001259", "0.0001274",
"0.0001288", "0.0001303", "0.0001318", "0.0001334", "0.0001349",
"0.0001365", "0.000138", "0.0001396", "0.0001413", "0.0001429",
"0.0001445", "0.0001462", "0.0001479", "0.0001496", "0.0001514",
"0.0001531", "0.0001549", "0.0001567", "0.0001585", "0.0001603",
"0.0001622", "0.0001641", "0.000166", "0.0001679", "0.0001698",
"0.0001718", "0.0001738", "0.0001758", "0.0001778", "0.0001799",
"0.000182", "0.0001841", "0.0001862", "0.0001884", "0.0001905",
"0.0001928", "0.000195", "0.0001972", "0.0001995", "0.0002018",
"0.0002042", "0.0002065", "0.0002089", "0.0002113", "0.0002138",
"0.0002163", "0.0002188", "0.0002213", "0.0002239", "0.0002265",
"0.0002291", "0.0002317", "0.0002344", "0.0002371", "0.0002399",
"0.0002427", "0.0002455", "0.0002483", "0.0002512", "0.0002541",
"0.000257", "0.00026", "0.000263", "0.0002661", "0.0002692",
"0.0002723", "0.0002754", "0.0002786", "0.0002818", "0.0002851",
"0.0002884", "0.0002917", "0.0002951", "0.0002985", "0.000302",
"0.0003055", "0.000309", "0.0003126", "0.0003162", "0.0003199",
"0.0003236", "0.0003273", "0.0003311", "0.000335", "0.0003388",
"0.0003428", "0.0003467", "0.0003508", "0.0003548", "0.0003589",
"0.0003631", "0.0003673", "0.0003715", "0.0003758", "0.0003802",
"0.0003846", "0.000389", "0.0003936", "0.0003981", "0.0004027",
"0.0004074", "0.0004121", "0.0004169", "0.0004217", "0.0004266",
"0.0004315", "0.0004365", "0.0004416", "0.0004467", "0.0004519",
"0.0004571", "0.0004624", "0.0004677", "0.0004732", "0.0004786",
"0.0004842", "0.0004898", "0.0004955", "5e-04", "0.0005012",
"0.000507", "0.0005129", "0.0005188", "0.0005248", "0.0005309",
"0.000537", "0.0005433", "0.0005495", "0.0005559", "0.0005623",
"0.0005689", "0.0005754", "0.0005821", "0.0005888", "0.0005957",
"0.0006026", "0.0006095", "0.0006166", "0.0006237", "0.000631",
"0.0006383", "0.0006457", "0.0006531", "0.0006607", "0.0006683",
"0.0006761", "0.0006839", "0.0006918", "0.0006998", "0.0007079",
"0.0007161", "0.0007244", "0.0007328", "0.0007413", "0.0007499",
"0.0007586", "0.0007674", "0.0007762", "0.0007852", "0.0007943",
"0.0008035", "0.0008128", "0.0008222", "0.0008318", "0.0008414",
"0.0008511", "0.000861", "0.000871", "0.000881", "0.0008913",
"0.0009016", "0.000912", "0.0009226", "0.0009333", "0.0009441",
"0.000955", "0.0009661", "0.0009772", "0.0009886", "0.001", "0.0010116",
"0.0010233", "0.0010351", "0.0010471", "0.0010593", "0.0010715",
"0.0010839", "0.0010965", "0.0011092", "0.001122", "0.001135",
"0.0011482", "0.0011614", "0.0011749", "0.0011885", "0.0012023",
"0.0012162", "0.0012303", "0.0012445", "0.0012589", "0.0012735",
"0.0012882", "0.0013032", "0.0013183", "0.0013335", "0.001349",
"0.0013646", "0.0013804", "0.0013964", "0.0014125", "0.0014289",
"0.0014454", "0.0014622", "0.0014791", "0.0014962", "0.0015136",
"0.0015311", "0.0015488", "0.0015668", "0.0015849", "0.0016032",
"0.0016218", "0.0016406", "0.0016596", "0.0016788", "0.0016982",
"0.0017179", "0.0017378", "0.0017579", "0.0017783", "0.0017989",
"0.0018197", "0.0018408", "0.0018621", "0.0018836", "0.0019055",
"0.0019275", "0.0019498", "0.0019724", "0.0019953", "0.0020184",
"0.0020417", "0.0020654", "0.0020893", "0.0021135", "0.002138",
"0.0021627", "0.0021878", "0.0022131", "0.0022387", "0.0022646",
"0.0022909", "0.0023174", "0.0023442", "0.0023714", "0.0023988",
"0.0024266", "0.0024547", "0.0024831", "0.0025119", "0.002541",
"0.0025704", "0.0026002", "0.0026303", "0.0026607", "0.0026915",
"0.0027227", "0.0027542", "0.0027861", "0.0028184", "0.002851",
"0.002884", "0.0029174", "0.0029512", "0.0029854", "0.00302",
"0.0030549", "0.0030903", "0.0031261", "0.0031623", "0.0031989",
"0.0032359", "0.0032734", "0.0033113", "0.0033497", "0.0033884",
"0.0034277", "0.0034674", "0.0035075", "0.0035481", "0.0035892",
"0.0036308", "0.0036728", "0.0037154", "0.0037584", "0.0038019",
"0.0038459", "0.0038905", "0.0039355", "0.0039811", "0.0040272",
"0.0040738", "0.004121", "0.0041687", "0.004217", "0.0042658",
"0.0043152", "0.0043652", "0.0044157", "0.0044668", "0.0045186",
"0.0045709", "0.0046238", "0.0046774", "0.0047315", "0.0047863",
"0.0048417", "0.0048978", "0.0049545", "0.005", "0.0050119",
"0.0050699", "0.0051286", "0.005188", "0.0052481", "0.0053088",
"0.0053703", "0.0054325", "0.0054954", "0.005559", "0.0056234",
"0.0056885", "0.0057544", "0.005821", "0.0058884", "0.0059566",
"0.0060256", "0.0060954", "0.006166", "0.0062373", "0.0063096",
"0.0063826", "0.0064565", "0.0065313", "0.0066069", "0.0066834",
"0.0067608", "0.0068391", "0.0069183", "0.0069984", "0.0070795",
"0.0071614", "0.0072444", "0.0073282", "0.0074131", "0.0074989",
"0.0075858", "0.0076736", "0.0077625", "0.0078524", "0.0079433",
"0.0080353", "0.0081283", "0.0082224", "0.0083176", "0.008414",
"0.0085114", "0.0086099", "0.0087096", "0.0088105", "0.0089125",
"0.0090157", "0.0091201", "0.0092257", "0.0093325", "0.0094406",
"0.0095499", "0.0096605", "0.0097724", "0.0098855", "0.01", "0.0101158",
"0.0102329", "0.0103514", "0.0104713", "0.0105925", "0.0107152",
"0.0108393", "0.0109648", "0.0110917", "0.0112202", "0.0113501",
"0.0114815", "0.0116145", "0.011749", "0.011885", "0.0120226",
"0.0121619", "0.0123027", "0.0124451", "0.0125893", "0.012735",
"0.0128825", "0.0130317", "0.0131826", "0.0133352", "0.0134896",
"0.0136458", "0.0138038", "0.0139637", "0.0141254", "0.0142889",
"0.0144544", "0.0146218", "0.0147911", "0.0149624", "0.0151356",
"0.0153109", "0.0154882", "0.0156675", "0.0158489", "0.0160325",
"0.0162181", "0.0164059", "0.0165959", "0.016788", "0.0169824",
"0.0171791", "0.017378", "0.0175792", "0.0177828", "0.0179887",
"0.018197", "0.0184077", "0.0186209", "0.0188365", "0.0190546",
"0.0192752", "0.0194984", "0.0197242", "0.0199526", "0.0201837",
"0.0204174", "0.0206538", "0.020893", "0.0211349", "0.0213796",
"0.0216272", "0.0218776", "0.0221309", "0.0223872", "0.0226464",
"0.0229087", "0.0231739", "0.0234423", "0.0237137", "0.0239883",
"0.0242661", "0.0245471", "0.0248313", "0.0251189", "0.0254097",
"0.025704", "0.0260016", "0.0263027", "0.0266073", "0.0269153",
"0.027227", "0.0275423", "0.0278612", "0.0281838", "0.0285102",
"0.0288403", "0.0291743", "0.0295121", "0.0298538", "0.0301995",
"0.0305492", "0.030903", "0.0312608", "0.0316228", "0.031989",
"0.0323594", "0.0327341", "0.0331131", "0.0334965", "0.0338844",
"0.0342768", "0.0346737", "0.0350752", "0.0354813", "0.0358922",
"0.0363078", "0.0367282", "0.0371535", "0.0375837", "0.0380189",
"0.0384592", "0.0389045", "0.039355", "0.0398107", "0.0402717",
"0.040738", "0.0412098", "0.0416869", "0.0421697", "0.042658",
"0.0431519", "0.0436516", "0.044157", "0.0446684", "0.0451856",
"0.0457088", "0.0462381", "0.0467735", "0.0473151", "0.047863",
"0.0484172", "0.0489779", "0.049545", "0.05", "0.0501187", "0.0506991",
"0.0512861", "0.05188", "0.0524807", "0.0530884", "0.0537032",
"0.054325", "0.0549541", "0.0555904", "0.0562341", "0.0568853",
"0.057544", "0.0582103", "0.0588844", "0.0595662", "0.060256",
"0.0609537", "0.0616595", "0.0623735", "0.0630957", "0.0638263",
"0.0645654", "0.0653131", "0.0660693", "0.0668344", "0.0676083",
"0.0683912", "0.0691831", "0.0699842", "0.0707946", "0.0716143",
"0.0724436", "0.0732825", "0.074131", "0.0749894", "0.0758578",
"0.0767361", "0.0776247", "0.0785236", "0.0794328", "0.0803526",
"0.0812831", "0.0822243", "0.0831764", "0.0841395", "0.0851138",
"0.0860994", "0.0870964", "0.0881049", "0.0891251", "0.0901571",
"0.0912011", "0.0922571", "0.0933254", "0.0944061", "0.0954993",
"0.0966051", "0.0977237", "0.0988553", "0.1", "0.1011579", "0.1023293",
"0.1035142", "0.1047129", "0.1059254", "0.1071519", "0.1083927",
"0.1096478", "0.1109175", "0.1122018", "0.1135011", "0.1148154",
"0.1161449", "0.1174898", "0.1188502", "0.1202264", "0.1216186",
"0.1230269", "0.1244515", "0.1258925", "0.1273503", "0.128825",
"0.1303167", "0.1318257", "0.1333521", "0.1348963", "0.1364583",
"0.1380384", "0.1396368", "0.1412538", "0.1428894", "0.144544",
"0.1462177", "0.1479108", "0.1496236", "0.15", "0.1513561", "0.1531087",
"0.1548817", "0.1566751", "0.1584893", "0.1603245", "0.162181",
"0.164059", "0.1659587", "0.1678804", "0.1698244", "0.1717908",
"0.1737801", "0.1757924", "0.1778279", "0.1798871", "0.1819701",
"0.1840772", "0.1862087", "0.1883649", "0.1905461", "0.1927525",
"0.1949845", "0.1972423", "0.1995262", "0.2", "0.2018366", "0.2041738",
"0.206538", "0.2089296", "0.2113489", "0.2137962", "0.2162719",
"0.2187762", "0.2213095", "0.2238721", "0.2264644", "0.2290868",
"0.2317395", "0.2344229", "0.2371374", "0.2398833", "0.242661",
"0.2454709", "0.2483133", "0.2511886", "0.2540973", "0.2570396",
"0.260016", "0.2630268", "0.2660725", "0.2691535", "0.2722701",
"0.2754229", "0.2786121", "0.2818383", "0.2851018", "0.2884032",
"0.2917427", "0.2951209", "0.2985383", "0.3", "0.3019952", "0.3054921",
"0.3090295", "0.3126079", "0.3162278", "0.3198895", "0.3235937",
"0.3273407", "0.3311311", "0.3349654", "0.3388442", "0.3427678",
"0.3467369", "0.3507519", "0.3548134", "0.3589219", "0.3630781",
"0.3672823", "0.3715352", "0.3758374", "0.3801894", "0.3845918",
"0.3890451", "0.3935501", "0.3981072", "0.4", "0.402717", "0.4073803",
"0.4120975", "0.4168694", "0.4216965", "0.4265795", "0.4315191",
"0.4365158", "0.4415704", "0.4466836", "0.4518559", "0.4570882",
"0.462381", "0.4677351", "0.4731513", "0.4786301", "0.4841724",
"0.4897788", "0.4954502", "0.5", "0.5102212", "0.5158276", "0.5213699",
"0.5268487", "0.5322649", "0.537619", "0.5429118", "0.5481441",
"0.5533164", "0.5584296", "0.5634842", "0.5684809", "0.5734205",
"0.5783035", "0.5831306", "0.5879025", "0.5926197", "0.597283",
"0.6", "0.6018928", "0.6064499", "0.6109549", "0.6154082", "0.6198106",
"0.6241626", "0.6284648", "0.6327177", "0.6369219", "0.6410781",
"0.6451866", "0.6492481", "0.6532631", "0.6572322", "0.6611558",
"0.6650346", "0.6688689", "0.6726593", "0.6764063", "0.6801105",
"0.6837722", "0.6873921", "0.6909705", "0.6945079", "0.6980048",
"0.7", "0.7014617", "0.7048791", "0.7082573", "0.7115968", "0.7148982",
"0.7181617", "0.7213879", "0.7245771", "0.7277299", "0.7308465",
"0.7339275", "0.7369732", "0.739984", "0.7429604", "0.7459027",
"0.7488114", "0.7516867", "0.7545291", "0.757339", "0.7601167",
"0.7628626", "0.7655771", "0.7682605", "0.7709132", "0.7735356",
"0.7761279", "0.7786905", "0.7812238", "0.7837281", "0.7862038",
"0.7886511", "0.7910704", "0.793462", "0.7958262", "0.7981634",
"0.8", "0.8004738", "0.8027577", "0.8050155", "0.8072475", "0.8094539",
"0.8116351", "0.8137913", "0.8159228", "0.8180299", "0.8201129",
"0.8221721", "0.8242076", "0.8262199", "0.8282092", "0.8301756",
"0.8321196", "0.8340413", "0.835941", "0.837819", "0.8396755",
"0.8415107", "0.8433249", "0.8451183", "0.8468913", "0.8486439",
"0.85", "0.8503764", "0.8520892", "0.8537823", "0.855456", "0.8571106",
"0.8587462", "0.8603632", "0.8619616", "0.8635417", "0.8651037",
"0.8666479", "0.8681743", "0.8696833", "0.871175", "0.8726497",
"0.8741075", "0.8755485", "0.8769731", "0.8783814", "0.8797736",
"0.8811498", "0.8825102", "0.8838551", "0.8851846", "0.8864989",
"0.8877982", "0.8890825", "0.8903522", "0.8916073", "0.8928481",
"0.8940746", "0.8952871", "0.8964858", "0.8976707", "0.8988421",
"0.9", "0.9011447", "0.9022763", "0.9033949", "0.9045007", "0.9055939",
"0.9066746", "0.9077429", "0.9087989", "0.9098429", "0.9108749",
"0.9118951", "0.9129036", "0.9139006", "0.9148862", "0.9158605",
"0.9168236", "0.9177757", "0.9187169", "0.9196474", "0.9205672",
"0.9214764", "0.9223753", "0.9232639", "0.9241422", "0.9250106",
"0.925869", "0.9267175", "0.9275564", "0.9283857", "0.9292054",
"0.9300158", "0.9308169", "0.9316088", "0.9323917", "0.9331656",
"0.9339307", "0.9346869", "0.9354346", "0.9361737", "0.9369043",
"0.9376265", "0.9383405", "0.9390463", "0.939744", "0.9404338",
"0.9411156", "0.9417897", "0.942456", "0.9431147", "0.9437659",
"0.9444096", "0.9450459", "0.945675", "0.9462968", "0.9469116",
"0.9475193", "0.94812", "0.9487139", "0.9493009", "0.9498813",
"0.95", "0.950455", "0.9510221", "0.9515828", "0.952137", "0.9526849",
"0.9532265", "0.9537619", "0.9542912", "0.9548144", "0.9553316",
"0.955843", "0.9563484", "0.9568481", "0.957342", "0.9578303",
"0.9583131", "0.9587902", "0.959262", "0.9597283", "0.9601893",
"0.960645", "0.9610955", "0.9615408", "0.9619811", "0.9624163",
"0.9628465", "0.9632718", "0.9636922", "0.9641078", "0.9645187",
"0.9649248", "0.9653263", "0.9657232", "0.9661156", "0.9665035",
"0.9668869", "0.9672659", "0.9676406", "0.968011", "0.9683772",
"0.9687392", "0.969097", "0.9694508", "0.9698005", "0.9701462",
"0.9704879", "0.9708257", "0.9711597", "0.9714898", "0.9718162",
"0.9721388", "0.9724577", "0.972773", "0.9730847", "0.9733927",
"0.9736973", "0.9739984", "0.974296", "0.9745903", "0.9748811",
"0.9751687", "0.9754529", "0.9757339", "0.9760117", "0.9762863",
"0.9765577", "0.9768261", "0.9770913", "0.9773536", "0.9776128",
"0.9778691", "0.9781224", "0.9783728", "0.9786204", "0.9788651",
"0.979107", "0.9793462", "0.9795826", "0.9798163", "0.9800474",
"0.9802758", "0.9805016", "0.9807248", "0.9809454", "0.9811635",
"0.9813791", "0.9815923", "0.981803", "0.9820113", "0.9822172",
"0.9824208", "0.982622", "0.9828209", "0.9830176", "0.983212",
"0.9834041", "0.9835941", "0.9837819", "0.9839675", "0.9841511",
"0.9843325", "0.9845118", "0.9846891", "0.9848644", "0.9850376",
"0.9852089", "0.9853782", "0.9855456", "0.9857111", "0.9858746",
"0.9860363", "0.9861962", "0.9863542", "0.9865104", "0.9866648",
"0.9868174", "0.9869683", "0.9871175", "0.987265", "0.9874107",
"0.9875549", "0.9876973", "0.9878381", "0.9879774", "0.988115",
"0.988251", "0.9883855", "0.9885185", "0.9886499", "0.9887798",
"0.9889083", "0.9890352", "0.9891607", "0.9892848", "0.9894075",
"0.9895287", "0.9896486", "0.9897671", "0.9898842", "0.99", "0.9901145",
"0.9902276", "0.9903395", "0.9904501", "0.9905594", "0.9906675",
"0.9907743", "0.9908799", "0.9909843", "0.9910875", "0.9911895",
"0.9912904", "0.9913901", "0.9914886", "0.991586", "0.9916824",
"0.9917776", "0.9918717", "0.9919647", "0.9920567", "0.9921476",
"0.9922375", "0.9923264", "0.9924142", "0.9925011", "0.9925869",
"0.9926718", "0.9927556", "0.9928386", "0.9929205", "0.9930016",
"0.9930817", "0.9931609", "0.9932392", "0.9933166", "0.9933931",
"0.9934687", "0.9935435", "0.9936174", "0.9936904", "0.9937627",
"0.993834", "0.9939046", "0.9939744", "0.9940434", "0.9941116",
"0.994179", "0.9942456", "0.9943115", "0.9943766", "0.994441",
"0.9945046", "0.9945675", "0.9946297", "0.9946912", "0.9947519",
"0.994812", "0.9948714", "0.9949301", "0.9949881", "0.995", "0.9950455",
"0.9951022", "0.9951583", "0.9952137", "0.9952685", "0.9953226",
"0.9953762", "0.9954291", "0.9954814", "0.9955332", "0.9955843",
"0.9956348", "0.9956848", "0.9957342", "0.995783", "0.9958313",
"0.995879", "0.9959262", "0.9959728", "0.9960189", "0.9960645",
"0.9961095", "0.9961541", "0.9961981", "0.9962416", "0.9962846",
"0.9963272", "0.9963692", "0.9964108", "0.9964519", "0.9964925",
"0.9965326", "0.9965723", "0.9966116", "0.9966503", "0.9966887",
"0.9967266", "0.9967641", "0.9968011", "0.9968377", "0.9968739",
"0.9969097", "0.9969451", "0.99698", "0.9970146", "0.9970488",
"0.9970826", "0.997116", "0.997149", "0.9971816", "0.9972139",
"0.9972458", "0.9972773", "0.9973085", "0.9973393", "0.9973697",
"0.9973998", "0.9974296", "0.997459", "0.9974881", "0.9975169",
"0.9975453", "0.9975734", "0.9976012", "0.9976286", "0.9976558",
"0.9976826", "0.9977091", "0.9977354", "0.9977613", "0.9977869",
"0.9978122", "0.9978373", "0.997862", "0.9978865", "0.9979107",
"0.9979346", "0.9979583", "0.9979816", "0.9980047", "0.9980276",
"0.9980502", "0.9980725", "0.9980945", "0.9981164", "0.9981379",
"0.9981592", "0.9981803", "0.9982011", "0.9982217", "0.9982421",
"0.9982622", "0.9982821", "0.9983018", "0.9983212", "0.9983404",
"0.9983594", "0.9983782", "0.9983968", "0.9984151", "0.9984332",
"0.9984512", "0.9984689", "0.9984864", "0.9985038", "0.9985209",
"0.9985378", "0.9985546", "0.9985711", "0.9985875", "0.9986036",
"0.9986196", "0.9986354", "0.998651", "0.9986665", "0.9986817",
"0.9986968", "0.9987118", "0.9987265", "0.9987411", "0.9987555",
"0.9987697", "0.9987838", "0.9987977", "0.9988115", "0.9988251",
"0.9988386", "0.9988518", "0.998865", "0.998878", "0.9988908",
"0.9989035", "0.9989161", "0.9989285", "0.9989407", "0.9989529",
"0.9989649", "0.9989767", "0.9989884", "0.999", "0.9990114",
"0.9990228", "0.9990339", "0.999045", "0.9990559", "0.9990667",
"0.9990774", "0.999088", "0.9990984", "0.9991087", "0.999119",
"0.999129", "0.999139", "0.9991489", "0.9991586", "0.9991682",
"0.9991778", "0.9991872", "0.9991965", "0.9992057", "0.9992148",
"0.9992238", "0.9992326", "0.9992414", "0.9992501", "0.9992587",
"0.9992672", "0.9992756", "0.9992839", "0.9992921", "0.9993002",
"0.9993082", "0.9993161", "0.9993239", "0.9993317", "0.9993393",
"0.9993469", "0.9993543", "0.9993617", "0.999369", "0.9993763",
"0.9993834", "0.9993905", "0.9993974", "0.9994043", "0.9994112",
"0.9994179", "0.9994246", "0.9994311", "0.9994377", "0.9994441",
"0.9994505", "0.9994567", "0.999463", "0.9994691", "0.9994752",
"0.9994812", "0.9994871", "0.999493", "0.9994988", "0.9995",
"0.9995045", "0.9995102", "0.9995158", "0.9995214", "0.9995268",
"0.9995323", "0.9995376", "0.9995429", "0.9995481", "0.9995533",
"0.9995584", "0.9995635", "0.9995685", "0.9995734", "0.9995783",
"0.9995831", "0.9995879", "0.9995926", "0.9995973", "0.9996019",
"0.9996064", "0.999611", "0.9996154", "0.9996198", "0.9996242",
"0.9996285", "0.9996327", "0.9996369", "0.9996411", "0.9996452",
"0.9996492", "0.9996533", "0.9996572", "0.9996612", "0.999665",
"0.9996689", "0.9996727", "0.9996764", "0.9996801", "0.9996838",
"0.9996874", "0.999691", "0.9996945", "0.999698", "0.9997015",
"0.9997049", "0.9997083", "0.9997116", "0.9997149", "0.9997182",
"0.9997214", "0.9997246", "0.9997277", "0.9997308", "0.9997339",
"0.999737", "0.99974", "0.999743", "0.9997459", "0.9997488",
"0.9997517", "0.9997545", "0.9997573", "0.9997601", "0.9997629",
"0.9997656", "0.9997683", "0.9997709", "0.9997735", "0.9997761",
"0.9997787", "0.9997812", "0.9997837", "0.9997862", "0.9997887",
"0.9997911", "0.9997935", "0.9997958", "0.9997982", "0.9998005",
"0.9998028", "0.999805", "0.9998072", "0.9998095", "0.9998116",
"0.9998138", "0.9998159", "0.999818", "0.9998201", "0.9998222",
"0.9998242", "0.9998262", "0.9998282", "0.9998302", "0.9998321",
"0.999834", "0.9998359", "0.9998378", "0.9998397", "0.9998415",
"0.9998433", "0.9998451", "0.9998469", "0.9998486", "0.9998504",
"0.9998521", "0.9998538", "0.9998555", "0.9998571", "0.9998587",
"0.9998604", "0.999862", "0.9998635", "0.9998651", "0.9998666",
"0.9998682", "0.9998697", "0.9998712", "0.9998726", "0.9998741",
"0.9998755", "0.999877", "0.9998784", "0.9998798", "0.9998811",
"0.9998825", "0.9998839", "0.9998852", "0.9998865", "0.9998878",
"0.9998891", "0.9998904", "0.9998916", "0.9998928", "0.9998941",
"0.9998953", "0.9998965", "0.9998977", "0.9998988", "0.9999",
"0.9999011", "0.9999023", "0.9999034", "0.9999045", "0.9999056",
"0.9999067", "0.9999077", "0.9999088", "0.9999098", "0.9999109",
"0.9999119", "0.9999129", "0.9999139", "0.9999149", "0.9999159",
"0.9999168", "0.9999178", "0.9999187", "0.9999196", "0.9999206",
"0.9999215", "0.9999224", "0.9999233", "0.9999241", "0.999925",
"0.9999259", "0.9999267", "0.9999276", "0.9999284", "0.9999292",
"0.99993", "0.9999308", "0.9999316", "0.9999324", "0.9999332",
"0.9999339", "0.9999347", "0.9999354", "0.9999362", "0.9999369",
"0.9999376", "0.9999383", "0.999939", "0.9999397", "0.9999404",
"0.9999411", "0.9999418", "0.9999425", "0.9999431", "0.9999438",
"0.9999444", "0.999945", "0.9999457", "0.9999463", "0.9999469",
"0.9999475", "0.9999481", "0.9999487", "0.9999493", "0.9999499",
"0.9999505", "0.999951", "0.9999516", "0.9999521", "0.9999527",
"0.9999532", "0.9999538", "0.9999543", "0.9999548", "0.9999553",
"0.9999558", "0.9999563", "0.9999568", "0.9999573", "0.9999578",
"0.9999583", "0.9999588", "0.9999593", "0.9999597", "0.9999602",
"0.9999606", "0.9999611", "0.9999615", "0.999962", "0.9999624",
"0.9999628", "0.9999633", "0.9999637", "0.9999641", "0.9999645",
"0.9999649", "0.9999653", "0.9999657", "0.9999661", "0.9999665",
"0.9999669", "0.9999673", "0.9999676", "0.999968", "0.9999684",
"0.9999687", "0.9999691", "0.9999695", "0.9999698", "0.9999701",
"0.9999705", "0.9999708", "0.9999712", "0.9999715", "0.9999718",
"0.9999721", "0.9999725", "0.9999728", "0.9999731", "0.9999734",
"0.9999737", "0.999974", "0.9999743", "0.9999746", "0.9999749",
"0.9999752", "0.9999755", "0.9999757", "0.999976", "0.9999763",
"0.9999766", "0.9999768", "0.9999771", "0.9999774", "0.9999776",
"0.9999779", "0.9999781", "0.9999784", "0.9999786", "0.9999789",
"0.9999791", "0.9999793", "0.9999796", "0.9999798", "0.99998",
"0.9999803", "0.9999805", "0.9999807", "0.9999809", "0.9999812",
"0.9999814", "0.9999816", "0.9999818", "0.999982", "0.9999822",
"0.9999824", "0.9999826", "0.9999828", "0.999983", "0.9999832",
"0.9999834", "0.9999836", "0.9999838", "0.999984", "0.9999842",
"0.9999843", "0.9999845", "0.9999847", "0.9999849", "0.999985",
"0.9999852", "0.9999854", "0.9999855", "0.9999857", "0.9999859",
"0.999986", "0.9999862", "0.9999864", "0.9999865", "0.9999867",
"0.9999868", "0.999987", "0.9999871", "0.9999873", "0.9999874",
"0.9999876", "0.9999877", "0.9999878", "0.999988", "0.9999881",
"0.9999883", "0.9999884", "0.9999885", "0.9999886", "0.9999888",
"0.9999889", "0.999989", "0.9999892", "0.9999893", "0.9999894",
"0.9999895", "0.9999896", "0.9999898", "0.9999899", "0.99999"
), class = "data.frame")
}


# ------------------------------------------------------------------------------


.jbLM =
function()
{
structure(list("5" = c(0.052470165, 0.052479988, 0.052481227,
0.052482714, 0.052484752, 0.0524849, 0.052487004, 0.052487547,
0.052492897, 0.052493878, 0.052493879, 0.052494842, 0.052496092,
0.052497834, 0.052499018, 0.052505397, 0.052509442, 0.052511664,
0.052515321, 0.052517838, 0.052529162, 0.05253247, 0.052537375,
0.05254833, 0.052553431, 0.052558648, 0.052559815, 0.052561665,
0.052570098, 0.052574164, 0.05258048, 0.05258408, 0.052586498,
0.052588564, 0.052593144, 0.052598993, 0.052600907, 0.052607888,
0.05260833, 0.05261147, 0.052611702, 0.052613837, 0.052620487,
0.052628488, 0.052631379, 0.052633052, 0.05263496, 0.052635328,
0.052641592, 0.052645432, 0.052651572, 0.052656332, 0.052660683,
0.052665999, 0.052673242, 0.052676104, 0.052688929, 0.052694803,
0.052699646, 0.052701458, 0.052709881, 0.052712268, 0.05272215,
0.052727874, 0.052730256, 0.052735346, 0.052743604, 0.05274666,
0.052760922, 0.052767973, 0.052775119, 0.052778247, 0.052781513,
0.052786578, 0.052791093, 0.052803672, 0.052807291, 0.052821458,
0.052823871, 0.052828785, 0.052833725, 0.052840874, 0.052843109,
0.052847147, 0.052853554, 0.052854431, 0.052861028, 0.052868051,
0.052877535, 0.052886001, 0.052888722, 0.052898288, 0.052898694,
0.052908141, 0.052910534, 0.052914546, 0.052923779, 0.052927527,
0.052935115, 0.052937428, 0.052942354, 0.052949011, 0.052953643,
0.052955157, 0.052959405, 0.052966149, 0.052973704, 0.052979684,
0.052991848, 0.053004827, 0.053010343, 0.053013686, 0.053025845,
0.053033555, 0.053044444, 0.053058405, 0.053066997, 0.053073765,
0.053081806, 0.053088497, 0.053090763, 0.053108566, 0.053114103,
0.053122934, 0.053128147, 0.053146206, 0.053152252, 0.053164623,
0.05316833, 0.053177206, 0.053182248, 0.05319418, 0.053199323,
0.053207293, 0.053213266, 0.053217012, 0.053225175, 0.053229997,
0.053235579, 0.053242451, 0.05324788, 0.05326003, 0.053268582,
0.053276668, 0.053286, 0.053297456, 0.053302415, 0.053308875,
0.053318566, 0.053325458, 0.053328303, 0.053336053, 0.053344035,
0.053358689, 0.053366123, 0.053379, 0.053385515, 0.053401057,
0.053409904, 0.053419693, 0.05342981, 0.053439077, 0.053447854,
0.053455487, 0.053471333, 0.053484097, 0.053494267, 0.053504441,
0.053516544, 0.053536177, 0.053548321, 0.05356068, 0.053567346,
0.053579452, 0.053587055, 0.053599905, 0.053615242, 0.053631305,
0.053642128, 0.053656104, 0.053669132, 0.053682175, 0.053698071,
0.053707687, 0.053716235, 0.053726155, 0.053745809, 0.053755976,
0.05377043, 0.053780103, 0.053789211, 0.053806224, 0.053817766,
0.053824376, 0.053845433, 0.053863096, 0.053870538, 0.053879744,
0.053890909, 0.053903248, 0.053919547, 0.05394363, 0.053957,
0.053965009, 0.053980621, 0.05400366, 0.054029899, 0.054039456,
0.054054511, 0.054071398, 0.054083595, 0.054101628, 0.054121831,
0.054135171, 0.054155278, 0.054169236, 0.054182059, 0.054193931,
0.054209747, 0.054225624, 0.054241778, 0.054263774, 0.054284711,
0.05430284, 0.054317464, 0.054336061, 0.054357414, 0.054374382,
0.054387071, 0.054405249, 0.054427059, 0.054446871, 0.054475202,
0.05450106, 0.054513578, 0.054526546, 0.054554829, 0.054572751,
0.054595561, 0.054614961, 0.054635283, 0.054657007, 0.054673574,
0.054692394, 0.054710538, 0.054731138, 0.05474872, 0.054763391,
0.054777874, 0.054792343, 0.054810068, 0.054829141, 0.054855706,
0.05487446, 0.054891618, 0.054910297, 0.054929487, 0.054947177,
0.054964138, 0.054984542, 0.055006999, 0.055028477, 0.055049614,
0.055073464, 0.055094798, 0.055114758, 0.055143275, 0.055167051,
0.05518981, 0.055216064, 0.055251392, 0.055276685, 0.055296083,
0.05532189, 0.055343876, 0.05536461, 0.055396205, 0.055426364,
0.055451719, 0.055475523, 0.055500743, 0.055533456, 0.055552921,
0.055575591, 0.05560366, 0.055628538, 0.055655587, 0.055674449,
0.05570588, 0.055734456, 0.055761618, 0.055791158, 0.055826565,
0.055857829, 0.055881359, 0.055912539, 0.055936362, 0.055970103,
0.056011042, 0.056037279, 0.056063243, 0.056083742, 0.056108859,
0.056139325, 0.056170292, 0.056204059, 0.05623625, 0.056263069,
0.056298258, 0.056333661, 0.056369701, 0.056405303, 0.056436116,
0.056475045, 0.05651771, 0.056546871, 0.056584349, 0.056632716,
0.056671386, 0.05670932, 0.056743841, 0.056782046, 0.056825988,
0.056864467, 0.056903272, 0.056941127, 0.056967605, 0.057003367,
0.057054254, 0.057095308, 0.057140073, 0.057180561, 0.057219623,
0.0572585, 0.057292375, 0.057333975, 0.057366877, 0.05740834,
0.057449003, 0.057494547, 0.057516542, 0.057521952, 0.057572976,
0.057610281, 0.057657893, 0.057708535, 0.057746393, 0.057793558,
0.057837331, 0.057876209, 0.057929694, 0.057973803, 0.058019725,
0.058068957, 0.058121873, 0.058174747, 0.058221805, 0.058286619,
0.058334838, 0.058380104, 0.058415891, 0.058465996, 0.058515601,
0.05855796, 0.058600887, 0.058655536, 0.058701084, 0.058760333,
0.058816393, 0.058862081, 0.058905556, 0.058971496, 0.059019469,
0.059063764, 0.059114972, 0.059171751, 0.059227574, 0.059274917,
0.05933176, 0.059390527, 0.059452501, 0.059511612, 0.059564306,
0.059636492, 0.059698885, 0.059766115, 0.059837585, 0.059887524,
0.059945926, 0.059999074, 0.060056947, 0.060122083, 0.060186488,
0.060244722, 0.060300044, 0.060352215, 0.060408161, 0.060470544,
0.060526999, 0.060598813, 0.060660754, 0.060724341, 0.060792375,
0.060847753, 0.060926621, 0.060992407, 0.061057209, 0.061120324,
0.0611963, 0.061270521, 0.061337538, 0.061417506, 0.061484316,
0.061562481, 0.061627421, 0.061694074, 0.06177382, 0.061851749,
0.061922786, 0.061995792, 0.062076488, 0.062159312, 0.062245855,
0.062315056, 0.062395949, 0.06247632, 0.062562632, 0.062644737,
0.062727552, 0.062815836, 0.062903098, 0.062989177, 0.063061467,
0.063144867, 0.063223598, 0.063317891, 0.063416288, 0.063510871,
0.063616702, 0.06370956, 0.06380296, 0.063900372, 0.063993006,
0.064080463, 0.064175429, 0.064261166, 0.064361682, 0.064460143,
0.064558649, 0.064658988, 0.06476212, 0.064871314, 0.064967746,
0.065064769, 0.065171274, 0.065262971, 0.065371605, 0.065479934,
0.065578652, 0.065700293, 0.065815076, 0.065912834, 0.066029784,
0.066140289, 0.066245342, 0.066359611, 0.066476329, 0.066584444,
0.066709781, 0.066818655, 0.066930523, 0.067050569, 0.067169022,
0.067288403, 0.067409223, 0.067525058, 0.06766065, 0.067778712,
0.067895012, 0.068018552, 0.068145529, 0.068267553, 0.068388902,
0.068519719, 0.068643022, 0.068766217, 0.068896974, 0.069016038,
0.069168397, 0.069299157, 0.069431158, 0.069560116, 0.069704215,
0.069843644, 0.069971697, 0.070103338, 0.07023748, 0.070377416,
0.070509735, 0.070653509, 0.070796704, 0.07094601, 0.071099757,
0.071238576, 0.071390633, 0.071552146, 0.071694361, 0.071836762,
0.071999384, 0.072158808, 0.072321327, 0.072486926, 0.072628402,
0.072795116, 0.07295204, 0.073125097, 0.073303727, 0.073474171,
0.073640301, 0.073817196, 0.073983465, 0.074142403, 0.074328202,
0.07451504, 0.074692498, 0.074867747, 0.075036873, 0.075213045,
0.075371473, 0.075552608, 0.075730952, 0.075924551, 0.076122715,
0.076311216, 0.076498756, 0.076695559, 0.076889651, 0.077069414,
0.077259419, 0.077457981, 0.077668035, 0.077829236, 0.077867934,
0.078073287, 0.07828708, 0.078488742, 0.078700243, 0.078918095,
0.07914362, 0.079349483, 0.07956642, 0.079789373, 0.080002642,
0.080214866, 0.080442838, 0.080649163, 0.080873901, 0.081107177,
0.081319937, 0.081567753, 0.081798954, 0.08203257, 0.082263671,
0.082501223, 0.082743095, 0.082966431, 0.083231188, 0.083480618,
0.083737812, 0.083986848, 0.084233095, 0.084484699, 0.084736427,
0.084995372, 0.08526503, 0.085529239, 0.085794454, 0.086063969,
0.086334599, 0.086603822, 0.08687205, 0.087151717, 0.087427125,
0.087721383, 0.088010904, 0.088313013, 0.088598517, 0.088880033,
0.089159514, 0.089450913, 0.089747583, 0.090045973, 0.090344842,
0.090647854, 0.090950884, 0.091240849, 0.091558786, 0.091875483,
0.092189983, 0.092493542, 0.092815099, 0.093121572, 0.093436734,
0.093764099, 0.094085097, 0.094423547, 0.09475989, 0.095098175,
0.095452041, 0.095791969, 0.096143076, 0.096496183, 0.096840964,
0.097184454, 0.097533137, 0.097873883, 0.09823798, 0.09859292,
0.098970052, 0.099347618, 0.099714553, 0.100080883, 0.100464315,
0.100855699, 0.101254573, 0.101657438, 0.102046963, 0.102431681,
0.102840317, 0.103244488, 0.103652006, 0.104065986, 0.104481629,
0.104906083, 0.105329388, 0.105753564, 0.106161729, 0.106582303,
0.106998509, 0.10742242, 0.107859271, 0.108306856, 0.108735774,
0.109187974, 0.109652307, 0.110120194, 0.110561495, 0.111027163,
0.11150901, 0.111977946, 0.112446572, 0.112932691, 0.11340247,
0.113878572, 0.11437796, 0.114882729, 0.115384302, 0.115870617,
0.116395596, 0.116912297, 0.11740143, 0.117901353, 0.118432072,
0.118969208, 0.119493672, 0.120021085, 0.120562712, 0.12109451,
0.121637082, 0.122187055, 0.122741896, 0.123296, 0.123864289,
0.124421527, 0.124998962, 0.125573542, 0.126174533, 0.126761174,
0.127350097, 0.127942086, 0.128531919, 0.12912565, 0.129744698,
0.130359095, 0.130966486, 0.131583949, 0.132213425, 0.132834131,
0.133471537, 0.13409182, 0.13473705, 0.135397919, 0.136050088,
0.136696613, 0.137355445, 0.138032578, 0.138709971, 0.139381378,
0.140052163, 0.14073491, 0.141428932, 0.142139672, 0.14284619,
0.143559704, 0.14426246, 0.14500191, 0.145712792, 0.14644532,
0.147200793, 0.14795636, 0.148706951, 0.149459648, 0.150227274,
0.150975582, 0.151752903, 0.152528495, 0.153330724, 0.154120583,
0.154920559, 0.155753484, 0.156563183, 0.157383151, 0.158215658,
0.159059398, 0.159886712, 0.160706772, 0.161563484, 0.162446575,
0.163307498, 0.164204058, 0.165094404, 0.165977168, 0.166867592,
0.16777023, 0.168688464, 0.169593089, 0.17051402, 0.17143549,
0.172362603, 0.173311887, 0.174254926, 0.175226169, 0.175972583,
0.176168081, 0.177113701, 0.178084983, 0.179057617, 0.180045234,
0.18102604, 0.182012732, 0.18303215, 0.184055821, 0.185083747,
0.186118818, 0.187166773, 0.18819944, 0.189208336, 0.190272574,
0.191346587, 0.192428787, 0.193492384, 0.19455919, 0.195636192,
0.196750706, 0.197866169, 0.19900669, 0.200130779, 0.201262417,
0.202423677, 0.203554436, 0.204708139, 0.205875288, 0.207036724,
0.208231326, 0.209442365, 0.210646426, 0.211847114, 0.21303464,
0.214250037, 0.21547567, 0.216741754, 0.21795791, 0.219225036,
0.220482653, 0.221772911, 0.223075154, 0.224364736, 0.225665912,
0.226972712, 0.228278037, 0.22960312, 0.230920243, 0.232234678,
0.233590175, 0.234943395, 0.236278433, 0.237675867, 0.239051915,
0.240419746, 0.241803662, 0.243195274, 0.24460318, 0.245986843,
0.247425103, 0.248847378, 0.250283034, 0.251755465, 0.253218577,
0.25468959, 0.256153165, 0.257608361, 0.259090004, 0.260591779,
0.262083691, 0.263593738, 0.265130052, 0.266662429, 0.268192743,
0.269732639, 0.271280597, 0.272840155, 0.27442519, 0.275971454,
0.277565429, 0.279151186, 0.280762347, 0.282365688, 0.283989397,
0.285596891, 0.287247189, 0.288899473, 0.290560509, 0.292225292,
0.293884349, 0.29556405, 0.297250202, 0.298924789, 0.300618801,
0.302317632, 0.302695636, 0.304035272, 0.305719545, 0.307444189,
0.309162685, 0.310877532, 0.312591498, 0.314346338, 0.316098633,
0.317840403, 0.319601215, 0.321376956, 0.32314503, 0.324896012,
0.326693289, 0.328497252, 0.330259791, 0.332062853, 0.333863106,
0.335674652, 0.337452884, 0.33925661, 0.341073618, 0.342869113,
0.344688059, 0.346502678, 0.346888283, 0.348327328, 0.350142652,
0.351950523, 0.353758149, 0.355564352, 0.357367144, 0.359166854,
0.360958041, 0.362749586, 0.364538941, 0.366298432, 0.368057348,
0.369799654, 0.371516272, 0.37321268, 0.3748495, 0.376431562,
0.378045922, 0.379663389, 0.381329847, 0.383006892, 0.384707289,
0.386411444, 0.388148858, 0.389912329, 0.391697412, 0.39349919,
0.395330177, 0.397180829, 0.399057667, 0.400953184, 0.402868023,
0.40480865, 0.406770746, 0.408736494, 0.409578047, 0.410736575,
0.412783862, 0.414853289, 0.416947189, 0.419061486, 0.421208875,
0.423352835, 0.425540946, 0.427762632, 0.430018276, 0.432310224,
0.434605949, 0.436943267, 0.439297493, 0.441709135, 0.44412421,
0.446579722, 0.449085514, 0.451591774, 0.454140122, 0.45671929,
0.459338165, 0.461985225, 0.464685287, 0.467411955, 0.468533586,
0.470142548, 0.472910039, 0.475741369, 0.47858597, 0.481446745,
0.484357093, 0.487307759, 0.490300277, 0.493325135, 0.496395494,
0.499499412, 0.502639164, 0.505792834, 0.509003801, 0.512252376,
0.515555907, 0.518906326, 0.522271745, 0.525689047, 0.528444967,
0.534633649, 0.538030903, 0.541425374, 0.544742542, 0.547994522,
0.551230963, 0.55444231, 0.557620928, 0.560762984, 0.563869092,
0.566922797, 0.569963126, 0.572977541, 0.575959158, 0.578910849,
0.581816713, 0.584683281, 0.587514046, 0.589172402, 0.590322203,
0.593088366, 0.595823957, 0.598534117, 0.601230269, 0.603904432,
0.60655138, 0.609166509, 0.61172416, 0.614251829, 0.61674597,
0.619213411, 0.621636174, 0.6240707, 0.626480892, 0.628822996,
0.631154318, 0.633481696, 0.635762469, 0.638024088, 0.640332879,
0.642675666, 0.64502398, 0.647392421, 0.649779028, 0.651144862,
0.6521659, 0.654557354, 0.656927015, 0.659282372, 0.661648238,
0.664007821, 0.666341212, 0.668687581, 0.671001732, 0.67330795,
0.675606659, 0.677876671, 0.680155267, 0.682420933, 0.684664482,
0.686898709, 0.689125837, 0.691330266, 0.69351974, 0.69570916,
0.697883097, 0.70005366, 0.702198771, 0.704333371, 0.706445037,
0.708566537, 0.710648863, 0.712716367, 0.714809464, 0.716965887,
0.719197024, 0.721465423, 0.72374083, 0.726047658, 0.728377098,
0.730223255, 0.730715257, 0.733079708, 0.735447269, 0.737861727,
0.740331757, 0.742751188, 0.745218075, 0.747685786, 0.750178554,
0.75268831, 0.755245642, 0.7577865, 0.760337191, 0.762904442,
0.765455832, 0.768060375, 0.770666115, 0.773268635, 0.775950199,
0.7786213, 0.781332048, 0.784057699, 0.78680819, 0.789585035,
0.792358048, 0.794555921, 0.795177757, 0.798051692, 0.800951015,
0.803905241, 0.806899751, 0.809929405, 0.812997153, 0.816098273,
0.819341174, 0.822601257, 0.826015744, 0.829559167, 0.833279682,
0.837282807, 0.841914723, 0.84674626, 0.851587127, 0.856433908,
0.861327173, 0.866186649, 0.871109095, 0.875986936, 0.880969371,
0.885820729, 0.890738489, 0.895645316, 0.900513619, 0.905332823,
0.910226659, 0.915074619, 0.919963454, 0.924840449, 0.929672851,
0.934584338, 0.939508185, 0.944426338, 0.94933579, 0.954166032,
0.959001596, 0.963822252, 0.968677394, 0.973533928, 0.978377081,
0.983171388, 0.988001833, 0.99285831, 0.99759517, 1.002475441,
1.00725245, 1.011992798, 1.01674803, 1.021484106, 1.026193148,
1.030974716, 1.035693934, 1.040433424, 1.045128756, 1.049749642,
1.054421002, 1.059096756, 1.063842849, 1.068419699, 1.073022058,
1.077680383, 1.082315675, 1.086856747, 1.091457231, 1.095973199,
1.100467839, 1.105031829, 1.109486153, 1.114012469, 1.118619328,
1.12307946, 1.12752654, 1.131872479, 1.136335799, 1.140764496,
1.145169214, 1.149517995, 1.153975562, 1.158320423, 1.162641185,
1.166972381, 1.171296555, 1.175514416, 1.179798295, 1.184038043,
1.188249018, 1.192486353, 1.196664813, 1.200837266, 1.204987952,
1.209187885, 1.21332695, 1.217359653, 1.218145707, 1.221371411,
1.225522459, 1.229548193, 1.233521145, 1.237613847, 1.241599977,
1.245553855, 1.24952797, 1.253454219, 1.257341712, 1.261197535,
1.26513069, 1.269013644, 1.272841752, 1.276657628, 1.280516189,
1.284321853, 1.288029272, 1.291804121, 1.295514278, 1.299229939,
1.302952046, 1.306687319, 1.310381746, 1.314072354, 1.317636773,
1.321364676, 1.325031896, 1.328619429, 1.332137817, 1.335666305,
1.339240732, 1.342779338, 1.346354466, 1.349828347, 1.353275753,
1.356768092, 1.360203447, 1.36356166, 1.366984289, 1.370356898,
1.373716524, 1.377087111, 1.3804117, 1.383715019, 1.38702761,
1.390404111, 1.393702374, 1.396852704, 1.400025155, 1.403231952,
1.406414295, 1.409536302, 1.412635759, 1.415813696, 1.418880849,
1.421969898, 1.425111722, 1.428174437, 1.431131855, 1.434120121,
1.437109742, 1.44011235, 1.443069887, 1.44615211, 1.449021331,
1.452074821, 1.455000796, 1.457931514, 1.460826278, 1.463787342,
1.466704236, 1.46958388, 1.472539203, 1.475279133, 1.478108787,
1.480982838, 1.483854417, 1.486700358, 1.489337911, 1.492063556,
1.49472949, 1.497400713, 1.500069737, 1.502772871, 1.505367226,
1.507990246, 1.510455589, 1.512998731, 1.515567372, 1.518049197,
1.520627632, 1.523143672, 1.525667936, 1.52826972, 1.530774309,
1.533231354, 1.535714651, 1.538212884, 1.540659081, 1.543078219,
1.545542939, 1.547955108, 1.550347116, 1.552728264, 1.555092705,
1.5574214, 1.559646744, 1.561889717, 1.564303206, 1.566537901,
1.568788321, 1.571030953, 1.573258377, 1.575469253, 1.577703323,
1.579884961, 1.582055073, 1.584102538, 1.586174757, 1.588293553,
1.590416812, 1.592479878, 1.594488975, 1.596607603, 1.598670903,
1.600792281, 1.602861174, 1.604916642, 1.60705445, 1.60902079,
1.611054488, 1.613044376, 1.615021114, 1.616983777, 1.61885745,
1.620842869, 1.622746926, 1.624632209, 1.626535176, 1.628458275,
1.630328774, 1.632184565, 1.634021005, 1.635899739, 1.637695706,
1.639480424, 1.641395616, 1.643147993, 1.644863353, 1.64649654,
1.648374607, 1.650237849, 1.651981305, 1.653735311, 1.655326522,
1.656993011, 1.658736868, 1.66044481, 1.662136961, 1.663750028,
1.665387567, 1.667006042, 1.668519569, 1.670106197, 1.671709125,
1.673312147, 1.674856827, 1.676359202, 1.677903361, 1.679465155,
1.681062879, 1.68255922, 1.68406578, 1.68555801, 1.687069553,
1.688611195, 1.69002579, 1.691372768, 1.692743386, 1.694191476,
1.695505006, 1.696914902, 1.69827621, 1.699591583, 1.701013037,
1.702293608, 1.70359653, 1.704927775, 1.706364857, 1.707714747,
1.709031837, 1.710255286, 1.711508017, 1.712761777, 1.714018815,
1.715400621, 1.716637781, 1.717922262, 1.719298997, 1.719540922,
1.720576895, 1.721715304, 1.722942953, 1.724106776, 1.725237907,
1.726496596, 1.727698521, 1.7288747, 1.730075855, 1.731171954,
1.732359913, 1.733595504, 1.734692913, 1.735845163, 1.736913904,
1.738047183, 1.739280165, 1.740354408, 1.741367023, 1.742436047,
1.743509804, 1.744551185, 1.745543187, 1.746595631, 1.747594421,
1.748609468, 1.749681791, 1.750618255, 1.751629102, 1.752544695,
1.753524936, 1.75451767, 1.755500097, 1.756464255, 1.757366121,
1.758293096, 1.759257851, 1.760265946, 1.761264926, 1.762150343,
1.763008311, 1.763958952, 1.764840492, 1.765795303, 1.766726363,
1.767707596, 1.768633906, 1.76955339, 1.770419974, 1.771292242,
1.772228925, 1.773247856, 1.77415568, 1.77497642, 1.775774977,
1.776604165, 1.777379731, 1.778195687, 1.77902606, 1.779871595,
1.780648021, 1.781429814, 1.782215087, 1.782983082, 1.783681728,
1.784416379, 1.785175287, 1.785817713, 1.786599071, 1.787338466,
1.788117574, 1.788881694, 1.789626285, 1.79039386, 1.791120547,
1.791912498, 1.792708587, 1.793450535, 1.794162874, 1.794810601,
1.795450085, 1.796118656, 1.796954172, 1.797653686, 1.798436176,
1.799059856, 1.799697037, 1.800351596, 1.801085189, 1.801718853,
1.802356178, 1.802963895, 1.803603575, 1.804261684, 1.804799889,
1.80540394, 1.806016296, 1.806600155, 1.807200711, 1.807726208,
1.808372043, 1.808985552, 1.809499636, 1.810038502, 1.810704917,
1.811254886, 1.81178381, 1.812364518, 1.812933526, 1.813499606,
1.814024561, 1.814648184, 1.81520018, 1.815749464, 1.816282284,
1.816801201, 1.817312871, 1.817805165, 1.818287774, 1.818795034,
1.819342973, 1.819859471, 1.820434232, 1.820933134, 1.821395982,
1.821947628, 1.822486132, 1.822964239, 1.823454109, 1.823889531,
1.824388057, 1.8248785, 1.825411436, 1.825835011, 1.826317349,
1.826835641, 1.827241546, 1.827696329, 1.828194134, 1.828679032,
1.829127806, 1.829609048, 1.83006277, 1.830491514, 1.830896874,
1.831341953, 1.831775368, 1.832163007, 1.832642787, 1.833097872,
1.833427218, 1.833873117, 1.834292945, 1.834701003, 1.835151056,
1.835520508, 1.835907731, 1.836305702, 1.836681135, 1.837084873,
1.837470822, 1.837802979, 1.838160314, 1.838519637, 1.838885759,
1.839275651, 1.839628391, 1.839939142, 1.840259407, 1.840581044,
1.841035419, 1.841399516, 1.841733161, 1.842051307, 1.842375513,
1.842768404, 1.843123262, 1.843509211, 1.843877602, 1.8442045,
1.844480364, 1.844788941, 1.845091169, 1.845446918, 1.845693089,
1.845970848, 1.846313861, 1.846599039, 1.846856231, 1.847175874,
1.847463673, 1.847762408, 1.848050504, 1.84828595, 1.848644539,
1.848936831, 1.849200492, 1.849494978, 1.849801546, 1.85006636,
1.850141123, 1.8504614, 1.850731155, 1.851015975, 1.851243705,
1.85148061, 1.851774971, 1.852044565, 1.852327826, 1.852607042,
1.852885351, 1.853091856, 1.853289609, 1.853527104, 1.8538286,
1.854103439, 1.854371464, 1.854616062, 1.85488047, 1.855143988,
1.855432533, 1.855681711, 1.855884277, 1.856143843, 1.8563751,
1.856574352, 1.85681122, 1.857013164, 1.857250581, 1.8574574,
1.85767943, 1.857897539, 1.858167231, 1.858352382, 1.858545867,
1.858754233, 1.858982871, 1.859166649, 1.859429853, 1.859679396,
1.859837348, 1.860120667, 1.860266732, 1.860431598, 1.860627322,
1.860805303, 1.861035974, 1.861239267, 1.861449155, 1.861670078,
1.861938628, 1.862092439, 1.862326917, 1.862571992, 1.862769436,
1.86296568, 1.863144599, 1.863340437, 1.86347717, 1.86363293,
1.86381832, 1.863957687, 1.86410605, 1.864302838, 1.86443252,
1.864543882, 1.86475927, 1.864973623, 1.865091999, 1.865240057,
1.86540875, 1.865599695, 1.865767607, 1.865912093, 1.866049718,
1.866245935, 1.866385573, 1.866519383, 1.866718011, 1.866837397,
1.86703303, 1.867186472, 1.867337111, 1.867512958, 1.867640259,
1.86778192, 1.867927188, 1.868034825, 1.868196141, 1.868331905,
1.868463971, 1.868617631, 1.868781817, 1.868975355, 1.869049271,
1.869224557, 1.869316851, 1.869498397, 1.869616048, 1.869789491,
1.869896711, 1.869994548, 1.870132837, 1.870264283, 1.87034501,
1.87051314, 1.870622215, 1.870708325, 1.870843078, 1.870994734,
1.871081677, 1.871237356, 1.871340748, 1.871443339, 1.8715665,
1.871685192, 1.871798715, 1.871899217, 1.872015052, 1.872096478,
1.872196367, 1.872276176, 1.872417925, 1.872483945, 1.87263454,
1.872749331, 1.872868863, 1.873008899, 1.873130967, 1.873246328,
1.873323064, 1.873453349, 1.873554812, 1.873639397, 1.87372022,
1.873871651, 1.873984101, 1.874079113, 1.874123861, 1.87415654,
1.874275554, 1.874352119, 1.874494968, 1.874585956, 1.874711972,
1.874855628, 1.874904579, 1.875027304, 1.875132526, 1.875194438,
1.875283952, 1.875380336, 1.875480899, 1.875601754, 1.875714944,
1.875827478, 1.875931762, 1.876020099, 1.876047824, 1.876151131,
1.876244849, 1.876330681, 1.876443389, 1.876527957, 1.876580913,
1.876656891, 1.876765498, 1.876881883, 1.876948365, 1.877014579,
1.877125125, 1.877161888, 1.877262602, 1.87734253, 1.877381585,
1.877442293, 1.877512929, 1.877615649, 1.877650794, 1.877724951,
1.877798229, 1.87784967, 1.877975502, 1.878020202, 1.878151172,
1.878221485, 1.878325806, 1.878389224, 1.87845106, 1.878493342,
1.878573721, 1.878629432, 1.878676416, 1.878811944, 1.878892178,
1.878925322, 1.878951753, 1.879045237, 1.879130989, 1.879211442,
1.87926171, 1.8793691, 1.879414956, 1.879462633, 1.879509959,
1.879569367, 1.879688823, 1.879824506, 1.879881912, 1.879946303,
1.880016967, 1.880071337, 1.880138786, 1.880175357, 1.880244357,
1.880306951, 1.880384977, 1.880528931, 1.880558149, 1.880572605,
1.880623114, 1.880662681, 1.880709789, 1.880753822, 1.880804606,
1.880840322, 1.880861816, 1.880913014, 1.880955307, 1.880964439,
1.881046539, 1.881095536, 1.881148717, 1.881205435, 1.881288228,
1.881348345, 1.881400325, 1.881430329, 1.881435006, 1.881476653,
1.881504869, 1.881513387, 1.881565983, 1.881663099, 1.881694868,
1.881736657, 1.881760925, 1.881773007, 1.881819437, 1.88185065,
1.881896147, 1.882025293, 1.882165714, 1.882198667, 1.882237335,
1.882265887, 1.882291944, 1.882321357, 1.882417017, 1.882433398,
1.882467213, 1.882512449, 1.882532624, 1.882552098, 1.882564904,
1.882596359, 1.88273094, 1.882752161, 1.88277869, 1.88281518,
1.882837527, 1.882889281, 1.882899245, 1.88296568, 1.882985475,
1.88302669, 1.883037421, 1.883059205, 1.883122296, 1.883169417,
1.883204626, 1.883272784, 1.883290337, 1.883302512, 1.883333876,
1.88335051, 1.883396561, 1.883426206, 1.883437231, 1.883476368,
1.883537702, 1.883565608, 1.883598869, 1.883671817, 1.88370509,
1.883718256, 1.883772535, 1.883795134, 1.883825164, 1.883862825,
1.883895913, 1.883945456, 1.883955054, 1.883969863, 1.88400399,
1.884007114, 1.884134975, 1.884145097, 1.884177685, 1.884203401,
1.884209634, 1.884225962, 1.884319578, 1.884325646, 1.884345072,
1.884358175, 1.884362145, 1.884382522, 1.884397617, 1.884416804,
1.884428978, 1.884434913, 1.884501154, 1.884548858, 1.884592467,
1.884634491, 1.88465272, 1.884671154, 1.884679435, 1.884717243,
1.884736992, 1.884741987, 1.884762476, 1.884795721, 1.884804257,
1.884813479, 1.884813724, 1.884833916, 1.884877054, 1.884879985,
1.884881567), "6" = c(0.000246485, 0.000246618, 0.000253587,
0.000259073, 0.000259532, 0.000267167, 0.000271747, 0.000271942,
0.000273371, 0.00027391, 0.000285898, 0.00028839, 0.00028888,
0.000289079, 0.00029089, 0.000291376, 0.000294129, 0.000300596,
0.000304032, 0.000305857, 0.00031113, 0.000312364, 0.000321426,
0.000321969, 0.000330719, 0.000331002, 0.000332664, 0.000332988,
0.000333587, 0.00033565, 0.000335927, 0.000340326, 0.000345593,
0.000345869, 0.000352482, 0.000354359, 0.000355907, 0.000363214,
0.000365998, 0.000367313, 0.000368867, 0.000372669, 0.000377003,
0.000382859, 0.00039174, 0.000394547, 0.000396383, 0.000399869,
0.000401283, 0.000404655, 0.000408318, 0.000414827, 0.000425427,
0.000431453, 0.000433697, 0.000437964, 0.000447794, 0.000449523,
0.000450879, 0.000454065, 0.000463426, 0.000468844, 0.000472636,
0.000478415, 0.000483601, 0.000489217, 0.00049154, 0.000493497,
0.000499022, 0.000500826, 0.000504023, 0.000505537, 0.000515972,
0.000518856, 0.000520827, 0.000526402, 0.000532287, 0.000533156,
0.000533812, 0.000537985, 0.000542936, 0.000545089, 0.000548143,
0.000560101, 0.000565961, 0.000573601, 0.000576895, 0.000580853,
0.000587662, 0.000590061, 0.000592558, 0.000596921, 0.000605138,
0.000611325, 0.000617308, 0.000619754, 0.000625917, 0.000636822,
0.00064243, 0.000648457, 0.000657901, 0.000668478, 0.000675889,
0.000680888, 0.000684118, 0.000686691, 0.000694101, 0.000704787,
0.000707929, 0.000716475, 0.000718466, 0.000721052, 0.000724169,
0.00072778, 0.0007332, 0.000734941, 0.00073764, 0.000741973,
0.000748402, 0.00076457, 0.000769571, 0.000782568, 0.000785651,
0.000791682, 0.000801812, 0.000808042, 0.000815136, 0.00082432,
0.000829941, 0.000836325, 0.00084193, 0.000852842, 0.000859333,
0.000864819, 0.000872527, 0.000877605, 0.000887824, 0.000895656,
0.000902313, 0.00091541, 0.000926477, 0.000940668, 0.000945049,
0.000954974, 0.000966432, 0.000972355, 0.000982462, 0.000996018,
0.001007897, 0.001013048, 0.00101758, 0.001024735, 0.001035965,
0.001046897, 0.001052992, 0.001066313, 0.001079129, 0.00108506,
0.001097272, 0.001108179, 0.001113339, 0.001124504, 0.001135263,
0.001141056, 0.001151709, 0.001161492, 0.001167748, 0.001177794,
0.001186519, 0.001189595, 0.00120074, 0.001212528, 0.001223066,
0.00123308, 0.001240056, 0.001251959, 0.001270731, 0.001277845,
0.001287622, 0.001296158, 0.001304658, 0.001321427, 0.001336762,
0.001343543, 0.001358865, 0.001370926, 0.001385303, 0.001400572,
0.001411287, 0.001433104, 0.001439686, 0.001455915, 0.001465133,
0.001482511, 0.001497064, 0.001507568, 0.001531063, 0.001537828,
0.001547516, 0.0015687, 0.001574207, 0.001584822, 0.001598446,
0.001613703, 0.001626805, 0.001643932, 0.001654442, 0.001669392,
0.001683267, 0.001698127, 0.001703123, 0.001715036, 0.00173058,
0.001749374, 0.001764104, 0.001787637, 0.001797907, 0.001817566,
0.001833269, 0.001847406, 0.001860595, 0.001874604, 0.001889999,
0.001904342, 0.001916885, 0.001939481, 0.001959775, 0.001976596,
0.001994139, 0.00200939, 0.002025805, 0.002041327, 0.002067451,
0.002089387, 0.002118721, 0.002140637, 0.002165125, 0.002182357,
0.00219909, 0.0022208, 0.002236674, 0.002253478, 0.002276949,
0.00230146, 0.002321761, 0.002339422, 0.002361516, 0.002393204,
0.002415941, 0.00244053, 0.002471367, 0.002489967, 0.002513338,
0.002546287, 0.002575781, 0.002597431, 0.002614888, 0.002634147,
0.002657245, 0.002677071, 0.002699378, 0.002723024, 0.002743961,
0.002770983, 0.002796584, 0.002818539, 0.002844162, 0.002868809,
0.002894069, 0.00292066, 0.00295473, 0.002987339, 0.003008433,
0.003028393, 0.003062206, 0.003088751, 0.003120064, 0.00314564,
0.003178599, 0.003218367, 0.003250871, 0.003282724, 0.003317722,
0.003346774, 0.003376342, 0.003407405, 0.003436065, 0.003466058,
0.003497027, 0.003524973, 0.003553681, 0.003590081, 0.003622855,
0.003655904, 0.003687171, 0.003718105, 0.003751636, 0.003780937,
0.003810347, 0.00384306, 0.003885464, 0.003925998, 0.003955115,
0.004001707, 0.004042745, 0.004075462, 0.0041286, 0.004161455,
0.004193849, 0.004224241, 0.004257358, 0.004290674, 0.004324179,
0.004368192, 0.004400867, 0.004435479, 0.00446958, 0.004510619,
0.004546774, 0.004577248, 0.004631653, 0.00466743, 0.004706942,
0.004764024, 0.004806533, 0.004835349, 0.004878557, 0.004916243,
0.004961585, 0.005007497, 0.005049152, 0.005094455, 0.00514834,
0.005199773, 0.005246314, 0.00528905, 0.005323536, 0.005374779,
0.005417338, 0.005476337, 0.005529093, 0.005541984, 0.005598167,
0.005642871, 0.005685682, 0.005724807, 0.005773339, 0.005824333,
0.005870692, 0.005910565, 0.005960266, 0.006019937, 0.006065174,
0.006113726, 0.006163506, 0.006206382, 0.006259026, 0.006316541,
0.006366299, 0.006420061, 0.006485454, 0.006559526, 0.006622545,
0.006682983, 0.006738329, 0.006814324, 0.006872915, 0.006933679,
0.006993052, 0.007058763, 0.007117375, 0.007187377, 0.007249843,
0.007321346, 0.007388371, 0.007453495, 0.00752541, 0.007589976,
0.007649719, 0.007729566, 0.007798538, 0.007874343, 0.00796197,
0.008033242, 0.008112115, 0.00817697, 0.008245501, 0.008316481,
0.008388391, 0.008456237, 0.008531967, 0.00859712, 0.008670801,
0.008751363, 0.008826103, 0.008909743, 0.008978521, 0.00905267,
0.009132298, 0.009210731, 0.009294423, 0.00937523, 0.00946327,
0.009548717, 0.00962587, 0.009708898, 0.009795012, 0.009870488,
0.009962751, 0.010050878, 0.010138563, 0.01021749, 0.010319616,
0.010411236, 0.010504919, 0.010600444, 0.010704447, 0.010791676,
0.010887163, 0.010980673, 0.011056466, 0.011149319, 0.01124667,
0.011351443, 0.011465882, 0.011577222, 0.011678878, 0.011772277,
0.011865739, 0.011986301, 0.012099775, 0.012202841, 0.01231285,
0.012429286, 0.012526831, 0.012639657, 0.012767042, 0.012885083,
0.013004042, 0.013121889, 0.013239984, 0.013348669, 0.013477585,
0.013598518, 0.013705249, 0.013824098, 0.013930125, 0.014048253,
0.014172707, 0.014304233, 0.014437267, 0.014568881, 0.014696099,
0.014822167, 0.014949302, 0.015095329, 0.015252369, 0.015386025,
0.015514958, 0.015652726, 0.015785441, 0.015934635, 0.016078068,
0.016211018, 0.016357346, 0.016510227, 0.016644534, 0.016798233,
0.016941769, 0.017101279, 0.017242105, 0.017397199, 0.01754059,
0.017714358, 0.017862421, 0.018034207, 0.018190656, 0.018354121,
0.018504397, 0.018662404, 0.018824565, 0.018977986, 0.019145639,
0.01932605, 0.019495535, 0.019666716, 0.019847258, 0.020031081,
0.020213571, 0.02039529, 0.020594578, 0.020773302, 0.020955742,
0.021124368, 0.021298347, 0.0214905, 0.021669721, 0.021862811,
0.022057032, 0.02226587, 0.022452273, 0.022656425, 0.022853017,
0.02306349, 0.023272089, 0.023483389, 0.023687817, 0.023894904,
0.024104811, 0.024281779, 0.024503637, 0.024721041, 0.024940641,
0.025167977, 0.02539707, 0.025637431, 0.025839939, 0.026066328,
0.026297204, 0.026534596, 0.026753468, 0.026969672, 0.027202202,
0.027442615, 0.027671905, 0.02790549, 0.028144237, 0.028399588,
0.028635174, 0.028879818, 0.029115736, 0.029362533, 0.029604482,
0.02985825, 0.030119744, 0.030371937, 0.030637129, 0.030900775,
0.031161184, 0.031425086, 0.031719194, 0.031930839, 0.031990283,
0.032288296, 0.032576889, 0.032846779, 0.03313892, 0.03341411,
0.033702927, 0.033978971, 0.0342843, 0.034574429, 0.034879193,
0.035161223, 0.035459844, 0.035747683, 0.036051533, 0.036359287,
0.036673165, 0.037002904, 0.03732728, 0.037656751, 0.038000172,
0.038321459, 0.038657221, 0.038975558, 0.039310193, 0.039643901,
0.040008388, 0.040347021, 0.040694385, 0.041034026, 0.041368597,
0.041718161, 0.042065413, 0.042422495, 0.042780302, 0.043144025,
0.04351107, 0.043899034, 0.044286008, 0.04468936, 0.045046493,
0.045429625, 0.04580407, 0.046196428, 0.046607983, 0.047009877,
0.047389878, 0.047780151, 0.048173407, 0.048596042, 0.048992478,
0.049445153, 0.049835888, 0.050235357, 0.050654457, 0.051093823,
0.051513066, 0.051954032, 0.05236864, 0.052795948, 0.053244296,
0.053700017, 0.054145201, 0.054589731, 0.055056418, 0.055514079,
0.055960662, 0.056424808, 0.056893481, 0.057365678, 0.057852164,
0.058325685, 0.058807887, 0.059261155, 0.059735377, 0.060240082,
0.060736415, 0.061224427, 0.061716891, 0.062227152, 0.062731164,
0.06324479, 0.063775158, 0.064305797, 0.064831251, 0.065371129,
0.065898878, 0.066423806, 0.066980024, 0.0675527, 0.068106688,
0.068712559, 0.069263362, 0.069804002, 0.070390413, 0.070986865,
0.071566338, 0.07216585, 0.072753836, 0.073364248, 0.073942902,
0.074562589, 0.075163804, 0.075770337, 0.076397274, 0.077017575,
0.077632396, 0.078235232, 0.078878657, 0.079497706, 0.080129892,
0.080769897, 0.081443266, 0.082098723, 0.082760907, 0.083419272,
0.084083935, 0.084779539, 0.085462831, 0.08618669, 0.086877244,
0.087568921, 0.088282426, 0.088999638, 0.089735005, 0.090473847,
0.091172681, 0.091931985, 0.092686606, 0.09342377, 0.09418967,
0.094952056, 0.095702426, 0.096472329, 0.097228548, 0.097982703,
0.098784307, 0.099562915, 0.100373278, 0.101162267, 0.101969366,
0.102765034, 0.103576577, 0.104399439, 0.105220722, 0.106026751,
0.1068887, 0.107735984, 0.108570582, 0.109447477, 0.110321891,
0.111194531, 0.112076163, 0.112966118, 0.113903338, 0.114813885,
0.115704952, 0.11659069, 0.117487094, 0.118405129, 0.11934409,
0.120283018, 0.121232628, 0.122177352, 0.123134424, 0.124078798,
0.125021484, 0.125970493, 0.126945053, 0.127899122, 0.128862533,
0.129858012, 0.130845589, 0.131820512, 0.132829465, 0.133848851,
0.134858075, 0.135893496, 0.136936106, 0.137978716, 0.139036256,
0.140088614, 0.1411303, 0.142185485, 0.143253991, 0.14433436,
0.145408738, 0.146480772, 0.147550095, 0.148661487, 0.149754632,
0.150854653, 0.151941654, 0.153083608, 0.154235075, 0.155368298,
0.156513889, 0.157663611, 0.158824578, 0.160001827, 0.160939311,
0.161181332, 0.162371293, 0.163551019, 0.164740708, 0.165949784,
0.167184126, 0.168363469, 0.169578991, 0.170809656, 0.172066527,
0.173300365, 0.174565509, 0.175816484, 0.177092902, 0.178388876,
0.179667137, 0.180938157, 0.182237872, 0.183528609, 0.184824968,
0.18609945, 0.187426281, 0.188766067, 0.19009978, 0.191433625,
0.192757563, 0.194089233, 0.195441687, 0.196775728, 0.198174218,
0.199556166, 0.200954458, 0.202342584, 0.203757183, 0.205149424,
0.206569262, 0.20796733, 0.20942396, 0.210862975, 0.212294992,
0.213724176, 0.2151523, 0.216617542, 0.2180836, 0.219558011,
0.221031132, 0.222500939, 0.223970653, 0.225472997, 0.226960793,
0.228434924, 0.229932951, 0.231410403, 0.232937581, 0.23441535,
0.23590675, 0.237442128, 0.238980361, 0.240501406, 0.242023056,
0.243573647, 0.245120579, 0.246656201, 0.248189204, 0.249736215,
0.251244117, 0.252768325, 0.254304515, 0.255824225, 0.257335218,
0.258880753, 0.260394098, 0.261938109, 0.263472, 0.265011515,
0.266551645, 0.268090393, 0.269655008, 0.271231912, 0.272806906,
0.274386916, 0.275961174, 0.277534874, 0.279138279, 0.280744703,
0.282344631, 0.283981926, 0.285637223, 0.287272342, 0.288940711,
0.290610709, 0.292292405, 0.293978784, 0.295682927, 0.297403313,
0.29911688, 0.299495273, 0.300840673, 0.302596905, 0.304339326,
0.306116666, 0.307897305, 0.309691984, 0.311518602, 0.313345364,
0.315185143, 0.317037183, 0.318889338, 0.320762703, 0.322651974,
0.324554982, 0.326463497, 0.328411381, 0.330344571, 0.332291897,
0.334250722, 0.33624776, 0.338241949, 0.340267382, 0.342313487,
0.344364995, 0.346405896, 0.346830594, 0.348495625, 0.350597168,
0.35270044, 0.354808546, 0.356927537, 0.359064983, 0.361248812,
0.363424827, 0.365609751, 0.367847791, 0.370065543, 0.372325169,
0.374598783, 0.376854706, 0.379157423, 0.381484855, 0.383811505,
0.386167815, 0.388525397, 0.390897001, 0.393309073, 0.395764055,
0.398228015, 0.400692694, 0.403164806, 0.405659139, 0.408185324,
0.410719813, 0.413277327, 0.415840449, 0.418455012, 0.421054878,
0.423685454, 0.426318738, 0.42898903, 0.430128892, 0.431689271,
0.434426187, 0.437163748, 0.439936488, 0.442731371, 0.445533371,
0.448354203, 0.451202122, 0.454069666, 0.456963785, 0.459906643,
0.462857778, 0.465824457, 0.468788869, 0.471785069, 0.474824375,
0.477858796, 0.480939197, 0.484039522, 0.487169492, 0.490313769,
0.493512722, 0.496721116, 0.499955775, 0.503225591, 0.504572431,
0.506501575, 0.509790148, 0.513113252, 0.516464448, 0.519844262,
0.523227971, 0.526629395, 0.530071305, 0.533558465, 0.537067353,
0.54058614, 0.544136728, 0.547728347, 0.551332541, 0.554979224,
0.558662864, 0.562357685, 0.566061298, 0.569846543, 0.572901344,
0.579727446, 0.583486893, 0.587157053, 0.590859926, 0.594491372,
0.598098154, 0.601637009, 0.605194222, 0.608718536, 0.612260202,
0.615784417, 0.619271219, 0.62272512, 0.62617206, 0.629628725,
0.63306172, 0.636475966, 0.639873819, 0.64187507, 0.643253385,
0.646598697, 0.649940787, 0.653263525, 0.656594778, 0.659915513,
0.663179766, 0.666502633, 0.669776396, 0.673032578, 0.676287381,
0.679513645, 0.682747923, 0.685955345, 0.689171514, 0.692374964,
0.695543528, 0.698732069, 0.701869318, 0.70501445, 0.708162965,
0.711274515, 0.71441099, 0.717506195, 0.720597108, 0.722362952,
0.723671084, 0.726737049, 0.729819492, 0.732860956, 0.735908533,
0.738963627, 0.741971446, 0.74498608, 0.748035879, 0.751073996,
0.754070386, 0.757059661, 0.760062072, 0.763046523, 0.766045306,
0.769055103, 0.771998857, 0.774989638, 0.777984287, 0.780971076,
0.783933592, 0.78693287, 0.789936518, 0.792921594, 0.795929339,
0.798951753, 0.801984288, 0.805019304, 0.808060388, 0.811180883,
0.814322933, 0.817497395, 0.820739316, 0.82397795, 0.827250215,
0.829877239, 0.830541885, 0.833867052, 0.837212534, 0.840617011,
0.843981949, 0.84737906, 0.850771122, 0.854219202, 0.857686177,
0.861166241, 0.864645635, 0.868211535, 0.87175644, 0.875326931,
0.878916281, 0.882509251, 0.886158877, 0.889830564, 0.893524845,
0.897218447, 0.900988798, 0.904728778, 0.908559583, 0.912378762,
0.91619454, 0.919182072, 0.92002264, 0.923945844, 0.927830849,
0.931752202, 0.935733904, 0.939722034, 0.943785832, 0.947925582,
0.952032871, 0.956235337, 0.960440997, 0.964713269, 0.969013021,
0.973372937, 0.977886421, 0.982436118, 0.987050311, 0.99171158,
0.996460095, 1.001241347, 1.006138131, 1.011226555, 1.016379611,
1.02163531, 1.027154645, 1.032884829, 1.038777747, 1.044907548,
1.051157528, 1.057747208, 1.064560746, 1.071435195, 1.078422169,
1.085496628, 1.092511796, 1.099597052, 1.106674673, 1.113892821,
1.121172612, 1.128420911, 1.135661698, 1.142898981, 1.150194156,
1.157451578, 1.164861607, 1.172116263, 1.179441966, 1.186936202,
1.194361503, 1.201789969, 1.209227704, 1.216656207, 1.224215683,
1.231667682, 1.239221396, 1.246698425, 1.254298421, 1.261787296,
1.269314132, 1.27671841, 1.284313532, 1.291864721, 1.299502675,
1.307025262, 1.314687112, 1.322201909, 1.329786664, 1.337394558,
1.345093327, 1.352841619, 1.360741237, 1.36853041, 1.376261873,
1.38389413, 1.391496689, 1.399129572, 1.40696014, 1.414708006,
1.42236544, 1.430096232, 1.437805462, 1.445545173, 1.453116092,
1.460674564, 1.468351205, 1.475946873, 1.483746698, 1.491522797,
1.499275547, 1.506944937, 1.514608044, 1.522205958, 1.529816998,
1.537503361, 1.545358609, 1.553031667, 1.554590054, 1.560639709,
1.568319972, 1.57603782, 1.583602347, 1.591142001, 1.598660037,
1.606199805, 1.613660901, 1.621251576, 1.628818948, 1.636258713,
1.643852732, 1.651544065, 1.65913225, 1.666680614, 1.674266415,
1.681905687, 1.689311328, 1.696902145, 1.704479291, 1.711836351,
1.719214615, 1.726586109, 1.734180428, 1.74162336, 1.749095306,
1.756620811, 1.763960035, 1.771329877, 1.778677365, 1.786159575,
1.793696649, 1.800858235, 1.808355585, 1.815676972, 1.822865331,
1.830123448, 1.837445185, 1.844769515, 1.851910714, 1.858921824,
1.86633693, 1.873476699, 1.880688952, 1.887678009, 1.894961537,
1.902155416, 1.9094204, 1.916554675, 1.923744931, 1.930842003,
1.937617243, 1.9445652, 1.951359257, 1.958427503, 1.965511132,
1.972386327, 1.979503795, 1.986653102, 1.993334431, 2.000354034,
2.007217762, 2.014133349, 2.021037813, 2.028029138, 2.034910074,
2.041658594, 2.0484236, 2.0551091, 2.061836865, 2.068453079,
2.075119735, 2.081713153, 2.088263612, 2.094843644, 2.101459576,
2.108202105, 2.114518074, 2.12111356, 2.127582795, 2.134053836,
2.140606807, 2.147061437, 2.1533559, 2.159650676, 2.166170118,
2.172547049, 2.179008197, 2.185475138, 2.191929842, 2.197896921,
2.204302226, 2.210336528, 2.216807173, 2.222892122, 2.229053624,
2.235151812, 2.241497319, 2.247539279, 2.253867524, 2.259937656,
2.265966753, 2.272032391, 2.278119029, 2.284177605, 2.290227785,
2.296125507, 2.302158162, 2.308120163, 2.314221456, 2.319994993,
2.325652575, 2.331750361, 2.337618067, 2.343299919, 2.348982322,
2.354754909, 2.360601636, 2.366506015, 2.372135085, 2.377476136,
2.383167142, 2.388781449, 2.394506849, 2.399963456, 2.405599009,
2.411201981, 2.416516667, 2.421969218, 2.427343379, 2.432828062,
2.438372167, 2.443839297, 2.449433857, 2.454737441, 2.459890993,
2.465284905, 2.470859237, 2.476282717, 2.48161123, 2.486910363,
2.492327341, 2.497513024, 2.502799682, 2.508219197, 2.513747793,
2.518902561, 2.523972199, 2.528853273, 2.533974288, 2.539146839,
2.544183082, 2.54908999, 2.553847318, 2.558811725, 2.563747331,
2.568768069, 2.573991901, 2.579176103, 2.584048872, 2.588897609,
2.593554903, 2.598150689, 2.602987673, 2.607902031, 2.612455565,
2.617399437, 2.622265412, 2.626914673, 2.631269002, 2.636060569,
2.640563209, 2.64523817, 2.649670012, 2.654292163, 2.659141407,
2.66378771, 2.668396452, 2.672997922, 2.677561738, 2.681897376,
2.686170403, 2.690472585, 2.694873192, 2.699595784, 2.704011652,
2.708409227, 2.712615778, 2.717278386, 2.721581397, 2.72568767,
2.729910178, 2.734005913, 2.738317653, 2.742313622, 2.746740379,
2.75090508, 2.755012955, 2.759030681, 2.762889471, 2.763609037,
2.766722182, 2.770774736, 2.775128288, 2.779053571, 2.783003082,
2.787079994, 2.791288762, 2.795367673, 2.799162897, 2.803125686,
2.80716279, 2.810994888, 2.815227341, 2.819181767, 2.823154699,
2.826984812, 2.830815091, 2.834446644, 2.838333687, 2.842103289,
2.845778073, 2.849611452, 2.853168252, 2.856664422, 2.860409848,
2.864213555, 2.867827334, 2.871710009, 2.875517019, 2.878983543,
2.882634443, 2.886202359, 2.88979874, 2.893306939, 2.896661592,
2.900175004, 2.903675467, 2.907130098, 2.910665622, 2.913966097,
2.917545904, 2.920967245, 2.924227493, 2.927530933, 2.931169898,
2.934187053, 2.93741082, 2.940751557, 2.943925809, 2.947074024,
2.95042207, 2.953453575, 2.956869782, 2.960188365, 2.963416107,
2.966788105, 2.96972458, 2.972558069, 2.975783309, 2.978722078,
2.981465692, 2.984528872, 2.987487628, 2.990569723, 2.993597245,
2.996472879, 2.999412667, 3.002256589, 3.005253483, 3.008234406,
3.010652024, 3.013534932, 3.016407552, 3.019664747, 3.022618291,
3.025532499, 3.028486643, 3.03135115, 3.034081558, 3.036699048,
3.040037365, 3.042814282, 3.045520809, 3.048102556, 3.051069039,
3.053658286, 3.05639731, 3.059264896, 3.0627063, 3.065402949,
3.067952821, 3.070682825, 3.07318895, 3.075859704, 3.078058032,
3.080774016, 3.083298099, 3.085922299, 3.08846848, 3.091268365,
3.094047362, 3.096467642, 3.099029936, 3.101614549, 3.104050342,
3.10663343, 3.108804939, 3.110981557, 3.113043106, 3.115304448,
3.117703984, 3.120096435, 3.122581149, 3.12486342, 3.127429403,
3.130191939, 3.13250667, 3.134927137, 3.13769589, 3.140457595,
3.14256888, 3.144529296, 3.147033607, 3.149106084, 3.151477651,
3.153581604, 3.155711547, 3.157738574, 3.160027442, 3.162232799,
3.164372806, 3.166488448, 3.169063894, 3.171632415, 3.173568211,
3.175830889, 3.177797057, 3.179614591, 3.181716734, 3.183769393,
3.185603172, 3.187435721, 3.189668843, 3.191488977, 3.193343536,
3.195379499, 3.197291571, 3.199692728, 3.201645009, 3.203295055,
3.205024714, 3.206706842, 3.208372779, 3.210354644, 3.212115627,
3.213923556, 3.21574783, 3.217808955, 3.219691555, 3.221465124,
3.22325564, 3.225067188, 3.226682332, 3.228764392, 3.230599919,
3.232752167, 3.23435939, 3.236300792, 3.238293365, 3.239945527,
3.241866887, 3.243595024, 3.245161882, 3.246549235, 3.248129485,
3.249845686, 3.251707384, 3.253623619, 3.254869415, 3.256417497,
3.258290733, 3.260065926, 3.261846396, 3.263556213, 3.265351052,
3.267140614, 3.268811983, 3.27018966, 3.271773684, 3.273233559,
3.275093701, 3.276767089, 3.278460949, 3.279697671, 3.281341028,
3.282803181, 3.284492289, 3.286402964, 3.287837053, 3.289580551,
3.289824571, 3.291201421, 3.292516019, 3.293780773, 3.295183207,
3.296904513, 3.298198254, 3.299511511, 3.300903705, 3.302711919,
3.30371588, 3.305039158, 3.306524243, 3.307604644, 3.309023536,
3.310608474, 3.312084522, 3.313488349, 3.314747143, 3.316026124,
3.317446162, 3.318616462, 3.319968387, 3.321177722, 3.322608603,
3.324018223, 3.325271473, 3.326268641, 3.327758158, 3.329134318,
3.330363987, 3.331345298, 3.332784851, 3.333767478, 3.335265304,
3.336440292, 3.337631353, 3.338998084, 3.339792137, 3.340999285,
3.342161667, 3.343396601, 3.344376789, 3.345736506, 3.346918231,
3.348272899, 3.34942441, 3.350526647, 3.352277932, 3.353791224,
3.354661003, 3.355697253, 3.356941477, 3.358455636, 3.359657659,
3.361050197, 3.36209321, 3.362775282, 3.364127548, 3.365261329,
3.366184889, 3.367079031, 3.368621855, 3.369770279, 3.370643724,
3.371801885, 3.373104775, 3.374590683, 3.375607452, 3.37661392,
3.377668368, 3.378597515, 3.379728412, 3.381074915, 3.382147753,
3.38329536, 3.384127007, 3.385112345, 3.38597737, 3.387107612,
3.388139154, 3.389064678, 3.390012423, 3.391041267, 3.391761332,
3.392562652, 3.393380495, 3.393977783, 3.395182948, 3.396020965,
3.397473156, 3.39804453, 3.399056571, 3.400117117, 3.401262851,
3.402035062, 3.402870994, 3.403325662, 3.404020163, 3.405031916,
3.405965376, 3.40657156, 3.407435804, 3.408228997, 3.409488735,
3.410454034, 3.411358508, 3.411896521, 3.412390726, 3.413530614,
3.414302395, 3.415106648, 3.415956404, 3.416787351, 3.417458693,
3.418094132, 3.418771128, 3.419465213, 3.420400613, 3.421076607,
3.421893229, 3.422176933, 3.423367252, 3.423917197, 3.424977414,
3.426180083, 3.426863582, 3.42768839, 3.428442145, 3.42913618,
3.429960334, 3.430382753, 3.431084074, 3.431441572, 3.432008014,
3.432757914, 3.433289567, 3.43400751, 3.434698921, 3.435231257,
3.435950702, 3.436378817, 3.436852587, 3.437544923, 3.43806602,
3.438744096, 3.439499486, 3.440248073, 3.440699826, 3.441084483,
3.441374402, 3.441870704, 3.442519813, 3.443035425, 3.443489049,
3.444229413, 3.444978974, 3.445578867, 3.446427517, 3.447063638,
3.44765875, 3.448206501, 3.448648217, 3.449121877, 3.44969943,
3.450325629, 3.450870557, 3.451423046, 3.451924058, 3.452466301,
3.452947579, 3.453437109, 3.453726011, 3.454320969, 3.454873386,
3.455807072, 3.456519443, 3.457032548, 3.457550901, 3.458241934,
3.458781487, 3.459170562, 3.459690588, 3.460096318, 3.460443907,
3.461028863, 3.461262514, 3.461945358, 3.462591816, 3.463313849,
3.463633193, 3.464111897, 3.464698768, 3.464985326, 3.465667621,
3.466114259, 3.466463464, 3.466679304, 3.467168858, 3.467531219,
3.467944251, 3.468160623, 3.468787091, 3.469554673, 3.470055153,
3.470463612, 3.471065236, 3.471700489, 3.472177884, 3.472420049,
3.472912304, 3.473377561, 3.47404502, 3.474726554, 3.475280201,
3.475592138, 3.476171267, 3.476565039, 3.477189822, 3.477821988,
3.478350414, 3.478805709, 3.479433079, 3.479852743, 3.480231196,
3.480698787, 3.481437487, 3.481880847, 3.481987965, 3.482588669,
3.482921349, 3.483829706, 3.484563277, 3.484645735, 3.485186279,
3.485734328, 3.486039558, 3.486283273, 3.48657518, 3.487129953,
3.487712441, 3.487944258, 3.488297078, 3.48884142, 3.489262719,
3.489451232, 3.490634491, 3.491493559, 3.491660782, 3.492020947,
3.492284671, 3.492851636, 3.493694634, 3.493783841, 3.493956654,
3.494435777, 3.494805779, 3.494899121, 3.495066503, 3.495936109,
3.496310048, 3.496796429, 3.497333088, 3.497809248, 3.49808808,
3.498180502, 3.49894318, 3.49898516, 3.499512312, 3.500047084,
3.500400626, 3.500634298, 3.500698624, 3.500986985, 3.501120749,
3.501364002, 3.50155378, 3.501789961, 3.501940435, 3.502169827,
3.502205169, 3.502765235, 3.502884532, 3.503065053, 3.503250681,
3.503625377, 3.50402256, 3.504606819, 3.504896275, 3.504918342,
3.505050988, 3.505416526, 3.505578757, 3.505780192, 3.506665255,
3.507205941, 3.507541167, 3.507623545, 3.507826336, 3.50831775,
3.508401156, 3.50840205, 3.50869149, 3.508965164, 3.509482273,
3.509650182, 3.510016516, 3.510273843, 3.510625921, 3.511109543,
3.511303434, 3.511347563, 3.511541381, 3.511785082, 3.512052225,
3.512246229, 3.512307335, 3.512888341, 3.513080935, 3.513148602,
3.513216063, 3.513582066, 3.51361344, 3.513674071, 3.514281942,
3.514525287, 3.514759158, 3.514857549, 3.514862308, 3.515173569,
3.515452764, 3.515565793, 3.515932833, 3.516320975, 3.516342971,
3.516560009, 3.516700529, 3.516720592, 3.517565526, 3.517832935,
3.518028782), "8" = c(4.0113e-05, 4.0127e-05, 4.033e-05, 4.0845e-05,
4.0984e-05, 4.1605e-05, 4.1992e-05, 4.2031e-05, 4.295e-05, 4.2979e-05,
4.3371e-05, 4.3794e-05, 4.4847e-05, 4.4915e-05, 4.499e-05, 4.5466e-05,
4.5666e-05, 4.6604e-05, 4.6707e-05, 4.7212e-05, 4.7535e-05, 4.7783e-05,
4.832e-05, 4.9849e-05, 5.0728e-05, 5.1081e-05, 5.2244e-05, 5.2324e-05,
5.3228e-05, 5.5268e-05, 5.5944e-05, 5.7022e-05, 5.8298e-05, 5.8345e-05,
5.9821e-05, 6.0144e-05, 6.0195e-05, 6.2007e-05, 6.2588e-05, 6.3681e-05,
6.3934e-05, 6.4839e-05, 6.5427e-05, 6.6905e-05, 6.788e-05, 6.9098e-05,
7.0569e-05, 7.0994e-05, 7.1343e-05, 7.268e-05, 7.3113e-05, 7.4354e-05,
7.5255e-05, 7.703e-05, 7.8031e-05, 7.9592e-05, 8.0085e-05, 8.0323e-05,
8.0975e-05, 8.3584e-05, 8.6302e-05, 8.8593e-05, 8.9164e-05, 9.0297e-05,
9.0748e-05, 9.173e-05, 9.3127e-05, 9.3561e-05, 9.4417e-05, 9.4905e-05,
9.6603e-05, 9.6805e-05, 9.7281e-05, 9.867e-05, 9.9086e-05, 9.9919e-05,
0.000100669, 0.000103024, 0.000103407, 0.000106, 0.000106667,
0.00010875, 0.00011117, 0.00011181, 0.000113051, 0.000114299,
0.000116402, 0.000118703, 0.000119564, 0.000122153, 0.000123566,
0.000126762, 0.000127571, 0.000129913, 0.000130302, 0.000131606,
0.000134256, 0.000135652, 0.000138673, 0.000140027, 0.000140244,
0.000141649, 0.000143069, 0.000144494, 0.000145066, 0.000145933,
0.000149336, 0.000151204, 0.000152106, 0.000154245, 0.000155987,
0.00015845, 0.000160081, 0.000162205, 0.000164821, 0.000165375,
0.000168886, 0.000171231, 0.000171808, 0.000172904, 0.000174872,
0.000177601, 0.000177952, 0.000180157, 0.000183334, 0.000185051,
0.000186748, 0.00018923, 0.000191922, 0.000192458, 0.000193666,
0.000194983, 0.00019702, 0.000200122, 0.000201385, 0.000203045,
0.000205098, 0.000207155, 0.000207953, 0.000210664, 0.000212518,
0.00021531, 0.000220107, 0.000222621, 0.000224265, 0.000225471,
0.000228673, 0.000229816, 0.000231477, 0.000234395, 0.000236399,
0.000238493, 0.000240272, 0.000243626, 0.000245832, 0.000248392,
0.000250731, 0.000253111, 0.000258134, 0.000261501, 0.000265408,
0.00026785, 0.000268816, 0.000271244, 0.000274927, 0.000277261,
0.000281004, 0.000283287, 0.000285963, 0.000288796, 0.000290863,
0.000295051, 0.000298186, 0.000301069, 0.000303248, 0.000306571,
0.000311192, 0.000314516, 0.000319015, 0.000321616, 0.000326769,
0.000329484, 0.000334272, 0.000337311, 0.000340297, 0.000343193,
0.000346429, 0.000351409, 0.0003564, 0.000363766, 0.000369437,
0.000376177, 0.000378448, 0.000381659, 0.000387327, 0.000391594,
0.000397114, 0.000400063, 0.000404807, 0.000410513, 0.000415242,
0.000422893, 0.000426624, 0.000430182, 0.000433672, 0.000439068,
0.000444206, 0.000447612, 0.00045102, 0.000456954, 0.000464583,
0.000471205, 0.000475862, 0.000481245, 0.000487205, 0.000494627,
0.000498919, 0.000503656, 0.000509194, 0.000515272, 0.000518081,
0.000523371, 0.000529576, 0.00053519, 0.000540811, 0.000545492,
0.000549632, 0.000555306, 0.000559738, 0.000565244, 0.000571749,
0.0005807, 0.000588242, 0.000599217, 0.000604127, 0.000611372,
0.00061804, 0.000628337, 0.00063307, 0.000639875, 0.000646157,
0.000655355, 0.000662592, 0.000668499, 0.000677019, 0.000681711,
0.00068956, 0.000695076, 0.000701492, 0.000710776, 0.000719169,
0.000723976, 0.000737912, 0.000749451, 0.000756599, 0.000765768,
0.000774647, 0.000784868, 0.000795009, 0.000804585, 0.000808807,
0.000819503, 0.000831773, 0.000839706, 0.000849455, 0.000861986,
0.000874413, 0.000884933, 0.000896151, 0.000906718, 0.000918016,
0.000928765, 0.000937372, 0.000944815, 0.000958586, 0.000969842,
0.000978446, 0.000991095, 0.001000563, 0.001009215, 0.001021071,
0.001035306, 0.001049018, 0.001061916, 0.001071104, 0.001084741,
0.001096268, 0.001111301, 0.001126362, 0.001137305, 0.001154894,
0.00116982, 0.001183228, 0.001198235, 0.001211047, 0.001225496,
0.001236371, 0.001251156, 0.001267817, 0.001285788, 0.001297004,
0.001313018, 0.001324379, 0.001338485, 0.001354135, 0.00136864,
0.0013848, 0.001399952, 0.001416367, 0.001435662, 0.001455764,
0.001473214, 0.001491908, 0.001510663, 0.001527533, 0.00154732,
0.001566909, 0.001586582, 0.001602017, 0.001616482, 0.001632141,
0.00164848, 0.00166434, 0.001689284, 0.001705477, 0.001718268,
0.001740629, 0.001755747, 0.001774348, 0.001790627, 0.00181156,
0.001829344, 0.001847576, 0.001868897, 0.001888759, 0.001910985,
0.001931256, 0.001952463, 0.001975452, 0.002001879, 0.002015152,
0.002019863, 0.002041689, 0.002065844, 0.002084108, 0.002103858,
0.002126843, 0.002147346, 0.002176134, 0.002207653, 0.002232437,
0.002262483, 0.002289486, 0.00231186, 0.002337795, 0.002362483,
0.002389332, 0.002426042, 0.002454606, 0.002482257, 0.002516172,
0.002547913, 0.002579081, 0.002604944, 0.0026356, 0.002670415,
0.002705408, 0.002734808, 0.002769112, 0.002802222, 0.00283155,
0.002858912, 0.002892193, 0.002918508, 0.00295696, 0.002991961,
0.003027919, 0.003057803, 0.003090697, 0.003127972, 0.003168696,
0.00320393, 0.003241954, 0.003279321, 0.003318442, 0.003356433,
0.003397385, 0.003437414, 0.003475622, 0.00351304, 0.003554539,
0.00359865, 0.003637147, 0.003676329, 0.003715017, 0.00375661,
0.003796236, 0.003844229, 0.003887101, 0.003928171, 0.00397561,
0.004023404, 0.004077748, 0.00413487, 0.004188968, 0.004246231,
0.004295759, 0.004342226, 0.004399193, 0.004448081, 0.004500881,
0.004549889, 0.004602567, 0.004662654, 0.004711663, 0.00476624,
0.004821099, 0.004876194, 0.004936241, 0.004996037, 0.005044459,
0.005106161, 0.005162731, 0.005217654, 0.005277418, 0.005334629,
0.005401948, 0.005463128, 0.005528598, 0.005596427, 0.005656876,
0.005729842, 0.005799221, 0.005863886, 0.005926529, 0.005983039,
0.006060199, 0.00613807, 0.006198629, 0.006262984, 0.006332134,
0.006405018, 0.006476121, 0.006552929, 0.006623015, 0.006705432,
0.006773875, 0.006845519, 0.006931178, 0.007013247, 0.007093266,
0.007179192, 0.00725718, 0.007351154, 0.007433705, 0.00751669,
0.007601691, 0.007688976, 0.007768893, 0.007860816, 0.007948041,
0.008038364, 0.008136249, 0.008231872, 0.008321528, 0.008411825,
0.008504824, 0.008593997, 0.008692825, 0.008797846, 0.008903259,
0.008996366, 0.009088201, 0.009200598, 0.009297868, 0.00940178,
0.009501122, 0.00961138, 0.00971612, 0.009827313, 0.009934467,
0.010021319, 0.010137622, 0.01024426, 0.01036051, 0.010476943,
0.010597068, 0.010717908, 0.010837417, 0.010958584, 0.011069569,
0.01119599, 0.01132146, 0.011448654, 0.011563187, 0.011689834,
0.011825385, 0.011952204, 0.012085688, 0.012226031, 0.012365944,
0.012514472, 0.012643689, 0.012791256, 0.012928558, 0.013081602,
0.013218705, 0.013378792, 0.013525152, 0.01368147, 0.013831806,
0.013991312, 0.014151555, 0.014311768, 0.014472783, 0.014631493,
0.014784498, 0.014958869, 0.015125887, 0.015286735, 0.015458222,
0.015637594, 0.015814127, 0.016010783, 0.016191586, 0.016382128,
0.01656162, 0.016743036, 0.016919233, 0.01710031, 0.017282261,
0.01747637, 0.017668133, 0.01787021, 0.018066962, 0.018264813,
0.018470668, 0.018680172, 0.018888479, 0.019092944, 0.019301574,
0.019465302, 0.019513077, 0.019731886, 0.019951208, 0.020164639,
0.020381497, 0.020590124, 0.020805183, 0.021041981, 0.02127306,
0.021501974, 0.021730666, 0.021965947, 0.022207238, 0.022446249,
0.02269493, 0.022948805, 0.023200088, 0.023434263, 0.023686974,
0.023932222, 0.02420679, 0.024490934, 0.02476461, 0.025038855,
0.02531164, 0.025572715, 0.025837127, 0.026111702, 0.026381474,
0.026662279, 0.026937006, 0.027245549, 0.027543716, 0.027828717,
0.028140012, 0.028432922, 0.028731019, 0.029045922, 0.029352994,
0.029665791, 0.029976169, 0.030274112, 0.030581931, 0.030909055,
0.031251467, 0.031583734, 0.031910215, 0.032263456, 0.032607759,
0.032960186, 0.033308706, 0.033671465, 0.034008894, 0.034362784,
0.034748078, 0.035117657, 0.035481814, 0.035849448, 0.036204865,
0.036576278, 0.036972513, 0.037354248, 0.037766069, 0.038147006,
0.038540329, 0.038952025, 0.039338998, 0.039730929, 0.04015352,
0.04057376, 0.040982485, 0.041411844, 0.041825225, 0.042267959,
0.042685428, 0.043133188, 0.043578345, 0.04403485, 0.044497823,
0.044936573, 0.045399494, 0.045876537, 0.046341113, 0.046823228,
0.047280776, 0.047787891, 0.04828274, 0.048774597, 0.049269698,
0.049764183, 0.0502466, 0.050735216, 0.051240896, 0.051743481,
0.052263773, 0.052802417, 0.053332289, 0.053875804, 0.054420092,
0.054952326, 0.05551223, 0.0560581, 0.056624377, 0.057207595,
0.057781968, 0.058349377, 0.058932672, 0.059507004, 0.060102617,
0.060675102, 0.061263766, 0.061875403, 0.062474742, 0.063080816,
0.063710522, 0.064337298, 0.06496146, 0.065591322, 0.066229881,
0.066873624, 0.067511479, 0.068164916, 0.068834802, 0.069473001,
0.070130056, 0.070802875, 0.071463387, 0.072150478, 0.072852004,
0.073549705, 0.074234518, 0.074928651, 0.075662102, 0.07637149,
0.077094314, 0.07780666, 0.078533026, 0.079258877, 0.079996209,
0.08071819, 0.081457169, 0.082223678, 0.082964093, 0.083740788,
0.084525266, 0.085309877, 0.086091976, 0.086876189, 0.087681038,
0.088466692, 0.089272387, 0.090091698, 0.090943661, 0.091773931,
0.092628442, 0.093496382, 0.094342775, 0.09518846, 0.096041448,
0.096890524, 0.097763712, 0.098652486, 0.099552709, 0.100460729,
0.101360769, 0.102277899, 0.103168297, 0.104091322, 0.105021345,
0.105969769, 0.106910274, 0.107884211, 0.108816771, 0.109761082,
0.110747068, 0.111717668, 0.11274081, 0.113733308, 0.114736182,
0.115772782, 0.116769565, 0.117802504, 0.118823783, 0.119861842,
0.120908887, 0.121970079, 0.123045008, 0.124139025, 0.125227387,
0.126321505, 0.127416415, 0.12852183, 0.129647312, 0.130771163,
0.131917621, 0.13305821, 0.134221177, 0.135393993, 0.1365608,
0.13771795, 0.138632941, 0.138867026, 0.140064908, 0.141254494,
0.142482426, 0.143715877, 0.144953375, 0.146189778, 0.147444786,
0.148678774, 0.1499224, 0.151166127, 0.152441136, 0.153696015,
0.154953696, 0.156269575, 0.15760261, 0.158898707, 0.160211327,
0.1615151, 0.16286521, 0.164227299, 0.16558438, 0.166939235,
0.168333542, 0.169725707, 0.171104256, 0.172524661, 0.173928857,
0.175370388, 0.176786885, 0.17824894, 0.179736659, 0.181195894,
0.182669972, 0.184149777, 0.1856612, 0.187147813, 0.188672571,
0.190181439, 0.191706098, 0.193263521, 0.194838288, 0.196405381,
0.19799745, 0.199565738, 0.201165687, 0.202780184, 0.204379329,
0.206015561, 0.207657017, 0.209311653, 0.210977284, 0.212648504,
0.214326304, 0.216023306, 0.217717944, 0.21943923, 0.221167162,
0.222907864, 0.224659926, 0.226409146, 0.228204159, 0.229986083,
0.231775477, 0.233583389, 0.235408882, 0.237254546, 0.239106291,
0.240983301, 0.242872591, 0.244756405, 0.246649027, 0.248540583,
0.250452688, 0.252382579, 0.254311892, 0.256270862, 0.258248494,
0.260222555, 0.262239797, 0.264239994, 0.266235641, 0.268301944,
0.270353653, 0.272404243, 0.274456814, 0.276553241, 0.278682664,
0.280800243, 0.282917078, 0.285068201, 0.287234898, 0.289397674,
0.291592579, 0.293756672, 0.295930047, 0.296412661, 0.29816527,
0.300401457, 0.302663645, 0.304924623, 0.307209611, 0.309497928,
0.311818338, 0.314129942, 0.31650366, 0.318830714, 0.321177586,
0.323534653, 0.325904396, 0.328311518, 0.33073116, 0.33314564,
0.33559132, 0.338058167, 0.340490468, 0.34299361, 0.345470026,
0.347992263, 0.350527983, 0.353069017, 0.35561004, 0.356142542,
0.358157364, 0.360764336, 0.363374824, 0.365998166, 0.36864503,
0.371308855, 0.373989063, 0.376707524, 0.379424497, 0.382145781,
0.384914548, 0.38768099, 0.390475338, 0.393297067, 0.39612223,
0.39897136, 0.401836321, 0.404728844, 0.407637732, 0.410556011,
0.413499551, 0.416447835, 0.419472934, 0.422488094, 0.425541065,
0.428602855, 0.431706659, 0.434806957, 0.437951622, 0.441124349,
0.444280475, 0.447486249, 0.450738395, 0.454001336, 0.457291456,
0.458696882, 0.460621015, 0.463987459, 0.467352659, 0.470783042,
0.474216303, 0.477690652, 0.481212393, 0.484735324, 0.488275983,
0.491861971, 0.495477958, 0.499168285, 0.502888294, 0.506620788,
0.510360589, 0.514112908, 0.517934311, 0.521806637, 0.525679282,
0.529642197, 0.533618728, 0.537622826, 0.541669816, 0.545756189,
0.549893325, 0.551612343, 0.554049027, 0.558268399, 0.562512231,
0.56677467, 0.571055001, 0.575440527, 0.579869196, 0.584367993,
0.58892245, 0.593519558, 0.598109914, 0.60279167, 0.607544237,
0.612307459, 0.617164866, 0.622102629, 0.627014756, 0.632074308,
0.637172244, 0.64124088, 0.65044007, 0.655513745, 0.660553185,
0.665584327, 0.670533717, 0.67543102, 0.680337977, 0.685199376,
0.689999408, 0.69478363, 0.699535807, 0.704211856, 0.708841676,
0.713476271, 0.718085856, 0.722689865, 0.727265561, 0.731791919,
0.734439117, 0.736270772, 0.740736732, 0.745181634, 0.749655777,
0.754101038, 0.758510722, 0.762872965, 0.767269427, 0.771611628,
0.775960883, 0.780269907, 0.784568454, 0.788896373, 0.793159983,
0.797471196, 0.801692075, 0.805923848, 0.810154453, 0.814386962,
0.818580093, 0.822762658, 0.826949325, 0.831135914, 0.835262279,
0.8394329, 0.841809132, 0.84357495, 0.847754738, 0.85190491,
0.856034225, 0.860151945, 0.864289837, 0.868374839, 0.872497364,
0.876611168, 0.880747097, 0.884864279, 0.889015623, 0.893189785,
0.897366509, 0.901495714, 0.905598463, 0.909755329, 0.913981309,
0.918155711, 0.922319175, 0.926511225, 0.93068738, 0.934965999,
0.939168053, 0.943419976, 0.947709798, 0.952065826, 0.956368213,
0.960730314, 0.965063399, 0.969475191, 0.973929476, 0.97832891,
0.982762271, 0.987228635, 0.990821102, 0.991756187, 0.996280877,
1.000822436, 1.005462788, 1.010115852, 1.014760169, 1.019503604,
1.024268488, 1.029024299, 1.033798623, 1.03860537, 1.043496949,
1.048546089, 1.053596065, 1.058658246, 1.063743447, 1.068961857,
1.074130336, 1.079492576, 1.084804685, 1.090149187, 1.095621179,
1.101182198, 1.106765951, 1.112413004, 1.116942833, 1.118220485,
1.124129381, 1.130059958, 1.136140212, 1.142228477, 1.14845495,
1.154856348, 1.161303828, 1.167905483, 1.174705768, 1.181569268,
1.188557071, 1.195525153, 1.202744857, 1.210075333, 1.217517918,
1.22517237, 1.23293905, 1.240723033, 1.248786428, 1.256936228,
1.265112808, 1.273469496, 1.281945433, 1.290599957, 1.299334386,
1.307988622, 1.316774133, 1.325689579, 1.334854, 1.343997395,
1.353256088, 1.36241604, 1.371806833, 1.381195167, 1.390730444,
1.400287204, 1.410009879, 1.419668214, 1.429419495, 1.439209833,
1.449124503, 1.459029191, 1.469024563, 1.478914513, 1.488922679,
1.499100736, 1.509448988, 1.519660179, 1.529905325, 1.540462117,
1.55109404, 1.561603618, 1.572344794, 1.583013743, 1.593858653,
1.604627538, 1.61562163, 1.626568161, 1.637558823, 1.648701076,
1.65984687, 1.671236244, 1.682544944, 1.69387451, 1.705402096,
1.716862842, 1.72869096, 1.740434885, 1.752099575, 1.764137688,
1.776299519, 1.788448865, 1.80051542, 1.812660791, 1.824973505,
1.837493252, 1.849723108, 1.862482445, 1.875012086, 1.887656881,
1.900247413, 1.913080164, 1.925983064, 1.939256027, 1.952250883,
1.965357638, 1.978595102, 1.99188788, 2.005494618, 2.01884326,
2.032286903, 2.046043776, 2.059318548, 2.072976369, 2.086933308,
2.089720898, 2.100351795, 2.113989607, 2.127861225, 2.141865623,
2.15586404, 2.169854302, 2.183961245, 2.198073637, 2.212194225,
2.226345669, 2.240675195, 2.254538606, 2.269078504, 2.283263033,
2.297547037, 2.311724278, 2.326381922, 2.340671499, 2.355249616,
2.36981316, 2.384504606, 2.398803297, 2.41323192, 2.427790812,
2.442808532, 2.457955711, 2.472738693, 2.487838046, 2.502865943,
2.517373964, 2.532197306, 2.547066402, 2.562177019, 2.577405329,
2.592081987, 2.606801812, 2.622080367, 2.636664356, 2.651898029,
2.667279364, 2.682565258, 2.697479523, 2.712686141, 2.728211348,
2.743647355, 2.759240763, 2.774512632, 2.78977269, 2.804772692,
2.820484578, 2.835991227, 2.851564983, 2.866455592, 2.882210053,
2.897641599, 2.913234422, 2.928843981, 2.944208251, 2.959783309,
2.974936169, 2.990259312, 3.005337859, 3.02105701, 3.036519485,
3.051989693, 3.068202553, 3.083810715, 3.098874967, 3.114368543,
3.130470135, 3.146529804, 3.162319098, 3.17783682, 3.193317715,
3.209070519, 3.224761864, 3.240299997, 3.256324084, 3.271653503,
3.287404134, 3.303094373, 3.318411013, 3.334341409, 3.349552387,
3.3653001, 3.380781845, 3.397059766, 3.412759509, 3.428204025,
3.443763699, 3.45946203, 3.475191939, 3.491205206, 3.50699097,
3.5226477, 3.537770938, 3.553298563, 3.568699813, 3.584610359,
3.600328468, 3.616416203, 3.632143424, 3.647792018, 3.664199627,
3.680472687, 3.696219579, 3.711330091, 3.726606105, 3.742696105,
3.758789654, 3.774038801, 3.789289924, 3.804964497, 3.820689724,
3.835753089, 3.851178938, 3.867159212, 3.883560141, 3.89887823,
3.914737319, 3.930404192, 3.946220222, 3.961827317, 3.977306201,
3.993304614, 4.009619112, 4.025312175, 4.041037247, 4.056813878,
4.072710448, 4.087816615, 4.1033018, 4.118381627, 4.134771235,
4.149629611, 4.164950609, 4.180167932, 4.195351712, 4.211302794,
4.226383775, 4.241631617, 4.256500285, 4.271245641, 4.287319197,
4.302373676, 4.317606995, 4.333048804, 4.348550002, 4.364914431,
4.38104059, 4.396763173, 4.412112513, 4.428092719, 4.44287666,
4.458588335, 4.473427664, 4.48887355, 4.50414533, 4.519793848,
4.535216758, 4.550587951, 4.566090749, 4.581441505, 4.59756706,
4.61290163, 4.628130657, 4.642783388, 4.657403349, 4.672373175,
4.687238046, 4.701789, 4.716382416, 4.731204434, 4.745455437,
4.760572009, 4.77650928, 4.792030454, 4.807425718, 4.821321921,
4.836661047, 4.85170574, 4.86709184, 4.881789457, 4.89647461,
4.910782461, 4.925868186, 4.94060752, 4.95619869, 4.971487264,
4.986211576, 5.001164347, 5.016375676, 5.030621216, 5.044201663,
5.058663282, 5.073680388, 5.088841994, 5.103053638, 5.117197321,
5.131273544, 5.134380963, 5.145855032, 5.159138686, 5.173477488,
5.186644592, 5.201280853, 5.214805696, 5.227330622, 5.241745887,
5.254894673, 5.268459318, 5.283004841, 5.296786297, 5.310845712,
5.325876701, 5.340519423, 5.353525665, 5.367478622, 5.381728018,
5.395559371, 5.410379192, 5.424678699, 5.438911802, 5.452265534,
5.466643231, 5.48075255, 5.49460159, 5.508978931, 5.52240022,
5.535653007, 5.549839573, 5.562474114, 5.57563891, 5.589995629,
5.60465439, 5.618024231, 5.630403598, 5.643269818, 5.657263236,
5.670827364, 5.683838562, 5.697062728, 5.711083223, 5.724531201,
5.737750867, 5.751756267, 5.765161971, 5.779504897, 5.791719373,
5.803462799, 5.815908504, 5.828651104, 5.841687592, 5.855539104,
5.868593607, 5.882023736, 5.895140755, 5.908940995, 5.922263803,
5.935454611, 5.946761525, 5.959872967, 5.973624177, 5.985209569,
5.998263632, 6.012458149, 6.026913239, 6.040136941, 6.052319518,
6.064561191, 6.077927028, 6.090845096, 6.104124622, 6.11619511,
6.12980548, 6.141068068, 6.153266557, 6.164402055, 6.177759267,
6.190131232, 6.202131302, 6.214626138, 6.227125308, 6.238958215,
6.250975415, 6.2640096, 6.275915508, 6.288314315, 6.300902514,
6.313010591, 6.324209744, 6.337699018, 6.349800828, 6.362390066,
6.374205935, 6.387470945, 6.399312886, 6.412990704, 6.425872985,
6.436548957, 6.447509657, 6.458792993, 6.469615379, 6.482024319,
6.494345431, 6.505249756, 6.516917822, 6.528070273, 6.539753146,
6.552745298, 6.565256616, 6.576020138, 6.586238014, 6.598313517,
6.611227764, 6.62165779, 6.631691928, 6.643143856, 6.654405712,
6.665349304, 6.675161128, 6.687524812, 6.700412571, 6.710947813,
6.72271854, 6.73404781, 6.743802513, 6.754745556, 6.764358264,
6.77599299, 6.78800906, 6.798329909, 6.808271806, 6.818809405,
6.827887977, 6.839550184, 6.850700139, 6.861178602, 6.870207805,
6.879254066, 6.890743559, 6.900743059, 6.912159782, 6.924031835,
6.934408526, 6.944290521, 6.954371085, 6.963929524, 6.972887765,
6.983690153, 6.994214959, 7.003506163, 7.014364115, 7.023548136,
7.032233185, 7.043480433, 7.053077519, 7.06209644, 7.072671111,
7.082895701, 7.092380057, 7.102958298, 7.112888324, 7.123329122,
7.135016676, 7.143210616, 7.15438583, 7.165528913, 7.174783264,
7.183183783, 7.194378458, 7.202487533, 7.212975632, 7.223547425,
7.233740913, 7.244496879, 7.252784999, 7.261741735, 7.271371778,
7.283980264, 7.294971948, 7.304183774, 7.314345756, 7.322620132,
7.332983189, 7.341071673, 7.351030053, 7.359314809, 7.368316796,
7.376044227, 7.385205998, 7.393639765, 7.402429304, 7.412172382,
7.421971208, 7.433279824, 7.441869409, 7.449495301, 7.458620858,
7.468116586, 7.476877057, 7.480549627, 7.48829792, 7.495888002,
7.504933055, 7.511813721, 7.520122526, 7.528313143, 7.537917085,
7.546768669, 7.55681745, 7.564926784, 7.574101101, 7.58196773,
7.591255125, 7.599334087, 7.605996181, 7.614358577, 7.623043273,
7.631761387, 7.640747402, 7.650165878, 7.661178115, 7.671126689,
7.678230842, 7.687413825, 7.696310665, 7.703719781, 7.71220506,
7.721774749, 7.730394139, 7.739837568, 7.749663857, 7.757353965,
7.765264454, 7.772941956, 7.78113766, 7.790470497, 7.796695259,
7.805076801, 7.813931739, 7.820670148, 7.829352207, 7.836973496,
7.843469695, 7.851640666, 7.859192298, 7.867418875, 7.874238322,
7.883461107, 7.893049003, 7.899517574, 7.90758053, 7.915469044,
7.922010337, 7.928475753, 7.938826418, 7.946700757, 7.953415341,
7.962181244, 7.969872187, 7.977714815, 7.985429219, 7.995471983,
8.002487092, 8.010970536, 8.017256447, 8.023576352, 8.030475646,
8.038546738, 8.048435928, 8.054407974, 8.059598579, 8.065803028,
8.071614966, 8.0767194, 8.083113198, 8.093146509, 8.10134046,
8.107499643, 8.114322677, 8.121199885, 8.126831527, 8.134036068,
8.1400024, 8.150661859, 8.157926265, 8.166849432, 8.173681591,
8.181803244, 8.187037133, 8.193053552, 8.199631792, 8.205287184,
8.209824025, 8.218350937, 8.224247712, 8.230609686, 8.237523214,
8.245977421, 8.250353049, 8.258655522, 8.263787204, 8.270447777,
8.278533936, 8.284318241, 8.292988023, 8.298839001, 8.304094188,
8.31097468, 8.317949728, 8.323279152, 8.329806942, 8.337938547,
8.346021825, 8.353586248, 8.359024652, 8.365610415, 8.371761657,
8.383609303, 8.392730448, 8.397994731, 8.405278849, 8.415566607,
8.420709105, 8.425982837, 8.432707987, 8.439875515, 8.444428389,
8.454760519, 8.461131021, 8.465772122, 8.471475692, 8.479378843,
8.483191067, 8.489319794, 8.49315754, 8.501660553, 8.507809466,
8.514683959, 8.523723324, 8.530037795, 8.537180004, 8.547095916,
8.553643464, 8.559813958, 8.563961619, 8.56952123, 8.574988793,
8.579247689, 8.586347924, 8.592757166, 8.598690054, 8.601964875,
8.609964172, 8.620012544, 8.625723297, 8.635510279, 8.641136148,
8.646636229, 8.650794204, 8.657791452, 8.661520245, 8.668001587,
8.67611085, 8.681212845, 8.685650576, 8.689391442, 8.693096236,
8.701203022, 8.704794271, 8.707449519, 8.715624964, 8.719105487,
8.726159921, 8.731491096, 8.735058318, 8.746802608, 8.752106717,
8.755956387, 8.764821969, 8.773056201, 8.777112255, 8.78298729,
8.789064623, 8.792356054, 8.795936376, 8.803520706, 8.806722627,
8.81172151, 8.816230606, 8.822873249, 8.825653293, 8.82875214,
8.837532973, 8.840020576, 8.84305092, 8.846240429, 8.849132796,
8.855021892, 8.857627283, 8.860117669, 8.866169433, 8.868367558,
8.872943973, 8.874248665, 8.87905568, 8.884684866, 8.89083719,
8.892363103, 8.899291747, 8.90159134, 8.90864608, 8.912877338,
8.918594727, 8.924331846, 8.925113111, 8.930173904, 8.933144335,
8.936133161, 8.940276194, 8.943236333, 8.944618537, 8.949125223,
8.95163792, 8.955305565, 8.958311633, 8.960902935, 8.968155841,
8.972275786, 8.977683076, 8.980357364, 8.981575201, 8.989526346,
8.992767427, 8.995902201, 9.001300482, 9.006438803, 9.010384489,
9.011012605, 9.016733574, 9.018605379, 9.019301735, 9.030618481,
9.036999639, 9.04124411, 9.044334493, 9.048868301, 9.050663182,
9.054258122, 9.056763471, 9.059305926, 9.060700291, 9.063163167,
9.066261546, 9.071557978, 9.076480182, 9.080526665, 9.084508751,
9.08765888, 9.091543029, 9.09520246, 9.097340909, 9.100876858,
9.103463676, 9.105668787, 9.114550928, 9.117121668, 9.11752028,
9.121378756, 9.125626245, 9.126540609, 9.135057066, 9.13532412,
9.138660689, 9.140089218, 9.14236083, 9.144000756, 9.14735692,
9.152942636, 9.154960493, 9.155201261, 9.156446515, 9.158331155,
9.16012823, 9.160722218, 9.167844299, 9.174298161, 9.177378716,
9.183358178, 9.185090345, 9.194145653, 9.196579267, 9.198709083,
9.201639817, 9.210227736, 9.218146401, 9.218869533, 9.220147215,
9.221308737, 9.226103226, 9.236823316, 9.23794137, 9.242590292,
9.246861434, 9.250164564, 9.250941408, 9.253412807, 9.254188225,
9.254346187, 9.263446662, 9.264235715, 9.26601677, 9.268338032,
9.269398789, 9.269722889, 9.270896451, 9.274161522, 9.27929605,
9.281179052, 9.285930354, 9.293614598, 9.296712663, 9.296860098,
9.299870167, 9.303183367, 9.304049533, 9.308433534, 9.313361664,
9.31821701, 9.323647023, 9.329855649, 9.333251634, 9.333660136,
9.337334946, 9.338164483, 9.343075012, 9.345022262, 9.34513263,
9.351094272, 9.359358664, 9.364297906), "10" = c(3.4839e-05,
3.4861e-05, 3.5187e-05, 3.5706e-05, 3.5908e-05, 3.6189e-05, 3.6279e-05,
3.6584e-05, 3.7652e-05, 3.7887e-05, 3.791e-05, 3.8996e-05, 3.9228e-05,
3.9351e-05, 3.9398e-05, 4.007e-05, 4.0214e-05, 4.1066e-05, 4.1071e-05,
4.1374e-05, 4.2437e-05, 4.2445e-05, 4.2833e-05, 4.2868e-05, 4.3323e-05,
4.3878e-05, 4.5029e-05, 4.5215e-05, 4.544e-05, 4.5776e-05, 4.5941e-05,
4.6807e-05, 4.8418e-05, 4.917e-05, 4.9578e-05, 4.9841e-05, 5.0441e-05,
5.0966e-05, 5.1757e-05, 5.2206e-05, 5.2273e-05, 5.2757e-05, 5.3657e-05,
5.3978e-05, 5.4158e-05, 5.4341e-05, 5.6117e-05, 5.6598e-05, 5.7374e-05,
5.7396e-05, 5.7937e-05, 5.8308e-05, 5.8468e-05, 5.915e-05, 5.9279e-05,
6.0185e-05, 6.2356e-05, 6.3296e-05, 6.3605e-05, 6.3768e-05, 6.4935e-05,
6.5981e-05, 6.6074e-05, 6.7255e-05, 6.7918e-05, 6.8445e-05, 6.8623e-05,
6.8902e-05, 7.0206e-05, 7.0799e-05, 7.2006e-05, 7.3314e-05, 7.419e-05,
7.4896e-05, 7.5702e-05, 7.661e-05, 7.6805e-05, 7.7895e-05, 7.9939e-05,
8.1323e-05, 8.1863e-05, 8.297e-05, 8.4342e-05, 8.5379e-05, 8.8252e-05,
8.9425e-05, 9.0363e-05, 9.0898e-05, 9.1771e-05, 9.2181e-05, 9.343e-05,
9.3687e-05, 9.4502e-05, 9.5749e-05, 9.7246e-05, 9.8175e-05, 9.8753e-05,
0.000101259, 0.000102234, 0.000102966, 0.00010349, 0.000105328,
0.00010693, 0.000108356, 0.000109865, 0.00011081, 0.000111366,
0.000116471, 0.00011784, 0.000118671, 0.000119241, 0.000120167,
0.000121351, 0.000124237, 0.000125997, 0.000127023, 0.00012967,
0.000131343, 0.000132312, 0.000134431, 0.000136157, 0.000138441,
0.000139388, 0.00014027, 0.00014115, 0.000142429, 0.000143962,
0.000145547, 0.000147546, 0.000149846, 0.000152037, 0.0001542,
0.000156212, 0.000158338, 0.000159753, 0.000161286, 0.000162434,
0.00016362, 0.000165485, 0.000166515, 0.000168482, 0.000169519,
0.000171676, 0.000173217, 0.000175773, 0.000178142, 0.000180885,
0.000182259, 0.000183484, 0.000184602, 0.000187106, 0.000189166,
0.000190503, 0.000193555, 0.000197096, 0.000200017, 0.000203144,
0.000204564, 0.00020695, 0.000208893, 0.000210594, 0.000211679,
0.000215336, 0.000217863, 0.000220213, 0.000223855, 0.000227646,
0.000228983, 0.00023183, 0.000234674, 0.000236659, 0.000240319,
0.000243862, 0.000248154, 0.000250293, 0.000253085, 0.00025528,
0.000258496, 0.000261117, 0.000265965, 0.00026952, 0.000271453,
0.000274236, 0.000276084, 0.000278344, 0.000282217, 0.000285789,
0.000286619, 0.000290032, 0.000293454, 0.000296564, 0.000299762,
0.000300583, 0.000303507, 0.000306053, 0.000310191, 0.00031309,
0.000317617, 0.000321662, 0.000324798, 0.000329566, 0.000333078,
0.000336457, 0.000342842, 0.000345676, 0.000348926, 0.000354222,
0.000357842, 0.000360589, 0.000363584, 0.000366949, 0.000370518,
0.000375225, 0.000380047, 0.000383565, 0.000387371, 0.000390888,
0.000394317, 0.000399223, 0.000403493, 0.00040783, 0.000411488,
0.00041471, 0.000418813, 0.000423098, 0.000427955, 0.000430956,
0.000435827, 0.000439639, 0.000443581, 0.00044939, 0.000456562,
0.000461501, 0.000465356, 0.000471579, 0.000475269, 0.000485071,
0.000489617, 0.000493112, 0.000500781, 0.000507449, 0.000511348,
0.000517195, 0.000521701, 0.00052824, 0.000533225, 0.000539569,
0.000545146, 0.000550646, 0.00055587, 0.0005604, 0.000567067,
0.000573596, 0.000582613, 0.000589624, 0.000594563, 0.000600728,
0.000609489, 0.000615844, 0.000624084, 0.000632156, 0.000642027,
0.000648117, 0.000658191, 0.000665218, 0.000675011, 0.000681309,
0.000689922, 0.000697727, 0.000705924, 0.000712431, 0.000720916,
0.000728143, 0.00073657, 0.000745649, 0.00075118, 0.000761206,
0.000769235, 0.000776708, 0.00078522, 0.000794522, 0.000807486,
0.000819041, 0.000828768, 0.000839762, 0.000850906, 0.000860377,
0.000871366, 0.000880981, 0.000892664, 0.000900773, 0.000911388,
0.000923239, 0.000932162, 0.000941754, 0.000950341, 0.000958985,
0.000969302, 0.000979129, 0.000987891, 0.000999306, 0.001011805,
0.001022876, 0.001038482, 0.001049271, 0.001066379, 0.001078467,
0.001090907, 0.001104433, 0.001117099, 0.001125667, 0.001137058,
0.001150195, 0.001165551, 0.001177292, 0.001193822, 0.001206389,
0.0012212, 0.001237652, 0.001248771, 0.001261159, 0.001278213,
0.001289666, 0.00130692, 0.001323798, 0.001338517, 0.001355981,
0.001371791, 0.001387156, 0.001403516, 0.001415292, 0.001432367,
0.001449261, 0.001471644, 0.001489693, 0.001503879, 0.001519771,
0.001537645, 0.001553678, 0.00157135, 0.001584364, 0.001588781,
0.001606021, 0.001624494, 0.001642344, 0.001662425, 0.001684175,
0.001699939, 0.001717802, 0.001737493, 0.001756536, 0.001778457,
0.001800134, 0.001821873, 0.001843665, 0.001867114, 0.001886742,
0.001912585, 0.001933293, 0.001958419, 0.001984986, 0.002011824,
0.002033592, 0.002058411, 0.002077427, 0.002106228, 0.002131332,
0.002156875, 0.002181862, 0.002212415, 0.002240264, 0.002268874,
0.002298982, 0.002322101, 0.002353018, 0.00237912, 0.002409121,
0.002435956, 0.002464936, 0.002493549, 0.002527661, 0.002559579,
0.002588474, 0.002618961, 0.002646067, 0.002682171, 0.002711641,
0.002744609, 0.002777509, 0.002809972, 0.002832035, 0.002861589,
0.002892272, 0.002919589, 0.002951047, 0.002979239, 0.003011641,
0.003046923, 0.003081247, 0.003118122, 0.003155955, 0.003190725,
0.003224746, 0.003257636, 0.003288451, 0.003327008, 0.003363772,
0.003402366, 0.003450126, 0.003488245, 0.003530667, 0.003576678,
0.003621338, 0.003665334, 0.003706738, 0.003749721, 0.003790898,
0.003831492, 0.003885118, 0.003926854, 0.003971993, 0.004016871,
0.004064893, 0.004110847, 0.004154862, 0.004196057, 0.00424633,
0.004289902, 0.004343434, 0.004391027, 0.004438262, 0.004488907,
0.00453876, 0.004585855, 0.004638955, 0.004690963, 0.004747075,
0.004796597, 0.004861256, 0.004912422, 0.004972589, 0.005037359,
0.005095556, 0.005155292, 0.005208575, 0.005270249, 0.005327455,
0.005386945, 0.005456113, 0.005522771, 0.005588694, 0.005653106,
0.0057249, 0.005784695, 0.005853493, 0.00592704, 0.00599627,
0.006064931, 0.006126662, 0.006196706, 0.006268111, 0.006346342,
0.006418132, 0.006493265, 0.006557153, 0.00663941, 0.006717969,
0.00679663, 0.006882484, 0.006963572, 0.007051409, 0.007132902,
0.007207094, 0.007291515, 0.007373353, 0.00745308, 0.007532234,
0.007618827, 0.007705484, 0.007795192, 0.007874946, 0.007966579,
0.008045207, 0.008148116, 0.008240174, 0.008342574, 0.008435475,
0.008544276, 0.008633119, 0.008738254, 0.008847872, 0.008943761,
0.009044631, 0.009152752, 0.009260627, 0.009364398, 0.009474209,
0.009580007, 0.009689336, 0.0098059, 0.009912301, 0.010026318,
0.010135789, 0.010252774, 0.01037755, 0.010494451, 0.010614601,
0.010728231, 0.010846762, 0.010966973, 0.01109405, 0.011231126,
0.011355152, 0.011488004, 0.011613384, 0.011745986, 0.011884016,
0.012020287, 0.012145758, 0.012283328, 0.012418749, 0.01256306,
0.012700897, 0.012845506, 0.012984035, 0.013141684, 0.013285889,
0.01344427, 0.013585014, 0.013725883, 0.013869291, 0.014021866,
0.014181313, 0.014340244, 0.014510828, 0.014671465, 0.014839474,
0.014997512, 0.015175273, 0.015345846, 0.015513889, 0.015641653,
0.015678518, 0.015855597, 0.016031518, 0.016213799, 0.016407443,
0.016582196, 0.016765322, 0.016953426, 0.017137043, 0.01731395,
0.01749392, 0.017685243, 0.01787874, 0.018081856, 0.018280393,
0.018493235, 0.018713949, 0.018929016, 0.019140656, 0.019338039,
0.01956398, 0.019766664, 0.019991905, 0.020215741, 0.020454612,
0.02067541, 0.020907709, 0.021125822, 0.021356905, 0.021595969,
0.021835443, 0.022093215, 0.022340206, 0.022585612, 0.022832572,
0.023078704, 0.023338873, 0.023591316, 0.023843808, 0.024109622,
0.024369936, 0.024649521, 0.02491085, 0.025199315, 0.025466027,
0.025759912, 0.026048458, 0.026341376, 0.026630651, 0.026923018,
0.02722684, 0.027532455, 0.02783278, 0.028146233, 0.028465643,
0.028779231, 0.02908904, 0.02940398, 0.029719986, 0.030031263,
0.030362293, 0.030693206, 0.031026584, 0.03136724, 0.031698462,
0.032067576, 0.032403974, 0.032756247, 0.033122079, 0.033474874,
0.033834055, 0.034215996, 0.034578242, 0.034956683, 0.035338467,
0.035705051, 0.036101017, 0.0364912, 0.036878644, 0.037272313,
0.037693437, 0.038105967, 0.03851715, 0.038923487, 0.039357676,
0.039785015, 0.040194802, 0.040631074, 0.041070081, 0.041497311,
0.041949593, 0.042387738, 0.042813993, 0.04328089, 0.043754079,
0.044212617, 0.044684699, 0.045175455, 0.045637771, 0.046158297,
0.04665672, 0.047165448, 0.047677942, 0.048187951, 0.048702154,
0.049212765, 0.049713911, 0.050259192, 0.050781946, 0.051294858,
0.051839854, 0.052364884, 0.052897394, 0.053449892, 0.054031423,
0.054586227, 0.055145805, 0.055716648, 0.056316645, 0.056921364,
0.057508719, 0.058081902, 0.058683707, 0.059290161, 0.059915692,
0.060558943, 0.061184939, 0.061810896, 0.062448509, 0.063094476,
0.063766002, 0.064442841, 0.06509929, 0.0657918, 0.066484331,
0.067162292, 0.067856165, 0.068548698, 0.069245337, 0.069956855,
0.070665378, 0.07139818, 0.072141631, 0.072869167, 0.073605191,
0.074381434, 0.075131682, 0.075885208, 0.076676274, 0.077451857,
0.078253819, 0.079023925, 0.079806628, 0.080609778, 0.081405364,
0.082218387, 0.08303654, 0.083877133, 0.084713607, 0.0855378,
0.086392884, 0.087255587, 0.088123692, 0.089006903, 0.08990238,
0.090804374, 0.091704515, 0.092614331, 0.093531637, 0.094459181,
0.095391986, 0.09634383, 0.097278836, 0.098241824, 0.099215899,
0.100190399, 0.101192284, 0.1021886, 0.103202898, 0.104196978,
0.105215656, 0.106246892, 0.107287354, 0.108302849, 0.109352749,
0.110416977, 0.111491333, 0.112566996, 0.113646424, 0.114736468,
0.115823121, 0.116946575, 0.118085967, 0.119207159, 0.120362494,
0.121529478, 0.122689837, 0.123850499, 0.125048358, 0.126276251,
0.127237547, 0.127477404, 0.128680814, 0.129890074, 0.131109078,
0.132354303, 0.133609219, 0.134863618, 0.136115512, 0.137404671,
0.138710429, 0.139988108, 0.141308462, 0.142598947, 0.143942517,
0.145264566, 0.146613574, 0.147953018, 0.149335655, 0.150722186,
0.152119039, 0.153508075, 0.15491008, 0.156330414, 0.157737093,
0.159187254, 0.160623968, 0.162090828, 0.163590666, 0.165095043,
0.166583755, 0.168083927, 0.169595874, 0.171147704, 0.172709969,
0.174262342, 0.17582164, 0.177414109, 0.178995934, 0.180622988,
0.182245919, 0.183883583, 0.185515529, 0.187184331, 0.188849526,
0.19053127, 0.19222642, 0.193940352, 0.195670465, 0.197419107,
0.199222182, 0.200978809, 0.202705572, 0.204504171, 0.206288955,
0.208080756, 0.209895899, 0.211747632, 0.213612338, 0.215450788,
0.217333474, 0.219218354, 0.221125106, 0.223022843, 0.224952302,
0.226863012, 0.228803645, 0.230814709, 0.23278008, 0.234787476,
0.236815192, 0.238785876, 0.24083577, 0.242875132, 0.244955746,
0.247043083, 0.249165084, 0.251311365, 0.253433753, 0.255579093,
0.25771999, 0.259911819, 0.262098837, 0.264300397, 0.266468691,
0.268695293, 0.270955965, 0.273226798, 0.27550936, 0.277813907,
0.280128908, 0.282482859, 0.284882654, 0.287261381, 0.289647453,
0.292056211, 0.294464514, 0.294985009, 0.29691527, 0.299363838,
0.301848854, 0.30435393, 0.306854384, 0.309393812, 0.311943509,
0.314501687, 0.317068956, 0.319666557, 0.322306256, 0.32496731,
0.327639532, 0.33032713, 0.33305561, 0.335804705, 0.338550408,
0.341330622, 0.344107233, 0.346912017, 0.349741794, 0.352570835,
0.355417051, 0.358303322, 0.361233221, 0.361819519, 0.364169762,
0.367111185, 0.370079483, 0.373082151, 0.376069153, 0.379092477,
0.382167704, 0.385250696, 0.388358276, 0.391482509, 0.394661838,
0.397875245, 0.401071094, 0.404289896, 0.407555887, 0.410832146,
0.414115468, 0.417428228, 0.420773062, 0.424162798, 0.427557863,
0.431010869, 0.434490086, 0.438020159, 0.441532644, 0.445098539,
0.448716627, 0.452314852, 0.455988944, 0.459602873, 0.463285176,
0.46703839, 0.470805093, 0.474601453, 0.478471566, 0.480107955,
0.48234221, 0.486204278, 0.490145679, 0.494109184, 0.498124023,
0.502162162, 0.506251654, 0.510334411, 0.514504378, 0.518676828,
0.522891639, 0.527117133, 0.531457624, 0.535774338, 0.540154209,
0.544576218, 0.549065169, 0.55357389, 0.558106326, 0.56271784,
0.567389922, 0.572100629, 0.576820782, 0.581606038, 0.586453308,
0.588467896, 0.591353361, 0.596308828, 0.601299823, 0.60633738,
0.611487368, 0.616663262, 0.621902903, 0.627195403, 0.632564256,
0.637997182, 0.643481519, 0.64906781, 0.654730473, 0.660443676,
0.666198138, 0.671994449, 0.677931635, 0.683961646, 0.690067991,
0.694981786, 0.706065211, 0.712204024, 0.718296612, 0.724285321,
0.730280787, 0.736182611, 0.742023032, 0.747866583, 0.753746892,
0.759515561, 0.765271058, 0.771000147, 0.776593371, 0.782204011,
0.787850288, 0.793420912, 0.798946527, 0.804475977, 0.807748303,
0.809980096, 0.815459908, 0.820854565, 0.826242793, 0.831650223,
0.837045, 0.84236412, 0.84771046, 0.853048129, 0.858360611, 0.86360293,
0.868807343, 0.874060167, 0.87931219, 0.884505107, 0.889670347,
0.894867557, 0.900015896, 0.905160656, 0.91031162, 0.915423168,
0.92055077, 0.925669078, 0.930830333, 0.935953819, 0.938902166,
0.941062292, 0.946140714, 0.951222822, 0.956358795, 0.96140277,
0.966535885, 0.971626935, 0.976782229, 0.981886052, 0.987004946,
0.992149453, 0.997270559, 1.002434505, 1.007510698, 1.012602291,
1.017771454, 1.022981379, 1.028171627, 1.033384842, 1.03859328,
1.043814733, 1.049004518, 1.054221192, 1.059499524, 1.064819805,
1.070118959, 1.075444076, 1.080800249, 1.086172687, 1.091587453,
1.097076761, 1.102522932, 1.108002798, 1.113603106, 1.119154324,
1.123604597, 1.124758891, 1.130471154, 1.136121409, 1.14192596,
1.147763363, 1.153646682, 1.159510105, 1.16549223, 1.171462212,
1.177553505, 1.1836856, 1.189837273, 1.196025897, 1.202344992,
1.20867178, 1.215108252, 1.221688561, 1.228312782, 1.234964929,
1.2417437, 1.24861593, 1.255572286, 1.262593216, 1.269773796,
1.276951715, 1.282634433, 1.284240673, 1.291666901, 1.29929172,
1.306899652, 1.314689571, 1.322484402, 1.330594626, 1.338670716,
1.346935887, 1.355301179, 1.363856899, 1.372651171, 1.381305266,
1.390212478, 1.399407152, 1.408579892, 1.418098234, 1.427765884,
1.43748996, 1.44729623, 1.457333001, 1.467506121, 1.47772342,
1.488191957, 1.498854693, 1.509577426, 1.520178964, 1.531207464,
1.542109745, 1.55325788, 1.564603887, 1.575976057, 1.587634094,
1.599394969, 1.61135625, 1.623183349, 1.635086157, 1.647373795,
1.659655455, 1.672234049, 1.684616356, 1.697260225, 1.709901044,
1.722692046, 1.735694219, 1.748418531, 1.761636173, 1.774636172,
1.787744977, 1.801187399, 1.814618032, 1.828323284, 1.841946911,
1.855616468, 1.869661223, 1.883706013, 1.897820389, 1.911847391,
1.926314469, 1.940660684, 1.955274629, 1.969858841, 1.98471957,
1.999233088, 2.013945116, 2.028822011, 2.043770247, 2.058811559,
2.07430097, 2.089793637, 2.105125588, 2.120945969, 2.136626563,
2.152305769, 2.168058387, 2.184083751, 2.200269368, 2.216093807,
2.231866961, 2.247997085, 2.264466089, 2.281025074, 2.297987337,
2.314831781, 2.331403502, 2.34852362, 2.365111578, 2.381888967,
2.399056067, 2.416373032, 2.433721275, 2.450940114, 2.46811018,
2.485782382, 2.503186023, 2.521057398, 2.524673517, 2.538288917,
2.555556038, 2.573395133, 2.591828054, 2.609962342, 2.628005009,
2.645725648, 2.664067835, 2.68216003, 2.70072058, 2.719313995,
2.737963386, 2.756510075, 2.7753762, 2.79462828, 2.81361609,
2.832914399, 2.8520055, 2.871109449, 2.890451253, 2.909918388,
2.929364094, 2.948970607, 2.969018161, 2.989366682, 3.009328338,
3.029478665, 3.050116871, 3.07036429, 3.090447669, 3.110311134,
3.130933778, 3.152006948, 3.172929199, 3.193626846, 3.214310165,
3.234981403, 3.255864356, 3.276977478, 3.298323082, 3.319133627,
3.340877054, 3.361852347, 3.38340846, 3.40473727, 3.425753869,
3.446711132, 3.468682444, 3.490317443, 3.511543177, 3.533809209,
3.555748128, 3.578176448, 3.600560012, 3.622876, 3.6453408, 3.6677256,
3.690252118, 3.712934183, 3.735816276, 3.758717115, 3.781520142,
3.804138792, 3.827218798, 3.849521563, 3.873545726, 3.896805064,
3.91942987, 3.943042032, 3.966268116, 3.989628329, 4.012959292,
4.035729249, 4.059595202, 4.082452724, 4.106266915, 4.129211396,
4.153264413, 4.176405381, 4.200006828, 4.223933497, 4.248385261,
4.272488294, 4.296258269, 4.319590537, 4.344223536, 4.368258411,
4.393009323, 4.417473224, 4.441210019, 4.465214529, 4.489722815,
4.51397265, 4.5387963, 4.562720216, 4.588092739, 4.612055772,
4.636285783, 4.66082539, 4.684934321, 4.709854471, 4.734326775,
4.758688128, 4.782575657, 4.807935253, 4.832181589, 4.856526611,
4.881329965, 4.905561835, 4.931743519, 4.956401189, 4.981208122,
5.006458711, 5.032528842, 5.058500808, 5.084928763, 5.111385563,
5.137515767, 5.162222655, 5.188318126, 5.213084091, 5.239645087,
5.266046793, 5.290539915, 5.316250764, 5.342768073, 5.368217715,
5.394190176, 5.419851123, 5.444319855, 5.469348116, 5.494274932,
5.519850335, 5.546205191, 5.571664359, 5.598419592, 5.624706859,
5.651075504, 5.677632607, 5.702901183, 5.727807112, 5.753717997,
5.779969887, 5.806463559, 5.833298168, 5.859627972, 5.886685629,
5.912316162, 5.938069904, 5.963404341, 5.990577388, 6.019705492,
6.047043923, 6.073733224, 6.098985766, 6.125251271, 6.15234576,
6.180336809, 6.206533887, 6.233071958, 6.260348112, 6.286616462,
6.31256301, 6.339235628, 6.367652926, 6.395661781, 6.422475677,
6.448846298, 6.474867242, 6.501826078, 6.528936695, 6.554704658,
6.580416578, 6.607299371, 6.635678507, 6.661365549, 6.686839545,
6.712426827, 6.738845633, 6.766577469, 6.793733778, 6.820943,
6.84717925, 6.87388465, 6.900189331, 6.926937272, 6.953351794,
6.978585571, 7.003119365, 7.028923336, 7.054782567, 7.081237514,
7.107467553, 7.13468381, 7.160855945, 7.18767732, 7.214012693,
7.24195315, 7.269535786, 7.295962871, 7.300437635, 7.321886502,
7.350015578, 7.377033547, 7.405497486, 7.433019252, 7.4589913,
7.484224312, 7.509480975, 7.535177543, 7.563174222, 7.589740204,
7.617272127, 7.641436082, 7.669156399, 7.695433766, 7.722645214,
7.75025712, 7.777826579, 7.803764626, 7.830208436, 7.857264015,
7.88457871, 7.913094416, 7.940003639, 7.967392357, 7.992002797,
8.019264826, 8.045657513, 8.071833114, 8.099880142, 8.126450761,
8.154322493, 8.179606484, 8.205665173, 8.233129794, 8.25945939,
8.286011596, 8.314744113, 8.340107029, 8.368131634, 8.395599891,
8.420679344, 8.447308602, 8.474552598, 8.503726279, 8.531367041,
8.558319543, 8.584233956, 8.611475787, 8.637903116, 8.664611492,
8.693746235, 8.720953586, 8.74855903, 8.772456788, 8.799850054,
8.827245834, 8.851809134, 8.8783896, 8.906608132, 8.932489203,
8.95780733, 8.983422846, 9.010221373, 9.0371145, 9.062998996,
9.090731069, 9.115455796, 9.140009666, 9.170371313, 9.196004519,
9.223438862, 9.248818112, 9.2762774, 9.30201158, 9.328802598,
9.357382862, 9.386877104, 9.412552139, 9.441367623, 9.471048822,
9.496103674, 9.525131654, 9.551257306, 9.578897685, 9.605466702,
9.630588252, 9.657389304, 9.685669283, 9.709879332, 9.736881557,
9.762753787, 9.788893738, 9.811176902, 9.838820336, 9.863660837,
9.887261363, 9.910426424, 9.935534192, 9.959137725, 9.988928301,
10.016689538, 10.046404856, 10.072271938, 10.098067293, 10.121528762,
10.148645822, 10.17463838, 10.202164932, 10.227687762, 10.252137493,
10.282389164, 10.307459581, 10.332604473, 10.361376287, 10.386602614,
10.409449749, 10.436629702, 10.463977187, 10.486895626, 10.512856511,
10.53733616, 10.559770896, 10.58748242, 10.615049283, 10.643387185,
10.66792312, 10.690427691, 10.714161557, 10.742950917, 10.767343569,
10.793840573, 10.820144261, 10.845541662, 10.867635802, 10.896811196,
10.921601791, 10.945795136, 10.968988916, 10.994512412, 11.017172413,
11.04649619, 11.07259058, 11.097565145, 11.124979819, 11.148803042,
11.174386067, 11.195239452, 11.219666001, 11.246408367, 11.272924695,
11.296640611, 11.320186017, 11.343780992, 11.366866179, 11.391055968,
11.416943157, 11.436395771, 11.457787482, 11.479790506, 11.505436132,
11.530921162, 11.554059966, 11.578151329, 11.60197661, 11.626205786,
11.650632776, 11.671926738, 11.699000853, 11.722696435, 11.747525054,
11.772679801, 11.796049636, 11.821827528, 11.847982547, 11.87092289,
11.893673892, 11.920151999, 11.940359165, 11.960495619, 11.984791142,
12.009248554, 12.034422178, 12.058971207, 12.082781797, 12.109605083,
12.134145975, 12.158536996, 12.180825092, 12.200972874, 12.227100925,
12.251323934, 12.276436472, 12.301577153, 12.322950293, 12.340865621,
12.366919573, 12.389719163, 12.414106871, 12.439295725, 12.443480346,
12.46642996, 12.490631842, 12.513669305, 12.535410496, 12.562195927,
12.587624772, 12.615840299, 12.638918433, 12.663558176, 12.687115435,
12.702322491, 12.722594963, 12.746093272, 12.767548646, 12.782877104,
12.802928816, 12.825806447, 12.845439364, 12.864280629, 12.884480543,
12.906659973, 12.92500153, 12.947961269, 12.973611654, 12.995645992,
13.018691178, 13.037676747, 13.054941256, 13.078951902, 13.099849563,
13.12333835, 13.14475832, 13.165475422, 13.189297731, 13.212481258,
13.233914782, 13.256440561, 13.277987935, 13.301333984, 13.331785838,
13.358700909, 13.382190949, 13.410263508, 13.434908815, 13.455217681,
13.476509756, 13.500285761, 13.518528902, 13.537942016, 13.554195237,
13.577337106, 13.598641088, 13.618107421, 13.638417538, 13.656885801,
13.672084372, 13.69840726, 13.728668551, 13.752636739, 13.774413846,
13.796929733, 13.819090162, 13.843651382, 13.864561637, 13.888910065,
13.914113522, 13.939477349, 13.955810111, 13.97030462, 13.984911227,
14.007891115, 14.023164362, 14.040083866, 14.060558448, 14.085774699,
14.109218436, 14.126490707, 14.145069439, 14.169762601, 14.181715868,
14.202540263, 14.217780128, 14.233008567, 14.253154286, 14.274288194,
14.302172335, 14.322892528, 14.342527965, 14.359778149, 14.380172919,
14.404278991, 14.422950341, 14.452868755, 14.470781228, 14.485054958,
14.504710581, 14.520918249, 14.545892945, 14.562123624, 14.583582307,
14.607684329, 14.624342841, 14.636170811, 14.657805307, 14.682201507,
14.707681199, 14.719869646, 14.736790538, 14.756476089, 14.777815555,
14.795499413, 14.81261785, 14.833309583, 14.852999413, 14.865302275,
14.894962315, 14.907404186, 14.925273211, 14.937704806, 14.952857268,
14.977251623, 14.997849832, 15.011729869, 15.031781857, 15.04748503,
15.061780164, 15.080766404, 15.105592656, 15.128226757, 15.145430732,
15.170290215, 15.192388708, 15.211049784, 15.23283873, 15.253217988,
15.266225446, 15.292935716, 15.314404681, 15.327898373, 15.344926989,
15.368638632, 15.387262912, 15.409783905, 15.418422658, 15.436405934,
15.459815976, 15.472947025, 15.485421696, 15.503766142, 15.527219161,
15.547216331, 15.566038949, 15.580929982, 15.600547329, 15.616316447,
15.626722849, 15.645141013, 15.659264075, 15.687032684, 15.719573889,
15.732579768, 15.749656457, 15.767014292, 15.779187017, 15.791819948,
15.803886261, 15.825898152, 15.843584285, 15.863395616, 15.877401778,
15.884423287, 15.901379717, 15.928668094, 15.944886607, 15.954386784,
15.969946535, 15.985863241, 15.998362398, 16.007685919, 16.025729738,
16.04021892, 16.064148284, 16.076368286, 16.091682351, 16.10276669,
16.116088157, 16.136769672, 16.15385302, 16.168704029, 16.184044891,
16.187701525, 16.217644701, 16.245031487, 16.264387034, 16.275463646,
16.307597783, 16.316609953, 16.329023567, 16.341326861, 16.357947133,
16.386639317, 16.407427476, 16.429602368, 16.434842005, 16.452636289,
16.468315762, 16.492599673, 16.509973746, 16.527888245, 16.538108131,
16.556026206, 16.579129335, 16.606937789, 16.640404858, 16.663002711,
16.679432007, 16.695640707, 16.714431652, 16.729942776, 16.746744788,
16.753502778, 16.781157035, 16.79481324, 16.820748003, 16.837823976,
16.850038065, 16.8661048, 16.878623257, 16.892773089, 16.927284421,
16.948865986, 16.972418153, 16.983787425, 16.989474628, 17.00791872,
17.023628424, 17.042585588, 17.055870598, 17.077316386, 17.087380085,
17.093416516, 17.104852613, 17.113116066, 17.120064468, 17.128449322,
17.132675956, 17.142984238, 17.159220707, 17.167290994, 17.17829603,
17.185347539, 17.212428949, 17.221453874, 17.239546898, 17.263520964,
17.281117635, 17.289435432, 17.303228437, 17.317145157, 17.322325686,
17.336825569, 17.342777063, 17.354949937, 17.39984682, 17.41940197,
17.428438947, 17.43161463, 17.437093177, 17.466910894, 17.47501107,
17.47962938, 17.488034758, 17.507413819, 17.528000056, 17.532284604,
17.541609361, 17.551432345, 17.578461966, 17.584806714, 17.586877531,
17.60568531, 17.626931921, 17.638038607, 17.643289652, 17.670931426,
17.68772271, 17.716503294, 17.72229762, 17.727878562, 17.752967124,
17.773472251, 17.788099198, 17.804632293, 17.819809897, 17.824314749,
17.835029965, 17.854614989, 17.869458321, 17.882727328, 17.909990014,
17.915060661, 17.922729578, 17.940601792, 17.956437207, 17.958347692,
17.972383091, 17.984372487, 17.985413022, 17.988150593, 18.008273291,
18.010604515, 18.012889652, 18.017162992, 18.044453239, 18.04800334,
18.049529565, 18.064452202, 18.077567715, 18.083380026, 18.090433925,
18.114071365, 18.118137519, 18.126885661, 18.135695877, 18.173946777,
18.203523664, 18.21307026, 18.224693051, 18.230022404, 18.254181917,
18.269022885, 18.271273493, 18.292291594, 18.307318467, 18.307946899,
18.308364555, 18.31630219, 18.353695546, 18.383430851, 18.391208
), "20" = c(2.339e-05, 2.3405e-05, 2.3519e-05, 2.384e-05, 2.3923e-05,
2.4142e-05, 2.4246e-05, 2.4257e-05, 2.4343e-05, 2.4416e-05, 2.4642e-05,
2.5679e-05, 2.5717e-05, 2.6083e-05, 2.6536e-05, 2.6728e-05, 2.6734e-05,
2.7574e-05, 2.7686e-05, 2.7964e-05, 2.8245e-05, 2.8432e-05, 2.8796e-05,
2.9036e-05, 2.934e-05, 2.9467e-05, 2.9914e-05, 3.0242e-05, 3.0672e-05,
3.0864e-05, 3.1147e-05, 3.1562e-05, 3.2249e-05, 3.2256e-05, 3.2484e-05,
3.2812e-05, 3.3198e-05, 3.3593e-05, 3.4357e-05, 3.4684e-05, 3.5087e-05,
3.5286e-05, 3.562e-05, 3.5764e-05, 3.6125e-05, 3.6533e-05, 3.6722e-05,
3.6998e-05, 3.722e-05, 3.7624e-05, 3.796e-05, 3.7985e-05, 3.8263e-05,
3.8579e-05, 3.8722e-05, 3.9789e-05, 4.0393e-05, 4.0811e-05, 4.1467e-05,
4.168e-05, 4.2238e-05, 4.3112e-05, 4.4084e-05, 4.4421e-05, 4.674e-05,
4.6869e-05, 4.7272e-05, 4.7542e-05, 4.7891e-05, 4.8685e-05, 4.995e-05,
5.0405e-05, 5.1703e-05, 5.3503e-05, 5.4029e-05, 5.4503e-05, 5.5373e-05,
5.5964e-05, 5.6303e-05, 5.6712e-05, 5.7272e-05, 5.8721e-05, 5.9176e-05,
6.0052e-05, 6.0469e-05, 6.1234e-05, 6.1672e-05, 6.3025e-05, 6.3718e-05,
6.5087e-05, 6.5352e-05, 6.5864e-05, 6.6471e-05, 6.7252e-05, 6.7714e-05,
6.9565e-05, 7.0392e-05, 7.1075e-05, 7.1609e-05, 7.2223e-05, 7.2343e-05,
7.2936e-05, 7.3738e-05, 7.5771e-05, 7.7717e-05, 7.848e-05, 8.0872e-05,
8.1932e-05, 8.347e-05, 8.3823e-05, 8.4615e-05, 8.5282e-05, 8.5652e-05,
8.6725e-05, 8.7278e-05, 8.7969e-05, 8.8379e-05, 8.9748e-05, 9.0097e-05,
9.1165e-05, 9.2829e-05, 9.4325e-05, 9.4941e-05, 9.5342e-05, 9.6262e-05,
9.7621e-05, 9.8888e-05, 9.9649e-05, 0.00010035, 0.000101335,
0.000102846, 0.000105743, 0.000106659, 0.000107226, 0.000108091,
0.000108636, 0.00010999, 0.000112197, 0.000113056, 0.000113808,
0.000114507, 0.000115255, 0.000116178, 0.000119685, 0.00012142,
0.000122493, 0.000123625, 0.000127134, 0.000128936, 0.000130497,
0.00013231, 0.000133497, 0.000134691, 0.000135658, 0.000137656,
0.000138347, 0.000140268, 0.000141504, 0.0001422, 0.000143731,
0.000145464, 0.000147095, 0.000149566, 0.000150626, 0.000152869,
0.000154157, 0.000155797, 0.000158325, 0.000160346, 0.000161569,
0.000162451, 0.00016381, 0.000165406, 0.000166394, 0.000168142,
0.000169909, 0.000172297, 0.000174682, 0.000176502, 0.000178356,
0.000179216, 0.000182802, 0.000185073, 0.000187006, 0.00018884,
0.000190996, 0.000194179, 0.000197068, 0.000199797, 0.000201743,
0.000204286, 0.000206488, 0.000209936, 0.000213365, 0.000215867,
0.000216835, 0.00021893, 0.000222558, 0.000225039, 0.000230972,
0.000233109, 0.00023623, 0.000239072, 0.000241845, 0.000244689,
0.000249244, 0.000251685, 0.00025466, 0.000258671, 0.000262603,
0.000266788, 0.000269628, 0.000273653, 0.000275816, 0.000278981,
0.000283083, 0.000286895, 0.000289482, 0.000292064, 0.000294195,
0.000296073, 0.000299839, 0.000302434, 0.00030596, 0.000308965,
0.000311424, 0.000314635, 0.000317664, 0.000320551, 0.00032294,
0.00032503, 0.00032869, 0.000332484, 0.000336026, 0.000340555,
0.000343925, 0.000348832, 0.00035428, 0.000357499, 0.000361642,
0.000367418, 0.000371612, 0.000377695, 0.000379808, 0.000384336,
0.000389196, 0.000394718, 0.000399181, 0.000403982, 0.00040926,
0.000414611, 0.000420274, 0.000424611, 0.00042969, 0.000433491,
0.000438909, 0.000444681, 0.000450244, 0.000453902, 0.000459696,
0.000463366, 0.000468344, 0.000474665, 0.000480969, 0.000486801,
0.000492886, 0.000499313, 0.00050593, 0.000512346, 0.000517065,
0.000525186, 0.000533393, 0.00053975, 0.000544525, 0.000549103,
0.0005547, 0.000561344, 0.000568144, 0.000575258, 0.000583671,
0.000587485, 0.000595357, 0.000601312, 0.000609046, 0.000614195,
0.00062035, 0.000627281, 0.000634654, 0.000642, 0.00064834, 0.000655551,
0.000662848, 0.000671058, 0.000680239, 0.000687156, 0.00069556,
0.000704877, 0.000712436, 0.000720719, 0.000727619, 0.000738568,
0.000747564, 0.000754166, 0.000761853, 0.00076895, 0.000780226,
0.000788824, 0.000800374, 0.00080936, 0.000817071, 0.000828763,
0.000837876, 0.000847952, 0.000859091, 0.000868771, 0.00087905,
0.000893344, 0.000905242, 0.000915969, 0.000925892, 0.000937254,
0.000944105, 0.000956744, 0.000967528, 0.000976676, 0.000987207,
0.000998741, 0.001011449, 0.001023135, 0.001035721, 0.001049429,
0.001060503, 0.001073242, 0.001085118, 0.001098214, 0.001108014,
0.001118794, 0.0011277, 0.001137371, 0.001149826, 0.001160378,
0.001161687, 0.001177279, 0.001189955, 0.001206121, 0.001222385,
0.001239739, 0.001253606, 0.00126512, 0.001279893, 0.001290073,
0.001305704, 0.00132096, 0.001335764, 0.001354889, 0.001370482,
0.001389627, 0.001403389, 0.001417448, 0.00143318, 0.001448996,
0.001464217, 0.001483137, 0.001496958, 0.001517504, 0.001534109,
0.001552898, 0.001571364, 0.001592578, 0.00161355, 0.00163087,
0.001648756, 0.001666053, 0.001682487, 0.001700536, 0.001719497,
0.001743259, 0.00176063, 0.001781515, 0.001804343, 0.001823211,
0.001840913, 0.001862139, 0.001879966, 0.001902285, 0.001927559,
0.00195009, 0.001971209, 0.001996912, 0.002023295, 0.002047133,
0.002071516, 0.002095616, 0.002123933, 0.00214877, 0.002175045,
0.002200652, 0.002226837, 0.002251468, 0.002273194, 0.002297723,
0.002321414, 0.002346435, 0.002377351, 0.002403049, 0.002426601,
0.002459711, 0.002490665, 0.002517327, 0.002545358, 0.002576057,
0.002603031, 0.002631347, 0.002661549, 0.002687523, 0.002722455,
0.002754744, 0.002786665, 0.002820427, 0.002851708, 0.002888028,
0.002920125, 0.002954685, 0.002986835, 0.003015703, 0.003049906,
0.003083914, 0.003122154, 0.003155291, 0.003192542, 0.003232372,
0.003265475, 0.003302501, 0.003338421, 0.003372545, 0.003415043,
0.003455903, 0.003498752, 0.003536687, 0.003586019, 0.003629313,
0.003669478, 0.003705867, 0.003748221, 0.003798474, 0.00383786,
0.003882106, 0.003925305, 0.00397289, 0.004024825, 0.004068506,
0.00411455, 0.004158467, 0.004211497, 0.004260341, 0.004309571,
0.004361062, 0.004413631, 0.004464356, 0.004516633, 0.004569218,
0.004620783, 0.004674806, 0.004724921, 0.004771783, 0.004819386,
0.004871086, 0.004925231, 0.004978855, 0.005035453, 0.005092426,
0.005149968, 0.005204531, 0.005274921, 0.005336039, 0.005399834,
0.005462169, 0.005527781, 0.005597183, 0.005658147, 0.005722023,
0.00579476, 0.005863545, 0.005934403, 0.006005717, 0.006072159,
0.006146696, 0.006221127, 0.006291758, 0.006365661, 0.00644214,
0.006521469, 0.006590293, 0.006665381, 0.006738967, 0.006817739,
0.00689996, 0.006972098, 0.007061182, 0.007144484, 0.007228005,
0.007303476, 0.007391813, 0.007479179, 0.007566231, 0.007652663,
0.007743889, 0.007831998, 0.007918137, 0.008009358, 0.008105319,
0.008192745, 0.008295796, 0.008392147, 0.008487878, 0.008599599,
0.00869887, 0.008801272, 0.008902367, 0.008999645, 0.009104777,
0.009208316, 0.00931229, 0.009418154, 0.009519975, 0.009633283,
0.009756424, 0.009861943, 0.009981448, 0.010098873, 0.01021596,
0.010329932, 0.010447395, 0.010570613, 0.010690873, 0.010811883,
0.010937048, 0.011055954, 0.011183922, 0.011317567, 0.011455511,
0.011571288, 0.011597712, 0.011735445, 0.011876069, 0.012009835,
0.012141962, 0.012286348, 0.012428562, 0.012570807, 0.012709545,
0.012857001, 0.013010937, 0.013160388, 0.013311762, 0.013461575,
0.013608056, 0.01376278, 0.013932067, 0.014095637, 0.014259644,
0.01443011, 0.014591499, 0.014751073, 0.014921655, 0.015089325,
0.015276327, 0.015451507, 0.015624237, 0.015809245, 0.01597986,
0.01616866, 0.016351404, 0.016548746, 0.016725605, 0.016904281,
0.017097985, 0.017290808, 0.017479317, 0.017669673, 0.017868601,
0.018066945, 0.01827421, 0.018489972, 0.01869718, 0.018919047,
0.019140176, 0.019351876, 0.019579916, 0.019807261, 0.020036362,
0.020269431, 0.020507983, 0.020745022, 0.020979088, 0.021209967,
0.021435342, 0.021689344, 0.021931172, 0.022192878, 0.022450465,
0.022705099, 0.022968175, 0.023229958, 0.023495024, 0.023759014,
0.024034806, 0.0243045, 0.024572776, 0.024848429, 0.025136363,
0.025428427, 0.025715371, 0.025997647, 0.026296504, 0.026581528,
0.026891179, 0.027197713, 0.027511705, 0.027818345, 0.028133322,
0.028445839, 0.028754167, 0.02908739, 0.029429882, 0.029744995,
0.030089201, 0.030432919, 0.030782308, 0.031122072, 0.031474938,
0.031836713, 0.032185056, 0.03254148, 0.032902541, 0.033286195,
0.033676629, 0.034065455, 0.034463716, 0.034855203, 0.035251733,
0.035642134, 0.036031395, 0.036430142, 0.036820879, 0.037235212,
0.037672619, 0.038087655, 0.038505179, 0.038949088, 0.039389628,
0.039825825, 0.040290519, 0.040737909, 0.041194873, 0.041637885,
0.042122393, 0.042597187, 0.043087469, 0.043572289, 0.044065971,
0.044553933, 0.045036921, 0.045543705, 0.0460437, 0.046561362,
0.047080359, 0.047616819, 0.048152184, 0.048681961, 0.049224882,
0.049773421, 0.050320693, 0.050888911, 0.051469173, 0.052038755,
0.052605626, 0.053175679, 0.053744595, 0.054370734, 0.054979419,
0.05558686, 0.056209296, 0.05682207, 0.057462141, 0.058095073,
0.05874974, 0.059411657, 0.060093836, 0.060778485, 0.061442327,
0.062144185, 0.062822774, 0.063519288, 0.064232733, 0.064942462,
0.065664884, 0.066382536, 0.067119083, 0.067870965, 0.068626289,
0.069383765, 0.070158746, 0.070926574, 0.071710956, 0.07251174,
0.073328441, 0.074158325, 0.074989496, 0.075836998, 0.076660762,
0.077484095, 0.078365573, 0.079252605, 0.080128556, 0.081013423,
0.08190712, 0.082823393, 0.083733689, 0.084625296, 0.085546448,
0.086484958, 0.087431266, 0.088371024, 0.089336105, 0.09031658,
0.091307492, 0.09232661, 0.093345124, 0.094368443, 0.095406477,
0.096427818, 0.097477359, 0.098546625, 0.099630265, 0.100706161,
0.10180417, 0.102902782, 0.104038184, 0.105173492, 0.106310264,
0.107452217, 0.108376928, 0.10863057, 0.109811234, 0.111015422,
0.112221133, 0.113430021, 0.114674723, 0.115909673, 0.117154835,
0.11840225, 0.119670833, 0.120935985, 0.122212891, 0.123530741,
0.124856864, 0.126184276, 0.127492343, 0.128832237, 0.130223208,
0.131625824, 0.133006868, 0.13442335, 0.135869698, 0.137340713,
0.138841154, 0.140335503, 0.141831969, 0.143358334, 0.144893702,
0.146419267, 0.147985139, 0.149560238, 0.15114867, 0.15276661,
0.154400409, 0.156043122, 0.157697237, 0.159385678, 0.161087419,
0.162806765, 0.164536979, 0.166279737, 0.168012197, 0.169777006,
0.171584828, 0.173372519, 0.175190418, 0.17702673, 0.178922294,
0.180776327, 0.182682813, 0.184552776, 0.186520582, 0.188461531,
0.190423978, 0.192408085, 0.194412914, 0.19644324, 0.198508694,
0.200561328, 0.20261208, 0.204737194, 0.206841619, 0.208997399,
0.211188408, 0.213357869, 0.215574724, 0.217803841, 0.220080959,
0.222354372, 0.224648362, 0.227019283, 0.22938731, 0.231744104,
0.234097746, 0.236522643, 0.238997896, 0.241483289, 0.244019516,
0.246551936, 0.249084591, 0.251665068, 0.254252195, 0.25684652,
0.259492087, 0.262135716, 0.264808568, 0.267489293, 0.270214725,
0.272999038, 0.275792356, 0.27863504, 0.281482482, 0.284363093,
0.287258414, 0.290203288, 0.29317944, 0.293826844, 0.29616221,
0.299192898, 0.302240066, 0.305281435, 0.308381827, 0.311509736,
0.314657921, 0.317836665, 0.32105776, 0.324306586, 0.327561156,
0.330873453, 0.334190044, 0.337528781, 0.340935399, 0.344359048,
0.347890757, 0.351387219, 0.354944441, 0.358510066, 0.362186203,
0.365851251, 0.369525307, 0.373191773, 0.376988659, 0.377748401,
0.380735374, 0.384561063, 0.388414747, 0.392337689, 0.396315202,
0.400311886, 0.404228721, 0.40833113, 0.412493166, 0.416681273,
0.420882997, 0.425121007, 0.429427971, 0.433759108, 0.438088001,
0.442461013, 0.446884241, 0.451413471, 0.455937187, 0.460535197,
0.465172232, 0.469881828, 0.474645011, 0.479402191, 0.484251849,
0.489189729, 0.494136254, 0.49912714, 0.504144297, 0.509278843,
0.514410423, 0.519602532, 0.524866162, 0.530174704, 0.535528243,
0.537816684, 0.540934514, 0.546425184, 0.551922901, 0.557543067,
0.563271729, 0.569044849, 0.574852965, 0.580722597, 0.586619961,
0.592601611, 0.598679633, 0.604809273, 0.610990893, 0.617234367,
0.623623711, 0.630112693, 0.636656802, 0.64326281, 0.649931071,
0.656627476, 0.663505984, 0.670453392, 0.67743513, 0.68454151,
0.691805353, 0.694798815, 0.699120504, 0.706477717, 0.713961322,
0.721558051, 0.729265449, 0.737043516, 0.744968124, 0.753009616,
0.761115039, 0.769340425, 0.777691134, 0.786228494, 0.794891402,
0.803683698, 0.812598052, 0.821660237, 0.830832196, 0.840210557,
0.849740367, 0.857388683, 0.874716476, 0.884235722, 0.893813601,
0.903287621, 0.912637989, 0.921935423, 0.931240899, 0.940449343,
0.949705872, 0.958889692, 0.968058628, 0.97713591, 0.986170158,
0.995173983, 1.004125292, 1.013068276, 1.021947866, 1.030862678,
1.03604984, 1.039790087, 1.048584894, 1.057391919, 1.066142826,
1.074847014, 1.083612736, 1.092270991, 1.100994856, 1.109737772,
1.118367596, 1.127023417, 1.135589981, 1.144272629, 1.152910687,
1.161473805, 1.170045775, 1.178651469, 1.187284762, 1.195774714,
1.204277009, 1.212825589, 1.221415025, 1.229864603, 1.238301115,
1.246738097, 1.251640141, 1.25527671, 1.263788549, 1.272326851,
1.280882088, 1.289431317, 1.2979597, 1.306578433, 1.315100907,
1.323661512, 1.332295528, 1.340889365, 1.349480031, 1.35806656,
1.366746197, 1.375441231, 1.384043917, 1.39280746, 1.401580923,
1.410335206, 1.419058664, 1.427864042, 1.436757488, 1.445669425,
1.45453694, 1.463450957, 1.472363269, 1.481336842, 1.490312328,
1.499417952, 1.508596269, 1.517805334, 1.52699762, 1.536190454,
1.545407034, 1.554857576, 1.562342881, 1.564252255, 1.573752308,
1.583199353, 1.592720477, 1.602424362, 1.612178836, 1.621968177,
1.631836477, 1.641871125, 1.651933243, 1.662010749, 1.672284249,
1.682634414, 1.693051014, 1.703527587, 1.714040797, 1.724646462,
1.735356464, 1.746223147, 1.757178521, 1.768181571, 1.779196135,
1.790464674, 1.802039274, 1.813701076, 1.822973242, 1.825619447,
1.837392048, 1.849344899, 1.861660605, 1.873745524, 1.886098001,
1.898672651, 1.911288695, 1.924322852, 1.937583825, 1.950852251,
1.964371635, 1.97794849, 1.991581829, 2.00560285, 2.019988902,
2.034397638, 2.04897311, 2.063485939, 2.078437491, 2.093667008,
2.108891816, 2.12444023, 2.140318565, 2.156264405, 2.172323878,
2.188793841, 2.205860861, 2.222711416, 2.239830062, 2.257248748,
2.274628294, 2.292495423, 2.310419448, 2.328598057, 2.346991589,
2.36578668, 2.38486793, 2.404056501, 2.4231701, 2.442625995,
2.462442751, 2.482432278, 2.50275071, 2.522986138, 2.543397005,
2.564021752, 2.585059927, 2.605903202, 2.627053127, 2.648280631,
2.670018483, 2.691818676, 2.713512777, 2.735391631, 2.757648891,
2.779921094, 2.802613231, 2.825626353, 2.848179486, 2.871805204,
2.89479692, 2.918204833, 2.94201004, 2.96586424, 2.989724786,
3.013804075, 3.037907762, 3.062029206, 3.086599924, 3.111724065,
3.136669961, 3.161716291, 3.186907067, 3.212605345, 3.237794427,
3.263759355, 3.289602086, 3.316231133, 3.342794, 3.369178829,
3.395305289, 3.422612993, 3.449983365, 3.477347668, 3.504277516,
3.531994593, 3.560155973, 3.588601519, 3.616883381, 3.645000352,
3.673504631, 3.701783851, 3.730631256, 3.760491666, 3.789469628,
3.795424042, 3.818307975, 3.847981486, 3.877909623, 3.907435929,
3.937826965, 3.967640744, 3.998306133, 4.02919948, 4.060607997,
4.091409204, 4.123320955, 4.153898734, 4.186216501, 4.217594688,
4.249388755, 4.282055915, 4.313644767, 4.346214253, 4.379265777,
4.411342987, 4.44333187, 4.476605079, 4.510131826, 4.54373602,
4.57640653, 4.609826122, 4.644098443, 4.678280048, 4.713180605,
4.74792371, 4.782090518, 4.816040729, 4.851225226, 4.886825977,
4.923146276, 4.957622406, 4.993794145, 5.029810976, 5.06662141,
5.102397214, 5.139041554, 5.176901205, 5.214184041, 5.250617479,
5.28853164, 5.326537716, 5.36373022, 5.401634472, 5.440956866,
5.478873672, 5.517080716, 5.556388903, 5.594919387, 5.633123355,
5.672080949, 5.710414022, 5.749796311, 5.791295074, 5.832352006,
5.872867261, 5.91364419, 5.954763551, 5.996507901, 6.03786911,
6.079425739, 6.121443477, 6.162680164, 6.204111513, 6.247054799,
6.28827861, 6.33084096, 6.37351583, 6.416250808, 6.459463569,
6.502197556, 6.545830325, 6.590496915, 6.633255961, 6.676262843,
6.719698052, 6.764827923, 6.808157607, 6.855661419, 6.899436227,
6.944256875, 6.987901135, 7.033203009, 7.079242389, 7.122810322,
7.167576581, 7.21353527, 7.260160366, 7.306462379, 7.353806179,
7.401272586, 7.449156804, 7.499190972, 7.546011146, 7.592959319,
7.640597528, 7.690651535, 7.737248491, 7.78685522, 7.834339849,
7.883073211, 7.933447607, 7.98325094, 8.031516636, 8.081555105,
8.130155592, 8.180966193, 8.230078823, 8.280139673, 8.332748425,
8.383959936, 8.434930378, 8.486984608, 8.53760183, 8.58799115,
8.639458286, 8.689450855, 8.739971736, 8.793609389, 8.844581607,
8.896138883, 8.948140736, 9.003183891, 9.057967096, 9.111398241,
9.166120076, 9.22018651, 9.27418226, 9.329430211, 9.382493946,
9.439149577, 9.49573727, 9.550743725, 9.606331254, 9.663390452,
9.718246069, 9.776090761, 9.831876873, 9.890579365, 9.945222269,
10.002728222, 10.057169048, 10.11586651, 10.172180325, 10.229874135,
10.289760473, 10.348630989, 10.407589922, 10.467793926, 10.527776161,
10.588986864, 10.647987097, 10.710776377, 10.775031058, 10.835754433,
10.895623744, 10.956059789, 11.015282277, 11.077751909, 11.140167267,
11.207365257, 11.267658924, 11.332493196, 11.391542151, 11.456294093,
11.519084635, 11.581049603, 11.639783087, 11.706117869, 11.767398353,
11.827717528, 11.887731868, 11.950975907, 12.014667432, 12.077083721,
12.141403213, 12.205786774, 12.270109131, 12.33676559, 12.401615137,
12.462450912, 12.524564161, 12.593733005, 12.657392189, 12.722473803,
12.787563703, 12.856800013, 12.922154564, 12.986755941, 13.051111219,
13.120267386, 13.189035996, 13.25472057, 13.323065251, 13.389254145,
13.455927473, 13.471002791, 13.52148907, 13.585255758, 13.652588717,
13.722205925, 13.788030681, 13.855530505, 13.922422613, 13.99554954,
14.062133903, 14.132630878, 14.206508156, 14.273228462, 14.345506667,
14.415980783, 14.488636895, 14.556675596, 14.62509698, 14.697948593,
14.772992934, 14.851940372, 14.921927961, 14.99327583, 15.0610092,
15.137370137, 15.216545286, 15.294928174, 15.366420092, 15.443652691,
15.519493707, 15.591270495, 15.666925518, 15.743038129, 15.815787288,
15.894855141, 15.971473067, 16.047721144, 16.128735327, 16.207910522,
16.286405014, 16.358932323, 16.436692413, 16.514961299, 16.587431397,
16.669240121, 16.738499698, 16.821075838, 16.900505671, 16.97945839,
17.056900521, 17.134348442, 17.216898794, 17.298216568, 17.375861096,
17.443966091, 17.52228353, 17.603146768, 17.681008055, 17.755242393,
17.833688368, 17.925314748, 18.008131174, 18.091921457, 18.185178465,
18.265290717, 18.33581754, 18.41497471, 18.487615993, 18.571171853,
18.653272444, 18.740173574, 18.833230713, 18.91918716, 18.996543874,
19.074665595, 19.163036008, 19.244447756, 19.326197582, 19.408173683,
19.481950313, 19.564460567, 19.658002924, 19.742945106, 19.832784007,
19.907186859, 19.994121041, 20.081694683, 20.167293294, 20.249475139,
20.328855845, 20.410350949, 20.487237442, 20.578656401, 20.665931734,
20.747391218, 20.825484342, 20.919613889, 21.007337383, 21.099204197,
21.195795313, 21.292173681, 21.389240337, 21.478477543, 21.573860303,
21.654448253, 21.742024213, 21.833663461, 21.938444372, 22.035568131,
22.125424619, 22.219256144, 22.300889982, 22.394669637, 22.461271105,
22.556483674, 22.654916987, 22.739718643, 22.834617363, 22.929554319,
23.023448758, 23.112076065, 23.193653067, 23.277642903, 23.372910817,
23.463352544, 23.552974204, 23.650688772, 23.750781452, 23.844679697,
23.934138959, 24.026694627, 24.132209405, 24.229567347, 24.319678695,
24.414371836, 24.504201578, 24.600448178, 24.699767017, 24.785139421,
24.877299094, 24.97042491, 25.070560177, 25.160042818, 25.273716803,
25.363460459, 25.445533792, 25.529326342, 25.628124491, 25.718457103,
25.810918646, 25.914867151, 26.018391197, 26.109077669, 26.209292305,
26.316204992, 26.43347714, 26.537757456, 26.627747223, 26.719957392,
26.824524947, 26.922383026, 27.030461588, 27.13204212, 27.241678778,
27.343616155, 27.457384224, 27.557998354, 27.658225678, 27.765644539,
27.850015718, 27.943389747, 28.056837429, 28.15776089, 28.242031954,
28.330707643, 28.431276204, 28.54031006, 28.646446023, 28.728844086,
28.810809172, 28.912473875, 29.026973914, 29.14943742, 29.239744083,
29.370191334, 29.473377094, 29.586959512, 29.679112999, 29.781706395,
29.90447304, 29.994376328, 30.126170324, 30.261980807, 30.373545934,
30.490150798, 30.58719532, 30.688695565, 30.800436634, 30.912027644,
31.016198804, 31.135852095, 31.15864159, 31.245223469, 31.343737758,
31.457105126, 31.565076562, 31.650272932, 31.73391576, 31.817578733,
31.917155969, 32.019105826, 32.123608831, 32.250400939, 32.360557049,
32.443320653, 32.547909497, 32.675847934, 32.780616103, 32.885735022,
33.001344841, 33.12624878, 33.223178967, 33.315102097, 33.442334165,
33.561476868, 33.694190738, 33.813764846, 33.925246699, 34.02232781,
34.141864365, 34.272526733, 34.376800813, 34.46198071, 34.547925389,
34.626621289, 34.723703916, 34.831913238, 34.96496818, 35.052961046,
35.16095201, 35.274157906, 35.40094168, 35.520629261, 35.642373329,
35.765110445, 35.86589917, 35.976707103, 36.076917958, 36.203829863,
36.350360384, 36.469534429, 36.587213311, 36.714732389, 36.842627291,
36.932219477, 37.047850134, 37.156513669, 37.274493026, 37.386979051,
37.499979877, 37.625176836, 37.736875939, 37.855723154, 37.993927329,
38.097156657, 38.191041973, 38.311091287, 38.430207403, 38.55559921,
38.650157959, 38.771023082, 38.873584718, 38.964402379, 39.05726144,
39.164824911, 39.310484701, 39.387752358, 39.484717593, 39.598244566,
39.74324648, 39.855190813, 39.969745552, 40.054085474, 40.160430516,
40.272295593, 40.376084878, 40.453484337, 40.569611241, 40.709021248,
40.847103888, 40.934904152, 41.031348196, 41.144710494, 41.258961353,
41.36669598, 41.461662051, 41.566329229, 41.65678098, 41.747095099,
41.876450405, 41.984675724, 42.108939254, 42.255511826, 42.355562916,
42.439963275, 42.545454881, 42.621295454, 42.749329503, 42.881820164,
43.0104063, 43.104953486, 43.184360606, 43.354029253, 43.4749484,
43.555393749, 43.666318054, 43.85606742, 44.051682984, 44.158147112,
44.29080016, 44.390965465, 44.556286766, 44.660895651, 44.788904289,
44.883199885, 45.026117579, 45.118583967, 45.259749897, 45.382219634,
45.521079058, 45.672070654, 45.838208999, 45.918770199, 46.041714873,
46.16997643, 46.275198204, 46.374454715, 46.448547857, 46.605590043,
46.750351285, 46.874463037, 46.996140729, 47.073178205, 47.165728878,
47.242088577, 47.301619054, 47.405367383, 47.467486261, 47.561767926,
47.691577344, 47.77367005, 47.871416553, 48.018263676, 48.198653263,
48.431707174, 48.54405848, 48.656490428, 48.758056134, 48.927771077,
49.002687604, 49.100000123, 49.23731429, 49.334258713, 49.565421015,
49.721288162, 49.794455232, 50.071743513, 50.211251045, 50.26765642,
50.399592256, 50.460774767, 50.57887838, 50.693753386, 50.761406703,
50.880252381, 50.959979139, 51.043516412, 51.231881356, 51.313760248,
51.430131009, 51.480152174, 51.557131414, 51.602928158, 51.669320811,
51.7946554, 51.981541947, 52.094358865, 52.225604631, 52.298685824,
52.402248733, 52.483023082, 52.595807902, 52.65046407, 52.76459214,
52.864390749, 53.06495084, 53.215414354, 53.348200245, 53.456550755,
53.660737398, 53.751834414, 53.955516983, 54.075093722, 54.246091425,
54.392003756, 54.553048569, 54.63516793, 54.770461387, 54.86377939,
54.970744406, 55.067825073, 55.126943103, 55.250223992, 55.34308901,
55.524397533, 55.652405424, 55.749573112, 55.951799281, 56.032670328,
56.227230719, 56.271589707, 56.329511434, 56.460676402, 56.551308679,
56.715289656, 56.800982473, 56.878069887, 56.978621317, 57.232207293,
57.271853826, 57.349112858, 57.516311356, 57.576940989, 57.744219975,
57.825553585, 57.976497341, 58.136213218, 58.183616201, 58.341376554,
58.408162613, 58.598549671, 58.682405351, 58.764242023, 58.85858274,
59.187076736, 59.330261934, 59.626221151, 59.895095069, 59.958164774,
60.122878873, 60.349044676, 60.47565427, 60.592016107, 60.778839491,
60.960887373, 61.084134613, 61.136246869, 61.305553797, 61.427402164,
61.523733177, 61.671908305, 61.76857039, 62.066495252, 62.138515119,
62.156065799, 62.229197391, 62.354830881, 62.429470891, 62.560546944,
62.636425633, 62.765326863, 62.803069412, 62.811369954, 62.835718635,
63.266259913, 63.319050644, 63.344072504, 63.413235799, 63.497480383,
63.668026913, 63.884742433, 63.958760583, 64.032977971, 64.207332582,
64.25362591, 64.275129345, 64.312530591, 64.377556522, 64.579393554,
64.589683811, 64.77986858, 64.832392144, 65.001633281, 65.059906486,
65.191877692, 65.287329894, 65.360237705, 65.733058624, 65.76350781,
65.994475666, 66.159244558, 66.26318932, 66.316439295, 66.48579639,
66.677906823, 66.79474387, 66.873170929, 67.130016676, 67.244180298,
67.276847579, 67.367319089, 67.613013371, 67.708809602, 67.786384978,
68.002444095, 68.071334931, 68.182617885, 68.190615455, 68.400271619,
68.486591771, 68.496685543, 68.526993689, 68.688526826, 68.700388882,
68.802851625, 68.95435695, 68.959541049, 69.091487863, 69.166426609,
69.19844385, 69.205142901, 69.483726806, 69.568686928, 69.643046114,
69.726188411, 69.754310044, 69.777047349, 70.211107473, 70.213906104,
70.516982926, 70.533103714, 70.563754305), "35" = c(2.114e-05,
2.1781e-05, 2.2061e-05, 2.2124e-05, 2.2253e-05, 2.2254e-05, 2.2396e-05,
2.3064e-05, 2.3541e-05, 2.3631e-05, 2.3657e-05, 2.373e-05, 2.3929e-05,
2.3956e-05, 2.4362e-05, 2.4477e-05, 2.4584e-05, 2.4892e-05, 2.4969e-05,
2.5495e-05, 2.5824e-05, 2.6079e-05, 2.6333e-05, 2.6435e-05, 2.6977e-05,
2.7048e-05, 2.7415e-05, 2.7496e-05, 2.7716e-05, 2.788e-05, 2.8463e-05,
2.9027e-05, 2.9361e-05, 2.959e-05, 3.0133e-05, 3.0694e-05, 3.0758e-05,
3.1354e-05, 3.2105e-05, 3.2777e-05, 3.3018e-05, 3.316e-05, 3.3847e-05,
3.4293e-05, 3.4605e-05, 3.5464e-05, 3.624e-05, 3.6481e-05, 3.6592e-05,
3.6659e-05, 3.6903e-05, 3.7036e-05, 3.7254e-05, 3.7709e-05, 3.781e-05,
3.7987e-05, 3.8939e-05, 3.9246e-05, 3.9604e-05, 4.0073e-05, 4.0501e-05,
4.0914e-05, 4.1032e-05, 4.1443e-05, 4.1608e-05, 4.1703e-05, 4.2201e-05,
4.2601e-05, 4.2994e-05, 4.3796e-05, 4.4617e-05, 4.4944e-05, 4.5144e-05,
4.5247e-05, 4.557e-05, 4.5754e-05, 4.6546e-05, 4.7257e-05, 4.7723e-05,
4.7909e-05, 4.8177e-05, 4.9148e-05, 5.0361e-05, 5.0544e-05, 5.1285e-05,
5.1495e-05, 5.2461e-05, 5.3174e-05, 5.3646e-05, 5.4307e-05, 5.5185e-05,
5.5533e-05, 5.5974e-05, 5.662e-05, 5.7194e-05, 5.8013e-05, 5.8547e-05,
5.912e-05, 5.9531e-05, 6.0224e-05, 6.0501e-05, 6.1098e-05, 6.1591e-05,
6.2106e-05, 6.2821e-05, 6.3693e-05, 6.4387e-05, 6.5381e-05, 6.5787e-05,
6.659e-05, 6.7917e-05, 6.8511e-05, 6.9024e-05, 6.9476e-05, 7.0682e-05,
7.1087e-05, 7.2354e-05, 7.308e-05, 7.4171e-05, 7.4725e-05, 7.6629e-05,
7.7305e-05, 7.8525e-05, 7.8927e-05, 8.0109e-05, 8.1002e-05, 8.1777e-05,
8.4199e-05, 8.5007e-05, 8.5459e-05, 8.6125e-05, 8.7182e-05, 8.8011e-05,
8.846e-05, 8.9655e-05, 9.0504e-05, 9.1488e-05, 9.2703e-05, 9.4129e-05,
9.4572e-05, 9.7239e-05, 9.8316e-05, 9.9037e-05, 0.000100048,
0.000101252, 0.000103677, 0.000105845, 0.000106832, 0.000109192,
0.000109844, 0.000110775, 0.000112322, 0.000113407, 0.00011415,
0.000116203, 0.000117297, 0.000119293, 0.000120244, 0.000121051,
0.000123352, 0.000124758, 0.000126406, 0.000128835, 0.000130681,
0.000132443, 0.000134748, 0.00013539, 0.000136779, 0.000137584,
0.000138953, 0.000140247, 0.000141336, 0.000142559, 0.000144217,
0.000146139, 0.000148278, 0.000150361, 0.000151848, 0.000154325,
0.000155679, 0.000156969, 0.00016169, 0.000163323, 0.000165206,
0.000167166, 0.000168764, 0.000171389, 0.000172961, 0.000174341,
0.000176685, 0.000178332, 0.000179699, 0.000182047, 0.000183618,
0.00018595, 0.000188893, 0.000190188, 0.000192094, 0.00019401,
0.000196609, 0.000199604, 0.000202132, 0.000205318, 0.000208665,
0.000210535, 0.000212923, 0.000215495, 0.000218003, 0.000219436,
0.00022197, 0.000223606, 0.000227475, 0.000228921, 0.000231646,
0.000234884, 0.00023801, 0.000241212, 0.000243356, 0.000247597,
0.000250032, 0.00025212, 0.00025487, 0.000257546, 0.000260209,
0.000263213, 0.000266572, 0.000269467, 0.000272967, 0.000275812,
0.00027978, 0.000282979, 0.000286577, 0.000289937, 0.000292627,
0.000297065, 0.000299549, 0.000302718, 0.000305201, 0.000310255,
0.000314136, 0.000317372, 0.00032169, 0.00032672, 0.000331448,
0.000334772, 0.0003398, 0.000342713, 0.000346923, 0.000350486,
0.000354466, 0.000359276, 0.000362608, 0.000366473, 0.000369092,
0.000373363, 0.000378225, 0.000380822, 0.000387069, 0.000392695,
0.000397891, 0.00040449, 0.000408574, 0.000411725, 0.00041602,
0.000420534, 0.00042417, 0.000429898, 0.000436294, 0.000441999,
0.000445965, 0.000452686, 0.000459505, 0.000466045, 0.000471354,
0.00047838, 0.000483853, 0.000489381, 0.000493843, 0.000499016,
0.000504524, 0.000511014, 0.0005171, 0.000524768, 0.000530954,
0.000537867, 0.000544043, 0.000550203, 0.000555852, 0.000563872,
0.000571215, 0.000579275, 0.000586291, 0.000592737, 0.000598307,
0.000604883, 0.000612681, 0.000619659, 0.000628705, 0.000635713,
0.000643223, 0.000651932, 0.000658816, 0.000666747, 0.000674472,
0.000680254, 0.00068761, 0.000695816, 0.000703018, 0.000709745,
0.000718531, 0.000728707, 0.000737885, 0.000747474, 0.000754817,
0.000764154, 0.000772726, 0.000783514, 0.000792349, 0.000800689,
0.000810534, 0.000818805, 0.00082905, 0.000836133, 0.000848171,
0.000857757, 0.000868244, 0.000881078, 0.000890269, 0.000899186,
0.000908361, 0.000917075, 0.000928576, 0.000938444, 0.000948362,
0.000959489, 0.00097066, 0.000982957, 0.00099203, 0.001002715,
0.001015314, 0.001024098, 0.001026735, 0.001042215, 0.00105715,
0.0010686, 0.001080544, 0.001094402, 0.001106417, 0.001119403,
0.001130924, 0.001144487, 0.00115638, 0.001172378, 0.001184661,
0.001197408, 0.001210626, 0.001225222, 0.001238577, 0.001252773,
0.001269318, 0.001281179, 0.00129669, 0.001310345, 0.001324101,
0.00133987, 0.001355296, 0.001368161, 0.001384103, 0.001402238,
0.00141546, 0.001430107, 0.001447049, 0.001467068, 0.00148263,
0.001499145, 0.001519187, 0.001538268, 0.001556218, 0.001572607,
0.001588914, 0.001607167, 0.001623399, 0.001638022, 0.001659139,
0.001680786, 0.00170425, 0.001722928, 0.001742661, 0.00176218,
0.001781692, 0.001800731, 0.001820189, 0.001845206, 0.00186924,
0.001894827, 0.001919658, 0.00194163, 0.001966641, 0.001987364,
0.002012574, 0.002037001, 0.002064755, 0.002092246, 0.0021143,
0.002136793, 0.002166385, 0.00219488, 0.002217065, 0.002239843,
0.002261637, 0.002287464, 0.002314566, 0.002340055, 0.002367317,
0.002396911, 0.002428037, 0.002455153, 0.002486752, 0.002515545,
0.002544427, 0.002572581, 0.002607963, 0.00263857, 0.002669137,
0.002700921, 0.00273004, 0.002759116, 0.002790789, 0.002823554,
0.002855719, 0.002889567, 0.002920933, 0.002956517, 0.002986238,
0.003018853, 0.003058631, 0.003090487, 0.003125742, 0.003162849,
0.003198021, 0.00323688, 0.003269633, 0.003307325, 0.003343699,
0.003380198, 0.003416259, 0.0034584, 0.003498564, 0.003544599,
0.003589042, 0.003630074, 0.003670961, 0.003709781, 0.003749282,
0.003794265, 0.003840086, 0.00388479, 0.003932327, 0.003976913,
0.004022136, 0.004074474, 0.004122794, 0.004169066, 0.004221604,
0.004270521, 0.004326132, 0.004377325, 0.004430089, 0.004481328,
0.004530429, 0.004584875, 0.0046405, 0.004690349, 0.004744396,
0.004799938, 0.004863017, 0.004918982, 0.004980464, 0.005032324,
0.005085539, 0.005143982, 0.005200816, 0.005261916, 0.005325935,
0.00538451, 0.005451245, 0.005518099, 0.005582301, 0.005647679,
0.005714775, 0.005777057, 0.005846814, 0.005918197, 0.005987433,
0.006056938, 0.006132189, 0.006199337, 0.006268819, 0.006337558,
0.006409757, 0.006479938, 0.006557962, 0.006638052, 0.006721134,
0.006796081, 0.006880799, 0.006964199, 0.007042618, 0.007126583,
0.007210164, 0.007300198, 0.007384714, 0.00747379, 0.007562205,
0.007647586, 0.00774033, 0.007830959, 0.007924905, 0.008014316,
0.008101219, 0.008194504, 0.008295678, 0.008395303, 0.00848982,
0.008589168, 0.008685489, 0.008778586, 0.008870953, 0.008975537,
0.009074746, 0.009183275, 0.009285707, 0.009394189, 0.009498362,
0.009607109, 0.009709715, 0.009824655, 0.009931608, 0.010045919,
0.010169999, 0.010292323, 0.010388806, 0.010413833, 0.010530703,
0.010650641, 0.010781765, 0.010912008, 0.011035159, 0.011162201,
0.011300925, 0.011429215, 0.011560107, 0.011688707, 0.011824651,
0.01196384, 0.012097182, 0.012246367, 0.01238869, 0.012537509,
0.012686415, 0.012840933, 0.012989611, 0.013140835, 0.0133002,
0.013454433, 0.013605412, 0.013759341, 0.013911729, 0.014063095,
0.014234343, 0.014400527, 0.014565745, 0.014726534, 0.014901144,
0.015075288, 0.015247174, 0.01543053, 0.015619522, 0.01579394,
0.015999384, 0.016187559, 0.016377075, 0.016565359, 0.016756173,
0.016951918, 0.017149043, 0.017349244, 0.01754551, 0.017739392,
0.017945834, 0.018154371, 0.0183603, 0.018572273, 0.018789974,
0.019005376, 0.019223781, 0.01944745, 0.019668082, 0.019897809,
0.020129674, 0.020357341, 0.020588249, 0.020815409, 0.021059415,
0.021307016, 0.021567119, 0.021815827, 0.022064732, 0.022319336,
0.022574063, 0.022825881, 0.02308199, 0.023344745, 0.023620929,
0.023906975, 0.024186046, 0.024459151, 0.024738315, 0.025018738,
0.025308047, 0.025600347, 0.025895505, 0.026194182, 0.026486324,
0.0267933, 0.027092387, 0.027417007, 0.027746206, 0.028059214,
0.02839247, 0.028722516, 0.029069087, 0.029405191, 0.029742434,
0.030073605, 0.030421206, 0.030762155, 0.031117217, 0.031468066,
0.031848664, 0.032210384, 0.03258932, 0.032969728, 0.033342029,
0.033711699, 0.034106108, 0.034505807, 0.034901405, 0.035287375,
0.03569849, 0.036091597, 0.036513184, 0.036929853, 0.037338196,
0.03778477, 0.03821308, 0.03864263, 0.039082521, 0.039542887,
0.040020474, 0.040477383, 0.040919613, 0.04138279, 0.041882714,
0.042351625, 0.042854724, 0.043350223, 0.043824778, 0.044310918,
0.044813869, 0.045350285, 0.045867652, 0.046385242, 0.046921857,
0.047447281, 0.04798214, 0.048538218, 0.049102894, 0.049678017,
0.050248628, 0.050824256, 0.051421315, 0.051993922, 0.052589395,
0.053194381, 0.053823054, 0.054437535, 0.055059517, 0.05567628,
0.056305334, 0.056936258, 0.057572288, 0.058229565, 0.058892945,
0.059556461, 0.060226178, 0.06090877, 0.061589757, 0.062308959,
0.063019989, 0.063738345, 0.064458579, 0.065191332, 0.065947626,
0.066689774, 0.067456172, 0.068220271, 0.069021765, 0.069796576,
0.070580651, 0.071392749, 0.072189252, 0.073016552, 0.07386058,
0.074694941, 0.075527431, 0.076376865, 0.077216452, 0.078100868,
0.078983589, 0.07987384, 0.08075131, 0.081680425, 0.082625606,
0.083568408, 0.084527497, 0.085496962, 0.086463912, 0.087449316,
0.088448917, 0.089465907, 0.090494069, 0.091502136, 0.092534577,
0.093599424, 0.094647955, 0.095766341, 0.096849178, 0.097944506,
0.099028783, 0.100170136, 0.101294736, 0.102201451, 0.102435915,
0.103605236, 0.104786181, 0.105964586, 0.107172469, 0.10839161,
0.109627594, 0.110866906, 0.112114411, 0.113388161, 0.114671249,
0.115971619, 0.117291206, 0.118623001, 0.119957505, 0.121289293,
0.122641322, 0.124061491, 0.125445588, 0.126830621, 0.128239542,
0.129667575, 0.131137468, 0.132619484, 0.134111252, 0.135582601,
0.137117821, 0.138642963, 0.140184452, 0.141776441, 0.143372654,
0.144984607, 0.146586603, 0.148226638, 0.149872426, 0.151554143,
0.153270084, 0.155013612, 0.156737894, 0.158482422, 0.160266537,
0.162060887, 0.163896654, 0.165742032, 0.167602855, 0.169488552,
0.171369521, 0.173275891, 0.175222383, 0.177186811, 0.179136524,
0.181148268, 0.183188715, 0.185210676, 0.187288837, 0.189395399,
0.191490935, 0.193639864, 0.195787246, 0.197959621, 0.200157277,
0.202425781, 0.204664006, 0.206923017, 0.209226614, 0.211569279,
0.213940919, 0.216332457, 0.218781201, 0.221219147, 0.22371581,
0.226194256, 0.228733434, 0.231316541, 0.233898237, 0.236507663,
0.239128454, 0.241793658, 0.244450769, 0.24717139, 0.249951817,
0.252779547, 0.255588122, 0.258427758, 0.261283255, 0.264205226,
0.267111755, 0.270067082, 0.273067169, 0.276135302, 0.279203586,
0.282309698, 0.285436517, 0.288616533, 0.291800778, 0.295043162,
0.295770828, 0.298346171, 0.30160941, 0.304920859, 0.308298726,
0.311699983, 0.315147279, 0.318632824, 0.322170219, 0.325717784,
0.329335287, 0.332982259, 0.33666076, 0.340378574, 0.344191876,
0.348053362, 0.351948503, 0.355833332, 0.359769664, 0.363776396,
0.36782594, 0.371912097, 0.376076539, 0.380245539, 0.38449416,
0.388742747, 0.389637178, 0.393073995, 0.397435358, 0.401874373,
0.406372738, 0.410901252, 0.415421142, 0.42003791, 0.424714686,
0.429450784, 0.434226093, 0.439051137, 0.443876081, 0.448793313,
0.453804835, 0.458846941, 0.463991437, 0.469193828, 0.474452066,
0.479744209, 0.485124383, 0.490573211, 0.496116947, 0.501678815,
0.507328628, 0.513060529, 0.51882964, 0.524656954, 0.53059892,
0.536553843, 0.542613844, 0.548801972, 0.555014507, 0.561340049,
0.567715849, 0.574188549, 0.57691988, 0.58070068, 0.587269454,
0.593951434, 0.60075521, 0.60761429, 0.614557521, 0.621668338,
0.628749378, 0.635992253, 0.643312781, 0.65076945, 0.658321331,
0.666010468, 0.673701852, 0.681592774, 0.689542891, 0.697591688,
0.70574286, 0.714012458, 0.722311617, 0.730872027, 0.739593419,
0.748329786, 0.757174293, 0.766147987, 0.769884316, 0.775235075,
0.784488493, 0.793945775, 0.803539288, 0.813247181, 0.823028037,
0.833056253, 0.843224836, 0.853620284, 0.864166658, 0.874829559,
0.885634043, 0.896641973, 0.907850993, 0.919266319, 0.930906055,
0.942823929, 0.954875911, 0.967203378, 0.977115098, 0.999609808,
1.012039651, 1.024403826, 1.036694585, 1.048943082, 1.061216094,
1.073458622, 1.085572186, 1.097646687, 1.109749821, 1.121685141,
1.133766871, 1.145626288, 1.157554086, 1.169422589, 1.181149193,
1.192886052, 1.20458977, 1.211507853, 1.216406986, 1.228121978,
1.239749851, 1.25147078, 1.263002162, 1.274673892, 1.286155969,
1.297675447, 1.309214334, 1.320690282, 1.332292174, 1.343809167,
1.355303167, 1.366728843, 1.378256972, 1.389918532, 1.401311901,
1.412743399, 1.42424855, 1.435654263, 1.447143137, 1.458596266,
1.470082863, 1.481545128, 1.493130336, 1.499694785, 1.504563801,
1.516058428, 1.527489414, 1.538939458, 1.550364186, 1.561846553,
1.573317223, 1.584781973, 1.596300752, 1.607867648, 1.61938151,
1.630888235, 1.642365283, 1.653925181, 1.665581818, 1.677313869,
1.689144875, 1.701027572, 1.712692508, 1.724456124, 1.736306899,
1.748174268, 1.759994063, 1.771981604, 1.78398656, 1.795997636,
1.808046978, 1.820145556, 1.832381782, 1.844674509, 1.856921374,
1.869183227, 1.881370005, 1.893846713, 1.906329713, 1.91623015,
1.918803967, 1.931234534, 1.943811188, 1.956494067, 1.969307054,
1.982069376, 1.994989708, 2.007943772, 2.021166425, 2.03426174,
2.047339624, 2.060636338, 2.074047526, 2.087443546, 2.100923579,
2.114596569, 2.128363889, 2.142249463, 2.156087403, 2.170175541,
2.184155785, 2.198513095, 2.212862123, 2.227297354, 2.241870437,
2.253333424, 2.256522858, 2.271375438, 2.286177751, 2.301261583,
2.316431474, 2.332105957, 2.347831697, 2.363599473, 2.379491729,
2.395529882, 2.411635039, 2.427960666, 2.444471371, 2.460846859,
2.477744579, 2.494585234, 2.511769865, 2.529237943, 2.54684658,
2.564680661, 2.582743513, 2.601040718, 2.619489059, 2.638173202,
2.657124469, 2.676269769, 2.695857688, 2.715307367, 2.735049731,
2.755163695, 2.775406263, 2.795986533, 2.816978309, 2.838012646,
2.859516478, 2.881358897, 2.90269923, 2.924674344, 2.947023426,
2.9696621, 2.992353788, 3.015162806, 3.038316104, 3.061751263,
3.085431218, 3.109292448, 3.133526883, 3.158014477, 3.183354633,
3.208879291, 3.23427488, 3.259526259, 3.285167743, 3.310554175,
3.336371475, 3.362172707, 3.388259364, 3.415032801, 3.441624981,
3.468493404, 3.496418366, 3.523887001, 3.551535042, 3.579155707,
3.607505048, 3.63615867, 3.664483266, 3.693869359, 3.723464268,
3.752763963, 3.782222225, 3.811872886, 3.841755023, 3.871631556,
3.902531008, 3.93293298, 3.963820976, 3.994686517, 4.025028621,
4.056731235, 4.08830789, 4.120303166, 4.151603653, 4.183148673,
4.2164921, 4.249455461, 4.282576856, 4.315177562, 4.348486875,
4.381681449, 4.415565832, 4.44877879, 4.482277467, 4.516759267,
4.550836157, 4.58563332, 4.592892531, 4.621420116, 4.656644565,
4.6918497, 4.727128816, 4.7634627, 4.799013439, 4.835033238,
4.871937258, 4.909051146, 4.947076645, 4.984353452, 5.02144922,
5.057896781, 5.095713527, 5.13281899, 5.170451523, 5.208951945,
5.24737281, 5.286267986, 5.324948674, 5.363574003, 5.403214857,
5.442084317, 5.482336028, 5.522495303, 5.562015177, 5.602776341,
5.643484615, 5.684144058, 5.726150441, 5.766912714, 5.809501037,
5.851045158, 5.893389255, 5.93635367, 5.977959163, 6.01989649,
6.063385706, 6.106935401, 6.150597741, 6.194174571, 6.23724292,
6.281656316, 6.326385208, 6.371384918, 6.415990432, 6.460869951,
6.505660789, 6.552941158, 6.597283365, 6.64238047, 6.688261346,
6.733824206, 6.780247586, 6.827751163, 6.875486892, 6.923819731,
6.97131241, 7.018546806, 7.066832588, 7.115316115, 7.164538463,
7.21279219, 7.264309006, 7.312839863, 7.361888298, 7.410750201,
7.46086377, 7.510664755, 7.561342546, 7.611361639, 7.660895797,
7.714019, 7.765816912, 7.817895741, 7.871225833, 7.923009526,
7.973523995, 8.025588785, 8.078273846, 8.130794229, 8.185596393,
8.23850096, 8.29130525, 8.347312552, 8.402102419, 8.457684498,
8.51075941, 8.565554345, 8.620357613, 8.675836533, 8.729986242,
8.78776728, 8.845142144, 8.902310542, 8.95889559, 9.017066156,
9.075905709, 9.131688291, 9.189967806, 9.247236756, 9.306131,
9.365735614, 9.425424212, 9.483899755, 9.543131042, 9.604669045,
9.665011344, 9.728275216, 9.788248186, 9.851663614, 9.910613499,
9.974132826, 10.035054177, 10.095004273, 10.158889074, 10.221147332,
10.285124, 10.349098897, 10.413227282, 10.476081023, 10.538970341,
10.602741336, 10.667449284, 10.734404374, 10.799259277, 10.86301981,
10.927647935, 10.993588124, 11.057225515, 11.124715062, 11.191689073,
11.26098097, 11.324918451, 11.393212239, 11.461839806, 11.530830011,
11.601563118, 11.668518153, 11.736312714, 11.807225052, 11.880901942,
11.949198379, 12.018517287, 12.090381584, 12.163291383, 12.23505575,
12.307118596, 12.381934637, 12.454435689, 12.524426337, 12.600569891,
12.674895479, 12.743521571, 12.813845949, 12.888316352, 12.964230893,
13.03347883, 13.107036892, 13.187626092, 13.260256338, 13.332844039,
13.406640571, 13.476018077, 13.551053477, 13.622745321, 13.699508423,
13.77274896, 13.848711509, 13.925576148, 14.005810452, 14.087932427,
14.163720053, 14.243410271, 14.322280275, 14.403901828, 14.487747788,
14.56676353, 14.647404998, 14.727869604, 14.818363654, 14.893415269,
14.975566129, 15.062592709, 15.142090413, 15.219445612, 15.298015019,
15.382567546, 15.464608993, 15.548582081, 15.635461942, 15.719251191,
15.803118781, 15.886267097, 15.978657133, 16.062453529, 16.144447682,
16.231236306, 16.312710065, 16.399252552, 16.413501272, 16.483035947,
16.575869078, 16.664124166, 16.754097643, 16.836204372, 16.927541256,
17.014808071, 17.106280388, 17.199313321, 17.285515507, 17.378266516,
17.468882853, 17.551926223, 17.643018055, 17.736101648, 17.8230864,
17.915431292, 18.008851953, 18.102768585, 18.192581257, 18.283087548,
18.382144354, 18.482893651, 18.588062705, 18.676547863, 18.7692377,
18.861120643, 18.957028427, 19.05840003, 19.164866294, 19.259437744,
19.351547773, 19.452706606, 19.543588542, 19.635623661, 19.733646369,
19.831238732, 19.925565945, 20.025403155, 20.118379514, 20.231543915,
20.328554614, 20.4278698, 20.524722507, 20.631924142, 20.733713687,
20.839927599, 20.947920035, 21.047295932, 21.151336977, 21.256740563,
21.365350808, 21.471483694, 21.575990144, 21.676284162, 21.783417293,
21.888286046, 21.99141798, 22.101359501, 22.208002503, 22.30929655,
22.422468215, 22.533823557, 22.647732331, 22.753648353, 22.85937594,
22.976106163, 23.072689346, 23.183598536, 23.279815971, 23.399501994,
23.509132484, 23.624709446, 23.737300371, 23.847272608, 23.961187607,
24.07743337, 24.190202345, 24.295303087, 24.408207231, 24.531496297,
24.647632382, 24.754070464, 24.870591427, 24.988383793, 25.101488686,
25.226170278, 25.344008914, 25.459458113, 25.565253755, 25.68121513,
25.809694448, 25.925922618, 26.059389168, 26.179509725, 26.286036885,
26.408198868, 26.534222468, 26.65673517, 26.78735186, 26.901265595,
27.03158618, 27.147857168, 27.261466946, 27.386508004, 27.507964938,
27.64335501, 27.766992496, 27.902944466, 28.019479694, 28.14586437,
28.280914824, 28.428013173, 28.550314489, 28.672427852, 28.808751552,
28.9592421, 29.093536311, 29.232739514, 29.334001505, 29.438523387,
29.567052035, 29.696050784, 29.840064713, 29.959667047, 30.112407956,
30.234583016, 30.36688697, 30.496516865, 30.623709211, 30.760430816,
30.874628206, 31.000163533, 31.139607951, 31.270829384, 31.405007068,
31.538074717, 31.670557691, 31.817389963, 31.968535989, 32.090092778,
32.222699584, 32.344565665, 32.467530606, 32.592130553, 32.727235312,
32.87932986, 33.001227412, 33.121900542, 33.274321226, 33.427908494,
33.543618315, 33.711085065, 33.855627568, 34.003211017, 34.164531305,
34.298318558, 34.438103361, 34.572447801, 34.720269922, 34.845036493,
34.994077264, 35.124154642, 35.287456868, 35.43597724, 35.598699642,
35.733645441, 35.883176223, 36.022957403, 36.167038583, 36.32208633,
36.488862233, 36.621749394, 36.776826541, 36.91653389, 37.057412825,
37.190317023, 37.319691751, 37.48362917, 37.618316074, 37.767584788,
37.928931894, 38.087329435, 38.251850768, 38.383779627, 38.519981763,
38.677674483, 38.821077154, 38.981929413, 39.152418785, 39.282349345,
39.454953206, 39.61493747, 39.754668596, 39.905517483, 40.048624329,
40.237858936, 40.403061351, 40.538046906, 40.722148354, 40.758463253,
40.884434126, 41.049132774, 41.219878264, 41.369198439, 41.507893355,
41.621818584, 41.783468881, 42.03460508, 42.195686153, 42.370342181,
42.518737339, 42.673582474, 42.902560698, 43.05949202, 43.193554189,
43.334353299, 43.517446064, 43.674869913, 43.814693155, 43.963797335,
44.132804445, 44.316804397, 44.477155468, 44.624599692, 44.791811073,
45.02124427, 45.178033422, 45.354535662, 45.539477133, 45.691955679,
45.895718254, 46.141368626, 46.302015722, 46.554979759, 46.738588609,
46.946452287, 47.113792886, 47.336403502, 47.501275637, 47.687546883,
47.892968505, 48.073711263, 48.259474864, 48.407617636, 48.568541189,
48.739627066, 48.919517497, 49.081461467, 49.217685717, 49.366185738,
49.572613135, 49.683612787, 49.834219095, 50.027081608, 50.204646082,
50.356046579, 50.557335771, 50.722151671, 50.970424544, 51.113103599,
51.331212753, 51.530208383, 51.705969659, 51.907744935, 52.098288421,
52.252827367, 52.445626341, 52.583286756, 52.702365326, 52.912402058,
53.081944961, 53.239256605, 53.396801286, 53.538609961, 53.725399909,
53.921845694, 54.122870748, 54.323661354, 54.531082041, 54.699552469,
54.947013956, 55.084530284, 55.272027031, 55.454877649, 55.620436979,
55.788766333, 55.995771602, 56.243776091, 56.462479108, 56.620773679,
56.785031272, 56.963576247, 57.149987224, 57.337833963, 57.50069253,
57.718188421, 57.897928702, 58.057970071, 58.293433633, 58.451139348,
58.669495931, 58.899025142, 59.042285507, 59.308110663, 59.545467535,
59.76636179, 59.970244643, 60.135709801, 60.271492305, 60.373365124,
60.58004772, 60.673447919, 60.896819644, 61.09509106, 61.302647877,
61.522925201, 61.693637153, 61.838876798, 62.06631538, 62.327016881,
62.499531319, 62.765252764, 62.900072692, 63.151103308, 63.343467006,
63.57036453, 63.860918671, 64.059648499, 64.253660522, 64.424883177,
64.666060618, 64.85554302, 65.0759968, 65.237490821, 65.44880355,
65.727460839, 65.894045276, 66.2084666, 66.474368574, 66.612287245,
66.854544293, 67.201520869, 67.427661123, 67.719680643, 67.921051436,
68.05234241, 68.237209321, 68.440631892, 68.711960887, 68.998996926,
69.420312747, 69.603426771, 69.794726112, 69.984334753, 70.22079556,
70.511271268, 70.617501681, 70.829861255, 71.03793747, 71.330014274,
71.482871876, 71.771735888, 71.916304023, 72.100197937, 72.34543968,
72.669943509, 72.833424533, 72.989328595, 73.171798214, 73.298525537,
73.436309865, 73.628016162, 73.877612387, 74.125927483, 74.283158243,
74.490681363, 74.75484971, 74.96373709, 75.319757437, 75.549868583,
75.677713211, 75.994843096, 76.156016037, 76.325611314, 76.659659325,
77.125292936, 77.299410302, 77.463056986, 77.592193548, 77.687519319,
77.785299441, 78.10468257, 78.309069539, 78.524639382, 78.794035207,
79.329247745, 79.647747949, 79.845542764, 80.140748485, 80.581739533,
80.898815101, 81.029824382, 81.331419238, 81.66189363, 81.844629267,
81.947865916, 82.282298831, 82.561106782, 82.835838199, 83.142854781,
83.350851894, 83.550934944, 83.806523069, 84.105202916, 84.437392898,
84.753197363, 85.12393925, 85.505531289, 85.685504682, 85.827567117,
85.879312927, 86.03277549, 86.174867573, 86.439059881, 86.59603768,
86.727455724, 87.188460801, 87.34944352, 87.503378305, 87.729001426,
87.926915899, 88.120173508, 88.36081338, 88.492157034, 88.65695608,
88.904368121, 89.08596032, 89.721590564, 89.996736033, 90.328041057,
90.504965884, 91.138974238, 91.390894653, 91.47017051, 91.612203566,
91.820408397, 92.066156446, 92.30576694, 92.517541093, 92.931681782,
93.184331431, 93.711322962, 94.03784139, 94.463193925, 94.9917164,
95.331687151, 95.388681778, 95.928735894, 95.990193627, 96.109304147,
96.753267192, 97.313734954, 97.506216742, 97.574715398, 97.694071569,
97.803842382, 97.943317876, 98.225379166, 98.415154052, 98.754755371,
98.889004089, 99.002218328, 99.241698479, 99.891010717, 100.309574995,
100.548846667, 100.60652688, 101.150603587, 101.309138641, 101.583135673,
102.034253024, 102.097979777, 102.246124336, 102.706379872, 103.219187876,
103.381794909, 103.732579264, 103.783725825, 104.933808555, 105.175550165,
105.250163169, 105.314491587, 105.37388747, 105.525008307, 105.628448336,
106.270050375, 106.394631647, 107.046283556, 107.638429388, 107.688922297,
107.733365474, 108.043445993, 108.721176327, 109.175799077, 109.218075437,
109.287442326, 109.420857653, 109.458811399, 109.959045411, 110.242607557,
110.539336252, 111.211375888, 111.620316462, 111.647446758, 111.853893602,
111.973790431, 112.212419505, 112.333996466, 112.442586253, 112.579127315,
112.965819913, 113.204043651, 113.390937027, 113.906347108, 113.934448013,
114.112537592, 114.582560379, 114.763713923, 114.89142731, 115.047249186,
115.839358312, 116.186530239, 116.954924575, 117.173546524, 117.468616154,
117.744636156, 117.873659844, 117.922609293, 118.456775516, 118.474326013
), "50" = c(2.3459e-05, 2.3488e-05, 2.3543e-05, 2.3764e-05, 2.3835e-05,
2.4066e-05, 2.4481e-05, 2.4767e-05, 2.5362e-05, 2.5394e-05, 2.5779e-05,
2.5963e-05, 2.6609e-05, 2.663e-05, 2.6708e-05, 2.6723e-05, 2.7115e-05,
2.751e-05, 2.8003e-05, 2.8053e-05, 2.812e-05, 2.8726e-05, 2.943e-05,
2.9601e-05, 2.9714e-05, 3.0511e-05, 3.0762e-05, 3.0778e-05, 3.0918e-05,
3.1132e-05, 3.1289e-05, 3.1764e-05, 3.2169e-05, 3.2333e-05, 3.2386e-05,
3.2493e-05, 3.2602e-05, 3.3069e-05, 3.3464e-05, 3.364e-05, 3.368e-05,
3.5003e-05, 3.5084e-05, 3.5423e-05, 3.6148e-05, 3.6602e-05, 3.676e-05,
3.7084e-05, 3.7567e-05, 3.9385e-05, 3.998e-05, 4.0446e-05, 4.0834e-05,
4.1636e-05, 4.2452e-05, 4.2682e-05, 4.3414e-05, 4.354e-05, 4.4152e-05,
4.4446e-05, 4.5069e-05, 4.5554e-05, 4.6351e-05, 4.6782e-05, 4.7255e-05,
4.8483e-05, 4.9277e-05, 4.9626e-05, 5.0393e-05, 5.0476e-05, 5.2121e-05,
5.2327e-05, 5.2631e-05, 5.3224e-05, 5.3272e-05, 5.3413e-05, 5.3976e-05,
5.4587e-05, 5.4871e-05, 5.5184e-05, 5.5754e-05, 5.6072e-05, 5.6285e-05,
5.7067e-05, 5.771e-05, 5.81e-05, 5.8739e-05, 5.9344e-05, 5.9685e-05,
5.9849e-05, 6.0782e-05, 6.136e-05, 6.1663e-05, 6.2335e-05, 6.2969e-05,
6.3937e-05, 6.4305e-05, 6.4762e-05, 6.5219e-05, 6.6729e-05, 6.7616e-05,
6.8914e-05, 6.9645e-05, 7.0327e-05, 7.0818e-05, 7.1639e-05, 7.2286e-05,
7.2754e-05, 7.333e-05, 7.4221e-05, 7.5533e-05, 7.686e-05, 7.7258e-05,
7.8116e-05, 7.8712e-05, 8.0337e-05, 8.0876e-05, 8.2162e-05, 8.2721e-05,
8.4618e-05, 8.4934e-05, 8.6727e-05, 8.7301e-05, 8.8069e-05, 8.9688e-05,
9.0613e-05, 9.1747e-05, 9.1991e-05, 9.283e-05, 9.3566e-05, 9.4011e-05,
9.5007e-05, 9.5754e-05, 9.6411e-05, 9.7037e-05, 9.872e-05, 0.000100588,
0.000102067, 0.000103002, 0.000103641, 0.000105375, 0.000106434,
0.000107652, 0.000110131, 0.000111512, 0.00011282, 0.000114182,
0.000115682, 0.000117045, 0.000117525, 0.000119305, 0.00012151,
0.000123645, 0.000124468, 0.000125206, 0.000126865, 0.000127895,
0.000128929, 0.000129616, 0.000131752, 0.000132798, 0.000133781,
0.000134839, 0.00013555, 0.000136587, 0.000137401, 0.000139155,
0.000141744, 0.000143216, 0.00014464, 0.000146355, 0.000147355,
0.000149285, 0.000150582, 0.000152599, 0.000153816, 0.000156117,
0.000157381, 0.000159278, 0.000162696, 0.000164433, 0.000165957,
0.00016874, 0.000170044, 0.000171683, 0.000173206, 0.000174891,
0.000178189, 0.000180469, 0.000182319, 0.000184166, 0.000185747,
0.00018812, 0.000189685, 0.000190699, 0.00019272, 0.000194591,
0.000196283, 0.000197791, 0.000200227, 0.000203458, 0.000206593,
0.000208516, 0.000211517, 0.000213873, 0.000216, 0.000220051,
0.000223917, 0.000226542, 0.00022942, 0.000234391, 0.000236134,
0.000238589, 0.000241011, 0.000242862, 0.000245164, 0.00024808,
0.00024998, 0.000252412, 0.000255631, 0.000257388, 0.000260221,
0.000262028, 0.000266213, 0.000268744, 0.000271955, 0.00027563,
0.000280144, 0.000282983, 0.00028449, 0.000287544, 0.000289502,
0.000294186, 0.000297042, 0.000299524, 0.0003029, 0.000306844,
0.000310461, 0.000314096, 0.000318257, 0.000322668, 0.000325278,
0.000327949, 0.000333193, 0.000335889, 0.000338897, 0.000342498,
0.000346947, 0.000351049, 0.000354633, 0.000359194, 0.000363175,
0.000367213, 0.000370694, 0.000375642, 0.000379881, 0.000384462,
0.000388529, 0.000394007, 0.000400529, 0.000405468, 0.000409403,
0.000414619, 0.000419494, 0.000422867, 0.000426866, 0.000431767,
0.000438888, 0.000442404, 0.000448617, 0.000453769, 0.000458339,
0.000462901, 0.000467147, 0.000474117, 0.000480834, 0.000487061,
0.000493104, 0.000498892, 0.000504579, 0.000508368, 0.000513132,
0.000520771, 0.00052589, 0.000531365, 0.000537328, 0.00054367,
0.000549957, 0.000554768, 0.000560357, 0.000566714, 0.000573017,
0.000576219, 0.000582422, 0.000591227, 0.000597267, 0.000606522,
0.000613536, 0.000620582, 0.000627816, 0.000634202, 0.00064191,
0.000649705, 0.000657993, 0.000665917, 0.00067556, 0.000681603,
0.000689062, 0.000696327, 0.000705395, 0.000712455, 0.000720583,
0.000731773, 0.000742481, 0.000753397, 0.000762779, 0.000774347,
0.000782097, 0.000788858, 0.000797841, 0.000806481, 0.000816836,
0.0008269, 0.000836126, 0.000846185, 0.000855992, 0.000866413,
0.000879189, 0.000890103, 0.000901946, 0.000910867, 0.000920016,
0.000929485, 0.000939344, 0.000950821, 0.000960742, 0.000969527,
0.000982373, 0.000995269, 0.001005952, 0.001013887, 0.00101764,
0.001031092, 0.001040227, 0.00105162, 0.001064265, 0.001075428,
0.00108765, 0.001102759, 0.001116054, 0.001126517, 0.001138989,
0.001154559, 0.001166312, 0.001182988, 0.001196336, 0.001209402,
0.001224765, 0.001238648, 0.001253211, 0.001267471, 0.001284077,
0.001300143, 0.001314828, 0.001331831, 0.001345023, 0.001359421,
0.00137541, 0.001391503, 0.001406173, 0.001422953, 0.001441677,
0.00145754, 0.001474604, 0.001490846, 0.001507149, 0.001524485,
0.001542421, 0.001561338, 0.001580136, 0.001596139, 0.001616435,
0.001633711, 0.001649927, 0.001666848, 0.00168734, 0.001706352,
0.00172684, 0.001745034, 0.001766447, 0.001787037, 0.001809534,
0.00183022, 0.00185187, 0.001874942, 0.001898782, 0.001919188,
0.00194302, 0.001965965, 0.001988155, 0.002008566, 0.002032904,
0.002053405, 0.002075796, 0.002101991, 0.002125058, 0.002147175,
0.002171125, 0.00219641, 0.002221926, 0.002247062, 0.002273785,
0.002302411, 0.002326196, 0.002351904, 0.002377832, 0.002406768,
0.002434797, 0.0024657, 0.002490857, 0.002519815, 0.002549419,
0.00257953, 0.002605721, 0.002633587, 0.00266221, 0.002689225,
0.002720565, 0.002754622, 0.002787795, 0.002818381, 0.002848583,
0.002881608, 0.002911603, 0.002943803, 0.002979864, 0.003013806,
0.003050346, 0.003085379, 0.003125404, 0.003156186, 0.003190661,
0.003223689, 0.003260933, 0.00329907, 0.003335294, 0.003372812,
0.003410377, 0.003447642, 0.003492924, 0.003536004, 0.003575697,
0.003620743, 0.003661108, 0.003701194, 0.003744165, 0.003786043,
0.003834304, 0.003876419, 0.003920351, 0.00396747, 0.00401226,
0.004060206, 0.004113954, 0.004168055, 0.004224192, 0.004277171,
0.004327769, 0.004376568, 0.004431651, 0.004479433, 0.004528825,
0.004583121, 0.004638677, 0.004690625, 0.004745775, 0.004800068,
0.004853048, 0.004912544, 0.004972842, 0.005026589, 0.005087044,
0.00514489, 0.005201285, 0.005258378, 0.005318877, 0.005382026,
0.005446991, 0.00551486, 0.005579086, 0.005641289, 0.005704254,
0.005774559, 0.005841232, 0.005912509, 0.00598763, 0.006053938,
0.006127602, 0.006202264, 0.006274129, 0.006344935, 0.006416923,
0.006495519, 0.006573688, 0.006648673, 0.006726146, 0.006807484,
0.006889941, 0.006967968, 0.007053212, 0.007135032, 0.007214538,
0.007303296, 0.007393899, 0.00747263, 0.007564349, 0.00765343,
0.007742087, 0.00783679, 0.007929342, 0.008022082, 0.008113583,
0.008215439, 0.008311359, 0.008410769, 0.008498085, 0.008588998,
0.008683844, 0.008786649, 0.00889275, 0.008987108, 0.00908454,
0.009180864, 0.009283315, 0.009396685, 0.00950194, 0.009611882,
0.009718261, 0.009834575, 0.009947845, 0.010064478, 0.010147393,
0.010171691, 0.010282565, 0.010394092, 0.010520564, 0.010646405,
0.010781068, 0.010908089, 0.011039389, 0.01117263, 0.011300701,
0.011443232, 0.011575723, 0.011701079, 0.011844982, 0.011984937,
0.012121946, 0.012267657, 0.012403403, 0.012548657, 0.012686032,
0.012834127, 0.012984192, 0.013135206, 0.013295089, 0.013453288,
0.013617049, 0.013771672, 0.013940871, 0.014090098, 0.014248777,
0.014415984, 0.014588807, 0.014757527, 0.014929293, 0.015105113,
0.015266442, 0.015440256, 0.015623385, 0.015808408, 0.015986042,
0.016169234, 0.016355021, 0.016546281, 0.016749596, 0.016939634,
0.017136453, 0.017332197, 0.017532255, 0.017719473, 0.01792359,
0.018120603, 0.01832482, 0.018538684, 0.018751458, 0.018974785,
0.019187349, 0.019407798, 0.019633469, 0.019869359, 0.020113225,
0.020342813, 0.020577374, 0.020818854, 0.021062674, 0.021309748,
0.021545894, 0.021787301, 0.02204259, 0.02229376, 0.022546329,
0.022810781, 0.023080218, 0.023336901, 0.023591619, 0.023877514,
0.02415078, 0.024435157, 0.02471709, 0.024997259, 0.025288061,
0.025576392, 0.025877362, 0.026162872, 0.026461011, 0.026769455,
0.027084737, 0.027393332, 0.027716192, 0.028025638, 0.028357765,
0.028683832, 0.029000594, 0.029334149, 0.029683788, 0.03003251,
0.030377505, 0.030721051, 0.031074617, 0.031419746, 0.031779013,
0.032149502, 0.032524962, 0.032895516, 0.033273137, 0.033665741,
0.034057633, 0.034439189, 0.034839301, 0.035229709, 0.035628259,
0.036025166, 0.03643291, 0.036840412, 0.037273194, 0.037687878,
0.038123051, 0.038584123, 0.039052778, 0.03949937, 0.039958963,
0.040412717, 0.040873459, 0.041338166, 0.041804122, 0.042281879,
0.042773719, 0.043267578, 0.043744457, 0.044242752, 0.044747407,
0.045281398, 0.045801368, 0.046310924, 0.046841218, 0.047387197,
0.047908436, 0.048468812, 0.049027799, 0.049577727, 0.050161472,
0.050728923, 0.051325128, 0.051920599, 0.052521316, 0.053114475,
0.053725993, 0.05434356, 0.054975799, 0.055622047, 0.056253651,
0.056913406, 0.057557087, 0.058220008, 0.058869274, 0.059531913,
0.060213622, 0.060913729, 0.061621254, 0.062342302, 0.063060591,
0.063789375, 0.064535323, 0.065258775, 0.066005175, 0.06676532,
0.067555614, 0.068351923, 0.06913658, 0.069926005, 0.070739206,
0.071554511, 0.072386235, 0.073207892, 0.074060271, 0.074907779,
0.075799857, 0.076653035, 0.077556075, 0.078455805, 0.079325089,
0.080226062, 0.081168184, 0.082079381, 0.083016742, 0.083965597,
0.084917429, 0.085871107, 0.086856105, 0.087864629, 0.08887092,
0.089870014, 0.090903023, 0.091942231, 0.093009541, 0.09408125,
0.095167955, 0.096261484, 0.097357539, 0.098466287, 0.099592672,
0.100490094, 0.100735279, 0.101898587, 0.103044783, 0.10422858,
0.105403942, 0.106641946, 0.107871006, 0.109108507, 0.110353368,
0.111614654, 0.11291433, 0.114217922, 0.115499196, 0.11679871,
0.118131187, 0.119503103, 0.120878347, 0.122282803, 0.123676609,
0.125077333, 0.126533967, 0.127937296, 0.129384364, 0.130864238,
0.132363833, 0.133890674, 0.135426968, 0.136963836, 0.138520687,
0.14010285, 0.141722503, 0.143332127, 0.144995039, 0.146655274,
0.148333918, 0.150033304, 0.151776839, 0.153498858, 0.155257507,
0.157028399, 0.158824125, 0.160620236, 0.162490146, 0.164373605,
0.166272364, 0.168208313, 0.170156806, 0.172113157, 0.174091447,
0.176099304, 0.178128077, 0.180165389, 0.182259664, 0.184359958,
0.186512365, 0.188662998, 0.190841814, 0.193054471, 0.195247673,
0.197459628, 0.1997243, 0.202009973, 0.204306783, 0.20661834,
0.209024949, 0.211426453, 0.213836388, 0.216284091, 0.218772834,
0.221277463, 0.223758905, 0.226303094, 0.228871458, 0.231511836,
0.234201788, 0.23688129, 0.239580913, 0.242329137, 0.245098927,
0.247906432, 0.250770279, 0.25362155, 0.25653149, 0.259495593,
0.262492913, 0.265483562, 0.268475712, 0.271556898, 0.274666929,
0.277811649, 0.280989967, 0.284254702, 0.287526184, 0.290830139,
0.294139915, 0.297517041, 0.298247234, 0.300957573, 0.304381945,
0.307857154, 0.311408235, 0.314983616, 0.318621206, 0.322266211,
0.325945496, 0.32969886, 0.333508846, 0.337311738, 0.341203143,
0.345140072, 0.349099669, 0.35318121, 0.357267863, 0.361376495,
0.36551871, 0.369703436, 0.373936893, 0.378303689, 0.382680486,
0.387080373, 0.391473565, 0.395956456, 0.39689192, 0.400545543,
0.405205791, 0.409909692, 0.414639694, 0.419464814, 0.424287332,
0.429245153, 0.434220292, 0.439278633, 0.444383367, 0.449610474,
0.4548177, 0.460149101, 0.465497097, 0.470949075, 0.476448025,
0.48202817, 0.48763691, 0.493261725, 0.499074283, 0.504923202,
0.51084046, 0.516772988, 0.522843707, 0.52905977, 0.535341804,
0.541680744, 0.548130981, 0.554622447, 0.56116211, 0.567881031,
0.574642129, 0.581504045, 0.588459976, 0.595473122, 0.598496273,
0.602598026, 0.60982204, 0.617123552, 0.624562334, 0.632093488,
0.639764885, 0.647537194, 0.655341087, 0.663290748, 0.671284045,
0.679451223, 0.687672516, 0.696078251, 0.704594591, 0.71333099,
0.722163226, 0.731022932, 0.740015691, 0.749224428, 0.758485446,
0.767991879, 0.777557872, 0.787268682, 0.797159079, 0.807203739,
0.811387308, 0.817388438, 0.827809029, 0.838304089, 0.849041795,
0.859933812, 0.870938066, 0.882297233, 0.893668559, 0.905282285,
0.91707555, 0.92901388, 0.94131805, 0.953798692, 0.966581558,
0.979494405, 0.99251355, 1.005912279, 1.019608707, 1.03358439,
1.0447379, 1.070051195, 1.084139794, 1.098264783, 1.112324365,
1.126336126, 1.140338894, 1.154326832, 1.168209383, 1.181980656,
1.195863565, 1.209651078, 1.22341245, 1.237115634, 1.250836692,
1.264368349, 1.277946024, 1.291405718, 1.304913607, 1.312844216,
1.31841107, 1.331833768, 1.345202296, 1.358670898, 1.372117493,
1.385431521, 1.398846222, 1.412274208, 1.425677911, 1.439060934,
1.45242263, 1.465753645, 1.478968304, 1.492416861, 1.505759181,
1.519054556, 1.532469636, 1.545748036, 1.559093227, 1.572336449,
1.58555107, 1.598825487, 1.612200318, 1.625605655, 1.638924332,
1.646614529, 1.652302606, 1.665629696, 1.678970294, 1.692352453,
1.705576898, 1.718837972, 1.732092568, 1.745499745, 1.758884463,
1.772296095, 1.785722584, 1.799064839, 1.812492065, 1.825842433,
1.839428099, 1.852864715, 1.866488081, 1.880024805, 1.893482252,
1.907206681, 1.920952511, 1.934695681, 1.94843163, 1.96223207,
1.976180607, 1.990061417, 2.003827614, 2.017702395, 2.031673648,
2.0456824, 2.059794608, 2.073900285, 2.087993512, 2.102135659,
2.116414439, 2.12782403, 2.130840127, 2.145305641, 2.159662516,
2.174250252, 2.188759784, 2.203429982, 2.218066887, 2.232782368,
2.247620377, 2.262452827, 2.277303267, 2.292284834, 2.307667149,
2.322686176, 2.337957291, 2.353418206, 2.368829403, 2.384522691,
2.400261944, 2.416116188, 2.431970497, 2.447796224, 2.464029208,
2.480235676, 2.496405767, 2.509364968, 2.513006366, 2.529506923,
2.546223402, 2.562867004, 2.579937372, 2.59674608, 2.613749406,
2.630917069, 2.648395712, 2.665829782, 2.683459195, 2.701323111,
2.719306098, 2.737544142, 2.755958392, 2.774236398, 2.792884937,
2.811743859, 2.830897975, 2.849941747, 2.86934231, 2.88904065,
2.908499813, 2.928519751, 2.948830908, 2.969203492, 2.989621838,
3.010321869, 3.031196135, 3.052218839, 3.073460635, 3.095207753,
3.116946584, 3.138758808, 3.161262425, 3.183376461, 3.20612686,
3.22949247, 3.25264956, 3.275987703, 3.299214249, 3.322915375,
3.347278459, 3.371272289, 3.395812635, 3.420495625, 3.445221198,
3.470968244, 3.496652725, 3.522615573, 3.548869409, 3.575169859,
3.601465679, 3.628237088, 3.6548304, 3.681834362, 3.709156127,
3.737287456, 3.765631662, 3.79371242, 3.821906137, 3.850877938,
3.880562659, 3.909414329, 3.939739178, 3.969232032, 3.99913138,
4.029597303, 4.05989882, 4.090958089, 4.121821482, 4.153134818,
4.183891831, 4.21573655, 4.247297258, 4.279564675, 4.312057396,
4.34405157, 4.37655697, 4.40929535, 4.442449794, 4.475782972,
4.509867869, 4.543902716, 4.577585767, 4.611646792, 4.645502348,
4.680052972, 4.714768897, 4.75036301, 4.786166455, 4.821796308,
4.858101504, 4.894813969, 4.93143891, 4.96841648, 4.97574453,
5.004064695, 5.039991843, 5.076984658, 5.113881328, 5.150564622,
5.188943745, 5.227038253, 5.265285297, 5.304154019, 5.343277989,
5.382007283, 5.421113171, 5.459991541, 5.499388172, 5.539764171,
5.578880646, 5.619165543, 5.660162791, 5.699110667, 5.739856515,
5.780374896, 5.820200439, 5.861894059, 5.902946123, 5.944869752,
5.986151675, 6.029896402, 6.072541755, 6.115104843, 6.157598315,
6.200229678, 6.243535202, 6.287384645, 6.331003517, 6.375355991,
6.420753913, 6.464189563, 6.509453868, 6.554418518, 6.597604931,
6.64249612, 6.688108226, 6.734755207, 6.780946656, 6.826979809,
6.87428588, 6.922132716, 6.969869036, 7.015784152, 7.064755157,
7.113807563, 7.163312303, 7.212206183, 7.259190737, 7.308788133,
7.358414484, 7.408115888, 7.459275551, 7.509503211, 7.559460414,
7.60829547, 7.657881609, 7.707626965, 7.76004499, 7.811546659,
7.864641557, 7.916580052, 7.968461789, 8.020531438, 8.072025641,
8.123355931, 8.176133988, 8.228021208, 8.28108358, 8.336015881,
8.389986033, 8.445052783, 8.500299117, 8.553743555, 8.607949207,
8.663723607, 8.716915679, 8.771813712, 8.827779666, 8.884756664,
8.941256189, 8.997020833, 9.052228803, 9.110019299, 9.169150478,
9.226380369, 9.284010934, 9.342120104, 9.397884181, 9.456236468,
9.514243663, 9.577263119, 9.637038242, 9.700254166, 9.759343741,
9.821968142, 9.883089268, 9.944710711, 10.005758327, 10.068853755,
10.128950957, 10.189705658, 10.250157123, 10.313030711, 10.376303949,
10.439876704, 10.504137306, 10.567384111, 10.632464217, 10.694772873,
10.759357306, 10.824258992, 10.890227395, 10.956332201, 11.018961138,
11.085759422, 11.149730253, 11.215657194, 11.284211111, 11.350873688,
11.42105375, 11.489884872, 11.555834794, 11.624359881, 11.695466403,
11.763181418, 11.835376702, 11.904687611, 11.970874046, 12.039159818,
12.111141852, 12.18207874, 12.25050605, 12.321797514, 12.392210204,
12.461935844, 12.530679289, 12.60482987, 12.677269161, 12.75082973,
12.82643928, 12.904426693, 12.981191771, 13.054282312, 13.129928443,
13.206790395, 13.285686021, 13.36430169, 13.441729122, 13.522188398,
13.595632482, 13.675601068, 13.756382909, 13.832626927, 13.909782074,
13.992814916, 14.074462184, 14.155081123, 14.227801516, 14.30255052,
14.382670482, 14.462331692, 14.544035603, 14.627520328, 14.705147346,
14.788951837, 14.874713353, 14.951328816, 15.034740965, 15.115404237,
15.198141608, 15.27701122, 15.355675662, 15.437375321, 15.519949609,
15.601838241, 15.682251363, 15.76994148, 15.854239237, 15.94789553,
16.029830683, 16.113493343, 16.193209612, 16.285209031, 16.370175113,
16.454913978, 16.542380838, 16.635495708, 16.721715846, 16.813355533,
16.901834313, 16.993225505, 17.081208408, 17.166000071, 17.263926495,
17.280912064, 17.354613683, 17.445969951, 17.532509084, 17.618981999,
17.709049244, 17.801017034, 17.898287709, 17.989425472, 18.076778923,
18.166513233, 18.258219447, 18.352160351, 18.447073118, 18.545046476,
18.649902424, 18.743627514, 18.844103962, 18.93995672, 19.033641351,
19.127994985, 19.22674408, 19.318026164, 19.413749936, 19.516583928,
19.602773848, 19.712924739, 19.813125862, 19.917604524, 20.026069348,
20.125966634, 20.233163542, 20.339660504, 20.432899073, 20.535459063,
20.637934472, 20.738099036, 20.842222876, 20.943493974, 21.057750829,
21.152752224, 21.258810794, 21.365330756, 21.472509672, 21.576662176,
21.682138699, 21.79679492, 21.901920871, 22.013025162, 22.119300064,
22.230514613, 22.342856542, 22.447431091, 22.562194449, 22.670868649,
22.773208826, 22.886540739, 22.98884591, 23.108571104, 23.223655072,
23.324155162, 23.431163689, 23.542148431, 23.655079594, 23.772178358,
23.873227636, 23.986110952, 24.091478021, 24.200026174, 24.303769388,
24.415115357, 24.54210985, 24.653305219, 24.772059155, 24.899970374,
25.021421978, 25.140523527, 25.264457866, 25.380700499, 25.495610867,
25.622686417, 25.740900614, 25.867493402, 25.975583099, 26.095653068,
26.247250157, 26.356410397, 26.457811175, 26.598902007, 26.744929718,
26.881351984, 27.000233574, 27.114869457, 27.243640518, 27.377283069,
27.494453511, 27.637869706, 27.77882592, 27.912335276, 28.031473058,
28.149779645, 28.277413553, 28.404469397, 28.539500311, 28.692251394,
28.820316465, 28.969216564, 29.106533313, 29.251407033, 29.384868735,
29.516607584, 29.652145378, 29.808736222, 29.942293555, 30.078973447,
30.220836834, 30.348247811, 30.487587464, 30.645138929, 30.794744679,
30.910658105, 31.054397499, 31.201859442, 31.351994051, 31.486716938,
31.629095174, 31.76833776, 31.908879054, 32.068536744, 32.215422436,
32.334144345, 32.499139359, 32.622631266, 32.755604953, 32.915189547,
33.069513382, 33.196956822, 33.336915224, 33.466944695, 33.60089337,
33.752829693, 33.887570142, 34.0383013, 34.18614147, 34.333045718,
34.488609854, 34.618283824, 34.746442657, 34.887682344, 35.030953666,
35.165769003, 35.308662753, 35.472956904, 35.611351779, 35.783350498,
35.939685606, 36.085659605, 36.231694088, 36.370554768, 36.510189545,
36.662841113, 36.813172716, 36.965498146, 37.131377622, 37.279864099,
37.424418437, 37.557674371, 37.726500793, 37.899042239, 38.041555716,
38.19152569, 38.32237562, 38.48669187, 38.661326717, 38.842753312,
38.990082559, 39.161519836, 39.340217699, 39.478061131, 39.600391868,
39.798948188, 39.984313755, 40.145165602, 40.328025617, 40.479059494,
40.675066737, 40.853873175, 41.016021773, 41.185484376, 41.337066484,
41.52946957, 41.688547783, 41.836886603, 42.049371011, 42.208699428,
42.369869275, 42.518030149, 42.718029559, 42.894670323, 43.046868838,
43.220827237, 43.255676848, 43.409486453, 43.576846337, 43.777735106,
43.966466885, 44.126601414, 44.310151734, 44.475294993, 44.643742514,
44.805261656, 44.918748931, 45.093559116, 45.27214507, 45.477031669,
45.631552739, 45.808374906, 45.986056965, 46.114997477, 46.313622809,
46.482072469, 46.669134063, 46.829000063, 47.044608739, 47.21489944,
47.413564871, 47.606513136, 47.765362379, 47.928166571, 48.119163358,
48.28578671, 48.471831971, 48.73757178, 48.898771783, 49.138229763,
49.296311563, 49.493673619, 49.693338565, 49.87322605, 50.026464039,
50.222446426, 50.472550114, 50.658326598, 50.871067335, 51.05283004,
51.285232344, 51.473541469, 51.688875653, 51.830154575, 52.024928209,
52.169819487, 52.377745069, 52.576514158, 52.815944306, 52.992547544,
53.13788021, 53.33948381, 53.582693777, 53.79317019, 54.001134178,
54.243661605, 54.436865821, 54.612322624, 54.834561462, 55.030217833,
55.194374063, 55.403473876, 55.564149625, 55.751016542, 55.956871954,
56.107914544, 56.298383326, 56.534283428, 56.680021851, 56.831775066,
57.05944334, 57.210817155, 57.413408574, 57.533574379, 57.734773131,
58.030316061, 58.215958367, 58.474773782, 58.669738265, 58.86115215,
59.011566287, 59.214911338, 59.407796866, 59.687801717, 60.021886403,
60.258800494, 60.363438864, 60.620823269, 60.831277305, 61.039043465,
61.259602596, 61.526145308, 61.844179164, 62.116227157, 62.286981508,
62.619974235, 62.814095515, 63.002643631, 63.257363825, 63.388562591,
63.597108503, 63.840729562, 64.019107782, 64.208715482, 64.50169267,
64.628354063, 64.823155124, 64.997809818, 65.44356564, 65.539881354,
65.800009126, 66.030437895, 66.280312317, 66.567077169, 66.668098229,
66.876019504, 67.153529398, 67.411280972, 67.568303116, 67.792183078,
68.036388017, 68.220476835, 68.538238371, 68.746208712, 68.978088023,
69.146200695, 69.329485779, 69.568059209, 69.78265228, 69.965392013,
70.13902224, 70.414842665, 70.672375649, 70.935584586, 71.15466812,
71.358832275, 71.734427789, 71.917152342, 72.25160482, 72.510782546,
72.81863373, 72.929605358, 73.188642125, 73.404066193, 73.573085135,
73.865746695, 74.118720987, 74.31762623, 74.663293522, 74.838072563,
75.018072795, 75.298859172, 75.493911358, 75.903057138, 76.098197981,
76.283834403, 76.805829014, 77.010369865, 77.370662448, 77.652782035,
77.879113174, 78.352568687, 78.621226741, 78.889946671, 79.133525545,
79.250266277, 79.629513093, 79.947720224, 80.289133592, 80.68991408,
80.768956526, 80.962180462, 81.297223925, 81.524013543, 81.831451907,
82.116362556, 82.290840653, 82.71199557, 82.973353985, 83.213231586,
83.464620336, 83.71403628, 83.845167159, 84.343488173, 84.447852247,
84.627562636, 84.845474233, 85.295608815, 85.495343603, 85.72941252,
85.809048562, 86.123827881, 86.342089604, 86.592962508, 86.809753015,
86.914876606, 87.052550521, 87.29994832, 87.550955527, 88.204022449,
88.417579649, 88.623297679, 88.909854371, 89.120577693, 89.354924901,
89.61333951, 89.792352134, 90.180709163, 90.658976948, 90.877280875,
91.174581973, 91.406927886, 91.93217966, 92.14488673, 92.410216048,
92.75437233, 93.041011508, 93.145634142, 93.268377993, 93.469652535,
93.517701033, 93.792905585, 94.067240082, 94.22546962, 94.555739116,
94.834516244, 95.25186722, 95.620761991, 95.993565375, 96.259822101,
96.431534571, 96.867753724, 96.994938808, 97.483226713, 97.653890118,
98.0395559, 98.427455395, 98.626083865, 98.822710254, 99.303279162,
99.418229472, 99.677670451, 99.969224317, 100.268062123, 100.592473231,
100.862156219, 101.156040921, 101.285539401, 101.752031408, 102.093685249,
102.327455735, 102.63438942, 103.245346725, 103.44414504, 103.559115814,
103.680413555, 103.897987566, 104.5600904, 105.121580857, 105.202608762,
105.436705503, 105.614519071, 105.913356763, 106.207835256, 106.389073407,
106.919107595, 107.354935631, 107.467156534, 107.862449231, 107.941813425,
108.061573573, 108.589190594, 109.007206208, 109.028718709, 110.023110606,
110.316757984, 110.415919632, 110.530547094, 112.388542346, 112.441772737,
112.650183117, 113.356986996, 113.459067202, 113.68578268, 113.846433546,
114.10317855, 114.39054056, 114.599508811, 114.811211115, 115.273832451,
115.483938518, 115.974501561, 116.019294721, 116.289748641, 117.199952687,
117.34192979, 117.474547792, 117.514755953, 117.620238882, 118.085802113,
118.412075151, 118.803923459, 119.399609207, 120.201470073, 120.612410313,
120.867811399, 121.151566203, 121.189271773, 121.279708169, 121.724592839,
121.901979262, 122.049292612, 122.117690135, 122.449730819, 122.886017516,
123.625635491, 123.811318881, 124.323971983, 124.389537568, 124.619442442,
125.353659296, 125.439631764, 125.537999334, 125.699394402, 126.061109276,
126.274495562, 126.462611783, 126.634908312, 126.915659094, 127.915750737,
128.011118659, 128.188445176, 128.726958042, 128.938960749, 129.130284989,
129.273995598, 129.276086753), "75" = c(1.9926e-05, 2.0008e-05,
2.0385e-05, 2.1582e-05, 2.1808e-05, 2.1893e-05, 2.1932e-05, 2.2121e-05,
2.2526e-05, 2.3232e-05, 2.3292e-05, 2.3594e-05, 2.379e-05, 2.4026e-05,
2.4275e-05, 2.4877e-05, 2.5259e-05, 2.5502e-05, 2.5611e-05, 2.5698e-05,
2.6289e-05, 2.6438e-05, 2.7329e-05, 2.7536e-05, 2.7644e-05, 2.7718e-05,
2.866e-05, 2.9818e-05, 3.0473e-05, 3.0589e-05, 3.0776e-05, 3.1023e-05,
3.1244e-05, 3.1409e-05, 3.1522e-05, 3.2579e-05, 3.3005e-05, 3.3525e-05,
3.3666e-05, 3.4297e-05, 3.4399e-05, 3.4562e-05, 3.4755e-05, 3.5491e-05,
3.6366e-05, 3.7106e-05, 3.7297e-05, 3.7574e-05, 3.8128e-05, 3.8675e-05,
3.8972e-05, 3.9115e-05, 4.1447e-05, 4.1503e-05, 4.1979e-05, 4.2266e-05,
4.2477e-05, 4.255e-05, 4.3575e-05, 4.3931e-05, 4.4393e-05, 4.4551e-05,
4.5011e-05, 4.5622e-05, 4.6149e-05, 4.6499e-05, 4.7175e-05, 4.783e-05,
4.8241e-05, 4.8287e-05, 4.8868e-05, 4.8988e-05, 4.9798e-05, 5.1023e-05,
5.1304e-05, 5.1746e-05, 5.3192e-05, 5.3692e-05, 5.3907e-05, 5.449e-05,
5.5097e-05, 5.5864e-05, 5.6597e-05, 5.6706e-05, 5.7617e-05, 5.844e-05,
5.8791e-05, 5.9081e-05, 5.985e-05, 6.0431e-05, 6.123e-05, 6.1972e-05,
6.2706e-05, 6.3098e-05, 6.3505e-05, 6.3749e-05, 6.4121e-05, 6.4656e-05,
6.5265e-05, 6.6212e-05, 6.7391e-05, 6.7794e-05, 6.854e-05, 6.93e-05,
7.0571e-05, 7.1609e-05, 7.2166e-05, 7.2955e-05, 7.3822e-05, 7.4783e-05,
7.5537e-05, 7.6474e-05, 7.7525e-05, 7.8667e-05, 7.9505e-05, 8.052e-05,
8.1199e-05, 8.2274e-05, 8.2997e-05, 8.3677e-05, 8.4412e-05, 8.5289e-05,
8.6208e-05, 8.7253e-05, 8.8733e-05, 8.9362e-05, 8.9914e-05, 9.102e-05,
9.1447e-05, 9.2998e-05, 9.4287e-05, 9.5269e-05, 9.6139e-05, 9.6581e-05,
9.7638e-05, 9.9416e-05, 0.00010098, 0.000101366, 0.000102071,
0.000103051, 0.000103755, 0.000104791, 0.000105291, 0.000107071,
0.000108582, 0.000109757, 0.000110478, 0.000111316, 0.000112261,
0.000112986, 0.000113815, 0.000115041, 0.000115839, 0.000117448,
0.000118267, 0.000118827, 0.000119618, 0.000120605, 0.0001222,
0.00012345, 0.000124872, 0.00012605, 0.000127025, 0.000128234,
0.000130543, 0.000131598, 0.000131996, 0.000133812, 0.000135429,
0.000137498, 0.000139364, 0.00014141, 0.000143611, 0.000145601,
0.00014695, 0.000148414, 0.000150361, 0.000152515, 0.000155253,
0.000156881, 0.000158615, 0.000160965, 0.000162523, 0.00016382,
0.00016542, 0.000168138, 0.000169684, 0.000172035, 0.000173632,
0.000175327, 0.000178322, 0.00018106, 0.0001827, 0.000184148,
0.000186076, 0.00018788, 0.000190469, 0.000192447, 0.000194425,
0.000196223, 0.00019755, 0.000200138, 0.000202466, 0.000204901,
0.00020708, 0.000209006, 0.000210608, 0.000212993, 0.000214926,
0.000217167, 0.000219635, 0.000221607, 0.000224589, 0.000227522,
0.000229108, 0.000230898, 0.00023364, 0.000237081, 0.000240049,
0.000244165, 0.00024663, 0.000249565, 0.000251697, 0.000255026,
0.000256873, 0.000260588, 0.000263516, 0.000265781, 0.000269424,
0.000272221, 0.000275057, 0.000276965, 0.000279901, 0.000282678,
0.000285924, 0.000289275, 0.000294005, 0.000296985, 0.000300995,
0.000305178, 0.000309034, 0.000312268, 0.000315977, 0.000320739,
0.000326553, 0.000331537, 0.000336519, 0.000339973, 0.000343915,
0.000347619, 0.000351954, 0.00035589, 0.000359344, 0.000363802,
0.000369434, 0.000374037, 0.000377876, 0.000383064, 0.000386987,
0.000390008, 0.000394365, 0.000398859, 0.000403114, 0.000407484,
0.000412573, 0.000416636, 0.000420671, 0.000426284, 0.000431252,
0.000437742, 0.000441717, 0.000447633, 0.000452455, 0.000462493,
0.000467497, 0.000473302, 0.000477663, 0.000482796, 0.000487487,
0.000494469, 0.000502626, 0.000508074, 0.000511598, 0.000518444,
0.000524746, 0.000529674, 0.000536019, 0.000542718, 0.00054819,
0.000554928, 0.000562425, 0.000568859, 0.00057505, 0.000580535,
0.000588557, 0.000595246, 0.00060263, 0.000610146, 0.000616631,
0.000624308, 0.000630953, 0.000637526, 0.000643568, 0.0006489,
0.000657013, 0.000666181, 0.000671517, 0.000680816, 0.000688438,
0.000696725, 0.000706144, 0.000714238, 0.000721153, 0.000732102,
0.000739255, 0.000746498, 0.000756852, 0.000764514, 0.000773356,
0.00078262, 0.000793557, 0.000802502, 0.000812878, 0.000823891,
0.000832964, 0.00084028, 0.000851717, 0.000863834, 0.000875031,
0.00088575, 0.000895026, 0.000904352, 0.000914788, 0.000925746,
0.000937002, 0.000947436, 0.000956921, 0.000967128, 0.000979563,
0.00099199, 0.001004034, 0.001007491, 0.001021799, 0.001034835,
0.001046194, 0.001059011, 0.001071378, 0.001082573, 0.001097574,
0.001109905, 0.001121798, 0.001135039, 0.001151245, 0.001163096,
0.001175644, 0.001188422, 0.001203852, 0.001217752, 0.001231812,
0.001243642, 0.001256653, 0.001274734, 0.001289672, 0.001303188,
0.00131862, 0.001333604, 0.001349962, 0.001364803, 0.001379406,
0.001398632, 0.001414492, 0.001429183, 0.001445884, 0.001463142,
0.001481469, 0.001496111, 0.00151357, 0.001533614, 0.001550292,
0.001567589, 0.001583405, 0.00159848, 0.001615945, 0.00163198,
0.001651671, 0.001670306, 0.001688008, 0.001707006, 0.001729937,
0.001748799, 0.001767137, 0.001783467, 0.001800725, 0.001819795,
0.001841482, 0.0018609, 0.001882537, 0.001904729, 0.001925837,
0.001949627, 0.001971304, 0.001993815, 0.002018549, 0.002039659,
0.002062082, 0.002085406, 0.002108171, 0.002132534, 0.002156637,
0.002178771, 0.002203661, 0.002227438, 0.002248875, 0.002278595,
0.002307573, 0.002333736, 0.002359829, 0.002384013, 0.002409816,
0.002436586, 0.002463893, 0.002495421, 0.002526547, 0.002557696,
0.002585832, 0.002617665, 0.002648936, 0.002678392, 0.002708905,
0.002743956, 0.002773472, 0.002804792, 0.00283807, 0.002873017,
0.002903934, 0.002936393, 0.002967045, 0.002998175, 0.003029929,
0.003063566, 0.003099063, 0.003137308, 0.003173732, 0.003214423,
0.003251052, 0.003286017, 0.003325581, 0.003365557, 0.003408106,
0.003443784, 0.003485741, 0.003531565, 0.003570438, 0.003610815,
0.003654142, 0.003696675, 0.003737082, 0.003782872, 0.003822526,
0.003866892, 0.003908356, 0.00395455, 0.003999571, 0.004045878,
0.004088984, 0.004142262, 0.004188075, 0.004238375, 0.004281831,
0.004331384, 0.00437894, 0.004427566, 0.004482056, 0.004533626,
0.004584114, 0.004639004, 0.004696475, 0.004749003, 0.004805075,
0.004861379, 0.004916152, 0.004979107, 0.005041756, 0.005100325,
0.005158877, 0.005220914, 0.005281421, 0.005339962, 0.00540595,
0.005463467, 0.005532255, 0.005597708, 0.005663057, 0.005725101,
0.005793743, 0.005861741, 0.005927356, 0.005999203, 0.006073196,
0.006141068, 0.006219749, 0.006286931, 0.00635649, 0.006427204,
0.006501247, 0.006573587, 0.006657182, 0.00673329, 0.00681637,
0.006897579, 0.006979872, 0.007064968, 0.007148851, 0.007229018,
0.007313727, 0.007399816, 0.007481987, 0.007561429, 0.007651177,
0.007737684, 0.007829306, 0.007918571, 0.008009056, 0.008102338,
0.008199902, 0.008293835, 0.008388583, 0.008489969, 0.008584689,
0.008682444, 0.00877732, 0.008874165, 0.008973627, 0.009078272,
0.009179862, 0.009281871, 0.009393476, 0.00949625, 0.009608853,
0.009721187, 0.009833293, 0.009920112, 0.009941647, 0.010053759,
0.010171958, 0.010287587, 0.010402661, 0.010515532, 0.010635243,
0.010750055, 0.010870614, 0.011000206, 0.011124979, 0.011257869,
0.011391389, 0.011514796, 0.011645585, 0.01179006, 0.0119261,
0.012053551, 0.01218608, 0.012321871, 0.012466541, 0.012615066,
0.012757039, 0.012907676, 0.01305624, 0.013220621, 0.013385722,
0.013546751, 0.013706813, 0.013861945, 0.014011276, 0.014182509,
0.014345854, 0.014512663, 0.014689881, 0.014865428, 0.015037275,
0.015214821, 0.015391351, 0.01556956, 0.015754579, 0.015931899,
0.016116862, 0.016302905, 0.01649429, 0.016682245, 0.016871728,
0.017066906, 0.017268368, 0.017469644, 0.017669236, 0.017880784,
0.018101856, 0.018311386, 0.01852651, 0.018741681, 0.018953572,
0.019179406, 0.01939771, 0.019618888, 0.019841391, 0.020076931,
0.020306719, 0.020538202, 0.020776628, 0.021020014, 0.021260945,
0.021503823, 0.021755277, 0.02200509, 0.022260713, 0.02252413,
0.022767544, 0.023026029, 0.023292419, 0.023568785, 0.023839924,
0.02411701, 0.024400725, 0.024677682, 0.024964285, 0.02526059,
0.025567869, 0.025857449, 0.026158686, 0.026460384, 0.026776368,
0.02708845, 0.027400763, 0.027710941, 0.028032315, 0.028361886,
0.028692197, 0.029028536, 0.029357258, 0.029715139, 0.030068717,
0.030421563, 0.030767706, 0.031115616, 0.031471277, 0.031833321,
0.032194297, 0.032547574, 0.032923893, 0.033290531, 0.033682553,
0.034094146, 0.03448713, 0.03489605, 0.035279167, 0.035686264,
0.036091058, 0.036516668, 0.036934399, 0.037354829, 0.037795679,
0.038240443, 0.038689519, 0.039131001, 0.039581493, 0.04005133,
0.040498717, 0.040957226, 0.041448159, 0.041932716, 0.042422303,
0.042913717, 0.043423282, 0.04391653, 0.044424687, 0.044935394,
0.04545522, 0.045998804, 0.046528047, 0.047059423, 0.047618004,
0.048165238, 0.048737574, 0.049293548, 0.049877354, 0.050473762,
0.051064636, 0.051672303, 0.052296108, 0.052897759, 0.053515141,
0.054137023, 0.054791101, 0.055432374, 0.056070315, 0.056718692,
0.05736932, 0.058047696, 0.058708263, 0.059393637, 0.060077602,
0.060783711, 0.061476152, 0.062188898, 0.062909171, 0.063633024,
0.064362186, 0.065123598, 0.065879092, 0.066621396, 0.067400083,
0.068172378, 0.068974947, 0.069775179, 0.070582042, 0.071418064,
0.072248371, 0.073093379, 0.073943475, 0.074809196, 0.07568846,
0.076565326, 0.077466479, 0.078366482, 0.079252339, 0.080184663,
0.081110283, 0.082040155, 0.083002517, 0.083970125, 0.084948654,
0.085941601, 0.086960742, 0.087992583, 0.08899445, 0.090020011,
0.091052404, 0.09211079, 0.093185065, 0.094248006, 0.095307949,
0.096406283, 0.097535818, 0.098625607, 0.099534964, 0.099775301,
0.100926136, 0.102106921, 0.103297996, 0.104503634, 0.105724141,
0.106947782, 0.108205992, 0.109433378, 0.11072439, 0.111967855,
0.113264713, 0.11457501, 0.115910299, 0.117258819, 0.11859499,
0.119967924, 0.121341871, 0.122731108, 0.124143687, 0.125602626,
0.12705951, 0.128534972, 0.130018761, 0.131521904, 0.133037789,
0.134596399, 0.136163596, 0.137714477, 0.139330786, 0.140950401,
0.142600892, 0.144248995, 0.145928493, 0.147651082, 0.149388627,
0.151120279, 0.152864596, 0.154638043, 0.156424667, 0.158263223,
0.160142953, 0.162014647, 0.163892394, 0.165831802, 0.167742732,
0.169709491, 0.171693788, 0.173660948, 0.175678456, 0.177729151,
0.17979807, 0.181900204, 0.184020595, 0.186167108, 0.188327432,
0.190563099, 0.192771184, 0.195056539, 0.197351085, 0.199661189,
0.201977288, 0.204351732, 0.206732127, 0.209165251, 0.211638347,
0.214092202, 0.216589294, 0.219095417, 0.221674396, 0.224235797,
0.226830966, 0.229512624, 0.232226657, 0.234941318, 0.237686438,
0.24044084, 0.243244063, 0.246108608, 0.248979895, 0.251888442,
0.254846968, 0.25782899, 0.260850077, 0.263921579, 0.267021944,
0.270164609, 0.273343468, 0.276547183, 0.279789516, 0.283120036,
0.28644187, 0.289827667, 0.29322163, 0.296697477, 0.300240509,
0.301020176, 0.303822864, 0.307442477, 0.311060961, 0.31474271,
0.31849776, 0.322254681, 0.326029374, 0.329871273, 0.333805906,
0.33782913, 0.341853439, 0.34589863, 0.349986043, 0.354100448,
0.35830011, 0.36258363, 0.366895362, 0.371307911, 0.375721798,
0.380250281, 0.384829638, 0.389461335, 0.394152556, 0.398876919,
0.403633452, 0.404596547, 0.408479137, 0.413354503, 0.418357859,
0.423395861, 0.428508426, 0.433673496, 0.43886669, 0.444141393,
0.449460405, 0.454891907, 0.460378974, 0.465909258, 0.471575238,
0.477262066, 0.483017407, 0.48887837, 0.494823834, 0.50083452,
0.506935618, 0.513120632, 0.519344357, 0.525821422, 0.532317374,
0.538783143, 0.545463385, 0.552146751, 0.558934882, 0.565834852,
0.572823254, 0.579915645, 0.58705497, 0.594359866, 0.601720611,
0.609257657, 0.616897194, 0.620159999, 0.62463396, 0.632393569,
0.640306494, 0.648337399, 0.656441161, 0.664700569, 0.673179752,
0.681669818, 0.690301043, 0.69909318, 0.70803116, 0.71705569,
0.726180231, 0.735548909, 0.744944668, 0.754505112, 0.764259579,
0.774320322, 0.784429532, 0.794655629, 0.805076063, 0.815638322,
0.826314588, 0.837193653, 0.848337813, 0.852889134, 0.85954841,
0.870978584, 0.882589914, 0.894466242, 0.906510638, 0.918772045,
0.931160869, 0.94386571, 0.956833184, 0.97005893, 0.983409303,
0.996936375, 1.010830571, 1.024997707, 1.039404075, 1.0542006,
1.069210224, 1.084432077, 1.100002002, 1.112642294, 1.141317016,
1.157242094, 1.173070036, 1.18897463, 1.204713797, 1.220591982,
1.236416104, 1.252055501, 1.267770984, 1.283380634, 1.298955412,
1.314632062, 1.33015199, 1.345696415, 1.361239212, 1.376724528,
1.392135468, 1.407487585, 1.416504332, 1.422844184, 1.438102779,
1.453350966, 1.468743558, 1.484085434, 1.499376048, 1.514703681,
1.529933508, 1.545238921, 1.560487812, 1.575774438, 1.591063512,
1.606280938, 1.621482583, 1.6367233, 1.651985594, 1.667254002,
1.68237079, 1.697580684, 1.712715106, 1.727904121, 1.743050873,
1.758237151, 1.773420278, 1.78869453, 1.797495188, 1.80391898,
1.819106079, 1.834352044, 1.849563365, 1.864871997, 1.880224102,
1.895492924, 1.910840162, 1.926145125, 1.941405416, 1.956831584,
1.972226825, 1.987661502, 2.002956008, 2.018485332, 2.033908516,
2.049275877, 2.064611737, 2.0799798, 2.095454639, 2.111095245,
2.12675087, 2.142222018, 2.15797565, 2.173704529, 2.189456198,
2.205400761, 2.221282222, 2.237220467, 2.253136312, 2.269053787,
2.28517093, 2.301236118, 2.317465544, 2.333534138, 2.346277805,
2.349648191, 2.365703175, 2.38190189, 2.398370982, 2.414859323,
2.431239836, 2.447629715, 2.464217304, 2.480965837, 2.497673828,
2.514382915, 2.531469327, 2.548521737, 2.565471271, 2.582595594,
2.599788525, 2.617005884, 2.634075327, 2.6514722, 2.668960707,
2.686513168, 2.70389386, 2.721777396, 2.739497723, 2.757252035,
2.771347314, 2.77528088, 2.79320824, 2.811452934, 2.82939088,
2.847771857, 2.866365006, 2.885207649, 2.904015954, 2.922697479,
2.941483914, 2.960662016, 2.980108786, 2.999380427, 3.018952193,
3.03875265, 3.058800515, 3.078608031, 3.098464055, 3.118829379,
3.139157872, 3.159575244, 3.180217975, 3.200909519, 3.221773896,
3.243112627, 3.264252457, 3.28578577, 3.3074665, 3.329095669,
3.350951741, 3.373147188, 3.39533351, 3.417678781, 3.440309289,
3.463187922, 3.486232028, 3.509345268, 3.532978825, 3.556651899,
3.580404835, 3.604247218, 3.628168054, 3.652336168, 3.676881682,
3.70192819, 3.72684761, 3.752158735, 3.77777219, 3.803777186,
3.829523494, 3.855742094, 3.88182545, 3.908357577, 3.935022192,
3.962317667, 3.989403595, 4.017343416, 4.045545161, 4.073390654,
4.102087489, 4.130816766, 4.159311438, 4.18759797, 4.216707911,
4.245456245, 4.275356415, 4.305198855, 4.334894942, 4.365032224,
4.395287998, 4.426471548, 4.457694052, 4.488730943, 4.519695158,
4.551107828, 4.583574006, 4.615375526, 4.648347284, 4.681061221,
4.714543963, 4.747577396, 4.781079783, 4.815308869, 4.849310177,
4.882601056, 4.916964185, 4.951199688, 4.985807049, 5.020554254,
5.056394255, 5.091424678, 5.127053192, 5.16217218, 5.197471957,
5.234126779, 5.270361693, 5.277682715, 5.307628203, 5.344796745,
5.382297963, 5.419496037, 5.456939775, 5.495003912, 5.532166183,
5.570861725, 5.608693106, 5.64763753, 5.686537552, 5.725483454,
5.765230705, 5.803804082, 5.844145514, 5.883305272, 5.923085301,
5.964095052, 6.00443908, 6.0448316, 6.08618218, 6.127861547,
6.169847985, 6.211983045, 6.253548125, 6.296546025, 6.33917013,
6.382581227, 6.425603351, 6.468285203, 6.511950696, 6.554148612,
6.596927605, 6.640524909, 6.684520057, 6.727936756, 6.774671692,
6.819953404, 6.863777119, 6.90748279, 6.952424696, 6.996961398,
7.041201208, 7.086495295, 7.134225719, 7.179714076, 7.225026377,
7.269975207, 7.316590615, 7.364152495, 7.410249915, 7.455832383,
7.504742777, 7.551713274, 7.601749543, 7.650030883, 7.699114264,
7.749564767, 7.797259042, 7.848053955, 7.897246087, 7.94717801,
7.996592629, 8.044889949, 8.09447516, 8.145849755, 8.198166523,
8.249633886, 8.301497645, 8.353618212, 8.405739836, 8.457896068,
8.510419824, 8.561409779, 8.614753222, 8.666467958, 8.717875551,
8.770023992, 8.82349278, 8.877000517, 8.930939228, 8.986553589,
9.039634575, 9.095522927, 9.151713275, 9.208592527, 9.263056447,
9.320469404, 9.377056696, 9.433645958, 9.491680389, 9.549224513,
9.607619134, 9.666848681, 9.724548232, 9.781688562, 9.839618541,
9.897930122, 9.956383476, 10.013881394, 10.073535228, 10.130293585,
10.189173995, 10.249983751, 10.311422134, 10.373434804, 10.434573126,
10.496463866, 10.55719245, 10.619278297, 10.679100645, 10.740707208,
10.805472699, 10.866241364, 10.92859632, 10.991682191, 11.05411833,
11.119366358, 11.182538908, 11.247821358, 11.312081147, 11.375355457,
11.438995013, 11.505189028, 11.571981389, 11.636623618, 11.704395262,
11.770973303, 11.834760219, 11.900196968, 11.967739342, 12.034635477,
12.103737641, 12.17111953, 12.238577634, 12.308895662, 12.375763365,
12.446143734, 12.516108013, 12.586104215, 12.657418491, 12.726160372,
12.798246266, 12.869004636, 12.936141032, 13.005716245, 13.078796026,
13.150464478, 13.223341613, 13.294087529, 13.364159649, 13.437415093,
13.511889343, 13.584978458, 13.660016824, 13.733562886, 13.806801166,
13.881113245, 13.952892544, 14.028039928, 14.100924398, 14.180216904,
14.259423078, 14.336756791, 14.40713669, 14.481695855, 14.559252676,
14.634481309, 14.711413257, 14.791474631, 14.869100162, 14.947552818,
15.031793992, 15.112869303, 15.195521906, 15.272167653, 15.355780651,
15.441853384, 15.52407653, 15.605226572, 15.691193553, 15.77467161,
15.856130316, 15.936933808, 16.019976263, 16.102652024, 16.182535717,
16.269014223, 16.351780713, 16.428327872, 16.512204866, 16.594821207,
16.679970777, 16.765054053, 16.849974904, 16.933127822, 17.017952455,
17.106935023, 17.19648364, 17.285326774, 17.304607969, 17.372270971,
17.460085835, 17.547278741, 17.631440514, 17.722730431, 17.811317211,
17.897301835, 17.982707413, 18.076088039, 18.16927638, 18.263898136,
18.358171517, 18.446661222, 18.531759673, 18.622134156, 18.716676115,
18.805334669, 18.902017815, 18.993888292, 19.094862128, 19.181136274,
19.275003927, 19.370737827, 19.467120717, 19.562055345, 19.64512511,
19.748171793, 19.839827758, 19.93778789, 20.031795843, 20.118916978,
20.204437147, 20.302982924, 20.407862244, 20.511769569, 20.597692651,
20.699736762, 20.804408365, 20.907905706, 21.005888065, 21.099619388,
21.193898124, 21.28074048, 21.381380688, 21.470932563, 21.578231155,
21.686058427, 21.794504976, 21.903250412, 22.003328965, 22.105037862,
22.221257761, 22.337541026, 22.456677064, 22.554926362, 22.659164215,
22.758755506, 22.864905868, 22.969425364, 23.081267442, 23.185856518,
23.295082775, 23.399981447, 23.516633181, 23.626276384, 23.738106656,
23.843910772, 23.957507751, 24.055839959, 24.167188195, 24.272502044,
24.378286958, 24.480550973, 24.57802788, 24.68133711, 24.796400786,
24.903087198, 25.013433162, 25.112367595, 25.220923041, 25.330708201,
25.449579765, 25.568182702, 25.678438284, 25.80768823, 25.917591706,
26.027798168, 26.152685572, 26.271315167, 26.381546263, 26.498783511,
26.616629148, 26.734514231, 26.85367742, 26.957470646, 27.099405707,
27.218349329, 27.328524982, 27.450770644, 27.581925572, 27.712541571,
27.852762474, 27.981626123, 28.118173978, 28.239999546, 28.350794453,
28.461865591, 28.587364988, 28.714356944, 28.850380935, 28.969908021,
29.12617478, 29.235195153, 29.374700744, 29.512633313, 29.64618087,
29.789214624, 29.910030091, 30.03279491, 30.176284284, 30.32211944,
30.476265898, 30.596481795, 30.714959588, 30.855075261, 30.970987043,
31.083902994, 31.223297639, 31.355319137, 31.485166049, 31.580677831,
31.702950699, 31.829512719, 31.943029383, 32.083042359, 32.206154047,
32.362078403, 32.491706552, 32.624351548, 32.738154461, 32.856838211,
33.018268215, 33.169821625, 33.30814675, 33.445449395, 33.589571954,
33.714444741, 33.846786265, 33.978939757, 34.148876609, 34.292813058,
34.444136502, 34.595927199, 34.721510068, 34.882286746, 35.013810563,
35.149287073, 35.270380352, 35.41909007, 35.583144206, 35.711651972,
35.839920573, 36.041803837, 36.162343164, 36.312263014, 36.470440381,
36.613487927, 36.762143808, 36.924122027, 37.059026101, 37.220382201,
37.369667734, 37.544005664, 37.729244247, 37.89057017, 38.038584742,
38.183405722, 38.35464719, 38.512190876, 38.679468137, 38.868850819,
39.011377799, 39.142450523, 39.30596865, 39.462846852, 39.624404526,
39.78832261, 39.953358704, 40.12666474, 40.266996137, 40.444407078,
40.6159048, 40.771834644, 40.92963362, 41.070938744, 41.226953373,
41.383947689, 41.571865356, 41.721915694, 41.867599504, 41.909143038,
42.031467459, 42.210274668, 42.360781507, 42.543526708, 42.715794078,
42.902716592, 43.088887824, 43.264244057, 43.456066573, 43.635969172,
43.798870706, 43.935138896, 44.067916026, 44.252385034, 44.417963645,
44.630029059, 44.830041235, 44.989434013, 45.175272988, 45.311409076,
45.510096405, 45.673558474, 45.852563254, 46.047641437, 46.237139164,
46.39251055, 46.601767237, 46.768514844, 46.957670058, 47.14206772,
47.348504686, 47.555871513, 47.716277307, 47.880854384, 48.070019998,
48.259259903, 48.472956642, 48.651268333, 48.799169225, 49.01739959,
49.240922184, 49.420236047, 49.682317958, 49.891484443, 50.064214616,
50.227744357, 50.404202686, 50.544198606, 50.749953891, 50.982728697,
51.143028814, 51.318085554, 51.468463292, 51.709410498, 51.887496301,
52.098173329, 52.260203224, 52.487273202, 52.693807247, 52.845124318,
53.012293732, 53.20033269, 53.343165262, 53.561980188, 53.71563949,
53.897688012, 54.07659027, 54.287568351, 54.465964384, 54.647452903,
54.835112197, 55.011298928, 55.182483061, 55.303688197, 55.511966273,
55.71434993, 55.896858167, 56.085747289, 56.285416279, 56.475630291,
56.663452853, 56.877077029, 57.018012688, 57.195300015, 57.387018459,
57.574874648, 57.866882074, 58.027368222, 58.208390607, 58.404040593,
58.623124666, 58.829725198, 59.166823694, 59.430100223, 59.60603961,
59.806570561, 60.003631179, 60.166573071, 60.320559683, 60.573054296,
60.786945053, 60.980205454, 61.218285773, 61.403517434, 61.619421285,
61.83886515, 62.061832259, 62.30657404, 62.518616714, 62.758532932,
62.927951721, 63.073012068, 63.320583569, 63.588250085, 63.841634594,
64.064181496, 64.326080396, 64.702158993, 64.944408273, 65.232640978,
65.456460594, 65.714085003, 65.833353152, 66.019562662, 66.224459687,
66.521079564, 66.751490521, 67.055417213, 67.244944786, 67.535851374,
67.896088983, 68.196473568, 68.285622329, 68.433520419, 68.655454659,
68.892914195, 69.078107255, 69.314079425, 69.696814695, 69.910080373,
70.13959349, 70.337922612, 70.520450063, 70.654999377, 70.919420464,
71.106304871, 71.328757439, 71.562597479, 71.732208206, 71.949178952,
72.420288867, 72.620514649, 72.887954861, 73.251697074, 73.477490762,
73.76481754, 74.056946606, 74.309308599, 74.533423702, 74.810856726,
75.2519076, 75.378175399, 75.569660254, 75.937143395, 76.234180576,
76.507831729, 76.652222311, 76.898308564, 77.022980299, 77.233955803,
77.63846302, 77.813969091, 78.051191873, 78.476617712, 78.850221025,
79.105752548, 79.516659362, 79.695450931, 80.054861545, 80.187698877,
80.423007372, 80.630659604, 80.833678117, 81.056453312, 81.25905194,
81.8108268, 81.889625514, 82.227491539, 82.395789473, 82.628076386,
82.763482731, 83.324100079, 83.631904737, 83.860563976, 84.046065875,
84.375308042, 84.680886989, 84.844806616, 84.977897121, 85.2524766,
85.597544194, 85.930482941, 86.125410227, 86.402207327, 86.663472862,
86.949491524, 87.067948762, 87.301402081, 87.604235639, 87.955850825,
88.349407016, 88.648514334, 88.854421481, 89.001137487, 89.519207705,
89.883249445, 90.242299437, 90.618754244, 90.738178414, 91.207054698,
91.301086864, 91.786519175, 92.103134996, 92.66012925, 92.842152604,
93.117521056, 93.286580601, 93.61848019, 93.993129831, 94.205392733,
94.33870803, 94.638488965, 94.878549297, 95.093153733, 95.217215465,
95.350325931, 95.638582933, 95.927191472, 96.094288691, 96.495907804,
96.80149314, 97.400607116, 97.888872944, 98.32235967, 98.549743596,
98.773698279, 98.873725165, 99.290987967, 99.698764828, 100.117475461,
100.804705123, 101.155015129, 101.245066689, 101.514609734, 101.818879871,
102.013354763, 102.279856122, 102.462346101, 103.092512895, 103.462272971,
103.585366391, 103.829659957, 104.092166849, 104.267376157, 104.766862567,
105.042716121, 105.201178247, 105.679878216, 105.967941348, 106.246069172,
107.114241456, 107.395560802, 107.827243507, 108.102552103, 108.585590472,
108.68997585, 108.97575373, 110.290141728, 111.013226163, 111.326648178,
111.62406051, 111.955968995, 112.50422195, 112.545540811, 112.87594895,
113.500456736, 113.833582295, 114.401290079, 114.631838397, 114.820868982,
115.095091011, 116.031059517, 116.348265352, 116.813482772, 117.067135774,
118.018491859, 118.373663213, 118.496056746, 119.66993259, 119.890087885,
119.990408531, 120.52910327, 121.002907001, 121.351553516, 121.382846723,
121.487422562, 121.635724257, 121.796160441, 122.783619768, 123.088969236,
123.247745169, 123.388805589, 124.008948741, 124.202106345, 124.47385446,
124.64764597, 125.140503348, 125.256850457, 126.046137541, 126.811279699,
127.4743812, 127.757850093, 127.921146115, 128.248710705, 128.50036943,
129.005970643, 129.136799378, 129.61042453, 129.79706749, 129.921992799,
130.057867798, 130.17787838, 131.503872695, 131.680741846, 131.785650209,
132.075579867, 132.499953229, 132.731408048, 133.044622802, 133.188601703
), "100" = c(2.2046e-05, 2.21e-05, 2.2301e-05, 2.2673e-05, 2.2864e-05,
2.2902e-05, 2.3154e-05, 2.3804e-05, 2.4213e-05, 2.4652e-05, 2.4896e-05,
2.501e-05, 2.5371e-05, 2.5492e-05, 2.5644e-05, 2.6348e-05, 2.6398e-05,
2.6884e-05, 2.7076e-05, 2.739e-05, 2.7511e-05, 2.7671e-05, 2.7876e-05,
2.7943e-05, 2.8361e-05, 2.8406e-05, 2.8641e-05, 2.873e-05, 2.9152e-05,
2.9693e-05, 2.9826e-05, 2.9937e-05, 3.0122e-05, 3.0193e-05, 3.0999e-05,
3.1048e-05, 3.1391e-05, 3.1818e-05, 3.2355e-05, 3.2758e-05, 3.2914e-05,
3.341e-05, 3.4117e-05, 3.4451e-05, 3.4561e-05, 3.4831e-05, 3.5301e-05,
3.5519e-05, 3.5736e-05, 3.5931e-05, 3.6099e-05, 3.6304e-05, 3.6699e-05,
3.6962e-05, 3.7482e-05, 3.7777e-05, 3.8233e-05, 3.8477e-05, 3.895e-05,
3.8983e-05, 3.9665e-05, 4.0027e-05, 4.0265e-05, 4.0605e-05, 4.0652e-05,
4.0934e-05, 4.148e-05, 4.2196e-05, 4.2878e-05, 4.3157e-05, 4.3617e-05,
4.393e-05, 4.4211e-05, 4.4694e-05, 4.495e-05, 4.5376e-05, 4.5985e-05,
4.6239e-05, 4.6505e-05, 4.7862e-05, 4.8719e-05, 4.9517e-05, 5.0171e-05,
5.0946e-05, 5.1352e-05, 5.1554e-05, 5.2033e-05, 5.2277e-05, 5.2765e-05,
5.3696e-05, 5.4182e-05, 5.4596e-05, 5.5192e-05, 5.5617e-05, 5.5971e-05,
5.6306e-05, 5.6901e-05, 5.7602e-05, 5.7909e-05, 5.8582e-05, 5.8838e-05,
5.9969e-05, 6.0744e-05, 6.1435e-05, 6.2104e-05, 6.233e-05, 6.3555e-05,
6.4948e-05, 6.5531e-05, 6.5983e-05, 6.6404e-05, 6.7249e-05, 6.7468e-05,
6.9046e-05, 6.9505e-05, 7.0207e-05, 7.12e-05, 7.2618e-05, 7.302e-05,
7.389e-05, 7.4158e-05, 7.4886e-05, 7.518e-05, 7.6717e-05, 7.7197e-05,
7.7796e-05, 7.9113e-05, 8.0554e-05, 8.1271e-05, 8.3116e-05, 8.4294e-05,
8.5022e-05, 8.5699e-05, 8.6484e-05, 8.787e-05, 8.8455e-05, 8.9181e-05,
8.9901e-05, 9.1068e-05, 9.1854e-05, 9.3009e-05, 9.4945e-05, 9.5928e-05,
9.6299e-05, 9.7374e-05, 9.8664e-05, 0.000100159, 0.000101589,
0.000102803, 0.000103672, 0.000104473, 0.000106613, 0.000107287,
0.000107851, 0.00010897, 0.000111109, 0.000111923, 0.000113873,
0.00011492, 0.000115728, 0.000116688, 0.000117544, 0.000119468,
0.000120704, 0.00012276, 0.000124797, 0.000126283, 0.000128897,
0.000129817, 0.000130835, 0.000132083, 0.000134923, 0.000136168,
0.000137598, 0.000139089, 0.000140576, 0.000142969, 0.000143792,
0.000145114, 0.000146444, 0.000148238, 0.000149753, 0.000151957,
0.00015366, 0.000155327, 0.000157008, 0.000158785, 0.000160364,
0.000162864, 0.000164797, 0.000167355, 0.000169446, 0.0001721,
0.000175441, 0.000177425, 0.000179381, 0.000182777, 0.000184808,
0.000187197, 0.000189022, 0.000190955, 0.000192868, 0.000194333,
0.000197013, 0.000198572, 0.000201138, 0.000204289, 0.00020611,
0.000207779, 0.000210597, 0.000212887, 0.000214564, 0.000217453,
0.000219513, 0.000221651, 0.000223656, 0.000225722, 0.000228047,
0.000230927, 0.000234075, 0.000238049, 0.00024159, 0.000243297,
0.00024555, 0.00024943, 0.00025173, 0.000255046, 0.000258552,
0.000261533, 0.000265449, 0.00026829, 0.000272163, 0.000274495,
0.000277602, 0.000279885, 0.000282717, 0.000284675, 0.0002869,
0.000291035, 0.000295737, 0.000299299, 0.000303633, 0.000306551,
0.000308745, 0.00031186, 0.000315565, 0.000319335, 0.00032332,
0.000327046, 0.000331575, 0.000336428, 0.000340346, 0.000344051,
0.000347137, 0.000350297, 0.000354921, 0.000359233, 0.000364497,
0.000366786, 0.000370382, 0.000373492, 0.00037742, 0.000381359,
0.000385329, 0.000389757, 0.000393919, 0.000398907, 0.000404194,
0.000409365, 0.000414027, 0.000420326, 0.000425612, 0.000430728,
0.000435118, 0.000442439, 0.000446824, 0.000452144, 0.000457252,
0.000463168, 0.000467292, 0.000472522, 0.000478102, 0.000482746,
0.000487238, 0.000492684, 0.000498803, 0.000505003, 0.00051364,
0.000519546, 0.000525379, 0.000531061, 0.000536924, 0.000544461,
0.000549227, 0.000555245, 0.000561661, 0.00056926, 0.00057622,
0.000583446, 0.000590375, 0.000595806, 0.000602372, 0.000608627,
0.000615206, 0.000622309, 0.000630504, 0.000637168, 0.000642875,
0.000652259, 0.000661019, 0.000666941, 0.000674891, 0.000681656,
0.000688418, 0.000696433, 0.000707135, 0.000713987, 0.000721125,
0.000727681, 0.000737443, 0.000746213, 0.000753625, 0.000764955,
0.000773955, 0.000782903, 0.000791077, 0.000798653, 0.000807501,
0.000816195, 0.000826463, 0.000839023, 0.000851726, 0.000861493,
0.000871512, 0.000880551, 0.000891295, 0.000903731, 0.000914804,
0.000927258, 0.000937222, 0.000945828, 0.000948729, 0.000959256,
0.000971721, 0.000983289, 0.000993458, 0.001006209, 0.001019447,
0.001030558, 0.00104123, 0.001053811, 0.001067216, 0.001081556,
0.001094365, 0.001106271, 0.001119245, 0.001133595, 0.001148798,
0.001159868, 0.001172947, 0.001186432, 0.001200825, 0.001215312,
0.001228586, 0.001242517, 0.001257438, 0.001271464, 0.001288909,
0.001304378, 0.001316997, 0.001331995, 0.001349445, 0.001363998,
0.001378511, 0.001396551, 0.001414905, 0.001433568, 0.001451659,
0.001469718, 0.00148923, 0.001507355, 0.001525868, 0.001542659,
0.001564372, 0.001582941, 0.00160133, 0.001617605, 0.001637256,
0.001657499, 0.001675688, 0.001694384, 0.001716933, 0.001733465,
0.001751349, 0.001772212, 0.00179033, 0.001812626, 0.001835029,
0.001854687, 0.001876432, 0.001894003, 0.001916278, 0.001943983,
0.001966535, 0.00198976, 0.002013911, 0.002036115, 0.002059274,
0.00208418, 0.00210994, 0.002134507, 0.002156575, 0.002184775,
0.002209424, 0.002233885, 0.002260416, 0.002290511, 0.002316387,
0.002342944, 0.002371213, 0.00240288, 0.002434141, 0.00245808,
0.002487761, 0.002516404, 0.002545423, 0.002574468, 0.002606061,
0.002635302, 0.002661895, 0.002692126, 0.002722019, 0.002757156,
0.002790119, 0.002821882, 0.002861482, 0.002895574, 0.002925966,
0.00296128, 0.002995106, 0.003030188, 0.003064064, 0.003096956,
0.003135267, 0.00316749, 0.00320432, 0.003245203, 0.003278626,
0.003318474, 0.003360552, 0.003398364, 0.003442344, 0.003482602,
0.003518428, 0.003560865, 0.003600476, 0.00364776, 0.003689881,
0.003736952, 0.003773659, 0.003820946, 0.003868191, 0.003916116,
0.003960839, 0.004008783, 0.00405502, 0.004105684, 0.004152341,
0.004202312, 0.004250609, 0.004308767, 0.00436127, 0.004410807,
0.004466608, 0.004518566, 0.004574437, 0.004625613, 0.004678856,
0.00473719, 0.004790274, 0.004849461, 0.00490517, 0.004961351,
0.005017706, 0.005081629, 0.005141006, 0.005198246, 0.005261217,
0.005320535, 0.005376729, 0.005436226, 0.005504335, 0.005571156,
0.00563334, 0.005699229, 0.005765702, 0.005828585, 0.005900045,
0.005971053, 0.00603885, 0.006101531, 0.006175246, 0.006241269,
0.006316367, 0.006392056, 0.006459509, 0.006531461, 0.006610934,
0.006688643, 0.006768093, 0.006852275, 0.006934984, 0.007016588,
0.00709985, 0.007180721, 0.007267687, 0.007355738, 0.007438997,
0.007526743, 0.007615891, 0.007702029, 0.007790993, 0.007885261,
0.007980859, 0.008074399, 0.0081662, 0.00825876, 0.008350244,
0.008453559, 0.008545673, 0.008645358, 0.008745536, 0.008841583,
0.008943474, 0.009046326, 0.009161898, 0.009268581, 0.009369447,
0.00948275, 0.009594321, 0.009701545, 0.009786526, 0.009810919,
0.009932886, 0.010042471, 0.010158005, 0.010276498, 0.010399671,
0.010525016, 0.010655423, 0.010777088, 0.010899858, 0.011024924,
0.011154067, 0.011280999, 0.011418972, 0.011557075, 0.011691381,
0.011829462, 0.011977053, 0.012109036, 0.012248515, 0.012392418,
0.012531099, 0.012670822, 0.012815535, 0.012960321, 0.013109524,
0.01327079, 0.013430489, 0.013593319, 0.013745128, 0.013914402,
0.014082288, 0.014242725, 0.014418746, 0.014588778, 0.014764624,
0.014943992, 0.015126911, 0.015303778, 0.015491292, 0.015674683,
0.015854716, 0.016044596, 0.016232589, 0.016416438, 0.01660696,
0.016804246, 0.016998376, 0.01718748, 0.017387313, 0.017584097,
0.017791622, 0.017993947, 0.018191053, 0.018386193, 0.018596614,
0.018822394, 0.01904355, 0.019266806, 0.019485516, 0.019702643,
0.019931634, 0.020172852, 0.020402193, 0.020621876, 0.020860086,
0.021098006, 0.021341012, 0.02159309, 0.02183181, 0.022086043,
0.022339586, 0.022605876, 0.022870076, 0.023147278, 0.023407464,
0.023671398, 0.023947395, 0.024231613, 0.024528724, 0.024826582,
0.02512088, 0.025418022, 0.025713575, 0.02601329, 0.026315852,
0.026636527, 0.02694876, 0.027267677, 0.027590679, 0.027913887,
0.028233198, 0.028552432, 0.02887989, 0.029220274, 0.029552872,
0.029898033, 0.030249539, 0.030607352, 0.030977626, 0.031331127,
0.031690849, 0.032050711, 0.032420636, 0.032790499, 0.033170926,
0.033542521, 0.033919157, 0.034302012, 0.034704942, 0.035110674,
0.035514701, 0.035915058, 0.036344273, 0.036773563, 0.037211272,
0.037652678, 0.038094323, 0.038539497, 0.038985772, 0.039432974,
0.039886864, 0.040339293, 0.040809738, 0.041278949, 0.041757134,
0.042252859, 0.042755216, 0.043262101, 0.043769459, 0.044263314,
0.04477855, 0.045297733, 0.045836776, 0.046370891, 0.046925339,
0.047471139, 0.048018869, 0.048595733, 0.049157487, 0.049724819,
0.050299999, 0.050874495, 0.051475952, 0.052070672, 0.052676457,
0.053287066, 0.053895259, 0.054539942, 0.055198054, 0.055858149,
0.056512273, 0.057169618, 0.057848225, 0.058536716, 0.059209824,
0.059897728, 0.060591742, 0.061296905, 0.062010538, 0.062730511,
0.063452818, 0.064191814, 0.064939497, 0.065703122, 0.066471837,
0.06725283, 0.068023069, 0.068797405, 0.069594645, 0.070404827,
0.071231306, 0.072059432, 0.072915402, 0.073791186, 0.074645355,
0.075526652, 0.076415047, 0.077312462, 0.078234166, 0.079150766,
0.08009144, 0.08103014, 0.08197015, 0.082935692, 0.083894165,
0.084865678, 0.085855617, 0.086856573, 0.087858083, 0.088874839,
0.089936314, 0.09099475, 0.092074988, 0.093146084, 0.094232442,
0.095321905, 0.096431373, 0.097547208, 0.098683731, 0.0995933,
0.099827242, 0.100975398, 0.102169037, 0.103356076, 0.104536807,
0.105763944, 0.106992721, 0.10823383, 0.109467507, 0.110769463,
0.112049482, 0.113362463, 0.114691779, 0.116049333, 0.117404465,
0.118787076, 0.120178138, 0.121588725, 0.123002351, 0.124433431,
0.125888464, 0.127349591, 0.12886263, 0.130359566, 0.131879476,
0.133399465, 0.134961313, 0.136538028, 0.138134912, 0.13972864,
0.141356954, 0.142996087, 0.144669832, 0.146350795, 0.14806779,
0.149818764, 0.151601433, 0.153360499, 0.15517908, 0.157014687,
0.158863887, 0.16070761, 0.162632037, 0.164548014, 0.166480385,
0.168425172, 0.170402059, 0.172395882, 0.174397754, 0.176438304,
0.178482295, 0.180570446, 0.182682089, 0.184804149, 0.186970571,
0.189164948, 0.191386359, 0.193642581, 0.195926858, 0.198240751,
0.200573818, 0.20295756, 0.205330883, 0.20774982, 0.210195783,
0.212702065, 0.215243494, 0.217822984, 0.220437537, 0.223044558,
0.225688219, 0.228358542, 0.231091417, 0.233853248, 0.236589112,
0.23948006, 0.242320858, 0.245190393, 0.248104698, 0.251094914,
0.254011376, 0.257050623, 0.26008293, 0.263215915, 0.266343363,
0.269526316, 0.272751899, 0.276023754, 0.279328391, 0.28262111,
0.286027925, 0.289408997, 0.292886668, 0.296383077, 0.299915538,
0.303558732, 0.304350055, 0.307226313, 0.310869675, 0.314593496,
0.318418643, 0.322253257, 0.326109848, 0.330038998, 0.333988443,
0.338064394, 0.342114253, 0.346222409, 0.350377858, 0.354588973,
0.358825423, 0.363089137, 0.367461504, 0.371930349, 0.376420158,
0.380967083, 0.385576181, 0.390247848, 0.394989961, 0.399751512,
0.404570381, 0.40947571, 0.410476221, 0.414475444, 0.419543777,
0.424752874, 0.429919498, 0.435181101, 0.440592749, 0.446040336,
0.451577491, 0.457107021, 0.462729794, 0.468495501, 0.474227251,
0.480103454, 0.486033308, 0.492043301, 0.498187011, 0.504444097,
0.510713081, 0.517072875, 0.523450555, 0.529962788, 0.536523114,
0.543274229, 0.550063019, 0.556955819, 0.563903635, 0.570992353,
0.578218935, 0.585544365, 0.59297132, 0.600401585, 0.607940009,
0.615657963, 0.623461627, 0.631397379, 0.634769974, 0.639499839,
0.647651737, 0.65594429, 0.664349968, 0.672928526, 0.681613398,
0.690483814, 0.699432038, 0.708524233, 0.717775239, 0.727220157,
0.736745298, 0.746360846, 0.756174717, 0.766164844, 0.776270447,
0.78651804, 0.796928869, 0.807600517, 0.818389337, 0.829384088,
0.84057276, 0.851921487, 0.863379215, 0.875087697, 0.879969668,
0.88686926, 0.899024963, 0.911400825, 0.92397949, 0.936771252,
0.949771466, 0.963005956, 0.976471183, 0.99028456, 1.004405345,
1.018659807, 1.033152566, 1.047819452, 1.06286159, 1.078099569,
1.093852635, 1.109915739, 1.126151959, 1.14293514, 1.156339871,
1.186881856, 1.203859696, 1.220860226, 1.237895505, 1.254904049,
1.271748183, 1.288825679, 1.305658144, 1.322448372, 1.33926091,
1.356065386, 1.37281981, 1.389619142, 1.406257522, 1.422980881,
1.43964158, 1.456307746, 1.473011247, 1.482822502, 1.489619808,
1.506135825, 1.522706056, 1.539224926, 1.555671109, 1.572176172,
1.588567986, 1.605056301, 1.621663618, 1.638023335, 1.654470689,
1.670894453, 1.687367434, 1.703879777, 1.720203925, 1.736595851,
1.753210728, 1.769607058, 1.786112273, 1.802470983, 1.818964293,
1.835511418, 1.851846055, 1.868359628, 1.884910383, 1.894402161,
1.901439261, 1.91795232, 1.934425073, 1.950872304, 1.967279343,
1.98375715, 2.000314666, 2.016866647, 2.033374692, 2.050029264,
2.0667086, 2.083248359, 2.100037277, 2.116552293, 2.133213734,
2.149958112, 2.16681886, 2.183579198, 2.20030173, 2.217175763,
2.234041363, 2.250879956, 2.267840196, 2.284537026, 2.301511876,
2.318523307, 2.335614945, 2.352642766, 2.369637843, 2.386780605,
2.403792516, 2.421098927, 2.438281062, 2.455445173, 2.472698794,
2.486527371, 2.490116137, 2.507579938, 2.525086997, 2.542548746,
2.560200833, 2.577756627, 2.5952444, 2.613182459, 2.630930061,
2.648940883, 2.666807427, 2.68458309, 2.702773711, 2.720776831,
2.739101141, 2.757189877, 2.775620497, 2.793740507, 2.811962028,
2.830350294, 2.848667268, 2.86734385, 2.886184052, 2.905064442,
2.923998668, 2.9390101, 2.943150294, 2.962183437, 2.981449704,
3.000642345, 3.019785564, 3.039437625, 3.058808576, 3.078618959,
3.09819757, 3.117819728, 3.137971791, 3.158011877, 3.178108836,
3.198371992, 3.21867403, 3.239156828, 3.259472616, 3.280245722,
3.301006653, 3.321587754, 3.342473633, 3.363725877, 3.384706536,
3.406021878, 3.427498187, 3.4490459, 3.470976187, 3.493322042,
3.515322682, 3.537269369, 3.559674066, 3.582290844, 3.604875953,
3.627348109, 3.650113317, 3.673366422, 3.696355935, 3.719719588,
3.743252029, 3.766870589, 3.790759509, 3.815096527, 3.839198126,
3.864137475, 3.888990564, 3.913846911, 3.938894285, 3.963921946,
3.989399698, 4.015418298, 4.041004099, 4.067597668, 4.093953122,
4.120446698, 4.146746566, 4.173615791, 4.200846986, 4.228095521,
4.255818103, 4.283401868, 4.310979784, 4.338932187, 4.36725408,
4.395586481, 4.424315563, 4.45309492, 4.482248547, 4.512078014,
4.541577692, 4.571499693, 4.601740925, 4.632323419, 4.662621363,
4.693679819, 4.724383488, 4.755875798, 4.787320315, 4.818018431,
4.850051024, 4.88226503, 4.914196953, 4.947380229, 4.980092608,
5.012759349, 5.045516966, 5.078501025, 5.111819227, 5.14501304,
5.178681681, 5.212507995, 5.247319271, 5.282283036, 5.316432459,
5.350616567, 5.38591507, 5.422136026, 5.430013901, 5.458016765,
5.494078735, 5.530486495, 5.56626393, 5.603365913, 5.640652422,
5.67819778, 5.715328843, 5.752699777, 5.79045162, 5.828795686,
5.866844549, 5.904919995, 5.943302877, 5.981175325, 6.020577244,
6.05977389, 6.09913535, 6.138060318, 6.178048443, 6.218006988,
6.258842036, 6.299323312, 6.339544871, 6.380274684, 6.421301103,
6.462368406, 6.50421716, 6.545153761, 6.586910134, 6.628721099,
6.671860275, 6.714431467, 6.75660349, 6.797944366, 6.842502434,
6.886304227, 6.930667363, 6.974141956, 7.017360195, 7.06012585,
7.104143951, 7.148391741, 7.193343667, 7.238369711, 7.28317896,
7.329649126, 7.375239126, 7.42205219, 7.466688907, 7.513390045,
7.560233044, 7.606967616, 7.653934088, 7.699869391, 7.745260637,
7.791972645, 7.839678273, 7.885948262, 7.933502658, 7.982642394,
8.031911489, 8.080226923, 8.130292559, 8.17911969, 8.228802789,
8.280008862, 8.327583392, 8.37816072, 8.426945429, 8.476087274,
8.525621151, 8.574839797, 8.625032565, 8.676859253, 8.7270657,
8.777219274, 8.82837904, 8.879377233, 8.93198319, 8.985064181,
9.037970095, 9.090763669, 9.144516265, 9.198725321, 9.252810474,
9.306190453, 9.359590358, 9.413947581, 9.470020327, 9.527052392,
9.57989044, 9.637771191, 9.693645481, 9.751169323, 9.805102991,
9.860345399, 9.915630897, 9.97369207, 10.029033191, 10.085468924,
10.140804448, 10.200965828, 10.259401641, 10.318475519, 10.376793771,
10.433439016, 10.492886126, 10.552438195, 10.608820264, 10.669812397,
10.728797175, 10.789650196, 10.850379627, 10.911164641, 10.973704381,
11.034539097, 11.093195434, 11.15545639, 11.21665481, 11.28052361,
11.33969023, 11.398303471, 11.459060377, 11.519178731, 11.581238884,
11.646276395, 11.708580212, 11.771354322, 11.836272243, 11.900028293,
11.961386512, 12.02415565, 12.090802651, 12.156212822, 12.220935193,
12.286957066, 12.355897377, 12.424869039, 12.491256977, 12.556353552,
12.621772406, 12.684840513, 12.752043393, 12.821182087, 12.890079436,
12.960909682, 13.024719386, 13.095689738, 13.17066366, 13.239831089,
13.310763255, 13.380822999, 13.451375527, 13.521087914, 13.594217229,
13.663094977, 13.735406222, 13.804844295, 13.872677343, 13.942369059,
14.010505666, 14.079112816, 14.156171216, 14.230128573, 14.304846923,
14.379184671, 14.456513715, 14.525274493, 14.599859838, 14.668405941,
14.741376372, 14.815257393, 14.89474761, 14.971833702, 15.04376377,
15.121781474, 15.200915822, 15.280298555, 15.352393181, 15.423803416,
15.502220507, 15.575922936, 15.654307822, 15.731577348, 15.809738635,
15.891520708, 15.96972721, 16.048772081, 16.123507835, 16.206514044,
16.283690581, 16.36451381, 16.445821935, 16.528591311, 16.610320297,
16.694194111, 16.776453764, 16.859360188, 16.943500651, 16.959063376,
17.021200811, 17.102583791, 17.19044808, 17.278698433, 17.358439571,
17.441968914, 17.528900251, 17.607674487, 17.688603229, 17.768515841,
17.855325113, 17.939175191, 18.029878048, 18.119546085, 18.201567426,
18.292648279, 18.380019316, 18.470153125, 18.559600468, 18.64664327,
18.735775238, 18.825143075, 18.914266356, 19.004073327, 19.096984826,
19.188955469, 19.275383735, 19.358411404, 19.449406902, 19.544713716,
19.630525983, 19.728506745, 19.82155505, 19.913445149, 20.007280735,
20.10484248, 20.20467446, 20.290535362, 20.388370672, 20.488837767,
20.590699927, 20.680168669, 20.781468635, 20.876650999, 20.966538192,
21.066923848, 21.178694831, 21.277648113, 21.382762457, 21.476992422,
21.574151326, 21.67093195, 21.780403148, 21.885141879, 21.979236407,
22.078798678, 22.182662889, 22.277610712, 22.376347003, 22.493111864,
22.593437901, 22.69058174, 22.794010536, 22.897957576, 22.999837422,
23.097414365, 23.199347339, 23.295235547, 23.406345878, 23.514225467,
23.62981113, 23.733972532, 23.83622068, 23.947061613, 24.062921186,
24.166106154, 24.276438763, 24.382758917, 24.492325139, 24.59497256,
24.703451588, 24.814289985, 24.92426164, 25.023657822, 25.13515211,
25.231797518, 25.339956542, 25.44574999, 25.552593312, 25.655126542,
25.767646071, 25.889412838, 25.999314167, 26.097166341, 26.215886485,
26.329591416, 26.434876333, 26.533924839, 26.653775203, 26.772271703,
26.891804262, 26.996351829, 27.129442949, 27.247047921, 27.369408734,
27.487745139, 27.614916255, 27.725308356, 27.836802817, 27.961156175,
28.070080767, 28.183588208, 28.311119443, 28.457562731, 28.575420886,
28.705347625, 28.840105259, 28.972406119, 29.098981419, 29.240005607,
29.353952402, 29.472733182, 29.602242944, 29.733795144, 29.866375751,
30.005439545, 30.133054049, 30.255454259, 30.389615649, 30.546894607,
30.642330982, 30.782051309, 30.93658643, 31.052559249, 31.198850709,
31.322558869, 31.471380506, 31.603979137, 31.72544234, 31.840240515,
31.985575088, 32.132429553, 32.265657367, 32.397182346, 32.521850173,
32.637346198, 32.769717315, 32.916455274, 33.045404394, 33.155448101,
33.28403114, 33.420225367, 33.569068106, 33.689533667, 33.82395603,
33.974557826, 34.104331221, 34.246316628, 34.386700889, 34.521054774,
34.682760996, 34.840849773, 34.957633644, 35.10322806, 35.239181792,
35.382101349, 35.514084069, 35.689628799, 35.83150553, 35.955312228,
36.096927987, 36.221487738, 36.356822395, 36.4763809, 36.62917011,
36.758019074, 36.911300201, 37.050911294, 37.183152376, 37.327622307,
37.466525528, 37.605935682, 37.749499387, 37.898630368, 38.033643277,
38.19092779, 38.375971347, 38.556926783, 38.696651303, 38.859005009,
39.043639208, 39.166149747, 39.313774656, 39.477573985, 39.625851318,
39.79004548, 39.947930284, 40.104762855, 40.22982726, 40.390248198,
40.429995205, 40.574490582, 40.748050125, 40.882094422, 41.020672834,
41.152868447, 41.358357078, 41.542392241, 41.719589136, 41.868949186,
42.030249947, 42.207014222, 42.362719817, 42.493494569, 42.613699879,
42.794910096, 42.983803365, 43.181119885, 43.341931375, 43.482367347,
43.692473268, 43.856660784, 43.987115093, 44.170155022, 44.374097191,
44.614708342, 44.807131236, 44.976003439, 45.125612323, 45.29206216,
45.448121491, 45.664460502, 45.838249671, 46.052650926, 46.208676983,
46.354814605, 46.493016039, 46.698729239, 46.864518739, 47.040453742,
47.250687059, 47.467711579, 47.644895878, 47.805933814, 48.008390469,
48.193214281, 48.336504316, 48.569059034, 48.722984792, 48.868317633,
49.045908322, 49.20577429, 49.368452121, 49.54178601, 49.776224491,
49.943205013, 50.160651053, 50.374419997, 50.563856656, 50.742178854,
50.930718463, 51.181323257, 51.342513377, 51.553318429, 51.765256457,
51.920671565, 52.136050864, 52.420457304, 52.549390616, 52.732572244,
52.940375574, 53.175574234, 53.389374086, 53.639780113, 53.822946793,
54.013939091, 54.18552773, 54.413064283, 54.553025502, 54.7274191,
54.940184147, 55.22585911, 55.382498294, 55.606066051, 55.75400631,
56.003101637, 56.209772696, 56.375454098, 56.547715023, 56.840923037,
57.014667061, 57.21297409, 57.408869512, 57.643609796, 57.84090301,
58.027265296, 58.263302623, 58.436553217, 58.618149965, 58.966297802,
59.216261903, 59.388679182, 59.631547824, 59.786333547, 59.929726319,
60.25588042, 60.482343125, 60.647192103, 60.842144648, 61.009536075,
61.22706126, 61.390499036, 61.568693499, 61.715428997, 62.011624773,
62.173440544, 62.427808782, 62.723659699, 62.928446544, 63.206595094,
63.463151485, 63.631145815, 63.813464004, 64.06230361, 64.227632067,
64.434075518, 64.711771639, 64.814283959, 64.982411312, 65.223856116,
65.584715844, 65.895664247, 66.164664316, 66.473435547, 66.69057511,
66.902991511, 67.068492794, 67.178923649, 67.486222539, 67.739891093,
68.031682592, 68.295234532, 68.471388235, 68.729551489, 68.904310088,
69.050840007, 69.243240404, 69.552693348, 69.735247484, 70.031229944,
70.415384293, 70.62426677, 70.78612381, 71.09006887, 71.345737201,
71.575888487, 71.746413144, 71.91680912, 72.101848831, 72.33307302,
72.46407485, 72.750530822, 72.991064102, 73.198649779, 73.315167541,
73.525982376, 73.665188495, 73.932091378, 74.066201909, 74.34601789,
74.585570188, 74.844896049, 75.018144794, 75.276530586, 75.442334375,
75.650678888, 76.086573621, 76.421626812, 76.763048913, 76.878500909,
76.996163783, 77.454951235, 77.627688589, 77.810469312, 78.165746475,
78.423740174, 78.654668466, 78.807962921, 79.116997279, 79.494288592,
79.846985486, 79.94463211, 80.338343816, 80.890953582, 81.123999444,
81.378086177, 81.648523841, 81.914838547, 82.159496175, 82.430691783,
82.777696792, 82.949558673, 83.508922841, 83.757115676, 83.914843051,
84.067801761, 84.167003793, 84.491795055, 84.676207138, 84.797461176,
85.048513174, 85.31743068, 85.468104826, 85.584192839, 85.793370652,
86.378376148, 86.887814755, 87.076988215, 87.29379477, 87.558841731,
88.189459906, 88.681822474, 88.958612062, 89.093633613, 89.299184416,
89.457458072, 89.81429918, 89.992189951, 90.200241424, 90.558764827,
90.909224696, 91.076723584, 91.360612969, 91.549382195, 91.768541045,
92.093095109, 92.362422978, 92.728440534, 92.968600798, 93.135484586,
93.512125227, 93.541457739, 93.900895311, 94.397608426, 94.546552378,
95.0077362, 95.347642911, 95.520700093, 96.110169384, 96.579173056,
97.087215736, 97.319315911, 97.6425654, 98.043919817, 98.219700571,
98.644221752, 98.959413669, 99.055218864, 99.33465987, 99.635709985,
100.187036999, 100.470373636, 100.721818452, 101.13885527, 101.502008151,
101.579829438, 102.162832984, 102.649431653, 103.082088613, 103.172853308,
103.280575482, 103.906497463, 104.294613095, 104.838589885, 105.003941502,
105.363634529, 105.558222606, 105.633229226, 105.863056616, 105.904711951,
106.263499197, 106.658475248, 106.768153111, 106.980548714, 107.332972858,
107.652033155, 107.988911218, 108.099169384, 108.253350292, 108.47652369,
108.605350246, 108.717487114, 109.006689741, 109.09121673, 109.335211685,
110.054657079, 110.188993107, 110.465654479, 110.53071703, 110.561634229,
111.30601108, 111.490988751, 111.85551994, 112.083124523, 112.914061873,
113.05544306, 113.319876123, 113.973685115, 113.990448239, 114.081532341,
114.398558877, 115.77304705, 116.126972557, 116.78344432, 116.843528542,
117.007980322, 117.035696543, 117.291383217, 117.446935054, 118.599175228,
119.09083066, 119.645537452, 119.723223718, 119.773566937, 119.871727805,
119.980721889, 120.463670909, 120.483280294, 120.827560431, 121.38938072,
121.926571322, 121.953680074, 121.956647797, 122.239692568, 122.404241886,
122.534630835, 122.921589103, 123.270801841, 123.93386657, 124.042245016,
124.690620322), "150" = c(1.9193e-05, 1.9534e-05, 1.9671e-05,
2.0084e-05, 2.021e-05, 2.0636e-05, 2.0807e-05, 2.0835e-05, 2.0981e-05,
2.146e-05, 2.1652e-05, 2.195e-05, 2.2011e-05, 2.2235e-05, 2.2336e-05,
2.2844e-05, 2.3186e-05, 2.3862e-05, 2.4049e-05, 2.4171e-05, 2.4402e-05,
2.4434e-05, 2.4645e-05, 2.5134e-05, 2.5538e-05, 2.5564e-05, 2.577e-05,
2.62e-05, 2.7058e-05, 2.7306e-05, 2.772e-05, 2.7849e-05, 2.8072e-05,
2.8329e-05, 2.861e-05, 2.8887e-05, 2.8895e-05, 2.9126e-05, 2.9324e-05,
2.9829e-05, 2.9865e-05, 3.0045e-05, 3.0236e-05, 3.0454e-05, 3.0778e-05,
3.1326e-05, 3.1539e-05, 3.1752e-05, 3.274e-05, 3.2894e-05, 3.3271e-05,
3.3461e-05, 3.3726e-05, 3.3987e-05, 3.472e-05, 3.5436e-05, 3.6077e-05,
3.6334e-05, 3.6574e-05, 3.6889e-05, 3.7535e-05, 3.7732e-05, 3.8173e-05,
3.869e-05, 3.9021e-05, 3.9609e-05, 3.9951e-05, 4.0102e-05, 4.1255e-05,
4.1421e-05, 4.1948e-05, 4.1983e-05, 4.2623e-05, 4.2773e-05, 4.2841e-05,
4.3289e-05, 4.3387e-05, 4.4053e-05, 4.4967e-05, 4.5979e-05, 4.6536e-05,
4.7625e-05, 4.8037e-05, 4.8141e-05, 4.8561e-05, 4.8893e-05, 4.9147e-05,
4.9754e-05, 4.9843e-05, 5.0666e-05, 5.1489e-05, 5.1695e-05, 5.2081e-05,
5.2574e-05, 5.3442e-05, 5.4403e-05, 5.4541e-05, 5.5237e-05, 5.5911e-05,
5.6737e-05, 5.7154e-05, 5.7583e-05, 5.8232e-05, 5.911e-05, 5.9854e-05,
6.0645e-05, 6.1105e-05, 6.2254e-05, 6.2764e-05, 6.3324e-05, 6.5343e-05,
6.6388e-05, 6.7192e-05, 6.7646e-05, 6.8932e-05, 6.9523e-05, 7.0198e-05,
7.0929e-05, 7.1466e-05, 7.213e-05, 7.2662e-05, 7.3679e-05, 7.421e-05,
7.506e-05, 7.5529e-05, 7.7139e-05, 7.8697e-05, 8.0818e-05, 8.2196e-05,
8.3085e-05, 8.4564e-05, 8.6169e-05, 8.751e-05, 8.8953e-05, 9.0361e-05,
9.1374e-05, 9.3146e-05, 9.3997e-05, 9.4567e-05, 9.5194e-05, 9.6302e-05,
9.7876e-05, 9.8466e-05, 9.9914e-05, 0.000100784, 0.000101322,
0.000102788, 0.000104147, 0.000106005, 0.000107282, 0.000108107,
0.000108975, 0.000110487, 0.000111871, 0.000112722, 0.000113746,
0.000114692, 0.00011608, 0.000116734, 0.000117664, 0.000118927,
0.000120358, 0.000121591, 0.000123532, 0.000125331, 0.00012686,
0.000128105, 0.000130109, 0.000132031, 0.00013369, 0.000134459,
0.000136028, 0.000138122, 0.000140526, 0.000142118, 0.000143348,
0.000145554, 0.000146727, 0.000148462, 0.000149742, 0.000150622,
0.00015177, 0.000153408, 0.000155541, 0.000157515, 0.000158895,
0.000161397, 0.000162646, 0.00016443, 0.000167144, 0.000169034,
0.000170997, 0.000172983, 0.000174226, 0.000175853, 0.000177872,
0.000179625, 0.000181604, 0.000184599, 0.000187262, 0.00018852,
0.000190828, 0.000192961, 0.00019542, 0.000196697, 0.000198948,
0.00020098, 0.000203465, 0.000205849, 0.000208268, 0.000210757,
0.000213404, 0.000215794, 0.000217388, 0.00022002, 0.000222179,
0.000225238, 0.000228063, 0.000230462, 0.000232855, 0.000235018,
0.000238782, 0.000240878, 0.000243928, 0.000246828, 0.000249956,
0.000252669, 0.000255538, 0.000258651, 0.000262776, 0.0002652,
0.000269441, 0.000272517, 0.000275902, 0.000278955, 0.000282513,
0.000286983, 0.000289904, 0.00029402, 0.000297814, 0.000300545,
0.000303462, 0.000306574, 0.000309958, 0.000313408, 0.000318715,
0.000323306, 0.00032535, 0.000329429, 0.000333873, 0.00033791,
0.00034114, 0.000345414, 0.000348628, 0.000352788, 0.00035724,
0.000362122, 0.000369308, 0.000373351, 0.000377817, 0.000382768,
0.000387126, 0.000391953, 0.000396849, 0.000401235, 0.000406611,
0.000412291, 0.000416487, 0.000419929, 0.000422842, 0.000428191,
0.000432475, 0.000436144, 0.00044213, 0.000447618, 0.000452392,
0.00045636, 0.000461904, 0.000467889, 0.000472916, 0.000477638,
0.000483113, 0.000489571, 0.000494953, 0.000501393, 0.000508688,
0.0005143, 0.000520005, 0.000525976, 0.000532358, 0.000540965,
0.00054878, 0.000554964, 0.00056093, 0.00056624, 0.000573386,
0.000580353, 0.000587579, 0.000594003, 0.000601605, 0.000610537,
0.000619026, 0.000625511, 0.000630155, 0.000636432, 0.000643337,
0.000648177, 0.000654898, 0.000662225, 0.000669399, 0.000676584,
0.000682966, 0.000690274, 0.000698852, 0.000706268, 0.000714553,
0.000724433, 0.000733288, 0.000741847, 0.000749472, 0.000757811,
0.000768654, 0.00077619, 0.000784174, 0.000797267, 0.000806705,
0.000816472, 0.00082639, 0.000837006, 0.000847608, 0.000858824,
0.000867792, 0.000877246, 0.000888657, 0.00089834, 0.000908524,
0.000919027, 0.000929917, 0.000940886, 0.000952286, 0.000962011,
0.000963912, 0.00097231, 0.00098131, 0.00099253, 0.001006106,
0.001020808, 0.001031191, 0.001043435, 0.001055399, 0.001066656,
0.001078003, 0.001088791, 0.001100852, 0.001116485, 0.00112754,
0.001140746, 0.00115572, 0.001169727, 0.00118503, 0.001198622,
0.001213586, 0.001224986, 0.001238795, 0.001250751, 0.001266422,
0.001280909, 0.001295769, 0.001309211, 0.001325859, 0.001343183,
0.001361506, 0.001375751, 0.001393051, 0.00141036, 0.001424626,
0.001440359, 0.001456627, 0.001476699, 0.001494511, 0.001513041,
0.001529335, 0.001547246, 0.001564226, 0.001588438, 0.001607157,
0.001624635, 0.001641001, 0.001662892, 0.001683322, 0.001702767,
0.001722895, 0.0017433, 0.00176189, 0.001784488, 0.001806122,
0.001827149, 0.001848395, 0.001872289, 0.001895292, 0.001916636,
0.001938777, 0.001959824, 0.001983709, 0.002006206, 0.002031265,
0.002053079, 0.002075256, 0.002096427, 0.002121656, 0.002150753,
0.002172669, 0.002198683, 0.002224995, 0.00225116, 0.002277903,
0.002299134, 0.002327301, 0.002352515, 0.002381546, 0.002409102,
0.002434976, 0.002461098, 0.0024917, 0.002519169, 0.002548491,
0.002574778, 0.002601063, 0.002632185, 0.002661109, 0.002695526,
0.002725712, 0.002759828, 0.002792948, 0.002824539, 0.002855699,
0.002887665, 0.002920797, 0.002953643, 0.002985262, 0.003025827,
0.003059289, 0.003093544, 0.003122492, 0.003157946, 0.003198171,
0.003238202, 0.003279517, 0.003322865, 0.003361008, 0.003398997,
0.003437128, 0.003475622, 0.003522534, 0.003556426, 0.003600314,
0.003642897, 0.003684975, 0.003729244, 0.003773638, 0.003812298,
0.003858262, 0.003905639, 0.003951276, 0.004000282, 0.004043139,
0.004096763, 0.004141174, 0.004192723, 0.00424513, 0.004293365,
0.004342762, 0.004393417, 0.004446093, 0.004493345, 0.004543036,
0.004596672, 0.00465577, 0.004711926, 0.004765578, 0.004820924,
0.004872585, 0.004930272, 0.004991704, 0.005051696, 0.00510741,
0.005167392, 0.00521966, 0.005284319, 0.005345782, 0.005404653,
0.005465697, 0.005528437, 0.005590563, 0.005658574, 0.005729979,
0.005799646, 0.005861727, 0.005929744, 0.006000484, 0.006075155,
0.006143664, 0.006210694, 0.006291906, 0.006361045, 0.006427652,
0.006498796, 0.006567852, 0.006643361, 0.006723564, 0.006800252,
0.006881885, 0.006971335, 0.007055518, 0.007138388, 0.00722363,
0.007308893, 0.007393707, 0.007476401, 0.007560889, 0.00764911,
0.007734609, 0.007832379, 0.007925015, 0.008024576, 0.008116677,
0.008215776, 0.008311802, 0.008410011, 0.008505267, 0.00860959,
0.00872122, 0.008826544, 0.008934385, 0.009038548, 0.0091452,
0.009262766, 0.009358389, 0.009464333, 0.009571874, 0.00968462,
0.009781088, 0.009800264, 0.009913899, 0.010030528, 0.010152605,
0.010268983, 0.010379763, 0.010501031, 0.010620926, 0.010741641,
0.010863561, 0.010991892, 0.011124946, 0.011261152, 0.011388096,
0.011524197, 0.011653015, 0.011793514, 0.011924286, 0.012067768,
0.012209084, 0.012349825, 0.012487791, 0.012632293, 0.012775675,
0.012933315, 0.013081562, 0.013230175, 0.013377804, 0.013529135,
0.013682885, 0.013847089, 0.014009601, 0.014170134, 0.014326938,
0.014495826, 0.014662663, 0.014838345, 0.015012568, 0.015180777,
0.015370509, 0.015550354, 0.015723164, 0.015905868, 0.016101772,
0.016296343, 0.016481016, 0.016675623, 0.01686963, 0.017066386,
0.017269545, 0.017466252, 0.017662553, 0.017852518, 0.018071317,
0.018278422, 0.018493621, 0.018709198, 0.018942724, 0.019154161,
0.019386807, 0.019606501, 0.019827563, 0.020062558, 0.020297526,
0.020530571, 0.020785934, 0.02102465, 0.021277045, 0.021517801,
0.021765457, 0.022036606, 0.022297734, 0.022554872, 0.022821922,
0.023093035, 0.023361845, 0.02362273, 0.023897726, 0.024167289,
0.024451412, 0.02473992, 0.02502249, 0.025319004, 0.025613555,
0.025912291, 0.026206372, 0.026507749, 0.026823907, 0.027144512,
0.027457266, 0.027766267, 0.028084779, 0.02841267, 0.028740019,
0.029069963, 0.029415573, 0.029740616, 0.030081189, 0.030428629,
0.030771617, 0.031122218, 0.031488884, 0.031855667, 0.03223248,
0.032612153, 0.03299113, 0.033365557, 0.033748549, 0.034137852,
0.0345204, 0.034912558, 0.035325096, 0.035740974, 0.036154227,
0.036568503, 0.036980524, 0.037395275, 0.037823129, 0.038273427,
0.038734396, 0.039186039, 0.039647387, 0.040115879, 0.040584245,
0.041060628, 0.04153671, 0.042026336, 0.042512805, 0.043016297,
0.043513623, 0.044020073, 0.044534623, 0.045044403, 0.04557793,
0.04611233, 0.046652778, 0.047213951, 0.047771452, 0.04833759,
0.04889839, 0.049475416, 0.050054063, 0.050650879, 0.051239675,
0.051824633, 0.052434897, 0.053039201, 0.053660115, 0.054270739,
0.054899867, 0.055529376, 0.056179656, 0.056847262, 0.057506689,
0.058178509, 0.058862007, 0.059545235, 0.060256048, 0.060952426,
0.061659312, 0.062373148, 0.06310494, 0.063832408, 0.064593336,
0.065363, 0.066136469, 0.066903439, 0.067691548, 0.068460885,
0.069272333, 0.070070673, 0.070900305, 0.071735613, 0.072558102,
0.073400094, 0.074258292, 0.075118721, 0.075996066, 0.076872782,
0.077773792, 0.078689172, 0.079608134, 0.080565914, 0.081515848,
0.082477758, 0.083445334, 0.084428006, 0.085440693, 0.08641772,
0.08742783, 0.088457596, 0.089527989, 0.09060016, 0.091642089,
0.092713407, 0.093816686, 0.094915618, 0.096024909, 0.097150816,
0.098278833, 0.099187494, 0.099420311, 0.100582466, 0.101791508,
0.103004986, 0.104238096, 0.105458942, 0.10671353, 0.107990265,
0.109281144, 0.110577412, 0.111919051, 0.113229859, 0.114539162,
0.115864351, 0.117242568, 0.118649068, 0.120039215, 0.121469304,
0.122903444, 0.124340647, 0.125800706, 0.127271154, 0.128754539,
0.130248399, 0.131788609, 0.133342899, 0.134912288, 0.136511893,
0.138138454, 0.139767533, 0.141442106, 0.143129121, 0.144835517,
0.146541167, 0.148248259, 0.149993595, 0.151753862, 0.153527447,
0.155378543, 0.157203852, 0.15907916, 0.1609826, 0.162879598,
0.164837943, 0.166800259, 0.168765308, 0.170794735, 0.172827461,
0.174875354, 0.17695226, 0.179081717, 0.181183364, 0.183334191,
0.185522116, 0.187744112, 0.189985164, 0.192214979, 0.194537345,
0.196840449, 0.199165841, 0.20156766, 0.20394501, 0.206359712,
0.208804865, 0.211316948, 0.213848944, 0.216424469, 0.219003041,
0.22163557, 0.22428433, 0.226976241, 0.229722482, 0.232512107,
0.235349671, 0.238158856, 0.241016354, 0.243914491, 0.246839054,
0.249769372, 0.252757062, 0.255770887, 0.258868539, 0.262012054,
0.265175421, 0.268380125, 0.271561593, 0.274840476, 0.278136381,
0.281479014, 0.284886925, 0.288363812, 0.291865211, 0.295435179,
0.299040808, 0.30261804, 0.306282524, 0.307106446, 0.310035419,
0.31375252, 0.317579034, 0.321415111, 0.325313513, 0.329335962,
0.333376713, 0.337431078, 0.341590916, 0.345788637, 0.350066005,
0.354400109, 0.358760477, 0.3631647, 0.367616163, 0.372151259,
0.37678165, 0.381416717, 0.386112466, 0.390876073, 0.39573402,
0.400637315, 0.405558728, 0.410558776, 0.415716074, 0.41675326,
0.420926539, 0.426192632, 0.431489381, 0.436835342, 0.442309107,
0.447786878, 0.453408541, 0.459151699, 0.464972806, 0.470776258,
0.476688303, 0.482683549, 0.48876739, 0.494923197, 0.501183456,
0.507603271, 0.514089741, 0.52063015, 0.527284328, 0.533965017,
0.540747284, 0.547654126, 0.554659915, 0.561747843, 0.568962194,
0.576231779, 0.583695213, 0.591271605, 0.598856913, 0.606550178,
0.614493165, 0.622544482, 0.630603408, 0.638847842, 0.647153934,
0.650758903, 0.655644563, 0.664252633, 0.673113214, 0.682002676,
0.690925119, 0.700102624, 0.709404192, 0.718903971, 0.728493696,
0.738213392, 0.748166759, 0.75815161, 0.768379541, 0.778804325,
0.789329795, 0.800090646, 0.81101207, 0.822091389, 0.833410635,
0.844869067, 0.856544944, 0.868448969, 0.880484751, 0.892731139,
0.905266124, 0.910512581, 0.917994341, 0.930900377, 0.944004477,
0.95725869, 0.971094652, 0.985034839, 0.999274465, 1.013748179,
1.028496444, 1.043504595, 1.058766383, 1.074362437, 1.090259424,
1.106479318, 1.123159991, 1.140002372, 1.157260429, 1.174864227,
1.192910943, 1.207578175, 1.240841278, 1.259211974, 1.277645082,
1.296094854, 1.314593943, 1.332868827, 1.351269936, 1.369512745,
1.387736524, 1.405960949, 1.424102732, 1.442431967, 1.46054306,
1.478661172, 1.496975553, 1.515113864, 1.533282444, 1.55124911,
1.561859847, 1.569334058, 1.587493623, 1.605454128, 1.623517221,
1.641622116, 1.659689521, 1.677621102, 1.695592187, 1.713396299,
1.73131183, 1.749206058, 1.767204558, 1.785148605, 1.80308047,
1.821244439, 1.839091394, 1.857137512, 1.875047997, 1.893062659,
1.910909754, 1.928944302, 1.946796762, 1.964785984, 1.982866742,
2.000825882, 2.011237075, 2.018865858, 2.03696872, 2.054973477,
2.073051003, 2.090998958, 2.108964087, 2.127146655, 2.145056429,
2.163159385, 2.181209698, 2.199508475, 2.217511699, 2.235612899,
2.253841889, 2.272138178, 2.290177541, 2.308377614, 2.326563586,
2.345132829, 2.363106448, 2.381290948, 2.399713124, 2.418071714,
2.436577696, 2.454945637, 2.473480023, 2.491883653, 2.510505791,
2.52901689, 2.547629814, 2.566081841, 2.584828533, 2.603365165,
2.622088871, 2.640703422, 2.655831769, 2.65981593, 2.678520915,
2.697518612, 2.716443611, 2.735496865, 2.754403211, 2.773623831,
2.792480822, 2.811560258, 2.830807396, 2.850045562, 2.869357298,
2.888643831, 2.908001201, 2.927437339, 2.946849542, 2.966481853,
2.986332364, 3.006053046, 3.025930198, 3.045817694, 3.06561017,
3.085613199, 3.105773008, 3.126007607, 3.14158016, 3.145934267,
3.166043322, 3.186483507, 3.207028467, 3.22760423, 3.247933277,
3.268428601, 3.288863601, 3.309617496, 3.330269307, 3.351458233,
3.372486851, 3.393619671, 3.414592513, 3.435666829, 3.457183712,
3.478217298, 3.499715354, 3.521251843, 3.542747873, 3.564545085,
3.586393267, 3.608061361, 3.630009467, 3.652116667, 3.67448017,
3.697060042, 3.719304345, 3.741944503, 3.764493639, 3.787439119,
3.810593076, 3.833436763, 3.857045589, 3.88085924, 3.904050632,
3.927678843, 3.951541294, 3.975164197, 3.998545803, 4.022410205,
4.046608818, 4.070666747, 4.094966776, 4.119364576, 4.144408409,
4.169368307, 4.194275591, 4.219314565, 4.244418621, 4.26982254,
4.295831205, 4.321497494, 4.347507952, 4.373192091, 4.399472713,
4.426167696, 4.452547566, 4.479274374, 4.505847816, 4.532807886,
4.559787862, 4.586744347, 4.614203063, 4.641851944, 4.669654904,
4.697860379, 4.726506128, 4.754877485, 4.783591543, 4.812553929,
4.841015454, 4.870784077, 4.899588477, 4.929850032, 4.958820097,
4.988868012, 5.019171499, 5.049264536, 5.079991369, 5.110468345,
5.141342273, 5.171769818, 5.202206148, 5.233531981, 5.265618989,
5.297424239, 5.329230952, 5.360388887, 5.392228974, 5.424302158,
5.457522997, 5.491318085, 5.524333854, 5.557793022, 5.591523044,
5.598425631, 5.625734834, 5.658909726, 5.692757032, 5.727708263,
5.76166739, 5.796059579, 5.830977458, 5.865891724, 5.901308015,
5.937181847, 5.973204332, 6.010197381, 6.046937261, 6.083377107,
6.120733855, 6.157543942, 6.195180617, 6.2319279, 6.268986265,
6.306535109, 6.344015798, 6.382819265, 6.421005234, 6.458883677,
6.496851871, 6.535821227, 6.574697541, 6.614502046, 6.653713176,
6.692404876, 6.73201565, 6.771502178, 6.811218201, 6.851336596,
6.892001377, 6.932095888, 6.97238859, 7.012197977, 7.053538929,
7.094922724, 7.135954971, 7.176082519, 7.21771982, 7.260138193,
7.302339997, 7.344436251, 7.387617176, 7.43073194, 7.472824681,
7.515289797, 7.558262196, 7.601229875, 7.644646291, 7.687549549,
7.731370237, 7.774633592, 7.819111591, 7.865362892, 7.908788589,
7.954824546, 8.00066184, 8.044969175, 8.091133069, 8.136581712,
8.181536939, 8.227849567, 8.273726489, 8.319949825, 8.367493151,
8.4147022, 8.462173776, 8.508938083, 8.55596717, 8.603684733,
8.65010308, 8.698290082, 8.745906215, 8.794144664, 8.842579227,
8.889973626, 8.937911795, 8.985965651, 9.034541751, 9.083400256,
9.132448391, 9.183641306, 9.233959793, 9.283475445, 9.332690501,
9.383477436, 9.434459243, 9.485131109, 9.53481109, 9.587020669,
9.636661988, 9.688625334, 9.74266181, 9.795485958, 9.847715312,
9.90100803, 9.954426896, 10.007044448, 10.060935169, 10.112845372,
10.165694834, 10.221635042, 10.274407151, 10.330138998, 10.384968517,
10.439436598, 10.495150245, 10.547223369, 10.601883674, 10.657423564,
10.714481752, 10.771604933, 10.827777792, 10.885914812, 10.941468882,
10.99628954, 11.052805569, 11.109586424, 11.169039554, 11.22540992,
11.283923167, 11.341493936, 11.398869817, 11.458448471, 11.517512311,
11.578094655, 11.637737748, 11.701115376, 11.760177398, 11.817682037,
11.876779282, 11.940457563, 11.998711092, 12.061251096, 12.123778137,
12.184532726, 12.243814892, 12.302404319, 12.366713319, 12.428273553,
12.489180737, 12.552285205, 12.614765471, 12.677802435, 12.741160533,
12.809990855, 12.876094097, 12.938700239, 13.005779667, 13.066810907,
13.132736534, 13.199271635, 13.263744061, 13.32547144, 13.38391306,
13.450964184, 13.515363931, 13.581827668, 13.647839277, 13.712560158,
13.776889488, 13.840654788, 13.904795712, 13.971390706, 14.037754614,
14.107264275, 14.177320382, 14.244277134, 14.315640702, 14.384747115,
14.451313616, 14.5207973, 14.592605489, 14.664770464, 14.735624382,
14.805895681, 14.880254811, 14.945643777, 15.015262426, 15.085064045,
15.156959713, 15.227818712, 15.298104463, 15.369893791, 15.436868805,
15.507064262, 15.583067445, 15.654311193, 15.728950203, 15.80077509,
15.874988042, 15.948747045, 16.0235222, 16.095675681, 16.169648772,
16.243664433, 16.257339341, 16.317876403, 16.393160185, 16.468323053,
16.549778793, 16.626672539, 16.704644232, 16.776084483, 16.850112888,
16.930944935, 17.006474709, 17.081800912, 17.157966859, 17.229967853,
17.311836774, 17.389505987, 17.464962448, 17.546944343, 17.619973874,
17.700473539, 17.781341025, 17.858531688, 17.939067883, 18.024300308,
18.108656892, 18.191702682, 18.275086484, 18.352289545, 18.435181201,
18.524213051, 18.606939533, 18.685845248, 18.779862628, 18.863677213,
18.942022897, 19.025344441, 19.113593468, 19.197976555, 19.282599711,
19.374818987, 19.455568787, 19.533865758, 19.625977466, 19.720134555,
19.80574557, 19.893856934, 19.980698312, 20.067676503, 20.157429462,
20.244480368, 20.329013717, 20.423460187, 20.515533314, 20.600505761,
20.677854194, 20.768568819, 20.862126693, 20.958305171, 21.03848757,
21.132206496, 21.218750879, 21.311443904, 21.400140478, 21.491405387,
21.584982012, 21.674882226, 21.772470134, 21.863813054, 21.960204743,
22.051875636, 22.144432777, 22.23665683, 22.334864406, 22.426668267,
22.521643962, 22.621333141, 22.719555619, 22.814302449, 22.904381999,
23.001590312, 23.09847899, 23.201154631, 23.292505139, 23.388805432,
23.480959657, 23.576356811, 23.676602655, 23.783607639, 23.874069358,
23.967784844, 24.065441693, 24.158248717, 24.271860009, 24.370945243,
24.464432081, 24.566975008, 24.680454748, 24.785918963, 24.886283143,
24.98765474, 25.081872053, 25.182738473, 25.29146074, 25.397597801,
25.510729163, 25.608929437, 25.714580843, 25.814945869, 25.923586724,
26.02385449, 26.128717324, 26.233394097, 26.347242555, 26.467502606,
26.58260024, 26.687726885, 26.791725101, 26.904794744, 27.005105467,
27.113556901, 27.230678889, 27.349422187, 27.472205265, 27.561775946,
27.675891458, 27.785407669, 27.905471748, 28.010743662, 28.119024351,
28.255959681, 28.366090776, 28.482690233, 28.593011554, 28.716137483,
28.848379789, 28.975609339, 29.09385115, 29.211511458, 29.322036341,
29.415416667, 29.546466952, 29.65675494, 29.774657616, 29.886013247,
30.013423337, 30.14877344, 30.262237238, 30.378875971, 30.502977191,
30.628946141, 30.750722187, 30.886540971, 30.996387278, 31.120769126,
31.231314254, 31.339916013, 31.474591113, 31.590275955, 31.713695285,
31.840412468, 31.958906487, 32.102335117, 32.220223108, 32.343764908,
32.494308237, 32.609088521, 32.750298078, 32.876956888, 33.004183315,
33.131028725, 33.265847583, 33.396736384, 33.532507857, 33.655499601,
33.792053151, 33.915559787, 34.053293363, 34.177517493, 34.296060601,
34.409175231, 34.525612732, 34.651971598, 34.764349063, 34.906524935,
35.054959728, 35.168178884, 35.300789348, 35.445425755, 35.585649861,
35.736789529, 35.852639323, 35.98090706, 36.121875635, 36.263500075,
36.400733612, 36.531761255, 36.639308143, 36.777941495, 36.90182676,
37.037841575, 37.198271369, 37.22901798, 37.335559798, 37.468358337,
37.610661041, 37.769280933, 37.950721892, 38.092929641, 38.234208768,
38.383413743, 38.548374369, 38.737128435, 38.884400974, 39.007842805,
39.117816833, 39.275036867, 39.436016798, 39.569855577, 39.753562278,
39.894173615, 40.052141688, 40.197149277, 40.388271955, 40.559939015,
40.669375515, 40.797054919, 40.942912505, 41.081039362, 41.235980577,
41.440599026, 41.570372808, 41.699385289, 41.832218177, 42.036006436,
42.179679553, 42.312449292, 42.446518419, 42.554894099, 42.695581178,
42.839869293, 42.992265095, 43.184195284, 43.339267045, 43.534362779,
43.639664105, 43.847321239, 43.983286668, 44.164412317, 44.354121117,
44.486164651, 44.667488984, 44.850350056, 45.010677476, 45.189950518,
45.340804719, 45.48955214, 45.669503811, 45.804299097, 45.970491386,
46.135609929, 46.260547811, 46.413035595, 46.61358044, 46.769199621,
46.95246609, 47.131562272, 47.280695703, 47.497928232, 47.671171505,
47.889778575, 48.057502992, 48.234766832, 48.379218539, 48.565833182,
48.70567038, 48.870711034, 49.034019268, 49.177575848, 49.446025513,
49.612925262, 49.805942923, 49.982022903, 50.230642697, 50.340802047,
50.527499932, 50.771769796, 50.912891957, 51.038929197, 51.184278055,
51.401579737, 51.580680264, 51.784200598, 51.930963587, 52.136288202,
52.340244245, 52.602457265, 52.826510571, 52.986701042, 53.166320573,
53.383225657, 53.513199748, 53.649661551, 53.844702233, 53.96133613,
54.101749437, 54.293453338, 54.498870653, 54.665547263, 54.791093403,
54.981203998, 55.174070122, 55.362409273, 55.576022497, 55.753879821,
55.868892737, 56.091871014, 56.242341216, 56.407749711, 56.52812885,
56.752497145, 56.959774428, 57.162513689, 57.316468389, 57.583588663,
57.751220218, 57.915950567, 58.111399044, 58.28025624, 58.3902017,
58.497702912, 58.715699062, 58.857102961, 59.057395559, 59.192871098,
59.365282328, 59.562288977, 59.709311559, 59.842923359, 60.060075784,
60.272861935, 60.432897458, 60.631825351, 60.780589145, 60.992930951,
61.30412323, 61.450269426, 61.589072139, 61.793443891, 61.993289019,
62.253216764, 62.38098701, 62.558077175, 62.858267142, 63.074169051,
63.249609061, 63.480668105, 63.763053696, 63.915814957, 64.133541183,
64.278247957, 64.421123154, 64.649828542, 64.694016207, 64.988496983,
65.127175486, 65.377450944, 65.642996175, 65.946463587, 66.189972953,
66.520650604, 66.622843776, 66.888118239, 67.117986059, 67.309833198,
67.623881065, 67.78697905, 68.063651458, 68.228142629, 68.506904902,
68.881425478, 69.153881373, 69.403622289, 69.550640287, 69.771972477,
69.966364723, 70.13595541, 70.372163008, 70.544781289, 70.911449354,
71.200924947, 71.43720875, 71.709039094, 72.072473156, 72.323387824,
72.570032622, 72.647190917, 72.92881354, 73.066667909, 73.208541723,
73.333393021, 73.629027484, 73.731324616, 73.946356107, 74.101281763,
74.462514843, 74.787408219, 75.171897598, 75.315913989, 75.671437978,
75.950215589, 76.284551853, 76.471897841, 76.606903611, 76.831762389,
77.592199984, 78.000089549, 78.348821347, 78.656252111, 78.864176574,
79.019745478, 79.169669205, 79.281759119, 79.383652176, 79.494247908,
79.625321656, 79.898881087, 80.098771763, 80.359679107, 80.534539039,
80.682389987, 80.827975463, 81.004162573, 81.299820867, 81.618379727,
81.838967704, 82.121672746, 82.295120837, 82.490611998, 82.619174098,
82.839030097, 82.932714321, 83.265248031, 83.501864961, 84.210345804,
84.791009168, 84.975345793, 85.192444544, 85.441976295, 85.65366605,
85.930748497, 86.285808882, 86.430963019, 86.691728745, 87.184743401,
87.325629811, 87.513258378, 87.872717068, 87.935569489, 88.361792203,
88.673071467, 88.832065294, 89.104969603, 89.37763522, 89.821518529,
90.133673838, 90.580064798, 90.792604009, 91.580021887, 91.882329816,
91.930480355, 92.104084999, 92.381628141, 92.682808946, 92.906264924,
93.163011101, 93.494509284, 93.605157884, 93.980370315, 94.060966077,
94.418170663, 94.63056749, 94.685817586, 94.889794037, 95.404784032,
95.455226955, 95.586432117, 96.121613836, 96.244536792, 96.528305296,
96.894083071, 97.695922436, 97.955149386, 98.135610033, 98.277113255,
98.463041877, 98.652794592, 98.867324954, 98.976560255, 99.025320562,
99.158531431, 100.000729557, 100.117119268, 100.274169133, 100.662016725,
101.222083557, 102.397954561, 102.586469906, 102.969614352, 103.621755538,
103.683474545, 104.673544848, 105.082933377, 105.293097192, 105.495550563,
105.698727572, 106.160343002, 106.367434622, 106.435859641, 106.464424914,
106.706338317, 107.193625078, 107.414232476, 107.749421956, 108.045236582,
108.263284444, 108.756273987, 108.888150044, 110.013150046, 110.017954219,
110.307032863, 110.63880078, 110.782784916, 111.247545613, 111.5687145,
112.171325905, 112.448993891, 112.86663226, 112.929461789, 113.056439816,
113.828463166, 113.941292245, 114.067054307), "200" = c(2.0152e-05,
2.0195e-05, 2.0251e-05, 2.0756e-05, 2.0857e-05, 2.1003e-05, 2.1187e-05,
2.1222e-05, 2.1714e-05, 2.1749e-05, 2.2095e-05, 2.2305e-05, 2.2353e-05,
2.2707e-05, 2.2821e-05, 2.2887e-05, 2.291e-05, 2.3076e-05, 2.3123e-05,
2.3436e-05, 2.3631e-05, 2.3769e-05, 2.4396e-05, 2.4418e-05, 2.5444e-05,
2.5452e-05, 2.5905e-05, 2.5974e-05, 2.6347e-05, 2.6631e-05, 2.6738e-05,
2.7928e-05, 2.8417e-05, 2.8832e-05, 2.8992e-05, 3.0162e-05, 3.0194e-05,
3.1025e-05, 3.1383e-05, 3.1498e-05, 3.1703e-05, 3.2056e-05, 3.2171e-05,
3.2476e-05, 3.3081e-05, 3.3197e-05, 3.3838e-05, 3.4075e-05, 3.4515e-05,
3.4645e-05, 3.4709e-05, 3.5476e-05, 3.5522e-05, 3.5584e-05, 3.5671e-05,
3.5803e-05, 3.6725e-05, 3.6797e-05, 3.7055e-05, 3.7405e-05, 3.7929e-05,
3.794e-05, 3.8233e-05, 3.8553e-05, 3.8635e-05, 3.8886e-05, 3.9507e-05,
3.9651e-05, 4.0113e-05, 4.0593e-05, 4.0958e-05, 4.1001e-05, 4.1534e-05,
4.1861e-05, 4.2042e-05, 4.2283e-05, 4.2586e-05, 4.3188e-05, 4.3748e-05,
4.4248e-05, 4.5175e-05, 4.581e-05, 4.6778e-05, 4.7686e-05, 4.772e-05,
4.8436e-05, 4.8717e-05, 4.9119e-05, 4.9784e-05, 5.0739e-05, 5.1294e-05,
5.1575e-05, 5.2508e-05, 5.3476e-05, 5.3689e-05, 5.4245e-05, 5.5088e-05,
5.5585e-05, 5.6159e-05, 5.7476e-05, 5.7674e-05, 5.8351e-05, 5.9147e-05,
5.9311e-05, 5.9985e-05, 6.0872e-05, 6.135e-05, 6.2221e-05, 6.3786e-05,
6.437e-05, 6.4944e-05, 6.5663e-05, 6.5955e-05, 6.6551e-05, 6.8447e-05,
6.9544e-05, 7.0231e-05, 7.1388e-05, 7.2164e-05, 7.2474e-05, 7.3752e-05,
7.4192e-05, 7.4847e-05, 7.5761e-05, 7.7063e-05, 7.8179e-05, 7.9315e-05,
8.0522e-05, 8.109e-05, 8.1863e-05, 8.228e-05, 8.3117e-05, 8.3437e-05,
8.4592e-05, 8.5764e-05, 8.6621e-05, 8.7724e-05, 8.9739e-05, 9.1109e-05,
9.225e-05, 9.3883e-05, 9.5049e-05, 9.6181e-05, 9.6917e-05, 9.838e-05,
0.000100026, 0.000100993, 0.000102875, 0.000103696, 0.000105951,
0.000107044, 0.000108297, 0.000108863, 0.000110459, 0.000111379,
0.000112922, 0.000114969, 0.000116721, 0.000118643, 0.000119905,
0.000121217, 0.000122549, 0.000123914, 0.000125307, 0.000128249,
0.000129723, 0.000131493, 0.000133209, 0.000135161, 0.000137516,
0.000139302, 0.000141555, 0.000142552, 0.000144167, 0.000145892,
0.000146761, 0.000147878, 0.000149105, 0.000151329, 0.000153886,
0.000156019, 0.000157999, 0.000159349, 0.000161668, 0.000163866,
0.000166097, 0.000167718, 0.000169714, 0.000171189, 0.000173273,
0.000175445, 0.000177666, 0.00018112, 0.000183929, 0.00018615,
0.000187754, 0.000189786, 0.000192555, 0.000195453, 0.000197684,
0.000199305, 0.00020309, 0.000204732, 0.000206531, 0.000208689,
0.000210837, 0.000212767, 0.000214059, 0.000216179, 0.000219672,
0.00022283, 0.000224891, 0.000226773, 0.00022874, 0.000231305,
0.000233352, 0.000235167, 0.000238213, 0.000240639, 0.000243626,
0.000244562, 0.000246585, 0.000248729, 0.000251414, 0.000253845,
0.000257622, 0.000260112, 0.000263559, 0.000266037, 0.000268317,
0.000271823, 0.000275657, 0.000278296, 0.000281895, 0.000284013,
0.000287202, 0.000289398, 0.00029144, 0.000294874, 0.000298893,
0.000302307, 0.000305537, 0.00030931, 0.000314239, 0.000319442,
0.000323679, 0.000328593, 0.000332341, 0.000338242, 0.000343011,
0.000347015, 0.000349618, 0.000352798, 0.000356134, 0.000361119,
0.000363746, 0.000367757, 0.000374762, 0.000378783, 0.000382872,
0.000389818, 0.000394078, 0.000398286, 0.000404227, 0.000409009,
0.0004137, 0.00041777, 0.000421069, 0.000425639, 0.000432055,
0.000436464, 0.000440767, 0.000445639, 0.000449705, 0.000454075,
0.000458389, 0.000463803, 0.00047042, 0.00047582, 0.000480776,
0.000487308, 0.000493526, 0.00050066, 0.000507037, 0.00051349,
0.000520061, 0.000525868, 0.000531695, 0.000536472, 0.000541609,
0.000546112, 0.00055178, 0.000557575, 0.000565099, 0.000570097,
0.000577047, 0.000584273, 0.000590978, 0.000599832, 0.000605756,
0.000610608, 0.000616041, 0.000622285, 0.000631054, 0.0006409,
0.000648108, 0.000653885, 0.000658938, 0.000666838, 0.000674352,
0.000682052, 0.000689258, 0.000695814, 0.000704416, 0.000712785,
0.000721927, 0.000729674, 0.000738607, 0.000748466, 0.000756999,
0.000766047, 0.000775984, 0.000785425, 0.000793882, 0.00080476,
0.000814114, 0.000822803, 0.000832547, 0.000843823, 0.00085559,
0.000865363, 0.000874788, 0.000885862, 0.000893898, 0.000903594,
0.000915996, 0.0009268, 0.000938447, 0.000947114, 0.00095839,
0.000968026, 0.000970225, 0.000982565, 0.000996548, 0.001010905,
0.001023333, 0.001035679, 0.001045822, 0.001054864, 0.001067806,
0.001079842, 0.001092359, 0.001105161, 0.00111777, 0.001134195,
0.001149398, 0.001166764, 0.001183124, 0.001196782, 0.001210757,
0.0012249, 0.001240435, 0.001256217, 0.001270154, 0.001284027,
0.0012977, 0.001313485, 0.001326426, 0.001342038, 0.001358277,
0.001375004, 0.001390287, 0.001404767, 0.001421084, 0.00143983,
0.001457104, 0.001475, 0.001490372, 0.001506386, 0.001522417,
0.00153792, 0.001556429, 0.001575399, 0.001593678, 0.001610227,
0.001628641, 0.00164678, 0.001668168, 0.001686167, 0.001703027,
0.001722586, 0.001741219, 0.001759175, 0.001776948, 0.001796058,
0.001815469, 0.001833886, 0.001852294, 0.001874285, 0.001894259,
0.001916703, 0.001941734, 0.001963493, 0.001984874, 0.002006677,
0.002033581, 0.002060522, 0.002088586, 0.002113066, 0.002135653,
0.002161631, 0.002188093, 0.002214044, 0.002238638, 0.002261883,
0.002293254, 0.002320055, 0.00234269, 0.002374344, 0.002400033,
0.002424315, 0.002453836, 0.002481938, 0.002510894, 0.002540843,
0.002571919, 0.002603082, 0.002631487, 0.002659711, 0.002685945,
0.002717969, 0.002745848, 0.002775532, 0.002809023, 0.002840281,
0.002873711, 0.002906551, 0.002941505, 0.002975328, 0.003008434,
0.003039883, 0.003070062, 0.003110114, 0.003147079, 0.003184444,
0.003220356, 0.003256776, 0.003295111, 0.003334392, 0.003367012,
0.00340775, 0.003451579, 0.003495862, 0.003531699, 0.003576338,
0.003619288, 0.003659601, 0.003698404, 0.0037389, 0.00377896,
0.003822234, 0.003871374, 0.003920403, 0.003966554, 0.004014205,
0.004058663, 0.004111438, 0.004162729, 0.004214637, 0.00427027,
0.004315017, 0.00435924, 0.004414803, 0.004464875, 0.004514677,
0.004564412, 0.00461548, 0.004670035, 0.004725049, 0.00478123,
0.00483727, 0.004892926, 0.004953834, 0.005009176, 0.005071247,
0.005124712, 0.005181856, 0.005242449, 0.005303965, 0.005367324,
0.005432855, 0.005496367, 0.005554884, 0.005626369, 0.005693257,
0.00575649, 0.00582174, 0.005893646, 0.005967856, 0.006036626,
0.00610568, 0.006181656, 0.006252979, 0.006320349, 0.00639452,
0.006468834, 0.006544049, 0.006619348, 0.006689091, 0.006770738,
0.006848493, 0.00693052, 0.00701242, 0.007100973, 0.007181018,
0.007261652, 0.007343513, 0.007434226, 0.007526616, 0.007616837,
0.007704705, 0.007792129, 0.007881717, 0.007976107, 0.008070397,
0.008167588, 0.008258611, 0.008364826, 0.008460304, 0.008557559,
0.008657552, 0.008756405, 0.008861054, 0.008955078, 0.009054476,
0.009159509, 0.009272135, 0.009386257, 0.00949445, 0.009610222,
0.009723458, 0.009807336, 0.009828022, 0.009937435, 0.010048381,
0.010166545, 0.010284558, 0.010401951, 0.010522375, 0.010642478,
0.010770985, 0.010891722, 0.011024196, 0.011147975, 0.011271685,
0.011400184, 0.011540153, 0.011673022, 0.01180631, 0.01193759,
0.012076398, 0.012219746, 0.012358641, 0.012496837, 0.012635585,
0.012790576, 0.012944521, 0.013084365, 0.013231485, 0.013376689,
0.013535632, 0.013690058, 0.013858493, 0.014009623, 0.014180047,
0.01433604, 0.01451211, 0.014685894, 0.014861898, 0.015035389,
0.0152129, 0.015391306, 0.01557073, 0.015753688, 0.015941245,
0.01612115, 0.016301051, 0.016490883, 0.016688306, 0.016884193,
0.017083506, 0.017274045, 0.017478591, 0.017690242, 0.017903908,
0.018111249, 0.018315432, 0.018524671, 0.018735287, 0.018954398,
0.019174714, 0.019397448, 0.019622147, 0.019862206, 0.020094141,
0.020325743, 0.020560344, 0.020803871, 0.021045475, 0.021304979,
0.021546942, 0.021808375, 0.022061835, 0.022308614, 0.022566747,
0.022825051, 0.023086706, 0.023355036, 0.023629717, 0.023901999,
0.024188581, 0.024462932, 0.024758525, 0.025040061, 0.025321219,
0.025616552, 0.025917624, 0.026233, 0.026526237, 0.026825614,
0.02712941, 0.027434686, 0.027757417, 0.028094173, 0.028425882,
0.028774075, 0.029128629, 0.029466391, 0.029804553, 0.030157938,
0.030510576, 0.030860871, 0.031227502, 0.031571762, 0.031940437,
0.032297757, 0.032677666, 0.033072488, 0.033455835, 0.033862178,
0.034241949, 0.034646075, 0.03504759, 0.03546324, 0.035877117,
0.036282523, 0.036710618, 0.037125415, 0.037546099, 0.037985822,
0.038436932, 0.038878656, 0.039320225, 0.039775991, 0.040234974,
0.040709713, 0.04118423, 0.041651861, 0.042134437, 0.042604893,
0.043110687, 0.043603822, 0.044115631, 0.044618251, 0.045139229,
0.045669184, 0.046185303, 0.046730033, 0.04728471, 0.047853463,
0.048393509, 0.048962693, 0.049529556, 0.05010064, 0.050684503,
0.051273555, 0.051869322, 0.052477744, 0.053064155, 0.053672988,
0.054317905, 0.054944593, 0.055600013, 0.056261605, 0.056907922,
0.057551213, 0.058242772, 0.058930676, 0.059627147, 0.060314288,
0.061028704, 0.06173077, 0.062458685, 0.063177926, 0.063922052,
0.064685776, 0.065450967, 0.066235286, 0.066998341, 0.067805351,
0.068614452, 0.069431235, 0.070241207, 0.071077527, 0.071920963,
0.072781587, 0.07363388, 0.074481867, 0.075348388, 0.07624503,
0.077111516, 0.078008936, 0.078912674, 0.079837399, 0.080774757,
0.08172481, 0.082679387, 0.083637884, 0.084618326, 0.085642411,
0.086653247, 0.087672529, 0.088736265, 0.089779693, 0.090844442,
0.091919044, 0.093008688, 0.094088156, 0.095211137, 0.096318466,
0.09745383, 0.098583549, 0.099504889, 0.099735237, 0.100898675,
0.102093323, 0.103317441, 0.104529016, 0.10576916, 0.107015788,
0.108287366, 0.109582562, 0.110887949, 0.11222076, 0.113523007,
0.114867457, 0.116221325, 0.117569605, 0.118931088, 0.12031573,
0.121716779, 0.123145194, 0.124599046, 0.126039324, 0.127527683,
0.129028204, 0.130567628, 0.132117967, 0.13371356, 0.135288529,
0.136905282, 0.138538488, 0.140178528, 0.141818574, 0.143488778,
0.145193657, 0.146919497, 0.14869637, 0.150448537, 0.152257986,
0.154086556, 0.155922644, 0.157751753, 0.159641815, 0.161557997,
0.163471018, 0.165437024, 0.167393577, 0.169353639, 0.171379134,
0.173442822, 0.175518327, 0.177622315, 0.179736078, 0.181879472,
0.184044142, 0.186238731, 0.188452924, 0.190694091, 0.192970338,
0.19530606, 0.197609652, 0.199999924, 0.202437455, 0.204845307,
0.207337467, 0.209826835, 0.212347265, 0.21493979, 0.217514711,
0.220174049, 0.222852309, 0.22553909, 0.228287915, 0.230991406,
0.233760491, 0.236583887, 0.239388158, 0.242306825, 0.245224322,
0.248232965, 0.251229088, 0.254275628, 0.257358953, 0.260476046,
0.263674125, 0.266889655, 0.270138217, 0.273438233, 0.276762604,
0.280147967, 0.283594724, 0.28705879, 0.290593035, 0.294105196,
0.29768812, 0.301323445, 0.305035351, 0.30879475, 0.309606847,
0.312558807, 0.316411753, 0.320357641, 0.324276431, 0.328302277,
0.33230534, 0.336388802, 0.340570584, 0.344751858, 0.349025341,
0.353327782, 0.357661617, 0.36205913, 0.366544906, 0.37115523,
0.375810139, 0.380399454, 0.385094535, 0.389857025, 0.394711563,
0.399714862, 0.404690292, 0.409794402, 0.414954712, 0.420164393,
0.421280913, 0.425451397, 0.430850562, 0.436276693, 0.441713799,
0.447288424, 0.452893653, 0.458609363, 0.464430468, 0.470301678,
0.476272284, 0.482345067, 0.488492612, 0.494727245, 0.501115006,
0.507541463, 0.514068578, 0.520678407, 0.527348394, 0.534110416,
0.540950515, 0.547967676, 0.555115727, 0.562273951, 0.569568744,
0.576993745, 0.584523257, 0.592154612, 0.599917687, 0.607713074,
0.615596818, 0.623706214, 0.631899671, 0.640173286, 0.64868089,
0.657352787, 0.66099726, 0.666030822, 0.674861752, 0.683847573,
0.693090088, 0.702383943, 0.711907928, 0.721478, 0.731196639,
0.741090993, 0.75124593, 0.761471301, 0.771817849, 0.782359533,
0.793055399, 0.803890477, 0.814967289, 0.826350107, 0.837894249,
0.849556789, 0.861286508, 0.873357332, 0.885643214, 0.898144556,
0.910897279, 0.923839894, 0.929234229, 0.937048905, 0.950483452,
0.964185531, 0.978139774, 0.992415034, 1.006864564, 1.021632708,
1.036648802, 1.051876216, 1.067555865, 1.083410827, 1.099645017,
1.116237278, 1.133331765, 1.15072834, 1.168418329, 1.186332043,
1.204615669, 1.223261081, 1.238470801, 1.273169543, 1.292347159,
1.311529653, 1.330670059, 1.349853801, 1.369047081, 1.388213322,
1.407304319, 1.426709589, 1.445872365, 1.464786706, 1.483819304,
1.502778243, 1.521724761, 1.540787839, 1.559851693, 1.578729313,
1.59754388, 1.608681357, 1.616405935, 1.635338881, 1.654352094,
1.673298939, 1.692326088, 1.711257385, 1.730005205, 1.748899431,
1.767764188, 1.786614085, 1.805358695, 1.824121789, 1.843152312,
1.8619079, 1.880847101, 1.899812894, 1.918751915, 1.937717768,
1.956617902, 1.975558833, 1.994452053, 2.013368214, 2.032226097,
2.051114718, 2.069848936, 2.080739048, 2.088784696, 2.107671953,
2.12650614, 2.145239446, 2.164244275, 2.183152253, 2.202120195,
2.22106058, 2.240005476, 2.259004505, 2.278132671, 2.297313919,
2.316392038, 2.335571106, 2.354455635, 2.373562077, 2.392589559,
2.41152266, 2.430558357, 2.449891957, 2.469061895, 2.488182368,
2.507534587, 2.526678336, 2.545952271, 2.565375668, 2.584685816,
2.604175643, 2.623738339, 2.643232023, 2.662589544, 2.682136236,
2.70143267, 2.720770616, 2.740346503, 2.755918928, 2.759998346,
2.779483311, 2.798965618, 2.818736607, 2.838420409, 2.858065761,
2.877775454, 2.897490856, 2.917290775, 2.937155286, 2.957221247,
2.977182257, 2.997333599, 3.017186493, 3.037287378, 3.057715604,
3.077984487, 3.098247306, 3.118381399, 3.138768253, 3.159556465,
3.18009058, 3.200655011, 3.22137379, 3.24197305, 3.25804043,
3.262639247, 3.283763241, 3.304663415, 3.32566555, 3.346558535,
3.36741467, 3.388671978, 3.409820994, 3.43088824, 3.45218008,
3.473297762, 3.494787495, 3.516290418, 3.53810963, 3.559551085,
3.581429511, 3.603085532, 3.625151794, 3.647123899, 3.668906374,
3.691107691, 3.713237742, 3.735328081, 3.75767604, 3.77994893,
3.802647408, 3.82521704, 3.848071954, 3.870857346, 3.893439895,
3.91651857, 3.939617184, 3.962905574, 3.985991838, 4.009139674,
4.032727531, 4.056432769, 4.079951494, 4.10288784, 4.12682965,
4.150209061, 4.174037009, 4.197907807, 4.222134602, 4.246292684,
4.270888707, 4.29545014, 4.320382513, 4.345398157, 4.370342067,
4.39514011, 4.420623161, 4.445904355, 4.471727421, 4.497060199,
4.522838878, 4.548421695, 4.574319959, 4.600608587, 4.626870834,
4.653712759, 4.679735756, 4.70619241, 4.732740786, 4.759932979,
4.786311996, 4.813505041, 4.840804004, 4.868596264, 4.896032786,
4.923706597, 4.951279582, 4.979749495, 5.007770049, 5.036012685,
5.064494689, 5.092807286, 5.121926393, 5.150879838, 5.180136559,
5.209265099, 5.238976942, 5.267946851, 5.297243587, 5.327156439,
5.357114331, 5.387585431, 5.418358571, 5.449299075, 5.480570649,
5.511944673, 5.54290934, 5.573704772, 5.605322731, 5.637600047,
5.669068858, 5.675839907, 5.700608277, 5.73260392, 5.764722816,
5.797378221, 5.82959778, 5.862893732, 5.896275244, 5.929752182,
5.962279553, 5.995864991, 6.029664318, 6.062924119, 6.096597423,
6.131125706, 6.166001317, 6.200955215, 6.235371192, 6.270626065,
6.305504811, 6.340851391, 6.376434899, 6.412344996, 6.449029864,
6.486107952, 6.523253198, 6.559788413, 6.596155112, 6.632863894,
6.6709388, 6.70831778, 6.745440251, 6.782495124, 6.819652048,
6.857331092, 6.895137591, 6.934844883, 6.973436519, 7.011632054,
7.04980394, 7.089633232, 7.129665924, 7.169588438, 7.209837693,
7.249017909, 7.288317451, 7.328104607, 7.367692211, 7.409012509,
7.448938379, 7.489816286, 7.530968115, 7.57284129, 7.6140567,
7.656704052, 7.697701349, 7.739735933, 7.780941629, 7.82270829,
7.865690779, 7.907877445, 7.952568275, 7.994742376, 8.037989705,
8.081740621, 8.124965856, 8.168367424, 8.210160355, 8.25438353,
8.297803297, 8.342518717, 8.387127707, 8.432084638, 8.477940592,
8.522383355, 8.567098754, 8.613157284, 8.657491388, 8.701602314,
8.747571812, 8.793588573, 8.839448239, 8.886751347, 8.933370305,
8.981891184, 9.029016412, 9.076264216, 9.122778433, 9.169554978,
9.217843901, 9.263555886, 9.31116457, 9.357686101, 9.406488016,
9.456028214, 9.502560463, 9.548775686, 9.595076003, 9.644533524,
9.693789182, 9.744191433, 9.792547839, 9.841497544, 9.892343687,
9.943908302, 9.992547083, 10.044400319, 10.093748426, 10.145844024,
10.197304545, 10.246132006, 10.296753523, 10.348986287, 10.401063827,
10.453163512, 10.505665317, 10.558428636, 10.611467936, 10.666707813,
10.718070794, 10.770700849, 10.82559692, 10.878163106, 10.932044654,
10.985684152, 11.040765336, 11.098269326, 11.152187783, 11.207014192,
11.260501252, 11.317316796, 11.373772604, 11.42902536, 11.482800203,
11.542751808, 11.59960954, 11.654219015, 11.709260207, 11.767109809,
11.823914576, 11.88188905, 11.938309265, 11.997821166, 12.053238765,
12.110268293, 12.165757508, 12.221069927, 12.280189254, 12.33776211,
12.394517957, 12.451235208, 12.51147935, 12.566408926, 12.624748307,
12.685415126, 12.744857001, 12.807068632, 12.866453031, 12.927582094,
12.988538031, 13.05050561, 13.114825874, 13.172118304, 13.231295283,
13.29038397, 13.350130208, 13.410077023, 13.477513535, 13.538225812,
13.598033309, 13.66034823, 13.719205511, 13.781196889, 13.842873487,
13.904289046, 13.972053947, 14.039540021, 14.104772572, 14.166658256,
14.228151526, 14.290295808, 14.353039874, 14.421000333, 14.485505536,
14.54403443, 14.608975867, 14.680061662, 14.742774725, 14.811337142,
14.876899483, 14.942661146, 15.007737883, 15.077635102, 15.146467875,
15.212249424, 15.27897034, 15.347162724, 15.414348932, 15.482647694,
15.552910981, 15.624618481, 15.637778142, 15.690885725, 15.75629806,
15.828040478, 15.892256244, 15.954810551, 16.021052292, 16.093963135,
16.164583403, 16.231645763, 16.299280777, 16.365562343, 16.436903018,
16.508347363, 16.578960665, 16.649966029, 16.725018085, 16.804216221,
16.870065426, 16.943493816, 17.011241643, 17.089289522, 17.163599668,
17.236208084, 17.311025008, 17.377592713, 17.455784588, 17.53403016,
17.604086821, 17.679517927, 17.755454279, 17.826014784, 17.900250248,
17.975559741, 18.049033206, 18.129299825, 18.198360032, 18.276337204,
18.351191001, 18.426554538, 18.501876768, 18.581200497, 18.655835394,
18.730734179, 18.815455867, 18.889338636, 18.966014472, 19.041235862,
19.122071823, 19.210593463, 19.287561546, 19.374313423, 19.452353985,
19.528672186, 19.610160808, 19.687230584, 19.771127925, 19.853091327,
19.946342483, 20.023199611, 20.106275433, 20.187888853, 20.274479827,
20.35769825, 20.447799259, 20.532358111, 20.605743056, 20.695548806,
20.781492123, 20.87525966, 20.950534045, 21.037425855, 21.122744795,
21.208456816, 21.285534474, 21.370044605, 21.462167102, 21.547551448,
21.638179902, 21.727493456, 21.814489819, 21.903135932, 21.985888911,
22.06797323, 22.156642982, 22.245559077, 22.337715974, 22.423427456,
22.511597225, 22.606582378, 22.69782431, 22.786405057, 22.872212873,
22.984831794, 23.077492876, 23.167839209, 23.242235743, 23.325954814,
23.416286217, 23.520016054, 23.608896044, 23.69722519, 23.803888628,
23.891973105, 23.993756718, 24.091329606, 24.196557855, 24.303824291,
24.398458356, 24.516417918, 24.614080281, 24.706031712, 24.805323573,
24.901494057, 24.988314202, 25.086974529, 25.189203291, 25.280087786,
25.378563236, 25.48147597, 25.57385701, 25.662900123, 25.766864441,
25.859993815, 25.965953361, 26.063432487, 26.163996905, 26.249012205,
26.346129934, 26.438925211, 26.549605915, 26.651230491, 26.738204198,
26.8337174, 26.93987968, 27.04052921, 27.142582338, 27.246680456,
27.337721113, 27.450218977, 27.551347906, 27.651054685, 27.75640794,
27.860175723, 27.957842514, 28.051384992, 28.154748106, 28.253734934,
28.359983633, 28.469085716, 28.57108331, 28.677516257, 28.783325771,
28.892817297, 28.995230774, 29.127116011, 29.251578834, 29.368316275,
29.490621465, 29.60738801, 29.716332979, 29.821014572, 29.944634482,
30.04725117, 30.187190177, 30.293066923, 30.407675372, 30.513113821,
30.612966185, 30.737700441, 30.830818565, 30.942142421, 31.065883643,
31.175937431, 31.288231839, 31.392983607, 31.5153586, 31.631284784,
31.716651856, 31.834132984, 31.947802071, 32.061448985, 32.176930392,
32.303512831, 32.410847686, 32.50292025, 32.636924492, 32.763861065,
32.898973872, 33.001236806, 33.127903145, 33.229624167, 33.363226458,
33.481377256, 33.601445989, 33.724592287, 33.830961315, 33.951069208,
34.072363621, 34.191983416, 34.30599673, 34.330183402, 34.428532395,
34.551833472, 34.668490943, 34.79932498, 34.907983601, 35.028534935,
35.142082265, 35.254042519, 35.378747492, 35.530705147, 35.634046114,
35.744612596, 35.881138874, 35.994641489, 36.110047704, 36.208604251,
36.312424832, 36.459356544, 36.598236591, 36.684691438, 36.803298263,
36.876163358, 37.026008279, 37.151221593, 37.289313497, 37.418381679,
37.528129901, 37.666274697, 37.776143172, 37.891090769, 38.012953716,
38.161154389, 38.331631852, 38.469408609, 38.586976629, 38.700651633,
38.852013564, 39.000027495, 39.145256145, 39.285081685, 39.424288748,
39.5948626, 39.713277612, 39.863862861, 39.999702153, 40.167406189,
40.294202454, 40.457615113, 40.600570489, 40.737342816, 40.889658031,
41.066008299, 41.229108869, 41.370208548, 41.525624165, 41.653925868,
41.857860962, 42.003292783, 42.120770234, 42.230888178, 42.336096821,
42.476502075, 42.626706394, 42.750890505, 42.875050884, 43.016043544,
43.169320732, 43.297143816, 43.418964277, 43.567327816, 43.669589019,
43.853565392, 43.978112794, 44.134888078, 44.273411989, 44.386554887,
44.484391348, 44.673781362, 44.860939023, 45.031774213, 45.131048115,
45.236409394, 45.40396908, 45.553048039, 45.691537076, 45.851697364,
46.018501735, 46.160060483, 46.322406332, 46.543942739, 46.688810357,
46.905156345, 47.102586634, 47.247537997, 47.342334829, 47.486766149,
47.634615578, 47.794906132, 47.960806925, 48.129549646, 48.255667004,
48.38034197, 48.519876256, 48.636998192, 48.864887825, 49.031438511,
49.189800039, 49.325161453, 49.506207756, 49.646988261, 49.845995054,
49.984133724, 50.159393254, 50.28457472, 50.371443346, 50.531470022,
50.689921903, 50.882050164, 51.014624826, 51.204563589, 51.370609616,
51.578028238, 51.678886161, 51.874191311, 52.094546518, 52.264595752,
52.474629109, 52.613783926, 52.776005046, 52.915480874, 53.140883198,
53.295251703, 53.492554782, 53.594744044, 53.804492164, 53.966797233,
54.21420487, 54.360148551, 54.518680833, 54.736114876, 54.887544663,
55.041600667, 55.19674103, 55.326150792, 55.44262995, 55.587981405,
55.741965678, 56.023026612, 56.200026637, 56.334664589, 56.582582073,
56.755445723, 56.980989843, 57.253805121, 57.457969699, 57.631204287,
57.80173358, 58.120986706, 58.326798151, 58.545139287, 58.803236153,
58.991861273, 59.209281505, 59.400525631, 59.582660907, 59.799362899,
60.007058884, 60.129541758, 60.216412858, 60.304569443, 60.517052706,
60.745108704, 60.806345046, 61.094995267, 61.450886065, 61.614245026,
61.851932122, 61.948777827, 62.257140597, 62.460485744, 62.774182268,
62.912146719, 63.042453453, 63.529974333, 63.720216526, 63.875635081,
64.015605131, 64.202085789, 64.337223608, 64.573860305, 64.65280254,
64.88038526, 65.121500369, 65.421710576, 65.817864871, 66.047977293,
66.288594611, 66.729360472, 67.060520635, 67.383502545, 67.561517221,
67.674579419, 67.833397856, 67.883428984, 67.991305343, 68.055560036,
68.299364299, 68.49578908, 68.806136011, 69.051525167, 69.179536824,
69.591174394, 69.705969558, 69.927882707, 70.237598635, 70.403099401,
70.742158579, 70.960755258, 71.271656913, 71.537668102, 71.685674992,
71.865725227, 72.058422665, 72.408666213, 72.697134438, 72.847824895,
73.241984523, 73.78463016, 74.185402681, 74.287086063, 74.4225662,
74.57686395, 74.65658518, 74.803187577, 75.000938311, 75.335364355,
75.622968014, 75.818286388, 75.906742846, 76.071424013, 76.220693814,
76.531531278, 76.67053788, 76.975909462, 77.227155471, 77.454304673,
77.868560447, 78.040710378, 78.264838105, 78.346221059, 78.62008108,
78.975369548, 79.235135036, 79.635080147, 79.713985877, 79.8313585,
80.210185001, 80.406909521, 80.516219672, 80.781166684, 81.199409251,
81.573583697, 81.957785253, 82.351015079, 82.502513021, 82.695454079,
82.762124811, 83.225764739, 83.476522192, 83.532794547, 83.797386618,
84.009413304, 84.413236574, 84.587293557, 84.888131407, 84.955706531,
85.242989836, 85.786719351, 85.903449223, 86.102043266, 86.200920381,
86.265359007, 86.351527926, 86.926231924, 87.039947818, 87.152864509,
87.26522543, 87.281532934, 87.391328884, 87.697722469, 88.116591223,
88.322515857, 88.527017159, 88.636699835, 89.098801755, 89.317936781,
89.367733676, 89.651087149, 89.733668449, 89.870248644, 89.945881951,
90.159486118, 90.723851598, 90.996907293, 91.549074065, 91.687069832,
91.89061272, 92.035021411, 92.670151751, 92.823396559, 93.213879153,
93.997480214, 94.118452758, 94.138527566, 94.229693472, 94.336374218,
94.633727783, 94.808029443, 94.916164536, 95.089277537, 95.703442565,
95.901304245, 95.920210837, 96.457554562, 96.5041075, 97.434630918,
97.537815644, 97.55832051, 98.159334126, 98.666771517, 98.961602686,
99.043951239, 99.758603186, 100.050922605, 100.354212124, 100.631198417,
100.631564726, 100.705014003, 100.848257738, 101.042344695),
    "300" = c(2.2884e-05, 2.3131e-05, 2.3355e-05, 2.37e-05, 2.3933e-05,
    2.4135e-05, 2.4289e-05, 2.4299e-05, 2.4382e-05, 2.4423e-05,
    2.4918e-05, 2.4998e-05, 2.5047e-05, 2.5211e-05, 2.5471e-05,
    2.6257e-05, 2.6293e-05, 2.7116e-05, 2.72e-05, 2.7604e-05,
    2.7948e-05, 2.8078e-05, 2.8393e-05, 2.8547e-05, 2.8639e-05,
    2.8864e-05, 2.9719e-05, 2.9849e-05, 3.0095e-05, 3.0943e-05,
    3.1567e-05, 3.1958e-05, 3.2405e-05, 3.2653e-05, 3.2817e-05,
    3.2857e-05, 3.2864e-05, 3.3649e-05, 3.4155e-05, 3.4445e-05,
    3.4669e-05, 3.4776e-05, 3.6093e-05, 3.6501e-05, 3.6896e-05,
    3.7188e-05, 3.7521e-05, 3.8087e-05, 3.8152e-05, 3.8796e-05,
    3.9322e-05, 3.9447e-05, 3.962e-05, 3.9858e-05, 3.9877e-05,
    4.0072e-05, 4.1273e-05, 4.1468e-05, 4.1988e-05, 4.2132e-05,
    4.2731e-05, 4.3277e-05, 4.3514e-05, 4.4568e-05, 4.5052e-05,
    4.5591e-05, 4.7707e-05, 4.8029e-05, 4.8205e-05, 4.8913e-05,
    4.9764e-05, 5.0061e-05, 5.0504e-05, 5.0648e-05, 5.1223e-05,
    5.1728e-05, 5.2817e-05, 5.3126e-05, 5.3462e-05, 5.4067e-05,
    5.4322e-05, 5.4654e-05, 5.5184e-05, 5.5394e-05, 5.5854e-05,
    5.6038e-05, 5.6903e-05, 5.7588e-05, 5.8941e-05, 5.9916e-05,
    6.0427e-05, 6.1004e-05, 6.1706e-05, 6.2414e-05, 6.3536e-05,
    6.4822e-05, 6.5567e-05, 6.6059e-05, 6.7023e-05, 6.8357e-05,
    6.8954e-05, 6.9477e-05, 7.0199e-05, 7.1351e-05, 7.1885e-05,
    7.337e-05, 7.429e-05, 7.5015e-05, 7.5327e-05, 7.5876e-05,
    7.6277e-05, 7.6994e-05, 7.7438e-05, 7.8465e-05, 7.8719e-05,
    7.9066e-05, 8.0349e-05, 8.1023e-05, 8.1633e-05, 8.2461e-05,
    8.3319e-05, 8.4564e-05, 8.5231e-05, 8.5623e-05, 8.7237e-05,
    8.8588e-05, 9.0054e-05, 9.0625e-05, 9.239e-05, 9.2897e-05,
    9.3957e-05, 9.5639e-05, 9.6208e-05, 9.6963e-05, 9.8459e-05,
    9.9153e-05, 0.00010009, 0.000102074, 0.000103091, 0.000104211,
    0.000105345, 0.000107184, 0.000108044, 0.000109195, 0.000109939,
    0.000110633, 0.000111468, 0.00011272, 0.000113938, 0.000115706,
    0.000116733, 0.000117599, 0.00011876, 0.0001202, 0.000121773,
    0.000123671, 0.000124761, 0.000126295, 0.0001278, 0.000128908,
    0.000131722, 0.000133277, 0.000134439, 0.000135612, 0.000136925,
    0.000138711, 0.000140267, 0.000141527, 0.000143236, 0.000144725,
    0.000146727, 0.000148034, 0.000149756, 0.000151502, 0.000153625,
    0.000156315, 0.000157827, 0.000158769, 0.000160446, 0.000162233,
    0.000163264, 0.000166467, 0.000168761, 0.000170043, 0.00017151,
    0.000173036, 0.000174715, 0.000175718, 0.000177873, 0.00017969,
    0.000181713, 0.000183717, 0.000185216, 0.000187307, 0.000188826,
    0.000190276, 0.000192852, 0.000194293, 0.000196543, 0.000198047,
    0.000199629, 0.00020098, 0.000203949, 0.000206834, 0.000210817,
    0.000213776, 0.000216366, 0.000218648, 0.000220034, 0.000223091,
    0.000225112, 0.000227466, 0.000230114, 0.000232614, 0.00023483,
    0.000237815, 0.000240823, 0.000244121, 0.000246098, 0.000248349,
    0.000250647, 0.000253655, 0.000257268, 0.000260008, 0.00026352,
    0.000267214, 0.000270067, 0.000273567, 0.000275872, 0.000278907,
    0.000282235, 0.000284717, 0.000287769, 0.000289736, 0.00029393,
    0.000297265, 0.000300602, 0.000304761, 0.000307582, 0.000309754,
    0.000311984, 0.00031713, 0.000320607, 0.000324775, 0.0003282,
    0.000330966, 0.000334841, 0.000339156, 0.000342518, 0.000345015,
    0.000348371, 0.000353096, 0.000356352, 0.00036096, 0.000364421,
    0.000367942, 0.000371279, 0.000374643, 0.000379541, 0.000383113,
    0.000389129, 0.000392768, 0.000397812, 0.000403055, 0.000407721,
    0.000413403, 0.000418972, 0.000425422, 0.000429509, 0.000435104,
    0.000439708, 0.000445499, 0.000451308, 0.000456607, 0.000460886,
    0.000466234, 0.000473684, 0.000480004, 0.000487854, 0.000495682,
    0.000500607, 0.000505601, 0.000515004, 0.000520202, 0.000526251,
    0.000531501, 0.000537354, 0.000543238, 0.000551059, 0.000557883,
    0.000562174, 0.00056731, 0.000574159, 0.000579694, 0.000585988,
    0.00059259, 0.000599635, 0.000607736, 0.000614255, 0.000620699,
    0.000628682, 0.000636078, 0.000644388, 0.000652748, 0.000658721,
    0.00066568, 0.00067302, 0.000681753, 0.000688882, 0.000697895,
    0.000706174, 0.000715253, 0.000725785, 0.000732608, 0.000741904,
    0.000749183, 0.000758342, 0.000767423, 0.000774185, 0.000782422,
    0.00078941, 0.000797926, 0.000805435, 0.000814911, 0.000825945,
    0.000834505, 0.000842131, 0.000852851, 0.000859576, 0.000870765,
    0.000880778, 0.000889357, 0.00089726, 0.000908901, 0.000920762,
    0.000931699, 0.000943385, 0.00095404, 0.000965404, 0.000979241,
    0.000990876, 0.000994095, 0.001006564, 0.001018609, 0.001027188,
    0.001040976, 0.001051683, 0.001061972, 0.001073225, 0.0010887,
    0.0011009, 0.00111421, 0.001129473, 0.001142047, 0.001157377,
    0.001171628, 0.00118332, 0.001197084, 0.001209418, 0.001223133,
    0.00123737, 0.00125163, 0.001264725, 0.001279128, 0.001292941,
    0.001308616, 0.001324614, 0.001341689, 0.00136051, 0.001371732,
    0.001390364, 0.001405824, 0.001423307, 0.001439632, 0.001457397,
    0.001475464, 0.001490993, 0.001505853, 0.001523527, 0.001540823,
    0.001564722, 0.001581381, 0.001600338, 0.001618664, 0.001639953,
    0.001660597, 0.001678944, 0.001698817, 0.001720121, 0.001737282,
    0.001753579, 0.001770327, 0.001789645, 0.001812984, 0.001837274,
    0.001859917, 0.001884732, 0.001905966, 0.001932764, 0.001952881,
    0.001973089, 0.001995903, 0.002018224, 0.002040717, 0.002064365,
    0.00208558, 0.002112743, 0.002138751, 0.00216388, 0.002187356,
    0.002210238, 0.002235065, 0.00226045, 0.002284933, 0.002310983,
    0.002333348, 0.002356228, 0.002384845, 0.002409941, 0.002439673,
    0.002466961, 0.002497262, 0.002526558, 0.002553027, 0.002582408,
    0.002610675, 0.002637448, 0.002664911, 0.002694394, 0.002724715,
    0.002759396, 0.002796159, 0.002827489, 0.002858438, 0.002890454,
    0.002925976, 0.002959639, 0.002995885, 0.003029399, 0.003064992,
    0.003095156, 0.003132294, 0.003170404, 0.003209059, 0.00324478,
    0.00328332, 0.003324867, 0.003361959, 0.003398393, 0.003437679,
    0.003479451, 0.003519221, 0.003563449, 0.003602646, 0.003646732,
    0.003687922, 0.003735917, 0.003772917, 0.003815744, 0.003861411,
    0.003909421, 0.003954494, 0.003998775, 0.004041693, 0.004087238,
    0.004132433, 0.004175841, 0.004227128, 0.004278099, 0.004325645,
    0.004377726, 0.004426967, 0.004476168, 0.004525912, 0.004575911,
    0.004628858, 0.004682209, 0.004737867, 0.004787095, 0.004841426,
    0.004901056, 0.004954193, 0.005011922, 0.005072716, 0.005134275,
    0.005194403, 0.005256411, 0.005319617, 0.005379437, 0.005439927,
    0.005512579, 0.005575191, 0.005640975, 0.005710066, 0.005778861,
    0.005843761, 0.005908897, 0.005977679, 0.006044593, 0.006111742,
    0.006187055, 0.006258299, 0.006332987, 0.006407304, 0.006479196,
    0.006553977, 0.006626807, 0.006699213, 0.006774113, 0.006852363,
    0.006935505, 0.007009843, 0.007090118, 0.00717207, 0.007252722,
    0.007339339, 0.007422485, 0.007517673, 0.007594548, 0.007683089,
    0.007769718, 0.007855689, 0.007940547, 0.008029773, 0.008121652,
    0.008216817, 0.008320884, 0.008417769, 0.008518291, 0.008615857,
    0.008710034, 0.008795701, 0.008894559, 0.009002039, 0.009100569,
    0.009205626, 0.009310164, 0.009419072, 0.009529012, 0.009639273,
    0.009764297, 0.009858818, 0.009882487, 0.009990796, 0.010104986,
    0.010222683, 0.010336144, 0.010454128, 0.010576047, 0.010702771,
    0.010826118, 0.010951288, 0.011073043, 0.011192522, 0.011326667,
    0.011455671, 0.011589508, 0.011725865, 0.011857226, 0.011992801,
    0.012132987, 0.012276623, 0.012416226, 0.012560286, 0.012709681,
    0.012857753, 0.013010055, 0.013164343, 0.013311002, 0.013466779,
    0.013622483, 0.013781375, 0.01394753, 0.014106087, 0.01426869,
    0.014431812, 0.014598149, 0.014764245, 0.01494359, 0.015119447,
    0.015292754, 0.015480755, 0.015658702, 0.015848093, 0.016028955,
    0.016214604, 0.016404948, 0.01659044, 0.016789895, 0.016971874,
    0.017172219, 0.017369483, 0.01756594, 0.01776476, 0.017970072,
    0.0181764, 0.018396022, 0.0186112, 0.018817853, 0.01904691,
    0.019257984, 0.019482767, 0.019704505, 0.019941197, 0.020166911,
    0.020386877, 0.02061486, 0.020853025, 0.021105038, 0.021352478,
    0.021605875, 0.021847156, 0.022101921, 0.022360361, 0.022608503,
    0.022872535, 0.023139083, 0.023409243, 0.023677963, 0.023945187,
    0.024209316, 0.024494456, 0.024782541, 0.025065533, 0.025361358,
    0.025651694, 0.025949562, 0.026248245, 0.026561564, 0.026863365,
    0.027181913, 0.027513533, 0.027828884, 0.028148705, 0.028489754,
    0.02881716, 0.029151592, 0.029495572, 0.029841905, 0.030190028,
    0.030543288, 0.0308907, 0.031248325, 0.031592645, 0.031955563,
    0.032339273, 0.032719098, 0.033114208, 0.033492237, 0.033889863,
    0.034281809, 0.034674062, 0.0350732, 0.035482658, 0.035883654,
    0.036302115, 0.036729249, 0.037163303, 0.037602414, 0.038038936,
    0.038486281, 0.038929846, 0.039364153, 0.039813314, 0.040277227,
    0.040740591, 0.041226823, 0.041705061, 0.042207152, 0.042707106,
    0.043219984, 0.043731038, 0.044255215, 0.044775908, 0.045305874,
    0.045825867, 0.046375403, 0.046910985, 0.04744583, 0.048002279,
    0.048576381, 0.049158511, 0.049721174, 0.050299301, 0.050910126,
    0.051496167, 0.052111424, 0.052720288, 0.053343951, 0.053951558,
    0.054582674, 0.055207366, 0.055842413, 0.056496094, 0.057170186,
    0.057817605, 0.058508558, 0.059181919, 0.059882243, 0.060570191,
    0.061290516, 0.062018615, 0.062736564, 0.063467595, 0.064215166,
    0.064991899, 0.065755225, 0.066525817, 0.067281527, 0.0680777,
    0.06888686, 0.069687049, 0.070494694, 0.071349398, 0.072196311,
    0.073045823, 0.073906182, 0.074787902, 0.07567595, 0.076564266,
    0.077491619, 0.078393856, 0.079295592, 0.080211463, 0.081167442,
    0.082112559, 0.083069086, 0.084055557, 0.085038768, 0.086056931,
    0.087077821, 0.088104701, 0.089135903, 0.090175277, 0.091229195,
    0.092322615, 0.093425288, 0.094543828, 0.095655298, 0.096777164,
    0.097903863, 0.099069047, 0.100004731, 0.100245184, 0.10141756,
    0.102645803, 0.103836845, 0.105032827, 0.106294331, 0.107578459,
    0.108864509, 0.110143732, 0.111453941, 0.112806079, 0.114155352,
    0.11549319, 0.116860374, 0.118243331, 0.119648636, 0.121062217,
    0.12248221, 0.12392802, 0.125398711, 0.126876273, 0.12836925,
    0.129869798, 0.13141331, 0.132998296, 0.134552259, 0.136133984,
    0.137772354, 0.139413264, 0.141074391, 0.142743284, 0.144445478,
    0.146156839, 0.147895466, 0.14965362, 0.151441876, 0.153215185,
    0.155035395, 0.15689497, 0.158800376, 0.160702263, 0.162660559,
    0.164565557, 0.166519767, 0.168544493, 0.170556042, 0.172597741,
    0.174652467, 0.176757554, 0.178839487, 0.180990937, 0.183181302,
    0.185363585, 0.187619958, 0.18987009, 0.192173519, 0.194494998,
    0.196830445, 0.199198065, 0.201606299, 0.204049321, 0.20648426,
    0.209018463, 0.211536923, 0.214107765, 0.216719452, 0.219370629,
    0.222043675, 0.224714574, 0.227436291, 0.230188693, 0.232925307,
    0.235724309, 0.23860618, 0.241474517, 0.244379461, 0.247333883,
    0.250392838, 0.25348737, 0.256602307, 0.259772928, 0.262981681,
    0.266196807, 0.269524997, 0.272823424, 0.276166406, 0.279558641,
    0.283006099, 0.28650936, 0.290047235, 0.293629806, 0.297265651,
    0.300869081, 0.304615043, 0.308378163, 0.312226479, 0.313049631,
    0.316054735, 0.319951689, 0.323860895, 0.327871304, 0.33190799,
    0.335990032, 0.340195833, 0.344472336, 0.34873973, 0.353059518,
    0.35746051, 0.36190279, 0.366376347, 0.370934063, 0.375631198,
    0.380302424, 0.385048967, 0.389917262, 0.394872966, 0.399788797,
    0.404820419, 0.409927469, 0.415061894, 0.420281026, 0.4255773,
    0.426704991, 0.43099395, 0.43641438, 0.44194655, 0.447604389,
    0.453337083, 0.459119207, 0.464999107, 0.470954833, 0.476945834,
    0.483082797, 0.489282505, 0.495607117, 0.501953082, 0.508375018,
    0.514965853, 0.521581999, 0.528298251, 0.535120984, 0.542150735,
    0.549201477, 0.556397996, 0.563684351, 0.571026715, 0.578538575,
    0.586210984, 0.593960415, 0.601787089, 0.609783654, 0.617933833,
    0.626093624, 0.634454753, 0.642930301, 0.651581001, 0.66031806,
    0.669227855, 0.67298294, 0.678247049, 0.68743351, 0.696759829,
    0.706259913, 0.715932246, 0.725669489, 0.735626385, 0.745795531,
    0.756068782, 0.766370528, 0.777021999, 0.787855045, 0.798843062,
    0.810109455, 0.82142437, 0.832986548, 0.844769684, 0.856708314,
    0.86884834, 0.881169011, 0.893731788, 0.906395705, 0.919409264,
    0.932668202, 0.94617688, 0.951786332, 0.959814021, 0.973842806,
    0.988210091, 1.00283384, 1.017554719, 1.032541373, 1.047997775,
    1.063843365, 1.07984955, 1.095971348, 1.112705404, 1.12974357,
    1.147117673, 1.164822628, 1.182858017, 1.201362821, 1.220294947,
    1.239552996, 1.259198879, 1.275166998, 1.31149343, 1.331640289,
    1.35176868, 1.371980105, 1.392297169, 1.41242234, 1.432472761,
    1.452492877, 1.472583093, 1.492556267, 1.512696501, 1.532823902,
    1.552891859, 1.573111003, 1.593115961, 1.613088055, 1.633147083,
    1.653154044, 1.6648807, 1.673202056, 1.693094867, 1.713055671,
    1.7330995, 1.752995465, 1.772990925, 1.792887221, 1.812853882,
    1.832868507, 1.852780775, 1.872620676, 1.892586266, 1.912697773,
    1.932566827, 1.952552564, 1.972547931, 1.992564305, 2.012536338,
    2.032456625, 2.052380369, 2.072251706, 2.092371885, 2.112532159,
    2.132497866, 2.152377659, 2.16390912, 2.172418573, 2.19231035,
    2.212045828, 2.232139446, 2.252223435, 2.27224692, 2.292427149,
    2.312551575, 2.332507353, 2.352282443, 2.372252781, 2.392273001,
    2.412589575, 2.432647182, 2.452896803, 2.473135053, 2.493156348,
    2.513551806, 2.533484891, 2.553723887, 2.573842654, 2.59410859,
    2.614650483, 2.634990628, 2.655522546, 2.675599263, 2.69607313,
    2.7164701, 2.737067178, 2.757262296, 2.777578268, 2.798103125,
    2.818801015, 2.839381149, 2.860120597, 2.876418531, 2.880610489,
    2.90103486, 2.921459904, 2.941875159, 2.962471785, 2.983169746,
    3.003813323, 3.024643378, 3.045468049, 3.066580373, 3.087611924,
    3.108672449, 3.12945314, 3.15044228, 3.171696143, 3.192751679,
    3.213865262, 3.235045311, 3.256183498, 3.277262076, 3.298478077,
    3.319684269, 3.341070224, 3.362176633, 3.383593746, 3.400318582,
    3.405057377, 3.426570051, 3.448257818, 3.46978818, 3.491430998,
    3.513196014, 3.534827663, 3.556401098, 3.57831645, 3.600127341,
    3.621853197, 3.643698306, 3.665874478, 3.688058215, 3.710322612,
    3.732388906, 3.754717555, 3.776965827, 3.799575035, 3.822123945,
    3.844612025, 3.867034188, 3.889420657, 3.912245583, 3.935020487,
    3.957603467, 3.980493503, 4.003670914, 4.026445517, 4.049438721,
    4.072446924, 4.095600441, 4.118563999, 4.142166648, 4.165512679,
    4.189095562, 4.212728349, 4.23620956, 4.260106671, 4.283688411,
    4.307682061, 4.33133387, 4.355552715, 4.37993191, 4.404277037,
    4.42816594, 4.452413817, 4.476339314, 4.50129219, 4.525865142,
    4.550494775, 4.575245757, 4.60007012, 4.624972707, 4.650006526,
    4.675147181, 4.700308264, 4.725653376, 4.75054382, 4.776164419,
    4.80175657, 4.827162329, 4.853034498, 4.878746682, 4.904812962,
    4.930867119, 4.956751432, 4.982778582, 5.00920553, 5.035519548,
    5.062934137, 5.089272012, 5.115982646, 5.143421169, 5.170304813,
    5.197062788, 5.224594751, 5.251507722, 5.278767275, 5.306336873,
    5.334225347, 5.362252448, 5.390520993, 5.419026581, 5.447126439,
    5.47538812, 5.503807018, 5.532825085, 5.56160233, 5.5905954,
    5.619464218, 5.648823163, 5.678217164, 5.707520468, 5.737494701,
    5.767033105, 5.773171013, 5.796614658, 5.826552881, 5.857034652,
    5.886835259, 5.916920754, 5.947858832, 5.978536113, 6.009248697,
    6.039979014, 6.071464334, 6.102191066, 6.133335233, 6.165288995,
    6.196793807, 6.228723225, 6.26110458, 6.292632433, 6.324416484,
    6.356887339, 6.389218908, 6.42220263, 6.455184718, 6.487829257,
    6.520499801, 6.552964742, 6.586071232, 6.618720719, 6.652413226,
    6.686385495, 6.721083387, 6.75452151, 6.789059867, 6.822933793,
    6.857885145, 6.893080317, 6.927663071, 6.962584099, 6.997710407,
    7.032677571, 7.068134322, 7.102923578, 7.137943975, 7.17435133,
    7.209661884, 7.246316206, 7.282646, 7.318911158, 7.355924293,
    7.39295461, 7.429704585, 7.466602868, 7.50307988, 7.540250382,
    7.578173067, 7.615977925, 7.654212232, 7.691476772, 7.729345627,
    7.767586599, 7.805249617, 7.843685646, 7.881245454, 7.921295401,
    7.961131074, 8.000097379, 8.039150074, 8.078895235, 8.118974796,
    8.158438498, 8.198694638, 8.238796506, 8.278685087, 8.318045299,
    8.359020349, 8.401817986, 8.441602996, 8.482768233, 8.524718928,
    8.565549915, 8.605374274, 8.648580619, 8.689435855, 8.732699111,
    8.773036651, 8.813583404, 8.856678932, 8.899891478, 8.940637122,
    8.982612798, 9.02582245, 9.069356064, 9.113220583, 9.156287625,
    9.200880806, 9.244813879, 9.289094036, 9.333565294, 9.379798999,
    9.42202912, 9.46784803, 9.513356519, 9.558033751, 9.603423299,
    9.649849209, 9.696920592, 9.742835268, 9.788279745, 9.832744869,
    9.878486419, 9.925089239, 9.972066845, 10.015913298, 10.062697817,
    10.108487383, 10.15376449, 10.200421563, 10.247881746, 10.295251584,
    10.34164177, 10.390086462, 10.436160343, 10.484757971, 10.531102255,
    10.57897784, 10.627984315, 10.674716037, 10.719885184, 10.76938425,
    10.815904075, 10.863547193, 10.912839118, 10.963782523, 11.013942819,
    11.060873851, 11.109156779, 11.157178853, 11.206253009, 11.257222385,
    11.308065643, 11.358022625, 11.407106249, 11.455304806, 11.506053432,
    11.558281701, 11.60819275, 11.658339793, 11.711264435, 11.764601853,
    11.816722092, 11.87010925, 11.919647778, 11.971702815, 12.023919863,
    12.075936841, 12.127955799, 12.181751407, 12.234864866, 12.286810036,
    12.34220727, 12.394998422, 12.447296541, 12.499279203, 12.556467612,
    12.607011051, 12.661487338, 12.714888278, 12.767481186, 12.817276531,
    12.870694961, 12.925563833, 12.981304556, 13.037271232, 13.091493957,
    13.150249762, 13.206170589, 13.259619461, 13.316337618, 13.374181723,
    13.427834689, 13.482205753, 13.539103878, 13.593870735, 13.65072701,
    13.711485116, 13.769411521, 13.826543674, 13.884591261, 13.942374711,
    14.000455841, 14.057824186, 14.117124959, 14.175702202, 14.239623067,
    14.294270122, 14.353205359, 14.411195964, 14.468068479, 14.530781807,
    14.593025823, 14.656614826, 14.669063662, 14.718251291, 14.778116081,
    14.835727353, 14.896122073, 14.956486932, 15.01586936, 15.080661886,
    15.142197481, 15.203720295, 15.26272313, 15.325734174, 15.389165506,
    15.449676712, 15.511510388, 15.572855753, 15.634006198, 15.696549319,
    15.752855, 15.813128427, 15.881758874, 15.947078106, 16.009525793,
    16.070382479, 16.13212618, 16.198920464, 16.265888842, 16.331714017,
    16.399673828, 16.462141049, 16.526716014, 16.596730395, 16.653888603,
    16.720984636, 16.792000409, 16.858953481, 16.92702052, 16.999528993,
    17.062534164, 17.123910464, 17.191244221, 17.258486761, 17.321998718,
    17.38637966, 17.457287481, 17.518099354, 17.582847661, 17.64692211,
    17.709896789, 17.771620979, 17.844799405, 17.913349711, 17.979433886,
    18.049893237, 18.119689345, 18.194530444, 18.262985323, 18.33421336,
    18.398423939, 18.467445541, 18.540778106, 18.612348107, 18.678522604,
    18.748297035, 18.827027957, 18.895636558, 18.967551076, 19.047419761,
    19.118693324, 19.190125579, 19.258209116, 19.322916635, 19.398321218,
    19.466864881, 19.536001991, 19.602412585, 19.676774064, 19.754746312,
    19.826999198, 19.898049326, 19.978958833, 20.050406773, 20.122084341,
    20.194290897, 20.269911089, 20.337739439, 20.418808526, 20.486328997,
    20.56261013, 20.643243349, 20.726926451, 20.806164193, 20.873064192,
    20.948301164, 21.034662126, 21.112215405, 21.190175651, 21.263888333,
    21.3460939, 21.426119335, 21.49015411, 21.569122445, 21.654765138,
    21.740202246, 21.817383991, 21.888878364, 21.970201883, 22.043760147,
    22.124022015, 22.208716751, 22.295243074, 22.37006097, 22.453346046,
    22.554223715, 22.632260032, 22.720541767, 22.817376838, 22.903178272,
    22.996204937, 23.080838345, 23.166444549, 23.245451451, 23.32259807,
    23.408628043, 23.497007248, 23.578679092, 23.660718733, 23.743037818,
    23.8210651, 23.906196806, 23.984371399, 24.058602524, 24.13570984,
    24.218701278, 24.29812912, 24.383499752, 24.459330573, 24.547368911,
    24.628875781, 24.730341726, 24.829504801, 24.925024647, 25.014406232,
    25.100089095, 25.185825667, 25.282681132, 25.35468651, 25.431775711,
    25.505977219, 25.595640725, 25.671859477, 25.767118647, 25.862499607,
    25.963674815, 26.053909409, 26.152043635, 26.232431236, 26.335996279,
    26.428579951, 26.52597517, 26.613990875, 26.715463927, 26.810783758,
    26.903927953, 27.005817517, 27.113552244, 27.197190195, 27.277107205,
    27.346873909, 27.448091104, 27.554982786, 27.6422186, 27.723810674,
    27.827960244, 27.926587899, 28.035485501, 28.132372959, 28.223678618,
    28.340178576, 28.416000177, 28.536696751, 28.634478436, 28.725512556,
    28.822603204, 28.914669592, 29.008964582, 29.094529658, 29.20625135,
    29.305283843, 29.398990346, 29.511065042, 29.631475124, 29.739783621,
    29.827191749, 29.922442699, 30.04162369, 30.141401384, 30.245681953,
    30.331124292, 30.448209597, 30.541510408, 30.561369793, 30.672494475,
    30.780929241, 30.909545208, 31.022431991, 31.158054935, 31.270188111,
    31.373148986, 31.491160056, 31.604744387, 31.722657746, 31.847421679,
    31.953091328, 32.068279632, 32.185033623, 32.304317783, 32.402845448,
    32.51111865, 32.598703332, 32.702851245, 32.810461891, 32.909719877,
    32.99324121, 33.094815686, 33.205366211, 33.315510962, 33.405459753,
    33.510547189, 33.626006964, 33.746066866, 33.850221994, 33.980389097,
    34.079604979, 34.196702756, 34.289971842, 34.393202827, 34.522701797,
    34.629195852, 34.725371084, 34.820880227, 34.92083893, 35.023972748,
    35.147083372, 35.283099164, 35.401258843, 35.51011167, 35.633796132,
    35.734464182, 35.818907767, 35.922135123, 36.054370322, 36.173090531,
    36.280918154, 36.417641567, 36.551753433, 36.666460168, 36.785063385,
    36.938363596, 37.061891623, 37.16103913, 37.300831818, 37.4202161,
    37.512344966, 37.613073215, 37.724507087, 37.887649375, 38.013977621,
    38.15150249, 38.28289798, 38.408784731, 38.547000908, 38.65250222,
    38.769868093, 38.88994737, 38.99008586, 39.137688057, 39.26365927,
    39.368133747, 39.530168008, 39.655982471, 39.776192766, 39.883373149,
    39.99432811, 40.137362005, 40.224960187, 40.33940252, 40.470296439,
    40.621003695, 40.716953042, 40.854855309, 40.967537447, 41.069795652,
    41.152016721, 41.271414581, 41.394090366, 41.468170713, 41.552694137,
    41.646984682, 41.835584569, 41.979000808, 42.10420279, 42.211351514,
    42.328428348, 42.504966444, 42.607923222, 42.733855874, 42.873267528,
    42.998585255, 43.110998886, 43.262738407, 43.448140976, 43.574210749,
    43.721159414, 43.86289527, 43.971714625, 44.165019858, 44.278524602,
    44.375225855, 44.499807374, 44.590270621, 44.696001541, 44.824895358,
    44.950785832, 45.144741466, 45.230270016, 45.331596071, 45.459664598,
    45.553099511, 45.69349694, 45.901872225, 46.01332241, 46.175265911,
    46.348729669, 46.437005533, 46.550351476, 46.703628155, 46.872419655,
    46.999707057, 47.232543913, 47.436191476, 47.572207175, 47.664101465,
    47.896300734, 48.089819717, 48.255729188, 48.3483684, 48.436822277,
    48.618436121, 48.797606752, 49.032024927, 49.188466421, 49.323349681,
    49.414212204, 49.502111061, 49.623631739, 49.865684217, 49.958856076,
    50.138635129, 50.272425894, 50.419378852, 50.538209175, 50.617749622,
    50.748168135, 50.820726566, 50.939628194, 51.068685018, 51.373324868,
    51.64642702, 51.874655062, 52.042578188, 52.178030628, 52.304513572,
    52.434147205, 52.543074649, 52.626831808, 52.754036525, 52.811614965,
    53.00385667, 53.164094381, 53.339990969, 53.646210198, 53.748578508,
    54.019036976, 54.163453505, 54.300288136, 54.474736258, 54.766912287,
    55.094810321, 55.173244652, 55.264211753, 55.43145331, 55.676022359,
    55.84439681, 56.027592039, 56.285613036, 56.454019976, 56.676770555,
    56.824388307, 57.01108887, 57.215661803, 57.435419951, 57.577975984,
    57.864305005, 58.169868483, 58.236392886, 58.367642191, 58.527080163,
    58.667064935, 58.825557464, 58.958943457, 58.989507119, 59.089437269,
    59.357501461, 59.614224581, 59.686204551, 59.80546501, 60.050999409,
    60.437658238, 60.663621891, 60.852227437, 60.968359186, 61.125960679,
    61.343114946, 61.539419401, 61.752846393, 62.192378273, 62.494208182,
    62.666415659, 62.827642987, 62.898918404, 63.058045705, 63.267572807,
    63.34982421, 63.414396141, 63.533776221, 63.692219485, 63.848773336,
    64.170091501, 64.429318134, 64.710950121, 64.852851386, 64.962947774,
    65.143039879, 65.332186156, 65.459118751, 65.578890164, 65.67080561,
    65.823783573, 65.996102493, 66.180813901, 66.310363582, 66.482963336,
    66.807704547, 66.959651074, 67.240388793, 67.416324357, 67.654921544,
    67.818154819, 68.017443634, 68.119851161, 68.378947474, 68.44002366,
    68.596258966, 68.619847288, 68.678903626, 69.040362459, 69.197496661,
    69.504011389, 69.548852898, 69.815462445, 69.841128832, 70.098281227,
    70.203767363, 70.310404639, 70.62535785, 70.931867587, 71.056631738,
    71.103693932, 71.190278919, 71.41457454, 71.575424281, 72.126466677,
    72.374119111, 72.388187928, 72.835804505, 73.264824979, 73.34026563,
    73.591888979, 73.661133291, 73.809734179, 73.86115672, 73.936565439,
    74.272126831, 74.513585214, 74.775096778, 74.850932568, 75.31252534,
    75.424842618, 75.528873428, 75.584956827, 75.595110519, 75.608130155,
    75.658669119, 76.154230117, 76.187842257, 76.249646813, 76.268395596,
    76.364943498, 76.394125555, 76.598522787, 77.031511229, 77.103355725,
    77.131211483, 77.358088615, 77.480792612, 77.718040932, 77.719374848,
    78.077622539, 78.549874316, 78.612048024, 78.867694625, 79.292918126,
    79.303189021, 79.370574168, 79.612957854, 79.763449682, 79.885949272,
    80.023758765, 80.11709534, 80.274853576, 80.612018386, 80.62324931,
    80.631394152, 80.995763814, 81.103803545, 81.184236131, 81.250025678,
    81.425645816, 81.534539473, 81.698623699, 82.31976964), "500" = c(1.7727e-05,
    1.7783e-05, 1.7991e-05, 1.8408e-05, 1.8545e-05, 1.8714e-05,
    1.8763e-05, 1.8792e-05, 1.8891e-05, 1.8948e-05, 1.8991e-05,
    1.9625e-05, 1.9958e-05, 2.0019e-05, 2.0322e-05, 2.0513e-05,
    2.0749e-05, 2.116e-05, 2.1339e-05, 2.1426e-05, 2.2285e-05,
    2.2341e-05, 2.2722e-05, 2.2838e-05, 2.3696e-05, 2.3734e-05,
    2.4212e-05, 2.4384e-05, 2.4771e-05, 2.4802e-05, 2.5052e-05,
    2.5173e-05, 2.551e-05, 2.5752e-05, 2.6148e-05, 2.6267e-05,
    2.6272e-05, 2.6388e-05, 2.7364e-05, 2.7789e-05, 2.7864e-05,
    2.8547e-05, 2.8818e-05, 2.924e-05, 2.9504e-05, 2.9683e-05,
    3.0157e-05, 3.0701e-05, 3.0823e-05, 3.1171e-05, 3.149e-05,
    3.1672e-05, 3.2433e-05, 3.3203e-05, 3.367e-05, 3.4681e-05,
    3.5084e-05, 3.5279e-05, 3.5363e-05, 3.5969e-05, 3.662e-05,
    3.6758e-05, 3.6845e-05, 3.7058e-05, 3.7293e-05, 3.744e-05,
    3.7536e-05, 3.8423e-05, 3.9167e-05, 4.0534e-05, 4.1733e-05,
    4.1938e-05, 4.2154e-05, 4.2597e-05, 4.3026e-05, 4.3405e-05,
    4.3973e-05, 4.4851e-05, 4.5284e-05, 4.6293e-05, 4.7064e-05,
    4.7453e-05, 4.7972e-05, 4.9222e-05, 4.992e-05, 5.0718e-05,
    5.1045e-05, 5.16e-05, 5.2104e-05, 5.2664e-05, 5.2941e-05,
    5.4143e-05, 5.4529e-05, 5.4793e-05, 5.6915e-05, 5.7641e-05,
    5.9264e-05, 5.9791e-05, 6.0639e-05, 6.1945e-05, 6.3041e-05,
    6.3524e-05, 6.4227e-05, 6.4942e-05, 6.5779e-05, 6.645e-05,
    6.7968e-05, 6.8493e-05, 6.9813e-05, 7.0478e-05, 7.1215e-05,
    7.175e-05, 7.229e-05, 7.4107e-05, 7.527e-05, 7.6396e-05,
    7.6833e-05, 7.8032e-05, 7.8746e-05, 7.97e-05, 8.0277e-05,
    8.1361e-05, 8.2358e-05, 8.3554e-05, 8.4799e-05, 8.5159e-05,
    8.6095e-05, 8.6799e-05, 8.7798e-05, 8.8394e-05, 8.8718e-05,
    8.9298e-05, 9.042e-05, 9.1361e-05, 9.4037e-05, 9.5389e-05,
    9.6244e-05, 9.709e-05, 9.7981e-05, 9.9118e-05, 0.000100306,
    0.000101361, 0.000102976, 0.000103808, 0.000105161, 0.00010596,
    0.000107509, 0.000109236, 0.000110472, 0.000111353, 0.000112327,
    0.000113264, 0.000114781, 0.00011574, 0.000116749, 0.000117981,
    0.000119903, 0.000120851, 0.000122887, 0.000123768, 0.000124762,
    0.000126462, 0.000127462, 0.000128524, 0.000129949, 0.00013171,
    0.000133386, 0.000135788, 0.000137465, 0.000138891, 0.000140846,
    0.000141898, 0.000143416, 0.000144303, 0.000146069, 0.000147818,
    0.000149558, 0.000151178, 0.000152496, 0.000154012, 0.000155818,
    0.000158094, 0.000159168, 0.000161254, 0.00016305, 0.000164538,
    0.000167199, 0.000169274, 0.000170696, 0.000173576, 0.000175257,
    0.00017653, 0.000178935, 0.000180739, 0.000183356, 0.00018655,
    0.000189811, 0.00019121, 0.00019278, 0.000195043, 0.000196566,
    0.000200963, 0.000202047, 0.000204421, 0.00020753, 0.000208746,
    0.000211526, 0.000213674, 0.000216373, 0.000218032, 0.000220208,
    0.000223549, 0.000227275, 0.000230777, 0.000232466, 0.000234382,
    0.000237106, 0.000239708, 0.00024261, 0.000245166, 0.000247327,
    0.000251654, 0.000254534, 0.000257794, 0.00026056, 0.000264745,
    0.0002672, 0.000270403, 0.000272799, 0.000275633, 0.000277761,
    0.000279295, 0.000281783, 0.000285206, 0.000288242, 0.000291333,
    0.000295802, 0.000300068, 0.000303487, 0.000307137, 0.000310975,
    0.000312997, 0.000316359, 0.000321214, 0.000324845, 0.000330302,
    0.000333369, 0.000336605, 0.000340501, 0.000345066, 0.000349918,
    0.000354898, 0.000358925, 0.00036488, 0.000369245, 0.000372564,
    0.000374768, 0.00038048, 0.000384701, 0.000389068, 0.000393812,
    0.000397423, 0.000402353, 0.000407619, 0.000411857, 0.000417505,
    0.000421575, 0.000426067, 0.000428931, 0.000435315, 0.000438808,
    0.000444196, 0.000448275, 0.000451998, 0.000457034, 0.000462493,
    0.000465263, 0.000470802, 0.000475549, 0.000478862, 0.000484596,
    0.000490522, 0.000495598, 0.000503378, 0.000509144, 0.000516116,
    0.000521923, 0.00052719, 0.000533543, 0.000540771, 0.000545602,
    0.000553064, 0.000558977, 0.000567465, 0.000573377, 0.000582895,
    0.000588935, 0.000595258, 0.000601478, 0.000609005, 0.000615293,
    0.000621284, 0.000628816, 0.000636894, 0.000643942, 0.000651241,
    0.000657426, 0.0006627, 0.000671226, 0.000681911, 0.000688554,
    0.000695512, 0.000703149, 0.000711624, 0.000719066, 0.000727415,
    0.000735671, 0.000744275, 0.000754932, 0.000766157, 0.000776249,
    0.000787796, 0.000797596, 0.000806906, 0.000816061, 0.000824969,
    0.000832518, 0.000844409, 0.000855563, 0.000863515, 0.000873225,
    0.000879768, 0.000891763, 0.000902706, 0.000912635, 0.000924044,
    0.0009318, 0.000943607, 0.000953494, 0.000963669, 0.000975029,
    0.00097888, 0.000987691, 0.000997279, 0.001009691, 0.001019756,
    0.001030717, 0.001041081, 0.001054215, 0.001065344, 0.001073662,
    0.001088098, 0.001102496, 0.00111247, 0.001129093, 0.001140409,
    0.00115381, 0.001167433, 0.001183115, 0.001198521, 0.0012154,
    0.001229553, 0.001243345, 0.001257961, 0.001274782, 0.001288804,
    0.001300539, 0.001314108, 0.001330962, 0.001346265, 0.001362809,
    0.001383477, 0.001398966, 0.001418319, 0.001433089, 0.001451006,
    0.001466043, 0.001482515, 0.001495569, 0.00151223, 0.001524874,
    0.001548381, 0.00156931, 0.001586793, 0.001604019, 0.001622668,
    0.001642967, 0.001661422, 0.001680567, 0.001703788, 0.001722511,
    0.001743123, 0.001762335, 0.00177988, 0.001799287, 0.00182004,
    0.001840221, 0.00186165, 0.001883112, 0.001904484, 0.001928344,
    0.001948684, 0.001972608, 0.00199577, 0.002018736, 0.002044888,
    0.002071021, 0.002097235, 0.002122945, 0.002149876, 0.00217529,
    0.00219909, 0.0022239, 0.002246988, 0.002270185, 0.002298586,
    0.002326894, 0.002352991, 0.00237824, 0.002406758, 0.002435488,
    0.002466303, 0.00249548, 0.002521254, 0.002549546, 0.002580839,
    0.002614884, 0.002648396, 0.002678341, 0.002711376, 0.002746241,
    0.002784082, 0.002816699, 0.002849323, 0.002881222, 0.002913566,
    0.002948418, 0.002980681, 0.003016866, 0.003053166, 0.003089022,
    0.003124675, 0.003162876, 0.003201588, 0.003238599, 0.003272894,
    0.003310682, 0.003348302, 0.003387331, 0.003432126, 0.003472907,
    0.003514139, 0.003549837, 0.003589247, 0.003629955, 0.003670133,
    0.00371799, 0.003763303, 0.003810067, 0.003855004, 0.003897008,
    0.003948839, 0.003995898, 0.004046852, 0.004091308, 0.004140213,
    0.004188211, 0.004235087, 0.0042824, 0.004332227, 0.004380904,
    0.004434517, 0.004485258, 0.004538118, 0.004589974, 0.004644285,
    0.00469734, 0.004751781, 0.004805988, 0.004866046, 0.004926999,
    0.004978297, 0.005041672, 0.005100242, 0.005159896, 0.005225132,
    0.005285516, 0.005344045, 0.005406639, 0.005471548, 0.005541083,
    0.00560061, 0.005668671, 0.005731306, 0.00580118, 0.005869786,
    0.005939337, 0.006007329, 0.00607617, 0.006146536, 0.006209105,
    0.006281796, 0.006353016, 0.006422175, 0.006491868, 0.006563836,
    0.006640692, 0.006718572, 0.006794301, 0.006873586, 0.006951689,
    0.007030237, 0.007112833, 0.007191698, 0.007270097, 0.007357195,
    0.007442439, 0.007528627, 0.007612709, 0.00769685, 0.007785474,
    0.007873259, 0.007970638, 0.008061777, 0.008151188, 0.008242842,
    0.008347416, 0.00844203, 0.008540573, 0.008637446, 0.008741672,
    0.008839618, 0.008941671, 0.009042671, 0.009149377, 0.009253212,
    0.009360084, 0.009467898, 0.009579029, 0.009697314, 0.009812891,
    0.009902731, 0.009924934, 0.01004341, 0.010160839, 0.010280894,
    0.010394195, 0.010515152, 0.010632054, 0.010753145, 0.010870971,
    0.010999679, 0.011122119, 0.011252893, 0.011377273, 0.011506134,
    0.011629036, 0.011766531, 0.011907155, 0.012044695, 0.012186392,
    0.012327229, 0.012473083, 0.012618368, 0.012766531, 0.012905646,
    0.013051729, 0.013210558, 0.013359191, 0.013521362, 0.013677249,
    0.013836351, 0.014001354, 0.014149048, 0.014311886, 0.014482396,
    0.014644934, 0.014807731, 0.014995878, 0.015171166, 0.015352825,
    0.015522532, 0.015688935, 0.015869923, 0.016056611, 0.016248947,
    0.016442098, 0.01662586, 0.016823216, 0.017015342, 0.017208924,
    0.017408074, 0.017612464, 0.017815905, 0.018003737, 0.018202284,
    0.018421603, 0.018639085, 0.018860198, 0.019077347, 0.01929289,
    0.019513928, 0.01975137, 0.019971156, 0.020207952, 0.020447182,
    0.020700311, 0.020943615, 0.02117714, 0.021416261, 0.021660473,
    0.021910526, 0.02215964, 0.022395157, 0.022661611, 0.022926701,
    0.023203637, 0.02346564, 0.023729936, 0.024006155, 0.024291853,
    0.024577178, 0.024848353, 0.025139984, 0.025432202, 0.025721293,
    0.02600402, 0.026319233, 0.026633027, 0.026952569, 0.027274568,
    0.027589022, 0.027906081, 0.02824184, 0.028566374, 0.02890281,
    0.02924324, 0.02959762, 0.029932227, 0.030284202, 0.030638711,
    0.030977415, 0.031331497, 0.031685632, 0.032065728, 0.032432391,
    0.032813811, 0.033181451, 0.033573165, 0.033965966, 0.03437743,
    0.034776785, 0.035181456, 0.035594747, 0.036013743, 0.036428654,
    0.036848414, 0.037281858, 0.037721108, 0.038181028, 0.038635977,
    0.039093152, 0.039554408, 0.040006857, 0.040464257, 0.040936093,
    0.041404087, 0.041885207, 0.042374164, 0.042873764, 0.043373339,
    0.043877803, 0.044392459, 0.044899095, 0.045430021, 0.045957049,
    0.046490207, 0.047044507, 0.047599772, 0.048150603, 0.048730087,
    0.049312599, 0.049893273, 0.050488539, 0.05109333, 0.051712332,
    0.052324527, 0.052938859, 0.053557827, 0.0541861, 0.054819561,
    0.055446066, 0.056108742, 0.056791013, 0.057457735, 0.058122283,
    0.058799883, 0.059506015, 0.060200977, 0.060922853, 0.061632516,
    0.062347456, 0.063090069, 0.063823191, 0.064580302, 0.065328658,
    0.066074961, 0.066852687, 0.067647855, 0.068449441, 0.069262015,
    0.070069705, 0.070876522, 0.071726028, 0.072559297, 0.073413516,
    0.07428699, 0.075155442, 0.076050621, 0.076933707, 0.077828158,
    0.078765439, 0.079672288, 0.080609223, 0.081546131, 0.08253084,
    0.083477668, 0.084465374, 0.085459994, 0.086449478, 0.087470002,
    0.088489321, 0.089550016, 0.090627782, 0.091695974, 0.09278169,
    0.093862415, 0.094973727, 0.096097182, 0.097245312, 0.098400938,
    0.09955397, 0.10051167, 0.100745279, 0.101958465, 0.103162363,
    0.104368375, 0.105607529, 0.106876297, 0.108135732, 0.109392875,
    0.11069744, 0.11201332, 0.113378493, 0.114721644, 0.11611093,
    0.117496588, 0.118880059, 0.120303768, 0.121769391, 0.123199732,
    0.124663139, 0.126122397, 0.127592563, 0.129114619, 0.130645527,
    0.132206987, 0.133751847, 0.135387498, 0.136984598, 0.138612526,
    0.140265034, 0.141921283, 0.143626405, 0.145378635, 0.14712739,
    0.14889396, 0.150706048, 0.152507023, 0.154329804, 0.156187798,
    0.158083891, 0.159982902, 0.161909059, 0.163852207, 0.165804151,
    0.167805186, 0.169810891, 0.17187395, 0.173939071, 0.176051551,
    0.178177183, 0.180351197, 0.182557626, 0.184751815, 0.186997434,
    0.189234347, 0.191526259, 0.193831608, 0.196205497, 0.198628468,
    0.201049966, 0.203517331, 0.206033258, 0.208503211, 0.210995201,
    0.213512453, 0.216114634, 0.218737409, 0.221400069, 0.224038969,
    0.226769501, 0.229529793, 0.232310671, 0.235138335, 0.238031694,
    0.240927908, 0.243917304, 0.246852235, 0.249891455, 0.252910707,
    0.256028504, 0.259177299, 0.262383481, 0.265629218, 0.26890343,
    0.272199737, 0.275541387, 0.27895663, 0.282327857, 0.285856507,
    0.289370675, 0.292986132, 0.296578276, 0.300261682, 0.303932607,
    0.307691835, 0.311513211, 0.315425705, 0.316281525, 0.319351896,
    0.323307816, 0.327279764, 0.331357641, 0.335570856, 0.339814632,
    0.344091159, 0.348403509, 0.352784598, 0.357158199, 0.361660983,
    0.366263062, 0.370920253, 0.375566717, 0.380351133, 0.385108638,
    0.389986574, 0.394946129, 0.399996546, 0.405054319, 0.410180469,
    0.415356909, 0.420615826, 0.425897041, 0.431344766, 0.432469727,
    0.436910902, 0.442509538, 0.448184789, 0.453840651, 0.459686868,
    0.465585783, 0.47151114, 0.477522262, 0.483715693, 0.49003185,
    0.496443416, 0.502848726, 0.509325627, 0.515945604, 0.522660364,
    0.529545806, 0.536434074, 0.543505098, 0.550634729, 0.55792048,
    0.565277486, 0.572798936, 0.580420835, 0.588091292, 0.595956787,
    0.603838857, 0.611901322, 0.620025291, 0.628340015, 0.63683342,
    0.645415402, 0.654079783, 0.662941319, 0.672020268, 0.681219062,
    0.685067114, 0.690465978, 0.699845077, 0.709394843, 0.719185127,
    0.729067247, 0.739295123, 0.749548319, 0.759836407, 0.770347176,
    0.781127874, 0.792128259, 0.803228249, 0.814530822, 0.826018006,
    0.837758601, 0.849710799, 0.861850189, 0.874202187, 0.886800598,
    0.899520663, 0.912621469, 0.925867319, 0.939349693, 0.953139238,
    0.967307531, 0.973184782, 0.981605903, 0.996244465, 1.011054819,
    1.026177918, 1.041528957, 1.05722549, 1.073136443, 1.089461743,
    1.106013386, 1.123035164, 1.140331321, 1.158204911, 1.176268558,
    1.194937424, 1.213789512, 1.233148437, 1.252814013, 1.272931499,
    1.293507888, 1.310094831, 1.347914446, 1.368997397, 1.390106943,
    1.411101797, 1.432213187, 1.453163783, 1.47421191, 1.495157288,
    1.516250281, 1.537146438, 1.558357387, 1.57950644, 1.600501685,
    1.621441631, 1.642367315, 1.663305906, 1.684128763, 1.705056276,
    1.717465687, 1.726162603, 1.74728048, 1.768372316, 1.789367163,
    1.810414388, 1.831338973, 1.85227706, 1.873159203, 1.894217334,
    1.915215721, 1.936245586, 1.957203048, 1.978355776, 1.999505854,
    2.020627059, 2.041761792, 2.06265371, 2.083650581, 2.10471774,
    2.125650899, 2.146618044, 2.167699471, 2.188736133, 2.209611668,
    2.230571658, 2.242686644, 2.251617637, 2.272585791, 2.29368983,
    2.314549013, 2.335665081, 2.356690225, 2.377848502, 2.398749402,
    2.419806603, 2.440856588, 2.462116242, 2.483173685, 2.504314658,
    2.52525356, 2.546400504, 2.567483312, 2.588685547, 2.609810849,
    2.630787309, 2.651753099, 2.672849518, 2.693943654, 2.715066824,
    2.73646266, 2.757480569, 2.778825597, 2.80001159, 2.821362841,
    2.842669667, 2.86391437, 2.885392251, 2.906841936, 2.92819327,
    2.949725164, 2.97098111, 2.988200105, 2.992545883, 3.014003201,
    3.035548395, 3.057013277, 3.078532186, 3.100233985, 3.121954919,
    3.143485934, 3.165000948, 3.186550741, 3.208172768, 3.229827686,
    3.25175922, 3.273709208, 3.295475884, 3.317362359, 3.339312972,
    3.361041203, 3.383190761, 3.40489858, 3.42662666, 3.44855929,
    3.47039153, 3.492419221, 3.514241966, 3.531215809, 3.535946369,
    3.558254201, 3.580300952, 3.602516069, 3.624872315, 3.646915918,
    3.669513878, 3.691438172, 3.713888148, 3.736308469, 3.758725662,
    3.781208553, 3.804112593, 3.826337032, 3.848694925, 3.871019349,
    3.893840141, 3.916139817, 3.938762623, 3.961724382, 3.984293544,
    4.006963148, 4.029931461, 4.052749686, 4.075579188, 4.09861402,
    4.121853873, 4.144771479, 4.167942103, 4.191517284, 4.215020148,
    4.238252855, 4.261309056, 4.285037719, 4.308323692, 4.331724901,
    4.355058651, 4.37860298, 4.402100123, 4.425842672, 4.44972818,
    4.473444734, 4.49736209, 4.521020312, 4.544760554, 4.568468034,
    4.592352166, 4.616238479, 4.640188739, 4.664437949, 4.688588414,
    4.712683916, 4.736755919, 4.761350155, 4.786060392, 4.810970362,
    4.835776919, 4.859837818, 4.884432505, 4.909407267, 4.934400349,
    4.959371413, 4.984335233, 5.009227017, 5.034557744, 5.059380904,
    5.084720389, 5.110443662, 5.135385611, 5.16036355, 5.186579623,
    5.212076284, 5.237707223, 5.263375099, 5.288670427, 5.314354765,
    5.339865038, 5.366196621, 5.391960743, 5.418436531, 5.444866418,
    5.47113315, 5.497361402, 5.524189925, 5.551064221, 5.578033251,
    5.604529297, 5.631532451, 5.658526091, 5.684958518, 5.712399376,
    5.739941203, 5.767304476, 5.794484434, 5.822064341, 5.849358308,
    5.855076347, 5.877687309, 5.905566023, 5.934614022, 5.961634481,
    5.989596649, 6.017720069, 6.045757496, 6.07406089, 6.102386187,
    6.130747145, 6.15893939, 6.187267972, 6.216107016, 6.245327291,
    6.274139284, 6.303702664, 6.332928671, 6.363132702, 6.391828563,
    6.420994407, 6.450010405, 6.479365359, 6.50925137, 6.538856277,
    6.569730675, 6.600015843, 6.630340794, 6.660902819, 6.691815416,
    6.722511051, 6.754263652, 6.785016193, 6.815966283, 6.846684449,
    6.877745805, 6.90880202, 6.939908808, 6.971921128, 7.003502936,
    7.034776354, 7.0664353, 7.098534171, 7.130539917, 7.162295333,
    7.194101964, 7.22596642, 7.258190147, 7.291164917, 7.324773521,
    7.356925707, 7.389530297, 7.423474228, 7.455760261, 7.488576038,
    7.522141716, 7.555018531, 7.588339528, 7.62145348, 7.654909128,
    7.689943998, 7.724621817, 7.758015029, 7.793238566, 7.826815383,
    7.861585672, 7.896190576, 7.930342295, 7.9644243, 7.999166692,
    8.034824165, 8.070220332, 8.105397817, 8.140242774, 8.174818448,
    8.211098741, 8.245201036, 8.282044821, 8.317873829, 8.353479786,
    8.389041124, 8.425041782, 8.460407839, 8.497160261, 8.533744848,
    8.568664918, 8.605523838, 8.642962345, 8.679795476, 8.716859898,
    8.754538942, 8.792981718, 8.830394793, 8.869133627, 8.906127958,
    8.9433922, 8.980354944, 9.018389543, 9.056671516, 9.096085726,
    9.134819557, 9.17421137, 9.214075491, 9.253618707, 9.292498096,
    9.332026869, 9.370493219, 9.409274537, 9.449021459, 9.48971517,
    9.5306034, 9.572122758, 9.613081636, 9.656210351, 9.697383253,
    9.736981218, 9.77607962, 9.816362029, 9.857432734, 9.898872538,
    9.939590093, 9.981122592, 10.020767572, 10.059825075, 10.101330796,
    10.143877454, 10.184840735, 10.226754963, 10.269772967, 10.313101036,
    10.353581143, 10.396429669, 10.439252639, 10.481105333, 10.522000081,
    10.564038924, 10.606199912, 10.650076642, 10.692380731, 10.735774294,
    10.778293955, 10.819785717, 10.862496117, 10.905647546, 10.948875753,
    10.991933952, 11.03695833, 11.079989033, 11.123080228, 11.168795573,
    11.212338286, 11.258279597, 11.303666586, 11.345954491, 11.390983152,
    11.435836584, 11.481614036, 11.528660714, 11.570434238, 11.615462125,
    11.65868236, 11.700316931, 11.744658725, 11.790121964, 11.835762221,
    11.880502608, 11.926107928, 11.973496249, 12.017983645, 12.066532372,
    12.115241279, 12.161493044, 12.210600208, 12.25707086, 12.302841703,
    12.353692517, 12.403390373, 12.448809576, 12.497752698, 12.546367687,
    12.595283535, 12.644112372, 12.691040632, 12.740738166, 12.790027422,
    12.836530034, 12.883923934, 12.930781988, 12.977968361, 13.029062074,
    13.080349176, 13.131906022, 13.181067834, 13.225047163, 13.274595722,
    13.326100112, 13.375513197, 13.424074524, 13.473910138, 13.522662432,
    13.571467304, 13.583278605, 13.62153765, 13.670268387, 13.720353151,
    13.773984815, 13.827040037, 13.872791782, 13.923674013, 13.975229157,
    14.024485967, 14.073718061, 14.125786107, 14.172298065, 14.223621964,
    14.275997998, 14.330267293, 14.384277573, 14.437095555, 14.490727094,
    14.540463983, 14.595616561, 14.647229974, 14.698224272, 14.74847984,
    14.801944669, 14.858099413, 14.909906967, 14.964985317, 15.017907707,
    15.073853655, 15.125821167, 15.174416744, 15.22850578, 15.288246736,
    15.339712563, 15.391103097, 15.44410959, 15.504738324, 15.562980915,
    15.621180943, 15.67439692, 15.72229515, 15.774534974, 15.833379409,
    15.882758006, 15.938258414, 15.991902054, 16.045356894, 16.106466633,
    16.164227727, 16.220779299, 16.280192709, 16.333173018, 16.388827573,
    16.446902081, 16.503364759, 16.561872339, 16.618912203, 16.672553605,
    16.733541168, 16.787504836, 16.845175649, 16.906495412, 16.962409815,
    17.02389628, 17.078121974, 17.131804732, 17.190566478, 17.247522058,
    17.303279808, 17.356264183, 17.41694397, 17.480291157, 17.532844013,
    17.593195859, 17.653760281, 17.70745985, 17.76297287, 17.820953431,
    17.881277186, 17.946674701, 18.004377169, 18.06469833, 18.121995639,
    18.172840451, 18.236411729, 18.290531417, 18.348568169, 18.415394253,
    18.477686102, 18.539721805, 18.600419414, 18.662010917, 18.72757394,
    18.782296987, 18.842686061, 18.905403082, 18.97856359, 19.03715941,
    19.097098854, 19.162605704, 19.233789819, 19.30814108, 19.374235742,
    19.443868777, 19.51253113, 19.569157473, 19.639216503, 19.701676626,
    19.768399772, 19.830234003, 19.88925016, 19.951571823, 20.013798989,
    20.068233249, 20.128331443, 20.18606867, 20.255140598, 20.315737773,
    20.378135268, 20.450182369, 20.512643432, 20.576309952, 20.636682586,
    20.700241558, 20.75846262, 20.83050942, 20.900457817, 20.974882106,
    21.043983107, 21.119410686, 21.184047531, 21.24865178, 21.311978073,
    21.376039868, 21.454420766, 21.518505355, 21.587484445, 21.667522128,
    21.745187798, 21.811795858, 21.874716156, 21.931866281, 21.99884695,
    22.058709273, 22.126284313, 22.204105741, 22.266784574, 22.346906418,
    22.417370775, 22.493837653, 22.567513284, 22.624410352, 22.692659159,
    22.764590049, 22.833074459, 22.911132845, 22.989957666, 23.05569752,
    23.144314228, 23.216522006, 23.288392784, 23.369594627, 23.444583857,
    23.52681675, 23.587631319, 23.653194219, 23.734973995, 23.801097841,
    23.884243953, 23.960228782, 24.04626963, 24.118637908, 24.203486417,
    24.266963229, 24.3436674, 24.431085037, 24.504590379, 24.584128823,
    24.659874118, 24.730169447, 24.808478872, 24.893258874, 24.967208034,
    25.034847083, 25.106807758, 25.180460379, 25.254989514, 25.338176016,
    25.412309077, 25.479314445, 25.545270102, 25.626355523, 25.699051545,
    25.783172, 25.850817367, 25.926617522, 26.01115069, 26.104713704,
    26.183670074, 26.251953279, 26.270434251, 26.335101792, 26.414426189,
    26.4840813, 26.557051121, 26.620754386, 26.706483838, 26.7961142,
    26.859980836, 26.930232214, 27.003527583, 27.079970372, 27.143575919,
    27.238863898, 27.325145504, 27.409137649, 27.488697787, 27.587225983,
    27.650702359, 27.734933876, 27.827287772, 27.917506841, 27.985401173,
    28.050168707, 28.1170351, 28.197831346, 28.29219971, 28.360285523,
    28.448750307, 28.50844264, 28.569487245, 28.64150682, 28.738339009,
    28.836134989, 28.907258971, 28.978370957, 29.053181598, 29.142181168,
    29.244159508, 29.358124722, 29.444160923, 29.522786587, 29.622356858,
    29.711525936, 29.805643696, 29.901196465, 29.984323467, 30.082080128,
    30.141589931, 30.226820702, 30.332886053, 30.426340064, 30.525366339,
    30.604117914, 30.694364542, 30.75502441, 30.822756083, 30.903805249,
    30.974394139, 31.071296947, 31.141872347, 31.239979795, 31.31830467,
    31.415000168, 31.51458897, 31.60070835, 31.705531702, 31.802114948,
    31.885111835, 31.973019286, 32.079611786, 32.180116783, 32.266831118,
    32.346506654, 32.433487956, 32.510794328, 32.583070835, 32.66186898,
    32.76474811, 32.841674688, 32.947540911, 33.043801247, 33.134557086,
    33.202298943, 33.294066152, 33.399245851, 33.471695598, 33.550967609,
    33.642942406, 33.711564341, 33.803878449, 33.915664742, 34.038237794,
    34.117399315, 34.20305049, 34.300131372, 34.411720584, 34.497866655,
    34.575927078, 34.718556114, 34.814612629, 34.939670601, 35.01887465,
    35.121716343, 35.210082838, 35.316131351, 35.443828943, 35.503893216,
    35.597413682, 35.708495817, 35.805683747, 35.92966752, 36.032208999,
    36.112871172, 36.221446679, 36.306179814, 36.392085493, 36.455710302,
    36.550528386, 36.661331131, 36.774441435, 36.896481046, 36.995915162,
    37.114205277, 37.307936677, 37.372325907, 37.468541607, 37.59589792,
    37.692726939, 37.780751253, 37.852353514, 37.937066126, 38.037629249,
    38.122702748, 38.227123795, 38.315172541, 38.432670613, 38.504429711,
    38.637043623, 38.743988896, 38.846835626, 38.938346458, 39.04925242,
    39.12090335, 39.259339913, 39.43272336, 39.571169039, 39.688484404,
    39.811952402, 39.88747621, 39.978514296, 40.152381637, 40.35420855,
    40.47737408, 40.629437242, 40.71406671, 40.802638002, 40.900231613,
    41.013359021, 41.066760418, 41.221772922, 41.254800113, 41.336641019,
    41.413266897, 41.532232229, 41.659154225, 41.832218643, 41.989340964,
    42.156357208, 42.350528262, 42.557426724, 42.716287116, 42.842791642,
    42.962978638, 43.088424316, 43.170713335, 43.252338646, 43.392347329,
    43.623411776, 43.742352433, 43.839051544, 43.963934149, 44.070506128,
    44.210742642, 44.349104483, 44.51938569, 44.570202015, 44.7407275,
    44.823576049, 44.922529232, 45.058292382, 45.246381369, 45.366841376,
    45.499271901, 45.643248931, 45.74817614, 45.873836129, 46.054139531,
    46.18166215, 46.265343449, 46.352181395, 46.493917035, 46.545156851,
    46.663235357, 46.765975967, 46.906295072, 47.037135557, 47.221724244,
    47.351247662, 47.454489265, 47.740961432, 47.848232088, 47.878701173,
    47.963836445, 48.008463222, 48.090772223, 48.203768141, 48.359235538,
    48.494209921, 48.570769658, 48.709369295, 48.828085676, 48.947716414,
    49.020031566, 49.223130906, 49.326867849, 49.517625629, 49.605716455,
    49.797397589, 49.870684792, 50.020020269, 50.097160278, 50.244484583,
    50.353299699, 50.504834441, 50.627467307, 50.72745666, 50.920703428,
    51.00395578, 51.041832635, 51.134321084, 51.159901503, 51.216717344,
    51.39030106, 51.551228399, 51.662520596, 51.884516052, 51.92966626,
    52.148738385, 52.263186595, 52.336500125, 52.394326305, 52.488747648,
    52.532897541, 52.604258799, 52.691128236, 53.05332628, 53.14078459,
    53.225779987, 53.281008836, 53.414404677, 53.441322584, 53.681770521,
    53.694230978, 53.843162277, 53.963210871, 54.31860899, 54.473101598,
    54.618985084, 54.877298374, 54.889409298, 55.02445277, 55.122625352,
    55.24297253, 55.279995439, 55.457580906, 55.664956501, 55.784405531,
    55.939632838, 56.065305511, 56.251033535, 56.442475283, 56.78838921,
    56.889339906, 56.99019936, 57.229974196, 57.302226227, 57.463457122,
    57.848818639, 57.861634863, 58.247427407, 58.457916107, 58.627570224,
    58.699933805, 58.885187424, 59.141793196, 59.393869028, 59.436952332,
    59.456462918, 59.810941211, 59.818702583, 59.859622556, 59.979651423,
    60.091443656, 60.193932725, 60.312181312, 60.690692628, 60.862491777,
    61.016853065, 61.493252421, 61.608483439, 61.615160437, 61.85277233,
    61.861245408, 61.878085226, 61.953124292, 62.005839737, 62.115374149,
    62.124593348, 62.526706878, 62.576074343, 62.801782775, 62.8111522,
    63.695240466, 63.73759467, 63.764742436, 63.914770902, 64.002200845,
    64.17089253, 64.222479267, 64.278562144, 64.309101302, 64.46963618,
    64.622212181, 65.106949353, 65.151571662, 65.279203719, 65.312775935,
    65.474624735, 65.500287029, 65.548105079), "800" = c(1.8108e-05,
    1.832e-05, 1.8396e-05, 1.9144e-05, 1.9624e-05, 1.9675e-05,
    1.9719e-05, 1.9796e-05, 1.9973e-05, 2.016e-05, 2.0678e-05,
    2.0689e-05, 2.0749e-05, 2.0854e-05, 2.097e-05, 2.1135e-05,
    2.1507e-05, 2.2497e-05, 2.2568e-05, 2.2834e-05, 2.286e-05,
    2.2917e-05, 2.3389e-05, 2.363e-05, 2.4641e-05, 2.5145e-05,
    2.5413e-05, 2.6076e-05, 2.6558e-05, 2.6929e-05, 2.6958e-05,
    2.7089e-05, 2.7236e-05, 2.7562e-05, 2.7962e-05, 2.8094e-05,
    2.8294e-05, 2.8501e-05, 2.9344e-05, 2.9468e-05, 2.9696e-05,
    3.0048e-05, 3.0703e-05, 3.1276e-05, 3.1572e-05, 3.2661e-05,
    3.2903e-05, 3.3799e-05, 3.4179e-05, 3.4695e-05, 3.5333e-05,
    3.5783e-05, 3.5933e-05, 3.6643e-05, 3.672e-05, 3.721e-05,
    3.7707e-05, 3.7969e-05, 3.8077e-05, 3.8672e-05, 3.918e-05,
    3.955e-05, 4.0456e-05, 4.1666e-05, 4.2088e-05, 4.2217e-05,
    4.3073e-05, 4.3228e-05, 4.3629e-05, 4.3808e-05, 4.4718e-05,
    4.5046e-05, 4.5476e-05, 4.615e-05, 4.6254e-05, 4.6668e-05,
    4.6919e-05, 4.7581e-05, 4.797e-05, 4.8509e-05, 4.8735e-05,
    4.9749e-05, 5.033e-05, 5.0938e-05, 5.2181e-05, 5.2254e-05,
    5.2694e-05, 5.3504e-05, 5.382e-05, 5.4814e-05, 5.5509e-05,
    5.5957e-05, 5.6232e-05, 5.6933e-05, 5.7824e-05, 5.8666e-05,
    5.9708e-05, 5.9804e-05, 6.0264e-05, 6.1252e-05, 6.1559e-05,
    6.3101e-05, 6.3653e-05, 6.3879e-05, 6.4122e-05, 6.5234e-05,
    6.5777e-05, 6.6376e-05, 6.7282e-05, 6.7513e-05, 6.804e-05,
    6.894e-05, 6.9292e-05, 6.9698e-05, 7.0611e-05, 7.1508e-05,
    7.2262e-05, 7.3036e-05, 7.3685e-05, 7.4491e-05, 7.6459e-05,
    7.7285e-05, 7.8008e-05, 7.9224e-05, 8.0211e-05, 8.1334e-05,
    8.2146e-05, 8.2705e-05, 8.3716e-05, 8.4527e-05, 8.5316e-05,
    8.5801e-05, 8.6968e-05, 8.7896e-05, 8.8317e-05, 8.9019e-05,
    8.9775e-05, 9.0345e-05, 9.178e-05, 9.2925e-05, 9.4182e-05,
    9.563e-05, 9.6442e-05, 9.7195e-05, 9.8317e-05, 0.000100427,
    0.000101681, 0.000103117, 0.000104871, 0.000106612, 0.000107576,
    0.000108377, 0.000110008, 0.000111909, 0.000113122, 0.000113767,
    0.000115216, 0.000116688, 0.000119062, 0.000120503, 0.000121683,
    0.000123859, 0.000125188, 0.000127855, 0.000129711, 0.000130975,
    0.00013207, 0.000133419, 0.000134469, 0.000135597, 0.000136166,
    0.000138857, 0.000140575, 0.000142381, 0.000143987, 0.00014497,
    0.000147178, 0.000149426, 0.000152749, 0.000154177, 0.000155289,
    0.00015752, 0.000159316, 0.000160825, 0.000162776, 0.000163847,
    0.000165209, 0.000167432, 0.000169541, 0.000170984, 0.000173275,
    0.000174144, 0.000176129, 0.000177875, 0.000180616, 0.000182766,
    0.000184611, 0.000185813, 0.000187442, 0.000191185, 0.000192918,
    0.00019445, 0.000198585, 0.000200485, 0.000203394, 0.000205631,
    0.000207426, 0.000209267, 0.000210978, 0.000213445, 0.000216995,
    0.000219402, 0.000223152, 0.000225723, 0.000229215, 0.000232169,
    0.000233247, 0.000236712, 0.000239604, 0.000241182, 0.000242707,
    0.000247053, 0.000249365, 0.000252222, 0.000254428, 0.000259007,
    0.000261546, 0.000265616, 0.000268184, 0.000270317, 0.000272599,
    0.000275709, 0.000279524, 0.000281582, 0.000286284, 0.000290296,
    0.000294075, 0.000298009, 0.000301891, 0.000304032, 0.000306879,
    0.000309991, 0.000313723, 0.000317039, 0.00031954, 0.000323694,
    0.000326621, 0.000330602, 0.000334408, 0.000337638, 0.000341859,
    0.000345932, 0.000349398, 0.000353956, 0.000358282, 0.000361813,
    0.00036561, 0.000369921, 0.000374151, 0.000377868, 0.000381661,
    0.000384815, 0.000389587, 0.000394591, 0.000399601, 0.000404665,
    0.000406739, 0.000414729, 0.000419794, 0.000424569, 0.00042887,
    0.000434022, 0.000438365, 0.000442833, 0.000447544, 0.000451709,
    0.000457116, 0.000464003, 0.000472329, 0.000477253, 0.000483172,
    0.000487043, 0.000492364, 0.000496786, 0.000500445, 0.000507175,
    0.000513619, 0.000519309, 0.000526175, 0.000532921, 0.00054041,
    0.000546011, 0.00055221, 0.000560436, 0.000566658, 0.000572262,
    0.000579151, 0.000589154, 0.000594311, 0.000602149, 0.00060875,
    0.000613302, 0.000621274, 0.000629165, 0.000636811, 0.000645304,
    0.000655765, 0.000664242, 0.000671972, 0.00068389, 0.000692002,
    0.000701345, 0.000708359, 0.000718033, 0.000724585, 0.000732792,
    0.000744117, 0.000753018, 0.000759807, 0.00076674, 0.000774994,
    0.000784532, 0.00079085, 0.000800376, 0.00081235, 0.000820168,
    0.000829896, 0.000840501, 0.000848969, 0.000860798, 0.000872561,
    0.00088513, 0.000894354, 0.000904182, 0.000913724, 0.000925379,
    0.000935995, 0.000949317, 0.000961676, 0.000970516, 0.000977623,
    0.000979744, 0.000989724, 0.001001614, 0.001014322, 0.001026698,
    0.001040247, 0.001052061, 0.001065608, 0.001076677, 0.001088945,
    0.001099753, 0.00111236, 0.00112602, 0.001139526, 0.001151324,
    0.001165954, 0.001179366, 0.001193053, 0.001207212, 0.001220908,
    0.001233998, 0.001247813, 0.001259462, 0.001274147, 0.001291047,
    0.001305615, 0.001326261, 0.001341772, 0.001359816, 0.001376192,
    0.001393094, 0.001408549, 0.001425503, 0.001440151, 0.001457609,
    0.001475398, 0.001492933, 0.001506534, 0.001519532, 0.001538918,
    0.001559533, 0.001579349, 0.001597292, 0.001618932, 0.001637961,
    0.001656966, 0.001676365, 0.001702013, 0.001722216, 0.001742225,
    0.001762931, 0.001784901, 0.001804889, 0.001827589, 0.001852849,
    0.001876463, 0.001898146, 0.001918747, 0.001935088, 0.001958096,
    0.001980329, 0.00200222, 0.00202425, 0.00204841, 0.002068816,
    0.002090524, 0.002115207, 0.002141229, 0.002169187, 0.002191191,
    0.002214917, 0.002245719, 0.002274631, 0.002295635, 0.002325634,
    0.002355626, 0.002383255, 0.002410676, 0.00243704, 0.002467735,
    0.002491641, 0.002517627, 0.0025505, 0.002576188, 0.002608096,
    0.002642165, 0.002674015, 0.002702292, 0.002732567, 0.002764067,
    0.002795308, 0.002829219, 0.002863027, 0.002895166, 0.002929261,
    0.002967645, 0.003002384, 0.003036496, 0.003073059, 0.003107634,
    0.003144398, 0.003178909, 0.003208575, 0.003242681, 0.003277743,
    0.003318352, 0.00336073, 0.003402726, 0.003444134, 0.003483601,
    0.003532343, 0.003570043, 0.003608339, 0.003647884, 0.003691381,
    0.003736881, 0.003778408, 0.003818663, 0.003859688, 0.003912182,
    0.00395865, 0.004000996, 0.004045923, 0.004092064, 0.004137124,
    0.004182144, 0.004228332, 0.004277583, 0.004331734, 0.004379976,
    0.004430171, 0.004479488, 0.004535438, 0.004584301, 0.004636742,
    0.004691653, 0.004746295, 0.004801342, 0.004856894, 0.004912587,
    0.004963274, 0.005022943, 0.005073144, 0.005136062, 0.005191773,
    0.00525092, 0.005311847, 0.005370738, 0.005434212, 0.00549046,
    0.005548888, 0.005607325, 0.005672382, 0.005742418, 0.005803947,
    0.005871165, 0.005935847, 0.006000986, 0.006067272, 0.0061338,
    0.006207709, 0.006279292, 0.006351454, 0.006422648, 0.006501579,
    0.006582876, 0.006660207, 0.00673653, 0.006821572, 0.006888887,
    0.0069739, 0.007056781, 0.007144438, 0.007223588, 0.00730703,
    0.007389568, 0.007474561, 0.007561729, 0.007650837, 0.007741739,
    0.007828452, 0.007918015, 0.008005355, 0.008099896, 0.008192255,
    0.008281328, 0.008372608, 0.00846794, 0.008558765, 0.008656178,
    0.008757307, 0.008858086, 0.008953954, 0.009052753, 0.009161917,
    0.009263644, 0.009373948, 0.009474421, 0.009591261, 0.009702701,
    0.009792787, 0.009818312, 0.009930949, 0.010042691, 0.010164239,
    0.010280746, 0.010395481, 0.010525292, 0.010649418, 0.010779804,
    0.010908005, 0.011043979, 0.011169642, 0.011300875, 0.011428678,
    0.01156349, 0.011694678, 0.011832287, 0.011969795, 0.012110226,
    0.012251462, 0.012387477, 0.012518108, 0.012671865, 0.012826036,
    0.012973635, 0.013130342, 0.013283597, 0.013440022, 0.01360439,
    0.013760411, 0.013923431, 0.014081029, 0.014245442, 0.014416161,
    0.01457826, 0.014754165, 0.014924356, 0.015088553, 0.015271326,
    0.015452731, 0.015629196, 0.015814648, 0.016001195, 0.016200431,
    0.01638547, 0.016575873, 0.016778334, 0.016954494, 0.017142099,
    0.01733968, 0.017549735, 0.017751063, 0.017973412, 0.018176172,
    0.018393707, 0.018602969, 0.018823097, 0.019048975, 0.019273237,
    0.019510431, 0.019745228, 0.01998926, 0.020228996, 0.0204622,
    0.020694007, 0.020946242, 0.021186846, 0.021431987, 0.021692925,
    0.021943025, 0.022209498, 0.022457031, 0.022717787, 0.022990181,
    0.023243914, 0.023509625, 0.023778184, 0.02405661, 0.024337971,
    0.02462817, 0.024925066, 0.025226504, 0.025532113, 0.025815674,
    0.026102691, 0.026395319, 0.026703785, 0.027024305, 0.027345658,
    0.02766503, 0.027988823, 0.028327582, 0.028648035, 0.02898199,
    0.029332886, 0.02966028, 0.030008881, 0.030353873, 0.030729176,
    0.031109381, 0.031456613, 0.031824369, 0.032197953, 0.032573675,
    0.032957024, 0.033346346, 0.033736774, 0.034138531, 0.034542956,
    0.03495637, 0.035343223, 0.035765313, 0.03617857, 0.03660814,
    0.03705231, 0.037493673, 0.037920611, 0.038357022, 0.038806771,
    0.039273388, 0.039734846, 0.040201606, 0.040669582, 0.041144628,
    0.041628665, 0.042117831, 0.042620894, 0.043107105, 0.043611714,
    0.044127726, 0.044647068, 0.045159922, 0.045697025, 0.046232414,
    0.046787859, 0.047336319, 0.047910045, 0.048472522, 0.049045535,
    0.04961829, 0.050188653, 0.050781205, 0.051371682, 0.051958667,
    0.052560485, 0.053157208, 0.053784047, 0.054401754, 0.055039863,
    0.055683523, 0.056340133, 0.056988735, 0.057674032, 0.058335417,
    0.059035447, 0.059731487, 0.060437231, 0.061176011, 0.06191328,
    0.062628485, 0.063356821, 0.064118005, 0.064883587, 0.065643543,
    0.066408781, 0.067174457, 0.067991015, 0.068786106, 0.069566743,
    0.070378622, 0.071204927, 0.072033051, 0.072875721, 0.07372602,
    0.074583551, 0.075461948, 0.076361995, 0.077274478, 0.078191769,
    0.079114021, 0.080062624, 0.081009828, 0.081974995, 0.082947265,
    0.083918924, 0.084915699, 0.085918491, 0.086928743, 0.087943248,
    0.088993768, 0.090038415, 0.091113914, 0.092218, 0.093307093,
    0.094422688, 0.095551754, 0.096649326, 0.097782304, 0.098937227,
    0.100115653, 0.101048376, 0.101307812, 0.102494369, 0.1037042,
    0.104923486, 0.106172414, 0.107427997, 0.108698138, 0.110001988,
    0.111340683, 0.112658782, 0.113995053, 0.115336216, 0.116694659,
    0.118118401, 0.119523936, 0.120941027, 0.122379923, 0.123819951,
    0.125303213, 0.126821494, 0.128347878, 0.129885104, 0.131440671,
    0.132978582, 0.134599778, 0.136215115, 0.137794612, 0.139436715,
    0.141105203, 0.142804532, 0.144526466, 0.146214136, 0.147958903,
    0.149741984, 0.151530833, 0.15336566, 0.155163613, 0.157016521,
    0.158868552, 0.160792145, 0.162714485, 0.164671886, 0.166646145,
    0.168630729, 0.170703353, 0.172792116, 0.174873016, 0.176963984,
    0.179110571, 0.181315566, 0.183515092, 0.185726194, 0.188021871,
    0.190322232, 0.192640421, 0.19495786, 0.197309518, 0.199713576,
    0.202114335, 0.204579527, 0.207080497, 0.209612438, 0.212155977,
    0.214732538, 0.217358358, 0.220040548, 0.222716697, 0.225434366,
    0.228170479, 0.231004873, 0.233821809, 0.236720136, 0.239585921,
    0.242494919, 0.245462352, 0.248480352, 0.251547379, 0.254610327,
    0.257741324, 0.260949404, 0.264214352, 0.267510763, 0.270806504,
    0.274152609, 0.277573353, 0.280999371, 0.284476376, 0.287994992,
    0.291562999, 0.295135517, 0.298820319, 0.302558213, 0.306363944,
    0.310169944, 0.314092376, 0.3180282, 0.318890956, 0.32200888,
    0.326014434, 0.330087841, 0.334235764, 0.338459875, 0.342727294,
    0.347067653, 0.351428731, 0.355880393, 0.360341673, 0.364869518,
    0.369500262, 0.37416318, 0.378884155, 0.383725073, 0.388595963,
    0.393526562, 0.398547983, 0.403568214, 0.408710896, 0.413957026,
    0.41922217, 0.424651641, 0.430072781, 0.435600229, 0.436757538,
    0.441213263, 0.446831337, 0.452617616, 0.458408819, 0.464283539,
    0.470285112, 0.47640661, 0.482600709, 0.488787522, 0.49518907,
    0.501746378, 0.508290286, 0.514950107, 0.521684445, 0.528476558,
    0.535428602, 0.542469421, 0.549705308, 0.556891228, 0.564324529,
    0.571779583, 0.579467637, 0.587176989, 0.595026504, 0.603004876,
    0.611128117, 0.6193244, 0.627717641, 0.636232539, 0.644906586,
    0.653706549, 0.662528296, 0.671576809, 0.680701405, 0.690086357,
    0.694004473, 0.69944969, 0.709065118, 0.718900923, 0.728758413,
    0.738838957, 0.749184966, 0.759532517, 0.770134785, 0.780902552,
    0.791884247, 0.803124252, 0.814489642, 0.826041483, 0.837762763,
    0.84976287, 0.861946318, 0.874396857, 0.887080374, 0.899935036,
    0.913093745, 0.926455431, 0.939986234, 0.953786185, 0.967854623,
    0.982139981, 0.988193427, 0.996825298, 1.011711845, 1.02702396,
    1.042537755, 1.058201888, 1.074338919, 1.090819039, 1.107663844,
    1.124848338, 1.14226482, 1.160044683, 1.178094985, 1.196547831,
    1.215469747, 1.234757703, 1.254634033, 1.274891981, 1.295589479,
    1.316700371, 1.333782333, 1.372994957, 1.394821015, 1.416636762,
    1.438428962, 1.460200089, 1.481965746, 1.50369274, 1.525399384,
    1.547155643, 1.568714789, 1.590392348, 1.612251521, 1.634063273,
    1.655584121, 1.676895085, 1.698664999, 1.720202876, 1.741906686,
    1.754665772, 1.763545591, 1.785236764, 1.806954113, 1.828602279,
    1.850329003, 1.871837154, 1.893563791, 1.915127728, 1.936964918,
    1.958612247, 1.980136181, 2.001858447, 2.023513867, 2.045235974,
    2.06707015, 2.088687657, 2.110313033, 2.131991984, 2.153524848,
    2.175215898, 2.197060081, 2.218741616, 2.240308538, 2.261896548,
    2.283666524, 2.296175731, 2.30535786, 2.326939706, 2.348653975,
    2.370430815, 2.392040905, 2.413767585, 2.435410771, 2.457044695,
    2.47867437, 2.500372555, 2.52227528, 2.543978378, 2.565502109,
    2.587474726, 2.609180821, 2.630815164, 2.652680153, 2.674499417,
    2.696595938, 2.718683058, 2.740331503, 2.762086462, 2.783871637,
    2.805822846, 2.827518584, 2.849265991, 2.871023123, 2.893029713,
    2.915057368, 2.937215711, 2.959295763, 2.981130328, 3.003084664,
    3.024909345, 3.046897759, 3.064464725, 3.068971026, 3.090868132,
    3.112926038, 3.134989693, 3.156965588, 3.178994416, 3.201099009,
    3.223279233, 3.245555342, 3.267541281, 3.289692158, 3.311788871,
    3.334260136, 3.35611272, 3.378358191, 3.400863368, 3.423584979,
    3.445840127, 3.468116323, 3.490487348, 3.512890167, 3.535305032,
    3.557583522, 3.579708416, 3.602167999, 3.619825877, 3.624819086,
    3.647336494, 3.669680173, 3.692299416, 3.715068244, 3.737650974,
    3.760296772, 3.782868989, 3.805465848, 3.828207357, 3.850773217,
    3.873221273, 3.895823367, 3.918678669, 3.941419403, 3.964431871,
    3.987249539, 4.010359009, 4.03298838, 4.056081296, 4.079003624,
    4.102179579, 4.125303019, 4.148478035, 4.17157967, 4.194600932,
    4.218092082, 4.24074436, 4.264094204, 4.28717423, 4.310509599,
    4.33386984, 4.356967534, 4.380425162, 4.404044097, 4.427364438,
    4.450800779, 4.474120262, 4.49730338, 4.521150499, 4.54466589,
    4.568351003, 4.592155495, 4.615847797, 4.639989116, 4.663767199,
    4.687144315, 4.711414035, 4.735128193, 4.759080324, 4.78286261,
    4.807224263, 4.831742455, 4.85569812, 4.879627193, 4.90334507,
    4.927441509, 4.951769228, 4.975977835, 4.999644346, 5.024419232,
    5.048871616, 5.073091037, 5.09757208, 5.121709321, 5.14614229,
    5.170942596, 5.195478018, 5.219581501, 5.244322736, 5.269353497,
    5.294191165, 5.319461167, 5.344489715, 5.369232467, 5.394327734,
    5.419256991, 5.444394473, 5.469689482, 5.494506301, 5.519489544,
    5.544702065, 5.570352523, 5.595817186, 5.62140891, 5.646649051,
    5.671793134, 5.697329263, 5.723386411, 5.749350788, 5.774754696,
    5.800297354, 5.826521679, 5.852360489, 5.878674019, 5.905077496,
    5.910348911, 5.930509312, 5.957051415, 5.98333971, 6.009551309,
    6.035893154, 6.062489134, 6.089126735, 6.114872779, 6.141972301,
    6.168937708, 6.195663328, 6.22220395, 6.249469512, 6.276307237,
    6.302700829, 6.330289012, 6.357714325, 6.385093266, 6.412467752,
    6.439947853, 6.467201243, 6.494652689, 6.521721953, 6.54907584,
    6.576828228, 6.604711725, 6.632759785, 6.660917527, 6.689375631,
    6.717816486, 6.74530722, 6.772730996, 6.801666723, 6.829443032,
    6.856946695, 6.885360548, 6.913156828, 6.941017305, 6.969340402,
    6.997456742, 7.025778765, 7.054696768, 7.083324019, 7.112767305,
    7.141062553, 7.170902413, 7.199866438, 7.229471894, 7.258822494,
    7.288514211, 7.318767771, 7.348609663, 7.378122354, 7.408658191,
    7.437776549, 7.467831767, 7.498244227, 7.527977713, 7.558506706,
    7.588436157, 7.619641555, 7.649396855, 7.679964952, 7.710925124,
    7.741734279, 7.7725748, 7.803554081, 7.834514967, 7.864899732,
    7.89616467, 7.926294361, 7.957419801, 7.98859122, 8.020154464,
    8.052567831, 8.084358009, 8.115935483, 8.14782649, 8.180149037,
    8.212061158, 8.244184485, 8.276755098, 8.3099166, 8.341836219,
    8.37438684, 8.407256912, 8.43912728, 8.472042891, 8.504287242,
    8.537592244, 8.57004224, 8.603559413, 8.636653255, 8.669472022,
    8.702802377, 8.734485979, 8.768579129, 8.802501687, 8.837033137,
    8.870907869, 8.905364036, 8.938025232, 8.972116286, 9.00591527,
    9.038633911, 9.073549689, 9.106730349, 9.141133812, 9.174955509,
    9.207592591, 9.2428251, 9.276629991, 9.313259575, 9.34823584,
    9.384128109, 9.419099334, 9.45444328, 9.489584143, 9.526758802,
    9.562030677, 9.597933929, 9.63373006, 9.669388973, 9.70711455,
    9.741911104, 9.778164019, 9.815022592, 9.850666055, 9.88870009,
    9.926389778, 9.963429807, 10.001473054, 10.038160685, 10.074074842,
    10.112379939, 10.149215506, 10.187473751, 10.225279753, 10.261614537,
    10.299430913, 10.335877306, 10.373854932, 10.411977733, 10.449300796,
    10.488413042, 10.527804705, 10.567651451, 10.605574309, 10.643183162,
    10.681341446, 10.718378229, 10.756382355, 10.792354992, 10.831035834,
    10.868930443, 10.907964126, 10.947173318, 10.985660616, 11.024217434,
    11.063032525, 11.10229286, 11.140853011, 11.178351131, 11.22024378,
    11.258902695, 11.298264951, 11.337875901, 11.37661399, 11.416590545,
    11.454861221, 11.496000824, 11.537514197, 11.581211312, 11.623268996,
    11.663948689, 11.706096768, 11.748514494, 11.792010995, 11.834943807,
    11.876059294, 11.916506532, 11.955151802, 11.997388925, 12.041376746,
    12.084687764, 12.124194152, 12.16641462, 12.208229005, 12.250020452,
    12.29059982, 12.331288497, 12.375993038, 12.421178336, 12.462423531,
    12.505840785, 12.545602273, 12.589586276, 12.633449543, 12.676704991,
    12.717104919, 12.725854311, 12.75890621, 12.803185347, 12.846956477,
    12.887408763, 12.929812105, 12.971812837, 13.015551264, 13.060954452,
    13.106038922, 13.151885831, 13.193875878, 13.237134797, 13.278221789,
    13.320536591, 13.366575277, 13.410801623, 13.45496763, 13.496338774,
    13.542961165, 13.58458004, 13.627782799, 13.674459065, 13.716921076,
    13.759896874, 13.804015471, 13.851231021, 13.896276299, 13.940961286,
    13.986928241, 14.035430493, 14.081194249, 14.129254566, 14.176111684,
    14.221503808, 14.267092254, 14.314898236, 14.362639031, 14.409911786,
    14.452472161, 14.497740436, 14.542122294, 14.594361677, 14.640527642,
    14.691247907, 14.736736195, 14.782023045, 14.832014555, 14.879037821,
    14.927184602, 14.977321402, 15.024492177, 15.069597555, 15.119366026,
    15.161165553, 15.211341219, 15.259074064, 15.302391861, 15.352365361,
    15.400399388, 15.447420586, 15.492569492, 15.539097329, 15.58838544,
    15.632300432, 15.679648173, 15.728780683, 15.774769702, 15.824090939,
    15.870693864, 15.924530329, 15.967222928, 16.009774691, 16.056217705,
    16.102911108, 16.155005563, 16.203652214, 16.253634626, 16.295566466,
    16.343661707, 16.393340959, 16.44511896, 16.491442576, 16.546682521,
    16.591800865, 16.641616897, 16.692129676, 16.739689538, 16.788103433,
    16.834781376, 16.887591915, 16.931813383, 16.980173526, 17.034829267,
    17.083965351, 17.135272684, 17.188500071, 17.243255006, 17.29417484,
    17.346945416, 17.400098777, 17.449644279, 17.501720105, 17.551649032,
    17.600055747, 17.647426189, 17.69543555, 17.749869156, 17.795323133,
    17.847595062, 17.898790653, 17.956214286, 18.01115285, 18.064615095,
    18.116348827, 18.16394281, 18.217933226, 18.263657535, 18.315622825,
    18.370950686, 18.422942096, 18.472909716, 18.528761796, 18.577302679,
    18.637939458, 18.695552125, 18.759171578, 18.81774598, 18.876230283,
    18.937783236, 18.993063313, 19.040266823, 19.092256301, 19.149795034,
    19.209554731, 19.264430527, 19.313724315, 19.357701428, 19.406127567,
    19.463191644, 19.52133347, 19.579825648, 19.640923149, 19.697188355,
    19.761363227, 19.822823887, 19.876953892, 19.930755551, 19.987698781,
    20.037497436, 20.094849913, 20.14896556, 20.202652529, 20.264412313,
    20.316432528, 20.372814708, 20.429050237, 20.47681634, 20.531778211,
    20.597783217, 20.666471564, 20.728648807, 20.790179206, 20.842658594,
    20.898482727, 20.962772654, 21.020528264, 21.076664713, 21.128133629,
    21.185860871, 21.249637596, 21.306166413, 21.362811664, 21.414222384,
    21.467465404, 21.519393215, 21.579028076, 21.639013578, 21.695241609,
    21.767743848, 21.827242271, 21.886920308, 21.95096918, 22.00969199,
    22.062642141, 22.126136839, 22.19103889, 22.249706856, 22.295384083,
    22.364282587, 22.439398618, 22.493700543, 22.560468994, 22.62263336,
    22.694660699, 22.7420245, 22.80334857, 22.855124314, 22.910316784,
    22.966593882, 23.033088862, 23.0452269, 23.091748471, 23.166073271,
    23.232216126, 23.294416456, 23.345878605, 23.41367427, 23.468059293,
    23.530098872, 23.595487177, 23.66347892, 23.74853224, 23.819304687,
    23.873000301, 23.938952286, 24.003030372, 24.072414024, 24.130564893,
    24.187879658, 24.25154512, 24.320598799, 24.382505003, 24.442988977,
    24.505268597, 24.567868164, 24.631170475, 24.699895516, 24.78510592,
    24.86591731, 24.921784862, 24.98374251, 25.051813078, 25.134747962,
    25.208454897, 25.275166473, 25.344752598, 25.402011779, 25.477893291,
    25.554215985, 25.620394572, 25.698030184, 25.761623634, 25.836094536,
    25.897512447, 25.980537471, 26.051301578, 26.131137646, 26.202714189,
    26.249335805, 26.312246527, 26.397654798, 26.455109912, 26.534302697,
    26.602916479, 26.669901001, 26.736490368, 26.806411053, 26.850931162,
    26.910845544, 26.972826549, 27.051265075, 27.11172063, 27.181539607,
    27.27364599, 27.343412698, 27.40546154, 27.489530994, 27.56861724,
    27.640403329, 27.730895, 27.797928966, 27.883414912, 27.94506799,
    28.012349941, 28.09960182, 28.155387677, 28.23331797, 28.296527751,
    28.352296255, 28.456367063, 28.527384844, 28.597007976, 28.675998508,
    28.743987091, 28.80178406, 28.876418169, 28.950411292, 28.99403289,
    29.051851402, 29.124272964, 29.245633421, 29.349149693, 29.444303806,
    29.511113935, 29.611513901, 29.67755941, 29.746734713, 29.804303738,
    29.874004298, 29.966400304, 30.059481443, 30.140053858, 30.184113425,
    30.258498249, 30.340569874, 30.420237966, 30.480111712, 30.549361206,
    30.618338708, 30.689686869, 30.762670329, 30.841300845, 30.908766087,
    30.984724386, 31.052509444, 31.114375697, 31.219550543, 31.288565322,
    31.346592583, 31.442325491, 31.482920075, 31.553446233, 31.616470247,
    31.712488548, 31.791879367, 31.849514313, 31.97490921, 32.10155949,
    32.154727796, 32.231557119, 32.383551428, 32.471192428, 32.586380662,
    32.642090416, 32.705419645, 32.813395036, 32.872394207, 32.938501103,
    33.042691646, 33.163711299, 33.24697904, 33.303407551, 33.474981596,
    33.571890086, 33.650330948, 33.716918506, 33.768241166, 33.849034892,
    33.957902318, 34.025784107, 34.070600872, 34.183715771, 34.264850362,
    34.341165386, 34.434633124, 34.582848795, 34.657164257, 34.737874012,
    34.827295923, 34.904711207, 34.952295742, 35.030753716, 35.146030754,
    35.265470551, 35.349261821, 35.436586401, 35.564698067, 35.667962894,
    35.726979466, 35.823395676, 35.92861265, 36.030279817, 36.111076989,
    36.170226009, 36.199100707, 36.268422088, 36.377488674, 36.563444028,
    36.742560773, 36.793771459, 36.850738069, 36.981404542, 37.046067509,
    37.092048614, 37.133464965, 37.18904043, 37.309450236, 37.356742994,
    37.472487112, 37.506258231, 37.632882335, 37.70357095, 37.736828432,
    37.870604853, 37.949787892, 38.06584847, 38.154829758, 38.32445379,
    38.374990472, 38.454050623, 38.598467851, 38.668630389, 38.7334744,
    38.870016804, 38.916918721, 38.999574278, 39.079043013, 39.197953322,
    39.235828801, 39.396273357, 39.465553658, 39.534940076, 39.660850353,
    39.782139399, 39.989529223, 40.059966861, 40.107695657, 40.160339447,
    40.276015789, 40.385203358, 40.44822256, 40.595801493, 40.665969696,
    40.771707141, 40.928050141, 41.017773512, 41.185954769, 41.42925149,
    41.574446943, 41.731435934, 41.899682242, 41.956198236, 42.074206027,
    42.171638415, 42.256334602, 42.292547066, 42.521986971, 42.60767022,
    42.664273845, 42.746931481, 42.815791799, 42.891293505, 42.950703299,
    43.207802475, 43.214876245, 43.300852981, 43.459519246, 43.502907387,
    43.781792882, 43.822128321, 43.942330435, 44.059662646, 44.133502799,
    44.149116692, 44.301077012, 44.365302902, 44.386362378, 44.489949113,
    44.55944061, 44.713072737, 44.878588431, 44.92563949, 45.019942391,
    45.108952638, 45.205374922, 45.293909669, 45.329287779, 45.389673565,
    45.535430403, 45.627405108, 45.672522552, 45.739260765, 45.795798236,
    46.001535182, 46.010393616, 46.121151083, 46.190330809, 46.341347053,
    46.44261159, 46.656854104, 46.744221017, 46.868678435, 46.915702756,
    47.018661733, 47.220469877, 47.319856561, 47.502775266, 47.643372978,
    47.767388086, 47.922447917, 47.952912452, 47.979743308, 48.005041858,
    48.060764028, 48.325449277, 48.455066789, 48.473588376, 48.55238717,
    48.619539962, 48.833069078, 49.049552803, 49.070927322, 49.101367705,
    49.172867767, 49.248725595, 49.371124641, 49.426604591, 49.548992728,
    49.559695873, 49.778622107, 49.906652592, 49.944490387, 50.035364068,
    50.097438934, 50.153143545, 50.38655689, 50.410027116, 50.673020494,
    50.67533773, 50.900979455, 50.905508289, 51.033639904, 51.119953645,
    51.13204761, 51.150539009, 51.154429444, 51.315629172, 51.437513261,
    51.471856818, 51.479165645, 51.75249531, 51.766494222, 51.770644281,
    51.872552385, 51.876477651, 51.90135813, 51.908601717, 52.103223498,
    52.345996749, 52.462643816, 52.582863613), "1000" = c(1.6312e-05,
    1.644e-05, 1.647e-05, 1.6674e-05, 1.6743e-05, 1.6977e-05,
    1.7228e-05, 1.7329e-05, 1.7638e-05, 1.7974e-05, 1.8197e-05,
    1.8746e-05, 1.8783e-05, 1.905e-05, 1.927e-05, 1.9461e-05,
    1.9515e-05, 1.9863e-05, 2.0116e-05, 2.0179e-05, 2.0503e-05,
    2.0533e-05, 2.0772e-05, 2.0887e-05, 2.1047e-05, 2.1216e-05,
    2.1416e-05, 2.1418e-05, 2.1862e-05, 2.1917e-05, 2.2223e-05,
    2.2508e-05, 2.2846e-05, 2.3222e-05, 2.3339e-05, 2.3582e-05,
    2.3817e-05, 2.4236e-05, 2.4525e-05, 2.5227e-05, 2.5525e-05,
    2.6144e-05, 2.6169e-05, 2.6234e-05, 2.6955e-05, 2.7632e-05,
    2.8096e-05, 2.8169e-05, 2.8283e-05, 2.8429e-05, 2.8713e-05,
    2.889e-05, 2.922e-05, 2.9624e-05, 2.9838e-05, 3.0416e-05,
    3.0454e-05, 3.0558e-05, 3.088e-05, 3.1198e-05, 3.2373e-05,
    3.2802e-05, 3.2976e-05, 3.3957e-05, 3.4327e-05, 3.4469e-05,
    3.5139e-05, 3.6114e-05, 3.6902e-05, 3.7256e-05, 3.7708e-05,
    3.8891e-05, 3.9062e-05, 3.9165e-05, 3.9583e-05, 4.0109e-05,
    4.1788e-05, 4.185e-05, 4.2854e-05, 4.3933e-05, 4.4399e-05,
    4.5153e-05, 4.567e-05, 4.6811e-05, 4.7028e-05, 4.7715e-05,
    4.8564e-05, 4.9524e-05, 5.0345e-05, 5.053e-05, 5.126e-05,
    5.1717e-05, 5.2136e-05, 5.2523e-05, 5.2825e-05, 5.3416e-05,
    5.3669e-05, 5.4506e-05, 5.6348e-05, 5.7044e-05, 5.7398e-05,
    5.7864e-05, 5.9048e-05, 6.0472e-05, 6.2245e-05, 6.3413e-05,
    6.4438e-05, 6.4985e-05, 6.6368e-05, 6.7011e-05, 6.7513e-05,
    6.7929e-05, 6.8446e-05, 6.8835e-05, 6.9521e-05, 7.0948e-05,
    7.1758e-05, 7.2395e-05, 7.2739e-05, 7.4286e-05, 7.4579e-05,
    7.6272e-05, 7.7155e-05, 7.7744e-05, 7.8642e-05, 7.9375e-05,
    8.0247e-05, 8.0743e-05, 8.2005e-05, 8.2848e-05, 8.5527e-05,
    8.6065e-05, 8.6858e-05, 8.8609e-05, 9.0163e-05, 9.0922e-05,
    9.2047e-05, 9.3653e-05, 9.4397e-05, 9.5128e-05, 9.6806e-05,
    9.7936e-05, 9.9832e-05, 0.000101571, 0.000102541, 0.000104264,
    0.000104886, 0.000106043, 0.000107462, 0.000109222, 0.000110983,
    0.000112697, 0.000114182, 0.000115536, 0.000116601, 0.000117609,
    0.000118624, 0.000120398, 0.000121823, 0.000122793, 0.000124321,
    0.000126299, 0.00012731, 0.000128523, 0.000132089, 0.000133564,
    0.000135094, 0.000136167, 0.000138445, 0.000139985, 0.000142129,
    0.000142907, 0.000144816, 0.00014634, 0.000147675, 0.000149561,
    0.000150785, 0.00015186, 0.000154185, 0.000156315, 0.000158553,
    0.000160683, 0.000161923, 0.000164841, 0.000166549, 0.000168104,
    0.000170054, 0.000171438, 0.000172805, 0.000174759, 0.000176986,
    0.000178736, 0.000179794, 0.000181653, 0.000184086, 0.00018668,
    0.000189816, 0.000191988, 0.000196232, 0.000198917, 0.000201589,
    0.000203, 0.000205967, 0.000207175, 0.000210615, 0.000213466,
    0.00021681, 0.000218531, 0.000221539, 0.000223381, 0.000225831,
    0.000227881, 0.000229873, 0.000231014, 0.000232763, 0.000235779,
    0.000239526, 0.000243188, 0.000245393, 0.000249045, 0.000250883,
    0.000254105, 0.000256315, 0.000258669, 0.000260829, 0.000263684,
    0.000266301, 0.000268689, 0.000272292, 0.000274302, 0.000278139,
    0.00028187, 0.000285539, 0.000288423, 0.000291796, 0.000294797,
    0.000297778, 0.000301864, 0.000305821, 0.000309287, 0.000314164,
    0.000319084, 0.000321902, 0.000325967, 0.000329153, 0.000332331,
    0.000338077, 0.000342615, 0.000346067, 0.000349778, 0.000353261,
    0.000357927, 0.000362417, 0.000367195, 0.000372311, 0.00037543,
    0.000379832, 0.0003845, 0.000388531, 0.000392055, 0.000396226,
    0.000402597, 0.000407659, 0.000411709, 0.00041536, 0.000420398,
    0.000423654, 0.00042964, 0.000433551, 0.000438541, 0.000444973,
    0.000450061, 0.0004554, 0.000460996, 0.000466087, 0.000472405,
    0.000476519, 0.000480512, 0.000486301, 0.000491034, 0.000495934,
    0.000500231, 0.00050675, 0.000512878, 0.000519226, 0.000523323,
    0.000530297, 0.000535991, 0.000542461, 0.000550206, 0.000557399,
    0.000564143, 0.00057216, 0.000578983, 0.000586364, 0.000591898,
    0.000601467, 0.000606201, 0.00061237, 0.000619963, 0.000625941,
    0.000631178, 0.000638871, 0.000648336, 0.000655038, 0.000661822,
    0.000668375, 0.00067671, 0.00068532, 0.000695343, 0.000704513,
    0.000711184, 0.000720437, 0.000727704, 0.000735472, 0.000741399,
    0.00075041, 0.000759214, 0.000769272, 0.000776441, 0.00078614,
    0.000793975, 0.0008037, 0.000811974, 0.000821817, 0.000832984,
    0.000843377, 0.000854464, 0.000864285, 0.000875329, 0.000886851,
    0.000896071, 0.00090722, 0.000918224, 0.000927451, 0.000939116,
    0.000951215, 0.00096511, 0.00097602, 0.000987317, 0.000992705,
    0.000996286, 0.001008338, 0.001021191, 0.001034468, 0.001049273,
    0.00106206, 0.001074138, 0.001085459, 0.001098149, 0.001107303,
    0.00112047, 0.001132658, 0.001147899, 0.00116219, 0.001176528,
    0.001190353, 0.001203562, 0.00121733, 0.001229309, 0.001242388,
    0.001258713, 0.001272572, 0.001287334, 0.001300453, 0.001318328,
    0.00133364, 0.00134739, 0.001362986, 0.001378303, 0.001394097,
    0.001409779, 0.001424474, 0.001436654, 0.001456329, 0.001472804,
    0.001492685, 0.00150902, 0.001528264, 0.001544577, 0.001561531,
    0.001579571, 0.001597537, 0.001615152, 0.00163201, 0.00165072,
    0.001670949, 0.001690924, 0.001711522, 0.001733907, 0.001757841,
    0.001776273, 0.001796075, 0.001814397, 0.00183428, 0.001857965,
    0.001880405, 0.00190128, 0.001921294, 0.001947219, 0.001967032,
    0.001989096, 0.002012908, 0.002037015, 0.002061155, 0.002083816,
    0.002110998, 0.00213535, 0.002159071, 0.002184693, 0.002209068,
    0.002234292, 0.002253956, 0.002282698, 0.002309833, 0.002334319,
    0.002357889, 0.002390175, 0.002419171, 0.002449895, 0.002484027,
    0.002512985, 0.002539188, 0.002567327, 0.002599104, 0.002625685,
    0.002654757, 0.002687361, 0.002721435, 0.002750738, 0.002781123,
    0.00281374, 0.002848015, 0.002876956, 0.002910757, 0.002943386,
    0.002972684, 0.003005049, 0.003036254, 0.003072399, 0.003106905,
    0.003139473, 0.003180708, 0.003216593, 0.003251694, 0.00328922,
    0.003324338, 0.003363825, 0.003402236, 0.00344843, 0.003486122,
    0.00352752, 0.00356777, 0.003612031, 0.003651362, 0.003697179,
    0.003737882, 0.003782545, 0.003826533, 0.003872156, 0.003911701,
    0.003955457, 0.004000382, 0.004052469, 0.004101301, 0.004147184,
    0.004194697, 0.004241262, 0.004288369, 0.004339907, 0.004394051,
    0.004451137, 0.004500002, 0.004554236, 0.004611872, 0.004667355,
    0.004719711, 0.004774229, 0.004830268, 0.004882428, 0.004934882,
    0.004986101, 0.005040784, 0.005101632, 0.005163274, 0.005226704,
    0.005289257, 0.005350263, 0.005412445, 0.005472033, 0.005535276,
    0.005601136, 0.005664036, 0.005733066, 0.005796037, 0.005857299,
    0.005925267, 0.005992233, 0.006057501, 0.006128742, 0.006200535,
    0.00627214, 0.006350154, 0.006423195, 0.00650077, 0.006574996,
    0.006654124, 0.006735702, 0.006818731, 0.006899771, 0.006975454,
    0.00706479, 0.007142061, 0.007225058, 0.007306496, 0.007394657,
    0.007482558, 0.007569182, 0.007663119, 0.007747601, 0.007838251,
    0.007930101, 0.008028204, 0.008120433, 0.008209601, 0.00830478,
    0.008398734, 0.008492907, 0.008583607, 0.008678857, 0.008776772,
    0.00887826, 0.008976361, 0.009082932, 0.00918953, 0.009297953,
    0.009403762, 0.00951546, 0.00962302, 0.009729068, 0.00985019,
    0.009938467, 0.009963858, 0.010080097, 0.010196991, 0.010315722,
    0.010437322, 0.010558435, 0.01067879, 0.010805449, 0.010931729,
    0.011062274, 0.011190088, 0.011317069, 0.011448161, 0.011581015,
    0.011707462, 0.011841585, 0.011978857, 0.012119373, 0.012265109,
    0.012399984, 0.012549156, 0.012693127, 0.012849814, 0.012996515,
    0.013137335, 0.01329749, 0.013448354, 0.013601525, 0.013761724,
    0.013920555, 0.014072876, 0.014240021, 0.014403816, 0.014567664,
    0.014728817, 0.014894604, 0.015081531, 0.015263896, 0.015435175,
    0.015619, 0.015798071, 0.015983946, 0.016166942, 0.016354234,
    0.016542838, 0.016742492, 0.016938717, 0.017140492, 0.017322702,
    0.017528381, 0.017731995, 0.017945509, 0.018155505, 0.018362455,
    0.018574234, 0.018788044, 0.019001751, 0.019224177, 0.019447634,
    0.019674075, 0.019904513, 0.020128669, 0.02036849, 0.020602089,
    0.020845288, 0.021083103, 0.021341087, 0.021606942, 0.02185917,
    0.022116756, 0.022365938, 0.022619579, 0.022883257, 0.023154996,
    0.023424856, 0.023705743, 0.023970958, 0.024239252, 0.024530042,
    0.024822481, 0.025115603, 0.025416476, 0.025707636, 0.025995277,
    0.026301741, 0.026605072, 0.026911905, 0.027208605, 0.027535344,
    0.027866076, 0.028189197, 0.028517451, 0.028844138, 0.029162815,
    0.029507343, 0.029847452, 0.030188175, 0.030552486, 0.030904678,
    0.031260369, 0.031621503, 0.031997861, 0.032365085, 0.032743534,
    0.033130562, 0.033511586, 0.033896154, 0.03428318, 0.034673362,
    0.035074117, 0.035481828, 0.035901596, 0.036323236, 0.036747672,
    0.037170823, 0.037596198, 0.038031219, 0.038474384, 0.038922554,
    0.039373353, 0.039826642, 0.040305016, 0.040783597, 0.041260709,
    0.041734888, 0.042215804, 0.042709929, 0.043205385, 0.043723684,
    0.044229306, 0.044764879, 0.045288526, 0.045807235, 0.046347416,
    0.046893041, 0.047438224, 0.047986797, 0.048551677, 0.049135507,
    0.049711363, 0.050304856, 0.050893895, 0.051488141, 0.052089747,
    0.052703854, 0.053315786, 0.053921535, 0.054559202, 0.055200639,
    0.055848837, 0.056495907, 0.05716728, 0.057833962, 0.058512883,
    0.059204573, 0.059889857, 0.060607973, 0.06133239, 0.062059847,
    0.062770234, 0.063506946, 0.064246986, 0.065011796, 0.06577357,
    0.066537431, 0.067323184, 0.06811618, 0.068945459, 0.069738331,
    0.070542392, 0.071383827, 0.072231818, 0.073067665, 0.073910188,
    0.074775834, 0.07564831, 0.076543624, 0.077445829, 0.078333793,
    0.079267801, 0.080207804, 0.081141087, 0.082081099, 0.083057134,
    0.084027762, 0.085001996, 0.086017223, 0.087022444, 0.088056712,
    0.089076272, 0.090103167, 0.091156724, 0.092257701, 0.093338566,
    0.094434656, 0.095570373, 0.096688792, 0.097851494, 0.099020154,
    0.100191692, 0.101163763, 0.101415416, 0.102602831, 0.103798926,
    0.105032367, 0.106287662, 0.107541063, 0.108817988, 0.110123263,
    0.111425787, 0.112732208, 0.114079945, 0.115440362, 0.116800969,
    0.118200238, 0.119645, 0.121068128, 0.122512388, 0.123958737,
    0.125446403, 0.126983672, 0.12848765, 0.130038502, 0.131603678,
    0.133165836, 0.134766794, 0.136383996, 0.138004527, 0.139633688,
    0.141321555, 0.143010054, 0.144734604, 0.146455281, 0.148231021,
    0.150044491, 0.151828251, 0.153645438, 0.155529055, 0.157368526,
    0.1592547, 0.161163296, 0.163114058, 0.165064881, 0.167039453,
    0.169054731, 0.171096338, 0.173168302, 0.175246152, 0.177406369,
    0.17956127, 0.181723467, 0.183923976, 0.186165027, 0.18841876,
    0.190704806, 0.19300599, 0.195331898, 0.197707201, 0.200057176,
    0.202489887, 0.204969748, 0.207439332, 0.209979971, 0.212534015,
    0.215126422, 0.217755023, 0.22047028, 0.223175171, 0.22591625,
    0.228689334, 0.231503425, 0.23431686, 0.237214253, 0.2401602,
    0.243121327, 0.246073868, 0.249101831, 0.252171456, 0.255278951,
    0.258449509, 0.261665598, 0.264919887, 0.268150942, 0.271437256,
    0.2747773, 0.278127265, 0.281602839, 0.285078214, 0.288629535,
    0.292188197, 0.295801906, 0.299460458, 0.303187012, 0.306952415,
    0.310778175, 0.314615373, 0.318526547, 0.319391535, 0.322502796,
    0.326514952, 0.330634042, 0.334767868, 0.338927465, 0.343205062,
    0.347521278, 0.351909804, 0.356342385, 0.360828888, 0.365307429,
    0.36993438, 0.374551907, 0.379327255, 0.384178358, 0.389107735,
    0.394045564, 0.399105969, 0.404230304, 0.409379319, 0.414662539,
    0.419995231, 0.425450578, 0.430911307, 0.436456937, 0.437615246,
    0.442092421, 0.447826878, 0.453632263, 0.459529452, 0.465422591,
    0.471439955, 0.47754601, 0.483760741, 0.49006621, 0.496398345,
    0.502918361, 0.509555152, 0.516174805, 0.522977637, 0.529930519,
    0.536941101, 0.54408714, 0.551390037, 0.558725267, 0.566251425,
    0.57374889, 0.5814305, 0.589190401, 0.597110067, 0.605107377,
    0.613254266, 0.621583327, 0.629977381, 0.638501498, 0.647191006,
    0.65595497, 0.664786222, 0.673917879, 0.683151218, 0.692483457,
    0.696518101, 0.702038189, 0.711705741, 0.721610776, 0.731603549,
    0.741825002, 0.752232523, 0.762713078, 0.773405938, 0.784205173,
    0.795335233, 0.806620319, 0.818131141, 0.829799025, 0.841666249,
    0.853779324, 0.866075962, 0.878600542, 0.891328845, 0.904274874,
    0.917378928, 0.930803915, 0.94455141, 0.958456037, 0.972623094,
    0.987104577, 0.99314629, 1.001812975, 1.01691887, 1.032348947,
    1.047859953, 1.063786567, 1.08001581, 1.09657264, 1.113410803,
    1.130508826, 1.148104394, 1.166092127, 1.184445502, 1.203225395,
    1.222341669, 1.241948911, 1.262011906, 1.282463931, 1.303269674,
    1.324761133, 1.342007596, 1.381450318, 1.403350107, 1.425306837,
    1.44726098, 1.469218581, 1.491196044, 1.513276475, 1.535148555,
    1.557164387, 1.579044306, 1.601124753, 1.623145177, 1.645172885,
    1.667020036, 1.689003743, 1.711132561, 1.733117404, 1.755037494,
    1.767859466, 1.77698794, 1.798886022, 1.820761967, 1.842642564,
    1.864718114, 1.88662657, 1.90849452, 1.930405264, 1.952220589,
    1.974017393, 1.996077254, 2.017880381, 2.039922185, 2.061815909,
    2.083682992, 2.105623458, 2.127365745, 2.149302662, 2.171221874,
    2.193164848, 2.21493192, 2.236955197, 2.258806181, 2.280827719,
    2.302735882, 2.31534215, 2.324688816, 2.346518515, 2.368345464,
    2.390457315, 2.412179597, 2.434075443, 2.456055356, 2.47829615,
    2.50033377, 2.522200333, 2.544005171, 2.565965716, 2.587791351,
    2.609814701, 2.631795293, 2.65378552, 2.675802654, 2.697890565,
    2.719957575, 2.741996776, 2.763942291, 2.785929406, 2.80807351,
    2.830118388, 2.85222842, 2.874075665, 2.896210325, 2.918613929,
    2.940564556, 2.962705842, 2.984830845, 3.006948737, 3.029012669,
    3.051155607, 3.07336328, 3.090875751, 3.095503691, 3.117590874,
    3.139834597, 3.162073002, 3.184466779, 3.206732896, 3.229126133,
    3.251258901, 3.273561703, 3.29574991, 3.318123497, 3.34059375,
    3.363101849, 3.385509475, 3.407832171, 3.430196479, 3.452781199,
    3.475314469, 3.497716682, 3.520377954, 3.542955073, 3.565545511,
    3.588139882, 3.610482081, 3.632800716, 3.650665528, 3.655632408,
    3.678456659, 3.700802962, 3.723495471, 3.746058732, 3.768542701,
    3.791009321, 3.813900852, 3.836555967, 3.858993532, 3.881552822,
    3.904346293, 3.927200933, 3.949912422, 3.972997432, 3.996029608,
    4.018559064, 4.041393333, 4.063960596, 4.087033708, 4.10992077,
    4.13283096, 4.155741067, 4.178809835, 4.201955179, 4.225076531,
    4.248137716, 4.271372459, 4.294395154, 4.31747589, 4.340521801,
    4.363612499, 4.386557532, 4.410614546, 4.433308059, 4.456820951,
    4.480201637, 4.503570282, 4.527187057, 4.550587572, 4.574134052,
    4.59757869, 4.621097051, 4.644777781, 4.668363965, 4.691949488,
    4.715557694, 4.739515576, 4.763549974, 4.787596715, 4.811309669,
    4.835349139, 4.858943758, 4.882638631, 4.906884217, 4.930929673,
    4.954526937, 4.978604575, 5.003078373, 5.027341109, 5.051369195,
    5.075340662, 5.098896084, 5.123327827, 5.147451055, 5.171851258,
    5.196368173, 5.22094711, 5.245418908, 5.269953524, 5.294354622,
    5.318892329, 5.343509472, 5.367827934, 5.392977825, 5.41749897,
    5.442197972, 5.46674117, 5.491904602, 5.516915798, 5.542095453,
    5.567133072, 5.592015004, 5.616814346, 5.641716692, 5.666517881,
    5.691090944, 5.715876204, 5.740863986, 5.766308822, 5.791065492,
    5.816448033, 5.842224299, 5.86760117, 5.893271137, 5.918958721,
    5.924174688, 5.944442716, 5.970448858, 5.996124254, 6.022119111,
    6.04827654, 6.073725873, 6.099687936, 6.125625996, 6.15159956,
    6.177687388, 6.204139422, 6.230543958, 6.25690007, 6.28318625,
    6.309109672, 6.335937217, 6.362262184, 6.388887237, 6.415254831,
    6.442435908, 6.469557337, 6.496755921, 6.523602328, 6.550244472,
    6.577177998, 6.603394374, 6.630080975, 6.657184174, 6.685051127,
    6.713158673, 6.74104784, 6.769586584, 6.797006468, 6.824661071,
    6.85204315, 6.879454005, 6.906781187, 6.933559623, 6.960884329,
    6.988637939, 7.016729342, 7.044111353, 7.071895703, 7.099697415,
    7.128711236, 7.157187681, 7.185487091, 7.21468549, 7.2430048,
    7.271447633, 7.299947401, 7.328367793, 7.357676668, 7.386730149,
    7.414428089, 7.443828794, 7.472587879, 7.501271287, 7.529826035,
    7.558076623, 7.586919756, 7.616411262, 7.646167108, 7.675254484,
    7.704245051, 7.73404699, 7.763033106, 7.791895415, 7.821608159,
    7.85155544, 7.880827692, 7.911011264, 7.941041698, 7.971913037,
    8.000580507, 8.030317394, 8.060547982, 8.09151538, 8.122528216,
    8.153044815, 8.183552811, 8.214168239, 8.244714092, 8.275561326,
    8.306715645, 8.337471038, 8.369523645, 8.400546315, 8.432096576,
    8.463275251, 8.494602666, 8.525526158, 8.556840703, 8.58808139,
    8.61895343, 8.652534318, 8.684256491, 8.716639044, 8.749239076,
    8.780252549, 8.813026563, 8.845398471, 8.877754166, 8.910062665,
    8.941728544, 8.973269754, 9.006118382, 9.038056541, 9.072003252,
    9.10486416, 9.138348391, 9.171536678, 9.203759441, 9.236561255,
    9.270373856, 9.303296365, 9.334896459, 9.367687404, 9.401974417,
    9.435158794, 9.468638725, 9.502661148, 9.536962044, 9.570708303,
    9.603986519, 9.639081701, 9.67161251, 9.704322808, 9.738456543,
    9.773003982, 9.807984587, 9.840148232, 9.874826499, 9.910390079,
    9.944614601, 9.97831588, 10.011866014, 10.047685301, 10.083194165,
    10.117057014, 10.152753854, 10.188438458, 10.223056627, 10.257172314,
    10.293545732, 10.329797296, 10.367142684, 10.402753157, 10.438100101,
    10.473479098, 10.508264303, 10.545557584, 10.580596315, 10.61796389,
    10.655041257, 10.692499468, 10.730316352, 10.768020086, 10.80416667,
    10.840860016, 10.87767198, 10.911250151, 10.947266938, 10.98428319,
    11.021973181, 11.058980232, 11.096938896, 11.134558529, 11.17211958,
    11.207814904, 11.244113238, 11.282030707, 11.320026208, 11.358941359,
    11.396487258, 11.433896285, 11.472263436, 11.51028198, 11.546431698,
    11.58531806, 11.621405676, 11.660764072, 11.700734147, 11.738697624,
    11.777316648, 11.817984446, 11.855700651, 11.893314497, 11.930670495,
    11.967166257, 12.004040225, 12.042088351, 12.083275053, 12.122218718,
    12.161701036, 12.202464871, 12.24241818, 12.279053387, 12.31937152,
    12.358787939, 12.365952866, 12.396032059, 12.432859562, 12.471765897,
    12.513941119, 12.55455533, 12.592817897, 12.634259198, 12.674930012,
    12.715201437, 12.757572913, 12.799445583, 12.840173712, 12.878353274,
    12.920316005, 12.963688681, 13.005699496, 13.045974356, 13.090219017,
    13.131869928, 13.173516151, 13.21447029, 13.256058634, 13.297801496,
    13.337866713, 13.379226231, 13.418165598, 13.459054944, 13.503523051,
    13.545801542, 13.589993354, 13.634415456, 13.673229894, 13.717166069,
    13.760833667, 13.802549102, 13.845978793, 13.88229718, 13.926969094,
    13.969741347, 14.020584773, 14.063497021, 14.106421511, 14.152376559,
    14.192885937, 14.233765885, 14.277472304, 14.321183454, 14.360570142,
    14.403758951, 14.448598585, 14.491167136, 14.535904218, 14.582947861,
    14.625599789, 14.6725745, 14.718997839, 14.761394561, 14.805949596,
    14.85064888, 14.900978513, 14.947169113, 14.988678489, 15.032559008,
    15.079656832, 15.122400472, 15.165768732, 15.209963169, 15.253789963,
    15.300347227, 15.34828837, 15.397736458, 15.446802477, 15.494129035,
    15.541429672, 15.587533448, 15.630930492, 15.676975481, 15.7252804,
    15.775569316, 15.817490142, 15.86337522, 15.910028822, 15.963110171,
    16.009430943, 16.054258848, 16.107653575, 16.156613747, 16.199784742,
    16.243867644, 16.287299954, 16.32696824, 16.374416464, 16.417341641,
    16.466423906, 16.516651137, 16.566030678, 16.615128131, 16.660956649,
    16.709823586, 16.751559968, 16.802611257, 16.851206243, 16.897381217,
    16.945059797, 16.996349675, 17.047885504, 17.094325939, 17.145028236,
    17.188949613, 17.23784932, 17.283040773, 17.335368552, 17.380830137,
    17.432191617, 17.477342334, 17.522343222, 17.571868284, 17.626732386,
    17.681486563, 17.731360134, 17.782254118, 17.831406757, 17.889086359,
    17.946475557, 17.993779179, 18.041402291, 18.084195459, 18.13442774,
    18.189057834, 18.230369028, 18.279297048, 18.332710638, 18.380173091,
    18.433078423, 18.482581177, 18.534257243, 18.581851907, 18.630773355,
    18.683385332, 18.736107211, 18.781480004, 18.830952675, 18.886293742,
    18.934812728, 18.982849997, 19.029241319, 19.076808466, 19.134419092,
    19.183730616, 19.237711915, 19.290681491, 19.339504392, 19.388826492,
    19.443137431, 19.505296464, 19.555609928, 19.602510563, 19.654802599,
    19.702613394, 19.749852122, 19.814617419, 19.874099968, 19.926560304,
    19.977392071, 20.036302582, 20.08310231, 20.135710948, 20.191863305,
    20.233985632, 20.295280106, 20.352643337, 20.412442544, 20.46905577,
    20.530274006, 20.574552244, 20.623417583, 20.671652792, 20.71322837,
    20.765396009, 20.829647709, 20.894267599, 20.948652775, 21.006803202,
    21.065032762, 21.124631625, 21.177418975, 21.233856595, 21.284287337,
    21.338137149, 21.391839824, 21.441121951, 21.498313528, 21.559519357,
    21.617162637, 21.670632555, 21.726960279, 21.786469651, 21.848778285,
    21.900742194, 21.962362608, 21.979373779, 22.01871292, 22.076232651,
    22.136137723, 22.187333124, 22.254325159, 22.319807377, 22.36891191,
    22.418047158, 22.463989396, 22.525807572, 22.587040357, 22.636714195,
    22.696197237, 22.760031787, 22.826483125, 22.890958094, 22.954007825,
    23.028399359, 23.081953225, 23.129389299, 23.188274871, 23.242092034,
    23.305993814, 23.358684395, 23.424059202, 23.497833855, 23.554383565,
    23.606629555, 23.661403119, 23.701969899, 23.780384594, 23.825702793,
    23.877373878, 23.966125205, 24.017205302, 24.072975967, 24.124798126,
    24.181005572, 24.246681188, 24.310037016, 24.367107045, 24.414851706,
    24.458241738, 24.516092786, 24.576605709, 24.632139396, 24.696667518,
    24.774270312, 24.822107577, 24.887105766, 24.936559127, 25.002189082,
    25.067648224, 25.129798395, 25.181038414, 25.261466633, 25.328060622,
    25.388807424, 25.452826854, 25.53190444, 25.597638426, 25.647577119,
    25.729393405, 25.786076209, 25.874358524, 25.934108501, 25.996712748,
    26.079586813, 26.153257474, 26.22619229, 26.334111008, 26.405668767,
    26.461895992, 26.509088203, 26.566222856, 26.631079676, 26.709727496,
    26.757476917, 26.82093597, 26.915255947, 26.958772112, 27.029781186,
    27.123676888, 27.177861957, 27.24512325, 27.324220533, 27.373956212,
    27.429742243, 27.517243075, 27.580190188, 27.669490052, 27.74731634,
    27.806532893, 27.880325086, 27.938234397, 28.005807934, 28.078041358,
    28.151126036, 28.233114182, 28.303061944, 28.362724087, 28.482449421,
    28.543715886, 28.60891968, 28.684275342, 28.73812249, 28.790255745,
    28.856823006, 28.948268523, 29.020813405, 29.105311365, 29.188942712,
    29.264382098, 29.323187336, 29.419749977, 29.506224844, 29.590684595,
    29.69251785, 29.762355789, 29.839526191, 29.911719723, 29.995192806,
    30.034259432, 30.112406705, 30.166103964, 30.223056562, 30.281651634,
    30.35005984, 30.416712998, 30.445453602, 30.518265699, 30.593615195,
    30.645713067, 30.713217323, 30.804048252, 30.883023345, 31.003901638,
    31.046771587, 31.122816878, 31.212532814, 31.298723674, 31.398619952,
    31.440883872, 31.496929889, 31.592603628, 31.674449132, 31.752243274,
    31.838803832, 31.910619642, 32.001721433, 32.072615429, 32.155554162,
    32.226116959, 32.309959846, 32.363491875, 32.426054232, 32.524080404,
    32.618500142, 32.66250955, 32.715304773, 32.766154792, 32.827417813,
    32.923754107, 32.999298518, 33.056013895, 33.120438294, 33.183914486,
    33.311475832, 33.380090976, 33.466218296, 33.524467037, 33.62453113,
    33.730024043, 33.770387513, 33.803389335, 33.905473534, 33.962510923,
    34.078107742, 34.170846129, 34.285455625, 34.340582899, 34.389445861,
    34.438864935, 34.492961414, 34.547874308, 34.597981374, 34.685305069,
    34.806943399, 34.845843342, 34.924446147, 34.99883105, 35.051775912,
    35.168066969, 35.250212765, 35.358180427, 35.42194198, 35.450930676,
    35.516288905, 35.614936573, 35.69816928, 35.760848373, 35.866269756,
    35.959431873, 35.998217411, 36.083350983, 36.194064776, 36.305490025,
    36.392964604, 36.435024894, 36.527155146, 36.627361762, 36.78730605,
    36.943207047, 37.016190187, 37.106458608, 37.289090809, 37.37285272,
    37.472465338, 37.518364903, 37.604077247, 37.645353034, 37.739465181,
    37.798799306, 37.848797149, 38.017653285, 38.151153304, 38.270592145,
    38.365140749, 38.396555631, 38.432455967, 38.548397827, 38.608158481,
    38.764807265, 38.842300142, 38.868837425, 38.970094398, 39.123902739,
    39.218598347, 39.277001895, 39.458020038, 39.527098458, 39.608606341,
    39.68270487, 39.730388971, 39.858581072, 39.890642873, 39.981130037,
    40.077966017, 40.147605449, 40.246266652, 40.389344258, 40.430330598,
    40.479263556, 40.564555212, 40.617019975, 40.665790342, 40.759113925,
    40.798994373, 40.9040053, 41.008785187, 41.114338928, 41.160073066,
    41.201166377, 41.390401437, 41.570880301, 41.610134677, 41.670708795,
    41.753299649, 41.808900843, 41.870709326, 42.027367466, 42.12826367,
    42.166004591, 42.217055121, 42.311566581, 42.447228452, 42.556030326,
    42.638700674, 42.77392623, 42.866600211, 43.049651544, 43.154833455,
    43.217707662, 43.433421485, 43.509396471, 43.527760411, 43.621500364,
    43.716025118, 43.837033557, 43.867298104, 44.086940006, 44.217152338,
    44.337085035, 44.528633326, 44.58608817, 44.607529512, 44.685977954,
    44.874825374, 44.905100576, 45.049288074, 45.063115702, 45.330354437,
    45.377069644, 45.4762267, 45.477203394, 45.572780757, 45.625621463,
    45.627282795, 45.753797945, 45.938842484, 46.03263932, 46.238160653,
    46.543987971, 46.638833574, 46.680236943, 46.768101045, 46.779913475,
    47.026021624, 47.055793392, 47.126439548, 47.579149862, 47.678981573,
    47.75136798, 47.896502887, 47.908145028, 48.149598767, 48.208613037,
    48.235845538, 48.310254024, 48.381232158, 48.413822693, 48.496609846,
    48.592803436, 48.671243351, 48.744190707, 48.954761599, 48.998392745,
    49.182614385, 49.198992275, 49.306254907), "1400" = c(1.7831e-05,
    1.7941e-05, 1.8217e-05, 1.8611e-05, 1.8815e-05, 1.9081e-05,
    1.9443e-05, 1.96e-05, 1.9702e-05, 1.9744e-05, 2.0181e-05,
    2.0782e-05, 2.1256e-05, 2.1813e-05, 2.1882e-05, 2.3072e-05,
    2.3142e-05, 2.418e-05, 2.4194e-05, 2.4341e-05, 2.4595e-05,
    2.469e-05, 2.505e-05, 2.5082e-05, 2.546e-05, 2.5485e-05,
    2.589e-05, 2.6002e-05, 2.6895e-05, 2.7061e-05, 2.7209e-05,
    2.7959e-05, 2.8171e-05, 2.889e-05, 2.9398e-05, 2.968e-05,
    2.9736e-05, 2.9963e-05, 3.0318e-05, 3.0687e-05, 3.0803e-05,
    3.0845e-05, 3.1211e-05, 3.1812e-05, 3.2497e-05, 3.3099e-05,
    3.388e-05, 3.4184e-05, 3.439e-05, 3.4522e-05, 3.4616e-05,
    3.4846e-05, 3.5136e-05, 3.5875e-05, 3.6194e-05, 3.6733e-05,
    3.6959e-05, 3.7536e-05, 3.7682e-05, 3.8127e-05, 3.8683e-05,
    3.8843e-05, 3.9177e-05, 4.0012e-05, 4.1324e-05, 4.162e-05,
    4.2317e-05, 4.2611e-05, 4.298e-05, 4.425e-05, 4.4502e-05,
    4.4771e-05, 4.5106e-05, 4.5176e-05, 4.5575e-05, 4.6577e-05,
    4.7041e-05, 4.7372e-05, 4.7525e-05, 4.8702e-05, 4.9524e-05,
    4.9639e-05, 5.0106e-05, 5.0529e-05, 5.162e-05, 5.2778e-05,
    5.3461e-05, 5.3734e-05, 5.4112e-05, 5.452e-05, 5.4948e-05,
    5.547e-05, 5.6164e-05, 5.7591e-05, 5.8096e-05, 5.8566e-05,
    5.9551e-05, 6.0805e-05, 6.1691e-05, 6.3108e-05, 6.4071e-05,
    6.4929e-05, 6.5518e-05, 6.579e-05, 6.6076e-05, 6.7158e-05,
    6.8531e-05, 7.0002e-05, 7.0515e-05, 7.081e-05, 7.0956e-05,
    7.2237e-05, 7.2873e-05, 7.325e-05, 7.4685e-05, 7.5447e-05,
    7.6158e-05, 7.7859e-05, 7.826e-05, 7.9464e-05, 7.9752e-05,
    8.0332e-05, 8.0767e-05, 8.1917e-05, 8.2542e-05, 8.3134e-05,
    8.3512e-05, 8.5169e-05, 8.5927e-05, 8.6353e-05, 8.7009e-05,
    8.8044e-05, 8.8568e-05, 9.0191e-05, 9.1324e-05, 9.285e-05,
    9.4366e-05, 9.5558e-05, 9.7074e-05, 9.7727e-05, 9.821e-05,
    9.8713e-05, 9.9614e-05, 0.000100618, 0.000101704, 0.000102758,
    0.000105049, 0.000105632, 0.000107586, 0.000109623, 0.000110474,
    0.000112455, 0.000113752, 0.000116114, 0.000116857, 0.000119142,
    0.000120064, 0.000123142, 0.000124947, 0.000127396, 0.000128653,
    0.000129438, 0.000131851, 0.000132872, 0.000134863, 0.000136806,
    0.000138167, 0.000141421, 0.000142609, 0.000143408, 0.000143968,
    0.000145835, 0.000147066, 0.000148516, 0.000149553, 0.000151104,
    0.000153825, 0.000156206, 0.000158041, 0.000159788, 0.000161304,
    0.000162707, 0.000164898, 0.000166645, 0.000168446, 0.000170277,
    0.000173114, 0.00017648, 0.000179495, 0.000182183, 0.000184009,
    0.000187362, 0.000189811, 0.000192369, 0.00019459, 0.000197003,
    0.000199655, 0.000201571, 0.000204675, 0.000207682, 0.000209182,
    0.000212542, 0.000214426, 0.000215959, 0.000218297, 0.000221451,
    0.000224371, 0.000225789, 0.000228894, 0.000231119, 0.00023454,
    0.000237453, 0.000239955, 0.000242609, 0.000244545, 0.000247264,
    0.00024904, 0.000252116, 0.000255853, 0.000258731, 0.00026079,
    0.000263787, 0.000267283, 0.000270197, 0.000273633, 0.00027623,
    0.000278359, 0.000281389, 0.000284484, 0.000288605, 0.000290886,
    0.000293991, 0.000295431, 0.000299053, 0.00030227, 0.000304083,
    0.000307362, 0.00031111, 0.000313991, 0.000317257, 0.000323222,
    0.00032635, 0.000330884, 0.000335135, 0.000339207, 0.00034301,
    0.00034574, 0.000349936, 0.000353446, 0.000357657, 0.000360733,
    0.000366342, 0.000372668, 0.000375599, 0.000379071, 0.000384658,
    0.000390366, 0.000396037, 0.000400398, 0.000407257, 0.000411043,
    0.000413765, 0.000418226, 0.000422363, 0.00042617, 0.000433043,
    0.000439303, 0.00044362, 0.000447518, 0.000451236, 0.000456108,
    0.00046104, 0.000466301, 0.000470775, 0.000474497, 0.000480046,
    0.000484521, 0.000490241, 0.000496422, 0.000502424, 0.000508108,
    0.000515968, 0.000521293, 0.000527903, 0.000533974, 0.000539036,
    0.000545418, 0.000553459, 0.000558079, 0.000564669, 0.000574388,
    0.000580491, 0.000585254, 0.000590991, 0.00059658, 0.000603943,
    0.00061044, 0.000616904, 0.000623054, 0.000630443, 0.000637127,
    0.000644168, 0.000650663, 0.000658346, 0.000667783, 0.00067755,
    0.00068299, 0.000690122, 0.000699591, 0.000707532, 0.000715653,
    0.000724076, 0.000734332, 0.000742258, 0.000750528, 0.000757049,
    0.00076384, 0.000773382, 0.000780601, 0.00078977, 0.000798184,
    0.000804483, 0.000812235, 0.00082303, 0.000833368, 0.000844874,
    0.000853381, 0.000863205, 0.000874652, 0.000883953, 0.000892882,
    0.000901772, 0.000910768, 0.000921226, 0.000931078, 0.000943045,
    0.000952734, 0.000962416, 0.000974802, 0.000984826, 0.000995642,
    0.000998921, 0.001008043, 0.001020199, 0.001034847, 0.00104597,
    0.001055822, 0.001064725, 0.00107654, 0.001088604, 0.001097583,
    0.001108285, 0.001121048, 0.001136141, 0.001151544, 0.001164786,
    0.00118078, 0.001195526, 0.001211035, 0.001227548, 0.00124237,
    0.001254335, 0.001267827, 0.001282371, 0.001296894, 0.001309091,
    0.001323164, 0.001338848, 0.001352798, 0.00137056, 0.001385016,
    0.001398836, 0.001417044, 0.001431063, 0.001442989, 0.001460997,
    0.001478356, 0.001497073, 0.00151401, 0.001530961, 0.001548813,
    0.001568046, 0.001584921, 0.001603088, 0.001621672, 0.001638888,
    0.001657925, 0.001680497, 0.001701322, 0.001721314, 0.001738647,
    0.001760024, 0.001782289, 0.001801516, 0.001826175, 0.001846628,
    0.001868996, 0.001893854, 0.00191508, 0.001936219, 0.001958154,
    0.00198371, 0.002003217, 0.002025618, 0.002047152, 0.002069159,
    0.002095042, 0.002120312, 0.002147799, 0.002175127, 0.002200075,
    0.00222602, 0.002249754, 0.002275208, 0.002304546, 0.002332326,
    0.002357866, 0.002384247, 0.002417239, 0.002445682, 0.002476115,
    0.002506389, 0.002532678, 0.002562547, 0.002595661, 0.002626736,
    0.002659281, 0.002691783, 0.002724389, 0.002758257, 0.002791711,
    0.00282071, 0.002859123, 0.002897628, 0.002926212, 0.00295774,
    0.002990777, 0.003027862, 0.003060847, 0.003099443, 0.00313426,
    0.003167127, 0.003202866, 0.003237372, 0.003275173, 0.00331299,
    0.00335026, 0.003387998, 0.003428177, 0.003470508, 0.003511506,
    0.003553344, 0.003596898, 0.003635832, 0.003673839, 0.003716018,
    0.003756955, 0.003800313, 0.003841522, 0.00388542, 0.003935598,
    0.003981581, 0.004024964, 0.004064957, 0.004113363, 0.004161181,
    0.004208517, 0.004258415, 0.004304457, 0.004359035, 0.004410823,
    0.004457618, 0.004510807, 0.004564723, 0.004617506, 0.004667974,
    0.004723028, 0.004776119, 0.004834308, 0.004889308, 0.004950506,
    0.005009818, 0.005067709, 0.005128123, 0.005185226, 0.005245835,
    0.005302674, 0.005362389, 0.005424943, 0.005483777, 0.005541555,
    0.005601424, 0.005666328, 0.005731092, 0.005801125, 0.005876458,
    0.005941965, 0.006009386, 0.006086423, 0.006156047, 0.006228982,
    0.006302115, 0.006370551, 0.006444773, 0.006521206, 0.006598042,
    0.006672904, 0.006752486, 0.00683559, 0.006907087, 0.006989825,
    0.007066569, 0.007152042, 0.007232803, 0.007314167, 0.007393841,
    0.007482537, 0.007568143, 0.00765811, 0.007745968, 0.007843669,
    0.007936871, 0.008028197, 0.008115132, 0.008206672, 0.008304237,
    0.008393474, 0.008493117, 0.008587774, 0.008684196, 0.008779884,
    0.00888822, 0.008986635, 0.009093139, 0.009192889, 0.009299264,
    0.009410611, 0.009519162, 0.00962003, 0.009732876, 0.009846762,
    0.009935242, 0.009959522, 0.010078135, 0.01019167, 0.010312979,
    0.010431056, 0.01055075, 0.010676978, 0.010804804, 0.010927679,
    0.01104412, 0.01117174, 0.011304265, 0.011430419, 0.011556717,
    0.011697134, 0.01183672, 0.011974694, 0.012116802, 0.012250956,
    0.012401834, 0.012542011, 0.01269006, 0.012843861, 0.012995813,
    0.01314788, 0.013299445, 0.013456905, 0.013604029, 0.013757391,
    0.01391732, 0.014084718, 0.014249747, 0.014414406, 0.014593065,
    0.01476755, 0.014933354, 0.015110561, 0.015289492, 0.015458509,
    0.015633904, 0.015820811, 0.016001214, 0.016192583, 0.016365278,
    0.016550122, 0.016742875, 0.016947451, 0.017140397, 0.017349719,
    0.017553093, 0.017752365, 0.017956966, 0.018168062, 0.018371116,
    0.018586213, 0.01882177, 0.019040949, 0.019251906, 0.019480295,
    0.019704174, 0.019919995, 0.020151233, 0.020383799, 0.020631969,
    0.020872085, 0.021114005, 0.021358482, 0.02161534, 0.021863279,
    0.022110226, 0.022373713, 0.022631341, 0.022892308, 0.023157784,
    0.023428334, 0.02369859, 0.023981441, 0.024274234, 0.024554533,
    0.024837347, 0.025120866, 0.025422795, 0.025705822, 0.026006082,
    0.026321311, 0.026630889, 0.026934877, 0.027246731, 0.027570721,
    0.027888284, 0.028223972, 0.028565219, 0.028903063, 0.029243131,
    0.029570553, 0.029900441, 0.030252719, 0.03060993, 0.030972154,
    0.03133065, 0.031703092, 0.032068587, 0.032427498, 0.03278975,
    0.03317027, 0.033561696, 0.033947039, 0.034347081, 0.034740006,
    0.035135155, 0.035548408, 0.035967298, 0.036380853, 0.036803288,
    0.037226308, 0.037657969, 0.038095012, 0.038543889, 0.038996519,
    0.039460793, 0.039916708, 0.04038132, 0.040838207, 0.041312822,
    0.041814392, 0.042291559, 0.04278887, 0.043273413, 0.04376992,
    0.044276841, 0.044803905, 0.045312161, 0.045842142, 0.046377646,
    0.04691486, 0.047467617, 0.048015043, 0.048582693, 0.049150278,
    0.049732473, 0.050325174, 0.050902754, 0.051485506, 0.0520896,
    0.052710511, 0.053330976, 0.053972506, 0.05459866, 0.055220276,
    0.055879048, 0.056521209, 0.05718659, 0.057874583, 0.058567676,
    0.059254534, 0.059952051, 0.060656017, 0.06136021, 0.062082199,
    0.062818063, 0.063557802, 0.064309565, 0.065091908, 0.065840773,
    0.066652898, 0.067429578, 0.068233059, 0.069043032, 0.069853244,
    0.070669721, 0.07150363, 0.072314971, 0.073130794, 0.073997433,
    0.074873937, 0.075756467, 0.076632126, 0.077541539, 0.078460833,
    0.079395226, 0.080327735, 0.081270587, 0.082254817, 0.083209335,
    0.084198798, 0.085207439, 0.086215355, 0.087245837, 0.088272613,
    0.089307031, 0.090386991, 0.091442429, 0.092510079, 0.093633603,
    0.09472008, 0.095843288, 0.096987019, 0.098131129, 0.099279874,
    0.100476453, 0.101438849, 0.101690879, 0.102904875, 0.104162496,
    0.105416044, 0.106667509, 0.107949782, 0.109232626, 0.110555224,
    0.111883944, 0.113203209, 0.114562951, 0.115906331, 0.117306447,
    0.118708657, 0.120104133, 0.121521496, 0.122966218, 0.124445082,
    0.125936753, 0.127436368, 0.128964483, 0.130504319, 0.132096003,
    0.1336863, 0.135276481, 0.136911111, 0.138554579, 0.140242268,
    0.141938269, 0.143639853, 0.145413528, 0.147140853, 0.148903696,
    0.150692133, 0.152512442, 0.154342111, 0.156183991, 0.158064246,
    0.159948347, 0.16186326, 0.163799273, 0.165739463, 0.167723915,
    0.169755963, 0.171802685, 0.173862164, 0.175960583, 0.178094205,
    0.180250519, 0.182441674, 0.184684602, 0.186948612, 0.189254457,
    0.191544006, 0.193859013, 0.196253067, 0.19865819, 0.201092421,
    0.203518097, 0.20598771, 0.208486394, 0.211028033, 0.213629583,
    0.216283244, 0.218866183, 0.221575107, 0.224254748, 0.227006951,
    0.229787496, 0.23258628, 0.235442661, 0.238352108, 0.241268666,
    0.244216287, 0.247228453, 0.250191699, 0.253247737, 0.256316727,
    0.259516443, 0.262718883, 0.265919953, 0.269178395, 0.272523273,
    0.27587615, 0.279282925, 0.282749436, 0.286185158, 0.289741431,
    0.293385233, 0.297030959, 0.30075067, 0.304507308, 0.308322805,
    0.312162702, 0.31602327, 0.319949115, 0.32082789, 0.323919928,
    0.32800637, 0.332108143, 0.336335469, 0.340551434, 0.344855098,
    0.349146596, 0.353522555, 0.357995419, 0.362504701, 0.367054137,
    0.371761268, 0.376467704, 0.381234707, 0.386070036, 0.390982324,
    0.395960356, 0.401035034, 0.406216225, 0.411399878, 0.416630596,
    0.42204557, 0.427474799, 0.432953557, 0.43850481, 0.439637128,
    0.444120589, 0.449814512, 0.455618014, 0.461540739, 0.467553056,
    0.473607745, 0.479753686, 0.485990488, 0.492335079, 0.498836804,
    0.505396717, 0.51210224, 0.518858833, 0.525725111, 0.532669852,
    0.539722327, 0.546858099, 0.554081562, 0.56141069, 0.568903591,
    0.57650255, 0.584234115, 0.592035779, 0.599906348, 0.608005073,
    0.616195718, 0.624589331, 0.633122478, 0.641746731, 0.650495154,
    0.659399327, 0.668324764, 0.677460417, 0.686782765, 0.696259463,
    0.700301934, 0.705899259, 0.715675488, 0.725611088, 0.735722033,
    0.746057065, 0.756582727, 0.767193548, 0.777981781, 0.78904289,
    0.800194613, 0.811567615, 0.823100716, 0.834898847, 0.84695932,
    0.8591844, 0.871573539, 0.884180699, 0.8970998, 0.910189396,
    0.923553666, 0.93717811, 0.951042684, 0.965058713, 0.979335098,
    0.99403733, 1.000124592, 1.009045808, 1.024271013, 1.039680589,
    1.055458033, 1.071505688, 1.087802123, 1.104611766, 1.121633766,
    1.138937738, 1.156798675, 1.174973474, 1.193468466, 1.212365717,
    1.231729056, 1.251584591, 1.271916935, 1.292545093, 1.313585597,
    1.335042505, 1.352472153, 1.392405154, 1.414543673, 1.43686263,
    1.45897607, 1.48107273, 1.50350833, 1.525800085, 1.547909482,
    1.570169267, 1.5923381, 1.614664762, 1.636791955, 1.65900005,
    1.681098133, 1.70317778, 1.725288768, 1.747378227, 1.76976488,
    1.782778916, 1.791981715, 1.814180111, 1.836370847, 1.858529791,
    1.880726916, 1.902889968, 1.925097569, 1.94722753, 1.969424357,
    1.991706858, 2.014070828, 2.036361554, 2.058676466, 2.080699189,
    2.103094261, 2.125066539, 2.147066555, 2.169353535, 2.191574977,
    2.213638703, 2.235763161, 2.257934161, 2.28023901, 2.302584566,
    2.324761302, 2.337511352, 2.346815031, 2.368966758, 2.391271784,
    2.413409591, 2.435552978, 2.457650243, 2.479793129, 2.501849394,
    2.524336938, 2.546429655, 2.568607336, 2.590855205, 2.613209996,
    2.635270247, 2.657583863, 2.67977755, 2.701883395, 2.724197234,
    2.746577633, 2.76894576, 2.791476991, 2.813944478, 2.836100422,
    2.858443026, 2.880696097, 2.903153652, 2.925446298, 2.947913348,
    2.970358702, 2.992546444, 3.014920703, 3.037222479, 3.059384704,
    3.081885344, 3.104212545, 3.1221715, 3.126934893, 3.14919205,
    3.17146431, 3.193885139, 3.216585498, 3.23910276, 3.26139601,
    3.28373027, 3.306140629, 3.328819697, 3.351293439, 3.373959545,
    3.396266108, 3.419012072, 3.441824954, 3.464493648, 3.4870455,
    3.509724987, 3.532051506, 3.554771465, 3.577219109, 3.599880612,
    3.62264978, 3.645040802, 3.667428476, 3.685154909, 3.690176794,
    3.71259837, 3.735831218, 3.75826397, 3.781220472, 3.803975884,
    3.826641724, 3.849586616, 3.872566093, 3.895255119, 3.918110252,
    3.940907203, 3.963847592, 3.986845659, 4.009553881, 4.032554549,
    4.056075874, 4.079107147, 4.101923222, 4.1248161, 4.147673198,
    4.170725359, 4.193650891, 4.216625512, 4.239771311, 4.262962181,
    4.28593134, 4.309075923, 4.331887794, 4.354919716, 4.377852362,
    4.401203027, 4.424622265, 4.447788967, 4.471319732, 4.494491303,
    4.517762387, 4.540916493, 4.56437975, 4.587635475, 4.611013723,
    4.634591771, 4.657716703, 4.681172443, 4.704258512, 4.727680408,
    4.751662216, 4.775630949, 4.799375101, 4.822920346, 4.847012334,
    4.870776346, 4.894275208, 4.917799179, 4.941433236, 4.965508646,
    4.98943228, 5.013190794, 5.037299313, 5.06129845, 5.085314705,
    5.109730231, 5.134200811, 5.157868971, 5.181704814, 5.205492551,
    5.229504309, 5.253819511, 5.277649258, 5.301522574, 5.325311026,
    5.349664055, 5.37413671, 5.398981341, 5.422677412, 5.447084989,
    5.471267717, 5.496017379, 5.520919705, 5.544971916, 5.569645408,
    5.594307293, 5.619380631, 5.643598633, 5.668194373, 5.691832144,
    5.71632921, 5.740995436, 5.765339322, 5.789863149, 5.814550383,
    5.839177576, 5.863681455, 5.888341346, 5.913130635, 5.9381396,
    5.943094981, 5.963447465, 5.989105859, 6.013914392, 6.039148704,
    6.064388766, 6.089931626, 6.114906488, 6.14063807, 6.166133609,
    6.191432074, 6.216686792, 6.24158708, 6.267433495, 6.292019684,
    6.317895135, 6.342935502, 6.368706992, 6.39413743, 6.419749472,
    6.444950483, 6.470879433, 6.495966135, 6.521601399, 6.547428989,
    6.573293606, 6.599667493, 6.625018651, 6.651138818, 6.67751432,
    6.704021371, 6.730062821, 6.755603606, 6.781644804, 6.807974411,
    6.834200405, 6.860836025, 6.887162462, 6.913342178, 6.940281284,
    6.966891035, 6.993895708, 7.020603287, 7.047296804, 7.072358263,
    7.099091796, 7.126616926, 7.152335727, 7.178526477, 7.204126957,
    7.230336709, 7.256502058, 7.283533298, 7.311191138, 7.339081494,
    7.366224283, 7.392941032, 7.420503728, 7.448082361, 7.475081667,
    7.50212611, 7.530232715, 7.557857142, 7.585262992, 7.613434275,
    7.641245297, 7.669204983, 7.69679902, 7.724253894, 7.753040358,
    7.781479345, 7.810388643, 7.838479852, 7.866158738, 7.894403754,
    7.92263794, 7.951301026, 7.979898607, 8.008146978, 8.035820435,
    8.063988604, 8.092353605, 8.121015088, 8.149615563, 8.177256764,
    8.206719413, 8.236145812, 8.265354014, 8.294641915, 8.323236457,
    8.351906726, 8.381429761, 8.41020102, 8.438678018, 8.468692193,
    8.498535515, 8.528851288, 8.558452306, 8.587794727, 8.617652046,
    8.647928708, 8.677810688, 8.706882254, 8.738057133, 8.766542253,
    8.795352902, 8.825000791, 8.853604781, 8.882366887, 8.912259335,
    8.941359824, 8.971910441, 9.003040639, 9.0325685, 9.063552796,
    9.095102748, 9.124930782, 9.155233753, 9.185071473, 9.215697931,
    9.247358748, 9.277479873, 9.308741793, 9.340122526, 9.371546338,
    9.403223964, 9.434258225, 9.465002121, 9.496099392, 9.530055097,
    9.562239177, 9.594208415, 9.625496374, 9.656909142, 9.687405356,
    9.719912779, 9.751652745, 9.782886456, 9.815346534, 9.845713223,
    9.878469807, 9.909888992, 9.94171453, 9.974679569, 10.00941817,
    10.040476713, 10.072877677, 10.106066174, 10.138047889, 10.171349801,
    10.203834184, 10.23558378, 10.269715313, 10.302368255, 10.334858227,
    10.368856342, 10.401547277, 10.43280348, 10.465735795, 10.497516923,
    10.532617352, 10.566311433, 10.599801776, 10.632991956, 10.664670284,
    10.697973848, 10.731601236, 10.763472955, 10.796222722, 10.82739145,
    10.860815763, 10.893553973, 10.924512156, 10.956755399, 10.990583987,
    11.025800065, 11.06099685, 11.096347767, 11.130434196, 11.163067337,
    11.199134945, 11.234011985, 11.269368734, 11.304387845, 11.341446697,
    11.37820082, 11.41201887, 11.445561399, 11.476793513, 11.510878799,
    11.548182173, 11.583837822, 11.619457603, 11.65383984, 11.690209306,
    11.724295129, 11.759833087, 11.794034973, 11.829217053, 11.862524217,
    11.899156462, 11.906123285, 11.933209308, 11.967090838, 12.001851064,
    12.037296007, 12.073697583, 12.109156117, 12.144370649, 12.182131016,
    12.21941042, 12.253657567, 12.291568426, 12.329046714, 12.367405548,
    12.408098049, 12.445229501, 12.482853644, 12.519304572, 12.55913932,
    12.596134817, 12.632016299, 12.671528505, 12.705914465, 12.739578814,
    12.774686387, 12.81630976, 12.854116714, 12.893962317, 12.929851152,
    12.969139888, 13.007732794, 13.046307261, 13.08559139, 13.125310613,
    13.164579981, 13.205254569, 13.246529852, 13.282731898, 13.318678576,
    13.358918939, 13.39653093, 13.433052873, 13.469031917, 13.508508501,
    13.545899121, 13.58018394, 13.61683416, 13.657073832, 13.695262213,
    13.73520164, 13.773689734, 13.814976363, 13.852312027, 13.897784269,
    13.934360393, 13.975412058, 14.017775214, 14.051730767, 14.09115577,
    14.127704395, 14.167873753, 14.209398833, 14.245705413, 14.282923621,
    14.317111627, 14.358878171, 14.402716347, 14.442466432, 14.482032655,
    14.526472438, 14.565763317, 14.605487159, 14.644731579, 14.68674541,
    14.726613209, 14.766155082, 14.809388935, 14.847757844, 14.886066221,
    14.927240125, 14.969691191, 15.01128237, 15.05347297, 15.094701095,
    15.138650958, 15.179364646, 15.223441962, 15.264190485, 15.306670381,
    15.344773763, 15.384057016, 15.43342212, 15.476165914, 15.518530034,
    15.559279257, 15.598846582, 15.634505575, 15.671366448, 15.711712561,
    15.752076947, 15.793430803, 15.838631265, 15.879485889, 15.918404272,
    15.965431106, 16.002836763, 16.045639074, 16.082835888, 16.126737456,
    16.168760989, 16.206079305, 16.250720863, 16.290902509, 16.327539773,
    16.370887493, 16.410487856, 16.45104139, 16.492869633, 16.536189862,
    16.569847576, 16.61464104, 16.655393506, 16.694050244, 16.734066475,
    16.775543007, 16.819962176, 16.860902805, 16.904973359, 16.950457987,
    16.998355971, 17.036009891, 17.078014944, 17.121025123, 17.158264308,
    17.204801211, 17.248207952, 17.293208549, 17.338506267, 17.381946465,
    17.42078027, 17.468614647, 17.520907535, 17.566355318, 17.606759332,
    17.654072879, 17.699799495, 17.741657194, 17.789275838, 17.835491312,
    17.88122275, 17.922041947, 17.977885118, 18.025175244, 18.067189375,
    18.1136232, 18.165575957, 18.206768316, 18.248437206, 18.296010955,
    18.347090183, 18.391092829, 18.437449124, 18.47793348, 18.522193401,
    18.564947938, 18.613807975, 18.66408711, 18.702975645, 18.744388855,
    18.783550179, 18.828238188, 18.885045547, 18.924226509, 18.967445602,
    19.005363159, 19.047125626, 19.09819107, 19.152194472, 19.199611854,
    19.247924173, 19.2910138, 19.33743649, 19.381901013, 19.426921453,
    19.473018914, 19.512539389, 19.558920115, 19.608999834, 19.66878986,
    19.717969512, 19.763690546, 19.816212359, 19.863869136, 19.917294436,
    19.963871525, 20.007901889, 20.059031663, 20.104103646, 20.148968851,
    20.208784874, 20.250611304, 20.260864673, 20.301633064, 20.356242161,
    20.410323694, 20.475885128, 20.529820987, 20.585500615, 20.637192051,
    20.68271839, 20.731569614, 20.776363445, 20.834937128, 20.909834124,
    20.956168503, 21.002129131, 21.052588409, 21.094520126, 21.152101549,
    21.188800537, 21.244103764, 21.310006433, 21.354955521, 21.40206369,
    21.451007678, 21.496271298, 21.556132338, 21.596261685, 21.647609367,
    21.699590413, 21.757692891, 21.799117089, 21.853644828, 21.909484857,
    21.964467698, 22.008776652, 22.068830939, 22.121923491, 22.162710405,
    22.215669156, 22.2693537, 22.323182076, 22.370740573, 22.424421703,
    22.466611751, 22.531483557, 22.570230473, 22.617088966, 22.682316728,
    22.724022677, 22.776742054, 22.846380802, 22.894232053, 22.937789069,
    22.98876521, 23.031455078, 23.086757448, 23.12985261, 23.194654071,
    23.248685814, 23.30577049, 23.366295531, 23.405813505, 23.453791538,
    23.515263701, 23.561628285, 23.601817044, 23.647521613, 23.681187305,
    23.724276935, 23.780825527, 23.832248062, 23.880707677, 23.92956275,
    23.981621568, 24.016026022, 24.072275342, 24.119481715, 24.16720295,
    24.231751128, 24.274181962, 24.312844376, 24.37659505, 24.438828699,
    24.487800035, 24.542615578, 24.58945204, 24.633320195, 24.695411749,
    24.770024218, 24.835249015, 24.889796744, 24.923664032, 24.96557347,
    25.025889114, 25.090011401, 25.126508622, 25.163440205, 25.223166277,
    25.267377228, 25.344636145, 25.39815572, 25.436128022, 25.487430653,
    25.55223161, 25.597565588, 25.652514018, 25.705482866, 25.74953022,
    25.792848257, 25.828800816, 25.869168934, 25.936397709, 26.002356204,
    26.053682635, 26.08087032, 26.132783828, 26.173515992, 26.229510266,
    26.272451472, 26.309083047, 26.359753651, 26.401396449, 26.439655782,
    26.52359056, 26.570079827, 26.618393095, 26.666802646, 26.727531139,
    26.791988234, 26.848400995, 26.885521682, 26.967373135, 27.054281972,
    27.126185196, 27.183460336, 27.223065766, 27.289476595, 27.375225828,
    27.439923738, 27.477301541, 27.521925285, 27.570839079, 27.616758862,
    27.656552659, 27.703057733, 27.761724127, 27.823437408, 27.882175375,
    27.922825706, 27.960352229, 28.032756295, 28.07334876, 28.131610108,
    28.169855066, 28.231016279, 28.278081515, 28.335405431, 28.413819878,
    28.480196918, 28.537600887, 28.598292655, 28.675957775, 28.72404327,
    28.769166625, 28.807193164, 28.90209439, 28.962740334, 29.018164949,
    29.137117605, 29.194046141, 29.264688281, 29.321271073, 29.403531514,
    29.445165682, 29.545930535, 29.606740538, 29.666936289, 29.776747342,
    29.80744734, 29.859842159, 29.901741074, 29.93032483, 29.963381947,
    29.998891358, 30.045012452, 30.11812948, 30.199097058, 30.238209676,
    30.287112238, 30.374915642, 30.431931488, 30.501493336, 30.542877905,
    30.635749315, 30.670782714, 30.785330861, 30.835762564, 30.865149921,
    30.914420871, 30.944687051, 31.01630189, 31.044910467, 31.090188515,
    31.171517457, 31.208208842, 31.276984199, 31.322863365, 31.397462852,
    31.426239419, 31.469240155, 31.498110677, 31.55080374, 31.61311541,
    31.67985269, 31.717604212, 31.810703749, 31.855962473, 31.885165866,
    31.936221169, 31.991485511, 32.095267577, 32.149774825, 32.215132505,
    32.255260932, 32.357580683, 32.422995529, 32.476985465, 32.541405968,
    32.606052982, 32.639744704, 32.68682831, 32.74459581, 32.886422118,
    32.962292367, 32.986547533, 33.078495458, 33.095588872, 33.22660229,
    33.305341441, 33.381124971, 33.47112108, 33.513695621, 33.594251757,
    33.664915995, 33.751212262, 33.801982951, 33.870467069, 33.967773124,
    34.016359868, 34.116978732, 34.213198621, 34.291303944, 34.35267169,
    34.405031834, 34.504573306, 34.599669406, 34.667354694, 34.756180964,
    34.772101323, 34.800954217, 34.829519917, 34.894981847, 34.962036857,
    34.989634698, 35.139006994, 35.191216342, 35.303532396, 35.381490532,
    35.40344537, 35.434872406, 35.511672376, 35.636608928, 35.719547598,
    35.830798778, 35.941286771, 35.995870806, 36.048051451, 36.134846943,
    36.174543236, 36.252959754, 36.320845061, 36.411175035, 36.465428134,
    36.536601048, 36.552838305, 36.621592403, 36.651809219, 36.683483414,
    36.689399464, 36.716580037, 36.751964023, 36.795326349, 36.831413385,
    36.843559234, 36.900053899, 36.989542159, 37.114403425, 37.199344052,
    37.221008725, 37.421660163, 37.546276656, 37.550029287, 37.597373401,
    37.683463608, 37.77477639, 37.792971714, 37.975615327, 38.092603058,
    38.235941387, 38.378966583, 38.415297286, 38.42722264, 38.567016786,
    38.727151144, 38.742254089, 38.766900609, 38.8570361, 38.930008015,
    38.971851011, 39.212871608, 39.233183416, 39.43810679, 39.486847253,
    39.540422443, 39.711002872, 39.74067893, 39.833629017, 39.841016861,
    40.044551158, 40.054603865, 40.370238296, 40.396922984, 40.489101459,
    40.53998817, 40.556387706, 40.569473614, 40.652730554, 40.672932824,
    40.74787794, 40.855127934, 40.96580759), "1600" = c(1.9608e-05,
    1.9648e-05, 1.9727e-05, 2.0009e-05, 2.0041e-05, 2.0139e-05,
    2.0404e-05, 2.0586e-05, 2.0881e-05, 2.098e-05, 2.0996e-05,
    2.1208e-05, 2.1646e-05, 2.2085e-05, 2.2228e-05, 2.2776e-05,
    2.305e-05, 2.3159e-05, 2.3327e-05, 2.3809e-05, 2.3967e-05,
    2.5013e-05, 2.5054e-05, 2.5344e-05, 2.6033e-05, 2.6367e-05,
    2.6606e-05, 2.6877e-05, 2.7451e-05, 2.7841e-05, 2.7963e-05,
    2.8765e-05, 2.9129e-05, 2.9248e-05, 3.0029e-05, 3.1115e-05,
    3.1445e-05, 3.1527e-05, 3.1743e-05, 3.2646e-05, 3.266e-05,
    3.3491e-05, 3.3662e-05, 3.3833e-05, 3.4055e-05, 3.4456e-05,
    3.454e-05, 3.5166e-05, 3.5674e-05, 3.5783e-05, 3.6293e-05,
    3.6576e-05, 3.6828e-05, 3.7116e-05, 3.7307e-05, 3.7549e-05,
    3.7707e-05, 3.8436e-05, 3.8537e-05, 3.9156e-05, 3.9638e-05,
    4.0333e-05, 4.0407e-05, 4.1254e-05, 4.1391e-05, 4.1557e-05,
    4.2304e-05, 4.2535e-05, 4.3134e-05, 4.3469e-05, 4.4021e-05,
    4.4534e-05, 4.4883e-05, 4.5663e-05, 4.6507e-05, 4.7421e-05,
    4.8536e-05, 4.8858e-05, 4.9498e-05, 5.0205e-05, 5.0792e-05,
    5.1395e-05, 5.2132e-05, 5.2864e-05, 5.3111e-05, 5.438e-05,
    5.4629e-05, 5.5095e-05, 5.5176e-05, 5.6272e-05, 5.6604e-05,
    5.7165e-05, 5.737e-05, 5.8394e-05, 5.8911e-05, 5.962e-05,
    5.9769e-05, 5.9977e-05, 6.0294e-05, 6.0669e-05, 6.0788e-05,
    6.2715e-05, 6.3488e-05, 6.3968e-05, 6.4149e-05, 6.5143e-05,
    6.5705e-05, 6.6558e-05, 6.7992e-05, 6.8934e-05, 6.935e-05,
    6.9942e-05, 7.0384e-05, 7.1346e-05, 7.2086e-05, 7.2511e-05,
    7.2964e-05, 7.3703e-05, 7.4205e-05, 7.4706e-05, 7.5241e-05,
    7.7104e-05, 7.7865e-05, 7.8385e-05, 8.0289e-05, 8.158e-05,
    8.193e-05, 8.2404e-05, 8.3532e-05, 8.4174e-05, 8.5311e-05,
    8.5934e-05, 8.6961e-05, 8.8601e-05, 8.9361e-05, 9.1574e-05,
    9.2368e-05, 9.3477e-05, 9.4418e-05, 9.5328e-05, 9.5785e-05,
    9.6351e-05, 9.7946e-05, 9.9014e-05, 9.9914e-05, 0.000101638,
    0.000103633, 0.000105045, 0.000106481, 0.000107, 0.00010876,
    0.000110241, 0.000111898, 0.000113577, 0.000115178, 0.000116739,
    0.000118434, 0.000119892, 0.00012238, 0.000123131, 0.000124411,
    0.000125164, 0.000126644, 0.000127636, 0.000128797, 0.000130438,
    0.000133073, 0.000134469, 0.000135637, 0.000136585, 0.000138459,
    0.000140227, 0.000141832, 0.00014371, 0.000145563, 0.000147242,
    0.000149543, 0.000150163, 0.000151477, 0.000152457, 0.000154833,
    0.000156335, 0.000158661, 0.000160509, 0.000162146, 0.000163496,
    0.00016585, 0.000168024, 0.000170576, 0.000173971, 0.000176986,
    0.000178693, 0.00018082, 0.000182338, 0.000184639, 0.000186151,
    0.000187686, 0.000190065, 0.00019204, 0.000194543, 0.000197008,
    0.000198837, 0.000199687, 0.000201194, 0.000204191, 0.000206865,
    0.000209102, 0.000211181, 0.000214209, 0.000216775, 0.000220638,
    0.000223163, 0.000225164, 0.000227157, 0.000229533, 0.000230987,
    0.000234294, 0.000236859, 0.000238718, 0.000241831, 0.000244894,
    0.000247702, 0.000252175, 0.000255984, 0.000258987, 0.000262327,
    0.000267135, 0.000270125, 0.000273046, 0.000276148, 0.000278128,
    0.00028031, 0.000285022, 0.000287534, 0.000291167, 0.000293897,
    0.000297144, 0.000299136, 0.000302594, 0.000307489, 0.000310602,
    0.000314946, 0.000319264, 0.0003233, 0.000325702, 0.00032958,
    0.000332149, 0.000334744, 0.000339388, 0.000343894, 0.000348425,
    0.00035381, 0.000357602, 0.000362248, 0.000366632, 0.000370196,
    0.000373131, 0.0003792, 0.000383007, 0.000388162, 0.000393247,
    0.000397325, 0.000401533, 0.000405681, 0.000409477, 0.000416028,
    0.000421962, 0.00042728, 0.000430528, 0.000435504, 0.000439555,
    0.000445165, 0.000452395, 0.000458133, 0.000464227, 0.000470577,
    0.000474971, 0.000480308, 0.000485912, 0.000493768, 0.000499034,
    0.000504568, 0.000510515, 0.000518386, 0.000525701, 0.00053136,
    0.000537427, 0.000543175, 0.000549798, 0.000556652, 0.00056581,
    0.000571205, 0.00057816, 0.000584698, 0.000590741, 0.000596818,
    0.000605053, 0.000611814, 0.000619932, 0.000627088, 0.000633434,
    0.000642642, 0.0006506, 0.000658292, 0.000667178, 0.000674553,
    0.000682585, 0.000690141, 0.000697951, 0.000705781, 0.000716598,
    0.000724589, 0.000733255, 0.000741133, 0.000748811, 0.00075455,
    0.000760989, 0.00076857, 0.000778212, 0.000787916, 0.000797474,
    0.000809724, 0.000818929, 0.000830413, 0.00083854, 0.000847903,
    0.000857198, 0.000865811, 0.000877492, 0.000887602, 0.000899217,
    0.000908817, 0.000920277, 0.000932576, 0.000944885, 0.000952942,
    0.000964371, 0.000974871, 0.000986834, 0.000997008, 0.001007407,
    0.0010103, 0.00102127, 0.001031801, 0.001041964, 0.001052419,
    0.001063847, 0.00107781, 0.001090016, 0.001102961, 0.001114682,
    0.001124333, 0.001137592, 0.001149969, 0.001167903, 0.001180698,
    0.001195647, 0.001209181, 0.001222212, 0.001237155, 0.001253557,
    0.001266142, 0.001279445, 0.001294372, 0.001306666, 0.001323518,
    0.001339331, 0.001357541, 0.001371849, 0.001385865, 0.001401686,
    0.001420666, 0.001437118, 0.001453388, 0.001467082, 0.001482054,
    0.001498343, 0.001515962, 0.001535743, 0.001550804, 0.001567058,
    0.001582374, 0.001600578, 0.001621506, 0.0016405, 0.001661329,
    0.001679951, 0.001702053, 0.001723209, 0.001743781, 0.001766747,
    0.001784785, 0.001804502, 0.001825268, 0.001846743, 0.001870399,
    0.0018882, 0.001905816, 0.001928426, 0.001951168, 0.00197417,
    0.001996004, 0.002015532, 0.002036926, 0.002057964, 0.002082128,
    0.002107694, 0.002131287, 0.002156351, 0.00218091, 0.002204043,
    0.002229679, 0.002254802, 0.002282737, 0.00230747, 0.002337905,
    0.002363414, 0.002391698, 0.002414529, 0.002440758, 0.002467858,
    0.002497623, 0.002528854, 0.002555771, 0.002587122, 0.002616973,
    0.00264712, 0.002680035, 0.002709456, 0.002738223, 0.002772198,
    0.002801418, 0.00283865, 0.002867849, 0.002901419, 0.002931064,
    0.002961736, 0.002996275, 0.00303342, 0.00306783, 0.003106326,
    0.003137532, 0.003177853, 0.003214556, 0.00325315, 0.003288313,
    0.003326135, 0.003367202, 0.003404877, 0.003445778, 0.003488188,
    0.003531544, 0.003575849, 0.003616207, 0.003656485, 0.003697793,
    0.00373393, 0.003779821, 0.003827249, 0.003872268, 0.003921892,
    0.003971794, 0.004011367, 0.004060488, 0.004105589, 0.00415167,
    0.004193045, 0.0042359, 0.00428922, 0.004332596, 0.00438236,
    0.004436557, 0.004490335, 0.004542976, 0.004598667, 0.004648224,
    0.004707669, 0.00476332, 0.004820933, 0.00487856, 0.004931938,
    0.004988346, 0.005036333, 0.005095563, 0.005157784, 0.005212247,
    0.005274377, 0.005333954, 0.005399004, 0.005459576, 0.005525137,
    0.005592215, 0.005656783, 0.005728735, 0.00579675, 0.00585996,
    0.005926989, 0.005999954, 0.006076654, 0.006144013, 0.006210273,
    0.00628234, 0.006350621, 0.006425043, 0.006498696, 0.006576208,
    0.006650547, 0.006731562, 0.006809107, 0.00689037, 0.006974664,
    0.007062075, 0.007146844, 0.00722866, 0.007315649, 0.0073949,
    0.007481442, 0.007564301, 0.007654682, 0.007743107, 0.007823353,
    0.007907365, 0.007993999, 0.008086297, 0.008178087, 0.008270875,
    0.008368043, 0.008466144, 0.008559625, 0.008660029, 0.008757896,
    0.008863688, 0.00896028, 0.009062079, 0.009167135, 0.009280718,
    0.009386579, 0.009487364, 0.009598223, 0.009713882, 0.009832575,
    0.009919475, 0.009946867, 0.010057721, 0.010174549, 0.010292708,
    0.010417993, 0.010539793, 0.010661047, 0.010785576, 0.010906694,
    0.011033031, 0.011173554, 0.011303163, 0.011428046, 0.011554525,
    0.011689424, 0.011827982, 0.011972646, 0.012112454, 0.012259515,
    0.012398831, 0.012540376, 0.012686258, 0.01283184, 0.01298467,
    0.013133881, 0.013280041, 0.013435068, 0.013581967, 0.013746377,
    0.013909884, 0.014068109, 0.014232, 0.014399898, 0.014569567,
    0.014751051, 0.014921487, 0.015093927, 0.015267785, 0.015440931,
    0.015619531, 0.015797833, 0.015991919, 0.016183064, 0.016376191,
    0.016555717, 0.016749219, 0.016936646, 0.017118963, 0.017324913,
    0.017528987, 0.017737855, 0.017945007, 0.018157602, 0.018364985,
    0.018590534, 0.018814443, 0.019049483, 0.019274873, 0.019504447,
    0.019733016, 0.019959117, 0.020203727, 0.020432253, 0.020669645,
    0.020915064, 0.021168859, 0.021405693, 0.021662116, 0.021919422,
    0.02217014, 0.022437214, 0.022691281, 0.022959496, 0.02322258,
    0.023490352, 0.023772717, 0.024053644, 0.024336686, 0.024608589,
    0.024897038, 0.025186381, 0.02547233, 0.025767037, 0.026060726,
    0.026353098, 0.026654712, 0.026963741, 0.027267906, 0.027585446,
    0.027912959, 0.028229763, 0.02853743, 0.028882079, 0.029210589,
    0.029546228, 0.029892296, 0.030233018, 0.030600219, 0.030956478,
    0.03130502, 0.031676159, 0.032046511, 0.032412659, 0.032790146,
    0.033181605, 0.033567581, 0.033953982, 0.034341506, 0.034731434,
    0.035130116, 0.035544055, 0.035945951, 0.03636141, 0.036788304,
    0.037217739, 0.03765034, 0.038096194, 0.038538564, 0.038978488,
    0.039442876, 0.039894266, 0.040372895, 0.04083233, 0.041309209,
    0.041805408, 0.042306804, 0.042806917, 0.043294524, 0.043806263,
    0.044314986, 0.044835526, 0.045353608, 0.045877006, 0.046402208,
    0.04695312, 0.047496679, 0.048053065, 0.048611292, 0.049181859,
    0.049767139, 0.050339708, 0.050951946, 0.05153766, 0.052142431,
    0.052767165, 0.05337718, 0.053996316, 0.054642896, 0.05525726,
    0.055896179, 0.056555187, 0.057225787, 0.057898253, 0.05857945,
    0.059271187, 0.059968963, 0.060689466, 0.061393938, 0.062134928,
    0.06283996, 0.063599242, 0.064355874, 0.065126401, 0.065893865,
    0.066675653, 0.067449514, 0.068249411, 0.069028923, 0.069839254,
    0.070639622, 0.071483182, 0.072338802, 0.073164123, 0.073991743,
    0.074853715, 0.07574516, 0.07665707, 0.077565151, 0.078504158,
    0.079411986, 0.08033844, 0.081280688, 0.082244622, 0.083215287,
    0.084190493, 0.085168159, 0.08617903, 0.087199267, 0.088231057,
    0.089277281, 0.090327973, 0.091394135, 0.092491532, 0.093592559,
    0.094700481, 0.095801566, 0.096934808, 0.098077559, 0.09923554,
    0.100408144, 0.101342244, 0.101581964, 0.102799278, 0.104035833,
    0.105252859, 0.106485856, 0.107765218, 0.109056564, 0.110361397,
    0.111704873, 0.113051946, 0.114382018, 0.115755541, 0.117129819,
    0.118539953, 0.119971105, 0.12139324, 0.122859848, 0.124296569,
    0.125760445, 0.127249288, 0.128759327, 0.130283119, 0.131840809,
    0.133413958, 0.13501097, 0.136635398, 0.138319555, 0.139949203,
    0.141627465, 0.14331139, 0.145025488, 0.146803641, 0.14852178,
    0.150343154, 0.152131926, 0.153963843, 0.155841704, 0.15772419,
    0.159569409, 0.161489262, 0.163464392, 0.165429376, 0.167406227,
    0.169459197, 0.171517532, 0.173574529, 0.175714045, 0.177821718,
    0.180004145, 0.182211284, 0.184468577, 0.186723296, 0.188979943,
    0.191235228, 0.193546528, 0.195906166, 0.198290112, 0.200685291,
    0.203120711, 0.205612479, 0.208131298, 0.21069092, 0.213270603,
    0.215872227, 0.218510376, 0.221216802, 0.223929011, 0.226692725,
    0.229506877, 0.232311141, 0.2351993, 0.238052004, 0.240955432,
    0.243965712, 0.246968563, 0.250028176, 0.253116323, 0.256213911,
    0.259343908, 0.262526436, 0.265779164, 0.269065249, 0.272438077,
    0.27583639, 0.279266913, 0.282695665, 0.286246667, 0.289859822,
    0.293391598, 0.297059828, 0.300787608, 0.304442371, 0.308265006,
    0.312135158, 0.316008857, 0.320004978, 0.320894683, 0.324051021,
    0.328118213, 0.332190459, 0.336370706, 0.340594626, 0.344888683,
    0.349314441, 0.353690315, 0.358136965, 0.36265863, 0.367235229,
    0.3719252, 0.376683997, 0.381489971, 0.386318183, 0.39127182,
    0.396243372, 0.401427221, 0.406567475, 0.411843889, 0.417142071,
    0.42248601, 0.427981978, 0.433494009, 0.4390649, 0.440231083,
    0.444743833, 0.450498579, 0.456359969, 0.462209895, 0.468136755,
    0.474254692, 0.480468468, 0.486756019, 0.493062126, 0.499584354,
    0.506166285, 0.512921878, 0.519646688, 0.526545855, 0.533532776,
    0.540649416, 0.547804238, 0.555107559, 0.562455713, 0.569944974,
    0.577616831, 0.585389378, 0.593243353, 0.60122625, 0.609410895,
    0.6175831, 0.625929333, 0.634386946, 0.64301333, 0.651771566,
    0.660633663, 0.669650653, 0.678880465, 0.688181525, 0.697680563,
    0.701783764, 0.707316414, 0.717190994, 0.727133311, 0.737301893,
    0.747652375, 0.758145893, 0.768837757, 0.779592087, 0.790609584,
    0.801792771, 0.813140822, 0.824785212, 0.836586167, 0.848635721,
    0.860853584, 0.873323418, 0.886025574, 0.898971384, 0.912164306,
    0.925455609, 0.939103405, 0.952905185, 0.967122956, 0.981559674,
    0.996261066, 1.002379448, 1.011148739, 1.026481668, 1.042014651,
    1.058027789, 1.074252012, 1.090714858, 1.107512391, 1.124734368,
    1.142158291, 1.160153153, 1.178468263, 1.197071052, 1.216008435,
    1.2354355, 1.255402234, 1.275580759, 1.296194001, 1.317422277,
    1.339089232, 1.356776216, 1.396857901, 1.419166027, 1.441499595,
    1.463914671, 1.486286028, 1.508553114, 1.530965741, 1.553256444,
    1.575408395, 1.597757432, 1.61999203, 1.642248544, 1.66445951,
    1.686623801, 1.708879757, 1.731187743, 1.753467545, 1.77584754,
    1.788943678, 1.798092967, 1.820355471, 1.842533792, 1.8647118,
    1.886936837, 1.909305865, 1.931529072, 1.954072842, 1.97632616,
    1.998608017, 2.021052713, 2.043302192, 2.065607133, 2.087935173,
    2.110229061, 2.132636253, 2.154995741, 2.177235488, 2.199402131,
    2.221576509, 2.243897723, 2.266290772, 2.288519429, 2.31070944,
    2.333081263, 2.34604377, 2.355550992, 2.377818707, 2.400255474,
    2.422738582, 2.445085654, 2.467456507, 2.489585904, 2.511891072,
    2.534260564, 2.556724906, 2.578914752, 2.601630169, 2.623749032,
    2.64633981, 2.66874498, 2.691173507, 2.713614835, 2.735922446,
    2.758327625, 2.780995364, 2.803597691, 2.82602563, 2.848557027,
    2.871113937, 2.893592411, 2.916118093, 2.938385762, 2.961034904,
    2.983444812, 3.005941158, 3.028652219, 3.05101414, 3.073272652,
    3.095837104, 3.118429574, 3.136015486, 3.140710814, 3.163138918,
    3.185804213, 3.208050254, 3.230456244, 3.252723084, 3.275477097,
    3.298061271, 3.320630926, 3.343295719, 3.365893743, 3.388335587,
    3.411000063, 3.433463548, 3.456103919, 3.478984207, 3.501678524,
    3.524249862, 3.54693322, 3.569697957, 3.592441598, 3.615246225,
    3.63796297, 3.660843769, 3.683646853, 3.701566223, 3.706492125,
    3.729503775, 3.752249619, 3.775286507, 3.798256196, 3.821089945,
    3.843911479, 3.866837404, 3.889765538, 3.912367522, 3.935198586,
    3.95849341, 3.981666366, 4.004585015, 4.027455577, 4.050548471,
    4.07333725, 4.096499729, 4.119721112, 4.142783164, 4.165786849,
    4.188684091, 4.211799178, 4.23542033, 4.258444631, 4.281890123,
    4.304955626, 4.328233819, 4.351453474, 4.37472522, 4.398061829,
    4.42090751, 4.443790214, 4.46696961, 4.489925058, 4.513195561,
    4.536385801, 4.5596758, 4.582575064, 4.605748125, 4.629605909,
    4.653126205, 4.676299127, 4.700037044, 4.723414306, 4.746634849,
    4.769973463, 4.793489829, 4.817049207, 4.840530591, 4.864497359,
    4.887951261, 4.911842579, 4.935306815, 4.959295444, 4.983237294,
    5.006667141, 5.030303609, 5.053877386, 5.077738734, 5.10201048,
    5.126551184, 5.150146789, 5.174040192, 5.197615803, 5.221386333,
    5.245090441, 5.269218155, 5.292958538, 5.317026774, 5.341222257,
    5.365501678, 5.390026963, 5.413952116, 5.437987855, 5.462037731,
    5.485389102, 5.509758948, 5.533935167, 5.55861352, 5.583254033,
    5.607737669, 5.632073412, 5.656230003, 5.679981489, 5.70419867,
    5.728780091, 5.753132828, 5.777633105, 5.802873749, 5.82829312,
    5.852921963, 5.877345802, 5.902088029, 5.926784915, 5.95170284,
    5.956932317, 5.976539626, 6.00132726, 6.026563375, 6.050891366,
    6.075930831, 6.10100674, 6.126025231, 6.151159708, 6.175888513,
    6.200908002, 6.22594573, 6.250873793, 6.275807847, 6.300432364,
    6.325435375, 6.351154795, 6.376822732, 6.401844701, 6.427899435,
    6.453341235, 6.47886146, 6.504204072, 6.529858887, 6.556075737,
    6.581907713, 6.607280077, 6.632756136, 6.658769962, 6.685091357,
    6.711490808, 6.737250795, 6.763185011, 6.788507367, 6.815523079,
    6.841317336, 6.867145296, 6.892899729, 6.918920643, 6.945285114,
    6.971541147, 6.997411954, 7.023948684, 7.04975532, 7.076827774,
    7.103543859, 7.130164155, 7.156144125, 7.181755538, 7.20777682,
    7.234404158, 7.261421502, 7.28755687, 7.314711435, 7.341336113,
    7.367887734, 7.393810989, 7.420697423, 7.448476829, 7.47569705,
    7.50287127, 7.530196223, 7.557572609, 7.583934051, 7.610549932,
    7.637731589, 7.665086607, 7.691881199, 7.718975188, 7.746156972,
    7.774414693, 7.80205666, 7.83087062, 7.859076354, 7.887065063,
    7.915279777, 7.941908508, 7.969120682, 7.996871424, 8.024686683,
    8.052752313, 8.080735981, 8.108699835, 8.137306191, 8.164735176,
    8.192991548, 8.220996876, 8.249405279, 8.276549495, 8.304691194,
    8.332483647, 8.360391301, 8.388989254, 8.419062944, 8.446804053,
    8.474675229, 8.503925906, 8.532266427, 8.561929567, 8.591129884,
    8.620195402, 8.648829425, 8.677572529, 8.707535185, 8.735724893,
    8.76430221, 8.792445756, 8.82094665, 8.850373881, 8.879906437,
    8.909855904, 8.938837154, 8.967751394, 8.996745769, 9.026304492,
    9.055684078, 9.084658305, 9.11347585, 9.142978391, 9.172816101,
    9.202146904, 9.232323031, 9.262932667, 9.292763445, 9.323302014,
    9.353149539, 9.382141103, 9.412471261, 9.440785925, 9.470015853,
    9.499454994, 9.531022442, 9.561485352, 9.591050563, 9.621325617,
    9.652616599, 9.683588346, 9.714670612, 9.746328276, 9.777442923,
    9.80953042, 9.839346426, 9.868991124, 9.900858858, 9.930867593,
    9.961954972, 9.993317993, 10.02530063, 10.057943074, 10.09043792,
    10.121011906, 10.15298009, 10.183747459, 10.213189069, 10.244521446,
    10.273848143, 10.304374765, 10.33804826, 10.369510676, 10.403616696,
    10.434119414, 10.466226387, 10.498922158, 10.530940443, 10.564481673,
    10.595683465, 10.628777174, 10.659549962, 10.692689923, 10.725488703,
    10.75635293, 10.789221417, 10.820740851, 10.853796554, 10.886242863,
    10.922147388, 10.95422346, 10.986197733, 11.018607838, 11.051462786,
    11.085296149, 11.115929815, 11.150454854, 11.184689508, 11.218197656,
    11.250994913, 11.283732314, 11.315691555, 11.349590332, 11.383478532,
    11.416762477, 11.44768531, 11.480623715, 11.513748435, 11.545544339,
    11.579777003, 11.615240741, 11.650970235, 11.685131131, 11.719985525,
    11.754139373, 11.761475492, 11.789629, 11.823465151, 11.857399017,
    11.891871173, 11.924475122, 11.959037157, 11.994176664, 12.028251435,
    12.061465241, 12.096112546, 12.13215272, 12.165511435, 12.203547152,
    12.23927245, 12.273735408, 12.30811362, 12.346834802, 12.381585417,
    12.417906017, 12.451604856, 12.489389194, 12.527012288, 12.562815567,
    12.59828709, 12.630836396, 12.668348689, 12.704121517, 12.739190978,
    12.774555877, 12.805485266, 12.842549733, 12.879094456, 12.915424885,
    12.95315268, 12.989435484, 13.025246843, 13.060779958, 13.095694982,
    13.134775737, 13.173183334, 13.212739565, 13.249454162, 13.285367867,
    13.320872408, 13.358066451, 13.397240583, 13.431764433, 13.470991203,
    13.508334144, 13.54452099, 13.57853487, 13.61352797, 13.65226941,
    13.693113204, 13.729390776, 13.766609316, 13.802679346, 13.841446755,
    13.879154544, 13.917690601, 13.957318139, 13.992388824, 14.031158169,
    14.069696325, 14.107525482, 14.146858162, 14.184152224, 14.220486359,
    14.261205632, 14.300037786, 14.340876401, 14.379109906, 14.420409474,
    14.458274758, 14.49739161, 14.534847084, 14.570076741, 14.60622325,
    14.644950766, 14.682386827, 14.720004683, 14.758984906, 14.799821017,
    14.83668079, 14.874403317, 14.913396762, 14.952721607, 14.996582213,
    15.03401045, 15.072766247, 15.113461069, 15.14935755, 15.187840487,
    15.226492162, 15.266485928, 15.306154044, 15.345960051, 15.382635992,
    15.418649399, 15.463932926, 15.503573428, 15.547200981, 15.591215516,
    15.62736016, 15.671998769, 15.705713984, 15.74634414, 15.781734275,
    15.818461095, 15.858422943, 15.899084153, 15.941888608, 15.983722119,
    16.027103789, 16.071712072, 16.112938766, 16.1553971, 16.196504528,
    16.236527087, 16.275561028, 16.311273455, 16.359528695, 16.406036871,
    16.448818256, 16.49033708, 16.529993071, 16.579253103, 16.637203761,
    16.674866379, 16.715299427, 16.766606419, 16.811127695, 16.850836947,
    16.890734465, 16.93634556, 16.976157376, 17.022172795, 17.063896712,
    17.106364998, 17.150379759, 17.19213418, 17.231503548, 17.273745227,
    17.31935129, 17.358169549, 17.401296875, 17.445995257, 17.494154846,
    17.534586447, 17.572755021, 17.609645732, 17.647421606, 17.67891636,
    17.730262271, 17.771399881, 17.815478631, 17.857317702, 17.900325493,
    17.946297417, 17.984045514, 18.019226649, 18.066491996, 18.108012114,
    18.152787064, 18.196015588, 18.236892167, 18.280939889, 18.326067691,
    18.378107337, 18.422755376, 18.472761828, 18.509803984, 18.56034217,
    18.602769612, 18.647500055, 18.692068495, 18.739540464, 18.775787521,
    18.814153634, 18.859371604, 18.90502837, 18.944135638, 18.981714708,
    19.030072503, 19.074001942, 19.119015824, 19.15190502, 19.19341947,
    19.234633149, 19.282267636, 19.324859856, 19.374693713, 19.420738199,
    19.460079179, 19.509073897, 19.560539872, 19.60356102, 19.642746263,
    19.6943337, 19.750659494, 19.760136154, 19.803178696, 19.854249396,
    19.890116428, 19.942864407, 19.981187444, 20.034104942, 20.07638789,
    20.132895418, 20.181209773, 20.233053118, 20.285619988, 20.327316759,
    20.36954867, 20.403895814, 20.447995722, 20.488015682, 20.524519424,
    20.568648548, 20.624641957, 20.682753936, 20.725554361, 20.78762286,
    20.828133805, 20.872930157, 20.917587139, 20.979322573, 21.041880885,
    21.086759022, 21.141800413, 21.190669952, 21.249117492, 21.304351712,
    21.354522955, 21.411996191, 21.464604221, 21.514155059, 21.553542685,
    21.584748074, 21.637298684, 21.688436817, 21.737904522, 21.786050471,
    21.833464433, 21.883446042, 21.932893752, 21.992641442, 22.042515609,
    22.081466348, 22.126777512, 22.18222527, 22.216450324, 22.293491865,
    22.359962337, 22.407936733, 22.445616563, 22.501558625, 22.547141032,
    22.623054476, 22.69091366, 22.73429762, 22.784541465, 22.825037373,
    22.866531616, 22.912266011, 22.95437647, 23.006652629, 23.050325172,
    23.088871084, 23.12764019, 23.170882246, 23.227082372, 23.265526191,
    23.331458681, 23.38301373, 23.446352516, 23.50309599, 23.556372768,
    23.607967213, 23.659282609, 23.714309954, 23.766279058, 23.819871055,
    23.860768398, 23.94430774, 24.01308824, 24.070126647, 24.115387817,
    24.171107678, 24.239440687, 24.286843451, 24.338798459, 24.392677126,
    24.458103069, 24.521377392, 24.568082718, 24.640511356, 24.691545779,
    24.747048057, 24.811608497, 24.862819798, 24.910276908, 24.965956215,
    25.007428883, 25.042230043, 25.093164383, 25.173819618, 25.224174306,
    25.276966566, 25.330070004, 25.372717679, 25.412123296, 25.462419047,
    25.509216911, 25.553764883, 25.584662084, 25.645225118, 25.688118429,
    25.741395106, 25.786230858, 25.845900415, 25.88750361, 25.942039158,
    26.000380361, 26.048367453, 26.099270646, 26.140294246, 26.200320236,
    26.25018939, 26.282010157, 26.346791107, 26.389578251, 26.437089604,
    26.495216299, 26.580603435, 26.629803073, 26.691484214, 26.747337301,
    26.843623193, 26.925133392, 26.955887647, 27.026045529, 27.075125608,
    27.120625714, 27.161729973, 27.224039066, 27.286711693, 27.345276498,
    27.426836551, 27.475963987, 27.557269686, 27.604269028, 27.686019623,
    27.738327574, 27.80257818, 27.868994024, 27.934557856, 27.9919186,
    28.050055367, 28.099790186, 28.176384648, 28.263431888, 28.314574092,
    28.384654866, 28.420857437, 28.482016571, 28.534125084, 28.586209073,
    28.655717515, 28.694753936, 28.752989333, 28.776198243, 28.826679329,
    28.88821201, 28.970385188, 29.037645252, 29.095543609, 29.127662593,
    29.178346659, 29.21808891, 29.256437237, 29.311458519, 29.396945312,
    29.465971776, 29.513974769, 29.541268448, 29.604064023, 29.67624075,
    29.738991003, 29.842024561, 29.914227423, 29.962348737, 30.023793139,
    30.062890127, 30.156809384, 30.197704197, 30.240681224, 30.284695816,
    30.334973768, 30.39014567, 30.402034559, 30.464131869, 30.49928459,
    30.5491443, 30.570208303, 30.680412976, 30.762882334, 30.797688397,
    30.846291531, 30.888630084, 30.955892322, 30.99327326, 31.038180846,
    31.08331476, 31.177114069, 31.219740799, 31.278481994, 31.307356672,
    31.397091692, 31.427870067, 31.518257919, 31.560807038, 31.656693868,
    31.709666696, 31.81093475, 31.97720113, 32.008186296, 32.062251627,
    32.128760312, 32.201684428, 32.24835578, 32.275283572, 32.381527085,
    32.440260139, 32.480574195, 32.511497111, 32.594310598, 32.657048537,
    32.686852256, 32.770629339, 32.886079338, 32.908666946, 32.983498442,
    33.081414235, 33.172397776, 33.205603604, 33.259229164, 33.339995403,
    33.426661359, 33.446689526, 33.581070378, 33.653683469, 33.762545454,
    33.841207286, 33.900457283, 33.939159982, 33.982114093, 34.047358415,
    34.113938239, 34.251111922, 34.289813067, 34.372027694, 34.453224165,
    34.502118774, 34.578105824, 34.700178181, 34.820598518, 34.903547937,
    34.998687655, 35.085454585, 35.109470558, 35.166703921, 35.182107392,
    35.480223295, 35.532236046, 35.582482125, 35.624753312, 35.633269296,
    35.729647242, 35.745811846, 35.788632725, 35.861119459, 35.884927747,
    36.016584275, 36.08383383, 36.150716721, 36.171502887, 36.384959313,
    36.451892434, 36.651091929, 36.677581245, 36.723017198, 36.786973784,
    36.790022964, 36.922170198, 37.17308079, 37.219391715, 37.331649375,
    37.342276474, 37.600697317, 37.668423705, 37.758140792, 37.826097596,
    37.877297373, 37.983718784, 37.995134734, 38.06848652, 38.119232119,
    38.18332962, 38.263766958, 38.487017648, 38.549110355, 38.565895129,
    38.651149164, 38.664741018, 38.672926811, 38.676696303, 38.844565215,
    38.927415905, 39.08920136, 39.109342143, 39.117129606, 39.119680444,
    39.187010546, 39.290260773, 39.322468768, 39.480664433, 39.482836853,
    39.544826561, 39.549907921, 39.613493916, 39.655546418, 39.680673602,
    39.879221838, 39.98433193, 40.033156282, 40.1705654, 40.337001074,
    40.404112195, 40.404179982, 40.471469348), "2400" = c(1.7178e-05,
    1.7437e-05, 1.7474e-05, 1.7721e-05, 1.7971e-05, 1.8153e-05,
    1.8376e-05, 1.8903e-05, 1.8981e-05, 1.8999e-05, 1.91e-05,
    1.9322e-05, 1.9456e-05, 1.9488e-05, 2.0038e-05, 2.0845e-05,
    2.138e-05, 2.2005e-05, 2.2142e-05, 2.2537e-05, 2.2595e-05,
    2.2684e-05, 2.312e-05, 2.377e-05, 2.4908e-05, 2.5018e-05,
    2.5166e-05, 2.5388e-05, 2.5756e-05, 2.6553e-05, 2.6588e-05,
    2.6661e-05, 2.6764e-05, 2.6861e-05, 2.7395e-05, 2.8357e-05,
    2.8654e-05, 2.8697e-05, 2.9443e-05, 3.0047e-05, 3.0104e-05,
    3.0159e-05, 3.0535e-05, 3.114e-05, 3.1683e-05, 3.2517e-05,
    3.3254e-05, 3.3919e-05, 3.3976e-05, 3.4779e-05, 3.5213e-05,
    3.5363e-05, 3.5624e-05, 3.5878e-05, 3.6538e-05, 3.6577e-05,
    3.7234e-05, 3.726e-05, 3.7699e-05, 3.7964e-05, 3.8244e-05,
    3.8499e-05, 3.9057e-05, 3.9522e-05, 3.9768e-05, 4.0305e-05,
    4.0658e-05, 4.1134e-05, 4.2235e-05, 4.2752e-05, 4.3105e-05,
    4.3585e-05, 4.3931e-05, 4.481e-05, 4.5382e-05, 4.566e-05,
    4.6139e-05, 4.6581e-05, 4.7066e-05, 4.741e-05, 4.7918e-05,
    4.8133e-05, 4.8824e-05, 4.9514e-05, 5.0328e-05, 5.0894e-05,
    5.1307e-05, 5.1739e-05, 5.2239e-05, 5.2509e-05, 5.2976e-05,
    5.3183e-05, 5.3507e-05, 5.4248e-05, 5.4371e-05, 5.4855e-05,
    5.5065e-05, 5.5405e-05, 5.57e-05, 5.6261e-05, 5.7648e-05,
    5.8271e-05, 5.867e-05, 5.8854e-05, 5.951e-05, 6.0263e-05,
    6.126e-05, 6.1816e-05, 6.2375e-05, 6.2857e-05, 6.3786e-05,
    6.5115e-05, 6.679e-05, 6.7969e-05, 6.9317e-05, 6.9939e-05,
    7.1844e-05, 7.3132e-05, 7.3791e-05, 7.4373e-05, 7.564e-05,
    7.7774e-05, 7.7979e-05, 7.913e-05, 7.9552e-05, 8.0045e-05,
    8.1189e-05, 8.145e-05, 8.2741e-05, 8.3656e-05, 8.4934e-05,
    8.5803e-05, 8.6824e-05, 8.7421e-05, 8.8203e-05, 8.9835e-05,
    9.0877e-05, 9.1476e-05, 9.3902e-05, 9.4631e-05, 9.5619e-05,
    9.7568e-05, 9.8753e-05, 0.000100161, 0.000101953, 0.000103111,
    0.000104311, 0.00010669, 0.000108172, 0.000109192, 0.000110088,
    0.000112323, 0.000114076, 0.00011498, 0.000116711, 0.000118339,
    0.000119571, 0.000120763, 0.000121844, 0.000122623, 0.000124201,
    0.000125346, 0.000126003, 0.000127776, 0.00013001, 0.000131695,
    0.000132976, 0.00013585, 0.00013714, 0.000138367, 0.000139583,
    0.000141045, 0.000143948, 0.000145445, 0.000147692, 0.000149832,
    0.00015145, 0.000152527, 0.000153448, 0.000154116, 0.000155534,
    0.000157445, 0.000158536, 0.000160169, 0.000161586, 0.000162579,
    0.000166052, 0.00016849, 0.000170624, 0.000171936, 0.000173872,
    0.000176034, 0.000177679, 0.000180457, 0.000182536, 0.000185257,
    0.000189009, 0.000191639, 0.000193225, 0.000195724, 0.000197973,
    0.000200201, 0.000202041, 0.000203402, 0.00020551, 0.000207531,
    0.000209424, 0.000211528, 0.000213214, 0.000217075, 0.000219678,
    0.000221447, 0.000224685, 0.000226149, 0.000228229, 0.000230797,
    0.000233824, 0.0002372, 0.000240453, 0.000242019, 0.000244964,
    0.000246839, 0.000249381, 0.000252412, 0.000254984, 0.000257801,
    0.000260351, 0.000263763, 0.000266824, 0.000269706, 0.000273499,
    0.000275559, 0.000279494, 0.000282232, 0.000284475, 0.000287297,
    0.000291561, 0.000294584, 0.000297794, 0.000301962, 0.000305252,
    0.000307996, 0.000312772, 0.000315531, 0.000318755, 0.000323499,
    0.000326694, 0.000330664, 0.000334312, 0.000338793, 0.000343186,
    0.000346828, 0.000349586, 0.000353464, 0.000357289, 0.000361042,
    0.000364642, 0.000370987, 0.000375025, 0.000380453, 0.00038464,
    0.000388846, 0.000395453, 0.000400012, 0.000404898, 0.000409761,
    0.00041559, 0.000419713, 0.000424505, 0.000430912, 0.000434929,
    0.000442776, 0.000448807, 0.000453865, 0.000457571, 0.000465744,
    0.00047165, 0.000475702, 0.000481828, 0.00048767, 0.000492853,
    0.000498283, 0.000504681, 0.00050994, 0.00051498, 0.000521487,
    0.000528681, 0.000535209, 0.000539164, 0.000544623, 0.000552155,
    0.000557323, 0.000564447, 0.000569849, 0.000577763, 0.00058411,
    0.000591151, 0.000597625, 0.000604459, 0.000610396, 0.000618998,
    0.000624474, 0.00063043, 0.000638556, 0.000646811, 0.000652603,
    0.000658358, 0.000666018, 0.000671239, 0.000679593, 0.000686094,
    0.000694392, 0.000701857, 0.000710419, 0.000718598, 0.000729937,
    0.000739465, 0.000746897, 0.000754684, 0.000762091, 0.000769664,
    0.000778649, 0.000787438, 0.000799968, 0.000811161, 0.000820619,
    0.000830402, 0.000839233, 0.000847345, 0.000858064, 0.000867324,
    0.000879508, 0.000891637, 0.000901385, 0.00091315, 0.000924437,
    0.000936487, 0.00094632, 0.000958511, 0.000972629, 0.000980697,
    0.000983118, 0.00099413, 0.001005918, 0.001019113, 0.001034105,
    0.001044248, 0.001058422, 0.00107034, 0.001084847, 0.001095882,
    0.001109201, 0.001122132, 0.001134029, 0.001146245, 0.001162419,
    0.001176039, 0.001188434, 0.001203787, 0.001218663, 0.001232436,
    0.001245984, 0.001263996, 0.001279096, 0.001291714, 0.001305906,
    0.001321977, 0.001333861, 0.001352614, 0.001370857, 0.00138673,
    0.001401786, 0.001416167, 0.001431409, 0.001446859, 0.001465124,
    0.001482466, 0.001500133, 0.001517755, 0.001534841, 0.001552331,
    0.001568846, 0.001586565, 0.00160552, 0.001621112, 0.001642942,
    0.001661675, 0.001680207, 0.001697626, 0.00171904, 0.001735997,
    0.001754478, 0.001772701, 0.001796223, 0.001818976, 0.001836072,
    0.001857108, 0.001874873, 0.001894533, 0.001917298, 0.001938081,
    0.00196265, 0.001985383, 0.002007206, 0.002028534, 0.002053734,
    0.002075714, 0.002101298, 0.002123986, 0.002150206, 0.002174722,
    0.002199332, 0.002225742, 0.002251239, 0.002275661, 0.00230621,
    0.002333786, 0.002360553, 0.002387843, 0.002418137, 0.002446432,
    0.002473129, 0.002499084, 0.002526203, 0.002555245, 0.002586402,
    0.002615502, 0.002651863, 0.002686339, 0.002716645, 0.002746473,
    0.002776418, 0.002808543, 0.002845148, 0.002881292, 0.002913609,
    0.002950416, 0.002986623, 0.003016033, 0.003050717, 0.003086049,
    0.003124963, 0.003162131, 0.003197467, 0.003234507, 0.003268726,
    0.003310684, 0.003347375, 0.003388629, 0.003431794, 0.003472548,
    0.00351216, 0.003556334, 0.003597877, 0.003640444, 0.00368133,
    0.003721665, 0.003767964, 0.003809081, 0.003856594, 0.003900152,
    0.00394266, 0.003987306, 0.004025994, 0.004067197, 0.004116095,
    0.004167899, 0.004217941, 0.004264051, 0.004313678, 0.004365274,
    0.004413741, 0.00446791, 0.004517147, 0.004568485, 0.00461457,
    0.004664479, 0.004720351, 0.004777366, 0.004838668, 0.004896029,
    0.004954582, 0.005010859, 0.005067983, 0.005122995, 0.005183631,
    0.005248049, 0.005311365, 0.00536894, 0.005427244, 0.005485687,
    0.005549502, 0.005625527, 0.005694803, 0.005757091, 0.005829949,
    0.005897995, 0.005963789, 0.006036973, 0.006107589, 0.006182345,
    0.006249977, 0.006325233, 0.006395935, 0.006475263, 0.006551027,
    0.006626429, 0.00670336, 0.006786473, 0.006865358, 0.006943197,
    0.007020769, 0.007101306, 0.007187275, 0.007270768, 0.007350367,
    0.00743613, 0.007520745, 0.00761086, 0.007692459, 0.007792366,
    0.007888121, 0.007979645, 0.00807733, 0.008173997, 0.008270828,
    0.008362489, 0.008459631, 0.008552975, 0.008653754, 0.008754328,
    0.008857084, 0.008953231, 0.009059186, 0.009167588, 0.009266781,
    0.009372374, 0.00947985, 0.009599794, 0.009715327, 0.00983675,
    0.009924768, 0.009952102, 0.010071156, 0.01018647, 0.010304928,
    0.010429585, 0.010557138, 0.010678739, 0.01079989, 0.01091722,
    0.011042485, 0.011161808, 0.011294579, 0.01142476, 0.01156346,
    0.011703474, 0.011832839, 0.011971103, 0.01211058, 0.012252592,
    0.012390961, 0.012542363, 0.012680313, 0.012823761, 0.012972982,
    0.013131328, 0.013281827, 0.013439499, 0.013604936, 0.013763899,
    0.013928818, 0.014095325, 0.014255072, 0.014418183, 0.014587162,
    0.014758513, 0.014931828, 0.015107095, 0.015278353, 0.015462594,
    0.015645494, 0.015835385, 0.01601119, 0.016196222, 0.016394421,
    0.016585983, 0.016773089, 0.016972546, 0.017184694, 0.017393817,
    0.017591183, 0.017794451, 0.017995498, 0.018197984, 0.018418062,
    0.018635599, 0.018869315, 0.019082504, 0.019291503, 0.019504017,
    0.019717064, 0.019960598, 0.020202869, 0.02043564, 0.020679515,
    0.020919757, 0.02116134, 0.021404922, 0.021647379, 0.021910124,
    0.022170309, 0.022420124, 0.022672084, 0.022937411, 0.023205852,
    0.023486991, 0.023764069, 0.024030969, 0.024319918, 0.024604837,
    0.024893858, 0.02517042, 0.025458572, 0.025754499, 0.026055543,
    0.026359237, 0.026666379, 0.026980776, 0.027305872, 0.027622481,
    0.027954964, 0.028286362, 0.028603047, 0.028936428, 0.029275817,
    0.029614225, 0.029952019, 0.030293205, 0.030642352, 0.031009775,
    0.031358948, 0.031735829, 0.032103997, 0.032477892, 0.032857532,
    0.033252433, 0.033628941, 0.034013696, 0.034402629, 0.03480071,
    0.035199981, 0.0356191, 0.036023859, 0.036435773, 0.036844972,
    0.037274241, 0.037717446, 0.038159516, 0.038598537, 0.039063441,
    0.039520243, 0.039970313, 0.040454153, 0.040928799, 0.041393215,
    0.041892881, 0.042361309, 0.042857362, 0.043371653, 0.043893933,
    0.044405253, 0.044946153, 0.045462704, 0.046007642, 0.046548841,
    0.047094555, 0.047649602, 0.048196448, 0.048755232, 0.049334034,
    0.049906955, 0.050483322, 0.051089235, 0.051670329, 0.052277327,
    0.052901455, 0.05353137, 0.054142319, 0.054786458, 0.055429137,
    0.056092204, 0.056745643, 0.057418575, 0.058079271, 0.058759564,
    0.059457664, 0.060155071, 0.060868665, 0.061592644, 0.062315054,
    0.063044267, 0.063779558, 0.064534261, 0.065284005, 0.066051685,
    0.066823465, 0.067628628, 0.068438803, 0.069240711, 0.070057348,
    0.070881036, 0.071712672, 0.072571553, 0.073435425, 0.074298605,
    0.075163985, 0.076034761, 0.076940045, 0.077842799, 0.078779043,
    0.079712029, 0.08068198, 0.081644575, 0.082617632, 0.083605125,
    0.08458859, 0.085555836, 0.08656141, 0.087606809, 0.088679207,
    0.089696287, 0.090735237, 0.091829309, 0.092914815, 0.094023333,
    0.09515461, 0.096266232, 0.097406381, 0.098576076, 0.099761519,
    0.100939449, 0.101907814, 0.102149726, 0.103339141, 0.104586084,
    0.105853005, 0.107120517, 0.108378213, 0.109668845, 0.110966013,
    0.112287156, 0.11359717, 0.114940835, 0.116325174, 0.117697907,
    0.119088503, 0.120505426, 0.121909432, 0.123365267, 0.124856359,
    0.126348009, 0.127852919, 0.129378992, 0.130914202, 0.132493523,
    0.134059548, 0.135637545, 0.137272472, 0.138919785, 0.140588076,
    0.14228093, 0.143982038, 0.145681888, 0.147453704, 0.149236438,
    0.151049783, 0.152853106, 0.154666927, 0.156501633, 0.158407308,
    0.160312385, 0.162266449, 0.164198001, 0.1661557, 0.168129124,
    0.170141295, 0.172198355, 0.174307857, 0.176415954, 0.178539637,
    0.180720427, 0.182902931, 0.185104554, 0.187390916, 0.189653674,
    0.191955441, 0.194269691, 0.196595039, 0.198985864, 0.201399085,
    0.203885789, 0.206357176, 0.208880364, 0.211361109, 0.213957377,
    0.216551459, 0.219176579, 0.221907478, 0.224635846, 0.227438882,
    0.230207054, 0.233059616, 0.235932373, 0.238861496, 0.241799465,
    0.244785191, 0.247799029, 0.250835805, 0.253942417, 0.257074188,
    0.260270103, 0.263511637, 0.266783666, 0.270091972, 0.27345334,
    0.276894703, 0.280364977, 0.283845425, 0.287312312, 0.290911744,
    0.294554357, 0.298255152, 0.301935999, 0.305693808, 0.309529316,
    0.313396626, 0.317250081, 0.321270054, 0.322139063, 0.325295904,
    0.329409678, 0.333567374, 0.33774375, 0.341953689, 0.346325147,
    0.350701812, 0.355123474, 0.359627458, 0.364204253, 0.368871471,
    0.373653969, 0.378411956, 0.383212247, 0.388107092, 0.393078723,
    0.398121124, 0.403205145, 0.408384823, 0.413613482, 0.418867692,
    0.424284446, 0.429769233, 0.435312983, 0.44098072, 0.442144527,
    0.446702385, 0.452546535, 0.458432153, 0.464429635, 0.470543747,
    0.476671433, 0.482888155, 0.489197648, 0.495591457, 0.502022942,
    0.508634811, 0.515402521, 0.522181099, 0.528995975, 0.535949028,
    0.543090116, 0.550329385, 0.557644341, 0.565103749, 0.572701362,
    0.580367712, 0.58819613, 0.596089679, 0.604138638, 0.612273067,
    0.620619492, 0.629006227, 0.637631434, 0.646353388, 0.655134243,
    0.664112224, 0.673185771, 0.682405463, 0.691863272, 0.701446786,
    0.705581565, 0.711232288, 0.721180558, 0.731195992, 0.741443715,
    0.751837201, 0.762385853, 0.773101373, 0.783986918, 0.795067615,
    0.806368064, 0.817903017, 0.829656217, 0.841693863, 0.853668951,
    0.866015967, 0.878563406, 0.891316592, 0.904359949, 0.917571555,
    0.93109899, 0.944825266, 0.958804168, 0.973031121, 0.987484295,
    1.002276938, 1.008530771, 1.017476252, 1.03275271, 1.04847124,
    1.064459315, 1.08078857, 1.097387345, 1.114358835, 1.131724573,
    1.149341986, 1.167266931, 1.185729253, 1.204415062, 1.223481726,
    1.242906313, 1.262987479, 1.283382427, 1.304316705, 1.325891181,
    1.347847948, 1.365481472, 1.40580525, 1.428460823, 1.450991213,
    1.473438129, 1.49602067, 1.518652624, 1.541116162, 1.563643747,
    1.586240885, 1.608594779, 1.63116568, 1.653751161, 1.676404984,
    1.699042017, 1.72162917, 1.744163959, 1.766817474, 1.789113743,
    1.802417191, 1.811635257, 1.834141336, 1.856609158, 1.87910716,
    1.901725223, 1.92435071, 1.946917157, 1.969717094, 1.992231574,
    2.014819142, 2.0373553, 2.059903609, 2.082201512, 2.104647547,
    2.127083039, 2.149376035, 2.171876532, 2.194552861, 2.216973039,
    2.239475447, 2.261934277, 2.284546308, 2.307115957, 2.329575983,
    2.352103239, 2.364951716, 2.374504381, 2.397049982, 2.419669689,
    2.44198323, 2.464759817, 2.487276033, 2.510036945, 2.532717128,
    2.555159038, 2.577707394, 2.600271866, 2.622796853, 2.645359057,
    2.668063194, 2.69058936, 2.713269637, 2.735773689, 2.758295254,
    2.780980611, 2.80369892, 2.826294373, 2.8487794, 2.871370699,
    2.893842225, 2.916405659, 2.939280951, 2.961612792, 2.984248203,
    3.006890565, 3.02961361, 3.052243329, 3.074858513, 3.097525875,
    3.120329855, 3.143230716, 3.161144138, 3.165870734, 3.1888734,
    3.211369087, 3.234139919, 3.256838528, 3.279636109, 3.302656357,
    3.325358664, 3.348341078, 3.371037115, 3.393646687, 3.416139481,
    3.439047403, 3.461909242, 3.48462072, 3.5075706, 3.530301218,
    3.553181285, 3.576009165, 3.59906504, 3.621699243, 3.644617717,
    3.667748017, 3.690259504, 3.713128569, 3.730866886, 3.735794329,
    3.758864544, 3.781433536, 3.804085816, 3.826952002, 3.849672079,
    3.872580984, 3.895434332, 3.918474292, 3.941200559, 3.96426954,
    3.987451614, 4.010215844, 4.033093005, 4.056386578, 4.079411973,
    4.102410733, 4.125749387, 4.148803156, 4.171877872, 4.19494414,
    4.218160662, 4.241304178, 4.264636928, 4.287901164, 4.311267137,
    4.334051415, 4.357064524, 4.380371316, 4.403578577, 4.426493532,
    4.449414154, 4.472535832, 4.495789125, 4.519043462, 4.542360934,
    4.5658265, 4.588539824, 4.611847808, 4.635107778, 4.658324432,
    4.681961104, 4.705212613, 4.728867723, 4.752110245, 4.775824215,
    4.799421178, 4.822757165, 4.846011137, 4.869538275, 4.892857015,
    4.916484808, 4.939423623, 4.962649097, 4.9861198, 5.009325377,
    5.032777036, 5.056163712, 5.079946684, 5.103444849, 5.126880829,
    5.150453208, 5.173832757, 5.197204073, 5.221088276, 5.244741587,
    5.268636866, 5.292648233, 5.316248211, 5.339513461, 5.363143551,
    5.387452981, 5.412081395, 5.436285098, 5.459868944, 5.484082695,
    5.507908495, 5.531307323, 5.554947907, 5.578762914, 5.602734511,
    5.626619461, 5.650028925, 5.674376384, 5.69843985, 5.722191109,
    5.746543591, 5.770078784, 5.793602674, 5.81750026, 5.841602902,
    5.865306004, 5.889584876, 5.913772507, 5.937732138, 5.962052368,
    5.967110903, 5.986579687, 6.011107636, 6.035057288, 6.059412629,
    6.083923037, 6.107534158, 6.132428395, 6.156502254, 6.181117885,
    6.205781981, 6.230481611, 6.254351216, 6.278350074, 6.302793164,
    6.326876927, 6.351342337, 6.375938596, 6.400840107, 6.425151745,
    6.450069878, 6.474727372, 6.499993369, 6.525027202, 6.550037234,
    6.57466392, 6.59916796, 6.624218204, 6.649284456, 6.674174086,
    6.699596346, 6.725030583, 6.749735652, 6.774004921, 6.799673519,
    6.824452054, 6.848917722, 6.874225516, 6.89903727, 6.92386197,
    6.949070285, 6.974604917, 6.999167132, 7.023879343, 7.049149731,
    7.074642688, 7.099473698, 7.125206782, 7.149628897, 7.17481879,
    7.20016764, 7.225965865, 7.251059055, 7.276512752, 7.301713799,
    7.327357036, 7.353079223, 7.379202432, 7.404450253, 7.429221149,
    7.454701634, 7.481095699, 7.507695273, 7.533941952, 7.560974652,
    7.587527163, 7.613640784, 7.638918521, 7.665867142, 7.691830998,
    7.717291199, 7.743225618, 7.769235129, 7.794958706, 7.821333697,
    7.846691841, 7.873400198, 7.898939259, 7.924686123, 7.950732841,
    7.976969698, 8.00281312, 8.028902501, 8.053659862, 8.080108627,
    8.107082651, 8.132653743, 8.158944232, 8.184581752, 8.210690121,
    8.237599273, 8.263748808, 8.291611799, 8.317527275, 8.344585297,
    8.371611933, 8.397904545, 8.425754419, 8.45256399, 8.479529924,
    8.505661285, 8.533133195, 8.559490641, 8.587413367, 8.614544253,
    8.640598881, 8.667004924, 8.694783588, 8.721377241, 8.747924273,
    8.775911121, 8.803331405, 8.831586999, 8.858405125, 8.885430621,
    8.913959505, 8.941519699, 8.968690845, 8.996629785, 9.024009773,
    9.050224709, 9.077306119, 9.104480936, 9.131817332, 9.159377714,
    9.188278773, 9.214645825, 9.242891709, 9.270908394, 9.298767502,
    9.327678091, 9.355292259, 9.383426355, 9.411902166, 9.438540198,
    9.467466525, 9.496963637, 9.524351293, 9.552120062, 9.580712591,
    9.608358535, 9.636483336, 9.66493032, 9.693457515, 9.721257111,
    9.750301491, 9.778960406, 9.807490339, 9.836827778, 9.866784461,
    9.897201746, 9.925997484, 9.953875666, 9.982107577, 10.010734819,
    10.03982484, 10.069970716, 10.097722622, 10.125617278, 10.154168507,
    10.184081807, 10.212766524, 10.241571158, 10.272141577, 10.301563646,
    10.330137108, 10.359668633, 10.38964189, 10.417549287, 10.446563363,
    10.476822321, 10.506277581, 10.538312212, 10.567160902, 10.597204745,
    10.625173765, 10.657469698, 10.686792236, 10.715657586, 10.744439088,
    10.772419593, 10.803605962, 10.833307463, 10.864584557, 10.894464225,
    10.923708658, 10.953589745, 10.983105603, 11.01485826, 11.045509612,
    11.074769333, 11.10487523, 11.135689233, 11.167770298, 11.196401773,
    11.224056242, 11.25569361, 11.286019219, 11.31672206, 11.347195545,
    11.376409629, 11.383993511, 11.407971755, 11.438796004, 11.470825216,
    11.504748594, 11.536677409, 11.567197457, 11.59746986, 11.628572748,
    11.659910735, 11.691564915, 11.720783522, 11.752107744, 11.780771998,
    11.8137527, 11.846291322, 11.873566196, 11.90388229, 11.934678364,
    11.964801822, 11.997067981, 12.031546296, 12.062734814, 12.095350284,
    12.126817421, 12.160490575, 12.191546489, 12.223823278, 12.254899083,
    12.288157596, 12.32128418, 12.35413014, 12.384381333, 12.419443852,
    12.45312663, 12.483135874, 12.516896682, 12.551271545, 12.585184095,
    12.618000185, 12.649908465, 12.68041151, 12.713074011, 12.743903329,
    12.77874738, 12.811508235, 12.844440768, 12.874802388, 12.906689029,
    12.939940049, 12.968628341, 13.003139573, 13.038543063, 13.070347555,
    13.10273164, 13.135767511, 13.170318546, 13.203922908, 13.237874146,
    13.270633228, 13.304522598, 13.336677651, 13.369552669, 13.399722555,
    13.432625278, 13.464680849, 13.499114314, 13.531274044, 13.566848576,
    13.602684205, 13.636829434, 13.67035139, 13.706866953, 13.741093796,
    13.770131076, 13.801968587, 13.83615658, 13.869423118, 13.902821459,
    13.938330245, 13.972967788, 14.00719326, 14.040425096, 14.074932047,
    14.109382312, 14.141443433, 14.176512898, 14.210593514, 14.244986639,
    14.278224326, 14.310589744, 14.341207379, 14.377948726, 14.408690363,
    14.443229524, 14.479035873, 14.51211821, 14.546488604, 14.578455699,
    14.613138033, 14.651970483, 14.689190358, 14.720059592, 14.757214584,
    14.794303984, 14.833084556, 14.867182797, 14.904650794, 14.938921947,
    14.973609043, 15.008017455, 15.041895751, 15.078546754, 15.109603065,
    15.144989174, 15.178437127, 15.213156459, 15.249252282, 15.280671333,
    15.312954917, 15.346938801, 15.381173289, 15.417751469, 15.454000121,
    15.487649877, 15.526093407, 15.563890453, 15.601219573, 15.641170188,
    15.677400919, 15.714078326, 15.751622962, 15.785983745, 15.823013755,
    15.859674291, 15.898039146, 15.93482367, 15.969735534, 16.009580417,
    16.044773812, 16.082829745, 16.126985378, 16.157163718, 16.196901956,
    16.233733727, 16.271885657, 16.3084295, 16.350748196, 16.38592205,
    16.421514743, 16.458203303, 16.494502013, 16.533405137, 16.573069511,
    16.607111527, 16.645373487, 16.681097722, 16.721040013, 16.757695439,
    16.793370432, 16.827347655, 16.865012281, 16.908519564, 16.950189977,
    16.979972553, 17.014887567, 17.049507415, 17.085173719, 17.118646667,
    17.153956847, 17.190247617, 17.233511783, 17.264723574, 17.297200814,
    17.332133021, 17.365855023, 17.40655102, 17.447540356, 17.478756028,
    17.515101543, 17.549724134, 17.585869928, 17.631101868, 17.676289195,
    17.721127984, 17.762388086, 17.803903217, 17.841310787, 17.87835534,
    17.911578029, 17.951778814, 17.986913324, 18.023452398, 18.072236234,
    18.107121732, 18.140482444, 18.189214697, 18.232866693, 18.272601155,
    18.316311833, 18.354632652, 18.365565399, 18.400917639, 18.444173703,
    18.484899384, 18.518604119, 18.566785153, 18.602079057, 18.654747824,
    18.694235602, 18.728499058, 18.76662534, 18.800232532, 18.836687777,
    18.870374068, 18.901521715, 18.931953622, 18.967163461, 19.003869911,
    19.046112026, 19.08824437, 19.130638349, 19.16939537, 19.211332752,
    19.24595799, 19.28392522, 19.318485533, 19.368420157, 19.416042404,
    19.453752916, 19.499588369, 19.542521343, 19.576806291, 19.62212728,
    19.663056893, 19.71400696, 19.756368865, 19.798902537, 19.840941302,
    19.882029316, 19.922159128, 19.964449323, 19.996558399, 20.034285781,
    20.064175575, 20.106548888, 20.160624527, 20.18536048, 20.224428438,
    20.274007611, 20.323866436, 20.377569333, 20.411644781, 20.454536806,
    20.497767618, 20.533715847, 20.573284147, 20.609948874, 20.651642368,
    20.690866691, 20.726748828, 20.761827113, 20.80498687, 20.845130874,
    20.899457687, 20.942248135, 20.964423066, 21.001486629, 21.054483475,
    21.090011477, 21.129752435, 21.16786368, 21.209827778, 21.256404704,
    21.290061976, 21.338857412, 21.387331963, 21.450562724, 21.490778467,
    21.538694703, 21.582845758, 21.62360905, 21.659987148, 21.685737242,
    21.728563865, 21.770176787, 21.815330008, 21.851625788, 21.899681723,
    21.954663215, 21.998638047, 22.053428717, 22.094894284, 22.144796508,
    22.186897441, 22.253479161, 22.284187978, 22.332233752, 22.373853579,
    22.415788295, 22.459238736, 22.494024435, 22.526910966, 22.572964459,
    22.616899706, 22.668259782, 22.720054314, 22.750433632, 22.785346852,
    22.843451195, 22.882612057, 22.928456385, 22.969111368, 22.995995512,
    23.027496203, 23.067154871, 23.102537411, 23.154232503, 23.193091021,
    23.236852454, 23.282557004, 23.31911965, 23.36209296, 23.42571173,
    23.475411499, 23.52164052, 23.559370977, 23.618556112, 23.656855911,
    23.733556935, 23.783429381, 23.813929793, 23.859191912, 23.891388205,
    23.926234722, 23.975712724, 24.020675868, 24.066062326, 24.120677358,
    24.167038827, 24.203135514, 24.249375678, 24.295417606, 24.334724023,
    24.380831961, 24.417310532, 24.455872228, 24.487010297, 24.530576465,
    24.583251364, 24.637609723, 24.691218855, 24.735911826, 24.795274171,
    24.840619875, 24.875995088, 24.947947427, 24.987209154, 25.054174834,
    25.13143555, 25.153574329, 25.202780519, 25.277129825, 25.327043671,
    25.396556956, 25.431642873, 25.483189731, 25.534043777, 25.571590125,
    25.623557684, 25.660291627, 25.739542246, 25.79154879, 25.803056747,
    25.853028036, 25.886286844, 25.927389932, 25.992145126, 26.016255324,
    26.042231623, 26.083773439, 26.118150584, 26.14294115, 26.182928797,
    26.230300769, 26.278425536, 26.352711911, 26.394304545, 26.456281347,
    26.503629664, 26.572383549, 26.625113091, 26.656145241, 26.670551336,
    26.734232112, 26.822014679, 26.840120016, 26.890677701, 26.933735372,
    27.006289692, 27.04661356, 27.098483752, 27.130072318, 27.160311453,
    27.204211372, 27.247874038, 27.306609763, 27.327955127, 27.473616661,
    27.503863355, 27.577761856, 27.64799816, 27.705348801, 27.777149632,
    27.832377341, 27.914380893, 27.958138221, 27.998299742, 28.065718124,
    28.09925855, 28.147589685, 28.176287902, 28.249552217, 28.279529192,
    28.317596103, 28.396777135, 28.427620286, 28.442051442, 28.491079598,
    28.522080348, 28.588767101, 28.662395793, 28.707666408, 28.742701171,
    28.784294068, 28.826840808, 28.844340279, 28.950034848, 29.022169913,
    29.055088464, 29.107846902, 29.132902008, 29.144022741, 29.175716109,
    29.204298157, 29.223452892, 29.306082201, 29.338438437, 29.363644872,
    29.392262279, 29.495521305, 29.549736011, 29.580042458, 29.656963298,
    29.692667434, 29.73705362, 29.757052647, 29.804236026, 29.82247234,
    29.864209623, 29.92210566, 29.93063263, 29.959537835, 30.038625373,
    30.072490151, 30.283490052, 30.297527243, 30.36262779, 30.387965438,
    30.450388416, 30.517136696, 30.539799408, 30.560837229, 30.576840205,
    30.623862338, 30.768741908, 30.885477707, 30.906308484, 30.937393567,
    30.978182687, 31.056075041, 31.071539842, 31.122357634, 31.291715029,
    31.306619664, 31.371314476, 31.421543285, 31.457476945, 31.49788851,
    31.533396535, 31.555900433, 31.688530089, 31.784683867, 31.84714738,
    31.879725638, 31.906355661, 31.960954068, 32.020208946, 32.073361081,
    32.088500579, 32.182330199, 32.248456316, 32.279806867, 32.339736411,
    32.356972287, 32.374786383, 32.420280009, 32.46249827, 32.520639729,
    32.523398832, 32.613391347, 32.762000787, 32.769422083, 32.831194625,
    32.874232547, 32.958124855, 32.9988881, 33.017075206, 33.135250264,
    33.135395583, 33.261582139, 33.314243134, 33.418348973, 33.467991181,
    33.505846218, 33.516651261, 33.528551668, 33.63001442, 33.799264125,
    33.816592197, 33.868146472, 33.95452163, 33.959913635, 33.962965333,
    34.167400107, 34.275508823, 34.321322032, 34.346509619, 34.401987518,
    34.449756523, 34.487266332, 34.498623383), "10000" = c(1.9113e-05,
    1.9205e-05, 1.9231e-05, 1.973e-05, 1.9808e-05, 1.9926e-05,
    2.0324e-05, 2.0587e-05, 2.0935e-05, 2.1019e-05, 2.1063e-05,
    2.1724e-05, 2.1845e-05, 2.2713e-05, 2.2943e-05, 2.307e-05,
    2.3081e-05, 2.3495e-05, 2.3499e-05, 2.3688e-05, 2.4026e-05,
    2.4122e-05, 2.4371e-05, 2.5164e-05, 2.5424e-05, 2.5572e-05,
    2.6309e-05, 2.632e-05, 2.6826e-05, 2.7639e-05, 2.7816e-05,
    2.8324e-05, 2.8952e-05, 2.9073e-05, 2.9275e-05, 2.9921e-05,
    2.9999e-05, 3.0115e-05, 3.0541e-05, 3.0875e-05, 3.1442e-05,
    3.1601e-05, 3.1816e-05, 3.194e-05, 3.2868e-05, 3.3397e-05,
    3.3485e-05, 3.3609e-05, 3.3857e-05, 3.42e-05, 3.4358e-05,
    3.503e-05, 3.5124e-05, 3.5725e-05, 3.6057e-05, 3.6577e-05,
    3.7825e-05, 3.8555e-05, 3.8993e-05, 3.9758e-05, 4.0642e-05,
    4.1027e-05, 4.1652e-05, 4.2936e-05, 4.3551e-05, 4.3716e-05,
    4.4037e-05, 4.4259e-05, 4.4539e-05, 4.4735e-05, 4.5896e-05,
    4.6437e-05, 4.6642e-05, 4.6919e-05, 4.7293e-05, 4.8493e-05,
    4.9248e-05, 4.9787e-05, 4.9943e-05, 5.0342e-05, 5.1434e-05,
    5.1851e-05, 5.2791e-05, 5.3464e-05, 5.3673e-05, 5.4326e-05,
    5.464e-05, 5.553e-05, 5.578e-05, 5.6338e-05, 5.6381e-05,
    5.6901e-05, 5.7152e-05, 5.7637e-05, 5.8319e-05, 5.8374e-05,
    5.8813e-05, 5.9602e-05, 6.0237e-05, 6.0759e-05, 6.1653e-05,
    6.2439e-05, 6.3446e-05, 6.4244e-05, 6.503e-05, 6.7763e-05,
    6.8607e-05, 6.8971e-05, 6.9901e-05, 7.0959e-05, 7.1428e-05,
    7.2114e-05, 7.361e-05, 7.4595e-05, 7.5327e-05, 7.6726e-05,
    7.8059e-05, 7.8583e-05, 7.9104e-05, 7.9599e-05, 8.0018e-05,
    8.1092e-05, 8.1436e-05, 8.3325e-05, 8.3949e-05, 8.5012e-05,
    8.633e-05, 8.6812e-05, 8.7775e-05, 8.9573e-05, 9.0935e-05,
    9.2026e-05, 9.3203e-05, 9.4522e-05, 9.5662e-05, 9.61e-05,
    9.7849e-05, 9.9217e-05, 9.9637e-05, 0.000101461, 0.000102683,
    0.000103715, 0.000104661, 0.000105469, 0.000106373, 0.000107443,
    0.000108663, 0.000109944, 0.000111038, 0.000112832, 0.000113964,
    0.000115699, 0.00011683, 0.000117959, 0.000119312, 0.000119697,
    0.000120544, 0.000121966, 0.000122756, 0.000124028, 0.000124799,
    0.000125608, 0.000128378, 0.000129729, 0.00013131, 0.000133061,
    0.000134293, 0.000135476, 0.00013796, 0.000139381, 0.000140881,
    0.000141841, 0.00014247, 0.000144129, 0.000146627, 0.000148061,
    0.000149751, 0.00015171, 0.000152925, 0.000154548, 0.000156115,
    0.000159157, 0.00016049, 0.000163224, 0.000165589, 0.000166848,
    0.000169405, 0.000171321, 0.000173023, 0.000174959, 0.000177255,
    0.000179062, 0.000181058, 0.000183189, 0.000185692, 0.000187527,
    0.000189032, 0.000190378, 0.000193243, 0.000194526, 0.00019777,
    0.000199371, 0.000201853, 0.000204935, 0.000207285, 0.000208971,
    0.000210847, 0.000212454, 0.000215876, 0.00021846, 0.000220332,
    0.000223133, 0.000226016, 0.000229636, 0.000232914, 0.00023645,
    0.000238326, 0.000241797, 0.000245161, 0.000247618, 0.000250668,
    0.000253731, 0.000257225, 0.00025975, 0.000262678, 0.00026568,
    0.000267737, 0.000269979, 0.000272333, 0.000276164, 0.000279397,
    0.000281898, 0.000283387, 0.000286795, 0.000289575, 0.000293066,
    0.000295805, 0.000300734, 0.000304314, 0.00030848, 0.000311122,
    0.000314662, 0.000317852, 0.000322351, 0.000326085, 0.000329782,
    0.000333962, 0.000337763, 0.000341106, 0.000344523, 0.000348615,
    0.000353732, 0.000359992, 0.000364439, 0.000368165, 0.000373202,
    0.000376621, 0.000380007, 0.000383722, 0.000387424, 0.000392995,
    0.000395967, 0.000400809, 0.00040448, 0.000408966, 0.000415003,
    0.000419893, 0.000425783, 0.000430954, 0.000436316, 0.000441486,
    0.000444828, 0.000449064, 0.000452615, 0.000458091, 0.000462015,
    0.000469942, 0.000474515, 0.000480316, 0.000486574, 0.000492201,
    0.000497833, 0.000502629, 0.000507922, 0.000514454, 0.000518318,
    0.000524265, 0.000530382, 0.000535941, 0.000541999, 0.000546711,
    0.000553182, 0.000560016, 0.000567406, 0.000573708, 0.000580453,
    0.000588058, 0.000594258, 0.000601884, 0.000608639, 0.000613954,
    0.000621157, 0.000626981, 0.000635032, 0.000642225, 0.000647378,
    0.000654467, 0.000660991, 0.000669902, 0.000677885, 0.000685055,
    0.000693242, 0.000702966, 0.000711407, 0.000722574, 0.000728942,
    0.00073585, 0.000744105, 0.000753806, 0.000761157, 0.000772299,
    0.000779454, 0.000790328, 0.000798951, 0.000810843, 0.000820614,
    0.000830988, 0.000839745, 0.000847953, 0.000856775, 0.000865299,
    0.000876475, 0.000888252, 0.000897385, 0.000909719, 0.000923365,
    0.000933342, 0.00094605, 0.000956811, 0.000969173, 0.000976318,
    0.000979332, 0.000993564, 0.001005039, 0.001016103, 0.001025172,
    0.001038597, 0.00104944, 0.001064093, 0.00107921, 0.001091534,
    0.001105862, 0.001118496, 0.001131185, 0.001142954, 0.001158096,
    0.0011722, 0.001186894, 0.001202164, 0.001218249, 0.001232575,
    0.001247647, 0.001265873, 0.001283783, 0.001297789, 0.00131253,
    0.00132636, 0.001340195, 0.001357354, 0.001372434, 0.001388906,
    0.001405144, 0.001424482, 0.001438795, 0.00145831, 0.001472921,
    0.001489218, 0.001506165, 0.001525709, 0.001543268, 0.001561492,
    0.00158057, 0.001596647, 0.001613427, 0.001633854, 0.001653302,
    0.001676213, 0.001697714, 0.001717371, 0.001739668, 0.001758515,
    0.001780935, 0.001802748, 0.001823756, 0.001844841, 0.0018657,
    0.00188685, 0.001908887, 0.001931815, 0.001953135, 0.001975967,
    0.002000222, 0.002026763, 0.002049139, 0.002071201, 0.002096148,
    0.002120228, 0.002145104, 0.002171967, 0.002199421, 0.002222931,
    0.002249921, 0.002275478, 0.002300627, 0.002326402, 0.002353507,
    0.002381964, 0.002407152, 0.002435657, 0.002459124, 0.002485891,
    0.002516222, 0.002548849, 0.002578078, 0.002602483, 0.002633057,
    0.00266164, 0.002699039, 0.002731672, 0.002762611, 0.002790618,
    0.002821572, 0.002856852, 0.002889224, 0.002918868, 0.002952661,
    0.002987048, 0.00301797, 0.003054713, 0.003087561, 0.00312267,
    0.003155237, 0.003192587, 0.003228129, 0.003271984, 0.003314523,
    0.003351523, 0.003388489, 0.003428068, 0.003467536, 0.00351275,
    0.003553854, 0.00359994, 0.003641945, 0.003688275, 0.003732734,
    0.003772217, 0.003813692, 0.003858764, 0.003903562, 0.003955557,
    0.003998601, 0.004042842, 0.00408987, 0.004136763, 0.00418031,
    0.004225705, 0.004272827, 0.004320931, 0.004376629, 0.00442466,
    0.004473494, 0.00452825, 0.004580761, 0.004635671, 0.004691539,
    0.004747212, 0.004801832, 0.0048537, 0.004908977, 0.004966246,
    0.005022952, 0.005080471, 0.005137943, 0.005198803, 0.005259809,
    0.005322021, 0.005381294, 0.005448784, 0.005506109, 0.005569297,
    0.005636612, 0.005709272, 0.005776645, 0.005842622, 0.005907972,
    0.005979773, 0.006050268, 0.006119947, 0.006191937, 0.006257924,
    0.006332057, 0.006397062, 0.006468974, 0.006543425, 0.006621174,
    0.006694334, 0.006767494, 0.006846333, 0.006922691, 0.007001868,
    0.007077291, 0.007156531, 0.007242269, 0.007322086, 0.007406104,
    0.0074938, 0.007583883, 0.00767322, 0.007758872, 0.00785313,
    0.007946915, 0.008041751, 0.00813417, 0.008232993, 0.00832743,
    0.008422489, 0.00850936, 0.008608189, 0.008703262, 0.008812769,
    0.008915303, 0.009013235, 0.009115237, 0.009228099, 0.009338106,
    0.00944284, 0.009553399, 0.009661682, 0.009771256, 0.009893945,
    0.009987961, 0.010011588, 0.010128678, 0.010246869, 0.010362978,
    0.010479596, 0.010597811, 0.010716364, 0.0108441, 0.010970138,
    0.011105286, 0.011233498, 0.011358782, 0.01148587, 0.011620296,
    0.011756402, 0.01189446, 0.01203156, 0.012173215, 0.012310305,
    0.012446006, 0.012598927, 0.012747672, 0.012899352, 0.013041951,
    0.013198545, 0.01336034, 0.013514248, 0.013672629, 0.013833003,
    0.013989302, 0.014157266, 0.014324415, 0.014494696, 0.014654952,
    0.014826421, 0.014993453, 0.01518531, 0.015374356, 0.015552943,
    0.015732018, 0.015910393, 0.016101102, 0.01628202, 0.016474503,
    0.016668722, 0.016870715, 0.017072664, 0.017272478, 0.017465778,
    0.017663856, 0.017868938, 0.018078108, 0.018286096, 0.018502116,
    0.018714076, 0.018933194, 0.019146969, 0.019364799, 0.019603091,
    0.019826448, 0.020048365, 0.020276861, 0.020506979, 0.020730198,
    0.02096665, 0.021216835, 0.021468721, 0.021716815, 0.021963973,
    0.022218165, 0.02249038, 0.022742974, 0.023009418, 0.023278825,
    0.023560105, 0.023836891, 0.024117304, 0.024389161, 0.02467489,
    0.024979292, 0.025279586, 0.025569242, 0.025864336, 0.026171801,
    0.026482004, 0.026786519, 0.027092973, 0.027436527, 0.027745921,
    0.028076733, 0.028403527, 0.028741013, 0.029091065, 0.029428384,
    0.029773884, 0.030118636, 0.030460549, 0.030815852, 0.031187915,
    0.031553373, 0.031926508, 0.032308221, 0.032680967, 0.033059975,
    0.033448324, 0.033845057, 0.034256549, 0.034651379, 0.035067622,
    0.035484032, 0.035912454, 0.036314418, 0.036738958, 0.037162515,
    0.037608517, 0.038045379, 0.038486902, 0.038955459, 0.039418652,
    0.039869928, 0.040317247, 0.0407707, 0.041235997, 0.041720606,
    0.042224011, 0.042726931, 0.043227051, 0.043731151, 0.044241674,
    0.044773434, 0.045292988, 0.045809581, 0.046339434, 0.046874789,
    0.047413108, 0.047965485, 0.048516821, 0.049091405, 0.049663424,
    0.050243036, 0.050813737, 0.051408716, 0.05201184, 0.052609203,
    0.053229173, 0.053845641, 0.054467917, 0.05510297, 0.055759368,
    0.056408494, 0.057058393, 0.057729424, 0.058414888, 0.059116772,
    0.0597981, 0.060490588, 0.061192425, 0.061907969, 0.06263199,
    0.063392313, 0.064143348, 0.064881775, 0.065648476, 0.066409027,
    0.067188155, 0.06797513, 0.068750824, 0.069569587, 0.070403169,
    0.071243789, 0.0720801, 0.072930627, 0.073781569, 0.074651084,
    0.075561893, 0.076449292, 0.077360133, 0.078263329, 0.079187385,
    0.080106821, 0.081054234, 0.082005024, 0.082998005, 0.083978622,
    0.084982081, 0.085987934, 0.086995793, 0.088034154, 0.089091603,
    0.090144015, 0.09119519, 0.092288066, 0.093383155, 0.094475291,
    0.095575296, 0.096716658, 0.09786501, 0.099034795, 0.100186764,
    0.101384212, 0.102353303, 0.102608275, 0.103822879, 0.105084515,
    0.106329987, 0.107601963, 0.108884933, 0.110179284, 0.111489639,
    0.112823351, 0.114184706, 0.115534168, 0.116934916, 0.118312661,
    0.119723719, 0.121179005, 0.122614932, 0.12408028, 0.125607617,
    0.127117844, 0.128675821, 0.130204332, 0.131748261, 0.133347317,
    0.134974832, 0.13660032, 0.138256084, 0.139894704, 0.141567156,
    0.143272747, 0.144988186, 0.146742392, 0.14851702, 0.150300215,
    0.152072053, 0.153883075, 0.155741877, 0.157615018, 0.159539699,
    0.161462968, 0.163447274, 0.165419698, 0.167403487, 0.169465999,
    0.171512491, 0.173594842, 0.175710734, 0.17781881, 0.179995191,
    0.182149252, 0.184372258, 0.186590762, 0.188863674, 0.191145993,
    0.193505482, 0.195826177, 0.198215164, 0.200615415, 0.203025168,
    0.205515597, 0.208023942, 0.21058194, 0.213181505, 0.215771723,
    0.218407288, 0.221053286, 0.223744485, 0.22647246, 0.229265894,
    0.232071778, 0.234900276, 0.237795354, 0.240701792, 0.243667921,
    0.246622937, 0.249648621, 0.252671452, 0.255739059, 0.258869934,
    0.262069996, 0.265331463, 0.26862322, 0.271982608, 0.275396827,
    0.278853574, 0.282297456, 0.285826785, 0.289398895, 0.293013057,
    0.296637045, 0.300384672, 0.304141693, 0.307925098, 0.311758599,
    0.315626989, 0.319586624, 0.323572304, 0.324482584, 0.327639688,
    0.331724196, 0.335898652, 0.340126904, 0.344356883, 0.348713883,
    0.353050529, 0.357533678, 0.362142031, 0.366734177, 0.371380188,
    0.37613222, 0.380915221, 0.385805748, 0.390686515, 0.395647902,
    0.400783906, 0.405924596, 0.411137691, 0.416437348, 0.421763918,
    0.427219507, 0.432745285, 0.438358055, 0.443950978, 0.445098911,
    0.449691877, 0.45551343, 0.461449668, 0.467499555, 0.473617463,
    0.479866822, 0.486134019, 0.492443033, 0.498912301, 0.505458898,
    0.512138571, 0.518810816, 0.525675686, 0.532626972, 0.539662164,
    0.546833215, 0.554171041, 0.561575312, 0.569053974, 0.576679115,
    0.584477513, 0.592267446, 0.600264156, 0.608407889, 0.61661681,
    0.624987398, 0.633488998, 0.642130529, 0.650973956, 0.659904965,
    0.66895086, 0.678146213, 0.687553174, 0.697080788, 0.706725016,
    0.710835446, 0.716522943, 0.726605261, 0.73675373, 0.747041251,
    0.757603248, 0.768271013, 0.779086525, 0.790086954, 0.801467527,
    0.812993436, 0.82463898, 0.836658917, 0.848745115, 0.860990569,
    0.873474269, 0.886261549, 0.899237068, 0.912496999, 0.926019843,
    0.939559171, 0.95349634, 0.967678427, 0.982123815, 0.99683852,
    1.011854472, 1.018074694, 1.027029105, 1.042510483, 1.05832214,
    1.074628134, 1.091323101, 1.108352608, 1.125464103, 1.143092581,
    1.161039627, 1.179357356, 1.198010015, 1.217044686, 1.236438616,
    1.256413992, 1.276822646, 1.297641298, 1.318759204, 1.340466779,
    1.362769541, 1.380813191, 1.421839241, 1.444706794, 1.467757033,
    1.490801155, 1.513604652, 1.53642429, 1.559431719, 1.582306189,
    1.605163235, 1.628119192, 1.651106655, 1.67394735, 1.696747518,
    1.719632869, 1.742427519, 1.765334369, 1.788280658, 1.811235982,
    1.824762899, 1.834066276, 1.857041227, 1.880129096, 1.903151676,
    1.926021295, 1.948868757, 1.971789649, 1.994524594, 2.017493213,
    2.040351274, 2.063197505, 2.086015042, 2.109074118, 2.132020323,
    2.155018084, 2.177747075, 2.200720009, 2.223396652, 2.246136027,
    2.269204726, 2.292060244, 2.314665324, 2.337673977, 2.360687947,
    2.383579603, 2.39678395, 2.406496153, 2.429571463, 2.452444775,
    2.475296436, 2.498075808, 2.520917811, 2.543595992, 2.566592949,
    2.589416066, 2.612389061, 2.63516765, 2.6580917, 2.681159045,
    2.703994885, 2.726822622, 2.74959839, 2.772526291, 2.795447428,
    2.818175682, 2.841298118, 2.864105138, 2.886974869, 2.910062207,
    2.933073042, 2.955872771, 2.97875241, 3.002012321, 3.024900935,
    3.047751948, 3.070702076, 3.093838732, 3.116737233, 3.139536846,
    3.162607839, 3.185366338, 3.203576708, 3.208258924, 3.231243422,
    3.254136622, 3.277409297, 3.300306828, 3.323307927, 3.346111609,
    3.368775711, 3.391500163, 3.414223943, 3.436873624, 3.459895214,
    3.482863538, 3.505974818, 3.529344276, 3.552351023, 3.575651971,
    3.598575047, 3.621358129, 3.644211801, 3.667127522, 3.69023184,
    3.713539776, 3.736469769, 3.759549309, 3.777783051, 3.782804279,
    3.805741046, 3.828673116, 3.851582242, 3.874483158, 3.89746273,
    3.920562307, 3.943494704, 3.966411152, 3.989837916, 4.012506628,
    4.03563262, 4.058749216, 4.081417425, 4.104222863, 4.126966589,
    4.150032311, 4.172942547, 4.195745795, 4.218713402, 4.242094246,
    4.265214382, 4.288523908, 4.311774814, 4.33523561, 4.358333093,
    4.381150221, 4.4041931, 4.427320514, 4.44999237, 4.473256417,
    4.496300112, 4.51941676, 4.542529576, 4.565793943, 4.588838271,
    4.612007785, 4.635021236, 4.658005277, 4.681102818, 4.703868097,
    4.726753352, 4.750079944, 4.772822559, 4.79591235, 4.819273371,
    4.842461023, 4.86585338, 4.889257759, 4.912191388, 4.935330689,
    4.958259575, 4.981911279, 5.004698315, 5.027582433, 5.050379316,
    5.07365985, 5.096505808, 5.119850501, 5.143041386, 5.166591282,
    5.189651594, 5.212705059, 5.23528152, 5.258117232, 5.281225356,
    5.304552059, 5.327354984, 5.349931119, 5.373401472, 5.396584264,
    5.419984303, 5.443216919, 5.46675659, 5.490020154, 5.513843839,
    5.537160747, 5.560406322, 5.583809855, 5.607145928, 5.630320082,
    5.653712015, 5.676892069, 5.700460256, 5.723920246, 5.748147543,
    5.771304051, 5.794657284, 5.818134764, 5.841428812, 5.864939105,
    5.887960256, 5.911108228, 5.934205964, 5.957674208, 5.980635103,
    5.985735184, 6.004379766, 6.027697822, 6.050831796, 6.074846313,
    6.097486343, 6.120615546, 6.144105443, 6.167482109, 6.191192951,
    6.214192013, 6.23791232, 6.261851599, 6.285425359, 6.308658778,
    6.3318006, 6.35539821, 6.379317781, 6.402478033, 6.426629586,
    6.449850911, 6.472757269, 6.496162126, 6.520086402, 6.543911011,
    6.567273118, 6.590330468, 6.613644713, 6.636766776, 6.660457667,
    6.683924627, 6.706662577, 6.72987785, 6.753048332, 6.777293778,
    6.801380579, 6.825213157, 6.849139063, 6.873269298, 6.897617483,
    6.92031197, 6.94323195, 6.966620608, 6.990256641, 7.014038558,
    7.03762782, 7.061594753, 7.084524704, 7.107751459, 7.132080726,
    7.155885644, 7.178844778, 7.202562218, 7.22676135, 7.251069521,
    7.274901836, 7.298602888, 7.321947079, 7.345241594, 7.369204399,
    7.393287777, 7.417370072, 7.440769256, 7.46464573, 7.488501421,
    7.512544493, 7.536186612, 7.559649809, 7.583013899, 7.607231066,
    7.630920846, 7.654824339, 7.679048681, 7.702645031, 7.726330728,
    7.750380717, 7.773797316, 7.796923189, 7.820934791, 7.84420086,
    7.866992807, 7.890810954, 7.914438545, 7.938991149, 7.963538553,
    7.986848288, 8.010409612, 8.034342682, 8.057893432, 8.0814986,
    8.105565027, 8.129491018, 8.153157557, 8.176225511, 8.199988856,
    8.224302502, 8.247244967, 8.270735097, 8.295116731, 8.31887978,
    8.342685312, 8.366776886, 8.391127462, 8.415567057, 8.438853282,
    8.463062293, 8.487153553, 8.512192163, 8.535564365, 8.558782883,
    8.584999974, 8.609331772, 8.633419705, 8.65772375, 8.681375063,
    8.706117403, 8.729982387, 8.753647425, 8.77723426, 8.800306384,
    8.825915771, 8.850230909, 8.875565671, 8.899660467, 8.923478601,
    8.94673013, 8.970247117, 8.99571305, 9.021029926, 9.044338109,
    9.067812183, 9.092436745, 9.116534101, 9.141173439, 9.16581246,
    9.19016182, 9.213718398, 9.238553696, 9.263319158, 9.287780195,
    9.312818417, 9.335650818, 9.360047671, 9.384148898, 9.407878145,
    9.432512292, 9.458497367, 9.483293222, 9.50781117, 9.532396941,
    9.55867118, 9.584239606, 9.610262997, 9.634404604, 9.65890613,
    9.68416484, 9.708085733, 9.731545409, 9.75665308, 9.781612532,
    9.806549653, 9.83164631, 9.855773889, 9.880832429, 9.906461607,
    9.931392805, 9.957818529, 9.982988124, 10.007216475, 10.032968506,
    10.056888974, 10.07924291, 10.102376488, 10.126335862, 10.151872437,
    10.176880172, 10.200285928, 10.22478936, 10.24899299, 10.273258454,
    10.296377443, 10.320424109, 10.345206656, 10.367762036, 10.3918676,
    10.414851575, 10.438671446, 10.462928126, 10.488210843, 10.513115787,
    10.540015512, 10.565615482, 10.590872799, 10.615257098, 10.638928263,
    10.662965975, 10.688630901, 10.711997907, 10.737824537, 10.761477787,
    10.786467629, 10.792231654, 10.812406907, 10.840110326, 10.864351032,
    10.890767298, 10.914839666, 10.941550397, 10.965596399, 10.991601924,
    11.015257328, 11.038812213, 11.063250376, 11.088158985, 11.110969924,
    11.136501029, 11.160050068, 11.185168048, 11.208420564, 11.23449531,
    11.258269627, 11.28333462, 11.308023927, 11.332149442, 11.35674268,
    11.382709595, 11.408635456, 11.434081077, 11.460691115, 11.486886744,
    11.511925957, 11.538227515, 11.561482114, 11.588038383, 11.613615497,
    11.640000906, 11.666570697, 11.691574658, 11.718095227, 11.741774989,
    11.767905802, 11.794080635, 11.8189974, 11.843191574, 11.869897942,
    11.895302369, 11.921198187, 11.945994383, 11.971029875, 11.999356859,
    12.02575441, 12.052458002, 12.078479965, 12.102351674, 12.128215565,
    12.155193375, 12.180918728, 12.206071525, 12.232945321, 12.258985202,
    12.284282214, 12.312340877, 12.334943443, 12.359099381, 12.387912068,
    12.412346753, 12.437384193, 12.464081128, 12.490051813, 12.515095577,
    12.540464028, 12.565115327, 12.590901432, 12.615516028, 12.641644221,
    12.667287805, 12.690342248, 12.716665273, 12.74584328, 12.772823717,
    12.79468774, 12.817717343, 12.844412416, 12.868887861, 12.894763409,
    12.920278037, 12.94797719, 12.974302091, 13.001002582, 13.02991896,
    13.058134468, 13.085473504, 13.113290603, 13.138432615, 13.161466991,
    13.190223232, 13.215796624, 13.238943258, 13.26440621, 13.291255433,
    13.318092951, 13.343115489, 13.369715506, 13.399482127, 13.42590911,
    13.452359664, 13.4787372, 13.506241878, 13.527237724, 13.553870152,
    13.578138904, 13.601917501, 13.631728658, 13.655675974, 13.681566582,
    13.708549454, 13.738020555, 13.759773624, 13.78536036, 13.807242198,
    13.83334189, 13.862074913, 13.88641803, 13.912677314, 13.935211416,
    13.961523516, 13.987809955, 14.016189606, 14.044468375, 14.077597999,
    14.107231333, 14.135385542, 14.162774649, 14.1873333, 14.211855811,
    14.238396169, 14.264868818, 14.292020793, 14.318581268, 14.344092476,
    14.370915605, 14.396728713, 14.424570272, 14.451920358, 14.478384546,
    14.508639244, 14.53039032, 14.5544646, 14.582625673, 14.61192027,
    14.636708016, 14.661911642, 14.686709221, 14.711020832, 14.735690136,
    14.763541904, 14.789784351, 14.813448353, 14.846229171, 14.86830997,
    14.894117104, 14.921738904, 14.955020808, 14.984595065, 15.014557113,
    15.039844056, 15.068733065, 15.095186713, 15.126635765, 15.156778159,
    15.181193589, 15.206289764, 15.232388602, 15.25468962, 15.282240409,
    15.305502241, 15.33381819, 15.358781265, 15.383394818, 15.408615071,
    15.438348368, 15.466865177, 15.494027617, 15.527721318, 15.556087058,
    15.580720574, 15.610593233, 15.637725393, 15.667592105, 15.695452943,
    15.728039465, 15.755593351, 15.782846731, 15.814585652, 15.845362251,
    15.872293979, 15.901956006, 15.931442266, 15.95630612, 15.984759421,
    16.015721082, 16.04144634, 16.05164719, 16.067132169, 16.090129135,
    16.113178777, 16.143404946, 16.172407612, 16.204191332, 16.236950768,
    16.268898532, 16.296974092, 16.322992924, 16.338979209, 16.361586192,
    16.388674794, 16.416639129, 16.452999313, 16.478163633, 16.50349834,
    16.528266993, 16.557166262, 16.577620842, 16.603543288, 16.630289228,
    16.65782226, 16.687659026, 16.717200793, 16.749410579, 16.771644772,
    16.793892317, 16.830283961, 16.855870419, 16.883081266, 16.910815692,
    16.944913092, 16.975130468, 17.002275256, 17.035655385, 17.061144001,
    17.09076142, 17.118666514, 17.14933933, 17.174606576, 17.209864013,
    17.240932162, 17.269894885, 17.297863787, 17.320020753, 17.345718806,
    17.366174563, 17.385332378, 17.403942865, 17.434170847, 17.464387864,
    17.496301624, 17.52159258, 17.547912196, 17.577732943, 17.596594383,
    17.626801495, 17.647201961, 17.677370291, 17.70462828, 17.751083617,
    17.776368386, 17.797823279, 17.81959781, 17.843597637, 17.862590964,
    17.900599573, 17.934218444, 17.959376549, 17.989139158, 18.024174897,
    18.059488246, 18.086663296, 18.111138854, 18.148657051, 18.180230185,
    18.208105326, 18.235472994, 18.261277404, 18.286004033, 18.315272052,
    18.348611163, 18.383736355, 18.420249564, 18.445069249, 18.468553199,
    18.499833977, 18.53260525, 18.564322782, 18.600765601, 18.630925991,
    18.655744493, 18.682030879, 18.705624299, 18.733848803, 18.759553745,
    18.793948319, 18.824964964, 18.840525753, 18.863035577, 18.888515695,
    18.920156247, 18.947292951, 18.979309166, 19.011339372, 19.039102594,
    19.075241114, 19.094528102, 19.127505157, 19.153849514, 19.176863805,
    19.198475326, 19.22932657, 19.246496105, 19.272057321, 19.297184385,
    19.320840451, 19.343128872, 19.359622668, 19.387155605, 19.420125166,
    19.443364598, 19.477563357, 19.501981701, 19.525499587, 19.561157541,
    19.589036889, 19.608136575, 19.63344721, 19.665400589, 19.683431536,
    19.709911865, 19.752103229, 19.780886616, 19.815262894, 19.853236895,
    19.884251557, 19.918852043, 19.939729968, 19.971313697, 20.003996899,
    20.043244746, 20.080171541, 20.126200331, 20.153193524, 20.178767288,
    20.211899696, 20.245437777, 20.267088983, 20.294749841, 20.316821507,
    20.355951972, 20.388050596, 20.404681547, 20.430763097, 20.462834763,
    20.48217512, 20.527059696, 20.561021129, 20.620512046, 20.641887509,
    20.682123183, 20.713720103, 20.732439408, 20.756800303, 20.781591036,
    20.825271323, 20.841334052, 20.86377364, 20.894405747, 20.924786397,
    20.95031274, 20.987350088, 21.011710295, 21.032226973, 21.058906283,
    21.072841234, 21.13060025, 21.1651364, 21.218803998, 21.271269265,
    21.308874393, 21.337807323, 21.353758961, 21.375514536, 21.401666373,
    21.419324351, 21.448392333, 21.48365833, 21.510465887, 21.562343035,
    21.595330316, 21.638457773, 21.673420571, 21.709197658, 21.728695918,
    21.774307486, 21.787296725, 21.827665014, 21.866367778, 21.895080195,
    21.903843845, 21.921600286, 21.946299211, 21.982785895, 22.039371515,
    22.082036147, 22.101947287, 22.121003254, 22.163009226, 22.184504353,
    22.214185064, 22.222782427, 22.25126491, 22.299124636, 22.317412928,
    22.350775586, 22.399291558, 22.438757012, 22.456301417, 22.499935415,
    22.532784789, 22.60156022, 22.636485459, 22.678030492, 22.69956242,
    22.735447784, 22.777606533, 22.788632485, 22.797931312, 22.837180453,
    22.86596757, 22.888916233, 22.946742326, 22.974604712, 23.003491087,
    23.035161967, 23.097804069, 23.122325995, 23.140576181, 23.184421908,
    23.224538776, 23.287732272, 23.319570439, 23.343978328, 23.38774935,
    23.446583199, 23.453492148, 23.505542654, 23.516144987, 23.562788454,
    23.577253453, 23.582717683, 23.629252236, 23.652089402, 23.681712286,
    23.697473803, 23.722171055, 23.750206714, 23.775301761, 23.805282649,
    23.815443127, 23.848098835, 23.867746378, 23.886847441, 23.90105248,
    23.933918917, 23.964668736, 24.030082728, 24.183934579, 24.190867612,
    24.244196373, 24.284390552, 24.359729962, 24.385272677, 24.448685167,
    24.522437854, 24.527607467, 24.542801663, 24.61371486, 24.675095637,
    24.720456848, 24.725428965, 24.790294908, 24.799948966, 24.800939572,
    24.810143823, 24.848215793, 24.91043489, 24.918278488, 24.935245201,
    24.968281895, 24.984504569, 25.026148115, 25.064518562, 25.116401007,
    25.134021561, 25.139076389, 25.187784998, 25.189022684, 25.213284168,
    25.323710031, 25.401259278, 25.460350133, 25.462677897, 25.471552555,
    25.477670604, 25.529011239, 25.607512567, 25.617872297, 25.625716314,
    25.672753602, 25.67911233, 25.718541815, 25.747157979, 25.761652248,
    25.843885294, 25.890676333, 25.91760945, 25.995443385, 26.034834368,
    26.03634578, 26.043951844, 26.051527127, 26.078781322, 26.096850322,
    26.120832803, 26.190551683, 26.205826895, 26.243900869, 26.265534583,
    26.302539129, 26.329828819, 26.367210457, 26.370845622, 26.385461048,
    26.47064484, 26.522115411, 26.553212865)), .Names = c("5",
"6", "8", "10", "20", "35", "50", "75", "100", "150", "200",
"300", "500", "800", "1000", "1400", "1600", "2400", "10000"), row.names = c("1e-05",
"1.01e-05", "1.02e-05", "1.04e-05", "1.05e-05", "1.06e-05", "1.07e-05",
"1.08e-05", "1.1e-05", "1.11e-05", "1.12e-05", "1.14e-05", "1.15e-05",
"1.16e-05", "1.17e-05", "1.19e-05", "1.2e-05", "1.22e-05", "1.23e-05",
"1.24e-05", "1.26e-05", "1.27e-05", "1.29e-05", "1.3e-05", "1.32e-05",
"1.33e-05", "1.35e-05", "1.36e-05", "1.38e-05", "1.4e-05", "1.41e-05",
"1.43e-05", "1.45e-05", "1.46e-05", "1.48e-05", "1.5e-05", "1.51e-05",
"1.53e-05", "1.55e-05", "1.57e-05", "1.58e-05", "1.6e-05", "1.62e-05",
"1.64e-05", "1.66e-05", "1.68e-05", "1.7e-05", "1.72e-05", "1.74e-05",
"1.76e-05", "1.78e-05", "1.8e-05", "1.82e-05", "1.84e-05", "1.86e-05",
"1.88e-05", "1.91e-05", "1.93e-05", "1.95e-05", "1.97e-05", "2e-05",
"2.02e-05", "2.04e-05", "2.07e-05", "2.09e-05", "2.11e-05", "2.14e-05",
"2.16e-05", "2.19e-05", "2.21e-05", "2.24e-05", "2.26e-05", "2.29e-05",
"2.32e-05", "2.34e-05", "2.37e-05", "2.4e-05", "2.43e-05", "2.45e-05",
"2.48e-05", "2.51e-05", "2.54e-05", "2.57e-05", "2.6e-05", "2.63e-05",
"2.66e-05", "2.69e-05", "2.72e-05", "2.75e-05", "2.79e-05", "2.82e-05",
"2.85e-05", "2.88e-05", "2.92e-05", "2.95e-05", "2.99e-05", "3.02e-05",
"3.05e-05", "3.09e-05", "3.13e-05", "3.16e-05", "3.2e-05", "3.24e-05",
"3.27e-05", "3.31e-05", "3.35e-05", "3.39e-05", "3.43e-05", "3.47e-05",
"3.51e-05", "3.55e-05", "3.59e-05", "3.63e-05", "3.67e-05", "3.72e-05",
"3.76e-05", "3.8e-05", "3.85e-05", "3.89e-05", "3.94e-05", "3.98e-05",
"4.03e-05", "4.07e-05", "4.12e-05", "4.17e-05", "4.22e-05", "4.27e-05",
"4.32e-05", "4.37e-05", "4.42e-05", "4.47e-05", "4.52e-05", "4.57e-05",
"4.62e-05", "4.68e-05", "4.73e-05", "4.79e-05", "4.84e-05", "4.9e-05",
"4.95e-05", "5.01e-05", "5.07e-05", "5.13e-05", "5.19e-05", "5.25e-05",
"5.31e-05", "5.37e-05", "5.43e-05", "5.5e-05", "5.56e-05", "5.62e-05",
"5.69e-05", "5.75e-05", "5.82e-05", "5.89e-05", "5.96e-05", "6.03e-05",
"6.1e-05", "6.17e-05", "6.24e-05", "6.31e-05", "6.38e-05", "6.46e-05",
"6.53e-05", "6.61e-05", "6.68e-05", "6.76e-05", "6.84e-05", "6.92e-05",
"7e-05", "7.08e-05", "7.16e-05", "7.24e-05", "7.33e-05", "7.41e-05",
"7.5e-05", "7.59e-05", "7.67e-05", "7.76e-05", "7.85e-05", "7.94e-05",
"8.04e-05", "8.13e-05", "8.22e-05", "8.32e-05", "8.41e-05", "8.51e-05",
"8.61e-05", "8.71e-05", "8.81e-05", "8.91e-05", "9.02e-05", "9.12e-05",
"9.23e-05", "9.33e-05", "9.44e-05", "9.55e-05", "9.66e-05", "9.77e-05",
"9.89e-05", "1e-04", "0.0001012", "0.0001023", "0.0001035", "0.0001047",
"0.0001059", "0.0001072", "0.0001084", "0.0001096", "0.0001109",
"0.0001122", "0.0001135", "0.0001148", "0.0001161", "0.0001175",
"0.0001189", "0.0001202", "0.0001216", "0.000123", "0.0001245",
"0.0001259", "0.0001274", "0.0001288", "0.0001303", "0.0001318",
"0.0001334", "0.0001349", "0.0001365", "0.000138", "0.0001396",
"0.0001413", "0.0001429", "0.0001445", "0.0001462", "0.0001479",
"0.0001496", "0.0001514", "0.0001531", "0.0001549", "0.0001567",
"0.0001585", "0.0001603", "0.0001622", "0.0001641", "0.000166",
"0.0001679", "0.0001698", "0.0001718", "0.0001738", "0.0001758",
"0.0001778", "0.0001799", "0.000182", "0.0001841", "0.0001862",
"0.0001884", "0.0001905", "0.0001928", "0.000195", "0.0001972",
"0.0001995", "0.0002018", "0.0002042", "0.0002065", "0.0002089",
"0.0002113", "0.0002138", "0.0002163", "0.0002188", "0.0002213",
"0.0002239", "0.0002265", "0.0002291", "0.0002317", "0.0002344",
"0.0002371", "0.0002399", "0.0002427", "0.0002455", "0.0002483",
"0.0002512", "0.0002541", "0.000257", "0.00026", "0.000263",
"0.0002661", "0.0002692", "0.0002723", "0.0002754", "0.0002786",
"0.0002818", "0.0002851", "0.0002884", "0.0002917", "0.0002951",
"0.0002985", "0.000302", "0.0003055", "0.000309", "0.0003126",
"0.0003162", "0.0003199", "0.0003236", "0.0003273", "0.0003311",
"0.000335", "0.0003388", "0.0003428", "0.0003467", "0.0003508",
"0.0003548", "0.0003589", "0.0003631", "0.0003673", "0.0003715",
"0.0003758", "0.0003802", "0.0003846", "0.000389", "0.0003936",
"0.0003981", "0.0004027", "0.0004074", "0.0004121", "0.0004169",
"0.0004217", "0.0004266", "0.0004315", "0.0004365", "0.0004416",
"0.0004467", "0.0004519", "0.0004571", "0.0004624", "0.0004677",
"0.0004732", "0.0004786", "0.0004842", "0.0004898", "0.0004955",
"5e-04", "0.0005012", "0.000507", "0.0005129", "0.0005188", "0.0005248",
"0.0005309", "0.000537", "0.0005433", "0.0005495", "0.0005559",
"0.0005623", "0.0005689", "0.0005754", "0.0005821", "0.0005888",
"0.0005957", "0.0006026", "0.0006095", "0.0006166", "0.0006237",
"0.000631", "0.0006383", "0.0006457", "0.0006531", "0.0006607",
"0.0006683", "0.0006761", "0.0006839", "0.0006918", "0.0006998",
"0.0007079", "0.0007161", "0.0007244", "0.0007328", "0.0007413",
"0.0007499", "0.0007586", "0.0007674", "0.0007762", "0.0007852",
"0.0007943", "0.0008035", "0.0008128", "0.0008222", "0.0008318",
"0.0008414", "0.0008511", "0.000861", "0.000871", "0.000881",
"0.0008913", "0.0009016", "0.000912", "0.0009226", "0.0009333",
"0.0009441", "0.000955", "0.0009661", "0.0009772", "0.0009886",
"0.001", "0.0010116", "0.0010233", "0.0010351", "0.0010471",
"0.0010593", "0.0010715", "0.0010839", "0.0010965", "0.0011092",
"0.001122", "0.001135", "0.0011482", "0.0011614", "0.0011749",
"0.0011885", "0.0012023", "0.0012162", "0.0012303", "0.0012445",
"0.0012589", "0.0012735", "0.0012882", "0.0013032", "0.0013183",
"0.0013335", "0.001349", "0.0013646", "0.0013804", "0.0013964",
"0.0014125", "0.0014289", "0.0014454", "0.0014622", "0.0014791",
"0.0014962", "0.0015136", "0.0015311", "0.0015488", "0.0015668",
"0.0015849", "0.0016032", "0.0016218", "0.0016406", "0.0016596",
"0.0016788", "0.0016982", "0.0017179", "0.0017378", "0.0017579",
"0.0017783", "0.0017989", "0.0018197", "0.0018408", "0.0018621",
"0.0018836", "0.0019055", "0.0019275", "0.0019498", "0.0019724",
"0.0019953", "0.0020184", "0.0020417", "0.0020654", "0.0020893",
"0.0021135", "0.002138", "0.0021627", "0.0021878", "0.0022131",
"0.0022387", "0.0022646", "0.0022909", "0.0023174", "0.0023442",
"0.0023714", "0.0023988", "0.0024266", "0.0024547", "0.0024831",
"0.0025119", "0.002541", "0.0025704", "0.0026002", "0.0026303",
"0.0026607", "0.0026915", "0.0027227", "0.0027542", "0.0027861",
"0.0028184", "0.002851", "0.002884", "0.0029174", "0.0029512",
"0.0029854", "0.00302", "0.0030549", "0.0030903", "0.0031261",
"0.0031623", "0.0031989", "0.0032359", "0.0032734", "0.0033113",
"0.0033497", "0.0033884", "0.0034277", "0.0034674", "0.0035075",
"0.0035481", "0.0035892", "0.0036308", "0.0036728", "0.0037154",
"0.0037584", "0.0038019", "0.0038459", "0.0038905", "0.0039355",
"0.0039811", "0.0040272", "0.0040738", "0.004121", "0.0041687",
"0.004217", "0.0042658", "0.0043152", "0.0043652", "0.0044157",
"0.0044668", "0.0045186", "0.0045709", "0.0046238", "0.0046774",
"0.0047315", "0.0047863", "0.0048417", "0.0048978", "0.0049545",
"0.005", "0.0050119", "0.0050699", "0.0051286", "0.005188", "0.0052481",
"0.0053088", "0.0053703", "0.0054325", "0.0054954", "0.005559",
"0.0056234", "0.0056885", "0.0057544", "0.005821", "0.0058884",
"0.0059566", "0.0060256", "0.0060954", "0.006166", "0.0062373",
"0.0063096", "0.0063826", "0.0064565", "0.0065313", "0.0066069",
"0.0066834", "0.0067608", "0.0068391", "0.0069183", "0.0069984",
"0.0070795", "0.0071614", "0.0072444", "0.0073282", "0.0074131",
"0.0074989", "0.0075858", "0.0076736", "0.0077625", "0.0078524",
"0.0079433", "0.0080353", "0.0081283", "0.0082224", "0.0083176",
"0.008414", "0.0085114", "0.0086099", "0.0087096", "0.0088105",
"0.0089125", "0.0090157", "0.0091201", "0.0092257", "0.0093325",
"0.0094406", "0.0095499", "0.0096605", "0.0097724", "0.0098855",
"0.01", "0.0101158", "0.0102329", "0.0103514", "0.0104713", "0.0105925",
"0.0107152", "0.0108393", "0.0109648", "0.0110917", "0.0112202",
"0.0113501", "0.0114815", "0.0116145", "0.011749", "0.011885",
"0.0120226", "0.0121619", "0.0123027", "0.0124451", "0.0125893",
"0.012735", "0.0128825", "0.0130317", "0.0131826", "0.0133352",
"0.0134896", "0.0136458", "0.0138038", "0.0139637", "0.0141254",
"0.0142889", "0.0144544", "0.0146218", "0.0147911", "0.0149624",
"0.0151356", "0.0153109", "0.0154882", "0.0156675", "0.0158489",
"0.0160325", "0.0162181", "0.0164059", "0.0165959", "0.016788",
"0.0169824", "0.0171791", "0.017378", "0.0175792", "0.0177828",
"0.0179887", "0.018197", "0.0184077", "0.0186209", "0.0188365",
"0.0190546", "0.0192752", "0.0194984", "0.0197242", "0.0199526",
"0.0201837", "0.0204174", "0.0206538", "0.020893", "0.0211349",
"0.0213796", "0.0216272", "0.0218776", "0.0221309", "0.0223872",
"0.0226464", "0.0229087", "0.0231739", "0.0234423", "0.0237137",
"0.0239883", "0.0242661", "0.0245471", "0.0248313", "0.0251189",
"0.0254097", "0.025704", "0.0260016", "0.0263027", "0.0266073",
"0.0269153", "0.027227", "0.0275423", "0.0278612", "0.0281838",
"0.0285102", "0.0288403", "0.0291743", "0.0295121", "0.0298538",
"0.0301995", "0.0305492", "0.030903", "0.0312608", "0.0316228",
"0.031989", "0.0323594", "0.0327341", "0.0331131", "0.0334965",
"0.0338844", "0.0342768", "0.0346737", "0.0350752", "0.0354813",
"0.0358922", "0.0363078", "0.0367282", "0.03