
# This library is free software; you can redistribute it and/or
# modify it under the terms of the GNU Library General Public
# License as published by the Free Software Foundation; either
# version 2 of the License, or (at your option) any later version.
#
# This library is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the 
# GNU Library General Public License for more details.
#
# You should have received a copy of the GNU Library General 
# Public License along with this library; if not, write to the 
# Free Foundation, Inc., 59 Temple Place, Suite 330, Boston, 
# MA  02111-1307  USA

# Copyrights (C)
# for this R-port: 
#   1999 - 2007, Diethelm Wuertz, GPL
#   Diethelm Wuertz <wuertz@itp.phys.ethz.ch>
#   info@rmetrics.org
#   www.rmetrics.org
# for the code accessed (or partly included) from other R-ports:
#   see R's copyright and license files
# for the code accessed (or partly included) from contributed R-ports
# and other sources
#   see Rmetrics's copyright file


################################################################################
# S3 METHODS:            TEST AND REPRESENTATION OF OBJECTS:
#  isWeekday              Tests if a date is a weekday or not
#  isWeekend              Tests if a date falls on a weekend or not
#  isBizday               Tests if a date is a business day or not
#  isHoliday              Tests if a date is a non-business day or not
#  getDayOfWeek           Returns the day of the week to a 'timeDate' object
#  getDayOfYear           Returns the day of the year to a 'timeDate' object
# S3 MEHOD:              SUBSETTING TIMEDATE OBJECTS:
#  [.timeDate             Extracts or replaces subsets from 'timeDate' objects
#  cut.timeDate           Extracts a piece from a 'timeDate' object
#  start.timeDate         Extracts the first entry of a 'timeDate' object
#  end.timeDate           Extracts the last entry of a 'timeDate' object
#  length.timeDate        Gets the length of a 'timeDate' object
#  blockStart             Creates start dates for equally sized blocks
#  blockEnd               Creates end dates for equally sized blocks
################################################################################


################################################################################
# S3 METHODS:            TEST AND REPRESENTATION OF OBJECTS:
#  isWeekday              Tests if a date is a weekday or not
#  isWeekend              Tests if a date falls on a weekend or not
#  isBizday               Tests if a date is a business day or not
#  isHoliday              Tests if a date is a non-business day or not
#  getDayOfWeek           Returns the day of the week to a 'timeDate' object
#  getDayOfYear           Returns the day of the year to a 'timeDate' object


isWeekday = 
function(x) 
{   # A function implemented by Diethelm Wuertz

    # Description:
    #   Test if a date is a weekday day or not
    
    # Arguments:
    #   x - an object of class "timeDate"
    
    # Value:
    #   returns a logical or a vector of logicals
    
    # Example:
    #   isWeekday(timeDate("2004-07-01"))
    #   isWeekday(Sys.timeDate())
    
    # Changes:
    #
    
    # FUNCTION:
    
    # Test for Weekdays:
    wday = as.POSIXlt(x@Data)$wday
    ans = (!(wday == 0 | wday == 6)) 
    names(ans) = x@Data
    
    # Return Value:
    ans
}


# ------------------------------------------------------------------------------

    
isWeekend = 
function(x) 
{   # A function implemented by Diethelm Wuertz

    # Description:
    #   Tests if a date is a weekend day or not
    
    # Arguments:
    #   x - an object of class "timeDate"
    
    # Value:
    #   Returns a logical or a vector of logicals
    
    # Example:
    #   isWeekend(timeDate("2004-07-01"))
    #   isWeekend(Sys.timeDate())
    
    # Changes:
    #
    
    # FUNCTION:
    
    # Return Value:
    return(!isWeekday(x)) 
}   


# ------------------------------------------------------------------------------

    
isBizday = 
function(x, holidays = holidayNYSE()) 
{   # A function implemented by Diethelm Wuertz

    # Description:
    #   Test if a date is a business day or not
    
    # Arguments:
    #   x - an object of class "timeDate"
    #   holidays - a holiday calendar
    
    # Value:
    #   Returns a logical or a vector of logicals
    
    # Example:
    #   x = timeSequence(from = "2005-05-15", to = "2005-07-15")
    #   h = holiday.NYSE(2005)
    #   cbind(as.character(x), is.bizday(x, h))
    
    # Changes:
    #
    
    # FUNCTION:
    
    # Test:
    char.x = substr(as.character(x), 1, 10)
    char.h = substr(as.character(holidays), 1, 10)
    Weekday = as.integer(isWeekday(x))
    nonHoliday = as.integer(!(char.x %in% char.h))
    
    # Business Days:
    bizdays = as.logical(Weekday*nonHoliday)
    names(bizdays) = x@Data
    
    # Return Value:
    bizdays
} 


# ------------------------------------------------------------------------------


isHoliday = 
function(x, holidays = holidayNYSE()) 
{   # A function implemented by Diethelm Wuertz

    # Description:
    #   Test if a date is a holiday or not
    
    # Arguments:
    #   x - an object of class "timeDate"
    #   holidays - a holiday calendar
    
    # Value:
    #   Returns a logical or a vector of logicals

    # Changes:
    #
    
    # FUNCTION:
    
    # Return Value:
    return(!isBizday(x, holidays)) 
}   


# ------------------------------------------------------------------------------


getDayOfWeek =
function(x)
{   # A function implemented by Diethelm Wuertz

    # Description:
    #   Returns day of week for time date objects
    
    # Example:
    #   weekDay(Sys.timeDate())
    #   weekDay(timeSequence("2005-05-15", "2005-07-15"))
    
    # FUNCTION:
    
    # Get Day of Week:
    wd = c("Sun", "Mon", "Tue", "Wed", "Thu", "Fri", "Sat")
    n = as.POSIXlt(x@Data)$wday + 1
    wdays = wd[n]
    names(wdays) = as.character(x@Data)
    
    # Return Value:
    wdays
}    


# ------------------------------------------------------------------------------


getDayOfYear =
function(x)
{   # A function implemented by Diethelm Wuertz

    # Description:
    #   Returns day of week for time date objects
    
    # Changes:
    #
    
    # FUNCTION:
    
    # Assign:
    yd = 1:366
    n = as.POSIXlt(x@Data)$yday + 1
    ydays = yd[n]
    names(ydays) = as.character(x@Data)
    
    # Return Value:
    ydays
}   


################################################################################
# S3 MEHOD:              SUBSETTING TIMEDATE OBJECTS:
#  [.timeDate             Extracts or replaces subsets from 'timeDate' objects
#  cut.timeDate           Extracts a piece from a 'timeDate' object
#  start.timeDate         Extracts the first object of a 'timeDate' object
#  end.timeDate           Extracts the last object of a 'timeDate' object
#  blockStart             Creates start dates for equally sized blocks
#  blockEnd               Creates end dates for equally sized blocks


"[.timeDate" =
function(x, ..., drop = TRUE)
{   # A function implemented by Diethelm Wuertz

    # Description:
    #   Extracts or replaces subsets from 'timeDate' objects
    
    # Arguments:
    #   x - a 'timeDate' object
    
    # Value:
    #   Returns a subset from a 'timeDate' object.
    
    # Changes:
    #
    
    # FUNCTION:
    
    # Set Timezone to GMT:
    myTZ = Sys.getenv("TZ")  
    Sys.setenv(TZ = "GMT")
    
    # Subsets:
    z = as.POSIXlt(x@Data)
    val <- lapply(z, "[", ..., drop = drop)
    attributes(val) <- attributes(z) 
    val = as.POSIXct(val)
    
    # Return Value:
    Sys.setenv(TZ = myTZ)
    new("timeDate", 
        Data = val, 
        Dim = length(as.character(val)),
        format = x@format,
        FinCenter = x@FinCenter)      
}   


# ------------------------------------------------------------------------------


cut.timeDate = 
function(x, from , to, ...)
{   # A function implemented by Diethelm Wuertz

    # Changes:
    #
    
    # FUNCTION:
    
    # Cut:
    X = timeDate(x, zone = x@FinCenter, FinCenter = "GMT")
    FROM = timeDate(from, zone = x@FinCenter, FinCenter = "GMT")
    TO = timeDate(to, zone = x@FinCenter, FinCenter = "GMT")
    test = (X >= FROM & X <= TO)
    ans = timeDate(X[test], zone = "GMT", FinCenter = x@FinCenter)
    
    # Return Value:
    ans
}


# ------------------------------------------------------------------------------


start.timeDate =
function(x, ...)
{   # A function implemented by Diethelm Wuertz

    # Description:
    #   Extracts the first object of a 'timeDate' vector

    # Arguments:
    #   x - a 'timeDate' object
    
    # Value:
    #   Returns from 'x' the earliest entry as an object of class 
    #   'timeDate'.
    
    # Changes:
    #
    
    # FUNCTION:
    
    # Set Timezone to GMT:
    myTZ = Sys.getenv("TZ")  
    Sys.setenv(TZ = "GMT")
    
    # Check Class Type:
    if (!inherits(x, "timeDate")) stop("Wrong class type")
    
    # First element:
    # print(x@FinCenter)
    xGMT = timeDate(x, zone=x@FinCenter, FinCenter="GMT")@Data
    z = as.numeric(as.POSIXct(xGMT))
    order(z)[1]
    
    # Return Value:
    Sys.setenv(TZ = myTZ)
    x[1]
}


# ------------------------------------------------------------------------------


end.timeDate =
function(x, ...)
{   # A function implemented by Diethelm Wuertz

    # Description:
    #   Extracts the last object of a 'timeDate' vector

    # Arguments:
    #   x - a 'timeDate' object
    
    # Value:
    #   Returns an object of class 'timeDate'.
    
    # Changes:
    #
    
    # FUNCTION:
    
    # Set Timezone to GMT:
    myTZ = Sys.getenv("TZ")  
    Sys.setenv(TZ = "GMT")
    
    # Check Class Type:
    if (!inherits(x, "timeDate")) stop("Wrong class type")
    
    # Last element:
    # print(x@FinCenter)
    xGMT = timeDate(x, zone = x@FinCenter, FinCenter = "GMT")@Data
    z = as.numeric(as.POSIXct(xGMT))
    n = order(z)[length(z)]
    
    # Return Value:
    Sys.setenv(TZ = myTZ)
    x[n]
}


# ------------------------------------------------------------------------------


length.timeDate = 
function(x) 
{   # A function implemented by Diethelm Wuertz

    # Description:
    #   Gets the length of a 'timeDate' vector

    # Arguments:
    #   x - a 'timeDate' object
    
    # Value:
    #   Returns the lengths of an object of class 'timeDate'.
    
    # Changes:
    #
    
    # FUNCTION:
    
    # Length:
    ans = length(x@Data)
    
    # Return Value:
    ans
}


# ------------------------------------------------------------------------------


blockStart =
function(x, block = 20)
{   # A function implemented by Diethelm Wuertz

    # Description:
    #   Computes start dates for numeric blocks of dates
    
    # Example:
    #   blockEnd(timeSequence(), block = 30)
    
    # Changes:
    #
    
    # FUNCTION:
    
    # Start Dates of Blocks:
    nx = length(as.character(x))
    fromIdx = seq(1, nx, by = block)
    from = x[fromIdx]
    
    # Return Value:
    from
}

# ------------------------------------------------------------------------------


blockEnd =
function(x, block = 20)
{   # A function implemented by Diethelm Wuertz

    # Description:
    #   Computes start dates for numeric blocks of dates
    
    # Example:
    #   blockEnd(timeSequence(), block = 30)
    
    # Changes:
    #
    
    # FUNCTION:
    
    # End Dates of Blocks:
    nx = length(as.character(x))
    fromIdx = seq(1, nx, by = block)
    toIdx = c(fromIdx[-1]-1, nx)
    to = x[toIdx]
    
    # Return Value:
    to
}


################################################################################

