% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/fviz_silhouette.R
\name{fviz_silhouette}
\alias{fviz_silhouette}
\title{Visualize Silhouette Information from Clustering}
\usage{
fviz_silhouette(sil.obj, label = FALSE, print.summary = TRUE, ...)
}
\arguments{
\item{sil.obj}{an object of class silhouette: pam, clara, fanny [in cluster 
package]; eclust and hcut [in factoextra].}

\item{label}{logical value. If true, x axis tick labels are shown}

\item{print.summary}{logical value. If true a summary of cluster silhouettes 
are printed in fviz_silhouette().}

\item{...}{other arguments to be passed to the function ggpubr::ggpar().}
}
\value{
return a ggplot
}
\description{
Silhouette (Si) analysis is a cluster validation approach that
 measures how well an observation is clustered and it estimates the average
 distance between clusters. fviz_silhouette() provides ggplot2-based elegant
 visualization of silhouette information from i) the result of
 \code{\link[cluster]{silhouette}}(), \code{\link[cluster]{pam}}(),
 \code{\link[cluster]{clara}}() and \code{\link[cluster]{fanny}}() [in
 cluster package]; ii) \code{\link{eclust}}() and \code{\link{hcut}}() [in
 factoextra].
 
 Read more: 
 \href{http://www.sthda.com/english/wiki/clustering-validation-statistics-4-vital-things-everyone-should-know-unsupervised-machine-learning}{Clustering
  Validation Statistics}.
}
\details{
- Observations with a large silhouhette Si (almost 1) are very well 
 clustered.
 
 - A small Si (around 0) means that the observation lies between two
 clusters.
 
 - Observations with a negative Si are probably placed in the wrong cluster.
}
\examples{
set.seed(123)

# Data preparation
# +++++++++++++++
data("iris")
head(iris)
# Remove species column (5) and scale the data
iris.scaled <- scale(iris[, -5])

# K-means clustering
# +++++++++++++++++++++
km.res <- kmeans(iris.scaled, 3, nstart = 2)

# Visualize kmeans clustering
fviz_cluster(km.res, iris[, -5], ellipse.type = "norm")+
theme_minimal()

# Visualize silhouhette information
require("cluster")
sil <- silhouette(km.res$cluster, dist(iris.scaled))
fviz_silhouette(sil)

# Identify observation with negative silhouette
neg_sil_index <- which(sil[, "sil_width"] < 0)
sil[neg_sil_index, , drop = FALSE]
\dontrun{
# PAM clustering
# ++++++++++++++++++++
require(cluster)
pam.res <- pam(iris.scaled, 3)
# Visualize pam clustering
fviz_cluster(pam.res, ellipse.type = "norm")+
theme_minimal()
# Visualize silhouhette information
fviz_silhouette(pam.res)

# Hierarchical clustering
# ++++++++++++++++++++++++
# Use hcut() which compute hclust and cut the tree
hc.cut <- hcut(iris.scaled, k = 3, hc_method = "complete")
# Visualize dendrogram
fviz_dend(hc.cut, show_labels = FALSE, rect = TRUE)
# Visualize silhouhette information
fviz_silhouette(hc.cut)
}
}
\seealso{
\code{\link{fviz_cluster}}, \code{\link{hcut}}, 
 \code{\link{hkmeans}},  \code{\link{eclust}}, \code{\link{fviz_dend}}
}
\author{
Alboukadel Kassambara \email{alboukadel.kassambara@gmail.com}
}
