% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/GenerateBoxData.R
\name{GenerateBoxData}
\alias{GenerateBoxData}
\title{Generate Thurstone's Box Data From length, width, and height box measurements}
\usage{
GenerateBoxData(
  XYZ,
  BoxStudy = 20,
  Reliability = 0.75,
  ModApproxErrVar = 0.1,
  SampleSize = NULL,
  NMinorFac = 50,
  epsTKL = 0.2,
  Seed = 1,
  SeedErrorFactors = 2,
  SeedMinorFactors = 3,
  PRINT = FALSE,
  LB = FALSE,
  LBVal = 1,
  Constant = 0
)
}
\arguments{
\item{XYZ}{(Matrix) Length, width, and height measurements for N boxes.   The Amazon Box data 
can be accessed by calling \code{data(AmxBoxes)}. The Thurstone Box data (20 hypothetical boxes) 
can be accessed by calling \code{data(Thurstone20Boxes)}.}

\item{BoxStudy}{(Integer) If BoxStudy = 20 then data will be generated for 
Thurstone's classic 20 variable box problem. If BoxStudy = 26 then data will 
be generated for Thurstone's 26 variable box problem. Default: \code{BoxStudy = 20}.}

\item{Reliability}{(Scalar [0, 1] ) The common reliability value for each 
measured variable. Default: Reliability = .75.}

\item{ModApproxErrVar}{(Scalar [0, 1] ) The proportion of reliable 
variance (for each variable) that is due to all minor common factors. 
Thus, if \code{x} (i.e., error free length) has variance \code{var(x)} and  
 \code{ModApproxErrVar = .10}, then \code{var(e.ma)/var(x + e.ma) = .10}.}

\item{SampleSize}{(Integer) Specifies the number of boxes to be sampled from 
the population.  If \code{SampleSize = NULL} then measurements will be 
generated for the original input box sizes.}

\item{NMinorFac}{(Integer) The number of minor factors to use while 
generating model approximation error. Default: \code{NMinorFac = 50.}}

\item{epsTKL}{(Numeric [0, 1])  A parameter of the 
Tucker, Koopman, and Linn (1969) algorithm that controls the spread of the influence of the minor factors.
Default: \code{epsTKL = .20.}}

\item{Seed}{(Integer)  Starting seed for box sampling.}

\item{SeedErrorFactors}{(Integer)  Starting seed for the error-factor scores.}

\item{SeedMinorFactors}{(Integer)  Starting seed for the minor common-factor scores.}

\item{PRINT}{(Logical) If PRINT = TRUE then the computed reliabilites will 
be printed. Default: \code{PRINT = FALSE}.  Setting \code{PRINT} to TRUE can be useful 
when \code{LB = TRUE}.}

\item{LB}{(lower bound; logical)  If LB = TRUE then minimum box measurements will be 
set to LBVal (inches) if they
fall below 0 after adding measurement error. If LB = FALSE then negative 
attribute values will not be modified. This argument has no effect
on data that include model approximation error.}

\item{LBVal}{(Numeric) If \code{LB = TRUE} then values in \code{BoxDataE} will be bounded 
from below at \code{LBVal}.  This can be used to avoid negative or very small box 
measurements.}

\item{Constant}{(Numeric) Optional value to add to all box measurements. 
Default: \code{Constant = 0}.}
}
\value{
\itemize{
  \item \strong{XYZ} The length (x), width (y), and height (z) measurements for the sampled boxes. 
         If \code{SampleSize = NULL} then \code{XYZ} contains the x, y, z values for the 
         original 98 boxes.
  \item \strong{BoxData} Error free box measurements.  
  \item \strong{BoxDataE} Box data with added measurement error. 
  \item \strong{BoxDataEME}  Box data with added (reliable) model approximation and (unreliable) measurement error.
  \item \strong{Rel.E}  Classical reliabilities for the scores in \code{BoxDataE}.
  \item \strong{Rel.EME}  Classical reliabilities for the scores in \code{BoxDataEME}.
  \item \strong{NMinorFac}  Number of minor common factors used to generate \code{BoxDataEME}.
  \item \strong{epsTKL}  Minor factor spread parameter for the Tucker, Koopman, Linn algorithm.
  \item \strong{SeedErrorFactors}  Starting seed for the error-factor scores.
  \item \strong{SeedMinorFactors} Starting seed for the minor common-factor scores. 
  }
}
\description{
Generate data for Thurstone's 20 variable and 26 variable Box Study From length, width, and height box measurements.
}
\details{
This function can be used with the Amazon boxes dataset (\code{data(AmzBoxes)}) or with any collection  
of user-supplied scores on three variables. The Amazon Boxes data were downloaded from the 
\code{BoxDimensions} website: (\url{https://www.boxdimensions.com/}). These data contain 
length (x),  width (y), and height (z) measurements for 98 Amazon shipping boxes.  In his 
classical monograph on Multiple Factor Analysis (Thurstone, 1947) Thurstone describes two data sets 
(one that he created from fictitious data and a second data set that he created from actual box measurements) 
that  were used to illustrate topics in factor analysis. The  first (fictitious) data set is 
 known as the Thurstone Box problem (see Kaiser and Horst, 1975).  To create his data for the Box problem, 
 Thurstone constructed 20 nonlinear combinations of fictitious length, width, and height measurements. 
 \strong{Box20} variables:
\enumerate{
     \item   x^2
     \item   y^2
     \item   z^2
     \item   xy
     \item   xz
     \item   yz
     \item   sqrt(x^2 + y^2)
     \item   sqrt(x^2 + z^2)
     \item   sqrt(y^2 + z^2)
     \item   2x + 2y
     \item   2x + 2z
     \item   2y + 2z 
     \item   log(x)
     \item   log(y)
     \item   log(z)
     \item   xyz
     \item   sqrt(x^2 + y^2 + z^2)
     \item   exp(x)
     \item   exp(y)
     \item   exp(z)
   }
   
 The second Thurstone Box problem contains measurements on the following 26 functions of length, width, and height. 
  \strong{Box26} variables:
  \enumerate{
  \item x
  \item y
  \item z
  \item xy 
  \item xz
  \item yz 
  \item x^2 * y
  \item x * y^2
  \item x^2 * z
  \item x * z^ 2
  \item y^2 * z
  \item y * z^2
  \item x/y
  \item y/x
  \item x/z
  \item  z/x
  \item  y/z
  \item  z/y
  \item 2x + 2y
  \item 2x + 2z
  \item 2y + 2z
  \item sqrt(x^2 + y^2)
  \item sqrt(x^2 + z^2)
  \item sqrt(y^2 + z^2)
  \item xyz
  \item sqrt(x^2 + y^2 + z^2)
  }

Note that when generating unreliable data (i.e., variables with 
reliability values less than 1) and/or data with model error, 
\strong{SampleSize} must be greater than \strong{NMinorFac}.
}
\examples{

  data(AmzBoxes)
  BoxList <- GenerateBoxData (XYZ = AmzBoxes[,2:4],
                              BoxStudy = 20,  
                              Reliability = .75,
                              ModApproxErrVar = .10,
                              SampleSize = 300, 
                              NMinorFac = 50,
                              epsTKL = .20,
                              Seed = 1,
                              SeedErrorFactors = 1,
                              SeedMinorFactors = 2,
                              PRINT = FALSE,
                              LB = FALSE,
                              LBVal = 1,
                              Constant = 0)
                                
   BoxData <- BoxList$BoxData
   
   RBoxes <- cor(BoxData)
   fout <- faMain(R = RBoxes,
                 numFactors = 3,
                 facMethod = "fals",
                 rotate = "geominQ",
                 rotateControl = list(numberStarts = 100,
                                      standardize = "CM")) 
                                      
  summary(fout)  
}
\references{
Cureton, E. E. & Mulaik, S. A. (1975). The weighted varimax rotation and the 
promax rotation. Psychometrika, 40(2), 183-195. 
Kaiser, H. F. and Horst, P.  (1975).  A score matrix for Thurstone's box problem.  
Multivariate Behavioral Research, 10(1), 17-26.  

Thurstone, L. L.  (1947).  Multiple Factor Analysis.  Chicago: 
University of Chicago Press. 

Tucker, L. R., Koopman, R. F., and Linn, R. L.  (1969).  Evaluation of factor 
analytic research procedures by means of simulated correlation matrices. 
\emph{Psychometrika, 34}(4), 421-459.
}
\seealso{
Other Factor Analysis Routines: 
\code{\link{BiFAD}()},
\code{\link{Box26}},
\code{\link{Ledermann}()},
\code{\link{SLi}()},
\code{\link{SchmidLeiman}()},
\code{\link{faAlign}()},
\code{\link{faEKC}()},
\code{\link{faIB}()},
\code{\link{faMB}()},
\code{\link{faMain}()},
\code{\link{faScores}()},
\code{\link{faSort}()},
\code{\link{faStandardize}()},
\code{\link{faX}()},
\code{\link{fals}()},
\code{\link{fapa}()},
\code{\link{fareg}()},
\code{\link{orderFactors}()},
\code{\link{print.faMB}()},
\code{\link{print.faMain}()},
\code{\link{promaxQ}()},
\code{\link{summary.faMB}()},
\code{\link{summary.faMain}()}
}
\author{
Niels G. Waller (nwaller@umn.edu)
}
\concept{Factor Analysis Routines}
