##' Compute convex hull of intersection of two sets of points
##' @param ps1 First set of points
##' @param ps2 Second set of points
##' @param tol Tolerance used to determine if a feasible point lies
##'   within the convex hulls of both points and to round off the
##'   points generated by the halfspace intersection, which sometimes
##'   produces points very close together.
##' @param return.chs If \code{TRUE} (default) return the convex hulls
##'   of the first and second sets of points, as well as the convex
##'   hull of the intersection.
##' @return List containing named elements: \code{ch1}, the convex
##'   hull of the first set of points, with volumes, areas and normals
##'   (see \code{\link{convhulln}}; \code{ch2}, the convex hull of the
##'   first set of points, with volumes, areas and normals; \code{ps},
##'   the intersection points of convex hulls \code{ch1} and
##'   \code{ch2}; and \code{ch}, the convex hull of the intersection
##'   points, with volumes, areas and normals.
##' @export
##' @examples
##' # Two overlapping boxes
##' ps1 <- rbox(0, C=0.5)
##' ps2 <- rbox(0, C=0.5) + 0.5
##' out <- intersectn(ps1, ps2)
##' message(paste("Volume of 1st convex hull:", out$ch1$vol))
##' message(paste("Volume of 2nd convex hull:", out$ch2$vol))
##' message(paste("Volume of intersection convex hull:", out$ch$vol))
##' @author David Sterratt
##' @seealso convhulln, halfspacen, inhulln
intersectn <- function(ps1, ps2, tol=0, return.chs=TRUE) {
  distinct <-
    any(apply(ps1, 2, min) > apply(ps2, 2, max)) ||
    any(apply(ps1, 2, max) < apply(ps2, 2, min))
  
  if (distinct & !return.chs) {
    return(list(ch=list(vol=0)))
  }
  
  ch1 <- convhulln(ps1, "n FA")
  ch2 <- convhulln(ps2, "n FA")

  if (distinct) {
    return(list(ch1=ch1, ch2=ch2, ch=list(vol=0)))
  }
  
  ## Find feasible point in which points could overlap
  fp <- feasible.point(ch1, ch2, tol=tol)
  if (all(is.na(fp))) {
    if (return.chs) {
      return(list(ch1=ch1, ch2=ch2, ch=list(vol=0)))
    }
    return(list(ch=list(vol=0)))
  }
  
  ## Find intesections of halfspaces about feasible point. Catch error
  ## when fixed point is not in intersection, due to precision issue.
  ps <- tryCatch(halfspacen(rbind(ch1$normals, ch2$normals), fp),
                 error=function(e) {
                   if (grepl("QH6023", e$message)) {
                     return(NA)
                   }
                   stop(e)
                 })
  if (all(is.na(ps)) || is.null(ps)) {
    if (return.chs) {
      return(list(ch1=ch1, ch2=ch2, ch=list(vol=0)))
    }
    return(list(ch=list(vol=0)))
  }
  
  ## Occasionally the halfspace creates points very close together. We
  ## can impose a tolerance to merge them
  if (tol != 0) {
    ps <- round(ps, ceiling(-log10(abs(tol/2))))
    ps <- ps[!duplicated(ps),]
  }
  ch <- convhulln(ps, "n FA")

  if (return.chs) {
    out <- list(ch1=ch1, ch2=ch2, ps=ps, ch=ch)
    class(out) <- "intersectn"
    return(out)
  }
  return(list(ch=ch))
}

##' Find point in intersection of convex hulls
##'
##' Find point that lies somewhere in interesction of two convex
##' hulls. If such a point does not exist, return \code{NA}. The
##' feasible point is found using a linear program similar to the one
##' suggested at \url{../doc/qhull/html/qhalf.html#notes}
##' 
##' @param ch1 First convex hull with normals
##' @param ch2 Second convex hull with normals
##' @param tol The point must be at least this far within the facets
##'   of both convex hulls
##' @export
feasible.point <- function(ch1, ch2, tol=0) {
  debug <- FALSE
  N <- ncol(ch1$p)

  objective.in <- c(rep(0, N), 1)
  const.mat <- round(rbind(cbind(ch1$normals[,-(N + 1)], 1),
                           cbind(ch2$normals[,-(N + 1)], 1),
                           c(rep(0, N), -1)), 6)
  const.rhs <- -c(ch1$normals[,N + 1], ch2$normals[,N + 1], tol)
  const.dir <- c(rep("<", length(const.rhs)))
  
  opt <- lpSolve::lp(direction = "max", objective.in, const.mat, const.dir, const.rhs)
  if ((opt$status == 2) || (opt$solution[N+1] == 0)) return(NA)
  return(opt$solution[1:N])
}

##' @method plot intersectn
##' @export 
plot.intersectn <- function(x, y, ...) {
  args <- list(...)
  add <- FALSE
  if ("add" %in% names(args)) {
    add <- args$add
    args$add <- NULL
  }
  xlim <- ylim <- NULL
  if ("xlim" %in% names(args)) {
    xlim <- args$xlim
    args$xlim <- NULL
  }
  if ("ylim" %in% names(args)) {
    ylim <- args$ylim
    args$xlim <- NULL
  }
  if (ncol(x$p) == 2) {
    if (!add) {
      p <- rbind(x$ch1$p, x$ch2$p)
      if (is.null(xlim)) xlim <- range(p[,1])
      if (is.null(ylim)) ylim <- range(p[,2])
      plot.new()
      do.call(plot.window, c(list(xlim=xlim, ylim=ylim), args))
    }
    plot(x$ch1, add=TRUE, lty=2)
    plot(x$ch2, add=TRUE, lty=2)
    plot(x$ch, add=TRUE, lwd=2)
  }
}

