% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/combine.subsets.R
\name{combine.subsets}
\alias{combine.subsets}
\title{Combine separate landmark configurations}
\usage{
combine.subsets(..., gpa = TRUE, CS.sets = NULL)
}
\arguments{
\item{...}{Class gpagen objects, Procrustes coordinates from class gpagen objects, or original landmarks.
As many data sets as desired can be supplied, separated by commas.  Additionally, arguments passed onto 
\code{\link{gpagen}} can be provided, but these arguments will be passed onto all GPAs performed.  Therefore,
it is recommended that GPA is performed first with \code{\link{gpagen}}, to maintain flexibility.  Naming
subsets is a good idea, as landmark names in the combined data set will take the subset name as a precursor.}

\item{gpa}{A logical argument to indicate if either (1) GPA should be performed (if original landmarks
are provided) or (2) \code{gpagen} objects are provided.  If TRUE, this function will check to see if 
the input is an object of class \code{gpagen}, and if not, perform GPA.  If FALSE, landmarks will be unchanged.
(One would choose gpa = FALSE if inputting aligned coordinates and centroid size, separately.  There might be
little reason to do this, unless one wishes to intentionally not scale configurations.)}

\item{CS.sets}{A list, array, or matrix of centroid sizes to use for scaling.  The default is NULL and should be
left so if gpa = TRUE.  If gpa = FALSE and CS.set is null, all centroid sizes become 1.0, meaning no scaling 
of configurations by relative size is performed.  If gpa = FALSE and CS.set is provided, scaling by relative 
size is performed according to the data input  (One could weight configurations via this method.).  If the 
CS.set input is a matrix, it is assumed that rows are specimens and columns correspond to the different landmark
sets.  Lists or arrays should be in the same order as the landmark sets.}
}
\value{
An object of class \code{combined.set} is a list containing the following
\item{cords}{An [p x k x n] array of scaled, concatenated landmark coordinates.}
\item{CS}{A matrix of columns representing original centroid sizes of subsets,
either input or found via GPA.}
\item{GPA}{If gpa = TRUE, the gpagen results for each subset.}
\item{gpa.coords.by.set}{A list of the coordinates following GPA for each subset.}
\item{adj.coords.by.set}{A list of the coordinates of each subset, after rescaling.}
\item{points.by.set}{A vector of the number of landmarks in each subset.}
}
\description{
Combine separate landmark configurations (subsets) into one landmark set
}
\details{
This function combines landmark configurations (either landmarks requiring GPA
or Procrustes coordinates following GPA) to create a different morphological data set.  
This might be of interest, for example, if one has landmarks digitized on separate images
collected from the same organisms.  (In the examples below, configurations for heads and tails
of larval salamanders were collected separately from images taken on the same individuals.)  An
attempt is made to scale configurations by their relative centroid sizes, following the procedure in
Davis et al. (2016); i.e., landmark coordinates are multiplied by CSi/(CSi + CSj + ...) before 
combining them, so that resulting combinations of landmarks are scaled to unit centroid size.  This is
only possible if GPA is performed on landmarks (gpa = TRUE) or centroid sizes are provided as an 
argument.  Objects of class \code{gpagen} can be used rather than original landmarks (recommended, 
especially if curves or surface sliding semilandmarks are used, as different arguments cannot be passed onto
onto separate GPAs via this function).

The procedure of Davis et al. (2016) is analogous to the "seperate subsets" method of Adams (1999)
for articulated structures.
}
\examples{
data(larvalMorph) 
head.gpa <- gpagen(larvalMorph$headcoords, curves = larvalMorph$head.sliders)
tail.gpa <- gpagen(larvalMorph$tailcoords, curves = larvalMorph$tail.sliders)
comb.lm <- combine.subsets(head = head.gpa, tail = tail.gpa, gpa = TRUE)
summary(comb.lm)
# Plot first specimen and color code landmarks 
# (configurations are actual relative size)
comb.lm$coords[,,1]
plot(comb.lm$coords[,,1], pch = 21, bg = c(rep(1,26), rep(2,64)), asp = 1)

# choose to override scaling by relative size 
comb.lm <- combine.subsets(head = head.gpa$coords, 
tail = tail.gpa$coords, gpa = FALSE, CS.sets = NULL)
summary(comb.lm)
plot(comb.lm$coords[,,1], pch = 21, bg = c(rep(1,26), rep(2,64)), asp = 1)
}
\references{
Davis, M.A., M.R. Douglas, M.L. Collyer, & M.E. Douglas, M. E. 2016.
 Deconstructing a species-complex: geometric morphometric and molecular analyses define species in the 
 Western Rattlesnake (Crotalus viridis). PloS one, 11(1), e0146166.

Adams, D. C. 1999. Methods for shape analysis of landmark data from articulated structures. 
 Evolutionary Ecology Research. 1:959-970.
}
\author{
Michael Collyer
}
\keyword{utilities}
