#'
#' H2O Model Related Functions
#'

# ------------------------------- Helper Functions --------------------------- #
# Used to verify data, x, y and turn into the appropriate things
.verify_dataxy <- function(data, x, y, autoencoder = FALSE) {
  if(!is(data,  "H2OFrame"))
    stop('`data` must be an H2OFrame object')
  if(!is.character(x) && !is.numeric(x))
    stop('`x` must be column names or indices')
  if( !autoencoder )
    if(!is.character(y) && !is.numeric(y))
      stop('`y` must be a column name or index')

  cc <- colnames(data)

  if(is.character(x)) {
    if(!all(x %in% cc))
      stop("Invalid column names: ", paste(x[!(x %in% cc)], collapse=','))
    x_i <- match(x, cc)
  } else {
    if(any( x < 1L | x > length(cc)))
      stop('out of range explanatory variable ', paste(x[x < 1L | x > length(cc)], collapse=','))
    x_i <- x
    x <- cc[x_i]
  }

  x_ignore <- c()
  if( !autoencoder ) {
    if(is.character(y)){
      if(!(y %in% cc))
        stop(y, ' is not a column name')
      y_i <- which(y == cc)
    } else {
      if(y < 1L || y > length(cc))
        stop('response variable index ', y, ' is out of range')
      y_i <- y
      y <- cc[y]
    }

    if(!autoencoder && (y %in% x)) {
      warning('removing response variable from the explanatory variables')
      x <- setdiff(x,y)
    }
    x_ignore <- setdiff(setdiff(cc, x), y)
    if( length(x_ignore) == 0L ) x_ignore <- ''
    return(list(x=x, y=y, x_i=x_i, x_ignore=x_ignore, y_i=y_i))
  } else {
    x_ignore <- setdiff(cc, x)
    if( !missing(y) ) stop("`y` should not be specified for autoencoder=TRUE, remove `y` input")
    return(list(x=x,x_i=x_i,x_ignore=x_ignore))
  }
}

.verify_datacols <- function(data, cols) {
  if(!is(data, "H2OFrame"))
    stop('`data` must be an H2OFrame object')
  if(!is.character(cols) && !is.numeric(cols))
    stop('`cols` must be column names or indices')

  cc <- colnames(data)
  if(length(cols) == 1L && cols == '')
    cols <- cc
  if(is.character(cols)) {
    if(!all(cols %in% cc))
      stop("Invalid column names: ", paste(cols[which(!cols %in% cc)], collapse=", "))
    cols_ind <- match(cols, cc)
  } else {
    if(any(cols < 1L | cols > length(cc)))
      stop('out of range explanatory variable ', paste(cols[cols < 1L | cols > length(cc)], collapse=','))
    cols_ind <- cols
    cols <- cc[cols_ind]
  }

  cols_ignore <- setdiff(cc, cols)
  if( length(cols_ignore) == 0L )
    cols_ignore <- ''
  list(cols=cols, cols_ind=cols_ind, cols_ignore=cols_ignore)
}

.build_cm <- function(cm, actual_names = NULL, predict_names = actual_names, transpose = TRUE) {
  categories <- length(cm)
  cf_matrix <- matrix(unlist(cm), nrow=categories)
  if(transpose)
    cf_matrix <- t(cf_matrix)

  cf_total <- apply(cf_matrix, 2L, sum)
  cf_error <- c(1 - diag(cf_matrix)/apply(cf_matrix,1L,sum), 1 - sum(diag(cf_matrix))/sum(cf_matrix))
  cf_matrix <- rbind(cf_matrix, cf_total)
  cf_matrix <- cbind(cf_matrix, round(cf_error, 3L))

  if(!is.null(actual_names))
    dimnames(cf_matrix) = list(Actual = c(actual_names, "Totals"), Predicted = c(predict_names, "Error"))
  cf_matrix
}




.h2o.startModelJob <- function(conn = h2o.getConnection(), algo, params, h2oRestApiVersion = .h2o.__REST_API_VERSION) {
  .key.validate(params$key)
  #---------- Force evaluate temporary ASTs ----------#
  ALL_PARAMS <- .h2o.__remoteSend(conn, method = "GET", .h2o.__MODEL_BUILDERS(algo), h2oRestApiVersion = h2oRestApiVersion)$model_builders[[algo]]$parameters

  params <- lapply(params, function(x) {if(is.integer(x)) x <- as.numeric(x); x})
  #---------- Check user parameter types ----------#
  error <- lapply(ALL_PARAMS, function(i) {
    e <- ""
    if (i$required && !(i$name %in% names(params)))
      e <- paste0("argument \"", i$name, "\" is missing, with no default\n")
    else if (i$name %in% names(params)) {
      # changing Java types to R types
      mapping <- .type.map[i$type,]
      type    <- mapping[1L, 1L]
      scalar  <- mapping[1L, 2L]
      if (is.na(type))
        stop("Cannot find type ", i$type, " in .type.map")
      if (scalar) { # scalar == TRUE
        if (type == "H2OModel")
            type <-  "character"
        if (!inherits(params[[i$name]], type))
          e <- paste0("\"", i$name , "\" must be of type ", type, ", but got ", class(params[[i$name]]), ".\n")
        else if ((length(i$values) > 1L) && !(params[[i$name]] %in% i$values)) {
          e <- paste0("\"", i$name,"\" must be in")
          for (fact in i$values)
            e <- paste0(e, " \"", fact, "\",")
          e <- paste(e, "but got", params[[i$name]])
        }
        if (inherits(params[[i$name]], 'numeric') && params[[i$name]] ==  Inf)
          params[[i$name]] <<- "Infinity"
        else if (inherits(params[[i$name]], 'numeric') && params[[i$name]] == -Inf)
          params[[i$name]] <<- "-Infinity"
      } else {      # scalar == FALSE
        k = which(params[[i$name]] == Inf | params[[i$name]] == -Inf)
        if (length(k) > 0)
          for (n in k)
            if (params[[i$name]][n] == Inf)
              params[[i$name]][n] <<- "Infinity"
            else
              params[[i$name]][n] <<- "-Infinity"
        if (!inherits(params[[i$name]], type))
          e <- paste0("vector of ", i$name, " must be of type ", type, ", but got ", class(params[[i$name]]), ".\n")
        else if (type == "character")
          params[[i$name]] <<- .collapse.char(params[[i$name]])
        else
          params[[i$name]] <<- .collapse(params[[i$name]])
      }
    }
    e
  })

  if(any(nzchar(error)))
    stop(error)

  #---------- Create parameter list to pass ----------#
  param_values <- lapply(params, function(i) {
    if(is(i, "H2OFrame"))
      i@frame_id
    else
      i
  })

  #---------- Validate parameters ----------#
  validation <- .h2o.__remoteSend(conn, method = "POST", paste0(.h2o.__MODEL_BUILDERS(algo), "/parameters"), .params = param_values, h2oRestApiVersion = h2oRestApiVersion)
  if(length(validation$messages) != 0L) {
    error <- lapply(validation$messages, function(i) {
      if( i$message_type == "ERROR" )
        paste0(i$message, ".\n")
      else ""
    })
    if(any(nzchar(error))) stop(error)
    warn <- lapply(validation$messages, function(i) {
      if( i$message_type == "WARN" )
        paste0(i$message, ".\n")
      else ""
    })
    if(any(nzchar(warn))) warning(warn)
  }

  #---------- Build! ----------#
  res <- .h2o.__remoteSend(conn, method = "POST", .h2o.__MODEL_BUILDERS(algo), .params = param_values, h2oRestApiVersion = h2oRestApiVersion)

  job_key  <- res$job$key$name
  dest_key <- res$job$dest$name

  new("H2OModelFuture",conn=conn, job_key=job_key, model_id=dest_key)
}

.h2o.createModel <- function(conn = h2o.getConnection(), algo, params, h2oRestApiVersion = .h2o.__REST_API_VERSION) {
 params$training_frame <- get("training_frame", parent.frame())
 tmp_train <- !.is.eval(params$training_frame)
 if( tmp_train ) {
    temp_train_key <- params$training_frame@frame_id
    .h2o.eval.frame(conn = conn, ast = params$training_frame@mutable$ast, frame_id = temp_train_key)
 }

 if (!is.null(params$validation_frame)){
    params$validation_frame <- get("validation_frame", parent.frame())
    tmp_valid <- !.is.eval(params$validation_frame)
    if( tmp_valid ) {
      temp_valid_key <- params$validation_frame@frame_id
      .h2o.eval.frame(conn = conn, ast = params$validation_frame@mutable$ast, frame_id = temp_valid_key)
    }
  }

  h2o.getFutureModel(.h2o.startModelJob(conn, algo, params, h2oRestApiVersion))
}

h2o.getFutureModel <- function(object) {
  .h2o.__waitOnJob(object@conn, object@job_key)
  h2o.getModel(object@model_id, object@conn)
}

#' Predict on an H2O Model
#'
#' Obtains predictions from various fitted H2O model objects.
#'
#' This method dispatches on the type of H2O model to select the correct
#' prediction/scoring algorithm.
#'
#' @param object a fitted \linkS4class{H2OModel} object for which prediction is
#'        desired
#' @param newdata A \linkS4class{H2OFrame} object in which to look for
#'        variables with which to predict.
#' @param ... additional arguments to pass on.
#' @return Returns an \linkS4class{H2OFrame} object with probabilites and
#'         default predictions.
#' @seealso \code{link{h2o.deeplearning}}, \code{link{h2o.gbm}},
#'          \code{link{h2o.glm}}, \code{link{h2o.randomForest}} for model
#'          generation in h2o.
#' @export
predict.H2OModel <- function(object, newdata, ...) {
  if (missing(newdata)) {
    stop("predictions with a missing `newdata` argument is not implemented yet")
  }

  tmp_data <- !.is.eval(newdata)
  if( tmp_data ) {
    key  <- newdata@frame_id
    .h2o.eval.frame(conn=h2o.getConnection(), ast=newdata@mutable$ast, frame_id=key)
  }

  # Send keys to create predictions
  url <- paste0('Predictions/models/', object@model_id, '/frames/', newdata@frame_id)
  res <- .h2o.__remoteSend(object@conn, url, method = "POST")
  res <- res$predictions_frame
  h2o.getFrame(res$name)
}
#' @rdname predict.H2OModel
#' @export
h2o.predict <- predict.H2OModel

h2o.crossValidate <- function(model, nfolds, model.type = c("gbm", "glm", "deeplearning"), params, strategy = c("mod1", "random"), ...)
{
  output <- data.frame()

  if( nfolds < 2 ) stop("`nfolds` must be greater than or equal to 2")
  if( missing(model) & missing(model.type) ) stop("must declare `model` or `model.type`")
  else if( missing(model) )
  {
    if(model.type == "gbm") model.type = "h2o.gbm"
    else if(model.type == "glm") model.type = "h2o.glm"
    else if(model.type == "deeplearning") model.type = "h2o.deeplearning"

    model <- do.call(model.type, c(params))
  }
  output[1, "fold_num"] <- -1
  output[1, "model_key"] <- model@model_id
  # output[1, "model"] <- model@model$mse_valid

  data <- params$training_frame
  data <- eval(data)
  data.len <- nrow(data)

  # nfold_vec <- h2o.sample(fr, 1:nfolds)
  nfold_vec <- sample(rep(1:nfolds, length.out = data.len), data.len)

  fnum_id <- as.h2o(nfold_vec, model@conn)
  fnum_id <- h2o.cbind(fnum_id, data)

  xval <- lapply(1:nfolds, function(i) {
      params$training_frame <- data[fnum_id$object != i, ]
      params$validation_frame <- data[fnum_id$object != i, ]
      fold <- do.call(model.type, c(params))
      output[(i+1), "fold_num"] <<- i - 1
      output[(i+1), "model_key"] <<- fold@model_id
      # output[(i+1), "cv_err"] <<- mean(as.vector(fold@model$mse_valid))
      fold
    })
  print(output)

  model
}

#' Model Performance Metrics in H2O
#'
#' Given a trained h2o model, compute its performance on the given
#' dataset
#'
#'
#' @param model An \linkS4class{H2OModel} object
#' @param data An \linkS4class{H2OFrame}. The model will make predictions
#'        on this dataset, and subsequently score them. The dataset should
#'        match the dataset that was used to train the model, in terms of
#'        column names, types, and dimensions. If data is passed in, then train and valid are ignored.
#' @param valid A logical value indicating whether to return the validation metrics (constructed during training).
#' @param ... Extra args passed in for use by other functions.
#' @return Returns an object of the \linkS4class{H2OModelMetrics} subclass.
#' @examples
#' \dontrun{
#' library(h2o)
#' localH2O <- h2o.init()
#' prosPath <- system.file("extdata", "prostate.csv", package="h2o")
#' prostate.hex <- h2o.uploadFile(localH2O, path = prosPath)
#' prostate.hex$CAPSULE <- as.factor(prostate.hex$CAPSULE)
#' prostate.gbm <- h2o.gbm(3:9, "CAPSULE", prostate.hex)
#' h2o.performance(model = prostate.gbm, data=prostate.hex)
#' }
#' @export
h2o.performance <- function(model, data=NULL, valid=FALSE, ...) {
  # Some parameter checking
  if(!is(model, "H2OModel")) stop("`model` must an H2OModel object")
  if(!is.null(data) && !is(data, "H2OFrame")) stop("`data` must be an H2OFrame object")

  missingData <- missing(data) || is.null(data)
  trainingFrame <- model@parameters$training_frame
  data.frame_id <- if( missingData ) trainingFrame else data@frame_id
  if( !missingData && data.frame_id == trainingFrame ) {
    warning("Given data is same as the training data. Returning the training metrics.")
    return(model@model$training_metrics)
  }
  else if( missingData && !valid ) return(model@model$training_metrics)    # no data, valid is false, return the training metrics
  else if( missingData &&  valid ) {
    if( is.null(model@model$validation_metrics@metrics) ) return(NULL)
    else                                                  return(model@model$validation_metrics)  # no data, but valid is true, return the validation metrics
  }
  else if( !missingData ) {
    mktmp <- !.is.eval(data)
    if( mktmp ) .h2o.eval.frame(conn=h2o.getConnection(), ast=data@mutable$ast, frame_id=data@frame_id)

    parms <- list()
    parms[["model"]] <- model@model_id
    parms[["frame"]] <- data.frame_id
    res <- .h2o.__remoteSend(model@conn, method = "POST", .h2o.__MODEL_METRICS(model@model_id,data.frame_id), .params = parms)

    ####
    # FIXME need to do the client-side filtering...  PUBDEV-874:   https://0xdata.atlassian.net/browse/PUBDEV-874
    model_metrics <- Filter(function(mm) { mm$frame$name==data.frame_id}, res$model_metrics)[[1]]   # filter on data.frame_id, R's builtin Filter function
    #
    ####
    metrics <- model_metrics[!(names(model_metrics) %in% c("__meta", "names", "domains", "model_category"))]
    model_category <- model_metrics$model_category
    Class <- paste0("H2O", model_category, "Metrics")
    metrics$frame <- list()
    metrics$frame$name <- data.frame_id
    new(Class     = Class,
        algorithm = model@algorithm,
        on_train  = missingData,
        metrics   = metrics)
  } else {
    warning("Shouldn't be here, returning NULL")
    return(NULL)
  }
}

#' Retrieve the AUC
#'
#' Retrieves the AUC value from an \linkS4class{H2OBinomialMetrics}.
#'
#' @param object An \linkS4class{H2OBinomialMetrics} object.
#' @param valid Retrieve the validation AUC
#' @param \dots extra arguments to be passed if `object` is of type
#'              \linkS4class{H2OModel} (e.g. train=TRUE)
#' @seealso \code{\link{h2o.giniCoef}} for the Gini coefficient,
#'          \code{\link{h2o.mse}} for MSE, and \code{\link{h2o.metric}} for the
#'          various threshold metrics. See \code{\link{h2o.performance}} for
#'          creating H2OModelMetrics objects.
#' @examples
#' \dontrun{
#' library(h2o)
#' h2o.init()
#'
#' prosPath <- system.file("extdata", "prostate.csv", package="h2o")
#' hex <- h2o.uploadFile(prosPath)
#'
#' hex[,2] <- as.factor(hex[,2])
#' model <- h2o.gbm(x = 3:9, y = 2, training_frame = hex, distribution = "bernoulli")
#' perf <- h2o.performance(model, hex)
#' h2o.auc(perf)
#' }
#' @export
h2o.auc <- function(object, valid=FALSE, ...) {
  if( is(object, "H2OModelMetrics") ) return( object@metrics$AUC )
  else if( is(object, "H2OModel") ) {
    model.parts <- .model.parts(object)
    if( valid ) {
      if( is.null(model.parts$vm) ) return( invisible(.warn.no.validation()) )
      else                          return( model.parts$vm@metrics$AUC )
    } else                          return( model.parts$tm@metrics$AUC )
  } else {
    warning(paste0("No AUC for ", class(object)))
    invisible(NULL)
  }
}

#'
#' Retrieve the AIC.
#' @param object An \linkS4class{H2OModel} or \linkS4class{H2OModelMetrics}.
#' @param valid Retrieve the validation AIC
#' @param \dots extra arguments to be passed if `object` is of type
#'              \linkS4class{H2OModel} (e.g. train=TRUE)
#' @export
h2o.aic <- function(object, valid=FALSE, ...) {
  if( is(object, "H2OModelMetrics") ) return( object@metrics$AIC )
  else if( is(object, "H2OModel") ) {
    model.parts <- .model.parts(object)
    if( valid ) {
      if( is.null(model.parts$vm) ) return( invisible(.warn.no.validation()) )
      else                          return( model.parts$vm@metrics$AIC )
    } else                          return( model.parts$tm@metrics$AIC )
  } else {
    warning(paste0("No AIC for ", class(object)))
    invisible(NULL)
  }
}

#'
#' Retrieve the R2 value
#'
#' Retrieves the R2 value from an H2O model.
#'
#' @param object An \linkS4class{H2OModel} object.
#' @param valid  Retrieve the validation set R2 if a validation set was passed in during model build time.
#' @param \dots extra arguments to be passed if `object` is of type
#'              \linkS4class{H2OModel} (e.g. train=TRUE)
#' @examples
#' \dontrun{
#' library(h2o)
#'
#' h <- h2o.init()
#' fr <- as.h2o(iris)
#'
#' m <- h2o.deeplearning(x=2:5,y=1,training_frame=fr)
#'
#' h2o.r2(m)
#' }
#' @export
h2o.r2 <- function(object, valid=FALSE, ...) {
  if( is(object, "H2OModelMetrics") ) return( object@metrics$r2 )
  else if( is(object, "H2OModel") ) {
    model.parts <- .model.parts(object)
    if( valid ) {
      if( is.null(model.parts$vm) ) return( invisible(.warn.no.validation()) )
      else                          return( model.parts$vm@metrics$r2 )
    } else                          return( model.parts$tm@metrics$r2 )
  } else {
    warning(paste0("No R2 for ", class(object)))
    invisible(NULL)
  }
}

#' Retrieve the GINI Coefficcient
#'
#' Retrieves the GINI coefficient from an \linkS4class{H2OBinomialMetrics}.
#'
#' @param object an \linkS4class{H2OBinomialMetrics} object.
#' @param valid TRUE to extract the metric from validation set metrics; otherwise, training is assumed
#' @param \dots extra arguments to be passed if `object` is of type
#'              \linkS4class{H2OModel} (e.g. train=TRUE)
#' @seealso \code{\link{h2o.auc}} for AUC,  \code{\link{h2o.giniCoef}} for the
#'          GINI coefficient, and \code{\link{h2o.metric}} for the various. See
#'          \code{\link{h2o.performance}} for creating H2OModelMetrics objects.
#'          threshold metrics.
#' @examples
#' \dontrun{
#' library(h2o)
#' h2o.init()
#'
#' prosPath <- system.file("extdata", "prostate.csv", package="h2o")
#' hex <- h2o.uploadFile(prosPath)
#'
#' hex[,2] <- as.factor(hex[,2])
#' model <- h2o.gbm(x = 3:9, y = 2, training_frame = hex, distribution = "bernoulli")
#' perf <- h2o.performance(model, hex)
#' h2o.giniCoef(perf)
#' }
#' @export
h2o.giniCoef <- function(object, valid=FALSE, ...) {
  if(is(object, "H2OModelMetrics")) return( object@metrics$Gini )
  else if( is(object, "H2OModel") ) {
    model.parts <- .model.parts(object)
    if( valid ) {
      if( is.null(model.parts$vm) ) return( invisible(.warn.no.validation()) )
      else                          return( model.parts$vm@metrics$Gini )
    } else                          return( model.parts$tm@metrics$Gini )
  }
  else{
    warning(paste0("No Gini for ",class(object)))
    invisible(NULL)
  }
}

#'
#' Retrieve the model coefficeints
#'
#' @param object an \linkS4class{H2OModel} object.
#' @export
h2o.coef <- function(object) {
  if( is(object, "H2OModel") ) {
    coefs <- object@model$coefficients_table
    if( is.null(coefs) ) stop("Can only extract coefficeints from GLMs")
    return( coefs$coefficients )
  } else stop("Can only extract coefficients from GLMs")
}

#'
#' Retrieve the normalized coefficients
#'
#' @param object an \linkS4class{H2OModel} object.
#' @export
h2o.coef_norm <- function(object) {
  if( is(object, "H2OModel") ) {
    coefs <- object@model$coefficients_table
    if( is.null(coefs) ) stop("Can only extract coefficeints from GLMs")
    return( coefs[,3] )  # the normalized coefs are 3rd column, (labels is 1st col)
  } else stop("Can only extract coefficients from GLMs")
}

#' Retrieves Mean Squared Error Value
#'
#' Retrieves the mean squared error value from an \linkS4class{H2OModelMetrics}
#' object.
#'
#' This function only supports \linkS4class{H2OBinomialMetrics},
#' \linkS4class{H2OMultinomialMetrics}, and \linkS4class{H2ORegressionMetrics} objects.
#'
#' @param object An \linkS4class{H2OModelMetrics} object of the correct type.
#' @param valid Retreive the validation metric.
#' @param \dots Extra arguments to be passed if `object` is of type \linkS4class{H2OModel} (e.g. train=TRUE)
#' @seealso \code{\link{h2o.auc}} for AUC, \code{\link{h2o.mse}} for MSE, and
#'          \code{\link{h2o.metric}} for the various threshold metrics. See
#'          \code{\link{h2o.performance}} for creating H2OModelMetrics objects.
#' @examples
#' \dontrun{
#' library(h2o)
#' h2o.init()
#'
#' prosPath <- system.file("extdata", "prostate.csv", package="h2o")
#' hex <- h2o.uploadFile(prosPath)
#'
#' hex[,2] <- as.factor(hex[,2])
#' model <- h2o.gbm(x = 3:9, y = 2, training_frame = hex, distribution = "bernoulli")
#' perf <- h2o.performance(model, hex)
#' h2o.mse(perf)
#' }
#' @export
h2o.mse <- function(object, valid=FALSE, ...) {
  if( is(object, "H2OModelMetrics") ) return( object@metrics$MSE )
  else if( is(object, "H2OModel") ) {
    metrics <- NULL # break out special for clustering vs the rest
    model.parts <- .model.parts(object)
    if( valid ) {
      if( is.null(model.parts$vm) ) return( invisible(.warn.no.validation()) )
      else                          metrics <- model.parts$vm@metrics
    } else                          metrics <- model.parts$tm@metrics

    if( is(object, "H2OClusteringModel") ) return( metrics$centroid_stats$within_cluster_sum_of_squares )
    return( metrics$MSE )

  # passed in something that's not an H2OModel or H2OModelMetrics
  } else {
    warning(paste0("No MSE for ",class(object)))
    invisible(NULL)
  }
}

#' Retrieve the Log Loss Value
#'
#' Retrieves the log loss output for a \linkS4class{H2OBinomialMetrics} or
#' \linkS4class{H2OMultinomialMetrics} object
#'
#' @param object a \linkS4class{H2OModelMetrics} object of the correct type.
#' @param valid Retreive the validation metric.
#' @param \dots Extra arguments to be passed if `object` is of type
#'        \linkS4class{H2OModel} (e.g. train=TRUE)
#' @export
h2o.logloss <- function(object, valid=FALSE, ...) {
  if( is(object, "H2OModelMetrics") ) return( object@metrics$logloss )
  else if( is(object, "H2OModel") ) {
    model.parts <- .model.parts(object)
    if( valid ) {
      if( is.null(model.parts$vm) ) return( invisible(.warn.no.validation()) )
      else                          return( model.parts$vm@metrics$logloss )
    } else                          return( model.parts$tm@metrics$logloss )
  } else  {
    warning(paste("No log loss for",class(object)))
    invisible(NULL)
  }
}

#'
#' Retrieve the variable importance.
#'
#' @param object An \linkS4class{H2OModel} object.
#' @param \dots further arguments to be passed on (currently unimplemented)
#' @export
h2o.varimp <- function(object, ...) {
  o <- object
  if( is(o, "H2OModel") ) {
    vi <- o@model$variable_importances
    if( is.null(vi) ) { vi <- object@model$standardized_coefficients_magnitude }  # no true variable importances, maybe glm coeffs? (return standardized table...)
    if( is.null(vi) ) {
      warning("This model doesn't have variable importances", call. = FALSE)
      return(invisible(NULL))
    }
    vi
  } else {
    warning( paste0("No variable importances for ", class(o)) )
    return(NULL)
  }
}

#'
#' Retrieve Model Score History
#'
#' @param object An \linkS4class{H2OModel} object.
#' @param \dots further arguments to be passed on (currently unimplemented)
#' @export
h2o.scoreHistory <- function(object, ...) {
  o <- object
  if( is(o, "H2OModel") ) {
    sh <- o@model$scoring_history
    if( is.null(sh) ) return(NULL)
    sh
  } else {
    warning( paste0("No score history for ", class(o)) )
    return(NULL)
  }
}

#'
#' Retrieve the respective weight matrix
#'
#' @param object An \linkS4class{H2OModel} or \linkS4class{H2OModelMetrics}
#' @param matrix_id An integer, ranging from 1 to number of layers + 1, that specifies the weight matrix to return.
#' @param \dots further arguments to be passed to/from this method.
#' @export
h2o.weights <- function(object, matrix_id=1, ...){
  o <- object
  if( is(o, "H2OModel") ) {
    sh <- o@model$weights[[matrix_id]]
    if( is.null(sh) ) return(NULL)
    sh
  } else {
    warning( paste0("No weights for ", class(o)) )
    return(NULL)
  }
}

#'
#' Return the respective bias vector
#'
#' @param object An \linkS4class{H2OModel} or \linkS4class{H2OModelMetrics}
#' @param vector_id An integer, ranging from 1 to number of layers + 1, that specifies the bias vector to return.
#' @param \dots further arguments to be passed to/from this method.
#' @export
h2o.biases <- function(object, vector_id=1, ...){
  o <- object
  if( is(o, "H2OModel") ) {
    sh <- o@model$biases[[vector_id]]
    if( is.null(sh) ) return(NULL)
    sh
  } else {
    warning( paste0("No biases for ", class(o)) )
    return(NULL)
  }
}

#'
#' Retrieve the Hit Ratios
#'
#' @param object An \linkS4class{H2OModel} object.
#' @param valid Retreive the validation metric.
#' @param \dots further arguments to be passed on (currently unimplemented)
#' @export
h2o.hit_ratio_table <- function(object, valid=FALSE, ...) {
  if( is(object, "H2OModelMetrics") ) return( object@metrics$hit_ratio_table )
  else if( is(object, "H2OModel") ) {
    model.parts <- .model.parts(object)
    if( valid ) {
      if( is.null(model.parts$vm) ) return( invisible(.warn.no.validation()) )
      else                          return( model.parts$vm@metrics$hit_ratio_table )
    } else                          return( model.parts$tm@metrics$hit_ratio_table )

  # if o is a data.frame, then the hrt was passed in -- just for pretty printing
  } else if( is(object, "data.frame") ) return(object)

  # warn if we got something unexpected...
  else warning( paste0("No hit ratio table for ", class(object)) )
  invisible(NULL)
}

#' H2O Model Metric Accessor Functions
#'
#' A series of functions that retrieve model metric details.
#'
#' Many of these functions have an optional thresholds parameter. Currently
#' only increments of 0.1 are allowed. If not specified, the functions will
#' return all possible values. Otherwise, the function will return the value for
#' the indicated threshold.
#'
#' Currently, the these functions are only supported by
#' \linkS4class{H2OBinomialMetrics} objects.
#'
#' @param object An \linkS4class{H2OModelMetrics} object of the correct type.
#' @param thresholds A value or a list of values between 0.0 and 1.0.
#' @param metric A specified paramter to retrieve.
#' @return Returns either a single value, or a list of values.
#' @seealso \code{\link{h2o.auc}} for AUC, \code{\link{h2o.giniCoef}} for the
#'          GINI coefficient, and \code{\link{h2o.mse}} for MSE. See
#'          \code{\link{h2o.performance}} for creating H2OModelMetrics objects.
#' @examples
#' \dontrun{
#' library(h2o)
#' h2o.init()
#'
#' prosPath <- system.file("extdata", "prostate.csv", package="h2o")
#' hex <- h2o.uploadFile(prosPath)
#'
#' hex[,2] <- as.factor(hex[,2])
#' model <- h2o.gbm(x = 3:9, y = 2, training_frame = hex, distribution = "bernoulli")
#' perf <- h2o.performance(model, hex)
#' h2o.F1(perf)
#' }
#' @export
h2o.metric <- function(object, thresholds, metric) {
  if(is(object, "H2OBinomialMetrics")){
    if(!missing(thresholds)) {
      t <- as.character(thresholds)
      t[t=="0"] <- "0.0"
      t[t=="1"] <- "1.0"
      if(!all(t %in% rownames(object@metrics$thresholds_and_metric_scores))) {
        stop(paste0("User-provided thresholds: ", paste(t,collapse=', '), ", are not a subset of the available thresholds: ", paste(rownames(object@metrics$thresholds_and_metric_scores), collapse=', ')))
      }
      else {
        output <- object@metrics$thresholds_and_metric_scores[t, metric]
        names(output) <- t
        output
      }
    }
    else {
      output <- object@metrics$thresholds_and_metric_scores[, metric]
      names(output) <- rownames(object@metrics$thresholds_and_metric_scores)
      output
    }
  }
  else{
    stop(paste0("No ", metric, " for ",class(object)))
  }
}

#' @rdname h2o.metric
#' @export
h2o.F0point5 <- function(object, thresholds){
  h2o.metric(object, thresholds, "f0point5")
}

#' @rdname h2o.metric
#' @export
h2o.F1 <- function(object, thresholds){
  h2o.metric(object, thresholds, "f1")
}

#' @rdname h2o.metric
#' @export
h2o.F2 <- function(object, thresholds){
  h2o.metric(object, thresholds, "f2")
}

#' @rdname h2o.metric
#' @export
h2o.accuracy <- function(object, thresholds){
  h2o.metric(object, thresholds, "accuracy")
}

#' @rdname h2o.metric
#' @export
h2o.error <- function(object, thresholds){
  h2o.metric(object, thresholds, "error")
}

#' @rdname h2o.metric
#' @export
h2o.maxPerClassError <- function(object, thresholds){
  1.0-h2o.metric(object, thresholds, "min_per_class_accuracy")
}

#' @rdname h2o.metric
#' @export
h2o.mcc <- function(object, thresholds){
  h2o.metric(object, thresholds, "absolute_MCC")
}

#' @rdname h2o.metric
#' @export
h2o.precision <- function(object, thresholds){
  h2o.metric(object, thresholds, "precision")
}

#' @rdname h2o.metric
#' @export
h2o.tpr <- function(object, thresholds){
  h2o.metric(object, thresholds, "tpr")
}

#' @rdname h2o.metric
#' @export
h2o.fpr <- function(object, thresholds){
  h2o.metric(object, thresholds, "fpr")
}

#' @rdname h2o.metric
#' @export
h2o.fnr <- function(object, thresholds){
  h2o.metric(object, thresholds, "fnr")
}

#' @rdname h2o.metric
#' @export
h2o.tnr <- function(object, thresholds){
  h2o.metric(object, thresholds, "tnr")
}

#' @rdname h2o.metric
#' @export
h2o.recall <- function(object, thresholds){
  h2o.metric(object, thresholds, "tpr")
}

#' @rdname h2o.metric
#' @export
h2o.sensitivity <- function(object, thresholds){
  h2o.metric(object, thresholds, "tpr")
}

#' @rdname h2o.metric
#' @export
h2o.fallout <- function(object, thresholds){
  h2o.metric(object, thresholds, "fpr")
}

#' @rdname h2o.metric
#' @export
h2o.missrate <- function(object, thresholds){
  h2o.metric(object, thresholds, "fnr")
}

#' @rdname h2o.metric
#' @export
h2o.specificity <- function(object, thresholds){
  h2o.metric(object, thresholds, "tnr")
}

#
#
h2o.find_threshold_by_max_metric <- function(object, metric) {
  if(!is(object, "H2OBinomialMetrics")) stop(paste0("No ", metric, " for ",class(object)))
  max_metrics <- object@metrics$max_criteria_and_metric_scores
  max_metrics[match(paste0("max ",metric),max_metrics$metric),"threshold"]
}

#
# No duplicate thresholds allowed
h2o.find_row_by_threshold <- function(object, threshold) {
  if(!is(object, "H2OBinomialMetrics")) stop(paste0("No ", threshold, " for ",class(object)))
  tmp <- object@metrics$thresholds_and_metric_scores
  if( is.null(tmp) ) return(NULL)
  res <- tmp[abs(as.numeric(tmp$threshold) - threshold) < 1e-8,]  # relax the tolerance
  if( nrow(res) == 0L ) {
    # couldn't find any threshold within 1e-8 of the requested value, warn and return closest threshold
    row_num <- which.min(abs(tmp$threshold - threshold))
    closest_threshold <- tmp$threshold[row_num]
    warning( paste0("Could not find exact threshold: ", threshold, " for this set of metrics; using closest threshold found: ", closest_threshold, ". Run `h2o.predict` and apply your desired threshold on a probability column.") )
    return( tmp[row_num,] )
  }
  else if( nrow(res) > 1L ) res <- res[1L,]
  res
}

#'
#' Retrieve the Model Centers
#'
#' @param object An \linkS4class{H2OClusteringModel} object.
#' @param \dots further arguments to be passed on (currently unimplemented)
#' @export
h2o.centers <- function(object, ...) { as.data.frame(object@model$centers[,-1]) }

#'
#' Retrieve the Model Centers STD
#'
#' @param object An \linkS4class{H2OClusteringModel} object.
#' @param \dots further arguments to be passed on (currently unimplemented)
#' @export
h2o.centersSTD <- function(object, ...) { as.data.frame(object@model$centers_std)[,-1] }

#'
#' Get the Within SS
#'
#' @param object An \linkS4class{H2OClusteringModel} object.
#' @param \dots further arguments to be passed on (currently unimplemented)
#' @export
h2o.withinss <- function(object, ...) { h2o.mse(object, ...) }

#'
#' Get the total within cluster sum of squares.
#'
#' @param object An \linkS4class{H2OClusteringModel} object.
#' @param valid Retreive the validation metric.
#' @param \dots further arguments to be passed on (currently unimplemented)
#' @export
h2o.tot_withinss <- function(object, valid=FALSE, ...) {
  model.parts <- .model.parts(object)
  if( valid ) {
    if( is.null(model.parts$vm) ) return( invisible(.warn.no.validation()) )
    else                          return( model.parts$vm@metrics$tot_withinss )
  } else                          return( model.parts$tm@metrics$tot_withinss )
}

#'
#' Get the between cluster sum of squares.
#'
#' @param object An \linkS4class{H2OClusteringModel} object.
#' @param valid Retreive the validation metric.
#' @param \dots further arguments to be passed on (currently unimplemented)
#' @export
h2o.betweenss <- function(object, valid=FALSE, ...) {
  model.parts <- .model.parts(object)
  if( valid ) {
    if( is.null(model.parts$vm) ) return( invisible(.warn.no.validation()) )
    else                          return( model.parts$vm@metrics$betweenss )
  } else                          return( model.parts$tm@metrics$betweenss )
}

#'
#' Get the total sum of squares.
#'
#' @param object An \linkS4class{H2OClusteringModel} object.
#' @param valid Retreive the validation metric.
#' @param \dots further arguments to be passed on (currently unimplemented)
#' @export
h2o.totss <- function(object,valid=FALSE, ...) {
  model.parts <- .model.parts(object)
  if( valid ) {
    if( is.null(model.parts$vm) ) return( invisible(.warn.no.validation()) )
    else                          return( model.parts$vm@metrics$totss )
  } else                          return( model.parts$tm@metrics$totss )
}

#'
#' Retrieve the number of iterations.
#'
#' @param object An \linkS4class{H2OClusteringModel} object.
#' @param \dots further arguments to be passed on (currently unimplemented)
#' @export
h2o.num_iterations <- function(object) { object@model$model_summary$number_of_iterations }

#'
#' Retrieve the centroid statistics
#'
#' @param object An \linkS4class{H2OClusteringModel} object.
#' @param valid Retrieve the validation metric.
#' @param \dots further arguments to be passed on (currently unimplemented)
#' @export
h2o.centroid_stats <- function(object, valid=FALSE, ...) {
  model.parts <- .model.parts(object)
  if( valid ) {
    if( is.null(model.parts$vm) ) return( invisible(.warn.no.validation()) )
    else                          return( model.parts$vm@metrics$centroid_stats )
  } else                          return( model.parts$tm@metrics$centroid_stats )
}

#'
#' Retrieve the cluster sizes
#'
#' @param object An \linkS4class{H2OClusteringModel} object.
#' @param valid Retrieve the validation metric.
#' @param \dots further arguments to be passed on (currently unimplemented)
#' @export
h2o.cluster_sizes <- function(object, valid=FALSE, ...) {
  model.parts <- .model.parts(object)
  if( valid ) {
    if( is.null(model.parts$vm) ) return( invisible(.warn.no.validation()) )
    else                          return( model.parts$vm@metrics$centroid_stats$size )
  } else                          return( model.parts$tm@metrics$centroid_stats$size )
}


#'
#' Retrieve the null deviance
#'
#' @param object An \linkS4class{H2OModel} or \linkS4class{H2OModelMetrics}
#' @param valid Retrieve the validation metric.
#' @param \dots further arguments to be passed to/from this method.
#' @export
h2o.null_deviance <- function(object, valid=FALSE, ...) {
  if( is(object, "H2OModelMetrics") ) return( object@metrics$null_deviance )
  else {
    model.parts <- .model.parts(object)
    if( valid ) {
      if( is.null(model.parts$vm) ) return( invisible(.warn.no.validation()) )
      else                          return( model.parts$vm@metrics$null_deviance )
    } else                          return( model.parts$tm@metrics$null_deviance )
  }
}

#' Retrieve the residual deviance
#'
#' @param object An \linkS4class{H2OModel} or \linkS4class{H2OModelMetrics}
#' @param valid Retrieve the validation metric.
#' @param \dots further arguments to be passed to/from this method.
#' @export
h2o.residual_deviance <- function(object, valid=FALSE, ...) {
  if( is(object, "H2OModelMetrics") ) return( object@metrics$residual_deviance )
  else {
    model.parts <- .model.parts(object)
    if( valid ) {
      if( is.null(model.parts$vm) ) return( invisible(.warn.no.validation()) )
      else                          return( model.parts$vm@metrics$residual_deviance )
    } else                          return( model.parts$tm@metrics$residual_deviance )
  }
}


#' Retrieve the residual degrees of freedom
#'
#' @param object An \linkS4class{H2OModel} or \linkS4class{H2OModelMetrics}
#' @param valid Retrieve the validation metric.
#' @param \dots further arguments to be passed to/from this method.
#' @export
h2o.residual_dof <- function(object, valid=FALSE, ...) {
  if( is(object, "H2OModelMetrics") ) return( object@metrics$residual_degrees_of_freedom )
  else {
    model.parts <- .model.parts(object)
    if( valid ) {
      if( is.null(model.parts$vm) ) return( invisible(.warn.no.validation()) )
      else                          return( model.parts$vm@metrics$residual_degrees_of_freedom )
    } else                          return( model.parts$tm@metrics$residual_degrees_of_freedom )
  }
}

#' Retrieve the null degrees of freedom
#'
#' @param object An \linkS4class{H2OModel} or \linkS4class{H2OModelMetrics}
#' @param valid Retrieve the validation metric.
#' @param \dots further arguments to be passed to/from this method.
#' @export
h2o.null_dof <- function(object, valid=FALSE, ...) {
  if( is(object, "H2OModelMetrics") ) return( object@metrics$null_degrees_of_freedom )
  else {
    model.parts <- .model.parts(object)
    if( valid ) {
      if( is.null(model.parts$vm) ) return( invisible(.warn.no.validation()) )
      else                          return( model.parts$vm@metrics$null_degrees_of_freedom )
    } else                          return( model.parts$tm@metrics$null_degrees_of_freedom )
  }
}

#' Access H2O Confusion Matrices
#'
#' Retrieve either a single or many confusion matrices from H2O objects.
#'
#' The \linkS4class{H2OModelMetrics} version of this function will only take
#' \linkS4class{H2OBinomialMetrics} or \linkS4class{H2OMultinomialMetrics}
#' objects. If no threshold is specified, all possible thresholds are selected.
#'
#' @param object Either an \linkS4class{H2OModel} object or an
#'        \linkS4class{H2OModelMetrics} object.
#' @param newdata An \linkS4class{H2OFrame} object that can be scored on.
#'        Requires a valid response column.
#' @param thresholds (Optional) A value or a list of valid values between 0.0 and 1.0.
#'        This value is only used in the case of
#'        \linkS4class{H2OBinomialMetrics} objects.
#' @param metrics (Optional) A metric or a list of valid metrics ("min_per_class_accuracy", "absolute_MCC", "tnr", "fnr", "fpr", "tpr", "precision", "accuracy", "f0point5", "f2", "f1").
#'        This value is only used in the case of
#'        \linkS4class{H2OBinomialMetrics} objects.
#' @param valid Retreive the validation metric.
#' @param ... Extra arguments for extracting train or valid confusion matrices.
#' @return Calling this function on \linkS4class{H2OModel} objects returns a
#'         confusion matrix corresponding to the \code{\link{predict}} function.
#'         If used on an \linkS4class{H2OBinomialMetrics} object, returns a list
#'         of matrices corresponding to the number of thresholds specified.
#' @seealso \code{\link{predict}} for generating prediction frames,
#'          \code{\link{h2o.performance}} for creating
#'          \linkS4class{H2OModelMetrics}.
#' @examples
#' \dontrun{
#' library(h2o)
#' h2o.init()
#' prosPath <- system.file("extdata", "prostate.csv", package="h2o")
#' hex <- h2o.uploadFile(prosPath)
#' hex[,2] <- as.factor(hex[,2])
#' model <- h2o.gbm(x = 3:9, y = 2, training_frame = hex, distribution = "bernoulli")
#' h2o.confusionMatrix(model, hex)
#' # Generating a ModelMetrics object
#' perf <- h2o.performance(model, hex)
#' h2o.confusionMatrix(perf)
#' }
#' @export
setGeneric("h2o.confusionMatrix", function(object, ...) {})

#' @rdname h2o.confusionMatrix
#' @export
setMethod("h2o.confusionMatrix", "H2OModel", function(object, newdata, valid=FALSE, ...) {
  model.parts <- .model.parts(object)
  if( missing(newdata) ) {
    if( valid ) {
      if( is.null(model.parts$vm) ) return( invisible(.warn.no.validation()) )
      else                          return( h2o.confusionMatrix(model.parts$vm, ...) )
    } else                          return( h2o.confusionMatrix(model.parts$tm, ...) )
  } else if( valid ) stop("Cannot have both `newdata` and `valid=TRUE`", call.=FALSE)

  # ok need to score on the newdata
  tmp <- !.is.eval(newdata)
  if( tmp ) {
    temp_key <- newdata@frame_id
    .h2o.eval.frame(conn = newdata@conn, ast = newdata@mutable$ast, frame_id = temp_key)
  }

  url <- paste0("Predictions/models/",object@model_id, "/frames/", newdata@frame_id)
  res <- .h2o.__remoteSend(object@conn, url, method="POST")

  # Make the correct class of metrics object
  metrics <- new(sub("Model", "Metrics", class(object)), algorithm=object@algorithm, metrics= res$model_metrics[[1L]])   # FIXME: don't think model metrics come out of Predictions anymore!!!
  h2o.confusionMatrix(metrics, ...)
})

#' @rdname h2o.confusionMatrix
#' @export
setMethod("h2o.confusionMatrix", "H2OModelMetrics", function(object, thresholds=NULL, metrics=NULL) {
  if( !is(object, "H2OBinomialMetrics") ) {
    if( is(object, "H2OMultinomialMetrics") )
      return(object@metrics$cm$table)
    warning(paste0("No Confusion Matrices for ",class(object)))
    return(NULL)
  }
  # H2OBinomial case
  if( is.null(metrics) && is.null(thresholds) ) {
    metrics = c("f1")
  }
  if( is(metrics, "list") ) metrics_list = metrics
  else {
    if( is.null(metrics) ) metrics_list = list()
    else metrics_list = list(metrics)
  }
  if( is(thresholds, "list") ) thresholds_list = thresholds
    else {
      if( is.null(thresholds) ) thresholds_list = list()
      else thresholds_list = list(thresholds)
  }

  # error check the metrics_list and thresholds_list
  if( !all(sapply(thresholds_list, f <- function(x) is.numeric(x) && x >= 0 && x <= 1)) )
    stop("All thresholds must be numbers between 0 and 1 (inclusive).")
  allowable_metrics <- c("min_per_class_accuracy", "absolute_MCC", "tnr", "fnr", "fpr", "tpr","precision", "accuracy", "f0point5", "f2", "f1")
  if( !all(sapply(metrics_list, f <- function(x) x %in% allowable_metrics)) )
      stop(paste("The only allowable metrics are ", paste(allowable_metrics, collapse=', ')))

  # make one big list that combines the thresholds and metric-thresholds
  metrics_thresholds = lapply(metrics_list, f <- function(x) h2o.find_threshold_by_max_metric(object, x))
  thresholds_list <- append(thresholds_list, metrics_thresholds)

  thresh2d <- object@metrics$thresholds_and_metric_scores
  actual_thresholds <- thresh2d$threshold
  d <- object@metrics$domain
  m <- lapply(thresholds_list,function(t) {
    row <- h2o.find_row_by_threshold(object,t)
    if( is.null(row) ) NULL
    else {
      tns <- row$tns; fps <- row$fps; fns <- row$fns; tps <- row$tps;
      rnames <- c(d, "Totals")
      cnames <- c(d, "Error", "Rate")
      col1 <- c(tns, fns, tns+fns)
      col2 <- c(fps, tps, fps+tps)
      col3 <- c(fps/(fps+tns), fns/(fns+tps), (fps+fns)/(fps+tns+fns+tps))
      col4 <- c( paste0(" =", fps, "/", fps+tns), paste0(" =", fns, "/", fns+tps), paste0(" =", fns+fps, "/", fps+tns+fns+tps) )
      fmts <- c("%i", "%i", "%f", "%s")
      tbl <- data.frame(col1,col2,col3,col4)
      colnames(tbl) <- cnames
      rownames(tbl) <- rnames
      header <-  "Confusion Matrix"
      if(t %in% metrics_thresholds) {
        m <- metrics_list[which(t == metrics_thresholds)]
        if( length(m) > 1) m <- m[[1]]
        header <- paste(header, "for max", m, "@ threshold =", t)
      } else {
        header <- paste(header, "@ threshold =", row$threshold)
      }
      attr(tbl, "header") <- header
      attr(tbl, "formats") <- fmts
      oldClass(tbl) <- c("H2OTable", "data.frame")
      tbl
    }
  })
  if( length(m) == 1L ) return( m[[1L]] )
  m
})

#' @export
plot.H2OModel <- function(x, ...) {
  if( is(x, "H2OBinomialModel") ) {
    if( !is.null(x@model$validation_metrics@metrics) ) metrics <- x@model$validation_metrics
    else                                               metrics <- x@model$training_metrics
    plot.H2OBinomialMetrics(metrics, ...)
  } else NULL
}

#' @export
plot.H2OBinomialMetrics <- function(x, type = "roc", ...) {
  # TODO: add more types (i.e. cutoffs)
  if(!type %in% c("roc")) stop("type must be 'roc'")
  if(type == "roc") {
    xaxis <- "False Positive Rate"; yaxis = "True Positive Rate"
    main <- paste(yaxis, "vs", xaxis)
    if( x@on_train ) main <- paste(main, "(on train)")
    else             main <- paste(main, "(on valid)")
    graphics::plot(x@metrics$thresholds_and_metric_scores$fpr, x@metrics$thresholds_and_metric_scores$tpr, main = main, xlab = xaxis, ylab = yaxis, ylim=c(0,1), xlim=c(0,1), ...)
    graphics::abline(0, 1, lty = 2)
  }
}

#' @export
screeplot.H2ODimReductionModel <- function(x, npcs, type = "barplot", main, ...) {
    if(x@algorithm != "pca") stop("x must be a H2O PCA model")
    if(missing(npcs))
      npcs = min(10, x@model$parameters$k)
    else if(!is.numeric(npcs) || npcs < 1 || npcs > x@model$parameters$k)
      stop(paste("npcs must be a positive integer between 1 and", x@model$parameters$k, "inclusive"))

    sdevH2O <- as.numeric(x@model$pc_importance[1,])
    if(missing(main))
      main = paste("h2o.prcomp(", strtrim(x@parameters$training_frame, 20), ")", sep="")
    if(type == "barplot")
      barplot(sdevH2O[1:npcs]^2, main = main, ylab = "Variances", ...)
    else if(type == "lines")
      lines(sdevH2O[1:npcs]^2, main = main, ylab = "Variances", ...)
    else
      stop("type must be either 'barplot' or 'lines'")
}

# Handles ellipses
.model.ellipses <- function(dots) {
  lapply(names(dots), function(type) {
    stop(paste0('\n  unexpected argument "',
                type,'", is this legacy code? Try ?h2o.shim'), call. = FALSE)
  })
}


# extract "bite size" pieces from a model
.model.parts <- function(object) {
  o  <- object
  m  <- object@model
  tm <- object@model$training_metrics
  vm <- object@model$validation_metrics
  if( is.null(vm@metrics) ) list(o=o,m=m,tm=tm)       # no validation metrics
  else                      list(o=o,m=m,tm=tm,vm=vm) #haz validation metrics
}

.warn.no.validation <- function() {
  warning("No validation metrics available.", call.=FALSE)
  NULL
}
