% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/3_1_validate.R
\name{validate}
\alias{validate}
\title{Validate high-dimensional Cox models with time-dependent AUC}
\usage{
validate(x, time, event, model.type = c("lasso", "alasso", "flasso",
  "enet", "aenet", "mcp", "mnet", "scad", "snet"), alpha, lambda,
  pen.factor = NULL, gamma, lambda1, lambda2, method = c("bootstrap",
  "cv", "repeated.cv"), boot.times = NULL, nfolds = NULL,
  rep.times = NULL, tauc.type = c("CD", "SZ", "UNO"), tauc.time,
  seed = 1001, trace = TRUE)
}
\arguments{
\item{x}{Matrix of training data used for fitting the model;
on which to run the validation.}

\item{time}{Survival time.
Must be of the same length with the number of rows as \code{x}.}

\item{event}{Status indicator, normally 0 = alive, 1 = dead.
Must be of the same length with the number of rows as \code{x}.}

\item{model.type}{Model type to validate. Could be one of \code{"lasso"},
\code{"alasso"}, \code{"flasso"}, \code{"enet"}, \code{"aenet"},
\code{"mcp"}, \code{"mnet"}, \code{"scad"}, or \code{"snet"}.}

\item{alpha}{Value of the elastic-net mixing parameter alpha for
\code{enet}, \code{aenet}, \code{mnet}, and \code{snet} models.
For \code{lasso}, \code{alasso}, \code{mcp}, and \code{scad} models,
please set \code{alpha = 1}.
\code{alpha=1}: lasso (l1) penalty; \code{alpha=0}: ridge (l2) penalty.
Note that for \code{mnet} and \code{snet} models,
\code{alpha} can be set to very close to 0 but not 0 exactly.}

\item{lambda}{Value of the penalty parameter lambda to use in the
model fits on the resampled data. From the fitted Cox model.}

\item{pen.factor}{Penalty factors to apply to each coefficient.
From the fitted \emph{adaptive lasso} or \emph{adaptive elastic-net} model.}

\item{gamma}{Value of the model parameter gamma for
MCP/SCAD/Mnet/Snet models.}

\item{lambda1}{Value of the penalty parameter lambda1 for fused lasso model.}

\item{lambda2}{Value of the penalty parameter lambda2 for fused lasso model.}

\item{method}{Validation method.
Could be \code{"bootstrap"}, \code{"cv"}, or \code{"repeated.cv"}.}

\item{boot.times}{Number of repetitions for bootstrap.}

\item{nfolds}{Number of folds for cross-validation and
repeated cross-validation.}

\item{rep.times}{Number of repeated times for repeated cross-validation.}

\item{tauc.type}{Type of time-dependent AUC.
Including \code{"CD"} proposed by Chambless and Diao (2006).,
\code{"SZ"} proposed by Song and Zhou (2008).,
\code{"UNO"} proposed by Uno et al. (2007).}

\item{tauc.time}{Numeric vector. Time points at which to evaluate
the time-dependent AUC.}

\item{seed}{A random seed for resampling.}

\item{trace}{Logical. Output the validation progress or not.
Default is \code{TRUE}.}
}
\description{
Validate high-dimensional Cox models with time-dependent AUC
}
\examples{
data(smart)
x <- as.matrix(smart[, -c(1, 2)])[1:500, ]
time <- smart$TEVENT[1:500]
event <- smart$EVENT[1:500]
y <- survival::Surv(time, event)

fit <- fit_lasso(x, y, nfolds = 5, rule = "lambda.1se", seed = 11)

# Model validation by bootstrap with time-dependent AUC
# Normally boot.times should be set to 200 or more,
# we set it to 3 here only to save example running time.
val.boot <- validate(
  x, time, event,
  model.type = "lasso",
  alpha = 1, lambda = fit$lambda,
  method = "bootstrap", boot.times = 3,
  tauc.type = "UNO", tauc.time = seq(0.25, 2, 0.25) * 365,
  seed = 1010
)

# Model validation by 5-fold cross-validation with time-dependent AUC
val.cv <- validate(
  x, time, event,
  model.type = "lasso",
  alpha = 1, lambda = fit$lambda,
  method = "cv", nfolds = 5,
  tauc.type = "UNO", tauc.time = seq(0.25, 2, 0.25) * 365,
  seed = 1010
)

# Model validation by repeated cross-validation with time-dependent AUC
val.repcv <- validate(
  x, time, event,
  model.type = "lasso",
  alpha = 1, lambda = fit$lambda,
  method = "repeated.cv", nfolds = 5, rep.times = 3,
  tauc.type = "UNO", tauc.time = seq(0.25, 2, 0.25) * 365,
  seed = 1010
)

# bootstrap-based discrimination curves has a very narrow band
print(val.boot)
summary(val.boot)
plot(val.boot)

# k-fold cv provides a more strict evaluation than bootstrap
print(val.cv)
summary(val.cv)
plot(val.cv)

# repeated cv provides similar results as k-fold cv
# but more robust than k-fold cv
print(val.repcv)
summary(val.repcv)
plot(val.repcv)
# # Test fused lasso, SCAD, and Mnet models
#
# data(smart)
# x = as.matrix(smart[, -c(1, 2)])[1:500,]
# time = smart$TEVENT[1:500]
# event = smart$EVENT[1:500]
# y = survival::Surv(time, event)
#
# set.seed(1010)
# val.boot = validate(
#   x, time, event, model.type = "flasso",
#   lambda1 = 5, lambda2 = 2,
#   method = "bootstrap", boot.times = 10,
#   tauc.type = "UNO", tauc.time = seq(0.25, 2, 0.25) * 365,
#   seed = 1010)
#
# val.cv = validate(
#   x, time, event, model.type = "scad",
#   gamma = 3.7, alpha = 1, lambda = 0.05,
#   method = "cv", nfolds = 5,
#   tauc.type = "UNO", tauc.time = seq(0.25, 2, 0.25) * 365,
#   seed = 1010)
#
# val.repcv = validate(
#   x, time, event, model.type = "mnet",
#   gamma = 3, alpha = 0.3, lambda = 0.05,
#   method = "repeated.cv", nfolds = 5, rep.times = 3,
#   tauc.type = "UNO", tauc.time = seq(0.25, 2, 0.25) * 365,
#   seed = 1010)
#
# print(val.boot)
# summary(val.boot)
# plot(val.boot)
#
# print(val.cv)
# summary(val.cv)
# plot(val.cv)
#
# print(val.repcv)
# summary(val.repcv)
# plot(val.repcv)
}
\references{
Chambless, L. E. and G. Diao (2006).
Estimation of time-dependent area under the ROC curve for long-term
risk prediction.
\emph{Statistics in Medicine} 25, 3474--3486.

Song, X. and X.-H. Zhou (2008).
A semiparametric approach for the covariate specific ROC curve with
survival outcome.
\emph{Statistica Sinica} 18, 947--965.

Uno, H., T. Cai, L. Tian, and L. J. Wei (2007).
Evaluating prediction rules for t-year survivors with censored
regression models.
\emph{Journal of the American Statistical Association} 102, 527--537.
}
