#!/usr/bin/Rscript
#
# --------------------------------------------------------------
# Authors:  Paul R. Staab
# Date:     2013-11-13
# Licence:  GPLv3 or later
# --------------------------------------------------------------

library(jaatha)

if (is.element("jaatha", loadedNamespaces()))
  attach(loadNamespace("jaatha"), name=paste("namespace", "jaatha", sep=":"), pos=3)

set.seed(13579)

# A Block
block.test <- new("Block")
block.test@border <- matrix(c(0, 0, 0.1, 0.1), 2, 2) 


# Theta-Tau Model 
dm.tt        <- dm.createThetaTauModel(11:12, 10)
sum.stats.tt <- dm.simSumStats(dm.tt, c(1, 5))
jaatha.tt    <- Jaatha.initialize(dm.tt, jsfs=sum.stats.tt) 
sim.data.tt  <- simulateWithinBlock(10, block.test, jaatha.tt)


# Migration Model
dm.mig        <- dm.addSymmetricMigration(dm.tt, 1, 5)
sum.stats.mig <- dm.simSumStats(dm.mig, c(1, 1, 5))
jaatha.mig    <- Jaatha.initialize(dm.mig, jsfs=sum.stats.mig) 

# Groups
dm.grp <- dm.tt
dm.grp <- dm.setLociLength(dm.grp, 100, 1) 
dm.grp <- dm.setLociNumber(dm.grp, 15, 1) 
dm.grp <- dm.setLociLength(dm.grp, 200, 2) 
dm.grp <- dm.addSampleSize(dm.grp, 5:6, 3)
sum.stats.grp <- dm.simSumStats(dm.grp, c(1, 5))

# Finite Sites Models
dm.sg <-  finalizeDM(dm.addOutgroup(dm.tt, "2*tau"))
dm.hky <- finalizeDM(dm.setMutationModel(dm.sg, "HKY", c(0.2, 0.2, 0.3, 0.3), 2))
dm.f81 <- finalizeDM(dm.setMutationModel(dm.sg, "F84", c(0.3, 0.2, 0.3, 0.2), 2))
dm.gtr <- finalizeDM(dm.setMutationModel(dm.sg, "GTR", 
                                         gtr.rates=c(0.2, 0.2, 0.1, 0.1, 0.1, 0.2)))


# Custom Simulation Interface
csi.sim.func <- function(x, jaatha) {
  list(poisson.vector=c(rpois(3, x[1]), rpois(3, x[2])))
}
csi.obs <- csi.sim.func(c(2:3))
csi.sum.stats <- list("poisson.vector"=list(method="poisson.independent",
                                            value=csi.obs$poisson.vector))
csi.par.ranges <- matrix(c(0.1, 0.1, 10, 10), 2, 2)
rownames(csi.par.ranges) <- c('x', 'y')

jaatha.csi <- new("Jaatha", csi.sim.func, csi.par.ranges, csi.sum.stats, 123)
jaatha.csi <- Jaatha.initialSearch(jaatha.csi, 20, 2)
jaatha.csi <- Jaatha.refinedSearch(jaatha.csi, 1, 20, max.steps=10)

sim.data.csi <- simulateWithinBlock(10, block.test, jaatha.csi)


# Smoothing
smooth.func <- function(x) {
  # x[1:2] = Matrix cell 
  # x[3:4] = Model Parameters
  x[1] * x[3]^2 + x[2] * x[4]^1.5 + prod(log(x[1:2]))
}

smooth.simfunc <- function(x, jaatha) {
  stopifnot(length(x) == 2)
  idxs <- cbind(rep(1:10, each=12), rep(1:12, 10), x[1], x[2])
  sampled.values <- sapply(apply(idxs, 1, smooth.func), function(x) rpois(1, x))
  list(mat=matrix(sampled.values, 10, 12, byrow=TRUE))
}

smooth.obs <- smooth.simfunc(c(3, 4))
smooth.sum.stats <- list("mat"=list(method="poisson.smoothing",
                                    model="(X1^2)*(X2^2)+log(X1)*log(X2)",
                                    value=smooth.obs$mat))

smooth.par.ranges <- matrix(c(2, 1, 7, 7), 2, 2)
rownames(smooth.par.ranges) <- c('x', 'y')

smooth.jaatha <- new("Jaatha", smooth.simfunc, smooth.par.ranges, smooth.sum.stats, 123)
smooth.sim.data <- simulateWithinBlock(10, block.test, smooth.jaatha)

save(list=ls(), file="test_setup.Rda")
