\name{algebra}
\alias{goedel.tnorm}
\alias{lukas.tnorm}
\alias{goguen.tnorm}
\alias{pgoedel.tnorm}
\alias{plukas.tnorm}
\alias{pgoguen.tnorm}

\alias{goedel.tconorm}
\alias{lukas.tconorm}
\alias{goguen.tconorm}
\alias{pgoedel.tconorm}
\alias{plukas.tconorm}
\alias{pgoguen.tconorm}

\alias{goedel.residuum}
\alias{lukas.residuum}
\alias{goguen.residuum}
\alias{goedel.biresiduum}
\alias{lukas.biresiduum}
\alias{goguen.biresiduum}

\alias{invol.neg}
\alias{strict.neg}

\alias{algebra}
\alias{is.algebra}

\title{Algebra for Fuzzy Sets}
\description{
    Compute triangular norms (t-norms), triangular conorms (t-conorms), residua,
    bi-residua, and negations.
}
\usage{
## a t-norm from concatenated arguments:
goedel.tnorm(..., na.rm=FALSE)
lukas.tnorm(..., na.rm=FALSE)
goguen.tnorm(..., na.rm=FALSE)

## a t-norm in a parallel (element-wise) manner:
pgoedel.tnorm(..., na.rm=FALSE)
plukas.tnorm(..., na.rm=FALSE)
pgoguen.tnorm(..., na.rm=FALSE)

## a t-conorm from concatenated arguments:
goedel.tconorm(..., na.rm=FALSE)
lukas.tconorm(..., na.rm=FALSE)
goguen.tconorm(..., na.rm=FALSE)

## a t-conorm in a parallel (element-wise) manner:
pgoedel.tconorm(..., na.rm=FALSE)
plukas.tconorm(..., na.rm=FALSE)
pgoguen.tconorm(..., na.rm=FALSE)

## compute a residuum (implication) 
goedel.residuum(x, y)
lukas.residuum(x, y)
goguen.residuum(x, y)

## compute a bi-residuum (equivalence) 
goedel.biresiduum(x, y)
lukas.biresiduum(x, y)
goguen.biresiduum(x, y)

## compute a negation
invol.neg(x)
strict.neg(x)

## algebra-related functions
algebra(name, stdneg=FALSE, ...)
is.algebra(a)
}

\arguments{
    \item{...}{
        For t-norms and t-conorms, these arguments are numeric vectors of values to compute t-norms or t-conorms from.
        Values outside the \eqn{[0,1]} interval cause an error. NA values are also permitted.

        For the \code{algebra()} function, these arguments are passed to the factory functions that
        create the algebra. (Reserved for future use).
    }
    \item{na.rm}{
        whether to ignore NA values: \code{TRUE} means that NA's are ignored, i.e.
        the computation is performed as if such values were not present in the arguments;
        \code{FALSE} means that the NA's in arguments are taken into considerations, details below.
    }
    \item{x}{
        Numeric vector of values to compute a residuum or bi-residuum from.
        Values outside the \eqn{[0,1]} interval cause an error. NA values are also permitted.
    }
    \item{y}{
        Numeric vector of values to compute a residuum or bi-residuum from.
        Values outside the \eqn{[0,1]} interval cause an error. NA values are also permitted.
    }
    \item{name}{
        The name of the algebra to be created. Must be one of: "goedel", "lukasiewicz", "goguen" (or
        an unambiguous abbreviation).
    }
    \item{stdneg}{
        If \code{TRUE} the use of a "standard" negation (i.e. involutive negation) is forced.
        Otherwise, the appropriate negation is used in the algebra (e.g. strict negation in Goedel
        and Goguen algebra and involutive negation in Lukasiewicz algebra).
    }
    \item{a}{
        An object to be checked if it is a valid algebra (i.e. a list returned by the
        \code{algebra} function).
    }
}
\details{
    \code{goedel.tnorm}, \code{lukas.tnorm}, and \code{goguen.tnorm} compute the Goedel,
    Lukasiewicz, and Goguen triangular norm (t-norm) from all values in the arguments. If the arguments are 
    vectors they are combined together firstly so that a numeric vector of length 1 is returned.

    \code{pgoedel.tnorm}, \code{plukas.tnorm}, and \code{pgoguen.tnorm} compute the same t-norms,
    but in a parallel manner (element-wisely). I.e. the values with indices 1 of all arguments are
    used to compute the t-norm, then the second values (while recycling the vectors if they do not have the same size)
    so that the result is a vector of values.

    \code{goedel.tconorm}, \code{lukas.tconorm}, \code{goguen.tconorm}, are similar to the
    previously mentioned functions, exept that they compute triangular conorms (t-conorms).
    \code{pgoedel.tconorm}, \code{plukas.tconorm}, and \code{pgoguen.tconorm} are their parallel
    (i.e. element-wise) alternatives.

    \code{goedel.residuum}, \code{lukas.residuum}, and \code{goguen.residuum} 
    compute residua (i.e. implications) and
    \code{goedel.biresiduum}, \code{lukas.biresiduum}, and \code{goguen.biresiduum} compute
    bi-residua.

    \code{invol.neg} and \code{strict.neg} compute the involutive and strict negation, respectively.

    Let \eqn{a}, \eqn{b} be values from the interval \eqn{[0, 1]}. Here are mathematical
    definitions of the realized functions:
    \itemize{
        \item{Goedel t-norm:} \eqn{min(a, b)};
        \item{Goguen t-norm:} \eqn{ab};
        \item{Lukasiewicz t-norm:} \eqn{max(0, a+b-1)};
        \item{Goedel t-conorm:} \eqn{max(a, b)};
        \item{Goguen t-conorm:} \eqn{a+b-ab};
        \item{Lukasiewicz t-conorm:} \eqn{min(1, a+b)};
        \item{Goedel residuum (standard Goedel implication):} \eqn{1} if \eqn{a \le b} and \eqn{b} otherwise;
        \item{Goguen residuum (implication):} \eqn{1} if \eqn{a \le b} and \eqn{b/a} otherwise;
        \item{Lukasiewicz residuum (standard Lukasiewicz implication):} \eqn{1} if \eqn{a \le b} and \eqn{1-a+b} otherwise;
        \item{Involutive negation:} \eqn{1-x};
        \item{Strict negation:} \eqn{1} if \eqn{x=0} and \eqn{0} otherwise.
    }
    Bi-residuum \eqn{B} is derived from t-norm \eqn{T} and residuum \eqn{R} as follows:
    \deqn{B(a, b) = T(R(a, b), R(b, a)).}

    The arguments have to be numbers from the interval \eqn{[0, 1]}. Values outside that range cause
    an error. Also NaN causes an error.

    If \code{na.rm=TRUE} then missing values (NA) are ignored. Otherwise, they are treated
    as unknown values accordingly to Kleene logic. See the examples below.

    \code{algebra} returns a named list of functions that together form Goedel, Goguen, or
    Lukasiewicz algebra:
    \itemize{
        \item{\code{"goedel"}:} strict negation and Goedel t-norm, t-conorm, residuum, and bi-residuum;
        \item{\code{"goguen"}:} strict negation and Goguen t-norm, t-conorm, residuum, and bi-residuum;
        \item{\code{"lukasiewicz"}:} involutive negation and Lukasiewicz t-norm, t-conorm, residuum, and bi-residuum.
    }

    \code{is.algebra} tests whether the given \code{a} argument is a valid algebra, i.e. a list
    returned by the \code{algebra} function.

}
\value{
    Functions for t-norms and t-conorms (such as \code{goedel.tnorm})
    return a numeric vector of size 1 that is the result of
    the appropriate t-norm or t-conorm applied on all values of all arguments.

    Parallel versions of t-norms and t-conorms (such as \code{pgoedel.tnorm})
    return a vector of results after applying
    the appropriate t-norm or t-conorm on argument in an element-wise (i.e. parallel, by indices)
    way. The resulting vector is of length of the longest argument (shorter arguments are recycled).
    
    Residua and bi-residua functions
    return a numeric vector of length of the longest argument (shorter argument is recycled).

    \code{strict.neg} and \code{invol.neg} compute negations and return a numeric vector of the same
    size as the argument \code{x}.

    \code{algebra} returns a list of functions of the requested algebra: 
    \code{"n"} (negation), \code{"t"} (t-norm),
    \code{"pt"} (parallel, i.e. element-wise, t-norm),
    \code{"c"} (t-conorm), \code{"pc"} (parallel t-conorm),
    \code{"r"} (residuum), and \code{"b"} (bi-residuum).
}
%\references{
%% ~put references to the literature/web site here ~
%}
\author{
    Michal Burda
}
%\note{
%%  ~~further notes~~
%}

%% ~Make other sections like Warning with \section{Warning }{....} ~

%\seealso{
%}
\examples{
    # direct and parallel version of functions
    goedel.tnorm(c(0.3, 0.2, 0.5), c(0.8, 0.1, 0.5))  # 0.1
    pgoedel.tnorm(c(0.3, 0.2, 0.5), c(0.8, 0.1, 0.5)) # c(0.3, 0.1, 0.5)

    # handling of missing values
    goedel.tnorm(c(0.3, 0, NA), na.rm=TRUE)    # 0
    goedel.tnorm(c(0.3, 0.7, NA), na.rm=TRUE)  # 0.3

    goedel.tnorm(c(0.3, 0, NA), na.rm=FALSE)   # 0
    goedel.tnorm(c(0.3, 0.7, NA), na.rm=FALSE) # NA

    goedel.tconorm(c(0.3, 1, NA), na.rm=TRUE)    # 1
    goedel.tconorm(c(0.3, 0.7, NA), na.rm=TRUE)  # 0.7

    goedel.tconorm(c(0.3, 1, NA), na.rm=FALSE)   # 1
    goedel.tconorm(c(0.3, 0.7, NA), na.rm=FALSE) # NA

    # algebras
    x <- runif(10)
    y <- runif(10)
    a <- algebra('goedel')
    a$n(x)     # negation
    a$t(x, y)  # t-norm
    a$pt(x, y) # parallel t-norm
    a$c(x, y)  # t-conorm
    a$pc(x, y) # parallel t-conorm
    a$r(x, y)  # residuum
    a$b(x, y)  # bi-residuum

    is.algebra(a) # TRUE

}
% Add one or more standard keywords, see file 'KEYWORDS' in the
% R documentation directory.
\keyword{ models }
\keyword{ robust }
