% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/frbe.R
\name{frbe}
\alias{frbe}
\title{Fuzzy Rule-Based Ensemble (FRBE) of time-series forecasts}
\usage{
frbe(d, h = 10)
}
\arguments{
\item{d}{A source time-series in the ts time-series format.  Note that the
frequency of the time-series must to be set properly.}

\item{h}{A forecasting horizon, i.e. the number of values to forecast.}
}
\value{
Result is a list of class \code{frbe} with the following elements:
\itemize{
\item \code{features} - a data frame with computed features of the given time-series;
\item \code{forecasts} - a data frame with forecasts to be ensembled;
\item \code{weights} - weights of the forecasting methods as inferred from the features
and the hard-wired fuzzy rule base;
\item \code{mean} - the resulting ensembled forecast (computed as a weighted sum
of forecasts).
}
}
\description{
This function computes the fuzzy rule-based ensemble of time-series
forecasts.  Several forecasting methods are used to predict future values of
given time-series and a weighted sum is computed from them with weights
being determined from a fuzzy rule base.
}
\details{
This function computes the fuzzy rule-based ensemble of time-series
forecasts.  The evaluation comprises of the following steps:
\enumerate{
\item Several features are extracted from the given time-series \code{d}:
\itemize{
\item length of the time-series;
\item strength of trend;
\item strength of seasonality;
\item skewness;
\item kurtosis;
\item variation coefficient;
\item stationarity;
\item frequency.
These features are used later to infer weights of the forecasting methods.
}
\item Several forecasting methods are applied on the given time-series \code{d} to
obtain forecasts. Actually, the following methods are used:
\itemize{
\item ARIMA - by calling \code{\link[forecast:auto.arima]{forecast::auto.arima()}};
\item Exponential Smoothing - by calling \code{\link[forecast:ets]{forecast::ets()}};
\item Random Walk with Drift - by calling \code{\link[forecast:naive]{forecast::rwf()}};
\item Theta - by calling [forecast::thetaf().
}
\item Computed features are input to the fuzzy rule-based inference mechanism
which yields into weights of the forecasting methods. The fuzzy rule base is
hardwired in this package and it was obtained by performing data mining with
the use of the \code{\link[=farules]{farules()}} function.
\item A weighted sum of forecasts is computed and returned as a result.
}
}
\examples{
  # prepare data (from the forecast package)
  library(forecast)
  horizon <- 10
  train <- wineind[-1 * (length(wineind)-horizon+1):length(wineind)]
  test <- wineind[(length(wineind)-horizon+1):length(wineind)]

  # perform FRBE
  f <- frbe(ts(train, frequency=frequency(wineind)), h=horizon)

  # evaluate FRBE forecasts
  evalfrbe(f, test)

  # display forecast results
  f$mean

}
\references{
Štěpnička, M., Burda, M., Štěpničková, L. Fuzzy Rule Base
Ensemble Generated from Data by Linguistic Associations Mining. FUZZY SET
SYST. 2015.
}
\seealso{
\code{\link[=evalfrbe]{evalfrbe()}}
}
\author{
Michal Burda
}
\keyword{models}
\keyword{robust}
