% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/orbbound.R
\name{orbbound}
\alias{orbbound}
\title{Sensitivity Analysis for Outcome Reporting Bias (ORB)}
\usage{
orbbound(x, k.suspect = 1, tau = x$tau, left = NULL,
  backtransf = x$backtransf)
}
\arguments{
\item{x}{An object of class \code{meta}.}

\item{k.suspect}{Number of studies with suspected outcome reporting
bias.}

\item{tau}{Square-root of between-study variance tau-squared.}

\item{left}{A logical indicating whether the cause of any selection
bias is due to missing studies on the left or right of the funnel
plot: left hand side if \code{left=TRUE}, right hand side if
\code{left=FALSE}. If not set, the linear regression test for
funnel plot asymmetry (i.e., function \code{metabias(...,
meth="linreg")}) is used to determine whether studies are missing
on the left or right hand side.}

\item{backtransf}{A logical indicating whether results should be
back transformed in printouts and plots. If
\code{backtransf=TRUE} (default), results for \code{sm="OR"} are
printed as odds ratios rather than log odds ratio, for example.}
}
\value{
An object of class \code{c("orbbound")} with corresponding
  \code{print} and \code{forest} function. The object is a list
  containing the following components:
\item{k.suspect, tau}{As defined above.}
\item{maxbias}{Maximum bias for given values of \code{k.suspect}.}
\item{fixed}{Adjusted treatment estimates and corresponding
  quantities for fixed effect model (a list with elements TE, seTE,
  lower, upper, z, p, level, df).}
\item{random}{Adjusted treatment estimates and corresponding
  quantities for random effects model (a list with elements TE,
  seTE, lower, upper, z, p, level, df).}
\item{left}{Whether selection bias expected on left or right}
  \item{x}{Meta-analysis object (i.e. argument \code{x} from
  function call).}
\item{call}{Function call.}
\item{version}{Version of R package metasens used to create
  object.}
}
\description{
Implementation of the method by Copas & Jackson (2004) to evaluate
outcome reporting bias in meta-analysis. An upper bound for outcome
reporting bias is estimated for a given number of studies suspected
with outcome reporting bias.
}
\details{
This function provides the method by Copas and Jackson (2004) to
estimate an upper bound for bias for a given number of studies with
suspected outcome reporting bias.

Based on the upper bound of outcome reporting bias, treatment
estimates and confidence limits adjusted for bias are calculated.

For comparison, the original meta-analysis is always considered in
the sensitivity analysis (i.e. value 0 is always added to
\code{k.suspect}).
}
\examples{
data(Fleiss93, package = "meta")

m1 <- metabin(event.e, n.e, event.c, n.c, data = Fleiss93, sm = "OR")

orb1 <- orbbound(m1, k.suspect = 1:5)
print(orb1, digits = 2)
forest(orb1, xlim = c(0.75, 1.5))

# Same result
#
orb2 <- orbbound(m1, k.suspect = 1:5, left = FALSE)
print(orb2, digits = 2)

# Assuming bias in other direction
#
orb3 <- orbbound(m1, k.suspect = 1:5, left = TRUE)
print(orb3, digits = 2)
}
\references{
Copas J, Jackson D (2004):
A bound for publication bias based on the fraction of unpublished
studies.
\emph{Biometrics},
  \bold{60}, 146--53
}
\seealso{
\code{\link{forest.orbbound}}, \code{\link{print.orbbound}}
}
\author{
Guido Schwarzer \email{sc@imbi.uni-freiburg.de}
}
