% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/do_possession.R
\name{do_possession}
\alias{do_possession}
\title{Compute when possessions start}
\usage{
do_possession(data, period_sel, time_point_start)
}
\arguments{
\item{data}{Play-by-play prepared data from a given game.}

\item{period_sel}{Period of interest. Options can be "xC", where x=1,2,....
If NULL, no filtering is done.}

\item{time_point_start}{Starting time point of the lineup.}
}
\value{
Data frame.  This is the meaning of the columns that might not be 
explanatory by themselves:
\itemize{
 \strong{time_start}: Time point when the action starts.
 \strong{time_end}: Time point when the action ends.
 \strong{poss_time}: Duration of the possession.
 \strong{possession}: Indicates when the possession starts. This is encoded
  with the Spanish word \emph{inicio} (\emph{start}, in English).
 \strong{points}: Number of points scored from a given action.
}
}
\description{
Compute when the possession starts for each team in a game.
}
\note{
1. A possession lasts 24 seconds in the ACB league.

2. Actions are given in Spanish. A bilingual basketball vocabulary (Spanish/English)
is provided in \url{https://www.uv.es/vivigui/docs/basketball_dictionary.xlsx}.

3. The \strong{game_code} column allows us to detect the source website, for example,
\url{https://live.acb.com/es/partidos/103389/jugadas}.
}
\examples{
\dontrun{
library(dplyr)
df0 <- acb_vbc_cz_pbp_2223

day_num <- unique(acb_vbc_cz_pbp_2223$day)
game_code <- unique(acb_vbc_cz_pbp_2223$game_code)

# Starting players:
acb_games_2223_sl <- acb_vbc_cz_sl_2223 \%>\%
  dplyr::filter(period == "1C")

# Prepare data:
df1 <- do_prepare_data(df0, day_num, 
                       acb_games_2223_sl, acb_games_2223_info,
                       game_code)

teams_game <- sort(unique(df1$team))
team_sel <- teams_game[1]

data <- df1
data <- data \%>\%
  mutate(row_num = row_number()) \%>\%
  mutate(time_point = ifelse(nchar(time_point) < 5, paste0("0", time_point), time_point))

# Filter by team:
data1 <- data \%>\%
  filter(team == team_sel)

# Set also the opponent team:
team_opp <- setdiff(unique(data$team), team_sel)

# Add the last row of games' data to have the real final 
# game score in case it is not available:
last_row_game <- data[nrow(data),]

last_row_game$time_point <- "00:00"
last_row_game$player <- NA
last_row_game$action <- NA
last_row_game$team <- team_sel

data1 <- bind_rows(data1, last_row_game)

# Get players out:
pl_out <- c(1, which(data1$action == "Sale de la pista"), nrow(data1))

i <- 1
data2 <- data1 \%>\%
  slice(pl_out[i]:pl_out[i + 1]) 

nr <- nrow(data2)

# Lineup:
lineup <- data2 \%>\%
  filter(action != "Sale de la pista") \%>\%
  # Avoid actions that are assigned to teams: 
  filter(player != team_sel) \%>\%
  distinct(player) \%>\%
  pull()

# Identify when the possessions start:
data2_rival <- data \%>\%
  filter(team == team_opp) \%>\%
  filter(between(row_num, data2$row_num[1], data2$row_num[nr]))

data3 <- rbind(data2, data2_rival) \%>\%
  arrange(row_num) \%>\%
  na.omit()

data4 <- do_possession(data3, NULL, "10:00") 
}

}
\author{
Guillermo Vinue
}
