#' findExposures
#'
#' This function is used to find the signature exposures of a set of samples
#' using fixed signatures found earlier. It does this using the least squares
#' optimization method with constraints to keep the output as non-negative 
#' using the lsei function from the package limSolve. 
#' 
#' @param scm Sample by component matrix
#' @param fixedSigs The fixed signatures
#' @param runName Name of the run used in file names, Default is ""
#' @param saveRes Whether or not to save the results, Default is FALSE
#' @param saveDir Where to save the results, must be provided if using saveDir
#' @keywords fixed signatures lsei
#' @return Returns the resulting matrix of exposures
#' @export
#' @examples
#' findExposures(t(scmExp), sigsExp)
findExposures = function(scm, fixedSigs,runName="", saveRes = FALSE, saveDir = NULL) {
   
   #saves input as a matrix
   sig_matrix = as.matrix(fixedSigs)
   out_exposures = data.frame()
   
   nSigs = ncol(sig_matrix)

   #Constraints to keep output non-negative
   G = diag(nSigs)
   H = rep(0, nSigs)

   #Performs least squares optimization on each column of the scm
   for (i in 1:ncol(scm)){
      fractions = limSolve::lsei(A = sig_matrix, B = scm[,i], G = G, H = H, type = 2)
      exposure_vec = round(fractions[["X"]], digits = 6)
      names(exposure_vec) = names(fixedSigs)
      rel_exposure = exposure_vec/sum(exposure_vec)
      out_exposures[1:nSigs,i] = rel_exposure
   }

   rownames(out_exposures) = colnames(fixedSigs)
   colnames(out_exposures) = colnames(scm)

   #Tries to recreate the NMF plots to save for the output.
   if (saveRes){
      if (is.null(saveDir)){
	stop('No saveDir provided for saveRes. Please set saveDir.')
      }
      toWrite = t(out_exposures)
      colnames(toWrite) = paste0(rep("S",nSigs),1:nSigs)
      rownames(toWrite) = colnames(scm)
      utils::write.table(toWrite,sep = '\t',
                  file=paste0(saveDir,"/",runName,"_sigExposureMatrix.txt"))

      toWrite = fixedSigs
      colnames(toWrite) = paste0(rep("S",nSigs),1:nSigs)
      utils::write.table(toWrite, sep = '\t',
                  file=paste0(saveDir,"/",runName,"_featureContribution.txt"))
   }

   return(out_exposures)
}
