#' fitComponent
#'
#' This function is used to fit a mixture model of either normal or poisson
#' distributions to the inpput data. This function is mainly used by the
#' fitModels function to create the components from the extracted features.
#' 
#' @param toFit Extracted features to fit models to.
#' @param min_prior Minimum prior probability of a cluster. Default is 0.001.
#' @param min_comp Minimum number of models to fit. Default is 2.
#' @param max_comp Maximum number of models to fit. Default is 10.
#' @param dist Type of distribution to fit. Either "norm" or "pois". Default "norm"
#' @param pR Peak Reduction reduces peaks in modeling to make modeling easier. Default is FALSE.
#' @param seed Seed to be used for modeling. Default is 77777
#' @param model_sel Type of model_selection method to be used. Default "BIC". See flexmix package for more options.
#' @param niter Max number of iterations for modeling. Default is 1000.
#' @param nrep Number of repetitions for modeling attempts. Default is 1.
#' @return Returns the flexmix object for the fit model.
#' @import flexmix
#' @export
#' @examples
#' fitComponent(featsExp$bp10MB[,2]) #Fits 2-10 normal distributions
#' 
#' #Tries to fit exactly 4 poisson distributions
#' fitComponent(featsExp$osCN[,2],dist="pois",min_comp = 4, max_comp = 4)
fitComponent = function(toFit,min_prior = 0.001,min_comp = 2, max_comp = 10,
                        dist = "norm", pR = FALSE, seed = 77777,
                        model_sel="BIC", niter = 10000, nrep = 1)
{
   supportedDists = c("norm","dist")
   
   #Performs either normal or poisson modelling
   distModel = switch(dist,
                      "norm" = flexmix::FLXMCnorm1(),
                      "pois" = flexmix::FLXMCmvpois(),
                      stop(paste("Unsupported distribution type.\nSupported",
                                 "types are:",
                                 paste(supportedDists,collapse=","))))
   
   #Creates new flexmix controller with parameters
   control = methods::new("FLXcontrol")
   control@minprior = min_prior
   control@iter.max = niter

   set.seed(seed)
   
   #Reduces the peaks if specified
   if (pR)
      toFit = reducePeaks(toFit)
   

   
   fit = flexmix::stepFlexmix(toFit~1,model = distModel,k = min_comp:max_comp,
                               nrep=nrep,control=control)

   #If you specified an exact number, the fit is already a single model
   if (min_comp != max_comp){
      if (!is(fit, "flexmix"))
         fit = flexmix::getModel(fit, which = model_sel)
   }
   
   #Checks for overlapping components
   params = flexmix::parameters(fit)
   finalParams = checkCompOverlap(params,dist=="pois")
   

   return(finalParams)
}
