% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/RNoughtIndices.R
\name{RNoughtIndices}
\alias{RNoughtIndices}
\title{R0 Index computation on s2dv_cube objects}
\usage{
RNoughtIndices(temp, method, mm, lon_dim, lat_dim, ncores = NULL)
}
\arguments{
\item{temp}{An array with temperature values, expressed in degrees Celsius. If
"caminade" is selected in "method", named spatial dimensions are required in the s2dv_cube object, 
ordered as ("latitude", "longitude"). Valid dimension names are "lon", "longitude", "lat", "latitude".}

\item{method}{a string indicating the R0 ento-epidemiological
model used to obtain the R0 outputs. Methods include "caminade",
"wesolowski", "liuhelmerssohn" and "mordecai". Alternatively, an 
extrapolation to real life data can be made by setting "method" to "empirical".}

\item{mm}{A Kramer probability matrix of dimensions (2, longitude, latitude) to be passed
if "caminade" is selected as the computation method. The spatial dimensions must
match those of temp and be equal in length. Valid dimension names are "lon", "longitude",
"lat", "latitude".}

\item{lon_dim}{a character vector indicating the longitude dimension name in
the element 'temp' if "caminade" is chosen in "method". Otherwise, can be set to NULL}

\item{lat_dim}{a character vector indicating the latitude dimension name in
the element 'temp' if "caminade" is chosen in "method". Otherwise, can be set to NULL}

\item{ncores}{An integer indicating the number of cores to use in parallel
computation for temporal subsetting.}
}
\value{
An s2dv_cube object containing the R0 Indices (unitless).
}
\description{
This function computes the environmental contribution to Aedes-borne
disease transmissibility. Values higher than 1 imply that the environmental conditions allow for 
the disease to be transmitted to more than 1 person (the infection can spread) while R0 values lower
 than 1 imply that the environmental conditions are not suitable for the disease to spread.

This function utilizes four possible ento-epidemiological models, each of them stated
in Caminade et al., 2015; Liu-Helmerssohn et al., 2014; and Mordecai et
al., 2017 and Wesolowski et al., 2015 respectively. Additionally, an adjustment to
real life data, using recorded DENV data recorded between 2014-2017 in the Caribbean, can also be performed.
}
\examples{
dims <- c(time = 100, lat = 5, lon = 4)
temp <- array(
    runif(prod(dims), min = 15, max = 35),
    dim = dims
  )
  
R0 <- RNoughtIndices(temp, method = "mordecai", mm = NULL, lon_dim = NULL, 
lat_dim = NULL, ncores = NULL)
# Caminade method (requires Kramer matrix)
mm <- array(runif(2 * 5 * 4, 0.1, 1.0), 
            dim = c(2, 4, 5))
names(dim(mm)) <- c("", "lon", "lat")
R0 <- RNoughtIndices(temp, method = "caminade", mm = mm, lon_dim = "lon", 
lat_dim = "lat", ncores = NULL)

}
\references{
Caminade, Cyril et al. (Jan. 2017). ‘Global risk model for vector-borne 
transmission of Zika virus reveals the role of El Niño 2015’.
In: Proceedings of the National Academy of Sciences 114.1.
Publisher: Proceedings of the National Academy of 
Sciences, pp. 119–124. doi: 10.1073/pnas.1614303114. 
url: https://www.pnas.org/doi/full/10.1073/pnas.1614303114.

Liu-Helmerssohn, Jing et al. (Mar. 2014). ‘Vectorial Capacity of Aedes
aegypti: Effects of Temperature and Implications for Global Dengue Epidemic Potential’. en.
In: PLOS ONE 9.3.
Publisher: Public Library of Science, e89783.
issn: 1932-6203. doi: 10.1371/journal.pone.0089783.
url: https://journals.plos.org/plosone/article?id=10.1371/journal.pone.0089783.

Mordecai, Erin A. et al. (Apr. 2017). ‘Detecting the impact of temperature on
transmission of Zika, dengue, and chikungunya using mechanistic models’. en. 
In: PLOS Neglected Tropical Diseases 11.4.
Publisher: Public Library of Science, e0005568.
issn: 1935-2735. doi: 10.1371/journal.pntd.0005568.
url: https://journals.plos.org/plosntds/article?id=10.1371/journal.pntd.0005568.

Wesolowski, Amy et al. (Sept. 2015). ‘Impact of human mobility on the emergence
of dengue epidemics in Pakistan’. 
In: Proceedings of the National Academy of Sciences 112.38.
Publisher: Proceedings of the National Academy of Sciences, pp. 11887–11892.
doi: 10.1073/pnas.1504964112.
url: https://www.pnas.org/doi/full/10.1073/pnas.1504964112.
}
\author{
Javier Corvillo, \email{javier.corvillo@bsc.es}
}
