library(testthat)
library(CoxMK)

test_that("Package loads correctly", {
  expect_true("CoxMK" %in% loadedNamespaces())
})

test_that("Example data loads correctly", {
  extdata_path <- system.file("extdata", package = "CoxMK")
  expect_true(dir.exists(extdata_path))
  
  # Check PLINK files exist
  expect_true(file.exists(file.path(extdata_path, "sample.bed")))
  expect_true(file.exists(file.path(extdata_path, "sample.bim")))  
  expect_true(file.exists(file.path(extdata_path, "sample.fam")))
  
  # Check phenotype and covariate files exist
  expect_true(file.exists(file.path(extdata_path, "tte_phenotype.txt")))
  expect_true(file.exists(file.path(extdata_path, "covariates.txt")))
})

test_that("Core functions work", {
  # Test with simple simulated data instead of loading PLINK files
  # This avoids dependency on internal functions and file I/O

  # Create simple test data
  set.seed(123)
  n_samples <- 20
  n_snps <- 10

  # Simulate genotype data (0, 1, 2 for SNP dosages)
  X <- matrix(sample(0:2, n_samples * n_snps, replace = TRUE),
              nrow = n_samples, ncol = n_snps)

  # Simulate SNP positions
  pos <- seq(1000, 1000 + 9*100, by = 100)

  # Test knockoff creation
  knockoffs <- create_knockoffs(
    X = X,
    pos = pos,
    M = 2,
    save_gds = FALSE,  # Don't save GDS for testing
    output_dir = tempdir()  # Explicitly set temp directory
  )

  expect_true("knockoffs" %in% names(knockoffs))
  expect_equal(length(knockoffs$knockoffs), 2)
  expect_equal(dim(knockoffs$knockoffs[[1]]), dim(X))

  # Test W statistics calculation
  t_orig <- rnorm(n_snps)
  t_knock <- matrix(rnorm(n_snps * 2), nrow = n_snps, ncol = 2)

  W_stats <- calculate_w_statistics(t_orig, t_knock)
  expect_equal(length(W_stats), n_snps)

  # Test knockoff filter
  selected <- knockoff_filter(W_stats, fdr = 0.1)
  expect_true(is.numeric(selected) || is.integer(selected))
})