##### QmdTemplate #####

#' Set the path to the Quarto task template.
#' @param file path to the Quarto task template.
#' @inheritParams base::readLines
#' @return The path to the Quato task template invisibly.
#' @export
setTaskQmdTemplate <- function(file,encoding="unknown"){
  if(file=="SQU4RE") {
    f <- system.file("template/SQU4REtemplate.Qmd",package="D4TAlink.light")
    file.copy(f,rfn<-tempfile(fileext=".Qmd"))
    dfn <- system.file("template/SQU4REtemplate.docx",package="D4TAlink.light")
    tin <- readLines(rfn,encoding=encoding,warn=FALSE)
    tin <- gsub("%TEMPLATEFILE%",dfn,tin,fixed=TRUE)
    writeLines(enc2utf8(tin),rfn,useBytes=TRUE)
    file <- rfn
  }
  assign(x="rmdTempl",value=file,pos=.pkgEnv)
  invisible(getTaskQmdTemplate())
}

#' Get the path to the Quarto task template.
#' @inheritParams D4getenv
#' @return The path to the quarto task template.
#' @export
getTaskQmdTemplate <- function(quiet=FALSE)
  D4getenv("rmdTempl","Quarto template",quiet=quiet)

## =======================================================================
#' Get path of R script file name.
#' @inheritParams D4TAlink-common-args
#' @return File path.
#' @export
qmdFn <- function(task,suffix=NA) {
  if(is.na(suffix)) return(file.path(getTaskPaths(task)[["code"]],paste0(task$task,".Qmd")))
  else return(file.path(getTaskPaths(task)[["code"]],paste0(task$task,".",suffix,".Qmd")))
}

## =======================================================================
#' Create task template in Quarto format.
#' @param overwrite overwrite Qmd file if exists, default FALSE
#' @inheritParams D4TAlink-common-args
#' @inheritParams base::readLines
#' @return the file name invisibly.
#' @export
initTaskQmd <- function(task,encoding="unknown",overwrite=FALSE,suffix=NA) {
  fn <- qmdFn(task,suffix=suffix)
  if(file.exists(fn)&&!overwrite) stop("The task Quarto file already exists. Set 'overwrite' to FALSE to overwrite the existing file.")
  tfn <- getTaskQmdTemplate()
  tin <- readLines(tfn,encoding=encoding,warn=FALSE)
  tin <- gsub("%DATE%"    ,task$date,tin,fixed=TRUE)
  tin <- gsub("%TASKID%",taskID(task),tin,fixed=TRUE)
  tin <- gsub("%TASKIDX%",taskID(task),tin,fixed=TRUE)
  tin <- gsub("%ROOT%",getTaskRoot(),tin,fixed=TRUE)
  tin <- gsub('"%SUFFIX%"',ifelse(is.na(suffix),"NA",paste0('"',suffix,'"')),tin,fixed=TRUE)
  tin <- gsub(" - %SUFFIX%",ifelse(is.na(suffix),"",paste0(" - ",suffix)),tin,fixed=TRUE)
  tin <- gsub("%SUFFIX%",ifelse(is.na(suffix),"",suffix),tin,fixed=TRUE)
  tin <- gsub("%YEAR%"    ,format(as.POSIXct(task$date),"%Y"),tin,fixed=TRUE)
  u <- unlist(task)
  for(n in names(u)) {
    tin <- gsub(sprintf("%%%s%%",toupper(n)),u[[n]],tin,fixed=TRUE)
    tin <- gsub(sprintf("%%%sX%%",toupper(n)),u[[n]],tin,fixed=TRUE)
  }
  for(e in names(task)[sapply(task,is.character)])
    tin <- gsub(paste0("%",toupper(e),"%"),task[[e]],tin,fixed=TRUE)
  tin <- gsub(paste0("%","TYPE","%"),"task report",tin,fixed=TRUE)
  writeLines(enc2utf8(tin),fn,useBytes=TRUE)
  invisible(fn)
}


#' Render the task from the Qmd file
#'
#' The template of the task is rendered towards pdf or html in the documentation directory of the specified task.
#' Note that on windows, Gnu zip may be required. The path to the executable must be added to the environment variables.
#' @param debug if TRUE execute in the global environment.
#' @inheritParams D4TAlink-common-args
#' @inheritParams quarto::quarto_render
#' @inheritDotParams quarto::quarto_render
#' @importFrom quarto quarto_render
#' @return the file name invisibly.
#' @export
renderTaskQmd <- function(task,output_format="docx",debug=FALSE,suffix=NA,...){
  ifn <- qmdFn(task,suffix=suffix)
  if(!file.exists(ifn)) stop("Quarto file not found, create it using 'initTaskQmd'.")
  odr <- getTaskPaths(task)[["doc"]]
  ofn <- gsub("Qmd$",output_format,ifn,ignore.case=TRUE)
  fn <- file.path(D4TAlink.light::docDir(task),basename(ofn))
  if(debug) stop("not implemented")
  else quarto::quarto_render(input=ifn,
                             output_file=basename(ofn),
                             output_format=output_format,...)
  file.rename(ofn,fn)
  if(grepl("docx$",fn)) formatTaskDocx(task,fn)
  invisible(fn)
}

