#'
#' Load the Scaler for the pre-trained Neural Networks for Determining the Number of Factors
#'
#' @description
#' Loads the scaler object within the \code{EFAfactors} package. This object is a \code{list} containing a mean vector and
#' a standard deviation vector, which were computed from the 10,000,000 datasets \code{\link[EFAfactors]{data.datasets.DNN}}
#' training the Deep Neural Network (DNN) or the 1,000,000 datasets \code{\link[EFAfactors]{data.datasets.LSTM}}
#' training the Long Short Term Memory (LSTM) Network. It serves as a tool for normalizing features in
#' \code{\link[EFAfactors]{NN}}.
#'
#' @param model A character string indicating the model type. Possible values are \code{"DNN"} (default) or \code{"LSTM"}.
#'
#' @seealso \code{\link[EFAfactors]{NN}}, \code{\link[EFAfactors]{normalizor}},
#' \code{\link[EFAfactors]{data.scaler.DNN}}, \code{\link[EFAfactors]{data.scaler.LSTM}}
#'
#' @return scaler objective.
#'
#' @examples
#' library(EFAfactors)
#'
#' scaler <- load.scaler()
#' print(scaler)
#'
#'
#' @export
#'
load.scaler <- function(model="DNN") {
  scaler.path <- system.file("data", paste0("data.scaler.", model,".rda"), package = "EFAfactors")

  local.env <- new.env()
  load(scaler.path, envir = local.env)
  data.scaler <- get(ls(local.env)[1], envir = local.env)

  return(data.scaler)
}

#'
#' Load the Tuned XGBoost Model
#'
#'
#' @description
#' Loads the tuned XGBoost model object within the \code{EFAfactors} package
#' into the global environment and retrieves it for use. Only the core model is retained to reduce the size.
#'
#' @return The tuned XGBoost model object
#'
#' @examples
#' library(EFAfactors)
#'
#' xgb.model <- load.xgb()
#' print(xgb.model)
#'
#'
#' @export
#'
load.xgb <- function() {
  xgb.path <- system.file("data", "model.xgb.rda", package = "EFAfactors")

  local.env <- new.env()
  load(xgb.path, envir = local.env)
  model.xgb <- get(ls(local.env)[1], envir = local.env)

  return(model.xgb)
}

#'
#' Load the the pre-trained Neural Networks for Determining the Number of Factors
#'
#' @description
#' Loads the pre-trained Deep Neural Network (DNN) from the \code{DNN.onnx} or Long Short
#' Term Memory (LSTM) Network form \code{LSTM.onnx}.
#' The function uses the \code{reticulate} package to import the \code{onnxruntime} Python library
#' and create an inference session for the model.
#' @seealso \link[EFAfactors]{NN}
#'
#' @param model A character string indicating the model type. Possible values are \code{"DNN"} (default) or \code{"LSTM"}.
#'
#' @return An ONNX runtime inference session object for the DNN or LSTM model.
#'
#' @note
#' Note that Python (suggested >= 3.11) and the libraries \code{numpy} and \code{onnxruntime} are required.
#'
#' First, please ensure that Python is installed on your computer and that Python is
#' included in the system's PATH environment variable. If not,
#' please download and install it from the official website (https://www.python.org/).
#'
#' If you encounter an error when running this function stating that the \code{numpy} and \code{onnxruntime}
#' modules are missing:
#'
#'  \code{Error in py_module_import(module, convert = convert) :}
#'
#'    \code{ModuleNotFoundError: No module named 'numpy'}
#'
#' or
#'
#'  \code{Error in py_module_import(module, convert = convert) :}
#'
#'    \code{ModuleNotFoundError: No module named 'onnxruntime'}
#'
#' this means that the \code{numpy} or \code{onnxruntime} library is missing from your Python environment. If you are using Windows or macOS,
#' please run the command \code{pip install numpy} or \code{pip install onnxruntime} in Command Prompt or Windows PowerShell (Windows), or Terminal (macOS).
#' If you are using Linux, please ensure that \code{pip} is installed and use the command \code{pip install numpy} or
#' \code{pip install onnxruntime} to install the missing libraries.
#'
#'
#' @export
#' @import reticulate
#' @importFrom reticulate import
#'
load.NN <- function(model="DNN") {
  NN.path <- system.file("extdata", paste0(model, ".onnx"), package = "EFAfactors")
  onnxruntime <- import("onnxruntime")
  NN <- onnxruntime$InferenceSession(NN.path)

  return(NN)
}
