% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/gareg_knots.R
\name{gareg_knots}
\alias{gareg_knots}
\title{Genetic-Algorithm–based Optimal Knot Selection}
\usage{
gareg_knots(
  y,
  x,
  ObjFunc = NULL,
  fixedknots = NULL,
  minDist = 3L,
  degree = 3L,
  type = c("ppolys", "ns", "bs"),
  intercept = TRUE,
  gaMethod = "cptga",
  cptgactrl = NULL,
  monitoring = FALSE,
  seed = NULL,
  ...
)
}
\arguments{
\item{y}{Numeric vector of responses (length \code{N}).}

\item{x}{Optional index/time vector aligned with \code{y}. If missing, it
defaults to \code{seq_along(y)}. Used to derive \code{x_unique} (candidate
knot positions) and passed to the objective function; the GA backend itself
does not use \code{x} directly.}

\item{ObjFunc}{Objective function or its name. If \code{NULL}, a default
is chosen:
\itemize{
  \item \code{fixknotsIC} when \code{fixedknots} is supplied;
  \item \code{varyknotsIC} otherwise.
}
A custom function must accept the chromosome and needed data via
named arguments (see the defaults for a template function).}

\item{fixedknots}{\code{NULL} (varying-knots search) or an integer giving the
number of interior knots for a fixed-\eqn{m} search. If non-\code{NULL}, the
method is \code{"fixknots"} and specialized operators are injected unless
overridden in \code{cptgactrl}.}

\item{minDist}{Integer minimum distance between adjacent changepoints.
If omitted (\code{missing()} or \code{NULL}), the value in \code{cptgactrl}
is used. If supplied here, it overrides the control value.}

\item{degree}{Integer polynomial degree for \code{"ppolys"} and \code{"bs"}.
Ignored for \code{"ns"} (always cubic). Must be provided for \code{"ppolys"}
and \code{"bs"}.}

\item{type}{One of \code{c("ppolys", "ns", "bs")}:
piecewise polynomials, natural cubic, or B-spline. See \code{\link{splineX}}.
The first option of `ppolys` is taken by default.}

\item{intercept}{Logical; include intercept column where applicable.
Default: \code{TRUE}.}

\item{gaMethod}{GA backend to call: function or name. Supports
\code{"cptga"} (single population) and \code{"cptgaisl"} (islands).}

\item{cptgactrl}{Control list built with \code{\link{cptgaControl}()}
(or a named list of overrides). When \code{gaMethod = "cptgaisl"},
island-specific knobs like \code{numIslands} and \code{maxMig} are
recognized. Other genetic algorithm parameters can be found in \link[changepointGA:cptga]{cptga} and
      \link[changepointGA:cptgaisl]{cptgaisl}.}

\item{monitoring}{Logical; print short progress messages (also forwarded
into the backend control).}

\item{seed}{Optional RNG seed; also stored into the backend control.}

\item{...}{Additional arguments passed to the GA backend. If the backend
does not accept \code{...}, unknown arguments are silently dropped
(the call is filtered against the backend formals).}
}
\value{
An object of class \code{"gareg"} with key slots:
\itemize{
  \item \code{call}, \code{method} (\code{"varyknots"} or \code{"fixknots"}), \code{N}.
  \item \code{objFunc}, \code{gaMethod}, \code{gaFit} (class \code{"cptga"} or \code{"cptgaisl"}), \code{ctrl}.
  \item \code{fixedknots}, \code{minDist}, \code{polydegree}, \code{type}, \code{intercept}.
  \item \code{bestFitness}, \code{bestChrom}, \code{bestnumbsol}, \code{bestsol}.
}
Use \code{summary(g)} to print GA settings and the best solution (extracted
from \code{g@gaFit}); \code{show(g)} prints a compact header.
}
\description{
Runs a GA-based search for changepoints/knots and returns a compact
\code{"gareg"} S4 result that stores the backend GA fit
(\code{"cptga"} or \code{"cptgaisl"}) plus the essential run settings.
}
\details{
\strong{Engine selection and controls.}
The function detects the engine from \code{gaMethod} and constructs a
matching control via \code{\link{cptgaControl}()}:
\itemize{
  \item \code{"cptga"} uses \code{.cptga.default}.
  \item \code{"cptgaisl"} uses \code{.cptgaisl.default} (supports
        \code{numIslands}, \code{maxMig}, etc.).
  \item see other details in \link[changepointGA:cptga]{cptga} and
        \link[changepointGA:cptgaisl]{cptgaisl}.
}
Top-level \code{monitoring}, \code{seed}, and \code{minDist} given to
\code{gareg_knots()} take precedence over the control list.

\strong{Fix-knots operators.}
When \code{fixedknots} is provided and the control does not already
override them, the following operators are injected:
\code{Popinitial_fixknots}, \code{crossover_fixknots}, \code{mutation_fixknots}.

\strong{Spline basis options.}
To build spline design matrices (via \code{\link{splineX}}):
\itemize{
  \item \code{type = "ppolys"}: Degree-\eqn{d} regression spline via truncated-power piecewise polynomials.
  \item \code{type = "ns"}: Degree-3 natural cubic spline with zero second-derivative at boundaries.
  \item \code{type = "bs"}: Degree-\eqn{d} B-spline basis (unconstrained).
}
}
\section{Argument precedence}{

Values are combined as \emph{control < core < \code{...}}. That is,
\code{cptgactrl} provides defaults, then core arguments from
\code{gareg_knots()} override those, and finally any matching names in
\code{...} override both.
}

\examples{
\donttest{
set.seed(1)
N <- 120
y <- c(rnorm(40, 0), rnorm(40, 3), rnorm(40, 0))
x <- seq_len(N)

# 1) Varying-knots with single-pop GA
g1 <- gareg_knots(
  y, x,
  minDist = 5,
  gaMethod = "cptga",
  cptgactrl = cptgaControl(popSize = 150, pcrossover = 0.9, maxgen = 500)
)
summary(g1)

# 2) Fixed knots (operators auto-injected unless overridden)
g2 <- gareg_knots(
  y, x,
  fixedknots = 5,
  minDist = 5
)
summary(g2)

# 3) Island GA with island-specific controls
g3 <- gareg_knots(
  y, x,
  gaMethod = "cptgaisl",
  minDist = 6,
  cptgactrl = cptgaControl(
    engine = "cptgaisl",
    numIslands = 8, maxMig = 250,
    popSize = 120, pcrossover = 0.9
  )
)
summary(g3)
}

}
\seealso{
\code{\link{cptgaControl}}, \code{changepointGA::cptga},
  \code{changepointGA::cptgaisl}, \code{\link{fixknotsIC}}, \code{\link{varyknotsIC}}
}
