% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/GeodRegr.R
\name{geo_reg}
\alias{geo_reg}
\title{Gradient descent for (robust) geodesic regression}
\usage{
geo_reg(
  manifold,
  x,
  y,
  estimator,
  c = NULL,
  p_tol = 1e-05,
  V_tol = 1e-05,
  max_iter = 1e+05
)
}
\arguments{
\item{manifold}{Type of manifold (\code{'euclidean'}, \code{'sphere'},
\code{'hyperbolic'}, or \code{'kendall'}).}

\item{x}{A vector, matrix, or data frame of independent variables; for
matrices and data frames, the rows and columns represent the subjects and
independent variables, respectively.}

\item{y}{A matrix or data frame whose columns represent points on the
manifold.}

\item{estimator}{M-type estimator (\code{'l2'}, \code{'l1'}, \code{'huber'},
or \code{'tukey'}).}

\item{c}{Multiplier of \eqn{\sigma}, the square root of the variance, used in
the cutoff parameter for the \code{'huber'} and \code{'tukey'} estimators;
should be \code{NULL} for the \code{'l2'} or \code{'l1'} estimators.}

\item{p_tol}{Termination condition for the distance between consecutive
updates of \code{p}.}

\item{V_tol}{Termination condition for the distance between columns of
consecutive updates of \code{V}, parallel transported to be in the same
tangent space. Can be a vector of positive real numbers for each
independent variable or a single positive number.}

\item{max_iter}{Maximum number of gradient descent steps before ending the
algorithm.}
}
\value{
A named list containing \item{p}{a vector representing the estimate
  of the initial point on the manifold} \item{V}{a matrix representing the
  estimate of the initial velocities for each independent variable; the
  columns represent the independent variables.} \item{iteration}{number of
  gradient descent steps taken.}
}
\description{
Finds \eqn{\mathrm{argmin}_{(p,V)\in M\times (T_pM) ^ n}\sum_{i=1} ^ {N}
\rho(d(\mathrm{Exp}(p,Vx_i),y_i))} through a gradient descent algorithm.
}
\details{
Each column of \code{x} should be centered to have an average of 0 for the
quickest and most accurate results. If all of the elements of a column of
\code{x} are equal, the resulting vector will consist of \code{NA}s. In the
case of the \code{'sphere'}, an error will be raised if all points are on a
pair of antipodes.
}
\examples{
# an example of multiple regression with two independent variables, with 64
# data points

x <- matrix(runif(2 * 64), ncol = 2)
x <- t(t(x) - colMeans(x))
y <- matrix(0L, nrow = 4, ncol = 64)
for (i in 1:64) {
  y[, i] <- exp_map('sphere', c(1, 0, 0, 0), c(0, runif(1), runif(1),
      runif(1)))
}
geo_reg('sphere', x, y, 'tukey', c = are_nr('tukey', 2, 6))

}
\references{
Fletcher, P. T. (2013). Geodesic regression and the theory of
  least squares on Riemannian manifolds. International Journal of Computer
  Vision, 105, 171-185.

  Kim, H. J., Adluru, N., Collins, M. D., Chung, M. K., Bendin, B. B.,
  Johnson, S. C., Davidson, R. J. and Singh, V. (2014). Multivariate general
  linear models (MGLM) on Riemannian manifolds with applications to
  statistical analysis of diffusion weighted images. 2014 IEEE Conference on
  Computer Vision and Pattern Recognition, 2705-2712.

  Shin, H.-Y. and Oh H.-S. (2020). Robust Geodesic Regression.
  <arXiv:2007.04518>
}
\seealso{
\code{\link{intrinsic_location}}.
}
\author{
Ha-Young Shin
}
