#' Preferred Interval Length for Regularizing Irregular Longitudinal Data
#'
#' This function calculates the optimal interval length for regularizing
#' irregular longitudinal data based on the given subject ID and time columns.
#'
#' @param data A data frame containing the irregular longitudinal data.
#' @param subject_col The column name for unique subject IDs.
#' @param time_col The column name for time points.
#'
#' @return Computed preferred interval length.
#'
#' @details
#' The function calculates the optimal interval length based on the observed range of time points and the average number of measurements per subject.
#' @examples
#' sdata <- sdata[1:100,]
#' intlen(sdata, "subject_id", "time")
#'
#' @export

intlen <- function(data, subject_col, time_col) {
  # Ensure the specified columns exist
  if (!(subject_col %in% names(data)) || !(time_col %in% names(data))) {
    stop("Specified columns not found in the dataset.")
  }

  # Convert to data frame (if not already)
  data <- as.data.frame(data)

  # Compute min and max time across all subjects
  min_time <- min(data[[time_col]], na.rm = TRUE)
  max_time <- max(data[[time_col]], na.rm = TRUE)

  # Compute average number of time points per subject
  avg_time_points <- mean(table(data[[subject_col]]))

  # Compute optimal interval length
  optimal_interval <- (max_time - min_time) / avg_time_points

  # Return formatted message
  return(paste("Optimal interval length for regularization according to your data:", round(optimal_interval, 4)))
}
