% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/ig_icmotif_count.R
\encoding{UTF-8}
\name{ig_icmotif_count}
\alias{ig_icmotif_count}
\title{Count interconnection motifs for tripartite networks with intra-guild interactions}
\usage{
ig_icmotif_count(mat, guilds, weighted = FALSE)
}
\arguments{
\item{mat}{A square block interaction matrix representing a tripartite network including intra-guild and inter-guild interactions. See details.}

\item{guilds}{A character vector matching rows of \code{mat} to indicate the guilds using ('a','b' and 'c'). See details.}

\item{weighted}{Logical. Default to FALSE. If TRUE, the arithmetic mean of the subgraph weights is provided for each motif. See details}
}
\value{
Return a matrix of counts (and mean weight) of 107 interconnection motifs.
}
\description{
Counting the frequencies of interconnection motifs for a tripartite interaction network with intra-guild interactions.
}
\details{
This function is designed for tripartite networks with intra-guild interactions. The input network should be nput as a block matrix (\eqn{M}) to represent three groups of nodes (a-nodes, b-nodes and c-nodes): three intra-guild interaction matrices (\eqn{m_{aa},m_{bb},m_{cc}}),
two inter-guild matrices of a and b-nodes (\eqn{m_{ab},m_{ba}}), and two inter-guild matrices of b- and c-nodes(\eqn{m_{bc},m_{cb}}).
\deqn{
  \left(
    \begin{array}{ccc}
      m_{aa} & m_{ab} & 0        \\
      m_{ba} & m_{bb} & m_{bc}   \\
       0     & m_{cb} & m_{cc}
    \end{array}
  \right)
}

\code{guilds} should be a vector of the same length as the row of \code{mat} like c("a","a"..."b","b"..."c","c"..)

\strong{Interconnection motifs in tripartite networks with intra-guild interactions}
\if{html}{\out{<br>}}An interconnection motif is defined to comprise three sets of connected nodes: the connector nodes (belonging to b-nodes), the nodes in one subnetwork (belonging to a-nodes in the P subnetwork), and the nodes in the other subnetwork (belonging to c-nodes in the Q subnetwork). Each guild has two nodes at most, resulting in a total of 107 distinct motif forms.
The algorithm for counting interconnection motifs is designed by extending the fast approach from Simmons et al.(2019). For interconnection motifs in tripartite networks without intra-guild interactions, please see \strong{icmotif_count} and \strong{icmotif_role}.

\strong{Weighted networks}
\if{html}{\out{<br>}}For weighted tripartite networks, the mean weight of a given motif is provided by averaging the weights of all motif occurrences isomorphic to the motif. The weight of a motif occurrence is the arithmetic mean of the weights of its links, following Mora et al. (2018) and Simmons et al. (2019).
}
\examples{

##A toy tripartite network with intra-guild negative interactions:
##Inter-guild mutualistic interactions and inter-guild antagonistic interactions.
set.seed(12)
##4 a-nodes, 5 b-nodes, and 3 c-nodes

##Intra-guild interaction matrices
mat_aa<-matrix(runif(16,-0.8,-0.2),4,4)
mat_aa <- mat_aa + t(mat_aa); diag(mat_aa) <- 0
mat_bb<-matrix(runif(25,-0.8,-0.2),5,5)
mat_bb <- mat_bb + t(mat_bb); diag(mat_bb) <- 0
mat_cc<-matrix(runif(9,-0.8,-0.2),3,3)
mat_cc <- mat_cc + t(mat_cc); diag(mat_cc) <- 0

##Inter-guild interaction matrices between a- and b-nodes.
mat_ab<-mat_ba<-matrix(sample(c(rep(0,8),runif(12,0,0.5))),4,5,byrow=TRUE)
mat_ba<-t(mat_ba)

##Inter-guild interaction matrices between b- and c-nodes.
mat_cb<-mat_bc<-matrix(sample(c(rep(0,8),runif(7,0,0.5))),3,5,byrow=TRUE)
mat_bc<--t(mat_bc)
mat<-rbind(cbind(mat_aa,mat_ab,matrix(0,4,3)),cbind(mat_ba,mat_bb,mat_bc))
mat<-rbind(mat,cbind(matrix(0,3,4),mat_cb,mat_cc))

##set the node names
rownames(mat)<-c(paste0("a",1:4),paste0("b",1:5),paste0("c",1:3))
colnames(mat)<-c(paste0("a",1:4),paste0("b",1:5),paste0("c",1:3))

#mat[mat!=0] <- 1
myguilds=c(rep("a",4),rep("b",5),rep("c",3))
ig_icmotif_count(mat,guilds=myguilds,TRUE)


}
\references{
Garcia-Callejas, D., Godoy, O., Buche, L., Hurtado, M., Lanuza, J.B., Allen-Perkins, A. et al. (2023) Non-random interactions within and across guilds shape the potential to coexist in multi-trophic ecological communities. Ecology Letters, 26, 831-842.

Pilosof, S., Porter, M. A., Pascual, M., & KC)fi, S. (2017). The multilayer nature of ecological networks. Nature Ecology & Evolution, 1(4), 0101.

Mora, B.B., Cirtwill, A.R. and Stouffer, D.B. (2018). pymfinder: a tool for the motif analysis of binary and quantitative complex networks. bioRxiv, 364703.

Simmons, B. I., Sweering, M. J., Schillinger, M., Dicks, L. V., Sutherland, W. J., & Di Clemente, R. (2019). bmotif: A package for motif analyses of bipartite networks. Methods in Ecology and Evolution, 10(5), 695-701.
}
