#' Hypothesis Test to Compare Two Median Absolute Deviations (MADs)
#' @description
#' Performs hypothesis tests and calculates confidence intervals to compare MADs of two samples.
#'
#' @param x,y numeric vectors of data values.
#' @param ratio a number representing the value of the ratio of the two MADs (or squared MADs) under the null hypothesis.
#' @param conf.level confidence level for the confidence interval.  The default is 0.95 (for a 95% confidence interval).
#' @param use.squared a logical indicating whether the squared MADs should be used.  Default is `FALSE`.
#' @param use.gld a logical indicating whether the Generalised Lambda Distribution (GLD) should be used to estimate the density of the data.  Default is `FALSE` in which case the function `density()` is used for kernel density estimation.
#' @param gld.est a character string indicating which GLD estimator to use if `use.gld = TRUE`.  See details below.
#' @param ... additional arguments, if desired, to be passed to function `density()` for density estimation (see help file for [density()] for more details).
#'
#' @return Hypothesis test results comparing two mads from independent samples and associated confidence interval for the ratio of mads (a list with class "htest").  This list includes values
#'\describe{
#' \item{\code{statistic}}{the value of the Z-statistic.}
#' \item{\code{p.value}}{the p-value for the test.}
#' \item{\code{conf.int}}{a confidence interval for the ratio of MADs (or squared MADs).}
#' \item{\code{estimate}}{the estimated ratio of MADs (or squared MADs).}
#' \item{\code{null.value}}{the specified hypothesized value of the ratio under the null hypothesis.}
#' \item{\code{alternative}}{a character string describing the alternative hypothesis.  Note, for this test only two-sided alternatives are allowed.}
#' \item{\code{data.name}}{a character string giving the names of the data.}
#' }
#' @details
#' This function carries out hypothesis tests and calculates confidence intervals for comparing the MADs between two populations.
#' Two independent samples are required, and the intervals are constructed for the ratio of mads (or squared mads as an analog to ratio of variances) as described in Arachchige and Prendergast (2024).
#' The interval is first computed for the log of the ratio before being back-transformed to the ratio scale.  The hypothesis test is carried out in a similar way, based on the
#' log ratio and using asymptotic normality of the test statistic.  The default for the ratio under the null hypothesis is one which tests against equality of mads.
#'
#' @references
#' Arachchige, C.N.P.G., & Prendergast, L.A. (2026) Confidence intervals for median absolute deviations. Communications in Statistics-Simulation and Computation, 55(1), 13-22.
#' @export
#'
#' @examples
#' # Create some data
#' set.seed(1234)
#' x <- rlnorm(100)
#' y <- rlnorm(50, meanlog = 0.5)
#'
#' # Calculate the mad, 95% confidence interval and test the hypothesis
#' mads.est <- madtest(x, y)
#' mads.est
madtest <- function(x, y, ratio = 1, conf.level = 0.95,
                    use.squared = FALSE, use.gld = FALSE, gld.est = "TM", ...)
{
  if (!is.numeric(x))
    stop("Argument 'x' must be numeric.")

  if (!is.numeric(y))
    stop("argument 'y' must be numeric")

  dname <- paste(deparse(substitute(x)), "and", deparse(substitute(y)))

  if (anyNA(x)) {
    count.x.na <- sum(is.na(x))
    warning(paste0(count.x.na), " missing values removed in ",
            deparse(substitute(x)), ".\n")
    x <- stats::na.omit(x)
  }
  if (anyNA(y)) {
    count.y.na <- sum(is.na(y))
    warning(paste0(count.y.na), " missing values removed in ",
            deparse(substitute(y)), ".\n")
    y <- stats::na.omit(y)
  }

  alpha <- 1 - conf.level
  crit <- stats::qnorm(1 - alpha/2)
  n.x <-length(x)
  n.y <- length(y)
  w.x <- n.x/(n.x + n.y)

  est.mad.x <- stats::mad(x)
  est.var.x <- varmad(x, use.gld = use.gld, gld.est = gld.est, ...)

  est.mad.y <- stats::mad(y)
  est.var.y <- varmad(y, use.gld = use.gld, gld.est = gld.est, ...)

  # results for squared ratio
  Rsq <- est.mad.x^2/est.mad.y^2
  est.var.Rsq <- 4*Rsq^2*(est.var.x/est.mad.x^2 + est.var.y/est.mad.y^2)

  sterr.log.Rsq <- sqrt(est.var.Rsq)/Rsq
  test.stat <- (log(Rsq) - log(ratio))/sterr.log.Rsq
  names(test.stat) <- "Z"

  pval <- 2 * (1 - stats::pnorm(abs(test.stat)))
  ci <- exp(log(Rsq) + c(-1, 1) * crit * sterr.log.Rsq)

  if(!use.squared){
    est <- sqrt(Rsq)
    names(est) <- "Ratio of mads"
    ci <- sqrt(ci)
  } else{
    est <- Rsq
    names(est) <- "Ratio of squared MADs"
  }
  attr(ci, "conf.level") <- conf.level
  names(ratio) <- names(est)
  res <- list(data.name = dname, statistic = test.stat, parameter = NULL, p.value = pval,
              alternative = "two.sided", estimate = est, null.value = ratio, conf.int = ci)
  class(res) <- "htest"
  return(res)

}
