# =============================================================================
# ECF-BASED PARAMETER ESTIMATION
# =============================================================================


#' Estimate stable distribution parameters using classical ECF regression
#'
#' @param x Numeric vector of data.
#' @param u Vector of frequency values.
#' @return A list with estimated parameters: \code{alpha}, \code{beta}, \code{gamma}, and \code{delta}.
#' @export
CDF <- function(x, u) {
  init_params <- stable_fit_init(x)
  gamma0 <- init_params[[3]]
  delta0 <- init_params[[4]]
  x <- (x - delta0) / gamma0

  ecf_result <- ecf_fn(x, u, method = "simple")
  y1 <- ecf_result[[1]]
  y3 <- ecf_result[[2]]

  z1 <- log(-log(y1))
  v1 <- log(abs(u))

  fit1 <- lm(z1 ~ v1)
  a1 <- coef(fit1)
  alpha1 <- clip(a1[2], 0.1, 2)
  gamma1 <- exp(a1[1] / alpha1)

  v3 <- -gamma1^alpha1 * eta0(u, alpha1, gamma1)
  X_mat <- cbind(v3, u)
  a3 <- solve(t(X_mat) %*% X_mat) %*% t(X_mat) %*% y3

  beta1 <- clip(a3[1], -1, 1)
  delta1 <- a3[2]

  return(list(
    alpha = alpha1,
    beta = beta1,
    gamma = gamma1 * gamma0,
    delta = delta1 * gamma0 + delta0
  ))
}

# =============================================================================
# Régression ECF pondérée
# =============================================================================



#' Estimate stable parameters using weighted ECF regression
#'
#' @param x Numeric vector of data.
#' @param u Vector of frequency values.
#' @return A list with estimated parameters: \code{alpha}, \code{beta}, \code{gamma}, and \code{delta}.
#' @importFrom stats predict residuals setNames
#' @export
ecf_regression <- function(x, u) {
  init <- stable_fit_init(x)
  gamma0 <- init$gamma
  delta0 <- init$delta
  x_norm <- (x - delta0) / gamma0

  ecf <- ecf_fn(x_norm, u, method = "kernel")
  z1 <- log(-log(pmax(ecf$magnitude, 1e-10)))
  v1 <- log(abs(u))

  reg1 <- lm(z1 ~ v1)
  residuals1 <- abs(z1 - predict(reg1))
  pred1 <- predict(reg1)
  fit_resid1 <- lm(residuals1 ~ pred1)
  weights1 <- 1 / pmax(predict(fit_resid1), 1e-8)^2
  reg1 <- lm(z1 ~ v1, weights = weights1)

  alpha1 <- min(max(coef(reg1)[2], 0.1), 2)
  gamma1 <- exp(coef(reg1)[1] / alpha1)

  y3 <- ecf$phase
  v3 <- -gamma1^alpha1 * eta0(u, alpha1, gamma1)
  Xmat <- cbind(v3, u)

  reg3 <- lm(y3 ~ Xmat - 1)
  residuals3 <- abs(y3 - predict(reg3))
  pred3 <- predict(reg3)
  fit_resid3 <- lm(residuals3 ~ pred3)
  weights3 <- 1 / pmax(predict(fit_resid3), 1e-8)^2
  reg3 <- lm(y3 ~ Xmat - 1, weights = weights3)

  beta1 <- min(max(coef(reg3)[1], -1), 1)
  delta1 <- coef(reg3)[2]

  return(list(
    alpha = alpha1,
    beta = beta1,
    gamma = as.numeric(gamma1 * gamma0),
    delta = as.numeric(delta1 * gamma0 + delta0)
  ))
}
# =============================================================================
# Régression ECF robuste
# =============================================================================

#' Estimate stable parameters using robust ECF regression
#'
#' @param x Numeric vector of data.
#' @param u Vector of frequency values.
#' @return A list with estimated parameters: \code{alpha}, \code{beta}, \code{gamma}, and \code{delta}.
#' @importFrom MASS rlm
#' @export
robust_ecf_regression <- function(x, u) {
  init_params <- stable_fit_init(x)
  gamma0 <- init_params$gamma
  delta0 <- init_params$delta
  x_norm <- (x - delta0) / gamma0

  ecf <- ecf_fn(x_norm, u, "kernel")
  z1 <- log(-log(pmax(ecf$magnitude, 1e-10)))
  v1 <- log(abs(u))

  reg1 <- rlm(z1 ~ v1)
  alpha1 <- min(max(coef(reg1)[2], 0.1), 2)
  gamma1 <- exp(coef(reg1)[1] / alpha1)

  v3 <- -gamma1^alpha1 * eta0(u, alpha1, gamma1)
  Xmat <- cbind(v3, u)
  reg2 <- rlm(ecf$phase ~ Xmat - 1)

  beta1 <- min(max(coef(reg2)[1], -1), 1)
  delta1 <- coef(reg2)[2]

  return(list(
    alpha = alpha1,
    beta = beta1,
    gamma = as.numeric(gamma1 * gamma0),
    delta = as.numeric(delta1 * gamma0 + delta0)
  ))
}

# =============================================================================
# Estimation par ECF avec pondération et filtrage
# =============================================================================

#' Estimate stable parameters using filtered and weighted ECF regression
#'
#' @param data Numeric vector of observations.
#' @param frequencies Vector of frequency values.
#' @return A list with estimated parameters: \code{alpha}, \code{beta}, \code{gamma}, and \code{delta}.
#' @export
fit_stable_ecf <- function(data, frequencies) {
  init_params <- stable_fit_init(data)
  gamma0 <- init_params$gamma
  delta0 <- init_params$delta
  normalized_data <- (data - delta0) / gamma0

  ecf <- ecf_fn(normalized_data, frequencies, method = "kernel")
  log_modulus <- log(-log(pmax(ecf$magnitude, 1e-10)))
  log_freq <- log(abs(frequencies))

  reg1 <- lm(log_modulus ~ log_freq)
  residuals1 <- abs(log_modulus - predict(reg1))
  weights1 <- 1 / pmax(predict(lm(residuals1 ~ predict(reg1))), 1e-8)^2
  reg1 <- lm(log_modulus ~ log_freq, weights = weights1)

  alpha <- min(max(coef(reg1)[2], 0.1), 2)
  gamma <- exp(coef(reg1)[1] / alpha)

  eta_vals <- -gamma^alpha * eta0(frequencies, alpha, gamma)
  X <- cbind(eta_vals, frequencies)

  valid <- apply(X, 1, function(row) all(is.finite(row))) & is.finite(ecf$phase)
  if (!any(valid)) stop("Filtered arrays are empty. Check input data.")

  X <- X[valid, , drop = FALSE]
  y3 <- ecf$phase[valid]

  reg2 <- lm(y3 ~ X - 1)
  residuals2 <- abs(y3 - predict(reg2))
  weights2 <- 1 / pmax(predict(lm(residuals2 ~ predict(reg2))), 1e-8)^2
  reg2 <- lm(y3 ~ X - 1, weights = weights2)

  beta <- min(max(coef(reg2)[1], -1), 1)
  delta <- coef(reg2)[2]

  return(list(
    alpha = alpha,
    beta = beta,
    gamma = as.numeric(gamma * gamma0),
    delta = as.numeric(delta * gamma0 + delta0)
  ))
}

# =============================================================================
# Estimation récursive par ECF
# =============================================================================

#' Estimate stable parameters using recursive ECF method
#'
#' @param data Numeric vector of observations.
#' @param u Vector of frequency values.
#' @return A list with estimated parameters: \code{alpha}, \code{beta}, \code{gamma}, and \code{delta}.
#' @export
estimate_stable_recursive_ecf <- function(data, u) {
  init_params <- stable_fit_init(data)
  gamma0 <- init_params$gamma
  delta0 <- init_params$delta
  x_norm <- (data - delta0) / gamma0

  ecf <- ecf_fn(x_norm, u, method = "recursive")
  z1 <- log(-log(pmax(ecf$magnitude, 1e-10)))
  v1 <- log(abs(u))

  fit1 <- lm(z1 ~ v1)
  alpha1 <- min(max(coef(fit1)[2], 0.1), 2)
  gamma1 <- exp(coef(fit1)[1] / alpha1)

  y3 <- ecf$phase
  v3 <- -gamma1^alpha1 * eta0(u, alpha1, gamma1)
  X <- cbind(v3, u)

  valid <- apply(X, 1, function(row) all(is.finite(row))) & is.finite(y3)
  if (!any(valid)) {
    warning("Filtered arrays are empty. Returning default parameter values.")
    return(list(
      alpha = alpha1,
      beta = 0,
      gamma = as.numeric(gamma1 * gamma0),
      delta = as.numeric(0 * gamma0 + delta0)
    ))
  }

  X <- X[valid, , drop = FALSE]
  y3 <- y3[valid]

  reg3 <- lm(y3 ~ X - 1)
  beta1 <- min(max(coef(reg3)[1], -1), 1)
  delta1 <- coef(reg3)[2]

  return(list(
    alpha = alpha1,
    beta = beta1,
    gamma = as.numeric(gamma1 * gamma0),
    delta = as.numeric(delta1 * gamma0 + delta0)
  ))
}

# =============================================================================
# Estimation par noyau ECF
# =============================================================================
#' Estimate stable parameters using kernel-based ECF method
#'
#' @param data Numeric vector of observations
#' @param u Vector of frequency values
#' @return A list with estimated parameters: \code{alpha}, \code{beta}, \code{gamma}, and \code{delta}.
#' @export
estimate_stable_kernel_ecf <- function(data, u) {
  init_params <- stable_fit_init(data)
  gamma0 <- init_params$gamma
  delta0 <- init_params$delta
  x_norm <- (data - delta0) / gamma0

  ecf <- ecf_fn(x_norm, u, method = "kernel")
  y3 <- ecf$phase
  z1 <- log(-log(pmax(ecf$magnitude, 1e-10)))
  v1 <- log(abs(u))

  valid <- is.finite(z1)
  z1 <- z1[valid]
  v1 <- v1[valid]

  fit1 <- lm(z1 ~ v1)
  alpha1 <- min(max(coef(fit1)[2], 0.1), 2)
  gamma1 <- exp(coef(fit1)[1] / alpha1)

  v3 <- -gamma1^alpha1 * eta0(u, alpha1, gamma1)
  X <- cbind(v3, u)

  valid <- apply(X, 1, function(row) all(is.finite(row))) & is.finite(y3)
  X <- X[valid, , drop = FALSE]
  y3 <- y3[valid]

  fit2 <- lm(y3 ~ X - 1)
  beta1 <- max(min(coef(fit2)[1], 1), -1)
  delta1 <- coef(fit2)[2]

  return(list(
    alpha = alpha1,
    beta = beta1,
    gamma = as.numeric(gamma1 * gamma0),
    delta = as.numeric(delta1 * gamma0 + delta0)
  ))
}

# =============================================================================
# Estimation par moindres carrés pondérés
# =============================================================================
#' Estimate stable parameters using weighted OLS on recursive ECF
#'
#' @param data Numeric vector of observations
#' @param u Vector of frequency values
#' @return A list with estimated parameters: \code{alpha}, \code{beta}, \code{gamma}, and \code{delta}.
#' @export
estimate_stable_weighted_ols <- function(data, u) {
  init_params <- stable_fit_init(data)
  gamma0 <- init_params$gamma
  delta0 <- init_params$delta
  x_norm <- (data - delta0) / gamma0

  ecf <- ecf_fn(x_norm, u, method = "recursive")
  y3 <- ecf$phase
  z1 <- log(-log(pmax(ecf$magnitude, 1e-10)))
  v1 <- log(abs(u))

  model1 <- lm(z1 ~ v1)
  alpha1 <- min(max(coef(model1)[2], 0.1), 2)
  gamma1 <- exp(coef(model1)[1] / alpha1)

  v3 <- -gamma1^alpha1 * eta0(u, alpha1, gamma1)
  X3 <- cbind(v3, u)

  valid <- apply(X3, 1, function(row) all(is.finite(row))) & is.finite(y3)
  if (!any(valid)) {
    warning("Filtered arrays are empty. Returning default parameter values.")
    return(list(
      alpha = alpha1,
      beta = 0,
      gamma = as.numeric(gamma1 * gamma0),
      delta = as.numeric(0 * gamma0 + delta0)
    ))
  }

  X3 <- X3[valid, , drop = FALSE]
  y3 <- y3[valid]

  fit2 <- lm(y3 ~ X3 - 1)
  resid2 <- residuals(fit2)
  weights <- 1 / pmax(abs(resid2), 1e-8)^2
  fit2 <- lm(y3 ~ X3 - 1, weights = weights)

  beta1 <- max(min(coef(fit2)[1], 1), -1)
  delta1 <- coef(fit2)[2]

  return(list(
    alpha = alpha1,
    beta = beta1,
    gamma = as.numeric(gamma1 * gamma0),
    delta = as.numeric(delta1 * gamma0 + delta0)
  ))
}

# =============================================================================
# Estimation à partir de la CDF
# =============================================================================
#' Estimate stable parameters using CDF-based ECF regression
#'
#' @param data Numeric vector of observations
#' @param u Vector of frequency values
#' @return A list with estimated parameters: \code{alpha}, \code{beta}, \code{gamma}, and \code{delta}.
#' @export
estimate_stable_from_cdf <- function(data, u) {
  init_params <- stable_fit_init(data)
  gamma0 <- init_params$gamma
  delta0 <- init_params$delta
  x_norm <- (data - delta0) / gamma0

  ecf <- ecf_fn(x_norm, u, method = "simple")
  y3 <- ecf$phase
  z1 <- log(-log(pmax(ecf$magnitude, 1e-10)))
  v1 <- log(abs(u))

  valid <- is.finite(z1) & is.finite(v1)
  z1 <- z1[valid]
  v1 <- v1[valid]

  model1 <- lm(z1 ~ v1)
  alpha1 <- min(max(coef(model1)[2], 0.1), 2)
  gamma1 <- exp(coef(model1)[1] / alpha1)

  v3 <- -gamma1^alpha1 * eta0(u, alpha1, gamma1)
  X3 <- cbind(v3, u)

  valid <- apply(X3, 1, function(row) all(is.finite(row))) & is.finite(y3)
  X3 <- X3[valid, , drop = FALSE]
  y3 <- y3[valid]

  fit2 <- lm(y3 ~ X3 - 1)
  resid2 <- residuals(fit2)
  weights <- 1 / pmax(abs(resid2), 1e-8)^2
  fit2 <- lm(y3 ~ X3 - 1, weights = weights)

  beta1 <- min(max(coef(fit2)[1], 1), -1)
  delta1 <- coef(fit2)[2]

  return(list(
    alpha = alpha1,
    beta = beta1,
    gamma = as.numeric(gamma1 * gamma0),
    delta = as.numeric(delta1 * gamma0 + delta0)
  ))
}

# =============================================================================
# Fonctions utilitaires ECF
# =============================================================================

# =============================================================================
# ECF UTILITY FUNCTIONS
# =============================================================================

#' Compute empirical characteristic function
#'
#' @param X Numeric vector of data
#' @param t_grid Vector of frequency values
#' @return Complex vector of empirical characteristic function values.
#' @export
ecf_empirical <- function(X, t_grid) {
  sapply(t_grid, function(t) mean(exp(1i * t * X)))
}

#' Extract magnitude and phase components from ECF
#'
#' @param phi_vals Complex vector of ECF values
#' @return A list with two numeric vectors: \code{y_vals} (log-log magnitude) and \code{arg_vals} (phase).
#' @export
ecf_components <- function(phi_vals) {
  abs_vals <- Mod(phi_vals)
  y_vals <- log(-log(pmax(abs_vals, 1e-10)))
  arg_vals <- Arg(phi_vals)
  list(y_vals = y_vals, arg_vals = arg_vals)
}


