# =============================================================================
# EM ALGORITHM FOR STABLE MIXTURE (RECURSIVE ECF)
# =============================================================================


#' EM algorithm for mixture of alpha-stable distributions using recursive ECF
#'
#' Performs Expectation-Maximization (EM) to estimate parameters of a two-component
#' alpha-stable mixture model using recursive kernel smoothing on the empirical characteristic function.
#'
#' @param data Numeric vector of observations.
#' @param max_iter Maximum number of EM iterations.
#' @param tol Convergence tolerance on log-likelihood.
#' @return A list with estimated component parameters (\code{alpha}, \code{beta}, \code{gamma}, \code{delta}) and mixture weight (\code{w}).
#' @importFrom stats kmeans
#' @export
em_estimate_stable_recursive_ecf <- function(data, max_iter = 100, tol = 1e-4) {
  u <- seq(0.1, 1, length.out = 10)
  labels <- kmeans(matrix(data, ncol = 1), centers = 2)$cluster
  w <- mean(labels == 1)

  params1 <- if (sum(labels == 1) < 5) {
    as.list(setNames(stable_fit_init(data), c("alpha", "beta", "gamma", "delta")))
  } else {
    estimate_stable_recursive_ecf(data[labels == 1], u)
  }

  params2 <- if (sum(labels == 2) < 5) {
    as.list(setNames(stable_fit_init(data), c("alpha", "beta", "gamma", "delta")))
  } else {
    estimate_stable_recursive_ecf(data[labels == 2], u)
  }

  log_likelihood_old <- -Inf

  for (iteration in 1:max_iter) {
    pdf1 <- pmax(r_stable_pdf(data, params1$alpha, params1$beta, params1$gamma, params1$delta), 1e-300)
    pdf2 <- pmax(r_stable_pdf(data, params2$alpha, params2$beta, params2$gamma, params2$delta), 1e-300)

    resp1 <- w * pdf1
    resp2 <- (1 - w) * pdf2
    total <- resp1 + resp2

    gamma1 <- resp1 / total
    gamma2 <- resp2 / total
    labels <- ifelse(gamma1 > gamma2, 1, 2)
    w <- mean(labels == 1)

    if (sum(labels == 1) >= 5) {
      params1 <- estimate_stable_recursive_ecf(data[labels == 1], u)
    } else {
      message("Cluster 0 too small. Reusing previous estimate.\n")
    }

    if (sum(labels == 2) >= 5) {
      params2 <- estimate_stable_recursive_ecf(data[labels == 2], u)
    } else {
      message("Cluster 1 too small. Reusing previous estimate.\n")
    }

    pdf1 <- pmax(r_stable_pdf(data, params1$alpha, params1$beta, params1$gamma, params1$delta), 1e-300)
    pdf2 <- pmax(r_stable_pdf(data, params2$alpha, params2$beta, params2$gamma, params2$delta), 1e-300)
    log_likelihood <- sum(log(w * pdf1 + (1 - w) * pdf2))

    if (abs(log_likelihood - log_likelihood_old) < tol) {
      message(sprintf("Converged after %d iterations.\n", iteration))
      break
    }

    log_likelihood_old <- log_likelihood
  }

  return(list(params1 = params1, params2 = params2, weight = w))
}


# =============================================================================
# EM ALGORITHM FOR STABLE MIXTURE (KERNEL ECF)
# =============================================================================


#' EM algorithm for mixture of alpha-stable distributions using kernel ECF
#'
#' Performs EM estimation of a two-component alpha-stable mixture using kernel-smoothed
#' empirical characteristic function (ECF).
#'
#' @param data Numeric vector of observations.
#' @param max_iter Maximum number of EM iterations.
#' @param tol Convergence tolerance on log-likelihood.
#' @return A list with estimated component parameters (\code{alpha}, \code{beta}, \code{gamma}, \code{delta}) and mixture weight (\code{w}).
#' @importFrom stats kmeans
#' @export
em_estimate_stable_kernel_ecf <- function(data, max_iter = 100, tol = 1e-4) {
  u <- seq(0.1, 1, length.out = 10)
  labels <- kmeans(matrix(data, ncol = 1), centers = 2)$cluster
  w <- mean(labels == 1)

  params1 <- if (sum(labels == 1) < 5) {
    as.list(setNames(stable_fit_init(data), c("alpha", "beta", "gamma", "delta")))
  } else {
    estimate_stable_kernel_ecf(data[labels == 1], u)
  }

  params2 <- if (sum(labels == 2) < 5) {
    as.list(setNames(stable_fit_init(data), c("alpha", "beta", "gamma", "delta")))
  } else {
    estimate_stable_kernel_ecf(data[labels == 2], u)
  }

  log_likelihood_old <- -Inf

  for (iteration in 1:max_iter) {
    pdf1 <- pmax(r_stable_pdf(data, params1$alpha, params1$beta, params1$gamma, params1$delta), 1e-300)
    pdf2 <- pmax(r_stable_pdf(data, params2$alpha, params2$beta, params2$gamma, params2$delta), 1e-300)

    resp1 <- w * pdf1
    resp2 <- (1 - w) * pdf2
    total <- resp1 + resp2

    gamma1 <- resp1 / total
    gamma2 <- resp2 / total
    labels <- ifelse(gamma1 > gamma2, 1, 2)
    w <- mean(labels == 1)

    if (sum(labels == 1) >= 5) {
      params1 <- estimate_stable_kernel_ecf(data[labels == 1], u)
    } else {
      message("Cluster 0 too small. Reusing previous estimate.\n")
    }

    if (sum(labels == 2) >= 5) {
      params2 <- estimate_stable_kernel_ecf(data[labels == 2], u)
    } else {
      message("Cluster 1 too small. Reusing previous estimate.\n")
    }

    pdf1 <- pmax(r_stable_pdf(data, params1$alpha, params1$beta, params1$gamma, params1$delta), 1e-300)
    pdf2 <- pmax(r_stable_pdf(data, params2$alpha, params2$beta, params2$gamma, params2$delta), 1e-300)
    log_likelihood <- sum(log(w * pdf1 + (1 - w) * pdf2))

    if (abs(log_likelihood - log_likelihood_old) < tol) {
      message(sprintf("Converged after %d iterations.\n", iteration))
      break
    }

    log_likelihood_old <- log_likelihood
  }

  return(list(params1 = params1, params2 = params2, weight = w))
}


# =============================================================================
# EM ALGORITHM FOR STABLE MIXTURE (WEIGHTED OLS)
# =============================================================================


#' EM algorithm for mixture of alpha-stable distributions using weighted OLS
#'
#' Performs EM estimation of a two-component alpha-stable mixture using weighted
#' least squares regression on the ECF.
#'
#' @param data Numeric vector of observations.
#' @param max_iter Maximum number of EM iterations.
#' @param tol Convergence tolerance on log-likelihood.
#' @return A list with estimated component parameters (\code{alpha}, \code{beta}, \code{gamma}, \code{delta}) and mixture weight (\code{w}).
#' @importFrom stats kmeans
#' @export
em_estimate_stable_weighted_ols <- function(data, max_iter = 100, tol = 1e-4) {
  u <- seq(0.1, 1, length.out = 10)
  labels <- kmeans(matrix(data, ncol = 1), centers = 2)$cluster
  w <- mean(labels == 1)

  params1 <- if (sum(labels == 1) < 5) {
    as.list(setNames(stable_fit_init(data), c("alpha", "beta", "gamma", "delta")))
  } else {
    estimate_stable_weighted_ols(data[labels == 1], u)
  }

  params2 <- if (sum(labels == 2) < 5) {
    as.list(setNames(stable_fit_init(data), c("alpha", "beta", "gamma", "delta")))
  } else {
    estimate_stable_weighted_ols(data[labels == 2], u)
  }

  log_likelihood_old <- -Inf

  for (iteration in 1:max_iter) {
    pdf1 <- pmax(r_stable_pdf(data, params1$alpha, params1$beta, params1$gamma, params1$delta), 1e-300)
    pdf2 <- pmax(r_stable_pdf(data, params2$alpha, params2$beta, params2$gamma, params2$delta), 1e-300)

    resp1 <- w * pdf1
    resp2 <- (1 - w) * pdf2
    total <- resp1 + resp2

    gamma1 <- resp1 / total
    gamma2 <- resp2 / total
    labels <- ifelse(gamma1 > gamma2, 1, 2)
    w <- mean(labels == 1)

    if (sum(labels == 1) >= 5) {
      params1 <- estimate_stable_weighted_ols(data[labels == 1], u)
    } else {
      message("Cluster 0 too small. Reusing previous estimate.\n")
    }

    if (sum(labels == 2) >= 5) {
      params2 <- estimate_stable_weighted_ols(data[labels == 2], u)
    } else {
      message("Cluster 1 too small. Reusing previous estimate.\n")
    }

    pdf1 <- pmax(r_stable_pdf(data, params1$alpha, params1$beta, params1$gamma, params1$delta), 1e-300)
    pdf2 <- pmax(r_stable_pdf(data, params2$alpha, params2$beta, params2$gamma, params2$delta), 1e-300)
    log_likelihood <- sum(log(w * pdf1 + (1 - w) * pdf2))

    if (abs(log_likelihood - log_likelihood_old) < tol) {
      message(sprintf("Converged after %d iterations.\n", iteration))
      break
    }

    log_likelihood_old <- log_likelihood
  }

  return(list(params1 = params1, params2 = params2, weight = w))
}


# =============================================================================
# EM ALGORITHM FOR STABLE MIXTURE (CDF-BASED ESTIMATION)
# =============================================================================


#' EM algorithm for mixture of alpha-stable distributions using CDF-based ECF
#'
#' Performs EM estimation of a two-component alpha-stable mixture using a simplified
#' empirical characteristic function derived from the cumulative distribution function (CDF).
#'
#' @param data Numeric vector of observations.
#' @param max_iter Maximum number of EM iterations.
#' @param tol Convergence tolerance on log-likelihood.
#' @return A list with estimated component parameters (\code{alpha}, \code{beta}, \code{gamma}, \code{delta}) and mixture weight (\code{w}).
#' @importFrom stats kmeans
#' @export
em_estimate_stable_from_cdf <- function(data, max_iter = 100, tol = 1e-4) {
  data <- as.numeric(data)
  u <- seq(0.1, 1, length.out = 10)

  # KMeans initialization
  labels <- kmeans(matrix(data, ncol = 1), centers = 2, nstart = 1)$cluster
  labels <- labels - 1  # Convert to 0/1
  w <- mean(labels == 0)

  params1 <- if (sum(labels == 0) < 5) {
    setNames(stable_fit_init(data), c("alpha", "beta", "gamma", "delta"))
  } else {
    estimate_stable_from_cdf(data[labels == 0], u)
  }

  params2 <- if (sum(labels == 1) < 5) {
    setNames(stable_fit_init(data), c("alpha", "beta", "gamma", "delta"))
  } else {
    estimate_stable_from_cdf(data[labels == 1], u)
  }

  log_likelihood_old <- -Inf

  for (iteration in 1:max_iter) {
    p1 <- pmax(r_stable_pdf(data, params1$alpha, params1$beta, params1$gamma, params1$delta), 1e-300)
    p2 <- pmax(r_stable_pdf(data, params2$alpha, params2$beta, params2$gamma, params2$delta), 1e-300)

    resp1 <- w * p1
    resp2 <- (1 - w) * p2
    total <- resp1 + resp2

    gamma1 <- resp1 / total
    gamma2 <- resp2 / total
    labels <- ifelse(gamma1 > gamma2, 0, 1)
    w <- mean(labels == 0)

    if (sum(labels == 0) >= 5) {
      params1 <- estimate_stable_from_cdf(data[labels == 0], u)
    } else {
      message("Cluster 0 too small. Reusing previous estimate.")
    }

    if (sum(labels == 1) >= 5) {
      params2 <- estimate_stable_from_cdf(data[labels == 1], u)
    } else {
      message("Cluster 1 too small. Reusing previous estimate.")
    }

    p1 <- pmax(r_stable_pdf(data, params1$alpha, params1$beta, params1$gamma, params1$delta), 1e-300)
    p2 <- pmax(r_stable_pdf(data, params2$alpha, params2$beta, params2$gamma, params2$delta), 1e-300)
    log_likelihood <- sum(log(w * p1 + (1 - w) * p2))

    if (abs(log_likelihood - log_likelihood_old) < tol) {
      message(sprintf("Converged after %d iterations.", iteration))
      break
    }

    log_likelihood_old <- log_likelihood
  }

  return(list(params1 = params1, params2 = params2, weight = w))
}

