#' @title Specifies Factor Loading Values for a Model.
#' @description
#' This function creates a factor loading matrix for a given Structural Equation Model (SEM).
#' The loadings must be provided as a vector and are assigned to the matrix column by column, where each column corresponds to a latent factor and each row corresponds to an observed item.
#' All factor loadings must be specified as values strictly greater than 0 and less than 1.
#' The resulting matrix has dimensions equal to the number of items by the number of factors.
#'
#' @author Fatih Orçan
#' @param nf An integer specifying the number of factors.
#' @param fl.loads A numeric vector of factor loadings. Values should be provided in column-wise order, corresponding to the items loading on each factor.
#' @return The function returns the factor loading matrix.
#' @export
#' @examples
#' # This example represents a three-factor CFA model
#' #  where the factors are indicated by 3, 3, and 2 items respectively.
#' #
#' loading.value(nf=3, fl.loads=c(.6,.6,.6,0,0,0,0,0,0,0,0,.7,.7,.7,0,0,0,0,0,0,0,0,.8,.8))

loading.value <-function(nf, fl.loads){

  fl.matx<-matrix(ncol = nf, fl.loads)
  colSums(fl.matx !=0)
  if(nf==1 & length(fl.loads)<=2){stop("Note: One-factor models should have at least 3 items.
                                      Please re-run the function.")}
  if(nf > 1 & any(colSums(fl.matx !=0)<2)){stop("Note: Multiple-factor models should have at least 2 items under each factor.
                                      Please re-run the function.")}
  if (any(fl.loads >= 1) | any(fl.loads < 0 )) {stop("Values of the factor loadings should be between 0 and 1.
                                      Please re-run the function.")}
    return(fl.matx)
}
