% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/NPC.R
\name{NPC}
\alias{NPC}
\title{Estimation of examinees' attribute profiles using the NPC method}
\usage{
NPC(
  Y,
  Q,
  distance = c("hamming", "whamming", "penalized"),
  gate = c("AND", "OR"),
  wg = 1,
  ws = 1
)
}
\arguments{
\item{Y}{A \eqn{N \times J} binary data matrix consisting of the responses from \eqn{N} examinees to
\eqn{J} items.}

\item{Q}{A \eqn{J \times K} binary Q-matrix where the entry \eqn{q_{jk}} describes whether the \eqn{k}th attribute
is required by the \eqn{j}th item.}

\item{distance}{The type of distance used to compute the loss function. The possible options include
(i) "\code{hamming}" representing the plain Hamming distance method, 
(ii) "\code{whamming}" representing the Hamming distance weighted by the inverse of item variance, and  
(iii) "\code{penalized}" representing the Hamming distance weighted by the inverse of item variance
      and specified penalizing weights for guess and slip.}

\item{gate}{A character string specifying the type of gate. The possible options include "\code{AND}" and "\code{OR}" standing
for conjunctive and disjunctive gate, respectively.}

\item{wg}{Additional argument for the "penalized" method. It is a weight
assigned to guesses in the DINA or DINO models. A large
value of weight results in a stronger impact on the
distance (i.e., larger loss function values) caused by guessing.}

\item{ws}{Additional input for the "penalized" method.
It is the weight assigned to slips in the DINA or DINO models.
A large value of weight results in a stronger impact on the
distance (i.e., larger loss function values) caused by slipping.}
}
\value{
The function returns a series of outputs, including:
\describe{
  \item{alpha.est}{A \eqn{N \times K}{}  matrix representing the estimated attribute profiles.
      1 = examinee masters the attribute, 0 = examinee does not master the attribute.}
  \item{est.ideal}{A \eqn{N \times J}{}  matrix indicating the estimated ideal response to all 
  items from all examinees. 1 = correct, 0 = incorrect.}
  \item{est.class}{A \eqn{N}{}-dimensional vector showing the class memberships for all examinees.}
  \item{n.tie}{The number of ties in the Hamming distance among the candidate
      attribute profiles for each person. When ties occur, one of
      the tied attribute profiles is randomly chosen.}
  \item{pattern}{All possible attribute profiles in the latent space.}
  \item{loss.matrix}{A \eqn{2^K \times N}{} matrix containing the values of the loss function 
  (the distances) between each examinee's observed response vector and the \eqn{2^K}{} ideal response vectors.}
}
}
\description{
The function estimates examinees' attribute profiles
    using the nonparametric classification (NPC) method (Chiu & Douglas, 2013).
    An examinee's attribute profile is estimated by minimizing the distance between
    the observed and ideal item responses.
}
\section{Details}{

The nonparametric classification (NPC) method (Chiu & Douglas, 2013) assigns examinees to the
proficiency classes they belong to by comparing their observed item response patterns with each of the ideal
item response patterns of the \eqn{2^K} proficiency classes. When there is no data perturbation, an 
examinee's ideal response pattern corresponding to the examinee's true attribute pattern and his/her 
observed item response patterns are identical, and thus the distance between them is 0. When data 
perturbations are small, this ideal response pattern remains the one most similar to the observed 
response pattern, which is exactly the setup of data conforming to the DINA or DINO model. Hence, based 
on this rationale, an examinee's attribute profile is obtained by minimizing the distance between the 
observed and the ideal item response patterns. The nonparametric nature of the NPC method furthermore 
makes it suitable for data obtained from small-scale settings.
}

\examples{
\dontrun{
library(GDINA)
N <- 500
Q <- sim30GDINA$simQ
gs <- data.frame(guess = rep(0.2, nrow(Q)), slip = rep(0.2, nrow(Q)))
sim <- simGDINA(N, Q, gs.parm = gs, model = "DINA")
Y <- extract(sim, what = "dat")
alpha <- extract(sim, what = "attribute")

# Estimate attribute profiles using NPC
result <- NPC(Y, Q, distance = "hamming", gate = "AND")
print(result)
result$alpha.est

# Check attributed agreement rate
PAR(alpha, result$alpha.est)
AAR(alpha, result$alpha.est)
}
}
\references{
Chiu, C. Y., & Douglas, J. A. (2013). A nonparametric approach to cognitive diagnosis by proximity to ideal response patterns.
\emph{Journal of Classification, 30}(2), 225-250.
\doi{10.1007/s00357-013-9132-9}
}
\seealso{
\code{\link{GNPC}}
}
