% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/Q.completeness.R
\name{Q.completeness}
\alias{Q.completeness}
\title{Check the completeness status of a binary Q-matrix}
\usage{
Q.completeness(raw.Q, model = NULL)
}
\arguments{
\item{raw.Q}{The Q-matrix that is to be checked, where \eqn{J} is the number
of items and \eqn{K} is the number of attributes. It must be a binary (0/1) matrix
or data frame that can be coerced to a matrix.}

\item{model}{Character string specifying the cognitive diagnosis model. Valid options
are "\code{DINA}", "\code{DINO}", or "\code{General}" (for general CDMs including
G-DINA, LCDM, etc.). Case-insensitive. If not specified or invalid, the default is
"\code{General}" with a warning.}
}
\value{
A list of class "\code{Qcompleteness}" containing:
  \item{is_complete}{Logical value indicating completeness: \code{TRUE} if complete,
    \code{FALSE} if incomplete, \code{NA} if uncertain.}
  \item{status}{Character string: "\code{complete}", "\code{incomplete}", or 
    "\code{uncertain}".}
  \item{message}{Character string with detailed explanation of the result.}
  \item{model}{The CDM used for assessment.}
  \item{K}{Number of attributes in the Q-matrix.}
  \item{J}{Number of items in the Q-matrix.}
  
  The function also prints the status message to the console as a side effect.
}
\description{
\code{Q.completeness} is used to examine whether a given Q-matrix is 
complete when data conform to a specified CDM. A Q-matrix is said 
to be complete if it allows for the unique identification of all possible attribute profiles 
among examinees. So far, the function can only be used for a binary Q-matrix with binary responses.
}
\details{
The conditions for one Q-matrix completeness are model-dependent: a Q-matrix may be complete for one CDM
but incomplete for another. This function implements the theoretical work developed by
Chiu et al. (2009) and \enc{Köhn}{Koehn} and Chiu (2017).

\strong{For DINA and DINO models:}

A Q-matrix is complete if and only if it contains all \eqn{K} single-attribute items
(Chiu et al., 2009). 

\strong{For More General CDMs:}

The function implements a sequential procedure based on Theorems 3-4 and
Propositions 1-2 in the work by \enc{Köhn}{Koehn} and Chiu (2017).
\enumerate{
  \item If Q contains all \eqn{K} single-attribute items, it is complete (Proposition 1).
  \item If Q has rank \eqn{< K}, it is incomplete (Theorem 3).
  \item For full-rank Q-matrices without all single-attribute items, the function
        examines non-nested attribute pairs using indicator vectors to determine
        if distinct expected response patterns \eqn{\mathbf{S}(\boldsymbol{\alpha})} can be guaranteed.
}

The theoretical framework establishes the sufficient conditions for Q completeness, 
which means completeness implies distinct expected item response patterns for 
all \eqn{2^K} possible attribute profiles.
}
\examples{
\dontrun{
# Example 1: Complete Q-matrix for DINA model
# (contains all 3 single-attribute items)
Q1 <- matrix(c(1, 0, 0,
               0, 1, 0,
               0, 0, 1,
               1, 1, 0,
               1, 0, 1), ncol = 3, byrow = TRUE)
result1 <- Q.completeness(Q1, model = "DINA")
print(result1$is_complete)  # TRUE

# Example 2: Incomplete Q-matrix for DINA model
# (missing single-attribute items)
Q2 <- matrix(c(1, 1, 0,
               1, 0, 1,
               0, 1, 1), ncol = 3, byrow = TRUE)
result2 <- Q.completeness(Q2, model = "DINA")
print(result2$is_complete)  # FALSE

# Example 3: Check completeness for general CDM
Q3 <- matrix(c(1, 0, 0,
               0, 1, 0,
               0, 0, 1,
               1, 1, 0,
               1, 0, 1,
               0, 1, 1), ncol = 3, byrow = TRUE)
result3 <- Q.completeness(Q3, model = "General")
}
}
\references{
Chiu, C.-Y., Douglas, J. A., & Li, X. (2009). Cluster analysis for cognitive diagnosis:
Theory and applications. \emph{Psychometrika, 74}(4), 633-665.
\doi{10.1007/s11336-009-9125-0}

Köhn, H.-F., & Chiu, C.-Y. (2017). A procedure for assessing the completeness of the
Q-matrices of cognitively diagnostic tests. \emph{Psychometrika, 82}(1), 112-132.
\doi{10.1007/s11336-016-9536-7}

Köhn, H.-F., & Chiu, C.-Y. (2018). How to build a complete Q-matrix for a 
#' cognitively diagnostic test. \emph{Journal of Classification, 35}(2), 273-299.
\doi{10.1007/s00357-018-9255-0}
}
