#   Copyright 1993, 1994, 1995, 1996  Bank of Canada.
#   Copyright 1997 (June), Paul Gilbert.
#   Copyright 1997 (Aug.), Bank of Canada.
#   Copyright 1998, Bank of Canada.

#   The user of this software has the right to use, reproduce and distribute it.
#   Bank of Canada makes no warranties with respect to the software or its 
#   fitness for any particular purpose. The software is distributed by the Bank
#   of Canada and by Paul Gilbert solely on an "as is" basis. By using the  
#   software, user agrees to accept the entire risk of using this software.

################################################################################



###########################################################################

# tagged.matrix data class  (matrix with a "tags" attribute)       <<<<<<<<<<<<

###########################################################################

tagged.matrix <-function(x, ...)UseMethod("tagged.matrix")

tagged.matrix.default <-function(mat, tags)
{if (length(tags) == 1) tags <- array(tags, dim(mat))
 attributes(tags) <- list(dim=attributes(tags)$dim) # drop any extra attributes
 attr(mat, "tags") <- tags 
 class(mat) <- c("tagged.matrix", class(mat))
 mat
}

tagged.matrix.TSdata <- function(data, input.tags, output.tags)
  {if(!is.null(input.data(data)))
       input.data(data) <- tagged.matrix(input.data(data), input.tags)
   if(!is.null(output.data(data)))
       output.data(data) <- tagged.matrix(output.data(data), output.tags)
   data
  }

tbind.tagged.matrix <-function(mat1, mat2)
{# aline and bind ts matrices and tags
 cls <- class(mat1)
 class(mat1) <- class(mat1)[-1]  # otherwise tbind calls this tbind
 if (0 == length(class(mat1))) class(mat1) <- NULL
 class(mat2) <- class(mat2)[-1]  # otherwise tbind calls this tbind
 if (0 == length(class(mat2))) class(mat2) <- NULL
 mat<-tbind(mat1, mat2)
 class(mat) <- cls
 if (is.null(attr(mat1, "tags"))) tags1 <- array("mat1", dim(mat1))
 else tag1 <- attr(mat1, "tags")
 if (is.null(attr(mat2, "tags"))) tags2 <- array("mat2", dim(mat2))
 else tag2 <- attr(mat2, "tags")
 tframe(tag1) <- tframe(mat1)
 tframe(tag2) <- tframe(mat2)
 tags <- tbind(tag1,tag2)
 # drop extra attributes (like tframes) from tags
 attributes(tags) <- list(dim=dim(tags)) 
 attr(mat, "tags") <- tags 
 mat
}

is.tagged.matrix <-function(mat)
 {r <-"tagged.matrix" == class(mat)[[1]]
  if (is.na(r)) r <-F
  r
 }

test.equal.tagged.matrix <-function(mat1, mat2)
{ test.equal.matrix(mat1,mat2) & 
  test.equal.matrix(attr(mat1,"tags"), attr(mat2, "tags"))
}


fprint <- function(x, ...)UseMethod("fprint")

fprint.tagged.matrix<- function(matrix, sub.title=NULL, super.title=NULL,
        digits=options()$digits, space=" ", file=NULL, append=F) 
 {# Formattted print of a matrix of class tagged.matrix.
  # Corresponding characters are printed after matrix numbers.
  # A character matrix (out) is returned invisibly.
  # If file is not NULL then elements of out are printed to lines of the file.
  tags <- attr(matrix, "tags")
  out <- NULL
  f <- frequency(matrix)
  s <- start(matrix)
  s <- s[1] + (s[2]-1)/f
  if (12 ==f) p <- c("Jan","Feb","Mar","Apr","May", "Jun","Jul","Aug", "Sep",
         "Oct","Nov","Dec")
  else if (4 == f) p <- c("Q1","Q2","Q3","Q4")
  else if (52 == f) p <- format(1:52)
  else p <-NULL
  pre.space <- paste(rep(" ",nchar(format(s))+nchar(p[1])),collapse="")
  if (!is.null(super.title))  out <- paste(pre.space, super.title, sep="")
  names <- format(dimnames(matrix)[[2]], digits=digits)
  if (!is.null(names))
    {ot <- pre.space
     for (i in seq(length(names)))
        ot <- paste(ot, space,names[i],sep="")
     out <- c(out, ot)
    }
  if (!is.null(sub.title)) out <- c(out,paste(pre.space, sub.title,sep=""))
  m <- format(signif(matrix[,], digits=digits))
  for (i in seq(nrow(m))) 
    {d <- (s+(i-1)/f) +.Options$ts.eps # +eps or trunc sometimes gets wrong year
     ot <- paste(trunc(d)," ", p[round(1+f*(d%%1))]," ", sep ="")
     for (j in seq(ncol(m))) 
       {ot <-paste(ot,space, m[i,j], sep="")
        if (!is.null(tags)) ot <- paste(ot,tags[i,j], sep="")
       }
      out <- c(out, ot)
    }
  if (!is.null(file)) write(out, file=file, append=append)
  invisible(out)
 }


splice.tagged.matrix <-function(mat1, mat2, tag1=NULL, tag2=NULL)
{# splice together 2 time series matrices as with splice.ts.
 # If data  is provided in both for a given period then mat1 takes priority.
 # The frequencies should be the same.
 # tag1 and tag2 are taken from the attributes of mat1 and mat2 unless
 #   they are specified (non NULL) in the argument. If specified they
 #   should be single character strings or matrices of character 
 #   strings of same dimension as mat1 and mat2. This second is useful for multiple
 # applications of the function. The result is the
 # resulting spliced matrix of class "tagged.matrix" and attribute tags 
 # (suitable for use with fprint).
 # In the case tags are not available as an attribute and are not specified 
 #   in the argument then they are set to "mat1" and "mat2".
 cls <- class(mat1)
 if (is.null(tag1)) tag1 <- attr(mat1, "tags")
 if (is.null(tag2)) tag2 <- attr(mat2, "tags")
 if (is.null(tag1)) tag1 <- "mat1"
 if (is.null(tag2)) tag2 <- "mat2"
 if (length(tag1) == 1) tag1 <- array(tag1, dim(mat1))
 if (length(tag2) == 1) tag2 <- array(tag2, dim(mat2))
 if (is.null(mat2))
   {attributes(tag1) <- list(dim=attributes(tag1)$dim) # drop extra attributes
    attr(mat1, "tags") <- tag1 
    return(mat1)
   }
 if (is.null(mat1))
   {attributes(tag2) <- list(dim=attributes(tag2)$dim) # drop extra attributes
    attr(mat2, "tags") <- tag2 
    return(mat2)
   }
 freq <- frequency(mat1)
 if (freq != frequency(mat2)) stop("frequencies must be the same.\n")
 p <- dim(mat1)[2]
 if (p != dim(mat2)[2]) stop("number of series must be the same.\n")
 tframe(tag1) <- tframe(mat1)
 tframe(tag2) <- tframe(mat2)

 fr <- c(freq,1)
 st <- min(fr %*% start(mat1), fr %*% start(mat2))
 strt <- c(st %/% freq, st %% freq)
 en <- max(fr %*% end(mat1), fr%*% end(mat2))
 tf <- list(start=strt, frequency=freq)
 if (fr %*% start(mat1) > st) 
    {tag1 <-tframed(rbind(matrix("?", fr %*% start(mat1) -st, p), tag1),tf)
     mat1 <-tframed(rbind(matrix(NA,  fr %*% start(mat1) -st, p), mat1), tf)
    }
 if (fr %*%   end(mat1) < en) 
    {tag1 <-tframed(rbind(tag1, matrix("?", en - fr %*% end(mat1), p)), tf)
     mat1 <-tframed(rbind(mat1, matrix(NA,  en - fr %*% end(mat1), p)), tf)
    }
 if (fr %*% start(mat2) > st) 
    {tag2 <-tframed(rbind(matrix("?", fr %*% start(mat2) -st, p), tag2), tf)
     mat2 <-tframed(rbind(matrix(NA,  fr %*% start(mat2) -st, p), mat2), tf)
    }
 if (fr %*%   end(mat2) < en) 
    {tag2 <-tframed(rbind(tag2,matrix("?", en - fr %*% end(mat2), p)), tf)
     mat2 <-tframed(rbind(mat2, matrix(NA, en - fr %*% end(mat2), p)), tf)
    }
 na <- is.na(mat1)
 mat1[na]  <- mat2[na]
 tag1[na] <- tag2[na]
 dimnames(mat1) <-list(round(time(mat1),digits=3),dimnames(mat1)[[2]])
 attributes(tag1) <- list(dim=attributes(tag1)$dim) # drop extra attributes
 attr(mat1, "tags") <- tag1
 class(mat1) <- cls 
 mat1
}

trim.na.tagged.matrix <-function(mat, Start=T, End=T)
{# trim NAs from the ends of a ts matrix of class "tagged.matrix".
 # (Observations for all series are dropped in a given period if any 
 #  one contains an NA in that period.)
 # if Start=F then beginning NAs are not trimmed.
 # If End=F   then ending NAs are not trimmed.
 sample <- ! apply(is.na(mat),1, any)
 if (Start) s <-min(time(mat)[sample])
 else       s <-start(mat)
 if (End)   e <-max(time(mat)[sample])
 else       e <-end(mat)
 window(mat,start=s, end=e, warn=F)
}

window.tagged.matrix <-function(mat, start=NULL, end=NULL, warn=T)
{# window a ts matrix of class "tagged.matrix".
 # With the default warn=T warnings will be issued if no truncation takes
 #  place because start or end is outside the range of data.
 tags <- attr(mat, "tags")
 class(mat) <- class(mat)[-1]
 if (0 == length(class(mat))) class(mat) <- NULL
 # The next line converts scalars tags to a matrix.
 if (length(tags) == 1) tags <- array(tags, dim(mat))
 # The next lines converts missing tags to a matrix.
 if (length(tags) == 0)
   {tags <- array("", dim(mat))
    if (warn) warning("missing tags converted to empty string.")
   }
 tframe(tags) <- tframe(mat)
 # The following is complicated by the fact that some versions of window
 #    look for missing arguments.
 if (is.null(start))
   {mat <- window(mat, end=end, warn=warn)
    tags<- window(tags,end=end, warn=warn)
   }
 else if (is.null(end))
   {mat <- window(mat, start=start, warn=warn)
    tags<- window(tags,start=start, warn=warn)
   }
 else
   {mat <- window(mat, start=start, end=end, warn=warn)
    tags<- window(tags,start=start, end=end, warn=warn)
   }
 attributes(tags) <- list(dim=dim(tags)) # drop extra attributes
 attr(mat, "tags") <- tags 
 class(mat) <- c("tagged.matrix",class(mat))
 mat
}

mon1.function.tests <- function( verbose=T, synopsis=T, fuzz.small=1e-10)
{# A short set of tests of the tagged.matrix class methods. 

  if (!is.TSdata(example.BOC.93.4.data.all))
     stop("Test data not found. Testing stopped.")
  if (synopsis & !verbose) cat("All tagged.matrix class tests ...")
  if (verbose) cat("dse1 test 1 ... ")
  z <- output.data(example.BOC.93.4.data.all)
  attr(z, "tags") <- array("a", dim(z))
  class(z) <- "tagged.matrix"
  ok <- is.tagged.matrix(z)
  all.ok <- ok
  if (verbose) 
    {if (ok) cat("ok\n")
     else    cat("failed!\n")
    }


  if (verbose) cat("tagged.matrix class test 2... ")
  zz <- z
  attr(zz, "tags") <- array("b", dim(z))
  ok <- test.equal(z,z) & (!test.equal(z,zz))
  all.ok <- all.ok & ok 
  if (verbose) 
    {if (ok) cat("ok\n")
     else    cat("failed!\n")
    }

  if (verbose) cat("tagged.matrix class test 3... ")
  zz <- window(z, start=c(1989,1))
  attr(zz, "tags") <- array("b", dim(zz))
  zzz <- tbind(window(z, start=c(1989,1)),zz)
  ok <-  (2*sum(window(output.data(example.BOC.93.4.data.all), start=c(1989,1)))) == 
            sum(zzz)
  ok <- ok & all("a" == attr(zzz, "tags")[,1:3]) & 
             all("b" == attr(zzz, "tags")[,4:6]) 
  all.ok <- all.ok & ok 
  if (verbose) 
    {if (ok) cat("ok\n")
     else    cat("failed!\n")
    }

  if (verbose) cat("tagged.matrix class test 4... ")
  zzz <- splice(zz, window(z, end=c(1990,1)))
  ok <- test.equal.matrix(z,zzz) & (!test.equal(z,zzz))
  zzz <- splice(zz, 
           window(output.data(example.BOC.93.4.data.all), end=c(1990,1)), tag2="x")
  ok <- ok & test.equal.matrix(z,zzz) & (!test.equal(z,zzz))
  all.ok <- all.ok & ok 
  if (verbose) 
    {if (ok) cat("ok\n")
     else    cat("failed!\n")
    }

  if (synopsis) 
    {if (verbose) cat("All tagged.matrix class tests completed")
     if (all.ok) cat(" ok\n\n")
     else    cat(", some failed!\n\n")
    }
  invisible(all.ok)
}

#   Copyright 1993, 1994, 1995, 1996  Bank of Canada.
#   Copyright 1997 (June), Paul Gilbert.
#   Copyright 1997 (Aug.), Bank of Canada.
#   Copyright 1998, Bank of Canada.

#   The user of this software has the right to use, reproduce and distribute it.
#   Bank of Canada makes no warranties with respect to the software or its 
#   fitness for any particular purpose. The software is distributed by the Bank
#   of Canada and by Paul Gilbert solely on an "as is" basis. By using the  
#   software, user agrees to accept the entire risk of using this software.

################################################################################


###########################################################################

# Simple monitoring functions and data checking        <<<<<<<<<<<<

###########################################################################


check.for.value.changes <- function(data.names, verification.data,
     discard.current=F,
     ignore.before= NULL,
     fuzz=1e-10)
  { # Check if data is modified or if more data is available.
    # data.names is an object of class c("TSPADIdata","TSdata").
    # verification.data is an object of class TSdata.
    # T is returned for any series which has a modified value.
    #   NA in one series and not in the other is counted as modified.
    # If data are not the same length then series are padded with NA
    #  so new NAs on the end will not count as a change.
    # It is assumed that changes happen at the end (not the beginning) of
    #   the data. The data is trimmed at the beginning to the point where
    #   all series are available. (this simplifies padding the end with NA)
    # If ignore.before is not NULL it should indicate a year and period
    #   before which data is trimmed, no comparison is performed and the
    #   data before is not returned. If there are NAs at the beginning then
    #   trimming as described above may make the data even shorter than
    #   indicated by ignore.before.
    # discard.current controls whether current period data is considered.
    #  (some series are available for a period from the first day of the
    #   period, and are updated daily. Usually these should be discarded
    #   by setting discard.current=T)

   data <-TSdata(data.names) 
   if (discard.current)
     {year.mo <- c(date()$y,date()$m) - c(0,1)
      data  <- window( data,  end=year.mo, warn=F )
     }
   if (!is.null(ignore.before)) 
     {data <- window(data, start= ignore.before)
      verification.data <-window(verification.data, start= ignore.before)
     }
   data <-trim.na(data, Start=T, End=F)
   verification.data <-trim.na(verification.data, Start=T, End=F)
   # which series are changed:
   if (is.null(input.series.names(data.names))) in.up <- NULL
   else
     {ld <-input.periods(data)
      lv <-input.periods(verification.data)
      l <- max(ld, lv)
      if (ld < l)
        input.data(data) <- ts(rbind(input.data(data), matrix(NA,l-ld, input.dimension(data))),
                         start=start(input.data(data)),
                     frequency=frequency(data))
      if (lv < l)
        input.data(verification.data) <- rbind(input.data(verification.data), 
                                        matrix(NA,l-lv, input.dimension(data)))
      z <- (is.na(input.data(data)) & is.na(input.data(verification.data)))   # both NA
      z <- (abs(input.data(data) - input.data(verification.data)) <= fuzz) | z
      z <- z & !is.na(z)
      in.up <- !apply(z,2, all)
     }
   if (is.null(output.series.names(data.names))) out.up <- NULL
   else
     {ld <-output.periods(data)
      lv <-output.periods(verification.data)
      l <- max(ld, lv)
      if (ld < l)
        output.data(data) <- ts(rbind(output.data(data), matrix(NA,l-ld, output.dimension(data))),
                         start=start(data), frequency=frequency(data))
      if (lv < l)
        output.data(verification.data) <- rbind(output.data(verification.data), 
                                      matrix(NA,l-lv, output.dimension(data)))
      z <- ( is.na(output.data(data)) & is.na(output.data(verification.data)))    # both NA
      z <- (abs(output.data(data) - output.data(verification.data)) <= fuzz) | z
      z <- z & !is.na(z)
      out.up <- !apply(z,2, all)
     }
   list(any(c(in.up,out.up)), input=in.up, output=out.up, data=data)   
  }

check.for.file.date.changes <- function(data.names, verification.dates)
  {# check file dates against previous dates
   # It is preferable to do file date checking with a Unix shell script rather 
   #   than in S, and then start S for further checks only when the time stamp
   #   on the database files has changed.
   up.in <-NULL
   if (!is.null(input.series.names(data.names)))
    {for (f in data.names$input$db) up.in <- c(up.in, file.date.info(f))
     inT <-any(verification.dates$input != up.in)
    }
   up.out <-NULL
   for (f in data.names$output$db) up.out <- c(up.out,file.date.info(f))
   outT <-any(verification.dates$output != up.out)
   list( any(c(inT,outT)), input=inT, output=outT, 
         dates=list(input=up.in, output=up.out))
  }


simple.monitoring <- function(model, data.names, 
   previous.data=NULL,
   mail.list=NULL,
   error.mail.list=whoami(),
   message.title="Simple Monitoring",
   message.subject="Simple Monitoring",
   message.footnote=NULL,
   show.start= c(0,-3),
   show.end  = c(0,12),    
   report.variables= series.names(data.names),
   data.sub.heading=NULL,
   data.tag=" ",
   forecast.tag="f",
   run.again=F,
   save.as=NULL)

{# automatic monitoring with e-mail of results
 # model is a TSmodel. data.names is a TSdata (names) object.
 # mail.list and error.mail.list should be single strings (not vectors)
 #   but the string can contain multiple user ids for mail
 #   If mail.list is NULL (default) then mail is not sent (useful for testing). 
 #   If error.mail.list  is  NULL   then mail is not sent (useful for testing).
 # The default for error.mail.list is the result of whoami().
 # This version does not allow for 
 #     -combining forecasts (ie. monitoring.data or overriding data)
 #     -input (policy) projections
 # See combination.monitoring for these features.
 # report.variables indicates output variables which are reported. It should
 #   correspond to a subset of names returned by series.names.
 # data.tag and forecast.tag are tags placed beside data points in the report.
 # If run.again is T then the monitoring is run regardless of data updates.
 # show.end is the number of periods after the end of data (ie forecasts)
 #   which should be displayed.
 # show.start is the number of periods before the end of data (ie history)
 #   which should be displayed. It is added to the end so it should be negative.
 

 # Step 0 -  prepare message files and error checking
    error.message <- c(message.title, paste(date(), collapse="."),
              "An error condition occurred running simple.monitoring.",
              "The message.file at the time of the error follows:") 
    message <- ""     
    on.exit(mail(error.mail.list,
                 subject=paste("error ",message.subject),
                 text= c(error.message, message)))
    if ( class(model)[1] == "TSestModel" ) model <- TSmodel(model)
    if (!is.null(data.names$pad.end))
       {if(!data.names$pad.end)
          warning("pad.end in data definition may disable retrieving all data.")
       } 
    else if (!is.null(data.names$pad))
       {if(!data.names$pad)
          warning("pad in data definition may disable retrieving all data.")
       } 

# The following line is useful for debugging
#mail(error.mail.list, subject=paste("checking ",message.subject), 
#                         text=paste(date(), collapse="."))

 # Step 1 - retrieve & check for updated data  or
 #            initialize system and if previous.data is NULL
    if (is.null(previous.data))
      {data <- TSdata(data.names)
       message <- "Initializing simple monitoring:"   
       status <- "Simple monitoring initialized."   
      }
    else if (run.again)
      {data <-previous.data  
       status <- "Simple monitoring re-run."   
      }
    else
      {updated.data<-check.for.value.changes(data.names,
                           verification.data=previous.data,
                           discard.current=T)
       if(updated.data[[1]])
         {data <-updated.data$data
          message <- c("data updates: ", 
               input.series.names(data)[ input.data(updated.data)],
              output.series.names(data)[output.data(updated.data)])
          status <- "Simple monitoring updated."   
         }
       else
         {on.exit()
          return(invisible(list(data=previous.data, 
                status="Simple monitoring updates not necessary.")))
         }
      }

 # Step 2 - check data
   # Sometimes data is available as soon as there are any days in a month (with
   #   ignore on in Fame). The following 4 lines trim these, but that may not be
   #   the best way to handle them.
   year.mo <- c(date()$y,date()$m) - c(0,1)
   data  <- window(data,  end=year.mo, warn=F )

 # Step 3 - run forecast
   pred<-forecast(model, data)$forecast[[1]]
   pred <-splice.tagged.matrix(output.data(data), pred, tag1=data.tag,tag2=forecast.tag) 
 
 # Step 4 - generate report and mail
    message <-c(message,"The forecasts are now:")
    #starting and end period for plots & printing:
    Start<-(output.end(data)+show.start) 
    End  <-(output.end(data)+show.end)

    report.variables$input<- 
            (report.variables$input == input.series.names(data.names))
    report.variables$output<- 
            (report.variables$output == output.series.names(data.names))
    rv <- tagged.matrix(pred[,report.variables$output, drop=F],
                 tags= (attr(pred,"tags")) [,report.variables$output, drop=F])
    tframe(rv) <- tframe(pred)
    inp <-tagged.matrix(input.data(data)[,report.variables$input, drop=F],tags= data.tag)
    tframe(inp) <-  tframe(input.data(data))
    rv <- window( tbind( inp, rv), start=Start, end=End, warn=F)   
    message <- c(message,fprint(rv, digits=5, sub.title=data.sub.heading)) 

    if (!is.null(message.footnote)) message <-c(message, message.footnote)
    mail(mail.list, subject=message.subject, text= message)

 # Step 4 - clean-up
    if (!is.null(save.as)) 
       assign(save.as,list(model=model, data=data, pred=pred), where=1)
    on.exit()
    #return latest data for comparison next time. Note - the forecast is NOT
    # returned (but may be saved above).
    invisible(list(data=data, status=status, message=message)) 
}

watch.data <- function(data.names, 
   previous.data=NULL,
   mail.list="gilp",
   error.mail.list=NULL,
   message.title="Data Monitor\n",
   message.subject="Data Monitor",
   message.footnote=NULL)

{# monitors data bases and check series for changes with e-mail of results.
 # this should be used with a script which watches for file date changes.
 #  ( see example in the file watch.data.readme)
 # data.names is a TSdata (names) object.
 # mail.list and error.mail.list should be single strings (not vectors)
 # If mail.list is null then mail is not sent (useful for testing).
 #  but the string can contain multiple user ids for mail
 # previous.data must normally be supplied. If it is not (ie. if it is NULL)
 # then the system will be initiallized and the returned result will be
 # the previous.data for the next time the function is called.

 # Step 0 - prepare message files 
    error.message <- c(message.title, paste(date(), collapse="."),
              "An error condition occurred running watch.data.",
              "The message.file at the time of the error follows:") 
    message <- ""     
    on.exit(mail(error.mail.list, subject=paste("error ",message.subject),
                 text= c(error.message, message)))

 # Step 1 - retrieve & check for updated data 
    data.names <- TSdata.name.to.form1(data.names)
    data.names$pad.end <- T
    #  Initialize system and exit if previous.data is NULL
    if (is.null(previous.data))
      {current.data <- TSdata(data.names)
       on.exit()
       #return latest data for comparison next time
       return(invisible(list(data=current.data,
           status="System watch.data initialized."))) 
      }
    update<-check.for.value.changes(data.names,
                           verification.data=previous.data$data,
                           discard.current=F)
    if (!update[[1]] )
        {on.exit()
         return(invisible(list(data=previous.data$data, 
             status="No data updates.")))
        }
    else
       message <- c(message, "data updates: ", 
              output.series.names(update$data)[update$output],)

 # Step 2 - mail 
    if(!is.null(message.footnote)) message <- c(message,message.footnote)
    mail(mail.list, subject=message.subject, text= message)

 # Step 3 - clean-up
    on.exit()
    #return latest data for comparison next time
    invisible(list(data=update$data, status="Data has been updated.")) 
}


###########################################################################

# Tests function for data retrieval for simple monitoring    <<<<<<<<<<<<

###########################################################################

simple.monitor.function.tests <- function( verbose=T, synopsis=T, 
         fuzz.small=1e-14, fuzz.large=1e-8,
         server.process = padi.server.process(),
         cleanup.script = padi.cleanup.script() )
{# Some of the tests here are really for functions defined in dse1 ... dse3
 #   but are not tested there to avoid assuming TSPADI (or Fame) access is
 # available. The main short coming of these tests is that they do not test
 #     functions which produce output or graphs.
 # These tests require access to Fame data bases and the files:
 #          monitoring.test.db    fake database 
 #          monitoring.test.info  comparison info. to check results

 # Note also that the test data is not real data (it may have been differenced
 #  or otherwise transformed) and is only intended to test that functions
 #  work as originally specified. 

  server <- local.host.netname()
  db     <- paste(DSE.HOME,"/data/monitoring.test.db",sep="")

  if (synopsis & !verbose) cat("All simple monitor tests ...")
  all.ok <- T

  if (verbose) cat("simple monitor test 0 ... ")
  # simulate a database server
  pid <- start.padi.server(server=server,
           dbname=db, 
           server.process=server.process)
  on.exit(cleanup.padi.server(pid, cleanup.script=cleanup.script))

  # wait for server to start 
     for (i in 1:30)
       {if (check.padi.server(server)) break
        sleep(1)
       }
  ok <- T
  all.ok <- all.ok & ok 
  if (verbose) 
    {if (ok) cat("ok\n")
     else    cat("failed!\n")
    }


  if (verbose) cat("simple monitor test 1 ... ")
  #  dbname=db would not be nec. with a public mode fame server
  test.data.names <- list(
      input=list(series="B14017", server=server, dbname=db),
#      input.transformations= "diff",
      output=list(series=c( "P484549", "I37026", "lfsa201","b3400"),
                    server=rep(server,4), dbname= rep(db,4) ),
#      output.transformations= rep("percent.change.vector",4),
      pad.end =T)
  class(test.data.names) <- c("TSPADIdata", "TSdata")
   
  z <-availability(test.data.names, verbose=F) 
  ok <- all(c(z$start==rep(c(1974,2),5), 
              z$end==rep(c(1993,9),5), 
              z$freq==rep(12,5) ))
  all.ok <- all.ok & ok 
  if (verbose) 
    {if (ok) cat("ok\n")
     else    cat("failed!\n")
    }


# the following sets ets.test.data, monitor.test.data, verification.data
#      and  monitoring.test
  source(paste(DSE.HOME,"/data/monitoring.test.info", sep=""))

  if (verbose) cat("simple monitor test 2 ... ") 
  v.data <- verification.data
  output.data(v.data) <- output.data(v.data)[,c(1,2,6,7)]
  tframe(output.data(v.data)) <- tframe(output.data(verification.data))
  ok <- is.TSdata(v.data)
  all.ok <- all.ok & ok 
  if (verbose) 
    {if (ok) cat("ok\n")
     else    cat("failed! (loading verification data)\n")
    }

  if (verbose) cat("simple monitor test 3 ... ")
  hist.data <-retrieve.and.verify.data(test.data.names, 
                                    verification.data=v.data)
  ok <- test.equal(hist.data, ets.test.data, fuzz=fuzz.small)
  all.ok <- all.ok & ok 
  if (verbose) 
    {if (ok) cat("ok\n")
     else    cat("failed! (retrieve.and.verify.data)\n")
    }


  if (verbose) cat("simple monitor test 4 ... ")
  monitoring<-simple.monitoring (monitoring.test.model, test.data.names, 
                   previous.data=NULL, mail.list=NULL, error.mail.list=NULL) 
  ok <-  monitoring$status == "Simple monitoring initialized."   
  if (verbose) cat("\n This test produces a warning: Input is not longer than output data. No forecasts produced...")
  # note that the following does not result in forecasts (and the forecast
  #   function produces a warning) because the input data does not extend
  #   beyond the output data.
  monitoring<-simple.monitoring (monitoring.test.model, test.data.names, 
           previous.data=monitoring$data, mail.list=NULL, error.mail.list=NULL) 
  ok <- ok & (monitoring$status == "Simple monitoring updates not necessary.")
  monitoring<-simple.monitoring (monitoring.test.model, test.data.names, 
                 previous.data=monitoring$data, 
                 mail.list=NULL, error.mail.list=NULL, run.again=T) 
  ok <- ok & (monitoring$status == "Simple monitoring re-run.")
  ok <- ok & monitoring$message[7] == 
          "1993 Sep   0.110000   0.383440   0.397520   0.355500   0.947460 "
  ok <- ok & sum(output.data(monitoring$data))==235.64806565791809589
  output.data(monitoring$data) <- window(output.data(monitoring$data), end=c(1993,8))
  monitoring<-simple.monitoring (monitoring.test.model, test.data.names, 
          previous.data=monitoring$data, mail.list=NULL, error.mail.list=NULL) 
  ok <- ok & (monitoring$status == "Simple monitoring updated.") &
      sum(output.data(monitoring$data)) == 235.64806565791809589
  all.ok <- all.ok & ok 
  if (verbose) 
    {if (ok) cat("ok\n")
     else    cat("failed! (simple.monitoring)\n")
    }


  if (verbose) cat("simple monitor test 5 ... ")

  watch <- watch.data(test.data.names, previous.data=NULL, mail.list=NULL)
  ok <- (watch$status == "System watch.data initialized.") & 
         sum(output.data(watch$data))== 235.64806565791809589

  watch <- watch.data(test.data.names, previous.data=watch, mail.list=NULL)
  ok <- (watch$status == "No data updates.") & 
           sum(input.data(watch$data))== -4.1300000572204575988
  watch$data <- window(watch$data, start=c(1993, 8))
  watch <- watch.data(test.data.names, previous.data=watch, mail.list=NULL)
  ok <- (watch$status == "Data has been updated.") & 
          sum(output.data(watch$data))== 235.64806565791809589

  all.ok <- all.ok & ok 
  if (verbose) 
    {if (ok) cat("ok\n")
     else    cat("failed! (watch.data)\n")
    }

  if (synopsis) 
    {if (verbose) cat("All simple monitor tests completed")
     if (all.ok) cat(" ok\n\n")
     else    cat(", some failed!\n\n")
    }
  invisible(all.ok)
}
#   Copyright 1993, 1994, 1995, 1996  Bank of Canada.
#   Copyright 1997 (June), Paul Gilbert.
#   Copyright 1997 (Aug.), Bank of Canada.
#   Copyright 1998, Bank of Canada.

#   The user of this software has the right to use, reproduce and distribute it.
#   Bank of Canada makes no warranties with respect to the software or its 
#   fitness for any particular purpose. The software is distributed by the Bank
#   of Canada and by Paul Gilbert solely on an "as is" basis. By using the  
#   software, user agrees to accept the entire risk of using this software.

################################################################################



###########################################################################

# Combination forecasting  functions.                       <<<<<<<<<<<<
#  These functions allow for the use of over-riding data    <<<<<<<<<<<<
#    which may come from other forecasts or monitoring and  <<<<<<<<<<<<
#    can be used to augment (and replace) actual data.      <<<<<<<<<<<<
#  Also, input (policy) variable forecasts can be used.     <<<<<<<<<<<<
#  NB. The combination is not in the sense of averaging     <<<<<<<<<<<<
#       together forecasts.                                 <<<<<<<<<<<<

###########################################################################

combine.and.forecast<- function(model, new.data, overlapping.period.forecast.tag="g", forecast.tag="f") 

{# model should be a TSmodel.
 # new data should be a list with $data and $overriding.data.
 # It can also contain elements data.tag and overriding.data.tag, character string
 #   tags which are passed along to construct.data.to.override.horizon.
 # $overriding.data is used in place of data and model forecasts to the horizon
 # for which it is available. $overriding.data should also include any input (policy)
 # variables to the forecast horzon.
 # best.guess in the result is a combination of available data, overriding.data,
 # and predictions. 
 # first splice and fill with model predictions.
 con.data <- construct.data.to.override.horizon(new.data, model, plot=F, 
                      forecast.tag=overlapping.period.forecast.tag) 
 pred <-l(model, con.data ,predictT=dim(con.data$input)[1])$estimates$pred 
   # do residual analysis ?
# forecast<-forecast(l(model, con.data), percent=c(80,100,120), horizon=6, plot=F)
#   pchange<-percent.change(forecast[[1]],forecast[[2]],forecast[[3]], base=base,lag=12,cumulate=T,e=T)
 best.guess <-splice.tagged.matrix(con.data$output, pred, 
                  tag1=con.data$output.tags,tag2=forecast.tag) 
# the following result could also include con.data and pred, but it should be possible to
#    reconstruct these from the information in the list.
 invisible(list(model=model,
                data=new.data$data,
                overriding.data=new.data$overriding.data, 
                override=con.data$override,
                best.guess=best.guess))
}

reconstruct.combined.forecast<- function(combined.forecast) 
{# use the result of combine.and.forecast to re-do and verify results
 con.data <- construct.data.to.override.horizon(combined.forecast, combined.forecast$model, plot=F)
 pred <-l(combined.forecast$model, con.data ,predictT=dim(con.data$input)[1])$estimates$pred 
 best.guess <-splice.tagged.matrix(con.data$output, pred) 
 all(combined.forecast$best.guess==best.guess)
}

graph.combined.forecast<- function(combined.forecast,verbose=F, 
       Start=start(combined.forecast$data$output),
       Title="Projection", select.inputs=NULL, select.outputs=NULL, pause=T)
{# if verbose is T additional information is provided
 # if pause is true graphics are paused between pages.
   if (pause) dev.ask(T)
   if (verbose)
     {graph(combined.forecast$data, Start=Start, Title="Data and combined.forecast")
      graph(combined.forecast$pred, Start=Start,
            Title="Model predictions (one step ahead for history)")
     }
   graph.data <- combined.forecast$data
   graph.data$output <- combined.forecast$best.guess
   if (is.null(select.inputs))  select.inputs  <- seq(dim(graph.data$input)[2])
   if (is.null(select.outputs)) select.outputs <- seq(dim(graph.data$output)[2])
   graph(graph.data, Start=Start, Title="Projection", 
           select.inputs=select.inputs, select.outputs=select.outputs)
#   graph(combined.forecast$forecast[[2]],combined.forecast$forecast[[1]],
#         combined.forecast$forecast[[3]], Start=Start,
#         Title="Projection using future policy=most recent value and 20% higher and lower")
#   graph(combined.forecast$pchange[[2]],combined.forecast$pchange[[1]],
#         combined.forecast$pchange[[3]],Start=Start, Title=
#    "Year over year percent change using future policy=most recent value and 20% higher and lower")
   invisible()
}

###########################################################################

# functions for misc. data retrieval, checking, and transformation <<<<<<<<<<<<

###########################################################################


construct.data.to.override.horizon <- function(new.data, model, plot=T, forecast.tag="f")
{# model should be a TSmodel.
 # new.data should be a list with $data and $overriding.data.
 # $overriding.data is used in place of $data and model forecasts to 
 # the horizon for which it is available. 
 #  Splice together $data and $overriding.data and if necessary
 #  calculate predictions for $overriding.data period and use them where $overriding.data
 #  or $data are not available, then return complete data set 
 #  to the end of the $overriding.data horizon, along with input data.
 #    (Typically the end of $overriding.data$output determines the periods
 #     for which forecast are combined and the end of $overriding.data$input
 #     determines how far into the future the model is used to extend the
 #     combined forecast. )
 #  Note that the $overriding.data is used in place of data in the 
 #  returned data set to allow for over-riding with anticipated data revisions.
 #  However, for any predictions during the combined.forecast period (ie. to augment
 #  $data and $overriding.data as returned by this function),  
 #  only $data is used and only to the last period for which observations
 #  for all variables are available.

 # if $overriding.data and $data overlap indicate override locations in 
 #     logical matrix dup:

 # tbind aligns the matrices
 dup <- tbind(output.data(new.data$data), output.data(new.data$overriding.data))
 if (!is.null(dup))
  {p <- output.dimension(new.data$data)
   dup <- (!is.na(dup[,1:p,drop=F])) & (!is.na(dup[,(p+1):(2*p),drop=F]))
  }

    # This can be used to provide a warning. eg
    #if (any(dup))
    #  {cat("WARNING:overriding is being used where data is available as follows:\n")
    #   print(dup)
    #  }

 z <- trim.na(tagged.matrix(new.data$data$output,
                              new.data$data$output.tags), End=F)
 z <- splice.tagged.matrix(new.data$overriding.data$output,z,
                    tag1=new.data$overriding.data$output.tags,
                    tag2=attr(z,"tags"))
 Start <- start(z)
 if (is.null(new.data$data$input)) z.in <-NULL
 else
   {# note that $overriding.data does not override actual data for $input, but 
    #  that could be changed by reversing the order in the next line. (extra  
    #  code is necessary to give a warning.)
    z.in <-trim.na.tagged.matrix(splice.tagged.matrix(new.data$data$input, 
              new.data$overriding.data$input,
              tag1=new.data$data$input.tags, 
              tag2=new.data$overriding.data$input.tags))
    Start <- latest.start(z, z.in)
    z.in <- window.tagged.matrix(z.in, start=Start, warn=F)
    if (any(is.na(z.in)))
       stop(paste("Input (exogenous) series data cannot be specified as NA. (note ",
                  "differenced data requires an overlap of one period at the end of ",
                  "historical data and the beginning of monitoring overriding data.)"))
   }
 z <- window.tagged.matrix(z, start=Start, warn=F)
 con.data <- list(output=z,  input=z.in)

 # now augment $data and $overriding.data with model predictions for 
 #  the combined forecast period if necessary.
 if (any(is.na(con.data$output)))    
   {z <- list(input=con.data$input, output=trim.na(new.data$data$output))
    class(z) <- "TSdata"
    pred <- l(model,z, predictT= dim(con.data$output)[1])$estimates$pred
    z <-splice.tagged.matrix(con.data$output,pred, 
                    tag1=con.data$output.tags, tag2=forecast.tag)
    con.data$output <- z
   }

 con.data<- TSdata(con.data)
 con.data$override <- dup
 if (plot && exists.graphics.device()) 
    {graph(con.data,Start=(end(output.data(data))-c(1,0)), 
           Title="Historical and overriding data data")
    }
  invisible(con.data)
}

get.overriding.data<- function(file="overriding.data", 
 first.input="",first.output="", second.output="", m=1, p=10)
{#Get other data eg(monitoring or other forecast data) 
  #   N.B. This cues on series names in the file
  # m is the number of input series
  # p is the number of output series
  z  <- scan(file=file,what=character())
  first.in   <- (1:length(z))[z==first.input] 
  if (0== length(first.in))
     stop(paste("Cannot find keying string:", first.input," in file", file))
  first.out  <- (1:length(z))[z==first.output] 
  if (0== length(first.out))
     stop(paste("Cannot find keying string:", first.output," in file", file))
  second.out <- (1:length(z))[z==second.output] 
  if (0== length(second.out))
     stop(paste("Cannot find keying string:", second.output," in file", file))
  input.periods <- (first.out-(first.in+m))/m     
  zz <- matrix(z[first.in:(first.out-1)],(input.periods+1),m)
  input.names <- zz[1,]
  input <-  matrix( as.numeric(zz[2:(1+input.periods),]), input.periods,m)
  dimnames(input) <- list(NULL,input.names)
  input <- ts(input, start=as.integer(z[1:2]),frequency=12)
  output.periods<- second.out-(first.out+1)
  zz <- matrix(z[first.out:length(z)],(output.periods+1),p)
  output.names <- zz[1,]
  output <-  matrix( as.numeric(zz[2:(1+output.periods),]), output.periods,p)
  dimnames(output) <- list(NULL,output.names)
  output <- ts(output, start=as.integer(z[1:2]),frequency=12)
  zz <-list(input=input , output=output)
  class(zz) <- "TSdata"
  zz
}


#graph.combined.forecast(combined.forecast,verbose=F, 
#      Start=start(combined.forecast$data$output),
#      Title="Projection", select.inputs=NULL, select.outputs=NULL)


restrict.overriding.data<-function(data, overriding.horizon=0)  
{#This function truncates overriding.data$output if it extends 
 #  overriding.horizon periods beyond the present. 
 year.mo <- c(date()$y,date()$m) - c(0,1) + c(0,overriding.horizon)
#check this - also note NAs should not be nec in overriding fame data
browser()
 data$output <-window(data$output, end=year.mo, warn=F )
 invisible(data)
}

###########################################################################

# functions for e-mail of results of combination forecasting <<<<<<<<<<<<

###########################################################################

combination.monitoring <- function(model, data.names,
   previous.data=NULL,
   overriding.data.names=NULL, 
   restrict.overriding.data=T, overriding.horizon=0,
   mail.list=NULL,
   error.mail.list=NULL,
   message.title="Combination Monitoring",
   message.subject="Combination Monitoring",
   message.footnote=NULL,
   show.start= c(0,-3),
   show.end  = c(0,12),    
   report.variables=series.names(data.names),
   data.sub.heading=NULL,
   data.tag=" ",
   future.input.data.tag="p",
   overriding.data.tag="m",
   overlapping.period.forecast.tag="g",
   forecast.tag="f",
   run.again=F,
   save.as=NULL)

{# automatic monitoring with e-mail of results.
 # This version allows for 
 #     -combining forecasts (ie. monitoring or other forecast data)
 #     -input (policy) projections
 # If these feature are not need see simple.monitoring.
 # mail.list and error.mail.list should be single strings (not vectors)
 #  but the string can contain multiple user ids for mail
 # If overriding.data.names=NULL then no overriding data is used.
 # report.variables indicates output variables which are reported. If NULL,
 # then all outputs are reported.
 # show.end is min of this and overriding.data$input if needed.

 # Step 0 - prepare message files and error checking
    error.message <- c(message.title, paste(date(), collapse="."),
              "An error condition occurred running combination.monitoring.",
              "The message.file at the time of the error follows:") 
    message <- ""     
    on.exit(mail(error.mail.list, subject=paste("error ", message.subject),
                 text= c(error.message, message)))
    if ( class(model)[1] == "TSestModel" ) model <- model$model
    if (!is.null(data.names$pad.end))
       {if(!data.names$pad.end)
          warning("pad.end in data definition may disable retrieving all data.")
       } 
    else if (!is.null(data.names$pad))
       {if(!data.names$pad)
          warning("pad in data definition may disable retrieving all data.")
       } 

# The following line can be removed if the code works reliably
   mail(error.mail.list,subject=paste("checking ",message.subject),
                           text=paste(date(), collapse="."))

 # Step 1 - retrieve & check for updated data  or
 #            initialize system and if previous.data is NULL
    if (is.null(previous.data))
      {data <- TSdata(data.names)
       message <- "Initializing combination monitoring:"   
       status <- "Combination monitoring initialized."   
      }
    else if (run.again)
      {data <-previous.data$data  
       overriding.update <- previous.data$overriding.data
       status <- "Combination monitoring re-run."   
      }
    else
      {updated.data<-check.for.value.changes(data.names,
                           verification.data=previous.data$data,
                           discard.current=T)
       if (is.null(overriding.data.names)) overriding.update<-list(F)
       else overriding.update<-check.for.value.changes(overriding.data.names,
                           verification.data=previous.data$overriding.data)
       if(updated.data[[1]] | overriding.update[[1]])
         {status <- "Combination monitoring updated."     
          if(updated.data[[1]])
            {data <-updated.data$data
             message <- c("data updates: ", 
                 series.names(data)$input[updated.data$input],
                 series.names(data)$output[updated.data$output])
            }
          if(overriding.update[[1]])
            {overriding.data <- overriding.update$data
             if(restrict.overriding.data & (!is.null(overriding.data$output))) 
                overriding.data <- restrict.overriding.data(overriding.data, 
                                 overriding.horizon=overriding.horizon)
             message <- c(message,"monitoring data updates: ",
             series.names(overriding.data)$input[ overriding.update$input],
             series.names(overriding.data)$output[overriding.update$output])
            }
         }
       else
         {on.exit()
          return(invisible(list(data=previous.data, 
                status="Combination monitoring updates not necessary.")))
         }
      }

 # Step 2 - check data and overriding data
   # Sometimes data is available as soon as there are any days in a month (with
   #   ignore on in Fame). The following 4 lines trim these, but that may not be
   #   the best way to handle them.
   year.mo <- c(date()$y,date()$m) - c(0,1)
   data  <- window(data,  end=year.mo, warn=F )
   fr <- c(frequency(data), 1)
      
   # check matching of starting date with end of available data.
   #   period for which all data is available in data
   end.ets <- end(trim.na(output.data(data))) 
   if (!is.null(overriding.data))
    {if (is.null(overriding.data$output))
     {overriding.data$output <- ts(matrix(NA, 1, output.dimension(data)),
                           end=end(data$output), 
                           frequency=frequency(data$output), 
                           names=dimnames(data$output)[[2]])
      if (!is.null(data$output.names))
         overriding.data$output.names <- data$output.names
     }
   else
     {if (!( (1+fr %*% end.ets) >= (fr %*%start(overriding.data$output))))
        stop(paste("Monitoring data (or NAs) must be indicated after ", end.ets))
      if (1== latest.end.index(output.data(data), output.data(overriding.data)))
         warning(paste("Overriding data file does not appear to be updated.",
         "True data is available past the end of the overriding data."))
    }}   

    if (is.null(overriding.data.names)) overriding.data <- NULL
    else
       overriding.data <- tagged.matrix(overriding.data,
          input.tags=future.input.data.tag, output.tags=overriding.data.tag)
    data <- tagged.matrix(data, input.tags=data.tag, output.tags=data.tag)

 # Step 3 - run forecast
   # warnings from this should be mailed!!!!
    combined.forecast<-combine.and.forecast(model, list(data, overriding.data),
           overlapping.period.forecast.tag=overlapping.period.forecast.tag, 
           forecast.tag=forecast.tag) 

 # Step 4 - write and mail files
    message <- c(message, "Projections are conditioned on forecast of ",
                            series.names(updated.data$data)$input, 
                          "                        with tranformation ",
                           data.names$input.transformations,
                          "The forecasts are now:")
    #starting and end period for plots & printing:
    Start<-(end(combined.forecast$data$output)+show.start) 
    End  <-(end(combined.forecast$data$output)+show.end)
    # this can be too long if sufficient input data is not provided, so: 
    if ((fr %*% end(combined.forecast$best.guess)) < ((End-c(0,1)) %*% fr))
       End  <-end(combined.forecast$best.guess)

    report.variables$input<- 
            (report.variables$input == series.names(data.names)$input)
    report.variables$output<- 
            (report.variables$output == series.names(data.names)$output)


    rv <- tagged.matrix(
              combined.forecast$best.guess[,report.variables$output, drop=F],
              tags= (attr(combined.forecast$best.guess,"tags")
                             ) [,report.variables$output, drop=F])
    tframe(rv) <- tframe(combined.forecast$best.guess)
    inp <- splice(combined.forecast$data$input, 
                  combined.forecast$overriding.data$input,
                  tag1=data.tag, tag2=future.input.data.tag)
    rv <-window(cbind(inp,rv), start=Start, end=End, warn=F) 
    message <- c(message,fprint(rv, digits=5, sub.title=data.sub.heading)) 

    if (any(combined.forecast$override))
       {message <- c(message, "WARNING: overriding data is being used where historical data is available as follows:",
              combined.forecast$override)
       }

#    print(window(tsmatrix(combined.forecast$data$input, combined.forecast$best.guess), 
#      start=Start), digits=print.digits)

# The following needs a postscipt viewer like gv or pageview
#    postscript(file=graphics.file, width=7, height=8, pointsize=14,
#        horizontal=F, onefile=F, print.it=F, append=F)
#    graph.combined.forecast(combined.forecast, Start=Start)
#    dev.off()
#    message <- c(message,"For graphic (in OpenWindows) type:\n    pageview ")
#    if("/" == substring(graphics.file,1,1) )
#             message <- c(message,graphics.file)
#    else
#      {pwd <- present.working.directory()
#       if("/tmp_mnt" == substring(pwd,1,8)) pwd <-substring(pwd,9)
#       message <- c(message,paste(pwd,"/",graphics.file, sep=""))
#      }
#    message <- c(message," in a command tool window. (Be patient. It takes a few seconds.)")

    if (!is.null(message.footnote)) message <-c(message, message.footnote)
    mail(mail.list, subject=message.subject, text= message)


 # Step 4 - clean-up
    if (!is.null(save.as)) 
      {assign(save.as, combined.forecast, where=1)
#       file.copy( graphics.file, save.as)   # save graph
      } 
    if (updated.data[[1]] ) previous.data$data   <- updated.data$data
    if ( overriding.update[[1]])
       previous.data$overriding.data<- overriding.update$data
    on.exit()
    #return latest data for comparison next time
    invisible(list(data=previous.data, status=status, message=message)) 
}


###########################################################################

# Tests function    <<<<<<<<<<<<

###########################################################################

combination.monitor.function.tests <- function( verbose=T, synopsis=T, 
         fuzz.small=1e-10,
         server.process = padi.server.process(),
         cleanup.script = padi.cleanup.script() )
{# Some of the tests here are really for functions defined in dse1 ... dse3
 #   but are not tested there to avoid assuming Fame access is available.
 # The main short coming of these tests is that they do not test
 #     functions which produce output or graphs.
 # These tests require access to Fame data bases and the files:
 #          monitoring.test.db    fake database 
 #          monitoring.test.info  comparison info. to check results
 #          monitoring.test.data  fake over-riding data 

 # Note also that the test data is not real data (it may have been differenced
 #  or otherwise transformed) and is only intended to test that functions
 #  work as originally specified. 

  server <- local.host.netname()
  db     <- paste(DSE.HOME,"/data/monitoring.test.db",sep="")

  if (synopsis & !verbose) cat("All combination monitor tests ...")
  all.ok <- T

  if (verbose) cat("combination monitor test 0 ... ")
  # simulated a database server
  pid <- start.padi.server(server=server, dbname=db, 
           server.process=server.process)
  on.exit(cleanup.padi.server(pid, cleanup.script=cleanup.script))

  # wait for server to start 
     for (i in 1:30)
       {if (check.padi.server(server)) break
        sleep(1)
       }
  ok <- T
  all.ok <- all.ok & ok 
  if (verbose) 
    {if (ok) cat("ok\n")
     else    cat("failed!\n")
    }

  if (verbose) cat("combination monitor test 1 ... ")
  #  dbname=db would not be nec. with a public mode fame server
  test.data.names <- list(
      input=list(series="B14017", dbname=db, server=server),
#      input.transformations= "diff",
      output=list(series=c( "P484549", "I37026", "lfsa201","b3400"),
                  dbname=rep(db, 4), server=rep(server,4) ),
#      output.transformations= rep("percent.change.vector",4),
      pad.end =T)
  class(test.data.names) <- c("TSPADIdata", "TSdata")

  source(paste(DSE.HOME,"/data/monitoring.test.info", sep=""))

  v.data <- verification.data
  v.data$output <- v.data$output[,c(1,2,6,7)]
  tframe(v.data$output) <- tframe(verification.data$output)
  ok <- is.TSdata(v.data)
  all.ok <- ok 
  if (verbose) 
    {if (ok) cat("ok\n")
     else    cat("failed! (loading verification data)\n")
    }

  if (verbose) cat("combination monitor test 2 ... ")
  data <-retrieve.and.verify.data(test.data.names, 
                                    verification.data=v.data)
  ok <- test.equal(data, ets.test.data, fuzz=fuzz.small)
  all.ok <- all.ok & ok 
  if (verbose) 
    {if (ok) cat("ok\n")
     else    cat("failed! (retrieve.and.verify.data)\n")
    }

  if (verbose) cat("combination monitor test 3 ... ")
  overriding.data <- get.overriding.data(
                   file=paste(DSE.HOME,"/data/monitoring.test.data", sep=""),
                   m=1, p=10,
                   first.input="diff(R90=B14017)", 
                   first.output="%change(CPI=P484549)", 
                   second.output="%change(GDP=I37026)"  )
  z.tf <-tframe(overriding.data$output)
  overriding.data$output <- overriding.data$output[,c(1,2,6,7)]
  tframe(overriding.data$output) <- z.tf
  ok <- test.equal(overriding.data, monitor.test.data, fuzz=fuzz.small)
  all.ok <- all.ok & ok 
  if (verbose) 
    {if (ok) cat("ok\n")
     else    cat("failed! (get.overriding.data)\n")
    }

  if (verbose) cat("combination monitor test 4 ... ")
  combined.forecast<-combine.and.forecast(monitoring.test.model,
                  list(data=data, overriding.data=overriding.data)) 
  ok <- fuzz.small > max(abs( combined.forecast$best.guess - 
                            best.guess.test))
  all.ok <- all.ok & ok 
  if (verbose) 
    {if (ok) cat("ok\n")
     else    cat("failed! (combine.and.forecast)\n")
    }

  if (synopsis) 
    {if (verbose) cat("All combination monitor tests completed")
     if (all.ok) cat(" ok\n\n")
     else    cat(", some failed!\n\n")
    }
  invisible(all.ok)
}




#   Copyright 1993, 1994, 1995, 1996  Bank of Canada.
#   Copyright 1997 (June), Paul Gilbert.
#   Copyright 1997 (Aug.), Bank of Canada.
#   Copyright 1998, Bank of Canada.

#   The user of this software has the right to use, reproduce and distribute it.
#   Bank of Canada makes no warranties with respect to the software or its 
#   fitness for any particular purpose. The software is distributed by the Bank
#   of Canada and by Paul Gilbert solely on an "as is" basis. By using the  
#   software, user agrees to accept the entire risk of using this software.

################################################################################



##############################################################################

#  These models work with TSPADIdata. See dse1d.s

##############################################################################

############################################################

#        Summary of functions in this file

############################################################

#      functions specific to .troll objects

#  to.troll				convert to a troll model
#  to.troll.TSestModel 
#  to.troll.troll 
#  to.troll.ARMA 
#  ted					edit a troll model


#   .troll specific methods for generic functions

#  print.troll 
#  is.troll 
#  test.equal.troll 			test equallity of 2 models
#  series.names.troll			extract series names
#  input.series.names.troll		extract input  series names 
#  output.series.names.troll		extract output series names 
#  check.consistent.dimensions.troll	check that model dimensions match data
#  minimum.startup.lag.troll		determine required startup lags 

#  l.troll 
#  simulate.troll  
#  monte.carlo.simulations.troll 


#     internal functions for troll class objects 

#   (These are used by the methods above, but not typically called
#         directly by the user.)

#  putTroll				put data on a file accessable by troll
#  getTroll				get data from a file written by troll
#  make.troll.simulation.database	arrange data in for a simulation
#  internal.make.trolldb		used by make.troll.simulation.database
#  internal.troll.call			write a script and call troll
#  to.internal.troll.model		convert a c("troll","TSmodel") model to 
#                                           troll's internal representation 
#  from.internal.troll.mod		reverse of to.internal.troll.model
#  internal.troll.model.shifts		gets the leads and lags in a model
#  to.internal.troll.LU.and.model	Write LU decomp. of a model in a file
#  internal.db.date.check 
#  internal.troll.simulation 


#    function for testing the programs in this file

#  troll.function.tests 


############################################################

#   Definition of class c("troll", "TSmodel") 

############################################################


###########################################################################
###########################################################################

#      functions specific to .troll objects

###########################################################################
###########################################################################
############################################################

#     methods for troll class objects ( create and modify)

############################################################



to.troll <- function(obj, ...)
 {# convert a model  to c("troll", "TSmodel")
   UseMethod("to.troll")
 }

to.troll.TSestModel <- function(obj)
 {# convert a model of class c("TSestModel") to c("troll", "TSmodel")
  # Note this sets names so model and data are consistent.
  to.troll(obj$model, names=series.names(obj$data))
 }

to.troll.troll <- function(obj) {obj}

to.troll.ARMA <- function(obj, names=NULL)
 {# convert a model of class c( "ARMA", "TSmodel") to c("troll","TSmodel")
  A <- obj$A
  B <- obj$B
  p <- dim(A)[2]
  C <- obj$C
  if (is.null(C)) m <- 0
  else            m <- dim(C)[3]
  TREND <- obj$TREND
  if (! all(A[1,,] == diag(1,p))) 
    {invA0 <- solve(A[1,  ,  ])
     for(l in 1:dim(A)[1]) A[l,  ,  ] <- invA0 %*% A[l,  ,  ]	# set A(0) = I      
     for(l in 1:dim(B)[1]) B[l,  ,  ] <- invA0 %*% B[l,  ,  ]
     if(!is.null(C)) for(l in 1:dim(C)[1]) C[l,  ,  ] <- invA0 %*% C[l,  ,  ]
     if(!is.null(TREND))
        {TREND <- invA0 %*% TREND
         stop("TREND not implemented.")
        }
    }
  if (is.null(names))
    {output.names <- output.series.names(obj) 
      input.names <-  input.series.names(obj) 
    }
  else
    {output.names <- names$output
      input.names <- names$input
    }
  l <- dim(A)[1] -1
  model<- c("ADDSYM"," ENDOGENOUS")
  model<- c(model,  paste((output.names), collapse=" "),",")
  if (1 < dim(B)[1])
    model<- c(model,  paste(paste("residual" ,1:length(output.names), sep=""), collapse=""))
  if (!is.null(obj$C))
      model <- c(model, "EXOGENOUS", paste(input.names,  collapse=" "),",")
  model <- c(model, " ;","ADDEQ BOTTOM"  ) 
  lhs <-  paste(output.names, " =  ", sep="")

  rhsL <- paste( "(-", array(rep(1:l, p*p), c(l,p,p)), ")",  sep="" )
  rhs <- aperm(array(rep(output.names, l*p), c(p,l,p)), c(2,3,1))
  rhs <- paste("- ", A[-1,,,drop=F], " * ", rhs, rhsL, sep="")
  rhs <- array(rhs, c(l,p,p))
  rhs[A[-1,,,drop=F]==0] <- ""

  b <- dim(B)[1] 
  rhsB <- paste( "(-", array(rep(0:(b-1), p*p), c(b,p,p)), ")" , sep="")
  rhsB <- paste("+ ", +B, "*", "residual",
        aperm(array(rep(1:p, b*p), c(p,b,p)), c(2,3,1)), rhsB, sep="")
  rhsB <- array(rhsB, c(b,p,p))
  rhsB[B==0] <- ""
   
  if (!is.null(C)) 
    {l <- dim(C)[1] -1
     rhsC <- paste( "(-", array(rep(0:l, p*m), c(l+1,p,m)), ")" , sep="")
     rhsC <- paste("+ ", C, "*", 
        aperm(array(rep(input.names, (l+1)*p), c(m,l+1,p)), c(2,3,1)), rhsC, sep="")
     rhsC <- array(rhsC, c(l+1,p,m))
     rhsC[C==0] <- ""
    }
  for (i in 1:p)
    {mdl <- paste(lhs[i], paste(rhs[,i,],  collapse=""))
     if (!is.null(rhsB))  mdl <- paste(mdl, paste(rhsB[,i,], collapse=""), collapse="")
     if (!is.null(C))     mdl <- paste(mdl, paste(rhsC[,i,], collapse=""), collapse="")
     mdl <- paste(mdl, ",", collapse="")
     model <- c(model,mdl)
    }
  # The residual is only necessary for models with MA components.
  # This relies on the true data being attached as dbase1 as is 
  # done in internal.troll.simulation
  if (1 < dim(B)[1])
    model <- c(model, paste("residual" ,1:length(output.names)," = ",
                       output.names, " - ","dbase1_",output.names,",",sep=""))
  model <- c(model," ;") 
  order <- list(m=m, p=p, a=dim(A)[1]-1, b=dim(B)[1]-1, c=dim(C)[1]-1)
  if (is.na(order$c)) order$c  <-NULL
  model <- list(troll.code=model, forward.looking=F, names=names,
            description="troll model converted from ARMA model.",
            order=order)
  class(model) <- c("troll", "TSmodel")
  # returning invisible(model) seems to cause lines like
  # max(abs(zz$estimates$pred - l(to.troll(zz$model), zz$data)$estimates$pred))
  #  to not print the result
  model
 }

ted <-function(obj, file=tempfile(), editor = "textedit")
{# edit a troll model definition.
 write(obj$model$troll.code, file)
 system.call(paste(editor, file), output = F)
 obj$model$troll.code<-scan(file, what="c", sep="\n")
 # this does not change forward.looking=
 class(obj$model) <- c("troll", "TSmodel")
 obj
}


###########################################################################
###########################################################################

#   .troll specific methods for generic functions

###########################################################################
###########################################################################

############################################################

#     methods for troll class objects (general)

############################################################


print.troll <- function(model, digits = 4)
  {if(!is.null(model$troll.code))  print(model$troll.code, digits = digits)
   else 
     {cat("file: ", model$path, paste(model$file.name,".mod",sep="")," forward.looking=")
      if(is.null(model$forward.looking)) cat("NULL\n")
      else cat(model$forward.looking,"\n")
     # print(from.internal.troll.mod(model$file.name, path=model$path))
     }
   invisible(model)
  }

is.troll <-function(obj)
  {r <- "troll" == class(obj)[1]
   if(is.na(r)) r <- F
   r
  }

test.equal.troll <- function(model1, model2, fuzz = 0)
  {r <- class(model1) == class(model2)
   if (is.na(r)) r <- F
   if (r) r <-  all(T==all.equal(model1$troll.code, model2$troll.code))
   r
  }

series.names.troll <- function(x)
 {list(input=input.series.names(x), output=output.series.names(x)) }

input.series.names.troll <- function(model, scratch.file.path="/tmp/", 
                                     leave.scratch.files=F) 
 {# return $names if available. Otherwise try to extract names from the file (slow).
  if(!is.null(model$input.names)) return(model$input.names)
  else
    {zzz <- from.internal.troll.mod(model$file.name, path=model$path, 
                   scratch.file.path=scratch.file.path, 
                   leave.scratch.files=leave.scratch.files)
     EX <- grep("EXOGENOUS", zzz$troll.code)
     commas <- grep(",", zzz$troll.code)
     endEX <- commas[commas > EX] [1]
     input <- paste((zzz$troll.code)[(EX+1):endEX], collapse=" ")
     brk <- substring(input, 1:nchar(input), 1:nchar(input))  # can get big
     trunc <- cumprod(!brk==",") # to remove , at end
     brk <- brk[1==trunc]
     input <- substring(input, 1, sum(trunc))
     wh <- brk == " "
     N <- length(wh)
     strt <- seq(N)[(!wh) & c(T,wh[-N])]  # start of tokens
     en   <- seq(N)[(!wh) & c(wh[-1],T)]  # end of tokens
     input <- substring(input, strt, en)
     }
  input
 }

output.series.names.troll <- function(model, 
                       scratch.file.path="/tmp/", leave.scratch.files=F) 
 {# return $names if available. Otherwise try to extract names from the file (slow).
  if(!is.null(model$output.names)) return(model$output.names)
  else
    {zzz <- from.internal.troll.mod(model$file.name, path=model$path, 
                   scratch.file.path=scratch.file.path, 
                   leave.scratch.files=leave.scratch.files)
     EN <- grep("ENDOGENOUS", zzz$troll.code)
     commas <- grep(",", zzz$troll.code)
     endEN <- commas[commas > EN] [1]
     output <- paste((zzz$troll.code)[(EN+1):endEN], collapse=" ")
     brk <- substring(output, 1:nchar(output), 1:nchar(output)) # can get big
     trunc <- cumprod(!brk==",") # to remove , at end
     brk <- brk[1==trunc]
     output <- substring(output, 1, sum(trunc))
     wh <- brk == " "
     N <- length(wh)
     strt <- seq(N)[(!wh) & c(T,wh[-N])]  # start of tokens
     en   <- seq(N)[(!wh) & c(wh[-1],T)]  # end of tokens
     output <- substring(output, strt, en)
     }
  output
 }

check.consistent.dimensions.troll <- function(model,data=NULL)
 {if ((!is.null(data)) && ("TSPADIdata"==class(data)[1]) )
    {# This works but can take some time (5 min+)
     warning("Dimension consistency check has been bypassed for troll data.")
     return(T)
    }

  nm <-series.names(model)
  p <- length(nm$output)
  m <- length(nm$input)
  # if (p!= d???) stop("Model ??? should be equal.")

  if (!is.null(data))
   {if ("TSPADIdata"==class(data)[1]) 
       {d <- list(output=getTroll(data$dbase,nm$output))
        if (!is.null(nm$input)) d$input <- getTroll(data$dbase,nm$input)
       }
    else d <- TSdata(data)
    if(p != dim(d$output)[2])
       stop("Model and data output dimensions do not correspond.")
    if(is.null(m))
      {if(!is.null(d$input))
        stop("Model and data input dimensions do not correspond.")
      }
    else
      {if(m != dim(d$input)[2])
         stop("Model and data input dimensions do not correspond.")
      }
   }
  return(T)
 }

minimum.startup.lag.troll <- function(model)
  {# This is a kludge.
   4
  } 
############################################################

#     methods for troll class objects ( to generate TSestModel)

############################################################


l.troll <- function(model, data, sampleT=NULL, predictT=sampleT,
   result=NULL, forward.values=NULL, initial.guess=NULL,
   troll.settings=list(solve.forward.algorithm="stack", canopt=NULL),
   scratch.file.path="/tmp/", leave.scratch.files=F)
 {# model should be an object of class ("troll","TSmodel").
  # sampleT indicates the number of periods to consider "in-sample" and 
  #    calculate one step ahead predictions.
  # predictT indicates the period beyond sampleT to which forecasts will be 
  #    calculated. (ie predictT-sampleT periods beyond sampleT.) Normally no 
  #    output data will be used beyond sampleT (except Troll may grab it from
  #    the database if it is supplied!). Input data must be available to 
  #    predictT.
  # sampleT and predictT indicate a number of periods. This is translated to 
  #    a date using the start date for the data.
  # If forward.values is supplied (a TSdata object)
  #   then these values are used as given (a troll NOFL solution is done. This
  #   is a forward looking soln, despite the troll terminology, but it is not
  #   necessarily "model consistent" since forward values are used as given.)
  # If forward.values is not supplied, then if the model has forward.looking=F
  #    a troll NOFL simulation is run.
  # If forward.values is not supplied, then if the model has forward.looking=T
  #   or NULL (or not set) then a forward looking solution (IDS) is done.
  #   IDS works with non-forward looking models, but gives a warning. If
  #   model$forward.looking is not set then warning message from IDS 
  #   simulate may result. 
  # If initial.guess is supplied it is used as the initial solution 
  #   of the simulation. This is especially valuable in forward looking
  #   models. If initial.guess is not supplied then a troll NOFL solution is 
  #   run first and used as the initial guess for the simulation.

  # If data, forward.values, or initial.guess are supplied they should be of 
  #   class TSdata. If they are of class c("TSPADIdata", "TSdata") then files 
  #    will be assumed to exist as indicated .

  # If model has an element $no.zeros then zeros will not be prepend or 
  #  appended to the database (to cover lags and leads). This is useful 
  #  for models where prepended 0s do not work. (eg logs are taken)

  # For more details on lags, leads and no.zeros see simulate.troll.

  shf <- start.shift(model,data,y0=y0)

  freq <- frequency(data)  
  Start <- add.date(start(data), shf$shift*shf$lags, freq)

  if ( (!is.null(forward.values)) & (!is.null(initial.guess)) )
    stop("forward.values and initial.guess cannot both be specified.")

  if ((!is.null(input.data(data))) && (!is.null(predictT)) && 
      (nrow(input.data(data)) < predictT))
     stop("input data must be at least as long as requested prediction." )

  # previously there could be problems by getting residuals here.
  # See version 21may97 or prior.
  output.names <- output.series.names(model)
  input.names  <-  input.series.names(model)
  p <- length(output.names)
  if (is.null(input.names)) m <- 0
  else m <- length(input.names)

  if(!is.null(forward.values)) solve.forward <- F
  else if (is.null(model$forward.looking)) solve.forward <- T
  else solve.forward <- model$forward.looking

  if ((!is.null(model$forward.looking))&(!is.null(forward.values)))
    if (!model$forward.looking)
      warning("model indicates forward.looking as F but forward.values has been supplied.")

  #if(!check.consistent.dimensions(model, data)) stop("dimension error.")

  rmfiles <- NULL
  if (!leave.scratch.files) on.exit(if (!is.null(rmfiles)) unlink(rmfiles))

  # The extraction of sampleT here may cause problems if class(data) == TSPADIdata
  #   but in addition to non TSPADIdata below it may be needed for 
  #   forward.values or initial.guess and to extract the sample for residuals stats
  if(is.null(sampleT))  sampleT <- periods(data)
  if(is.null(predictT))  predictT <- sampleT
  if(sampleT > predictT) 
    {warning("predictT < sampleT. Using sampleT.")
     predictT <- sampleT
    }

  rmfiles <- NULL
  if (!leave.scratch.files) on.exit(if (!is.null(rmfiles)) unlink(rmfiles))

 # databases (excluding noise) are written with make.troll.simulation.database
 # by l.troll, simulate.troll and monte.carlo.simulations.troll.
 # If any of data, forward.values, or initial.guess are already TSPADIdata
 #  (or NULL) then they are simply returned unchanged.


 sim.bases <- make.troll.simulation.database(data,  
                      forward.values, 
                      initial.guess,
                      p,m, (!shf$shift)*shf$lags, 
                      (!shf$shift)*(predictT-sampleT),
                      names)
 rmfiles<- c(rmfiles, sim.bases$rmfiles)
 sim.bases$rmfiles <-NULL

#  if (is.TSPADIdata(noise)) sim.bases <- append(sim.bases, noise)
# else{
     # the following (residual) is only necessary for models with MA terms.
     # noise names is used to write zeros to the database. It must correspond
     # to the names used in the model (otherwise data will not be found for 
     # equations with those terms, so the following works with models
     # converted with to.troll. 
#     if (is.null(noise.names))
        noise.names <-paste("residual",1:length(output.names),sep="")
     simnoisedb <- tempfile()
     rmfiles <- c(rmfiles,paste(simnoisedb,".db", sep=""))
     sim.bases <- append(sim.bases, list(internal.make.trolldb(
                  tframed(matrix(0,predictT+shf$terminal.periods,
                         length(noise.names)), 
                        list(start=Start,frequency=freq)),
                  simnoisedb,
                  prepend=matrix(0,shf$lags,length(noise.names)),  
                  append=NULL,
                  names=noise.names)))

 #   }

  z <- internal.troll.simulation(model, sim.bases, 
                     output.names, solve.forward,
                     Start,freq,sampleT, predictT=predictT,
                     dbnick.names=c("data","forward.values",
                               "initial.guess", "noise"),
                     solve.forward.algorithm=
                            troll.settings$solve.forward.algorithm,
                     canopt=troll.settings$canopt,
                     scratch.file.path=scratch.file.path, 
                     leave.scratch.files=leave.scratch.files)

  if((!is.null(result)) && (result == "pred")) return(z$pred)
  if (is.TSPADIdata(data))
    {data <- list(output=getTroll(data$dbase,output.names)) 
             # only for residual.stats
     warning("It is not certain that residual stats are calculated with correct data and subset.")
    }
  r <- residual.stats(z$pred, output.data(data), sampleT)
  if(is.null(result)) 
    {r <- (list(estimates = r, data = data, model = model, log=z$log,
                forward.values= forward.values))
     class(r) <- "TSestModel"
     return(r)
    }
  else 
    {if(result == "like") return(r$like[1]) # neg.log.like. from residual.stats
     else  return(r[[result]])
    }
 }


############################################################

#      simulation methods for troll class objects 

############################################################


simulate.troll <- function(model, data=NULL, y0=NULL, input=NULL, 
          forward.values=NULL,initial.guess=NULL,
          input0=NULL, sampleT=100,
          Start=NULL,   freq=frequency(data),
          noise=NULL, sd=1, SIGMA=NULL, seed=NULL,
          noise.model=NULL, noise.names=NULL,
          noise.baseline=0,
          troll.settings=list(solve.forward.algorithm="stack", canopt=NULL),
          scratch.file.path="/tmp/", leave.scratch.files=F) 
 {# model should be an object of class ( "troll", "TSmodel") or of
  #   class "TSestModel" with $model of class ( "troll", "TSmodel"). In 
  #   the later case SIGMA is set to model$estimates$cov and then $model used.
  # If forward.values is supplied (a TSdata object)
  #   then these values are used as given (a troll NOFL solution is done. This
  #   is a forward looking soln, despite the troll terminology, but it is not
  #   necessarily "model consistent" since forward values are used as given.)
  # If forward.values is not supplied, then if the model has forward.looking=F
  #    a troll NOFL simulation is run.
  # If forward.values is not supplied, then if the model has forward.looking=T
  #   or NULL (or not set) then a forward looking solution (IDS) is done.
  #   IDS works with non-forward looking models, but gives a warning. If
  #   model$forward.looking is not set then warning message from IDS 
  #   simulate may result. 
  # If initial.guess is supplied it is used as the initial solution 
  #   of the simulation. This is especially valuable in forward looking
  #   models. If initial.guess is not supplied then a troll NOFL solution is 
  #   run first and used as the initial guess for the simulation.
  # forward.values and initial.guess should not both be supplied. Mechanically
  #   they are both written to the data base as the (pre-simulation) values
  #   of the endogenous variables. If forward.values is supplied then a NOFL
  #   simulation is done and if initial.guess is supplied an IDS simulation
  #   is done. 
  # Warning! If the model has forward looking equations but has forward.looking=F 
  #    then the (NOFL) simulation may pick up spurious forward values, either 
  #    from forward.values or initial.values OR FROM THE DATABASE.

  # If data, forward.values, or initial.guess are supplied they should be of 
  #   class TSdata. If they are of class c("TSPADIdata", "TSdata") then files 
  #    will be assumed to exist as indicated .

  # noise.names are the troll variable names to write noise into.

  # lags is calculated to give the number of periods of zeros to prepended 
  #   as initial conditions or the number of periods that the Start must be
  #   shifted from the beginning of data if zeros are not to be prepended.
  #   lags is set to max(model$order$a,model$order$b, 
  #   model$order$c) if available, otherwise to the number of periods in y0
  #   if available, and otherwise defaults to 20.
  # shift is calculated to  determine if zeros should be prepended or Start
  #   should be shifted.  For "TSPADIdata" shift is set T, otherwise it is
  #   set to model$no.zeros if that is available, otherwise it is set to F.
  # If Start is not specified (or NULL) then if shift is T Start is set
  #     to start(data)+lags, otherwise, zeros will be prepended to
  #     data and Start will be set to start(data).
  # Noise is generated from Start-(b+1) for b+1+sampleT+terminal.periods,
  #        where b is model$order$b if it is available and lags otherwise, and
  #        terminal.periods is max(model$order$a.leads, model$order$b.leads,
  #        model$order$c.leads) or 0 if these are not available.
  # noise.baseline is added to noise. It should be either a scalar, a matrix of
  #   the same dimension as noise (or noise generated by noise.model), or a 
  #   vector of length equal to the dimension of the noise process (which will
  #   be replicated for all periods.) 

  if (is.TSestModel(model)) 
    {if(is.null(SIGMA)) SIGMA <- model$estimates$cov
     model <- model$model
    }
  if (!is.troll(model)) stop("model must be of class('troll', 'TSmodel').")

  shf <- start.shift(model,data,y0=y0)

  if(is.null(model$order)) b <- shf$lags
  else b<- model$order$b              # used for generating noise

  if (is.null(Start)) Start<- add.date(start(data), shf$shift*shf$lags,freq)

 # The extraction of sampleT here may cause problems if class(data) == TSPADIdata
 #   but in addition to non TSPADIdata below it may be needed for 
 #   forward.values or initial.guess ans noise.
  if(is.null(sampleT))  sampleT <- periods(data)

  if ( (!is.null(forward.values)) & (!is.null(initial.guess)) )
    stop("forward.values and initial.guess cannot both be specified.")

  if ((!is.null(input.data(data))) && (!is.null(sampleT)) && 
      (nrow(input.data(data)) < sampleT))
     stop("input data must be at least as long as requested prediction." )

  #  problems by getting residuals ?
  output.names <- output.series.names(model)
  input.names  <-  input.series.names(model)
  p <- length(output.names)
  if (is.null( input.names)) m <- 0
  else m <- length(input.names)

  if(!is.null(forward.values)) solve.forward <- F
  else if (is.null(model$forward.looking)) solve.forward <- T
  else solve.forward <- model$forward.looking

  if ((!is.null(model$forward.looking))&(!is.null(forward.values)))
    if (!model$forward.looking)
      warning("model indicates forward.looking as F but forward.values has been supplied.")

  #if(!check.consistent.dimensions(model, data)) stop("dimension error.")

  rmfiles <- NULL
  if (!leave.scratch.files) on.exit(if (!is.null(rmfiles)) unlink(rmfiles))

 # databases (excluding noise) are written with make.troll.simulation.database
 # by l.troll, simulate.troll and monte.carlo.simulations.troll.
 # If any of data, forward.values, or initial.guess are already TSPADIdata
 #  (or NULL) then they are simply returned unchanged.

 if ((!is.null(data))  && (!is.TSPADIdata(data))) 
    output.data(data)<- tframed(matrix(0,sampleT,p),
                                list(start=Start,frequency=freq))

 sim.bases <- make.troll.simulation.database(data,  
                      forward.values, 
                      initial.guess,
                      p,m, (!shf$shift)*shf$lags, 0, names)
 rmfiles<- c(rmfiles, sim.bases$rmfiles)
 sim.bases$rmfiles <-NULL

 if (is.TSPADIdata(noise))
    {sim.bases <- append(sim.bases, noise)
     if (!is.null(noise.baseline) && !all(noise.baseline==0))
        warning("noise.baseline is not used with noise of class TSPADIdata.")
    }
 else
    {if (is.null(noise.names))
        noise.names <-paste("residual",1:length(output.names),sep="")
     # b not b+1 is theoretical what should be in the 
     #    following, but b+1=dim(B)[1] is consistent with ARMA models and
     #    make.TSnoise chokes on b.
     if (is.null(noise.names)) 
        stop("noise process dimension indicated by noise.names is NULL.")
     else if(0==length(noise.names)) 
        stop("noise process dimension indicated by noise.names is 0.")
     noise <- make.TSnoise(sampleT+shf$terminal.periods,
                       length(noise.names),b+1, 
                       noise=noise,seed=seed, 
                       SIGMA=SIGMA, sd=sd, noise.model=noise.model, 
                       noise.baseline=noise.baseline,
                       start=Start, frequency=freq)

     simnoisedb <- tempfile()
     rmfiles <- c(rmfiles,paste(simnoisedb,".db", sep=""))
     sim.bases <- append(sim.bases, list(internal.make.trolldb(noise$w,
                              simnoisedb,
                              # rbind ing zeros just prevents warnings
                              # from checks in internal.troll.simulation
                              prepend=rbind(noise$w0, 
                                 matrix(0,shf$lags,length(noise.names))),
                              append=NULL,
                              names=noise.names)))
    }
  z <- internal.troll.simulation(model, sim.bases, 
                     output.names, solve.forward,
                     Start,freq,0, predictT=sampleT,
                     dbnick.names=c("data","forward.values",
                               "initial.guess", "noise"),
                     solve.forward.algorithm=
                            troll.settings$solve.forward.algorithm,
                     canopt=troll.settings$canopt,
                     scratch.file.path=scratch.file.path, 
                     leave.scratch.files=leave.scratch.files)

  r <- list(input=input, output=z$pred,  
            model=model, input0=input0, log=z$log,
            description = "data generated by simulate.troll", 
            noise = noise,
            noise.model=noise.model,
            forward.values=forward.values, )
  class(r) <- "TSdata"
  invisible(r)
 }

############################################################

#     methods for monte.carlo.simulations of troll class objects 

############################################################


monte.carlo.simulations.troll <- function(model, simulation.args=NULL,
     replications=10, seed=NULL, horizons=NULL, select.series=NULL, 
     scratch.file.path="/tmp/", leave.scratch.files=F,
     spm.minsize=10, show.count=F)
{# multiple simulations with noise diced into shocks.
 # model should be  of class ( "troll", "TSmodel") or of
  #   class "TSestModel" with $model of class ( "troll", "TSmodel"). In 
  #   the later case SIGMA is set to model$estimates$cov and then $model used.
#  defaults for simulation.args are:
#     simulation.args=list(data=NULL,input=NULL, y0=NULL,input0=NULL, 
#                          forward.values=NULL, initial.guess=NULL, 
#                          sampleT=100, 
#                          Start=start(simulation.args$data),   
#                          freq=frequency(simulation.args$data),
#                          noise.model=NULL, noise.names=NULL,
#                          SIGMA=NULL,
#                          troll.settings=list(
#                             solve.forward.algorithm="stack", canopt=NULL))

# data probably should not be an argument to simulate but is used to specify a database
#   to use for internal troll models (ie. models not converted from S).

  if (!is.troll(model)) stop("model must be of class('troll', 'TSmodel').")

  seed <- set.seed(seed)

  if (is.TSestModel(model)) 
    {if(is.null(simulation.args$SIGMA) & is.null(simulation.args$sd))
          simulation.args$SIGMA <- model$estimates$cov
     model <- model$model
    }

 rmfiles <- NULL
 if (!leave.scratch.files) on.exit(if (!is.null(rmfiles)) unlink(rmfiles))

 # convert once to troll internal model representation 
  if( is.null(model$troll.code)) trollname <- model
  else    
    {trollname <- to.internal.troll.model(model, "simtrollname", 
                          scratch.file.path=scratch.file.path,
                          leave.scratch.files=leave.scratch.files)
     rmfiles <- c(rmfiles,"simtrollname.mod")
    }

  shf <- start.shift(model,simulation.args$data,y0=simulation.args$y0)

  output.names <- output.series.names(model)
  input.names  <-  input.series.names(model)
  p <- length(output.names)
  m <- length( input.names)
 arglist <- simulation.args
 if (is.null(arglist$sampleT)) arglist$sampleT <-100 
 if (is.null(arglist$freq))    arglist$freq <-frequency(arglist$data)
 if (is.null(arglist$Start))   arglist$Start<-add.date(start(arglist$data),
                                     shf$shift*shf$lags, arglist$freq)


 if (is.null(arglist$troll.settings$solve.forward.algorithm))   
          arglist$troll.settings$solve.forward.algorithm <- "stack"
 if (is.null(arglist$troll.settings$canopt))   
         arglist$troll.settings$canopt <- NULL

 if (is.null(select.series)) select.series <- seq(p)
 if (is.null(horizons))      horizons <- seq(arglist$sampleT)

 # zz following could be assigned just before the for loop, but if it is large
 #   it may require resetting options("object.size"), so it's good to
 #   try before running other things that can take some time.
 zz <- array(NA, c(length(horizons), length(select.series), replications))
# set dimnames rather than $names so select.series in plot will work with names
  if (is.numeric(select.series))
        dimnames(zz) <- list(NULL,output.names[select.series],NULL)
  else  dimnames(zz) <- list(NULL,select.series,NULL)

 #  make the main troll databases once (but not noise).
 
 if ((!is.null(arglist$data))  && (!is.TSPADIdata(arglist$data))) 
    arglist$data$output <- tframed(matrix(0,arglist$sampleT,p),
                            list(start=arglist$Start,frequency=arglist$freq))

 sim.bases <- make.troll.simulation.database(arglist$data,  
                      arglist$forward.values, 
                      arglist$initial.guess,
                      p,m, (!shf$shift)*shf$lags, 0, names)
 rmfiles<- c(rmfiles, sim.bases$rmfiles)
 sim.bases$rmfiles <-NULL

 # save LU decompositon of forward looking model (from simulate wihout noise)
 #   THIS CURRENTLY relies on noise terms being found in the database to solve
 #   It would be better to write out the baseline to a database.
  if( (!is.null(model$forward.looking))
       && model$forward.looking   && is.null(model$LU.info))
    {trollname<-to.internal.troll.LU.and.model(trollname,"monte.carlo.LU.file",
                   sim.bases, arglist$Start, arglist$sampleT, arglist$freq,
                   c("data","forward.values", "initial.guess"),
                   spm.minsize=spm.minsize, 
                   solve.forward.algorithm=
                     arglist$troll.settings$solve.forward.algorithm,
                   canopt=arglist$troll.settings$canopt,
                   scratch.file.path="/tmp/", leave.scratch.files=F)
     rmfiles<- c(rmfiles, "monte.carlo.LU.file")
    }
 arglist <- append(list(trollname), arglist)

 arglist$data           <- sim.bases$data
 arglist$forward.values <- sim.bases$fv
 arglist$initial.guess  <- sim.bases$ig
 for ( i in seq(replications))
   {if (show.count) cat(".")
    zz[,,i] <-(do.call("simulate.troll",arglist)$output)[horizons, select.series]
    if (show.count) cat(i)
   }
# st <- (arglist$Start)[1]+((arglist$Start)[2]-1)/(arglist$freq)
# zz<- tframed(zz, list(start=st,end=st+(dim(zz)[1]-1)/(arglist$freq), 
#                       frequency=arglist$freq))
# zz<- tframed(zz, list(start=st, frequency=arglist$freq))
  zz<- tframed(zz, list(start=arglist$Start, frequency=arglist$freq))

  zz <-list(simulations=zz,model=model, seed=seed, version=version, 
              simulation.args=simulation.args,
              description = "data generated by monte.carlo.simulation.troll")
  class(zz) <- c("monte.carlo.simulations")
  invisible(zz)
 }

############################################################

#     internal functions for troll class objects 

############################################################

#  The following relies on Troll being able to access the same type of
#     database (ie. Fame) and thus the default padi server must be
#     a fame.server.
putTroll <- function(data, dbname, names, ...)
   {global.assign("putTroll.start.time", rbind(putTroll.start.time,proc.time()))
    z <-putpadi(data,  dbname=dbname, series.names=names, ...)
    global.assign("putTroll.end.time", rbind(putTroll.end.time,proc.time()))
z}
getTroll <- function(dbname,names,...)
   {global.assign("getTroll.start.time", rbind(getTroll.start.time,proc.time()))
    z <-getpadi(names, dbname=dbname, ...)
    global.assign("getTroll.end.time", rbind(getTroll.end.time,proc.time()))
z}


null.timing.summary <- function()
  {getTroll.start.time <- getTroll.end.time <-putTroll.start.time <- putTroll.end.time <-NULL
   invisible()
  }

timing.summary <- function()
  {cat("numeric vector, giving the user, system and elapsed  times ",
       "for  the  currently  running  S-PLUS  process, in units of ",
       "seconds.  If there have been any child  processes  spawned ",
       "during  the  current  session,  the cumulative sums of the ",
       "user and system times for them is also returned\n")
   put <- putTroll.end.time - putTroll.start.time 
   put <- apply(put, 2, sum)
   cat("put total ", put,"\n")
   get <- getTroll.end.time - getTroll.start.time
   get <- apply(get, 2, sum) 
   cat("get total ", get)
   cat("\ntime from first put to last get\n")
   all <- getTroll.end.time[dim(getTroll.end.time)[1],] -putTroll.start.time[1,]
   cat("overall    ", all,"\n")
   percent <- (get+put)*100/all
   cat("percent    ", percent,"\n")
   invisible(rbind(put,get,all,percent))
  }

putTroll <- function(data, dbname, names, ...)
   {putpadi(data,  dbname=dbname, series.names=names, ...)}
getTroll <- function(dbname,names,...)
   {getpadi(names, dbname=dbname, ...)}

make.troll.simulation.database <- function(data,
                      forward.values, initial.guess,
                      p,m, lags, leads, names,
                      scratch.file.path="/tmp/", leave.scratch.files=F)
 {# if any of data forward.values or initial.guess are already TSPADIdata
  #  (or NULL) then they are simply returned unchanged.
  simindb   <- tempfile()  #"simindb"
  simindbfv <- tempfile()  #"simindbfv"
  simindbig <- tempfile()  #"simindbig"

  if (0==leads) append  <- NULL
  else append <- matrix(0,leads,p)

  if (0==lags)
       {prepend  <- NULL
        prependm <- NULL
       }
  else
       {prepend <- matrix(0,lags,p)
        if (m!=0) prependm <- matrix(0,lags,m)
        else prependm <- NULL
       }

 rmfiles <-NULL
 if (!leave.scratch.files) on.exit(if (!is.null(rmfiles)) unlink(rmfiles))
 if ((!is.null(data))  && ("TSPADIdata"!=class(data)[1])) 
    {tr.data<- internal.make.trolldb(output.data(data),simindb, 
                                  prepend=prepend,  append=append,
                                  names=output.names)
   
     if (!is.null(input.data(data)))
     tr.data<- internal.make.trolldb(input.data(data),simindb, 
                                  prepend=prependm,
                                  names=input.names)
     rmfiles<- c(rmfiles,paste(simindb,".db", sep=""))
    }
 else tr.data <-data

 # splicing below is to make sure initial conditions are not truncated.
 # This is skipped if the values are already in a database.
 if ((!is.null(forward.values)) && ("TSPADIdata"!=class(forward.values)[1]))
   {tr.fv<- internal.make.trolldb(
                            splice(forward.values$output, output.data(data)),
                                 simindbfv,
                                  prepend=prepend,  append=append,
                                  names=output.names)
    rmfiles<- c(rmfiles,paste(simindbfv,".db", sep=""))
   }
 else tr.fv <- forward.values

 if ((!is.null(initial.guess))  && ("TSPADIdata"!=class(initial.guess)[1])) 
   {tr.ig<- internal.make.trolldb(
                            splice(initial.guess$output, output.data(data)),
                                 simindbig,
                                  prepend=prepend,  append=append,
                                  names=output.names)
    rmfiles<- c(rmfiles,paste(simindbig,".db", sep=""))
   }
 else tr.ig <- initial.guess

 on.exit()
 list(data=tr.data, fv=tr.fv, ig=tr.ig, rmfiles=rmfiles)
 }

internal.make.trolldb <- function(data, simindb, prepend=NULL, append=NULL, names=NULL)
    {# NB - data is a matrix, NOT TSdata.
     #  The returned object is class ("TSPADIdata", "TSdata")
     # prepend is a matrix (initial conditions) to prepend.
     # append is a matrix to append (eg zeros so simulation is not truncated).
     # the returned object has elements start and end which are S type dates
     #   (two element vectors) indicating the bounds of the resulting database
     #   (ie. start and end of data together with prepend and append).

     f <- frequency(data)
     if (is.null(names)) names <- dimnames(data)[[2]]

     if (!is.null(append))
       data<- tframed(rbind(data, append),list(frequency=f, start=start(data)),
                       names=names)
     if (!is.null(prepend)) 
       data <- tframed(rbind(prepend, data), list(frequency=f, end=end(data)),
                       names=names)

 # prepend above can result in negative starting dates and also 1 1 cause
 #  problems in Fame. If the start is before 1,1 then fix by shifting by 4000
     st <- start(data)
     if ((st[1]<=0) | all(st==1))
       {# A 1,1 starting point causes problems with Fame.
        #   kludge.fix is used in internal.troll.simulation to correct this shift
        kludge.fix <-c(4000,0)
        st <- st + kludge.fix
        data<- tframed(data, list(frequency=f, start=kludge.fix+start(data)),
                      names=names) 
       }
     else kludge.fix <- NULL
     id <- putTroll(data, simindb, names)
     id <-list(dbase=simindb, start=start(data), end=end(data), freq=f,kludge.fix=kludge.fix)
     class(id) <- c("TSPADIdata", "TSdata")
     id
}

internal.troll.call <- function(script)
 {scriptf <- tempfile()
  write(script,paste(scriptf,".inp", sep=""))
  # this is done with 2 file so QUITerror can be used.
  tmpf<- tempfile()
  write(c("do smp_ok = true;",     #  prevent log messages
           paste("INPUT QUITerror", scriptf, ";"),
          "trexit;"), 
     file=paste(tmpf,".inp", sep=""))
  on.exit(unlink(c(paste(scriptf,".inp", sep=""),paste(tmpf,".inp", sep=""))))
  system.call(paste("troll ", tmpf," >/dev/null"))
  log <- scan("troll.log", what="c", sep="\n")
  if((!is.na(charmatch("ERROR", log)))|(!is.na(charmatch("FATAL ERROR", log))))
    {if (length(log)<200)stop(paste("Troll did not complete. Log follows:\n",
                paste(log, collapse="\n"), collapse="\n"))
     else 
       {global.assign(".troll.log",log)
        stop("Troll did not complete. See .troll.log for details.")
    }  }
  if(!is.na(charmatch("WARNING", log)))
    warning(paste("There were warnings in Troll.",
            paste(log[!is.na(charmatch("WARNING", log))], collapse="\n"),
           "See $log in the result for more details.",  collapse="\n") )
  log
}

to.internal.troll.model <- function(model, file.name, 
                              scratch.file.path="/tmp/", leave.scratch.files=F)
 {# convert a model of class c("troll","TSmodel") to troll's internal
  #     representation (in a file file.name".mod" for troll to use).
  if(is.null(model$troll.code)) stop("model list does not have troll.code.")
  log <-internal.troll.call(c(
    "usemod;", 
    "modedit;", 
     model$troll.code,
     paste("filemod ",file.name,";"), #write internal format file(.mod is added)
    ))

  model$troll.code <- NULL
  model$file.name <-file.name
  model$log   <- log
  model$names <- series.names(model)
  model$description <-
      paste("Coverted to internal troll representation from\n", 
             model$description)
  class(model) <- c("troll", "TSmodel")
  invisible(model)
 }

from.internal.troll.mod <- function(model.file, path=NULL, scratch.file.path="/tmp/", leave.scratch.files=F)
 {# omit .mod on file name
  # path should NOT include final /
  if(0!=length(grep("/",model.file)))
     stop("File name cannot include path. Use path= in call.")
  rmfiles <-NULL
  if (!leave.scratch.files) on.exit(if (!is.null(rmfiles)) unlink(rmfiles))

  # unfortunately it seems the following cannot be done to path.
  asciiname <- "zottempmod" # this will get .inp appended
  rmfiles   <- c(rmfiles, "zottempmod.inp")
  if (is.null(path))
     script <- paste("usemod ", model.file,";")
  else 
     script <- c(paste("access mod id ",path,"/", model.file, sep=""),
                 paste("usemod mod_", model.file," ;", sep="")) 
  script <- c(script,
     paste("sourcemod to input ",asciiname,";")) # write ascii file (.inp added)
  log <- internal.troll.call(script)

  model <- list(troll.code=scan(paste(asciiname,".inp", sep=""), what="c", sep="\n"), 
                log=log,
                forward.looking=NULL,
                description=paste("From troll file", model.file))
  warning("model$forward.looking not determined and has not been set.")
  class(model) <- c( "troll", "TSmodel")
  invisible(model)
 }

internal.troll.model.shifts <- function(model, scratch.file.path="/tmp/", leave.scratch.files=F)
 {# this gets the leads and lags in a model
  if(!is.troll(model)) stop("model is not of class troll.")
  if (!leave.scratch.files) on.exit(if (!is.null(rmfiles)) unlink(rmfiles))

  # write the model to file if necessary
  if(is.null(model$troll.code))
    {file.mod <- model$file.name
     if(! ("character"==mode(file.mod))) 
        stop("model does not seem to be in the correct form. ")
     rmfiles <- NULL
    }
  else 
    {file.mod <-to.internal.troll.model(model, "simtrollname", 
                     scratch.file.path=scratch.file.path, 
                     leave.scratch.files=leave.scratch.files)$file.name
     rmfiles <- paste(file.mod,".mod", sep="")
    }

  endofile <- tempfile()
  exofile <- tempfile()
  rmfiles <-c(rmfiles, endofile,exofile)

  log <- internal.troll.call(c(
     paste("usemod ", file.mod,";"), 
     paste('do xwrite(convert(symtab(modsym("ENDOGENOUS"))),"' ,
             endofile, '");' ,sep=""),
     paste('do xwrite(convert(symtab(modsym("EXOGENOUS"))),"' ,
             exofile, '");' ,sep=""),
    ))

  shifts.in <- scan(exofile)
  shifts.in <- matrix(shifts.in,length(shifts.in)%/%2, 2)
  # column 1 is lags and column 2 for each variable
  shifts.out <- scan(endofile)
  shifts.out <- matrix(shifts.out,length(shifts.out)%/%2, 2)

  # this does not distinguish MA terms (b is set to 1 ).
  obj <- list(input=shifts.in, output=shifts.out,
              order=list(m=dim(shifts.in)[1], p=dim(shifts.out)[1],
                  a=abs(min(shifts.out)),  a.leads=max(shifts.out),
                  c=abs(min(shifts.in)),   c.leads=max(shifts.in),
                  b=1, b.leads=NULL),
               log=log)
  invisible(obj)
 }


to.internal.troll.LU.and.model <- function(model, LU.file.name, fmdbs, Start,
            predictT, freq, dbnick.names, model.file.name=NULL, spm.minsize=2, 
            solve.forward.algorithm="stack", canopt=NULL,
            scratch.file.path="/tmp/", leave.scratch.files=F)
 {# Write the LU decomposition of a model in a file LU.file.name.mod. If
  # the model is not already in troll's internal representation then
  # model.file.name must be supplied and the internal representation is
  # written to that file.
  # This function should not really need fmdbs, but the (current)
  # troll code has to run a simulation (requiring data) before the LU
  # decompostion is calculated. Thus this code resembles 
  # internal.troll.simulation in many respects. (See it for additional
  #     documentation of arguments.)
  if(!is.troll(model)) stop("model is not of class troll.")
  if(!is.null(model$troll.code)) 
     {if(is.null(model.file.name))
        stop("model.file.name must be supplied for models not already in troll internal representation.")
      model <- to.internal.troll.model(model, model.file.name, 
                       scratch.file.path=scratch.file.path, 
                       leave.scratch.files=leave.scratch.files)
     }

############## following code could be removed if simulate was ############
#                not required for getting LU decompostition   

  if (0==length(fmdbs))
     stop("A database must be supplied to to.internal.troll.LU.and.model.")

  if ((!is.null(fmdbs[[1]]$kludge.fix)) & all(Start==c(1,1)))
     Start<- Start+fmdbs[[1]]$kludge.fix

  internal.db.date.check(model,fmdbs, Start, freq, predictT, dbnick.names)

##############   above code could be removed if simulate was   ############
#                not required for getting LU decompostition   

# if (!leave.scratch.files) on.exit(if (!is.null(rmfiles)) unlink(rmfiles))
# out.db <- paste(scratch.file.path,"zotStrollsimoutdb.db", sep="")
  LU.info <-list(file.name=LU.file.name, simstack=predictT,
                  spm.minsize=spm.minsize, do.stack=1)
  sc <- paste(Start[1],c("A","Q","M","W")[c(1,4,12,52)==freq], Start[2], sep="")
# Send <- add.date(Start,LU.info$simstack-1,freq)
# stp <- paste(Send[1],c("A","Q","M","W")[c(1,4,12,52)==freq], Send[2], sep="")

  script<- paste("usemod ",model$file.name,";")
  if (!is.null(canopt)) script <-c(script,  paste(canopt,";"))

  for ( i in seq(length(fmdbs)))
     if (!is.null(fmdbs[[i]]$dbase)) script <-c(script,
        paste("access dbase",i," type fame id ",fmdbs[[i]]$dbase,
                  " mode r;", sep=""),
        paste("search first dbase",i," ;", sep="")) 
  script <-c(script,
           paste("dosave spm_minsize = ",LU.info$spm.minsize, ";"),
                 "dosave save_fct=true;",
           paste("simulate ", solve.forward.algorithm, LU.info$simstack,";"),
           paste("simstart ",      sc,  ";"), 
           paste("dostack ",LU.info$do.stack, ";"),
    #        paste("access  dbout type fame id ",out.db," mode wu;"),
    #              "filesim dbout: endogenous;",
                   "delaccess all;",
            paste("dofile ",LU.file.name,"= get_fct();") #write (.dat is added)
           )

# rmfiles <- c(rmfiles,out.db)
  LU.info$log <- internal.troll.call(script)
  model$LU.info <-LU.info
  invisible(model)
 }

internal.db.date.check <- function(model,fmdbs, Start, freq, predictT, dbnick.names)
 {# required database start and end dates:
  lags <- max(model$order$a,model$order$b,model$order$c)
  if (is.na(lags)) 
     {warning("model lag values are not available.")
      db.start <- Start # this won't catch many problems
     }
  else db.start <- add.date(Start,-lags,freq)
  db.end <- add.date(Start, predictT-1,freq)
  leads <- max(model$order$a.leads,model$order$b.leads,model$order$c.leads)
  if (!is.na(leads)) db.end <- add.date(db.end,leads,freq)
  freq.time <- c(freq,1)
  #The following is overly stringent. eg - noise does not
  # need as many lags in some models. 
  for ( i in seq(length(fmdbs)))  
    {if(( !is.null(fmdbs[[i]]$freq)) && (fmdbs[[i]]$freq !=freq))
        stop(paste("Specified simulation frequency (", freq,
             ") does not agree with ", dbnick.names[i]," database (", 
             fmdbs[[i]]$freq,").", sep=""))
     if( !is.null(fmdbs[[i]]$start))
        if ( db.start %*% freq.time < fmdbs[[i]]$start %*% freq.time)
           warning(paste( dbnick.names[i]," database starts at", 
              (fmdbs[[i]]$start)[1],(fmdbs[[i]]$start)[2],
             ". Specified start (", Start[1], Start[2],
             ") and model lags (", lags,
            ") imply the database should start by", db.start[1],db.start[2]))
 
     if( !is.null(fmdbs[[i]]$end))
        if ( db.end %*% freq.time > fmdbs[[i]]$end %*% freq.time)
           warning(paste(dbnick.names[i]," database ends at", 
             (fmdbs[[i]]$end)[1],(fmdbs[[i]]$end)[2],
             ". Specified start (", Start[1],Start[2],
             "), model leads (", leads,
             ") and simulation periods (",predictT,
             ") imply the database must not end before ",
             db.end[1], db.end[2]))
    }
  invisible()
  }

internal.troll.simulation <- function(model,fmdbs, output.names, solve.forward,
       Start, freq, sampleT, predictT=sampleT, 
       dbnick.names=seq(length(fmdbs)),
       solve.forward.algorithm="stack", canopt=NULL,
       scratch.file.path="/tmp/", leave.scratch.files=F)
 {# a simulation is done with simper 1 for sampleT periods and then if
  #    predictT > sampleT another simulation is done for (predictT-sampleT) 
  #    more periods, without setting simper.
  #    simper 1 uses data instead of previous forecasts for the lagged values
  #    of variables.
  # solve.forward [F=> troll NOFL, T=> IDS] simulation.
  # fmdbs should be a list of objects, each as returned by 
  #   internal.make.trolldb, indicating databases to open.
  # It must contain at least one database.
  # They are opened read only and sequencially and each is indicated as 
  #   "search first", so the last one will be accessed first in finding data.
  # The reason for opening multiple databases is to avoid copying a large
  #    database when only a few variables (eg. shocks) are being changed. 
  #    The name of the database is actually the element $dbname in each of
  #    the sublists in fmdbs. If either fmdbs[[i]] or fmdbs[[i]]$dbname is
  #    NULL then there is no attempt to open a database for that element,
  #    so it is not necessary to check for NULLs in the list.
  # Start(the S form start date) and freq are each checked against non NULL 
  #    database in fmdbs.
  # Simulated values of endogenous variables are written to a 
  # scratch database which is removed and the data returned.
  # dbnick.names is used for reporting errors and warnings.

  if(!is.troll(model)) stop("model is not of class troll.")

  if (0==length(fmdbs))
     stop("A database must be supplied to internal.troll.simulation.")

  if ((!is.null(fmdbs[[1]]$kludge.fix)) & all(Start==c(1,1)))
     Start<- Start+fmdbs[[1]]$kludge.fix

  internal.db.date.check(model,fmdbs, Start, freq, predictT, dbnick.names)

  # write the model to file if necessary
  if(is.null(model$troll.code))
    {file.mod <- model$file.name
     if(! ("character"==mode(file.mod))) 
        stop("model does not seem to be in the correct form. ")
     rmfiles <- NULL
    }
  else 
    {file.mod <-to.internal.troll.model(model, "simtrollname", 
                     scratch.file.path=scratch.file.path,
                     leave.scratch.files=leave.scratch.files)$file.name
     rmfiles <- paste(file.mod,".mod", sep="")
    }

  if (!leave.scratch.files) on.exit(if (!is.null(rmfiles)) unlink(rmfiles))
  out.db <- paste(scratch.file.path,"zotStrollsimoutdb.db", sep="")

  sc <- paste(Start[1],c("A","Q","M","W")[c(1,4,12,52)==freq], Start[2], sep="")
  Send <- add.date(Start,sampleT-1,freq)
  stp <- paste(Send[1],c("A","Q","M","W")[c(1,4,12,52)==freq], Send[2], sep="")

  script<- paste("usemod ",file.mod,";")
  if (!is.null(canopt)) script <-c(script,  paste(canopt,";"))

  for ( i in seq(length(fmdbs)))
     if (!is.null(fmdbs[[i]]$dbase)) script <-c(script,
          paste("access dbase",i," type fame id ",
                      fmdbs[[i]]$dbase," mode r;",sep=""),
          paste("search first dbase",i," ;", sep="")) 
  if (solve.forward & !is.null(model$LU.info))
       script <-c(script,
                "dosave use_fct=true;",
          paste("dosave set_fct(",model$LU.info$file.name, ");"),
          paste("dosave spm_minsize = ",model$LU.info$spm.minsize, ";"))
        
  if (sampleT >0 )
    {if (solve.forward) 
       {warning("simper not supported with stack in troll.")
        script <-c(script,
               paste("simulate ", solve.forward.algorithm, sampleT,";"),
             #  this works with solve.forward.algorithm= stack or oldstack 
             #  oldstack may be faster on machines with less memory
             # "simper 1;", 
               paste("simstart ",      sc,  ";"), 
              "dostack 1;")
       }
     else               
       script <-c(script,
              "simulate nofl;",
              "simper 1;", 
               paste("simstart ",sc,";"),
               paste("dotil    ",stp, ";"))
     script <-c(script,
             paste("access  dbout type fame id ",out.db," mode wu;"),
            "filesim dbout: endogenous;")
    }
  if ((!is.null(predictT)) && (sampleT < predictT))
    {Start <- add.date(Send,1,freq)
     sc <- paste(Start[1],c("A","Q","M","W")[c(1,4,12,52)==freq], Start[2], sep="")
     Send <- add.date(Start,predictT-sampleT-1,freq)
     stp <- paste(Send[1],c("A","Q","M","W")[c(1,4,12,52)==freq], Send[2], sep="")
     if (solve.forward) 
        script <-c(script,
             paste("simulate ", solve.forward.algorithm, predictT-sampleT,";"),
             paste("simstart ",      sc,  ";"), 
                   "dostack 1;")
     else               
        script <-c(script,
              "simulate nofl;",
               paste("simstart ",sc,";"),
               paste("dotil    ",stp, ";"))
     if(sampleT == 0)
       script <-c(script,
               paste("access  dbout type fame id ",out.db," mode wu;"))
     script <-c(script,
              "filesim dbout: endogenous;")
    }
  rmfiles <- c(rmfiles,out.db)
  log <- internal.troll.call(script)
  pred <- getTroll(out.db,output.names)

  if (!is.null(fmdbs[[1]]$kludge.fix)) 
      pred<- tframed(pred, list(frequency=fmdbs[[1]]$freq, 
                    start=start(pred)-fmdbs[[1]]$kludge.fix))

  # if the troll simulation explodes then troll seems to just stop without any
  #  warning message. In fact, it seems to be a problem of NAs or infs not 
  #   passed correctly through Fame. In any case pred is not complete.
  if(dim(pred)[1] < predictT ) 
    {if (length(log)<200)stop(paste("troll simulation did not complete to the end date. Log follows:\n",
                paste(log, collapse="\n"), collapse="\n"))
     else 
       {global.assign(".troll.log",log)
        stop("troll simulation did not complete to the end date. See .troll.log for details.")
    }  }
  if(dim(pred)[1] > predictT ) 
       warning("Simulation is longer than expected.")
  list(pred=pred,log=log)
 }


############################################################

#     testing for troll class objects and methods

############################################################

#  simple.test.model <- TSmodel(list(A=array(c(1,.5),c(2,1,1)), B=1))
#  simple.test.data  <- list(output=matrix(10,10,1))
#  class(simple.test.data) <- "TSdata"
#  l(simple.test.model,simple.test.data)$estimates$pred
#  l(to.troll(simple.test.model),simple.test.data)$estimates$pred

troll.function.tests <- function( verbose=T, synopsis=T, fuzz.small=1e-6, skip.to=0)
{# A short set of tests for troll objects
  max.error <- NA
  if (!is.TSdata(example.BOC.93.4.data.all))
     stop("Test data not found. Testing stopped.\n")

  if (synopsis & !verbose) cat("All troll tests ...")
  all.ok <-T

 if(skip.to <= 0)
  {
  if (verbose) cat("troll test 0 ... ")
  # this test uses kludge.fix
  simple.test.model <- TSmodel(list(A=array(c(1,.5),c(2,1,1)), B=1))
  simple.test.data  <- list(output=matrix(10,10,1))
  class(simple.test.data) <- "TSdata"
  z  <- l(simple.test.model,simple.test.data)$estimates$pred
  zz <- l(to.troll(simple.test.model),simple.test.data)$estimates$pred
  error <- max(abs(z - zz))
  ok <- fuzz.small > error
  if (!ok) {if (is.na(max.error)) max.error <- error
            else max.error <- max(error, max.error)}
  all.ok <- all.ok & ok
  if (verbose) 
    {if (ok) cat("ok\n")
     else  cat("failed! (error magnitude= ", error,")\n")
    }
  }

 if(T)   #skip.to <= 1)
  {
  if (verbose) cat("troll test 1 ... ")
  VARmodel <- est.VARX.ar(example.BOC.93.4.data.all, re.add.means=F)
  trollmodel  <- l(to.troll(VARmodel), VARmodel$data)
  error <- max(abs(trollmodel$estimates$pred - VARmodel$estimates$pred))
  ok <- fuzz.small > error
  if (!ok) {if (is.na(max.error)) max.error <- error
            else max.error <- max(error, max.error)}
  all.ok <- all.ok & ok
  if (verbose) 
    {if (ok) cat("ok\n")
     else  cat("failed! (error magnitude= ", error,")\n")
    }
  }

 if(skip.to <= 1)
  {
  if (verbose) cat("troll test 2 ... ")
  ok <- is.troll(trollmodel$model) & !is.troll(VARmodel$model) 
  all.ok <- all.ok & ok
  if (verbose) 
    {if (ok) cat("ok\n")
     else  cat("failed!\n")
    }

  if (verbose) cat("troll test 3 ... ")
  trollname <- to.internal.troll.model(to.troll(VARmodel), "simtrollname")
  trollmodel  <- l(trollname, VARmodel$data)
  error <- max(abs(trollmodel$estimates$pred - VARmodel$estimates$pred))
  ok <- fuzz.small > error
  if (!ok) {if (is.na(max.error)) max.error <- error
            else max.error <- max(error, max.error)}
  all.ok <- all.ok & ok
  if (verbose) 
    {if (ok) cat("ok\n")
     else  cat("failed! (error magnitude= ", error,")\n")
    }
  }

 if(skip.to <= 1)
  {
  if (verbose) cat("troll test 4 ... ")
  ok <- is.troll(trollmodel$model) & !is.troll(VARmodel$model) 
  all.ok <- all.ok & ok
  if (verbose) 
    {if (ok) cat("ok\n")
     else  cat("failed!\n")
    }
  }

 if(skip.to <= 1)
  {
  if (verbose) cat("troll test 5 ... ")
  names <- series.names(trollname)
  #  the following produces a warning because model and data do not have the same names
  # ok <- all(c(input.names ==series.names(trollmodel)$input, 
  #            output.names==series.names(trollmodel)$output))
  ok <- all(c(input.names ==series.names(trollmodel$model)$input, 
              output.names==series.names(trollmodel$model)$output))
  all.ok <- all.ok & ok
  if (verbose) 
    {if (ok) cat("ok\n")
     else  cat("failed!\n")
    }
  }

 if(skip.to <= 2)
  {
  if (verbose) cat("troll test 6 ... ")
  VARmodelB <- VARmodel
  B <- VARmodel$estimates$cov
  VARmodelB$B <- array(B, c(1,dim(B)))  # has B != I
  VARmodelB <- set.parameters(VARmodelB)
  VARmodelB <- l(VARmodelB,VARmodel$data)
  trollmodelB  <- l(to.troll(VARmodelB), VARmodel$data)
  error <- max(abs(trollmodelB$estimates$pred - VARmodelB$estimates$pred))
  ok <- fuzz.small > error
  if (!ok) {if (is.na(max.error)) max.error <- error
            else max.error <- max(error, max.error)}
  all.ok <- all.ok & ok
  if (verbose) 
    {if (ok) cat("ok\n")
     else    cat("failed! (error magnitude= ", error,")\n")
    }
  }

 if(skip.to <= 3)
  {
  if (verbose) cat("troll test 7 ... ")
  zz<- VARmodel
  zz$data$output <- window(output.data(zz), end=c(1969,6))
  z <- to.troll(zz)
  z  <- l(z,        zz$data, sampleT=100, predictT=363)  #troll
  zz <- l(zz$model, zz$data, sampleT=100, predictT=363)  # ARMA 
  error <- max(abs(zz$estimates$pred - z$estimates$pred))
  ok <- fuzz.small > error
  if (!ok) {if (is.na(max.error)) max.error <- error
            else max.error <- max(error, max.error)}
  all.ok <- all.ok & ok 
  if (verbose) 
    {if (ok) cat("ok\n")
     else    cat("failed! (error magnitude= ", error,")\n")
    }
  }

 if(skip.to <= 4)
  {
  if (verbose) cat("troll test 8 ... ")
  zz<- VARmodelB
  zz$data$output <- window(output.data(zz), end=c(1969,6))
  z <- to.troll(zz)
  z  <- l(z,        zz$data, sampleT=100, predictT=363)  #troll
  zz <- l(zz$model, zz$data, sampleT=100, predictT=363)  # ARMA is exploding with VARB ???
  error <-  max(abs(zz$estimates$pred - z$estimates$pred))
  ok <- ok & (fuzz.small > error)
  if (!ok) {if (is.na(max.error)) max.error <- error
            else max.error <- max(error, max.error)}
  all.ok <- all.ok & ok 
  if (verbose) 
    {if (ok) cat("ok\n")
     else    cat("failed! (error magnitude= ", error,")\n")
    }
  }

 if(skip.to <= 5)
  {
  if (verbose) cat("troll test 9 ... ")
  # forecast uses  l(model, data, sampleT=sampleT, predictT=predictT)
  zz<- VARmodel
  zz$data$output <- window(output.data(zz), end=c(1969,6))
  z <- forecast(to.troll(VARmodel),zz$data, percent=c(90,100,110))
  zz <- forecast(zz, percent=c(90,100,110))
  error <- max(abs(c((z$forecast)[[1]]-(zz$forecast)[[1]],
          (z$forecast)[[2]]-(zz$forecast)[[2]],
          (z$forecast)[[3]]-(zz$forecast)[[3]])))
  ok <- fuzz.small > error
  if (!ok) {if (is.na(max.error)) max.error <- error
            else max.error <- max(error, max.error)}
  all.ok <- all.ok & ok 
  if (verbose) 
    {if (ok) cat("ok\n")
     else    cat("failed! (error magnitude= ", error,")\n")
    }
  }

 if(skip.to <= 5)
  {
  if (verbose) cat("troll test 10... ")
  zz<- VARmodelB
  zz$data$output <- window(output.data(zz), end=c(1969,6))
  z <- forecast(to.troll(VARmodelB), zz$data, percent=c(90,100,110))
  zz <- forecast(zz, percent=c(90,100,110))
  error <- max(abs(c((z$forecast)[[1]]-(zz$forecast)[[1]],
                     (z$forecast)[[2]]-(zz$forecast)[[2]], 
                     (z$forecast)[[3]]-(zz$forecast)[[3]])))
  ok <- fuzz.small > error
  if (!ok) {if (is.na(max.error)) max.error <- error
            else max.error <- max(error, max.error)}
  all.ok <- all.ok & ok 
  if (verbose) 
    {if (ok) cat("ok\n")
     else    cat("failed! (error magnitude= ", error,")\n")
    }
  }

 if(skip.to <= 6)
  {
  if (verbose) cat("troll test 11... ")
  z  <- feather.forecasts(to.troll(VARmodel),VARmodel$data, from.periods=c(250,300))
  zz <- feather.forecasts(VARmodel,   from.periods=c(250,300))
  error <- max(abs(c(
        (z$feather.forecasts)[[1]][-(1:249),] - 
                      (zz$feather.forecasts)[[1]][-(1:249),],
        (z$feather.forecasts)[[2]][-(1:299),] - 
                      (zz$feather.forecasts)[[2]][-(1:299),])))
  ok <- fuzz.small > error
  if (!ok) {if (is.na(max.error)) max.error <- error
            else max.error <- max(error, max.error)}
  all.ok <- all.ok & ok 
  if (verbose) 
    {if (ok) cat("ok\n")
     else    cat("failed! (error magnitude= ", error,")\n")
    }
  }

 if(skip.to <= 6)
  {
  if (verbose) cat("troll test 12... ")
  z <- to.internal.troll.model(to.troll(VARmodel), "simtrollname")
  z  <- feather.forecasts(z, VARmodel$data, from.periods=c(250,300))
  zz <- feather.forecasts(VARmodel,         from.periods=c(250,300))
  error <- max(abs(c(
       (z$feather.forecasts)[[1]][-(1:249),] -
                     (zz$feather.forecasts)[[1]][-(1:249),],
       (z$feather.forecasts)[[2]][-(1:299),] -
                     (zz$feather.forecasts)[[2]][-(1:299),])))
  ok <- fuzz.small > error
  if (!ok) {if (is.na(max.error)) max.error <- error
            else max.error <- max(error, max.error)}
  all.ok <- all.ok & ok 
  if (verbose) 
    {if (ok) cat("ok\n")
     else    cat("failed! (error magnitude= ", error,")\n")
    }
  }

 if(skip.to <= 7)
  {
  if (verbose) cat("troll test 13... ")
  trollmodel <- to.troll(VARmodel)
  zzz  <- simulate(trollmodel, data=VARmodel$data, input=input.data(VARmodel),
                   seed=c(45,21,45,24,63,0,14,11,36,39,0,1))
  zz  <- simulate(VARmodel$model, input=input.data(VARmodel), 
                  Start=start(zzz$output), freq=frequency(zzz$output),
                  seed=get.seed(zzz))
  error <- max(abs(zz$output - zzz$output))
  ok <- fuzz.small > error
  if (!ok) {if (is.na(max.error)) max.error <- error
            else max.error <- max(error, max.error)}
  all.ok <- all.ok & ok 
  if (verbose) 
    {if (ok) cat("ok\n")
     else    cat("failed! (error magnitude= ", error,")\n")
    }
  }

 if(skip.to <= 7)
  {
  if (verbose) cat("troll test 14... ")
  trollmodel <- to.troll(VARmodel)
  zzz  <- simulate(trollmodel, data=VARmodel$data, input=input.data(VARmodel),
                   SIGMA=VARmodel$estimates$cov,
                   seed=c(45,21,45,24,63,0,14,11,36,39,0,1))
  zz  <- simulate(VARmodel, input=input.data(VARmodel),
                  Start=start(zzz$output), freq=frequency(zzz$output),
                  seed=get.seed(zzz))
  error <- max(abs(zz$output - zzz$output))
  ok <- fuzz.small > error
  if (!ok) {if (is.na(max.error)) max.error <- error
            else max.error <- max(error, max.error)}
  all.ok <- all.ok & ok 
  if (verbose) 
    {if (ok) cat("ok\n")
     else    cat("failed! (error magnitude= ", error,")\n")
    }
  }

 if(skip.to <= 7)
  {
#  if (verbose) cat("troll test 15... ")
#browser()
#  zzz <- simulate(trollmodel,  input=input.data(VARmodel), 
#                  forward.values=VARmodel$data)
#  error <- max(abs(10.0 - sum(zzz$output)))
#  ok <- fuzz.small > error
#  if (!ok) {if (is.na(max.error)) max.error <- error
#            else max.error <- max(error, max.error)}
#  all.ok <- all.ok & ok 
#  if (verbose) 
#    {if (ok) cat("ok\n")
#     else    cat("failed! (error magnitude= ", error,")\n")
#    }
  }

 if(skip.to <= 7)
  {
  if (verbose) cat("troll test 16... ")
  zzz <- simulate(trollmodel, data=VARmodel$data, input=input.data(VARmodel),
                  SIGMA=VARmodel$estimates$cov,
                  initial.guess=VARmodel$data, seed=get.seed(zz))
  error <- max(abs(zz$output - zzz$output))
  ok <- fuzz.small > error
  if (!ok) {if (is.na(max.error)) max.error <- error
            else max.error <- max(error, max.error)}
  all.ok <- all.ok & ok 
  if (verbose) 
    {if (ok) cat("ok\n")
     else    cat("failed! (error magnitude= ", error,")\n")
    }
  }

 if(skip.to <= 8)
  {
  if (verbose) cat("troll test 17... ")
  zzz <- monte.carlo.simulations(trollmodel,  
              simulation.args=list(SIGMA=VARmodel$estimates$cov,
                   data=VARmodel$data, input=input.data(VARmodel),
                   y0=tframed(matrix(0,20,3), list(frequency=12,end=c(1961,2))),
                   input0=matrix(0,20,1)),
              seed=get.seed(zz),replications=3)
  zz <- monte.carlo.simulations(VARmodel,
              simulation.args=list(input=input.data(VARmodel),
                   y0=tframed(matrix(0,20,3), list(frequency=12,end=c(1961,2))),
                   # Start and freq should be picked up from input 
                   #Start=start(VARmodel$data), freq=frequency(VARmodel$data),
                   input0=matrix(0,20,1), sampleT=100),
              seed=get.seed(zzz),replications=3, Spawn=F)
  error <- max(abs(zz$simulations - zzz$simulations ))
  ok <- fuzz.small > error
  if (!ok) {if (is.na(max.error)) max.error <- error
            else max.error <- max(error, max.error)}
  all.ok <- all.ok & ok 
  if (verbose) 
    {if (ok) cat("ok\n")
     else    cat("failed! (error magnitude= ", error,")\n")
    }
  }

 if(skip.to <= 9)
  {
  if (verbose) cat("troll test 18... ")
#  zzzz <- l(zl, c(),  80, forward.values=zl$data)
  warning("need to test forward looking models.")
  ok <- T
  all.ok <- all.ok & ok 
  if (verbose) 
    {if (ok) cat("ok\n")
     else    cat("failed forward looking models.\n")
    }
  }

  if (synopsis) 
    {if (verbose) cat("All troll tests completed")
     if (all.ok) cat(" ok\n\n")
     else    cat(", some failed! (max. error magnitude= ", max.error,")\n")
    }
  invisible(all.ok)
}

