# PeruAPIs - Access Peruvian Data via Public APIs and Curated Datasets
# Version 0.1.0
# Copyright (c) 2025 Renzo Caceres Rossi
# Licensed under the MIT License.
# See the LICENSE file in the root directory for full license text.

# get_peru_gdp


library(testthat)

test_that("get_peru_gdp() returns a tibble with the correct structure", {
  skip_on_cran()
  result <- get_peru_gdp()

  # Basic structure
  expect_s3_class(result, "tbl_df")
  expect_named(result, c("indicator", "country", "year", "value", "value_label"))
  expect_equal(ncol(result), 5)
  expect_equal(nrow(result), 13) # 2010–2022
})

test_that("get_peru_gdp() returns correct data types", {
  skip_on_cran()
  result <- get_peru_gdp()

  expect_type(result$indicator, "character")
  expect_type(result$country, "character")
  expect_type(result$year, "integer")
  expect_type(result$value, "double")
  expect_type(result$value_label, "character")
})

test_that("get_peru_gdp() returns only Peru GDP indicator", {
  skip_on_cran()
  result <- get_peru_gdp()

  expect_true(all(result$indicator == "GDP (current US$)"))
  expect_true(all(result$country == "Peru"))
})

test_that("get_peru_gdp() covers years 2010 to 2022", {
  skip_on_cran()
  result <- get_peru_gdp()

  expect_true(all(result$year %in% 2010:2022))
  expect_equal(length(unique(result$year)), 13)
})

test_that("get_peru_gdp() returns values without missing years", {
  skip_on_cran()
  result <- get_peru_gdp()

  expect_false(any(is.na(result$year)))
  expect_true(all(!is.na(result$value) | is.na(result$value))) # allow NA values
})

test_that("get_peru_gdp() formats value_label correctly", {
  skip_on_cran()
  result <- get_peru_gdp()

  expect_true(all(grepl(",", result$value_label)))
})

test_that("get_peru_gdp() returns years sorted in descending order", {
  skip_on_cran()
  result <- get_peru_gdp()

  expect_equal(result$year, sort(result$year, decreasing = TRUE))
})

test_that("get_peru_gdp() has consistent indicator and country", {
  skip_on_cran()
  result <- get_peru_gdp()

  expect_equal(length(unique(result$indicator)), 1)
  expect_equal(length(unique(result$country)), 1)
})
