// Title: The CH statistic for the Laplace distribution
// Ref. (book or article): Goodness of Fit via Non-Parametric Likelihood Ratios
// Gerda Claeskens and Nils Lid Hjort
// Scandinavian Journal of Statistics, Vol. 31, No. 4 (Dec., 2004), pp. 487-513


#include <R.h>
#include "Rmath.h"

extern "C" {

  void stat103(double *x, int *xlen, double *level, int *nblevel, char **name, int *getname, double *statistic, int *pvalcomp, double *pvalue, double *critvalL, double *critvalR, int *usecrit, int *alter, int *decision, double *paramstat, int *nbparamstat) {

// If the test statistic can only be in category 3 or 4 (see just below), INDICATE the following line accordingly. Else, leave it commented.
// 0: two.sided=bilateral, 1: less=unilateral, 2: greater=unilateral, 3: bilateral test that rejects H0 only for large values of the test statistic, 
// 4: bilateral test that rejects H0 only for small values of the test statistic
    alter[0] = 3;

    int i, j = 0, n = xlen[0];
    if (getname[0] == 1) {    
// Here, INDICATE the name of your statistic
      const char *nom = "$CH$";
// Here, INDICATE the number of parameters of your statistic
      nbparamstat[0] = 4;
// Here, INDICATE the default values of the parameters:
      if (name[0][0] == '1') { // To prevent writing in a non declared address (because maybe paramstat has not be initialized with a sufficient length since the correct value of nbparamstat[0] may be unkown yet).
	paramstat[0] = 4.0;
	paramstat[1] = 1.0;
	paramstat[2] = 1.0;
	paramstat[3] = 1.0;
     }
// The following 7 lines should NOT be modified
      const char *space = " ";
      while (nom[j] != '\0') {
	name[j][0] = nom[j];
	j++;
      }
      for (i = j; i < 50; i++) name[i][0] = space[0];
      return;
    }

    // Initialization of the parameters
    int m0, version, criterion, orthofam;
    // m0 <= 10
    // version = Z,T; criterion = AIC, BIC; orthofam = poly, cos
    if (nbparamstat[0] == 0) {
      nbparamstat[0] = 4;
      paramstat[0] = 4.0;
      paramstat[1] = 1.0;
      paramstat[2] = 1.0;
      paramstat[3] = 1.0;
      m0 = 4;
      version = 1;
      criterion = 1;
      orthofam = 1;
    } else if (nbparamstat[0] == 1) {
      nbparamstat[0] = 4;
      m0 = (int)paramstat[0];
      paramstat[1] = 1.0;
      paramstat[2] = 1.0;
      paramstat[3] = 1.0;
      version = 1;
      criterion = 1;
      orthofam = 1;
    } else if (nbparamstat[0] == 2) {
      nbparamstat[0] = 4;
      m0 = (int)paramstat[0];
      version = (int)paramstat[1];
      paramstat[2] = 1.0;
      paramstat[3] = 1.0;
      criterion = 1;
      orthofam = 1;
    } else if (nbparamstat[0] == 3) {
      nbparamstat[0] = 4;
      m0 = (int)paramstat[0];
      version = (int)paramstat[1];
      criterion = (int)paramstat[2];
      paramstat[3] = 1.0;
      orthofam = 1;
    } else if (nbparamstat[0] == 4) {
      m0 = (int)paramstat[0];
      version = (int)paramstat[1];
      criterion = (int)paramstat[2];
      orthofam = (int)paramstat[3];
    } else {
      Rf_error("Number of parameters in stat103 should be at most: 4");
    }

    // If necessary, we check if some parameter values are out of parameter space
    if ((m0 < 0) || (m0 > 10)) {
      Rf_warning("m0 should be >=0 and <= 10 in stat103!\n");
      statistic[0] = R_NaN;
      return;
    }
    if ((version != 1) && (version != 2)) {
      Rf_warning("version should be 1 or 2 in stat103!\n");
      statistic[0] = R_NaN;
      return;
    }
    if ((criterion != 1) && (criterion != 2)) {
      Rf_warning("criterion should be 1 or 2 in stat103!\n");
      statistic[0] = R_NaN;
      return;
    }
    if ((orthofam != 1) && (orthofam != 2)) {
      Rf_warning("orthofam should be 1 or 2 in stat103!\n");
      statistic[0] = R_NaN;
      return;
    }


    if (n > 3) {
// Computation of the value of the test statistic
      void R_rsort(double* x, int n);
      double CHZ103(double *zhat, int n, int *S, int cardS, int orthofam);
      double CHT103(double *zhat, int n, int *S, int cardS, int orthofam);
      
      R_rsort(x, n); 		// we sort the data

      double *zhat;
      zhat = new double[n];
      
      double muhat, bhat, tmp = 0.0;
      // calculate mu^ and b^ by using the maximum likelihood estimators 
      // mu^ = the sample median
      // b^ = 1/n * \sum_{i=1}^{n} |xi - mu^|
      
      // calculate mu^     
      if (n % 2 == 0) {		// check if n is divisible by 2
	muhat = (x[n / 2 - 1] + x[n / 2]) / 2.0;
      } else {
	muhat = x[n / 2];
      }
      
      // calculate b^
      for (i = 0; i < n; i++) {
	tmp = tmp + fabs(x[i] - muhat);
      }
      bhat = tmp / (double)n;

      if (fabs(bhat) < R_pow(10.0, -30.0)) Rf_error("bhat too small in stat103; maybe your values of x are all the same?");
      
      for (i = 0; i < n; i++) zhat[i] = (x[i] - muhat) / bhat;

      int cardS;
      int *arr;
      arr = new int[m0];
      for (i = 0; i < m0; i++) arr[i] = i + 1;
      int *S;
      S = new int[m0];
      
      double stat, statvalue = 0.0, crit;

      //      int breakloop;
      int maxS;

      tmp = 0.0; // empty set
      int  count = (int)R_pow(2.0, (double)m0);
      // The outer for loop will run 2 ^ m0 - 1 times to compute all subsets
      // excluding the emptyset (set i = 0 below to also get the empyset).
      // Here variable i will act as a binary counter
      for (i = 1; i < count; i++) {
	// The inner for loop will run m0 times, as the maximum number
	// of elements a set can have is m0
	// This loop below will generate a different subset S, for each value of i
	cardS = 0;
	for (j = 0; j < m0; j++) {
	  // This if condition will check if jth bit in binary representation of  i  is set or not
	  // if the value of (i & (1 << j)) is greater than 0 , include arr[j] in the current subset
	  // otherwise exclude arr[j]
	  if ((i & (1 << j)) > 0) {
	    S[cardS] = arr[j];
	    cardS = cardS + 1;
	  }
	}
	if (orthofam == 1) {
	  //	  breakloop = 0;
	  maxS = S[0];
	  for (j = 0; j < cardS; j++) {
	    if (S[j] > maxS) maxS = S[j];
	    //	    if (S[j] % 2 != 0) {
	    //	      breakloop = 1;
	    //	    }
	  }
	  //	  if (breakloop == 1) continue; // remove odd degrees
	  if (maxS > 1) {
	    if (maxS % 2 != 0) continue; // Do not keep S when max(S) is odd unless S = {1}
	  }
	}
	// Now, using that S, i.e., S = {S[0], ..., S[cardS]}:
	if (version == 1) { // Z
	  stat = CHZ103(zhat, n, S, cardS, orthofam);
	  if (criterion == 1) { // AIC
	    crit = stat - 2.0 * cardS;
	  } else if (criterion == 2) { // BIC
	    crit = stat - log((double)n) * cardS;	      
	  }
	} else if (version == 2) { // T
	  stat = CHT103(zhat, n, S, cardS, orthofam);
	  if (criterion == 1) { // AIC
	    crit = stat - 2.0 * cardS;
	  } else if (criterion == 2) { // BIC
	    crit = stat - log((double)n) * cardS;	      
	  }
	}

	if (crit > tmp) {
	  tmp = crit;
	  statvalue = stat;
	}
      }
   
      statistic[0] = statvalue; // Here is the test statistic value
      

if (pvalcomp[0] == 1) {
	// If possible, computation of the p-value.
	#include "pvalues/pvalue103.cpp"
}

// We take the decision to reject or not to reject the null hypothesis H0
    for (i = 0; i <= (nblevel[0] - 1);i++) {
    if (usecrit[0] == 1) { // We use the provided critical values
	if (statistic[0] > critvalR[i]) decision[i] = 1; else decision[i] = 0; // two.sided (but in this case only one possible critical value)
      } else {
		if (pvalue[0] < level[i]) decision[i] = 1; else decision[i] = 0; // We use the p-value
      }
    }
    
// If applicable, we free the unused array of pointers
    delete[] zhat;
    delete[] arr;
    delete[] S;
    
}

// We return
    return;
   
        
  }


 
  double psi103(int orthofam, int j, double x) {
    double res = 0.0;
    if (orthofam == 1) {
      if (j == 0) res = 1.0;
      if (j == 1) res = 0.707107 * x;    
      if (j == 2) res = 0.223607 * (-2.0 + R_pow(x, 2.0));
      if (j == 3) res = 0.0481125 * (-12.0 * x + R_pow(x, 3.0));
      if (j == 4) res = 0.00763274 * (-24.0 + R_pow(x, 4.0) - 33.6 * (-2.0 + R_pow(x, 2.0)));
      if (j == 5) res = 0.000977577 * (-360.0 * x + R_pow(x, 5.0) - 73.3333 * (-12.0 * x + R_pow(x, 3.0)));
      if (j == 6) res = 0.000103512 * (-720.0 + R_pow(x, 6.0) - 1944.0 * (-2.0 + R_pow(x, 2.0)) - 134.295 * (-24.0 + R_pow(x, 4.0) - 33.6 * (-2.0 + R_pow(x, 2.0))));
      if (j == 7) res = 9.44636 * R_pow(10.0, -6.0) * (-20160.0 * x + R_pow(x, 7.0) - 7280.0 * (-12.0 * x + R_pow(x, 3.0)) - 223.486 * (-360.0 * x + R_pow(x, 5.0) - 73.3333 * (-12.0 * x + R_pow(x, 3.0))));
      if (j == 8) res = 7.50613 * R_pow(10.0, -7.0) * (-40320.0 + R_pow(x, 8.0) - 177408.0 * (-2.0 + R_pow(x, 2.0)) - 20904.2 * (-24.0 + R_pow(x, 4.0) - 33.6 * (-2.0 + R_pow(x, 2.0))) - 343.561 * (-720.0 + R_pow(x, 6.0) - 1944.0 * (-2.0 + R_pow(x, 2.0)) - 134.295 * (-24.0 + R_pow(x, 4.0) - 33.6 * (-2.0 + R_pow(x, 2.0)))));
      if (j == 9) res = 5.32183 * R_pow(10.0, -8.0) * (-1.8144 * R_pow(10.0, 6.0) * x + R_pow(x, 9.0) - 1.008 * R_pow(10.0, 6.0) * (-12.0 * x + R_pow(x, 3.0)) - 51546.7 * (-360.0 * x + R_pow(x, 5.0) - 73.3333 * (-12.0 * x + R_pow(x, 3.0))) - 501.94 * (-20160.0 * x + R_pow(x, 7.0) - 7280.0 * (-12.0 * x + R_pow(x, 3.0)) - 223.486 * (-360.0 * x + R_pow(x, 5.0) - 73.3333 * (-12.0 * x + R_pow(x, 3.0)))));
      if (j == 10) res = 3.38415 * R_pow(10.0, -9.0) * (-3.6288 * R_pow(10.0, 6.0) + R_pow(x, 10.0) - 2.35872 * R_pow(10.0, 7.0) * (-2.0 + R_pow(x, 2.0)) - 4.15039 * R_pow(10.0, 6.0) * (-24.0 + R_pow(x, 4.0) - 33.6 * (-2.0 + R_pow(x, 2.0))) - 111817.0 * (-720.0 + R_pow(x, 6.0) - 1944.0 * (-2.0 + R_pow(x, 2.0)) - 134.295 * (-24.0 + R_pow(x, 4.0) - 33.6 * (-2.0 + R_pow(x, 2.0)))) - 700.875 * (-40320.0 + R_pow(x, 8.0) - 177408.0 * (-2.0 + R_pow(x, 2.0)) - 20904.2 * (-24.0 + R_pow(x, 4.0) - 33.6 * (-2.0 + R_pow(x, 2.0))) - 343.561 * (-720.0 + R_pow(x, 6.0) - 1944.0 * (-2.0 + R_pow(x, 2.0)) - 134.295 * (-24.0 + R_pow(x, 4.0) - 33.6 * (-2.0 + R_pow(x, 2.0))))));
    }
    if (orthofam == 2) {
      if (j == 0) res = 1.0;
      if (j == 1) res = 1.69031 * (-0.5 + cos(x));
      if (j == 2) res = 1.63272 * (-0.2 - 0.571429 * (-0.5 + cos(x)) + cos(2.0 * x));
      if (j == 3) res = 1.63361 * (-0.1 - 0.226891 * (-0.5 + cos(x)) - 0.543424 * (-0.2 - 0.571429 * (-0.5 + cos(x)) + cos(2.0 * x)) + cos(3.0 * x));
      if (j == 4) res = 1.63516 * (-0.0588235 - 0.113769 * (-0.5 + cos(x)) - 0.210583 * (-0.2 - 0.571429 * (-0.5 + cos(x)) + cos(2.0 * x)) - 0.539487 * (-0.1 - 0.226891 * (-0.5 + cos(x)) - 0.543424 * (-0.2 - 0.571429 * (-0.5 + cos(x)) + cos(2.0 * x)) + cos(3.0 * x)) + cos(4.0 * x));
      if (j == 5) res = 1.63602 * (-0.0384615 - 0.0676986 * (-0.5 + cos(x)) - 0.103347 * (-0.2 - 0.571429 * (-0.5 + cos(x)) + cos(2.0 * x)) - 0.206563 * (-0.1 - 0.226891 * (-0.5 + cos(x)) - 0.543424 * (-0.2 - 0.571429 * (-0.5 + cos(x)) + cos(2.0 * x)) + cos(3.0 * x)) - 0.538006 * (-0.0588235 - 0.113769 * (-0.5 + cos(x)) - 0.210583 * (-0.2 - 0.571429 * (-0.5 + cos(x)) + cos(2.0 * x)) - 0.539487 * (-0.1 - 0.226891 * (-0.5 + cos(x)) - 0.543424 * (-0.2 - 0.571429 * (-0.5 + cos(x)) + cos(2.0 * x)) + cos(3.0 * x)) + cos(4.0 * x)) + cos(5.0 * x));
      if (j == 6) res = 1.63652 * (-0.027027 - 0.0449064 * (-0.5 + cos(x)) - 0.0605593 * (-0.2 - 0.571429 * (-0.5 + cos(x)) + cos(2.0 * x)) - 0.100032 * (-0.1 - 0.226891 * (-0.5 + cos(x)) - 0.543424 * (-0.2 - 0.571429 * (-0.5 + cos(x)) + cos(2.0 * x)) + cos(3.0 * x)) - 0.20472 * (-0.0588235 - 0.113769 * (-0.5 + cos(x)) - 0.210583 * (-0.2 - 0.571429 * (-0.5 + cos(x)) + cos(2.0 * x)) - 0.539487 * (-0.1 - 0.226891 * (-0.5 + cos(x)) - 0.543424 * (-0.2 - 0.571429 * (-0.5 + cos(x)) + cos(2.0 * x)) + cos(3.0 * x)) + cos(4.0 * x)) - 0.537212 * (-0.0384615 - 0.0676986 * (-0.5 + cos(x)) - 0.103347 * (-0.2 - 0.571429 * (-0.5 + cos(x)) + cos(2.0 * x)) - 0.206563 * (-0.1 - 0.226891 * (-0.5 + cos(x)) - 0.543424 * (-0.2 - 0.571429 * (-0.5 + cos(x)) + cos(2.0 * x)) + cos(3.0 * x)) - 0.538006 * (-0.0588235 - 0.113769 * (-0.5 + cos(x)) - 0.210583 * (-0.2 - 0.571429 * (-0.5 + cos(x)) + cos(2.0 * x)) - 0.539487 * (-0.1 - 0.226891 * (-0.5 + cos(x)) - 0.543424 * (-0.2 - 0.571429 * (-0.5 + cos(x)) + cos(2.0 * x)) + cos(3.0 * x)) + cos(4.0 * x)) + cos(5.0 * x)) + cos(6.0 * x));
      if (j == 7) res = 1.63683 * (-0.02 - 0.0320166 * (-0.5 + cos(x)) - 0.0397866 * (-0.2 - 0.571429 * (-0.5 + cos(x)) + cos(2.0 * x)) - 0.0579348 * (-0.1 - 0.226891 * (-0.5 + cos(x)) - 0.543424 * (-0.2 - 0.571429 * (-0.5 + cos(x)) + cos(2.0 * x)) + cos(3.0 * x)) - 0.0983732 * (-0.0588235 - 0.113769 * (-0.5 + cos(x)) - 0.210583 * (-0.2 - 0.571429 * (-0.5 + cos(x)) + cos(2.0 * x)) - 0.539487 * (-0.1 - 0.226891 * (-0.5 + cos(x)) - 0.543424 * (-0.2 - 0.571429 * (-0.5 + cos(x)) + cos(2.0 * x)) + cos(3.0 * x)) + cos(4.0 * x)) - 0.203683 * (-0.0384615 - 0.0676986 * (-0.5 + cos(x)) - 0.103347 * (-0.2 - 0.571429 * (-0.5 + cos(x)) + cos(2.0 * x)) - 0.206563 * (-0.1 - 0.226891 * (-0.5 + cos(x)) - 0.543424 * (-0.2 - 0.571429 * (-0.5 + cos(x)) + cos(2.0 * x)) + cos(3.0 * x)) - 0.538006 * (-0.0588235 - 0.113769 * (-0.5 + cos(x)) - 0.210583 * (-0.2 - 0.571429 * (-0.5 + cos(x)) + cos(2.0 * x)) - 0.539487 * (-0.1 - 0.226891 * (-0.5 + cos(x)) - 0.543424 * (-0.2 - 0.571429 * (-0.5 + cos(x)) + cos(2.0 * x)) + cos(3.0 * x)) + cos(4.0 * x)) + cos(5.0 * x)) - 0.53673 * (-0.027027 - 0.0449064 * (-0.5 + cos(x)) - 0.0605593 * (-0.2 - 0.571429 * (-0.5 + cos(x)) + cos(2.0 * x)) - 0.100032 * (-0.1 - 0.226891 * (-0.5 + cos(x)) - 0.543424 * (-0.2 - 0.571429 * (-0.5 + cos(x)) + cos(2.0 * x)) + cos(3.0 * x)) - 0.20472 * (-0.0588235 - 0.113769 * (-0.5 + cos(x)) - 0.210583 * (-0.2 - 0.571429 * (-0.5 + cos(x)) + cos(2.0 * x)) - 0.539487 * (-0.1 - 0.226891 * (-0.5 + cos(x)) - 0.543424 * (-0.2 - 0.571429 * (-0.5 + cos(x)) + cos(2.0 * x)) + cos(3.0 * x)) + cos(4.0 * x)) - 0.537212 * (-0.0384615 - 0.0676986 * (-0.5 + cos(x)) - 0.103347 * (-0.2 - 0.571429 * (-0.5 + cos(x)) + cos(2.0 * x)) - 0.206563 * (-0.1 - 0.226891 * (-0.5 + cos(x)) - 0.543424 * (-0.2 - 0.571429 * (-0.5 + cos(x)) + cos(2.0 * x)) + cos(3.0 * x)) - 0.538006 * (-0.0588235 - 0.113769 * (-0.5 + cos(x)) - 0.210583 * (-0.2 - 0.571429 * (-0.5 + cos(x)) + cos(2.0 * x)) - 0.539487 * (-0.1 - 0.226891 * (-0.5 + cos(x)) - 0.543424 * (-0.2 - 0.571429 * (-0.5 + cos(x)) + cos(2.0 * x)) + cos(3.0 * x)) + cos(4.0 * x)) + cos(5.0 * x)) +  cos(6.0 * x)) + cos(7.0 * x));
      if (j == 8) res = 1.63705 * (-0.0153846 - 0.024015 * (-0.5 + cos(x)) - 0.0282147 * (-0.2 - 0.571429 * (-0.5 + cos(x)) + cos(2.0 * x)) - 0.0377136 * (-0.1 - 0.226891 * (-0.5 + cos(x)) - 0.543424 * (-0.2 - 0.571429 * (-0.5 + cos(x)) + cos(2.0 * x)) + cos(3.0 * x)) - 0.056539 * (-0.0588235 - 0.113769 * (-0.5 + cos(x)) - 0.210583 * (-0.2 - 0.571429 * (-0.5 + cos(x)) + cos(2.0 * x)) - 0.539487 * (-0.1 - 0.226891 * (-0.5 + cos(x)) - 0.543424 * (-0.2 - 0.571429 * (-0.5 + cos(x)) + cos(2.0 * x)) + cos(3.0 * x)) + cos(4.0 * x)) - 0.0974017 * (-0.0384615 - 0.0676986 * (-0.5 + cos(x)) - 0.103347 * (-0.2 - 0.571429 * (-0.5 + cos(x)) + cos(2.0 * x)) - 0.206563 * (-0.1 - 0.226891 * (-0.5 + cos(x)) - 0.543424 * (-0.2 - 0.571429 * (-0.5 + cos(x)) + cos(2.0 * x)) + cos(3.0 * x)) - 0.538006 * (-0.0588235 - 0.113769 * (-0.5 + cos(x)) - 0.210583 * (-0.2 - 0.571429 * (-0.5 + cos(x)) + cos(2.0 * x)) - 0.539487 * (-0.1 - 0.226891 * (-0.5 + cos(x)) - 0.543424 * (-0.2 - 0.571429 * (-0.5 + cos(x)) + cos(2.0 * x)) + cos(3.0 * x)) + cos(4.0 * x)) + cos(5.0 * x)) - 0.203037 * (-0.027027 - 0.0449064 * (-0.5 + cos(x)) - 0.0605593 * (-0.2 - 0.571429 * (-0.5 + cos(x)) + cos(2.0 * x)) - 0.100032 * (-0.1 - 0.226891 * (-0.5 + cos(x)) - 0.543424 * (-0.2 - 0.571429 * (-0.5 + cos(x)) + cos(2.0 * x)) + cos(3.0 * x)) - 0.20472 * (-0.0588235 - 0.113769 * (-0.5 + cos(x)) - 0.210583 * (-0.2 - 0.571429 * (-0.5 + cos(x)) + cos(2.0 * x)) - 0.539487 * (-0.1 - 0.226891 * (-0.5 + cos(x)) - 0.543424 * (-0.2 - 0.571429 * (-0.5 + cos(x)) + cos(2.0 * x)) + cos(3.0 * x)) + cos(4.0 * x)) - 0.537212 * (-0.0384615 - 0.0676986 * (-0.5 + cos(x)) - 0.103347 * (-0.2 - 0.571429 * (-0.5 + cos(x)) + cos(2.0 * x)) - 0.206563 * (-0.1 - 0.226891 * (-0.5 + cos(x)) - 0.543424 * (-0.2 - 0.571429 * (-0.5 + cos(x)) + cos(2.0 * x)) + cos(3.0 * x)) - 0.538006 * (-0.0588235 - 0.113769 * (-0.5 + cos(x)) - 0.210583 * (-0.2 - 0.571429 * (-0.5 + cos(x)) + cos(2.0 * x)) - 0.539487 * (-0.1 - 0.226891 * (-0.5 + cos(x)) -  0.543424 * (-0.2 - 0.571429 * (-0.5 + cos(x)) +  cos(2.0 * x)) + cos(3.0 * x)) + cos(4.0 * x)) + cos(5.0 * x)) + cos(6.0 * x)) - 0.536415 * (-0.02 - 0.0320166 * (-0.5 + cos(x)) - 0.0397866 * (-0.2 - 0.571429 * (-0.5 + cos(x)) + cos(2.0 * x)) - 0.0579348 * (-0.1 - 0.226891 * (-0.5 + cos(x)) - 0.543424 * (-0.2 - 0.571429 * (-0.5 + cos(x)) + cos(2.0 * x)) + cos(3.0 * x)) - 0.0983732 * (-0.0588235 - 0.113769 * (-0.5 + cos(x)) - 0.210583 * (-0.2 - 0.571429 * (-0.5 + cos(x)) + cos(2.0 * x)) - 0.539487 * (-0.1 - 0.226891 * (-0.5 + cos(x)) - 0.543424 * (-0.2 - 0.571429 * (-0.5 + cos(x)) + cos(2.0 * x)) + cos(3.0 * x)) + cos(4.0 * x)) - 0.203683 * (-0.0384615 - 0.0676986 * (-0.5 + cos(x)) - 0.103347 * (-0.2 - 0.571429 * (-0.5 + cos(x)) + cos(2.0 * x)) - 0.206563 * (-0.1 - 0.226891 * (-0.5 + cos(x)) - 0.543424 * (-0.2 - 0.571429 * (-0.5 + cos(x)) + cos(2.0 * x)) + cos(3.0 * x)) - 0.538006 * (-0.0588235 - 0.113769 * (-0.5 + cos(x)) - 0.210583 * (-0.2 - 0.571429 * (-0.5 + cos(x)) + cos(2.0 * x)) - 0.539487 * (-0.1 - 0.226891 * (-0.5 + cos(x)) - 0.543424 * (-0.2 - 0.571429 * (-0.5 + cos(x)) +  cos(2.0 * x)) + cos(3.0 * x)) + cos(4.0 * x)) + cos(5.0 * x)) - 0.53673 * (-0.027027 - 0.0449064 * (-0.5 + cos(x)) - 0.0605593 * (-0.2 - 0.571429 * (-0.5 + cos(x)) + cos(2.0 * x)) - 0.100032 * (-0.1 - 0.226891 * (-0.5 + cos(x)) - 0.543424 * (-0.2 - 0.571429 * (-0.5 + cos(x)) + cos(2.0 * x)) + cos(3.0 * x)) - 0.20472 * (-0.0588235 - 0.113769 * (-0.5 + cos(x)) - 0.210583 * (-0.2 - 0.571429 * (-0.5 + cos(x)) + cos(2.0 * x)) - 0.539487 * (-0.1 - 0.226891 * (-0.5 + cos(x)) -  0.543424 * (-0.2 - 0.571429 * (-0.5 + cos(x)) +  cos(2.0 * x)) + cos(3.0 * x)) + cos(4.0 * x)) - 0.537212 * (-0.0384615 - 0.0676986 * (-0.5 + cos(x)) - 0.103347 * (-0.2 - 0.571429 * (-0.5 + cos(x)) + cos(2.0 * x)) - 0.206563 * (-0.1 - 0.226891 * (-0.5 + cos(x)) -  0.543424 * (-0.2 - 0.571429 * (-0.5 + cos(x)) +  cos(2.0 * x)) + cos(3.0 * x)) - 0.538006 * (-0.0588235 - 0.113769 * (-0.5 + cos(x)) -  0.210583 * (-0.2 - 0.571429 * (-0.5 + cos(x)) +  cos(2.0 * x)) -  0.539487 * (-0.1 - 0.226891 * (-0.5 + cos(x)) - 0.543424 * (-0.2 - 0.571429 * (-0.5 + cos(x)) + cos(2.0 * x)) + cos(3.0 * x)) + cos(4.0 * x)) + cos(5.0 * x)) + cos(6.0 * x)) + cos(7.0 * x)) + cos(8.0 * x));
      if (j == 9) res = 1.6372 * (-0.0121951 - 0.0187007 * (-0.5 + cos(x)) - 0.0211108 * (-0.2 - 0.571429 * (-0.5 + cos(x)) + cos(2.0 * x)) - 0.0265633 * (-0.1 - 0.226891 * (-0.5 + cos(x)) - 0.543424 * (-0.2 - 0.571429 * (-0.5 + cos(x)) + cos(2.0 * x)) + cos(3.0 * x)) - 0.0365565 * (-0.0588235 - 0.113769 * (-0.5 + cos(x)) - 0.210583 * (-0.2 - 0.571429 * (-0.5 + cos(x)) + cos(2.0 * x)) - 0.539487 * (-0.1 - 0.226891 * (-0.5 + cos(x)) - 0.543424 * (-0.2 - 0.571429 * (-0.5 + cos(x)) + cos(2.0 * x)) + cos(3.0 * x)) + cos(4.0 * x)) - 0.0556919 * (-0.0384615 - 0.0676986 * (-0.5 + cos(x)) - 0.103347 * (-0.2 - 0.571429 * (-0.5 + cos(x)) + cos(2.0 * x)) - 0.206563 * (-0.1 - 0.226891 * (-0.5 + cos(x)) - 0.543424 * (-0.2 - 0.571429 * (-0.5 + cos(x)) + cos(2.0 * x)) + cos(3.0 * x)) - 0.538006 * (-0.0588235 - 0.113769 * (-0.5 + cos(x)) - 0.210583 * (-0.2 - 0.571429 * (-0.5 + cos(x)) + cos(2.0 * x)) - 0.539487 * (-0.1 - 0.226891 * (-0.5 + cos(x)) - 0.543424 * (-0.2 - 0.571429 * (-0.5 + cos(x)) + cos(2.0 * x)) + cos(3.0 * x)) + cos(4.0 * x)) + cos(5.0 * x)) - 0.0967804 * (-0.027027 - 0.0449064 * (-0.5 + cos(x)) - 0.0605593 * (-0.2 - 0.571429 * (-0.5 + cos(x)) + cos(2.0 * x)) - 0.100032 * (-0.1 - 0.226891 * (-0.5 + cos(x)) - 0.543424 * (-0.2 - 0.571429 * (-0.5 + cos(x)) + cos(2.0 * x)) + cos(3.0 * x)) - 0.20472 * (-0.0588235 - 0.113769 * (-0.5 + cos(x)) - 0.210583 * (-0.2 - 0.571429 * (-0.5 + cos(x)) + cos(2.0 * x)) - 0.539487 * (-0.1 - 0.226891 * (-0.5 + cos(x)) - 0.543424 * (-0.2 - 0.571429 * (-0.5 + cos(x)) + cos(2.0 * x)) + cos(3.0 * x)) + cos(4.0 * x)) - 0.537212 * (-0.0384615 - 0.0676986 * (-0.5 + cos(x)) - 0.103347 * (-0.2 - 0.571429 * (-0.5 + cos(x)) + cos(2.0 * x)) - 0.206563 * (-0.1 - 0.226891 * (-0.5 + cos(x)) - 0.543424 * (-0.2 - 0.571429 * (-0.5 + cos(x)) + cos(2.0 * x)) + cos(3.0 * x)) - 0.538006 * (-0.0588235 - 0.113769 * (-0.5 + cos(x)) - 0.210583 * (-0.2 - 0.571429 * (-0.5 + cos(x)) + cos(2.0 * x)) -  0.539487 * (-0.1 - 0.226891 * (-0.5 + cos(x)) -  0.543424 * (-0.2 - 0.571429 * (-0.5 + cos(x)) +  cos(2.0 * x)) + cos(3.0 * x)) + cos(4.0 * x)) + cos(5.0 * x)) + cos(6.0 * x)) - 0.202607 * (-0.02 - 0.0320166 * (-0.5 + cos(x)) - 0.0397866 * (-0.2 - 0.571429 * (-0.5 + cos(x)) + cos(2.0 * x)) - 0.0579348 * (-0.1 - 0.226891 * (-0.5 + cos(x)) - 0.543424 * (-0.2 - 0.571429 * (-0.5 + cos(x)) + cos(2.0 * x)) + cos(3.0 * x)) - 0.0983732 * (-0.0588235 - 0.113769 * (-0.5 + cos(x)) - 0.210583 * (-0.2 - 0.571429 * (-0.5 + cos(x)) + cos(2.0 * x)) - 0.539487 * (-0.1 - 0.226891 * (-0.5 + cos(x)) - 0.543424 * (-0.2 - 0.571429 * (-0.5 + cos(x)) + cos(2.0 * x)) + cos(3.0 * x)) + cos(4.0 * x)) - 0.203683 * (-0.0384615 - 0.0676986 * (-0.5 + cos(x)) - 0.103347 * (-0.2 - 0.571429 * (-0.5 + cos(x)) + cos(2.0 * x)) - 0.206563 * (-0.1 - 0.226891 * (-0.5 + cos(x)) - 0.543424 * (-0.2 - 0.571429 * (-0.5 + cos(x)) + cos(2.0 * x)) + cos(3.0 * x)) - 0.538006 * (-0.0588235 - 0.113769 * (-0.5 + cos(x)) - 0.210583 * (-0.2 - 0.571429 * (-0.5 + cos(x)) + cos(2.0 * x)) -  0.539487 * (-0.1 - 0.226891 * (-0.5 + cos(x)) -  0.543424 * (-0.2 - 0.571429 * (-0.5 + cos(x)) +  cos(2.0 * x)) + cos(3.0 * x)) + cos(4.0 * x)) + cos(5.0 * x)) - 0.53673 * (-0.027027 - 0.0449064 * (-0.5 + cos(x)) - 0.0605593 * (-0.2 - 0.571429 * (-0.5 + cos(x)) + cos(2.0 * x)) - 0.100032 * (-0.1 - 0.226891 * (-0.5 + cos(x)) - 0.543424 * (-0.2 - 0.571429 * (-0.5 + cos(x)) + cos(2.0 * x)) + cos(3.0 * x)) - 0.20472 * (-0.0588235 - 0.113769 * (-0.5 + cos(x)) - 0.210583 * (-0.2 - 0.571429 * (-0.5 + cos(x)) + cos(2.0 * x)) - 0.539487 * (-0.1 - 0.226891 * (-0.5 + cos(x)) -  0.543424 * (-0.2 - 0.571429 * (-0.5 + cos(x)) +  cos(2.0 * x)) + cos(3.0 * x)) + cos(4.0 * x)) - 0.537212 * (-0.0384615 - 0.0676986 * (-0.5 + cos(x)) - 0.103347 * (-0.2 - 0.571429 * (-0.5 + cos(x)) + cos(2.0 * x)) - 0.206563 * (-0.1 - 0.226891 * (-0.5 + cos(x)) -  0.543424 * (-0.2 - 0.571429 * (-0.5 + cos(x)) +  cos(2.0 * x)) + cos(3.0 * x)) - 0.538006 * (-0.0588235 - 0.113769 * (-0.5 + cos(x)) -  0.210583 * (-0.2 - 0.571429 * (-0.5 + cos(x)) +  cos(2.0 * x)) -  0.539487 * (-0.1 - 0.226891 * (-0.5 + cos(x)) -  0.543424 * (-0.2 - 0.571429 * (-0.5 + cos(x)) + cos(2.0 * x)) + cos(3.0 * x)) + cos(4.0 * x)) + cos(5.0 * x)) + cos(6.0 * x)) + cos(7.0 * x)) - 0.536196 * (-0.0153846 - 0.024015 * (-0.5 + cos(x)) - 0.0282147 * (-0.2 - 0.571429 * (-0.5 + cos(x)) + cos(2.0 * x)) - 0.0377136 * (-0.1 - 0.226891 * (-0.5 + cos(x)) - 0.543424 * (-0.2 - 0.571429 * (-0.5 + cos(x)) + cos(2.0 * x)) + cos(3.0 * x)) - 0.056539 * (-0.0588235 - 0.113769 * (-0.5 + cos(x)) - 0.210583 * (-0.2 - 0.571429 * (-0.5 + cos(x)) + cos(2.0 * x)) - 0.539487 * (-0.1 - 0.226891 * (-0.5 + cos(x)) - 0.543424 * (-0.2 - 0.571429 * (-0.5 + cos(x)) + cos(2.0 * x)) + cos(3.0 * x)) + cos(4.0 * x)) - 0.0974017 * (-0.0384615 - 0.0676986 * (-0.5 + cos(x)) - 0.103347 * (-0.2 - 0.571429 * (-0.5 + cos(x)) + cos(2.0 * x)) - 0.206563 * (-0.1 - 0.226891 * (-0.5 + cos(x)) - 0.543424 * (-0.2 - 0.571429 * (-0.5 + cos(x)) + cos(2.0 * x)) + cos(3.0 * x)) - 0.538006 * (-0.0588235 - 0.113769 * (-0.5 + cos(x)) - 0.210583 * (-0.2 - 0.571429 * (-0.5 + cos(x)) + cos(2.0 * x)) - 0.539487 * (-0.1 - 0.226891 * (-0.5 + cos(x)) -  0.543424 * (-0.2 - 0.571429 * (-0.5 + cos(x)) +  cos(2.0 * x)) + cos(3.0 * x)) + cos(4.0 * x)) + cos(5.0 * x)) - 0.203037 * (-0.027027 - 0.0449064 * (-0.5 + cos(x)) - 0.0605593 * (-0.2 - 0.571429 * (-0.5 + cos(x)) + cos(2.0 * x)) - 0.100032 * (-0.1 - 0.226891 * (-0.5 + cos(x)) - 0.543424 * (-0.2 - 0.571429 * (-0.5 + cos(x)) + cos(2.0 * x)) + cos(3.0 * x)) - 0.20472 * (-0.0588235 - 0.113769 * (-0.5 + cos(x)) - 0.210583 * (-0.2 - 0.571429 * (-0.5 + cos(x)) + cos(2.0 * x)) - 0.539487 * (-0.1 - 0.226891 * (-0.5 + cos(x)) -  0.543424 * (-0.2 - 0.571429 * (-0.5 + cos(x)) +  cos(2.0 * x)) + cos(3.0 * x)) + cos(4.0 * x)) - 0.537212 * (-0.0384615 - 0.0676986 * (-0.5 + cos(x)) - 0.103347 * (-0.2 - 0.571429 * (-0.5 + cos(x)) + cos(2.0 * x)) - 0.206563 * (-0.1 - 0.226891 * (-0.5 + cos(x)) -  0.543424 * (-0.2 - 0.571429 * (-0.5 + cos(x)) +  cos(2.0 * x)) + cos(3.0 * x)) - 0.538006 * (-0.0588235 - 0.113769 * (-0.5 + cos(x)) - 0.210583 * (-0.2 - 0.571429 * (-0.5 + cos(x)) +  cos(2.0 * x)) -  0.539487 * (-0.1 - 0.226891 * (-0.5 + cos(x)) -  0.543424 * (-0.2 - 0.571429 * (-0.5 + cos(x)) + cos(2.0 * x)) + cos(3.0 * x)) + cos(4.0 * x)) + cos(5.0 * x)) + cos(6.0 * x)) - 0.536415 * (-0.02 - 0.0320166 * (-0.5 + cos(x)) - 0.0397866 * (-0.2 - 0.571429 * (-0.5 + cos(x)) + cos(2.0 * x)) - 0.0579348 * (-0.1 - 0.226891 * (-0.5 + cos(x)) - 0.543424 * (-0.2 - 0.571429 * (-0.5 + cos(x)) + cos(2.0 * x)) + cos(3.0 * x)) - 0.0983732 * (-0.0588235 - 0.113769 * (-0.5 + cos(x)) - 0.210583 * (-0.2 - 0.571429 * (-0.5 + cos(x)) + cos(2.0 * x)) - 0.539487 * (-0.1 - 0.226891 * (-0.5 + cos(x)) -  0.543424 * (-0.2 - 0.571429 * (-0.5 + cos(x)) +  cos(2.0 * x)) + cos(3.0 * x)) + cos(4.0 * x)) - 0.203683 * (-0.0384615 - 0.0676986 * (-0.5 + cos(x)) - 0.103347 * (-0.2 - 0.571429 * (-0.5 + cos(x)) + cos(2.0 * x)) - 0.206563 * (-0.1 - 0.226891 * (-0.5 + cos(x)) -  0.543424 * (-0.2 - 0.571429 * (-0.5 + cos(x)) +  cos(2.0 * x)) + cos(3.0 * x)) - 0.538006 * (-0.0588235 - 0.113769 * (-0.5 + cos(x)) - 0.210583 * (-0.2 - 0.571429 * (-0.5 + cos(x)) +  cos(2.0 * x)) -  0.539487 * (-0.1 - 0.226891 * (-0.5 + cos(x)) -  0.543424 * (-0.2 - 0.571429 * (-0.5 + cos(x)) + cos(2.0 * x)) + cos(3.0 * x)) + cos(4.0 * x)) + cos(5.0 * x)) - 0.53673 * (-0.027027 - 0.0449064 * (-0.5 + cos(x)) - 0.0605593 * (-0.2 - 0.571429 * (-0.5 + cos(x)) +  cos(2.0 * x)) - 0.100032 * (-0.1 - 0.226891 * (-0.5 + cos(x)) -  0.543424 * (-0.2 - 0.571429 * (-0.5 + cos(x)) +  cos(2.0 * x)) + cos(3.0 * x)) - 0.20472 * (-0.0588235 - 0.113769 * (-0.5 + cos(x)) -  0.210583 * (-0.2 - 0.571429 * (-0.5 + cos(x)) +  cos(2.0 * x)) -  0.539487 * (-0.1 - 0.226891 * (-0.5 + cos(x)) -  0.543424 * (-0.2 - 0.571429 * (-0.5 + cos(x)) + cos(2.0 * x)) + cos(3.0 * x)) + cos(4.0 * x)) - 0.537212 * (-0.0384615 - 0.0676986 * (-0.5 + cos(x)) -  0.103347 * (-0.2 - 0.571429 * (-0.5 + cos(x)) +  cos(2.0 * x)) -  0.206563 * (-0.1 - 0.226891 * (-0.5 + cos(x)) -  0.543424 * (-0.2 - 0.571429 * (-0.5 + cos(x)) + cos(2.0 * x)) + cos(3.0 * x)) -  0.538006 * (-0.0588235 - 0.113769 * (-0.5 + cos(x)) - 0.210583 * (-0.2 - 0.571429 * (-0.5 + cos(x)) + cos(2.0 * x)) -  0.539487 * (-0.1 - 0.226891 * (-0.5 + cos(x)) - 0.543424 * (-0.2 - 0.571429 * (-0.5 + cos(x)) + cos(2.0 * x)) + cos(3.0 * x)) + cos(4.0 * x)) +  cos(5.0 * x)) + cos(6.0 * x)) + cos(7.0 * x)) + cos(8.0 * x)) + cos(9.0 * x));
      if (j == 10) res = 1.63731 * (-0.00990099 - 0.0149869 * (-0.5 + cos(x)) - 0.0164291 * (-0.2 - 0.571429 * (-0.5 + cos(x)) + cos(2.0 * x)) - 0.0197798 * (-0.1 - 0.226891 * (-0.5 + cos(x)) - 0.543424 * (-0.2 - 0.571429 * (-0.5 + cos(x)) + cos(2.0 * x)) + cos(3.0 * x)) - 0.0256038 * (-0.0588235 - 0.113769 * (-0.5 + cos(x)) - 0.210583 * (-0.2 - 0.571429 * (-0.5 + cos(x)) + cos(2.0 * x)) - 0.539487 * (-0.1 - 0.226891 * (-0.5 + cos(x)) - 0.543424 * (-0.2 - 0.571429 * (-0.5 + cos(x)) + cos(2.0 * x)) + cos(3.0 * x)) + cos(4.0 * x)) - 0.0358319 * (-0.0384615 - 0.0676986 * (-0.5 + cos(x)) - 0.103347 * (-0.2 - 0.571429 * (-0.5 + cos(x)) + cos(2.0 * x)) - 0.206563 * (-0.1 - 0.226891 * (-0.5 + cos(x)) - 0.543424 * (-0.2 - 0.571429 * (-0.5 + cos(x)) + cos(2.0 * x)) + cos(3.0 * x)) - 0.538006 * (-0.0588235 - 0.113769 * (-0.5 + cos(x)) - 0.210583 * (-0.2 - 0.571429 * (-0.5 + cos(x)) + cos(2.0 * x)) - 0.539487 * (-0.1 - 0.226891 * (-0.5 + cos(x)) - 0.543424 * (-0.2 - 0.571429 * (-0.5 + cos(x)) + cos(2.0 * x)) + cos(3.0 * x)) + cos(4.0 * x)) + cos(5.0 * x)) - 0.0551364 * (-0.027027 - 0.0449064 * (-0.5 + cos(x)) - 0.0605593 * (-0.2 - 0.571429 * (-0.5 + cos(x)) + cos(2.0 * x)) - 0.100032 * (-0.1 - 0.226891 * (-0.5 + cos(x)) - 0.543424 * (-0.2 - 0.571429 * (-0.5 + cos(x)) + cos(2.0 * x)) + cos(3.0 * x)) - 0.20472 * (-0.0588235 - 0.113769 * (-0.5 + cos(x)) - 0.210583 * (-0.2 - 0.571429 * (-0.5 + cos(x)) + cos(2.0 * x)) - 0.539487 * (-0.1 - 0.226891 * (-0.5 + cos(x)) - 0.543424 * (-0.2 - 0.571429 * (-0.5 + cos(x)) + cos(2.0 * x)) + cos(3.0 * x)) + cos(4.0 * x)) - 0.537212 * (-0.0384615 - 0.0676986 * (-0.5 + cos(x)) - 0.103347 * (-0.2 - 0.571429 * (-0.5 + cos(x)) + cos(2.0 * x)) - 0.206563 * (-0.1 - 0.226891 * (-0.5 + cos(x)) - 0.543424 * (-0.2 - 0.571429 * (-0.5 + cos(x)) + cos(2.0 * x)) + cos(3.0 * x)) - 0.538006 * (-0.0588235 - 0.113769 * (-0.5 + cos(x)) - 0.210583 * (-0.2 - 0.571429 * (-0.5 + cos(x)) + cos(2.0 * x)) - 0.539487 * (-0.1 - 0.226891 * (-0.5 + cos(x)) -  0.543424 * (-0.2 - 0.571429 * (-0.5 + cos(x)) +  cos(2.0 * x)) + cos(3.0 * x)) + cos(4.0 * x)) + cos(5.0 * x)) + cos(6.0 * x)) - 0.0963581 * (-0.02 - 0.0320166 * (-0.5 + cos(x)) - 0.0397866 * (-0.2 - 0.571429 * (-0.5 + cos(x)) + cos(2.0 * x)) - 0.0579348 * (-0.1 - 0.226891 * (-0.5 + cos(x)) - 0.543424 * (-0.2 - 0.571429 * (-0.5 + cos(x)) + cos(2.0 * x)) + cos(3.0 * x)) - 0.0983732 * (-0.0588235 - 0.113769 * (-0.5 + cos(x)) - 0.210583 * (-0.2 - 0.571429 * (-0.5 + cos(x)) + cos(2.0 * x)) - 0.539487 * (-0.1 - 0.226891 * (-0.5 + cos(x)) - 0.543424 * (-0.2 - 0.571429 * (-0.5 + cos(x)) + cos(2.0 * x)) + cos(3.0 * x)) + cos(4.0 * x)) - 0.203683 * (-0.0384615 - 0.0676986 * (-0.5 + cos(x)) - 0.103347 * (-0.2 - 0.571429 * (-0.5 + cos(x)) + cos(2.0 * x)) - 0.206563 * (-0.1 - 0.226891 * (-0.5 + cos(x)) - 0.543424 * (-0.2 - 0.571429 * (-0.5 + cos(x)) + cos(2.0 * x)) + cos(3.0 * x)) - 0.538006 * (-0.0588235 - 0.113769 * (-0.5 + cos(x)) - 0.210583 * (-0.2 - 0.571429 * (-0.5 + cos(x)) + cos(2.0 * x)) - 0.539487 * (-0.1 - 0.226891 * (-0.5 + cos(x)) -  0.543424 * (-0.2 - 0.571429 * (-0.5 + cos(x)) +  cos(2.0 * x)) + cos(3.0 * x)) + cos(4.0 * x)) + cos(5.0 * x)) - 0.53673 * (-0.027027 - 0.0449064 * (-0.5 + cos(x)) - 0.0605593 * (-0.2 - 0.571429 * (-0.5 + cos(x)) + cos(2.0 * x)) - 0.100032 * (-0.1 - 0.226891 * (-0.5 + cos(x)) - 0.543424 * (-0.2 - 0.571429 * (-0.5 + cos(x)) + cos(2.0 * x)) + cos(3.0 * x)) - 0.20472 * (-0.0588235 - 0.113769 * (-0.5 + cos(x)) - 0.210583 * (-0.2 - 0.571429 * (-0.5 + cos(x)) + cos(2.0 * x)) - 0.539487 * (-0.1 - 0.226891 * (-0.5 + cos(x)) -  0.543424 * (-0.2 - 0.571429 * (-0.5 + cos(x)) +  cos(2.0 * x)) + cos(3.0 * x)) + cos(4.0 * x)) - 0.537212 * (-0.0384615 - 0.0676986 * (-0.5 + cos(x)) - 0.103347 * (-0.2 - 0.571429 * (-0.5 + cos(x)) + cos(2.0 * x)) - 0.206563 * (-0.1 - 0.226891 * (-0.5 + cos(x)) -  0.543424 * (-0.2 - 0.571429 * (-0.5 + cos(x)) +  cos(2.0 * x)) + cos(3.0 * x)) - 0.538006 * (-0.0588235 - 0.113769 * (-0.5 + cos(x)) -  0.210583 * (-0.2 - 0.571429 * (-0.5 + cos(x)) +  cos(2.0 * x)) -  0.539487 * (-0.1 - 0.226891 * (-0.5 + cos(x)) -  0.543424 * (-0.2 - 0.571429 * (-0.5 + cos(x)) + cos(2.0 * x)) + cos(3.0 * x)) + cos(4.0 * x)) + cos(5.0 * x)) + cos(6.0 * x)) + cos(7.0 * x)) - 0.202305 * (-0.0153846 - 0.024015 * (-0.5 + cos(x)) - 0.0282147 * (-0.2 - 0.571429 * (-0.5 + cos(x)) + cos(2.0 * x)) - 0.0377136 * (-0.1 - 0.226891 * (-0.5 + cos(x)) - 0.543424 * (-0.2 - 0.571429 * (-0.5 + cos(x)) + cos(2.0 * x)) + cos(3.0 * x)) - 0.056539 * (-0.0588235 - 0.113769 * (-0.5 + cos(x)) - 0.210583 * (-0.2 - 0.571429 * (-0.5 + cos(x)) + cos(2.0 * x)) - 0.539487 * (-0.1 - 0.226891 * (-0.5 + cos(x)) - 0.543424 * (-0.2 - 0.571429 * (-0.5 + cos(x)) + cos(2.0 * x)) + cos(3.0 * x)) + cos(4.0 * x)) - 0.0974017 * (-0.0384615 - 0.0676986 * (-0.5 + cos(x)) - 0.103347 * (-0.2 - 0.571429 * (-0.5 + cos(x)) + cos(2.0 * x)) - 0.206563 * (-0.1 - 0.226891 * (-0.5 + cos(x)) - 0.543424 * (-0.2 - 0.571429 * (-0.5 + cos(x)) + cos(2.0 * x)) + cos(3.0 * x)) - 0.538006 * (-0.0588235 - 0.113769 * (-0.5 + cos(x)) - 0.210583 * (-0.2 - 0.571429 * (-0.5 + cos(x)) + cos(2.0 * x)) - 0.539487 * (-0.1 - 0.226891 * (-0.5 + cos(x)) -  0.543424 * (-0.2 - 0.571429 * (-0.5 + cos(x)) +  cos(2.0 * x)) + cos(3.0 * x)) + cos(4.0 * x)) + cos(5.0 * x)) - 0.203037 * (-0.027027 - 0.0449064 * (-0.5 + cos(x)) - 0.0605593 * (-0.2 - 0.571429 * (-0.5 + cos(x)) + cos(2.0 * x)) - 0.100032 * (-0.1 - 0.226891 * (-0.5 + cos(x)) - 0.543424 * (-0.2 - 0.571429 * (-0.5 + cos(x)) + cos(2.0 * x)) + cos(3.0 * x)) - 0.20472 * (-0.0588235 - 0.113769 * (-0.5 + cos(x)) - 0.210583 * (-0.2 - 0.571429 * (-0.5 + cos(x)) + cos(2.0 * x)) - 0.539487 * (-0.1 - 0.226891 * (-0.5 + cos(x)) -  0.543424 * (-0.2 - 0.571429 * (-0.5 + cos(x)) +  cos(2.0 * x)) + cos(3.0 * x)) + cos(4.0 * x)) - 0.537212 * (-0.0384615 - 0.0676986 * (-0.5 + cos(x)) - 0.103347 * (-0.2 - 0.571429 * (-0.5 + cos(x)) + cos(2.0 * x)) - 0.206563 * (-0.1 - 0.226891 * (-0.5 + cos(x)) - 0.543424 * (-0.2 - 0.571429 * (-0.5 + cos(x)) +  cos(2.0 * x)) + cos(3.0 * x)) - 0.538006 * (-0.0588235 - 0.113769 * (-0.5 + cos(x)) -  0.210583 * (-0.2 - 0.571429 * (-0.5 + cos(x)) +  cos(2.0 * x)) -  0.539487 * (-0.1 - 0.226891 * (-0.5 + cos(x)) -  0.543424 * (-0.2 - 0.571429 * (-0.5 + cos(x)) + cos(2.0 * x)) + cos(3.0 * x)) + cos(4.0 * x)) + cos(5.0 * x)) + cos(6.0 * x)) - 0.536415 * (-0.02 - 0.0320166 * (-0.5 + cos(x)) - 0.0397866 * (-0.2 - 0.571429 * (-0.5 + cos(x)) + cos(2.0 * x)) - 0.0579348 * (-0.1 - 0.226891 * (-0.5 + cos(x)) - 0.543424 * (-0.2 - 0.571429 * (-0.5 + cos(x)) + cos(2.0 * x)) + cos(3.0 * x)) - 0.0983732 * (-0.0588235 - 0.113769 * (-0.5 + cos(x)) - 0.210583 * (-0.2 - 0.571429 * (-0.5 + cos(x)) + cos(2.0 * x)) - 0.539487 * (-0.1 - 0.226891 * (-0.5 + cos(x)) -  0.543424 * (-0.2 - 0.571429 * (-0.5 + cos(x)) +  cos(2.0 * x)) + cos(3.0 * x)) + cos(4.0 * x)) - 0.203683 * (-0.0384615 - 0.0676986 * (-0.5 + cos(x)) - 0.103347 * (-0.2 - 0.571429 * (-0.5 + cos(x)) + cos(2.0 * x)) -  0.206563 * (-0.1 - 0.226891 * (-0.5 + cos(x)) -  0.543424 * (-0.2 - 0.571429 * (-0.5 + cos(x)) +  cos(2.0 * x)) + cos(3.0 * x)) - 0.538006 * (-0.0588235 - 0.113769 * (-0.5 + cos(x)) -  0.210583 * (-0.2 - 0.571429 * (-0.5 + cos(x)) +  cos(2.0 * x)) -  0.539487 * (-0.1 - 0.226891 * (-0.5 + cos(x)) -  0.543424 * (-0.2 - 0.571429 * (-0.5 + cos(x)) + cos(2.0 * x)) + cos(3.0 * x)) + cos(4.0 * x)) + cos(5.0 * x)) - 0.53673 * (-0.027027 - 0.0449064 * (-0.5 + cos(x)) - 0.0605593 * (-0.2 - 0.571429 * (-0.5 + cos(x)) +  cos(2.0 * x)) - 0.100032 * (-0.1 - 0.226891 * (-0.5 + cos(x)) -  0.543424 * (-0.2 - 0.571429 * (-0.5 + cos(x)) +  cos(2.0 * x)) + cos(3.0 * x)) - 0.20472 * (-0.0588235 - 0.113769 * (-0.5 + cos(x)) -  0.210583 * (-0.2 - 0.571429 * (-0.5 + cos(x)) +  cos(2.0 * x)) -  0.539487 * (-0.1 - 0.226891 * (-0.5 + cos(x)) -  0.543424 * (-0.2 - 0.571429 * (-0.5 + cos(x)) + cos(2.0 * x)) + cos(3.0 * x)) + cos(4.0 * x)) - 0.537212 * (-0.0384615 - 0.0676986 * (-0.5 + cos(x)) -  0.103347 * (-0.2 - 0.571429 * (-0.5 + cos(x)) +  cos(2.0 * x)) -  0.206563 * (-0.1 - 0.226891 * (-0.5 + cos(x)) -  0.543424 * (-0.2 - 0.571429 * (-0.5 + cos(x)) + cos(2.0 * x)) + cos(3.0 * x)) -  0.538006 * (-0.0588235 - 0.113769 * (-0.5 + cos(x)) -  0.210583 * (-0.2 - 0.571429 * (-0.5 + cos(x)) + cos(2.0 * x)) -  0.539487 * (-0.1 - 0.226891 * (-0.5 + cos(x)) - 0.543424 * (-0.2 - 0.571429 * (-0.5 + cos(x)) + cos(2.0 * x)) + cos(3.0 * x)) + cos(4.0 * x)) +  cos(5.0 * x)) + cos(6.0 * x)) + cos(7.0 * x)) + cos(8.0 * x)) - 0.536039 * (-0.0121951 - 0.0187007 * (-0.5 + cos(x)) - 0.0211108 * (-0.2 - 0.571429 * (-0.5 + cos(x)) + cos(2.0 * x)) - 0.0265633 * (-0.1 - 0.226891 * (-0.5 + cos(x)) - 0.543424 * (-0.2 - 0.571429 * (-0.5 + cos(x)) + cos(2.0 * x)) + cos(3.0 * x)) - 0.0365565 * (-0.0588235 - 0.113769 * (-0.5 + cos(x)) - 0.210583 * (-0.2 - 0.571429 * (-0.5 + cos(x)) + cos(2.0 * x)) - 0.539487 * (-0.1 - 0.226891 * (-0.5 + cos(x)) - 0.543424 * (-0.2 - 0.571429 * (-0.5 + cos(x)) + cos(2.0 * x)) + cos(3.0 * x)) + cos(4.0 * x)) - 0.0556919 * (-0.0384615 - 0.0676986 * (-0.5 + cos(x)) - 0.103347 * (-0.2 - 0.571429 * (-0.5 + cos(x)) + cos(2.0 * x)) - 0.206563 * (-0.1 - 0.226891 * (-0.5 + cos(x)) - 0.543424 * (-0.2 - 0.571429 * (-0.5 + cos(x)) + cos(2.0 * x)) + cos(3.0 * x)) - 0.538006 * (-0.0588235 - 0.113769 * (-0.5 + cos(x)) - 0.210583 * (-0.2 - 0.571429 * (-0.5 + cos(x)) + cos(2.0 * x)) - 0.539487 * (-0.1 - 0.226891 * (-0.5 + cos(x)) -  0.543424 * (-0.2 - 0.571429 * (-0.5 + cos(x)) +  cos(2.0 * x)) + cos(3.0 * x)) + cos(4.0 * x)) + cos(5.0 * x)) - 0.0967804 * (-0.027027 - 0.0449064 * (-0.5 + cos(x)) - 0.0605593 * (-0.2 - 0.571429 * (-0.5 + cos(x)) + cos(2.0 * x)) - 0.100032 * (-0.1 - 0.226891 * (-0.5 + cos(x)) - 0.543424 * (-0.2 - 0.571429 * (-0.5 + cos(x)) + cos(2.0 * x)) + cos(3.0 * x)) - 0.20472 * (-0.0588235 - 0.113769 * (-0.5 + cos(x)) - 0.210583 * (-0.2 - 0.571429 * (-0.5 + cos(x)) + cos(2.0 * x)) - 0.539487 * (-0.1 - 0.226891 * (-0.5 + cos(x)) -  0.543424 * (-0.2 - 0.571429 * (-0.5 + cos(x)) +  cos(2.0 * x)) + cos(3.0 * x)) + cos(4.0 * x)) - 0.537212 * (-0.0384615 - 0.0676986 * (-0.5 + cos(x)) - 0.103347 * (-0.2 - 0.571429 * (-0.5 + cos(x)) + cos(2.0 * x)) - 0.206563 * (-0.1 - 0.226891 * (-0.5 + cos(x)) -  0.543424 * (-0.2 - 0.571429 * (-0.5 + cos(x)) +  cos(2.0 * x)) + cos(3.0 * x)) - 0.538006 * (-0.0588235 - 0.113769 * (-0.5 + cos(x)) -  0.210583 * (-0.2 - 0.571429 * (-0.5 + cos(x)) +  cos(2.0 * x)) -  0.539487 * (-0.1 - 0.226891 * (-0.5 + cos(x)) -  0.543424 * (-0.2 - 0.571429 * (-0.5 + cos(x)) + cos(2.0 * x)) + cos(3.0 * x)) + cos(4.0 * x)) + cos(5.0 * x)) + cos(6.0 * x)) - 0.202607 * (-0.02 - 0.0320166 * (-0.5 + cos(x)) - 0.0397866 * (-0.2 - 0.571429 * (-0.5 + cos(x)) + cos(2.0 * x)) - 0.0579348 * (-0.1 - 0.226891 * (-0.5 + cos(x)) - 0.543424 * (-0.2 - 0.571429 * (-0.5 + cos(x)) + cos(2.0 * x)) + cos(3.0 * x)) - 0.0983732 * (-0.0588235 - 0.113769 * (-0.5 + cos(x)) - 0.210583 * (-0.2 - 0.571429 * (-0.5 + cos(x)) + cos(2.0 * x)) - 0.539487 * (-0.1 - 0.226891 * (-0.5 + cos(x)) -  0.543424 * (-0.2 - 0.571429 * (-0.5 + cos(x)) +  cos(2.0 * x)) + cos(3.0 * x)) + cos(4.0 * x)) - 0.203683 * (-0.0384615 - 0.0676986 * (-0.5 + cos(x)) - 0.103347 * (-0.2 - 0.571429 * (-0.5 + cos(x)) + cos(2.0 * x)) - 0.206563 * (-0.1 - 0.226891 * (-0.5 + cos(x)) -  0.543424 * (-0.2 - 0.571429 * (-0.5 + cos(x)) +  cos(2.0 * x)) + cos(3.0 * x)) - 0.538006 * (-0.0588235 - 0.113769 * (-0.5 + cos(x)) -  0.210583 * (-0.2 - 0.571429 * (-0.5 + cos(x)) +  cos(2.0 * x)) -  0.539487 * (-0.1 - 0.226891 * (-0.5 + cos(x)) -  0.543424 * (-0.2 - 0.571429 * (-0.5 + cos(x)) + cos(2.0 * x)) + cos(3.0 * x)) + cos(4.0 * x)) + cos(5.0 * x)) - 0.53673 * (-0.027027 - 0.0449064 * (-0.5 + cos(x)) - 0.0605593 * (-0.2 - 0.571429 * (-0.5 + cos(x)) +  cos(2.0 * x)) - 0.100032 * (-0.1 - 0.226891 * (-0.5 + cos(x)) -  0.543424 * (-0.2 - 0.571429 * (-0.5 + cos(x)) +  cos(2.0 * x)) + cos(3.0 * x)) - 0.20472 * (-0.0588235 - 0.113769 * (-0.5 + cos(x)) -  0.210583 * (-0.2 - 0.571429 * (-0.5 + cos(x)) +  cos(2.0 * x)) -  0.539487 * (-0.1 - 0.226891 * (-0.5 + cos(x)) -  0.543424 * (-0.2 - 0.571429 * (-0.5 + cos(x)) + cos(2.0 * x)) + cos(3.0 * x)) + cos(4.0 * x)) - 0.537212 * (-0.0384615 - 0.0676986 * (-0.5 + cos(x)) - 0.103347 * (-0.2 - 0.571429 * (-0.5 + cos(x)) +  cos(2.0 * x)) -  0.206563 * (-0.1 - 0.226891 * (-0.5 + cos(x)) -  0.543424 * (-0.2 - 0.571429 * (-0.5 + cos(x)) + cos(2.0 * x)) + cos(3.0 * x)) -  0.538006 * (-0.0588235 - 0.113769 * (-0.5 + cos(x)) -  0.210583 * (-0.2 - 0.571429 * (-0.5 + cos(x)) + cos(2.0 * x)) -  0.539487 * (-0.1 - 0.226891 * (-0.5 + cos(x)) - 0.543424 * (-0.2 - 0.571429 * (-0.5 + cos(x)) + cos(2.0 * x)) + cos(3.0 * x)) + cos(4.0 * x)) +  cos(5.0 * x)) + cos(6.0 * x)) + cos(7.0 * x)) - 0.536196 * (-0.0153846 - 0.024015 * (-0.5 + cos(x)) - 0.0282147 * (-0.2 - 0.571429 * (-0.5 + cos(x)) + cos(2.0 * x)) - 0.0377136 * (-0.1 - 0.226891 * (-0.5 + cos(x)) - 0.543424 * (-0.2 - 0.571429 * (-0.5 + cos(x)) + cos(2.0 * x)) + cos(3.0 * x)) - 0.056539 * (-0.0588235 - 0.113769 * (-0.5 + cos(x)) - 0.210583 * (-0.2 - 0.571429 * (-0.5 + cos(x)) + cos(2.0 * x)) - 0.539487 * (-0.1 - 0.226891 * (-0.5 + cos(x)) -  0.543424 * (-0.2 - 0.571429 * (-0.5 + cos(x)) +  cos(2.0 * x)) + cos(3.0 * x)) + cos(4.0 * x)) - 0.0974017 * (-0.0384615 - 0.0676986 * (-0.5 + cos(x)) - 0.103347 * (-0.2 - 0.571429 * (-0.5 + cos(x)) + cos(2.0 * x)) - 0.206563 * (-0.1 - 0.226891 * (-0.5 + cos(x)) -  0.543424 * (-0.2 - 0.571429 * (-0.5 + cos(x)) +  cos(2.0 * x)) + cos(3.0 * x)) - 0.538006 * (-0.0588235 - 0.113769 * (-0.5 + cos(x)) -  0.210583 * (-0.2 - 0.571429 * (-0.5 + cos(x)) +  cos(2.0 * x)) -  0.539487 * (-0.1 - 0.226891 * (-0.5 + cos(x)) -  0.543424 * (-0.2 - 0.571429 * (-0.5 + cos(x)) + cos(2.0 * x)) + cos(3.0 * x)) + cos(4.0 * x)) + cos(5.0 * x)) - 0.203037 * (-0.027027 - 0.0449064 * (-0.5 + cos(x)) - 0.0605593 * (-0.2 - 0.571429 * (-0.5 + cos(x)) +  cos(2.0 * x)) - 0.100032 * (-0.1 - 0.226891 * (-0.5 + cos(x)) -  0.543424 * (-0.2 - 0.571429 * (-0.5 + cos(x)) +  cos(2.0 * x)) + cos(3.0 * x)) - 0.20472 * (-0.0588235 - 0.113769 * (-0.5 + cos(x)) -  0.210583 * (-0.2 - 0.571429 * (-0.5 + cos(x)) +  cos(2.0 * x)) -  0.539487 * (-0.1 - 0.226891 * (-0.5 + cos(x)) -  0.543424 * (-0.2 - 0.571429 * (-0.5 + cos(x)) + cos(2.0 * x)) + cos(3.0 * x)) + cos(4.0 * x)) -  0.537212 * (-0.0384615 - 0.0676986 * (-0.5 + cos(x)) -  0.103347 * (-0.2 - 0.571429 * (-0.5 + cos(x)) +  cos(2.0 * x)) -  0.206563 * (-0.1 - 0.226891 * (-0.5 + cos(x)) -  0.543424 * (-0.2 - 0.571429 * (-0.5 + cos(x)) + cos(2.0 * x)) + cos(3.0 * x)) -  0.538006 * (-0.0588235 - 0.113769 * (-0.5 + cos(x)) -  0.210583 * (-0.2 - 0.571429 * (-0.5 + cos(x)) + cos(2.0 * x)) -  0.539487 * (-0.1 - 0.226891 * (-0.5 + cos(x)) - 0.543424 * (-0.2 - 0.571429 * (-0.5 + cos(x)) + cos(2.0 * x)) + cos(3.0 * x)) + cos(4.0 * x)) +  cos(5.0 * x)) + cos(6.0 * x)) - 0.536415 * (-0.02 - 0.0320166 * (-0.5 + cos(x)) - 0.0397866 * (-0.2 - 0.571429 * (-0.5 + cos(x)) +  cos(2.0 * x)) - 0.0579348 * (-0.1 - 0.226891 * (-0.5 + cos(x)) -  0.543424 * (-0.2 - 0.571429 * (-0.5 + cos(x)) +  cos(2.0 * x)) + cos(3.0 * x)) - 0.0983732 * (-0.0588235 - 0.113769 * (-0.5 + cos(x)) -  0.210583 * (-0.2 - 0.571429 * (-0.5 + cos(x)) +  cos(2.0 * x)) -  0.539487 * (-0.1 - 0.226891 * (-0.5 + cos(x)) -  0.543424 * (-0.2 - 0.571429 * (-0.5 + cos(x)) + cos(2.0 * x)) + cos(3.0 * x)) + cos(4.0 * x)) -  0.203683 * (-0.0384615 - 0.0676986 * (-0.5 + cos(x)) -  0.103347 * (-0.2 - 0.571429 * (-0.5 + cos(x)) +  cos(2.0 * x)) -  0.206563 * (-0.1 - 0.226891 * (-0.5 + cos(x)) -  0.543424 * (-0.2 - 0.571429 * (-0.5 + cos(x)) + cos(2.0 * x)) + cos(3.0 * x)) -  0.538006 * (-0.0588235 - 0.113769 * (-0.5 + cos(x)) -  0.210583 * (-0.2 - 0.571429 * (-0.5 + cos(x)) + cos(2.0 * x)) -  0.539487 * (-0.1 - 0.226891 * (-0.5 + cos(x)) - 0.543424 * (-0.2 - 0.571429 * (-0.5 + cos(x)) + cos(2.0 * x)) + cos(3.0 * x)) + cos(4.0 * x)) +  cos(5.0 * x)) - 0.53673 * (-0.027027 - 0.0449064 * (-0.5 + cos(x)) -  0.0605593 * (-0.2 - 0.571429 * (-0.5 + cos(x)) +  cos(2.0 * x)) -  0.100032 * (-0.1 - 0.226891 * (-0.5 + cos(x)) -  0.543424 * (-0.2 - 0.571429 * (-0.5 + cos(x)) + cos(2.0 * x)) + cos(3.0 * x)) -  0.20472 * (-0.0588235 - 0.113769 * (-0.5 + cos(x)) -  0.210583 * (-0.2 - 0.571429 * (-0.5 + cos(x)) + cos(2.0 * x)) -  0.539487 * (-0.1 - 0.226891 * (-0.5 + cos(x)) - 0.543424 * (-0.2 - 0.571429 * (-0.5 + cos(x)) + cos(2.0 * x)) + cos(3.0 * x)) + cos(4.0 * x)) - 0.537212 * (-0.0384615 - 0.0676986 * (-0.5 + cos(x)) -  0.103347 * (-0.2 - 0.571429 * (-0.5 + cos(x)) + cos(2.0 * x)) -  0.206563 * (-0.1 - 0.226891 * (-0.5 + cos(x)) - 0.543424 * (-0.2 - 0.571429 * (-0.5 + cos(x)) + cos(2.0 * x)) + cos(3.0 * x)) -  0.538006 * (-0.0588235 - 0.113769 * (-0.5 + cos(x)) - 0.210583 * (-0.2 - 0.571429 * (-0.5 + cos(x)) + cos(2.0 * x)) - 0.539487 * (-0.1 - 0.226891 * (-0.5 + cos(x)) - 0.543424 * (-0.2 - 0.571429 * (-0.5 + cos(x)) + cos(2.0 * x)) + cos(3.0 * x)) + cos(4.0 * x)) +  cos(5.0 * x)) + cos(6.0 * x)) + cos(7.0 * x)) + cos(8.0 * x)) + cos(9.0 * x)) + cos(10.0 * x));
    }
    return(res);
  }

  double psijbar103(int orthofam, int j, double *z, int lenz) {
    double psi103(int orthofam, int j, double x);
    int i;
    double psijbar = 0.0;
    for (i = 0; i < lenz; i++) {
      psijbar = psijbar + psi103(orthofam, j, z[i]);
    }
    psijbar = psijbar / (double)lenz;
  
    return(psijbar);
  }

  double phi103(double z) {
    return(0.5 * exp(-fabs(z)));
  }

  // Integrand of c_S(a)
  // Here, n is NOT the sample size, it's here because
  // Rdqagi needs a vector-valued function as input
  void integrandfn103(double *z, int n, void *ex) {
    // n is the length of z
    double phi103(double z);
    double psi103(int orthofam, int j, double x);
    int i, j;
   
    int orthofam = (int)((double*)ex)[0];
    int cardS = (int)((double*)ex)[1];
    int *S;
    S = new int[cardS];
    for (i = 0; i < cardS; i++) S[i] = (int)((double*)ex)[i + 2];
    double *a;
    a = new double[cardS];
    for (i = 0; i < cardS; i++) a[i] = ((double*)ex)[i + cardS + 2];
    double sum;
    for (i = 0; i < n; i++) { // n is NOT the sample size
      sum  = 0.0;
      for (j = 0; j < cardS; j++) {
	sum = sum + a[j] * psi103(orthofam, S[j], z[i]);
      }
      z[i] = exp(sum) * phi103(z[i]);
    }
    delete[] S;
    delete[] a;
    
    return;
  }

  void integrandfnprime103(double *z, int n, void *ex) {
    // n is the length of z
    double phi103(double z);
    double psi103(int orthofam, int j, double x);
    int i, k;
  
    int orthofam = (int)((double*)ex)[0];
    int cardS = (int)((double*)ex)[1];
    int *S;
    S = new int[cardS];
    for (i = 0; i < cardS; i++) S[i] = (int)((double*)ex)[i + 2];
    double *a;
    a = new double[cardS];
    for (i = 0; i < cardS; i++) a[i] = ((double*)ex)[i + cardS + 2];
    int j = (int)((double*)ex)[2 * cardS + 2];
    double sum;
    for (i = 0; i < n; i++) { // n is NOT the sample size
      sum  = 0.0;
      for (k = 0; k < cardS; k++) {
	sum = sum + a[k] * psi103(orthofam, S[k], z[i]);
      }
      z[i] = psi103(orthofam, S[j], z[i]) * exp(sum) * phi103(z[i]);
    }
    delete[] S;
    delete[] a;
    
    return;
  }
  
  double Csa103(int cardS, double *a, void *ex) {
    int i;
    void integrandfn103(double *x, int n, void *ex); 
    void Rdqagi(integr_fn f, void *ex, double *bound, int *inf,
		double *epsabs, double *epsrel,
		double *result, double *abserr, int *neval, int *ier,
		int *limit, int *lenw, int *last,
		int *iwork, double *work);
    double res;

    double *exin;
    exin = new double[2 * cardS + 2]; 
    exin[0] = ((double*)ex)[0]; // orthofam
    exin[1] = (double)cardS;
    for (i = 1; i <= cardS; i++) exin[i + 1] = ((double*)ex)[i]; // S
    for (i = 1; i <= cardS; i++) exin[i + cardS + 1] = a[i - 1];
    double *bound;  
    bound = new double[1];
    bound[0] = 0.0; // finite bound of the integration (has no meaning if interval is doubly-infinite)
    int *inf;  /* indicating the kind of integration range involved
                    inf = 1 corresponds to  (bound,+infinity),
                    inf = -1            to  (-infinity,bound),
                    inf = 2             to (-infinity,+infinity) */
    inf = new int[1];
    inf[0] = 2; // corresponds to (-Inf, +Inf)
    double *epsabs;  
    epsabs = new double[1];
    epsabs[0] = 0.0001220703; // absolute accuracy requested
    double *epsrel;  
    epsrel = new double[1];
    epsrel[0] = 0.0001220703; // relative accuracy requested
    /* if  epsabs <= 0
                    and epsrel < max(50*rel.mach.acc.,0.5d-28),
                    the routine will end with ier = 6
    */
    double *result;  
    result = new double[1]; // output value (final estimate of the integral)
    double *abserr;  
    abserr = new double[1]; // output abs.err (estimate of the modulus of the absolute error)
    int *neval;  
    neval = new int[1]; // number of integrand function evaluations
    int *ier;  
    ier = new int[1]; /* "OK",
	       "maximum number of subdivisions reached",
	       "roundoff error was detected",
	       "extremely bad integrand behaviour",
	       "roundoff error is detected in the extrapolation table",
	       "the integral is probably divergent",
	       "the input is invalid" */
    /*
                    ier = 0 normal and reliable termination of the
                            routine. it is assumed that the requested
                            accuracy has been achieved.
                  - ier > 0 abnormal termination of the routine. the
                            estimates for result and error are less
                            reliable. it is assumed that the requested
                            accuracy has not been achieved.
           error messages
                    ier = 1 maximum number of subdivisions allowed
                            has been achieved. one can allow more
                            subdivisions by increasing the value of
                            limit (and taking the according dimension
                            adjustments into account). however, if
                            this yields no improvement it is advised
                            to analyze the integrand in order to
                            determine the integration difficulties. if
                            the position of a local difficulty can be
                            determined (e.g. singularity,
                            discontinuity within the interval) one
                            will probably gain from splitting up the
                            interval at this point and calling the
                            integrator on the subranges. if possible,
                            an appropriate special-purpose integrator
                            should be used, which is designed for
                            handling the type of difficulty involved.
                        = 2 the occurrence of roundoff error is
                            detected, which prevents the requested
                            tolerance from being achieved.
                            the error may be under-estimated.
                        = 3 extremely bad integrand behaviour occurs
                            at some points of the integration
                            interval.
                        = 4 the algorithm does not converge.
                            roundoff error is detected in the
                            extrapolation table.
                            it is assumed that the requested tolerance
                            cannot be achieved, and that the returned
                            result is the best which can be obtained.
                        = 5 the integral is probably divergent, or
                            slowly convergent. it must be noted that
                            divergence can occur with any other value
                            of ier.
                        = 6 the input is invalid, because
                            (epsabs <= 0 and
                             epsrel < max(50*rel.mach.acc.,0.5d-28))
                             or limit < 1 or leniw < limit*4.
                            result, abserr, neval, last are set to
                            zero. exept when limit or leniw is
                            invalid, iwork(1), work(limit*2+1) and
                            work(limit*3+1) are set to zero, work(1)
                            is set to a and work(limit+1) to b.

    */
    int *limit;  
    limit = new int[1];
    limit[0] = 10000;
    /*
                   dimensioning parameter for iwork
                   limit determines the maximum number of subintervals
                   in the partition of the given integration interval
                   (a,b), limit >= 1.
                   if limit < 1, the routine will end with ier = 6.

    */
    int *lenw;  
    lenw = new int[1];
    lenw[0] = 4 * limit[0];
    /*
                   dimensioning parameter for work
                   lenw must be at least limit*4.
                   if lenw < limit*4, the routine will end
                   with ier = 6.

    */
    int *last;  
    last = new int[1];
    /*
  output subdivisions (number of subintervals produced in the subdivision process, which
                   determines the number of significant elements
                   actually in the work arrays.)
    */
    int *iwork;  
    iwork = new int[limit[0]];
    /*
                   vector of dimension at least limit, the first
                   k elements of which contain pointers
                   to the error estimates over the subintervals,
                   such that work(limit*3+iwork(1)),... ,
                   work(limit*3+iwork(k)) form a decreasing
                   sequence, with k = last if last <= (limit/2+2), and
                   k = limit+1-last otherwise

    */
    double *work;  
    work = new double[lenw[0]];
    /*
                  vector of dimension at least lenw
                   on return
                   work(1), ..., work(last) contain the left
                    end points of the subintervals in the
                    partition of (a,b),
                   work(limit+1), ..., work(limit+last) contain
                    the right end points,
                   work(limit*2+1), ...,work(limit*2+last) contain the
                    integral approximations over the subintervals,
                   work(limit*3+1), ..., work(limit*3)
                    contain the error estimates.

    */

    
 // see https://cran.r-project.org/doc/manuals/r-release/R-exts.html#Integration
 // https://github.com/SurajGupta/r-source/blob/master/src/library/stats/R/integrate.R
    // https://github.com/krlmlr/cxxr/blob/master/src/appl/integrate.c
    Rdqagi(integrandfn103, exin, bound, inf, epsabs, epsrel, result, abserr, neval, ier,
	   limit, lenw, last, iwork, work);
    
    double bignumber = R_pow(10.0, 30.0);
    if (ier[0] > 0) res = 1.797693 * R_pow(10.0, 308.0); else res = result[0];
    if (fabs(res) >= bignumber) res = bignumber;

    
    delete[] exin;
    delete[] bound;
    delete[] inf;
    delete[] epsabs;
    delete[] epsrel;
    delete[] result;
    delete[] abserr;
    delete[] neval;
    delete[] ier;
    delete[] limit;
    delete[] lenw;
    delete[] last;
    delete[] iwork;
    delete[] work;


    
    return res;

  }

  double Csaprime103(int j, int cardS, double *a, void *ex) {
    int i;
    void integrandfnprime103(double *x, int n, void *ex); 
    void Rdqagi(integr_fn f, void *ex, double *bound, int *inf,
		double *epsabs, double *epsrel,
		double *result, double *abserr, int *neval, int *ier,
		int *limit, int *lenw, int *last,
		int *iwork, double *work);
    double res;
    double *exin;
    exin = new double[2 * cardS + 3]; 
    exin[0] = ((double*)ex)[0]; // orthofam
    exin[1] = (double)cardS;
    for (i = 1; i <= cardS; i++) exin[i + 1] = ((double*)ex)[i]; // S
    for (i = 1; i <= cardS; i++) exin[i + cardS + 1] = a[i - 1];
    exin[2 * cardS + 2] = (double)j; // j
    double *bound;  
    bound = new double[1];
    bound[0] = 0.0;
    int *inf;  
    inf = new int[1];
    inf[0] = 2;
    double *epsabs;  
    epsabs = new double[1];
    epsabs[0] = 0.0001220703;
    double *epsrel;  
    epsrel = new double[1];
    epsrel[0] = 0.0001220703;
    double *result;  
    result = new double[1];
    double *abserr;  
    abserr = new double[1];
    int *neval;  
    neval = new int[1];
    int *ier;  
    ier = new int[1];
    int *limit;  
    limit = new int[1];
    limit[0] = 10000;
    int *lenw;  
    lenw = new int[1];
    lenw[0] = 4 * limit[0];
    int *last;  
    last = new int[1];
    int *iwork;  
    iwork = new int[limit[0]];
    double *work;  
    work = new double[lenw[0]];


    
 // see https://cran.r-project.org/doc/manuals/r-release/R-exts.html#Integration
    Rdqagi(integrandfnprime103, exin, bound, inf, epsabs, epsrel, result, abserr, neval, ier,
	   limit, lenw, last, iwork, work);


    double bignumber = R_pow(10.0, 30.0);
    if (ier[0] > 0) res = 1.797693 * R_pow(10.0, 308.0); else res = result[0];
    if (fabs(res) >= bignumber) res = bignumber;
    
    delete[] exin;
    delete[] bound;
    delete[] inf;
    delete[] epsabs;
    delete[] epsrel;
    delete[] result;
    delete[] abserr;
    delete[] neval;
    delete[] ier;
    delete[] limit;
    delete[] lenw;
    delete[] last;
    delete[] iwork;
    delete[] work;

    return res;

  }

  void ahatS103(double *ahat, double *zhat, int n, int *S, int cardS, int orthofam) {
    double psijbar103(int orthofam, int j, double *z, int lenz);
    typedef double optimfn(int n, double *par, void *ex);
    typedef void optimgr(int n, double *par, double *gr, void *ex);
    // https://github.com/SurajGupta/r-source/blob/master/src/appl/optim.c
    // https://fossies.org/linux/R/src/appl/lbfgsb.c
    void lbfgsb(int n, int lmm, double *x, double *lower,
		double *upper, int *nbd, double *Fmin, optimfn fn,
		optimgr gr, int *fail, void *ex, double factr,
		double pgtol, int *fncount, int *grcount,
		int maxit, char *msg, int trace, int nREPORT);
    double Csa103(int cardS, double *a, void *ex);
    int i, j;
    
    int lmm = 5; // maximum number of variable metric corrections allowed in the limited memory matrix.
    
    double *ain;
    ain = new double[cardS];
    for (i = 0; i < cardS; i++) ain[i] = 0.0;  // initial values (will be modified at output)
    
    double *loweroptim, *upperoptim;
    loweroptim  = new double[cardS];
    upperoptim = new double[cardS];
    int maxS = S[0]; for (i = 1; i < cardS; i++) {if (S[i] > maxS) maxS = S[i];}
    double mepsilon = -0.001;
    double mepsilon2 = -0.00001;
    double tmp = M_SQRT2 + mepsilon2;
    for (i = 0; i < cardS; i++) {
      loweroptim[i] = R_NegInf; // lower bounds
      upperoptim[i] = R_PosInf; // upper bounds
    }
    
    int *nbdoptim;
    nbdoptim = new int[cardS];
    for (i = 0; i < cardS; i++) nbdoptim[i] = 0; // 0 if x(i) is unbounded, 1 if x(i) has only a lower bound, 2 if x(i) has both lower and upper bounds, 3 if x(i) has only an upper bound.
    if (orthofam == 1) {
      if (maxS > 1) {
	for (j = 0; j < cardS; j++) {
	  if (S[j] == maxS)  {
	    upperoptim[j] = mepsilon;
	    nbdoptim[j] = 3;
	  }
	}
      } else {
	upperoptim[0] = tmp;
	loweroptim[0] = -tmp;
	nbdoptim[0] = 2;
      }
    }

    double *Fminoptim;
    Fminoptim = new double[1];
    Fminoptim[0] = 0.0; // To display some information during the calculations
    
    double minusCHZwon103(int cardS, double *a, void *ex);
    
    void minusgrCHZwon103(int n, double *par, double *gr, void *ex);
    
    int *failoptim;
    failoptim = new int[1];
    failoptim[0] = 0; // Contains some value indicating what kind of error occured
    
    double *exin; // values to be passed to the functions minusCHZwon103 and minusgrCHZwon103
    exin = new double[n + cardS + 2];
    exin[0] = (double)orthofam;
    exin[1] = (double)n;
    for (i = 0; i < n; i++) exin[i + 2] = zhat[i];
    for (i = 0; i < cardS; i++) exin[n + i + 2] = (double)S[i];
    
    
    double factroptim = 100000000.0; // decrease for higher accuracy. The iteration will stop when (f^k - f^{k+1})/max{|f^k|,|f^{k+1}|,1} <= factr*epsmch where epsmch is the machine precision. Typical values for factr: 1.d+12 for low accuracy; 1.d+7 for moderate accuracy; 1.d+1 for extremely high accuracy.
    
    double pgtoloptim = 0.0; // The iteration will stop when max{|proj g_i | i = 1, ..., n} <= pgtol where pg_i is the ith component of the projected gradient.
    
    int *fncountoptim; // output
    fncountoptim = new int[1];
    fncountoptim[0] = 0; 
    
    int *grcountoptim; // output
    grcountoptim = new int[1];
    grcountoptim[0] = 0;
    
    int maxitoptim = 100;
    
    char *msgoptim;
    msgoptim = new char[60]; // Always 60.
    msgoptim[0] = '\0'; // Used to display some messages durint the calculations.
    
    int traceoptim = 0; // To display (if = 1) some information (See below) during the calculations
    // If 2: iprint = 0
    // If 3: iprint = nREPORT (voir plus bas)
    // If 4: iprint = 99
    // If 5: iprint = 100
    // If 6: iprint = 101
    // Sinon: iprint = -1
    //        iprint is an integer variable that must be set by the user.
    // 	 It controls the frequency and type of output generated:
    // 	  iprint<0    no output is generated;
    // 	  iprint=0    print only one line at the last iteration;
    // 	  0<iprint<99 print also f and |proj g| every iprint iterations;
    // 	  iprint=99   print details of every iteration except n-vectors;
    // 	  iprint=100  print also the changes of active set and final x;
    // 	  iprint>100  print details of every iteration including x and g;
    // 	 When iprint > 0, the file iterate.dat will be created to
    // 			  summarize the iteration.
    int nREPORToptim = 10;
    
		
    //        This subroutine solves bound (or not) constrained optimization problems by
    //	 using the compact formula of the limited memory BFGS updates.
    // This computes argmax_a Cs(a):
    lbfgsb(cardS, lmm, ain, // cardS is for number of variables
	   loweroptim, upperoptim, nbdoptim, Fminoptim, minusCHZwon103, minusgrCHZwon103,
	   failoptim, exin, factroptim, pgtoloptim, fncountoptim,
	   grcountoptim, maxitoptim, msgoptim,
	   traceoptim, nREPORToptim);

    
    for (i = 0; i < cardS; i++) ahat[i] = ain[i];
    // Now ahat contains \hat{a}_S
    

    
    /*
      https://github.com/microsoft/microsoft-r-open/blob/master/source/src/appl/optim.c
      https://github.com/SurajGupta/r-source/blob/master/src/library/stats/src/optim.c
      
      void nmmin(int n, double *xin, double *x, double *Fmin, optimfn fn,
      int *fail, double abstol, double intol, void *ex,
      double alpha, double beta, double gamma, int trace,
      int *fncount, int maxit); 
      
      See ?optim in R
      
      n:       input. Number of parameters (npar = length(par))
      xin:     input. Starting parameters on entry (dpar of length = npar; dpar = par / parscale)
      x:       output. Final parameters on exit (opar of length = npar)
      Fmin:    output. Final value returned.
      fn:      input. A function to be minimized (or maximized), with first
      argument the vector of parameters over which minimization is
      to take place.  It should return a scalar result.
      fail:    output if failure
      abstol:  input, default = -Inf. The absolute convergence tolerance. Only useful for
      non-negative functions, as a tolerance for reaching zero.
      intol:   input, default = sqrt(.Machine$double.eps). Relative convergence tolerance.  The algorithm stops if
      it is unable to reduce the value by a factor of 'intol * (abs(val) + intol)’ at a step.
      ex:      input. Extra parameters for fn.
      alpha:   input, default = 1. Scaling parameter (reflection factor)
      beta:    input, default = 0.5. Scaling parameter (contraction factor)
      gamma:   input, default = 2. Scaling parameter (expansion factor)
      trace:   input; default = 0. Non-negative integer. If positive, tracing information on
      the progress of the optimization is produced. Higher values
      may produce more tracing information
      fncount: output
      maxit:   input; default = 500. The maximum number of iterations.
      
      nmmin(cardS, ain, ahatS, Fmin, minusCsa103,
      fail, abstol, intol, exin,
      alpha, beta, gamma, trace,
      fncount, maxit);*/
    
    
    
    delete[] ain;
    delete[] loweroptim;
    delete[] upperoptim;
    delete[] nbdoptim;
    delete[] Fminoptim;
    delete[] failoptim;
    delete[] exin;
    delete[] fncountoptim;
    delete[] grcountoptim;
    delete[] msgoptim;
    
    return;
     
  }
  
  double CHZwon103(int cardS, double *a, void *ex) {
    double psijbar103(int orthofam, int j, double *z, int lenz);
    double Csa103(int cardS, double *a, void *ex);
    int i, j;   
    int orthofam = (int)((double*)ex)[0];
    int n = (int)((double*)ex)[1];
    double *zhat;
    zhat = new double[n];
    for (i = 0; i < n; i++) zhat[i] = ((double*)ex)[i + 2];
    int *S;
    S = new int[cardS];
    for (i = 0; i < cardS; i++) S[i] = (int)((double*)ex)[n + i + 2];
   
    double *exin;
    exin = new double[cardS + 1];
    exin[0] = (double)orthofam;
    for (i = 1; i <= cardS; i++) exin[i] = (double)S[i - 1];


    double sum = 0.0;
    for (j = 0; j < cardS; j++) {
      sum = sum + a[j] * psijbar103(orthofam, S[j], zhat, n);
    }

    double tmp = Csa103(cardS, a, exin);

    delete[] zhat;
    delete[] S;
    delete[] exin;


    double bignumber = R_pow(10.0, 30.0);
    if ((fabs(tmp) >= bignumber) || (fabs(sum) >= bignumber)) return(-bignumber); // CHANGED HERE!
    sum = sum - log(tmp);

    return(sum);
  }

  double minusCHZwon103(int cardS, double *a, void *ex) {
    double CHZwon103(int cardS, double *a, void *ex);
    return -CHZwon103(cardS, a, ex); // because we want to maximise and nmmin minimises
}

  
  void minusgrCHZwon103(int cardS, double *a, double *gr, void *ex) {
    double psijbar103(int orthofam, int j, double *z, int lenz);
    double Csa103(int cardS, double *a, void *ex);
    double Csaprime103(int j, int cardS, double *a, void *ex);

    int i, j;
    int orthofam = (int)((double*)ex)[0];
    int n = (int)((double*)ex)[1];
    double *zhat;
    zhat = new double[n];
    for (i = 0; i < n; i++) zhat[i] = ((double*)ex)[i + 2];
    int *S;
    S = new int[cardS];
    for (i = 0; i < cardS; i++) S[i] = (int)((double*)ex)[n + i + 2];
    
    double *exin;
    exin = new double[cardS + 1];
    exin[0] = (double)orthofam;
    for (i = 1; i <= cardS; i++) exin[i] = (double)S[i - 1];
    
    double tmp = Csa103(cardS, a, exin);
    double bignumber = R_pow(10.0, 30.0);

    double tmp2, tmp3;
    for (j = 0; j < cardS; j++) {
      tmp2 = Csaprime103(j, cardS, a, exin);
      tmp3 = -psijbar103(orthofam, S[j], zhat, n) + tmp2 / tmp; // CHANGED HERE!!
      if (fabs(tmp3) >= bignumber) gr[j] = -bignumber; else gr[j] = tmp3; // CHANGED HERE||
    }
    
    delete[] zhat;
    delete[] S;
    delete[] exin;

    return;
  }
  
  double CHZ103(double *zhat, int n, int *S, int cardS, int orthofam) {
    void ahatS103(double *ahat, double *zhat, int n, int *S, int cardS, int orthofam);
    double CHZwon103(int cardS, double *a, void *ex);
    double res;
    int i;
    
    double *ahat;
    ahat = new double[cardS];

    ahatS103(ahat, zhat, n, S, cardS, orthofam);
    
    double *exin;
    exin = new double[n + cardS + 2];
    exin[0] = (double)orthofam;
    exin[1] = (double)n;
    for (i = 0; i < n; i++) exin[i + 2] = zhat[i];
    for (i = 0; i < cardS; i++) exin[n + i + 2] = (double)S[i];

    res = CHZwon103(cardS, ahat, exin);
    res = (double)(2 * n) * res;

    delete[] ahat;
    delete[] exin;
    
    return(res);
  }

  double CHT103(double *zhat, int n, int *S, int cardS, int orthofam) {
    double psijbar103(int orthofam, int j, double *z, int lenz);
    int j;
    double sum = 0.0;
    for (j = 0; j < cardS; j++) {
      sum = sum + R_pow(psijbar103(orthofam, S[j], zhat, n), 2.0);
    }
    sum = (double)n * sum;
    return(sum);
  }

  
}

  
