% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/rdmc_tune.R
\name{rdmc_tune}
\alias{rdmc_tune}
\title{Robust discrete matrix completion with hyperparameter tuning}
\usage{
rdmc_tune(
  X,
  values = NULL,
  lambda = fraction_grid(),
  relative = TRUE,
  splits = holdout_control(),
  loss = c("pseudo_huber", "absolute", "truncated"),
  loss_const = NULL,
  ...
)
}
\arguments{
\item{X}{a matrix or data frame of discrete ratings with missing values.}

\item{values}{an optional numeric vector giving the possible values of the 
ratings.  Currently, these are assumed to be the same for all columns.  If 
\code{NULL}, the unique values of the observed parts of \code{X} are used.}

\item{lambda}{a numeric vector giving values of the regularization 
parameter.  See \code{\link{fraction_grid}()} for the default values.}

\item{relative}{a logical indicating whether the values of the 
regularization parameter should be considered relative to a certain 
reference value computed from the data at hand.  If \code{TRUE} (the 
default), the values of \code{lambda} are multiplied with the largest 
singular value of the median-centered data matrix with missing values 
replaced by zeros.}

\item{splits}{an object inheriting from class \code{"split_control"}, as 
generated by \code{\link{holdout_control}()} for repeated holdout validation 
or \code{\link{cv_folds_control}()} for \eqn{K}-fold cross-validation, or a 
list of index vectors giving different validation sets of observed cells as 
generated by \code{\link{create_splits}()}.  Cells in the validation set
will be set to \code{NA} for fitting the algorithm with the training set of 
observed cells.}

\item{loss}{a character string specifying the robust loss function for the 
loss part of the objective function.  Possible values are 
\code{"pseudo_huber"} (the default) for the pseudo-Huber loss, 
\code{"absolute"} for the absolute loss, and \code{"truncated"} for the 
truncated absolute loss.  See \sQuote{Details} for more information.}

\item{loss_const}{tuning constant for the loss function.  For the 
pseudo-Huber loss, the default value is the average step size between the 
rating categories in \code{values}.  For the truncated absolute loss, 
the default is half the range of the rating categories in \code{values}.  
This is ignored for the absolute loss, which does not have a tuning 
parameter.  See \sQuote{Details} for more information.}

\item{\dots}{additional arguments to be passed down to \code{\link{rdmc}()}.}
}
\value{
An object of class \code{"rdmc_tuned"} with the following components: 
\item{lambda}{a numeric vector containing the values of the regularization 
parameter.}
\item{tuning_loss}{a numeric vector containing the (average) values of the 
loss function on the validation set(s) for each value of the regularization 
parameter.}
\item{lambda_opt}{numeric; the optimal value of the regularization 
parameter.}
\item{fit}{an object of class \code{"\link{rdmc}"} containing the results 
from the algorithm with the optimal regularization parameter on the full 
(observed) data matrix.}

The class structure is still experimental and may change in the future. 
The following accessor functions are available:
\itemize{
  \item \code{\link{get_completed}()} to extract the completed (i.e., 
  imputed) data matrix with the optimal value of the regularization 
  parameter,
  \item \code{\link{get_lambda}()} to extract the optimal value of the 
  regularization parameter,
  \item \code{\link{get_nb_iter}()} to extract the number of iterations with 
  the optimal value of the regularization parameter.
}
}
\description{
Perform robust discrete matrix completion with a low-rank constraint on a 
latent continuous matrix, implemented via an ADMM algorithm.  The 
regularization parameter is thereby selected via repeated holdout validation 
or cross-validation.
}
\details{
For the loss part of the objective function, the pseudo-Huber loss 
(\code{loss = "pseudo_huber"}) is given by
\deqn{\rho(x) = \code{loss\_const}^2 (\sqrt{1 + (x/\code{loss\_const})^2} - 1).}{rho(x) = \code{loss_const}^2 (sqrt(1 + (x/\code{loss_const})^2) - 1).}
The absolute loss
(\code{loss = "absolute"}) is given by
\deqn{\rho(x) = |x|,}{rho(x) = |x|,}
and the truncated absolute loss (\code{loss = "truncated"}) is defined as
\deqn{\rho(x) = \min (|x|, \code{loss\_const}).}{rho(x) = min(|x|, \code{loss_const}).}
}
\examples{
# toy example derived from MovieLens 100K dataset
data("MovieLensToy")
# robust discrete matrix completion with hyperparameter tuning
set.seed(20250723)
fit <- rdmc_tune(MovieLensToy, 
                 lambda = fraction_grid(nb_lambda = 6),
                 splits = holdout_control(R = 5))
# extract completed matrix with optimal regularization parameter
X_hat <- get_completed(fit)
head(X_hat)
# extract optimal value of regularization parameter
get_lambda(fit)
# extract number of iterations with optimal regularization parameter
get_nb_iter(fit)

}
\references{
Archimbaud, A., Alfons, A., and Wilms, I. (2025) Robust Matrix Completion 
for Discrete Rating-Scale Data. arXiv:2412.20802. 
\doi{10.48550/arXiv.2412.20802}.
}
\seealso{
\code{\link{rdmc}()}, \code{\link{fraction_grid}()}, 

\code{\link{holdout_control}()}, \code{\link{cv_folds_control}()}, 
\code{\link{create_splits}()}
}
\author{
Andreas Alfons
}
\keyword{multivariate}
