#' relevance_test: Test the relevance of the exogenous regressors.
#'
#' @description This test is needed only if the endogenous regressor `P` is close to be normally
#' distributed. In this case, 2sCOPE can leverage correlated exogenous regressors to achieve model identification.
#'  This function conducts a test for the relevance of exogenous regressor(s),
#'   i.e. the effect of \eqn{W^*} on \eqn{P^*}. Test statistics greater than 10
#'   are reported in a table.
#'   The `formula` argument must be in the form `Y ~ X | P` or `Y ~ X | P | W`, where
#' `X` respresents the explanatory variable(s), `P` represents the endogenous
#' explanatory variable(s), and `W` represents the exogenous explanatory
#' variable(s).
#'
#' @details This test is needed only if the endogenous regressor `P` is close to be normally
#' distributed. If the endogenous regressor
#' `P` is found to have insufficient nonnormality (the Kolmogorov-Smirnov (KS) normality test p-value > 0.05),
#' then 2sCOPE can leverage correlated exogenous regressors to achieve model identification. To compensate
#' for the lack of nonnormality of endogenous regressor `P`, at least one exogenous
#' and continuous regressor `W` needs to satisfy the following two conditions: (1) sufficient
#' nonnormality, and (2) sufficient association with the endogenous regressor `P`. A conservative
#' rule of thumb for such a `W` is the p-value from the KS test on `W` being < 0.001 and a sufficient
#'  association with P (F statistic for the effect of \eqn{W*} on \eqn{P*} > 10 in the first-stage regression.
#' This function `relevance_test()` checks the condition (2) above.
#' When these conditions are met, 2sCOPE is expected to yield consistent estimates
#' even if `P` is normally distributed. When these conditions are not met, Yang, Qian, and
#' Xie (2025) suggest gauging potential bias of 2sCOPE for data at hand via a bootstrap
#' procedure described there, and using 2sCOPE only if the potential bias is small.
#' In order for this function to work as intended, the user must
#' supply a `ccf` object as an argument to the function.
#'
#' @param ccf_obj an object of class ccf returned from the function `CCF()` containing the model of interest for which
#'   the relevance test will be conducted for.
#'   
#' @return No return value; prints out the results of the relevance test.
#'
#' @examples
#' data("diapers") #load data
#'
#' ### Specify logPrice as endogenous using the 1-bar option,
#' #run the copula control function
#' cop_ctrl_fn <- CCF(logVol ~ logPrice+Fshare+week+Q2+Q3+Q4|logPrice,
#'         data = diapers)
#'
#' relevance_test(cop_ctrl_fn) #run relevance test
#'
#' @references
#' Qian, Y., Koschmann, A., & Xie, H. (2025).
#' \emph{EXPRESS: A Practical Guide to Endogeneity Correction Using Copulas.}
#' Journal of Marketing. <doi:10.1177/00222429251410844>\cr
#'
#' Yang, F., Qian, Y., & Xie, H. (2025).
#' \emph{Addressing Endogeneity Using a Two-Stage Copula Generated Regressor Approach.}
#' Journal of Marketing Research, 62(4), 601-623.
#' <doi:10.1177/00222437241296453>\cr
#' @export
relevance_test <- function(ccf_obj) {
  if(!inherits(ccf_obj, "ccf")){
    stop("The relevance test is intended for objects of class ccf")
  }
  if(is.null(ccf_obj$wstar)){
    message("No exogenous regressors specified. Thus, there are no test results.")
  } else if(is.null(attr(ccf_obj, "f_stat"))) {
      message("No pair of endogenous and exogenous variables were significant.")
  } else {
      cat("\n F-Statistics: \n")
      print(attr(ccf_obj, "f_stat"), row.names = F)
  }
}
