library(testthat)

test_that("One Endogenous Regressor", {
  dat <- read.csv(test_path("testdata", "Example1.csv"))
  ccf1 <- CCF(logVol ~ logPrice + Fshare + week + Q2 + Q3 + Q4|logPrice,
                              data=dat)
  ccf2 <- CCF(logVol ~ logPrice + Fshare + week + Q2 + Q3 + Q4|logPrice|Fshare + week + Q2 + Q3 + Q4,
                              data=dat)
  expect_equal(ccf1, ccf2)
  tscope1 <- tscope(logVol ~ logPrice + Fshare + week + Q2 + Q3 + Q4|logPrice,
                                    data=dat, nboot = 50)
  tscope2 <- tscope(logVol ~ logPrice + Fshare + week + Q2 + Q3 + Q4|logPrice|Fshare + week + Q2 + Q3 + Q4,
              data=dat, nboot = 50)
  expect_equal(tscope1$Est, tscope2$Est)
})

test_that("Two Endogenous Regressors", {
  dat <- read.csv(test_path("testdata", "Example1.csv"))
  ccf1 <- CCF(logVol ~ logPrice + Fshare + week + Q2 + Q3 + Q4|logPrice + Fshare,
              data=dat)
  ccf2 <- CCF(logVol ~ logPrice + Fshare + week + Q2 + Q3 + Q4|logPrice + Fshare|week + Q2 + Q3 + Q4,
              data=dat)
  expect_equal(ccf1, ccf2)
  tscope1 <- tscope(logVol ~ logPrice + Fshare + week + Q2 + Q3 + Q4|logPrice + Fshare,
                    data=dat, nboot = 50)
  tscope2 <- tscope(logVol ~ logPrice + Fshare + week + Q2 + Q3 + Q4|logPrice + Fshare|week + Q2 + Q3 + Q4,
                    data=dat, nboot = 50)
  expect_equal(tscope1$Est, tscope2$Est)
})

test_that("Expect message about running P&G, wstar is null, ccf == pstar, one endogenous regressor", {
  dat <- read.csv(test_path("testdata", "Example1.csv"))
  expect_message(ccf <- CCF(logVol ~ logPrice | logPrice, data = dat))
  expect_message(pg <- tscope(logVol ~ logPrice | logPrice, data = dat, nboot = 50))
  expect_null(ccf$wstar)
  expect_equal(unname(ccf$ccf), unname(ccf$pstar))
})

test_that("Expect message about running P&G, wstar is null, ccf == pstar, two endogenous regressors", {
  dat <- read.csv(test_path("testdata", "Example1.csv"))
  expect_message(ccf <- CCF(logVol ~ logPrice + Fshare|logPrice + Fshare, data = dat))
  expect_message(pf <- tscope(logVol ~ logPrice + Fshare|logPrice + Fshare, data = dat, nboot = 50))
  expect_null(ccf$wstar)
  expect_equal(unname(ccf$ccf), unname(ccf$pstar))
})


test_that("Exogenous regressor specified twice", {
  dat <- read.csv(test_path("testdata", "Example1.csv"))
  ccf1 <- CCF(logVol ~ logPrice + Fshare + week + Q2 + Q3 + Q4|logPrice,
              data=dat)
  ccf1_twice <- CCF(logVol ~ logPrice + Fshare + Fshare + week + Q2 + Q3 + Q4|logPrice,
              data=dat)
  ccf2 <- CCF(logVol ~ logPrice + Fshare + week + Q2 + Q3 + Q4|logPrice|Fshare + week + Q2 + Q3 + Q4,
              data=dat)
  ccf2_twice <- CCF(logVol ~ logPrice + Fshare + Fshare + week + Q2 + Q3 + Q4|logPrice|Fshare + week + Q2 + Q3 + Q4,
              data=dat)
  expect_equal(ccf1, ccf2_twice)
  expect_equal(ccf2, ccf1_twice)
  tscope1 <- tscope(logVol ~ logPrice + Fshare + week + Q2 + Q3 + Q4|logPrice,
                    data=dat, nboot = 50)
  tscope1_twice <- tscope(logVol ~ logPrice + Fshare + Fshare + week + Q2 + Q3 + Q4|logPrice,
                    data=dat, nboot = 50)
  tscope2 <- tscope(logVol ~ logPrice + Fshare + week + Q2 + Q3 + Q4|logPrice|Fshare + week + Q2 + Q3 + Q4,
                    data=dat, nboot = 50)
  tscope2_twice <- tscope(logVol ~ logPrice + Fshare + Fshare + week + Q2 + Q3 + Q4|logPrice|Fshare + Fshare + week + Q2 + Q3 + Q4,
                    data=dat, nboot = 50)
  expect_equal(tscope1$Est, tscope2_twice$Est)
  expect_equal(tscope2$Est, tscope1_twice$Est)
})

test_that("Endogenous regressor specified twice", {
  dat <- read.csv(test_path("testdata", "Example1.csv"))
  ccf1 <- CCF(logVol ~ logPrice + Fshare + week + Q2 + Q3 + Q4|logPrice,
              data=dat)
  ccf1_twice <- CCF(logVol ~ logPrice + Fshare + week + Q2 + Q3 + Q4|logPrice + logPrice,
                    data=dat)
  ccf2 <- CCF(logVol ~ logPrice + Fshare + week + Q2 + Q3 + Q4|logPrice|Fshare + week + Q2 + Q3 + Q4,
              data=dat)
  ccf2_twice <- CCF(logVol ~ logPrice + Fshare + week + Q2 + Q3 + Q4|logPrice + logPrice|Fshare + week + Q2 + Q3 + Q4,
                    data=dat)
  expect_equal(ccf1, ccf2_twice)
  expect_equal(ccf2, ccf1_twice)
  tscope1 <- tscope(logVol ~ logPrice + Fshare + week + Q2 + Q3 + Q4|logPrice,
                    data=dat, nboot = 50)
  tscope1_twice <- tscope(logVol ~ logPrice + Fshare + week + Q2 + Q3 + Q4|logPrice + logPrice,
                          data=dat, nboot = 50)
  tscope2 <- tscope(logVol ~ logPrice + Fshare + week + Q2 + Q3 + Q4|logPrice|Fshare + week + Q2 + Q3 + Q4,
                    data=dat, nboot = 50)
  tscope2_twice <- tscope(logVol ~ logPrice + Fshare  + week + Q2 + Q3 + Q4|logPrice + logPrice|Fshare + week + Q2 + Q3 + Q4,
                          data=dat, nboot = 50)
  expect_equal(tscope1$Est, tscope2_twice$Est)
  expect_equal(tscope2$Est, tscope1_twice$Est)
})


test_that("One Endogenous higher order Regressor", {
  dat <- read.csv(test_path("testdata", "Example1.csv"))

  expect_error(
    ccf1 <- CCF(
      logVol ~ logPrice + Fshare + I(logPrice * Fshare) +
        week + Q2 + Q3 + Q4 | logPrice + Fshare + I(logPrice * Fshare),
      data = dat
    )
  )

  expect_warning(
    ccf2 <- CCF(
      logVol ~ logPrice + Fshare + I(logPrice * Fshare) +
        week + Q2 + Q3 + Q4 |
        logPrice + Fshare + I(logPrice * Fshare) | week + Q2 + Q3 + Q4,
      data = dat
    )
  )

  expect_error(
    tscope1 <- tscope(
      logVol ~ logPrice + Fshare + I(logPrice * Fshare) +
        week + Q2 + Q3 + Q4 | logPrice + Fshare + I(logPrice * Fshare),
      data = dat,
      nboot = 50
    )
  )

  expect_warning(
    tscope2 <- tscope(
      logVol ~ logPrice + Fshare + I(logPrice * Fshare) +
        week + Q2 + Q3 + Q4 |
        logPrice + Fshare + I(logPrice * Fshare) | week + Q2 + Q3 + Q4,
      data = dat,
      nboot = 50
    )
  )
})

test_that("No endogenous regressors specified", {
  dat <- read.csv(test_path("testdata", "Example1.csv"))
  expect_error(ccf <- CCF(logVol ~ logPrice + Fshare + week + Q2 + Q3 + Q4,
              data=dat))
  expect_error(tscope <- tscope(logVol ~ logPrice + Fshare + week + Q2 + Q3 + Q4,
                    data=dat, nboot = 50))
})

test_that("Regressors specified as endogenous not in original formula", {
  dat <- read.csv(test_path("testdata", "Example1.csv"))
  expect_error(ccf <- CCF(logVol ~ logPrice + week + Q2 + Q3 + Q4|Fshare,
                          data=dat))
  expect_error(ccf <- CCF(logVol ~ logPrice + week + Q2 + Q3 + Q4|Fshare|logPrice + week + Q2 + Q3 + Q4,
                          data=dat))
  expect_error(tscope <- tscope(logVol ~ logPrice + week + Q2 + Q3 + Q4|Fshare,
                                data=dat, nboot = 50))
  expect_error(tscope <- tscope(logVol ~ logPrice + week + Q2 + Q3 + Q4|Fshare|logPrice + week + Q2 + Q3 + Q4,
                                data=dat, nboot = 50))
})


test_that("Regressors in original formula not specified as endogenous", {
  dat <- read.csv(test_path("testdata", "Example1.csv"))
  expect_error(ccf <- CCF(logVol ~ logPrice + Fshare + week + Q2 + Q3 + Q4|Fshare| week + Q2 + Q3 + Q4,
                          data=dat))
  expect_error(tscope <- tscope(logVol ~ logPrice + Fshare + week + Q2 + Q3 + Q4|Fshare| week + Q2 + Q3 + Q4,
                                data=dat, nboot = 50))
})


test_that("Regressors specified as exogenous not in original formula", {
  dat <- read.csv(test_path("testdata", "Example1.csv"))
  expect_error(ccf <- CCF(logVol ~  Fshare + week + Q2 + Q3 + Q4|Fshare| logPrice +week + Q2 + Q3 + Q4,
                       data=dat))
  expect_error(tscope <- tscope(logVol ~  Fshare + week + Q2 + Q3 + Q4|Fshare| logPrice +week + Q2 + Q3 + Q4,
                                data=dat, nboot = 50))
})


test_that("Regressors specified as both endogenous and exogenous.", {
  dat <- read.csv(test_path("testdata", "Example1.csv"))
  expect_error(ccf <- CCF(logVol ~  Fshare + week + Q2 + Q3 + Q4|Fshare|Fshare + logPrice +week + Q2 + Q3 + Q4,
                          data=dat))
  expect_error(tscope <- tscope(logVol ~  Fshare + week + Q2 + Q3 + Q4|Fshare|Fshare + logPrice +week + Q2 + Q3 + Q4,
                                data=dat, nboot = 50))
})




