#' Extract Model Coefficients from a `safe` Object
#'
#' Retrieves the coefficients at specified values of `lambda` from a fitted `safe()` model.
#'
#' This function extracts coefficients for specified `lambda` values from a `safe()` object.
#' If `s`, the vector of `lambda` values, contains values not originally used in the model fitting,
#' the `coef` function employs linear interpolation between the closest `lambda` values from the 
#' original sequence to estimate coefficients at the new `lambda` values.
#'
#' @importFrom methods rbind2
#' @importFrom stats coef predict
#' @param object Fitted `safe()` object.
#' @param s Values of the penalty parameter `lambda` for which coefficients are requested.
#'   Defaults to the entire sequence used during the model fit.
#' @param ... Not used.
#' @seealso \code{\link{safe}}, \code{\link{predict.safe}}
#'
#' @return Returns a matrix or vector of coefficients corresponding to the specified `lambda` values.
#'
#' @examples
#' set.seed(1)
#' n <- 100
#' p <- 5
#' x <- matrix(rnorm(n * p), nrow = n, ncol = p)
#' beta_true <- rep(0.1, 5)
#' gamma_true <- c(rep(1, 3), -1, -1)
#' mu <- x %*% beta_true
#' k <- rpois(n, lambda = exp(mu))
#' alpha_val <- 1
#' theta <- exp(x %*% gamma_true) / alpha_val
#' y <- rgamma(n, shape = alpha_val, scale = theta)
#' lambda_val <- 1
#' fit <- safe(x, y, k, 1, ind_p = c(1, 1, 1, 0, 0))
#' coef(fit)
#'
#' @method coef safe
#' @export

coef.safe <- function(object, s = NULL, ...) {
    nbeta <- object$beta
    if (!is.null(s)) {
      vnames <- dimnames(nbeta)[[1]]
      dimnames(nbeta) <- list(NULL, NULL)
      lambda <- object$lambda
      lamlist <- lambda.interp(lambda, s)
      nbeta <- nbeta[,lamlist$left, drop=FALSE]%*%Diagonal(x=lamlist$frac) +
               nbeta[,lamlist$right,drop=FALSE]%*%Diagonal(x=1-lamlist$frac)
      dimnames(nbeta) <- list(vnames, paste(seq(along = s)))
    }
    ngamma <- object$gamma
    if (!is.null(s)) {
      dimnames(ngamma) <- list(NULL, NULL)
      lambda <- object$lambda
      lamlist <- lambda.interp(lambda, s)
      ngamma <- ngamma[,lamlist$left, drop=FALSE]%*%Diagonal(x=lamlist$frac) +
               ngamma[,lamlist$right,drop=FALSE]%*%Diagonal(x=1-lamlist$frac)
      dimnames(ngamma) <- list(vnames, paste(seq(along = s)))
    }
    return(list(beta=nbeta, gamma=ngamma))
}

#' Make Predictions from a `safe` Object
#'
#' Produces fitted values for new predictor data using a fitted `safe()` object.
#'
#' This function generates predictions at specified `lambda` values from a fitted `safe()` object.
#' It is essential to provide a new matrix of predictor values (`newx`) at which these predictions are to be made.
#'
#' @param object Fitted `safe()` object from which predictions are to be derived.
#' @param newx Matrix of new predictor values for which predictions are desired.
#'   This must be a matrix and is a required argument.
#' @param s Values of the penalty parameter `lambda` for which predictions are requested.
#'   Defaults to the entire sequence used during the model fit.
#' @param ... Not used.
#' @seealso \code{\link{safe}}, \code{\link{coef.safe}}
#'
#' @return Returns the fitted values.
#'
#' @examples
#' set.seed(1)
#' n <- 100
#' p <- 5
#' x <- matrix(rnorm(n * p), nrow = n, ncol = p)
#' beta_true <- rep(0.1, 5)
#' gamma_true <- c(rep(1, 3), -1, -1)
#' mu <- x %*% beta_true
#' k <- rpois(n, lambda = exp(mu))
#' alpha_val <- 1
#' theta <- exp(x %*% gamma_true) / alpha_val
#' y <- rgamma(n, shape = alpha_val, scale = theta)
#' lambda_val <- 1
#' fit <- safe(x, y, k, 1, ind_p = c(1, 1, 1, 0, 0))
#' set.seed(234)
#' newx <- matrix(rnorm(n * p), nrow = n, ncol = p)
#' predict(fit, newx)
#'
#' @method predict safe
#' @export

predict.safe <- function(object, newx, s=NULL, ...) {
  newx.row <- as.integer(NROW(newx))
  coeffs <- coef(object, s)
  gamma <- coeffs$gamma
  beta <- coeffs$beta
  khat <- exp(newx %*% beta)
  muhat <- exp(newx %*% gamma)
  yhat <- khat * muhat
  return(yhat)
}