% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/data_simulation.R
\name{simulate_spatial_data}
\alias{simulate_spatial_data}
\title{Simulate Spatial Transcriptomics Data}
\usage{
simulate_spatial_data(
  n_spots = 500,
  n_genes = 200,
  n_svg = 50,
  grid_type = c("hexagonal", "square", "random"),
  pattern_types = c("gradient", "hotspot", "periodic", "cluster"),
  mean_counts = 50,
  dispersion = 5
)
}
\arguments{
\item{n_spots}{Integer. Number of spatial locations. Default is 500.}

\item{n_genes}{Integer. Total number of genes. Default is 200.}

\item{n_svg}{Integer. Number of spatially variable genes. Default is 50.}

\item{grid_type}{Character. Type of spatial layout.
\itemize{
  \item \code{"hexagonal"} (default): Visium-like hexagonal grid
  \item \code{"square"}: Square grid
  \item \code{"random"}: Random spatial distribution
}}

\item{pattern_types}{Character vector. Types of spatial patterns for SVGs.
Any combination of:
\itemize{
  \item \code{"gradient"}: Linear spatial gradient
  \item \code{"hotspot"}: Localized expression hotspots
  \item \code{"periodic"}: Periodic/oscillating patterns
  \item \code{"cluster"}: Clustered expression
}
Default is all four types.}

\item{mean_counts}{Numeric. Mean expression level for baseline. Default is 50.}

\item{dispersion}{Numeric. Dispersion parameter for Negative Binomial.
Smaller values = more overdispersion. Default is 5.}
}
\value{
A list containing:
  \itemize{
    \item \code{counts}: Matrix of gene counts (genes × spots)
    \item \code{spatial_coords}: Matrix of spatial coordinates (spots × 2)
    \item \code{gene_info}: Data.frame with gene metadata including
      \code{is_svg} (TRUE/FALSE) and \code{pattern_type}
    \item \code{logcounts}: Log-normalized counts (log2(counts + 1))
  }
}
\description{
Generates a simulated spatial transcriptomics dataset with a mixture of
spatially variable genes (SVGs) and non-spatially variable genes.
Uses scientifically accurate count distributions (Negative Binomial).
}
\details{
\strong{Spatial Patterns:}
\itemize{
  \item \strong{Gradient}: Expression increases linearly along x-axis
  \item \strong{Hotspot}: High expression in circular regions
  \item \strong{Periodic}: Sine wave pattern along x-axis
  \item \strong{Cluster}: Expression in spatially defined clusters
}

\strong{Count Distribution:}
Counts are drawn from Negative Binomial distribution:
\deqn{X \sim NB(\mu, \phi)}
where mu is the mean (modulated by spatial pattern) and phi is dispersion.
}
\examples{
# Set seed for reproducibility before calling
set.seed(42)
sim_data <- simulate_spatial_data(n_spots = 200, n_genes = 50, n_svg = 10)
str(sim_data, max.level = 1)

\donttest{
# Use with SVG detection (requires RANN)
if (requireNamespace("RANN", quietly = TRUE)) {
    results <- CalSVG_MERINGUE(sim_data$counts, sim_data$spatial_coords,
                               network_method = "knn", k = 10, verbose = FALSE)
}
}

}
