#' Testing true long memory against spurious long memory
#'
#' Tests the null hypothesis of true long memory against the alternative of spurious long memory (e.g., due to level shifts or trends) using the Qu (2011) test.
#'
#' @param x Numeric vector of time series data
#' @param demean Logical, demean the series (default: TRUE)
#' @param alpha Numeric, significance level: 0.01, 0.025, 0.05, or 0.10 (default: 0.05)
#' @param filter Logical, apply a pre-whitening filter to reduce short-memory dynamics that may otherwise affect the size of the test in finite samples (default: TRUE).
#' @param print_results Logical, print results (default: TRUE)
#'
#' @return A list with elements:
#' \item{d_estimate}{Local Whittle estimate of the long memory parameter.}
#' \item{test_stat_eps02}{Test statistic with trimming parameter epsilon = 0.02.}
#' \item{test_stat_eps05}{Test statistic with trimming parameter epsilon = 0.05.}
#' \item{critical_value_eps02}{Critical value for epsilon = 0.02 at the chosen significance level alpha.}
#' \item{critical_value_eps05}{Critical value for epsilon = 0.05 at the chosen significance level alpha.}
#' \item{reject_eps02}{Logical, TRUE if the null is rejected for epsilon = 0.02.}
#' \item{reject_eps05}{Logical, TRUE if the null is rejected for epsilon = 0.05.}
#' \item{alpha}{Significance level.}
#' \item{m}{Bandwidth parameter m.}
#' \item{n}{Sample size.}
#'
#' @details
#' The test uses bandwidth parameter m = n^0.7 with trimming parameters 0.02 or 0.05 as recommended by Qu (2011). Critical values are based on these settings. Whether the pre-whitening filter is applied does not affect the asymptotic critical values.
#'
#' @references
#' Qu, Z. (2011). A Test Against Spurious Long Memory. Journal of Business & Economic Statistics, 29(3), 423-438.
#'
#' @examples
#' data(RV5min)
#' result <- Longmemorytest(log(RV5min))
#'
#' # Simulated example: ARFIMA(0,d,0) with d = 0.3 (true long memory)
#' set.seed(123)
#' n  <- 2000
#' d0 <- 0.3
#' e  <- rnorm(n)
#' x  <- fracdiff::fracdiff.sim(n, d = d0)$series
#'
#' # Apply the test (null = true long memory)
#' out <- Longmemorytest(x, alpha = 0.05)
#' out$reject_eps02    # typically FALSE
#' out$d_estimate
#'
#' # Example: Spurious long memory due to a structural break
#' set.seed(123)
#' n <- 2000
#' x <- c(rnorm(n/2, 0, 1), rnorm(n/2, 2, 1))   # one-time mean shift
#'
#' out <- Longmemorytest(x, alpha = 0.05)
#' out$reject_eps02
#' out$reject_eps05
#'
#' @importFrom stats fft optimize
#' @export
Longmemorytest <- function(x, demean = TRUE, alpha = 0.05, filter = TRUE, print_results = TRUE) {

  # Validation
  if (!is.numeric(x)) stop("x must be numeric")
  if (!alpha %in% c(0.01, 0.025, 0.05, 0.10)) {
    stop("alpha must be 0.01, 0.025, 0.05, or 0.10")
  }
  if (is.matrix(x)) x <- as.vector(x)
  if (anyNA(x)) x <- stats::na.omit(x)

  n <- length(x)

  # Preprocess
  if (demean) x <- x - mean(x)

  if (filter==TRUE) {x <- filterx(x, n)}  # user-defined prewhitening; ensure this exists

  # Bandwidth m = n^0.7
  mm <- 0.7
  m <- min(round(n^mm),n-1)
  trm1 <- max(1,round(0.02 * m))
  trm2 <- max(1,round(0.05 * m))

  # Critical values
  cv_table <- matrix(
    c(1.118, 1.022,
      1.252, 1.155,
      1.374, 1.277,
      1.517, 1.426),
    nrow = 4, ncol = 2, byrow = TRUE,
    dimnames = list(c("0.10", "0.05", "0.025", "0.01"),
                    c("eps02", "eps05"))
  )
  alpha_key <- format(alpha, nsmall = 2)
  if (!alpha_key %in% rownames(cv_table)) {
    stop("alpha must be one of 0.01, 0.025, 0.05, or 0.10")
  }
  cv_eps02 <- cv_table[alpha_key, "eps02"]
  cv_eps05 <- cv_table[alpha_key, "eps05"]

  # Periodogram
  freq <- (1:(n-1)) * (2 * pi / n)
  xf <- stats::fft(x)
  px <- (Re(xf)^2 + Im(xf)^2) / (2 * pi * n)
  perdx <- px[2:n]  # drop zero frequency

  # Local Whittle objective
  fn <- function(h) {
    lambda <- freq[1:m]^(2 * h - 1)
    Gh <- mean(perdx[1:m] * lambda)
    log(Gh) - (2 * h - 1) * mean(log(freq[1:m]))
  }

  est <- stats::optimize(fn, c(0, 1.5), tol = 1e-5)
  hhat <- est$minimum
  d_est <- hhat - 0.5

  # Test statistic
  lambda_hat <- freq[1:m]^(2 * hhat - 1)
  Ghat <- mean(perdx[1:m] * lambda_hat)
  comp1 <- (perdx[1:m] * lambda_hat) / Ghat
  comp2 <- log(freq[1:m]) - mean(log(freq[1:m]))
  stat <- cumsum((comp1 - 1) * comp2) / sqrt(sum(comp2^2))

  z1 <- max(abs(stat[trm1:m]))
  z2 <- max(abs(stat[trm2:m]))

  reject_eps02 <- z1 > cv_eps02
  reject_eps05 <- z2 > cv_eps05

  if (print_results) {
    cat("\n--- Spurious Long Memory Test (Qu, 2011) ---\n")
    cat(sprintf("n = %d, m = %d (n^%.1f), alpha = %.2f%%\n", n, m, mm, alpha * 100))
    cat(sprintf("d estimate: %.4f\n\n", d_est))
    cat(sprintf("epsilon = 0.02: W = %.3f (cv = %.3f) %s\n",
                z1, cv_eps02, ifelse(reject_eps02, "**REJECT**", "Does Not Reject")))
    cat(sprintf("epsilon = 0.05: W = %.3f (cv = %.3f) %s\n",
                z2, cv_eps05, ifelse(reject_eps05, "**REJECT**", "Does Not Reject")))
    cat("--------------------------------------------\n\n")
  }

  invisible(list(
    d_estimate = d_est,
    test_stat_eps02 = z1,
    test_stat_eps05 = z2,
    critical_value_eps02 = cv_eps02,
    critical_value_eps05 = cv_eps05,
    reject_eps02 = reject_eps02,
    reject_eps05 = reject_eps05,
    alpha = alpha,
    m = m,
    n = n
  ))
}
