/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * Copyright by The HDF Group.                                               *
 * All rights reserved.                                                      *
 *                                                                           *
 * This file is part of HDF5.  The full HDF5 copyright notice, including     *
 * terms governing use, modification, and redistribution, is contained in    *
 * the LICENSE file, which can be found at the root of the source code       *
 * distribution tree, or in https://www.hdfgroup.org/licenses.               *
 * If you do not have access to either file, you may request a copy from     *
 * help@hdfgroup.org.                                                        *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#include "H5Omodule.h" 
#define H5Z_FRIEND     

#include "H5private.h"   
#include "H5Eprivate.h"  
#include "H5FLprivate.h" 
#include "H5MMprivate.h" 
#include "H5Opkg.h"      
#include "H5Zpkg.h"      

static herr_t H5O__pline_encode(H5F_t *f, uint8_t *p, const void *mesg);
static void  *H5O__pline_decode(H5F_t *f, H5O_t *open_oh, unsigned mesg_flags, unsigned *ioflags,
                                size_t p_size, const uint8_t *p);
static void  *H5O__pline_copy(const void *_mesg, void *_dest);
static size_t H5O__pline_size(const H5F_t *f, const void *_mesg);
static herr_t H5O__pline_reset(void *_mesg);
static herr_t H5O__pline_free(void *_mesg);
static herr_t H5O__pline_pre_copy_file(H5F_t *file_src, const void *mesg_src, bool *deleted,
                                       const H5O_copy_t *cpy_info, void *_udata);
static herr_t H5O__pline_debug(H5F_t *f, const void *_mesg, FILE *stream, int indent, int fwidth);

#define H5O_SHARED_TYPE        H5O_MSG_PLINE
#define H5O_SHARED_DECODE      H5O__pline_shared_decode
#define H5O_SHARED_DECODE_REAL H5O__pline_decode
#define H5O_SHARED_ENCODE      H5O__pline_shared_encode
#define H5O_SHARED_ENCODE_REAL H5O__pline_encode
#define H5O_SHARED_SIZE        H5O__pline_shared_size
#define H5O_SHARED_SIZE_REAL   H5O__pline_size
#define H5O_SHARED_DELETE      H5O__pline_shared_delete
#undef H5O_SHARED_DELETE_REAL
#define H5O_SHARED_LINK H5O__pline_shared_link
#undef H5O_SHARED_LINK_REAL
#define H5O_SHARED_COPY_FILE H5O__pline_shared_copy_file
#undef H5O_SHARED_COPY_FILE_REAL
#define H5O_SHARED_POST_COPY_FILE H5O__pline_shared_post_copy_file
#undef H5O_SHARED_POST_COPY_FILE_REAL
#undef H5O_SHARED_POST_COPY_FILE_UPD
#define H5O_SHARED_DEBUG      H5O__pline_shared_debug
#define H5O_SHARED_DEBUG_REAL H5O__pline_debug
#include "H5Oshared.h" 

const H5O_msg_class_t H5O_MSG_PLINE[1] = {{
    H5O_PLINE_ID,                              
    "filter pipeline",                         
    sizeof(H5O_pline_t),                       
    H5O_SHARE_IS_SHARABLE | H5O_SHARE_IN_OHDR, 
    H5O__pline_shared_decode,                  
    H5O__pline_shared_encode,                  
    H5O__pline_copy,                           
    H5O__pline_shared_size,                    
    H5O__pline_reset,                          
    H5O__pline_free,                           
    H5O__pline_shared_delete,                  
    H5O__pline_shared_link,                    
    NULL,                                      
    NULL,                                      
    H5O__pline_pre_copy_file,                  
    H5O__pline_shared_copy_file,               
    H5O__pline_shared_post_copy_file,          
    NULL,                                      
    NULL,                                      
    H5O__pline_shared_debug                    
}};

const unsigned H5O_pline_ver_bounds[] = {
    H5O_PLINE_VERSION_1,     
    H5O_PLINE_VERSION_2,     
    H5O_PLINE_VERSION_2,     
    H5O_PLINE_VERSION_2,     
    H5O_PLINE_VERSION_2,     
    H5O_PLINE_VERSION_2,     
    H5O_PLINE_VERSION_LATEST 
};

H5FL_DEFINE(H5O_pline_t);

static void *
H5O__pline_decode(H5F_t H5_ATTR_UNUSED *f, H5O_t H5_ATTR_UNUSED *open_oh, unsigned H5_ATTR_UNUSED mesg_flags,
                  unsigned H5_ATTR_UNUSED *ioflags, size_t p_size, const uint8_t *p)
{
    H5O_pline_t       *pline = NULL;               
    H5Z_filter_info_t *filter;                     
    size_t             name_length;                
    size_t             i;                          
    const uint8_t     *p_end     = p + p_size - 1; 
    void              *ret_value = NULL;

    FUNC_ENTER_PACKAGE

    assert(f);
    assert(p);

    
    if (NULL == (pline = H5FL_CALLOC(H5O_pline_t)))
        HGOTO_ERROR(H5E_RESOURCE, H5E_NOSPACE, NULL, "memory allocation failed");

    
    if (H5_IS_BUFFER_OVERFLOW(p, 1, p_end))
        HGOTO_ERROR(H5E_OHDR, H5E_OVERFLOW, NULL, "ran off end of input buffer while decoding");
    pline->version = *p++;
    if (pline->version < H5O_PLINE_VERSION_1 || pline->version > H5O_PLINE_VERSION_LATEST)
        HGOTO_ERROR(H5E_PLINE, H5E_CANTLOAD, NULL, "bad version number for filter pipeline message");

    
    if (H5_IS_BUFFER_OVERFLOW(p, 1, p_end))
        HGOTO_ERROR(H5E_OHDR, H5E_OVERFLOW, NULL, "ran off end of input buffer while decoding");
    pline->nused = *p++;
    if (pline->nused > H5Z_MAX_NFILTERS) {

        
        pline->nused = 0;

        HGOTO_ERROR(H5E_PLINE, H5E_CANTLOAD, NULL, "filter pipeline message has too many filters");
    }

    
    if (pline->version == H5O_PLINE_VERSION_1) {
        if (H5_IS_BUFFER_OVERFLOW(p, 6, p_end))
            HGOTO_ERROR(H5E_OHDR, H5E_OVERFLOW, NULL, "ran off end of input buffer while decoding");
        p += 6;
    }

    
    pline->nalloc = pline->nused;
    if (NULL == (pline->filter = (H5Z_filter_info_t *)H5MM_calloc(pline->nalloc * sizeof(pline->filter[0]))))
        HGOTO_ERROR(H5E_RESOURCE, H5E_NOSPACE, NULL, "memory allocation failed");

    
    for (i = 0, filter = &pline->filter[0]; i < pline->nused; i++, filter++) {
        
        if (H5_IS_BUFFER_OVERFLOW(p, 2, p_end))
            HGOTO_ERROR(H5E_OHDR, H5E_OVERFLOW, NULL, "ran off end of input buffer while decoding");
        UINT16DECODE(p, filter->id);

        
        if (pline->version > H5O_PLINE_VERSION_1 && filter->id < H5Z_FILTER_RESERVED)
            name_length = 0;
        else {
            if (H5_IS_BUFFER_OVERFLOW(p, 2, p_end))
                HGOTO_ERROR(H5E_OHDR, H5E_OVERFLOW, NULL, "ran off end of input buffer while decoding");
            UINT16DECODE(p, name_length);
            if (pline->version == H5O_PLINE_VERSION_1 && name_length % 8)
                HGOTO_ERROR(H5E_PLINE, H5E_CANTLOAD, NULL, "filter name length is not a multiple of eight");
        }

        
        if (H5_IS_BUFFER_OVERFLOW(p, 2, p_end))
            HGOTO_ERROR(H5E_OHDR, H5E_OVERFLOW, NULL, "ran off end of input buffer while decoding");
        UINT16DECODE(p, filter->flags);

        
        if (H5_IS_BUFFER_OVERFLOW(p, 2, p_end))
            HGOTO_ERROR(H5E_OHDR, H5E_OVERFLOW, NULL, "ran off end of input buffer while decoding");
        UINT16DECODE(p, filter->cd_nelmts);

        
        if (name_length) {
            size_t actual_name_length;            
            size_t max = (size_t)(p_end - p + 1); 

            
            actual_name_length = strnlen((const char *)p, max);
            if (actual_name_length == max)
                HGOTO_ERROR(H5E_OHDR, H5E_NOSPACE, NULL, "filter name not null terminated");
            actual_name_length += 1; 

            
            if (actual_name_length > H5Z_COMMON_NAME_LEN) {
                filter->name = (char *)H5MM_malloc(actual_name_length);
                if (NULL == filter->name)
                    HGOTO_ERROR(H5E_RESOURCE, H5E_NOSPACE, NULL, "memory allocation failed for filter name");
            }
            else
                filter->name = filter->_name;

            strncpy(filter->name, (const char *)p, actual_name_length);

            if (H5_IS_BUFFER_OVERFLOW(p, name_length, p_end))
                HGOTO_ERROR(H5E_OHDR, H5E_OVERFLOW, NULL, "ran off end of input buffer while decoding");
            p += name_length;
        }

        
        if (filter->cd_nelmts) {

            
            if (filter->cd_nelmts > H5Z_COMMON_CD_VALUES) {
                filter->cd_values = (unsigned *)H5MM_malloc(filter->cd_nelmts * sizeof(unsigned));
                if (NULL == filter->cd_values)
                    HGOTO_ERROR(H5E_RESOURCE, H5E_NOSPACE, NULL, "memory allocation failed for client data");
            }
            else
                filter->cd_values = filter->_cd_values;

            
            for (size_t j = 0; j < filter->cd_nelmts; j++) {
                if (H5_IS_BUFFER_OVERFLOW(p, 4, p_end))
                    HGOTO_ERROR(H5E_OHDR, H5E_OVERFLOW, NULL, "ran off end of input buffer while decoding");
                UINT32DECODE(p, filter->cd_values[j]);
            }

            if (pline->version == H5O_PLINE_VERSION_1)
                if (filter->cd_nelmts % 2) {
                    if (H5_IS_BUFFER_OVERFLOW(p, 4, p_end))
                        HGOTO_ERROR(H5E_OHDR, H5E_OVERFLOW, NULL,
                                    "ran off end of input buffer while decoding");
                    p += 4; 
                }
        }
    }

    
    ret_value = pline;

done:
    if (!ret_value && pline) {
        H5O__pline_reset(pline);
        H5O__pline_free(pline);
    }

    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5O__pline_encode(H5F_t H5_ATTR_UNUSED *f, uint8_t *p , const void *mesg)
{
    const H5O_pline_t       *pline = (const H5O_pline_t *)mesg; 
    const H5Z_filter_info_t *filter;                            
    size_t                   i, j;                              

    FUNC_ENTER_PACKAGE_NOERR

    
    assert(p);
    assert(mesg);

    
    *p++ = (uint8_t)pline->version;
    *p++ = (uint8_t)(pline->nused);
    if (pline->version == H5O_PLINE_VERSION_1) {
        *p++ = 0; 
        *p++ = 0; 
        *p++ = 0; 
        *p++ = 0; 
        *p++ = 0; 
        *p++ = 0; 
    }             

    
    for (i = 0, filter = &pline->filter[0]; i < pline->nused; i++, filter++) {
        const char *name;        
        size_t      name_length; 

        
        UINT16ENCODE(p, filter->id);

        
        if (pline->version > H5O_PLINE_VERSION_1 && filter->id < H5Z_FILTER_RESERVED) {
            name_length = 0;
            name        = NULL;
        } 
        else {
            
            if (NULL == (name = filter->name)) {
                H5Z_class2_t *cls; 

                H5Z_find(true, filter->id, &cls);
                if (cls)
                    name = cls->name;
            }
            name_length = name ? strlen(name) + 1 : 0;

            
            UINT16ENCODE(p, pline->version == H5O_PLINE_VERSION_1 ? H5O_ALIGN_OLD(name_length) : name_length);
        } 

        
        UINT16ENCODE(p, filter->flags);

        
        UINT16ENCODE(p, filter->cd_nelmts);

        
        if (name_length > 0) {
            
            H5MM_memcpy(p, name, name_length);
            p += name_length;

            
            if (pline->version == H5O_PLINE_VERSION_1)
                while (name_length++ % 8)
                    *p++ = 0;
        } 

        
        for (j = 0; j < filter->cd_nelmts; j++)
            UINT32ENCODE(p, filter->cd_values[j]);

        
        if (pline->version == H5O_PLINE_VERSION_1)
            if (filter->cd_nelmts % 2)
                UINT32ENCODE(p, 0);
    } 

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

static void *
H5O__pline_copy(const void *_src, void *_dst )
{
    const H5O_pline_t *src = (const H5O_pline_t *)_src; 
    H5O_pline_t       *dst = (H5O_pline_t *)_dst;       
    size_t             i;                               
    H5O_pline_t       *ret_value = NULL;                

    FUNC_ENTER_PACKAGE

    
    if (!dst && NULL == (dst = H5FL_MALLOC(H5O_pline_t)))
        HGOTO_ERROR(H5E_RESOURCE, H5E_NOSPACE, NULL, "memory allocation failed");

    
    *dst = *src;

    
    dst->nalloc = dst->nused;
    if (dst->nalloc) {
        
        if (NULL == (dst->filter = (H5Z_filter_info_t *)H5MM_calloc(dst->nalloc * sizeof(dst->filter[0]))))
            HGOTO_ERROR(H5E_RESOURCE, H5E_NOSPACE, NULL, "memory allocation failed");

        
        for (i = 0; i < src->nused; i++) {
            
            dst->filter[i] = src->filter[i];

            
            if (src->filter[i].name) {
                size_t namelen; 

                namelen = strlen(src->filter[i].name) + 1;

                
                if (namelen > H5Z_COMMON_NAME_LEN) {
                    dst->filter[i].name = (char *)H5MM_strdup(src->filter[i].name);
                    if (NULL == dst->filter[i].name)
                        HGOTO_ERROR(H5E_RESOURCE, H5E_NOSPACE, NULL,
                                    "memory allocation failed for filter name");
                } 
                else
                    dst->filter[i].name = dst->filter[i]._name;
            } 

            
            if (src->filter[i].cd_nelmts > 0) {
                
                if (src->filter[i].cd_nelmts > H5Z_COMMON_CD_VALUES) {
                    if (NULL == (dst->filter[i].cd_values =
                                     (unsigned *)H5MM_malloc(src->filter[i].cd_nelmts * sizeof(unsigned))))
                        HGOTO_ERROR(H5E_RESOURCE, H5E_NOSPACE, NULL, "memory allocation failed");

                    H5MM_memcpy(dst->filter[i].cd_values, src->filter[i].cd_values,
                                src->filter[i].cd_nelmts * sizeof(unsigned));
                } 
                else
                    dst->filter[i].cd_values = dst->filter[i]._cd_values;
            } 
        }     
    }         
    else
        dst->filter = NULL;

    
    ret_value = dst;

done:
    if (!ret_value && dst) {
        H5O__pline_reset(dst);
        if (!_dst)
            H5O__pline_free(dst);
    } 

    FUNC_LEAVE_NOAPI(ret_value)
} 

static size_t
H5O__pline_size(const H5F_t H5_ATTR_UNUSED *f, const void *mesg)
{
    const H5O_pline_t *pline = (const H5O_pline_t *)mesg; 
    size_t             i;                                 
    size_t             ret_value = 0;                     

    FUNC_ENTER_PACKAGE_NOERR

    
    ret_value = (size_t)(1 +                                               
                         1 +                                               
                         (pline->version == H5O_PLINE_VERSION_1 ? 6 : 0)); 

    
    for (i = 0; i < pline->nused; i++) {
        size_t      name_len; 
        const char *name;     

        
        if (pline->version > H5O_PLINE_VERSION_1 && pline->filter[i].id < H5Z_FILTER_RESERVED)
            name_len = 0;
        else {
            
            if (NULL == (name = pline->filter[i].name)) {
                H5Z_class2_t *cls; 

                H5Z_find(true, pline->filter[i].id, &cls);
                if (cls)
                    name = cls->name;
            }
            name_len = name ? strlen(name) + 1 : 0;
        } 

        ret_value +=
            2 + 
            (size_t)((pline->version == H5O_PLINE_VERSION_1 || pline->filter[i].id >= H5Z_FILTER_RESERVED)
                         ? 2
                         : 0) + 
            2 +                 
            2 +                 
            (pline->version == H5O_PLINE_VERSION_1 ? (size_t)H5O_ALIGN_OLD(name_len)
                                                   : name_len); 

        ret_value += pline->filter[i].cd_nelmts * 4;
        if (pline->version == H5O_PLINE_VERSION_1)
            if (pline->filter[i].cd_nelmts % 2)
                ret_value += 4;
    } 

    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5O__pline_reset(void *mesg)
{
    H5O_pline_t *pline = (H5O_pline_t *)mesg; 
    size_t       i;                           

    FUNC_ENTER_PACKAGE_NOERR

    

    assert(pline);

    
    if (pline->filter) {
        
        for (i = 0; i < pline->nused; i++) {
            if (pline->filter[i].name && pline->filter[i].name != pline->filter[i]._name)
                assert((strlen(pline->filter[i].name) + 1) > H5Z_COMMON_NAME_LEN);
            if (pline->filter[i].name != pline->filter[i]._name)
                pline->filter[i].name = (char *)H5MM_xfree(pline->filter[i].name);
            if (pline->filter[i].cd_values && pline->filter[i].cd_values != pline->filter[i]._cd_values)
                assert(pline->filter[i].cd_nelmts > H5Z_COMMON_CD_VALUES);
            if (pline->filter[i].cd_values != pline->filter[i]._cd_values)
                pline->filter[i].cd_values = (unsigned *)H5MM_xfree(pline->filter[i].cd_values);
        } 

        
        pline->filter = (H5Z_filter_info_t *)H5MM_xfree(pline->filter);
    }

    
    pline->nused = pline->nalloc = 0;

    
    pline->version = H5O_PLINE_VERSION_1;

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

static herr_t
H5O__pline_free(void *mesg)
{
    FUNC_ENTER_PACKAGE_NOERR

    assert(mesg);

    mesg = H5FL_FREE(H5O_pline_t, mesg);

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

static herr_t
H5O__pline_pre_copy_file(H5F_t H5_ATTR_UNUSED *file_src, const void *mesg_src, bool H5_ATTR_UNUSED *deleted,
                         const H5O_copy_t *cpy_info, void *_udata)
{
    const H5O_pline_t         *pline_src = (const H5O_pline_t *)mesg_src;       
    H5O_copy_file_ud_common_t *udata     = (H5O_copy_file_ud_common_t *)_udata; 
    herr_t                     ret_value = SUCCEED;                             

    FUNC_ENTER_PACKAGE

    
    assert(pline_src);
    assert(cpy_info);
    assert(cpy_info->file_dst);

    
    if (pline_src->version > H5O_pline_ver_bounds[H5F_HIGH_BOUND(cpy_info->file_dst)])
        HGOTO_ERROR(H5E_OHDR, H5E_BADRANGE, FAIL, "pline message version out of bounds");

    
    if (udata)
        if (NULL == (udata->src_pline = (H5O_pline_t *)H5O__pline_copy(pline_src, NULL)))
            HGOTO_ERROR(H5E_PLINE, H5E_CANTINIT, FAIL, "unable to copy");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5O__pline_debug(H5F_t H5_ATTR_UNUSED *f, const void *mesg, FILE *stream, int indent, int fwidth)
{
    const H5O_pline_t *pline = (const H5O_pline_t *)mesg;

    FUNC_ENTER_PACKAGE_NOERR

    
    assert(f);
    assert(pline);
    assert(stream);
    assert(indent >= 0);
    assert(fwidth >= 0);

    Rfprintf(stream, "%*s%-*s %llu/%llu\n", indent, "", fwidth, "Number of filters:",
            (unsigned long long)pline->nused, (unsigned long long)pline->nalloc);

    
    for (size_t i = 0; i < pline->nused; i++) {
        
        char name[64];

        memset(name, 0, 64);
        snprintf(name, sizeof(name), "Filter at position %llu", (unsigned long long)i);

        Rfprintf(stream, "%*s%-*s\n", indent, "", fwidth, name);
        Rfprintf(stream, "%*s%-*s 0x%04x\n", indent + 3, "", MAX(0, fwidth - 3),
                "Filter identification:", (unsigned)(pline->filter[i].id));
        if (pline->filter[i].name)
            Rfprintf(stream, "%*s%-*s \"%s\"\n", indent + 3, "", MAX(0, fwidth - 3),
                    "Filter name:", pline->filter[i].name);
        else
            Rfprintf(stream, "%*s%-*s NONE\n", indent + 3, "", MAX(0, fwidth - 3), "Filter name:");
        Rfprintf(stream, "%*s%-*s 0x%04x\n", indent + 3, "", MAX(0, fwidth - 3),
                "Flags:", pline->filter[i].flags);
        Rfprintf(stream, "%*s%-*s %llu\n", indent + 3, "", MAX(0, fwidth - 3),
                "Num CD values:", (unsigned long long)pline->filter[i].cd_nelmts);

        
        for (size_t j = 0; j < pline->filter[i].cd_nelmts; j++) {
            char field_name[32];

            snprintf(field_name, sizeof(field_name), "CD value %lu", (unsigned long)j);
            Rfprintf(stream, "%*s%-*s %u\n", indent + 6, "", MAX(0, fwidth - 6), field_name,
                    pline->filter[i].cd_values[j]);
        }
    }

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

herr_t
H5O_pline_set_version(H5F_t *f, H5O_pline_t *pline)
{
    unsigned version;             
    herr_t   ret_value = SUCCEED; 

    FUNC_ENTER_NOAPI(FAIL)

    
    assert(f);
    assert(pline);

    
    version = MAX(pline->version, H5O_pline_ver_bounds[H5F_LOW_BOUND(f)]);

    
    if (version > H5O_pline_ver_bounds[H5F_HIGH_BOUND(f)])
        HGOTO_ERROR(H5E_PLINE, H5E_BADRANGE, FAIL, "Filter pipeline version out of bounds");

    
    pline->version = version;

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 
