% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/ba_analysis.R
\name{ba_analysis}
\alias{ba_analysis}
\title{Bland-Altman Analysis for Method Comparison}
\usage{
ba_analysis(
  x,
  y = NULL,
  data = NULL,
  conf_level = 0.95,
  type = c("absolute", "percent"),
  na_action = c("omit", "fail")
)
}
\arguments{
\item{x}{Numeric vector of measurements from method 1 (reference method),
or a formula of the form \code{method1 ~ method2}.}

\item{y}{Numeric vector of measurements from method 2 (test method).
Ignored if \code{x} is a formula.}

\item{data}{Optional data frame containing the variables specified in
\code{x} and \code{y} (or in the formula).}

\item{conf_level}{Confidence level for intervals (default: 0.95).}

\item{type}{Type of difference calculation: \code{"absolute"} (default) for
\code{y - x}, or \code{"percent"} for \code{100 * (y - x) / mean}.}

\item{na_action}{How to handle missing values: \code{"omit"} (default) removes
pairs with any NA, \code{"fail"} stops with an error.}
}
\value{
An object of class \code{c("ba_analysis", "valytics_comparison", "valytics_result")},
which is a list containing:

\describe{
\item{input}{List with original data and metadata:
\itemize{
\item \code{x}, \code{y}: Numeric vectors (after NA handling)
\item \code{n}: Number of paired observations
\item \code{n_excluded}: Number of pairs excluded due to NAs
\item \code{var_names}: Named character vector with variable names
}
}
\item{results}{List with statistical results:
\itemize{
\item \code{differences}: Numeric vector of differences (y - x or percent)
\item \code{averages}: Numeric vector of means ((x + y) / 2)
\item \code{bias}: Mean difference (point estimate)
\item \code{bias_se}: Standard error of the bias
\item \code{bias_ci}: Named numeric vector with lower and upper CI for bias
\item \code{sd_diff}: Standard deviation of differences
\item \code{loa_lower}: Lower limit of agreement (bias - 1.96 * SD)
\item \code{loa_upper}: Upper limit of agreement (bias + 1.96 * SD)
\item \code{loa_lower_ci}: Named numeric vector with CI for lower LoA
\item \code{loa_upper_ci}: Named numeric vector with CI for upper LoA
}
}
\item{settings}{List with analysis settings:
\itemize{
\item \code{conf_level}: Confidence level used
\item \code{type}: Type of difference calculation
\item \code{multiplier}: Multiplier for LoA (default: 1.96 for 95\\%)
}
}
\item{call}{The matched function call.}
}
}
\description{
Performs Bland-Altman analysis to assess agreement between two measurement
methods. Calculates bias (mean difference), limits of agreement, and
confidence intervals following the approach of Bland & Altman (1986, 1999).
}
\details{
The Bland-Altman method assesses agreement between two quantitative
measurements by analyzing the differences against the averages. The key
outputs are:

\itemize{
\item \strong{Bias}: The mean difference between methods, indicating systematic
difference. A bias significantly different from zero suggests one method
consistently measures higher or lower than the other.
\item \strong{Limits of Agreement (LoA)}: The interval within which 95\\% of
differences are expected to lie (bias +/- 1.96 x SD). These define the
range of disagreement between methods.
\item \strong{Confidence Intervals}: CIs for bias and LoA quantify the
uncertainty in these estimates due to sampling variability.
}

The confidence intervals for limits of agreement are calculated using the
exact method from Bland & Altman (1999), which accounts for the uncertainty
in both the mean and standard deviation.
}
\section{Assumptions}{

The standard Bland-Altman analysis assumes:
\itemize{
\item Differences are approximately normally distributed
\item No proportional bias (constant bias across the measurement range)
\item No repeated measurements per subject
}
}

\examples{
# Simulated method comparison data
set.seed(42)
method_a <- rnorm(50, mean = 100, sd = 15)
method_b <- method_a + rnorm(50, mean = 2, sd = 5)  # Method B has +2 bias

# Basic analysis
ba <- ba_analysis(method_a, method_b)
ba

# Using formula interface with data frame
df <- data.frame(reference = method_a, test = method_b)
ba <- ba_analysis(reference ~ test, data = df)

# Percentage differences
ba_pct <- ba_analysis(method_a, method_b, type = "percent")

}
\references{
Bland JM, Altman DG (1986). Statistical methods for assessing agreement
between two methods of clinical measurement. \emph{Lancet}, 1(8476):307-310.
\doi{10.1016/S0140-6736(86)90837-8}

Bland JM, Altman DG (1999). Measuring agreement in method comparison studies.
\emph{Statistical Methods in Medical Research}, 8(2):135-160.
\doi{10.1177/096228029900800204}
}
\seealso{
\code{\link[=plot.ba_analysis]{plot.ba_analysis()}} for visualization,
\code{\link[=summary.ba_analysis]{summary.ba_analysis()}} for detailed summary
}
