#' @importFrom utils combn
#' @title Dummy and null players
#' @description Given a game, this function identifies its dummy players and null players.
#' @param v A characteristic function, as a vector.
#' @param binary A logical value. By default, \code{binary=FALSE}. Should be set to \code{TRUE} if \code{v} is introduced in binary order instead of lexicographic order.
#' @param tol A tolerance parameter, as a non-negative number.\cr
#'            By default, \code{tol=100*.Machine$double.eps}.
#' @return Two different vectors are returned: one containing the dummy players and the other containing the null players.
#' @details Given a game \eqn{v\in G^N}, \eqn{i \in N} is said to be a dummy player
#' if \eqn{v(S) + v(\{i\}) = v(S \cup \{i\})} for all \eqn{S \subset N \setminus \{i\}}.
#'
#' A dummy player \eqn{i \in N} is said to be a null player if \eqn{v(\{i\})=0}.
#' @examples
#' v <- c(0,1,0,1,0,1,1)
#' dummynull(v)
#' # Checking if a particular player is a dummy player:
#' 2 %in% dummynull(v)$dummy # player 2 is a dummy player in v
#' 2 %in% dummynull(v)$null # player 2 is not a null player in v
#' @export

dummynull <- function(v, binary = FALSE, tol = 100*.Machine$double.eps) {

  nC <- length(v)
  n <- log2(nC + 1)
  if (n > floor(n)) {
    stop("'v' must have length 2^n-1 for some n.")
  }
  if (binary == FALSE) { # Si el juego se introdujo en lexicográfico, lo pasamos a binario.
    v <- lex2bin(v)
  }

  notiteres <- c()
  for (i in 1:n) { # empieza bucle que identifica jugadores no títeres
    # Obtenemos las coaliciones en las que está el jugador i.
    others <- setdiff(1:n,i) # jugadores que no son i
    combs <- unlist( # todas las combinaciones de others, como lista
      lapply(1:length(others), function(k) combn(others, k, simplify = FALSE)),
      recursive = FALSE
    )
    # Comprobaciones v(S)+v({i})=v(Su{i}).
    check <- TRUE # por defecto, i es nulo
    for (S in 1:length(combs)) {
      vi <- v[2^(i-1)]
      vS <- v[getcoalitionnumber(combs[[S]])]
      vSi <- v[getcoalitionnumber(c(i,combs[[S]]))]

      if (abs(vi+vS-vSi) > tol) {
        notiteres <- c(notiteres,i) # si v(S)+v({i})!=v(Su{i}), i es no títere
        break # pasamos a estudiar el jugador i+1
      }
    }

  } # termina bucle que identifica jugadores no títeres

  # Los títeres son los que no son no titeres.
  # Si no hay títeres, tampoco hay nulos.
  if (length(notiteres)==n) {
    return(list(dummy = NULL, null = NULL))
  }
  # Si hay títeres, vemos cuáles de ellos son nulos (v(i)=0).
  titeres <- setdiff(1:n,notiteres)
  nulos <- c()
  for (i in 1:length(titeres)) {
    if (abs(v[2^(i-1)]) < tol) {
      nulos <- c(nulos,titeres[i])
    }
  }
  return(list(dummy = titeres, null = nulos))

}
