#' @title Lorenz dominance relation
#' @description Given two awards vectors, this function returns the Lorenz dominance relation between them.
#' @param x A vector.
#' @param y A vector.
#' @return There are four possible outputs:
#'
#' \item{-1}{if the introduced vectors are not Lorenz-comparable.}
#'
#' \item{0}{if the vectors are Lorenz-equal.}
#'
#' \item{1}{if the vectors are not Lorenz-equal and the first one Lorenz-dominates the second one.}
#'
#' \item{2}{if the vectors are not Lorenz-equal and the second one Lorenz-dominates the first one.}
#' @details In order to compare two vectors \eqn{x,y\in \mathbb{R}^n} through the Lorenz criterion,
#' both of them must be rearranged in non-decreasing order; thus, let \eqn{\bar{x}} and \eqn{\bar{y}}
#' be the vectors obtained by rearranging \eqn{x} and \eqn{y}, respectively, in non-decreasing order.
#' It is said that \eqn{x} Lorenz-dominates \eqn{y} (or that \eqn{y} is Lorenz-dominated by \eqn{x})
#' if all the cumulative sums of \eqn{\bar{x}} are not less than those of \eqn{\bar{y}}.
#' That is, \eqn{x} Lorenz-dominates \eqn{y} if \eqn{\sum_{j=1}^{n}\bar{x}_j=\sum_{j=1}^{n}\bar{y}_j}
#' and, for each \eqn{k=1,\dots,n-1}, \deqn{\sum_{j=1}^{k}\bar{x}_j \geq \sum_{j=1}^{k}\bar{y}_j.}
#'
#' If \eqn{x} Lorenz-dominates \eqn{y} and \eqn{y} Lorenz-dominates \eqn{x},
#' then \eqn{x} and \eqn{y} are said to be Lorenz-equal.
#'
#' If \eqn{x} does not Lorenz-dominate \eqn{y} and \eqn{y} does not Lorenz-dominate \eqn{x},
#' then \eqn{x} and \eqn{y} are not Lorenz-comparable.
#' @examples
#' lorenzdominancerelation(c(1,2,3), c(1,1,4))
#' lorenzdominancerelation(c(1,2,7,2), c(1,1,4,6))
#' @references Lorenz, M. O. (1905). Methods of Measuring the Concentration of Wealth. \emph{Publications of the American Statistical Association}, 9(70), 209-219.
#' @export

lorenzdominancerelation <- function(x, y) {

  ################################
  ###Comprobación datos entrada###
  ################################

  n=length(x)
  ny=length(y)
  tol <- 100*.Machine$double.eps

  if(n!=ny)
  {
    stop("'x' and 'y' must have the same length.")
  }

  if(abs(sum(x)-sum(y))>tol)
  {
    stop("The sum of 'x' and the sum of 'y' must be equal.")
  }
  ################################
  #####Creación de variables######
  ################################

  Pre=7 #Nivel de precisión.


  if (sum(x==sort(x))!=n || sum(y==sort(y))!=n)
  {#Ordenamos los vectores de menor a mayor. Si no están ordenados, sacamos un mensaje diciendo que el resultado es para vectores ordenados.
    message('The coordinates of at least one of the vectors have been rearranged in non-decreasing order.\n')
    x=sort(x)
    y=sort(y)
  }

  ################################
  ######Cuerpo de la función######
  ################################

  sum1 = cumsum(x)
  sum2 = cumsum(y)
  dom1 = sum(round(sum1,Pre) >= round(sum2,Pre))
  dom2 = sum(round(sum2,Pre) >= round(sum1,Pre))

  if (dom1 == n & dom2 < n)
  {
    cod <- 1 # first dominates second
    mensaje <- paste("The first vector Lorenz-dominates the second vector.")
  } else if (dom2 == n & dom1 < n) {
    cod <- 2 # second dominates first
    mensaje <- paste("The second vector Lorenz-dominates the first vector.")
  } else if (dom1 == n & dom2 == n) {
    cod <- 0
    mensaje <- paste("The vectors are Lorenz-equal.")
  } else {
    cod <- -1 # not Lorenz-comparable
    mensaje <- paste("The vectors are not Lorenz-comparable.")
  }


  ################################
  ######Salidas de la función#####
  ################################
  message(mensaje)
  return(cod)
}# Fin de la función

