#' @title Per capita nucleolus
#' @description Given a game, this function computes its per capita nucleolus.
#' @param v A characteristic function, as a vector.
#' @param binary A logical value. By default, \code{binary=FALSE}. Should be set to \code{TRUE} if \code{v} is introduced in binary order instead of lexicographic order.
#' @param tol A tolerance parameter, as a non-negative number.\cr
#'            By default, \code{tol=100*.Machine$double.eps}.
#' @return The per capita nucleolus of the game, as a vector.
#' @details Given a game \eqn{v\in G^N} and an allocation \eqn{x \in I(v)}, the per capita excess
#' of each coalition \eqn{S \in 2^{N}} with respect to \eqn{x} is defined as
#' \deqn{e^{p}(v,x,S) = \frac{v(S)-\sum_{i \in S}x_{i}}{|S|}.}
#' The per capita excesses of all non-empty coalitions, sorted in non-increasing order, are stored
#' in the per capita excesses vector, \eqn{\theta^{p}(x)}.
#' For any game \eqn{v\in G^N} with a non-empty set of imputations, the per capita nucleolus
#' is defined as the only imputation \eqn{pcn(v) \in I(v)} that satisfies
#' \eqn{\theta^{p}(pcn(v))_{i} \leqslant \theta^{p}(y)_{i}} for each \eqn{i \in \{1,\dots,2^{N}-1\}}
#' and for all \eqn{y \in I(v)}.
#' This function is programmed following the algorithm of Potters, J.A., et al. (1996).
#' @examples
#' nucleoluspcvalue(c(1,5,10,6,11,15,16))
#' nucleoluspcvalue(c(0,0,0,30,30,80,100))
#'
#' # Computing the per capita nucleolus of a random essential game:
#' n <- 10 # number of players in the game
#' v <- c(rep(0,n),runif(2^n-(n+1),min=10,max=20)) # random essential game
#' nucleoluspcvalue(v)
#'
#' # What if the game is a cost game?
#' cost.v <- airfieldgame(c(1,5,10,15)) # cost game
#' -nucleoluspcvalue(-cost.v) # per capita nucleolus of the cost game
#' @references Grotte, J. (1970). \emph{Computation of and Observations on the Nucleolus, the Normalized Nucleolus and the Central Games}. Master’s thesis), Cornell University, Ithaca.
#' @references Potters, J. A., Reijnierse, J. H., & Ansing, M. (1996). Computing the nucleolus by solving a prolonged simplex algorithm. \emph{Mathematics of Operations Research}, 21(3), 757-768.
#' @seealso \link{excesses}, \link{leastcore}, \link{nucleolusvalue}, \link{prenucleolusvalue}
#' @export

nucleoluspcvalue <- function(v, binary = FALSE, tol = 100*.Machine$double.eps){

  nC=length(v)
  n=log(nC+1)/log(2)

  if(n>floor(n)) {
    stop("'v' must have length 2^n-1 for some n.")
  }

  if (binary == FALSE) { # Si el juego se introdujo en lexicográfico, lo pasamos a binario.
    v <- lex2bin(v)
  }

  ################################
  #####Creación de variables######
  ################################

  nucleoluspc=c()
  vind=(v[2.^(seq(1,n)-1)])

  #Check if the game is essential.
  if(v[nC]-sum(vind)+tol<0) {
    stop("'v' must be essential.")
  } else if(abs(v[nC]-sum(vind))<tol) { #Si el juego es degenerado, el nucleolus coincide con los valores individuales de la f. caracteristica.
    nucleolus=v[vind]
  }

  #Modificaciones de la función característica:
  #Juego 0-normalizado:
  v=zeronormalizedgame(v,binary=TRUE) # aquí tuve que meter binary=TRUE

  #El nucleolo per capita de un juego no cambia si ponemos a cero las coaliciones con valor negativo
  v[v<0]=0

  ################################
  ######Cuerpo de la función######
  ################################

  #Escogemos i=n.
  # c=max(v[1:(nC-1)])

  #Creamos el simplex (usamos la notación del artículo):
  #1) Las variables z en orden binario,
  #2) La variables xi (correspondiente al jugador elegido). En este caso i=n.
  #3) Las otras variables x (x1,...,x(n-1)). Las de los jugadores no elegidos.
  #4) La variable t, que añadiremos en el bucle, irá en la posición nC+n+1
  #El orden es el siguiente:
  # z1, z2, z3,..., z(nC-1) xn  x1,...,x(n-1),   t
  # La variable t la añadimos dentro del bucle

  # Las variables básicas son: xn y todas las z's

  IndicesBasicas=c(1:nC)
  IndicesNoBasicas=c((nC+1):(nC+n-1))

  # La matriz correspondiente a las variables básicas es la identidad, pero no la guardamos para ahorrar espacio.
  # Calculamos solo la matriz de coeficientes para las variables no basicas.
  #
  #
  ################################################
  # Aquí van los cambios con respecto al nucleolus
  ################################################
  A <- matrix(0,2^n-2,n)
  A <- rbind(A, c(rep(1, n - 1), v[length(v)]))  # Añadiendo la última fila
  caux <- numeric(n - 1)  # Vector auxiliar de coeficientes


  #Coaliciones que no contienen al jugador i=n. Estas van al principio.
  for (ii in 1:(2^(n-1)-1))  # Só é preciso obter a metade das coalicións (nas que non está i=n)
  {
   # A[ii,]=c(-as.numeric(intToBits(ii)[1:(n-1)]), c-v[ii])
    coal <- as.integer(intToBits(ii)[1:(n-1)])  # Obtiene la coalición en binario
    s <- sum(coal)  # Número de jugadores en la coalición
    caux[ii] <- v[ii ]/ s  # valor pc da coalición ii
    caux[nC - ii] <- v[nC - ii ]/ (n - s) ## valor de la complementaria

    # Construcción de las filas de A
    A[ii, ] <- c(-coal / s, -v[ii ] / s)  # Fila de A correspondente a y_S -X(S)/|S| = c-v(S)/|S|, i = n \notin S. Falta c, que se añade en un paso final
    A[nC - ii, ] <- c(coal / (n - s), v[length(v)] / (n - s) - v[nC - ii ] / (n - s))
  }

  # Añadiendo el término c en la última columna de A
  A[1:(nrow(A)-1), n] <- A[1:(nrow(A)-1), n] + max(caux)

  # Hasta aquí
  ################################################

  #Ya tenemos todos los parámetros del simplex. Ahora vamos a resolverlo.

  #iterada=0
  while(length(IndicesBasicas)>n)
  {#Cuando tengamos n variables básicas, estas serán elementales y tendremos la solución.
    #iterada=iterada+1 #No debería haber más de n-1 iteraciones

    # Añadimos la nueva columna t (la suma de los coeficientes de las variables z's en cada fila)
    # La ponemos como la última variable, y por tanto justo antes del término independiente.
    if(any(IndicesNoBasicas<=nC-1))
    {#Aquí no es como en matlab. Si no hay ninguna no básica para sumar da error.
      A=cbind(A[,1:(dim(A)[2]-1)], t(t(as.numeric(IndicesBasicas<=nC-1)))+rowSums(as.matrix(A[,which(IndicesNoBasicas<=nC-1)])),A[,dim(A)[2]])
      A[abs(A) < tol] <- 0 # reemplazamos por 0 todos los elementos de A cuyo valor absoluto sea menor que TOL ### MODIFICADO, REV NUCLEOLUS
    }else
    {
     A=cbind(A[,1:dim(A)[2]-1],as.numeric(IndicesBasicas<=nC-1),A[,dim(A)[2]])
    }
    # Añadimos la variable t a las no basicas
    IndicesNoBasicas=c(IndicesNoBasicas,nC+n)

    ####OPTIMIZACIÓN####
    # Tamaño del tableau q= filas (ecuaciones)  , t+1= columnas
    q=dim(A)[1]
    t=dim(A)[2]-1
    #Paso 1
    #Pasos=1;
    # Metemos la variable t en la base. Pivotaremos en la columna t
    ColPivote=as.matrix(A[,t])

    # Buscamos la fila para pivotar. Además esta será la fila que, en los demás pasos,
    # desempeñará el papel de la última fila del tableau del simplex (ya que coincide con la fila de las r's).
    # Buscaremos pues los elementos con coordenadas estrictamente positivas para incluirlos
    # en la base.

    w=A[,t+1]/ColPivote # Cocientes columna n+1 entre columna t

    w[ColPivote <= tol] <- Inf # Los cocientes correspondientes a coeficientes negativos o nulos de la columna pivote no los consideramos.
    Filas= which(w==min(w))
    r=which(IndicesBasicas[Filas]==min(IndicesBasicas[Filas]))
    r=Filas[r]

    #Por lo tanto, pivotamos en fila r y columna t.
    # Guardamos la fila pivote

    Faux=A[r,]; #Fila auxiliar
    # Cambiamos la columna de la variable t por el vector r de la base canonica
    A[,t]=rep(0,q)
    A[r,t]=1;
    # Pivotamos en el elemento (r,t) de la matriz A
    Aux=c()
    for(i in 1:length(Faux))
    {
      Aux=cbind(Aux,ColPivote/ColPivote[r]*A[r,i])
    }
    A=A-Aux;
    # Recomponemos la fila r
    A[r,]=Faux/ColPivote[r]
    A[r,t]=1/ColPivote[r]
    A[abs(A) < tol] <- 0  ### MODIFICADO, REV NUCLEOLUS
    # Actualizamos las variables basicas y no basicas
    momentanea=IndicesBasicas[r]
    IndicesBasicas[r]=nC+n
    IndicesNoBasicas[t]=momentanea

    control=which(A[r,1:t]<  -tol);

    while(length(control)>0)
    {#Mientras haya elementos negativos en la fila pivote, seguimos operando.
      #Pasos=Pasos+1;
      #Usamos la Regla de Bland (entra en la base la columna de menor índice con el
      # coeficiente de control estrictamente negativo)
      C=which(IndicesNoBasicas[control]==max(IndicesNoBasicas[control]))
      C=control[C]; # Meteremos la columna C en la base
      ColPivote=A[,C]; #ColPivote[abs(ColPivote)<tol] <- 0
      # Buscamos la fila para pivotar. No volveremos a pivotar en la fila r
      w=A[,t+1]/ColPivote

      #Para elegir la fila en la que pivotar tenemos que considerar sólo
      # los cocientes correspondientes a valores de la columna pivote estrictamente mayores que cero.
      # Como vamos a calcular el mínimo de estos cocientes, los que no nos interesan los ponemos iguales a Inf.
      # Para evitar elegir la fila r le damos también valor Inf al cociente.
      w[ColPivote<=tol]<-Inf; w[r]<-Inf ### MODIFICADO, REV NUCLEOLUS
      # Regla de Bland (De entre las filas con el mínimo cociente no negativo, sale de la base la de menor índice)
      Filas= which(w==min(w));
      R=which(IndicesBasicas[Filas]==max(IndicesBasicas[Filas]))#F se confunde con FALSE
      R=Filas[R]
      #Empezamos a pivotar como antes
      # pivot=A[R,C];
      Faux=A[R,];
      # Cambiamos la  columna C por el vector R de la base canonica
      A[,C]=rep(0,q)
      A[R,C]=1;
      # Pivotamos en el elemento (R,C) de la matriz A
      Aux=c()
      for(i in 1:length(Faux))
      {
        Aux=cbind(Aux,ColPivote/ColPivote[R]*A[R,i])
      }
      A=A-Aux;
      #Pivotamos el elemento R de d
      A[R,]=Faux/ColPivote[R]
      A[R,C]=1/ColPivote[R];
      A[abs(A)<tol] <- 0 ### MODIFICADO, REV NUCLEOLUS
      # Actualizamos las variables básicas y no básicas
      momentanea=IndicesBasicas[R];
      IndicesBasicas[R]=IndicesNoBasicas[C];
      IndicesNoBasicas[C]=momentanea;
      # Actualizamos el control
      control=which(A[r,1:t]<  -tol);
    }

    ####Ahora empezamos el proceso de simplificación del simplex####

    #Eliminamos variables (columnas)
    # Variables con coeficientes estrictamente positivos en la fila r de la
    # matriz A (la que da el valor optimo de t en el paso anterior)
    ColBorrar=IndicesNoBasicas[A[r,1:(dim(A)[2]-1)]>0];

    # Si se corresponde con una variable ys entonces se elimina la columna
    # Si se corresponde con una variable xj, se pone a cero toda la columna y
    # se añade la ecuación xj=0 (que será básica). O sea, xj pasa a ser básica
    # y se elimina su columna en la matriz A

    for(ii in ColBorrar)
    {
      if(ii<=(nC-1))
      {
        A=A[,-which(IndicesNoBasicas==ii)]
        IndicesNoBasicas=IndicesNoBasicas[-which(IndicesNoBasicas==ii)];
      }else
      {
        # Añadimos una nueva ecuacion xi=0, que será ahora la última fila de A
        # Luego xi pasa a ser variable básica
        A=A[,-which(IndicesNoBasicas==ii)]
        A=rbind(A,rep(0,dim(A)[2]))
        IndicesNoBasicas=IndicesNoBasicas[-which(IndicesNoBasicas==ii)]
        IndicesBasicas=c(IndicesBasicas,ii);
      }
    }
    #Eliminamos ecuaciones (filas)

    # Eliminamos la fila r
    A=as.matrix(A)
    A=A[-r,]
    # Eliminamos la columna t
    IndicesBasicas=IndicesBasicas[-which(IndicesBasicas==nC+n)]
    # Eliminamos las filas elementales (y las correspondientes columnas)
    # Si A solo tiene una columna (la de los términos independientes)
    # hemos acabado
    A=as.matrix(A)
    if(dim(A)[2]==1)
    {
      # Todas las variables restantes son basicas. Ya solo me queda la identidad. Elimino las variables que se
      # corresponden con variables ys
      elementales=IndicesBasicas<=nC-1;
      elementales=as.numeric(elementales)
      A=A[which(elementales==0),];
      IndicesBasicas=IndicesBasicas[which(elementales==0)]
    }else
    {
      # Primero vemos que ecuaciones de A tienen todos los coeficientes de las variables nulos
      S=rowSums(as.matrix(A[,1:dim(A)[2]-1]!=0));
      # Las otras no pueden ser elementales, ya que al menos tendrán otro coeficiente no nulo correspondiente a la variable básica.
      # De estas, aquellas en las que S=0,  sólo serán elementales aquellas que tengan como variable básica
      # una ys. (Recordemos que las ys van numeradas de 1 a nC-1.

      elementales=(IndicesBasicas*as.numeric(S==0)>0)&(IndicesBasicas*as.numeric(S==0)<=nC-1);
      elementales=as.numeric(elementales)
      A=A[which(elementales==0),];
      IndicesBasicas=IndicesBasicas[which(elementales==0)];
    }
  }

  O=sort(IndicesBasicas,index.return=T)$ix
  nucleoluspc=A[c(O[2:length(O)],O[1])]
  nucleoluspc=vind+nucleoluspc
  ################################
  ######Salidas de la función#####
  ################################

  return(nucleoluspc)

}# Fin de la función

